/* Routines for setup and initialization of H-type functions */

#include "internal.h"

/* ****************************************************************** */

TVIL_COMPLEX TVIL_H00000x (TVIL_REAL x, TVIL_REAL qq)
{
  TVIL_COMPLEX lnbarx = TVIL_LOG (TVIL_FABS(x)/qq);
  if (x<0) lnbarx = lnbarx - I * PI;

  return PI2*PI2/30.L + 6.L*Zeta3 - 6.L*lnbarx*Zeta3;
}

/* ****************************************************************** */

TVIL_COMPLEX TVIL_H0000xx (TVIL_REAL x, TVIL_REAL qq)
{
  TVIL_COMPLEX lnbarx = TVIL_LOG (TVIL_FABS(x)/qq);
  if (x<0) lnbarx = lnbarx - I * PI;

  return -PI2*PI2/18.L + 6.L*Zeta3 - 6.L*lnbarx*Zeta3;
}

/* ****************************************************************** */

TVIL_COMPLEX TVIL_H00xx00 (TVIL_REAL x, TVIL_REAL qq)
{
  TVIL_COMPLEX lnbarx = TVIL_LOG (TVIL_FABS(x)/qq);
  if (x<0) lnbarx = lnbarx - I * PI;

  return 16.L*A4TRANS - (7.L*PI2*PI2)/60.L - (2.L*PI2*LN2*LN2)/3.L +
      (2.L*TVIL_POW(LN2,4))/3.L + 6.L*Zeta3 - 6.L*lnbarx*Zeta3;
}

/* ****************************************************************** */

TVIL_COMPLEX TVIL_H000xxx (TVIL_REAL x, TVIL_REAL qq)
{
  TVIL_COMPLEX lnbarx = TVIL_LOG (TVIL_FABS(x)/qq);
  if (x<0) lnbarx = lnbarx - I * PI;

  return (-11.L*PI2*PI2)/180.L - 9.L*LS2*LS2 + 6.L*Zeta3 - 6.L*lnbarx*Zeta3;
}

/* ****************************************************************** */

TVIL_COMPLEX TVIL_H00x0xx (TVIL_REAL x, TVIL_REAL qq)
{
  TVIL_COMPLEX lnbarx = TVIL_LOG (TVIL_FABS(x)/qq);
  if (x<0) lnbarx = lnbarx - I * PI;

  return -PI2*PI2/10.L + 6.L*Zeta3 - 6.L*lnbarx*Zeta3;
}

/* ****************************************************************** */

TVIL_COMPLEX TVIL_H00xx0x (TVIL_REAL x, TVIL_REAL qq)
{
  TVIL_COMPLEX lnbarx = TVIL_LOG (TVIL_FABS(x)/qq);
  if (x<0) lnbarx = lnbarx - I * PI;

  return -PI2*PI2/24.L - (27.L*LS2*LS2)/2.L + 6.L*Zeta3 - 6.L*lnbarx*Zeta3;
}

/* ****************************************************************** */

TVIL_COMPLEX TVIL_H00xxxx (TVIL_REAL x, TVIL_REAL qq)
{
  TVIL_COMPLEX lnbarx = TVIL_LOG (TVIL_FABS(x)/qq);
  if (x<0) lnbarx = lnbarx - I * PI;

  return (-77.L*PI2*PI2)/1080.L - (27.L*LS2*LS2)/2.L + 6.L*Zeta3 - 6.L*lnbarx*Zeta3;
}

/* ****************************************************************** */

TVIL_COMPLEX TVIL_H0xxx0x (TVIL_REAL x, TVIL_REAL qq)
{
  TVIL_COMPLEX lnbarx = TVIL_LOG (TVIL_FABS(x)/qq);
  if (x<0) lnbarx = lnbarx - I * PI;

  return 32.L*A4TRANS - (11.L*PI2*PI2)/45.L - (4.L*PI2*LN2*LN2)/3.L +
      (4.L*TVIL_POW(LN2,4))/3.L + 6.L*Zeta3 - 6.L*lnbarx*Zeta3;
}

/* ****************************************************************** */

TVIL_COMPLEX TVIL_H0xxxxx (TVIL_REAL x, TVIL_REAL qq)
{
  TVIL_COMPLEX lnbarx = TVIL_LOG (TVIL_FABS(x)/qq);
  if (x<0) lnbarx = lnbarx - I * PI;

  return (7.L*PI2*PI2)/30.L - 6.L*LS2*LS2 + 4.L*PI*LS3 - 
    6.L*LSP4 + 6.L*Zeta3 - (26.L*TVIL_LOG(3.L)*Zeta3)/3.L - 6.L*lnbarx*Zeta3;
}

/* ****************************************************************** */

TVIL_COMPLEX TVIL_Hxxxxxx (TVIL_REAL x, TVIL_REAL qq)
{
  TVIL_COMPLEX lnbarx = TVIL_LOG (TVIL_FABS(x)/qq);
  if (x<0) lnbarx = lnbarx - I * PI;

  return 16.L*A4TRANS - (17.L*PI2*PI2)/90.L - (2.L*PI2*LN2*LN2)/3.L +
      (2.L*TVIL_POW(LN2,4))/3.L - 9.L*LS2*LS2 + 6.L*Zeta3 - 6.L*lnbarx*Zeta3;
}

/* ****************************************************************** */

TVIL_COMPLEX TVIL_H0000xy (TVIL_REAL X, TVIL_REAL Y, TVIL_REAL qq)
{
  TVIL_REAL temp, x, y, r, rn;
  TVIL_COMPLEX lnyox, lnyox2, lnyox3, result;
  int n;

  TVIL_REAL H0000xytableA[] = 
{0, -5.771508654754528, -0.9908202605290377, -0.4554997637691356, 
-0.2722553172561659, -0.18340689113100384, -0.13264301025945624, 
-0.10065188364596245, -0.07910335295050075, -0.06386299666644402, 
-0.05267152266389563, -0.044203457720624506, -0.037637189824228756, 
-0.03244045032314036, -0.02825566197451801, -0.024835167451514817, 
-0.022002920684058513, -0.019630936576686797, -0.017624326231044263, 
-0.015911510773175264, -0.014437658599678937, -0.013160185208758335, 
-0.012045605508904489, -0.011067292553681857, -0.010203855801378753, 
-0.00943795038884998, -0.008755391144546015, -0.008144485256976947, 
-0.007595523964870454, -0.007100391345518764, -0.006652260325854115, 
-0.006245354357771404, -0.00587475901885709, -0.00553627192336581, 
-0.005226282284630873, -0.004941673612817055, -0.004679744600887353, 
-0.004438144411544276, -0.0042148194431325114, -0.004007969303397238, 
-0.003816010213595261, -0.0036375444426091225, -0.0034713346609491907, 
-0.0033162823294021204, -0.0031714094124283415, -0.0030358428439824087, 
-0.0029088012819856734, -0.00278958377382435, -0.0026775600239582753, 
-0.0025721620098190234, -0.0024728767365571197, -0.00237923995711857, 
-0.0022908307133314987, -0.0022072665775220157, -0.002128199493719271};

  TVIL_REAL H0000xytableAL[] = 
{0, 6.289868133696452, 2.519934066848226, 1.707733822343262, 
1.320730922313002, 1.0847514045170683, 0.9232187630234829, 
0.8048386287528784, 0.7140153690363196, 0.6419756690081806, 
0.5833609945235882, 0.5346961183563276, 0.4936210803474843, 
0.4584735278777824, 0.42804693073942135, 0.40144343343261524, 
0.37798055165205013, 0.35712994121711616, 0.33847602069749777, 
0.32168729427003384, 0.30649603279040655, 0.2926835938334291, 
0.28006963046066846, 0.2685040345737776, 0.2578608371971804, 
0.24803353146632695, 0.23893144485609086, 0.2304768953898546, 
0.2226029406763021, 0.21525158018351165, 0.20837230756127628, 
0.20192093586638576, 0.19585863741054743, 0.19015115377287606, 
0.18476814175476985, 0.1796826287106614, 0.17487055646735347, 
0.17031039744545307, 0.16598282997502245, 0.1618704624114089, 
0.15795759769408763, 0.15423003158950496, 0.15067487912096633, 
0.14728042469138464, 0.14403599220610044, 0.14093183214699528, 
0.13795902306937347, 0.1351093854154818, 0.13237540588311347, 
0.12975017087012625, 0.1272273077481267, 0.1248009329106759, 
0.12246560570078144, 0.12021628745522983, 0.11804830501433407};

  TVIL_REAL H0000xytableAL2[] = 
{0, -1., -0.25, -0.1111111111111111, -0.0625, -0.04, -0.027777777777777776, 
-0.02040816326530612, -0.015625, -0.012345679012345678, -0.01, 
-0.008264462809917356, -0.006944444444444444, -0.005917159763313609, 
-0.00510204081632653, -0.0044444444444444444, -0.00390625, 
-0.0034602076124567475, -0.0030864197530864196, -0.002770083102493075, 
-0.0025, -0.0022675736961451248, -0.002066115702479339, 
-0.001890359168241966, -0.001736111111111111, -0.0016, 
-0.0014792899408284023, -0.0013717421124828531, -0.0012755102040816326, 
-0.0011890606420927466, -0.0011111111111111111, -0.001040582726326743, 
-0.0009765625, -0.0009182736455463728, -0.0008650519031141869, 
-0.0008163265306122449, -0.0007716049382716049, -0.0007304601899196494, 
-0.0006925207756232687, -0.0006574621959237344, -0.000625, 
-0.000594883997620464, -0.0005668934240362812, -0.0005408328826392645, 
-0.0005165289256198347, -0.0004938271604938272, -0.0004725897920604915, 
-0.0004526935264825713, -0.00043402777777777775, 
-0.00041649312786339027, -0.0004, -0.00038446751249519417, 
-0.0003698224852071006, -0.000355998576005696, -0.0003429355281207133};

  TVIL_REAL H0000xytableAL3[] = 
{0, 0.16666666666666666, 0.08333333333333333, 0.05555555555555555, 
0.041666666666666664, 0.03333333333333333, 0.027777777777777776, 
0.023809523809523808, 0.020833333333333332, 0.018518518518518517, 
0.016666666666666666, 0.015151515151515152, 0.013888888888888888, 
0.01282051282051282, 0.011904761904761904, 0.011111111111111112, 
0.010416666666666666, 0.00980392156862745, 0.009259259259259259, 
0.008771929824561403, 0.008333333333333333, 0.007936507936507936, 
0.007575757575757576, 0.007246376811594203, 0.006944444444444444, 
0.006666666666666667, 0.00641025641025641, 0.006172839506172839, 
0.005952380952380952, 0.005747126436781609, 0.005555555555555556, 
0.005376344086021506, 0.005208333333333333, 0.005050505050505051, 
0.004901960784313725, 0.004761904761904762, 0.004629629629629629, 
0.0045045045045045045, 0.0043859649122807015, 0.004273504273504274, 
0.004166666666666667, 0.0040650406504065045, 0.003968253968253968, 
0.003875968992248062, 0.003787878787878788, 0.003703703703703704, 
0.0036231884057971015, 0.0035460992907801418, 0.003472222222222222, 
0.003401360544217687, 0.0033333333333333335, 0.0032679738562091504, 
0.003205128205128205, 0.0031446540880503146, 0.0030864197530864196};

  TVIL_REAL H0000xytableB[] = 
{0, -3.606170709478783, 1.3030853547393915, -0.7020569031595942, 
0.44899638107339945, -0.3161415493031639, 0.23677227874029094, 
-0.18513020717421758, 0.14942386507238703, -0.12358586339347805, 
0.10421485228998595, -0.08927459372384941, 0.07748095467614455, 
-0.06798911436855568, 0.06022342172231385, -0.053779853118521376, 
0.04836757525410332, -0.04377260022553214, 0.03983443999339681, 
-0.03643068369021216, 0.033466554347372, -0.030867680989995816, 
0.02857499751036313, -0.026541078850438687, 0.024727467512195206, 
-0.023102694466010182, 0.021640794766869628, -0.020320180772936194, 
0.019122777326742807, -0.018033351205936965, 0.01703898628391136, 
-0.01612866913134285, 0.015292959146436497, -0.01452372397143166, 
0.013813925762716603, -0.01315744738809714, 0.012548950206661622, 
-0.011983757006052656, 0.011457755111565354, -0.010967315770349928, 
0.010509226744095143, -0.010080635681143882, 0.009679002332162678, 
-0.009302058057522974, 0.008947771375507549, -0.008614318537730273, 
0.00830005830630419, -0.008003510257300686, 0.007723336055267219, 
-0.007458323240407891, 0.007207371148400615, -0.0069694786465391845, 
0.006743733421913201, -0.006529302599998385, 0.0063254245071484305};

  x = X; y = Y;
  if (x < y) {temp = x; x = y; y = temp;}
  lnyox = TVIL_CLOG (y/x);
  lnyox2 = lnyox * lnyox;
  lnyox3 = lnyox2 * lnyox;

  if (y < 0.5 * x) {
    result = TVIL_H00000x (x, qq);
    rn = r = y/x;
    for (n = 1; n < 55; n++) {
      result += rn * (H0000xytableA[n] + 
                      H0000xytableAL[n] * lnyox +
                      H0000xytableAL2[n] * lnyox2 +
                      H0000xytableAL3[n] * lnyox3);
      rn *= r;
    }
  }
  else {
    result = TVIL_H0000xx (x, qq);
    rn = r = (y/x - 1.L);
    for (n = 1; n < 55; n++) {
      result += rn * H0000xytableB[n]; 
      rn *= r;
    }
  }

  return result;
}

/* ****************************************************************** */

TVIL_COMPLEX TVIL_H00xy00 (TVIL_REAL X, TVIL_REAL Y, TVIL_REAL qq)
{
  TVIL_REAL temp, x, y, r, rn;
  TVIL_COMPLEX lnyox, lnyox2, lnyox3, result;
  int n;

  TVIL_REAL H00xy00tableA[] = 
     {0, -9.79207768635047, 0.7511047763270091, -0.4191462310912395, 
     0.20621701959959626, -0.13722069745019216, 0.09193846614273049, 
     -0.06843562038543838, 0.05165372870784368, -0.04105357365173534, 
     0.033019836413298975, -0.02737467270849847, 0.022910932916912707, 
     -0.019558060340949424, 0.016822339978673106, -0.014671608279901882, 
     0.012873979604384522, -0.011413178354922782, 0.010168751558653266, 
     -0.009131772457131593, 0.008234679617107499, -0.007472254436954373, 
     0.006804238346668323, -0.00622741874813848, 0.005716602689091299, 
     -0.005269726260277052, 0.0048703735969110595, -0.004517170409153615, 
     0.004199050275951688, -0.003915069785280917, 0.003657549501413143, 
     -0.003425824577737352, 0.003214431127257694, -0.0030228897954743816, 
     0.0028472274732536385, -0.002687091549342435, 0.0025395370938717243, 
     -0.002404300086827793, 0.0022791603410184103, -0.002163917021350853, 
     0.002056870781620146, -0.0019578666706048797, 0.0018655861862000023, 
     -0.001779908241255207, 0.0016997971848202176, -0.0016251572911653855, 
     0.001555165964983285, -0.0014897467282653787, 0.0014282395391937985, 
     -0.0013705827057139408, 0.0013162416688140832, -0.0012651662216932036, 
     0.0012169197168695745, -0.0011714609604231707, 0.001128430461449892};
 
  TVIL_REAL H00xy00tableAL[] = 
     {0, 6.579736267392905, -1.7898681336964524, 1.1932454224643019, 
     -0.846322955737115, 0.6770583645896922, -0.5560671556765954, 
     0.47662899057993885, -0.41465878512479337, 0.3685855867775942, 
     -0.3307887564948964, 0.30071705135899685, -0.2752172909572561, 
     0.2540467301143903, -0.2356676439785089, 0.219956467713275, 
     -0.20607463987008415, 0.19395260223066746, -0.1830943936935481, 
     0.17345784665704553, -0.16473093770369465, 0.15688660733685203, 
     -0.14971876918632868, 0.1432092574825752, -0.13721649707794614, 
     0.13172783719482833, -0.12664281113977002, 0.12195233665311186, 
     -0.11758314864287198, 0.11352855731035913, -0.10973387879588299, 
     0.10619407625408032, -0.10286750884284943, 0.09975031160518737, 
     -0.09681021752945451, 0.09404421131432726, -0.09142690304533586, 
     0.08895590566573218, -0.08661096715723433, 0.08439017312756168, 
     -0.08227717257978025, 0.08027041227295637, -0.07835654621183032, 
     0.07653430095109008, -0.07479267556974536, 0.07313061611263992, 
     -0.07153897338250136, 0.0700168675658524, -0.06855662734583841, 
     0.06715751250204577, -0.06581304280177581, 0.06452259098213316, 
     -0.0632806453842238, 0.06208667094301203, -0.060935950144224024};
 
  TVIL_REAL H00xy00tableAL2[] = 
     {0, -1., 0.25, -0.1111111111111111, 0.0625, -0.04, 
     0.027777777777777776, -0.02040816326530612, 0.015625, 
     -0.012345679012345678, 0.01, -0.008264462809917356, 
     0.006944444444444444, -0.005917159763313609, 0.00510204081632653, 
     -0.0044444444444444444, 0.00390625, -0.0034602076124567475, 
     0.0030864197530864196, -0.002770083102493075, 0.0025, 
     -0.0022675736961451248, 0.002066115702479339, -0.001890359168241966, 
     0.001736111111111111, -0.0016, 0.0014792899408284023, 
     -0.0013717421124828531, 0.0012755102040816326, -0.0011890606420927466, 
     0.0011111111111111111, -0.001040582726326743, 0.0009765625, 
     -0.0009182736455463728, 0.0008650519031141869, -0.0008163265306122449, 
     0.0007716049382716049, -0.0007304601899196494, 0.0006925207756232687, 
     -0.0006574621959237344, 0.000625, -0.000594883997620464, 
     0.0005668934240362812, -0.0005408328826392645, 0.0005165289256198347, 
     -0.0004938271604938272, 0.0004725897920604915, -0.0004526935264825713, 
     0.00043402777777777775, -0.00041649312786339027, 0.0004, 
     -0.00038446751249519417, 0.0003698224852071006, -0.000355998576005696, 
     0.0003429355281207133};
 
  TVIL_REAL H00xy00tableAL3[] = 
     {0, 0.3333333333333333, -0.16666666666666666, 
     0.1111111111111111, -0.08333333333333333, 0.06666666666666667, 
     -0.05555555555555555, 0.047619047619047616, -0.041666666666666664, 
     0.037037037037037035, -0.03333333333333333, 0.030303030303030304, 
     -0.027777777777777776, 0.02564102564102564, -0.023809523809523808, 
     0.022222222222222223, -0.020833333333333332, 0.0196078431372549, 
     -0.018518518518518517, 0.017543859649122806, -0.016666666666666666, 
     0.015873015873015872, -0.015151515151515152, 0.014492753623188406, 
     -0.013888888888888888, 0.013333333333333334, -0.01282051282051282, 
     0.012345679012345678, -0.011904761904761904, 0.011494252873563218, 
     -0.011111111111111112, 0.010752688172043012, -0.010416666666666666, 
     0.010101010101010102, -0.00980392156862745, 0.009523809523809525, 
     -0.009259259259259259, 0.009009009009009009, -0.008771929824561403, 
     0.008547008547008548, -0.008333333333333333, 0.008130081300813009, 
     -0.007936507936507936, 0.007751937984496124, -0.007575757575757576, 
     0.007407407407407408, -0.007246376811594203, 0.0070921985815602835, 
     -0.006944444444444444, 0.006802721088435374, -0.006666666666666667, 
     0.006535947712418301, -0.00641025641025641, 0.006289308176100629, 
     -0.006172839506172839};
 
  TVIL_REAL H00xy00tableB[] = 
     {0, -3.606170709478783, 1.3444198410583277, -0.7433913894785305, 
     0.4854159390174413, -0.34764617887231086, 0.26410724373726213, 
     -0.20904077140173205, 0.17052736121150414, -0.14237169300359953, 
     0.12106815577282903, -0.10449992028742372, 0.09132132775302415, 
     -0.08064049248844635, 0.07184548325983225, -0.06450403333709659, 
     0.05830309176370446, -0.053011010668709224, 0.04845312414495573, 
     -0.04449553120072153, 0.04103406033139695, -0.03798659580825092, 
     0.035287637580184106, -0.03288437720791906, 0.03073382269569433, 
     -0.02880066171427778, 0.027055652884628245, -0.025474400177053547, 
     0.024036408956822714, -0.02272435161000705, 0.021523490880894364, 
     -0.020421223127560756, 0.01940671364231006, -0.01847060329077066, 
     0.017604770863683256, -0.016802139293195906, 0.016056516660319542, 
     -0.015362464988553981, 0.014715191374112147, -0.014110457182627087, 
     0.013544501943618573, -0.013013979268006029, 0.012515902652041028, 
     -0.012047599451010346, 0.01160667163591271, -0.011190962206949526, 
     0.010798526344768966, -0.01042760654587789, 0.010076611121519571, 
     -0.009744095546565093, 0.00942874623192927, -0.009129366364862737, 
     0.008844863519416657, -0.008574238786983476, 0.008316577216074247};

  x = X; y = Y;
  if (x < y) {temp = x; x = y; y = temp;}
  lnyox = TVIL_CLOG (y/x);
  lnyox2 = lnyox * lnyox;
  lnyox3 = lnyox2 * lnyox;

  if (y < 0.5 * x) {
    result = TVIL_H00000x (x, qq);
    rn = r = y/x;
    for (n = 1; n < 55; n++) {
      result += rn * (H00xy00tableA[n] + 
                      H00xy00tableAL[n] * lnyox +
                      H00xy00tableAL2[n] * lnyox2 +
                      H00xy00tableAL3[n] * lnyox3);
      rn *= r;
    }
  }
  else {
    result = TVIL_H00xx00 (x, qq);
    rn = r = (y/x - 1.L);
    for (n = 1; n < 55; n++) {
      result += rn * H00xy00tableB[n]; 
      rn *= r;
    }
  }

  return result;
}

/* ****************************************************************** */

TVIL_COMPLEX TVIL_Hxx00yy (TVIL_REAL X, TVIL_REAL Y, TVIL_REAL qq)
{
  TVIL_REAL temp, x, y, r, rn;
  TVIL_COMPLEX lnyox, lnyox2, result;
  int n;

  TVIL_REAL Hxx00yytableA[] = 
      {0, -5.803808507330341, -1.1070235056985824, 
     -0.39659361124680936, -0.18764389686941385, -0.10403747946959108, 
     -0.06392637368694487, -0.042216980337300924, -0.029409935226498707, 
     -0.021349313511786594, -0.016013382287494937, -0.012335206198216042, 
     -0.009714170980258285, -0.0077939341522410555, -0.006353655064130437, 
     -0.005251338280434115, -0.004392763214592366, -0.003713682498096005, 
     -0.003169230399269085, -0.002727410486290549, -0.002364974043723156, 
     -0.002064743885343223, -0.0018138364827109954, -0.001602456357289339, 
     -0.0014230632207882232, -0.0012697868405164292, -0.001138009579365884, 
     -0.0010240643396874916, -0.0009250131615211044, -0.0008384829898046564, 
     -0.0007625424951059889, -0.0006956087331940197, -0.000636375736878303, 
     -0.0005837593978922917, -0.0005368545667244762, -0.0004949014002961771, 
     -0.0004572587696264341, -0.00042338310080741737, -0.000392811429245965, 
     -0.0003651477445470175, -0.00034005192288846797, 
     -0.00031723070704706147, -0.0002964303167277138, 
     -0.00027743036440311397, -0.00026003882231806497, 
     -0.0002440878402823199, -0.00022943025550087315, 
     -0.00021593666798258582, -0.00020349298027200202, 
     -0.00019199832002797512, -0.00018136327957818388, 
     -0.00017150841895413604, -0.0001623629887732292, 
     -0.00015386383723026142, -0.0001459544718103445};
 
  TVIL_REAL Hxx00yytableAL[] = 
     {0, 4.710131866303548, 1.6328437109295515, 0.8396735850641455, 
     0.5160080232652293, 0.35128966376209514, 0.2556253487463144, 
     0.19493365057579692, 0.15390900397981067, 0.12482139132031672, 
     0.10341336426723124, 0.0871781336316711, 0.07455918422684954, 
     0.06454727506201285, 0.05646402596920924, 0.04983936503964295, 
     0.04433896518515787, 0.03971961617652872, 0.03580086331623464, 
     0.03244646126274395, 0.029551937447689886, 0.027036064359842765, 
     0.024834894479562414, 0.02289751239201923, 0.021182960405754042, 
     0.019657980545688153, 0.018295333751607606, 0.017072533268224235, 
     0.015970879315570147, 0.014974715666670801, 0.014070851569770601, 
     0.01324810819596009, 0.012496959807985639, 0.011809247650515442, 
     0.011177950157322133, 0.010596997126397376, 0.010061118483949666, 
     0.009565720454195376, 0.00910678359030076, 0.00868077835475943, 
     0.008284594872754611, 0.007915484196918454, 0.007571008972245399, 
     0.007249001816492226, 0.006947530064186927, 0.0066648657835836345, 
     0.0063994601821420055, 0.00614992167984893, 0.005914997060393681, 
     0.005693555215049564, 0.005484573078630137, 0.005287123425333204, 
     0.0051003642479615086, 0.004923529489493095, 0.0047559209332825245};

  TVIL_REAL Hxx00yytableAL2[] = 
     {0, -1., -0.5833333333333334, -0.4111111111111111, 
     -0.31726190476190474, -0.25825396825396824, -0.21773689273689273, 
     -0.18820148105862392, -0.1657179625929626, -0.1480310720506799, 
     -0.1337542806350856, -0.12198834642544183, -0.11212458325713812, 
     -0.10373630170481388, -0.0965155303434345, -0.09023441835885304, 
     -0.08472077527594087, -0.0798420550358441, -0.07549456873314729, 
     -0.07159603748964931, -0.06808033817926941, -0.06489372398910566, 
     -0.061992058464711355, -0.05933875955645778, -0.05690324977588434, 
     -0.054659772846073464, -0.05258647957556952, -0.05066471411053239, 
     -0.048878451129775895, -0.04721384801073383, -0.04565888546875645, 
     -0.04420307692878923, -0.04283723176682806, -0.0415532611213545, 
     -0.04034401760492567, -0.03920316220692631, -0.038125053154143065, 
     -0.0371046526161107, -0.03613744799971259, -0.03521938523902766, 
     -0.03434681200045703, -0.03351642912537886, -0.03272524894936139, 
     -0.031970559388009204, -0.031249892879642034, -0.030560999435430815, 
     -0.029901823176904922, -0.029270481845468755, -0.028665248853805114, 
     -0.02808453751874225, -0.027526887172407814, -0.02699095089570893, 
     -0.02647548465727964, -0.025979337673544503, -0.025501443832673828};
 
  TVIL_REAL Hxx00yytableB[] = 
     {0, -3.606170709478783, 1.2512855644747463, -0.6502571128949493, 
     0.40349958040532413, -0.2769477382316584, 0.2029143496567781, 
     -0.1556410524701204, 0.12350357552552654, -0.10060172816807414, 
     0.08367027685191816, -0.07077801775591966, 0.06072083684993612, 
     -0.052714983972890705, 0.046231770792222265, -0.04090364270863467, 
     0.0364683840117164, -0.032734697929903384, 0.029560269323487598, 
     -0.026837380162998276, 0.024483238744118313, -0.022433332269491985, 
     0.02063676627266295, -0.019052938404268707, 0.017649126081082545, 
     -0.016398711194005544, 0.015279856125493874, -0.014274504227028348, 
     0.013367616731442974, -0.012546584110916748, 0.01180076763039252, 
     -0.011121139110225886, 0.010499995506321483, -0.009930731015061187, 
     0.009407653789464038, -0.008925837531778946, 0.008481000558839866, 
     -0.00806940666249338, 0.007687783376888871, -0.007333254235995304, 
     0.007003282342622416, -0.006695623134924875, 0.006408284671608699, 
     -0.00613949409480033, 0.00588766919331174, -0.005651394196297593, 
     0.005429399091112341, -0.005220541889360816, 0.005023793369147683, 
     -0.0048382239050493575, 0.004662992064719984, -0.0044973347056663365, 
     0.004340558350199891, -0.00419203165293526, 0.004051178805057246};

  x = X; y = Y;
  if (x < y) {temp = x; x = y; y = temp;}
  lnyox = TVIL_CLOG (y/x);
  lnyox2 = lnyox * lnyox;

  if (y < 0.5 * x) {
    result = TVIL_H0000xx (x, qq);
    rn = r = y/x;
    for (n = 1; n < 55; n++) {
      result += rn * (Hxx00yytableA[n] + 
                      Hxx00yytableAL[n] * lnyox +
                      Hxx00yytableAL2[n] * lnyox2);
      rn *= r;
    }
  }
  else {
    result = TVIL_H0xxx0x (x, qq);
    rn = r = (y/x - 1.L);
    for (n = 1; n < 55; n++) {
      result += rn * Hxx00yytableB[n]; 
      rn *= r;
    }
  }

  return result;
}

/* ****************************************************************** */

TVIL_COMPLEX TVIL_Hxx0yxx (TVIL_REAL x, TVIL_REAL y, TVIL_REAL qq)
{
  TVIL_REAL r, rn;
  TVIL_COMPLEX lnyox, lnyox2, lnyox3, result;
  int n;
  
  lnyox = TVIL_CLOG(y/x);
  lnyox2 = lnyox  * lnyox;
  lnyox3 = lnyox2  * lnyox;

  TVIL_REAL Hxx0yxxtableA[] = 
      {0,-7.17120335576568, -4.770786863068025, -3.643141859329301, 
     -2.976352507227782, -2.5310036189189566, -2.2102281744852936, 
     -1.9669857405038749, -1.7755223624616479, -1.6204849056779387, 
     -1.4921183676049312, -1.38390980430293, -1.2913349040190347, 
     -1.211147018675306, -1.1409523441336409, -1.0789448913293271, 
     -1.0237350221853105, -0.9742350165134863, -0.9295806197089809, 
     -0.8890759795708817, -0.8521541906742954, -0.8183484975970985, 
     -0.7872709290008437, -0.7585962087489505, -0.7320494774569928, 
     -0.7073968073103726, -0.6844377928392268, -0.6629997040534177, 
     -0.6429328290467945, -0.6241067318527449, -0.606407221502185, 
     -0.5897338787820147, -0.5739980240414656, -0.5591210365572382, 
     -0.5450329561999209, -0.53167131335898, -0.5189801446294756, 
     -0.5069091605998957, -0.4954130388970907, -0.4844508209424492, 
     -0.47398539502065434, -0.46398305153042935, -0.4544130988783056, 
     -0.44524753054401983, -0.4364607355050638, -0.42802924554613536, 
     -0.41993151406437873, -0.4121477218655252, -0.40465960616992336, 
     -0.39745030964276085, -0.39050424675438716, -0.38380698518435025, 
     -0.37734514032215666, -0.3711062812013789, -0.36507884644160693, 
     -0.35925206897289885, -0.3536159084864234, -0.34816099069815465, 
     -0.3428785526341339, -0.3377603932494973, -0.3327988287820708, 
     -0.3279866523172766, -0.32331709710635176, -0.31878380323609823, 
     -0.3143807872969408, -0.31010241473810474, -0.30594337463520366, 
     -0.30189865662725535, -0.2979635298077941, -0.29413352337890697, 
     -0.2904044088981544, -0.2867721839668873, -0.2832330572247561, 
     -0.27978343452955245, -0.27641990621417567, -0.27313923532369117, 
     -0.26993834674533324, -0.2668143171530848, -0.2637643656962408, 
     -0.26078584536829363, -0.257876234998645, -0.2550331318151497, 
     -0.2522542445304148, -0.2495373869091722, -0.24688047177798583, 
     -0.24428150544208632, -0.24173858247730343, -0.23924988086791957, 
     -0.23681365746384253, -0.2344282437328125, -0.2320920417854528, 
     -0.22980352065286586, -0.22756121279818867, -0.22536371084507228, 
     -0.22320966450745705, -0.22109777770629258, -0.21902680586001128, 
     -0.2169955533366217, -0.21500287105624588, -0.21304765423380767, 
     -0.21112884025237058};
 
  TVIL_REAL Hxx0yxxtableAL[] = 
     {0, 4., 2.888888888888889, 2.299259259259259, 
     1.9261678004535148, 1.6659027462836986, 1.4726476196028866, 
     1.3227621890501868, 1.202711730370738, 1.1041442648931712, 
     1.0216051547131357, 0.9513698057702162, 0.8908019330821674, 
     0.8379794676182001, 0.7914657923327295, 0.7501641594940478, 
     0.7132218629696401, 0.6799653071622836, 0.649854890756228, 
     0.6224529599568244, 0.5974005974171258, 0.5744005165582854, 
     0.5532042577651595, 0.5336024692596654, 0.5154174351319091, 
     0.49849726408944117, 0.4827113217132884, 0.4679466050623757, 
     0.4541048393252593, 0.4411001333789486, 0.4288570720688297, 
     0.4173091527321683, 0.40639749528743474, 0.3960697713800168, 
     0.38627931018658795, 0.3769843476381838, 0.36814739280702297, 
     0.3597346905736244, 0.351715763853482, 0.3440630219119889, 
     0.3367514238502791, 0.3297581883650753, 0.3230625424937986, 
     0.31664550334371305, 0.3104896878403842, 0.3045791463695734, 
     0.29889921686896503, 0.29343639648371395, 0.28817822835757445, 
     0.28311320150883246, 0.2782306620528089, 0.2735207342925209, 
     0.26897425041590967, 0.2645826877196444, 0.26033811243215016, 
     0.256233129337238, 0.25226083650863257, 0.24841478455814034, 
     0.24468893987890186, 0.24107765143236579, 0.23757562068515362, 
     0.23417787435137488, 0.23087973963846603, 0.22767682173130985, 
     0.22456498328112687, 0.22154032569314583, 0.21859917203096998, 
     0.21573805137638433, 0.21295368450152533, 0.2102429707262352, 
     0.20760297584735501, 0.20503092103894857, 0.2025241726332127, 
     0.20008023270131786, 0.1976967303618006, 0.19537141375154088, 
     0.19310214260092046, 0.190886881360588, 0.18872369283243387, 
     0.18661073226198893, 0.1845462418535739, 0.18252854567319587, 
     0.18055604490747068, 0.1786272134497923, 0.17674059378760218, 
     0.17489479316698273, 0.17308848001292304, 0.17132038058552407, 
     0.1695892758541342, 0.16789399857296625, 0.16623343054315362, 
     0.16460650004747687, 0.16301217944514554, 0.16144948291506417, 
     0.15991746433696222, 0.15841521530062647, 0.15694186323426207, 
     0.1554965696437177, 0.15407852845496448, 0.15268696445280858, 
     0.15132113180936044};
 
  TVIL_REAL Hxx0yxxtableB[] = 
     {0, -2.6759565668419647, 0.8076906476165622, 
     -0.3845247324846479, 0.22409036295871276, -0.1465401051537547, 
     0.10324708396793292, -0.07664686926049558, 0.059143131402314086, 
     -0.047015939227991854, 0.03826930297945497, -0.03175429240711414, 
     0.02677161705321085, -0.02287587903255217, 0.019772484679844826, 
     -0.017260278758420955, 0.015198112410752196, -0.013484577691433075, 
     0.012045299517347507, -0.010824727687532454, 0.009780695839648579, 
     -0.008880730515275603, 0.0080994953619839, -0.007416988342716535, 
     0.00681724867829514, -0.006287415224468018, 0.005817031216611146, 
     -0.005397524376068885, 0.005021813592000823, -0.004684008146971785, 
     0.0043791754125867, -0.004103159764159814, 0.0038524402033980887, 
     -0.0036240175139667374, 0.0034153241508877996, -0.0032241517762809924, 
     0.003048592599838279, -0.0028869915982926986, 0.0027379073676870036, 
     -0.002600079870831884, 0.002472403726131972, -0.002353905975820275, 
     0.0022437274952280817, -0.002141107377199497, 0.002045369759696103, 
     -0.001955912669294818, 0.0018721985355548143, -0.0017937460962647866, 
     0.0017201234652828301, -0.0016509421759837162, 0.0015858520464931691, 
     -0.0015245367396447274, 0.0014667099122778193, -0.0014121118661420684, 
     0.0013605066271007993, -0.0013116793911590874, 0.00126543428559589, 
     -0.0012215924015395174, 0.001179990061016739, -0.001140477287074504, 
     0.001102916450226643, -0.0010671810683768879, 0.0010331547406474684, 
     -0.0010007301983055161, 0.0009698084583168176, -0.0009402980670369931, 
     0.000912114423235244, -0.0008851791710801082, 0.0008594196549452371, 
     -0.0008347684289431029, 0.000811162814996505, -0.0007885445040354405, 
     0.0007668591955751897, -0.0007460562715128627, 0.0007260885004773399, 
     -0.0007069117695072125, 0.0006884848402061161, -0.0006707691268571792, 
     0.0006537284942650656, -0.0006373290733462517, 0.0006215390927088033, 
     -0.0006063287246567406, 0.0005916699442242018, -0.0005775363999942773, 
     0.000563903295589932, -0.0005507472808414793, 0.0005380463517377643, 
     -0.0005257797583600315, 0.0005139279200789548, -0.0005024723473673331, 
     0.0004913955696449618, -0.00048068106862964033, 0.00047031321671955895, 
     -0.00046027721997694837, 0.0004505590653255259, -0.00044114547160849227, 
     0.0004320238441883617, -0.0004231822327989271, 0.00041460929238491817, 
     -0.0004062942466910441, 0.000398226854380776};
 
  TVIL_REAL Hxx0yxxtableC[] = 
     {0, 2.3265870915803015, 2.2558960783562805, 1.8212156257749559, 
     1.4923901848114964, 1.261341516685507, 1.0932804754914427, 
     0.9660552164362657, 0.8664292044707946, 0.7862452865629423, 
     0.720256565158887, 0.6649488091183019, 0.6178833600881172, 
     0.5773155309218692, 0.5419639774761948, 0.510866119368717, 
     0.4832844919071698, 0.45864430363278175, 0.43649071791800276, 
     0.4164589538441205, 0.398252929292225, 0.3816297251355881, 
     0.36638809704213715, 0.35235985348179233, 0.3394032971914295, 
     0.32739817470854116, 0.3162417433354313, 0.30584567659332845, 
     0.2961336062005154, 0.28703915245820155, 0.2785043331266954, 
     0.2704782683220972, 0.2629161189232382, 0.25577821065515044, 
     0.24902930691985142, 0.24263800162612392, 0.2365762094635951, 
     0.23081873579569953, 0.22534291198637302, 0.2201282847986069, 
     0.215156350708345, 0.21041032771152968, 0.20587495857475713, 
     0.20153634057299555, 0.19738177763315976, 0.19339965150720118, 
     0.18957930916892737, 0.1859109640928988, 0.1823856094530979, 
     0.17899494159052698, 0.17573129235571527, 0.17258756914476525, 
     0.16955720162434343, 0.16663409428852918, 0.16381258411396976, 
     0.1610874026836145, 0.15845364223685102, 0.1559067251779298, 
     0.15344237663741928, 0.15105659973493096, 0.14874565323702515, 
     0.14650603134329787, 0.14433444536720727, 0.14222780710707197, 
     0.1401832137275852, 0.13819793399373337, 0.13626939571768307, 
     0.1343951742954255, 0.13257298222409677, 0.13080065950322012, 
     0.12907616483389733, 0.1273975675394199, 0.12576304013906475, 
     0.12417085151412971, 0.12261936061169187, 0.12110701063724282, 
     0.11963232369237232, 0.11819389581811329, 0.11679039240850378, 
     0.11542054396242382, 0.11408314214488219, 0.11277703613170784, 
     0.1115011292140799, 0.11025437564154855, 0.1090357776841836, 
     0.1078443828962676, 0.10667928156554687, 0.10553960433348998, 
     0.1044245199732951, 0.10333323331355168, 0.10226498329651232, 
     0.10121904116087871, 0.10019470873986433, 0.09919131686607201, 
     0.09820822387542871, 0.09724481420305836, 0.09630049706455328, 
     0.09537470521663205, 0.09446689379165156, 0.09357653920087722, 
     0.09269269444023234};
 
  TVIL_REAL Hxx0yxxtableCL[] = 
     {0, 4.644934066848226, 2.220567308492198, 1.4533560751286048, 
     1.1152464830063975, 0.9235331306124643, 0.7975701115890522, 
     0.7069497674710449, 0.6377663787362929, 0.5827302065227348, 
     0.5376185538375832, 0.49979465337677537, 0.46751397909386133, 
     0.4395685227095246, 0.4150906129948624, 0.39343810399358137, 
     0.3741237978586313, 0.3567702345957645, 0.34107970188381587, 
     0.3268137371005917, 0.31377875027611113, 0.3018157103943566, 
     0.29079259871212504, 0.28059878917568026, 0.27114079804794167, 
     0.2623390238861646, 0.2541252154226907, 0.24644048225767892, 
     0.23923371569358856, 0.23246032320231713, 0.22608120536871337, 
     0.22006192219509474, 0.21437200866540937, 0.20898440897412893, 
     0.2038750058502728, 0.19902222665485095, 0.19440671188954478, 
     0.19001103476986836, 0.18581946283253264, 0.18181775434083297, 
     0.17799298365204208, 0.1743333908113904, 0.17082825150821043, 
     0.16746776422352078, 0.16424295195416033, 0.16114557634649837, 
     0.15816806243565526, 0.15530343248171632, 0.15254524763628882, 
     0.14988755637161455, 0.14732484876867386, 0.14485201589691826, 
     0.14246431363167733, 0.14015733035008965, 0.13792695802594587, 
     0.135769366310805, 0.1336809792453257, 0.13165845429271456, 
     0.12969866342697278, 0.12779867604340056, 0.12595574348856703, 
     0.12416728503246689, 0.12243087512753105, 0.12074423181807775, 
     0.11910520618015033, 0.11751177268585987, 0.11596202039866269, 
     0.11445414491672078, 0.11298644099084339, 0.11155729575168283, 
     0.11016518248802121, 0.10880865492427155, 0.10748634195084748, 
     0.10619694276592895, 0.10493922239145392, 0.10371200752997067, 
     0.10251418273235577, 0.10134468684939459, 0.10020250974287873, 
     0.0990866892342421, 0.09799630827086701, 0.09693049229207584, 
     0.0958884067785089, 0.09486925497009731, 0.09387227573919274, 
     0.09289674160663138, 0.09194195688960055, 0.09100725597116059, 
     0.09009200168216082, 0.08919558378708804, 0.08831741756610922, 
     0.08745694248622356, 0.08661362095503151, 0.08578693715116516, 
     0.08497639592591148, 0.08418152177100355, 0.083401857847956, 
     0.08263696507468979, 0.0818864212655232, 0.0811498203209132, 
     0.08041142994939152};
 
  TVIL_REAL Hxx0yxxtableCL2[] = 
     {0, -1., -0.4375, -0.19444444444444445, -0.0849609375, 
     -0.028802083333333332, 0.002658420138888889, 0.021332908163265305, 
     0.03283125559488932, 0.04005289767364556, 0.04460699535551525, 
     0.04743788097531217, 0.04912504459184314, 0.050037248249814995, 
     0.0504162860338134, 0.05042469376194829, 0.05017402344228727, 
     0.04974217756410621, 0.049184352129238114, 0.04854012348577984, 
     0.04783814180305653, 0.04709930157015638, 0.04633892124472436, 
     0.04556826535397117, 0.04479562239985248, 0.04402707784047107, 
     0.04326707468961388, 0.042518824226114436, 0.04178460963864882, 
     0.04106601235290838, 0.04036408196103562, 0.039679464634130156, 
     0.03901250071476344, 0.038363299253967593, 0.037731795179499004, 
     0.037117793295303826, 0.03652100223804877, 0.03594106073392141, 
     0.03537755792392641, 0.034830049100270384, 0.03429806787909263, 
     0.033781135596623914, 0.033278768535970275, 0.03279048345505984, 
     0.03231580178189861, 0.031854252763125444, 0.03140537579001272, 
     0.030968722078128962, 0.030543855839575012, 0.030130355057557286, 
     0.029727811950203388, 0.02933583319253828, 0.028954039951342436, 
     0.028582067776374794, 0.028219566382524337, 0.027866199350359264, 
     0.02752164376688673, 0.027185589823820088, 0.02685774038703802, 
     0.026537810548027792, 0.02622552716578656, 0.02592062840579549, 
     0.02562286328119039, 0.025331991200056792, 0.02504778152181892, 
     0.02477001312492507, 0.024498473987418994, 0.02423296078149824, 
     0.023973278482772524, 0.02371923999462835, 0.023470665787864633, 
     0.023227383555576086, 0.022989227883115457, 0.022756039932854643, 
     0.022527667143381818, 0.02230396294271056, 0.022084786475034264, 
     0.02187000234053035, 0.0216594803477018, 0.02145309527773511, 
     0.021250726660352767, 0.021052258560642724, 0.020857579376355934, 
     0.020666581645174806, 0.020479161861469683, 0.020295220302076266, 
     0.020114660860644035, 0.019937390890123325, 0.019763321052977085, 
     0.019592365178721403, 0.019424440128417236, 0.019259465665753515, 
     0.019097364334379526, 0.01893806134116122, 0.018781484445052964, 
     0.018627563851291794, 0.01847623211063677, 0.01832742402339047, 
     0.018181076547953777, 0.018037128713678383, 0.01788988668989317};
 
  TVIL_REAL Hxx0yxxtableCL3[] = 
     {0, 0.16666666666666666, 0.14583333333333334, 
     0.11805555555555555, 0.09830729166666667, 0.08411458333333334, 
     0.0735134548611111, 0.06531343005952381, 0.05878575642903646, 
     0.05346623173466435, 0.049046961466471355, 0.04531623377944484, 
     0.042123867405785456, 0.039360434581071906, 0.03694433470567068, 
     0.03481348123815325, 0.03291979578595298, 0.03122546512396166, 
     0.029700337060839282, 0.028320070843747408, 0.02706479895235437, 
     0.02591814263899915, 0.024866476724879027, 0.023898372990828833, 
     0.02300417349804916, 0.022175659760297985, 0.021405793536277067, 
     0.020688511769311708, 0.020018562910774588, 0.01939137519159507, 
     0.018802949788603238, 0.018249773558694376, 0.017728747278709368, 
     0.017237126265556544, 0.016772470951475858, 0.0163326055178894, 
     0.0159155830936041, 0.015519656331869336, 0.015143252419530497, 
     0.014784951757466436, 0.014443469697318986, 0.014117640834616354, 
     0.013806405449791984, 0.013508797761607369, 0.01322393571612599, 
     0.012951012081732357, 0.012689286659111688, 0.01243807944643424, 
     0.012196764625650076, 0.011964765256912415, 0.011741548585589758, 
     0.011526621880794875, 0.011319528736405723, 0.011119845775621558, 
     0.010927179708541927, 0.01074116469936272, 0.010561460005783813, 
     0.010387747858305114, 0.010219731551405573, 0.010057133722278781, 
     0.0098996947959425, 0.009747171578232895, 0.009599335980508335, 
     0.009455973861880356, 0.00931688397651018, 0.009181877014998192, 
     0.009050774730185522, 0.008923409138809889, 0.008799621791436214, 
     0.008679263103936824, 0.00856219174454334, 0.008448274071147534, 
     0.00833738361410374, 0.008229400600291772, 0.008124211514645515, 
     0.0080217086957465, 0.007921789962430382, 0.007824358268663109, 
     0.0077293213842176915, 0.007636591598926175, 0.00754608544849831, 
     0.007457723460091823, 0.007371429915991894, 0.007287132633911815, 
     0.007204762762565075, 0.007124254591283078, 0.007045545372563967, 
     0.0069685751565380745, 0.006893286636425545, 0.006819625004142846, 
     0.006747537815288046, 0.006676974862800926, 0.006607888058653738, 
     0.006540231322982617, 0.006473960480118682, 0.006409033161022448, 
     0.006345408711665525, 0.006283048106940458, 0.0062219138697129014, 
     0.00616196999466095, 0.006103181876574102};

  if (y < 2. * x) {     
    result = TVIL_H0xxx0x (x,qq);     
    rn = r = 0.25L * y/x;
    for (n=1; n < 100; n++) {
      result += rn * (Hxx0yxxtableA[n] + Hxx0yxxtableAL[n] * lnyox);
      rn *= r;
    }
  }
  else if (y > 6.47 * x) {
    result = TVIL_H00000x (y, qq);
    rn = r = 4.L * x/y;
    for (n=1; n < 100; n++) {
      result += rn * (Hxx0yxxtableC[n] +
                     -Hxx0yxxtableCL[n] * lnyox +
                     +Hxx0yxxtableCL2[n] * lnyox2 +
                     -Hxx0yxxtableCL3[n] * lnyox3);
      rn *= r;
    }
  }
  else {
    result = -10.9900479678501708324319020119L - 6.L * Zeta3 * TVIL_Ap(x,qq); 
    rn = r = 0.25L * y/x - 1.L;
    for (n=1; n < 100; n++) {
      result += rn * Hxx0yxxtableB[n];
      rn *= r;
    }
  }

  return result;
}

/* ****************************************************************** */

TVIL_COMPLEX TVIL_H00xyxx (TVIL_REAL x, TVIL_REAL y, TVIL_REAL qq)
{
  TVIL_REAL r, rn;
  TVIL_COMPLEX lnyox, lnyox2, lnyox3, result;
  int n;
  
  TVIL_REAL H00xyxxtableA[] = 
     {0, -13.289868133696451, -0.9669930816025213, 
     -0.12106297685467804, 0.02269210315254011, 0.05305705124267418, 
     0.056962172362317404, 0.053817860206711726, 0.04898548386116558, 
     0.04409630208175254, 0.03963634922671204, 0.035711810187721085, 
     0.03230179376483015, 0.02934741462276283, 0.026783839197765377, 
     0.024551306237142002, 0.022598195814536655, 0.02088113647382457, 
     0.019364107976896472, 0.018017307514450152, 0.016816067264322616, 
     0.015739915948772273, 0.014771799271321738, 0.013897445421188783, 
     0.013104853496650144, 0.012383882466561, 0.011725920932249775, 
     0.011123621285347031, 0.01057068502372949, 0.010061688703854748, 
     0.009591942228289858, 0.009157372938046443, 0.008754430373090143, 
     0.008380007654238283, 0.00803137628966647, 0.007706131872125611, 
     0.0074021486506852825, 0.007117541366178745, 0.006850633057925321, 
     0.006599927800311574, 0.006364087526459925, 0.006141912254062708, 
     0.005932323154408588, 0.0057343480065412036, 0.005547108659678668, 
     0.00536981019260855, 0.00520173151196892, 0.005042217174641751, 
     0.004890670254892031, 0.0047465461059366994, 0.004609346889550437, 
     0.004478616767085412, 0.00435393766167708, 0.004234925515049275, 
     0.004121226973719345, 0.004012516448939682, 0.003908493502721968, 
     0.0038088805190383294, 0.003713420624995103, 0.003621875831605604, 
     0.003534025367892218, 0.0034496641855440617, 0.003368601614341886, 
     0.003290660151117803, 0.0032156743672108173, 0.0031434899212660133, 
     0.0030739626658519053, 0.0030069578377760475, 0.002942349323196156, 
     0.002880018989680172, 0.00281985607828696, 0.0027617566495394775, 
     0.0027056230778606067, 0.0026513635896526257, 0.002598891840736391, 
     0.002548126529336018, 0.00249899104120778, 0.0024514131238756003, 
     0.0024053245872563828, 0.0023606610282419054, 0.002317361577054807, 
     0.0022753686634186185, 0.002234627800779078, 0.0021950873869894553, 
     0.002156698520028768, 0.0021194148274609717, 0.0020831923084674935, 
     0.002047989187396551, 0.00201376577787209, 0.0019804843565942643, 
     0.0019481090460433201, 0.0019166057053704982, 0.0018859418288241937, 
     0.0018560864511177047, 0.0018270100591973489, 0.0017986845099169854, 
     0.0017710829531677398, 0.001744179760050358, 0.0017179504557125838, 
     0.0016923716565056934, 0.0016674210111430328};
 
  TVIL_REAL H00xyxxtableAL[] = 
     {0, 6., 1., 0.35555555555555557, 0.17142857142857143, 
     0.09752380952380953, 0.06156806156806157, 0.0417541642031438, 
     0.029836829836829837, 0.022188027416785588, 0.017026391565091257, 
     0.01340133141683688, 0.01077123920036829, 0.008810746195614866, 
     0.007315649807923605, 0.0061529940453463215, 0.00523345637677516, 
     0.004495383730580068, 0.003895206396532606, 0.0034014889128663856, 
     0.0029911298017116076, 0.0026468711258993264, 0.0023556288196188413, 
     0.0021073501929259756, 0.0018942176268713314, 0.0017100842105384022, 
     0.001550067629055722, 0.0014102538317303814, 0.0012874780227249117, 
     0.001179160880107527, 0.001083184734649434, 0.0009977990083940814, 
     0.000921547317059402, 0.0008532107787074236, 0.0007917635633530104, 
     0.0007363377721153913, 0.0006861954872945445, 0.0006407063781225025, 
     0.0005993296430478138, 0.0005615993608668193, 0.0005271125393553616, 
     0.0004955193119724905, 0.00046651485534595554, 0.00043983269306615544, 
     0.0004152391223474259, 0.0003925285548402621, 0.00037151960530774306, 
     0.0003520517949759243, 0.00033398276233356533, 0.0003171858946400654, 
     0.00030154830964451766, 0.00028696912996573824, 0.0002733580029520103, 
     0.000260633827181856, 0.00024872365350918027, 0.00023756173402786907, 
     0.00022708869678988967, 0.0002172508277588653, 0.0002079994444768006, 
     0.00019929034839068997, 0.0001910833448278782, 0.00018334182130379817, 
     0.00017603237625676764, 0.00016912449148308558, 0.00016259024253303052, 
     0.00015640404215798816, 0.00015054241259800468, 0.0001449837830897447, 
     0.00013970830947526876, 0.00013469771321711444, 0.00012993513748712685, 
     0.00012540501830545417, 0.00012109296897048317, 0.00011698567624719628, 
     0.0001130708069762813, 0.00010933692393415447, 0.00010577340991891872, 
     0.00010237039916256808, 0.00009911871527832853, 0.0000960098150463013, 
     0.00009303573742258913, 0.00009018905722856146, 0.00008746284303931697, 
     0.00008485061884497837, 0.00008234632910626645, 0.00007994430686775546, 
     0.00007763924462908719, 0.0000754261677068827, 0.00007330040984870924, 
     0.00007125759088573047, 0.00006929359623301264, 0.0000674045580662466, 
     0.00006558683802119102, 0.00006383701127772334, 0.0000621518519042368, 
     0.0000605283193504564, 0.00005896354598773992, 0.000057454825605744105, 
     0.00005599960278310456, 0.000054595463057624134, 0.00005324012382849211};
 
  TVIL_REAL H00xyxxtableAL2[] = 
     {0, -1., -0.3333333333333333, -0.17777777777777778, 
     -0.11428571428571428, -0.08126984126984127, -0.06156806156806157, 
     -0.04871319157033443, -0.03978243978243978, -0.03328204112517838, 
     -0.028377319275152094, -0.02456910759753428, -0.02154247840073658, 
     -0.019089950090498877, -0.017069849551821746, -0.015382485113365802, 
     -0.013955883671400425, -0.01273692056997686, -0.011685619189597818, 
     -0.01077138155741022, -0.009970432672372026, -0.009264048940647642, 
     -0.008637305671935751, -0.008078175739549573, -0.007576870507485326, 
     -0.007125350877243342, -0.006716959725908129, -0.006346142242786716, 
     -0.006008230772716254, -0.00569927758718638, -0.00541592367324717, 
     -0.005155294876702753, -0.004914919024316812, -0.00469265928289083, 
     -0.004486660192333725, -0.0042953036706731154, -0.004117172923767266, 
     -0.0039510226650887655, -0.0037957544059694878, -0.0036503958456343254, 
     -0.003514083595702411, -0.003386048631812019, -0.003265603987421689, 
     -0.003152134300307447, -0.003045086897214456, -0.002943964161301966, 
     -0.0028483169740260304, -0.0027577390606447405, -0.0026718620986685227, 
     -0.0025903514728938676, -0.00251290258037098, -0.002439237604708775, 
     -0.002369102692250756, -0.0023022654734397283, -0.0022385128815826227, 
     -0.0021776492285887996, -0.0021194945033723035, -0.0020638828637092204, 
     -0.0020106612966090728, -0.001959688425841785, -0.0019108334482787822, 
     -0.0018639751832552813, -0.0018190012213199324, -0.0017758071605723986, 
     -0.0017342959203523252, -0.0016943771233782052, -0.0016559665385780515, 
     -0.0016189855778354825, -0.0015833608407197128, -0.0015490237019968162, 
     -0.0015159099373498132, -0.0014839593832812075, -0.001453115627645798, 
     -0.0014233257276742213, -0.001394539952707469, -0.001366711549176931, 
     -0.001339796525639637, -0.0013137534559196236, -0.001288543298618271, 
     -0.0012641292314429671, -0.0012404764989678552, -0.0012175522725855798, 
     -0.0011953255215373319, -0.0011737668940222006, -0.0011528486074877304, 
     -0.001132544347293202, -0.0011128291730169164, -0.001093679431749799, 
     -0.001075072677781069, -0.0010569875981383352, -0.0010394039434951897, 
     -0.0010223024640047401, -0.0010056648496582622, -0.0009894736748047117, 
     -0.0009737123464997101, -0.0009583650563822264, -0.0009434167358038387, 
     -0.0009288530139595296, -0.000914660178790708, -0.0009008251404507983, 
     -0.000887335397141535};
 
  TVIL_REAL H00xyxxtableC[] = 
     {0, -2.2653441971127455, 0.3031253907398125, 0.2594319089125304, 
     0.1954223295332606, 0.15149392686662788, 0.12137308667201503, 
     0.09992003367833141, 0.08406826016981268, 0.07198488138710939, 
     0.06253258882759302, 0.05497701253003841, 0.0488265326643646, 
     0.04374148696116405, 0.03948068574885861, 0.035868734447709216, 
     0.03277539956280122, 0.03010219054008656, 0.027773402597548038, 
     0.025729996223518285, 0.02392532670501184, 0.02232210801044078, 
     0.020890217347775332, 0.01960508303889128, 0.01844648404888097, 
     0.017397644550895437, 0.016444542959218055, 0.015575378905917153, 
     0.014780157937692277, 0.014050364932089823, 0.013378705068325997, 
     0.012758896731596425, 0.012185504699704947, 0.01165380483632161, 
     0.011159673620146013, 0.010699497395567525, 0.010270097391864965, 
     0.009868667432393612, 0.009492721918967523, 0.00914005218444652, 
     0.008808689697894896, 0.008496874910404425, 0.008203030766951544, 
     0.007925740096191498, 0.007663726237608897, 0.007415836382769102, 
     0.007181027201232715, 0.006958352397103688, 0.006746951903091604, 
     0.0065460424683999666, 0.006354909437049024, 0.0061728995462337006, 
     0.0059994146014404545, 0.005833905907434118, 0.005675869352774105, 
     0.005524841060943822, 0.005380393534048518, 0.005242132225814785, 
     0.005109692489678793, 0.004982736855380443, 0.004860952593929939, 
     0.00474404953628081, 0.004631758115691577, 0.0045238276077207195, 
     0.004420024545186425, 0.0043201312883248645, 0.004223944732873823, 
     0.004131275140955373, 0.004041945081484222, 0.003955788468431157, 
     0.0038726496866606512, 0.003792382796268878, 0.00371485080739931, 
     0.0036399250184298766, 0.0035674844112267897, 0.0034974150978618053, 
     0.0034296098138050833, 0.003363967453146711, 0.0033003926418760775, 
     0.0032387953456681764, 0.0031790905089967717, 0.00312119772272246, 
     0.003065040917594397, 0.0030105480813624316, 0.0029576509974256903, 
     0.0029062850031477366, 0.0028563887661502895, 0.002807904077059863, 
     0.002760775657326733, 0.002714950980865516, 0.002670380108382929, 
     0.0026270155333627415, 0.002584812038771613, 0.0025437265636338996, 
     0.0025037180786993867, 0.0024647474704963604, 0.0024267774331241942, 
     0.002389772367195435, 0.0023536982853878045, 0.002318522724112259, 
     0.002294658322427868};
 
  TVIL_REAL H00xyxxtableCL[] = 
     {0, 3.144934066848226, 0.5699752750680849, 0.28366681948226935, 
     0.18546461851440593, 0.1356028847053723, 0.10539841670202049, 
     0.08521684817788971, 0.07085570305816472, 0.06017178767415825, 
     0.051953555993096756, 0.04546414520256157, 0.040230069189885566, 
     0.03593368626992976, 0.03235434804170108, 0.029334291545215926, 
     0.02675793151952626, 0.02453879385812589, 0.022610997270675923, 
     0.020923542244883145, 0.01943638769904172, 0.018117697386749492, 
     0.016941870224394567, 0.015888107235179097, 0.014939352857354803, 
     0.014081501923717378, 0.013302798117995781, 0.012593372392857806, 
     0.011944885021668895, 0.0113502452996259, 0.010803390064048018, 
     0.010299107222650199, 0.009832894046253626, 0.00940084254932681, 
     0.008999546149657044, 0.00862602317206406, 0.008277653781167786, 
     0.00795212769279736, 0.007647400591659313, 0.007361657623451097, 
     0.007093282667984005, 0.0068408323616579066, 0.006603014041538995, 
     0.0063786669431569396, 0.0061667461102455174, 0.005966308574712265, 
     0.005776501444959889, 0.005596551604720992, 0.005425756776194239, 
     0.005263477743091294, 0.005109131563238527, 0.004962185628197663, 
     0.0048221524502061455, 0.0046885850755598564, 0.004561073039133981, 
     0.004439238787671262, 0.004322734510246542, 0.0042112393233321105, 
     0.004104456765453622, 0.004002112562795005, 0.003903952632488476, 
     0.003809741294880403, 0.003719259669932254, 0.003632304236210753, 
     0.003548685533735206, 0.0034682269943592725, 0.00339076388543248, 
     0.0033161423542662494, 0.0032442185624639514, 0.0031748579005012106, 
     0.0031079342740920747, 0.0030433294548745777, 0.002980932488817444, 
     0.0029206391565065547, 0.0028623514801309306, 0.0028059772725665686, 
     0.002751429724463754, 0.0026986270256890623, 0.0026474920178653147, 
     0.0025979518750983135, 0.0025499378102842724, 0.0025033848046616416, 
     0.00245823135850995, 0.0024144192611102555, 0.002371893378270076, 
     0.0023306014558831995, 0.0022904939381439967, 0.0022515237991690452, 
     0.0022136463868978142, 0.002176819278250568, 0.002141002144616999, 
     0.0021061566268345557, 0.002072246218892224, 0.002039236159664528, 
     0.0020070933320426444, 0.0019757861688855074, 0.0019452845652642994, 
     0.00191555979651932, 0.0018865844416894938, 0.0018583323119120978, 
     0.0018461198976401447};
 
  TVIL_REAL H00xyxxtableCL2[] = 
     {0, -0.75, -0.078125, -0.005208333333333333, 
     0.009195963541666666, 0.01208984375, 0.012026638454861112, 
     0.011161627737032312, 0.01013500349862235, 0.009148183953825129, 
     0.008259613930232942, 0.0074785252904760605, 0.006797519925768528, 
     0.00620448389778889, 0.005686970059064114, 0.005233704296158512, 
     0.00483497459793171, 0.0044825972355188115, 0.004169740119403678, 
     0.0038907156705757806, 0.003640785967356354, 0.0034159939272717025, 
     0.003213022476427887, 0.0030290792183634532, 0.0028618028032834832, 
     0.002709187164393746, 0.0025695202204833833, 0.0024413341955944173, 
     0.0023233652370024433, 0.0022145204728455654, 0.002113851030921681, 
     0.002020529846389728, 0.0019338333293778282, 0.00185312615538956, 
     0.0017778485922894942, 0.0017075058962295756, 0.0016416594021635984, 
     0.0015799190081383693, 0.0015219368106894873, 0.0014674016947836242, 
     0.0014160347184560282, 0.0013675851616206966, 0.0013218271320547658, 
     0.001278556640500603, 0.0012375890721409242, 0.0011987569941275802, 
     0.001161908248966095, 0.0011269042918337407, 0.001093618736700303, 
     0.0010619360817139968, 0.0010317505889376824, 0.0010029652973541018, 
     0.000975491151248516, 0.000949246228739647, 0.0009241550574592431, 
     0.0009001480062530808, 0.0008771607433535518, 0.0008551337528065131, 
     0.0008340119020639474, 0.0008137440546128824, 0.0007942827223277077, 
     0.0007755837529303653, 0.0007576060485397872, 0.0007403113118041256, 
     0.0007236638165498064, 0.0007076302002611443, 0.0006921792760322862, 
     0.0006772818619172585, 0.0006629106258502837, 0.0006490399445227385, 
     0.0006356457747897206, 0.0006227055363420233, 0.000610198004521725, 
     0.0005981032122843189, 0.0005864023604197804, 0.0005750777352411842, 
     0.0005641126330342194, 0.0005534912906356821, 0.0005431988215750459, 
     0.0005332211572716338, 0.000523544992831674, 0.0005141577370354743, 
     0.0005050474661457814, 0.0004962028812047315, 0.00048761326851920353, 
     0.00047926846306328556, 0.0004711588145524129, 0.00046327515596685296, 
     0.0004556087743229288, 0.000448151383508959, 0.00044089509901958237, 
     0.00043383241443714746, 0.00042695617952236104, 0.00042025957978856875, 
     0.00041373611744503297, 0.00040737959360449984, 0.0004011840916593257, 
     0.00039514396173855954, 0.00038925380616573963, 0.00038350846584384566, 
     0.0003835378554008345};
 
  TVIL_REAL H00xyxxtableCL3[] = 
     {0, 0.16666666666666666, 0.0625, 0.034722222222222224, 
     0.022786458333333332, 0.01640625, 0.012532552083333334, 
     0.009974888392857142, 0.008182525634765625, 0.00686928077980324, 
     0.005873235066731771, 0.0050966089422052555, 0.0044772293832567, 
     0.003973872233659794, 0.0035582376377923147, 0.003210321068763733, 
     0.002915623626904562, 0.0026634070501827143, 0.0024455666587325847, 
     0.0022558828181660686, 0.002089511460326321, 0.0019426297023441986, 
     0.001812184526050425, 0.001695711418515993, 0.0015912014265762053, 
     0.001497002302122894, 0.0014117440349161907, 0.0013342820851265506, 
     0.0012636536329164079, 0.001199043518510456, 0.0011397574778729946, 
     0.0010852009284638814, 0.0010348620182079886, 0.0009882979788212196, 
     0.0009451240619263478, 0.0009050045099098826, 0.0008676451416381397, 
     0.0008327872287162275, 0.0008002024098641037, 0.000769688445516886, 
     0.0007410656564492269, 0.0007141739223365542, 0.0006888701410746298, 
     0.0006650260691282932, 0.0006425264784630539, 0.0006212675776941528, 
     0.0006011556546922391, 0.0005821059055621093, 0.0005640414210796046, 
     0.0005468923066527778, 0.000530594915914525, 0.0005150911813472418, 
     0.0005003280280356992, 0.0004862568588598998, 0.000472833101267917, 
     0.0004600158072831338, 0.0004477672996593641, 0.000436052858147808, 
     0.0004248404407192154, 0.00041410043532326857, 0.0004038054383895373, 
     0.00039393005680032797, 0.0003844507305115584, 0.00037534557337674194, 
     0.00036659423005216214, 0.0003581777471373551, 0.0003500784569415265, 
     0.0003422798724696725, 0.0003347665923970791, 0.00032752421495181567, 
     0.0003205392597553943, 0.00031379909678493767, 0.0003072918817185062, 
     0.00030100649701080923, 0.0002949324981211645, 0.00028906006438079643, 
     0.0002833799540436738, 0.0002778834631151829, 0.00027256238759695167, 
     0.00026740898882488944, 0.00026241596161167473, 0.0002575764049350991, 
     0.0002528837949403713, 0.0002483319600481365, 0.00024391505798095374, 
     0.00023962755453962764, 0.0002354642039773856, 0.00023142003083469038, 
     0.00022749031311067671, 0.0002236705666590014, 0.0002199565307064538, 
     0.00021634415440213335, 0.00021282958431348813, 0.00020940915279313028, 
     0.0002060793671472006, 0.00020283689954222523, 0.00019967857759297292, 
     0.00019660137557883954, 0.00019360240624082412, 0.00019067891311526123, 
     0.00018782826336418806};
 
  TVIL_REAL H00xyxxtableB[] = 
     {0, -3.2027181327479024, 1.0836780701489368, 
     -0.5586508236979613, 0.345704442565157, -0.23712107307382535, 
     0.17378044034801188, -0.13338962730616433, 0.10594518330463032, 
     -0.08638813320563787, 0.07192549347710508, -0.060907975150488625, 
     0.05230828719048042, -0.04545819225801295, 0.03990707934314597, 
     -0.03534171964407373, 0.03153865207400919, -0.028334833241990487, 
     0.025608942182765104, -0.023269126570914456, 0.02124476658862029, 
     -0.019480814495540844, 0.017933826282116316, -0.01656912946201866, 
     0.01535876885765941, -0.014279994689045394, 0.01331413483913646, 
     -0.012445743325006975, 0.011661950056237547, -0.0109519591229299, 
     0.01030665795254565, -0.0097183101112574, 0.009180311838921151, 
     -0.008686997596723645, 0.008233483633042299, -0.007815541278211935, 
     0.007429493662858111, -0.007072131023536793, 0.006740640857096691, 
     -0.006432550012346022, 0.006145676435929143, -0.005878088770232909, 
     0.005628072371859694, -0.0053941006069346265, 0.005174810504262467, 
     -0.00496898202398785, 0.004775520339042938, -0.004593440637659568, 
     0.004421855043916928, -0.004259961324530659, 0.004107033107577346, 
     -0.003962411385456999, 0.003825497112353804, -0.0036957447374920682, 
     0.0035726565409773814, -0.0034557776600199763, 0.003344691710724629, 
     -0.0032390169250685052, 0.003138402734717355, -0.003042526743391949, 
     0.002951092037931785, -0.0028638247953020335, 0.0027804721487847406, 
     -0.0027008002816604204, 0.002624592720995189, -0.0025516488078090177, 
     0.0024817823230267877, -0.002414820251287156, 0.0023506016669782304, 
     -0.0022889767288382795, 0.0022298057711591934, -0.0021729584810964847, 
     0.002118313152857612, -0.00206575601064192, 0.0020151805931611783, 
     -0.0019664871934027273, 0.001919582348022239, -0.0018743783713905254, 
     0.0018307929298716358, -0.0017887486523996687, 0.0017481727738485886, 
     -0.0017089968080675477, 0.0016711562477856323, -0.001634590288884713, 
     0.0015992415767975, -0.0015650559730196884, 0.0015319823399267407, 
     -0.0014999723422693678, 0.0014689802638817306, -0.0014389628382804054, 
     0.0014098790919603588, -0.0013816901993086023, 0.00135435934815821, 
     -0.0013278516150980885, 0.001302133849734835, -0.001277174567177547, 
     0.0012529438480829297, -0.0012294132456565632, 0.0012065556990621396, 
     -0.0011843454527359493, 0.001162757981151022};

  lnyox = TVIL_CLOG(y/x);
  lnyox2 = lnyox  * lnyox;
  lnyox3 = lnyox2  * lnyox;

  if (y < 2. * x) {     
    result = TVIL_H00x0xx (x,qq);     
    rn = r = 0.25L * y/x;
    for (n=1; n < 100; n++) {
      result += rn * (H00xyxxtableA[n] + 
                      H00xyxxtableAL[n] * lnyox +
                      H00xyxxtableAL2[n] * lnyox2);
      rn *= r;
    }
  }
  else if (y > 6.47 * x) {
    result = TVIL_H00000x (y, qq);
    rn = r = 4.L * x/y;
    for (n=1; n < 100; n++) {
      result += rn * (H00xyxxtableC[n] +
                     -H00xyxxtableCL[n] * lnyox +
                     +H00xyxxtableCL2[n] * lnyox2 +
                     -H00xyxxtableCL3[n] * lnyox3);
      rn *= r;
    }
  }
  else {
    result = -9.403507956757080718L - 6.L * Zeta3 * TVIL_Ap(x,qq); 
    rn = r = 0.25L * y/x - 1.L;
    for (n=1; n < 100; n++) {
      result += rn * H00xyxxtableB[n];
      rn *= r;
    }
  }

  return result;
}

/* ****************************************************************** */

TVIL_COMPLEX TVIL_H00xxxy (TVIL_REAL x, TVIL_REAL y, TVIL_REAL qq)
{
  TVIL_REAL r, rn;
  TVIL_COMPLEX lnyox, lnyox2, lnyox3, result;
  int n;
  
  TVIL_REAL H00xxxytableA[] = 
      {0, -2.0018374258594345, -0.5237717540537673, 
     -0.17977238330662387, -0.07532513896071008, -0.03698279344431988, 
     -0.020537181387706258, -0.012517297358194722, -0.008182086212874022, 
     -0.005641000802617092, -0.0040538915808804345, -0.0030114338806556153, 
     -0.0022983379215537098, -0.0017939851302012047, -0.0014271725329481444, 
     -0.0011539939564214213, -0.0009463589133344908, -0.000785719826366258, 
     -0.0006594912764238116, -0.0005589272023354966, -0.0004778203183260859, 
     -0.00041168190788527185, -0.0003572111461558853, -0.0003119438105629195, 
     -0.00027401491572767305, -0.00024199532297983082, 
     -0.0002147773587920781, -0.00019149350092271928, 
     -0.00017145775193257406, -0.00015412281851842323, 
     -0.00013904845885774595, -0.00012587782453086488, 
     -0.00011431959489204814, -0.00010413435574443687, 
     -0.00009512412067096044, -0.00008712420218687674, 
     -0.00007999685606429336, -0.00007362627524114822, 
     -0.00006791461925357034, -0.00006277884428970507, 
     -0.00005814815671434321, -0.00005396195542147482, -0.000050168159922686, 
     -0.00004672184468368209, -0.00004358411801364419, 
     -0.00004072119731956327, -0.00003810364286181029, 
     -0.000035705720089247126, -0.00003350486677942201, 
     -0.00003148124599539414, -0.000029617369617840967, 
     -0.000027897780160569198, -0.000026308780911088536, 
     -0.000024838206293217418, -0.00002347522583073245, 
     -0.00002221017628030099, -0.00002103441746034766, 
     -0.000019940208078124577, -0.000018920598487465914, 
     -0.000017969337823749914, -0.00001708079338344079, 
     -0.000016249880461371667, -0.000015472001144029248, 
     -0.00001474299079294827, -0.000014059071148057986, 
     -0.000013416809143772704, -0.000012813080666684826, 
     -0.000012245038597668416, -0.00001171008457689898, 
     -0.000011205844010884181, -0.000010730143908646078, 
     -0.00001028099319179823, -9.856565172144843e-6, -9.4551819320144e-6, 
     -9.075300377999613e-6, -8.715499769080541e-6, -8.374470546064993e-6, 
     -8.05100431156166e-6, -7.743984828867321e-6, -7.452379924669301e-6, 
     -7.175234194731711e-6, -6.911662424079725e-6, -6.660843643899264e-6, 
     -6.4220157566655e-6, -6.194470669101962e-6, -5.977549879621921e-6, 
     -5.770640473058848e-6, -5.5731714808756425e-6, -5.384610569757469e-6, 
     -5.204461025629754e-6, -5.032259003777795e-6, -4.867571018943172e-6, 
     -4.709991652091184e-6, -4.559141453031559e-6, -4.4146650202736035e-6, 
     -4.276229241443063e-6, -4.143521679312646e-6, -4.016249090028573e-6, 
     -3.894136061475339e-6, -3.776923760930652e-6, -3.6643687822400234e-6};
 
  TVIL_REAL H00xxxytableAL[] = 
     {0, -1.6449340668482262, -0.5724670334241131, 
     -0.2566446889494088, -0.1383168500453899, -0.08503840067123256, 
     -0.057326974104334044, -0.04123697347471053, -0.031091672826478456, 
     -0.02428517447305728, -0.01949604432690355, -0.015997840783815498, 
     -0.01336437827741245, -0.011332105477754165, -0.009730851742749125, 
     -0.008446718429967762, -0.0074011186889088865, -0.006538399712167894, 
     -0.005818250483743376, -0.0052108899355867155, -0.004693929076570069, 
     -0.004250272777776204, -0.0038666851943456454, -0.0035327892399835414, 
     -0.003240356205963635, -0.0029827931703419186, -0.002754767656171317, 
     -0.0025519290804592726, -0.0023706994780381407, -0.002208114483812626, 
     -0.002061701235477137, -0.0019293837135462982, -0.0018094086908285248, 
     -0.0017002873175084102, -0.00160074867898799, -0.0015097026015761447, 
     -0.001426209659552695, -0.0013494568329290857, -0.0012787376310495688, 
     -0.0012134357695534892, -0.0011530116927521358, -0.0010969913883026317, 
     -0.001044957059156533, -0.0009965393084890237, -0.0009514105634925371, 
     -0.00090927951855483, -0.0008698864211390545, -0.000832999057404811, 
     -0.0007984093213278323, -0.0007659302723587169, -0.0007353936037006042, 
     -0.0007066474569941902, -0.000679554530278615, -0.0006539904350921846, 
     -0.0006298422659117842, -0.0006070073511343956, -0.000585392159739722, 
     -0.0005649113418449286, -0.0005454868847343912, -0.0005270473687490305, 
     -0.0005095273097561609, -0.0004928665768747335, -0.00047700987577062315, 
     -0.00046190628921681867, -0.0004475088677783068, 
     -0.00043377426446765746, -0.0004206624080545056, 
     -0.00040813621042464984, -0.000396161303992313, -0.0003847058056893615, 
     -0.00037374010450101636, -0.00036323666990098827, 
     -0.00035316987886915147, -0.00034351585945962756, 
     -0.00033425234913416965, -0.0003253585662890933, 
     -0.00031681509359005045, -0.00030860377189027496, 
     -0.00030070760364930144, -0.0002931106648917589, 
     -0.00028579802485424743, -0.0002787556725622961, -0.0002719704496627201, 
     -0.0002654299889096323, -0.0002591226577666472, -0.00025303750664476543, 
     -0.0002471642213454049, -0.0002414930793227417, -0.00023601490941871708, 
     -0.0002307210547591275, -0.0002256033385304064, -0.00022065403238415507, 
     -0.0002158658272415001, -0.0002112318062910401, -0.00020674541999413446, 
     -0.00020240046292873543, -0.00019819105231878506, 
     -0.00019411160811047892, -0.00019015683446923018, 
     -0.0001863217025828076, -0.0001826014346662519};

  TVIL_REAL H00xxxytableB[] = 
     {0, -3.1367204059992617, 1.02405686873614, -0.5111637812120541, 
     0.3075805973977935, -0.2058676274757182, 0.14764174262949725, 
     -0.11115024596756605, 0.08674814846215805, -0.06961418050897154, 
     0.05711622772615627, -0.04771657458910572, 0.04046719489009554, 
     -0.034757424578047806, 0.03017931562756071, -0.026451825996391745, 
     0.02337617155577874, -0.020808503661621524, 0.01864262747624206, 
     -0.016798751751043776, 0.015215977090940374, -0.013847166406816285, 
     0.012655371080989252, -0.011611295698685334, 0.010691469973331352, 
     -0.009876910933648566, 0.00915213057259579, -0.008504390579798005, 
     0.007923136227167631, -0.007399561798916166, 0.006926273738759701, 
     -0.0064970271725368475, 0.0061065180828012095, -0.005750218088812486, 
     0.0054242421295496775, -0.005125241765171428, 0.004850318578515205, 
     -0.004596953460932227, 0.004362948536394483, -0.004146379205745648, 
     0.003945554343880191, -0.003758983102766872, 0.0035853470959516386, 
     -0.0034234769897689473, 0.003272332720786766, -0.003130986711196491, 
     0.0029986095737634865, -0.0028744578929544404, 0.002757863744529584, 
     -0.002648225676471537, 0.002545000922863391, -0.0024476986617249343, 
     0.002355874159801613, -0.0022691236733813494, 0.0021870799955673575, 
     -0.0021094085579867108, 0.0020358040093961054, -0.001965987205638705, 
     0.001899702555371882, -0.001836715674295628, 0.0017768113075631651, 
     -0.0017197914858891356, 0.001665473885781362, -0.0016136903684677662, 
     0.0015642856755997405};

  TVIL_REAL H00xxxytableC[] = 
    {0, 0.585870317864344, 1.0749440871750642, 0.7945196888382672, 
     0.5895917448978106, 0.4545582718893676, 0.36275303657612046, 
     0.29765529667343843, 0.24972739347131054, 0.21331138625131096, 
     0.1849083442465826, 0.16226500145691228, 0.14387679028992437, 
     0.12870682605896683, 0.11602064385723486, 0.10528540022463197, 
     0.09610634001964423, 0.08818555963660289, 0.0812945329914469, 
     0.07525537461357738, 0.06992779118333772, 0.06519982184902946, 
     0.060981154473962684, 0.057198226177995797, 0.05379058097785075, 
     0.05070812693471472, 0.04790904614809814, 0.04535818480595726, 
     0.04302580051470128, 0.04088657851551592, 0.03891885236746102, 
     0.03710398161514831, 0.035425851071982116, 0.033870465112308616, 
     0.032425616772186465, 0.031080616189315652, 0.029826066439047106, 
     0.02865367747524076, 0.027556110895544186, 0.026526849787374185, 
     0.025560089093999896, 0.024650642857412478, 0.023793865410588803, 
     0.0229855841540511, 0.02222204199591147, 0.021499847887642205, 
     0.02081593416989338, 0.020167519669225566, 0.019552077669473557, 
     0.01896730802973494, 0.01841111284176682, 0.017881575118397965, 
     0.017376940085753394, 0.01689559871905688, 0.01643607321722688, 
     0.015997004157563333, 0.015577139110255201, 0.015175322524600337, 
     0.0147904867258322, 0.014421643884191105, 0.014067878837091624, 
     0.013728342661517989, 0.013402246907613043, 0.013088858416213359, 
     0.012787494653153207, 0.012497519501785221, 0.012218339462571623, 
     0.011949400214974447, 0.011690183502372685, 0.011440204305489612, 
     0.011199008273934477, 0.010966169389041384, 0.010741287834302499, 
     0.010523988052408256, 0.010313916970279662, 0.01011074237555459, 
     0.009914151429811035, 0.009723849305410181, 0.009539557934250181, 
     0.009361014857962643, 0.009187972170179859, 0.009020195542470036, 
     0.008857463326396446, 0.008699565724918073, 0.008546304027026178, 
     0.008397489900113308, 0.008252944735107915, 0.008112499039886454, 
     0.007975991876902632, 0.007843270341356208, 0.007714189076566506, 
     0.007588609823523402, 0.007466401001864631, 0.007347437319776609, 
     0.007231599410539449, 0.007118773493638315, 0.007008851058545002, 
     0.006901728569437787, 0.006797307189276052, 0.006695492521780521, 
     0.006611859862366555};
 
  TVIL_REAL H00xxxytableCL[] = 
      {0, -3.894934066848226, -1.4733962917801415, 
     -0.804765568127423, -0.5256161807781027, -0.3791501568585194, 
     -0.2908160363011469, -0.23250205316651362, -0.1915268235960676, 
     -0.16139293732498686, -0.13844758503845897, -0.12048881936677944, 
     -0.10611549485993722, -0.09439667297840391, -0.0846915674493769, 
     -0.07654589365154811, -0.06962941177263608, -0.06369682031828511, 
     -0.05856245566358826, -0.05408345410681581, -0.050148265495223034, 
     -0.04666864483296625, -0.04357395944447823, -0.04080707127702656, 
     -0.0383213115387549, -0.03607822615007485, -0.03404587377863209, 
     -0.032197525754650756, -0.03051066214090776, -0.028966188705421368, 
     -0.027547820517899387, -0.0262415925348443, -0.025035467900106485, 
     -0.02391902211084343, -0.022883186578186235, -0.021920039052144534, 
     -0.02102263129543962, -0.020184846568046973, -0.019401281124499896, 
     -0.01866714517210502, -0.017978179692290683, -0.01733058626319474, 
     -0.01672096759316628, -0.016146276921738936, -0.01560377479622663, 
     -0.015090992010366285, -0.01460569771292576, -0.014145871871448031, 
     -0.013709681418881759, -0.013295459526101151, -0.012901687536919414, 
     -0.012526979178559354, -0.012170066723106159, -0.011829788826935672, 
     -0.01150507981762116, -0.011194960233065421, -0.010898528446931541, 
     -0.010614953238933982, -0.010343467189069158, -0.010083360792110987, 
     -0.009833977203239233, -0.009594707537967609, -0.00936498665997112, 
     -0.00914428939928617, -0.008932127150925822, -0.008728044810425384, 
     -0.008531618008382945, -0.008342450610828937, -0.00816017245636795, 
     -0.00798443730458409, -0.007814920973272011, -0.007651319644719056, 
     -0.007493348323578776, -0.007340739430892208, -0.007193241520572635, 
     -0.007050618106207846, -0.006912646587381483, -0.00677911726589759, 
     -0.006649832443332239, -0.0065246055922515995, -0.006403260594243657, 
     -0.00628563103862433, -0.006171559576310465, -0.0060608973239119615, 
     -0.005953503313592334, -0.005849243984688793, -0.005747992713476381, 
     -0.005649629377811419, -0.005554039953702582, -0.0054611161411379765, 
     -0.005370755016747068, -0.005282858711101009, -0.005197334108656421, 
     -0.0051140925685288175, -0.005033049664444813, -0.004954124942368973, 
     -0.00487724169443347, -0.0048023267479182065, -0.004729310268136915, 
     -0.0046581255741824645, -0.004611721237897026};
 
  TVIL_REAL H00xxxytableCL2[] = 
     {0, -0.75, -0.234375, -0.08333333333333333, -0.032958984375, 
     -0.0137890625, -0.005594889322916667, -0.0017523940728635203, 
     0.00017075538635253907, 0.001168960490554729, 0.0016894442694527763, 
     0.0019501629822463418, 0.0020644755235740115, 0.0020944497269474766, 
     0.0020754579919745747, 0.0020282866014746576, 0.0019654306225108443, 
     0.0018945178921172827, 0.001820243263293511, 0.0017454965785438726, 
     0.0016720383510939873, 0.001600913806469657, 0.0015327115033696027, 
     0.001467727481187457, 0.0014060708201278156, 0.0013477322303520093, 
     0.001292628956354538, 0.0012406343101957077, 0.0011915971190656492, 
     0.0011453544951925063, 0.001101740153088354, 0.0010605897427283005, 
     0.0010217441772264095, 0.0009850516123224874, 0.000950368522143078, 
     0.0009175601733289971, 0.0008865007035928135, 0.0008570729455246915, 
     0.0008291680918595764, 0.0008026852677678019, 0.0007775310545861888, 
     0.000753618994780464, 0.000730869097799995, 0.000709207359472893, 
     0.0006885653027460001, 0.0006688795442454127, 0.0006500913888640359, 
     0.0006321464530556135, 0.0006149943165070315, 0.0005985882012158699, 
     0.0005828846766077036, 0.0005678433891095446, 0.0005534268144969155, 
     0.0005396000313134176, 0.0005263305136957979, 0.0005135879420051747, 
     0.0005013440297527001, 0.000489572365406228, 0.00047824826776709085, 
     0.0004673486537085726, 0.0004568519171673062, 0.0004467378183738378, 
     0.00043698738239797945, 0.00042758280616779666, 0.0004185073731979724, 
     0.0004097453753339396, 0.0004012820408827782, 0.0003931034685607557, 
     0.0003851965667409129, 0.0003775489975326414, 0.00037014912526916945, 
     0.0003629859690186454, 0.0003560491587704594, 0.0003493288949809166, 
     0.0003428159111917038, 0.00033650143946106314, 0.00033037717837149533, 
     0.0003244352633993941, 0.00031866823945150564, 0.0003130690353907162, 
     0.00030763094038958216, 0.00030234758196440284, 0.00029721290555564706, 
     0.000292221155532323, 0.0002873668575085405, 0.0002826448018701781, 
     0.0002780500284183236, 0.0002735778120441017, 0.0002692236493567174, 
     0.000264983246193089, 0.00026085250594340653, 0.0002568275186323608, 
     0.00025290455070072333, 0.0002490800354364453, 0.00024535056400853766, 
     0.00024171287706072683, 0.00023816385682529054, 0.00023470051972058738, 
     0.00023132000939864143, 0.000228019590211745, 0.00023324891292081119};
 
  TVIL_REAL H00xxxytableCL3[] = 
     {0, -0.125, -0.078125, -0.049479166666666664, -0.03369140625, 
     -0.02451171875, -0.018778483072916668, -0.014957972935267858, 
     -0.012272834777832031, -0.010303709242078993, -0.008809804916381836, 
     -0.007644902576099743, -0.006715841591358185, -0.005960807777368105, 
     -0.00533735632364239, -0.004815481572101514, -0.0043734354330808856, 
     -0.003995110573562081, -0.0036683499876946574, -0.0033838242271533665, 
     -0.003134267190466744, -0.002913944553510884, -0.0027182767890743457, 
     -0.0025435671277736806, -0.002386802139864234, -0.0022455034531843232, 
     -0.0021176160523742817, -0.002001423127689825, -0.0018954804493746117, 
     -0.0017985652777656842, -0.001709636216809492, -0.0016278013926958222, 
     -0.0015522930273119827, -0.0014824469682318295, -0.0014176860928895218, 
     -0.001357506764864824, -0.0013014677124572095, -0.0012491808430743414, 
     -0.0012003036147961554, -0.0011545326682753292, -0.0011115984846738404, 
     -0.0010712608835048312, -0.0010333052116119445, -0.0009975391036924398, 
     -0.0009637897176945809, -0.0009319013665412294, -0.0009017334820383588, 
     -0.0008731588583431641, -0.000846062131619407, -0.0008203384599791668, 
     -0.0007958923738717877, -0.0007726367720208627, -0.0007504920420535488, 
     -0.0007293852882898498, -0.0007092496519018755, -0.0006900237109247007, 
     -0.0006716509494890463, -0.0006540792872217119, -0.0006372606610788231, 
     -0.0006211506529849029, -0.000605708157584306, -0.000590895085200492, 
     -0.0005766760957673376, -0.0005630183600651129, -0.0005498913450782433, 
     -0.0005372666207060327, -0.0005251176854122897, -0.0005134198087045087, 
     -0.0005021498885956187, -0.0004912863224277235, -0.0004808088896330914, 
     -0.00047069864517740653, -0.0004609378225777593, 
     -0.00045150974551621387, -0.00044239874718174675, 
     -0.00043359009657119465, -0.00042506993106551075, 
     -0.0004168251946727744, -0.00040884358139542756, -0.0004011134832373342, 
     -0.00039362394241751204, -0.0003863646074026487, -0.000379325692410557, 
     -0.0003724979400722047, -0.00036587258697143066, 
     -0.00035944133180944147, -0.0003531963059660784, -0.0003471300462520356, 
     -0.00034123546966601507, -0.0003355058499885021, 
     -0.00032993479605968073, -0.00032451623160320006, 
     -0.00031924437647023215, -0.0003141137291896954, -0.0003091190507208009, 
     -0.0003042553493133379, -0.0002995178663894594, -0.0002949020633682593, 
     -0.0002904036093612362, -0.0002860183696728918, -0.0002817423950462821};

  TVIL_REAL H00xxxytableD[] = 
     {0, -1.7565021624839214, 0.4486609263461504, 
     -0.19666799319246364, 0.10924868529394295, -0.06926246094895291, 
     0.04775912516360917, -0.03489772034546111, 0.026604194270673367, 
     -0.02094810232142802, 0.016919899144109213, -0.01395031293885434, 
     0.011698655113410304, -0.009950963585654834, 0.008567396603633088, 
     -0.007453437993533912, 0.006543351739846642, -0.005790274969628928, 
     0.005160071840011441, -0.004627401297154714, 0.0041731322929471105, 
     -0.003782603579979171, 0.003444426862955732, -0.003149647751071713, 
     0.0028911473055271072, -0.002663208458957223, 0.0024611973740282208, 
     -0.0022813261970076513, 0.0021204742855948948, -0.0019760520032654737, 
     0.0018458958800162465, -0.0017281871486281553, 0.0016213878847469697, 
     -0.0015241905340876447, 0.001435477713169881, -0.0013542899615538882, 
     0.0012797996976804695, -0.001211290051059335, 0.001148137554654336, 
     -0.0010897979134598058, 0.0010357942399446586, -0.000985707279527106, 
     0.0009391672504795593, -0.0008958470005708232, 0.0008554562431110046, 
     -0.0008177366821258894, 0.0007824578733044684, -0.0007494136964902495, 
     0.000718419338594653, -0.000689308704239683, 0.0006619321862085234, 
     -0.0006361547396795855, 0.0006118542138454012, -0.00058891990234051, 
     0.0005672512802868224, -0.0005467569009965211, 0.0005273534296762701, 
     -0.0005089647950302033, 0.0004915214426041574, -0.0004749596761626571, 
     0.00045922107543360224, -0.00044425198026612305, 0.00043000303268337975, 
     -0.0004164287695217614, 0.0004034872593697211, -0.00039113977838497045, 
     0.0003793505203038601, -0.00036808633658276394, 0.0003573165031457331, 
     -0.00034701251067009026, 0.00033714787573404826, 
     -0.00032769797048786884, 0.0003186398688008873, -0.0003099522070879044, 
     0.0003016150582358489, -0.0002936098172401759, 0.0002859190973243344, 
     -0.00027852663545832975, 0.0002714172063168899, -0.00026457654382654487, 
     0.00025799126954619304, -0.000251648827209281, 0.00024553742282913937, 
     -0.00023964596983362124, 0.00023396403875213803, 
     -0.00022848181102846937, 0.00022319003657717603, 
     -0.00021807999474081694, 0.0002131434583400828, -0.0002083726605399665, 
     0.00020376026428266592, -0.00019929933406247257, 0.00019498330983979783, 
     -0.000190805982911043, 0.0001867614735684998, -0.00018284421040011713, 
     0.0001790489110929939, -0.00017537056461704224, 0.0001718044146765702, 
     -0.00016834594432769983, 0.00016499086166869156};

  lnyox = TVIL_CLOG(y/x);
  lnyox2 = lnyox  * lnyox;
  lnyox3 = lnyox2  * lnyox;

  if (y < 0.5 * x) {
    result = TVIL_H00xx0x (x,qq);
    rn = r = y/x;
    for (n=1; n < 100; n++) {
      result += rn * (H00xxxytableA[n] - H00xxxytableAL[n] * lnyox); 
      rn *= r;
    }
  } 
  else if (y < 1.6 * x) {
    result = TVIL_H00xxxx (x,qq);
    rn = r = y/x - 1.L;
    for (n=1; n < 100; n++) {
      result += rn * H00xxxytableD[n]; 
      rn *= r;
    }
  } 
  else if (y < 6. * x) {
    result = -9.267575567256300L - 6.L * Zeta3 * TVIL_Ap(x,qq);
    rn = r = y/(4.L*x) - 1.L;
    for (n=1; n < 64; n++) {
      result += rn * H00xxxytableB[n]; 
      rn *= r;
    }
  }
  else {
    result = TVIL_H00000x (y, qq);
    rn = r = 4.L * x/y;
    for (n=1; n < 100; n++) {
      result += rn * (H00xxxytableC[n] +
                     +H00xxxytableCL[n] * lnyox +
                     +H00xxxytableCL2[n] * lnyox2 +
                     +H00xxxytableCL3[n] * lnyox3);
      rn *= r;
    }
  }

  return result;
}

/* ****************************************************************** */

TVIL_COMPLEX TVIL_H00y0xx (TVIL_REAL x, TVIL_REAL y, TVIL_REAL qq)
{
  TVIL_REAL r, rn;
  TVIL_COMPLEX lnyox, lnyox2, lnyox3, result;
  int n;
  
  TVIL_REAL H00y0xxtableA[] = 
      {0, -1.7715086547545278, -0.7408202605290377, 
     -0.40611704771975266, -0.2566303172561659, -0.17700689113100387, 
     -0.1295565905063698, -0.09898591113450883, -0.07812679045050075, 
     -0.06325333350534057, -0.052271522663895675, -0.04393025233847847, 
     -0.03744428858966081, -0.03230039920448213, -0.028151538692552214, 
     -0.024756155105835842, -0.021941885527808458, -0.019583044429801577, 
     -0.017586222283475283, -0.01588081733159641, -0.014412658599678996, 
     -0.013139617646888563, -0.012028530172520396, -0.011052998722542062, 
     -0.01019179947421825, -0.00942771038884993, -0.008746637949629849, 
     -0.008136958551284323, -0.0075890162597475885, -0.007094735884676462, 
     -0.006647322054249172, -0.006241023108130073, -0.005870944321591465, 
     -0.005532899017413384, -0.005223289025450578, -0.004939008056798727, 
     -0.004677363104164278, -0.004436010123188033, -0.004212901103033823, 
     -0.0040062402772409605, -0.0038144477135952787, -0.0036361288947266007, 
     -0.0034700491883323387, -0.0033151123285743295, -0.0031703422039043444, 
     -0.003034867382924633, -0.0029079079175394207, -0.002788764048108669, 
     -0.002676806503510737, -0.0025714681437167958, -0.0024722367365571096, 
     -0.0023786486960459124, -0.002290283638649235, -0.002206759637577549, 
     -0.002127729074613491};
 
  TVIL_REAL H00y0xxtableAL[] = 
     {0, 3.2898681336964524, 2.144934066848226, 1.596622711232151, 
     1.273855922313002, 1.0607514045170683, 0.909329874134594, 
     0.7960922730677472, 0.7081559940363196, 0.637860442670732, 
     0.5803609945235884, 0.5324421739536229, 0.49188496923637315, 
     0.4571080294708639, 0.42695363627878, 0.4005545445437263, 
     0.37724812977705013, 0.35651931634432965, 0.3379616174053167, 
     0.32124991272753495, 0.30612103279040653, 0.29235965473397985, 
     0.2797878874103304, 0.2682574659866156, 0.2576438233082915, 
     0.24784153146632698, 0.23876075755522602, 0.23032447959957875, 
     0.22246627886872192, 0.21512857391019172, 0.20826119645016516, 
     0.2018202343122251, 0.19576708467617246, 0.19006767435055363, 
     0.18469181364567153, 0.17961265786518035, 0.17480625605583083, 
     0.17025117094356768, 0.1659281572822101, 0.16181988839633785, 
     0.15791072269408762, 0.1541865034921181, 0.15063438673353519, 
     0.14724269216468885, 0.14400077432480818, 0.1408989103362957, 
     0.13792820199597822, 0.13508049008400413, 0.13234827914700237, 
     0.1297246712908693, 0.12720330774812671, 0.12477831717464677, 
     0.12244426978817335, 0.12019613659243704, 0.11802925304054959};
 
  TVIL_REAL H00y0xxtableB[] = 
     {0, -2.4041138063191885, 0.7020569031595942, 
     -0.32914904655084054, 0.19014882195016747, -0.12364683533791176, 
     0.08677668376924741, -0.06423451201385233, 0.04945610402928957, 
     -0.039246715766299956, 0.031900338120411015, -0.026438707826544672, 
     0.02226826980820068, -0.019011956510902345, 0.01642091895935105, 
     -0.014325558749997136, 0.012607065347161384, -0.011180202275942103, 
     0.009982534269668844, -0.008967481477756264, 0.008099722339680843, 
     -0.007352079497221442, 0.006703366159663909, -0.006136868725103578, 
     0.005639259567403007, -0.005199806064228096, 0.00480978708500146, 
     -0.004462057005823708, 0.0041507161154377, -0.003870858744220093, 
     0.003618378854547591, -0.0033898185850389922, 0.0031822492353346665, 
     -0.0029931769867646285, 0.0028204676533238743, -0.002662286196320071, 
     0.00251704778280204, -0.0023833779368807873, 0.0022600799032780824, 
     -0.002146107768977938, 0.002040544210554558, -0.0019425819792737165, 
     0.0018515084232989532, -0.0017666924907055945, 0.0016875737690661974, 
     -0.001613653204899805, 0.0015444852150559019, -0.0014796709564563887, 
     0.0014188525638071797, -0.001361708199383832, 0.0013079477866839787, 
     -0.001257309322068162, 0.0012095556766017965, -0.0011644718150307931, 
     0.0011218623708515799, -0.001081549526303828, 0.0010433711542404758, 
     -0.0010071791855473106, 0.0009728381713567241, -0.0009402240139391241, 
     0.0009092228440294864, -0.0008797300255928495, 0.000851649271760041, 
     -0.0008248918579652417, 0.0007993759202603035, -0.0007750258284293168, 
     0.0007517716249272957, -0.000729548521861044, 0.0007082964492497942, 
     -0.0006879596486771902, 0.0006684863071958896, -0.0006498282269914515, 
     0.0006319405268685463, -0.0006147813721038703, 0.00059831172962619, 
     -0.0005824951458458169, 0.0005672975447700186, -0.0005526870443148708, 
     0.0005386337889628119, -0.0005251097971247162, 0.0005120888217479344, 
     -0.000499546222872755, 0.00048745885098107, -0.00047580494010507, 
     0.0004645640097739358, -0.00045371677497306473, 0.00044324506337613256, 
     -0.000433131739186441, 0.0004233606329910218, -0.0004139164770911932, 
     0.0004047848458261852, -0.00039595210045411874, 0.0003874053381969639, 
     -0.00037913234509363847, 0.0003711215523395754, -0.0003633619958211473, 
     0.00035584327858046924, -0.00034855553597066755, 0.0003414894032833342, 
     -0.0003346359856497595, 0.00032798683003531365};

  TVIL_REAL H00y0xxtableC[] = 
     {0, -9.771508654754527, -1.2408202605290377, 
     -0.5048824798185181, -0.2878803172561659, -0.1898068911310038, 
     -0.13572943001254256, -0.10231785615741606, -0.08007991545050075, 
     -0.06447265982754746, -0.0530715226638957, -0.04447666310277054, 
     -0.037830091058796644, -0.03258050144179869, -0.028359785256483916, 
     -0.024914179797193903, -0.022063955840308513, -0.019678828723572017, 
     -0.017662430178613242, -0.01594220421475412, -0.014462658599678935, 
     -0.013180752770628162, -0.01206268084528861, -0.011081586384821707, 
     -0.010215912128539256, -0.009448190388849947, -0.008764144339462127, 
     -0.00815201196266957, -0.00760203166999332, -0.007106046806361066, 
     -0.0066571985974590575, -0.006249685607412708, -0.005878573716122715, 
     -0.005539644829318235, -0.005229275543811196, -0.004944339168835382, 
     -0.004682126097610428, -0.004440278699900491, -0.0042167377832312, 
     -0.004009698329553488, -0.0038175727135952714, -0.0036389599904916164, 
     -0.003472620133566043, -0.0033174523302298836, -0.0031724766209523664, 
     -0.0030368183050401704, -0.0029096946464319123, -0.0027904034995400173, 
     -0.0026783135444058, -0.002572855875921251, -0.002473516736557116, 
     -0.002379831218191214, -0.002291377788013735, -0.0022077735174664964, 
     -0.0021286699128250652, -0.002053749262200602, -0.0019827214270896343, 
     -0.0019153210176665186, -0.001851304900217357, -0.0017904499927947626, 
     -0.0017325513116035612, -0.0017505745439953813, -0.0016944131909985144, 
     -0.0016409074191613138, -0.0015898924680629312, -0.0015412161579330536, 
     -0.0014947377543294699, -0.0014503269505709987, -0.001407862954171446, 
     -0.0013672336653019757, -0.001328334936837386, -0.0012910699068568732, 
     -0.001255348395603345, -0.0012210863598844518, -0.0011882053987462143, 
     -0.001156632304985819, -0.0011262986577091072, -0.0010971404516955663, 
     -0.0010690977598189835, -0.0010421144251971867, -0.0010161377801158472, 
     -0.0009911183890980516, -0.0009670098137776403, -0.000943768397487374, 
     -0.0009213530676948425, -0.0008997251546162197, -0.0008788482245111959, 
     -0.0008586879263168326, -0.0008392118504141205, -0.0008203893984429569, 
     -0.0008021916631882897, -0.0007845913176573183, -0.0007675625125521807, 
     -0.000751080781420034, -0.0007351229528304165, -0.0007196670689911255, 
     -0.0007046923102686647, -0.0006901789251288828, -0.0006761081650574485, 
     -0.0006624622240595998, -0.0006492241823748482};
 
  TVIL_REAL H00y0xxtableCL[] = 
     {0, 9.289868133696451, 2.894934066848226, 1.8188449334543733, 
     1.367605922313002, 1.1087514045170683, 0.9371076519123718, 
     0.8135849844380096, 0.7198747440363196, 0.6460908953456292, 
     0.5863609945235884, 0.5369500627590323, 0.49535719145859536, 
     0.4598390262847009, 0.4291402252000628, 0.40233232232150407, 
     0.37871297352705013, 0.3577405660899026, 0.3389904239896788, 
     0.3221246758125328, 0.30687103279040656, 0.29300753293287846, 
     0.2803513735110066, 0.26875060316093957, 0.2580778510860693, 
     0.24822553146632698, 0.23910213215695564, 0.2306293111801305, 
     0.22273960248388225, 0.2153745864568316, 0.2084834186723874, 
     0.2020216374205464, 0.19595019014492246, 0.19023463319519845, 
     0.18484446986386815, 0.17975259955614245, 0.1749348568788761, 
     0.17036962394733846, 0.16603750266783485, 0.16192103642647998, 
     0.1580044726940876, 0.15427355968689183, 0.1507153715083975, 
     0.1473181572180804, 0.1440712100873927, 0.14096475395769487, 
     0.13798984414276871, 0.13513828074695938, 0.13240253261922458, 
     0.12977567044938318, 0.1272513077481267, 0.12482354864670503, 
     0.12248694161338955, 0.12023643831802261, 0.11806735698811854, 
     0.11597534969404188, 0.11395637307994151, 0.11200666212395151, 
     0.11012270656777007, 0.10830122970381631, 0.10653916924913555, 
     0.10733142063336483, 0.1055999912915541, 0.10392354085680952, 
     0.10229949140567485, 0.10072542371529623, 0.09919906523539339, 
     0.0977182791379043, 0.09628105433330511, 0.0948854963554834, 
     0.09352981902826424, 0.09221233683648503, 0.09093145793308785, 
     0.08968567772121164, 0.08847357295686689, 0.08729379632358328, 
     0.08614507143553948, 0.08502618823020572, 0.0839359987155276, 
     0.08287341304022404, 0.08183739585891488, 0.08082696296658941, 
     0.07984117817941463, 0.07887915044109986, 0.07794003113601439, 
     0.07702301159202574, 0.07612732075761058, 0.07525222303921186, 
     0.07439701628609091, 0.07356102991107011, 0.07274362313659374, 
     0.07194418335646399, 0.07116212460444804, 0.07039688612171012, 
     0.06964793101570671, 0.0689147450038041, 0.06819683523543832, 
     0.06749372918714854, 0.06680497362527675, 0.0661301336315486, 
     0.0654687916871319};
 
  TVIL_REAL H00y0xxtableCL2[] = 
     {0, -2., -0.5, -0.2222222222222222, -0.125, -0.08, 
     -0.05555555555555555, -0.04081632653061224, -0.03125, 
     -0.024691358024691357, -0.02, -0.01652892561983471, 
     -0.013888888888888888, -0.011834319526627219, -0.01020408163265306, 
     -0.008888888888888889, -0.0078125, -0.006920415224913495, 
     -0.006172839506172839, -0.00554016620498615, -0.005, 
     -0.0045351473922902496, -0.004132231404958678, -0.003780718336483932, 
     -0.003472222222222222, -0.0032, -0.0029585798816568047, 
     -0.0027434842249657062, -0.002551020408163265, -0.0023781212841854932, 
     -0.0022222222222222222, -0.002081165452653486, -0.001953125, 
     -0.0018365472910927456, -0.0017301038062283738, -0.0016326530612244899, 
     -0.0015432098765432098, -0.0014609203798392988, -0.0013850415512465374, 
     -0.0013149243918474688, -0.00125, -0.001189767995240928, 
     -0.0011337868480725624, -0.001081665765278529, -0.0010330578512396695, 
     -0.0009876543209876543, -0.000945179584120983, -0.0009053870529651426, 
     -0.0008680555555555555, -0.0008329862557267805, -0.0008, 
     -0.0007689350249903883, -0.0007396449704142012, -0.000711997152011392, 
     -0.0006858710562414266, -0.0006611570247933885, -0.0006377551020408163, 
     -0.0006155740227762388, -0.0005945303210463733, -0.0005745475438092502, 
     -0.0005555555555555556, -0.0002687449610319806, -0.00026014568158168577, 
     -0.0002519526329050139, -0.000244140625, -0.00023668639053254438, 
     -0.0002295684113865932, -0.00022276676319893073, 
     -0.00021626297577854672, -0.00021003990758244065, 
     -0.00020408163265306123, -0.00019837333862328903, 
     -0.00019290123456790122, -0.00018765246762994934, 
     -0.00018261504747991235, -0.00017777777777777779, 
     -0.00017313019390581717, -0.00016866250632484398, 
     -0.0001643655489809336, -0.0001602307322544464, -0.00015625, 
     -0.00015241579027587258, -0.000148720999405116, -0.00014515894904920887, 
     -0.0001417233560090703, -0.0001384083044982699, -0.00013520822065981613, 
     -0.0001321178491214163, -0.00012913223140495868, 
     -0.00012624668602449185, -0.0001234567901234568, 
     -0.00012075836251660427, -0.00011814744801512288, 
     -0.00011562030292519366, -0.00011317338162064282, 
     -0.00011080332409972299, -0.00010850694444444444, 
     -0.00010628122010840685, -0.00010412328196584757, 
     -0.00010203040506070809, -0.0001};
 
  TVIL_REAL H00y0xxtableCL3[] = 
     {0, 0.3333333333333333, 0.16666666666666666, 
     0.1111111111111111, 0.08333333333333333, 0.06666666666666667, 
     0.05555555555555555, 0.047619047619047616, 0.041666666666666664, 
     0.037037037037037035, 0.03333333333333333, 0.030303030303030304, 
     0.027777777777777776, 0.02564102564102564, 0.023809523809523808, 
     0.022222222222222223, 0.020833333333333332, 0.0196078431372549, 
     0.018518518518518517, 0.017543859649122806, 0.016666666666666666, 
     0.015873015873015872, 0.015151515151515152, 0.014492753623188406, 
     0.013888888888888888, 0.013333333333333334, 0.01282051282051282, 
     0.012345679012345678, 0.011904761904761904, 0.011494252873563218, 
     0.011111111111111112, 0.010752688172043012, 0.010416666666666666, 
     0.010101010101010102, 0.00980392156862745, 0.009523809523809525, 
     0.009259259259259259, 0.009009009009009009, 0.008771929824561403, 
     0.008547008547008548, 0.008333333333333333, 0.008130081300813009, 
     0.007936507936507936, 0.007751937984496124, 0.007575757575757576, 
     0.007407407407407408, 0.007246376811594203, 0.0070921985815602835, 
     0.006944444444444444, 0.006802721088435374, 0.006666666666666667, 
     0.006535947712418301, 0.00641025641025641, 0.006289308176100629, 
     0.006172839506172839, 0.006060606060606061, 0.005952380952380952, 
     0.005847953216374269, 0.005747126436781609, 0.005649717514124294, 
     0.005555555555555556, 0.00546448087431694, 0.005376344086021506, 
     0.005291005291005291, 0.005208333333333333, 0.005128205128205128, 
     0.005050505050505051, 0.004975124378109453, 0.004901960784313725, 
     0.004830917874396135, 0.004761904761904762, 0.004694835680751174, 
     0.004629629629629629, 0.0045662100456621, 0.0045045045045045045, 
     0.0044444444444444444, 0.0043859649122807015, 0.004329004329004329, 
     0.004273504273504274, 0.004219409282700422, 0.004166666666666667, 
     0.00411522633744856, 0.0040650406504065045, 0.004016064257028112, 
     0.003968253968253968, 0.00392156862745098, 0.003875968992248062, 
     0.0038314176245210726, 0.003787878787878788, 0.003745318352059925, 
     0.003703703703703704, 0.003663003663003663, 0.0036231884057971015, 
     0.0035842293906810036, 0.0035460992907801418, 0.0035087719298245615, 
     0.003472222222222222, 0.003436426116838488, 0.003401360544217687, 
     0.003367003367003367, 0.0033333333333333335};

  lnyox = TVIL_CLOG(y/x);
  lnyox2 = lnyox  * lnyox;
  lnyox3 = lnyox2  * lnyox;

  if (y < 0.5 * x) {     
    result = TVIL_H0000xx (x,qq);     
    rn = r = y/x;
    for (n=1; n < 55; n++) {
      result += rn * (H00y0xxtableA[n] + H00y0xxtableAL[n] * lnyox); 
      rn *= r;
    }
  }
  else if (x < 0.67 * y) {
    result = TVIL_H00000x (y, qq);
    rn = r = x/y;
    for (n=1; n < 100; n++) {
      result += rn * (H00y0xxtableC[n] +
                     -H00y0xxtableCL[n] * lnyox +
                     +H00y0xxtableCL2[n] * lnyox2 +
                     -H00y0xxtableCL3[n] * lnyox3);
      rn *= r;
    }
  }
  else {
    result = TVIL_H00x0xx (x,qq); 
    rn = r = y/x - 1.L;
    for (n=1; n < 100; n++) {
      result += rn * H00y0xxtableB[n];
      rn *= r;
    }
  }

  return result;
}

/* ****************************************************************** */

TVIL_COMPLEX TVIL_H00xx0y (TVIL_REAL x, TVIL_REAL y, TVIL_REAL qq)
{
  TVIL_REAL r, rn;
  TVIL_COMPLEX lnyox, lnyox2, lnyox3, result;
  int n;

  TVIL_REAL H00xx0ytableA[] = 
     {0, -1.2020569031595942, -1.9020950655332, -0.6590780520133783, 
     -0.24072188193900024, -0.09194118272731955, -0.03574665133693089, 
     -0.013699278434281092, -0.004895226250996698, -0.0013943258841968564, 
     -0.000052440035756440295, 0.000409953850718181, 0.0005215928186071972, 
     0.0005016369970803902, 0.000439779500366555, 0.0003713305619680307, 
     0.0003089308983357242, 0.0002560155440084477, 0.0002125046783741924, 
     0.0001771653277412577, 0.0001485504530944204, 0.00012533804925313197, 
     0.000106424927892812, 0.00009092665015419943, 0.0000781468086645809, 
     0.00006754078451781664, 0.00005868297050082294, 0.000051239859941321075, 
     0.00004494890184891798, 0.00003960223108800953, 0.000035034263323519, 
     0.00003111226050489524, 0.000027729146033883545, 
     0.000024798013480764602, 0.000022247909019335044, 
     0.000020020573864687474, 0.000018067913121144752, 
     0.000016350017049415744, 0.000014833604793540456, 
     0.000013490793086851083, 0.000012298116446707308, 
     0.000011235743141047028, 0.000010286844445372978, 9.437084605518065e-6, 
     8.674206370740123e-6, 7.98769259505172e-6, 7.368488694176056e-6, 
     6.808774024563702e-6, 6.301772776190484e-6, 5.841596925376669e-6, 
     5.4231153114037145e-6, 5.0418440899202555e-6, 4.6938547476060855e-6, 
     4.375696599973056e-6, 4.0843312785476815e-6, 3.8170771771706136e-6, 
     3.571562200321843e-6, 3.3456834558631687e-6, 3.137572772676789e-6, 
     2.945567122852877e-6, 2.768183182094708e-6, 2.604095395097872e-6, 
     2.452117015244726e-6, 2.3111836768840988e-6, 2.1803391291319585e-6, 
     2.0587228183877038e-6, 1.945559057331403e-6, 1.8401475575247073e-6, 
     1.741855137182835e-6, 1.650108443773668e-6, 1.5643875553028597e-6, 
     1.4842203430731615e-6, 1.4091774974134275e-6, 1.3388681291692794e-6, 
     1.2729358754293152e-6, 1.2110554442334998e-6, 1.1529295450563026e-6, 
     1.0982861571029456e-6, 1.0468760950066436e-6, 9.984708351368221e-7, 
     9.528605736386364e-7, 9.098524873102365e-7, 8.692691749329073e-7, 
     8.309472582790356e-7, 7.947361245624895e-7, 7.604967948021679e-7, 
     7.28100904068274e-7, 6.974297816070285e-7, 6.683736198803691e-7, 
     6.408307230976185e-7, 6.147068269818368e-7, 5.899144820131763e-7, 
     5.663724943344439e-7, 5.440054171440556e-7, 5.227430888987694e-7, 
     5.025202127889594e-7, 4.832759734202385e-7, 4.649536874817839e-7, 
     4.475004847376285e-7, 4.308670164959727e-7, 4.150071891556584e-7};
 
  TVIL_REAL H00xx0ytableAL[] = 
     {0, 3.2898681336964524, 1.8949340668482262, 0.7355116001210398, 
     0.292605922313002, 0.11864822991389368, 0.04776241381713364, 
     0.018200944819276088, 0.005789064504211461, 0.000658575240690018, 
     -0.0013447392656538715, -0.002007677796757923, -0.002108885576070141, 
     -0.0019890184252975285, -0.0017958670658744236, -0.0015917162425927445, 
     -0.001400998132089104, -0.0012316189569759883, -0.0010845249512847766, 
     -0.0009579675420502143, -0.0008493688264805399, -0.0007561022691022212, 
     -0.0006757879956779678, -0.0006063762483947133, -0.0005461432595488813, 
     -0.0004936551715838677, -0.0004477241321129699, -0.00040736635278833255, 
     -0.00037176551696914417, -0.0003402421722370902, -0.0003122286586977885, 
     -0.00028724877783382896, -0.00026490137282959847, 
     -0.0002448470760261917, -0.0002267975972167352, -0.0002105070425788419, 
     -0.00019576485551160705, -0.00018239005463639713, 
     -0.0001702265118467411, -0.0001591390669379994, -0.0001490103175962737, 
     -0.00013973795669619538, -0.00013123255489069785, 
     -0.00012341570693180648, -0.00011621847627509008, 
     -0.00010958008524704355, -0.00010344680814354523, 
     -0.00009777103265130871, -0.00009251046139180374, 
     -0.00008762743052034838, -0.00008308832644432496, 
     -0.00007886308505888895, -0.00007492476060104447, 
     -0.0000712491534217291, -0.0000678144877693937, -0.00006460113214815794, 
     -0.00006159135602178123, -0.0000587691176300309, 
     -0.00005611987850828032, -0.00005363044098415787, 
     -0.00005128880549436221, -0.00004908404503940755, 
     -0.00004700619449135579, -0.00004504615280370716, 
     -0.00004319559645296622, -0.00004144690267888357, 
     -0.000039793081290526167, -0.00003822771397560265, 
     -0.00003674490019513893, -0.000035339208868723215, 
     -0.00003400563516089278, -0.00003273956176944731, 
     -0.00003153672419337755, -0.000030393179525467495, 
     -0.00002930527837084057, -0.000028269639542874425, 
     -0.000027283127230708082, -0.00002634283036893198, 
     -0.000025446043973267884, -0.000024590252232989385, 
     -0.000023773113176042693, -0.000022992444743664964, 
     -0.000022246212130011545, -0.00002153251625865854, 
     -0.000020849583282141204, -0.000020195755003039884, 
     -0.00001956948012667159, -0.00001896930626442417, 
     -0.00001839387261616532, -0.000017841903267278025, 
     -0.00001731220104247283, -0.000016803641865126284, 
     -0.000016315169575238653, -0.000015845791164911804, 
     -0.000015394572393218064, -0.000014960633747139462, 
     -0.000014543146717872768, -0.000014141330364939031, 
     -0.000013754448143402098, -0.0000133818049715545, 
     -0.000013022744518746687};
 
  TVIL_REAL H00xx0ytableAL2[] = 
      {0, 0., -0.25, -0.16666666666666666, -0.10416666666666667, 
     -0.06666666666666667, -0.044444444444444446, -0.030952380952380953, 
     -0.022470238095238095, -0.016931216931216932, -0.013174603174603174, 
     -0.010533910533910534, -0.008615921115921115, -0.007181707181707182, 
     -0.00608161679590251, -0.005219040219040219, -0.004529758436008436, 
     -0.0039698863228275, -0.0035086554694397833, -0.003123983000143991, 
     -0.002799681398752606, -0.0025236578089298124, -0.0022867241274221292, 
     -0.002081792962126789, -0.001903322895801695, -0.001746928323318147, 
     -0.001609100155441417, -0.0014870026389447277, -0.001378323362295441, 
     -0.0012811610320441044, -0.0011939404758328114, -0.0011153475420696399, 
     -0.001044278723754498, -0.0009798018054567263, -0.0009211248512220922, 
     -0.0008675715663919069, -0.0008185615749325739, -0.000773594519937168, 
     -0.0007322371613776611, -0.0006941128411030171, -0.0006588928305505413, 
     -0.0006262891856344104, -0.0005960488156363078, -0.000567948535654055, 
     -0.0005417909203024175, -0.00051740081356199, -0.0004946223786142583, 
     -0.000473316594159984, -0.0004533591215645844, -0.00043463848131054475, 
     -0.0004170544884952282, -0.00040051690612511186, 
     -0.00038494428220803463, -0.0003702629425055381, -0.0003564061155656982, 
     -0.00034331317053696566, -0.00033092895144063654, 
     -0.0003192031941913904, -0.00030809001481062117, -0.0002975474590623801, 
     -0.0002875371052254442, -0.0002780237129524042, -0.0002689749122019958, 
     -0.0002603609270999068, -0.0002521543303149219, -0.0002443298241550385, 
     -0.00023686404511130975, -0.00022973538902135518, 
     -0.00022292385440264088, -0.00021641090182841126, 
     -0.00021017932749534845, -0.00020421314936893434, 
     -0.0001984975044961657, -0.00019301855625077275, 
     -0.00018776341042765348, -0.00018272003923439913, 
     -0.00017787721234153466, -0.00017322443425193983, 
     -0.00016875188733597667, -0.00016445037995391773, 
     -0.00016031129915287977, -0.00015632656748290358, 
     -0.00015248860352720043, -0.00014879028578584306, 
     -0.00014522491959112686, -0.00014178620676715624, 
     -0.00013846821777651873, -0.0001352653661237057, 
     -0.00013217238480866828, -0.00012918430464493917, 
     -0.0001262964342754409, -0.00012350434173571864, 
     -0.00012080383742913797, -0.00011819095839178491, 
     -0.00011566195373659048, -0.00011321327117673646, 
     -0.00011084154453783151, -0.00010854358217679635, 
     -0.00010631635623298322, -0.00010415699264386574, 
     -0.00010206276186376407};
 
  TVIL_REAL H00xx0ytableC[] = 
     {0, -0.03876335736944325, 0.20191880611871815, 
     0.13033185206855996, 0.06547502443932326, 0.03242005030391808, 
     0.016503660709830417, 0.008763343748911707, 0.0048924242268701, 
     0.0028851915927378185, 0.0017997604326185817, 0.0011849666350145132, 
     0.0008192808569375037, 0.0005908102182682419, 0.0004412247494701174, 
     0.0003390380872512295, 0.00026660052046851294, 0.00021361851428081246, 
     0.00017384302295225796, 0.00014332949219134948, 0.00011949561559959308, 
     0.00010059433615458268, 0.0000854089892620926, 0.00007307083175966039, 
     0.00006294611889215956, 0.000054563978095665623, 
     0.000047568981955649615, 0.00004168913694940354, 
     0.000036713772406235434, 0.000032477955680011264, 
     0.00002885131140363663, 0.000025729875158409063, 0.0000230300765060977, 
     0.00002068424057992245, 0.000018637188196512605, 0.00001684364078509043, 
     0.000015266221707819627, 0.000013873904099454792, 
     0.000012640796175861197, 0.000011545183809694343, 
     0.000010568770809843384, 9.6960722700571e-6, 8.913927260401552e-6, 
     8.211105179777983e-6, 7.577986071427276e-6, 7.0062996897169916e-6, 
     6.488911495394719e-6, 6.019646334366875e-6, 5.593142529022429e-6, 
     5.204730632138191e-6, 4.850332272397443e-6, 4.52637543962068e-6, 
     4.229723278128437e-6, 3.957614024127108e-6, 3.7076101722741866e-6, 
     3.4775553139458486e-6, 3.265537375286684e-6, 3.069857212292486e-6, 
     2.889001704866954e-6, 2.721620641234975e-6, 2.5665068054850203e-6, 
     2.422578779985194e-6, 2.288866055402305e-6, 2.1644961075503196e-6, 
     2.0486831550786914e-6, 1.9407183572917824e-6, 1.8399612489337744e-6, 
     1.7458322399955522e-6, 1.6578060346415848e-6, 1.5754058451370464e-6, 
     1.4981982949251282e-6, 1.4257889203690133e-6, 1.357818193627671e-6, 
     1.2939580000849472e-6, 1.2339085130299918e-6, 1.177395416167751e-6, 
     1.1241674312470366e-6, 1.0739941138176175e-6, 1.0266638850220328e-6, 
     9.819822715215544e-7, 9.397703292565295e-7, 8.9986322983932e-7, 
     8.621089910485515e-7, 8.263673351998386e-7, 7.925086611638215e-7, 
     7.604131175322531e-7, 7.299697659350678e-7, 7.010758248180136e-7, 
     6.736359851289457e-7, 6.47561790354486e-7, 6.227710742173059e-7, 
     5.991874501046894e-7, 5.767398469657838e-7, 5.553620870004354e-7, 
     5.349925009774065e-7, 5.155735774732106e-7, 4.970516427227046e-7, 
     4.793765681257213e-7, 4.6250150276630257e-7, 4.463826285775962e-7, 
     4.309789360306151e-7};
 
  TVIL_REAL H00xx0ytableCL[] = 
     {0, 3.144934066848226, 0.9943420334241131, 0.36179181948226935, 
     0.15225290591838056, 0.07009376768013592, 0.03445772244924768, 
     0.017928283747653727, 0.00985129377934957, 0.005716969509715456, 
     0.0035027120830794256, 0.002260991653330491, 0.0015313018742280575, 
     0.001082075231592847, 0.0007928676671711775, 0.0005988215673941918, 
     0.00046373110157571024, 0.0003666218470057749, 0.00029488120134782625, 
     0.00024064287283112365, 0.00019882795428534484, 0.00016605092797984599, 
     0.00013998936344746616, 0.0001190091739817059, 0.00010193481534577077, 
     0.00008790412392270996, 0.00007627403869290985, 0.00006655779173532222, 
     0.00005838209078577434, 0.000051457325039004046, 
     0.000045556451404781605, 0.0000404997878589639, 0.000036143902133849905, 
     0.000032373387350508056, 0.0000290947033133241, 0.000026231515740659272, 
     0.000023721134941613834, 0.00002151177036015907, 0.00001956039665352605, 
     0.00001783108239514566, 0.00001629367175253646, 0.000014922737626211663, 
     0.000013696745114782997, 0.000012597379077678735, 
     0.000011609000569355103, 0.000010718205108743735, 9.913461892222339e-6, 
     9.184817702685105e-6, 8.523652802209942e-6, 7.922478804169894e-6, 
     7.374770608707567e-6, 6.874826104934401e-6, 6.417648606328921e-6, 
     5.998847976321858e-6, 5.614557181814412e-6, 5.261361630837699e-6, 
     4.936239142781268e-6, 4.636508793142304e-6, 4.3597871907343704e-6, 
     4.103951000083657e-6, 3.867104728010696e-6, 3.6475529610320957e-6, 
     3.443776376964063e-6, 3.2544109660601833e-6, 3.0782299889841226e-6, 
     2.9141282747235043e-6, 2.7611085242365536e-6, 2.6182693376210687e-6, 
     2.484794725858415e-6, 2.3599449042861513e-6, 2.243048195162612e-6, 
     2.1334938920360784e-6, 2.030725959958606e-6, 1.934237463574562e-6, 
     1.8435656303267022e-6, 1.7582874689181636e-6, 1.6780158741258757e-6, 
     1.6023961583922437e-6, 1.5311029585865525e-6, 1.463837473140244e-6, 
     1.4003249905999382e-6, 1.3403126756580159e-6, 1.2835675830375917e-6, 
     1.2298748733313564e-6, 1.1790362081100399e-6, 1.1308683044000569e-6, 
     1.0852016310438489e-6, 1.0418792315532337e-6, 1.0007556598905872e-6, 
     9.616960172028802e-7, 9.245750789218428e-7, 8.892765028574095e-7, 
     8.556921099746559e-7, 8.237212304768343e-7, 7.932701086361201e-7, 
     7.642513605341322e-7, 7.365834795089455e-7, 7.10190384665163e-7, 
     6.850010082981566e-7, 6.609489185210033e-7, 6.37971973770057e-7};
 
  TVIL_REAL H00xx0ytableCL2[] = 
     {0, -0.5, -0.09375, 0.012152777777777778, 
     0.022623697916666668, 0.016686197916666666, 0.01053466796875, 
     0.006396027649340987, 0.0038879439944312687, 0.00241006484527857, 
     0.0015376332449534582, 0.001014024525134094, 0.0006919401351536278, 
     0.0004879915148496406, 0.0003547993163445107, 0.00026508528699138164, 
     0.00020284109383075707, 0.0001584548384385511, 0.000126008282503208, 
     0.00010176061709387379, 0.00008328432311517708, 0.00006896331766177152, 
     0.00005769533812357489, 0.000048711474171063574, 0.00004146420098306361, 
     0.000035556301926585536, 0.000030694726184956055, 
     0.00002665998430441769, 0.000023285435115798682, 
     0.000020443001232627516, 0.000018033145893691996, 
     0.000015977727478493996, 0.000014214831180011887, 
     0.00001269498098801689, 0.000011378329572547689, 
     0.000010232550388479065, 9.231240328898502e-6, 8.35269782977163e-6, 
     7.578979996182494e-6, 6.895169114649263e-6, 6.28879772062885e-6, 
     5.749394744469061e-6, 5.268124847028049e-6, 4.837500010582539e-6, 
     4.451147542957636e-6, 4.1036224148319475e-6, 3.790254652609979e-6, 
     3.5070246132341943e-6, 3.250460558573199e-6, 3.017554158905416e-6, 
     2.805690484061871e-6, 2.6125897574681295e-6, 2.436258704469176e-6, 
     2.2749497603365377e-6, 2.127126743894005e-6, 1.9914358712695863e-6, 
     1.8666811971409803e-6, 1.7518037403440569e-6, 1.6458636863017116e-6, 
     1.548025167661247e-6, 1.4575432124086092e-6, 1.3737525199100606e-6, 
     1.296057783211766e-6, 1.2239253231663851e-6, 1.156875838647811e-6, 
     1.0944781089166271e-6, 1.0363435104240749e-6, 9.821212320388977e-7, 
     9.314940906872957e-7, 8.841748643842982e-7, 8.399030721472793e-7, 
     7.984421407575694e-7, 7.595769071298101e-7, 7.231114124503731e-7, 
     6.888669504922129e-7, 6.566803377974987e-7, 6.264023779003191e-7, 
     5.978964955706405e-7, 5.710375203057071e-7, 5.45710601065862e-7, 
     5.218102366226985e-7, 4.992394079205356e-7, 4.779088005991331e-7, 
     4.5773610732958897e-7, 4.3864540091278453e-7, 4.205665702110501e-7, 
     4.0343481195460856e-7, 3.871901723065001e-7, 3.7177713280145536e-7, 
     3.5714423591110955e-7, 3.4324374604326e-7, 3.300313422678058e-7, 
     3.174658394861609e-7, 3.0550893513249775e-7, 2.941249788211698e-7, 
     2.832807626410865e-7, 2.729453300498501e-7, 2.63089801542541e-7, 
     2.536872154659988e-7, 2.4471238252258887e-7, 2.361417526606321e-7};
 
  TVIL_REAL H00xx0ytableCL3[] = 
     {0, 0.08333333333333333, 0.052083333333333336, 
     0.022569444444444444, 0.009440104166666666, 0.0039713541666666664, 
     0.0016954210069444445, 0.0007353283110119048, 0.0003236134847005208, 
     0.00014425207067418982, 0.00006500879923502605, 0.00002957112861402107, 
     0.000013558401001824273, 6.2588697824722685e-6, 2.906169919740586e-6, 
     1.3562747173839144e-6, 6.357683256889383e-7, 2.9918851842190705e-7, 
     1.4128427547230213e-7, 6.692432195944875e-8, 3.178909840547324e-8, 
     1.513767673468609e-8, 7.2248028435283095e-9, 3.455341108246353e-9, 
     1.6556844290644475e-9, 7.947285614780716e-10, 3.8208104771232687e-10, 
     1.8396495095449633e-10, 8.869738756298172e-11, 
     4.2819428598316933e-11, 2.0696057184765242e-11, 
     1.0014221225429647e-11, 4.850638407761551e-12, 2.351824682961738e-12, 
     1.1413266844781415e-12, 5.543586753421553e-13, 2.69479911630541e-13, 
     1.3109833538926155e-13, 6.382418959775194e-14, 3.109383595796399e-14, 
     1.5158245029528123e-14, 7.394265868067544e-15, 3.609105959415151e-15, 
     1.7625866313425836e-15, 8.612639221333813e-16, 4.210623619318933e-16, 
     2.059544161623435e-16, 1.0078620365391385e-16, 
     4.9343245538895585e-17, 2.416812026394892e-17, 
     1.1842378929334987e-17, 5.805087710458331e-18, 2.846725704167067e-18, 
     1.396506949214033e-18, 6.853228547068867e-19, 3.3643121958338077e-19, 
     1.6521175961683876e-19, 8.115665384686816e-20, 3.987870059716798e-20, 
     1.960139520877748e-20, 9.637352644315595e-21, 4.739681628351932e-21, 
     2.331617575237644e-21, 1.147303886228047e-21, 5.646886315028669e-22, 
     2.7800055704756524e-22, 1.3689421369766471e-22, 
     6.742550823914828e-23, 3.321697832369805e-23, 1.6367786420372953e-23, 
     8.066980450040956e-24, 3.976680503541316e-24, 1.96072441494051e-24, 
     9.669325881898405e-25, 4.769329657963403e-25, 2.352869297928612e-25, 
     1.1609552456884598e-25, 5.729389524176815e-26, 
     2.8279679061641975e-26, 1.3960854220304263e-26, 6.89317177127523e-27, 
     3.4040354426050524e-27, 1.6812614076281052e-27, 
     8.305026230452084e-28, 4.103078435282875e-28, 2.0274034621397737e-28, 
     1.0019145016388416e-28, 4.9519912149965735e-29, 
     2.4478592937767155e-29, 1.2101776283839942e-29, 
     5.9836560514541935e-30, 2.9589507946751505e-30, 
     1.4633941430186886e-30, 7.2382936106300725e-31, 
     3.5806452435563656e-31, 1.7714771204963072e-31, 
     8.765121169122353e-32, 4.3373795476069377e-32, 
     2.1465602863156783e-32, 1.0624389295905882e-32, 
     5.2590727014734124e-33};
 
  TVIL_REAL H00xx0ytableB[] = 
     {0, -3.7638985318796165, 1.3443793388193528, 
     -0.7099590303821657, 0.4445328486578516, -0.30661560374033897, 
     0.225191383127224, -0.17286248662678608, 0.13712696465115531, 
     -0.11158171053002897, 0.09265701402694604, -0.07822882246428087, 
     0.06696619808223199, -0.0579992987890677, 0.05073938744816935, 
     -0.044776031867448475, 0.039815766513902576, -0.03564410495726714, 
     0.03210124611851566, -0.02906609705541509, 0.02644550337039564, 
     -0.024166829624553025, 0.022172747036227992, -0.02041750706044057, 
     0.018864234721167786, -0.017482934135792466, 0.016248999405295478, 
     -0.015142089357022616, 0.014145267769671547, -0.013244339698268872, 
     0.012427334302131823, -0.011684098279601806, 0.011005973628752991, 
     -0.010385540285983479, 0.009816409106270524, -0.009293054218288821, 
     0.008810676407830731, -0.008365091124769488, 0.007952636160719445, 
     -0.007570095139313923, 0.007214633793054649, -0.006883746637788161, 
     0.006575212147169082, -0.006287054910892992, 0.006017513558489863, 
     -0.005765013464718285, 0.005528143437795993, -0.005305635738921479, 
     0.005096348899186439, -0.0048992528944617275, 0.004713416315084034, 
     -0.004537995228979406, 0.004372223487186121, -0.004215404261888981, 
     0.004066902640859043, -0.003926139130033905, 0.0037925839390041795, 
     -0.003665751943291246, 0.0035451982332287816, -0.00343051417257294, 
     0.0033213239011284017, -0.0032172812250660762, 0.0031180668465283887, 
     -0.003023385890818806, 0.0029329656951570466, -0.002846553827817364, 
     0.0027639163105917983, -0.0026848360210469466, 0.0026091112540656343, 
     -0.002536554424761951, 0.0024669908970946022, -0.002400257924433703, 
     0.002336203690005647, -0.0022746864365876494, 0.002215573676080315, 
     -0.002158741470680222, 0.002104073778328037, -0.0020514618559406804, 
     0.0020008037146649576, -0.0019520036220291752, 0.0019049716464304394, 
     -0.00185962323988904, 0.001815878855436311, -0.001773663595886248, 
     0.0017329068910804841, -0.0016935422009966017, 0.0016555067423760655, 
     -0.0016187412367644812, 0.0015831896780670997, -0.00154879911790963, 
     0.0015155194672612732, -0.0014833033129258231, 0.0014521057476398114, 
     -0.0014218842126358109, 0.0013925983516357714, -0.001364209875335033, 
     0.0013366824355236886, -0.001309981508069313, 0.001284074284054722, 
     -0.0012589295684271693, 0.0012345176855720135};
 
  TVIL_REAL H00xx0ytableD[] = 
     {0, -2.307877162034201, 0.6672791419531447, 
     -0.31246372251570226, 0.18074954407313984, -0.11777767924446753, 
     0.08284001211717511, -0.06145151095848243, 0.04740787520488685, 
     -0.03769054203953798, 0.03068714255733636, -0.02547252806481687, 
     0.021484902745292588, -0.018367036434150746, 0.015882934632334963, 
     -0.013871593879388043, 0.012220108296376538, -0.0108473948746683, 
     0.009694001415375, -0.008715530460870603, 0.007878283624816566, 
     -0.007156308828193388, 0.006529356676731321, -0.005981439500739899, 
     0.005499798121260266, -0.005074149593074805, 0.004696131850149933, 
     -0.0043588884638621015, 0.004056754512038266, -0.003785016361598105, 
     0.003539726131145227, -0.0033175570538864665, 0.0031156897491577775, 
     -0.002931722075932334, 0.002763597139575578, -0.0026095453899614798, 
     0.002468037743887603, -0.0023377473959664435, 0.002217518524673305, 
     -0.0021063405062394804, 0.0020033265554467885, -0.001907695945363148, 
     0.001818759136543747, -0.001735905283912533, 0.0016585916964625895, 
     -0.00158633490846913, 0.0015187030865938336, -0.0014553095491906412, 
     0.0013958072154041318, -0.0013398838346368326, 0.0012872578734491984, 
     -0.0012376749583265052, 0.0011909047900673796, -0.0011467384596478322, 
     0.0011049861069395359, -0.0010654748731194956, 0.0010280471054006529, 
     -0.0009925587791565897, 0.0009588781078608087, -0.0009268843157138425, 
     0.0008964665515515861, -0.0008675229257462465, 0.0008399596544326427, 
     -0.0008136902976029641, 0.000788635079482273, -0.0007647202811820866, 
     0.0007418776969770977, -0.0007200441466989997, 0.0006991610377232757, 
     -0.0006791739708660152, 0.0006600323852301969, -0.0006416892376626372, 
     0.0006241007130191522, -0.0006072259618991474, 0.0005910268629125304, 
     -0.0005754678068905294, 0.000560515500755286, -0.0005461387890273914, 
     0.0005323084911813069, -0.0005189972532604482, 0.0005061794123405921, 
     -0.0004938308725855033, 0.00048192899177518776, -0.00047045247730738654, 
     0.00045938129077896907, -0.00044869656034756224, 0.00043838050015664596, 
     -0.0004284163361807965, 0.00041878823791294045, -0.0004094812553734069, 
     0.00040048126097211176, -0.00039177489580114036, 0.0003833495199759793, 
     -0.00037519316668026276, 0.0003672944996016542, -0.00035964277347581933, 
     0.0003522277974817531, -0.00034503990125534005, 0.00033806990330925555, 
     -0.0003313090816664153, 0.00032474914653138836};

  lnyox = TVIL_CLOG(y/x);
  lnyox2 = lnyox  * lnyox;
  lnyox3 = lnyox2  * lnyox;

  if (y < 0.5 * x) {     
    result = TVIL_H00xx00 (x,qq);     
    rn = r = y/x;
    for (n=1; n < 100; n++) {
      result += rn * (H00xx0ytableA[n] 
                    + H00xx0ytableAL[n] * lnyox 
                    + H00xx0ytableAL2[n] * lnyox2); 
      rn *= r;
    }
  }
  else if (y < 1.5 * x) {
    result = TVIL_H00xx0x (x, qq);
    rn = r = (y/x - 1.L);
    for (n=1; n < 100; n++) {
      result += rn * H00xx0ytableD[n]; 
      rn *= r;
    }
  }
  else if (y < 6.47 * x) {
    result = -7.2191002928335420630003L - 6.L * Zeta3 * TVIL_Ap(x,qq);
    rn = r = (y/(4.L* x) - 1.L);
    for (n=1; n < 100; n++) {
      result += rn * H00xx0ytableB[n]; 
      rn *= r;
    }
  }
  else {
    result = TVIL_H00000x (y, qq);
    rn = r = 4.L*x/y;
    for (n=1; n < 100; n++) {
      result += rn * (H00xx0ytableC[n] +
                     -H00xx0ytableCL[n] * lnyox +
                     +H00xx0ytableCL2[n] * lnyox2 +
                     -H00xx0ytableCL3[n] * lnyox3);
      rn *= r;
    }
  }

  return result;
}

/* ****************************************************************** */

TVIL_COMPLEX TVIL_H00xy0x (TVIL_REAL x, TVIL_REAL y, TVIL_REAL qq)
{
  TVIL_REAL r, rn;
  TVIL_COMPLEX lnyox, lnyox2, lnyox3, result;
  int n;

  TVIL_REAL H00xy0xtableA[] = 
     {0, -4.546838320513397, -0.2193748991909774, 
     -0.03890059082876296, -0.011885355417493804, -0.00481878965352267, 
     -0.0023172687648259638, -0.0012498339706712436, -0.0007324770417662663, 
     -0.00045725645147776106, -0.00030000156260123756, 
     -0.00020490431533110032, -0.00014467597713596625, 
     -0.00010503834863934136, -0.00007809246333001507, 
     -0.00005925925962280143, -0.00004577636725988654, 
     -0.000035919110178528335, -0.000028577960679732062, 
     -0.00002302008118482135, -0.000018750000000122835, 
     -0.000015425671402363728, -0.000012806502288104094, 
     -0.000010720373354866641, -9.042245370377566e-6, 
     -7.680000000009901e-6, -6.564896187111202e-6, -5.6450292694743e-6, 
     -4.88077884214233e-6, -4.241595631726547e-6, -3.703703703700212e-6, 
     -3.248437230984891e-6, -2.86102294921875e-6, -2.5296794643192566e-6, 
     -2.244944385249026e-6, -1.9991670137664563e-6, -1.786122542313251e-6, 
     -1.6007162672027064e-6, -1.4387550740235788e-6, 
     -1.2967696172150323e-6, -1.1718750000250244e-6, 
     -1.0616609118913045e-6, -9.641044626390594e-7, -8.775006208466829e-7, 
     -8.004063929978322e-7, -7.315957933212713e-7, -6.700233346895734e-7, 
     -6.147942867712475e-7, -5.651403356676798e-7, -5.20399576681152e-7, 
     -4.799999999971494e-7, -4.4344580449329207e-7, 
     -4.1030601169445013e-7, -3.802049583501166e-7, 
     -3.5281432934908263e-7, -3.2784645859207373e-7, 
     -3.050486776373651e-7, -2.841985331375474e-7, -2.650997269759703e-7, 
     -2.4757866007307783e-7, -2.3148148146912018e-7, 
     -2.1667156224758344e-7, -2.0302732695043346e-7, 
     -1.904403876817895e-7, -1.7881393432617188e-7, 
     -1.6806134239821802e-7, -1.5810496650781047e-7, 
     -1.488750923650528e-7, -1.4030902409367663e-7, 
     -1.3235028832603302e-7, -1.2494793835693407e-7, 
     -1.1805594443692335e-7, -1.1163265889457819e-7, 
     -1.0564034581828086e-7, -1.0004476670710805e-7, 
     -9.481481481887766e-8, -8.99221921507598e-8, -8.534112311275877e-8, 
     -8.104810107767424e-8, -7.702166269357713e-8, -7.32421874954925e-8, 
     -6.969171938153718e-8, -6.635380699060445e-8, -6.321336146891676e-8, 
     -6.025652891927802e-8, -5.747057627014662e-8, -5.4843788808989213e-8, 
     -5.2365378172547494e-8, -5.002539955889507e-8, -4.781467719677801e-8, 
     -4.572473707303848e-8, -4.374774635607093e-8, -4.187645841202681e-8, 
     -4.0104163337439935e-8, -3.842464292580505e-8, -3.683212990157614e-8, 
     -3.5321270991373055e-8, -3.388709324952943e-8, 
     -3.2524973554715064e-8, -3.1230610673949055e-8, 
     -3.0000000005026006e-8};
 
  TVIL_REAL H00xy0xtableAL[] = 
     {0, 2.355065933151774, 0.26086629990922017, 
     0.07483679253207265, 0.031326007097467246, 0.016009218376386525, 
     0.009260527899834614, 0.0058310945117247825, 0.003906280608974966, 
     0.00274348939111857, 0.0020000009075317737, 0.0015026297665059196, 
     0.001157407438117486, 0.0009103322771372435, 0.0007288629749004627, 
     0.0005925925928180059, 0.0004882812500452693, 0.0004070832485335435, 
     0.00034293552812261763, 0.00029158769499967874, 0.00025000000000009737, 
     0.00021595939963288402, 0.00018782870022539255, 0.00016437905810801068, 
     0.00014467592592593004, 0.00012799999999998923, 0.00011379153390987662, 
     0.00010161052685058597, 0.00009110787172011969, 0.00008200418221329664, 
     0.00007407407407406669, 0.00006713436944043305, 0.00006103515625000694, 
     0.0000556529482149265, 0.000050885406065549754, 0.0000466472303206969, 
     0.00004286694101508781, 0.00003948433459025763, 0.00003644846187491474, 
     0.000033716010047368794, 0.000031250000000010436, 
     0.00002901873159124463, 0.000026994924954115707, 
     0.000025155017797179324, 0.000023478587528172334, 
     0.000021947873799721407, 0.000020547382263501335, 
     0.000019263554318416087, 0.000018084490740741255, 
     0.000016999719504640076, 0.000015999999999995185, 
     0.000015077157352751114, 0.000014223941738736312, 
     0.000013433908528516503, 0.000012701315856324114, 
     0.000012021036814423985, 0.000011388483965014962, 
     0.00001079954425923374, 0.000010250522776669019, 9.738093962874395e-6, 
     9.259259259257469e-6, 8.811310197774647e-6, 8.391796180055866e-6, 
     7.998496282701079e-6, 7.62939453125347e-6, 7.282658170234407e-6, 
     6.956618526864078e-6, 6.649754125342966e-6, 6.360675758192852e-6, 
     6.088113263264666e-6, 5.830903790086245e-6, 5.587981369674366e-6, 
     5.358367626887711e-6, 5.1411634967128694e-6, 4.9355418237813364e-6, 
     4.740740740746124e-6, 4.556057734368679e-6, 4.380844320125976e-6, 
     4.21450125591763e-6, 4.056474234293006e-6, 3.906250000004774e-6, 
     3.763352846318513e-6, 3.627341448906446e-6, 3.4978060011875423e-6, 
     3.3743656192644633e-6, 3.2566659881952398e-6, 3.1443772246474155e-6, 
     3.0371919338273434e-6, 2.934823441019807e-6, 2.837004180328284e-6, 
     2.743484224965176e-6, 2.6540299454223815e-6, 2.5684227829420037e-6, 
     2.4864581274278152e-6, 2.4079442898054804e-6, 2.3327015599948486e-6, 
     2.260561342590922e-6, 2.1913653630613317e-6, 2.1249649380800095e-6, 
     2.061220304258593e-6, 1.9999999999985307e-6};
 
  TVIL_REAL H00xy0xtableAL2[] = 
     {0, -0.5, -0.125, -0.05555555555555555, -0.03125, -0.02, 
     -0.013888888888888888, -0.01020408163265306, -0.0078125, 
     -0.006172839506172839, -0.005, -0.004132231404958678, 
     -0.003472222222222222, -0.0029585798816568047, -0.002551020408163265, 
     -0.0022222222222222222, -0.001953125, -0.0017301038062283738, 
     -0.0015432098765432098, -0.0013850415512465374, -0.00125, 
     -0.0011337868480725624, -0.0010330578512396695, -0.000945179584120983, 
     -0.0008680555555555555, -0.0008, -0.0007396449704142012, 
     -0.0006858710562414266, -0.0006377551020408163, -0.0005945303210463733, 
     -0.0005555555555555556, -0.0005202913631633715, -0.00048828125, 
     -0.0004591368227731864, -0.00043252595155709344, 
     -0.00040816326530612246, -0.00038580246913580245, 
     -0.0003652300949598247, -0.00034626038781163435, -0.0003287310979618672, 
     -0.0003125, -0.000297441998810232, -0.0002834467120181406, 
     -0.00027041644131963225, -0.00025826446280991736, 
     -0.0002469135802469136, -0.00023629489603024575, 
     -0.00022634676324128565, -0.00021701388888888888, 
     -0.00020824656393169514, -0.0002, -0.00019223375624759708, 
     -0.0001849112426035503, -0.000177999288002848, -0.00017146776406035664, 
     -0.00016528925619834712, -0.00015943877551020407, 
     -0.0001538935056940597, -0.00014863258026159333, 
     -0.00014363688595231256, -0.0001388888888888889, -0.0001343724805159903, 
     -0.00013007284079084288, -0.00012597631645250694, -0.0001220703125, 
     -0.00011834319526627219, -0.0001147842056932966, 
     -0.00011138338159946537, -0.00010813148788927336, 
     -0.00010501995379122033, -0.00010204081632653062, 
     -0.00009918666931164451, -0.00009645061728395061, 
     -0.00009382623381497467, -0.00009130752373995617, 
     -0.00008888888888888889, -0.00008656509695290859, 
     -0.00008433125316242199, -0.0000821827744904668, -0.0000801153661272232, 
     -0.000078125, -0.00007620789513793629, -0.000074360499702558, 
     -0.00007257947452460444, -0.00007086167800453515, 
     -0.00006920415224913495, -0.00006760411032990806, 
     -0.00006605892456070815, -0.00006456611570247934, 
     -0.00006312334301224592, -0.0000617283950617284, 
     -0.000060379181258302136, -0.00005907372400756144, 
     -0.00005781015146259683, -0.00005658669081032141, 
     -0.000055401662049861494, -0.00005425347222222222, 
     -0.000053140610054203425, -0.000052061640982923784, 
     -0.000051015202530354045, -0.00005};
 
  TVIL_REAL H00xy0xtableC[] = 
     {0, -3.0684295518521365, -0.03934936685846757, 
     0.0027452673348180336, 0.0026663611838204124, 0.0016624714709731515, 
     0.0010448843218034494, 0.0006836312481927683, 0.0004651665461721946, 
     0.0003275760242312741, 0.00023760271767920294, 0.00017678073864044652, 
     0.00013445149846287534, 0.00010423181548953029, 0.00008216847941961537, 
     0.00006573811464525412, 0.000053285749825298324, 
     0.000043699203320406094, 0.00003621450016005926, 
     0.000030296371185835513, 0.000025563011019939046, 
     0.000021737613032881178, 0.000018616496166922553, 
     0.000016047740536688276, 0.000013916614545127804, 
     0.000012135473435649808, 0.000010636652666800035, 9.367399170421059e-6, 
     8.286209850343774e-6, 7.360155190420372e-6, 6.562901285100145e-6, 
     5.873232941288626e-6, 5.273940270360367e-6, 4.750971716113718e-6, 
     4.292784289768377e-6, 3.889841113584476e-6, 3.5342199530734736e-6, 
     3.2193060556085456e-6, 2.9395495217050325e-6, 2.6902724335060735e-6, 
     2.4675146131931426e-6, 2.2679095691042166e-6, 2.0885841792423786e-6, 
     1.9270771506870793e-6, 1.7812724142191252e-6, 1.6493444629800175e-6, 
     1.529713292066799e-6, 1.4210070934505748e-6, 1.3220312447619947e-6, 
     1.231742428813428e-6, 1.1492269536610832e-6, 1.0736825258326031e-6, 
     1.0044028735540913e-6, 9.40764731106135e-7, 8.82216786443199e-7, 
     8.282702669917065e-7, 7.784908969964699e-7, 7.324920069255585e-7, 
     6.899286136101584e-7, 6.504923208145516e-7, 6.139069152332761e-7, 
     5.799245536246085e-7, 5.483224538023738e-7, 5.189000162287482e-7, 
     4.914763145423606e-7, 4.65887902969164e-7, 4.419868965597952e-7, 
     4.1963928686777993e-7, 3.987234612633331e-7, 3.791288987586376e-7, 
     3.60755019157603e-7, 3.4351016566300403e-7, 3.273107038802863e-7, 
     3.1208022253540025e-7, 2.977488232438686e-7, 2.8425248838781175e-7, 
     2.715325176247485e-7, 2.595350248064243e-7, 2.4821048816076875e-7, 
     2.3751334751296954e-7, 2.2740164311559534e-7, 2.1783669134209788e-7, 
     2.087827930890624e-7, 2.002069712439619e-7, 1.9207873401842363e-7, 
     1.8436986133188646e-7, 1.7705421176528743e-7, 1.7010754789606824e-7, 
     1.635073780803037e-7, 1.5723281297021477e-7, 1.5126443525007087e-7, 
     1.4558418124422785e-7, 1.401752332009648e-7, 1.3502192118761317e-7, 
     1.3010963364855577e-7, 1.254247357800386e-7, 1.2095449496612502e-7, 
     1.1668701260004397e-7, 1.1261116168593787e-7, 1.0871652967874021e-7, 
     1.0499336607558068e-7};
 
  TVIL_REAL H00xy0xtableCL[] = 
     {0, -2.394934066848226, -0.10405425835602827, 
     -0.02038057083711583, -0.00823623333404255, -0.0042382297397251766, 
     -0.00241978455225756, -0.0014800237231200196, -0.0009559911851987613, 
     -0.0006459710911219567, -0.00045320356972232054, -0.0003281442738433651, 
     -0.00024401176837735074, -0.00018562235845216433, 
     -0.0001439963432712069, -0.00011362117687562358, -0.0000909994979494072, 
     -0.00007384698110110528, -0.00006063265690219214, 
     -0.000050306656076863615, -0.00004213410662235404, 
     -0.00003559105335306274, -0.000030297651127895703, 
     -0.000025974338079732784, -0.000022412509182257086, 
     -0.000019454535076529343, -0.000016979921354511248, 
     -0.000014895574698290152, -0.000013128860928253444, 
     -0.000011622589817382763, -0.00001033134826410334, 
     -9.218789326404792e-6, -8.255607070545193e-6, -7.4180090299777195e-6, 
     -6.686553523264335e-6, -6.045257135267367e-6, -5.480904096209069e-6, 
     -4.982507856696823e-6, -4.540888333338078e-6, -4.148337745812778e-6, 
     -3.798354802103878e-6, -3.4854319794355704e-6, -3.204884323005959e-6, 
     -2.9527109116095626e-6, -2.725482178403134e-6, -2.520247810864595e-6, 
     -2.334461118614815e-6, -2.1659166466922698e-6, 
     -2.0126984945854345e-6, -1.8731373287910155e-6, 
     -1.7457744865003778e-6, -1.629331888186884e-6, -1.522686728293809e-6, 
     -1.4248501116522563e-6, -1.3349489606128104e-6, 
     -1.2522106432307017e-6, -1.1759498731506262e-6, 
     -1.1055575124412591e-6, -1.0404909736661489e-6, 
     -9.802659701606434e-7, -9.244494063226077e-7, -8.726532346853595e-7, 
     -8.245291351740706e-7, -7.797638954773042e-7, -7.380753908665122e-7, 
     -6.992090778439269e-7, -6.629349293140031e-7, -6.290447500538576e-7, 
     -5.973498205056745e-7, -5.67678824653627e-7, -5.398760242429958e-7, 
     -5.137996470643357e-7, -4.893204616358449e-7, -4.663205145159033e-7, 
     -4.446920097828412e-7, -4.243363130273403e-7, -4.0516306459457187e-7, 
     -3.870893888545624e-7, -3.700391880254032e-7, -3.5394251057064483e-7, 
     -3.38734985477763e-7, -3.2435731483095724e-7, -3.1075481804555127e-7, 
     -2.978770219554501e-7, -2.856772916584469e-7, -2.741124976426751e-7, 
     -2.631427152547247e-7, -2.527309530373519e-7, -2.428429068720702e-7, 
     -2.334467372175024e-7, -2.245128670452766e-7, -2.160137983474954e-7, 
     -2.079239453285742e-7, -2.0021948260394704e-7, 
     -1.9287820691259514e-7, -1.8587941101282244e-7, 
     -1.7920376857401027e-7, -1.728332290036571e-7, -1.667509212609555e-7, 
     -1.6094106580729155e-7, -1.5538889393195403e-7};
 
  TVIL_REAL H00xy0xtableCL2[] = 
     {0, -0.5, 0.03125, 0.019965277777777776, 0.00830078125, 
     0.003583984375, 0.0017164442274305556, 0.000916087870695153, 
     0.0005364962986537388, 0.0003381014290730579, 0.00022548073814028784, 
     0.00015714938178019278, 0.00011343588248905853, 0.00008425718979867695, 
     0.00006409198436539611, 0.00004974645355044191, 0.000039287210285815066, 
     0.00003149845332000182, 0.000025590500768486506, 
     0.000021035948122031634, 0.000017473889517083252, 
     0.000014652004026625768, 0.000012390490253662217, 
     0.000010559015552035567, 9.061638638912192e-6, 7.826740740537667e-6, 
     6.800174247631372e-6, 5.940520640520894e-6, 5.215756879954644e-6, 
     4.600878267883116e-6, 4.076180946611491e-6, 3.626005845588901e-6, 
     3.2378097089735435e-6, 2.9014708106649042e-6, 2.608764986185085e-6, 
     2.3529665830648943e-6, 2.128541944706447e-6, 1.9309120779790863e-6, 
     1.7562674999866405e-6, 1.60142276335127e-6, 1.463701388604092e-6, 
     1.3408442696353659e-6, 1.2309363251583118e-6, 1.1323474263643515e-6, 
     1.0436845643471742e-6, 9.637529191371217e-7, 8.915240183296354e-7, 
     8.26109572467777e-7, 7.667398791719463e-7, 7.127459222510928e-7, 
     6.635444730991011e-7, 6.18625642444387e-7, 5.775424405476896e-7, 
     5.399019903928098e-7, 5.053581066792402e-7, 4.7360500758808276e-7, 
     4.4437196946411847e-7, 4.174187691198578e-7, 3.9253178625597966e-7, 
     3.6952066092596557e-7, 3.48215419154248e-7, 3.28463994608651e-7, 
     3.101300863054246e-7, 2.9309130222156756e-7, 2.772375468259279e-7, 
     2.624696172526985e-7, 2.486979783954437e-7, 2.3584169181053487e-7, 
     2.2382747715759585e-7, 2.125888881097719e-7, 2.0206558735036611e-7, 
     1.922027075255821e-7, 1.829502869198098e-7, 1.7426277022057587e-7, 
     1.6609856609449423e-7, 1.5841965444400342e-7, 1.5119123719090832e-7, 
     1.4438142726446226e-7, 1.3796097118189416e-7, 1.3190300121696534e-7, 
     1.2618281367321456e-7, 1.2077767022626814e-7, 1.1566661968504404e-7, 
     1.1083033785416523e-7, 1.0625098346723781e-7, 1.0191206840942535e-7, 
     9.779834066352738e-8, 9.389567860124909e-8, 9.019099540451883e-8, 
     8.667215254395906e-8, 8.332788136582793e-8, 8.01477119473826e-8, 
     7.712190847576977e-8, 7.424141048902631e-8, 7.149777939111072e-8, 
     6.888314971741178e-8, 6.639018468403126e-8, 6.401203560429729e-8, 
     6.174230480027881e-8, 5.9575011676274536e-8, 5.750456165597528e-8};
 
  TVIL_REAL H00xy0xtableCL3[] = 
     {0, -0.125, -0.015625, -0.0026041666666666665, -0.00048828125, 
     -0.00009765625, -0.000020345052083333332, -4.359654017857143e-6, 
     -9.5367431640625e-7, -2.1192762586805554e-7, -4.76837158203125e-8, 
     -1.0837208140980114e-8, -2.483526865641276e-9, -5.73121584378756e-10, 
     -1.330460820879255e-10, -3.104408582051595e-11, 
     -7.275957614183426e-12, -1.7119900268666884e-12, 
     -4.0421986745463475e-13, -9.573628439715034e-14, 
     -2.2737367544323207e-14, -5.4136589391245725e-15, 
     -1.2918958832001822e-15, -3.089316242435218e-16, 
     -7.401486830834377e-17, -1.7763568394002505e-17, 
     -4.270088556250602e-18, -1.02798428206033e-18, 
     -2.4781763942525814e-19, -5.981805089575197e-20, 
     -1.4456028966473392e-20, -3.497426362856466e-21, 
     -8.470329472543003e-22, -2.0534132054649706e-22, 
     -4.982546748554708e-23, -1.2100470675061433e-23, 
     -2.941086622410765e-24, -7.153994486945105e-25, 
     -1.7414328685326897e-25, -4.2419518592462956e-26, 
     -1.0339757656912847e-26, -2.5218921114421576e-27, 
     -6.154617652924313e-28, -1.5028717524582625e-28, 
     -3.671788940665073e-29, -8.97548407718129e-30, 
     -2.1950912145280328e-30, -5.370967865334548e-31, 
     -1.314768175368353e-31, -3.2198404294735173e-32, 
     -7.888609052210118e-33, -1.9334826108358132e-33, 
     -4.740750632337812e-34, -1.1628256267998405e-34, 
     -2.853229547240349e-35, -7.003381615953585e-36, 
     -1.7195803074886033e-36, -4.22353057979657e-37, 
     -1.0376777717603641e-37, -2.5502250322924203e-38, 
     -6.269303204385533e-39, -1.5416319355046394e-39, 
     -3.791917260717056e-40, -9.329320244621329e-41, 
     -2.2958874039497803e-41, -5.651415148184075e-42, 
     -1.3914469114847154e-42, -3.426697617835493e-43, 
     -8.440762514521251e-44, -2.0796081557516126e-44, 
     -5.1247486695307594e-45, -1.2631422777012436e-45, 
     -3.1139965873884822e-46, -7.678347749725025e-47, 
     -1.893646573411915e-47, -4.670994881082724e-48, 
     -1.1523836055302771e-48, -2.8435439616980868e-49, 
     -7.017720674703611e-50, -1.7322221918572204e-50, 
     -4.276423536147513e-51, -1.0559070459623488e-51, 
     -2.6075753269192154e-52, -6.440396891788424e-53, 
     -1.5909313750548783e-53, -3.930536338370876e-54, 
     -9.712081068648967e-55, -2.4001119882293423e-55, 
     -5.93209497090775e-56, -1.46636055460641e-56, 
     -3.6251691488880693e-57, -8.963330313184787e-58, 
     -2.216475702445151e-58, -5.481606575939621e-59, 
     -1.3558229030914487e-59, -3.353877707647268e-60, 
     -8.297353703814855e-61, -2.052953493727387e-61, 
     -5.080012471723381e-62, -1.2571748036083116e-62, 
     -3.111507638930571e-63};
 
  TVIL_REAL H00xy0xtableB[] = 
     {0, -3.832395696766864, 1.4103728688791697, -0.7645688151148226, 
     0.4896392980417352, -0.34444150169983806, 0.25742724806637457, 
     -0.2007314811474131, 0.161519133849482, -0.1331563848551563, 
     0.11191240524929004, -0.09554859281736211, 0.08265093889994449, 
     -0.07228818881260686, 0.06382538688596241, -0.05681674663246943, 
     0.05094135694405122, -0.04596313912571007, 0.04170509816857665, 
     -0.0380322984291125, 0.034840328843722845, -0.03204731621161359, 
     0.029588287049970186, -0.02741111755534184, 0.02547357825948874, 
     -0.023741146492021418, 0.022185365951507927, -0.02078260179240453, 
     0.019513085449164306, -0.01836017431163945, 0.017309772524879866, 
     -0.01634987388882598, 0.01547019818651001, -0.014661899650618437, 
     0.013917331601694428, -0.013229855172077738, 0.01259368288743193, 
     -0.01200375000203949, 0.01145560807723868, -0.010945336497238993, 
     0.010469468534934809, -0.010024929285577385, 0.009608983331552629, 
     -0.009219190426099193, 0.008853367816441849, -0.008509558088991478, 
     0.008186001627086732, -0.007881112937391617, 0.007593460233768351, 
     -0.007321747774292979, 0.007064800533519518, -0.006821550862343559, 
     0.00659102684515044, -0.006372342110924876, 0.0061646868936681065, 
     -0.005967320169411682, 0.005779562723601416, -0.005600791024661706, 
     0.005430431797949013, -0.005267957209715119, 0.005112880583651665, 
     -0.004964752583503789, 0.004823157804469224, -0.0046877117239232735, 
     0.0045580579686616595, -0.004433865861523178, 0.004314828215099742, 
     -0.0042006593443928986, 0.0040910932738413815, -0.003985882117213549, 
     0.0038847946115067766, -0.003787614788285778, 0.0036941407678761974, 
     -0.003604183663552918, 0.003517566584361876, -0.0034341237265212675, 
     0.003353699544489809, -0.003276147993788853, 0.003201331838541091, 
     -0.0031291220174578414, 0.0030593970626836397, -0.0029920425665031923, 
     0.002926950691442051, -0.002864019719757669, 0.0028031536387294127, 
     -0.002744261758521381, 0.002687258359716236, -0.0026320623679066887, 
     0.002578597052988135, -0.002526789751024995, 0.002476571606767812, 
     -0.002427877335081029, 0.002380644999705057, -0.002334815807922957, 
     0.0022903339198337364, -0.0022471462710525613, 0.0022052024077645915, 
     -0.0021644543331549846, 0.002124856364324005, -0.0020863649988741516, 
     0.0020489387904266618};
 
  TVIL_REAL H00xy0xtableD[] = 
     {0, -2.452232128461682, 0.7583843664343507, -0.3738714225570137, 
     0.22514879183270928, -0.15158250311714414, 0.10957758482508934, 
     -0.0832192793947759, 0.06553423194499704, -0.05306023081827901, 
     0.04391389261739405, -0.03699653839547673, 0.03163083244863561, 
     -0.027380043663681273, 0.02395192022919039, -0.021144677168985042, 
     0.018815218772632626, -0.016859697751699824, 0.015201215014397492, 
     -0.013781806247041386, 0.012557085385875758, -0.011492581655719453, 
     0.010561183360107413, -0.009741321222501475, 0.009015655904713968, 
     -0.008370115521407836, 0.0077931801511706205, -0.007275343294871732, 
     0.006808701856185741, -0.006386640661998576, 0.006003587343847221, 
     -0.005654820155701144, 0.005336316021280834, -0.005044629442031571, 
     0.004776795286726321, -0.0045302502138355304, 0.004302768743442178, 
     -0.0040924109303653875, 0.0038974792870612005, -0.003716483128859117, 
     0.0035481089112774523, -0.003391195432539594, 0.0032447130078251387, 
     -0.0031077459026068984, 0.002979477453412462, -0.002859177414962483, 
     0.0027461911599326633, -0.0026399304268653473, 0.0025398653670322025, 
     -0.0024455176853775823, 0.002356454706394606, -0.00227228422470789, 
     0.002192650023657604, -0.002117227964389733, 0.0020457225637115863, 
     -0.001977863991942473, 0.0019134054327092774, -0.0018521207555283906, 
     0.0017938024594167082, -0.0017382598519558826, 0.0016853174334135108, 
     -0.0016348134598785388, 0.0015865986630386267, -0.001540535107330798, 
     0.001496495167828382, -0.0014543606144648594, 0.0014140217901028418, 
     -0.0013753768715868114, 0.001338331204315213, -0.0013027967020672975, 
     0.0012686913048529757, -0.0012359384884451316, 0.0012044668200244207, 
     -0.001174209555034347, 0.0011451042709242228, -0.0011170925339621094, 
     0.0010901195957396265, -0.0010641341163746363, 0.0010390879117539243, 
     -0.0010149357224526346, 0.0009916350022259436, -0.0009691457241960313, 
     0.0009474302030579346, -0.0009264529318048058, 0.0009061804316295181, 
     -0.0008865811137980057, 0.0008676251524124847, -0.000849284367091678, 
     0.0008315321146920749, -0.0008143431892805262, 0.0007976937296454011, 
     -0.0007815611337022006, 0.0007659239792109105, -0.0007507619502773163, 
     0.0007360557691597437, -0.0007217871329468612, 0.0007079386547118689, 
     -0.0006944938087840848, 0.0006814368798110731, -0.000668752915313426, 
     0.0006564276814604402};

  lnyox = TVIL_CLOG(y/x);
  lnyox2 = lnyox  * lnyox;
  lnyox3 = lnyox2  * lnyox;

  if (y < 0.5 * x) {     
    result = TVIL_H0000xx (x,qq);
    rn = r = y/x;
    for (n=1; n < 100; n++) {
      result += rn * (H00xy0xtableA[n] 
                    + H00xy0xtableAL[n] * lnyox 
                    + H00xy0xtableAL2[n] * lnyox2); 
      rn *= r;
    }
  }
  else if (y < 1.5 * x) {
    result = TVIL_H00xx0x (x, qq);
    rn = r = (y/x - 1.L);
    for (n=1; n < 100; n++) {
      result += rn * H00xy0xtableD[n]; 
      rn *= r;
    }
  }
  else if (y < 6.47 * x) {
    result = -7.37081890178383365960648778L - 6.L * Zeta3 * TVIL_Ap(x,qq);
    rn = r = (y/(4.L* x) - 1.L);
    for (n=1; n < 100; n++) {
      result += rn * H00xy0xtableB[n]; 
      rn *= r;
    }
  }
  else {
    result = TVIL_H00000x (y, qq);
    rn = r = 4.L*x/y;
    for (n=1; n < 100; n++) {
      result += rn * (H00xy0xtableC[n] +
                     +H00xy0xtableCL[n] * lnyox +
                     +H00xy0xtableCL2[n] * lnyox2 +
                     +H00xy0xtableCL3[n] * lnyox3);
      rn *= r;
    }
  }

  return result;
}

/* ****************************************************************** */

TVIL_COMPLEX TVIL_H0xyxxy (TVIL_REAL x, TVIL_REAL y, TVIL_REAL qq)
{
  TVIL_REAL r, rn;
  TVIL_COMPLEX lnyox, lnyox2, lnyox3, result;
  int n;

  TVIL_REAL H0xyxxytableA[] = 
     {0, -5., -0.5578703703703703, -0.09913580246913581, 
     -0.022110689315408702, -0.005895757597571657, -0.0018702317387822198, 
     -0.0007042900382162473, -0.0003097663721546232, -0.000154640983596654, 
     -0.00008502588227017905, -0.000050254114096208755, 
     -0.00003138452816573706, -0.00002046893559725157, 
     -0.000013829165881557104, -9.622650109638615e-6, 
     -6.8659629150144794e-6, -5.006656173975466e-6, 
     -3.7209996255646362e-6, -2.812405962491567e-6, -2.157776521066482e-6, 
     -1.677933689697068e-6, -1.3207265838036512e-6, 
     -1.0510694877493304e-6, -8.449028750850438e-7, -6.854406632537833e-7, 
     -5.607856518275605e-7, -4.6238342897279424e-7, -3.840013021044869e-7, 
     -3.2104250060974094e-7, -2.700783643948213e-7, 
     -2.2852462784409347e-7, -1.944144237065131e-7, 
     -1.6623713473355173e-7, -1.4282267012976278e-7, 
     -1.232574658352298e-7, -1.06822896276704e-7, -9.294969027163774e-8, 
     -8.118389176376313e-8, -7.116122822750638e-8, -6.258765724291877e-8, 
     -5.522449157606733e-8, -4.887694459095076e-8, -4.33852502642664e-8, 
     -3.861773520567292e-8, -3.446538080926251e-8, -3.08375303802943e-8, 
     -2.765848150315061e-8, -2.4864766897187687e-8, 
     -2.2402973770339716e-8, -2.0227986636158453e-8, 
     -1.830156485803571e-8, -1.6591186090882456e-8, -1.506910194780858e-8, 
     -1.3711563825649354e-8, -1.2498185758590793e-8, 
     -1.1411418084085544e-8, -1.0436111083339332e-8, -9.55915196155443e-9, 
     -8.769161832823747e-9, -8.056241976710532e-9, -7.411760694397611e-9, 
     -6.828173731200117e-9, -6.298872540799211e-9, -5.8180557153865544e-9, 
     -5.3806197496947645e-9, -4.98206598815944e-9, -4.618421156430481e-9, 
     -4.286169327171724e-9, -3.98219353606324e-9, -3.703725563352776e-9, 
     -3.448302642042155e-9, -3.2137300560685934e-9, 
     -2.9980487588180047e-9, -2.799507280533015e-9, 
     -2.6165373079142367e-9, -2.4477324147029622e-9, 
     -2.2918295017092746e-9, -2.147692571396526e-9, 
     -2.0142985180171567e-9, -1.890724661264768e-9, 
     -1.7761377909759638e-9, -1.6697845238236598e-9, 
     -1.5709828012120412e-9, -1.4791143815536753e-9, 
     -1.3936182004771612e-9, -1.3139844898554839e-9, 
     -1.2397495613396567e-9, -1.1704911727277594e-9, 
     -1.105824406328517e-9, -1.0453979977690702e-9, 
     -9.888910616814057e-10, -9.360101675765507e-10, 
     -8.864867251445692e-10, -8.400746433402494e-10, 
     -7.965482320462488e-10, -7.557003189464212e-10, 
     -7.173405575760918e-10, -6.81293905414484e-10, 
     -6.473992534080244e-10, -6.155081905135674e-10};
 
  TVIL_REAL H0xyxxytableAL[] = 
     {0, 3., 0.4861111111111111, 0.11, 0.03072987528344671, 
     0.010370622323003275, 0.004184146314377195, 0.0019735454549450076, 
     0.0010539306324510703, 0.0006177285713118969, 0.0003879177322815684, 
     0.0002566532913514061, 0.00017687755488052944, 0.00012597861479456812, 
     0.00009220683648964006, 0.00006906169092669979, 0.000052759657530502066, 
     0.000041004639502432355, 0.00003235335719037635, 
     0.000025870850909841243, 0.000020935371064498435, 
     0.00001712380429765053, 0.000014142248661849423, 
     0.000011782770572046298, 9.895801889896504e-6, 8.372129312630723e-6, 
     7.130907379785522e-6, 6.111535581124009e-6, 5.268062727203476e-6, 
     4.5652737429049065e-6, 3.975914796126812e-6, 3.478700235396734e-6, 
     3.0568639511594176e-6, 2.6970947488465306e-6, 2.3887458384266142e-6, 
     2.1232421851831665e-6, 1.8936321742283025e-6, 1.6942455650488228e-6, 
     1.5204304516826837e-6, 1.3683494567443115e-6, 1.234820698262328e-6, 
     1.1171928597405451e-6, 1.0132464259726425e-6, 9.2111513332285e-7, 
     8.392231391815126e-7, 7.662344911177193e-7, 7.01012277147418e-7, 
     6.425854390432468e-7, 5.901216839440126e-7, 5.429052739754287e-7, 
     5.003187369446169e-7, 4.6182774370977286e-7, 4.269685544798783e-7, 
     3.95337558121203e-7, 3.6658252377564447e-7, 3.403952589078247e-7, 
     3.1650542695229955e-7, 2.9467532456292314e-7, 2.7469545576808934e-7, 
     2.5638077017382086e-7, 2.3956745632281135e-7, 2.2411020064330328e-7, 
     2.0987983806505043e-7, 1.9676133308884265e-7, 1.846520404582163e-7, 
     1.7346020305949688e-7, 1.6310365163503562e-7, 1.5350867662484614e-7, 
     1.44609047185342e-7, 1.3634515635569537e-7, 1.2866327460114794e-7, 
     1.2151489667814626e-7, 1.148561690352201e-7, 1.0864738686453261e-7, 
     1.0285255151581073e-7, 9.743898032897191e-8, 9.237696207677455e-8, 
     8.763945216918299e-8, 8.320180258559341e-8, 7.904152219333376e-8, 
     7.51380637005394e-8, 7.147263399488595e-8, 6.802802505024948e-8, 
     6.478846295243082e-8, 6.173947291203579e-8, 5.886775840532525e-8, 
     5.6161092818948146e-8, 5.360822217750751e-8, 5.119877770856538e-8, 
     4.8923197151925716e-8, 4.677265385218974e-8, 4.473899278849349e-8, 
     4.281467279542371e-8, 4.099271431641551e-8, 3.926665210721296e-8, 
     3.76304923737164e-8, 3.607867388702416e-8, 3.4606032669798975e-8, 
     3.320776989319017e-8, 3.187942266323077e-8, 3.061683741060088e-8};
 
  TVIL_REAL H0xyxxytableAL2[] = 
     {0, -0.5, -0.125, -0.03888888888888889, -0.014583333333333334, 
     -0.006507936507936508, -0.003373015873015873, -0.001965098393669822, 
     -0.0012482309357309358, -0.0008447253545292761, -0.0005994169229463347, 
     -0.0004412285653490269, -0.0003344261374447133, -0.0002596209981508759, 
     -0.00020562644360790623, -0.00016565875809551185, 
     -0.00013543245400771069, -0.00011214589769858343, 
     -0.00009391327366203405, -0.00007943277158715695, 
     -0.0000677856611192188, -0.00005831078320426595, 
     -0.00005052398936614752, -0.00004406517310776336, 
     -0.00003866268757366699, -0.00003410897699046893, 
     -0.000030243594001871133, -0.000026941174889970454, 
     -0.000024102800840456915, -0.000021649708574869066, 
     -0.00001951865489533295, -0.000017658461240620136, 
     -0.000016027410629843768, -0.000014591267449433058, 
     -0.000013321757248188267, -0.000012195389684574449, 
     -0.000011192539856528917, -0.000010296725895482213, 
     -9.494036871434965e-6, -8.772676710086096e-6, -8.122598305759567e-6, 
     -7.535208244168854e-6, -7.003127164113113e-6, -6.519994233405224e-6, 
     -6.080306807419199e-6, -5.6792883039682125e-6, -5.312778828030176e-6, 
     -4.977144231895299e-6, -4.669200186774124e-6, -4.386148534271297e-6, 
     -4.12552372752809e-6, -3.885147597482366e-6, -3.6630910160739466e-6, 
     -3.4576412953790097e-6, -3.267274374845677e-6, 
     -3.0906310197037756e-6, -2.9264963912240766e-6, 
     -2.773782460761312e-6, -2.631512829838938e-6, -2.498809592147531e-6, 
     -2.374881933549511e-6, -2.2590162156258552e-6, -2.150567329036213e-6, 
     -2.0489511366389495e-6, -1.9536378542465587e-6, 
     -1.8641462401279175e-6, -1.7800384837593947e-6, 
     -1.700915700555825e-6, -1.6264139529340455e-6, 
     -1.5562007295263819e-6, -1.489971824036601e-6, -1.427448563416806e-6, 
     -1.3683753419872312e-6, -1.3125174240247563e-6, 
     -1.2596589823778036e-6, -1.2096013449637103e-6, 
     -1.162161424684811e-6, -1.117170311457072e-6, -1.074472007760107e-6, 
     -1.0339222914566158e-6, -9.953876916487387e-7, -9.587445650856359e-7, 
     -9.238782621504256e-7, -8.906823727689567e-7, -8.590580437260953e-7, 
     -8.289133598712746e-7, -8.001627825644343e-7, -7.727266394735279e-7, 
     -7.465306605003313e-7, -7.215055551950275e-7, -6.975866275328055e-7, 
     -6.747134243768213e-7, -6.528294143493228e-7, -6.318816941834177e-7, 
     -6.118207199378154e-7, -5.926000607310363e-7, -5.741761728944654e-7, 
     -5.565081926591294e-7, -5.395577456824833e-7, -5.232887718917511e-7, 
     -5.076673642719785e-7};
 
  TVIL_REAL H0xyxxytableC[] = 
     {0, -1.893829290521217, 0.1733144102723232, 0.14314759539822597, 
     0.10693668774301324, 0.0812332913152306, 0.06277377862958178, 
     0.04915106770575772, 0.038849242865190536, 0.030894429083372643, 
     0.024641473156461646, 0.019650536265279517, 0.015613973918725765, 
     0.012311659789973728, 0.00958287805435415, 0.00730816016794944, 
     0.005397259375023944, 0.0037810117521402953, 0.00240571660112171, 
     0.001229185265249344, 0.0002179166812376343, -0.0006549523902822374, 
     -0.0014111556771840839, -0.002068421491650041, -0.0026413203584760847, 
     -0.0031419111284282653, -0.003580238445972689, -0.0039647193489054655, 
     -0.004302446320150671, -0.004599426772116333, -0.004860773729838147, 
     -0.005090858733121428, -0.005293435258081236, -0.005471738964086842, 
     -0.005628569595963556, -0.005766358269096106, -0.005887223035237948, 
     -0.005993014997146509, -0.0060853567588044916, -0.006165674627429021, 
     -0.006235225696295463, -0.0062951207134444576, -0.006346343465624418, 
     -0.006389767268167701, -0.006426169041487767, -0.006456241367148119, 
     -0.006480602846128919, -0.0064998070252829665, -0.0065143501121536496, 
     -0.006524677661097822, -0.006531190383279234, -0.006534249208211371, 
     -0.006534179704062643, -0.006531275947043391, -0.006525803916199821, 
     -0.006518004478307887, -0.006508096017859677, -0.006496276759017983, 
     -0.006482726819605683, -0.0064676100314623175, -0.0064510755566636185, 
     -0.006433259325002841, -0.006414285314658555, -0.006394266695015152, 
     -0.006373306848081267, -0.006351500282793313, -0.006328933454643762, 
     -0.006305685501485858, -0.006281828904999523, -0.006257430086123936, 
     -0.00623254994174352, -0.006207244329029818, -0.006181564503076886, 
     -0.006155557512799989, -0.006129266559487895, -0.0061027313218916315, 
     -0.00607598825129016, -0.006049070839585238, -0.006022009863137445, 
     -0.0059948336047566594, -0.005967568055996887, -0.005940237101673997, 
     -0.005912862688319628, -0.005885464978104403, -0.005858062489602953, 
     -0.005830672226631847, -0.0058033097962650465, -0.005775989517020359, 
     -0.00574872451810985, -0.005721526830559348, -0.0056944074709225564, 
     -0.005667376518244609, -0.0056404431848668946, -0.005613615881608497, 
     -0.0055869022778083435, -0.005560309356667556, -0.005533843466289899, 
     -0.005507510366781765, -0.005481315273740461, -0.005455262898429021, 
     -0.005418913823326536};
 
  TVIL_REAL H0xyxxytableCL[] = 
     {0, -2., -0.3248775943990546, -0.1780488906953509, 
     -0.10794843429489148, -0.06466857334904913, -0.03624111656783163, 
     -0.016878259768595416, -0.0033209484243580667, 0.006379022395829492, 
     0.013436408754181395, 0.018635825278401047, 0.022499739929363688, 
     0.025385413915747584, 0.02754286982436241, 0.029150673880992323, 
     0.030338660015575517, 0.031202733227660207, 0.03181473598847406, 
     0.03222916077986704, 0.03248780245587715, 0.032623037350458, 
     0.03266017001201346, 0.032619136182270206, 0.032515754413808357, 
     0.03236265672884696, 0.0321699880684205, 0.031945937180456824, 
     0.03169714325325892, 0.03142901001102131, 0.031145950232723478, 
     0.030851577492567582, 0.030548857532098878, 0.03024022851663284, 
     0.029927697133841635, 0.02961291580895685, 0.029297245065263787, 
     0.028981804128989153, 0.028667512178637465, 0.028355122109261402, 
     0.028045248278176615, 0.027738389388424656, 0.027434947426599125, 
     0.027135243385345426, 0.02683953035521964, 0.026548004456152935, 
     0.026260813988377825, 0.02597806711092998, 0.025699838298633447, 
     0.0254261737826607, 0.025157096142909577, 0.02489260819068131, 
     0.02463269625602152, 0.02437733297445785, 0.024126479651841826, 
     0.02388008827287192, 0.02363810320808242, 0.023400462665183683, 
     0.023167099923280924, 0.02293794438239504, 0.022712922455634093, 
     0.022491958327131297, 0.022274974595328557, 0.022061892818219197, 
     0.021852633974674615, 0.021647118853882697, 0.021445268383158377, 
     0.02124700390289134, 0.021052247396130236, 0.020860921679228043, 
     0.020672950559059784, 0.020488258961545814, 0.020306773035550212, 
     0.020128420235656655, 0.01995312938683887, 0.019780830733626877, 
     0.01961145597601325, 0.019444938294036825, 0.019281212362717733, 
     0.019120214358789595, 0.018961881960479522, 0.01880615434141646, 
     0.018652972159602817, 0.018502277542257355, 0.018354014067227937, 
     0.018208126741577943, 0.018064561977867596, 0.017923267568580425, 
     0.01778419265908271, 0.01764728771945031, 0.01751250451545025, 
     0.017379796078923967, 0.017249116677783526, 0.01712042178580181, 
     0.016993668052350433, 0.016868813272216357, 0.016745816355607665, 
     0.01662463729844156, 0.01650523715299227, 0.016387577998963447, 
     0.0162562814008219};
 
  TVIL_REAL H0xyxxytableCL2[] = 
     {0, -0.25, 0.026041666666666668, -0.019444444444444445, 
     -0.04700753348214286, -0.06036427331349206, -0.06637948143075097, 
     -0.0686103126009655, -0.06883963681654073, -0.06799407655959407, 
     -0.06657504725072716, -0.06486307011799483, -0.0630189105926035, 
     -0.06113599546287946, -0.05926873640074436, -0.05744839967638495, 
     -0.05569227745021364, -0.05400912804943074, -0.05240247376080019, 
     -0.050872635358436745, -0.04941800455671499, -0.04803584762274223, 
     -0.0467228157002505, -0.04547526909304419, -0.04428948220449073, 
     -0.04316177126133637, -0.042088571793058596, -0.04106648333411533, 
     -0.04009229277050895, -0.039162983856229884, -0.03827573788620914, 
     -0.0374279288414178, -0.036617115212710355, -0.03584102996879567, 
     -0.03509756963562481, -0.03438478311847195, -0.0337008606708424, 
     -0.0330441232607957, -0.032413012481739986, -0.03180608108534567, 
     -0.0312219841681655, -0.03065947101347804, -0.030117377570750267, 
     -0.02959461954346898, -0.029090186049520423, -0.02860313381514466, 
     -0.028132581862604616, -0.02767770665229556, -0.027237737641528585, 
     -0.026811953224276684, -0.02639967701851549, -0.026000274470251807, 
     -0.025613149745796232, -0.025237742886229517, -0.02487352720028958, 
     -0.024520006874043238, -0.02417671477769015, -0.02384321045167416, 
     -0.023519078255950728, -0.023203925667785067, -0.022897381714842444, 
     -0.022599095531588946, -0.02230873502815826, -0.0220259856618674, 
     -0.021750549302491602, -0.02148214318324507, -0.02122049893016824, 
     -0.02096536166330211, -0.020716489163642934, -0.020473651100422905, 
     -0.020236628313760816, -0.020005212148175956, -0.019779203832864137, 
     -0.019558413905000974, -0.019342661672668397, -0.0191317747142995, 
     -0.0189255884118074, -0.018723945514808715, -0.01852669573357398, 
     -0.018333695358538786, -0.018144806904391478, -0.01795989877691908, 
     -0.017778844960943643, -0.017601524727817876, -0.01742782236107356, 
     -0.017257626898929575, -0.01709083189246962, -0.016927335178394114, 
     -0.016767038665336582, -0.016609848132813586, -0.01645567304194891, 
     -0.016304426357178582, -0.016156024378203397, -0.016010386581510626, 
     -0.015867435470837393, -0.015727096435994316, -0.015589297619510716, 
     -0.01545396979060177, -0.015321046225994122, -0.015190462597179266, 
     -0.015056522015793905};
 
  TVIL_REAL H0xyxxytableCL3[] = 
     {0, 0., 0.0625, 0.0625, 0.056640625, 0.05078125, 
     0.045735677083333336, 0.04150390625, 0.037952423095703125, 
     0.034947713216145836, 0.032380294799804685, 0.030164718627929688, 
     0.028234978516896565, 0.026539921760559082, 0.02503957280090877, 
     0.023702370127042134, 0.022503129119286314, 0.02142154355533421, 
     0.020441077801580023, 0.019548141019186005, 0.01873146561902104, 
     0.017981634702491214, 0.01729071914912145, 0.01665199617923463, 
     0.016059729053604716, 0.015508993093631318, 0.014995537126020659, 
     0.014515672263138867, 0.014066181958393634, 0.01364424875481346, 
     0.013247394233047683, 0.012873429472672872, 0.012520413945376034, 
     0.012186621215051494, 0.011870510167162132, 0.011570700755984637, 
     0.01128595346397447, 0.01101515182736483, 0.010757287507249795, 
     0.010511447483962162, 0.010276803030652319, 0.01005260018420985, 
     0.009838151481538016, 0.009632828769359307, 0.009436056928247201, 
     0.009247308378028653, 0.009066098253314588, 0.008891980155654098, 
     0.008724542403427854, 0.008563404712694728, 0.008408215252256424, 
     0.008258648024585726, 0.008114400531277518, 0.007975191687571244, 
     0.007840759955455509, 0.00771086166905969, 0.007585269529593336, 
     0.007463771250117979, 0.007346168333014768, 0.007232274965216635, 
     0.007121917018164722, 0.007014931141074426, 0.006911163937497583, 
     0.006810471216377711, 0.006712717309843514, 0.006617774450895628, 
     0.0065255222049329965, 0.006435846949755162, 0.006348641399279351, 
     0.006263804166738756, 0.00618123936359096, 0.0061008562307719475, 
     0.006022568799288926, 0.005946295577460723, 0.005871959262393264, 
     0.0057994864735242775, 0.005728807506290031, 0.005659856104160944, 
     0.005592569247465555, 0.005526886957575965, 0.005462752115164976, 
     0.005400110291367543, 0.005338909590788642, 0.005279100505397759, 
     0.005220635778438091, 0.005163470277557587, 0.005107560876439936, 
     0.005052866344277538, 0.004999347242486151, 0.004946965828112883, 
     0.004895685963436194, 0.004845473031299095, 0.004796293855755188, 
     0.004748116627642114, 0.004700910834728611, 0.004654647196110166, 
     0.004609297600554414, 0.004564835048521214, 0.004521233597604059, 
     0.004478468311159267, 0.004436515209907435};
 
  TVIL_REAL H0xyxxytableB[] = 
     {0, -4.46174402057842, 1.6878607023212888, -0.9209228826750575, 
     0.5900508061969214, -0.41441277337672533, 0.3089926768655442, 
     -0.2403153849559233, 0.19286542590750178, -0.15859526282017034, 
     0.13297044119648305, -0.11326784738468865, 0.09776713805548942, 
     -0.08533562811320572, 0.07520147410429556, -0.06682321386828394, 
     0.05981142497972208, -0.05387992167663397, 0.048814374574228214, 
     -0.0444515680967183, 0.04066535208137678, -0.0373569196999512, 
     0.034447948437782705, -0.03187567639913741, 0.029589312059499995, 
     -0.027547378802056877, 0.025715725163576952, -0.024066016038960747, 
     0.022574575985693734, -0.021221493447636436, 0.01998992051530074, 
     -0.018865520758518523, 0.0178360302799363, -0.016890906125774457, 
     0.016021042669228047, -0.015218541302766647, 0.014476522249988882, 
     -0.01378896988906661, 0.013150604914649533, -0.012556778127493257, 
     0.012003381755200493, -0.011486775062431953, 0.011003721669632371, 
     -0.010551336513795473, 0.010127040786961694, -0.009728523505581815, 
     0.00935370861486079, -0.009000726732681263, 0.0086678907970541, 
     -0.008353675011578882, 0.008056696585638956, -0.007775699852434204, 
     0.007509542416146764, -0.007257183036237624, 0.0070176710038425405, 
     -0.006790136802234031, 0.0065737838784031, -0.006367881372709383, 
     0.006171757689493448, -0.005984794783221947, 0.005806423090822493, 
     -0.005636117009214181, 0.005473390863326114, -0.005317795298578273, 
     0.005168914053284429, -0.005026361058371592, 0.004889777831890124, 
     -0.0047588311306051845, 0.0046332108448814135, -0.004512628065414373, 
     0.004396813402577721, -0.004285515344946866, 0.004178498917400603, 
     -0.004075544345548743, 0.003976445886843649, -0.0038810108045110314, 
     0.0037890582751878704, -0.0037004187026731805, 0.0036149326337039656, 
     -0.0035324502922396694, 0.003452830854337119, -0.0033759414483697514, 
     0.0033016569021807746, -0.003229859406645455, 0.003160437995612775, 
     -0.0030932874828539236, 0.0030283082015369472, -0.002965405681930794, 
     0.002904493624959495, -0.0028454850208619042, 0.0027883022477690755, 
     -0.002732869872929214, 0.002679115056783324, -0.002626970193210315, 
     0.0025763696262876834, -0.0025272539380618853, 0.0024795664316384775, 
     -0.002433247855058024, 0.002388246122303622, -0.002344509359620675, 
     0.0023020097683009166};
 
  TVIL_REAL H0xyxxytableD[] = 
     {0, -2.9076415309340558, 0.9075528692303481, -0.444529021879714, 
     0.26572809604126735, -0.1777554768694839, 0.12780651820329558, 
     -0.09662408670107187, 0.07579727698279626, -0.06116592307686569, 
     0.05047555006610807, -0.042415741881200564, 0.03618133519416877, 
     -0.031254783711776785, 0.027290741079000147, -0.02405138780562699, 
     0.02136848178463538, -0.019120199303275548, 0.017216516624285083, 
     -0.01558970724338038, 0.014188002964374155, -0.012971267577422466, 
     0.011907983253944196, -0.010973112526879616, 0.0101465561586494, 
     -0.009412023993733865, 0.008756196807222824, -0.008168096313968753, 
     0.007638606158674219, -0.007160103817632393, 0.006726174940648658, 
     -0.006331389641343069, 0.005971125810778327, -0.005641428462986112, 
     0.005338896933981568, -0.005060593790053414, 0.004803970787526465, 
     -0.004566808322994893, 0.004347165629798769, -0.0041433395900270995, 
     0.003953830495936596, -0.0037773134492230104, 0.0036126143591247925, 
     -0.003458689711285018, 0.003314609443645304, -0.0031795423944797965, 
     0.0030527438892752334, -0.0029335451137307705, 0.0028213439843883734, 
     -0.002715597279881146, 0.002615813837243075, -0.002521548651263951, 
     0.0024323977421330384, -0.0023479936788640687, 0.0022680016642300307, 
     -0.0021921161019405402, 0.0021200575791884557, -0.002051570207965897, 
     0.0019864192770975356, -0.0019243891740729074, 0.0018652815417350421, 
     -0.0018089136399023903, 0.0017551168862296948, -0.0017037355541899418, 
     0.0016546256090878703, -0.0016076536655911175, 0.001562696052458859, 
     -0.0015196379720223072, 0.0014783727435769556, -0.001438801121224173, 
     0.0014008306778859544, -0.0013643752482395896, 0.001329354424202362, 
     -0.0012956930973631944, 0.001263321043421476, -0.0012321725442727555, 
     0.0012021860438837205, -0.0011733038345390773, 0.001145471770427726, 
     -0.0011186390058724595, 0.0010927577558038257, -0.0010677830763382967, 
     0.001043672663550733, -0.0010203866687329437, 0.0009978875286088392, 
     -0.0009761398091349846, 0.0009551100616550473, -0.0009347666903015402, 
     0.0009150798296483415, -0.0008960212317160792, 0.0008775641615202016, 
     -0.0008596833004296296, 0.0008423546566741469, -0.0008255554824008108, 
     0.0008092641967362201, -0.0007934603143615092, 0.0007781243791520671, 
     -0.0007632379024747016, 0.0007487833057715284, -0.0007347438670927479, 
     0.0007211036712702042};

  lnyox = TVIL_CLOG(y/x);
  lnyox2 = lnyox  * lnyox;
  lnyox3 = lnyox2  * lnyox;

  if (y < 0.5 * x) {
    result = TVIL_H00x0xx (x,qq);
    rn = r = y/x;
    for (n=1; n < 100; n++) {
      result += rn * (H0xyxxytableA[n] + 
                      H0xyxxytableAL[n] * lnyox + 
                      H0xyxxytableAL2[n] * lnyox2);
      rn *= r;
    }
  }
  else if (y < 1.6 * x) {
    result = TVIL_H0xxxxx (x,qq);
    rn = r = y/x - 1.L;
    for (n=1; n < 100; n++) {
      result += rn * H0xyxxytableD[n];
      rn *= r;
    }
  }
  else if (y < 6. * x) {
    result = -13.325948659996127462L - 6.L * Zeta3 * TVIL_Ap(x,qq);
    rn = r = y/(4.L*x) - 1.L;
    for (n=1; n < 64; n++) {
      result += rn * H0xyxxytableB[n];
      rn *= r;
    }
  }
  else {
    result = TVIL_H0000xx (y, qq);
    rn = r = 4.L * x/y;
    for (n=1; n < 100; n++) {
      result += rn * (H0xyxxytableC[n] 
                     +H0xyxxytableCL[n] * lnyox 
                     +H0xyxxytableCL2[n] * lnyox2
                     +H0xyxxytableCL3[n] * lnyox3);
      rn *= r;
    }
  }

  return result;
}

/* ****************************************************************** */

TVIL_COMPLEX TVIL_H00yxyx (TVIL_REAL x, TVIL_REAL y, TVIL_REAL qq)
{
  TVIL_REAL r, rn;
  TVIL_COMPLEX lnyox, lnyox2, lnyox3, result;
  int n;

  TVIL_REAL H00yxyxtableA[] = 
     {0, -1.5479430968404058, -0.06628527877928236, 
     -0.006013133100024864, -0.0012565424995303748, -0.0005429208076882179, 
     -0.00031842964941132757, -0.0002074264683930396, 
     -0.00014219693920383686, -0.0001009927672878716, 
     -0.00007380953218811489, -0.00005526909950618017, 
     -0.00004226384992596835, -0.00003291614993198787, 
     -0.000026051715650225568, -0.000020914194910195504, 
     -0.000017003563198997238, -0.000013981443347461545, 
     -0.000011613976721705294, -9.736431331362152e-6, 
     -8.230744739693364e-6, -7.010959380318245e-6, -6.013585436323936e-6, 
     -5.191105846785796e-6, -4.507524291977443e-6, -3.935265571618351e-6, 
     -3.4529862216682757e-6, -3.044007276621232e-6, 
     -2.6951783875554098e-6, -2.396045018151705e-6, -2.138231244982706e-6, 
     -1.914977721463788e-6, -1.7207925252279861e-6, 
     -1.5511849685016866e-6, -1.4024609665763318e-6, 
     -1.271564494251785e-6, -1.155953841333315e-6, -1.0535043542751414e-6, 
     -9.624314898002755e-7, -8.812295574387769e-7, -8.086226626313544e-7, 
     -7.435251989626099e-7, -6.850098601921659e-7, -6.322816086142482e-7, 
     -5.846563876035117e-7, -5.415436329409432e-7, -5.024318413271902e-7, 
     -4.6687661118043236e-7, -4.3449069198371377e-7, 
     -4.049356726710922e-7, -3.7791501320786035e-7, -3.531681814096223e-7, 
     -3.304657027615226e-7, -3.0960496727338945e-7, -2.904066663202017e-7, 
     -2.7271175556476427e-7, -2.5637885866882797e-7, 
     -2.412820415205713e-7, -2.2730889887947758e-7, 
     -2.1435890524156034e-7, -2.0234198981170795e-7, 
     -1.9117730209261916e-7, -1.8079214004391953e-7, 
     -1.7112101725500268e-7, -1.6210484928991086e-7, 
     -1.5369024244544462e-7, -1.4582887072974615e-7, 
     -1.3847692901065148e-7, -1.3159465207618452e-7, 
     -1.2514589085460473e-7, -1.1909773830799366e-7, 
     -1.1342019858198318e-7, -1.0808589389805641e-7, 
     -1.0306980444113894e-7, -9.834903714639362e-8, -9.390261984377932e-8, 
     -8.971131769238711e-8, -8.57574692415865e-8, -8.202483980320719e-8, 
     -7.849849011722189e-8, -7.516465855007946e-8, -7.201065528619457e-8, 
     -6.902476716434412e-8, -6.619617197625336e-8, -6.351486118819093e-8, 
     -6.097157017106918e-8, -5.855771513303032e-8, -5.6265336043045125e-8, 
     -5.408704491659095e-8, -5.2015978906639477e-8, 
     -5.0045757706383837e-8, -4.81704448255585e-8, -4.638451235088823e-8, 
     -4.468280884402222e-8, -4.3060530068028836e-8, -4.15131922667952e-8, 
     -4.003660775106336e-8, -3.8626862570823624e-8, -3.728029607680258e-8, 
     -3.599348219419139e-8, -2.6456786080566956e-6};
 
  TVIL_REAL H00yxyxtableAL[] = 
     {0, 1.0887664832879436, 0.06665136596654848, 
     0.003524648957387709, -0.0011551691899625933, -0.0010375836099557022, 
     -0.0006754636432639293, -0.0004333983799966759, -0.0002867223298754863, 
     -0.00019669025767321614, -0.00013951244223977483, 
     -0.00010187560601047128, -0.00007627552343351132, 
     -0.000058351699759020975, -0.00004548097430499353, 
     -0.00003603185243482929, -0.000028958269436163335, 
     -0.000023570945324908037, -0.000019404458681708046, 
     -0.000016137588414549017, -0.00001354423196453057, 
     -0.000011462389107299852, -9.774106606182764e-6, 
     -8.392236470960101e-6, -7.251524773771063e-6, -6.30250931594681e-6, 
     -5.507273403733252e-6, -4.836447350762219e-6, -4.267062096940133e-6, 
     -3.7809933355511067e-6, -3.3638204404508167e-6, 
     -3.003980462037761e-6, -2.6921344995786835e-6, -2.420688617036703e-6, 
     -2.183428378181675e-6, -1.9752377215987406e-6, -1.791881010173032e-6, 
     -1.6298328057488942e-6, -1.4861439883947052e-6, 
     -1.3583357641968449e-6, -1.2443152269941285e-6, 
     -1.1423076918983264e-6, -1.0508021638788313e-6, 
     -9.685071564654698e-7, -8.943147137556771e-7, -8.272709703805006e-7, 
     -7.665519498046856e-7, -7.114435809253089e-7, -6.613251279900092e-7, 
     -6.156553952462022e-7, -5.739611971926719e-7, -5.358276865727287e-7, 
     -5.008902118710688e-7, -4.688274389900825e-7, -4.39355521727846e-7, 
     -4.1222314551257775e-7, -3.872073007540416e-7, -3.641096678363139e-7, 
     -3.427535165032966e-7, -3.2298103919345577e-7, 
     -3.0465105155689046e-7, -2.876370045588277e-7, 
     -2.7182526173004746e-7, -2.571136026552947e-7, 
     -2.4340992000441755e-7, -2.3063108255428727e-7, -2.18701940919936e-7, 
     -2.0755445626946508e-7, -1.971269352671039e-7, 
     -1.8736335697590179e-7, -1.782127795400016e-7, 
     -1.6962881622475692e-7, -1.6156917187706902e-7, 
     -1.539952321240879e-7, -1.4687169869351864e-7, 
     -1.4016626514425897e-7, -1.3384932806762512e-7, 
     -1.2789372947826523e-7, -1.222745266776769e-7, 
     -1.1696878635674869e-7, -1.119554001192174e-7, 
     -1.0721491896564987e-7, -1.0272940458614331e-7, 
     -9.848229557663834e-8, -9.445828692466114e-8, -9.06432213106117e-8, 
     -8.702399094475773e-8, -8.358844881157562e-8, -8.032532832514711e-8, 
     -7.72241705146436e-8, -7.427525795979586e-8, -7.146955478459982e-8, 
     -6.879865209502116e-8, -6.625471831456187e-8, -6.383045393146997e-8, 
     -6.151905022416628e-8, -5.93141515780426e-8, -5.720982104793783e-8, 
     -5.520050885700588e-8, -5.328102355493788e-8, -3.886825059595448e-6};
 
  TVIL_REAL H00yxyxtableAL2[] = 
     {0, -0.25, -0.036458333333333336, -0.00798611111111111, 
     -0.0024483816964285714, -0.0009768725198412698, -0.00047134171890031266, 
     -0.00025940013994561983, -0.00015649799237254987, 
     -0.00010091993737563782, -0.00006844520579171265, 
     -0.00004829679217216003, -0.000035190304357214175, 
     -0.000026330671140142776, -0.00002014760314455338, 
     -0.00001571444432354641, -0.000012461483493098061, 
     -0.000010026059691991769, -8.170288070682427e-6, 
     -6.733996470556322e-6, -5.606826041035188e-6, -4.711168536303925e-6, 
     -3.991439725433649e-6, -3.407168576038599e-6, -2.928445748155955e-6, 
     -2.5328657945517278e-6, -2.203435488025686e-6, -1.927119301437979e-6, 
     -1.6938126070610009e-6, -1.495606712499644e-6, -1.326256023295854e-6, 
     -1.180787148515242e-6, -1.0552089709901362e-6, -9.462953944500218e-7, 
     -8.514209889752808e-7, -7.684355399411732e-7, -6.955674870864235e-7, 
     -6.313490139052001e-7, -5.745575013214496e-7, -5.241694503563882e-7, 
     -4.793239782858142e-7, -4.392937182211531e-7, -4.034614830611753e-7, 
     -3.7130144670001085e-7, -3.4236388695189644e-7, 
     -3.162627531909837e-7, -2.9266548666781525e-7, -2.71284646830833e-7, 
     -2.5187099287129456e-7, -2.34207743505751e-7, -2.1810579513656038e-7, 
     -2.0339972300180948e-7, -1.899444247312922e-7, 
     -1.7761229310598309e-7, -1.6629082646487042e-7, 
     -1.558806023971127e-7, -1.462935540771716e-7, -1.3745149959580005e-7, 
     -1.2928488348861816e-7, -1.2173169681431257e-7, 
     -1.1473654793482443e-7, -1.0824986087280442e-7, 
     -1.0222718198119188e-7, -9.662857882486735e-8, -9.1418117778677e-8, 
     -8.656340899607392e-8, -8.203520918305839e-8, -7.780707409103632e-8, 
     -7.385505387434488e-8, -7.015742548764587e-8, -6.66944571608855e-8, 
     -6.344820071408775e-8, -6.0402308084533e-8, -5.75418689542012e-8, 
     -5.4853266801577097e-8, -5.232405107204684e-8, -4.994282347590575e-8, 
     -4.769913669132104e-8, -4.558340397881412e-8, -4.358681841006078e-8, 
     -4.1701280582149274e-8, -3.99193338331441e-8, -3.8234106099439277e-8, 
     -3.663925766293935e-8, -3.512893412909419e-8, -3.369772405735375e-8, 
     -3.234062073549239e-8, -3.1052987649993674e-8, 
     -2.9830527257571426e-8, -2.8669252709023574e-8, 
     -2.7565462216901613e-8, -2.651571579372435e-8, 
     -2.5516814118347675e-8, -2.4565779315203507e-8, 
     -2.365983745493836e-8, -2.2796402605943872e-8, 
     -2.1973062284745697e-8, -2.1187564169521625e-8, 
     -2.0437803955428343e-8, -1.972181424316656e-8, -1.4277497295949218e-6};
 
  TVIL_REAL H00yxyxtableC[] = 
     {0, -1.5795867438169813, -0.018418253829740383, 
     0.009758330737013488, 0.006650200100248638, 0.0038658156894913476, 
     0.002180382876794451, 0.001171479357063826, 0.0005538130225843374, 
     0.00016722332207315393, -0.00007861199362786675, 
     -0.00023620162809684375, -0.00033707796159149206, 
     -0.00040074984809852776, -0.0004396098926429809, 
     -0.00046171353339917454, -0.00047240236370479916, 
     -0.00047527961637904015, -0.00047281159331771794, 
     -0.00046670715967191134, -0.0004581622476124911, 
     -0.00044802037647224263, -0.0004368798374117591, -0.0004251663663479219, 
     -0.00041318310124429866, -0.00040114535349422553, 
     -0.00038920508188943284, -0.0003774682927160239, -0.0003660075223262463, 
     -0.00035487086384511715, -0.00034408854094841734, -0.000333677724694368, 
     -0.00032364608144470203, -0.0003139943974128957, 
     -0.00030471852667960167, -0.0002958108404716511, 
     -0.00028726130675093067, -0.00027905829443855903, 
     -0.0002711891716698639, -0.00026364074944045315, -0.000256399608863183, 
     -0.0002494523406193355, -0.0002427857180763663, -0.00023638682026741985, 
     -0.00023024311699084907, -0.00022434252533628553, 
     -0.00021867344472063614, -0.0002132247758362734, -0.0002079859276374, 
     -0.00020294681551826128, -0.00019809785309412783, 
     -0.00019342993942687847, -0.00018893444309991785, 
     -0.00018460318421103378, -0.00018042841509277076, 
     -0.0001764028003701356, -0.00017251939681128467, 
     -0.00016877163330799964, -0.0001651532912310726, 
     -0.00016165848533516037, -0.0001582816453335155, -0.0001550174982215078, 
     -0.00015186105139637727, -0.00014880757659667398, 
     -0.00014585259466705879, -0.00014299186114081174, 
     -0.00014022135262278714, -0.0001375372539487059, 
     -0.00013493594609188114, -0.00013241399478557646, 
     -0.0001299681398272091, -0.0001275952850299314, -0.00012529248878692667, 
     -0.00012305695521416804, -0.000120886025838228, -0.0001187771717967882, 
     -0.00011672798652072585, -0.00011473617886810158, 
     -0.00011279956668178246, -0.00011091607074391243, 
     -0.00010908370910196752, -0.00010730059174255367, 
     -0.0001055649155905173, -0.00010387495981239222, 
     -0.00010222908140442022, -0.00010062571104670678, 
     -0.00009906334920627754, -0.00009754056247286268, 
     -0.00009605598011237435, -0.000094608290824005, -0.00009319623968782453, 
     -0.00009181862529062963, -0.0000904742970186485, 
     -0.00008916215250640708, -0.00008788113523190482, 
     -0.00008663023224874354, -0.00008540847204667771, 
     -0.00008421492253245103, -0.00008304868912346539, 
     -0.00008190891294724417, -0.00008601659993178737};
 
  TVIL_REAL H00yxyxtableCL[] = 
     {0, -1.4112335167120564, -0.09193738538829793, 
     -0.014849784929117482, 0.0004300047643115146, 0.005388765603540138, 
     0.007101268070572209, 0.00752378697477582, 0.007400526992062867, 
     0.007051437008622784, 0.006621219527949362, 0.00617641292364822, 
     0.005747122199022479, 0.00534600718336509, 0.004977277392315901, 
     0.004641083397975716, 0.0043357061516381655, 0.004058659894496157, 
     0.003807245748767625, 0.0035788226049503474, 0.003370931114042559, 
     0.0031813414133182397, 0.003008061880454308, 0.0028493287846982837, 
     0.0027035874300549677, 0.0025694703840264815, 0.0024457756657599704, 
     0.002331446285496858, 0.002225551724640581, 0.0021272715189835393, 
     0.002035880889315952, 0.001950738260546825, 0.0018712744702783961, 
     0.0017969834610299835, 0.0017274142604457353, 0.0016621640716132247, 
     0.0016008723160165954, 0.001543215491981424, 0.0014889027304221718, 
     0.0014376719467156778, 0.00138928650244677, 0.0013435323036705927, 
     0.0013002152733796474, 0.001259159145259908, 0.0012202035337844502, 
     0.0011832022424269977, 0.0011480217774652913, 0.0011145400396456026, 
     0.001082645170034622, 0.0010522345298112842, 0.0010232137966498076, 
     0.0009954961628003836, 0.0009690016220567422, 0.0009436563345695398, 
     0.0009193920599708211, 0.0008961456505592342, 0.0008738585973930135, 
     0.0008524766230768952, 0.0008319493158345632, 0.0008122298001501768, 
     0.0007932744398583104, 0.0007750425700753451, 0.0007574962548093714, 
     0.0007406000674700193, 0.0007243208918329977, 0.0007086277413038133, 
     0.0006934915945773028, 0.0006788852460094914, 0.000664783169210385, 
     0.0006511613925343912, 0.0006379973852923672, 0.0006252699536386571, 
     0.0006129591452002045, 0.0006010461616150001, 0.0005895132782355176, 
     0.0005783437703308098, 0.0005675218451900623, 0.0005570325795915688, 
     0.0005468618621554163, 0.0005369963401464178, 0.0005274233703367816, 
     0.0005181309735762334, 0.0005091077927514823, 0.0005003430538473776, 
     0.0004918265298493887, 0.00048354850725146597, 0.00047549975495523905, 
     0.00046767149536616173, 0.0004600553775098914, 0.00045264345200807504, 
     0.000445428147767055, 0.0004384022502458972, 0.0004315588811818242, 
     0.00042489147966165083, 0.00041839378443737766, 0.0004120598173926915, 
     0.0004058838680749803, 0.00039986047921454883, 0.0003939844331591715, 
     0.0003882507391579776, 0.0003903253785420243};
 
  TVIL_REAL H00yxyxtableCL2[] = 
     {0, -0.125, 0.0234375, -0.008680555555555556, 
     -0.018513997395833332, -0.019306640625, -0.017647298177083333, 
     -0.015583915450946003, -0.013676367487226214, -0.012039869634772944, 
     -0.010664665396251376, -0.009511881348937573, -0.008541330480486685, 
     -0.007718572087732053, -0.007015761619988097, -0.006410815335304361, 
     -0.005886286353644617, -0.005428352284129359, -0.005025998041558661, 
     -0.004670382479609359, -0.004354354812155635, -0.0040720867245976145, 
     -0.0038187919388523966, -0.0035905114035325154, -0.003383947712155125, 
     -0.0031963365722494363, -0.003025346308444636, -0.002868998711781585, 
     -0.0027256062544083123, -0.0025937219394674012, -0.002472098974964268, 
     -0.0023596581388208105, -0.002255461205984571, -0.0021586891847225724, 
     -0.0020686243921376004, -0.0019846356130654, -0.001906165749633353, 
     -0.0018327214938346568, -0.001763864651977448, -0.001699204824788186, 
     -0.0016383932054568559, -0.0015811173038658508, -0.0015270964415384659, 
     -0.0014760778906605994, -0.0014278335535282847, -0.0013821570972197975, 
     -0.0013388614731560178, -0.0012977767632466838, -0.00125874830410524, 
     -0.0012216350488052235, -0.0011863081322024416, -0.0011526496112398256, 
     -0.0011205513561076065, -0.0010899140718265271, -0.0010606464328969066, 
     -0.001032664316224185, -0.0010058901196828643, -0.0009802521554887725, 
     -0.0009556841090737805, -0.0009321245554458097, -0.0009095165261097215, 
     -0.0008878071205538189, -0.0008669471570988466, -0.0008468908585834803, 
     -0.0008275955689405372, -0.0008090214972165607, -0.0007911314860166122, 
     -0.0007738908017264716, -0.0007572669441848179, -0.0007412294737556408, 
     -0.000725749853992344, -0.0007108013082949492, -0.0006963586891449007, 
     -0.0006823983586619689, -0.000668898079367805, -0.0006558369141635426, 
     -0.000643195134636767, -0.0006309541369081476, -0.0006190963643117832, 
     -0.0006076052362772336, -0.000596465082846623, -0.0005856610843181224, 
     -0.0005751792155585142, -0.0005650061945732083, -0.0005551294349627002, 
     -0.000545537001930667, -0.0005362175715411978, -0.0005271603929515074, 
     -0.0005183552533723077, -0.0005097924455311276, -0.0005014627374346227, 
     -0.0004933573442445345, -0.0004854679020987112, -0.00047778644372366866, 
     -0.0004703053756987606, -0.00046301745724427524, -0.0004559157804168495, 
     -0.0004489937516055932, -0.00044224507423137654, -0.0004356637325599396, 
     -0.0004320614004973855};
 
  TVIL_REAL H00yxyxtableCL3[] = 
     {0, 0., 0.03125, 0.0234375, 0.0166015625, 0.01220703125, 
     0.009379069010416666, 0.007476806640625, 0.0061359405517578125, 
     0.0051517486572265625, 0.004404878616333008, 0.0038224458694458008, 
     0.0033579195539156594, 0.0029804036021232605, 0.002668678095298154, 
     0.002407740770528714, 0.002186717712902464, 0.0019975552859250456, 
     0.0018341749936452187, 0.001691912113528815, 0.0015671335952220033, 
     0.0014569722767527353, 0.001359138394536527, 0.001271783563886686, 
     0.00119340106993208, 0.0011227517265921527, 0.0010588080261871387, 
     0.001000711563844912, 0.0009477402246873057, 0.000899282638882842, 
     0.0008548181084047459, 0.0008139006963479111, 0.0007761465136559913, 
     0.0007412234841159147, 0.0007088430464447609, 0.0006787533824324118, 
     0.0006507338562286047, 0.0006245904215371707, 0.0006001518073980777, 
     0.0005772663341376646, 0.0005557992423369202, 0.0005356304417524156, 
     0.0005166526058059723, 0.0004987695518462199, 0.00048189485884729045, 
     0.0004659506832706147, 0.0004508667410191794, 0.000436579429171582, 
     0.0004230310658097035, 0.0004101692299895834, 0.00039794618693589384, 
     0.0003863183860104313, 0.0003752460210267744, 0.00036469264414492486, 
     0.00035462482595093774, 0.00034501185546235037, 0.00033582547474452314, 
     0.00032703964361085596, 0.0003186303305394116, 0.00031057532649245146, 
     0.000302854078792153, 0.000295447542600246, 0.0002883380478836688, 
     0.00028150918003255645, 0.00027494567253912163, 0.00026863331035301635, 
     0.00026255884270614486, 0.00025670990435225436, 0.00025107494429780933, 
     0.00024564316121386174, 0.0002404044448165457, 0.00023534932258870327, 
     0.00023046891128887965, 0.00022575487275810693, 0.00022119937359087337, 
     0.00021679504828559732, 0.00021253496553275537, 0.0002084125973363872, 
     0.00020442179069771375, 0.0002005567416186671, 0.00019681197120875602, 
     0.00019318230370132434, 0.0001896628462052785, 0.00018624897003610233, 
     0.00018293629348571533, 0.00017972066590472074, 0.0001765981529830392, 
     0.0001735650231260178, 0.00017061773483300754, 0.00016775292499425108, 
     0.00016496739802984036, 0.00016225811580160003, 0.0001596221882351161, 
     0.0001570568645948477, 0.00015455952536040044, 0.00015212767465666894, 
     0.0001497589331947297, 0.00014745103168412965, 0.0001452018046806181, 
     0.0001430091848364459, 0.00014087119752314106};
 
  TVIL_REAL H00yxyxtableB[] = 
     {0, -5.0628893982817065, 2.047382391789873, -1.1672770690327166, 
     0.772255800105581, -0.5560106642317105, 0.422943722005418, 
     -0.3344389283970494, 0.27220196060490776, -0.2265614535962275, 
     0.19197570946928966, -0.16506657253788587, 0.14367099553650503, 
     -0.12634738119636682, 0.11210243516911511, -0.10023223091851192, 
     0.09022574480502424, -0.08170416516629384, 0.07438150688205568, 
     -0.06803835230488098, 0.06250392269519765, -0.05764357655678174, 
     0.05334992643815072, -0.04953641910391005, 0.04613262441662885, 
     -0.04308072973108008, 0.04033289801304472, -0.03784925358608762, 
     0.03559632988468433, -0.03354586137338588, 0.031673834686541384, 
     -0.02995973701153302, 0.028385955987024904, -0.02693729702286894, 
     0.025600592374667525, -0.024364382473258565, 0.02321865456803054, 
     -0.022154627143935946, 0.021164571131532365, -0.02024166087140699, 
     0.01937984927931747, -0.01857376280219966, 0.017818612642342, 
     -0.01711011941962562, 0.0164444489858171, -0.01581815753485574, 
     0.015228144494758036, -0.014671611959751725, 0.014146029640477948, 
     -0.013649104487013425, 0.013178754282899536, -0.012733084625178342, 
     0.012310368800975639, -0.011909030149633903, 0.01152762656408979, 
     -0.011164836838731098, 0.01081944861544039, -0.010490347716601843, 
     0.0101765086848512, -0.009876986375362617, 0.009590908468362032, 
     -0.009317468788042203, 0.009055921329706233, -0.008805574910256138, 
     0.00856578836845793, -0.008335966251073511, 0.008115554929215956, 
     -0.007904039096376617, 0.0077009386056712615, -0.007505805609109486, 
     0.007318221966233182, -0.007137796893401787, 0.006964164828413531, 
     -0.0067969834881177445, 0.006635932099256966, -0.006480709785032618, 
     0.006331034091859974, -0.006186639642505641, 0.0060472769033169185, 
     -0.005912711054585209, 0.005782720954259468, -0.005657098186260945, 
     0.005535646185565204, -0.005418179433026761, 0.0053045227136389655, 
     -0.005194510432558343, 0.005087985983788426, -0.004984801166921723, 
     0.0048848156477873175, -0.004787896459252228, 0.004693917538782634, 
     -0.004602759299691443, 0.004514308233285714, -0.004428456539384857, 
     0.004345101782911832, -0.0042641465744674405, 0.004185498272984997, 
     -0.004109068708731341, 0.004034773925072337, -0.003962533937558449, 
     0.003892272509010289};
 
  TVIL_REAL H00yxyxtableD[] = 
     {0, -3.5763898990065126, 1.2274618429079514, 
     -0.6340363571501366, 0.3921105816691218, -0.2686028418892603, 
     0.1965681572973041, -0.1506651404515779, 0.11950340571515362, 
     -0.09731903267077452, 0.0809296391355988, -0.06845643375456997, 
     0.05872964975859569, -0.050988753484022216, 0.044721181103483644, 
     -0.0395708473007439, 0.03528386813419134, -0.03167511369705323, 
     0.028606910226645223, -0.025975085708470087, 0.023699594477764326, 
     -0.021718076488503886, 0.019981343838727014, -0.018450160884602357, 
     0.017092909842187854, -0.015883873393459307, 0.01480195422462823, 
     -0.013829708581359575, 0.012952608580543446, -0.012158473258279574, 
     0.011437025523104386, -0.010779544063061183, 0.010178587577407859, 
     -0.009627774607790565, 0.009121606481659317, -0.008655323956217471, 
     0.008224790406107402, -0.007826396067246858, 0.007456979096075285, 
     -0.007113760142763737, 0.0067942878502373115, -0.006496393236645102, 
     0.006218151339527252, -0.0059578488262846795, 0.005713956530412623, 
     -0.005485106073199955, 0.0052700698888442, -0.0050677440966888235, 
     0.00487713376475549, -0.004697340189411833, 0.004527549881099592, 
     -0.004367024998802188, 0.004215095018880194, -0.00407114945901863, 
     0.003934631506857297, -0.003805032426633137, 0.0036818866368170107, 
     -0.0035647673680440244, 0.0034532828242303856, -0.003347072781132945, 
     0.003245805566136853, -0.003149175371072986, 0.003056899856630486, 
     -0.0029687180126531697, 0.002884388243465315, -0.002803686651504797, 
     0.0027264054960669864, -0.0026523518069780346, 0.002581346135601335, 
     -0.002513221427802692, 0.002447822005413555, -0.0023850026443839175, 
     0.00232462773924606, -0.0022665705447497773, 0.002210712486606601, 
     -0.0021569425342180043, 0.002105156629080245, -0.002055257163273058, 
     0.0020071525030649754, -0.001960756553216672, 0.0019159883580457666, 
     -0.0018727717357406703, 0.0018310349427849801, -0.0017907103656840573, 
     0.0017517342374773215, -0.0017140463767783408, 0.0016775899473140856, 
     -0.0016423112361383895, 0.001608159448875818, -0.0015750865205135132, 
     0.001543046940402494, -0.00151199759025844, 0.0014818975940669194, 
     -0.0014527081789009364, 0.0014243925457509416, -0.0013969157495502514, 
     0.0013702445876532339, -0.0013443474960905726, 0.0013191944529861999, 
     -0.0012947568885748732, 0.0012710076013084212};
 
  TVIL_REAL H00yxyxtableE[] = 
     {0, 2.115819489896223, -1.5261975767837797, 1.211112021756433, 
     -1.0109381608448256, 0.8709855746324195, -0.7669526366543788, 
     0.6862411600816268, -0.6216106387528415, 0.5685791112946103, 
     -0.5242119289812345, 0.4865010344690192, -0.45402270325719357, 
     0.42573750451792086, -0.4008676899069004, 0.3788190047198143, 
     -0.3591291273973895, 0.3414326767348997, -0.32543686305383734, 
     0.3109041710931282, -0.2976398035479569, 0.2854824184432958, 
     -0.2742971900632005, 0.26397053774846296, -0.2544060708689525, 
     0.2455214333480462, -0.23724582226102467, 0.22951801760188661, 
     -0.22228480395283515, 0.21549969566984917, -0.20912189934085199, 
     0.20311546334847613, -0.19744857617409292, 0.19209298384089216, 
     -0.18702350346083407, 0.18221761481853038, -0.17765511571654502, 
     0.1733178297236498, -0.16918935722914114, 0.1652548624724195, 
     -0.16150089060563927, 0.1579152099460984, -0.15448667544994207, 
     0.15120511013938798, -0.14806120177985435, 0.1450464125600454, 
     -0.14215289989954977, 0.13937344681216424, -0.13670140050248106, 
     0.13413061807888454, -0.13165541843636103, 0.12927053950407266, 
     -0.12697110017077093, 0.12475256630006519, -0.12261072033072223, 
     0.12054163402730676, -0.11854164400581119, 0.11660732970928565, 
     -0.11473549355135162, 0.11292314298208292, -0.11116747426207368, 
     0.10946585775741124, -0.10781582459142199, 0.1062150545090334, 
     -0.10466136482687145, 0.1031527003571897, -0.10168712420674156, 
     0.1002628093630393, -0.09887803099033442, 0.09753115936630244, 
     -0.09622065339799428, 0.09494505466226942, -0.0937029819217792, 
     0.0924931260727256, -0.09131424548517401, 0.09016516170072492, 
     -0.08904475545591516, 0.08795196300288516, -0.08688577270165938, 
     0.08584522186089084, -0.08482939380615202, 0.08383741515684526, 
     -0.08286845329458771, 0.0819217140075187, -0.08099643929640762, 
     0.08009190532972232, -0.07920742053597168, 0.07834232382267359, 
     -0.0774959829122341, 0.07666779278586805, -0.0758571742274524, 
     0.07506357245989317, -0.07428645586721043, 0.07352531479611095, 
     -0.07277966043133127, 0.07204902373949955, -0.07133295447668797, 
     0.07063102025521337, -0.06994280566559412, 0.06926791144989229, 
     -0.06860595372296145};

  lnyox = TVIL_CLOG(y/x);
  lnyox2 = lnyox  * lnyox;
  lnyox3 = lnyox2  * lnyox;

  if (y < 0.1545 * x) {
    result = TVIL_H0000xx (x,qq);
    rn = r = 4.L * y/x;
    for (n=1; n < 100; n++) {
      result += rn * (H00yxyxtableA[n] + 
                      H00yxyxtableAL[n] * lnyox + 
                      H00yxyxtableAL2[n] * lnyox2);
      rn *= r;
    }
  }
  else if (y < 0.5 * x) {
    result = -1.9978533782973847021L - 6.L * Zeta3 * TVIL_Ap(x,qq);
    rn = r = 0.25L * x/y - 1.L;
    for (n=1; n < 100; n++) {
      result += rn * H00yxyxtableE[n];
      rn *= r;
    }
  }
  else if (y < 1.6 * x) {
    result = TVIL_H00xxxx (x,qq);
    rn = r = y/x - 1.L;
    for (n=1; n < 100; n++) {
      result += rn * H00yxyxtableD[n];
      rn *= r;
    }
  }
  else if (y < 6.47 * x) {
    result = -11.926489899965666927L - 6.L * Zeta3 * TVIL_Ap(x,qq);
    rn = r = y/(4.L*x) - 1.L;
    for (n=1; n < 64; n++) {
      result += rn * H00yxyxtableB[n];
      rn *= r;
    }
  }
  else {
    result = TVIL_H0000xx (y, qq);
    rn = r = 4.L * x/y;
    for (n=1; n < 100; n++) {
      result += rn * (H00yxyxtableC[n] 
                     +H00yxyxtableCL[n] * lnyox 
                     +H00yxyxtableCL2[n] * lnyox2
                     +H00yxyxtableCL3[n] * lnyox3);
      rn *= r;
    }
  }

  return result;
}

/* ****************************************************************** */

TVIL_COMPLEX TVIL_H000yxx (TVIL_REAL x, TVIL_REAL y, TVIL_REAL qq)
{
  TVIL_REAL r, rn;
  TVIL_COMPLEX lnyox, lnyox2, lnyox3, result;
  int n;

  TVIL_REAL H000yxxtableA[] = 
     {0, -4.934802200544679, 0.181359075880536, 0.06416886443839245, 
     0.01824065528652331, 0.00568765732570582, 0.0020279936076613913, 
     0.0008303784708569284, 0.00038530927290828307, 0.00019833657466611956, 
     0.00011084005079510887, 0.00006606620850958446, 0.00004144445952961552, 
     0.00002709841763970486, 0.000018336610611332925, 
     0.000012772272513824738, 9.120050915420242e-6, 6.6540544725676726e-6, 
     4.947522973913302e-6, 3.7407457396060606e-6, 2.87085237714575e-6, 
     2.232966834369581e-6, 1.7579520006122944e-6, 1.399262109422847e-6, 
     1.1249610003716986e-6, 9.127566396924097e-7, 7.468431736063384e-7, 
     6.158518997987441e-7, 5.114972253687302e-7, 4.2766675900978684e-7, 
     3.598003484687612e-7, 3.044602216878288e-7, 2.5902945997568657e-7, 
     2.2149787583445148e-7, 1.9030820782085703e-7, 1.6424445281687158e-7, 
     1.4234997473894957e-7, 1.2386688202234267e-7, 1.0819075105133585e-7, 
     9.483652769909667e-8, 8.341264431149258e-8, 7.360122597154836e-8, 
     6.51428463844348e-8, 5.782470888647166e-8, 5.147142462894457e-8, 
     4.5937773632236993e-8, 4.110298958251918e-8, 3.686622282912294e-8, 
     3.314291979844472e-8, 2.986191923854593e-8, 2.696311220799447e-8, 
     2.4395547708989718e-8, 2.211589235089474e-8, 2.0087172599356766e-8, 
     1.8277743611553216e-8, 1.666044055010884e-8, 1.52118774719557e-8, 
     1.3911866047129616e-8, 1.2742931957334123e-8, 1.1689911216903878e-8, 
     1.0739612123164597e-8, 9.880531287059948e-9, 9.102614377126135e-9, 
     8.397053952338465e-9, 7.756118156014871e-9, 7.173005166626683e-9, 
     6.641719208650797e-9, 6.156964661718342e-9, 5.714055403948934e-9, 
     5.3088370128055424e-9, 4.937619845635598e-9, 4.59712134938899e-9, 
     4.284416218441509e-9, 3.99689324187995e-9, 3.732217865738486e-9, 
     3.488299648522417e-9, 3.263263915566119e-9, 3.055427023918397e-9, 
     2.8632747382395895e-9, 2.685443292648509e-9, 2.520702776036353e-9, 
     2.3679425310838766e-9, 2.2261583017303956e-9, 2.094440901508139e-9, 
     1.9719662070941944e-9, 1.857986308571456e-9, 1.75182167099748e-9, 
     1.6528541815936132e-9, 1.5605209737171705e-9, 1.4743089332095013e-9, 
     1.393749805092945e-9, 1.3184158292300155e-9, 1.2479158427192895e-9, 
     1.1818917947034175e-9, 1.12001562609021e-9, 1.0619864725938766e-9, 
     1.0075281546221808e-9, 9.5638692197848e-10, 9.083294252103403e-10, 
     8.631408887995251e-10, 8.206234643205615e-10};
 
  TVIL_REAL H000yxxtableAL[] = 
     {0, 2., -0.08333333333333333, -0.04259259259259259, 
     -0.015178571428571428, -0.005825396825396826, -0.0025362754529421197, 
     -0.001249515790332117, -0.0006829802142302142, -0.00040509740400807937, 
     -0.0002559415558641565, -0.00016988488307311618, 
     -0.00011729712232267373, -0.00008364020849716657, 
     -0.0000612662697133213, -0.000045913444870944965, 
     -0.00003509041513745161, -0.00002728114426495112, 
     -0.00002153094097331834, -0.00001722055299696602, 
     -0.00001393777911636367, -0.000011401898120023105, 
     -9.417799753649876e-6, -7.847377660470193e-6, -6.591250805055673e-6, 
     -5.5768274001546275e-6, -4.750355641251745e-6, -4.071534009943252e-6, 
     -3.5097967507848016e-6, -3.0417151576482198e-6, 
     -2.6491541862230683e-6, -2.3179480404736783e-6, 
     -2.0369372845762235e-6, -1.7972610392488546e-6, -1.59183131399521e-6, 
     -1.4149388401693745e-6, -1.2619548377061558e-6, 
     -1.129103452889574e-6, -1.0132867353725668e-6, -9.11949012380908e-7, 
     -8.229710461920767e-7, -7.445868803661778e-7, -6.753180960454628e-7, 
     -6.139215199513645e-7, -5.593473937424663e-7, -5.107057297721709e-7, 
     -4.672391109254757e-7, -4.283005916369801e-7, -3.9333565875786354e-7, 
     -3.6186744010243437e-7, -3.334845237283267e-7, -3.078308857786447e-7, 
     -2.8459752896444556e-7, -2.6351551484282383e-7, 
     -2.4435013642855585e-7, -2.2689602747459586e-7, 
     -2.1097304406795101e-7, -1.9642278536357192e-7, 
     -1.8310564511278382e-7, -1.7089830550894104e-7, 
     -1.596916008303248e-7, -1.4938869122859495e-7, 
     -1.3990349742789059e-7, -1.3115935556301026e-7, 
     -1.2308785828574557e-7, -1.1562785391424952e-7, 
     -1.0872458003480357e-7, -1.0232891178187852e-7, 
     -9.639670817509113e-8, -9.088824250383929e-8, -8.576770492103708e-8, 
     -8.100276721619808e-8, -7.656420124957197e-8, -7.242554379537221e-8, 
     -6.856280160583356e-8, -6.49541914035714e-8, -6.15799102658433e-8, 
     -5.842193250410288e-8, -5.546382968483701e-8, -5.269061089888172e-8, 
     -5.008858077928397e-8, -4.764521310315338e-8, -4.5349038099831234e-8, 
     -4.3189541833602864e-8, -4.115707624036074e-8, -3.924277857933795e-8, 
     -3.743849921767882e-8, -3.57367368008988e-8, -3.41305799793292e-8, 
     -3.261365496207905e-8, -3.118007825810671e-8, -2.98244140405671e-8, 
     -2.8541635637292725e-8, -2.7327090708445433e-8, 
     -2.6176469723204323e-8, -2.5085777391830816e-8, 
     -2.4051306748424743e-8, -2.3069615613885667e-8, 
     -2.213750519864877e-8, -2.1252000631212632e-8, -2.0410333221781885e-8};
 
  TVIL_REAL H000yxxtableAL2[] = 
      {0, -0.5, -0.041666666666666664, -0.005555555555555556, 
     -0.0008928571428571428, -0.00015873015873015873, 
     -0.000030062530062530064, -5.946434517863089e-6, 
     -1.214063714063714e-6, -2.5392182254927353e-7, -5.412544112234515e-8, 
     -1.1715463446395053e-8, -2.568063545314858e-9, 
     -5.689248469620608e-10, -1.27180289863212e-10, 
     -2.865211127952822e-11, -6.4987147559413805e-12, 
     -1.4827727072379621e-12, -3.400962796760088e-13, 
     -7.837211565791382e-14, -1.813611137981211e-14, 
     -4.212801714242069e-15, -9.819479682773535e-16, 
     -2.2959556359721695e-16, -5.383690298888642e-17, 
     -1.2657165763917785e-17, -2.9829293372732343e-18, 
     -7.045633235488142e-19, -1.6676190352827451e-19, 
     -3.954668250640267e-20, -9.395129940080408e-21, 
     -2.2357527620498063e-21, -5.328765883754623e-22, 
     -1.2719478473111267e-22, -3.040279204832785e-23, 
     -7.276527455044927e-24, -1.7436905579870962e-24, 
     -4.183307965848347e-25, -1.0047278254818223e-25, 
     -2.415629337322263e-26, -5.8135636425113326e-27, 
     -1.4004368906983036e-27, -3.376555383338853e-28, 
     -8.148076191805385e-29, -1.9678412850898846e-29, 
     -4.75623062164047e-30, -1.1504260641211123e-30, 
     -2.7846048406317475e-31, -6.744728172100362e-32, 
     -1.63474160619811e-32, -3.9646612085673353e-33, 
     -9.621095924498485e-34, -2.3361156179630844e-34, 
     -5.675522579489829e-35, -1.37959007665169e-35, 
     -3.355199886168747e-36, -8.163995862017745e-37, 
     -1.987446904462634e-37, -4.840490998949849e-38, 
     -1.1794445690618057e-38, -2.8751026224818946e-39, 
     -7.011495353566468e-40, -1.7105805278403377e-40, 
     -4.1749025708052427e-41, -1.0193275285141491e-41, 
     -2.4896634208669978e-42, -6.083060347653866e-43, 
     -1.4868034381315363e-43, -3.635218210115723e-44, 
     -8.890960014585371e-45, -2.1752240816773354e-45, 
     -5.323443212575638e-46, -1.3031991664041277e-46, 
     -3.1912066502782224e-47, -7.816666776674318e-48, 
     -1.915170795036625e-48, -4.6936370347163716e-49, 
     -1.1506004376759937e-49, -2.8213027274528396e-50, 
     -6.919618557535707e-51, -1.6975369267916807e-51, 
     -4.1654153559798926e-52, -1.0223436387320094e-52, 
     -2.509762185773651e-53, -6.16258614834427e-54, 
     -1.513512282029835e-54, -3.7179131775008705e-55, 
     -9.134836841670466e-56, -2.244856495279343e-56, 
     -5.517732717400886e-57, -1.3564854393088894e-57, 
     -3.335417418008016e-58, -8.202836670980154e-59, 
     -2.0176928097406576e-59, -4.9638824415311606e-60, 
     -1.2214108953876172e-60, -3.0058991412721545e-61, 
     -7.398741115849627e-62, -1.8214221653330494e-62, 
     -4.484678889365381e-63, -1.1043803465997512e-63};
 
  TVIL_REAL H000yxxtableC[] = 
     {0, -0.03876335736944325, 0.668888878520501, 0.5053024888448541, 
     0.37222810288839064, 0.28446408751629876, 0.22528199601575083, 
     0.18367638711826378, 0.15328140716952146, 0.13034347332324822, 
     0.11255828401021752, 0.09845256655364765, 0.08704919132067174, 
     0.07767892267685375, 0.06987038854583225, 0.06328337096922641, 
     0.05766699868732189, 0.052832757236897974, 0.048636597243375296, 
     0.044966789695318296, 0.04173550522489728, 0.03887286273554986, 
     0.03632264989286498, 0.03403919714600585, 0.031985061465902444, 
     0.030129287471508823, 0.028446086256194193, 0.02691382042517642, 
     0.025514216375859333, 0.0242317471375554, 0.0230531445775869, 
     0.02196701069262753, 0.020963505486386813, 0.02003409454936347, 
     0.0191713435510777, 0.018368749871718396, 0.01762060384356447, 
     0.016921873755937893, 0.01626811005126917, 0.015655365111422574, 
     0.015080125780027178, 0.014539256344362839, 0.014029950150533616, 
     0.01354968837865528, 0.01309620478325278, 0.012667455425011461, 
     0.012261592596300126, 0.011876942284255424, 0.011511984629165374, 
     0.011165336928174242, 0.010835738809413194, 0.010522039263015594, 
     0.01022318526582335, 0.009938211778072973, 0.009666232924659852, 
     0.009406434202063099, 0.009158065575744587, 0.008920435352675773, 
     0.00869290473028774, 0.008474882937143555, 0.008265822892452259, 
     0.008065217321550503, 0.00787259527297384, 0.007687518989972804, 
     0.007509581095503565, 0.0073384020550075965, 0.007173627885829219, 
     0.0070149280860198485, 0.006861993758640098, 0.006714535910576332, 
     0.006572283907404284, 0.0064349840680160125, 0.006302398384625596, 
     0.0061743033554237745, 0.0060504889185968745, 0.005930757477689384, 
     0.005814923009397496, 0.005702810245853784, 0.005594253924318744, 
     0.005489098097948897, 0.005387195501976453, 0.0052884069702237215, 
     0.00519260089739615, 0.005099652743059673, 0.005009444573618101, 
     0.004921864638971008, 0.004836806980857391, 0.004754171070180108, 
     0.00467386147086482, 0.00459578752803858, 0.004519863078520381, 
     0.0044460061818018095, 0.004374138869862633, 0.004304186914316144, 
     0.004236079609513889, 0.004169749570360945, 0.004105132543702541, 
     0.004042167232241735, 0.003980795130037316, 0.003920960368712072, 
     0.0038626095735748364};
 
  TVIL_REAL H000yxxtableCL[] = 
     {0, 3.144934066848226, 0.9762252750680849, 0.49460431948226935, 
     0.30777907163940593, 0.2140940956428723, 0.15964074435682374, 
     0.12480892988361889, 0.10098125145982212, 0.08385151657140867, 
     0.07105733686561401, 0.06120734729660936, 0.053434994997813234, 
     0.047175660709097914, 0.042047491372114325, 0.03778403423856106, 
     0.034194342808993264, 0.031138402278494726, 0.02851147080646428, 
     0.02623380438002759, 0.02424373840167382, 0.02249292210265041, 
     0.020942968618131826, 0.01956305698518488, 0.01832818719507133, 
     0.01721789145080649, 0.016215269398503488, 0.015306256903210937, 
     0.014479065508040085, 0.013723748219148908, 0.01303185988132876, 
     0.012396189148261195, 0.01181054518633981, 0.011269586612830028, 
     0.010768683307499532, 0.010303804020036254, 0.009871424373887308, 
     0.009468451112861997, 0.009092159369906533, 0.008740140442352131, 
     0.008410258094769318, 0.008100611822524192, 0.007809505827554756, 
     0.007535422705689852, 0.007277001038931199, 0.007033016239086137, 
     0.00680236411039648, 0.006584046695466055, 0.006377160046245967, 
     0.006180883624221187, 0.005994471084430474, 0.005817242239001575, 
     0.005648576029407234, 0.005487904364141122, 0.005334706701152313, 
     0.005188505273092719, 0.005048860868961488, 0.004915369098663069, 
     0.004787657077802726, 0.004665380479104601, 0.004548220904459067, 
     0.004435883538037075, 0.004328095046351396, 0.004224601695763677, 
     0.00412516766186734, 0.004029573508531156, 0.00393761481725868, 
     0.003849100949980846, 0.0037638539305161253, 0.0036817074317573245, 
     0.00360250585722046, 0.0035261035069558814, 0.0034523638190060053, 
     0.003381158678623616, 0.0033123677883617215, 0.003245878092929058, 
     0.003181583253390238, 0.0031193831658897263, 0.0030591835206056936, 
     0.003000895397103136, 0.0029444348926638093, 0.002889722780530698, 
     0.0028366841953230454, 0.002785248343159773, 0.0027353482342789386, 
     0.002686920436162745, 0.0026399048453748354, 0.002594244476492336, 
     0.002549885266671716, 0.0025067758945274644, 0.002464867612127641, 
     0.002424114089022382, 0.0023844712673218274, 0.0023458972269300585, 
     0.0023083520601226417, 0.0022717977547282195, 0.0022361980852402628, 
     0.002201518511244266, 0.002167726082599146, 0.002134789350859881, 
     0.00210267828647215};
 
  TVIL_REAL H000yxxtableCL2[] = 
     {0, -0.5, -0.15625, -0.08159722222222222, 
     -0.051920572916666664, -0.03669921875, -0.027692328559027778, 
     -0.02184900815795068, -0.017806121281215122, -0.014872584603661162, 
     -0.01266454023028177, -0.010953485932888736, -0.009595788290303966, 
     -0.00849710249413108, -0.007593168793595712, -0.006838864830540379, 
     -0.006201673173043223, -0.005657629757658244, -0.005188726227208922, 
     -0.004781195730378176, -0.004424352764978724, -0.004109790249537488, 
     -0.0038308126557632595, -0.0035820285939664946, -0.0033590532490885474, 
     -0.003158287855030614, -0.0029767540767092074, -0.00281196810812957, 
     -0.002661843888676481, -0.0025246179345293036, -0.0023987904003899296, 
     -0.0022830784581148603, -0.0021763791148953254, -0.0020777393323943827, 
     -0.0019863318412438355, -0.0019014354340674076, -0.0018224188066715444, 
     -0.0017487272301754423, -0.0016798714968468762, -0.0016154187035368715, 
     -0.0015549845290402581, -0.0014982267327798224, -0.0014448396572466638, 
     -0.0013945495595346074, -0.001347110630969854, -0.0013023015904099389, 
     -0.0012599227578833079, -0.0012197935320830135, -0.0011817502087439914, 
     -0.0011456440878343198, -0.0011113398263248612, -0.0010787140004934021, 
     -0.0010476538475998188, -0.0010180561615971801, -0.0009898263215242314, 
     -0.0009628774345189627, -0.0009371295781293596, -0.0009125091288785931, 
     -0.0008889481659500529, -0.0008663839404590606, -0.0008447584021263999, 
     -0.000824017776307455, -0.0008041121852952176, -0.0007849953086346902, 
     -0.0007666240778840442, -0.0007489584018539161, -0.0007319609188665506, 
     -0.00071559677301461, -0.0006998334117764321, -0.000684640402669675, 
     -0.00066998926690637, -0.0006558533282559252, -0.0006422075755340694, 
     -0.000629028537319676, -0.0006162941676617902, -0.0006039837416792638, 
     -0.000592077760078003, -0.0005805578617183297, -0.0005694067434593682, 
     -0.0005586080865904525, -0.0005481464892327685, -0.0005380074041590946, 
     -0.0005281770815366689, -0.0005186425161488403, -0.0005093913986960744, 
     -0.0005004120708167821, -0.000491693483503929, -0.00048322515862501584, 
     -0.00047499715328122507, -0.00046700002676674005, 
     -0.0004592248099117869, -0.0004516629766131475, -0.0004443064173740041, 
     -0.00043714741469124053, -0.0004301786201429479, 
     -0.00042339303304204386, -0.0004167839805337768, -0.0004103450990255832, 
     -0.00040407031684743536, -0.0003979538380495473, -0.0003919901272522229};
 
  TVIL_REAL H000yxxtableCL3[] = 
     {0, 0.08333333333333333, 0.03125, 0.017361111111111112, 
     0.011393229166666666, 0.008203125, 0.006266276041666667, 
     0.004987444196428571, 0.0040912628173828125, 0.00343464038990162, 
     0.0029366175333658856, 0.0025483044711026278, 0.00223861469162835, 
     0.001986936116829897, 0.0017791188188961574, 0.0016051605343818665, 
     0.001457811813452281, 0.0013317035250913572, 0.0012227833293662923, 
     0.0011279414090830343, 0.0010447557301631605, 0.0009713148511720993, 
     0.0009060922630252124, 0.0008478557092579965, 0.0007956007132881027, 
     0.000748501151061447, 0.0007058720174580954, 0.0006671410425632753, 
     0.0006318268164582039, 0.000599521759255228, 0.0005698787389364973, 
     0.0005426004642319407, 0.0005174310091039943, 0.0004941489894106098, 
     0.0004725620309631739, 0.0004525022549549413, 0.00043382257081906985, 
     0.00041639361435811373, 0.00040010120493205183, 0.000384844222758443, 
     0.0003705328282246134, 0.0003570869611682771, 0.0003444350705373149, 
     0.0003325130345641466, 0.00032126323923152697, 0.0003106337888470764, 
     0.00030057782734611957, 0.00029105295278105466, 0.0002820207105398023, 
     0.0002734461533263889, 0.0002652974579572625, 0.0002575455906736209, 
     0.0002501640140178496, 0.0002431284294299499, 0.0002364165506339585, 
     0.0002300079036415669, 0.00022388364982968206, 0.000218026429073904, 
     0.0002124202203596077, 0.00020705021766163429, 0.00020190271919476866, 
     0.00019696502840016398, 0.0001922253652557792, 0.00018767278668837097, 
     0.00018329711502608107, 0.00017908887356867756, 0.00017503922847076324, 
     0.00017113993623483624, 0.00016738329619853955, 0.00016376210747590783, 
     0.00016026962987769714, 0.00015689954839246884, 0.0001536459408592531, 
     0.00015050324850540461, 0.00014746624906058226, 0.00014453003219039822, 
     0.0001416899770218369, 0.00013894173155759145, 0.00013628119379847583, 
     0.00013370449441244472, 0.00013120798080583736, 0.00012878820246754955, 
     0.00012644189747018566, 0.00012416598002406825, 0.00012195752899047687, 
     0.00011981377726981382, 0.0001177321019886928, 0.00011571001541734519, 
     0.00011374515655533836, 0.0001118352833295007, 0.0001099782653532269, 
     0.00010817207720106667, 0.00010641479215674406, 0.00010470457639656514, 
     0.0001030396835736003, 0.00010141844977111262, 0.00009983928879648646, 
     0.00009830068778941977, 0.00009680120312041206, 0.00009533945655763061, 
     0.00009391413168209403};
 
  TVIL_REAL H000yxxtableB[] = 
     {0, -3.815750659947664, 1.3860794915520769, -0.7442121755235209, 
     0.4732952950546744, -0.33121908990023374, 0.246559341385313, 
     -0.19165419187768962, 0.15382658371297356, -0.12655383810005827, 
     0.10618224813076427, -0.09052723070491839, 0.07821323255630956, 
     -0.06833686583902354, 0.06028372371781432, -0.05362342261596523, 
     0.04804677314005938, -0.043326741526815364, 0.03929340131047899, 
     -0.035817405327431044, 0.03279880803063215, -0.030159339377459675, 
     0.02783695940144168, -0.025781952448958145, 0.023954081085427188, 
     -0.02232048216870204, 0.02085409104369139, -0.019532447039255357, 
     0.01833677795424868, -0.01725129119018108, 0.016262619689355426, 
     -0.015359385065653763, 0.014531850322964328, -0.013771641683524578, 
     0.013071524183421196, -0.012425219432156548, 0.011827256684500465, 
     -0.011272850416094646, 0.010757799125443862, -0.010278401241864833, 
     0.009831384900153166, -0.009413849018865446, 0.009023213641659478, 
     -0.008657177907651213, 0.00831368433492529, -0.007990888352100378, 
     0.00768713221148043, -0.007400922575300298, 0.007130911193558395, 
     -0.006875878193453713, 0.006634717583403414, -0.006406424640836586, 
     0.006190084908335398, -0.005984864566839887, 0.005790001991760767, 
     -0.005604800328087099, 0.005428620945828258, -0.005260877657989216, 
     0.005101031600934232, -0.0049485866913104234, 0.0048030855861428, 
     -0.004664106083969118, 0.004531257911399515, -0.004404179848558509, 
     0.0042825371552973335, -0.004166019258830159, 0.004054337676021036, 
     -0.003947224141446805, 0.0038444289192320624, -0.003745719277616605, 
     0.0036508781095444948, -0.0035597026825654327, 0.003472003500473435, 
     -0.00338760327935117, 0.003306336001665078, -0.0032280460659868115, 
     0.0031525875041811396, -0.0030798232810720014, 0.0030096246302368686, 
     -0.002941870439754017, 0.002876446775248429, -0.0028132462829347643, 
     0.002752167803630006, -0.0026931159144347114, 0.0026360005526263747, 
     -0.0025807366482548805, 0.0025272438330014683, -0.0024754460027217456, 
     0.0024252712212952977, -0.0023766513627206595, 0.002329521762142517, 
     -0.002283821256200233, 0.0022394917785079627, -0.0021964781608220657, 
     0.0021547280341566513, -0.0021141918554377845, 0.002074821717732135, 
     -0.0020365734522485093, 0.001999403404745914, -0.0019632713508622585, 
     0.001928137848442469};
 
  TVIL_REAL H000yxxtableD[] = 
     {0, -2.4041138063191885, 0.7290764556560972, 
     -0.3558339016213172, 0.2131309206453023, -0.1430448302260282, 
     0.10321102816893954, -0.07829238998087833, 0.06160917076709504, 
     -0.049859756071874275, 0.04125420770032101, -0.0347510905994573, 
     0.029709679247851464, -0.02571752373734991, 0.022498984197721, 
     -0.019863948949923035, 0.01767771843766348, -0.01584260907888949, 
     0.014286329121345505, -0.012954414057363204, 0.011805174910503376, 
     -0.010806246537495163, 0.009932180550705015, -0.00916273569417068, 
     0.008481643350620629, -0.007875702669067643, 0.007334108177865795, 
     -0.006847943865816088, 0.006409798120285363, -0.006013467531674084, 
     0.005653726813179585, -0.0053261484470570756, 0.005026960110539579, 
     -0.004752931075885328, 0.004501281027200493, -0.004269606363685611, 
     0.004055820248733429, -0.003858103542888962, 0.0036748644134373763, 
     -0.003504704905559663, 0.0033463931329787843, -0.0031988400308526725, 
     0.003061079832772394, -0.002932253603425778, 0.0028115952907869558, 
     -0.0026984198654747696, 0.0025921131968162296, -0.0024921233801354636, 
     0.002397953281632033, -0.0023091541087840385, 0.002225319847709514, 
     -0.002146082436038393, 0.0020711075619000664, -0.002000090997641621, 
     0.001932755391660922, -0.0018688474538978656, 0.0018081354805756648, 
     -0.0017504071721185666, 0.0016954677051096616, -0.001643138024946173, 
     0.0015932533307041125, -0.0015456617278044997, 0.0015002230275133466, 
     -0.0014568076752162762, 0.0014152957918750055, -0.0013755763151699772, 
     0.0013375462286211639, -0.001301109868507039, 0.001266178299710901, 
     -0.0012326687527481417, 0.00120050411519604, -0.0011696124715828446, 
     0.0011399266865150983, -0.0011113840264479723, 0.0010839258160467889, 
     -0.0010574971255607301, 0.0010320464860419064, -0.0010075256296029832, 
     0.0009838892522215882, -0.000961094796875874, 0.0009391022550381209, 
     -0.0009178739847665729, 0.0008973745438236562, -0.0008775705364145918, 
     0.0008584304722870312, -0.0008399246370621306, 0.00082202497278255, 
     -0.0008047049677650309, 0.0007879395549360493, -0.0007717050179099265, 
     0.0007559789041408916, -0.0007407399445449806, 0.0007259679790452028, 
     -0.0007116438875449295, 0.0006977495258806191, -0.0006842676663464204, 
     0.0006711819424204092, -0.0006584767973556748, 0.0006461374363296144, 
     -0.0006341497818719478, 0.00062250043231648};

  lnyox = TVIL_CLOG(y/x);
  lnyox2 = lnyox  * lnyox;
  lnyox3 = lnyox2  * lnyox;

  if (y < 0.5 * x) {
    result = TVIL_H0000xx (x,qq);
    rn = r = y/x;
    for (n=1; n < 100; n++) {
      result += rn * (H000yxxtableA[n] + 
                      H000yxxtableAL[n] * lnyox + 
                      H000yxxtableAL2[n] * lnyox2);
      rn *= r;
    }
  }
  else if (y < 1.5 * x) {
    result = TVIL_H000xxx (x,qq);
    rn = r = y/x - 1.L;
    for (n=1; n < 100; n++) {
      result += rn * H000yxxtableD[n];
      rn *= r;
    }
  }
  else if (y < 6.47 * x) {
    result = -7.1562457830666209564090640 - 6.L * Zeta3 * TVIL_Ap(x,qq);
    rn = r = y/(4.L*x) - 1.L;
    for (n=1; n < 100; n++) {
      result += rn * H000yxxtableB[n];
      rn *= r;
    }
  }
  else {
    result = TVIL_H00000x (y, qq);
    rn = r = 4.L * x/y;
    for (n=1; n < 100; n++) {
      result += rn * (H000yxxtableC[n] 
                     -H000yxxtableCL[n] * lnyox 
                     +H000yxxtableCL2[n] * lnyox2
                     -H000yxxtableCL3[n] * lnyox3);
      rn *= r;
    }
  }

  return result;
}

/* ****************************************************************** */

TVIL_COMPLEX TVIL_H00yyxx (TVIL_REAL x, TVIL_REAL y, TVIL_REAL qq)
{
  TVIL_REAL r, rn;
  TVIL_COMPLEX lnyox, lnyox2, lnyox3, result;
  int n;

  TVIL_REAL H00yyxxtableA[] = 
     {0, -1.6765777138248021, -0.21839146157457381, 
     -0.05407439970406864, -0.01361904911154041, -0.004358300185905051, 
     -0.0025322703936159637, -0.0019379690265873268, -0.00146189299575922, 
     -0.001058634296486741, -0.0007603352052413285, -0.0005551093150004207, 
     -0.0004154402885480543, -0.0003182960073345397, -0.0002486734691895623, 
     -0.00019744455984393333, -0.0001589606677725758, 
     -0.00012956074427410039, -0.00010677080091643899, 
     -0.0000888709522340875, -0.00007464227774823089, 
     -0.00006320734064551756, -0.000053925344961631786, 
     -0.00004632182044595929, -0.000040040861966035464, 
     -0.00003481227178521221, -0.000030428635507141785, 
     -0.000026729099127805053, -0.000023587731983671534, 
     -0.00002090507442736556, -0.000018601927760258036, 
     -0.000016614742773641378, -0.000014892161200088038, 
     -0.000013392397580612108, -0.000012081239930416782, 
     -0.000010930510348191122, -9.916870549398307e-6, 
     -9.020888246721955e-6, -8.22630236385433e-6, -7.519440951433267e-6, 
     -6.8887572114601285e-6, -6.324457489875912e-6, -5.818201341172599e-6, 
     -5.36285841705174e-6, -4.952310416677747e-6, -4.581288968134727e-6, 
     -4.245242311552319e-6, -3.94022518508832e-6, -3.6628074931409203e-6, 
     -3.409998248281517e-6, -3.179181988468314e-6, -2.9680654268267196e-6, 
     -2.7746325284370706e-6, -2.5971065541389023e-6, 
     -2.433917885817874e-6, -2.2836766666063842e-6, -2.145149464882049e-6, 
     -2.0172393121246496e-6, -1.8989685787462963e-6, 
     -1.7894642445104588e-6, -1.687945195456873e-6, -1.593711240771345e-6, 
     -1.506133593478871e-6, -1.4246466003313057e-6, 
     -1.3487405405052266e-6, -1.2779553410770266e-6, 
     -1.211875080786385e-6, -1.1501231732090595e-6, 
     -1.0923581368395444e-6, -1.0382698733037119e-6, 
     -9.875763864438912e-7, -9.400208847212106e-7, -8.953692175704899e-7, 
     -8.534076032742397e-7, -8.139406118019194e-7, -7.767893710597251e-7, 
     -7.417899692562679e-7, -7.087920297277736e-7, -6.776574376803012e-7, 
     -6.482592009770552e-7, -6.204804293939139e-7, -5.942134187425014e-7, 
     -5.693588279650492e-7, -5.458249387789475e-7, -5.23526988724941e-7, 
     -5.023865695798681e-7, -4.823310840567374e-7, -4.6329325455221885e-7, 
     -4.4521067843163886e-7, -4.2802542497908334e-7, 
     -4.1168366969783446e-7, -3.961353621348486e-7, 
     -3.8133392383155795e-7, -3.67235973379825e-7, -3.538010758932242e-7, 
     -3.409915144957577e-7, -3.287720816877289e-7, -3.1710988867609035e-7, 
     -3.0597419095795047e-7, -2.953362286242961e-7, -2.851690800092079e-7};
 
  TVIL_REAL H00yyxxtableAL[] = 
     {0, 1.322467033424113, -0.13519185004538986, 
     -0.13291283845240337, -0.05878341421982013, -0.02194236701346516, 
     -0.008813430586005996, -0.004563628719167273, -0.0029308289080373514, 
     -0.002040823639541441, -0.001448616651406027, -0.0010416724437153004, 
     -0.0007638813042690972, -0.0005732153529895903, -0.00043969005249255144, 
     -0.00034376499761311076, -0.00027317581437441606, 
     -0.00022015874672748852, -0.0001796464857230483, 
     -0.00014822165161151193, -0.0001235173600659948, 
     -0.00010386007287052508, -0.00008804604929227362, 
     -0.00007519611165300691, -0.00006465899115847488, 
     -0.000055945883768872375, -0.000048685459338841044, 
     -0.00004259247136552521, -0.00003744553993193704, 
     -0.000033071213115399327, -0.000029332388244261965, 
     -0.000026119802298237992, -0.000023345710297468504, 
     -0.000020939141805838723, -0.000018842308079892767, 
     -0.00001700785673684477, -0.00001539675664184748, 
     -0.000013976655643790253, -0.000012720596089895714, 
     -0.00001160600321390353, -0.00001061388321133023, 
     -9.728183596343034e-6, -8.935280000961471e-6, -8.22356212399591e-6, 
     -7.583097901843188e-6, -7.005359748142812e-6, -6.483000316696065e-6, 
     -6.009667985695929e-6, -5.579854361364131e-6, -5.188767716323178e-6, 
     -4.832227530708864e-6, -4.506576279738895e-6, -4.208605375489212e-6, 
     -3.935492771972097e-6, -3.6847502182242095e-6, -3.454178522056139e-6, 
     -3.241829488803432e-6, -3.0459734412846406e-6, 
     -2.8650714218883325e-6, -2.697751335100753e-6, -2.542787416497951e-6, 
     -2.3990825182445667e-6, -2.265652786167562e-6, 
     -2.1416143732170093e-6, -2.026171891526878e-6, -1.918608352684802e-6, 
     -1.8182763850779283e-6, -1.724590549798527e-6, 
     -1.6370206037714721e-6, -1.5550855814774255e-6, 
     -1.478348585677153e-6, -1.4064121935329684e-6, -1.338914397993776e-6, 
     -1.2755250156863435e-6, -1.2159425021865208e-6, 
     -1.1598911237171764e-6, -1.107118441271512e-6, 
     -1.0573930690868397e-6, -1.0105026744572452e-6, 
     -9.662521902086778e-7, -9.244622148793639e-7, -8.849675788459253e-7, 
     -8.47616057389814e-7, -8.122672140756797e-7, -7.787913598685022e-7, 
     -7.47068615196605e-7, -7.16988063712486e-7, -6.884469878462548e-7, 
     -6.613501774157051e-7, -6.356093035768169e-7, -6.111423512893152e-7, 
     -5.878731042513233e-7, -5.657306769401445e-7, -5.446490889955886e-7, 
     -5.24566877709059e-7, -5.054267448452679e-7, -4.8717523443211e-7, 
     -4.6976243851489793e-7, -4.531417281899305e-7, -4.372695075144419e-7, 
     -4.2210498813993396e-7};
 
  TVIL_REAL H00yyxxtableAL2[] = 
     {0, -0.125, -0.033854166666666664, -0.050868055555555555, 
     -0.03392275855654762, -0.016777266245039682, -0.0074700322960821, 
     -0.0034473175111254073, -0.0018160573437682585, -0.0010990018713499762, 
     -0.0007261945906792728, -0.000502892082440933, -0.0003590755164307726, 
     -0.00026323696690017757, -0.00019774677850814242, 
     -0.0001518336724436923, -0.00011881315051722492, 
     -0.00009450518645070483, -0.00007624501212150423, 
     -0.00006228617504930821, -0.000051451085762026684, 
     -0.00004292568584360509, -0.000036135311322259915, 
     -0.00003066693086291809, -0.000026218898634845626, 
     -0.000022567691876846213, -0.000019545460056173685, 
     -0.000017024611767075103, -0.000014907065784267077, 
     -0.000013116643900280318, -0.000011593613106384507, 
     -0.000010290719947233706, -9.170275145597257e-6, -8.2019869389195e-6, 
     -7.361334491550319e-6, -6.628335185038767e-6, -5.986602123684419e-6, 
     -5.422617537725636e-6, -4.925168251638152e-6, -4.48490384338559e-6, 
     -4.093988431334803e-6, -3.7458244507508233e-6, 
     -3.4348321783883017e-6, -3.1562727202062935e-6, 
     -2.9061051019336287e-6, -2.6808702811311026e-6, 
     -2.477596534778621e-6, -2.2937219125232703e-6, -2.127030386324369e-6, 
     -1.9755990475557487e-6, -1.8377542576315126e-6, 
     -1.7120350883469503e-6, -1.5971627233303284e-6, 
     -1.4920147546262725e-6, -1.3956035152447931e-6, 
     -1.3070577521709548e-6, -1.2256070744520668e-6, 
     -1.1505687149026943e-6, -1.0813362273279726e-6, 
     -1.017369808310828e-6, -9.581879869103874e-7, -9.033604697039068e-7, 
     -8.525019645288477e-7, -8.052668356614022e-7, -7.613444672786691e-7, 
     -7.204552319051288e-7, -6.823469769449041e-7, -6.467919559924789e-7, 
     -6.135841429116026e-7, -5.825368760898972e-7, -5.534807881501636e-7, 
     -5.262619829999717e-7, -5.007404276493832e-7, -4.7678853090250147e-7, 
     -4.5428988497904687e-7, -4.331381494680206e-7, -4.132360598560076e-7, 
     -3.9449454528968485e-7, -3.7683194229339334e-7, 
     -3.6017329292438464e-7, -3.444497173573015e-7, 
     -3.2959785218452886e-7, -3.1555934683282107e-7, 
     -3.022804114563191e-7, -2.8971141049461785e-7, -2.778064968011774e-7, 
     -2.6652328186833537e-7, -2.558225382142004e-7, -2.45667930465396e-7, 
     -2.3602577207785863e-7, -2.2686480499404554e-7, 
     -2.1815599984612284e-7, -2.0987237458708112e-7, 
     -2.0198882967047463e-7, -1.9448199810906447e-7, 
     -1.8733010892690084e-7, -1.8051286268159526e-7, 
     -1.7401131787654714e-7, -1.6780778720914626e-7, 
     -1.6188574271258562e-7, -1.5622972894771553e-7};
 
  TVIL_REAL H00yyxxtableAL3[] = 
     {0, 0., -0.03125, -0.015625, -0.0048828125, -0.0009765625, 
     -0.000040690104166666664, 0.00006103515625, 0.0000286102294921875, 
     6.357828776041667e-6, -9.5367431640625e-8, -7.152557373046875e-7, 
     -3.228584925333659e-7, -7.450580596923828e-8, -2.66092164175851e-10, 
     7.885197798411052e-9, 3.7107383832335472e-9, 8.731149137020111e-10, 
     -8.084397349092695e-13, -9.822542779147625e-11, 
     -4.661160346586257e-11, -1.1098000825205374e-11, 
     -2.5837917664003644e-15, 1.2647660696529783e-12, 
     6.061817714453355e-13, 1.4548362514688052e-13, 
     -8.540177112501205e-18, -1.684455044584057e-14, 
     -8.12098404396571e-15, -1.960237527853792e-15, 
     -2.8912057932946785e-20, 2.2920033926343564e-16, 
     1.1102060839662115e-16, 2.691408688402937e-17, 
     -9.965093497109416e-23, -3.1720899855846545e-18, 
     -1.54198230526374e-18, -3.7507677695604487e-19, 
     -3.4828657370653795e-25, 4.4480004288573253e-20, 
     2.1684022770194775e-20, 5.288786041510921e-21, 
     -1.2309235305848625e-27, -6.303504008566205e-22, 
     -3.080120542555244e-22, -7.529183548468374e-23, 
     -4.3901824290560656e-30, 9.01098772638409e-24, 
     4.4116296706625125e-24, 1.0803991030193908e-24, 
     -1.5777218104420236e-32, -1.2975382544391074e-25, 
     -6.362927883684457e-26, -1.5607181601490176e-26, 
     -5.706459094480699e-35, 1.8799559236137542e-27, 
     9.231926444994863e-28, 2.267490705788212e-28, 
     -2.0753555435207283e-37, -2.7382832828845724e-29, 
     -1.346322612831054e-29, -3.3106293758140675e-30, 
     -7.583834521434113e-40, 4.006912532590069e-31, 1.972152262593352e-31, 
     4.8545286463836356e-32, -2.7828938229694307e-42, 
     -5.887021681439159e-33, -2.9002239164225352e-33, 
     -7.145479214374362e-34, -1.0249497339061519e-44, 
     8.680247636422237e-35, 4.279844320798244e-35, 1.0553040791009369e-35, 
     -3.78729314682383e-47, -1.2839532962674993e-36, 
     -6.335295869717903e-37, -1.5632548249953265e-37, 
     -1.4035441349407222e-49, 1.9045984418351786e-38, 
     9.403954806569747e-39, 2.321964149770308e-39, -5.215150653838431e-52, 
     -2.8325165080068e-40, -1.3993980366935176e-40, 
     -3.4573363259486905e-41, -1.9424162137297933e-54, 
     4.2223216624358976e-42, 2.0871703672269542e-42, 
     5.159297536965505e-43, -7.250338297776139e-57, 
     -6.3073829778841915e-44, -3.119412233627463e-44, 
     -7.714675416498028e-45, -2.7116458061828974e-59, 
     9.440326496503964e-46, 4.670994881082707e-46, 1.1557100736699481e-46, 
     -1.0160024943446761e-61, -1.4154529942674945e-47, 
     -7.006492321624092e-48};
 
  TVIL_REAL H00yyxxtableC[] = 
     {0, -0.6010284515797971, 0.282594290922158, 0.1572511455354056, 
     0.11214929547182594, 0.08547270215010111, 0.06734599847238176, 
     0.05459632625807288, 0.045343078455553056, 0.03841099679429158, 
     0.033069464180799577, 0.02885432058559024, 0.0254606951980384, 
     0.022681601573941935, 0.02037231159351678, 0.018429002944863077, 
     0.01677551415214321, 0.015354867618789247, 0.014123691485465991, 
     0.013048457238846073, 0.012102885394690932, 0.011266120816163554, 
     0.01052142621198462, 0.009855231403758036, 0.0092564312333445, 
     0.008715860077269932, 0.00822589367656774, 0.0077801440052158175, 
     0.007373222986023832, 0.007000557746103854, 0.006658244871282, 
     0.006342934465768774, 0.006051737203158884, 0.005782149266947894, 
     0.005531991324171614, 0.005299358591088891, 0.005082579729063083, 
     0.004880182817436278, 0.0046908670343461965, 0.004513478968919717, 
     0.004346992712664636, 0.004190493051269039, 0.004043161212913141, 
     0.003904262734827195, 0.003773137093050127, 0.003649188806294201, 
     0.0035318797773768603, 0.0034207226777846693, 0.00331527521483622, 
     0.0032151351483403353, 0.0031199359459419653, 0.0030293429845515783, 
     0.002943050220179081, 0.002860777260780626, 0.0027822667868818225, 
     0.0027072822731653693, 0.0026356059712251225, 0.0025670371195487085, 
     0.0025013903517029285, 0.0024384942778272775, 0.002378190218025707, 
     0.0023203310691952946, 0.002264780289332506, 0.002211410985486725, 
     0.0021601050933472998, 0.002110752638004292, 0.0020632510667558295, 
     0.002017504645980748, 0.0019734239150825295, 0.0019309251913633967, 
     0.0018899301204256616, 0.001850365267337885, 0.001812161744360196, 
     0.0017752548715081256, 0.0017395838666576797, 0.001705091562264597, 
     0.0016717241460951014, 0.0016394309236502122, 0.0016081641002159998, 
     0.0015778785806927025, 0.0015485317855501815, 0.0015200834814291268, 
     0.0014924956250595986, 0.0014657322193034237, 0.001439759180246728, 
     0.0014145442143754054, 0.0013900567049611385, 0.0013662676068701657, 
     0.0013431493490824848, 0.0013206757442766714, 0.001298821904895972, 
     0.0012775641651654463, 0.0012568800085786098, 0.0012367480004156816, 
     0.0012171477248948696, 0.0011980597265935223, 0.001179465455807922, 
     0.0011613472175492786, 0.001143688123899582, 0.0011264720494744904, 
     0.0011096835897618092};
 
  TVIL_REAL H00yyxxtableCL[] = 
     {0, 1.6449340668482262, 0.37998351671205655, 
     0.20063480627080374, 0.12003768178359925, 0.08147580275826301, 
     0.060241288178809, 0.046989007015476425, 0.0380022679028408, 
     0.031561885064745776, 0.026757378070143004, 0.0230599625467822, 
     0.020142407081275415, 0.017792256453371652, 0.015866136959474218, 
     0.014264134642395074, 0.012914702712446307, 0.011765398794419317, 
     0.010776997840577616, 0.009919635906289184, 0.009170216866959712, 
     0.008510627585966414, 0.007926483977847629, 0.007406233704367805, 
     0.006940503361526094, 0.006521616377362859, 0.006143232100473775, 
     0.005800072234517048, 0.005487711102169146, 0.0052024131493617525, 
     0.004941005823542665, 0.004700779228440114, 0.0044794062517380185, 
     0.00427487849273745, 0.004085454490196159, 0.00390961760393677, 
     0.003746041531119278, 0.0035935619036954114, 0.0034511527623442783, 
     0.003317906965702864, 0.0031930197944130398, 0.003075775163549334, 
     0.0029655339760741046, 0.00286172424265404, 0.002763832665782988, 
     0.002671397443390135, 0.002584002092490311, 0.002501270129611296, 
     0.002422860473729801, 0.0023484634608066093, 0.00227779737791943, 
     0.0022106054403682527, 0.0021466531476874635, 0.002085725964801246, 
     0.002027627283043815, 0.0019721766227817563, 0.001919208045198217, 
     0.0018685687456484119, 0.0018201178050492551, 0.0017737250791651318, 
     0.0017292702085114557, 0.0016866417340109816, 0.0016457363055804686, 
     0.001606457972559248, 0.001568717546367217, 0.0015324320270395909, 
     0.0014975240863637905, 0.00146392160126873, 0.0014315572319121702, 
     0.0014003680395974461, 0.001370295140243308, 0.0013412833896435865, 
     0.0013132810971985735, 0.0012862397651871166, 0.001260113850985761, 
     0.0012348605499357837, 0.0012104395968165977, 0.0011868130841097874, 
     0.001163945295436282, 0.001141802552723512, 0.0011203530758130229, 
     0.0010995668533545716, 0.00107941552395257, 0.0010598722666368215, 
     0.0010409116998235872, 0.0010225097880165288, 0.0010046437555714044, 
     0.0009872920069145421, 0.0009704340526641358, 0.0009540504411561187, 
     0.0009381226949234969, 0.0009226332517202325, 0.0009075654097186064, 
     0.0008929032765429574, 0.0008786317218332244, 0.0008647363330591948, 
     0.0008512033743311026, 0.0008380197479745568, 0.0008251729586579167, 
     0.0008126510798784676, 0.0008004427226302202};
 
  TVIL_REAL H00yyxxtableCL2[] = 
     {0, 0., 0.03125, -0.005208333333333333, -0.00634765625, 
     -0.004524739583333334, -0.003384060329861111, -0.0027035667782738097, 
     -0.0022446632385253905, -0.0019080136818860575, -0.0016494253703526088, 
     -0.0014449328743190131, -0.0012796787633782341, -0.0011437823021103585, 
     -0.0010303866227240543, -0.0009345818532912469, -0.0008527623248405161, 
     -0.0007822235648795093, -0.0007209000237631731, -0.0006671890981718396, 
     -0.0006198299910770583, -0.0005778184697127502, -0.0005403457251190087, 
     -0.0005067537705975955, -0.0004765024094207082, -0.00044914443494116344, 
     -0.0004243067794198416, -0.0004016760216961834, -0.00038098712959953445, 
     -0.0003620146310441959, -0.0003445656283038915, -0.0003284742250833588, 
     -0.000313597046534298, -0.0002998096120702777, -0.00028700337895937444, 
     -0.00027508331750047114, -0.00026396591045219043, 
     -0.0002535774933052214, -0.00024385287010205968, 
     -0.00023473415333356498, -0.00022616978707427028, 
     -0.00021811372075392915, -0.0002105247073849096, 
     -0.00020336570510517211, -0.0001966033648761742, 
     -0.00019020759033545586, -0.00018415115832732745, 
     -0.00017840939066085984, -0.00017295986927863639, 
     -0.00016778218834443431, -0.00016285773783670073, 
     -0.00015816951411690597, -0.00015370195366640332, 
     -0.00014944078678287136, -0.00014537290852196017, 
     -0.0001414862645806416, -0.00013776975016135102, 
     -0.00013421312014263252, -0.0001308069091225905, 
     -0.00012754236010404862, -0.00012441136076144264, 
     -0.0001214063863744396, -0.00011852044863644033, 
     -0.00011574704965104298, -0.00011308014051917101, 
     -0.00011051408399632126, -0.00010804362076528262, 
     -0.00010566383892638612, -0.00010337014635626387, 
     -0.00010115824562838979, -0.00009902411122532003, 
     -0.0000969639688043729, -0.00009497427630617659, 
     -0.00009305170671965282, -0.00009119313233809508, 
     -0.00008939561035945623, -0.00008765636970014698, 
     -0.00008597279890586236, -0.00008434243505546558, 
     -0.00008276295356498521, -0.00008123215880851938, 
     -0.00007974797548145032, -0.00007830844063900011, 
     -0.00007691169634992372, -0.00007555598291114622, 
     -0.00007423963257449897, -0.00007296106374147447, 
     -0.00007171877558617045, -0.00007051134307039271, 
     -0.00006933741231828295, -0.00006819569632088293, 
     -0.00006708497094377456, -0.00006600407121338867, 
     -0.0000649518878597776, -0.0000639273640956318, -0.00006292949261310818, 
     -0.00006195731278165073, -0.00006100990803144127, 
     -0.00006008640340843494, -0.00005918596328812786, 
     -0.00005830778923628428};
 
  TVIL_REAL H00yyxxtableCL3[] = 
     {0, 0., -0.020833333333333332, -0.010416666666666666, 
     -0.005859375, -0.00390625, -0.002875434027777778, 
     -0.0022379557291666665, -0.0018068949381510417, -0.001498752170138889, 
     -0.0012694040934244791, -0.0010932286580403645, -0.0009544094403584799, 
     -0.0008427103360493978, -0.0007512434607460385, -0.0006752201666434606, 
     -0.0006112179738314202, -0.0005567322174708048, -0.0005098919295792088, 
     -0.0004692741931648925, -0.0004337795572913213, -0.0004025464432555646, 
     -0.0003748911015973687, -0.0003502647019217875, -0.0003282221497331348, 
     -0.0003083990802271804, -0.0002904946501225471, -0.0002742585039400075, 
     -0.0002594807880014998, -0.00024598441820151973, 
     -0.00023361903419539886, -0.00022225622929497403, 
     -0.00021178575518066673, -0.00020211247853324748, 
     -0.00019315392275188656, -0.00018483826867857103, 
     -0.00017710271818516937, -0.00016989214668607837, 
     -0.00016315798726429919, -0.00015685730165288962, 
     -0.00015095200287119402, -0.00014540820164737366, 
     -0.0001401956544245943, -0.00013528729515658276, -0.0001306588365505912, 
     -0.000126288429136173, -0.00012215636869437375, -0.00011824484430053899, 
     -0.00011453772061105536, -0.00011102034913343315, 
     -0.00010767940411671288, -0.0001045027394289449, 
     -0.00010147926338445759, -0.00009859882897240149, 
     -0.00009585213734054488, -0.00009323065272103574, 
     -0.00009072652726095047, -0.00008833253445039406, 
     -0.0000860420100330771, -0.00008384879944542285, -0.0000817472109657953, 
     -0.00007973197386980717, -0.00007779820098446092, 
     -0.00007594135511603217, -0.00007415721889653282, 
     -0.00007244186765327069, -0.00007079164495708967, 
     -0.00006920314054868088, -0.00006767317038002666, 
     -0.00006619875854051017, -0.00006477712086527782, 
     -0.00006340565004773154, -0.00006208190209910676, 
     -0.000060803584016419915, -0.00005956854253603982, 
     -0.00005837475386407945, -0.00005722031428700075, 
     -0.000056103431576512606, -0.00005502241711222566, 
     -0.00005397567865377981, -0.00005296171370143068, 
     -0.00005197910339049738, -0.000051026506870744645, 
     -0.00005010265612679331, -0.0000492063512001047, 
     -0.00004833645577703761, -0.000047491893110992113, 
     -0.0000466716422497857, -0.000045874734542199196, 
     -0.000045100250400123994, -0.00004434731629497181, 
     -0.00004361510196900521, -0.00004290281784403727, 
     -0.000042209712611555683, -0.00004153507098977106, 
     -0.000040878211634388715, -0.00004023848519107402, 
     -0.00003961527247863754, -0.000039007982792919245, 
     -0.00003841605232221318, -0.00003783894266585343};
 
  TVIL_REAL H00yyxxtableB[] = 
    {0, -5.222598374646595, 2.127888411692993, -1.2146758037754257, 
     0.8025305529209008, -0.5763041123577907, 0.4369526928800926, 
     -0.34427148940483066, 0.27914627312197693, -0.2314466568906187, 
     0.1953575681416547, -0.1673299200058482, 0.14508949708937058, 
     -0.12712001212086788, 0.11237659893292108, -0.10011888288149962, 
     0.08980955854905914, -0.08105052915851566, 0.07354143985598029, 
     -0.06705201973931998, 0.06140319342085714, -0.056453908501857615, 
     0.05209177570196362, -0.04822630525807937, 0.04478394451246511, 
     -0.04170438634384649, 0.038937788115382455, -0.03644265219054563, 
     0.03418419336027544, -0.03213306890782122, 0.030264381730608855, 
     -0.028556891165652132, 0.026992383305344026, -0.02555516486443171, 
     0.02423165354772599, -0.023010044373298417, 0.021880036213790695, 
     -0.020832606404798726, 0.019859823967746364, -0.01895469404169211, 
     0.0181110276833789, -0.01732333239984922, 0.016586719712269563, 
     -0.01589682677879218, 0.015249749676905207, -0.014641986398056327, 
     0.014070387966657763, -0.013532116382546122, 0.013024608316321372, 
     -0.012545543672796698, 0.012092818288367887, -0.011664520150682152, 
     0.011258908629187223, -0.010874396287390998, 0.01050953291544164, 
     -0.010162991477704667, 0.009833555716559734, -0.00952010919241732, 
     0.009221625572369605, -0.008937160007072615, 0.00866584145832363, 
     -0.008406865859092364, 0.00815949000409171, -0.007923026082829358, 
     0.00769683677887134, -0.0074803308691068875, 0.007272959265407094, 
     -0.007074211448446321, 0.00688361224979512, -0.006700718943854902, 
     0.0065251186159199425, -0.006356425776732317, 0.006194280197433092, 
     -0.006038344941886925, 0.00588830457603326, -0.005743863536251568, 
     0.005604744640768086, -0.0054706877299174495, 0.005341448422639201, 
     -0.00521679697796538, 0.005096517251466728, -0.004980405737692748, 
     0.0048682706905836875, -0.004759931314666146, 0.004655217020582458, 
     -0.0045539667391588864, 0.004456028288799406, -0.004361257791509367, 
     0.004269519133314216, -0.004180683465249624, 0.004094628741466528, 
     -0.0040112392913229725, 0.00393040542262865, -0.003852023053471617, 
     0.0037759933702932466, -0.003702222510090079, 0.0036306212648124945, 
     -0.0035611048062024015, 0.0034935924294674675, -0.003428007314329603, 
     0.003364276302112124};
 
  TVIL_REAL H00yyxxtableD[] = 
    {0, -3.699337093989723, 1.2630449505870924, -0.6444769442108692, 
     0.39330075627434846, -0.265982939178367, 0.19233587442207264, 
     -0.14580276956759025, 0.11447718246156759, -0.09235628033406117, 
     0.07614014283716418, -0.06388971451620931, 0.054403567491575995, 
     -0.04690441981336844, 0.04087098612990412, -0.03594287518495315, 
     0.03186435475360727, -0.028449814651284447, 0.02556185044809435, 
     -0.02309695351129569, 0.02097592993168019, -0.019137341114599696, 
     0.017532922790526522, -0.016124327959132993, 0.014880773375404098, 
     -0.013777313745390908, 0.012793559123807806, -0.011912709905077347, 
     0.011120822509903875, -0.010406244754845893, 0.009759177479167633, 
     -0.009171331128589342, 0.008635654469841563, -0.008146118607753756, 
     0.007697543771862198, -0.0072854594494434315, 0.006905990716815238, 
     -0.006555765300985184, 0.00623183715600982, -0.005931623279741268, 
     0.0056528512099120005, -0.005393515183106845, 0.005151839358980914, 
     -0.00492624683635837, 0.0047153334405559445, -0.00451784545942531, 
     0.004332660661889913, -0.00415877205669987, 0.003995273947956462, 
     -0.0038413499231573786, 0.0036962624732923294, -0.0035593439961174576, 
     0.0034299889756682586, -0.0033076471652912034, 0.0031918176295164277, 
     -0.003082043523163625, 0.0029779075051241037, -0.002879027700051279, 
     0.002785054134323814, -0.002695665583604356, 0.0026105667784916365, 
     -0.0025294859224692023, 0.002452172482845019, -0.0023783952208601515, 
     0.002307940431790929, -0.002240610369816206, 0.0021762218357835833, 
     -0.0021146049088797245, 0.002055601805668001, -0.00199906585206615, 
     0.0019448605556510532, -0.0018928587672419703, 0.0018429419220650047, 
     -0.001794999351971658, 0.001748927661199482, -0.0017046301590453493, 
     0.0016620163435906452, -0.0016210014312886898, 0.0015815059278114867, 
     -0.0015434552360668494, 0.0015067792977477078, -0.0014714122651711759, 
     0.001437292200512989, -0.0014043607998498028, 0.0013725631396936482, 
     -0.0013418474439447674, 0.001312164869404915, -0.001283469308184717, 
     0.0012557172055059267, -0.0012288673915423934, 0.0012028809260651316, 
     -0.0011777209547652286, 0.0011533525762362634, -0.0011297427187166836, 
     0.0011068600258211739, -0.001084674750605703, 0.0010631586573720164, 
     -0.0010422849305868965, 0.0010220280901789278, -0.0010023639123727035, 
     0.0009832693552918779};
 
  TVIL_REAL H00yyxxtableE[] = 
    {0, -5.057914916112982, 2.0309901076844796, -1.1477577132154109, 
     0.7522846395124226, -0.5365763229645484, 0.4044152036727893, 
     -0.31693111361003995, 0.2557197366008313, -0.21106198130383858, 
     0.17739693912592752, -0.15134072101491577, 0.13073093991582158, 
     -0.114129585541043, 0.10054814218051718, -0.08928778057253206, 
     0.07984246141205208, -0.07183804295011059, 0.06499281477255808, 
     -0.05909122073136665, 0.05396594405199491, -0.049485434178298475, 
     0.04554505783962467, -0.04206071442932211, 0.038964158584423154, 
     -0.036199525627652725, 0.033720717662527516, -0.03148941418902828, 
     0.029473541778935547, -0.027646085221481098, 0.02598415547599118, 
     -0.02446825273599178, 0.023081679140962777, -0.021810067282958708, 
     0.02064099905526086, -0.01956369553147044, 0.018568763097822752, 
     -0.0176479844405331, 0.01679414553002532, -0.01600089166894265, 
     0.015262607141102839, -0.014574314129694847, 0.013931587449273518, 
     -0.01333048231937687, 0.012767472943693218, -0.012239400081829676, 
     0.011743426136570667, -0.01127699654750267, 0.010837806496822519, 
     -0.010423772106384545, 0.010033005445320996, -0.00966379278166916, 
     0.009314575604641417, -0.008983934020620718, 0.008670572188913869, 
     -0.008373305515327391, 0.00809104936479589, -0.007822809090227227, 
     0.007567671204744659, -0.00732479554965957, 0.007093408331653429, 
     -0.006872795920476452, 0.006662299313547191, -0.006461309186622667, 
     0.006269261460580447, -0.006085633323622978, 0.005909939656137147, 
     -0.005741729812230454, 0.005580584717795917, -0.0054261142499773376, 
     0.005277954867237642, -0.0051357674629780555, 0.004999235418901153, 
     -0.004868062837128738, 0.004741972932537248, -0.004620706568910668, 
     0.004504020924377603, -0.004391688273232461, 0.0042834948726723915, 
     -0.004179239944238719, 0.004078734740857333, -0.003981801691346601, 
     0.0038882736151209965, -0.0037979930005782995, 0.003710811341330739, 
     -0.0036265885250365048, 0.003545192270117298, -0.0034664976061180275, 
     0.0033903863938836176, -0.003316746882101229, 0.003245473297089494, 
     -0.003176465463014228, 0.0031096284499766394, -0.003044872247658927, 
     0.0029821114624263897, -0.0029212650359776364, 0.002862255983807507, 
     -0.0028050111519030882, 0.00274946099023358, -0.0026955393417214093, 
     0.0026431832454963636};

  lnyox = TVIL_CLOG(y/x);
  lnyox2 = lnyox  * lnyox;
  lnyox3 = lnyox2  * lnyox;

  if (y < 0.1545 * x) {
    result = TVIL_H0000xx (x,qq);
    rn = r = 4.L * y/x;
    for (n=1; n < 100; n++) {
      result += rn * (H00yyxxtableA[n] + 
                      H00yyxxtableAL[n] * lnyox + 
                      H00yyxxtableAL2[n] * lnyox2 + 
                      H00yyxxtableAL3[n] * lnyox3);
      rn *= r;
    }
  }
  else if (y < 0.5 * x) {
    result = -11.883106027610596939339217L - 6.L * Zeta3 * TVIL_Ap(y,qq);
    rn = r = 0.25L * x/y - 1.L;
    for (n=1; n < 100; n++) {
      result += rn * H00yyxxtableE[n];
      rn *= r;
    }
  }
  else if (y < 1.6 * x) {
    result = TVIL_H00xxxx (x,qq);
    rn = r = y/x - 1.L;
    for (n=1; n < 100; n++) {
      result += rn * H00yyxxtableD[n];
      rn *= r;
    }
  }
  else if (y < 6.47 * x) {
    result = -12.1311341137951050868474L - 6.L * Zeta3 * TVIL_Ap(x,qq);
    rn = r = y/(4.L*x) - 1.L;
    for (n=1; n < 100; n++) {
      result += rn * H00yyxxtableB[n];
      rn *= r;
    }
  }
  else {
    result = TVIL_H00xx00 (y, qq);
    rn = r = 4.L * x/y;
    for (n=1; n < 100; n++) {
      result += rn * (H00yyxxtableC[n] 
                     -H00yyxxtableCL[n] * lnyox 
                     +H00yyxxtableCL2[n] * lnyox2
                     -H00yyxxtableCL3[n] * lnyox3);
      rn *= r;
    }
  }

  return result;
}

/* ****************************************************************** */

TVIL_COMPLEX TVIL_H00yxxx (TVIL_REAL x, TVIL_REAL y, TVIL_REAL qq)
{
  TVIL_REAL r, rn;
  TVIL_COMPLEX lnyox, lnyox2, lnyox3, result;
  int n;


  TVIL_REAL H00yxxxtableA[] = 
     {0, -1.9265606588002084, -0.5755739672502368, 
     -0.23428149218953095, -0.11443845225497752, -0.06319567089933167, 
     -0.038122719406037474, -0.024578734915809416, -0.016686817954433275, 
     -0.011805182713655602, -0.008636376635237341, -0.006496131552550613, 
     -0.0050018026351498195, -0.003928677103461428, -0.0031391972552154376, 
     -0.002546104657894622, -0.0020923121920952, -0.001739449084641916, 
     -0.0014611027331012363, -0.0012387029774744397, -0.0010589440184318576, 
     -0.0009121283559881976, -0.0007910786624915002, -0.0006904080577794791, 
     -0.0006060215713503199, -0.0005347697350494995, -0.0004742041241677966, 
     -0.0004224023689512053, -0.00037784123862377825, -0.0003393034665547055, 
     -0.00030580857114627944, -0.00027656095149153966, 
     -0.00025091056157899584, -0.00022832284127868735, 
     -0.00020835552757524106, -0.00019064062750895892, 
     -0.00017487029757759552, -0.0001607857040617032, -0.0001481681757551636, 
     -0.00013683213259297418, -0.00012661939962448242, 
     -0.00011739460879786486, -0.00010904146026820544, 
     -0.0001014596668859756, -0.00009456244477113952, 
     -0.00008827444273837609, -0.00008253002620600431, 
     -0.00007727184884263835, -0.00007244965886592957, 
     -0.00006801929755788821, -0.00006394185590990689, 
     -0.00006018296188931538, -0.00005671217602928101, 
     -0.000053502477189724235, -0.00005052982365120537, 
     -0.000047772777364987996, -0.00004521218132846984, 
     -0.00004283088179274703, -0.00004061348842151862, 
     -0.00003854616667304711, -0.000036616457620748895, 
     -0.000034813121203702816, -0.000033125999538064085, 
     -0.000031545897449643893, -0.000030064477827218045, 
     -0.000028674169761878515, -0.000027368087743162585, 
     -0.00002613996043847228, -0.000024984067797099864, 
     -0.000023895185401055823, -0.00002286853513761152, 
     -0.000021899741397732945, -0.00002098479211426732, 
     -0.000020120004047038828, -0.000019301991801548394, 
     -0.000018527640135936758, -0.000017794079169072268, 
     -0.00001709866215257404, -0.000016438945512536542, 
     -0.000015812670903737713, -0.000015217749051073717, 
     -0.00001465224518061214, -0.000014114365866619445, 
     -0.000013602447141725728, -0.000013114943735489138, 
     -0.000012650419322392883, -0.000012207537674072695, 
     -0.000011785054622606885, -0.000011381810752239078, 
     -0.000010996724746146138, -0.000010628787322982311, 
     -0.000010277055705072608, -9.940648566420599e-6, 
     -9.618741414247287e-6, -9.310562362682377e-6, -9.01538826156864e-6, 
     -8.732541147184333e-6, -8.461384985098721e-6, -8.201322678404682e-6, 
     -7.951793317266094e-6, -7.712269648116176e-6};
 
  TVIL_REAL H00yxxxtableAL[] = 
     {0, 1.870926791185962, 0.7905293287447546, 0.4178656886468093, 
     0.25261201313306086, 0.16704736422986624, 0.11775132275132275, 
     0.08704317315162612, 0.06674196428689216, 0.052679771174271695, 
     0.042566991037665036, 0.035067319424270375, 0.029361090443070684, 
     0.024924240361488485, 0.021409726061489526, 0.018580707939409247, 
     0.01627128546900968, 0.014362569985787518, 0.012767633577328822, 
     0.01142177297710698, 0.01027605339206093, 0.009292931385002423, 
     0.008443227621109904, 0.0077039951893356886, 0.007056993840293377, 
     0.006487581529565879, 0.005983898077903946, 0.005536256387154409, 
     0.005136683166827636, 0.004778568732314183, 0.004456397312694259, 
     0.0041655374366561665, 0.00390207760733637, 0.003662696442415153, 
     0.003444559276034419, 0.0032452352471503657, 0.0030626303725266282, 
     0.0028949331837828593, 0.002740570308532201, 0.0025981699736339348, 
     0.002466531858898448, 0.0023446020713108165, 0.002231452271072875, 
     0.002126262181852344, 0.0020283048734430365, 0.0019369343265337735, 
     0.0018515748845829062, 0.001771712272974501, 0.001696885925261613, 
     0.00162668240384479, 0.0015607297405275417, 0.0014986925530590667, 
     0.0014402678185776883, 0.0013851812050152588, 0.0013331838779576462, 
     0.001284049713914687, 0.0012375728620195102, 0.001193565605308533, 
     0.0011518564802963191, 0.0011122886198449825, 0.0010747182895680235, 
     0.001039013592393233, 0.0010050533195873922, 0.000972725929641885, 
     0.0009419286390315582, 0.0009125666110713209, 0.0008845522309721502, 
     0.0008578044567961199, 0.000832248237372794, 0.0008078139894051456, 
     0.0007844371269923627, 0.000762057637655812, 0.0007406196996934674, 
     0.0007200713363269235, 0.000700364102656679, 0.0006814528019200572, 
     0.0006632952279614811, 0.00064585193118662, 0.0006290860055865842, 
     0.0006129628946942597, 0.000597450214575208, 0.0005825175921669173, 
     0.000568136517465431, 0.000554280208221497, 0.0005409234859517326, 
     0.0005280426621972585, 0.0005156154340739536, 0.0005036207882576776, 
     0.0004920389126353611, 0.0004808511149314433, 0.00047003974768705534, 
     0.0004595881390326179, 0.00044948052874811987, 0.00043970200915501023, 
     0.00043023847042717193, 0.00042107654994783997, 0.00041220358537398874, 
     0.0004036075711016922, 0.00039527711785378254, 0.00038720141513673423, 
     0.0003793701963364085};
 
  TVIL_REAL H00yxxxtableC[] = 
     {0, -3.994196190694689, -0.16877976765409552, 
     0.024584969673348143, 0.016154659525827838, 0.006211446640891923, 
     0.002967858809826639, 0.0018697405005172973, 0.0012932787712492632, 
     0.0009158585906402067, 0.0006625264626614185, 0.000491061461932887, 
     0.0003723431576026828, 0.0002879422341382517, 0.0002265022975739228, 
     0.00018086000982089713, 0.00014634516232404796, 0.00011982788222487068, 
     0.00009916301375120994, 0.00008285104269024432, 0.00006982477609561037, 
     0.00005931212460961967, 0.00005074606172998867, 0.000043704412507801043, 
     0.00003786887277946636, 0.0000329966598936297, 0.000028900605794652795, 
     0.000025434983666653178, 0.000022485286543995145, 
     0.000019960767594646474, 0.000017788935152628318, 
     0.00001591144797460957, 0.000014281024766961684, 
     0.000012859096137226515, 0.000011614005344147636, 
     0.000010519618478363476, 9.554242763618261e-6, 8.699778644435234e-6, 
     7.941050637608827e-6, 7.265275879303895e-6, 6.661639472701273e-6, 
     6.1209532201183e-6, 5.635379865810643e-6, 5.1982091139343535e-6, 
     4.803674798402643e-6, 4.446804938091186e-6, 4.123298207143483e-6, 
     3.829421727836227e-6, 3.561926156442194e-6, 3.3179748573607018e-6, 
     3.095084604327061e-6, 2.891075752240153e-6, 2.7040302209928376e-6, 
     2.5322559477990734e-6, 2.3742567152463406e-6, 2.2287064627015253e-6, 
     2.094427349553607e-6, 1.97037096842272e-6, 1.855602211372496e-6, 
     1.749285377376645e-6, 1.650672178763901e-6, 1.5590913612072899e-6, 
     1.4739396984908255e-6, 1.3946741617279197e-6, 1.3208050944728078e-6, 
     1.2518902514996147e-6, 1.18752958091846e-6, 1.127360647555472e-6, 
     1.0710546107910143e-6, 1.0183126828524926e-6, 9.688630043214266e-7, 
     9.224578826870519e-7, 8.788713474452935e-7, 8.378969817361429e-7, 
     7.993459960265893e-7, 7.630455140389103e-7, 7.288370451266543e-7, 
     6.965751207219173e-7, 6.661260754081436e-7, 6.373669556881792e-7, 
     6.101845416808052e-7, 5.844744688433715e-7, 5.60140438427837e-7, 
     5.370935067698729e-7, 5.152514447172363e-7, 4.945381595509824e-7, 
     4.7488317266385185e-7, 4.5622114705346294e-7, 4.384914593800596e-7, 
     4.2163781194337295e-7, 4.0560788046249495e-7, 3.9035299400665864e-7, 
     3.7582784383214243e-7, 3.619902182386162e-7, 3.48800760873515e-7, 
     3.3622275019097544e-7, 3.242218980172563e-7, 3.1276616539148425e-7, 
     3.018255940425808e-7, 2.913721520334062e-7, 2.813795922541795e-7};
 
  TVIL_REAL H00yxxxtableCL[] = 
     {0, 2.677202369051736, 0.12745043366550945, 
     0.02358894237434983, 0.017563608977895193, 0.011527773019324608, 
     0.006888354551881095, 0.0041958195046097905, 0.0026922023724315395, 
     0.0018117478541301603, 0.001266736942310003, 0.0009140956751773092, 
     0.0006775937107235915, 0.0005140054674247774, 0.00039775031499920916, 
     0.00031315769398523587, 0.0002503181818081097, 0.00020278026160058715, 
     0.00016623273285730955, 0.0001377270817584512, 0.00011520455942920845, 
     0.00009720066308556067, 0.00008265593545102385, 0.00007079214917870354, 
     0.00006102964923299798, 0.00005293115847195446, 0.0000461629322237639, 
     0.000040467491579166584, 0.00003564421235648719, 0.00003153532526198472, 
     0.00002801569611855957, 0.000024985281343698435, 
     0.000022363499859287103, 0.00002008499345870448, 
     0.000018096403800205893, 0.00001635390117818418, 
     0.000014821274411297478, 0.000013468443216531575, 
     0.00001227029131847867, 0.000011205744947709595, 
     0.000010257040466039142, 9.409138773071628e-6, 8.649254382938984e-6, 
     7.96647464734079e-6, 7.3514502685701415e-6, 6.796142510548105e-6, 
     6.293615746563418e-6, 5.837866446072407e-6, 5.423681593439284e-6, 
     5.046520990905027e-6, 4.7024190311227175e-6, 4.387902409085607e-6, 
     4.099920937348833e-6, 3.835789175836082e-6, 3.5931370213096383e-6, 
     3.369867746934275e-6, 3.1641222585342863e-6, 2.974248555932638e-6, 
     2.7987755666126067e-6, 2.6363906637400256e-6, 2.4859202982603122e-6, 
     2.346313270771327e-6, 2.2166262474469146e-6, 2.0960111888265363e-6, 
     1.983704413475996e-6, 1.8790170624998587e-6, 1.7813267673567434e-6, 
     1.6900703537659484e-6, 1.6047374398026715e-6, 1.5248648074524136e-6, 
     1.4500314446571678e-6, 1.3798541698257328e-6, 1.3139837633787473e-6, 
     1.2521015415492865e-6, 1.1939163166851116e-6, 1.139161695964907e-6, 
     1.087593676967557e-6, 1.0389885041025104e-6, 9.931407546713296e-7, 
     9.498616274111477e-7, 9.089774098746335e-7, 8.703281040157107e-7, 
     8.337661919490502e-7, 7.991555260958651e-7, 7.66370329870669e-7, 
     7.352942967471555e-7, 7.058197770032507e-7, 6.778470427170033e-7, 
     6.512836226929418e-7, 6.260436999654794e-7, 6.020475653711724e-7, 
     5.792211214216146e-7, 5.574954313575795e-7, 5.368063088347585e-7, 
     5.17093944192474e-7, 4.98302563697976e-7, 4.803801185480338e-7, 
     4.6327800075314496e-7, 4.46950783333517e-7, 4.3135598252499235e-7, 
     4.1645383993157684e-7};
 
  TVIL_REAL H00yxxxtableCL2[] = 
     {0, -0.75, 0.046875, 0.0625, 0.028076171875, 0.01111328125, 
     0.004923502604166667, 0.002576015757865646, 0.0015124616168794179, 
     0.0009532989017547123, 0.0006333762123471215, 0.0004394824089191411, 
     0.00031601306464937, 0.0002339592324067171, 0.00017745938331241765, 
     0.00013739339000207296, 0.00010826536383348231, 0.00008663058890762606, 
     0.00007025832925695574, 0.00005766307463391525, 0.00004783108829762231, 
     0.000040055444402645585, 0.000033833575497431366, 
     0.00002880196956674451, 0.000024693565604888462, 0.00002130937008889447, 
     0.0000184991866440073, 0.000016148306202030137, 0.000014168168649137755, 
     0.000012489715952008591, 0.000011058597925762227, 9.831671676072257e-6, 
     8.774416464806984e-6, 7.859004367119131e-6, 7.062846135968492e-6, 
     6.367485115352961e-6, 5.757748628367552e-6, 5.221091627901759e-6, 
     4.747085182164027e-6, 4.3270149731690096e-6, 3.953564012614339e-6, 
     3.620560304651753e-6, 3.3227749446723e-6, 3.055759644725618e-6, 
     2.8157152729815764e-6, 2.599384935251608e-6, 2.4039665873901898e-6, 
     2.2270412746395218e-6, 2.06651393876883e-6, 1.9205643824551117e-6, 
     1.7876064813045271e-6, 1.6662541230483797e-6, 1.555292657379606e-6, 
     1.4536548785254388e-6, 1.3604007509515972e-6, 1.2747002378943535e-6, 
     1.1958187113438116e-6, 1.1231045172550038e-6, 1.0559783462207518e-6, 
     9.939241215276334e-7, 9.364811664841457e-7, 8.83237453539237e-7, 
     8.338237708666493e-7, 7.879086692461318e-7, 7.45194074389098e-7, 
     7.05411468256758e-7, 6.683185581389222e-7, 6.336963648901678e-7, 
     6.013466722296162e-7, 5.710897877820782e-7, 5.427625738798148e-7, 
     5.162167123054881e-7, 4.91317172341444e-7, 4.679408558642763e-7, 
     4.4597539692266224e-7, 4.2531809637236625e-7, 4.05874974807077e-7, 
     3.8755992929330085e-7, 3.70293981354796e-7, 3.540046053092119e-7, 
     3.3862512748016843e-7, 3.240941880282381e-7, 3.1035525819452597e-7, 
     2.97356206656204e-7, 2.850489094758348e-7, 2.733888988036043e-7, 
     2.623350460788735e-7, 2.5184927598760583e-7, 2.4189630787611536e-7, 
     2.324434217084661e-7, 2.2346024599259975e-7, 2.1491856539559044e-7, 
     2.0679214602705033e-7, 1.9905657659653924e-7, 1.916891238500764e-7, 
     1.846686008661116e-7, 1.7797524694570093e-7, 1.715906179678181e-7, 
     1.6549748620102533e-7, 1.596797486691277e-7, 1.5412234326266156e-7};
 
  TVIL_REAL H00yxxxtableCL3[] = 
     {0, 0.16666666666666666, 0.041666666666666664, 
     0.008680555555555556, 0.0013020833333333333, 0.00013020833333333333, 
     0.000013563368055555555, 5.8128720238095235e-6, 2.5431315104166665e-6, 
     7.064254195601851e-7, 1.2715657552083333e-7, 1.4449610854640152e-8, 
     1.655684577094184e-9, 7.64162112505008e-10, 3.54789552234468e-10, 
     1.034802860683865e-10, 1.9402553637822468e-11, 
     2.2826533691555846e-12, 2.6947991163642317e-13, 
     1.2764837919620045e-13, 6.063298011819522e-14, 1.804552979708191e-14, 
     3.445055688533819e-15, 4.1190883232469576e-16, 
     4.9343245538895844e-17, 2.3684757858670008e-17, 
     1.1386902816668272e-17, 3.4266142735344336e-18, 6.60847038467355e-19, 
     7.975740119433596e-20, 9.637352644315595e-21, 4.663235150475288e-21, 
     2.2587545260114674e-21, 6.844710684883235e-22, 1.328679132947922e-22, 
     1.6133960900081913e-23, 1.96072441494051e-24, 9.538659315926806e-25, 
     4.643820982753839e-25, 1.4139839530820986e-25, 2.757268708510092e-26, 
     3.3625228152562104e-27, 4.103078435282875e-28, 
     2.0038290032776833e-28, 9.791437175106862e-29, 
     2.9918280257270965e-29, 5.853576572074754e-30, 7.161290487112731e-31, 
     8.765121169122353e-32, 4.2931205726313566e-32, 2.103629080589365e-32, 
     6.444942036119378e-33, 1.2642001686234163e-33, 1.550434169066454e-34, 
     1.9021530314935662e-35, 9.33784215460478e-36, 4.5855474866362756e-36, 
     1.4078435265988565e-36, 2.7671407246943043e-37, 
     3.4003000430565605e-38, 4.179535469590356e-39, 2.055509247339519e-39, 
     1.0111779361912151e-39, 3.1097734148737763e-40, 
     6.122366410532747e-41, 7.535220197578766e-42, 9.276312743231436e-43, 
     4.5689301571139905e-43, 2.2508700038723336e-43, 
     6.932027185838709e-44, 1.3665996452082027e-44, 1.684189703601658e-45, 
     2.0759977249256548e-46, 1.0237796999633366e-46, 
     5.049724195765107e-47, 1.5569982936942412e-47, 3.073022948080739e-48, 
     3.791391948930782e-49, 4.678480449802408e-50, 2.3096295891429604e-50, 
     1.1403796096393368e-50, 3.5196901532078297e-51, 
     6.953534205117908e-52, 8.587195855717897e-53, 1.0606209167032522e-53, 
     5.240715117827835e-54, 2.5898882849730578e-54, 8.000373294097807e-55, 
     1.5818919922420666e-55, 1.95514740614188e-56, 2.4167794325920462e-57, 
     1.1951107084246382e-57, 5.9106018731870694e-58, 
     1.8272021919798735e-58, 3.6155277415771966e-59, 
     4.4718369435296907e-60, 5.53156913587657e-61, 2.7372713249698492e-61, 
     1.3546699924595682e-61, 4.1905826786943714e-62, 8.297353703814856e-63};
 
  TVIL_REAL H00yxxxtableB[] = 
     {0, -3.1849802808219985, 1.0599842568308924, 
     -0.5365616934027327, 0.32621540689142026, -0.22007141919879852, 
     0.15881575770684617, -0.12016860573872232, 0.09418017798574005, 
     -0.07584544258038849, 0.06241677863003516, -0.052281097696991116, 
     0.044439552345540756, -0.038246220850450256, 0.033268078828708796, 
     -0.029205855737543935, 0.025847259179058904, -0.02303825756385366, 
     0.020664861725203248, -0.018641232021714655, 0.016901717042267013, 
     -0.015395404039602637, 0.014082313644099582, -0.012930694758642723, 
     0.011915070198291739, -0.011014803826110267, 0.010213035853938639, 
     -0.009495881933957366, 0.00885182383991141, -0.008271241048932874, 
     0.007746047149327843, -0.007269405074980974, 0.0068355022080555826, 
     -0.006439371374714502, 0.006076747326854104, -0.005743950886193233, 
     0.005437794816641609, -0.005155506886366847, 0.004894666620941412, 
     -0.004653153030586458, 0.004429101186743059, -0.004220865975314439, 
     0.004026991701565868, -0.0038461864908068398, 0.003677300638690909, 
     -0.003519308229379869, 0.003371291469470975, -0.003232427288397708, 
     0.0031019758379765144, -0.002979270589444291, 0.0028637097792071882, 
     -0.0027547489972919066, 0.002651894747241261, -0.0025546988345507096, 
     0.0024627534639736, -0.0023756869451308305, 0.002293159921636583, 
     -0.0022148620520241703, 0.002140509081626032, -0.002069840253631027, 
     0.0020026160151333638, -0.0019386159803615108, 0.0018776371186436172, 
     -0.0018194921392002549, 0.001764008048696023, -0.0017110248607437033, 
     0.0016603944393326982, -0.0016119794605252964, 0.001565652478794334, 
     -0.001521295086117454, 0.0014787971534408711, -0.0014380561454163873, 
     0.0013989765004303871, -0.0013614690689086712, 0.0013254506037181375, 
     -0.0012908432972139013, 0.0012575743601140569, -0.001225575637937078, 
     0.001194783261220059, -0.0011651373261591135, 0.001136581602684394, 
     -0.001109063267308329, 0.0010825326583727027, -0.0010569430515732347, 
     0.001032250453863758, -0.0010084134140396404, 0.0009853928484750926, 
     -0.0009631518806441868, 0.0009416556931932587, -0.0009208713914549602, 
     0.0009007678774034315, -0.0008813157331474405, 0.0008624871131453016, 
     -0.0008442556444031419, 0.0008265963339876904, -0.0008094854832471661, 
     0.0007929006081898204, -0.0007768203655200078, 0.0007612244838768952, 
     -0.0007460936998616675, 0.0007314096984758184};
 
  TVIL_REAL H00yxxxtableD[] = 
     {0, -1.8198877365225914, 0.47915495005989944, 
     -0.21364882245175737, 0.1199004178119594, -0.0765221747384657, 
     0.0530090881372382, -0.038864870544442806, 0.029704765933973743, 
     -0.02343672572633592, 0.0189607537046217, -0.015653870698121738, 
     0.013141915365755222, -0.011189204959188152, 0.009641314231429039, 
     -0.008393645708050106, 0.00737331635083545, -0.0065282836800161104, 
     0.005820584593752799, -0.005222000991059738, 0.004711205610161743, 
     -0.004271837424938262, 0.003891176091279548, -0.0035592114765256415, 
     0.0032679792245609534, -0.0030110788581377114, 0.002783319283488974, 
     -0.0025804546119787084, 0.002398984929349546, -0.0022360043866614713, 
     0.0020890841909370836, -0.0019561816248017993, 0.0018355686825062975, 
     -0.0017257756336972044, 0.0016255460503082977, -0.0015338007109609749, 
     0.0014496084353051262, -0.0013721623685198745, 0.0013007605823972502, 
     -0.0012347901179284547, 0.0011737137889373441, -0.001117059213999151, 
     0.0010644096567988019, -0.0010153963420222794, 0.0009696919812618778, 
     -0.0009270052959818242, 0.0008870763658432039, -0.0008496726632481044, 
     0.0008145856608050338, -0.0007816279190350793, 0.0007506305781698305, 
     -0.0007214411912111062, 0.0006939218462025988, -0.000667947534427074, 
     0.0006434047283970315, -0.0006201901393709489, 0.0005982096289527091, 
     -0.0005773772533174312, 0.0005576144219107092, -0.0005388491552163632, 
     0.0005210154284813253, -0.000504052590206585, 0.0004879048458259385, 
     -0.00047252079835292813, 0.0004578530389241956, -0.0004438577811399385, 
     0.0004304945339282808, -0.000417725808364009, 0.000405516854472971, 
     -0.00039383542456776185, 0.00038265156010164493, -0.0003719373994072062, 
     0.0003616670040134068, -0.0003518162015173163, 0.00034236244323146675, 
     -0.00033328467503999763, 0.00032456322008122987, 
     -0.00031617967203495355, 0.00030811679793288745, 
     -0.00030035844953329126, 0.0002928894824080097, -0.00028569568198434636, 
     0.0002787636958668728, -0.0002720809718370701, 0.00026563570099287357, 
     -0.0002594167655468558, 0.00025341369085188803, -0.00024761660126750094, 
     0.00024201617951952206, -0.00023660362924052934, 0.00023137064040975293, 
     -0.00022630935743875088, 0.00022141234967388234, 
     -0.00021667258410865684, 0.00021208340011875696, 
     -0.00020763848605018578, 0.0002033318575068124, -0.00019915783719778852, 
     0.00019511103621806553, -0.00019118633664671456, 0.00018737887535808466};

  lnyox = TVIL_CLOG(y/x);
  lnyox2 = lnyox  * lnyox;
  lnyox3 = lnyox2  * lnyox;

  if (y < 0.5 * x) {
    result = TVIL_H000xxx (x,qq);
    rn = r = y/x;
    for (n=1; n < 100; n++) {
      result += rn * (H00yxxxtableA[n] + 
                      H00yxxxtableAL[n] * lnyox);
      rn *= r;
    }
  }
  else if (y < 1.5 * x) {
    result = TVIL_H00xxxx (x,qq);
    rn = r = y/x - 1.L;
    for (n=1; n < 100; n++) {
      result += rn * H00yxxxtableD[n];
      rn *= r;
    }
  }
  else if (y < 6.47 * x) {
    result = -9.34923561975823495052641L - 6.L * Zeta3 * TVIL_Ap(x,qq);
    rn = r = y/(4.L*x) - 1.L;
    for (n=1; n < 100; n++) {
      result += rn * H00yxxxtableB[n];
      rn *= r;
    }
  }
  else {
    result = TVIL_H00000x (y, qq);
    rn = r = 4.L * x/y;
    for (n=1; n < 100; n++) {
      result += rn * (H00yxxxtableC[n] 
                     -H00yxxxtableCL[n] * lnyox 
                     +H00yxxxtableCL2[n] * lnyox2
                     -H00yxxxtableCL3[n] * lnyox3);
      rn *= r;
    }
  }

  return result;
}

/* ****************************************************************** */

TVIL_COMPLEX TVIL_H0xxx0y (TVIL_REAL x, TVIL_REAL y, TVIL_REAL qq)
{
  TVIL_REAL r, rn;
  TVIL_COMPLEX lnyox, lnyox2, lnyox3, result;
  int n;

  TVIL_REAL H0xxx0ytableA[] = 
     {0, -1.8787030962832836, -0.5539687846855071, 
     -0.2301165183859894, -0.11687260735951034, -0.06757271034213204, 
     -0.04271222145478448, -0.028799354092528515, -0.020388877050949807, 
     -0.014994621827894053, -0.01136919315817922, -0.008838522430554014, 
     -0.007015905369333909, -0.005668307691761144, -0.004649397425124063, 
     -0.0038640182221505726, -0.003248404792528721, -0.0027586972419367077, 
     -0.00236402870524173, -0.0020422321684209583, -0.001777098891349566, 
     -0.0015565838814767547, -0.0013716050177582295, -0.0012152232905011278, 
     -0.0010820730382126705, -0.0009679594050302356, -0.0008695696566670318, 
     -0.0007842632921926097, -0.0007099175068014643, -0.0006448120751341776, 
     -0.0005875426689989546, -0.000536954928422479, -0.0004920938469102674, 
     -0.0004521645734422185, -0.00041650180735453224, -0.0003845457188603314, 
     -0.0003558228670971522, -0.00032993097578296846, -0.0003065267088202501, 
     -0.00028531579531808636, -0.00026604500682859907, 
     -0.0002484956040279118, -0.0002324779561455514, -0.00021782710166677393, 
     -0.00020439906859886288, -0.00019206781082011334, 
     -0.0001807226465808089, -0.0001702661082067362, -0.00016061213003119626, 
     -0.00015168451571890277, -0.0001434156373226147, 
     -0.00013574532729510697, -0.00012861993177070463, 
     -0.00012199149911910095, -0.00011581708235742831, 
     -0.00011005813771503926, -0.00010468000465844136, 
     -0.00009965145514139877, -0.00009494430185735868, 
     -0.00009053305692483573, -0.00008639463379991298, 
     -0.00008250808633826192, -0.00007885437986570633, 
     -0.0000754161898963665, -0.0000721777247889946, -0.00006912456917801146, 
     -0.00006624354547442723, -0.00006352259111828052, 
     -0.00006095064959070789, -0.000058517573470285096, 
     -0.00005621403805309371, -0.000054031464255845184, 
     -0.00005196194969193899, -0.00004999820695618369, 
     -0.00004813350827892348, -0.00004636163581769859, 
     -0.00004467683694700723, -0.00004307378398647251, 
     -0.00004154753787663457, -0.00004009351537126921, 
     -0.0000387074593669155, -0.00003738541203530544, 
     -0.00003612369046358319, -0.00003491886454139246, 
     -0.00003376773686378707, -0.00003266732444506985, 
     -0.00003161484206159086, -0.00003060768706166699, 
     -0.000029643425498488952, -0.00002871977945747523, 
     -0.000027834615463287524, -0.000026985933863872124, 
     -0.000026171859099640077, -0.00002539063077541997, 
     -0.000024640595461262195, -0.000023920199155674612, 
     -0.00002322798035154019, -0.00002256256365090725, 
     -0.000021922653880138607, -0.00002130703066163326, 
     -0.000020714543402558623};
 
  TVIL_REAL H0xxx0ytableAL[] = 
     {0, 1.870926791185962, 0.8261146020412241, 0.46510656718966487, 
     0.3001286726236158, 0.21093162361898615, 0.1570723684271729, 
     0.12193502712157067, 0.09766891267882406, 0.08016447430586512, 
     0.06709599864835374, 0.0570641834747387, 0.04918456364039419, 
     0.0428749581418737, 0.03773893505899829, 0.03349865809262183, 
     0.02995452547230397, 0.026960020949180125, 0.024405546017023466, 
     0.02220773787010763, 0.020302239432478567, 0.01863869976400596, 
     0.017177249778480858, 0.01588597461583338, 0.01473907223439889, 
     0.013715492699100246, 0.0127979195269155, 0.011971997958810446, 
     0.011225743852313477, 0.010549086310259078, 0.009933510451007016, 
     0.009371775950558039, 0.008857693476096393, 0.008385945751646473, 
     0.007951943325450515, 0.007551707532570056, 0.0071817749289435, 
     0.0068391187967065835, 0.0065210843120022, 0.006225334715346533, 
     0.0059498063947027455, 0.005692671228652995, 0.005452304874779928, 
     0.005227259950968606, 0.005016243262826154, 0.004818096392182757, 
     0.004631779089717952, 0.004456355016721046, 0.0042909794625927995, 
     0.004134888730321409, 0.003987390935197482, 0.0038478580050828157, 
     0.003715718705647877, 0.0035904525427310876, 0.003471584417597022, 
     0.0033586799303657196, 0.0032513412430327142, 0.0031492034269218795, 
     0.00305193123060891, 0.0029592162137225676, 0.0028707741998967406, 
     0.0027863430087704355, 0.0027056804325277226, 0.002628562427208837, 
     0.0025547814930491283, 0.002484145221531022, 0.002416474989761706, 
     0.002351604785295372, 0.0022893801466694367, 0.002229657206773967, 
     0.0021723018277683484, 0.002117188817637141, 0.002064201219670267, 
     0.002013229667187947, 0.0019641717967309974, 0.0019169317137213084, 
     0.001871419505281825, 0.0018275507955039442, 0.0017852463389746542, 
     0.0017444316488359598, 0.0017050366560537483, 0.0016669953969295306, 
     0.0016302457262027635, 0.0015947290533690787, 0.0015603901000854022, 
     0.0015271766767506274, 0.0014950394765436805, 0.0014639318853725154, 
     0.0014338098063403637, 0.0014046314974717397, 0.0013763574215622275, 
     0.0013489501071246523, 0.0013223740195013978, 0.0012965954412996624, 
     0.0012715823613845016, 0.001247304371734605, 0.0012237325715287756, 
     0.0012008394778878022, 0.0011785989427475225, 0.0011569860753849867, 
     0.0011359771701612597};
 
  TVIL_REAL H0xxx0ytableC[] = 
     {0, -1.0068196431612202, -0.01822939308325064, 
     0.00764502272173293, 0.005206433629285597, 0.0032474516601847603, 
     0.00212362952835584, 0.0014572855794514788, 0.0010407825413994997, 
     0.000768079257283113, 0.0005824479233060947, 0.0004518883357315523, 
     0.00035747787281277895, 0.0002875624527525238, 0.0002347076770102066, 
     0.00019402363340487256, 0.00016220660712038717, 0.0001369710845949393, 
     0.00011670234204199753, 0.00010023792398144479, 0.00008672670800075901, 
     0.00007553593771900134, 0.00006618863600469082, 0.000058320684259330346, 
     0.0000516508878031544, 0.00004595977359001529, 0.00004107435850603039, 
     0.00003685706301970555, 0.000033197543951370045, 0.00003000660997045003, 
     0.00002721164125971528, 0.000024753107874029118, 0.00002258189914631515, 
     0.00002065725774660468, 0.000018945168716404612, 0.00001741709384222571, 
     0.000016048970304577916, 0.000014820413132832512, 
     0.000013714075980951326, 0.000012715135739903771, 
     0.000011810874647217002, 0.0000109903396324939, 0.000010244063208593133, 
     9.563833680213952e-6, 8.942505081850676e-6, 8.3738392837246e-6, 
     7.852374269647982e-6, 7.373313807008211e-6, 6.932434679404911e-6, 
     6.5260083990850476e-6, 6.150734905922426e-6, 5.80368622761308e-6, 
     5.48225844888812e-6, 5.184130636446383e-6, 4.907229606799733e-6, 
     4.649699618535885e-6, 4.409876228142055e-6, 4.186263676919426e-6, 
     3.977515281475297e-6, 3.782416386394236e-6, 3.599869508591899e-6, 
     3.428881361429849e-6, 3.26855149521832e-6, 3.118062331099377e-6, 
     2.97667039896716e-6, 2.8436986182401925e-6, 2.718529483920958e-6, 
     2.600599040246285e-6, 2.4893915409893295e-6, 2.384434709643989e-6, 
     2.285295524735094e-6, 2.1915764657061387e-6, 2.1029121635321586e-6, 
     2.0189664076298934e-6, 1.9394294669906173e-6, 1.864015688909428e-6, 
     1.7924613433671268e-6, 1.724522685153085e-6, 1.6599742092969142e-6, 
     1.5986070783849977e-6, 1.5402277029437423e-6, 1.484656458332799e-6, 
     1.431726523557551e-6, 1.3812828291224372e-6, 1.333181102540516e-6, 
     1.2872870014200895e-6, 1.2434753251918756e-6, 1.2016292975419702e-6, 
     1.1616399124954727e-6, 1.123405337869197e-6, 1.086830370493145e-6, 
     1.051825938201313e-6, 1.0183086441230953e-6, 9.862003492759674e-7, 
     9.554277898758474e-7, 9.25922226150229e-7, 8.97619119766566e-7, 
     8.704578372794801e-7, 8.443813772594952e-7, 8.193361189969705e-7, 
     7.952715908810175e-7};
 
  TVIL_REAL H0xxx0ytableCL[] = 
     {0, 1.3977484741547672, 0.11616632758994828, 
     0.02403879403011605, 0.01030945947441241, 0.005789617864866462, 
     0.0036158796284056203, 0.0024073297172761744, 0.001680535839148054, 
     0.0012180856263854945, 0.0009103343739232275, 0.0006978345992218829, 
     0.0005465010625854822, 0.00043586393971095536, 0.00035313722630810415, 
     0.0002900598647840328, 0.0002411353825690768, 0.0002026114801541057, 
     0.00017186776982301087, 0.0001470370446834856, 0.00012676457270247824, 
     0.00011005136325803618, 0.00009614986424751271, 0.00008449317041064488, 
     0.00007464610920602469, 0.00006627088737929323, 0.000059102600730124835, 
     0.00005293153392749458, 0.000047590204809769916, 
     0.000042943769711445316, 0.00003888284025827921, 
     0.000035318050914909194, 0.00003217591166737446, 
     0.000029395613799806706, 0.000026926549343595596, 
     0.000024726369761271152, 0.00002275945553006332, 
     0.000020995701339942398, 0.000019409545546829737, 
     0.000017979190003167688, 0.00001668596927284802, 
     0.000015513837812984356, 0.0000144489508769794, 0.000013479320305061003, 
     0.00001259453048082803, 0.000011785502878337684, 
     0.000011044300046374561, 0.00001036396175270824, 9.738367472886878e-6, 
     9.162120553306763e-6, 8.630450280265045e-6, 8.139128800720763e-6, 
     7.684400408480343e-6, 7.262921163466938e-6, 6.871707176157811e-6, 
     6.5080901830961e-6, 6.1696792772639275e-6, 5.854327850462628e-6, 
     5.560104962619555e-6, 5.28527048215536e-6, 5.028253447748243e-6, 
     4.787633189421404e-6, 4.562122819359107e-6, 4.350554763024944e-6, 
     4.1518680512592415e-6, 3.965097135880085e-6, 3.789362026366067e-6, 
     3.62385957464426e-6, 3.467855759808705e-6, 3.320678845540486e-6, 
     3.1817133007349467e-6, 3.050394388894586e-6, 2.9262033446537717e-6, 
     2.808663066724254e-6, 2.6973342658869236e-6, 2.5918120146535643e-6, 
     2.491722652088859e-6, 2.3967210031902206e-6, 2.306487877315392e-6, 
     2.2207278145463427e-6, 2.139167052684611e-6, 2.0615516908737757e-6, 
     1.9876460287116723e-6, 1.917231062209742e-6, 1.8501031201315702e-6, 
     1.7860726261415273e-6, 1.7249629738552136e-6, 1.66660950333829e-6, 
     1.6108585688768103e-6, 1.557566688963939e-6, 1.5065997704351733e-6, 
     1.4578323995543258e-6, 1.4111471936205197e-6, 1.3664342073452763e-6, 
     1.3235903888495893e-6, 1.2825190806633672e-6, 1.2431295615821736e-6, 
     1.205336625656092e-6, 1.1690601949591193e-6, 1.1342249631202741e-6, 
     1.1007600668943909e-6};
 
  TVIL_REAL H0xxx0ytableCL2[] = 
     {0, -0.3333333333333333, -0.0030864197530864196, 
     0.010516689529035207, 0.006214118782705888, 0.0034894183926343658, 
     0.0020961352666095056, 0.001348082911578604, 0.0009161206017626423, 
     0.0006502824129760479, 0.00047797815125805274, 0.0003614896792071658, 
     0.00027994897259293547, 0.0002211836962866472, 0.00017777092042268309, 
     0.00014500985938675578, 0.00011982554571406118, 0.00010014916434847022, 
     0.0000845541365969785, 0.00007203512788883098, 0.00006186983124092799, 
     0.000053530228268441866, 0.0000466242670405837, 0.00004085671559423794, 
     0.000036002381608115345, 0.000031887472366254125, 
     0.000028376414345645103, 0.000025362397622663865, 
     0.000022760501679853383, 0.000020502636204230063, 
     0.000018533775145526996, 0.000016809123737896456, 
     0.000015291966340791976, 0.00001395201644545929, 
     0.000012764140795174908, 0.000011707364841159676, 
     0.000010764091630271417, 9.919483951975805e-6, 9.160972340921187e-6, 
     8.47786081499828e-6, 7.861009039539789e-6, 7.302574647736389e-6, 
     6.795803206306057e-6, 6.334856140816959e-6, 5.914669074162759e-6, 
     5.5308346624143975e-6, 5.179505263622394e-6, 4.857311741395087e-6, 
     4.5612954556447056e-6, 4.288851079245981e-6, 4.037678339916499e-6, 
     3.8057411502761863e-6, 3.591232877603511e-6, 3.392546734878899e-6, 
     3.2082504589887898e-6, 3.037064590220789e-6, 2.8778437869598273e-6, 
     2.729560706661524e-6, 2.5912920633103577e-6, 2.462206536257646e-6, 
     2.3415542584067597e-6, 2.2286576554088283e-6, 2.122903443630037e-6, 
     2.023735624570099e-6, 1.930649338286967e-6, 1.8431854591284433e-6, 
     1.760925834424015e-6, 1.6834890813462998e-6, 1.610526869395615e-6, 
     1.5417206262881592e-6, 1.4767786137610015e-6, 1.4154333272103053e-6, 
     1.3574391793706875e-6, 1.3025704336030682e-6, 1.2506193569343521e-6, 
     1.2013945669081439e-6, 1.1547195496639447e-6, 1.1104313295484359e-6, 
     1.068379273047991e-6, 1.0284240119763958e-6, 9.904364727061922e-7, 
     9.542969998384905e-7, 9.198945641002374e-7, 8.871260454700427e-7, 
     8.558955835893724e-7, 8.261139884369616e-7, 7.976982050492304e-7, 
     7.705708267740624e-7, 7.446596521630501e-7, 7.198972811497441e-7, 
     6.962207466384894e-7, 6.735711780485862e-7, 6.51893493729098e-7, 
     6.311361194869617e-7, 6.112507307605805e-7, 5.921920162275108e-7, 
     5.739174608622817e-7, 5.563871466623432e-7, 5.395635694397118e-7, 
     5.234114702357516e-7, 5.078976800590243e-7};
 
  TVIL_REAL H0xxx0ytableCL3[] = 
     {0, 0.07407407407407407, 0.01646090534979424, 
     0.004420057918000305, 0.0016257684296093074, 0.0007643369630871537, 
     0.00042139098120853386, 0.0002576005067344652, 0.00016914954645054923, 
     0.00011708486092503506, 0.00008441381307724471, 0.00006286824234705721, 
     0.000048081682513645694, 0.00003759565236930872, 
     0.000029952371231216134, 0.00002424953590691439, 0.00001990805899180423, 
     0.00001654457600918, 0.000013898373596357014, 0.000011787922237166506, 
     0.000010084168107111123, 8.693647963425543e-6, 7.547540856645542e-6, 
     6.594407021853605e-6, 5.795273302313984e-6, 5.120245082944576e-6, 
     4.5461312348769374e-6, 4.054753593181964e-6, 3.631726725013222e-6, 
     3.2655657469674727e-6, 2.9470261988885675e-6, 2.66861020921494e-6, 
     2.4241932637179234e-6, 2.2087394238837866e-6, 2.018082092133444e-6, 
     1.848753826119117e-6, 1.697853192598553e-6, 1.5629398320654665e-6, 
     1.4419511832656225e-6, 1.3331359644234242e-6, 1.2350007110068724e-6, 
     1.1462665559620002e-6, 1.065834096489697e-6, 9.927546841337105e-7, 
     9.262068465561477e-7, 8.654768316514376e-7, 8.0994248051793e-7, 
     7.590598019538623e-7, 7.123517498082691e-7, 6.693988047373953e-7, 
     6.298310404179033e-7, 5.933214160852133e-7, 5.5958008619347e-7, 
     5.283495569099835e-7, 4.994005502556027e-7, 4.7252846166587016e-7, 
     4.4755031687309116e-7, 4.243021503018499e-7, 4.026367404127743e-7, 
     3.824216482340161e-7, 3.635375141684003e-7, 3.4587657543662144e-7, 
     3.293413725145786e-7, 3.1384361788587026e-7, 2.99303204550365e-7, 
     2.856473351604702e-7, 2.7280975552212284e-7, 2.6073007859774554e-7, 
     2.493531871646298e-7, 2.3862870498048666e-7, 2.2851052774216599e-7, 
     2.189564063379391e-7, 2.0992757592452642e-7, 2.0138842523711296e-7, 
     1.9330620128859435e-7, 1.8565074525368978e-7, 1.7839425588131534e-7, 
     1.7151107724888157e-7, 1.649775080767636e-7, 1.5877163016997738e-7, 
     1.5287315385537076e-7, 1.4726327854337888e-7, 1.419245667694746e-7, 
     1.3684083026683488e-7, 1.3199702679264573e-7, 1.2737916657944673e-7, 
     1.2297422741300776e-7, 1.1877007745201245e-7, 1.1475540500450571e-7, 
     1.109196545635269e-7, 1.0725296848121451e-7, 1.0374613372831736e-7, 
     1.0039053324567361e-7, 9.717810144684968e-8, 9.41012834776523e-8, 
     9.115299787940322e-8, 8.832660233936186e-8, 8.561586224406986e-8, 
     8.301492178017217e-8, 8.051827735287719e-8, 7.81207531150298e-8};
 
  TVIL_REAL H0xxx0ytableB[] = 
     {0, -3.1583714815146466, 1.0491173392529898, 
     -0.5309474723357924, 0.3228978994124212, -0.21793209668387814, 
     0.15734855514283355, -0.11911511160996877, 0.0933963371054626, 
     -0.07524540558822218, 0.0619466210113503, -0.05190547484215264, 
     0.04413446773212103, -0.037994892698764805, 0.03305847034984721, 
     -0.02902914274134587, 0.025696847985791985, -0.022909137629899455, 
     0.02055316823273714, -0.018543943801302157, 0.016816443850037947, 
     -0.015320234113889847, 0.014015702720973409, -0.012871384104711009, 
     0.011862025277928073, -0.010967167856274117, 0.010170094243977138, 
     -0.009457034781809755, 0.008816564458873143, -0.00823913905600181, 
     0.007716735037663759, -0.007242567471703227, 0.006810867219296644, 
     -0.006416703565936843, 0.006055841993858384, -0.005724629351982535, 
     0.0054199005490219365, -0.005138902276260525, 0.004879230295715853, 
     -0.004638777603023364, 0.0044156913606012045, -0.0042083369442428836, 
     0.004015267790488807, -0.003835199998633162, 0.003666990848905325, 
     -0.003509620561204215, 0.0033621767471901693, -0.0032238411103895225, 
     0.003093878030171173, -0.002971624730526258, 0.0028564827869796755, 
     -0.00274791076735017, 0.0026454178365188093, -0.002548558183472371, 
     0.0024569261519180327, -0.002370151974709946, 0.002287898027970471, 
     -0.0022098555337515574, 0.0021357416508614507, -0.002065296902476508, 
     0.0019982828966873126, -0.0019344803024509694, 0.0018736870487468286, 
     -0.0018157167192313865, 0.0017603971184988155, -0.0017075689892904829, 
     0.0016570848627535549, -0.0016088080262026017, 0.0015626115948528757, 
     -0.0015183776757225955, 0.0014759966133882272, -0.0014353663085581547, 
     0.0013963916015370386, -0.001358983713611374, 0.0013230597402179446, 
     -0.0012885421904793397, 0.0012553585683198719, -0.001223440990924215, 
     0.0011927258407809624, -0.0011631534479735472, 0.0011346677997496107, 
     -0.0011072162747238535, 0.0010807493993545353, -0.0010552206245851617, 
     0.0010305861207648673, -0.0010068045891572913, 0.0009838370885216097, 
     -0.0009616468754035805, 0.0009401992569114431, -0.0009194614548733402, 
     0.0008994024803814483, -0.0008799930178247905, 0.0008612053175991305, 
     -0.0008430130967596352, 0.0008253914469511821, -0.0008083167490132062, 
     0.0007917665937116407, -0.0007757197081005114, 0.0007601558870607253, 
     -0.000745055929604093, 0.0007304015795671421};
 
  TVIL_REAL H0xxx0ytableD[] = 
     {0, -1.8030853547393915, 0.4747428871050507, 
     -0.21195363509823276, 0.11908957801926866, -0.07607598509009696, 
     0.05273867470114305, -0.0386890676119989, 0.029584225711408042, 
     -0.023350561686140196, 0.018897066984978873, -0.01560548807102161, 
     0.013104307565004887, -0.01115939970174481, 0.009617296314256937, 
     -0.008374010106936114, 0.0073570594645541165, -0.006514673305220674, 
     0.005809076466801236, -0.005212183782947548, 0.004702763812987744, 
     -0.0042645257796382705, 0.003884801632412969, -0.00355362070875824, 
     0.003263048821469288, -0.003006708820021032, 0.002779427838661711, 
     -0.002576974371907678, 0.002395859957376759, -0.002233187940478032, 
     0.0020865369675249754, -0.0019538703840194915, 0.001833465158783936, 
     -0.0017238556689446116, 0.0016237888981148528, -0.001532188474488588, 
     0.0014481256103094176, -0.0013707954696202008, 0.0012994978357077013, 
     -0.0012336212069197566, 0.0011726296432509786, -0.0011160518331147407, 
     0.0010634719621338661, -0.0010145220523439967, 0.0009688755073048239, 
     -0.0009262416509615875, 0.0008863610891839182, -0.0008490017553403999, 
     0.0008139555270085312, -0.0007810353214583296, 0.0007500725940173947, 
     -0.0007209151766952837, 0.0006934254051862266, -0.0006674784911016915, 
     0.0006429611034137081, -0.0006197701289340377, 0.0005978115864635681, 
     -0.0005769996732187888, 0.0005572559254352713, -0.0005385084777874264, 
     0.0005206914085501287, -0.0005037441593421824, 0.00048761101989952413, 
     -0.0004722406696806389, 0.0004575857692511265, -0.0004436025953640028, 
     0.0004302507144760836, -0.0004174926901424605, 0.0004052938203302575, 
     -0.000393621901205782, 0.000382447014389314, -0.0003717413350503244, 
     0.00036147895854221884, -0.0003516357435575945, 0.00034218917002902684, 
     -0.00033311821021210394, 0.00032440321157143145, -0.0003160257902505886, 
     0.00030796873404684473, -0.000300215913933681, 0.00029275220328120276, 
     -0.00028556340401842946, 0.0002786361790639661, -0.0002719579904241847, 
     0.00026551704242207616, -0.0002593022295764678, 0.0002533030887013004, 
     -0.00024750975483893965, 0.0002419129206807723, -0.00023650379916322143, 
     0.0002312740889583454, -0.00022621594260582, 0.00022132193705775067, 
     -0.00021658504642977126, 0.00021199861677156606, 
     -0.00020755634268756934, 0.00020325224565438855, 
     -0.00019908065389566892, 0.00019503618368784816, 
     -0.00019111372198170304, 0.00018573532585999624};
 
  TVIL_REAL H0xxx0ytableF[] = 
     {0, -4.019619913866701, 1.4867400752395576, -0.8031760252636851, 
     0.5112067312028203, -0.3570597913834937, 0.26488026592407526, 
     -0.20500758035837255, 0.16375193128718868, -0.13403226805952595, 
     0.11186655551898207, -0.094867082805986, 0.08152733944731877, 
     -0.07085657832675935, 0.062180417921701, -0.05502627743006429, 
     0.049054641968123154, -0.04401627702001672, 0.03972473646560186, 
     -0.03603820198334326, 0.03284719104016192, -0.030066055554496145, 
     0.027626987837670054, -0.02547572051419995, 0.02356839301902526, 
     -0.021869235489755286, 0.02034883447431914, -0.018982818768376868, 
     0.017750852657788398, -0.016635856835486684, 0.015623399844787162, 
     -0.014701218581236328, 0.013858837418015055, -0.0130872633793166, 
     0.012378740449517749, -0.011726550230929781, 0.011124849197649854, 
     -0.010568535046783603, 0.010053136336921101, -0.009574720879504499, 
     0.009129819320162342, -0.008715361092301225, 0.008328620500946786, 
     -0.00796717114254049, 0.0076288472167846694, -0.007311710562494145, 
     0.007014022467851243, -0.0067342194793703625, 0.006470892573056294, 
     -0.006222769163176776, 0.005988697514524039, -0.005767633197467096, 
     0.005558627284961114, -0.005360816039690968, 0.0051734118798116065, 
     -0.004995695444988832, 0.004827008611973758, -0.004666748331824787, 
     0.004514361179972854, -0.004369338526291841, 0.004231212245737814, 
     -0.004099550901404388, 0.003973956341369996, -0.003854060658782061, 
     0.0037395234714755754, -0.0036300294832581573, 0.0035252862939745447, 
     -0.003425022429726122, 0.003328985568277925, -0.0032369409378300634, 
     0.0031486698700406214, -0.003063968490527971, 0.0029826465321067127, 
     -0.00290452625776911, 0.0028294414819515384, -0.0027572366799558925, 
     0.0026877661765567365, -0.0026208934058397255, 0.0025564902352055085, 
     -0.0024944363472529122, 0.002434618673940324, -0.002376930878027253, 
     0.0023212728773297545, -0.002267550407792949, 0.0022156746217991464, 
     -0.0021655617184979407, 0.002117132603270925, -0.0020703125737335836, 
     0.002025031029934763, -0.0019812212066438488, 0.0019388199258206762, 
     -0.0018977673675462218, 0.0018580068578557757, -0.0018194846720628699, 
     0.0017821498522936005, -0.0017459540380689244, 0.001710851308878475, 
     -0.0016767980377847902, 0.0016437527551826994, -0.0016116760219160513, 
     0.001580530311023835};

  lnyox = TVIL_CLOG(y/x);
  lnyox2 = lnyox  * lnyox;
  lnyox3 = lnyox2  * lnyox;

  if (y < 0.5 * x) {
    result = TVIL_H00xx0x (x,qq);
    rn = r = y/x;
    for (n=1; n < 100; n++) {
      result += rn * (H0xxx0ytableA[n] + 
                      H0xxx0ytableAL[n] * lnyox);
      rn *= r;
    }
  }
  else if (y < 1.6 * x) {
    result = TVIL_H0xxx0x (x,qq);
    rn = r = y/x - 1.L;
    for (n=1; n < 100; n++) {
      result += rn * H0xxx0ytableD[n];
      rn *= r;
    }
  }
  else if (y < 5.54 * x) {
    result = -9.459609917903611726046994L - 6.L * Zeta3 * TVIL_Ap(x,qq);
    rn = r = y/(4.L*x) - 1.L;
    for (n=1; n < 100; n++) {
      result += rn * H0xxx0ytableB[n];
      rn *= r;
    }
  }
  else if (y < 14.56 * x) {
    result = -12.3708114600309836671344L - 6.L * Zeta3 * TVIL_Ap(x,qq);
    rn = r = y/(9.L*x) - 1.L;
    for (n=1; n < 100; n++) {
      result += rn * H0xxx0ytableF[n];
      rn *= r;
    }
  }
  else {
    result = TVIL_H00000x (y, qq);
    rn = r = 9.L * x/y;
    for (n=1; n < 100; n++) {
      result += rn * (H0xxx0ytableC[n] 
                     -H0xxx0ytableCL[n] * lnyox 
                     +H0xxx0ytableCL2[n] * lnyox2
                     -H0xxx0ytableCL3[n] * lnyox3);
      rn *= r;
    }
  }

  return result;
}

/* ****************************************************************** */

TVIL_COMPLEX TVIL_H0xxyyy (TVIL_REAL x, TVIL_REAL y, TVIL_REAL qq)
{
  TVIL_REAL r, rn;
  TVIL_COMPLEX lnyox, lnyox2, result;
  int n;

  TVIL_REAL H0xxyyytableA[] = 
     {0, -1.394453945748151, -0.11746173391306594, 
     -0.017232606186386477, -0.004258971127450625, -0.0017356084096962052, 
     -0.0009550125249925185, -0.0006025028158655245, -0.00040652599025097353, 
     -0.00028603128314970503, -0.00020767259014323596, 
     -0.0001547131794351338, -0.00011781301913792259, 
     -0.00009143256976876734, -0.00007214692434195073, 
     -0.00005776823413367932, -0.00004685954811036016, 
     -0.00003845376579358013, -0.000031885651554929954, 
     -0.00002668853507882136, -0.000022529170521608333, 
     -0.00001916571494557447, -0.000016420054402947816, 
     -0.000014159230922495167, -0.000012282759224777938, 
     -0.000010713826004854908, -9.393092592314398e-6, 
     -8.274270964695369e-6, -7.3209254919329084e-6, 
     -6.5041334723814126e-6, -5.800755020055241e-6, -5.192140441834945e-6, 
     -4.663155197058903e-6, -4.201437789743603e-6, -3.7968301708964723e-6, 
     -3.4409370718908405e-6, -3.126782529604526e-6, -2.848540273255792e-6, 
     -2.6013206743368724e-6, -2.381001327296567e-6, 
     -2.1840915172163322e-6, -2.0076231787112176e-6, 
     -1.8490626929563633e-6, -1.7062391728675518e-6, 
     -1.5772858678817583e-6, -1.4605920639473072e-6, 
     -1.3547634222626598e-6, -1.2585891364206716e-6, 
     -1.171014624497139e-6, -1.0911187343230137e-6, 
     -1.0180946445792957e-6, -9.512338048281503e-7, -8.899123842104094e-7, 
     -8.335797989196832e-7, -7.817489685089341e-7, -7.339880150376474e-7, 
     -6.899131704435319e-7, -6.491826989640866e-7, -6.114916749930909e-7, 
     -5.76567484037408e-7, -5.441659366966561e-7, -5.140679038114851e-7, 
     -4.860763958975289e-7, -4.600140223225006e-7, -4.357207758881812e-7, 
     -4.1305209694307244e-7, -3.9187717819302766e-7, 
     -3.7207747725228764e-7, -3.535454088930118e-7, 
     -3.3618319307549496e-7, -3.1990183831039787e-7, 
     -3.0462024282999145e-7, -2.902643985188951e-7, -2.767666846510261e-7, 
     -2.640652402601748e-7, -2.5210340548772735e-7, 
     -2.4082922354475723e-7, -2.301949960320072e-7, -2.201568853093173e-7, 
     -2.1067455842012764e-7, -2.017108677771166e-7, -1.932315644188617e-7, 
     -1.8520504016893769e-7, -1.7760209548014352e-7, 
     -1.7039573013774305e-7, -1.6356095433530377e-7, 
     -1.5707461793221183e-7, -1.5091525595940894e-7, 
     -1.4506294866459645e-7, -1.3949919458456948e-7, 
     -1.3420679530430315e-7, -1.2916975071319095e-7, 
     -1.243731637012354e-7, -1.198031533544236e-7, -1.154467758110564e-7, 
     -1.1129195203122253e-7, -1.0732740181145893e-7, 
     -1.0354258344724505e-7, -9.992763850849367e-8, -9.647334124862106e-8, 
     -9.317105221697342e-8};
 
  TVIL_REAL H0xxyyytableAL[] = 
    {0, 1.6775329665758867, 0.20795550971087473, 
     0.026100223840701344, 0.0006293029714070124, -0.002154269397530137, 
     -0.0017819055848048062, -0.0012156589417500951, -0.0008154261982272819, 
     -0.0005588860153622585, -0.00039449629395331746, -0.000286533113805317, 
     -0.00021349129551867857, -0.0001626416414251497, 
     -0.00012631642013075377, -0.00009976826878837839, 
     -0.00007997174721851891, -0.00006494526132106401, 
     -0.00005335807163970713, -0.00004429616171267003, 
     -0.000037118892087129136, -0.00003136896156788528, 
     -0.000026714505138797343, -0.000022911049696338338, 
     -0.00001977602263636473, -0.000017171362928080592, 
     -0.000014991463172106093, -0.000013154681366021222, 
     -0.000011597282016792417, -0.000010269055435288451, 
     -9.130112469702579e-6, -8.148513192294989e-6, -7.298494380082572e-6, 
     -6.559131764676279e-6, -5.913321267630672e-6, -5.346996571244072e-6, 
     -4.848523404254733e-6, -4.408227108297144e-6, -4.0180215478218556e-6, 
     -3.6711156733397486e-6, -3.361780019398671e-6, 
     -3.0851597808892642e-6, -2.83712432458986e-6, -2.614145378716688e-6, 
     -2.4131979281058738e-6, -2.2316791874967303e-6, 
     -2.0673420456130003e-6, -1.9182401517450525e-6, 
     -1.7826824150502955e-6, -1.659195149358145e-6, 
     -1.5464904557953666e-6, -1.4434397165064085e-6, 
     -1.3490512934432519e-6, -1.2624517004282259e-6, 
     -1.182869654893771e-6, -1.1096225258288529e-6, -1.042104782603112e-6, 
     -9.797781201850487e-7, -9.221629934462968e-7, -8.688313395693994e-7, 
     -8.194003052529988e-7, -7.735268261627914e-7, -7.309029312694121e-7, 
     -6.912516654217327e-7, -6.543235405806199e-7, -6.198934402654761e-7, 
     -5.877579134892927e-7, -5.577328042155485e-7, -5.296511705151855e-7, 
     -5.033614544196858e-7, -4.787258691886649e-7, -4.5561897552606176e-7, 
     -4.339264223419426e-7, -4.135438310933275e-7, -3.9437580565090664e-7, 
     -3.763350521143995e-7, -3.5934159510813133e-7, 
     -3.4332207888855175e-7, -3.2820914313538674e-7, 
     -3.1394086461826235e-7, -3.004602570646536e-7, 
     -2.8771482253107144e-7, -2.756561484211297e-7, -2.642395450213923e-7, 
     -2.5342371905540537e-7, -2.431704793021912e-7, 
     -2.3344447079965495e-7, -2.242129345659517e-7, -2.154454901314782e-7, 
     -2.0711393848809736e-7, -1.9919208333671496e-7, 
     -1.9165556875473015e-7, -1.8448173161573025e-7, 
     -1.7764946727900593e-7, -1.7113910722937673e-7, 
     -1.6493230749132581e-7, -1.5901194676803525e-7, 
     -1.5336203336772996e-7, -1.4796762007864094e-7, 
     -1.428147262414863e-7, -1.378902663460389e-7};
 
  TVIL_REAL H0xxyyytableAL2[] = 
     {0, -0.375, -0.09114583333333333, -0.025520833333333333, 
     -0.008399600074404762, -0.0032808624751984126, -0.0015077063416907167, 
     -0.0007940627807135201, -0.00046424981372472065, 
     -0.00029317784613313696, -0.00019606174087455523, 
     -0.0001369892383477316, -0.0000990898568597171, -0.0000737265585920473, 
     -0.00005616002999289723, -0.00004364066954979314, 
     -0.00003449914577857944, -0.000027683012570445033, 
     -0.000022507193477526782, -0.00001851329661212645, 
     -0.000015387143756886278, -0.000012908781720858352, 
     -0.000010921286655647422, -9.310789419412805e-6, 
     -7.993389591545502e-6, -6.906404356766241e-6, -6.00240649044764e-6, 
     -5.2450932946415584e-6, -4.606379658010986e-6, -4.064323320255307e-6, 
     -3.6016246466049295e-6, -3.2045286627361034e-6, 
     -2.8620124533091218e-6, -2.5651774700766178e-6, 
     -2.3067906501069106e-6, -2.0809347486707117e-6, 
     -1.8827396211952466e-6, -1.7081740612561909e-6, 
     -1.5538833342053936e-6, -1.4170614758051053e-6, 
     -1.2953502445932708e-6, -1.1867586585710126e-6, 
     -1.0895985388129914e-6, -1.0024325820429083e-6, 
     -9.240323008660472e-7, -8.533437815520703e-7, -7.894596699983005e-7, 
     -7.315961461990733e-7, -6.790739146954939e-7, -6.31302443837896e-7, 
     -5.87766845490441e-7, -5.480169102974643e-7, -5.116579101859879e-7, 
     -4.783428556658619e-7, -4.4776595542940715e-7, 
     -4.1965707332346363e-7, -3.937770156941934e-7, -3.699135124771052e-7, 
     -3.478777798297319e-7, -3.275015718268521e-7, -3.086346447260483e-7, 
     -2.911425703209702e-7, -2.7490484552441097e-7, -2.598132540309566e-7, 
     -2.457704430697366e-7, -2.3268868416584825e-7, -2.204887917190014e-7, 
     -2.0909917726771413e-7, -1.9845502068796048e-7, 
     -1.8849754239827852e-7, -1.7917336300750106e-7, 
     -1.7043393882644657e-7, -1.6223506333622852e-7, 
     -1.5453642611651805e-7, -1.473012219307142e-7, 
     -1.4049580367736684e-7, -1.3408937377787605e-7, 
     -1.2805370930385551e-7, -1.2236291677380553e-7, 
     -1.1699321308468408e-7, -1.119227295035759e-7, 
     -1.0713133603960818e-7, -1.0260048385632221e-7, -9.83130636780697e-8, 
     -9.425327839755467e-8, -9.040652831118608e-8, -8.675930759934652e-8, 
     -8.329911083416184e-8, -8.001434844139351e-8, -7.689427016865509e-8, 
     -7.39288957218197e-8, -7.110895182740463e-8, -6.842581506274496e-8, 
     -6.587145986948654e-8, -6.343841123069939e-8, -6.111970154890376e-8, 
     -5.8908831312517965e-8, -5.679973318254594e-8, 
     -5.4786739170471435e-8, -5.286455061296073e-8, -5.102821067965768e-8};
 
  TVIL_REAL H0xxyyytableC[] = 
     {0, -0.9632803294001042, 0.01563764461301982, 
     0.03481135245205951, 0.027890866823153837, 0.021977286835722185, 
     0.017747577290113017, 0.01468090279168265, 0.012390514297191702, 
     0.010631909392857474, 0.009249175695491814, 0.008139817621226174, 
     0.007234315950625875, 0.006484161414380468, 0.005854644639877069, 
     0.005320374418071914, 0.004862408940817411, 0.004466369823612272, 
     0.004121172009008472, 0.003818149660012894, 0.0035504426719814583, 
     0.003312558384563122, 0.0031000533664138803, 0.002909298953313105, 
     0.0027373061550266605, 0.0025815932726350383, 0.0024400846631008537, 
     0.0023110325054099433, 0.002192955751371616, 0.0020845920543308637, 
     0.0019848595975676055, 0.0018928265451617003, 0.0018076864133792746, 
     0.0017287380783741599, 0.0016553694424866823, 0.0015870440084769673, 
     0.001523289780777346, 0.001463690040845769, 0.0014078756410021568, 
     0.0013555185356621158, 0.0013063263263894255, 0.001260037641864564, 
     0.0012164182088007288, 0.0011752574973248817, 0.0011363658460948208, 
     0.0010995719897364407, 0.0010647209250383232, 0.0010316720634817938, 
     0.0010002976266882404, 0.0009704812486758331, 0.0009421167547798017, 
     0.0009151070919736227, 0.0008893633893444849, 0.0008648041307922512, 
     0.0008413544247692582, 0.0008189453581641858, 0.0007975134233412876, 
     0.0007770000089443326, 0.0007573509464173677, 0.000738516105326204, 
     0.0007204490315213705, 0.0007031066229945827, 0.0006864488389706056, 
     0.0006704384383645458, 0.0006550407442373746, 0.0006402234313133821, 
     0.0006259563339934438, 0.0006122112726167897, 0.0005989618959991463, 
     0.0005861835385131889, 0.0005738530901836452, 0.0005619488784487184, 
     0.0005504505603956287, 0.000539339024414264, 0.0005285963003319761, 
     0.0005182054771967975, 0.000508150627967806, 0.0004984167404517332, 
     0.0004889896538956691, 0.00047985600070810975, 0.00047100315283571327, 
     0.0004624191723718837, 0.0004540927660165145, 0.0004460132430445641, 
     0.00043817047647521403, 0.0004305548671636909, 0.00042315731056486865, 
     0.000415969165941894, 0.0004089822278146438, 0.00040218869946211967, 
     0.0003955811683101792, 0.00038915258305151384, 0.00038289623235872186, 
     0.0003768057250638573, 0.0003708749716891184, 0.0003650981672235155, 
     0.0003594697750495341, 0.0003539845119321065, 0.00034863733398970287, 
     0.00034342342357414617, 0.00033833817699190826};
 
  TVIL_REAL H0xxyyytableCL[] = 
     {0, 0.935463395592981, 0.16329877334736786, 
     0.08013965681068157, 0.05347991599169528, 0.03949691560589045, 
     0.030734944279133505, 0.02477566377882973, 0.020507508930852722, 
     0.017330888585225756, 0.01489353722926494, 0.01297628440751266, 
     0.011436584357171188, 0.010178276419291006, 0.009134452566730584, 
     0.008257304911728722, 0.0075118767407504105, 0.006872087557396259, 
     0.0063181305417652385, 0.005834724504215135, 0.005409912796714552, 
     0.005034221097595976, 0.0047000559521737945, 0.004401268097838903, 
     0.004132830642016706, 0.0038905986247070447, 0.003671127126806556, 
     0.0034715320812496882, 0.0032893826301014974, 0.0031226170609306866, 
     0.0029694765603216883, 0.0028284525670167066, 0.002698244603436319, 
     0.0025777262517203098, 0.002465917512321217, 0.0023619622029056654, 
     0.0022651093663854397, 0.0021746978895636135, 0.0020901437094100065, 
     0.0020109291174859423, 0.0019365937753636593, 0.00186672713288402, 
     0.0018009620025056187, 0.001738969091047121, 0.0016804523279528106, 
     0.0016251448591685657, 0.001572805599572764, 0.001523216256007638, 
     0.0014761787483276725, 0.001431512968311425, 0.0013890548263813148, 
     0.0013486545443164588, 0.0013101751588969726, 0.0012734912069753959, 
     0.0012384875670618477, 0.0012050584363165273, 0.0011731064250115626, 
     0.0011425417531703021, 0.0011132815363094886, 0.00108524914907381, 
     0.001058373657124298, 0.0010325893089716688, 0.0010078350805736403, 
     0.0009840542664747518, 0.0009611941120856685, 0.0009392054823989062, 
     0.0009180425630380499, 0.000897662590053365, 0.0008780256053211057, 
     0.0008590942347875813, 0.0008408334871311783, 0.0008232105707036018, 
     0.0008061947268619874, 0.0007897570780215982, 0.0007738704889491305, 
     0.0007585094399830371, 0.0007436499110130417, 0.0007292692751789276, 
     0.0007153462013611467, 0.0007018605646348249, 0.000688793363946111, 
     0.000676126646347003, 0.0006638434371931078, 0.0006519276757693388, 
     0.0006403641558623126, 0.0006291384708459932, 0.0006182369628896762, 
     0.0006076466759353415, 0.0005973553121252635, 0.000587351191391044, 
     0.0005776232139423357, 0.0005681608254178048, 0.0005589539844826902, 
     0.0005499931326768899, 0.0005412691663351332, 0.0005327734104166572, 
     0.0005244975940961171, 0.0005164338279803719, 0.0005085745828274585, 
     0.0005009126696546216, 0.0004934412211318292};
 
  TVIL_REAL H0xxyyytableCL2[] = 
     {0, 0., 0.0625, 0.041666666666666664, 0.027018229166666668, 
     0.018717447916666668, 0.013829210069444445, 0.010738118489583334, 
     0.008650943211146763, 0.007165077249839823, 0.00606260443490649, 
     0.00521753193840148, 0.004552637686403741, 0.0040182167123922535, 
     0.0035809523830953194, 0.0032177473790254442, 0.0029121284958816334, 
     0.002652059430682302, 0.0024285603407192763, 0.0022348086297765708, 
     0.002065536934925993, 0.0019166203004784181, 0.0017847870218013848, 
     0.0016674122522195025, 0.001562368169243023, 0.0014679135264011626, 
     0.0013826111014320795, 0.0013052652103247822, 0.0012348738593936227, 
     0.0011705917145016559, 0.0011117011593872134, 0.0010575894697493074, 
     0.0010077306583565959, 0.00096167092156747, 0.0009190168871010487, 
     0.0008794260586451212, 0.0008425989965947413, 0.0008082728807631768, 
     0.0007762161806368854, 0.0007462242189280945, 0.0007181154599776516, 
     0.0006917283896838696, 0.0006669188807654037, 0.000643557958270495, 
     0.0006215298967678486, 0.0006007305936708508, 0.0005810661734613997, 
     0.0005624517857990419, 0.0005448105670859762, 0.0005280727403607521, 
     0.0005121748326837634, 0.0004970589926652341, 0.00048267239363408736, 
     0.0004689667102812281, 0.00045589765853341495, 0.00044342459000313314, 
     0.0004315101336784682, 0.00042011987861496913, 0.00040922209230897025, 
     0.00039878747020105735, 0.00038878891240535466, 0.0003792013243061908, 
     0.00037000143812565615, 0.00036116765295762625, 0.0003526798910975025, 
     0.00034451946878167407, 0.00033666897969433936, 0.0003291121898083139, 
     0.000321833942306148, 0.00031482007148276567, 0.00030805732466464053, 
     0.0003015332912963715, 0.00029523633844604527, 0.0002891555520681737, 
     0.0002832806834391467, 0.00027760210024662275, 0.0002721107418724252, 
     0.00026679807845947113, 0.00026165607339799095, 0.0002566771489056389, 
     0.00025185415441075234, 0.0002471803374785993, 0.00024264931704748763, 
     0.00023825505876553586, 0.00023399185224012568, 0.0002298542900308956, 
     0.00022583724823388985, 0.0002219358685193977, 0.00021814554149932495, 
     0.00021446189131182287, 0.00021088076132152537, 0.0002073982008432603, 
     0.00020401045280563025, 0.00020071394227851513, 0.00019750526579543078, 
     0.00019438118140786863, 0.00019133859941432028, 0.00018837457371172103, 
     0.00018548629372158845, 0.00018267107684723731, 0.00017992636142216657};
 
  TVIL_REAL H0xxyyytableB[] = 
     {0, -5.687681695528182, 2.4313166353200573, -1.435430251243721, 
     0.9734246325143795, -0.7140796458240717, 0.55127468383879, 
     -0.4412046872004001, 0.3627332938740514, -0.30450893332181156, 
     0.25993686077688866, -0.22494774140627868, 0.19690751732815706, 
     -0.17404346455319222, 0.15512330249026157, -0.1392666941288901, 
     0.12582983269821266, -0.11433224132277864, 0.10440891328847125, 
     -0.09577818053458736, 0.0882196330332968, -0.0815586282838542, 
     0.07565522162362212, -0.07039612338107488, 0.06568876694013498, 
     -0.0614568736820084, 0.057637095596486124, -0.05417644457773721, 
     0.051030303334415494, -0.04816087136311465, 0.04553593989964899, 
     -0.04312791813656407, 0.040913053148995134, -0.038870800458415525, 
     0.03698331269481359, -0.035235021552973596, 0.033612293975685575, 
     -0.03210314779099389, 0.030697015272593088, -0.029384545559907906, 
     0.02815743876667846, -0.027008306068564778, 0.02593055119709031, 
     -0.024918269657104048, 0.023966162685783773, -0.023069463526404238, 
     0.022223874032366503, -0.021425509971164745, 0.020670853683035104, 
     -0.019956712979580894, 0.019280185354986757, -0.018638626735299764, 
     0.018029624116532462, -0.017450971545426866, 0.016900648981871047, 
     -0.016376803652563204, 0.01587773356426827, -0.015401872894068101, 
     0.014947779015103904, -0.014514120950850286, 0.014099669080080035, 
     -0.013703285939301276, 0.013323917990328615, -0.012960588238405239, 
     0.01261238960143192, -0.012278478943800474, 0.011958071699420248, 
     -0.011650437018054386, 0.01135489337728757, -0.01107080460952835, 
     0.01079757629957464, -0.010534652513580923, 0.010281512824878051, 
     -0.010037669606111495, 0.009802665560665235, -0.009576071469398106, 
     0.009357484131397854, -0.009146524479807124, 0.008942835855839364, 
     -0.008746082425918771, 0.00855594772847945, -0.008372133338372546, 
     0.008194357638080244, -0.0080223546860428, 0.00785587317338692, 
     -0.007694675461216385, 0.007538536691401933, -0.007387243964499021, 
     0.007240595579038848, -0.0070984003269890916, 0.006960476840673668, 
     -0.00682665298688213, 0.006696765304295165, -0.0065706584807078885, 
     0.00644818486685201, -0.006329204023905331, 0.0062135823020358924, 
     -0.006101192447561637, 0.005991913236517201, -0.005885629132610044, 
     0.005782229967720497};
 
  TVIL_REAL H0xxyyytableD[] = 
     {0, -4.304699888023509, 1.5877936463196338, -0.8516605951028584, 
     0.5389824929068298, -0.37499700619748677, 0.27751724052478993, 
     -0.2145141271020673, 0.17127291279360085, -0.14021924521339152, 
     0.11711347838587019, -0.09942433558805853, 0.0855609156146769, 
     -0.07448059422779235, 0.06547585493155444, -0.0580522367160929, 
     0.051855346640446255, -0.04662556590530813, 0.042169028575342896, 
     -0.03833850091404479, 0.03502047071296685, -0.03212623781337045, 
     0.029585644991477276, -0.027342588874228448, 0.02535175422582896, 
     -0.023576203824182077, 0.021985576287475314, -0.02055472220129119, 
     0.019262660473232746, -0.018091771536241946, 0.01702716772780217, 
     -0.016056197605899228, 0.015168052508345152, -0.014353451875105432, 
     0.013604389763588968, -0.012913929286272687, 0.012276034859378887, 
     -0.011685434495024226, 0.011137506123446899, -0.010628183255932748, 
     0.010153876306329739, -0.009711406661088435, 0.009297951183724803, 
     -0.008910995302728096, 0.008548293194135428, -0.00820783385497933, 
     0.007887812089339877, -0.007586603608186499, 0.0073027435877366575, 
     -0.00703490814645116, 0.006781898293993071, -0.00654262598110939, 
     0.006316101941034933, -0.006101425063466153, 0.00589777308360746, 
     -0.005704394402989501, 0.005520600887073432, -0.005345761508183379, 
     0.005179296721927574, -0.00502067348168109, 0.004869400809477609, 
     -0.0047250258532538225, 0.0045871303701815695, -0.0044553275841143075, 
     0.004329259372215237, -0.004208593741829246, 0.004093022563778337, 
     -0.003982259532639884, 0.0038760383283242767, -0.0037741109564996, 
     0.003676246248195689, -0.0035822285013255356, 0.0034918562489444035, 
     -0.003404941140873543, 0.003321306926885697, -0.003240788531017322, 
     0.003163231207765977, -0.003088489771974822, 0.0030164278951201507, 
     -0.0029469174615198014, 0.0028798379786851517, -0.0028150760366599525, 
     0.0027525248117364976, -0.00269208361042298, 0.0026336574499634586, 
     -0.0025771566720906944, 0.002522496587028234, -0.0024695971450567765, 
     0.0024183826332256223, -0.0023687813950267777, 0.0023207255710605812, 
     -0.0022741508589104475, 0.0022289962906131776, -0.0021852040262625087, 
     0.00214271916241919, -0.0021014895541226187, 0.0020614656494085328, 
     -0.0020226003353357265, 0.0019848487946135095, -0.0019481683720016338, 
     0.0019125184497266981};
 
  TVIL_REAL H0xxyyytableE[] = 
     {0, -4.487729693649713, 1.7014832547521614, -0.9257683913505929, 
     0.5898215232163679, -0.4112861447569545, 0.30421504897387813, 
     -0.23461595859651918, 0.18668193281017537, -0.15219595185797655, 
     0.12652189746985434, -0.10687399176356434, 0.09149238493159642, 
     -0.07921903481926136, 0.06926560333658166, -0.0610798037655032, 
     0.05426508250882369, -0.04853056464131893, 0.043658865883772865, 
     -0.03948481973490557, 0.03588107575195692, -0.0327481390545617, 
     0.030007348677564185, -0.027595841948522302, 0.025462886651147622, 
     -0.023567171489075382, 0.021874778519865648, -0.020357647886159836, 
     0.018992402609452817, -0.017759439928076787, 0.01664222216339555, 
     -0.01562671850030037, 0.014700962015426235, -0.013854695508312075, 
     0.013079086334413957, -0.012366495276490962, 0.011710288048710615, 
     -0.011104680669025995, 0.010544611913433179, -0.01002563755944664, 
     0.009543842262967195, -0.009095765784344817, 0.008678340952414309, 
     -0.008288841278333719, 0.007924836540148436, -0.007584154980885255, 
     0.00726485101768818, -0.006965177562155773, 0.006683562214105815, 
     -0.0064185867212456634, 0.006168969202407203, -0.005933548717324706, 
     0.005711271835443272, -0.005501180913110319, 0.005302403835209054, 
     -0.00511414501580286, 0.0049356774842287164, -0.00476633590954499, 
     0.004605510438294907, -0.004452641238987588, 0.004307213662165223, 
     -0.004168753937937208, 0.0040368253438416565, -0.003911024785185699, 
     0.003790979737899791, -0.0036763455106488202, 0.003566802788664493, 
     -0.003462055426656238, 0.0033618284623521248, -0.003265866325825069, 
     0.0031739312228631425, -0.003085801673321495, 0.0030012711877102473, 
     -0.0029201470672809266, 0.0028422493146180797, -0.0027674096432604572, 
     0.00269547057619944, -0.0026262846242582484, 0.0025597135363669924, 
     -0.0024956276146352934, 0.0024339050879028596, -0.0023744315381332664, 
     0.0023170993746196057, -0.002261807351503107, 0.002208460124576436, 
     -0.002156967843759886, 0.0021072457780078944, -0.0020592139697310973, 
     0.0020127969161104716, -0.0019679232749394986, 0.001924525592861452, 
     -0.0018825400540752719, 0.0018419062477678682, -0.0018025669526957164, 
     0.001764467937486413, -0.0017275577753634588, 0.0016917876721165943, 
     -0.001657111306247064, 0.0016234846803135183, -0.0015908659825910844, 
     0.0015592154582344223};

  lnyox = TVIL_CLOG(y/x);
  lnyox2 = lnyox  * lnyox;

  if (y < 0.155 * x) {
    result = TVIL_H0000xx (x,qq);
    rn = r = 4.L*y/x;
    for (n=1; n < 100; n++) {
      result += rn * (H0xxyyytableA[n] + 
                      H0xxyyytableAL[n] * lnyox +
                      H0xxyyytableAL2[n] * lnyox2);
      rn *= r;
    }
  }
  else if (y < 0.5 * x) {
    result = -13.35025921012291505528939L - 6.L * Zeta3 * TVIL_Ap(y,qq);
    rn = r = x/(4.L*y) - 1.L;
    for (n=1; n < 100; n++) {
      result += rn * H0xxyyytableE[n];
      rn *= r;
    }
  }
  else if (y < 1.6 * x) {
    result = TVIL_H0xxxxx (x,qq);
    rn = r = y/x - 1.L;
    for (n=1; n < 100; n++) {
      result += rn * H0xxyyytableD[n];
      rn *= r;
    }
  }
  else if (y < 6.47 * x) {
    result = -15.1926244255576197354684L - 6.L * Zeta3 * TVIL_Ap(x,qq);
    rn = r = y/(4.L*x) - 1.L;
    for (n=1; n < 100; n++) {
      result += rn * H0xxyyytableB[n];
      rn *= r;
    }
  }
  else {
    result = TVIL_H000xxx (y, qq);
    rn = r = 4.L * x/y;
    for (n=1; n < 100; n++) {
      result += rn * (H0xxyyytableC[n] 
                     -H0xxyyytableCL[n] * lnyox 
                     +H0xxyyytableCL2[n] * lnyox2);
      rn *= r;
    }
  }

  return result;
}

/* ****************************************************************** */

TVIL_COMPLEX TVIL_H0xxxxy (TVIL_REAL x, TVIL_REAL y, TVIL_REAL qq)
{
  TVIL_REAL r, rn;
  TVIL_COMPLEX lnyox, lnyox2, lnyox3, result;
  int n;

  TVIL_REAL H0xxxxytableA[] = 
     {0, -1.7675544522665516, -0.3577926178617032, 
     -0.10013827471358111, -0.03607585071442304, -0.015994402745572518, 
     -0.008325424508411297, -0.004872037634407313, -0.0031012254455798686, 
     -0.0020991300252876, -0.0014883009976347193, -0.0010941470543540514, 
     -0.0008281554718869412, -0.0006420412270983093, -0.0005078673352673009, 
     -0.0004086751928026906, -0.0003337517839613865, -0.0002760976652216289, 
     -0.0002310055870477178, -0.0001952293431684656, -0.00016648039025338784, 
     -0.00014311351537840263, -0.00012392520014922417, 
     -0.00010802114867597718, -0.00009472737403286129, 
     -0.00008352935987318763, -0.00007402969938426736, -0.000065918127363999, 
     -0.00005895000933336414, -0.00005293069363638506, 
     -0.000047703987636639484, -0.000043143573994513836, 
     -0.00003914654910041419, -0.000035628511028195466, 
     -0.000032519791088008575, -0.000029762537884479294, 
     -0.000027308442860442566, -0.00002511695279678097, 
     -0.00002315385502621041, -0.000021390150144870248, 
     -0.000019801148119551566, -0.00001836573918644036, 
     -0.00001706580241057239, -0.000015885723337800086, 
     -0.000014811998610426358, -0.000013832910295506565, 
     -0.00001293825639535738, -0.000012119126866277976, 
     -0.000011367716678289534, -0.000010677169163768325, 
     -0.000010041444243349654, -9.455207170923639e-6, 
     -8.913734271711233e-6, -8.412832808057008e-6, -7.9487726345621e-6, 
     -7.518227726475444e-6, -7.118226005151312e-6, -6.746106159103133e-6, 
     -6.399480382135292e-6, -6.0762021316799165e-6, -5.774338159015145e-6, 
     -5.492144184966324e-6, -5.228043695114391e-6, -4.9806094115279715e-6, 
     -4.748547066846724e-6, -4.5306811637764345e-6, -4.32594245080439e-6, 
     -4.133356884896173e-6, -3.9520358854602354e-6, 
     -3.7811677120762356e-6, -3.620009822283577e-6, -3.46788208585909e-6, 
     -3.3241607490855818e-6, -3.188273057026369e-6, 
     -3.0596924541878745e-6, -2.937934294514626e-6, -2.822552000702052e-6, 
     -2.7131336205683044e-6, -2.6092987348936575e-6, 
     -2.510695676879446e-6, -2.4169990283359157e-6, -2.327907361995688e-6, 
     -2.2431412030643883e-6, -2.1624411863447402e-6, 
     -2.0855663880748257e-6, -2.01229281406425e-6, -1.942412027843942e-6, 
     -1.8757299044088812e-6, -1.8120654967646859e-6, 
     -1.7512500039198211e-6, -1.6931258302218833e-6, 
     -1.6375457270418256e-6, -1.5843720087837872e-6, 
     -1.5334758360572677e-6, -1.4847365596073494e-6, 
     -1.4380411192701006e-6, -1.3932834928150927e-6, 
     -1.350364190064582e-6, -1.3091897881475226e-6, -1.269672504163336e-6, 
     -1.231729801901448e-6};
 
  TVIL_REAL H0xxxxytableAL[] = 
     {0, 1.1719536193447295, 0.3168708519673948, 
     0.11903850330308469, 0.057462386783118206, 0.03314649231862221, 
     0.021533846072570046, 0.015141134714803592, 0.01124405253854647, 
     0.008687643271417442, 0.006917393699558512, 0.005639694242112531, 
     0.004686796122648348, 0.003956971291556988, 0.003385516652770231, 
     0.002929642758184195, 0.0025601165841689887, 0.002256407495349588, 
     0.002003749730001737, 0.00179130144545546, 0.001610955618583985, 
     0.0014565514585759664, 0.0013233391147385134, 0.001207608715566282, 
     0.0011064284732439477, 0.0010174566659140402, 0.0009388045929225086, 
     0.0008689352926315067, 0.0008065877368829946, 0.0007507194300256417, 
     0.0007004624755668, 0.0006550896154356614, 0.0006139877354127027, 
     0.0005766370175517076, 0.0005425944044449205, 0.0005114803851922211, 
     0.00048296836163265206, 0.000456776034559036, 0.0004326583829074676, 
     0.00041040190786318433, 0.0003898198879327764, 0.00037074844699287263, 
     0.0003530432799098789, 0.00033657691296918853, 0.0003212364015501222, 
     0.000306921387061489, 0.0002935424504605776, 0.0002810197117185521, 
     0.00026928163411898905, 0.00025826399985037893, 0.00024790902940810945, 
     0.0002381646221854486, 0.00022898369955898288, 0.00022032363495709585, 
     0.00021214575799212548, 0.0002044149218563615, 0.00019709912492209858, 
     0.00019016917891997065, 0.00018359841725594641, 0.00017736243801199235, 
     0.00017143887699554175, 0.00016580720688828308, 0.00016044855911935942, 
     0.00015534556557124444, 0.00015048221763407375, 0.00014584374046889588, 
     0.00014141648063265778, 0.00013718780546636688, 0.00013314601285986205, 
     0.00012928025018783248, 0.00012558044136699265, 0.00012203722111768036, 
     0.00011864187562794765, 0.00011538628891725967, 0.00011226289428255566, 
     0.00010926463028362443, 0.00010638490078916603, 0.0001036175386609491, 
     0.00010095677270231588, 0.00009839719753993471, 0.00009593374614500989, 
     0.00009356166473285335, 0.0000912764898084249, 0.00008907402715068588, 
     0.00008695033255084037, 0.00008490169413914756, 0.00008292461615231578, 
     0.00008101580400881967, 0.00007917215057306939, 0.0000773907235014189, 
     0.00007566875357371623, 0.0000740036239236364, 0.00007239286008953617, 
     0.00007083412081515383, 0.00006932518953625345, 0.00006786396649537437, 
     0.00006644846143227654, 0.00006507678680254113, 0.00006374715148115759, 
     0.00006245785491185736, 0.000061207281666491};
 
  TVIL_REAL H0xxxxytableC[] = 
     {0, -0.9464613263673327, 0.043745301052738834, 
     0.027661536657969416, 0.013068077174878434, 0.006760244062935916, 
     0.0038883864158926187, 0.002445110271852425, 0.0016481163582718048, 
     0.0011710132742743745, 0.0008659775472780104, 0.0006604868674497673, 
     0.0005162302495518194, 0.00041158879846670845, 0.00033364751454754277, 
     0.000274311466716928, 0.00022830014649115774, 0.0001920529045347342, 
     0.00016310140525583857, 0.0001396940041653771, 0.00012056242680275694, 
     0.00010477217128303487, 0.00009162395108290528, 0.00008058722495967153, 
     0.00007125444288809198, 0.00006330897985064199, 0.00005650229926476348, 
     0.0000506374529152434, 0.000045557001796770914, 0.00004113406661162514, 
     0.000037265623404476414, 0.000033867429530913136, 
     0.00003087014689213523, 0.00002821635361764726, 0.000025858221464876852, 
     0.000023755696588828217, 0.000021875064176260145, 
     0.00002018780815892834, 0.0000186696994689745, 0.000017300062563928627, 
     0.0000160611819435692, 0.000014937819300579272, 0.000013916818632140117, 
     0.00001298678168745813, 0.00001213779996471952, 0.000011361232409471903, 
     0.000010649520230350925, 9.99603200289777e-6, 9.394933600338709e-6, 
     8.841078562758742e-6, 8.329915361414189e-6, 7.857408684526262e-6, 
     7.419972403886686e-6, 7.014412307837156e-6, 6.6378770285672195e-6, 
     6.2878158678875375e-6, 5.961942449387304e-6, 5.6582033068606645e-6, 
     5.374750667453944e-6, 5.109918809724496e-6, 4.862203476912565e-6, 
     4.630243908334758e-6, 4.412807120197836e-6, 4.208774123932489e-6, 
     4.017127817468549e-6, 3.836942324415453e-6, 3.6673735892493737e-6, 
     3.507651064456474e-6, 3.357070349048135e-6, 3.2149866576902293e-6, 
     3.0808090164819138e-6, 2.953995095679269e-6, 2.8340466017965996e-6, 
     2.7205051618734333e-6, 2.6129486415498694e-6, 2.5109878461811176e-6, 
     2.414263560738832e-6, 2.3224438898551957e-6, 2.2352218642019813e-6, 
     2.1523132835755675e-6, 2.0734547706764453e-6, 1.9984020127093057e-6, 
     1.926928170656053e-6, 1.8588224384471101e-6, 1.7938887363255619e-6, 
     1.7319445245059485e-6, 1.6728197248106292e-6, 1.6163557393521217e-6, 
     1.562404556545783e-6, 1.5108279358060239e-6, 1.461496663220138e-6, 
     1.4142898713233158e-6, 1.3690944168307365e-6, 1.3258043108300604e-6, 
     1.2843201965108235e-6, 1.2445488700153471e-6, 1.206402840446787e-6, 
     1.1697999254708098e-6, 1.134662879304096e-6, 1.1009190502006998e-6, 
     1.068500064830811e-6};
 
  TVIL_REAL H0xxxxytableCL[] = 
     {0, 1.5232010529118827, 0.14707264874754997, 
     0.03958305220158904, 0.017926379621714828, 0.009834863385773138, 
     0.005929232382712244, 0.0038189081356117915, 0.002592786668813834, 
     0.0018378918759861384, 0.0013497713754620623, 0.0010206992458221072, 
     0.0007908494623444575, 0.0006254039037832423, 0.0005032292766581114, 
     0.0004110088742554716, 0.00034006855343555125, 0.0002845907138109031, 
     0.00024057199260236028, 0.0002051943631808603, 0.00017643401419534873, 
     0.0001528113739175463, 0.000133227370943154, 0.00011685378615097817, 
     0.00010305835178391838, 0.00009135265894845006, 0.00008135533313814266, 
     0.00007276561390750511, 0.00006534414037313265, 0.000058898802134909246, 
     0.000053274199936683365, 0.000048343711297599604, 
     0.000044003458046746845, 0.00004016767754683126, 0.00003676514039046085, 
     0.00003373635563278982, 0.00003103137393903881, 0.00002860804845139003, 
     0.00002643064878809512, 0.000024468749490140717, 
     0.000022696333245019193, 0.000021091063294977863, 
     0.000019633689944845677, 0.000018307563987852178, 
     0.000017098235855613683, 0.000015993123866340105, 
     0.00001498123845293497, 0.000014052951962987606, 
     0.000013199805729418249, 0.000012414347757470153, 
     0.000011689995668217415, 0.000011020920561497852, 
     0.00001040194827325474, 9.828475150151742e-6, 9.296395983577346e-6, 
     8.802042163145833e-6, 8.342128447699838e-6, 7.913707026081114e-6, 
     7.51412776341618e-6, 7.141003711449851e-6, 6.792181111504867e-6, 
     6.465713242245164e-6, 6.159837566578934e-6, 5.87295571674991e-6, 
     5.603615927132038e-6, 5.350497583037984e-6, 5.1123976030533235e-6, 
     4.888218413698783e-6, 4.676957309970057e-6, 4.47769702462373e-6, 
     4.289597353880829e-6, 4.111887708254824e-6, 3.94386047509445e-6, 
     3.784865094671675e-6, 3.634302764662281e-6, 3.4916216990097313e-6, 
     3.3563128767229944e-6, 3.2279062243778273e-6, 3.1059671831714963e-6, 
     2.9900936174928648e-6, 2.8799130272559205e-6, 2.7750800308254515e-6, 
     2.6752740893401574e-6, 2.5801974466968906e-6, 2.4895732624727427e-6, 
     2.403143917691213e-6, 2.320669475637285e-6, 2.2419262819389043e-6, 
     2.1667056898973926e-6, 2.094812898599648e-6, 2.0260658927087454e-6, 
     1.960294474031013e-6, 1.8973393760176097e-6, 1.8370514532951333e-6, 
     1.7792909391482874e-6, 1.7239267646116487e-6, 1.6708359334786799e-6, 
     1.6199029481144804e-6, 1.5710192814731137e-6, 1.5240828911783591e-6, 
     1.4789977719350634e-6};
 
  TVIL_REAL H0xxxxytableCL2[] = 
     {0, -0.4444444444444444, -0.024691358024691357, 
     0.008763907940862674, 0.007925621094345374, 0.005115920676048705, 
     0.0032201808123782528, 0.002084469228355917, 0.0014030596906003375, 
     0.0009816625861577947, 0.0007111671394827893, 0.0005309809151726876, 
     0.00040678122897865195, 0.00031852861235745185, 0.00025412676164942677, 
     0.00020602845668485833, 0.00016937503881931553, 0.00014094722297246486, 
     0.00011855555052254765, 0.00010067564811465971, 0.00008622370828675565, 
     0.00007441456664153332, 0.00006466976003685466, 0.000056556608350625834, 
     0.00004974701451154269, 0.00004398907459200141, 0.00003908717911786961, 
     0.00003488784759891447, 0.00003126950009450598, 0.000028134974631263555, 
     0.000025405987205458912, 0.000023018984282719774, 
     0.000020922005689266133, 0.00001907228895385508, 
     0.000017434423480987056, 0.000015978916465100768, 
     0.000014681069969413119, 0.000013520095185110266, 
     0.000012478408938967145, 0.000011541071305289759, 
     0.000010695333250395488, 9.930270661424373e-6, 9.236486628613097e-6, 
     8.605867983305044e-6, 8.031385213456123e-6, 7.506927249608749e-6, 
     7.02716442893616e-6, 6.5874343425910326e-6, 6.183646354651523e-6, 
     5.812201425153878e-6, 5.469924531386133e-6, 5.1540075030163316e-6, 
     4.86196049958042e-6, 4.591570687514034e-6, 4.340866936710226e-6, 
     4.108089567664379e-6, 3.891664350538119e-6, 3.690180095395424e-6, 
     3.502369285027961e-6, 3.3270912933536934e-6, 3.1633178074017458e-6, 
     3.0101201325936625e-6, 2.8666581119399218e-6, 2.732170431916924e-6, 
     2.6059661227936752e-6, 2.4874170903395633e-6, 2.3759515402127397e-6, 
     2.271048176749846e-6, 2.172231075039449e-6, 2.0790651396231554e-6, 
     1.9911520753867743e-6, 1.9081268065535407e-6, 1.8296542884796985e-6, 
     1.755426664433335e-6, 1.6851607259195632e-6, 1.618595640572702e-6, 
     1.5554909163131964e-6, 1.4956245744838802e-6, 1.4387915081371863e-6, 
     1.3848020046261335e-6, 1.3334804142278962e-6, 1.2846639487588823e-6, 
     1.2382015960745337e-6, 1.1939531380278394e-6, 1.1517882609236148e-6, 
     1.1115857487813928e-6, 1.073232750834103e-6, 1.0366241156646526e-6, 
     1.0016617852368703e-6, 9.682542428271054e-7, 9.363160095219159e-7, 
     9.05767184527533e-7, 8.765330250483761e-7, 8.485435619435533e-7, 
     8.217332477696228e-7, 7.960406341714323e-7, 7.714080758962724e-7, 
     7.477814589848083e-7, 7.251099509395309e-7, 7.033457708915381e-7, 
     6.8244397798254e-7};
 
  TVIL_REAL H0xxxxytableCL3[] = 
     {0, 0.09259259259259259, 0.0257201646090535, 
     0.0086114921505868, 0.003410303307422649, 0.001555205563740848, 
     0.0007991375231576872, 0.0004542058250748397, 0.00028063742261189495, 
     0.0001854782971805928, 0.00012930287570137605, 0.00009400768422676758, 
     0.0000706587551251951, 0.00005454746293798338, 0.00004304031121216454, 
     0.000034584792139263654, 0.000028222724040942366, 
     0.000023338948459348958, 0.00001952499084268175, 
     0.000016501686569809404, 0.000014073460168823303, 
     0.000012100323479050176, 0.000010480176804990798, 9.137308525306947e-6, 
     8.014738518035585e-6, 7.069010119391214e-6, 6.2665794064542e-6, 
     5.5812688255294194e-6, 4.992443595628343e-6, 4.483687376593267e-6, 
     4.041828178452157e-6, 3.6562134534611093e-6, 3.3181647741157023e-6, 
     3.0205634864709205e-6, 2.7575329426564723e-6, 2.524192680868631e-6, 
     2.316466716205505e-6, 2.130932891673721e-6, 1.964703647521723e-6, 
     1.8153310205250627e-6, 1.6807304680878986e-6, 1.5591194201019014e-6, 
     1.4489674293783939e-6, 1.3489555134995267e-6, 1.2579428237450304e-6, 
     1.1749391878267257e-6, 1.0990823866447893e-6, 1.0296192659210975e-6, 
     9.658899694551443e-7, 9.073147252062869e-7, 8.533827283098421e-7, 
     8.036427538589863e-7, 7.576952023722989e-7, 7.151853365097852e-7, 
     6.757975119871308e-7, 6.39250241208549e-7, 6.052919567697683e-7, 
     5.736973651259396e-7, 5.44264299502651e-7, 5.168109964329057e-7, 
     4.911737328201835e-7, 4.672047707024579e-7, 4.447705653556131e-7, 
     4.2375019936994884e-7, 4.0403401113399637e-7, 3.855223909849246e-7, 
     3.681247223106764e-7, 3.517584482579316e-7, 3.363482475272773e-7, 
     3.218253051162402e-7, 3.0812666587844386e-7, 2.951946604655576e-7, 
     2.8297639465914823e-7, 2.714232943241817e-7, 2.604906992595653e-7, 
     2.501375001125607e-7, 2.40325813287035e-7, 2.3102068943024305e-7, 
     2.221898516457309e-7, 2.1380346006489747e-7, 2.0583389982837757e-7, 
     1.982555898904874e-7, 1.9104481037373778e-7, 1.8417954647281884e-7, 
     1.7763934714437174e-7, 1.714051970252733e-7, 1.65459400202303e-7, 
     1.5978547461354254e-7, 1.5436805599975392e-7, 1.4919281044491586e-7, 
     1.4424635465132717e-7, 1.3951618318813363e-7, 1.3499060203446437e-7, 
     1.3065866781100225e-7, 1.265101321579899e-7, 1.2253539077445582e-7, 
     1.1872543668374929e-7, 1.1507181733510119e-7, 1.1156659519056826e-7, 
     1.0820231148197409e-7, 1.0497195285385657e-7};
 
  TVIL_REAL H0xxxxytableB[] = 
     {0, -2.7221166278184046, 0.8407068480549538, 
     -0.4072692801419656, 0.24046308534883187, -0.15883609233521884, 
     0.11280576897982418, -0.08428629494376033, 0.06538722889638769, 
     -0.05221464234205848, 0.042664848228129035, -0.03551954553544501, 
     0.030033189925170335, -0.025728607170497995, 0.02228879845895765, 
     -0.01949644761587971, 0.017198513059239723, -0.01528467835321557, 
     0.013673788858662933, -0.012305061208051263, 0.011132237486755448, 
     -0.010119607927417697, 0.00923924917001971, -0.008469071147690847, 
     0.00779141279959953, -0.007192017111090683, 0.006659272697587128, 
     -0.006183645530530852, 0.005757248194113746, -0.005373509894595206, 
     0.00502692115308027, -0.004712834464910593, 0.004427307326724278, 
     -0.004166977640789943, 0.003928964081110481, -0.0037107858637004283, 
     0.003510297718060225, -0.0033256368542375, 0.003155179460971673, 
     -0.002997504825886982, 0.0028513655884375923, -0.002715662955934269, 
     0.0025894259581528806, -0.0024717940053872994, 0.002362002162028034, 
     -0.002259368662917131, 0.0021632842903618506, -0.0020732033014157067, 
     0.001988635652105665, -0.00190914031092722, 0.0018343194906125114, 
     -0.001763813656798141, 0.00169729719624634, -0.0016344746468463973, 
     0.0015750774076345642, -0.001518860860220308, 0.001465601843847596, 
     -0.001415096435290137, 0.0013671579922282474, -0.001321615424961401, 
     0.0012783116664988015, -0.0012371023154212268, 0.0011978544295672764, 
     -0.001160445451684835, 0.0011247622508009636, -0.0010907002652798033, 
     0.0010581627354234737, -0.0010270600150789654, 0.000997308953088878, 
     -0.0009688323366022195, 0.0009415583892738208, -0.0009154203182525867, 
     0.0008903559046110376, -0.0008663071325191248, 0.0008432198530290826, 
     -0.0008210434788276359, 0.0007997307067378638, -0.0007792372651243491, 
     0.0007595216836795782, -0.0007405450833533226, 0.0007222709844354696, 
     -0.0007046651310211602, 0.0006876953302791659, -0.000671331305113643, 
     0.0006555445589586932, -0.0006403082515770949, 0.0006255970848513304, 
     -0.0006113871976585305, 0.0005976560690128213, -0.0005843824287402109, 
     0.0005715461750238397, -0.0005591282982221922, 0.0005471108104206852, 
     -0.0005354766802287113, 0.000524209772380434, -0.0005132947917390456, 
     0.0005027172313413319, -0.0004924633241527396, 0.00048251999823311655, 
     -0.00047287483504027637, 0.00046351603062284624};
 
  TVIL_REAL H0xxxxytableD[] = 
     {0, -1.453820765467028, 0.3480399477631727, 
     -0.14835729014382976, 0.08121095843053454, -0.05103328123901877, 
     0.034983882952256545, -0.02545709137846989, 0.019347481935684144, 
     -0.015198041472921925, 0.01225241643284258, -0.010086541009301812, 
     0.008447782144846677, -0.007178068076543449, 0.006174404597168607, 
     -0.005367361206456142, 0.004708756434202222, -0.004164308045448902, 
     0.003709086138782193, -0.003324611158885063, 0.002996950414491502, 
     -0.0027154389884974602, 0.002471801514051511, -0.002259537431462495, 
     0.002073483121186506, -0.0019094950613825514, 0.0017642172412521708, 
     -0.0016349081661626225, 0.0015193106246025627, -0.001415552550998794, 
     0.001322070780209082, -0.0012375518465057855, 0.0011608856078629524, 
     -0.0010911286159021429, 0.0010274749594311923, -0.0009692328884770263, 
     0.0009158059452883402, -0.0008666776359373637, 0.0008213989031485004, 
     -0.0007795778302468382, 0.0007408711334011568, -0.000704977095810545, 
     0.0006716296711587543, -0.0006405935403255571, 0.0006116599492217603, 
     -0.0005846431898082617, 0.0005593776131695969, -0.0005357150846554498, 
     0.0005135228078699081, -0.0004926814576541625, 0.00047308357291735414, 
     -0.0004546321687919681, 0.0004372395345635108, -0.0004208261894893118, 
     0.00040531997324316896, -0.00039065525150868037, 0.0003767722203577644, 
     -0.0003636162956211427, 0.0003511375755869677, -0.00033929036713409305, 
     0.00032803276688321364, -0.0003173262901849248, 0.0003071355418012019, 
     -0.0002974279230103515, 0.0002881733706031772, -0.0002793441238628166, 
     0.0002709145161511929, -0.0002628607881766795, 0.00025516092040310046, 
     -0.00024779448239007875, 0.00024074249713770265, 
     -0.00023398731875175286, 0.00022751252195535247, -0.0002213028021539223, 
     0.00021534388491697943, -0.00020962244387616536, 0.00020412602615693285, 
     -0.00019884298456409024, 0.00019376241583104727, 
     -0.00018887410432094284, 0.0001841684706364182, -0.00017963652465494316, 
     0.0001752698225594393, -0.00017106042748043712, 0.00016700087340698012, 
     -0.0001630841320596619, 0.00015930358245116257, -0.0001556529828879673, 
     0.00015212644519205977, -0.0001487184109436798, 0.0001454236295660635, 
     -0.00014223713809073855, 0.00013915424245769097, -0.0001361705002187745, 
     0.00013328170452529836, -0.0001304838692919745, 0.00012777321543948453, 
     -0.00012514615812696717, 0.00012259929489385042, 
     -0.00012012939463775604, 0.00011773338736177957};
 
  TVIL_REAL H0xxxxytableF[] = 
     {0, -3.5912266258392047, 1.25365015412144, -0.6522514381539645, 
     0.403962322764129, -0.2762608171130646, 0.20148856177156463, 
     -0.1537658869080048, 0.12136839849934149, -0.09832851595565442, 
     0.08133750694615917, -0.06843619458804859, 0.05840233941480663, 
     -0.05044032441010991, 0.0440136148217911, -0.03874942685856744, 
     0.03438204234484791, -0.030717815475625212, 0.027612861007504128, 
     -0.024958421664321995, 0.022671030897649394, -0.02068575247163603, 
     0.01895144238313138, -0.017427369009295937, 0.01608076335173182, 
     -0.014885017498896966, 0.013818342142125654, -0.01286275397142588, 
     0.012003303327363896, -0.011227479010079129, 0.010524745218051551, 
     -0.009886178081921179, 0.009304178011743556, -0.008772240286119247, 
     0.008284770769065862, -0.007836936874892958, 0.0074245462722562, 
     -0.007043947573176822, 0.006691948563007857, -0.00636574851396059, 
     0.006062881873715496, -0.005781171193398323, 0.00551868760036576, 
     -0.005273717463349118, 0.005044734164495313, -0.00483037410248544, 
     0.004629416216476991, -0.004440764452095913, 0.004263432695667469, 
     -0.004096531787090383, 0.003939258289653376, -0.0037908847500819627, 
     0.0036507512268358993, -0.0035182579012204133, 0.0033928586158473916, 
     -0.0032740552096617477, 0.00316139253914256, -0.0030544540921993877, 
     0.0029528581153490372, -0.0028562541864844185, 0.0027643201753366066, 
     -0.0026767595418964118, 0.0025932989298348657, -0.0025136860174947246, 
     0.002437687593375159, -0.0023650878261304285, 0.0022956867006891798, 
     -0.0022292985915976365, 0.0021657509409848123, -0.002104882999596911, 
     0.0020465445714744008, -0.00199059466949529, 0.0019368999285501066, 
     -0.0018853325149403566, 0.0018357670782205948, -0.0017880759505655534, 
     0.0017421211945606236, -0.0016977410308148012, 0.0016547262839089257, 
     -0.0016127791349426178, 0.0015714405386681113, -0.0015299621645186478, 
     0.0014870801314842451, -0.0014406148862784185, 0.0013867632643743723, 
     -0.0013188454758646627, 0.0012250867330788466, -0.0010846888963740365};

  lnyox = TVIL_CLOG(y/x);
  lnyox2 = lnyox  * lnyox;
  lnyox3 = lnyox2  * lnyox;

  if (y < 0.5 * x) {
    result = TVIL_H00xxxx (x,qq);
    rn = r = y/x;
    for (n=1; n < 100; n++) {
      result += rn * (H0xxxxytableA[n] + 
                      H0xxxxytableAL[n] * lnyox);
      rn *= r;
    }
  }
  else if (y < 1.6 * x) {
    result = TVIL_H0xxxxx (x,qq);
    rn = r = y/x - 1.L;
    for (n=1; n < 100; n++) {
      result += rn * H0xxxxytableD[n];
      rn *= r;
    }
  }
  else if (y < 5.59 * x) {
    result = -11.065385063863448169332393L - 6.L * Zeta3 * TVIL_Ap(x,qq);
    rn = r = y/(4.L*x) - 1.L;
    for (n=1; n < 100; n++) {
      result += rn * H0xxxxytableB[n];
      rn *= r;
    }
  }
  else if (y < 14.56 * x) {
    result = -13.622846650979722025603266L - 6.L * Zeta3 * TVIL_Ap(x,qq);
    rn = r = y/(9.L*x) - 1.L;
    for (n=1; n < 86; n++) {
      result += rn * H0xxxxytableF[n];
      rn *= r;
    }
  }
  else {
    result = TVIL_H00000x (y, qq);
    rn = r = 9.L * x/y;
    for (n=1; n < 100; n++) {
      result += rn * (H0xxxxytableC[n] 
                     -H0xxxxytableCL[n] * lnyox 
                     +H0xxxxytableCL2[n] * lnyox2
                     -H0xxxxytableCL3[n] * lnyox3);
      rn *= r;
    }
  }

  return result;
}

/* ****************************************************************** */
TVIL_COMPLEX TVIL_H0xxxyy (TVIL_REAL x, TVIL_REAL y, TVIL_REAL qq)
{
  TVIL_REAL r, rn;
  TVIL_COMPLEX lnyox, lnyox2, lnyox3, result;
  int n;

  TVIL_REAL H0xxxyytableA[] = 
     {0, -0.9701351305356795, 0.030230514248257918, 
     0.04442300104500347, 0.03570712896676171, 0.028671856870820047, 
     0.02355089337926542, 0.019750024400405054, 0.01685218921281486, 
     0.014588915887374574, 0.012784213442346112, 0.011319323803982864, 
     0.010111846180161421, 0.009103155548983907, 0.008250602573710316, 
     0.0075225393147969185, 0.0068950628880843365, 0.006349832325824792, 
     0.005872572941705358, 0.005452031380919277, 0.005079232398715547, 
     0.004746941570735752, 0.004449271054633419, 0.004181386344878775, 
     0.003939285398022381, 0.00371963033494419, 0.003519617828199685, 
     0.0033368782898736157, 0.003169396736502695, 0.0030154501362428636, 
     0.0028735574077858993, 0.002742439217203556, 0.002620985425875977, 
     0.0025082285597783733, 0.0024033220523644837, 0.002305522298030714, 
     0.00221417376726605, 0.00212869659693063, 0.0020485761931403655, 
     0.001973354479700659, 0.0019026224990216505, 0.0018360141301710289, 
     0.0017732007340371873, 0.0017138865713646932, 0.0016578048678496625, 
     0.0016047144231822897, 0.001554396679144756, 0.0015066531765711043, 
     0.0014613033428885086, 0.001418182561658363, 0.0013771404834664164, 
     0.0013380395440231597, 0.001300753660704013, 0.0012651670832013757, 
     0.0012311733776505485, 0.0011986745266672465, 0.0011675801303069162, 
     0.0011378066951147645, 0.0011092770002525884, 0.0010819195312228996, 
     0.001055667973010275, 0.001030460755563467, 0.0010062406454816723, 
     0.0009829543785708952, 0.0009605523286234216, 0.0009389882083630681, 
     0.0009182187990061859, 0.0008982037053258447, 0.0008789051334847209, 
     0.0008602876892297138, 0.0008423181943255912, 0.0008249655193522582, 
     0.0008082004312057672, 0.0007919954538314147, 0.0007763247408819609, 
     0.0007611639591383703, 0.0007464901816572503, 0.0007322817897206875, 
     0.0007185183827624564, 0.0007051806955313219, 0.0006922505218288435, 
     0.0006797106442269997, 0.0006675447692311711, 0.0006557374674075146, 
     0.0006442741180413303, 0.0006331408579354099, 0.0006223245339951482, 
     0.000611812659280967, 0.0006015933722387971, 0.000591655398846404, 
     0.000581988017437594, 0.0005725810259881037, 0.0005634247116665463, 
     0.0005545098224713947, 0.0005458275407908492, 0.0005373694587367461, 
     0.0005291275551165875, 0.000521094173919452, 0.0005132620042021177, 
     0.0005056240612713033, 0.0004981736690666182};
 
  TVIL_REAL H0xxxyytableAL[] = 
     {0, 0.878965214508547, 0.1599610350217345, 0.09037413639063974, 
     0.06502796787716504, 0.05005700915193459, 0.040011045469386615, 
     0.03287534276338331, 0.0276051160534065, 0.023590723886315616, 
     0.02045421307237939, 0.017950843464470446, 0.015916374766697362, 
     0.014237171406591568, 0.012832499627251664, 0.011643656638210268, 
     0.010627079493865578, 0.009749851740845852, 0.008986698776823723, 
     0.008317932792024071, 0.007728018326721361, 0.007204552470240619, 
     0.006737527675737931, 0.00631879073687899, 0.0059416401979702875, 
     0.005600522954470484, 0.005290802925027438, 0.005008582768856888, 
     0.0047505651121129665, 0.004513943527256734, 0.004296316149054172, 
     0.004095616677923491, 0.003910058858004665, 0.0037380914850633894, 
     0.003578361707402652, 0.003429684906159007, 0.0032910198315717603, 
     0.0031614479653887524, 0.0030401563022693433, 0.002926422913293941, 
     0.0028196047858023055, 0.002719127535461601, 0.0026244766658398283, 
     0.002535190113113199, 0.0024508518628103006, 0.0023710864646565334, 
     0.0022955543028738675, 0.002223947504424172, 0.00215598638797094, 
     0.0020914163727861904, 0.002030005280231818, 0.001971540971410239, 
     0.0019158292735881532, 0.0018626921554282746, 0.001811966117216786, 
     0.001763500767387608, 0.001717157560909058, 0.001672808678667084, 
     0.0016303360299753894, 0.0015896303628661738, 0.0015505904689469784, 
     0.0015131224714153386, 0.0014771391863576656, 0.0014425595487663951, 
     0.001409308095826526, 0.0013773145009794297, 0.0013465131530933088, 
     0.0013168427757767063, 0.0012882460824814397, 0.0012606694635687113, 
     0.001234062701969158, 0.0012083787144644587, 0.0011835733159634495, 
     0.001159605004446803, 0.0011364347645173823, 0.001114025887723622, 
     0.00109234380802519, 0.001071355950947544, 0.001051031595128083, 
     0.001031341745094156, 0.0010122590142346934, 0.0009937575170346337, 
     0.0009758127697364925, 0.0009584015986778276, 0.000941502055628359, 
     0.0009250933395172245, 0.0009091557240003121, 0.000893670490370654, 
     0.000878619865362265, 0.0008639869634402023, 0.0008497557332076105, 
     0.000835910907594577, 0.0008224379575242129, 0.0008093230487788812, 
     0.0007965530018142478, 0.0007841152542911443, 0.0007719978261153598, 
     0.0007601892867936585, 0.000748678724930759, 0.000737455719706891, 
     0.0007265103141890225};
 
  TVIL_REAL H0xxxyytableAL2[] = 
     {0, 0., 0.0625, 0.044270833333333336, 0.030859375, 
     0.022670200892857144, 0.017463611421130953, 0.013958033441981196, 
     0.011477313579974713, 0.009649627077357637, 0.008258597310643778, 
     0.007171489463611011, 0.0063030847412225985, 0.005596515136200329, 
     0.005012565599748467, 0.004523424596961729, 0.0041088847481238365, 
     0.003753951156237329, 0.0034472892713380363, 0.0031801902142543233, 
     0.002945864475933789, 0.0027389494722620126, 0.002555159636736333, 
     0.0023910335148826386, 0.0022437481241091768, 0.002110980760774004, 
     0.001990804799385826, 0.0018816101931643826, 0.0017820421604530805, 
     0.0016909534221386988, 0.0016073666493978032, 0.0015304446843943888, 
     0.0014594667353955509, 0.0013938092051377655, 0.0013329301424699134, 
     0.0012763565496929816, 0.001223673957195415, 0.0011745178106767798, 
     0.0011285663168873392, 0.0010855344701909194, 0.0010451690406830833, 
     0.0010072443496176322, 0.0009715586928296891, 0.0009379313001323977, 
     0.0009061997401143526, 0.000876217696726406, 0.000847853057535011, 
     0.0008209862643040836, 0.0007955088852349842, 0.0007713223751946803, 
     0.0007483369959427804, 0.0007264708729985759, 0.000705649169579948, 
     0.0006858033611621444, 0.000666870596776209, 0.0006487931352974819, 
     0.0006315178467461965, 0.0006149957701005019, 0.0005991817203599562, 
     0.0005840339386372366, 0.000569513779931906, 0.0005555854339805635, 
     0.0005422156752054148, 0.0005293736383168986, 0.000517030616580868, 
     0.0005051598801495732, 0.0004937365121888027, 0.0004827372608196758, 
     0.00047214040513994955, 0.0004619256338023414, 0.0004520739348112918, 
     0.0004425674953590327, 0.0004333896106603407, 0.0004245246008659263, 
     0.00041595773523958243, 0.0004076751628761299, 0.0003996638493176745, 
     0.0003919115184962885, 0.0003844065994932649, 0.0003771381776597053, 
     0.00037009594969135694, 0.0003632701822931453, 0.0003566516741064779, 
     0.00035023172060572906, 0.00034400208169990237, 0.0003379549518017571, 
     0.0003320829321500836, 0.00032637900519166764, 0.000320836510848095, 
     0.0003154491245091817, 0.0003102108366096978, 0.00030511593365939243, 
     0.0003001589806082886, 0.00029533480443996675, 0.0002906384788952188, 
     0.00028606531023715796, 0.00028161082397671326, 0.0002772707524845159, 
     0.0002730410234215804, 0.00026891774892696626, 0.00026489721550584233};
 
  TVIL_REAL H0xxxyytableC[] = 
     {0, -0.8848090046795724, -0.009639699869155716, 
     -0.004542949817107669, -0.003990348905037823, -0.00291406744458222, 
     -0.00202907971753062, -0.0014150656458194837, -0.0010058264841250738, 
     -0.0007326997906835697, -0.0005472419680420892, -0.0004183692391163972, 
     -0.00032660677541389245, -0.0002597146944804419, 
     -0.00020988353240488215, -0.0001720283071541541, 
     -0.00014276375961923824, -0.00011978557253868833, 
     -0.00010149149803471656, -0.00008674511683251229, 
     -0.00007472554990550868, -0.00006482987371235726, 
     -0.000056608475645480675, -0.00004972140520565552, 
     -0.00004390836879128771, -0.0000389677567467827, 
     -0.00003474175592001338, -0.00003110563042012439, 
     -0.000027959901459713213, -0.00002522457250617932, 
     -0.000022834816599858363, -0.00002073772186129249, 
     -0.000018889811607774807, -0.00001725513754887367, 
     -0.00001580380118561988, -0.000014510798141958602, 
     -0.000013355108162142181, -0.000012318973525742118, 
     -0.000011387323086460405, -0.000010547309678028611, 
     -9.787936381087375e-6, -9.099752894382605e-6, -8.474607552846756e-6, 
     -7.905443774393155e-6, -7.386132175468849e-6, -6.911331473678844e-6, 
     -6.476372740276118e-6, -6.07716268307384e-6, -5.710102510387423e-6, 
     -5.372019607668393e-6, -5.06010979447665e-6, -4.771888353426896e-6, 
     -4.505148359795489e-6, -4.257925109679449e-6, -4.028465660591688e-6, 
     -3.815202672429152e-6, -3.6167318775938564e-6, -3.431792623476022e-6, 
     -3.259251023837449e-6, -3.098085332036883e-6, -2.9473732118116493e-6, 
     -2.806280633082209e-6, -2.6740521630559268e-6, 
     -2.5500024584330714e-6, -2.433508794092989e-6, 
     -2.3240044883329846e-6, -2.220973105412296e-6, 
     -2.1239433335203832e-6, -2.032484450913238e-6, -1.94620230530936e-6, 
     -1.8647357420895826e-6, -1.7877534257151652e-6, 
     -1.7149510063242131e-6, -1.6460485899004741e-6, 
     -1.5807884759071718e-6, -1.518933130988209e-6, -1.46026337138169e-6, 
     -1.4045767301680362e-6, -1.3516859884721676e-6, 
     -1.301417852327692e-6, -1.2536117591508724e-6, 
     -1.2081187997139237e-6, -1.164800743193676e-6, -1.123529154339022e-6, 
     -1.0841845930795203e-6, -1.0466558880141655e-6, 
     -1.0108394761958486e-6, -9.766388024824169e-7, -9.439637724756766e-7, 
     -9.127302537290873e-7, -8.82859620485157e-7, -8.542783377149485e-7, 
     -8.269175806834343e-7, -8.007128866633324e-7, -7.756038357730831e-7, 
     -7.515337582274791e-7, -7.284494655670862e-7, -7.06301003679256e-7, 
     -6.850414256429838e-7, -6.646265826253256e-7, -6.450149312314123e-7};
 
  TVIL_REAL H0xxxyytableCL[] = 
     {0, 0.8494371185386919, 0.01656483359535362, 
     -0.009321133400193908, -0.008147074810154656, -0.005535666905668476, 
     -0.003640878101067303, -0.0024308385961208686, -0.0016710931449379863, 
     -0.001186370710773187, -0.000868477383577359, -0.0006535026901336496, 
     -0.0005036958249296189, -0.00039636624453710014, -0.0003175334990582716, 
     -0.00025834472278465405, -0.0002130378417379792, 
     -0.00017776273832861218, -0.00014988335777035828, 
     -0.00012755425478857462, -0.0001094571837888276, -0.0000946332312782693, 
     -0.00008237335447805146, -0.00007214546213173323, -0.000063544869933635, 
     -0.000056260022074538886, -0.00005004837799912006, 
     -0.00004471919004835515, -0.00004012103005352537, 
     -0.00003613263885411015, -0.00003265613372811984, 
     -0.000029611910761859682, -0.000026934780273454765, 
     -0.000024571009297833156, -0.00002247603826233301, 
     -0.000020612703625251897, -0.000018949843668500107, 
     -0.000017461196908943884, -0.000016124525768908804, 
     -0.000014920914954697225, -0.000013834206296253038, 
     -0.000012850540887518431, -0.000011957986132849814, 
     -0.000011146230382301935, -0.000010406331677303744, -9.730510050907e-6, 
     -9.111975066895496e-6, -8.544782009937759e-6, -8.023711479855653e-6, 
     -7.5441681896659265e-6, -7.102095588439867e-6, -6.693903578078029e-6, 
     -6.316407106974996e-6, -5.966773833004882e-6, -5.642479376024623e-6, 
     -5.341268943634962e-6, -5.061124326748561e-6, -4.8002354340637644e-6, 
     -4.556975675001003e-6, -4.32988061543019e-6, -4.117629424643297e-6, 
     -3.919028709493366e-6, -3.7329983955953747e-6, -3.558559368489698e-6, 
     -3.394822631725833e-6, -3.2409797755558213e-6, -3.096294580642426e-6, 
     -2.9600956069458455e-6, -2.8317696396147183e-6, 
     -2.7107558819731095e-6, -2.5965408011377544e-6, 
     -2.4886535448886236e-6, -2.38666185953688e-6, -2.2901684480059644e-6, 
     -2.198807715427185e-6, -2.1122428564691515e-6, -2.03016324455228e-6, 
     -1.9522820881967977e-6, -1.878334324141675e-6, 
     -1.8080747206587592e-6, -1.7412761677600643e-6, 
     -1.6777281338316383e-6, -1.6172352706879244e-6, 
     -1.5596161511794752e-6, -1.5047021253496215e-6, 
     -1.4523362827606538e-6, -1.4023725100300111e-6, 
     -1.3546746338597794e-6, -1.3091156409323015e-6, 
     -1.2655769670013269e-6, -1.2239478483493278e-6, 
     -1.184124729522443e-6, -1.1460107219078865e-6, 
     -1.1095151082957188e-6, -1.0745528890772572e-6, -1.04104436618441e-6, 
     -1.00891476127506e-6, -9.780938650255648e-7, -9.485157147079001e-7, 
     -9.201182975106936e-7, -8.928432773144796e-7};
 
  TVIL_REAL H0xxxyytableCL2[] = 
     {0, -0.1111111111111111, -0.00720164609053498, 
     -0.009846060051821369, -0.0065935252320711854, -0.003916446040106051, 
     -0.002343540734632126, -0.001461092510750758, -0.0009556662860266839, 
     -0.0006542556237433316, -0.00046627211606904614, 
     -0.00034385313731446923, -0.00026094786214498635, 
     -0.0002028411352525431, -0.00016089091874052914, -0.0001298255684179163, 
     -0.00010631328205900991, -0.00008817917952528499, 
     -0.00007396192223076974, -0.00006265407820065345, 
     -0.000053545176308785486, -0.00004612398895293279, 
     -0.00004001602079988921, -0.00003494252353042558, 
     -0.00003069302009487038, -0.000027106512424248996, 
     -0.000024058393385890512, -0.000021451180783733193, 
     -0.000019207858743925495, -0.000017267027235173578, 
     -0.000015579324383260468, -0.000014104757122991015, 
     -0.000012810688332615057, -0.000011670303997627356, 
     -0.000010661435197387105, -9.76564501856956e-6, -8.967515140958821e-6, 
     -8.254084241668099e-6, -7.614402787488804e-6, -7.039177747167607e-6, 
     -6.5204872832522864e-6, -6.051550281317942e-6, 
     -5.6265391315122025e-6, -5.240426835832715e-6, -4.888861516646465e-6, 
     -4.56806292069665e-6, -4.274736672859208e-6, -4.006002925741994e-6, 
     -3.7593367411422324e-6, -3.5325180762748948e-6, 
     -3.323589667865597e-6, -3.130821437819631e-6, -2.9526803056711895e-6, 
     -2.787804500861302e-6, -2.6349816338802146e-6, -2.493129918473044e-6, 
     -2.3612820444043712e-6, -2.238571287095463e-6, -2.124219510979217e-6, 
     -2.0175267809405995e-6, -1.9178623432991553e-6, 
     -1.8246567764765782e-6, -1.737395143386247e-6, 
     -1.6556110039637654e-6, -1.5788811681512376e-6, 
     -1.5068210878736642e-6, -1.43908080176398e-6, -1.3753413591373097e-6, 
     -1.315311660417491e-6, -1.258725660231488e-6, -1.2053398869970429e-6, 
     -1.1549312392706593e-6, -1.1072950245897248e-6, 
     -1.062243211193048e-6, -1.0196028669695323e-6, -9.79214763373753e-7, 
     -9.409321249500626e-7, -9.046195075986456e-7, -8.701517908603799e-7, 
     -8.374132713448861e-7, -8.062968460218061e-7, -7.767032754761213e-7, 
     -7.485405184254068e-7, -7.217231298366611e-7, -6.961717158850335e-7, 
     -6.718124397853334e-7, -6.485765732156943e-7, -6.264000886549043e-7, 
     -6.052232884823628e-7, -5.8499046715239e-7, -5.656496031612593e-7, 
     -5.471520778831759e-7, -5.29452418666818e-7, -5.125080638624248e-7, 
     -4.962791476954435e-7, -4.807283031204979e-7, -4.6582048098242674e-7, 
     -4.5152278398238626e-7, -4.3780431409918634e-7, 
     -4.2463603225140274e-7, -4.119906291064035e-7};
 
  TVIL_REAL H0xxxyytableCL3[] = 
     {0, 0., -0.012345679012345678, -0.0054869684499314125, 
     -0.002362444749276025, -0.0011007807075479686, -0.0005632484759989036, 
     -0.0003152853384581836, -0.0001912050397520406, -0.00012412181260070974, 
     -0.00008519818429691317, -0.0000611627212055496, 
     -0.000045509944982441155, -0.00003485273337902297, 
     -0.00002732480776636581, -0.000021843023114755844, 
     -0.00001774879173591381, -0.000014624956559889038, 
     -0.000012197751269484458, -0.000010281941068894044, 
     -8.748852099382215e-6, -7.507042313412139e-6, -6.49022561468161e-6, 
     -5.649505173756711e-6, -4.948248525040301e-6, -4.358628393236135e-6, 
     -3.859240454549544e-6, -3.43343307751709e-6, -3.068117247917629e-6, 
     -2.7529061976903004e-6, -2.479485104366698e-6, 
     -2.2411437099523938e-6, -2.032425864896146e-6, 
     -1.8488640270973834e-6, -1.6867761924026845e-6, 
     -1.543109189994014e-6, -1.4153167495003505e-6, 
     -1.3012638847494459e-6, -1.1991513660244679e-6, 
     -1.1074556501600495e-6, -1.0248807953281696e-6, 
     -9.503197340456204e-7, -8.828229027707732e-7, -8.215726914465767e-7, 
     -7.658625251242449e-7, -7.150796533735388e-7, -6.686909237820132e-7, 
     -6.262309695433883e-7, -5.872923596541597e-7, -5.515173521873714e-7, 
     -5.185909628644106e-7, -4.882351174473591e-7, -4.602037008871552e-7, 
     -4.3427835137492907e-7, -4.10264875497054e-7, -3.879901831482789e-7, 
     -3.6729965890971015e-7, -3.4805490117442286e-7, 
     -3.301317721210997e-7, -3.1341871125531176e-7, -2.978152730977267e-7, 
     -2.832308560441189e-7, -2.695835947267206e-7, -2.567993925870667e-7, 
     -2.4481107499995486e-7, -2.335576463048513e-7, -2.229836366161937e-7, 
     -2.1303852638722273e-7, -2.0367623846575302e-7, 
     -1.948546888635661e-7, -1.8653538871186015e-7, -1.786830909326315e-7, 
     -1.7126547605214094e-7, -1.642528723441678e-7, -1.576180061393744e-7, 
     -1.5133577869082156e-7, -1.4538306645944557e-7, 
     -1.3973854208955622e-7, -1.343825136935122e-7, 
     -1.2929678036533488e-7, -1.244645021023909e-7, 
     -1.1987008253850539e-7, -1.1549906308609067e-7, 
     -1.1133802725341733e-7, -1.0737451404968022e-7, 
     -1.0359693951811722e-7, -9.9994525548763e-8, -9.65572352196998e-8, 
     -9.327571400081618e-8, -9.014123622873419e-8, -8.714565632711082e-8, 
     -8.428136430416176e-8, -8.154124511001978e-8, -7.891864148131458e-8, 
     -7.640731993990751e-8, -7.400143964769176e-8, -7.169552385034802e-8, 
     -6.948443367041874e-8, -6.736334403446058e-8, -6.532772154072684e-8, 
     -6.337330409314234e-8};
 
  TVIL_REAL H0xxxyytableB[] = 
     {0, -4.455844660071442, 1.6718092439969705, -0.902558101298034, 
     0.5717440217360757, -0.3969580141251979, 0.29263354944767744, 
     -0.22508468200954324, 0.17871349140285825, -0.1454426095355402, 
     0.12073023997056685, -0.10185555355338691, 0.08710436913142562, 
     -0.07535141319808576, 0.06583230683201545, -0.05801258557679164, 
     0.05150916237710103, -0.04604148348952839, 0.041400178156711445, 
     -0.0374263755237587, 0.03399772479146502, -0.031018740937133672, 
     0.02841400836203582, -0.02612331311122382, 0.02409810153299913, 
     -0.022298867105934525, 0.020693197016608527, -0.019254294470124316, 
     0.017959848588180644, -0.016791161366097478, 0.01573246688141833, 
     -0.014770395787614058, 0.013893550666381364, -0.013092166735232143, 
     0.012357838829847907, -0.011683300253428525, 0.011062242519049815, 
     -0.01048916755818888, 0.009959265874778843, -0.00946831556260893, 
     0.009012598197918708, -0.00858882845730619, 0.008194094957893585, 
     -0.007825810319126148, 0.007481668838324463, -0.007159610480963123, 
     0.0068577901309206084, -0.006574551240195098, 0.006308403172856868, 
     -0.00605800166274718, 0.0058221319051179245, -0.005599693884044609, 
     0.00538968960392893, -0.005191211947775728, 0.005003434929568707, 
     -0.00482560514486201, 0.004657034254142387, -0.004497092358786565, 
     0.004345202150489993, -0.004200833732638517, 0.00406350002684799, 
     -0.003932752690305445, 0.0038081784800128725, -0.0036893960088898677, 
     0.003576052846203512, -0.003467822921183623, 0.003364404194130734, 
     -0.003265516563982955, 0.003170899985300566, -0.003080312771057116, 
     0.0029935300605789784, -0.0029103424345236803, 0.0028305546609911007, 
     -0.0027539845587713687, 0.002680461965391608, -0.0026098277990665286, 
     0.0025419332049155946, -0.0024766387769079804, 0.002413813847957608, 
     -0.002353335841432942, 0.002295089678085816, -0.0022389672330540363, 
     0.0021848668381654656, -0.0021326928252768745, 0.002082355106827558, 
     -0.0020337687901830995, 0.0019868538226950837, -0.0019415346647135884, 
     0.0018977399880657608, -0.0018554023977598322, 0.0018144581748932552, 
     -0.0017748470389393587, 0.0017365119277618064, -0.0016993987938626251, 
     0.001663456415509729, -0.0016286362215155886, 0.0015948921285515614, 
     -0.0015621803899838822, 0.0015304594553086311, -0.0014996898393452695, 
     0.0014698340004225497};
 
  TVIL_REAL H0xxxyytableD[] = 
     {0, -2.850879122556482, 0.8582466928359385, 
     -0.40512963257183937, 0.23407677856103462, -0.15186804460990305, 
     0.10624171018584176, -0.07836435155100864, 0.0601167374277702, 
     -0.047536561222154716, 0.038505190595679195, -0.031807290184038245, 
     0.026705630091365132, -0.022732172464511608, 0.019578320933918838, 
     -0.017033997443968753, 0.014952223036379544, -0.01322769893151956, 
     0.01178340649905438, -0.010561972081027505, 0.009519955520716753, 
     -0.008623984215957883, 0.007848081820511011, -0.007171787888175066, 
     0.006578811908894972, -0.006056055092467299, 0.0055928894845578775, 
     -0.0051806199210167655, 0.004812077723069517, -0.00448131054761404, 
     0.0041833432600262375, -0.003913991848031823, 0.0036697173560564875, 
     -0.0034475103057006427, 0.0032447985476252226, -0.003059373273763684, 
     0.0028893292153256904, -0.002733016003886243, 0.0025889983781205985, 
     -0.002456023445899434, 0.002332993608737891, -0.002218944057330375, 
     0.0021130239777688246, -0.0020144807859177813, 0.001922646845377439, 
     -0.0018369282321388195, 0.0017567951935757423, -0.0016817740161754796, 
     0.001611440069411995, -0.0015454118354656622, 0.0014833457684201932, 
     -0.001424931853908068, 0.0013698897623090729, -0.0013179655066026474, 
     0.001268928530668254, -0.0012225691658732966, 0.001178696403700665, 
     -0.0011371359403557553, 0.0010977284560801256, -0.001060328097544726, 
     0.0010248011364074924, -0.0009910247810651048, 0.0009588861219418998, 
     -0.0009282811934498083, 0.0008991141381108662, -0.00087129646033114, 
     0.0008447463590114713, -0.000819388129625305, 0.0007951516276275093, 
     -0.0007719717861139583, 0.0007497881815575097, -0.0007285446422249722, 
     0.0007081888945509909, -0.0006886722433246147, 0.0006699492820461784, 
     -0.0006519776302473957, 0.0006347176949458262, -0.0006181324537342315, 
     0.0006021872572926592, -0.000586849649362187, 0.0005720892024391086, 
     -0.0005578773676411493, 0.0005441873373666892, -0.0005309939195170275, 
     0.0005182734221830982, -0.0005060035478140233, 0.0004941632959874245, 
     -0.0004827328739922067, 0.00047169361451502976, -0.00046102789979317647, 
     0.00045071909166009453, -0.00044075146696649344, 0.0004311101579103496, 
     -0.0004217810968542296, 0.0004127509652486208, -0.0004040071463160048, 
     0.0003955376811827162, -0.00038733122817460815, 0.0003793770250185783, 
     -0.0003716648537154155, 0.0003641850078705022};
 
  TVIL_REAL H0xxxyytableE[] = 
     {0, -5.735668132712307, 2.462603840720206, -1.4576628734661734, 
     0.9900167790391754, -0.7268644378825239, 0.5613492246053227, 
     -0.44927498582780645, 0.3692780905391686, -0.30986601981046236, 
     0.2643521617007389, -0.2286049030530302, 0.19994659492597638, 
     -0.1765731811869663, 0.15722934941736202, -0.14101750340407493, 
     0.12728068532891323, -0.11552824994510263, 0.10538722605265663, 
     -0.09656963505121922, 0.08885001493416464, -0.08204964091200218, 
     0.07602524160253525, -0.0706607953840826, 0.06586147626126418, 
     -0.06154912493124557, 0.057658818557666594, -0.054136243033282766, 
     0.0509356588531598, -0.048018311244777484, 0.04535117638458623, 
     -0.0429059644229684, 0.04065832057190833, -0.038587180276427754, 
     0.036674245229865385, -0.03490355488451727, 0.0332611339647207, 
     -0.03173470087416704, 0.03031342520077624, -0.028987725043761344, 
     0.027749096821793134, -0.026589971715813635, 0.025503594062850674, 
     -0.0244839179276758, 0.023525518796448056, -0.022623517904882953, 
     0.021773517166389453, -0.020971543028402297, 0.02021399787719999, 
     -0.019497617847758903, 0.018819436087192238, -0.018176750677039246, 
     0.01756709654812709, -0.01698822082744762, 0.016438061143835773, 
     -0.01591472649166825, 0.015416480312081903, -0.014941725501552159, 
     0.014488991099851179, -0.014056920444859768, 0.01364426061159991, 
     -0.013249852978133988, 0.012872624782415229, -0.012511581552405924, 
     0.012165800307326481, -0.011834423441189247, 0.011516653211162093, 
     -0.011211746763093338, 0.010919011634957145, -0.010637801686252696, 
     0.010367513407682731, -0.010107582570891565, 0.009857481182781113, 
     -0.0096167147130478, 0.009384819567180298, -0.009161360780301164, 
     0.008945929909987011, -0.008738143108614973, 0.008537639357903236, 
     -0.008344078850179116, 0.008157141502552803, -0.007976525591626924, 
     0.007801946497656247, -0.007633135548209067, 0.00746983895239073, 
     -0.007311816817585753, 0.007158842241472259, -0.007010700472772503, 
     0.006867188134836768, -0.006728112506723647, 0.006593290856945818, 
     -0.006462549825503504, 0.006335724850234155, -0.006212659633871589, 
     0.006093205648535616, -0.005977221674668163, 0.005864573371697553, 
     -0.0057551328779522565, 0.005648778437561703, -0.005545394052277255, 
     0.005444869156323308};
 
  TVIL_REAL H0xxxyytableF[] = 
     {0, -5.291406274519789, 2.1786291405571925, -1.2524512674453052, 
     0.8315262381090299, -0.5991691374295054, 0.45538230515704026, 
     -0.3593914205413129, 0.2917319134585367, -0.24204946923414364, 
     0.2043806398084927, -0.17507484858605946, 0.15178651517786268, 
     -0.1329479506450003, 0.11747648209870021, -0.104603411265293, 
     0.09376986590452188, -0.08456108835704312, 0.0766636854429379, 
     -0.0698370635651165, 0.06389388833815224, -0.058686436128588246, 
     0.05409688182308903, -0.050030270991121825, 0.04640935698412596, 
     -0.04317075559736759, 0.040262044903429876, -0.03763955264633221, 
     0.03526665024433207, -0.03311242450053999, 0.031150634003477687, 
     -0.0293588822857405, 0.027717957575681483, -0.02621130171250949, 
     0.024824580027129978, -0.02354533075371547, 0.022362677539604577, 
     -0.021267092355920534, 0.020250198923757096, -0.019304608905966608, 
     0.018423784748075666, -0.017601924310616874, 0.016833863411840073, 
     -0.01611499316251017, 0.01544118957385812, -0.014808753393494446, 
     0.01421435850064455, -0.013655007492957857, 0.013127993338803262, 
     -0.01263086616399418, 0.012161404400016616, -0.011717589649614588, 
     0.011297584730913265, -0.010899714447755247, 0.010522448705235686, 
     -0.010164387648436229, 0.009824248551376313, -0.009500854224063625, 
     0.009193122739714805, -0.008900058312933199, 0.008620743183855287, 
     -0.008354330383830702, 0.00810003727578608, -0.00785713977768025, 
     0.007624967191025176, -0.007402897569046981, 0.0071903535716412335, 
     -0.006986798768236394, 0.006791734367266733, -0.006604696375956299, 
     0.006425253233063044, -0.006253004021548531, 0.006087577477910171, 
     -0.005928632205571293, 0.005775858833866391, -0.005628985452466559, 
     0.005487788689365629, -0.005352114635291213, 0.005221917061956139, 
     -0.005097326121828184, 0.0049787708760106976, -0.004867196980198318, 
     0.004764452699720558, -0.004673972812778711, 0.004601989838136869, 
     -0.004559678961216873, 0.004566956539791908, -0.00465920764963842}; 

  lnyox = TVIL_CLOG(y/x);
  lnyox2 = lnyox  * lnyox;
  lnyox3 = lnyox2  * lnyox;

  if (y < 0.155 * x) {
    result = TVIL_H00xx0x (x,qq);
    rn = r = 4.L*y/x;
    for (n=1; n < 100; n++) {
      result += rn * (H0xxxyytableA[n] + 
                      H0xxxyytableAL[n] * lnyox +
                      H0xxxyytableAL2[n] * lnyox2);
      rn *= r;
    }
  }
  else if (y < 0.5 * x) {
    result = -15.268461152599217213771832646388L - 6.L * Zeta3 * TVIL_Ap(y,qq);
    rn = r = x/(4.L*y) - 1.L;
    for (n=1; n < 100; n++) {
      result += rn * H0xxxyytableE[n];
      rn *= r;
    }
  }
  else if (y < 1.6 * x) {
    result = TVIL_H0xxxxx (x,qq);
    rn = r = y/x - 1.L;
    for (n=1; n < 100; n++) {
      result += rn * H0xxxyytableD[n];
      rn *= r;
    }
  }
  else if (y < 5.59 * x) {
    result = -13.285145261974023210263387L - 6.L * Zeta3 * TVIL_Ap(x,qq);
    rn = r = y/(4.L*x) - 1.L;
    for (n=1; n < 100; n++) {
      result += rn * H0xxxyytableB[n];
      rn *= r;
    }
  }
  else if (y < 14.56 * x) {
    result = -17.2471361638484718668742249L - 6.L * Zeta3 * TVIL_Ap(x,qq);
    rn = r = y/(9.L*x) - 1.L;
    for (n=1; n < 86; n++) {
      result += rn * H0xxxyytableF[n];
      rn *= r;
    }
  }
  else {
    result = TVIL_H0000xx (y, qq);
    rn = r = 9.L * x/y;
    for (n=1; n < 100; n++) {
      result += rn * (H0xxxyytableC[n] 
                     -H0xxxyytableCL[n] * lnyox 
                     +H0xxxyytableCL2[n] * lnyox2
                     -H0xxxyytableCL3[n] * lnyox3);
      rn *= r;
    }
  }

  return result;
}

/* ****************************************************************** */
TVIL_COMPLEX TVIL_H0xyyxx (TVIL_REAL x, TVIL_REAL y, TVIL_REAL qq)
{
  TVIL_REAL r, rn;
  TVIL_COMPLEX lnyox, lnyox2, lnyox3, result;
  int n;

  TVIL_REAL H0xyyxxtableA[] = 
     {0, -1.0009187129297172, -0.03883721069547208, 
     -0.00040178575569213234, 0.0012078795448267844, 0.0008396720511033885, 
     0.0005318766080981466, 0.0003451754798245971, 0.00023249089141519862, 
     0.00016217174076431338, 0.00011664386533837383, 0.00008615005138238978, 
     0.0000651026114782491, 0.00005018637963845737, 0.00003936686294136913, 
     0.000031356344019049687, 0.000025316740955636507, 
     0.00002068876306580866, 0.000017090653748394856, 
     0.000014256478563648095, 0.00001199754250298264, 
     0.000010177699034382358, 8.697208958220088e-6, 7.481980677915513e-6, 
     6.476268223724826e-6, 5.637633609483561e-6, 4.933418215388181e-6, 
     4.33823625938016e-6, 3.832171014489617e-6, 3.3994609854895355e-6, 
     3.0275321467019436e-6, 2.7062775779920764e-6, 2.42751597442358e-6, 
     2.1845808601180055e-6, 1.97200626076527e-6, 1.785284228564264e-6, 
     1.6206763620769695e-6, 1.4750662387403904e-6, 1.3458430906229041e-6, 
     1.230809516293474e-6, 1.1281078140957376e-6, 1.03616083801255e-6, 
     9.53624251150622e-7, 8.79347778100832e-7, 8.123436029434416e-7, 
     7.517604722778757e-7, 6.968623768054086e-7, 6.470109256840856e-7, 
     6.016507133957063e-7, 5.602971226839952e-7, 5.225261192290557e-7, 
     4.879656815700931e-7, 4.5628857897401763e-7, 4.272062647006573e-7, 
     4.0046369564866433e-7, 3.758349241339731e-7, 3.531193354395407e-7, 
     3.3213842723564787e-7, 3.12733045132427e-7, 2.947610033695884e-7, 
     2.780950316606467e-7, 2.6262099903125385e-7, 2.482363735506145e-7, 
     2.3484888348992323e-7, 2.223753509217432e-7, 2.1074067331444846e-7, 
     1.9987693244884565e-7, 1.8972261312869925e-7, 1.8022191678522413e-7, 
     1.713241572784791e-7, 1.629832280498072e-7, 1.5515713133918782e-7, 
     1.4780756149879827e-7, 1.4089953554958418e-7, 1.3440106507439828e-7, 
     1.282828643466698e-7, 1.2251809028024977e-7, 1.1708211037281921e-7, 
     1.119522953176596e-7, 1.0710783338967226e-7, 1.0252956408217163e-7, 
     9.819982879026123e-8, 9.410233661217701e-8, 9.022204357827238e-8, 
     8.654504382374402e-8, 8.305847140033467e-8, 7.975041157797327e-8, 
     7.660982062291369e-8, 7.362645315720434e-8, 7.079079630764383e-8, 
     6.809400994278223e-8, 6.552787237575147e-8, 6.30847309802368e-8, 
     6.075745722801249e-8, 5.8539405710251915e-8, 5.642437675223235e-8, 
     5.440658227289859e-8, 5.248061457773192e-8, 5.064141780609871e-8, 
     4.8884261783253256e-8, 4.720471805289874e-8};
 
  TVIL_REAL H0xyyxxtableAL[] = 
     {0, 0.8224670334241132, 0.07155837917801415, 
     0.013662507640780136, 0.00482843025209072, 0.0023144561574192776, 
     0.0012752331866176861, 0.0007630211175141861, 0.00048498494346749645, 
     0.00032352363121167276, 0.0002245758707538703, 0.00016115159648129679, 
     0.0001189189769451791, 0.00008986705724369164, 0.00006931474338234614, 
     0.00005441831979243009, 0.000043390247166980835, 
     0.000035072549691608386, 0.000028694831635621814, 
     0.000023732268301298332, 0.000019819682533771845, 
     0.000016698108475877662, 0.00001418072612525312, 
     0.000012130652826276535, 0.000010446167346853053, 9.050695256940749e-6, 
     7.885905415681389e-6, 6.906876548713815e-6, 6.078664418630283e-6, 
     5.373831314538866e-6, 4.770646210744478e-6, 4.251758546372079e-6, 
     3.8032106031903443e-6, 3.4136947377475687e-6, 3.07398958082435e-6, 
     2.7765283624854174e-6, 2.515065701496448e-6, 2.284418424383892e-6, 
     2.0802625078338377e-6, 1.89897290440148e-6, 1.73749637875527e-6, 
     1.593249933537555e-6, 1.4640392045299133e-6, 1.347992537879895e-6, 
     1.2435074566742865e-6, 1.1492069715340123e-6, 1.0639037554792356e-6, 
     9.86570634136992e-7, 9.163161726163325e-7, 8.523643950619857e-7, 
     7.940378704400171e-7, 7.4074355217405e-7, 6.919608800417589e-7, 
     6.472317479240812e-7, 6.061520163671243e-7, 5.683643088723162e-7, 
     5.335518787391254e-7, 5.01433371732042e-7, 4.717583408258101e-7, 
     4.443033943492074e-7, 4.1886887920559025e-7, 3.9527601744369013e-7, 
     3.7336442802858855e-7, 3.5298997680794153e-7, 3.340229068483075e-7, 
     3.1634620890183207e-7, 2.9985419805103713e-7, 2.8445126780687694e-7, 
     2.7005079729399555e-7, 2.5657419080189565e-7, 2.4395003203683293e-7, 
     2.3211333797858438e-7, 2.2100489941191995e-7, 2.105706970326236e-7, 
     2.0076138357805898e-7, 1.9153182374845966e-7, 1.828406848052299e-7, 
     1.7465007168790758e-7, 1.6692520130807095e-7, 1.596341113779863e-7, 
     1.5274739973222584e-7, 1.462379906169015e-7, 1.400809248661852e-7, 
     1.3425317127001707e-7, 1.2873345676926493e-7, 1.2350211340260567e-7, 
     1.1854094017942077e-7, 1.1383307827040913e-7, 1.0936289809699824e-7, 
     1.0511589706586004e-7, 1.0107860683922501e-7, 9.723850915806075e-8, 
     9.35839593459563e-8, 9.010411671880245e-8, 8.678888121085517e-8, 
     8.362883560304377e-8, 8.061519280575068e-8, 7.773974770688161e-8, 
     7.499483314784114e-8, 7.237327963587997e-8, 6.986837844191797e-8};
 
  TVIL_REAL H0xyyxxtableAL2[] = 
     {0, 0., 0.03125, 0.013020833333333334, 0.00498046875, 
     0.0020484561011904763, 0.000943816654265873, 0.00048808308390827925, 
     0.0002788143875675919, 0.00017235330336145155, 0.00011322742190240252, 
     0.00007797606173607171, 0.000055740560303788756, 
     0.000041068801555647845, 0.0000310264758451475, 0.00002394056801978792, 
     0.00001881055338550314, 0.000015013748902098706, 
     0.000012149322513379675, 9.951662969979321e-6, 8.240267504881846e-6, 
     6.8897190182652785e-6, 5.8111416679529195e-6, 4.940434650717552e-6, 
     4.230626421412012e-6, 3.64679908920973e-6, 3.1626534314625892e-6, 
     2.7581432016405134e-6, 2.417819667298569e-6, 2.1296560976685165e-6, 
     1.8842017575967417e-6, 1.6739654410321386e-6, 1.4929610718417016e-6, 
     1.3363691691607765e-6, 1.2002821123066834e-6, 1.081510672862761e-6, 
     9.774357949845979e-7, 8.858941110617819e-7, 8.050888336308248e-7, 
     7.335198958850678e-7, 6.699288082326609e-7, 6.13254849621687e-7, 
     5.626000508682879e-7, 5.172010432205325e-7, 4.7640630161409687e-7, 
     4.3965765258216766e-7, 4.064751727892528e-7, 3.764447979801328e-7, 
     3.4920810985176104e-7, 3.244538816478141e-7, 3.0191105069046037e-7, 
     2.8134285390167904e-7, 2.625419153021059e-7, 2.453261160007057e-7, 
     2.295351099279374e-7, 2.15027374502687e-7, 2.0167770606791016e-7, 
     1.893750864352583e-7, 1.7802086013167903e-7, 1.6752717262608345e-7, 
     1.5781562846344403e-7, 1.488161352618881e-7, 1.4046590526044668e-7, 
     1.3270859079682337e-7, 1.2549353394793257e-7, 1.1877511374135782e-7, 
     1.1251217697100489e-7, 1.0666754082719199e-7, 1.0120755736217502e-7, 
     9.610173132264918e-8, 9.132238414443311e-8, 8.684435796457217e-8, 
     8.264475439767006e-8, 7.870270357507598e-8, 7.499915958110914e-8, 
     7.151671895907833e-8, 6.82394594173253e-8, 6.515279625499174e-8, 
     6.224335435952673e-8, 5.949885391211788e-8, 5.690800818073191e-8, 
     5.446043198953923e-8, 5.214655963339235e-8, 4.9957571161108177e-8, 
     4.7885326085246104e-8, 4.592230369197222e-8, 4.406154922506624e-8, 
     4.229662530537048e-8, 4.0621568022869115e-8, 3.9030847204709885e-8, 
     3.751933042018937e-8, 3.608225033416873e-8, 3.4715175064552146e-8, 
     3.3413981238185646e-8, 3.2174829473540655e-8, 3.099414204845245e-8, 
     2.9868582537520716e-8, 2.8795037227006484e-8, 2.77705981355705e-8, 
     2.6792547487336156e-8, 2.5858343499819365e-8};
 
  TVIL_REAL H0xyyxxtableC[] = 
     {0, -0.9015426773696957, 0.43683688231907947, 
     0.2724394055268168, 0.19828629766815362, 0.14948846763856713, 
     0.11728230693126342, 0.09486716724736441, 0.07864059783749072, 
     0.06649140520364774, 0.05713725332693239, 0.04976438170155528, 
     0.04383692462803492, 0.0389903012042017, 0.0349693131350685, 
     0.03159083035445134, 0.028720538025695394, 0.026258006867486354, 
     0.024126848154031395, 0.022268064505231628, 0.02063546151664616, 
     0.01919241954359513, 0.0179095822607202, 0.01676317506841031, 
     0.015733763821169068, 0.01480532632399057, 0.01396454926282288, 
     0.013200289820690434, 0.012503159103204558, 0.011865196700704824, 
     0.01127961416845208, 0.010740591142237918, 0.010243112027020787, 
     0.009782834231848104, 0.00935598113183186, 0.008959254559721868, 
     0.008589762832565675, 0.008244961219308455, 0.007922602434830093, 
     0.007620695263079596, 0.007337469808486521, 0.007071348181037488, 
     0.00682091965848568, 0.006584919555468761, 0.006362211176001101, 
     0.006151770341979853, 0.005952672082862711, 0.005764079145746558, 
     0.005585232044679606, 0.00541544041623366, 0.005254075487512537, 
     0.0051005634947160735, 0.004954379916553934, 0.004815044408338828, 
     0.00468211634037656, 0.0045551908590180535, 0.004433895401009918, 
     0.0043178866020266785, 0.004206847548852253, 0.004100485331893515, 
     0.0039985288607920985, 0.003900726911044988, 0.0038068463739074658, 
     0.0037166706855626696, 0.003629998414706502, 0.0035466419904023423, 
     0.0034664265543793643, 0.0033891889239412963, 0.0033147766533691636, 
     0.0032430471831836564, 0.003173867067915163, 0.0031071112741415763, 
     0.003042662541520287, 0.00298041080038223, 0.002920252640189971, 
     0.0028620908238035816, 0.002805833843060087, 0.002751395511665409, 
     0.0026986945918311906, 0.002647654451470504, 0.0025982027491030526, 
     0.002550271143917813, 0.0025037950287041554, 0.0024587132835956348, 
     0.002414968048777564, 0.002372504514493417, 0.0023312707268488175, 
     0.002291217408057808, 0.002252297789906356, 0.0022144674593244794, 
     0.002177684215062572, 0.0021419079345608924, 0.002107100450184952, 
     0.0020732254340748053, 0.0020402482909239366, 0.0020081360580643938, 
     0.00197685731228977, 0.0019463820828972227, 0.0019166817704745303, 
     0.001887729070998715, 0.001854276074057832};
 
  TVIL_REAL H0xyyxxtableCL[] = 
     {0, 2.4674011002723395, 0.5428001375340424, 0.3204452076559101, 
     0.19760340239672722, 0.1349848087032723, 0.09828560389076056, 
     0.0751331055018614, 0.059574867523662825, 0.048598456501346414, 
     0.04054675944996655, 0.03445105108116884, 0.02971459473205004, 
     0.025953445179418673, 0.022911300438679418, 0.02041160875109357, 
     0.01832942053761164, 0.016574204251980244, 0.015079001701789016, 
     0.013793381766919376, 0.012678742619457463, 0.011705107064740948, 
     0.010848891112908937, 0.010091321255774877, 0.009417292950817304, 
     0.008814534726150798, 0.008272987543657842, 0.0077843381023541425, 
     0.007341663780922058, 0.006939159591372953, 0.0065719260997239804, 
     0.006235803171806031, 0.005927238517521037, 0.005643182913439226, 
     0.005381006061413362, 0.005138428543001961, 0.004913466426916316, 
     0.0047043858964146415, 0.004509665866553999, 0.004327967014187941, 
     0.004158105986694777, 0.0039990338173190795, 0.003849817776399311, 
     0.0037096260436986325, 0.003577714708609941, 0.0034534167003504052, 
     0.0033361323254944716, 0.0032253211498977936, 0.003120495009701082, 
     0.0030212119743086435, 0.0029270711150301957, 0.00283770795801202, 
     0.002752790520369999, 0.0026720158450117743, 0.0025951069632337473, 
     0.0025218102253800057, 0.0024518929491124863, 0.002385141342528832, 
     0.0023213586657669445, 0.002260363600086147, 0.0022019887979015857, 
     0.002146079591022608, 0.002092492837530217, 0.0020410958904234756, 
     0.0019917656734516815, 0.001944387851495375, 0.001898856084519918, 
     0.0018550713555461043, 0.001812941364300723, 0.0017723799792575116, 
     0.0017333067416816438, 0.0016956464160705468, 0.0016593285820587344, 
     0.0016242872634397699, 0.001590460590467391, 0.0015577904920410952, 
     0.0015262224147683553, 0.0014957050662339118, 0.001466190180102937, 
     0.0014376323009449435, 0.0014099885868939662, 0.0013832186284619035, 
     0.0013572842819995542, 0.0013321495164568204, 0.0013077802722324576, 
     0.0012841443310268577, 0.0012612111957206323, 0.0012389519793989125, 
     0.0012173393027277667, 0.0011963471989662406, 0.0011759510259663495, 
     0.0011561273845748829, 0.0011368540429059475, 0.0011181098660025335, 
     0.0010998747504496658, 0.0010821295635415013, 0.0010648560866405084, 
     0.0010480369623991014, 0.0010316556455431467, 0.0010156963569429955, 
     0.0009924732836133579};
 
  TVIL_REAL H0xyyxxtableCL2[] = 
     {0, 0., 0.078125, -0.0026041666666666665, 
     -0.020100911458333332, -0.023177083333333334, -0.022016059027777778, 
     -0.019893246605282737, -0.017734375454130626, -0.01579298392805473, 
     -0.014112651158892918, -0.01267469199360879, -0.011445296858224575, 
     -0.0103906171560744, -0.009481076379958849, -0.008692062751130013, 
     -0.0080034759911574, -0.007398996103785602, -0.006865368566495867, 
     -0.006391792199621997, -0.005969421648532652, -0.005590972100502806, 
     -0.005250408121475297, -0.004942699296333084, -0.004663628074009148, 
     -0.004409638146190537, -0.00417771425846871, -0.003965286438340956, 
     -0.0037701532559379605, -0.00359041998693051, -0.0034244485023294797, 
     -0.003270816435733028, -0.0031282837302692686, -0.0029957650877693825, 
     -0.002872307164004577, -0.0027570696004879727, -0.002649309173604202, 
     -0.0025483664893102814, -0.002453654766550333, -0.002364650342498836, 
     -0.002280884603557389, -0.0022019371020382977, -0.0021274296629859666, 
     -0.002057021321140941, -0.0019904039565787003, -0.0019272985205482035, 
     -0.0018674517616487682, -0.001810633377616243, -0.0017566335303423274, 
     -0.001705260671874934, -0.0016563396374767927, -0.0016097099686977001, 
     -0.0015652244351161849, -0.0015227477281470005, -0.00148215530426663, 
     -0.0014433323583205667, -0.0014061729103571488, -0.001370578991775316, 
     -0.0013364599185527136, -0.0013037316409973301, -0.0012723161608903825, 
     -0.0012421410081016582, -0.0012131387697949406, -0.001185246666228527, 
     -0.0011584061679175146, -0.0011325626495797977, -0.0011076650768528005, 
     -0.0010836657222563007, -0.001060519907299684, -0.001038185767999068, 
     -0.0010166240413890285, -0.0009957978708918856, -0.0009756726286504616, 
     -0.0009562157531427289, -0.0009373966005830202, -0.0009191863087779852, 
     -0.0009015576722492878, -0.0008844850275617235, -0.0008679441479072306, 
     -0.0008519121460940722, -0.0008363673851779324, -0.00082128939604921, 
     -0.0008066588013596492, -0.0007924572452326612, -0.0007786673282562099, 
     -0.0007652725473057434, -0.0007522572397880558, -0.0007396065319357708, 
     -0.0007273062908168791, -0.0007153430797549029, -0.0007037041168832064, 
     -0.0006923772365820885, -0.0006813508535698789, -0.0006706139294396155, 
     -0.0006601559414512215, -0.0006499668534056648, -0.000640037088442551, 
     -0.0006303575036161362, -0.00062091936611701, -0.0006117143310178096, 
     -0.0006055518443818779};
 
  TVIL_REAL H0xyyxxtableCL3[] = 
     {0, 0., -0.0625, -0.041666666666666664, -0.029296875, 
     -0.021484375, -0.016520182291666668, -0.01318359375, 
     -0.010829925537109375, -0.009100172254774306, -0.007785987854003906, 
     -0.006760120391845703, -0.005941241979598999, -0.00527527928352356, 
     -0.004725046988044467, -0.004264226059118906, -0.003873721434501931, 
     -0.003539378638379276, -0.0032505066758151064, -0.0029988958049216308, 
     -0.002778148313609563, -0.002583212611409094, -0.0024100525570299233, 
     -0.00225540866986762, -0.002116622976286268, -0.001991508696029953, 
     -0.001878252026409139, -0.0017753372356220571, -0.0016814889075808814, 
     -0.0015956269613335086, -0.0015168312923150263, -0.0014443137363673916, 
     -0.0013773956607987439, -0.0013154899184380422, -0.001258086213079024, 
     -0.0012047391532766566, -0.0011550584403576159, -0.001108700762490307, 
     -0.001065363061466743, -0.001024776910788995, -0.0009867037986714315, 
     -0.0009509311519595347, -0.0009172689698549776, -0.0008855469620240995, 
     -0.0008556121058564096, -0.0008273265536032237, -0.0008005658328205732, 
     -0.0007752172936895832, -0.000751178764943859, -0.0007283573867207351, 
     -0.0007066685939984976, -0.0006860352286386938, -0.000666386761619478, 
     -0.000647658609978027, -0.0006297915353999416, -0.0006127311133983442, 
     -0.0005964272636923745, -0.0005808338337858063, -0.0005659082289111702, 
     -0.0005516110824831065, -0.0005379059620290908, -0.000524759106262489, 
     -0.000512139189553604, -0.0005000171105565347, -0.0004883658021777322, 
     -0.00047716006043800407, -0.00046637639009321723, 
     -0.00045599286514825437, -0.0004459890026306682, -0.0004363456481906122, 
     -0.00042704487226674816, -0.00041806987570791295, 
     -0.00040940490387070754, -0.0004010351683266746, -0.0003929467754117346, 
     -0.00038512666093708074, -0.0003775625304564958, -0.0003702428045515184, 
     -0.00036315656865429466, -0.00035629352697936746, 
     -0.0003496439601809964, -0.0003431986863926491, -0.00033694902534073295, 
     -0.00033088676525602984, -0.0003250041323341531, -0.0003192937625211049, 
     -0.00031374867542204656, -0.0003083622501510356, -0.0003031282029570093, 
     -0.00029804056647696165, -0.00029309367048127854, 
     -0.0002882821239887563, -0.0002836007986401006, -0.00027904481322882276, 
     -0.0002746095192975555, -0.00027029048771600865, -0.0002660834961641719, 
     -0.0002619845174510422, -0.00025798970860517614, 
     -0.00025409540067881703, -0.0002502980892122785};
 
  TVIL_REAL H0xyyxxtableB[] = 
     {0, -4.579021951072399, 1.7160519814933797, -0.9211108557663468, 
     0.5796695671013951, -0.3998854062628255, 0.2930326508782866, 
     -0.22415753291814064, 0.17708796305332658, -0.14346311606979498, 
     0.11859199580176828, -0.09967132142372849, 0.08493961601400532, 
     -0.07324369962531406, 0.06380238619781924, -0.05607095294637915, 
     0.049660014184732044, -0.04428515894675208, 0.03973468732891155, 
     -0.03584836604260195, 0.03250309725257915, -0.029603041132494484, 
     0.02707267604091192, -0.024851837658026992, 0.022892116878747, 
     -0.02115420685234463, 0.019605923529633328, -0.018220711036837582, 
     0.016976500684683676, -0.015854831073540666, 0.014840163148569802, 
     -0.013919342340616115, 0.013081172760751571, -0.012316077534986439, 
     0.011615825920131547, -0.010973312603886743, 0.010382378086956053, 
     -0.009837661633881317, 0.009334480214172823, -0.008868728313508777, 
     0.008436794602389951, -0.008035492297231826, 0.007662000702073551, 
     -0.007313815925835459, 0.006988709165694775, -0.006684691257901516, 
     0.006399982442839196, -0.006132986486118988, 0.00588226845317931, 
     -0.005646535559787258, 0.005424620621559642, -0.005215467707199767, 
     0.005018119666509122, -0.004831707258445267, 0.004655439648961639, 
     -0.004488596084975649, 0.004330518581067699, -0.004180605480607963, 
     0.004038305773892542, -0.0039031140733096375, 0.003774566160164895, 
     -0.003652235030070301, 0.0035357273741463834, -0.0034246804420315197, 
     0.0033187592401033483, -0.0032176540246157174, 0.0031210780548212966, 
     -0.003028765575734325, 0.0029404700041136115, -0.0028559622946154194, 
     0.0027750294659647312, -0.002697473269492885, 0.002623108984549388, 
     -0.0025517643271659343, 0.002483278459973426, -0.00241750109278365, 
     0.0023542916644762224, -0.0022935185979039894, 0.002235058620467805, 
     -0.0021787961438329744, 0.002124622696980283, -0.0020724364074178945, 
     0.002022141525937948, -0.0019736479907932767, 0.0019268710276038334, 
     -0.0018817307816863462, 0.0018381519798408622, -0.0017960636189295138, 
     0.0017553986788508384, -0.0017160938577513742, 0.0016780893275285943, 
     -0.001641328507868632, 0.0016057578572313887, -0.0015713266793468547, 
     0.0015379869439218804, -0.0015056931203780088, 0.001474402023549901, 
     -0.00144407267037176, 0.0014146661466671745, -0.001386145483237085, 
     0.0013584755405120232};
 
  TVIL_REAL H0xyyxxtableD[] = 
     {0, -2.907641530934056, 0.8608407860425027, -0.3997227148423782, 
     0.22796102402340665, -0.14642058714794587, 0.10163517169647576, 
     -0.07450859148187698, 0.056880482293886106, -0.04480074004090404, 
     0.036172703012288664, -0.029801498379464126, 0.024966527455897763, 
     -0.021212682211764377, 0.01824127880081347, -0.015849827592837155, 
     0.013897183686144095, -0.012282564699902025, 0.010932476711847284, 
     -0.009792320269311231, 0.008820857325571608, -0.007986477790866087, 
     0.007264627249552297, -0.006636001143566543, 0.006085255363102064, 
     -0.005600071282989377, 0.005170468220868002, -0.004788291293307998, 
     0.004446825383772866, -0.004140500974449126, 0.0038646677047609016, 
     -0.0036154184213084946, 0.0033894512621111842, -0.003183960669355728, 
     0.0029965506041585233, -0.0028251649453318395, 0.0026680312940232562, 
     -0.002523615314898944, 0.002390583416927659, -0.002267772078691464, 
     0.00215416250086602, -0.00204885955502808, 0.001951074216899852, 
     -0.0018601088406413882, 0.0017753447613456045, -0.0016962318146697285, 
     0.001622279442369877, -0.001553049115485017, 0.0014881478568735808, 
     -0.0014272226846410586, 0.0013699558299200018, -0.0013160606081691654, 
     0.001265277843952103, -0.001217372766049936, 0.0011721323035473578, 
     -0.0011293627248233386, 0.0010888875706649482, -0.0010505458403890285, 
     0.0010141903962076907, -0.0009796865563536867, 0.0009469108518859132, 
     -0.000915749925780828, 0.0008860995560092424, -0.0008578637869026715, 
     0.0008309541553128153, -0.0008052889999301091, 0.0007807928437085361, 
     -0.0007573958406900222, 0.0007350332796705885, -0.0007136451381333738, 
     0.0006931756807166107, -0.0006735730972092658, 0.0006547891756913495, 
     -0.0006367790069749205, 0.0006195007169682115, -0.0006029152239896788, 
     0.0005869860184100906, -0.0005716789623065787, 0.0005569621070792728, 
     -0.0005428055272141571, 0.0005291811685797472, -0.0005160627098240192, 
     0.0005034254355951002, -0.0004912461204474091, 0.0004795029224167127, 
     -0.00046817528535503163, 0.0004572438492113243, -0.00044669036752798143, 
     0.0004364976314977167, -0.0004266493999916341, 0.0004171303350281066, 
     -0.0004079259422044908, 0.0003990225156604121, -0.0003904070871830462, 
     0.000382067379102086, -0.0003739917606554283, 0.0003661692075364904, 
     -0.0003585892643608693, 0.0003512420098141214, -0.00034411802426408404, 
     0.00033720835964063593};

  TVIL_REAL H0xyyxxtableE[] = 
    {0, -5.733355716211488, 2.4496709636469096, -1.441170670829355, 
    0.9724986709405867, -0.709373173992519, 0.5443478329040027, 
    -0.43296319865814387, 0.35372973483142167, -0.29509248877502525, 
    0.2503337006860176, -0.21530673538241835, 0.18732747282961337, 
    -0.16458997855260005, 0.1458395158110235, -0.13018033466105958, 
    0.11695792973865042, -0.10568433974012771, 0.09598929928467015, 
    -0.08758744330158172, 0.08025577250128028, -0.07381784794495429, 
    0.06813249996920447, -0.06308562790605064, 0.05858415506139243, 
    -0.05455151171403085, 0.05092421790442875, -0.047649268780671425, 
    0.04468211305320288, -0.041985074905327116, 0.03952611105401739, 
    -0.03727782364618975, 0.035216670264978614, -0.03332232711875029, 
    0.03157717224025992, -0.029965863420814427, 0.028474991460153514, 
    -0.027092793694462738, 0.02580891607173446, -0.0246142145594483, 
    0.023500588598070752, -0.022460840802886833, 0.021488558274172183, 
    -0.020578011781310575, 0.019724069799319163, -0.018922124940650098, 
    0.018168030774483087, -0.017458047385353743, 0.01678879431223927, 
    -0.01615720974302809, 0.015560515029139696, -0.014996183739880485, 
    0.014461914602913552, -0.013955607781478243, 0.013475344025056516, 
    -0.013019366301488183, 0.01258606357782766, -0.012173956466704987, 
    0.01178168449636718, -0.01140799479735538, 0.011051732028071162, 
    -0.01071182938623946, 0.010387300574250117, -0.01007723260418171, 
    0.00978077934349442, -0.00949715571534747, 0.009225632478601831, 
    -0.008965531522101245, 0.008716221616026315, -0.008477114570189122, 
    0.00824766175524855, -0.0080273509481208, 0.007815703467454681, 
    -0.0076122715690373575, 0.007416636074478489, -0.007228404209560794, 
    0.007047207631304116, -0.006872700625120085, 0.006704558455479657, 
    -0.006542475855314152, 0.006386165640954239, -0.006235357440808589, 
    0.0060897965272184084, -0.005949242742016622, 0.0058134695072886025, 
    -0.005682262913690774, 0.005555420879447165, -0.005432752373823851, 
    0.005314076699487114, -0.005199222828691945, 0.005088028788730711, 
    -0.004980341092504193, 0.004876014210464559, -0.004774910080527245, 
    0.004676897652860738, -0.004581852466743746, 0.004489656256931723, 
    -0.0044001965872022064, 0.004313366508953665, -0.004229064242917848, 
    0.00414719288221316};

  lnyox = TVIL_CLOG(y/x);
  lnyox2 = lnyox  * lnyox;
  lnyox3 = lnyox2  * lnyox;

  if (y < 0.155 * x) {
    result = TVIL_H00xx0x (x,qq);
    rn = r = 4.L*y/x;
    for (n=1; n < 100; n++) {
      result += rn * (H0xyyxxtableA[n] + 
                      H0xyyxxtableAL[n] * lnyox +
                      H0xyyxxtableAL2[n] * lnyox2);
      rn *= r;
    }
  }
  else if (y < 0.5 * x) {
    result = -15.232182693589549044416286380172L - 6.L * Zeta3 * TVIL_Ap(y,qq);
    rn = r = x/(4.L*y) - 1.L;
    for (n=1; n < 100; n++) {
      result += rn * H0xyyxxtableE[n];
      rn *= r;
    }
  }
  else if (y < 1.6 * x) {
    result = TVIL_H0xxxxx (x,qq);
    rn = r = y/x - 1.L;
    for (n=1; n < 100; n++) {
      result += rn * H0xyyxxtableD[n];
      rn *= r;
    }
  }
  else if (y < 6.47 * x) {
    result = -13.412622352667375198843495151454L - 6.L * Zeta3 * TVIL_Ap(x,qq);
    rn = r = y/(4.L*x) - 1.L;
    for (n=1; n < 100; n++) {
      result += rn * H0xyyxxtableB[n];
      rn *= r;
    }
  }
  else {
    result = TVIL_H00xx00 (y, qq);
    rn = r = 4.L * x/y;
    for (n=1; n < 100; n++) {
      result += rn * (H0xyyxxtableC[n] 
                     -H0xyyxxtableCL[n] * lnyox 
                     +H0xyyxxtableCL2[n] * lnyox2
                     -H0xyyxxtableCL3[n] * lnyox3);
      rn *= r;
    }
  }

  return result;
}

/* ****************************************************************** */
TVIL_COMPLEX TVIL_H0xyy0x (TVIL_REAL x, TVIL_REAL y, TVIL_REAL qq)
{
  TVIL_REAL r, rn;
  TVIL_COMPLEX lnyox, lnyox2, lnyox3, result;
  int n;

  if (y > x) return TVIL_H0xyy0x(y,x,qq);

  lnyox = TVIL_CLOG(y/x);
  lnyox2 = lnyox  * lnyox;
  lnyox3 = lnyox2  * lnyox;

  TVIL_REAL H0xyy0xtableA[] = 
     {0, -2.4041138063191885, -0.9498473505055757, 
     -1.412482379884174, -0.3824601531213455, -0.5164709094119858, 
     -0.16954635642205798, -0.25830461783395814, -0.0918342831201888, 
     -0.15296163098723847, -0.0565018714595538, -0.10049537650237907, 
     -0.03786020234848946, -0.07081316623756695, -0.026954324179434757, 
     -0.052470452733868675, -0.020075342353123485, -0.04037685650474681, 
     -0.015481381505103383, -0.03199874489139608, -0.012273041838154697, 
     -0.025963433735031925, -0.009950342313450385, -0.02147623340527611, 
     -0.008218422346855435, -0.018051881333454124, -0.006894775904858016, 
     -0.015380790900754322, -0.005861795785418221, -0.01325807954695051, 
     -0.005041068093905243, -0.011543871180870177, -0.0043787611587437335, 
     -0.010140080740061381, -0.003836971851625046, -0.008976310022219372, 
     -0.003388414874669334, -0.008000991183383765, -0.003013060931645409, 
     -0.00717566237641977, -0.002695949672502746, -0.006471172268203613, 
     -0.0024257319897306685, -0.0058651057682879496, -0.0021936771717831565, 
     -0.005340002242507689, -0.0019929833821262515, -0.004882099503332805, 
     -0.0018182902970553302, -0.004480433662060655, -0.001665329084723778, 
     -0.004126184238960464, -0.0015306673315663001, -0.003812191114468502, 
     -0.0014115206625691273, -0.0035325937119097066, -0.0013056118991084291, 
     -0.0032825583373416755, -0.0012110645593647901, -0.00305806991822054, 
     -0.0011263214791944508, -0.002855771344242699, -0.0010500820206406165, 
     -0.002672838382109338, -0.0009812531821662162, -0.002506881447428867, 
     -0.0009189112101729539, -0.002355867846323118, -0.0008622712173314516, 
     -0.002218059757496954, -0.000810662959212895, -0.0020919644191889726, 
     -0.0007635113863433563, -0.0019762938537010387, -0.0007203209278884737, 
     -0.0018699321000194748, -0.0006806627124905693, -0.0017719083978978366, 
     -0.0006441641167529995, -0.0016813751203878203, -0.0006105001702765424, 
     -0.0015975895184233092, -0.0005793864505577765, -0.0015198985436476135, 
     -0.0005505731804158948, -0.001447726170735726, -0.0005238403013684695, 
     -0.0013805627599768491, -0.0004989933432084682, -0.0013179560936061228, 
     -0.00047585994637023015, -0.0012595037917655318, -0.0004542869220386182, 
     -0.0012048468708289522, -0.0004341377572317611, -0.0011536642517338228, 
     -0.00041529048967984015, -0.0011056680616254304, -0.0003976358912899632, 
     -0.0010605996005890754, -0.00038107591013403485};
 
  TVIL_REAL H0xyy0xtableAL[] = 
     {0, 6.579736267392906, 2.210131866303547, 4.193245422464302, 
     1.4314548220406624, 2.4826139201452477, 0.9270809924715527, 
     1.7288512128021611, 0.6653468838094468, 1.3158701672764603, 
     0.5114357962891801, 1.0579197472521689, 0.41189773730586016, 
     0.8824755503722148, 0.3429605332632571, 0.7558331297245247, 
     0.2927300870295686, 0.6603242154489757, 0.2546787715449149, 
     0.5858366412475464, 0.22495650422178362, 0.5261828199124607, 
     0.2011588787221603, 0.4773722515589643, 0.18171321154490827, 
     0.4367198464446333, 0.165550827019242, 0.40235459814989916, 
     0.15192194168643874, 0.3729339818408522, 0.1402860224925113, 
     0.34747036306985685, 0.1302441817874889, 0.3252214229520911, 
     0.12149602707807872, 0.3056187111480296, 0.11381128144445679, 
     0.2882197292679868, 0.10701062200893764, 0.272674992478568, 
     0.10095243891559515, 0.25870488832568, 0.09552349492165942, 
     0.24608310019765828, 0.09063221382467165, 0.2346245250258577, 
     0.08620377705867163, 0.22417632776299912, 0.08217648705316194, 
     0.21461122367958493, 0.07849903297060071, 0.20582236813147084, 
     0.07512840908604636, 0.19771942303018067, 0.0720283117885528, 
     0.190225496124773, 0.06916789208910477, 0.1832747355821417, 
     0.06652077530956094, 0.17681042208905498, 0.06406428376440691, 
     0.17078344261273207, 0.061778815228779864, 0.16515105976057787, 
     0.05964734208883698, 0.159875912136761, 0.05765500480004431, 
     0.1549251967201392, 0.05578877964626892, 0.15026999579162204, 
     0.05403720548562775, 0.14588471949272425, 0.05239015766208715, 
     0.14174664151729674, 0.05083865988561886, 0.13783551030000601, 
     0.04937472687166638, 0.1341332217771072, 0.04799123204913183, 
     0.1306235426514845, 0.04668179581485364, 0.12729187530826291, 
     0.045440690718674244, 0.12412505725586298, 0.04426276067053895, 
     0.12111118932566603, 0.04314335181682576, 0.11823948793749363, 
     0.04207825317246492, 0.11550015759242241, 0.04106364544478385, 
     0.11288428043785284, 0.040096056764388974, 0.11038372029936513, 
     0.03917232426298494, 0.10799103901818717, 0.03828956061949423, 
     0.10569942329400857, 0.03744512484316753, 0.10350262052742265, 
     0.03663659668254348};
 
  TVIL_REAL H0xyy0xtableAL2[] = 
     {0, 0., -0.5, -1., -0.7916666666666666, -0.8333333333333334, 
     -0.7055555555555556, -0.7, -0.6169642857142857, -0.6039682539682539, 
     -0.5457936507936508, -0.5325396825396825, -0.48942400192400193, 
     -0.4774170274170274, -0.444100740529312, -0.43354164354164354, 
     -0.40696612415362415, -0.39773282273282273, -0.3759996630094669, 
     -0.36790611349434876, -0.3497739657143682, -0.3426418721089221, 
     -0.3272640144825622, -0.3209402826275891, -0.30771873703501446, 
     -0.30207665422028057, -0.29057666456092063, -0.28551257157148485, 
     -0.27541017625269976, -0.27083938199768054, -0.2618880309502483, 
     -0.2577411052206704, -0.2497496997147825, -0.2459694057840315, 
     -0.2387874388843334, -0.23532628516831045, -0.22883353562439354, 
     -0.22565184847538589, -0.21975107549199904, -0.2168154878613, 
     -0.2114271519468188, -0.20870941653348177, -0.20376779804901388, 
     -0.20124385147201576, -0.1966941521761161, -0.1943433730368489, 
     -0.19013952109202345, -0.18794413812706828, -0.1840471045165605, 
     -0.18199172143118852, -0.178368213533177, -0.17643942503252646, 
     -0.1730608620090735, -0.1712469415357528, -0.1680886428410086, 
     -0.16637928704524615, -0.16341982389531945, -0.1618059422730958, 
     -0.1590266150042072, -0.15750015572009146, -0.15488456932061348, 
     -0.15343837419513895, -0.1509720910747298, -0.1495997742169128, 
     -0.1472700282408053, -0.14596587396017752, -0.143761333452693, 
     -0.14252020997737014, -0.14043078014774374, -0.13924806636896142, 
     -0.13726472368761838, -0.13613624669403018, -0.1342508993275399, 
     -0.1331728809197045, -0.1313782505457483, -0.13034726125919238, 
     -0.1286367825214791, -0.12764970195558153, -0.12601743655016384, 
     -0.12507141901512445, -0.12351198197363791, -0.12260442664063004, 
     -0.12111292282943031, -0.12024144770865257, -0.1188134169232239, 
     -0.11797583610788066, -0.11660720543001248, -0.11580150913664425, 
     -0.11448855145349891, -0.11371288846474663, -0.11245218623613985, 
     -0.11170484841425309, -0.11049326192652469, -0.10977267051731006, 
     -0.10860730998626852, -0.10791200347577476, -0.10679020446296059, 
     -0.10611882778464828, -0.1050381294749654, -0.10438942439471939, 
     -0.1033475503527924};
 
  TVIL_REAL H0xyy0xtableAL3[] = 
     {0, 0., -0.3333333333333333, 0., -0.16666666666666666, 0., 
     -0.1111111111111111, 0., -0.08333333333333333, 0., -0.06666666666666667, 
     0., -0.05555555555555555, 0., -0.047619047619047616, 0., 
     -0.041666666666666664, 0., -0.037037037037037035, 0., 
     -0.03333333333333333, 0., -0.030303030303030304, 0., 
     -0.027777777777777776, 0., -0.02564102564102564, 0., 
     -0.023809523809523808, 0., -0.022222222222222223, 0., 
     -0.020833333333333332, 0., -0.0196078431372549, 0., 
     -0.018518518518518517, 0., -0.017543859649122806, 0., 
     -0.016666666666666666, 0., -0.015873015873015872, 0., 
     -0.015151515151515152, 0., -0.014492753623188406, 0., 
     -0.013888888888888888, 0., -0.013333333333333334, 0., 
     -0.01282051282051282, 0., -0.012345679012345678, 0., 
     -0.011904761904761904, 0., -0.011494252873563218, 0., 
     -0.011111111111111112, 0., -0.010752688172043012, 0., 
     -0.010416666666666666, 0., -0.010101010101010102, 0., 
     -0.00980392156862745, 0., -0.009523809523809525, 0., 
     -0.009259259259259259, 0., -0.009009009009009009, 0., 
     -0.008771929824561403, 0., -0.008547008547008548, 0., 
     -0.008333333333333333, 0., -0.008130081300813009, 0., 
     -0.007936507936507936, 0., -0.007751937984496124, 0., 
     -0.007575757575757576, 0., -0.007407407407407408, 0., 
     -0.007246376811594203, 0., -0.0070921985815602835, 0., 
     -0.006944444444444444, 0., -0.006802721088435374, 0., 
     -0.006666666666666667};
 
  TVIL_REAL H0xyy0xtableD[] = 
     {0, -3.606170709478783, 1.1994857742101015, -0.5984573226303043, 
     0.3583467481102707, -0.23844186390619648, 0.17000927804351956, 
     -0.12729062831167723, 0.09884520361541065, -0.07895637335069727, 
     0.06450800957722107, -0.053683559634000746, 0.0453658770374085, 
     -0.038837266390966844, 0.033619576632724, -0.02938429268412622, 
     0.025899671384210524, -0.02299844993084313, 0.020557460260752208, 
     -0.018484384298519563, 0.016708928065145782, -0.015176806714433281, 
     0.013845562079967004, -0.012681601425626658, 0.011658066294194963, 
     -0.010753275817378628, 0.009949574120390697, -0.009232466254203582, 
     0.008589962981235405, -0.008012078661461006, 0.007490442685745521, 
     -0.007017996037192094, 0.006588752318997055, -0.006197608061668003, 
     0.005840191031531549, -0.005512738084818812, 0.005211996170553493, 
     -0.004935141602119661, 0.004679713844813082, -0.004443560911995443, 
     0.004224794101420167, -0.004021750289971531, 0.0038329603784409577, 
     -0.0036571227664014302, 0.003493080961514792, -0.003339804603073875, 
     0.0031963733176748367, -0.0030619629342073427, 0.0029358336723128847, 
     -0.002817319988011697, 0.002705821816095705, -0.0026007969940197166, 
     0.002501754688631565, -0.0024082496769027295, 0.0023198773562130427, 
     -0.0022362693797739466, 0.0021570898292869005, -0.002082031850595065, 
     0.0020108146894293414, -0.0019431810737997731, 0.001878894897481995, 
     -0.0018177391656713472, 0.0017595141694475122, -0.0017040358603911921, 
     0.001651134400669104, -0.0016006528672752513, 0.0015524460919841324, 
     -0.0015063796210169186, 0.001462328780512087, -0.0014201778356833594, 
     0.0013798192330864902, -0.0013411529167410968, 0.0013040857099966563, 
     -0.001268530756020019, 0.0012344070106380656, -0.0012016387820125492, 
     0.0011701553122709007, -0.0011398903967804466, 0.0011107820372457079, 
     -0.0010827721252390384, 0.0010558061531522056, -0.00102983294988777, 
     0.0010048044389003958, -0.000980675416454771, 0.0009574033481931326, 
     -0.0009349481823053392, 0.0009132721777713721, -0.0008923397463029395, 
     0.0008721173067499998, -0.0008525731508616843, 0.0008336773194011454, 
     -0.0008154014877119206, 0.0007977188599209073, -0.0007806040710412193, 
     0.0007640330963081334, -0.0007479831671439653, 0.0007324326932038902, 
     -0.0007173611900051436, 0.0007027492116873655, -0.0006885782884926334, 
     0.0006789570033403114};

  if (y < 0.5 * x) {
    result = TVIL_H00xx00 (x,qq);
    rn = r = y/x;
    for (n=1; n < 100; n++) {
      result += rn * (H0xyy0xtableA[n] + 
                      H0xyy0xtableAL[n] * lnyox +
                      H0xyy0xtableAL2[n] * lnyox2 + 
                      H0xyy0xtableAL3[n] * lnyox3);
      rn *= r;
    }
  }
  else {
    result = TVIL_H0xxx0x (x,qq);
    rn = r = y/x - 1.L;
    for (n=1; n < 100; n++) {
      result += rn * H0xyy0xtableD[n];
      rn *= r;
    }
  }

  return result;
}

/* ****************************************************************** */

int TVIL_AnalyticHTYPE (TVIL_HTYPE *foo, TVIL_REAL qq)
{
  int retval;

  retval = TVIL_Hanalytic (foo->arg[0],
			   foo->arg[1],
			   foo->arg[2],
			   foo->arg[3],
			   foo->arg[4],
			   foo->arg[5],
			   qq,
			   &(foo->value));

  return retval;
}

/* ****************************************************************** */

int TVIL_Hanalytic (TVIL_REAL a1,
		    TVIL_REAL a2,
		    TVIL_REAL a3,
		    TVIL_REAL a4,
		    TVIL_REAL a5,
		    TVIL_REAL a6,
		    TVIL_REAL QQ,
		    TVIL_COMPLEX *result)
{
  int success = 1;

  if (SAME7(0,a1,a2,a3,a4,a5,a6))
    *result = TVIL_ComplexInfinity;

  else if (SAME6(0,a2,a3,a4,a5,a6))
    *result = TVIL_H00000x (a1, QQ);
  else if (SAME6(0,a1,a3,a4,a5,a6))
    *result = TVIL_H00000x (a2, QQ);
  else if (SAME6(0,a1,a2,a4,a5,a6))
    *result = TVIL_H00000x (a3, QQ);
  else if (SAME6(0,a1,a2,a3,a5,a6))
    *result = TVIL_H00000x (a4, QQ);
  else if (SAME6(0,a1,a2,a3,a4,a6))
    *result = TVIL_H00000x (a5, QQ);
  else if (SAME6(0,a1,a2,a3,a4,a5))
    *result = TVIL_H00000x (a6, QQ);
	
  else if (SAME5(0,a1,a2,a3,a4) && SAME2(a5,a6))
    *result = TVIL_H0000xx (a6, QQ);
  else if (SAME5(0,a1,a2,a3,a5) && SAME2(a4,a6))
    *result = TVIL_H0000xx (a6, QQ);
  else if (SAME5(0,a1,a2,a4,a5) && SAME2(a3,a6))
    *result = TVIL_H0000xx (a6, QQ);
  else if (SAME5(0,a2,a3,a4,a5) && SAME2(a1,a6))
    *result = TVIL_H0000xx (a6, QQ);
  else if (SAME5(0,a1,a2,a3,a6) && SAME2(a4,a5))
    *result = TVIL_H0000xx (a5, QQ);
  else if (SAME5(0,a1,a2,a4,a6) && SAME2(a3,a5))
    *result = TVIL_H0000xx (a5, QQ);
  else if (SAME5(0,a1,a3,a4,a6) && SAME2(a2,a5))
    *result = TVIL_H0000xx (a5, QQ);
  else if (SAME5(0,a1,a3,a5,a6) && SAME2(a2,a4))
    *result = TVIL_H0000xx (a4, QQ);
  else if (SAME5(0,a2,a3,a5,a6) && SAME2(a1,a4))
    *result = TVIL_H0000xx (a4, QQ);
  else if (SAME5(0,a1,a4,a5,a6) && SAME2(a2,a3)) 
    *result = TVIL_H0000xx (a3, QQ);
  else if (SAME5(0,a2,a4,a5,a6) && SAME2(a1,a3))
    *result = TVIL_H0000xx (a3, QQ);
  else if (SAME5(0,a3,a4,a5,a6) && SAME2(a1,a2))
    *result = TVIL_H0000xx (a2, QQ);

  else if (SAME5(0,a1,a3,a4,a5) && SAME2(a2,a6))
    *result = TVIL_H00xx00 (a6, QQ);
  else if (SAME5(0,a2,a3,a4,a6) && SAME2(a1,a5))
    *result = TVIL_H00xx00 (a5, QQ);
  else if (SAME5(0,a1,a2,a5,a6) && SAME2(a3,a4))
    *result = TVIL_H00xx00 (a4, QQ);

  else if (SAME4(0,a1,a2,a3) && SAME3(a4,a5,a6))
    *result = TVIL_H000xxx (a6, QQ);
  else if (SAME4(0,a2,a4,a5) && SAME3(a1,a3,a6))
    *result = TVIL_H000xxx (a6, QQ);
  else if (SAME4(0,a1,a4,a6) && SAME3(a2,a3,a5))
    *result = TVIL_H000xxx (a5, QQ);
  else if (SAME4(0,a3,a5,a6) && SAME3(a1,a2,a4))
    *result = TVIL_H000xxx (a4, QQ);

  else if (SAME4(0,a1,a2,a4) && SAME3(a3,a5,a6))
    *result = TVIL_H00x0xx (a6, QQ);
  else if (SAME4(0,a2,a3,a5) && SAME3(a1,a4,a6))
    *result = TVIL_H00x0xx (a6, QQ);
  else if (SAME4(0,a1,a3,a6) && SAME3(a2,a4,a5))
    *result = TVIL_H00x0xx (a5, QQ);
  else if (SAME4(0,a4,a5,a6) && SAME3(a1,a2,a3))
    *result = TVIL_H00x0xx (a3, QQ);

  else if (SAME4(0,a1,a2,a5) && SAME3(a3,a4,a6))
    *result = TVIL_H00xx0x (a6, QQ);
  else if (SAME4(0,a1,a3,a5) && SAME3(a2,a4,a6)) 
    *result = TVIL_H00xx0x (a6, QQ);
  else if (SAME4(0,a1,a3,a4) && SAME3(a2,a5,a6))
    *result = TVIL_H00xx0x (a6, QQ);
  else if (SAME4(0,a2,a3,a4) && SAME3(a1,a5,a6))
    *result = TVIL_H00xx0x (a6, QQ);
  else if (SAME4(0,a3,a4,a5) && SAME3(a1,a2,a6))
    *result = TVIL_H00xx0x (a6, QQ);
  else if (SAME4(0,a1,a4,a5) && SAME3(a2,a3,a6))
    *result = TVIL_H00xx0x (a6, QQ);
  else if (SAME4(0,a2,a3,a6) && SAME3(a1,a4,a5))
    *result = TVIL_H00xx0x (a5, QQ);
  else if (SAME4(0,a1,a2,a6) && SAME3(a3,a4,a5))
    *result = TVIL_H00xx0x (a5, QQ);
  else if (SAME4(0,a2,a4,a6) && SAME3(a1,a3,a5))
    *result = TVIL_H00xx0x (a5, QQ);
  else if (SAME4(0,a3,a4,a6) && SAME3(a1,a2,a5))
    *result = TVIL_H00xx0x (a5, QQ);
  else if (SAME4(0,a1,a5,a6) && SAME3(a2,a3,a4))
    *result = TVIL_H00xx0x (a4, QQ);
  else if (SAME4(0,a2,a5,a6) && SAME3(a1,a3,a4))
    *result = TVIL_H00xx0x (a4, QQ);

  else if (SAME3(0,a1,a2) && SAME4(a3,a4,a5,a6))
    *result = TVIL_H00xxxx (a6, QQ);
  else if (SAME3(0,a1,a3) && SAME4(a2,a4,a5,a6))
    *result = TVIL_H00xxxx (a6, QQ);
  else if (SAME3(0,a1,a4) && SAME4(a2,a3,a5,a6))
    *result = TVIL_H00xxxx (a6, QQ);
  else if (SAME3(0,a2,a3) && SAME4(a1,a4,a5,a6))
    *result = TVIL_H00xxxx (a6, QQ);
  else if (SAME3(0,a2,a4) && SAME4(a1,a3,a5,a6))
    *result = TVIL_H00xxxx (a6, QQ);
  else if (SAME3(0,a2,a5) && SAME4(a1,a3,a4,a6))
    *result = TVIL_H00xxxx (a6, QQ);
  else if (SAME3(0,a3,a5) && SAME4(a1,a2,a4,a6))
    *result = TVIL_H00xxxx (a6, QQ);
  else if (SAME3(0,a4,a5) && SAME4(a1,a2,a3,a6))
    *result = TVIL_H00xxxx (a6, QQ);
  else if (SAME3(0,a1,a6) && SAME4(a2,a3,a4,a5))
    *result = TVIL_H00xxxx (a5, QQ);
  else if (SAME3(0,a3,a6) && SAME4(a1,a2,a4,a5))
    *result = TVIL_H00xxxx (a5, QQ);
  else if (SAME3(0,a4,a6) && SAME4(a1,a2,a3,a5))
    *result = TVIL_H00xxxx (a5, QQ);
  else if (SAME3(0,a5,a6) && SAME4(a1,a2,a3,a4))
    *result = TVIL_H00xxxx (a4, QQ);

  else if (SAME3(0,a1,a5) && SAME4(a2,a3,a4,a6))
    *result = TVIL_H0xxx0x (a6, QQ);
  else if (SAME3(0,a3,a4) && SAME4(a1,a2,a5,a6))
    *result = TVIL_H0xxx0x (a6, QQ);
  else if (SAME3(0,a2,a6) && SAME4(a1,a3,a4,a5))
    *result = TVIL_H0xxx0x (a5, QQ);

  else if (SAME2(0,a1) && SAME5(a2,a3,a4,a5,a6))
    *result = TVIL_H0xxxxx (a6, QQ);
  else if (SAME2(0,a2) && SAME5(a1,a3,a4,a5,a6))
    *result = TVIL_H0xxxxx (a6, QQ);
  else if (SAME2(0,a3) && SAME5(a1,a2,a4,a5,a6))
    *result = TVIL_H0xxxxx (a6, QQ);
  else if (SAME2(0,a4) && SAME5(a1,a2,a3,a5,a6))
    *result = TVIL_H0xxxxx (a6, QQ);
  else if (SAME2(0,a5) && SAME5(a1,a2,a3,a4,a6))
    *result = TVIL_H0xxxxx (a6, QQ);
  else if (SAME2(0,a6) && SAME5(a1,a2,a3,a4,a5))
    *result = TVIL_H0xxxxx (a5, QQ);

  else if (SAME6(a1,a2,a3,a4,a5,a6))
    *result = TVIL_Hxxxxxx (a6, QQ);

  else if (SAME5(0,a1,a2,a3,a4) && (a5>0) && (a6>0))
    *result = TVIL_H0000xy (a5, a6, QQ);

  else if (SAME5(0,a1,a2,a3,a5) && (a4>0) && (a6>0))
    *result = TVIL_H0000xy (a4, a6, QQ);
  else if (SAME5(0,a1,a2,a4,a5) && (a3>0) && (a6>0))
    *result = TVIL_H0000xy (a3, a6, QQ);
  else if (SAME5(0,a2,a3,a4,a5) && (a1>0) && (a6>0))
    *result = TVIL_H0000xy (a1, a6, QQ);
  else if (SAME5(0,a1,a2,a3,a6) && (a4>0) && (a5>0))
    *result = TVIL_H0000xy (a4, a5, QQ);
  else if (SAME5(0,a1,a2,a4,a6) && (a3>0) && (a5>0))
    *result = TVIL_H0000xy (a3, a5, QQ);
  else if (SAME5(0,a1,a3,a4,a6) && (a2>0) && (a5>0))
    *result = TVIL_H0000xy (a2, a5, QQ);
  else if (SAME5(0,a1,a3,a5,a6) && (a2>0) && (a4>0))
    *result = TVIL_H0000xy (a2, a4, QQ);
  else if (SAME5(0,a2,a3,a5,a6) && (a1>0) && (a4>0))
    *result = TVIL_H0000xy (a1, a4, QQ);
  else if (SAME5(0,a1,a4,a5,a6) && (a2>0) && (a3>0)) 
    *result = TVIL_H0000xy (a2, a3, QQ);
  else if (SAME5(0,a2,a4,a5,a6) && (a1>0) && (a3>0))
    *result = TVIL_H0000xy (a1, a3, QQ);
  else if (SAME5(0,a3,a4,a5,a6) && (a1>0) && (a2>0))
    *result = TVIL_H0000xy (a1, a2, QQ);

  else if (SAME5(0,a1,a3,a4,a5) && (a2>0) && (a6>0))
    *result = TVIL_H00xy00 (a2, a6, QQ);
  else if (SAME5(0,a2,a3,a4,a6) && (a1>0) && (a5>0))
    *result = TVIL_H00xy00 (a1, a5, QQ);
  else if (SAME5(0,a1,a2,a5,a6) && (a3>0) && (a4>0))
    *result = TVIL_H00xy00 (a3, a4, QQ);

  else if (SAME3(0,a1,a5) && SAME2(a2,a3) && SAME2(a4,a6) && (a2>0) && (a4>0))
    *result = TVIL_Hxx00yy (a2, a4, QQ);
  else if (SAME3(0,a1,a5) && SAME2(a2,a4) && SAME2(a3,a6) && (a2>0) && (a3>0))
    *result = TVIL_Hxx00yy (a2, a3, QQ);
  else if (SAME3(0,a2,a6) && SAME2(a1,a4) && SAME2(a3,a5) && (a1>0) && (a3>0))
    *result = TVIL_Hxx00yy (a1, a3, QQ);
  else if (SAME3(0,a2,a6) && SAME2(a1,a3) && SAME2(a4,a5) && (a1>0) && (a4>0))
    *result = TVIL_Hxx00yy (a1, a4, QQ);
  else if (SAME3(0,a3,a4) && SAME2(a1,a2) && SAME2(a5,a6) && (a1>0) && (a5>0))
    *result = TVIL_Hxx00yy (a1, a5, QQ);
  else if (SAME3(0,a3,a4) && SAME2(a1,a6) && SAME2(a2,a5) && (a1>0) && (a2>0))
    *result = TVIL_Hxx00yy (a1, a2, QQ);

  else if (SAME4(a1,a2,a5,a6) && SAME2(0,a3) && (a1>0) && (a4>0))
    *result = TVIL_Hxx0yxx (a1, a4, QQ);
  else if (SAME4(a1,a2,a5,a6) && SAME2(0,a4) && (a1>0) && (a3>0))
    *result = TVIL_Hxx0yxx (a1, a3, QQ);
  else if (SAME4(a1,a3,a4,a5) && SAME2(0,a2) && (a1>0) && (a6>0))
    *result = TVIL_Hxx0yxx (a1, a6, QQ);
  else if (SAME4(a1,a3,a4,a5) && SAME2(0,a6) && (a1>0) && (a2>0))
    *result = TVIL_Hxx0yxx (a1, a2, QQ);
  else if (SAME4(a2,a3,a4,a6) && SAME2(0,a1) && (a2>0) && (a5>0))
    *result = TVIL_Hxx0yxx (a2, a5, QQ);
  else if (SAME4(a2,a3,a4,a6) && SAME2(0,a5) && (a2>0) && (a1>0))
    *result = TVIL_Hxx0yxx (a2, a1, QQ);

  else if (SAME3(0,a1,a2) && SAME3(a3,a5,a6) && (a3>0) && (a4>0))
    *result = TVIL_H00xyxx (a3, a4, QQ);
  else if (SAME3(0,a1,a3) && SAME3(a2,a4,a5) && (a2>0) && (a6>0))
    *result = TVIL_H00xyxx (a2, a6, QQ);
  else if (SAME3(0,a1,a4) && SAME3(a3,a5,a6) && (a3>0) && (a2>0))
    *result = TVIL_H00xyxx (a3, a2, QQ);
  else if (SAME3(0,a1,a6) && SAME3(a2,a4,a5) && (a2>0) && (a3>0))
    *result = TVIL_H00xyxx (a2, a3, QQ);
  else if (SAME3(0,a2,a3) && SAME3(a1,a4,a6) && (a1>0) && (a5>0))
    *result = TVIL_H00xyxx (a1, a5, QQ);
  else if (SAME3(0,a2,a4) && SAME3(a3,a5,a6) && (a3>0) && (a1>0))
    *result = TVIL_H00xyxx (a3, a1, QQ);
  else if (SAME3(0,a2,a5) && SAME3(a1,a4,a6) && (a1>0) && (a3>0))
    *result = TVIL_H00xyxx (a1, a3, QQ);
  else if (SAME3(0,a3,a5) && SAME3(a1,a4,a6) && (a1>0) && (a2>0))
    *result = TVIL_H00xyxx (a1, a2, QQ);
  else if (SAME3(0,a3,a6) && SAME3(a2,a4,a5) && (a2>0) && (a1>0))
    *result = TVIL_H00xyxx (a2, a1, QQ);
  else if (SAME3(0,a4,a5) && SAME3(a1,a2,a3) && (a1>0) && (a6>0))
    *result = TVIL_H00xyxx (a1, a6, QQ);
  else if (SAME3(0,a4,a6) && SAME3(a1,a2,a3) && (a1>0) && (a5>0))
    *result = TVIL_H00xyxx (a1, a5, QQ);
  else if (SAME3(0,a5,a6) && SAME3(a1,a2,a3) && (a1>0) && (a4>0))
    *result = TVIL_H00xyxx (a1, a4, QQ);

  else if (SAME3(0,a1,a2) && SAME3(a3,a4,a5) && (a3>0) && (a6>0))
    *result = TVIL_H00xxxy (a3, a6, QQ);
  else if (SAME3(0,a1,a2) && SAME3(a3,a4,a6) && (a3>0) && (a5>0))
    *result = TVIL_H00xxxy (a3, a5, QQ);
  else if (SAME3(0,a1,a3) && SAME3(a2,a4,a6) && (a2>0) && (a5>0))
    *result = TVIL_H00xxxy (a2, a5, QQ);
  else if (SAME3(0,a1,a3) && SAME3(a2,a5,a6) && (a2>0) && (a4>0))
    *result = TVIL_H00xxxy (a2, a4, QQ);
  else if (SAME3(0,a1,a4) && SAME3(a2,a3,a6) && (a2>0) && (a5>0))
    *result = TVIL_H00xxxy (a2, a5, QQ);
  else if (SAME3(0,a1,a4) && SAME3(a2,a5,a6) && (a2>0) && (a3>0))
    *result = TVIL_H00xxxy (a2, a3, QQ);
  else if (SAME3(0,a1,a6) && SAME3(a2,a3,a4) && (a2>0) && (a5>0))
    *result = TVIL_H00xxxy (a2, a5, QQ);
  else if (SAME3(0,a1,a6) && SAME3(a3,a4,a5) && (a3>0) && (a2>0))
    *result = TVIL_H00xxxy (a3, a2, QQ);
  else if (SAME3(0,a2,a3) && SAME3(a1,a4,a5) && (a1>0) && (a6>0))
    *result = TVIL_H00xxxy (a1, a6, QQ);
  else if (SAME3(0,a2,a3) && SAME3(a1,a5,a6) && (a1>0) && (a4>0))
    *result = TVIL_H00xxxy (a1, a4, QQ);
  else if (SAME3(0,a2,a4) && SAME3(a1,a3,a5) && (a1>0) && (a6>0))
    *result = TVIL_H00xxxy (a1, a6, QQ);
  else if (SAME3(0,a2,a4) && SAME3(a1,a5,a6) && (a1>0) && (a3>0))
    *result = TVIL_H00xxxy (a1, a3, QQ);
  else if (SAME3(0,a2,a5) && SAME3(a1,a3,a4) && (a1>0) && (a6>0))
    *result = TVIL_H00xxxy (a1, a6, QQ);
  else if (SAME3(0,a2,a5) && SAME3(a3,a4,a6) && (a3>0) && (a1>0))
    *result = TVIL_H00xxxy (a3, a1, QQ);
  else if (SAME3(0,a3,a5) && SAME3(a1,a2,a6) && (a1>0) && (a4>0))
    *result = TVIL_H00xxxy (a1, a4, QQ);
  else if (SAME3(0,a3,a5) && SAME3(a2,a4,a6) && (a2>0) && (a1>0))
    *result = TVIL_H00xxxy (a2, a1, QQ);
  else if (SAME3(0,a3,a6) && SAME3(a1,a2,a5) && (a1>0) && (a4>0))
    *result = TVIL_H00xxxy (a1, a4, QQ);
  else if (SAME3(0,a3,a6) && SAME3(a1,a4,a5) && (a1>0) && (a2>0))
    *result = TVIL_H00xxxy (a1, a2, QQ);
  else if (SAME3(0,a4,a5) && SAME3(a1,a2,a6) && (a1>0) && (a3>0))
    *result = TVIL_H00xxxy (a1, a3, QQ);
  else if (SAME3(0,a4,a5) && SAME3(a2,a3,a6) && (a2>0) && (a1>0))
    *result = TVIL_H00xxxy (a2, a1, QQ);
  else if (SAME3(0,a4,a6) && SAME3(a1,a2,a5) && (a1>0) && (a3>0))
    *result = TVIL_H00xxxy (a1, a3, QQ);
  else if (SAME3(0,a4,a6) && SAME3(a1,a3,a5) && (a1>0) && (a2>0))
    *result = TVIL_H00xxxy (a1, a2, QQ);
  else if (SAME3(0,a5,a6) && SAME3(a1,a3,a4) && (a1>0) && (a2>0))
    *result = TVIL_H00xxxy (a1, a2, QQ);
  else if (SAME3(0,a5,a6) && SAME3(a2,a3,a4) && (a2>0) && (a1>0))
    *result = TVIL_H00xxxy (a2, a1, QQ);

  else if (SAME4(0,a1,a2,a4) && SAME2(a5,a6) && (a5>0) && (a3>0))
    *result = TVIL_H00y0xx (a5, a3, QQ);
  else if (SAME4(0,a1,a2,a4) && SAME2(a3,a6) && (a3>0) && (a5>0))
    *result = TVIL_H00y0xx (a3, a5, QQ);
  else if (SAME4(0,a1,a2,a4) && SAME2(a3,a5) && (a3>0) && (a6>0))
    *result = TVIL_H00y0xx (a3, a6, QQ);
  else if (SAME4(0,a4,a5,a6) && SAME2(a1,a2) && (a1>0) && (a3>0))
    *result = TVIL_H00y0xx (a1, a3, QQ);
  else if (SAME4(0,a4,a5,a6) && SAME2(a2,a3) && (a2>0) && (a1>0))
    *result = TVIL_H00y0xx (a2, a1, QQ);
  else if (SAME4(0,a4,a5,a6) && SAME2(a1,a3) && (a1>0) && (a2>0))
    *result = TVIL_H00y0xx (a1, a2, QQ);
  else if (SAME4(0,a1,a3,a6) && SAME2(a2,a5) && (a2>0) && (a4>0))
    *result = TVIL_H00y0xx (a2, a4, QQ);
  else if (SAME4(0,a1,a3,a6) && SAME2(a2,a4) && (a2>0) && (a5>0))
    *result = TVIL_H00y0xx (a2, a5, QQ);
  else if (SAME4(0,a1,a3,a6) && SAME2(a4,a5) && (a4>0) && (a2>0))
    *result = TVIL_H00y0xx (a4, a2, QQ);
  else if (SAME4(0,a2,a3,a5) && SAME2(a1,a6) && (a1>0) && (a4>0))
    *result = TVIL_H00y0xx (a1, a4, QQ);
  else if (SAME4(0,a2,a3,a5) && SAME2(a4,a6) && (a4>0) && (a1>0))
    *result = TVIL_H00y0xx (a4, a1, QQ);
  else if (SAME4(0,a2,a3,a5) && SAME2(a1,a4) && (a1>0) && (a6>0))
    *result = TVIL_H00y0xx (a1, a6, QQ);

  else if (SAME4(0,a1,a2,a5) && SAME2(a3,a4) && (a3>0) && (a6>0))
    *result = TVIL_H00xx0y (a3, a6, QQ);
  else if (SAME4(0,a1,a2,a6) && SAME2(a3,a4) && (a3>0) && (a5>0))
    *result = TVIL_H00xx0y (a3, a5, QQ);
  else if (SAME4(0,a1,a3,a4) && SAME2(a2,a6) && (a2>0) && (a5>0))
    *result = TVIL_H00xx0y (a2, a5, QQ);
  else if (SAME4(0,a1,a3,a5) && SAME2(a2,a6) && (a2>0) && (a4>0))
    *result = TVIL_H00xx0y (a2, a4, QQ);
  else if (SAME4(0,a1,a4,a5) && SAME2(a2,a6) && (a2>0) && (a3>0))
    *result = TVIL_H00xx0y (a2, a3, QQ);
  else if (SAME4(0,a1,a5,a6) && SAME2(a3,a4) && (a3>0) && (a2>0))
    *result = TVIL_H00xx0y (a3, a2, QQ);
  else if (SAME4(0,a2,a3,a4) && SAME2(a1,a5) && (a1>0) && (a6>0))
    *result = TVIL_H00xx0y (a1, a6, QQ);
  else if (SAME4(0,a2,a3,a6) && SAME2(a1,a5) && (a1>0) && (a4>0))
    *result = TVIL_H00xx0y (a1, a4, QQ);
  else if (SAME4(0,a2,a4,a6) && SAME2(a1,a5) && (a1>0) && (a3>0))
    *result = TVIL_H00xx0y (a1, a3, QQ);
  else if (SAME4(0,a3,a4,a6) && SAME2(a1,a5) && (a1>0) && (a2>0))
    *result = TVIL_H00xx0y (a1, a2, QQ);
  else if (SAME4(0,a2,a5,a6) && SAME2(a3,a4) && (a3>0) && (a1>0))
    *result = TVIL_H00xx0y (a3, a1, QQ);
  else if (SAME4(0,a3,a4,a5) && SAME2(a2,a6) && (a2>0) && (a1>0))
    *result = TVIL_H00xx0y (a2, a1, QQ);

  else if (SAME4(0,a1,a2,a5) && SAME2(a3,a6) && (a3>0) && (a4>0))
    *result = TVIL_H00xy0x (a3, a4, QQ);
  else if (SAME4(0,a1,a2,a6) && SAME2(a3,a5) && (a3>0) && (a4>0))
    *result = TVIL_H00xy0x (a3, a4, QQ);
  else if (SAME4(0,a1,a2,a5) && SAME2(a4,a6) && (a4>0) && (a3>0))
    *result = TVIL_H00xy0x (a4, a3, QQ);
  else if (SAME4(0,a1,a2,a6) && SAME2(a4,a5) && (a4>0) && (a3>0))
    *result = TVIL_H00xy0x (a4, a3, QQ);
  else if (SAME4(0,a1,a3,a4) && SAME2(a2,a5) && (a2>0) && (a6>0))
    *result = TVIL_H00xy0x (a2, a6, QQ);
  else if (SAME4(0,a1,a3,a5) && SAME2(a2,a4) && (a2>0) && (a6>0))
    *result = TVIL_H00xy0x (a2, a6, QQ);
  else if (SAME4(0,a1,a4,a5) && SAME2(a2,a3) && (a2>0) && (a6>0))
    *result = TVIL_H00xy0x (a2, a6, QQ);
  else if (SAME4(0,a1,a5,a6) && SAME2(a2,a3) && (a2>0) && (a4>0))
    *result = TVIL_H00xy0x (a2, a4, QQ);
  else if (SAME4(0,a1,a5,a6) && SAME2(a2,a4) && (a2>0) && (a3>0))
    *result = TVIL_H00xy0x (a2, a3, QQ);
  else if (SAME4(0,a1,a3,a4) && SAME2(a5,a6) && (a5>0) && (a2>0))
    *result = TVIL_H00xy0x (a5, a2, QQ);
  else if (SAME4(0,a1,a3,a5) && SAME2(a4,a6) && (a4>0) && (a2>0))
    *result = TVIL_H00xy0x (a4, a2, QQ);
  else if (SAME4(0,a1,a4,a5) && SAME2(a3,a6) && (a3>0) && (a2>0))
    *result = TVIL_H00xy0x (a3, a2, QQ);
  else if (SAME4(0,a2,a3,a4) && SAME2(a1,a6) && (a1>0) && (a5>0))
    *result = TVIL_H00xy0x (a1, a5, QQ);
  else if (SAME4(0,a2,a3,a6) && SAME2(a1,a4) && (a1>0) && (a5>0))
    *result = TVIL_H00xy0x (a1, a5, QQ);
  else if (SAME4(0,a2,a4,a6) && SAME2(a1,a3) && (a1>0) && (a5>0))
    *result = TVIL_H00xy0x (a1, a5, QQ);
  else if (SAME4(0,a2,a5,a6) && SAME2(a1,a3) && (a1>0) && (a4>0))
    *result = TVIL_H00xy0x (a1, a4, QQ);
  else if (SAME4(0,a2,a5,a6) && SAME2(a1,a4) && (a1>0) && (a3>0))
    *result = TVIL_H00xy0x (a1, a3, QQ);
  else if (SAME4(0,a3,a4,a5) && SAME2(a1,a2) && (a1>0) && (a6>0))
    *result = TVIL_H00xy0x (a1, a6, QQ);
  else if (SAME4(0,a3,a4,a6) && SAME2(a1,a2) && (a1>0) && (a5>0))
    *result = TVIL_H00xy0x (a1, a5, QQ);
  else if (SAME4(0,a3,a4,a5) && SAME2(a1,a6) && (a1>0) && (a2>0))
    *result = TVIL_H00xy0x (a1, a2, QQ);
  else if (SAME4(0,a2,a3,a4) && SAME2(a5,a6) && (a5>0) && (a1>0))
    *result = TVIL_H00xy0x (a5, a1, QQ);
  else if (SAME4(0,a2,a3,a6) && SAME2(a4,a5) && (a4>0) && (a1>0))
    *result = TVIL_H00xy0x (a4, a1, QQ);
  else if (SAME4(0,a2,a4,a6) && SAME2(a3,a5) && (a3>0) && (a1>0))
    *result = TVIL_H00xy0x (a3, a1, QQ);
  else if (SAME4(0,a3,a4,a6) && SAME2(a2,a5) && (a2>0) && (a1>0))
    *result = TVIL_H00xy0x (a2, a1, QQ);

  else if (SAME2(0,a1) && SAME3(a2,a4,a5) && SAME2(a3,a6) && (a2>0) && (a6>0))
    *result = TVIL_H0xyxxy (a2, a3, QQ);
  else if (SAME2(0,a1) && SAME3(a3,a5,a6) && SAME2(a2,a4) && (a3>0) && (a2>0))
    *result = TVIL_H0xyxxy (a3, a2, QQ);
  else if (SAME2(0,a2) && SAME3(a1,a4,a6) && SAME2(a3,a5) && (a1>0) && (a3>0))
    *result = TVIL_H0xyxxy (a1, a3, QQ);
  else if (SAME2(0,a2) && SAME3(a3,a5,a6) && SAME2(a1,a4) && (a3>0) && (a1>0))
    *result = TVIL_H0xyxxy (a3, a1, QQ);
  else if (SAME2(0,a3) && SAME3(a2,a4,a5) && SAME2(a1,a6) && (a2>0) && (a1>0))
    *result = TVIL_H0xyxxy (a2, a1, QQ);
  else if (SAME2(0,a3) && SAME3(a1,a4,a6) && SAME2(a2,a5) && (a1>0) && (a2>0))
    *result = TVIL_H0xyxxy (a1, a2, QQ);
  else if (SAME2(0,a4) && SAME3(a1,a2,a3) && SAME2(a5,a6) && (a1>0) && (a5>0))
    *result = TVIL_H0xyxxy (a1, a5, QQ);
  else if (SAME2(0,a4) && SAME3(a3,a5,a6) && SAME2(a1,a2) && (a3>0) && (a1>0))
    *result = TVIL_H0xyxxy (a3, a1, QQ);
  else if (SAME2(0,a5) && SAME3(a1,a2,a3) && SAME2(a4,a6) && (a1>0) && (a4>0))
    *result = TVIL_H0xyxxy (a1, a4, QQ);
  else if (SAME2(0,a5) && SAME3(a1,a4,a6) && SAME2(a2,a3) && (a1>0) && (a2>0))
    *result = TVIL_H0xyxxy (a1, a2, QQ);
  else if (SAME2(0,a6) && SAME3(a1,a2,a3) && SAME2(a4,a5) && (a1>0) && (a4>0))
    *result = TVIL_H0xyxxy (a1, a4, QQ);
  else if (SAME2(0,a6) && SAME3(a2,a4,a5) && SAME2(a1,a3) && (a2>0) && (a1>0))
    *result = TVIL_H0xyxxy (a2, a1, QQ);

  else if (SAME3(0,a1,a2) && SAME2(a3,a5) && SAME2(a4,a6) && (a3>0) && (a4>0))
    *result = TVIL_H00yxyx (a4, a3, QQ);
  else if (SAME3(0,a1,a2) && SAME2(a3,a6) && SAME2(a4,a5) && (a3>0) && (a4>0))
    *result = TVIL_H00yxyx (a4, a3, QQ);
  else if (SAME3(0,a1,a3) && SAME2(a2,a5) && SAME2(a4,a6) && (a2>0) && (a4>0))
    *result = TVIL_H00yxyx (a4, a2, QQ);
  else if (SAME3(0,a1,a3) && SAME2(a2,a4) && SAME2(a5,a6) && (a2>0) && (a5>0))
    *result = TVIL_H00yxyx (a5, a2, QQ);
  else if (SAME3(0,a1,a4) && SAME2(a3,a6) && SAME2(a2,a5) && (a3>0) && (a2>0))
    *result = TVIL_H00yxyx (a2, a3, QQ);
  else if (SAME3(0,a1,a4) && SAME2(a5,a6) && SAME2(a2,a3) && (a5>0) && (a2>0))
    *result = TVIL_H00yxyx (a2, a5, QQ);
  else if (SAME3(0,a1,a6) && SAME2(a4,a5) && SAME2(a2,a3) && (a4>0) && (a2>0))
    *result = TVIL_H00yxyx (a2, a4, QQ);
  else if (SAME3(0,a1,a6) && SAME2(a4,a2) && SAME2(a5,a3) && (a4>0) && (a5>0))
    *result = TVIL_H00yxyx (a5, a4, QQ);
  else if (SAME3(0,a2,a3) && SAME2(a1,a6) && SAME2(a4,a5) && (a1>0) && (a4>0))
    *result = TVIL_H00yxyx (a4, a1, QQ);
  else if (SAME3(0,a2,a3) && SAME2(a1,a4) && SAME2(a6,a5) && (a1>0) && (a6>0))
    *result = TVIL_H00yxyx (a6, a1, QQ);
  else if (SAME3(0,a2,a4) && SAME2(a5,a6) && SAME2(a1,a3) && (a5>0) && (a1>0))
    *result = TVIL_H00yxyx (a1, a5, QQ);
  else if (SAME3(0,a2,a4) && SAME2(a5,a3) && SAME2(a1,a6) && (a5>0) && (a1>0))
    *result = TVIL_H00yxyx (a1, a5, QQ);
  else if (SAME3(0,a2,a5) && SAME2(a4,a6) && SAME2(a1,a3) && (a4>0) && (a1>0))
    *result = TVIL_H00yxyx (a1, a4, QQ);
  else if (SAME3(0,a2,a5) && SAME2(a4,a1) && SAME2(a6,a3) && (a4>0) && (a6>0))
    *result = TVIL_H00yxyx (a6, a4, QQ);
  else if (SAME3(0,a3,a5) && SAME2(a4,a6) && SAME2(a1,a2) && (a4>0) && (a1>0))
    *result = TVIL_H00yxyx (a1, a4, QQ);
  else if (SAME3(0,a3,a5) && SAME2(a1,a6) && SAME2(a4,a2) && (a1>0) && (a4>0))
    *result = TVIL_H00yxyx (a4, a1, QQ);
  else if (SAME3(0,a3,a6) && SAME2(a4,a5) && SAME2(a1,a2) && (a4>0) && (a1>0))
    *result = TVIL_H00yxyx (a1, a4, QQ);
  else if (SAME3(0,a3,a6) && SAME2(a2,a5) && SAME2(a1,a4) && (a2>0) && (a1>0))
    *result = TVIL_H00yxyx (a1, a2, QQ);
  else if (SAME3(0,a4,a5) && SAME2(a2,a3) && SAME2(a1,a6) && (a2>0) && (a1>0))
    *result = TVIL_H00yxyx (a1, a2, QQ);
  else if (SAME3(0,a4,a5) && SAME2(a2,a1) && SAME2(a3,a6) && (a2>0) && (a3>0))
    *result = TVIL_H00yxyx (a3, a2, QQ);
  else if (SAME3(0,a4,a6) && SAME2(a1,a3) && SAME2(a2,a5) && (a1>0) && (a2>0))
    *result = TVIL_H00yxyx (a2, a1, QQ);
  else if (SAME3(0,a4,a6) && SAME2(a1,a2) && SAME2(a3,a5) && (a1>0) && (a3>0))
    *result = TVIL_H00yxyx (a3, a1, QQ);
  else if (SAME3(0,a5,a6) && SAME2(a2,a3) && SAME2(a1,a4) && (a2>0) && (a1>0))
    *result = TVIL_H00yxyx (a1, a2, QQ);
  else if (SAME3(0,a5,a6) && SAME2(a1,a3) && SAME2(a2,a4) && (a1>0) && (a2>0))
    *result = TVIL_H00yxyx (a2, a1, QQ);

  else if (SAME3(0,a1,a2) && SAME2(a3,a4) && SAME2(a5,a6) && (a5>0) && (a3>0))
    *result = TVIL_H00yyxx (a5, a3, QQ);
  else if (SAME3(0,a1,a3) && SAME2(a2,a6) && SAME2(a4,a5) && (a2>0) && (a4>0))
    *result = TVIL_H00yyxx (a4, a2, QQ);
  else if (SAME3(0,a1,a4) && SAME2(a2,a6) && SAME2(a3,a5) && (a2>0) && (a3>0))
    *result = TVIL_H00yyxx (a3, a2, QQ);
  else if (SAME3(0,a1,a6) && SAME2(a3,a4) && SAME2(a2,a5) && (a3>0) && (a2>0))
    *result = TVIL_H00yyxx (a2, a3, QQ);
  else if (SAME3(0,a2,a3) && SAME2(a1,a5) && SAME2(a4,a6) && (a1>0) && (a4>0))
    *result = TVIL_H00yyxx (a4, a1, QQ);
  else if (SAME3(0,a4,a6) && SAME2(a1,a5) && SAME2(a2,a3) && (a1>0) && (a2>0))
    *result = TVIL_H00yyxx (a2, a1, QQ);
  else if (SAME3(0,a2,a4) && SAME2(a1,a5) && SAME2(a3,a6) && (a1>0) && (a3>0))
    *result = TVIL_H00yyxx (a3, a1, QQ);
  else if (SAME3(0,a3,a6) && SAME2(a1,a5) && SAME2(a2,a4) && (a1>0) && (a2>0))
    *result = TVIL_H00yyxx (a2, a1, QQ);
  else if (SAME3(0,a2,a5) && SAME2(a3,a4) && SAME2(a1,a6) && (a3>0) && (a1>0))
    *result = TVIL_H00yyxx (a1, a3, QQ);
  else if (SAME3(0,a4,a5) && SAME2(a2,a6) && SAME2(a1,a3) && (a2>0) && (a1>0))
    *result = TVIL_H00yyxx (a1, a2, QQ);
  else if (SAME3(0,a3,a5) && SAME2(a2,a6) && SAME2(a1,a4) && (a2>0) && (a1>0))
    *result = TVIL_H00yyxx (a1, a2, QQ);
  else if (SAME3(0,a5,a6) && SAME2(a3,a4) && SAME2(a1,a2) && (a3>0) && (a1>0))
    *result = TVIL_H00yyxx (a1, a3, QQ);

  else if (SAME4(0,a1,a2,a3) && SAME2(a5,a6) && (a5>0) && (a4>0))
    *result = TVIL_H000yxx (a5, a4, QQ);
  else if (SAME4(0,a1,a2,a3) && SAME2(a4,a5) && (a4>0) && (a6>0))
    *result = TVIL_H000yxx (a4, a6, QQ);
  else if (SAME4(0,a1,a4,a6) && SAME2(a3,a5) && (a3>0) && (a2>0))
    *result = TVIL_H000yxx (a3, a2, QQ);
  else if (SAME4(0,a1,a4,a6) && SAME2(a2,a5) && (a2>0) && (a3>0))
    *result = TVIL_H000yxx (a2, a3, QQ);
  else if (SAME4(0,a1,a2,a3) && SAME2(a4,a6) && (a4>0) && (a5>0))
    *result = TVIL_H000yxx (a4, a5, QQ);
  else if (SAME4(0,a1,a4,a6) && SAME2(a2,a3) && (a2>0) && (a5>0))
    *result = TVIL_H000yxx (a2, a5, QQ);
  else if (SAME4(0,a2,a4,a5) && SAME2(a3,a6) && (a3>0) && (a1>0))
    *result = TVIL_H000yxx (a3, a1, QQ);
  else if (SAME4(0,a3,a5,a6) && SAME2(a2,a4) && (a2>0) && (a1>0))
    *result = TVIL_H000yxx (a2, a1, QQ);
  else if (SAME4(0,a2,a4,a5) && SAME2(a1,a6) && (a1>0) && (a3>0))
    *result = TVIL_H000yxx (a1, a3, QQ);
  else if (SAME4(0,a2,a4,a5) && SAME2(a1,a3) && (a1>0) && (a6>0))
    *result = TVIL_H000yxx (a1, a6, QQ);
  else if (SAME4(0,a3,a5,a6) && SAME2(a1,a4) && (a1>0) && (a2>0))
    *result = TVIL_H000yxx (a1, a2, QQ);
  else if (SAME4(0,a3,a5,a6) && SAME2(a1,a2) && (a1>0) && (a4>0))
    *result = TVIL_H000yxx (a1, a4, QQ);

  else if (SAME3(0,a1,a2) && SAME3(a4,a5,a6) && (a4>0) && (a3>0))
    *result = TVIL_H00yxxx (a4, a3, QQ);
  else if (SAME3(0,a1,a4) && SAME3(a2,a3,a5) && (a2>0) && (a6>0))
    *result = TVIL_H00yxxx (a2, a6, QQ);
  else if (SAME3(0,a1,a6) && SAME3(a2,a3,a5) && (a2>0) && (a4>0))
    *result = TVIL_H00yxxx (a2, a4, QQ);
  else if (SAME3(0,a1,a3) && SAME3(a4,a5,a6) && (a4>0) && (a2>0))
    *result = TVIL_H00yxxx (a4, a2, QQ);
  else if (SAME3(0,a2,a4) && SAME3(a1,a3,a6) && (a1>0) && (a5>0))
    *result = TVIL_H00yxxx (a1, a5, QQ);
  else if (SAME3(0,a2,a5) && SAME3(a1,a3,a6) && (a1>0) && (a4>0))
    *result = TVIL_H00yxxx (a1, a4, QQ);
  else if (SAME3(0,a3,a5) && SAME3(a1,a2,a4) && (a1>0) && (a6>0))
    *result = TVIL_H00yxxx (a1, a6, QQ);
  else if (SAME3(0,a3,a6) && SAME3(a1,a2,a4) && (a1>0) && (a5>0))
    *result = TVIL_H00yxxx (a1, a5, QQ);
  else if (SAME3(0,a5,a6) && SAME3(a1,a2,a4) && (a1>0) && (a3>0))
    *result = TVIL_H00yxxx (a1, a3, QQ);
  else if (SAME3(0,a4,a5) && SAME3(a1,a3,a6) && (a1>0) && (a2>0))
    *result = TVIL_H00yxxx (a1, a2, QQ);
  else if (SAME3(0,a2,a3) && SAME3(a4,a5,a6) && (a4>0) && (a1>0))
    *result = TVIL_H00yxxx (a4, a1, QQ);
  else if (SAME3(0,a4,a6) && SAME3(a2,a3,a5) && (a2>0) && (a1>0))
    *result = TVIL_H00yxxx (a2, a1, QQ);

  else if (SAME3(0,a1,a5) && SAME3(a2,a3,a4) && (a2>0) && (a6>0))
    *result = TVIL_H0xxx0y (a2, a6, QQ);
  else if (SAME3(0,a1,a5) && SAME3(a2,a3,a6) && (a2>0) && (a4>0))
    *result = TVIL_H0xxx0y (a2, a4, QQ);
  else if (SAME3(0,a1,a5) && SAME3(a2,a4,a6) && (a2>0) && (a3>0))
    *result = TVIL_H0xxx0y (a2, a3, QQ);
  else if (SAME3(0,a1,a5) && SAME3(a3,a4,a6) && (a3>0) && (a2>0))
    *result = TVIL_H0xxx0y (a3, a2, QQ);
  else if (SAME3(0,a2,a6) && SAME3(a1,a3,a4) && (a1>0) && (a5>0))
    *result = TVIL_H0xxx0y (a1, a5, QQ);
  else if (SAME3(0,a2,a6) && SAME3(a1,a3,a5) && (a1>0) && (a4>0))
    *result = TVIL_H0xxx0y (a1, a4, QQ);
  else if (SAME3(0,a2,a6) && SAME3(a1,a4,a5) && (a1>0) && (a3>0))
    *result = TVIL_H0xxx0y (a1, a3, QQ);
  else if (SAME3(0,a2,a6) && SAME3(a3,a4,a5) && (a3>0) && (a1>0))
    *result = TVIL_H0xxx0y (a3, a1, QQ);
  else if (SAME3(0,a3,a4) && SAME3(a1,a2,a5) && (a1>0) && (a6>0))
    *result = TVIL_H0xxx0y (a1, a6, QQ);
  else if (SAME3(0,a3,a4) && SAME3(a1,a2,a6) && (a1>0) && (a5>0))
    *result = TVIL_H0xxx0y (a1, a5, QQ);
  else if (SAME3(0,a3,a4) && SAME3(a1,a5,a6) && (a1>0) && (a2>0))
    *result = TVIL_H0xxx0y (a1, a2, QQ);
  else if (SAME3(0,a3,a4) && SAME3(a2,a5,a6) && (a2>0) && (a1>0))
    *result = TVIL_H0xxx0y (a2, a1, QQ);

  else if (SAME2(0,a1) && SAME2(a2,a3)  && SAME3(a4,a5,a6) && (a2>0) && (a4>0))
    *result = TVIL_H0xxyyy (a2, a4, QQ);
  else if (SAME2(0,a1) && SAME2(a4,a6)  && SAME3(a2,a3,a5) && (a4>0) && (a2>0))
    *result = TVIL_H0xxyyy (a4, a2, QQ);
  else if (SAME2(0,a2) && SAME2(a1,a3)  && SAME3(a4,a5,a6) && (a1>0) && (a4>0))
    *result = TVIL_H0xxyyy (a1, a4, QQ);
  else if (SAME2(0,a3) && SAME2(a1,a2)  && SAME3(a4,a5,a6) && (a1>0) && (a4>0))
    *result = TVIL_H0xxyyy (a1, a4, QQ);
  else if (SAME2(0,a4) && SAME2(a1,a6)  && SAME3(a2,a3,a5) && (a1>0) && (a2>0))
    *result = TVIL_H0xxyyy (a1, a2, QQ);
  else if (SAME2(0,a6) && SAME2(a1,a4)  && SAME3(a2,a3,a5) && (a1>0) && (a2>0))
    *result = TVIL_H0xxyyy (a1, a2, QQ);
  else if (SAME2(0,a2) && SAME2(a4,a5)  && SAME3(a1,a3,a6) && (a4>0) && (a1>0))
    *result = TVIL_H0xxyyy (a4, a1, QQ);
  else if (SAME2(0,a4) && SAME2(a2,a5)  && SAME3(a1,a3,a6) && (a2>0) && (a1>0))
    *result = TVIL_H0xxyyy (a2, a1, QQ);
  else if (SAME2(0,a5) && SAME2(a2,a4)  && SAME3(a1,a3,a6) && (a2>0) && (a1>0))
    *result = TVIL_H0xxyyy (a2, a1, QQ);
  else if (SAME2(0,a3) && SAME2(a5,a6)  && SAME3(a1,a2,a4) && (a5>0) && (a1>0))
    *result = TVIL_H0xxyyy (a5, a1, QQ);
  else if (SAME2(0,a5) && SAME2(a3,a6)  && SAME3(a1,a2,a4) && (a3>0) && (a1>0))
    *result = TVIL_H0xxyyy (a3, a1, QQ);
  else if (SAME2(0,a6) && SAME2(a3,a5)  && SAME3(a1,a2,a4) && (a3>0) && (a1>0))
    *result = TVIL_H0xxyyy (a3, a1, QQ);

  else if (SAME2(0,a1) && SAME4(a2,a3,a4,a5) && (a2>0) && (a6>0))
    *result = TVIL_H0xxxxy (a2, a6, QQ);
  else if (SAME2(0,a1) && SAME4(a2,a3,a5,a6) && (a2>0) && (a4>0))
    *result = TVIL_H0xxxxy (a2, a4, QQ);
  else if (SAME2(0,a1) && SAME4(a2,a4,a5,a6) && (a2>0) && (a3>0))
    *result = TVIL_H0xxxxy (a2, a3, QQ);
  else if (SAME2(0,a1) && SAME4(a3,a4,a5,a6) && (a3>0) && (a2>0))
    *result = TVIL_H0xxxxy (a3, a2, QQ);
  else if (SAME2(0,a2) && SAME4(a1,a3,a4,a6) && (a1>0) && (a5>0))
    *result = TVIL_H0xxxxy (a1, a5, QQ);
  else if (SAME2(0,a2) && SAME4(a1,a3,a5,a6) && (a1>0) && (a4>0))
    *result = TVIL_H0xxxxy (a1, a4, QQ);
  else if (SAME2(0,a2) && SAME4(a1,a4,a5,a6) && (a1>0) && (a3>0))
    *result = TVIL_H0xxxxy (a1, a3, QQ);
  else if (SAME2(0,a3) && SAME4(a1,a2,a4,a5) && (a1>0) && (a6>0))
    *result = TVIL_H0xxxxy (a1, a6, QQ);
  else if (SAME2(0,a3) && SAME4(a1,a2,a4,a6) && (a1>0) && (a5>0))
    *result = TVIL_H0xxxxy (a1, a5, QQ);
  else if (SAME2(0,a4) && SAME4(a1,a2,a3,a5) && (a1>0) && (a6>0))
    *result = TVIL_H0xxxxy (a1, a6, QQ);
  else if (SAME2(0,a4) && SAME4(a1,a2,a3,a6) && (a1>0) && (a5>0))
    *result = TVIL_H0xxxxy (a1, a5, QQ);
  else if (SAME2(0,a5) && SAME4(a1,a2,a3,a4) && (a1>0) && (a6>0))
    *result = TVIL_H0xxxxy (a1, a6, QQ);
  else if (SAME2(0,a6) && SAME4(a1,a2,a3,a4) && (a1>0) && (a5>0))
    *result = TVIL_H0xxxxy (a1, a5, QQ);
  else if (SAME2(0,a5) && SAME4(a1,a2,a3,a6) && (a1>0) && (a4>0))
    *result = TVIL_H0xxxxy (a1, a4, QQ);
  else if (SAME2(0,a6) && SAME4(a1,a2,a3,a5) && (a1>0) && (a4>0))
    *result = TVIL_H0xxxxy (a1, a4, QQ);
  else if (SAME2(0,a5) && SAME4(a1,a2,a4,a6) && (a1>0) && (a3>0))
    *result = TVIL_H0xxxxy (a1, a3, QQ);
  else if (SAME2(0,a6) && SAME4(a1,a2,a4,a5) && (a1>0) && (a3>0))
    *result = TVIL_H0xxxxy (a1, a3, QQ);
  else if (SAME2(0,a3) && SAME4(a1,a4,a5,a6) && (a1>0) && (a2>0))
    *result = TVIL_H0xxxxy (a1, a2, QQ);
  else if (SAME2(0,a4) && SAME4(a1,a3,a5,a6) && (a1>0) && (a2>0))
    *result = TVIL_H0xxxxy (a1, a2, QQ);
  else if (SAME2(0,a5) && SAME4(a1,a3,a4,a6) && (a1>0) && (a2>0))
    *result = TVIL_H0xxxxy (a1, a2, QQ);
  else if (SAME2(0,a2) && SAME4(a3,a4,a5,a6) && (a3>0) && (a1>0))
    *result = TVIL_H0xxxxy (a3, a1, QQ);
  else if (SAME2(0,a3) && SAME4(a2,a4,a5,a6) && (a2>0) && (a1>0))
    *result = TVIL_H0xxxxy (a2, a1, QQ);
  else if (SAME2(0,a4) && SAME4(a2,a3,a5,a6) && (a2>0) && (a1>0))
    *result = TVIL_H0xxxxy (a2, a1, QQ);
  else if (SAME2(0,a6) && SAME4(a2,a3,a4,a5) && (a2>0) && (a1>0))
    *result = TVIL_H0xxxxy (a2, a1, QQ);

  else if (SAME2(0,a1) && SAME3(a2,a3,a4) && SAME2(a5,a6) && (a2>0) && (a5>0))
    *result = TVIL_H0xxxyy (a2, a6, QQ);
  else if (SAME2(0,a1) && SAME3(a2,a3,a6) && SAME2(a4,a5) && (a2>0) && (a4>0))
    *result = TVIL_H0xxxyy (a2, a4, QQ);
  else if (SAME2(0,a1) && SAME3(a2,a4,a6) && SAME2(a3,a5) && (a2>0) && (a3>0))
    *result = TVIL_H0xxxyy (a2, a3, QQ);
  else if (SAME2(0,a1) && SAME3(a3,a4,a6) && SAME2(a2,a5) && (a3>0) && (a2>0))
    *result = TVIL_H0xxxyy (a3, a2, QQ);
  else if (SAME2(0,a2) && SAME3(a1,a3,a4) && SAME2(a5,a6) && (a1>0) && (a5>0))
    *result = TVIL_H0xxxyy (a1, a5, QQ);
  else if (SAME2(0,a2) && SAME3(a1,a3,a5) && SAME2(a4,a6) && (a1>0) && (a4>0))
    *result = TVIL_H0xxxyy (a1, a4, QQ);
  else if (SAME2(0,a2) && SAME3(a1,a4,a5) && SAME2(a3,a6) && (a1>0) && (a3>0))
    *result = TVIL_H0xxxyy (a1, a3, QQ);
  else if (SAME2(0,a3) && SAME3(a1,a2,a5) && SAME2(a4,a6) && (a1>0) && (a4>0))
    *result = TVIL_H0xxxyy (a1, a4, QQ);
  else if (SAME2(0,a3) && SAME3(a1,a2,a6) && SAME2(a4,a5) && (a1>0) && (a4>0))
    *result = TVIL_H0xxxyy (a1, a4, QQ);
  else if (SAME2(0,a4) && SAME3(a1,a2,a5) && SAME2(a3,a6) && (a1>0) && (a3>0))
    *result = TVIL_H0xxxyy (a1, a3, QQ);
  else if (SAME2(0,a4) && SAME3(a1,a2,a6) && SAME2(a3,a5) && (a1>0) && (a3>0))
    *result = TVIL_H0xxxyy (a1, a3, QQ);
  else if (SAME2(0,a3) && SAME3(a1,a5,a6) && SAME2(a2,a4) && (a1>0) && (a2>0))
    *result = TVIL_H0xxxyy (a1, a2, QQ);
  else if (SAME2(0,a6) && SAME3(a1,a3,a4) && SAME2(a2,a5) && (a1>0) && (a2>0))
    *result = TVIL_H0xxxyy (a1, a2, QQ);
  else if (SAME2(0,a6) && SAME3(a1,a3,a5) && SAME2(a2,a4) && (a1>0) && (a2>0))
    *result = TVIL_H0xxxyy (a1, a2, QQ);
  else if (SAME2(0,a4) && SAME3(a1,a5,a6) && SAME2(a2,a3) && (a1>0) && (a2>0))
    *result = TVIL_H0xxxyy (a1, a2, QQ);
  else if (SAME2(0,a6) && SAME3(a1,a4,a5) && SAME2(a2,a3) && (a1>0) && (a2>0))
    *result = TVIL_H0xxxyy (a1, a2, QQ);
  else if (SAME2(0,a2) && SAME3(a3,a4,a5) && SAME2(a1,a6) && (a3>0) && (a1>0))
    *result = TVIL_H0xxxyy (a3, a1, QQ);
  else if (SAME2(0,a3) && SAME3(a2,a5,a6) && SAME2(a1,a4) && (a2>0) && (a1>0))
    *result = TVIL_H0xxxyy (a2, a1, QQ);
  else if (SAME2(0,a5) && SAME3(a2,a3,a4) && SAME2(a1,a6) && (a2>0) && (a1>0))
    *result = TVIL_H0xxxyy (a2, a1, QQ);
  else if (SAME2(0,a5) && SAME3(a2,a3,a6) && SAME2(a1,a4) && (a2>0) && (a1>0))
    *result = TVIL_H0xxxyy (a2, a1, QQ);
  else if (SAME2(0,a4) && SAME3(a2,a5,a6) && SAME2(a1,a3) && (a2>0) && (a1>0))
    *result = TVIL_H0xxxyy (a2, a1, QQ);
  else if (SAME2(0,a5) && SAME3(a2,a4,a6) && SAME2(a1,a3) && (a2>0) && (a1>0))
    *result = TVIL_H0xxxyy (a2, a1, QQ);
  else if (SAME2(0,a5) && SAME3(a3,a4,a6) && SAME2(a1,a2) && (a3>0) && (a1>0))
    *result = TVIL_H0xxxyy (a3, a1, QQ);
  else if (SAME2(0,a6) && SAME3(a3,a4,a5) && SAME2(a1,a2) && (a3>0) && (a1>0))
    *result = TVIL_H0xxxyy (a3, a1, QQ);

  else if (SAME2(0,a1) && SAME3(a2,a5,a6) && SAME2(a3,a4) && (a2>0) && (a3>0))
    *result = TVIL_H0xyyxx (a2, a3, QQ);
  else if (SAME2(0,a1) && SAME3(a3,a4,a5) && SAME2(a2,a6) && (a3>0) && (a2>0))
    *result = TVIL_H0xyyxx (a3, a2, QQ);
  else if (SAME2(0,a2) && SAME3(a1,a5,a6) && SAME2(a3,a4) && (a1>0) && (a3>0))
    *result = TVIL_H0xyyxx (a1, a3, QQ);
  else if (SAME2(0,a5) && SAME3(a1,a2,a6) && SAME2(a3,a4) && (a1>0) && (a3>0))
    *result = TVIL_H0xyyxx (a1, a3, QQ);
  else if (SAME2(0,a6) && SAME3(a1,a2,a5) && SAME2(a3,a4) && (a1>0) && (a3>0))
    *result = TVIL_H0xyyxx (a1, a3, QQ);
  else if (SAME2(0,a3) && SAME3(a1,a4,a5) && SAME2(a2,a6) && (a1>0) && (a2>0))
    *result = TVIL_H0xyyxx (a1, a2, QQ);
  else if (SAME2(0,a4) && SAME3(a1,a3,a5) && SAME2(a2,a6) && (a1>0) && (a2>0))
    *result = TVIL_H0xyyxx (a1, a2, QQ);
  else if (SAME2(0,a5) && SAME3(a1,a3,a4) && SAME2(a2,a6) && (a1>0) && (a2>0))
    *result = TVIL_H0xyyxx (a1, a2, QQ);
  else if (SAME2(0,a2) && SAME3(a3,a4,a6) && SAME2(a1,a5) && (a3>0) && (a1>0))
    *result = TVIL_H0xyyxx (a3, a1, QQ);
  else if (SAME2(0,a3) && SAME3(a2,a4,a6) && SAME2(a1,a5) && (a2>0) && (a1>0))
    *result = TVIL_H0xyyxx (a2, a1, QQ);
  else if (SAME2(0,a4) && SAME3(a2,a3,a6) && SAME2(a1,a5) && (a2>0) && (a1>0))
    *result = TVIL_H0xyyxx (a2, a1, QQ);
  else if (SAME2(0,a6) && SAME3(a2,a3,a4) && SAME2(a1,a5) && (a2>0) && (a1>0))
    *result = TVIL_H0xyyxx (a2, a1, QQ);

  else if (SAME3(0,a1,a5) && SAME2(a2,a6)  && SAME2(a3,a4) && (a2>0) && (a3>0))
    *result = TVIL_H0xyy0x (a2, a3, QQ);
  else if (SAME3(0,a2,a6) && SAME2(a1,a5)  && SAME2(a3,a4) && (a1>0) && (a3>0))
    *result = TVIL_H0xyy0x (a1, a3, QQ);
  else if (SAME3(0,a3,a4) && SAME2(a1,a5)  && SAME2(a2,a6) && (a1>0) && (a2>0))
    *result = TVIL_H0xyy0x (a1, a2, QQ);

  else
    success = 0;

  return success;
}

/* ****************************************************************** */
/* Sets initial value for the H function at a point near t = 0. */

int TVIL_InitialH (TVIL_HTYPE  *foo, 
		   TVIL_REAL    qq,
		   TVIL_COMPLEX tinit)
{
  TVIL_REAL u = (foo->arg[0])/_a;
  TVIL_REAL v = (foo->arg[1])/_a;
  TVIL_REAL w = (foo->arg[2])/_a;
  TVIL_REAL x = (foo->arg[3])/_a;
  TVIL_REAL y = (foo->arg[4])/_a;
  TVIL_REAL z = (foo->arg[5])/_a;

  TVIL_REAL Hexp[9];
  TVIL_REAL u2, u3, u4, u5, u6, u7, u8;
  TVIL_REAL v2, v3, v4, v5, v6, v7, v8;
  TVIL_REAL w2, w3, w4, w5, w6, w7, w8;
  TVIL_REAL x2, x3, x4, x5, x6, x7, x8;
  TVIL_REAL y2, y3, y4, y5, y6, y7, y8;
  TVIL_REAL z2, z3, z4, z5, z6, z7, z8;

  TVIL_REAL lnbara;
  TVIL_COMPLEX ttothei, result;
  int i, nTerms = 9;

  lnbara = TVIL_LOG (_a/qq);

  u2 = u*u;
  u3 = u2*u;
  u4 = u2*u2;
  u5 = u2*u3;
  u6 = u3*u3;
  u7 = u4*u3;
  u8 = u4*u4;

  v2 = v*v;
  v3 = v2*v;
  v4 = v2*v2;
  v5 = v2*v3;
  v6 = v3*v3;
  v7 = v4*v3;
  v8 = v4*v4;

  w2 = w*w;
  w3 = w2*w;
  w4 = w2*w2;
  w5 = w2*w3;
  w6 = w3*w3;
  w7 = w4*w3;
  w8 = w4*w4;

  x2 = x*x;
  x3 = x2*x;
  x4 = x2*x2;
  x5 = x2*x3;
  x6 = x3*x3;
  x7 = x4*x3;
  x8 = x4*x4;

  y2 = y*y;
  y3 = y2*y;
  y4 = y2*y2;
  y5 = y2*y3;
  y6 = y3*y3;
  y7 = y4*y3;
  y8 = y4*y4;

  z2 = z*z;
  z3 = z2*z;
  z4 = z2*z2;
  z5 = z2*z3;
  z6 = z3*z3;
  z7 = z4*z3;
  z8 = z4*z4;

  Hexp[0] = -10.03527847976879 - 7.212341418957566*lnbara;
 
  Hexp[1] = 7.212341418957566 - 1.2020569031595942*u - 1.2020569031595942*v - 
     1.2020569031595942*w - 1.2020569031595942*x - 1.2020569031595942*y - 
     1.2020569031595942*z;
 
  Hexp[2] = 3.606170709478783 - 1.2020569031595942*u + 0.2638365705403415*u2 - 
     1.2020569031595942*v + 0.14013624956353604*u*v + 
     0.2638365705403415*v2 - 1.2020569031595942*w + 
     0.14013624956353604*u*w + 0.14013624956353604*v*w + 
     0.2638365705403415*w2 - 1.2020569031595942*x + 
     0.14013624956353604*u*x + 0.14013624956353604*v*x + 
     0.11383876382476714*w*x + 0.2638365705403415*x2 - 
     1.2020569031595942*y + 0.11383876382476714*u*y + 
     0.14013624956353604*v*y + 0.14013624956353604*w*y + 
     0.14013624956353604*x*y + 0.2638365705403415*y2 - 
     1.2020569031595942*z + 0.14013624956353604*u*z + 
     0.11383876382476714*v*z + 0.14013624956353604*w*z + 
     0.14013624956353604*x*z + 0.14013624956353604*y*z + 
     0.2638365705403415*z2;
 
  Hexp[3] = 2.4041138063191885 - 1.2020569031595942*u + 0.527673141080683*u2 - 
     0.10795675579460708*u3 - 1.2020569031595942*v + 
     0.2802724991270721*u*v - 0.04287703318427488*u2*v + 
     0.527673141080683*v2 - 0.04287703318427488*u*v2 - 
     0.10795675579460708*v3 - 1.2020569031595942*w + 
     0.2802724991270721*u*w - 0.04287703318427488*u2*w + 
     0.2802724991270721*v*w - 0.0262974857387689*u*v*w - 
     0.04287703318427488*v2*w + 0.527673141080683*w2 - 
     0.04287703318427488*u*w2 - 0.04287703318427488*v*w2 - 
     0.10795675579460708*w3 - 1.2020569031595942*x + 
     0.2802724991270721*u*x - 0.04287703318427488*u2*x + 
     0.2802724991270721*v*x + 0.12075917025180338*u*v*x + 
     0.04340568960536035*lnbara*u*v*x - 0.19685380218203927*v2*x - 
     0.04340568960536035*lnbara*v2*x + 0.22767752764953428*w*x - 
     0.1786014099503857*u*w*x - 0.04340568960536035*lnbara*u*w*x - 
     0.02462464095262132*v*w*x + 0.12168202803800214*w2*x + 
     0.04340568960536035*lnbara*w2*x + 0.527673141080683*x2 - 
     0.04287703318427488*u*x2 - 0.19685380218203927*v*x2 - 
     0.04340568960536035*lnbara*v*x2 + 0.12168202803800214*w*x2 + 
     0.04340568960536035*lnbara*w*x2 - 0.10795675579460708*x3 - 
     1.2020569031595942*y + 0.22767752764953428*u*y - 
     0.03229474095976225*u2*y + 0.2802724991270721*v*y - 
     0.02462464095262132*u*v*y - 0.04287703318427488*v2*y + 
     0.2802724991270721*w*y - 0.02462464095262132*u*w*y - 
     0.03321759874596101*v*w*y - 0.04287703318427488*w2*y + 
     0.2802724991270721*x*y - 0.02462464095262132*u*x*y + 
     0.4356328212545243*v*x*y + 0.13021706881608106*lnbara*v*x*y - 
     0.4865549479459145*w*x*y - 0.13021706881608106*lnbara*w*x*y - 
     0.04287703318427488*x2*y + 0.527673141080683*y2 - 
     0.03229474095976225*u*y2 - 0.04287703318427488*v*y2 - 
     0.04287703318427488*w*y2 - 0.04287703318427488*x*y2 - 
     0.10795675579460708*y3 - 1.2020569031595942*z + 
     0.2802724991270721*u*z - 0.04287703318427488*u2*z + 
     0.22767752764953428*v*z - 0.1786014099503857*u*v*z - 
     0.04340568960536035*lnbara*u*v*z + 0.12168202803800214*v2*z + 
     0.04340568960536035*lnbara*v2*z + 0.2802724991270721*w*z + 
     0.12075917025180338*u*w*z + 0.04340568960536035*lnbara*u*w*z - 
     0.02462464095262132*v*w*z - 0.19685380218203927*w2*z - 
     0.04340568960536035*lnbara*w2*z + 0.2802724991270721*x*z - 
     0.0262974857387689*u*x*z - 0.02462464095262132*v*x*z - 
     0.02462464095262132*w*x*z - 0.04287703318427488*x2*z + 
     0.2802724991270721*y*z - 0.02462464095262132*u*y*z - 
     0.4865549479459145*v*y*z - 0.13021706881608106*lnbara*v*y*z + 
     0.4356328212545243*w*y*z + 0.13021706881608106*lnbara*w*y*z - 
     0.03321759874596101*x*y*z - 0.04287703318427488*y2*z + 
     0.527673141080683*z2 - 0.04287703318427488*u*z2 + 
     0.12168202803800214*v*z2 + 0.04340568960536035*lnbara*v*z2 - 
     0.19685380218203927*w*z2 - 0.04340568960536035*lnbara*w*z2 - 
     0.04287703318427488*x*z2 - 0.04287703318427488*y*z2 - 
     0.10795675579460708*z3;
 
  Hexp[4] = 1.8030853547393915 - 1.2020569031595942*u + 
     0.7915097116210246*u2 - 0.32387026738382124*u3 + 
     0.05780009836815717*u4 - 1.2020569031595942*v + 
     0.4204087486906081*u*v - 0.12863109955282465*u2*v + 
     0.019595873400524336*u3*v + 0.7915097116210246*v2 - 
     0.12863109955282465*u*v2 + 0.014860391371313229*u2*v2 - 
     0.32387026738382124*v3 + 0.019595873400524336*u*v3 + 
     0.05780009836815717*v4 - 1.2020569031595942*w + 
     0.4204087486906081*u*w - 0.12863109955282465*u2*w + 
     0.019595873400524336*u3*w + 0.4204087486906081*v*w - 
     0.0788924572163067*u*v*w + 0.009467502496847111*u2*v*w - 
     0.12863109955282465*v2*w + 0.009467502496847111*u*v2*w + 
     0.019595873400524336*v3*w + 0.7915097116210246*w2 - 
     0.12863109955282465*u*w2 + 0.014860391371313229*u2*w2 - 
     0.12863109955282465*v*w2 + 0.009467502496847111*u*v*w2 + 
     0.014860391371313229*v2*w2 - 0.32387026738382124*w3 + 
     0.019595873400524336*u*w3 + 0.019595873400524336*v*w3 + 
     0.05780009836815717*w4 - 1.2020569031595942*x + 
     0.4204087486906081*u*x - 0.12863109955282465*u2*x + 
     0.019595873400524336*u3*x + 0.4204087486906081*v*x + 
     0.3297232435513899*u*v*x + 0.13021706881608106*lnbara*u*v*x + 
     0.09849335296892119*u2*v*x + 0.022352066577107733*lnbara*u2*v*x - 
     0.5580071393420976*v2*x - 0.13021706881608106*lnbara*v2*x - 
     0.18693837131393218*u*v2*x - 0.056425067024902395*lnbara*u*v2*x + 
     0.056907625247539464*v3*x + 0.012370155645114489*lnbara*v3*x + 
     0.3415162914743014*w*x - 0.5032499626471368*u*w*x - 
     0.13021706881608106*lnbara*u*w*x - 0.07638626512936658*u2*w*x - 
     0.022352066577107733*lnbara*u2*w*x - 0.07387392285786396*v*w*x + 
     0.007362480745074675*u*v*w*x + 0.5854099721783019*v2*w*x + 
     0.1642900692638757*lnbara*v2*w*x + 0.33249181690998614*w2*x + 
     0.13021706881608106*lnbara*w2*x + 0.2084140365865988*u*w2*x + 
     0.056425067024902395*lnbara*u*w2*x - 0.567682337066967*v*w2*x - 
     0.1642900692638757*lnbara*v*w2*x - 0.02302537153791993*w3*x - 
     0.012370155645114489*lnbara*w3*x + 0.7915097116210246*x2 - 
     0.12863109955282465*u*x2 + 0.014860391371313229*u2*x2 - 
     0.5580071393420976*v*x2 - 0.13021706881608106*lnbara*v*x2 - 
     0.18693837131393218*u*v*x2 - 0.056425067024902395*lnbara*u*v*x2 + 
     0.3204490485619901*v2*x2 + 0.0898488456982695*lnbara*v2*x2 + 
     0.33249181690998614*w*x2 + 0.13021706881608106*lnbara*w*x2 + 
     0.2084140365865988*u*w*x2 + 0.056425067024902395*lnbara*u*w*x2 - 
     0.37639381622218754*v*w*x2 - 0.10851422401340087*lnbara*v*w*x2 + 
     0.0892695937352878*w2*x2 + 0.018665378315131372*lnbara*w2*x2 - 
     0.32387026738382124*x3 + 0.019595873400524336*u*x3 + 
     0.056907625247539464*v*x3 + 0.012370155645114489*lnbara*v*x3 - 
     0.02302537153791993*w*x3 - 0.012370155645114489*lnbara*w*x3 + 
     0.05780009836815717*x4 - 1.2020569031595942*y + 
     0.3415162914743014*u*y - 0.09688422287928675*u2*y + 
     0.014286380309095197*u3*y + 0.4204087486906081*v*y - 
     0.07387392285786396*u*v*y + 0.008548106272223455*u2*v*y - 
     0.12863109955282465*v2*y + 0.009179528839111438*u*v2*y + 
     0.019595873400524336*v3*y + 0.4204087486906081*w*y - 
     0.07387392285786396*u*w*y + 0.008548106272223455*u2*w*y - 
     0.09965279623788302*v*w*y + 0.007362480745074675*u*v*w*y + 
     0.012927559000443166*v2*w*y - 0.12863109955282465*w2*y + 
     0.009179528839111438*u*w2*y + 0.012927559000443166*v*w2*y + 
     0.019595873400524336*w3*y + 0.4204087486906081*x*y - 
     0.07387392285786396*u*x*y + 0.008548106272223455*u2*x*y + 
     1.209235662151512*v*x*y + 0.39065120644824314*lnbara*v*x*y + 
     0.20631857427634712*u*v*x*y + 0.05077906612931307*lnbara*u*v*x*y - 
     0.2756916181666832*v2*x*y - 0.08594186774137386*lnbara*v2*x*y - 
     1.3620020422256827*w*x*y - 0.39065120644824314*lnbara*w*x*y - 
     0.19235861493067452*u*w*x*y - 0.05077906612931307*lnbara*u*w*x*y + 
     0.007362480745074675*v*w*x*y + 0.2937072269357538*w2*x*y + 
     0.08594186774137386*lnbara*w2*x*y - 0.12863109955282465*x2*y + 
     0.009179528839111438*u*x2*y - 0.2756916181666832*v*x2*y - 
     0.08594186774137386*lnbara*v*x2*y + 0.2937072269357538*w*x2*y + 
     0.08594186774137386*lnbara*w*x2*y + 0.019595873400524336*x3*y + 
     0.7915097116210246*y2 - 0.09688422287928675*u*y2 + 
     0.009916328431553665*u2*y2 - 0.12863109955282465*v*y2 + 
     0.008548106272223455*u*v*y2 + 0.014860391371313229*v2*y2 - 
     0.12863109955282465*w*y2 + 0.008548106272223455*u*w*y2 + 
     0.012927559000443166*v*w*y2 + 0.014860391371313229*w2*y2 - 
     0.12863109955282465*x*y2 + 0.008548106272223455*u*x*y2 - 
     0.16375636262563784*v*x*y2 - 0.04883140080603039*lnbara*v*x*y2 + 
     0.18240339396159638*w*x*y2 + 0.04883140080603039*lnbara*w*x*y2 + 
     0.014860391371313229*x2*y2 - 0.32387026738382124*y3 + 
     0.014286380309095197*u*y3 + 0.019595873400524336*v*y3 + 
     0.019595873400524336*w*y3 + 0.019595873400524336*x*y3 + 
     0.05780009836815717*y4 - 1.2020569031595942*z + 
     0.4204087486906081*u*z - 0.12863109955282465*u2*z + 
     0.019595873400524336*u3*z + 0.3415162914743014*v*z - 
     0.5032499626471368*u*v*z - 0.13021706881608106*lnbara*u*v*z - 
     0.07638626512936658*u2*v*z - 0.022352066577107733*lnbara*u2*v*z + 
     0.33249181690998614*v2*z + 0.13021706881608106*lnbara*v2*z + 
     0.2084140365865988*u*v2*z + 0.056425067024902395*lnbara*u*v2*z - 
     0.02302537153791993*v3*z - 0.012370155645114489*lnbara*v3*z + 
     0.4204087486906081*w*z + 0.3297232435513899*u*w*z + 
     0.13021706881608106*lnbara*u*w*z + 0.09849335296892119*u2*w*z + 
     0.022352066577107733*lnbara*u2*w*z - 0.07387392285786396*v*w*z + 
     0.007362480745074675*u*v*w*z - 0.567682337066967*v2*w*z - 
     0.1642900692638757*lnbara*v2*w*z - 0.5580071393420976*w2*z - 
     0.13021706881608106*lnbara*w2*z - 0.18693837131393218*u*w2*z - 
     0.056425067024902395*lnbara*u*w2*z + 0.5854099721783019*v*w2*z + 
     0.1642900692638757*lnbara*v*w2*z + 0.056907625247539464*w3*z + 
     0.012370155645114489*lnbara*w3*z + 0.4204087486906081*x*z - 
     0.0788924572163067*u*x*z + 0.009467502496847111*u2*x*z - 
     0.07387392285786396*v*x*z + 0.007362480745074675*u*v*x*z - 
     0.37639381622218754*v2*x*z - 0.10851422401340087*lnbara*v2*x*z - 
     0.07387392285786396*w*x*z + 0.007362480745074675*u*w*x*z + 
     0.7764813235894199*v*w*x*z + 0.21702844802680174*lnbara*v*w*x*z - 
     0.37639381622218754*w2*x*z - 0.10851422401340087*lnbara*w2*x*z - 
     0.12863109955282465*x2*z + 0.009467502496847111*u*x2*z + 
     0.5854099721783019*v*x2*z + 0.1642900692638757*lnbara*v*x2*z - 
     0.567682337066967*w*x2*z - 0.1642900692638757*lnbara*w*x2*z + 
     0.019595873400524336*x3*z + 0.4204087486906081*y*z - 
     0.07387392285786396*u*y*z + 0.008548106272223455*u2*y*z - 
     1.3620020422256827*v*y*z - 0.39065120644824314*lnbara*v*y*z - 
     0.19235861493067452*u*v*y*z - 0.05077906612931307*lnbara*u*v*y*z + 
     0.2937072269357538*v2*y*z + 0.08594186774137386*lnbara*v2*y*z + 
     1.209235662151512*w*y*z + 0.39065120644824314*lnbara*w*y*z + 
     0.20631857427634712*u*w*y*z + 0.05077906612931307*lnbara*u*w*y*z + 
     0.007362480745074675*v*w*y*z - 0.2756916181666832*w2*y*z - 
     0.08594186774137386*lnbara*w2*y*z - 0.09965279623788302*x*y*z + 
     0.007362480745074675*u*x*y*z + 0.007362480745074675*v*x*y*z + 
     0.007362480745074675*w*x*y*z + 0.012927559000443166*x2*y*z - 
     0.12863109955282465*y2*z + 0.008548106272223455*u*y2*z + 
     0.18240339396159638*v*y2*z + 0.04883140080603039*lnbara*v*y2*z - 
     0.16375636262563784*w*y2*z - 0.04883140080603039*lnbara*w*y2*z + 
     0.012927559000443166*x*y2*z + 0.019595873400524336*y3*z + 
     0.7915097116210246*z2 - 0.12863109955282465*u*z2 + 
     0.014860391371313229*u2*z2 + 0.33249181690998614*v*z2 + 
     0.13021706881608106*lnbara*v*z2 + 0.2084140365865988*u*v*z2 + 
     0.056425067024902395*lnbara*u*v*z2 + 0.0892695937352878*v2*z2 + 
     0.018665378315131372*lnbara*v2*z2 - 0.5580071393420976*w*z2 - 
     0.13021706881608106*lnbara*w*z2 - 0.18693837131393218*u*w*z2 - 
     0.056425067024902395*lnbara*u*w*z2 - 0.37639381622218754*v*w*z2 - 
     0.10851422401340087*lnbara*v*w*z2 + 0.3204490485619901*w2*z2 + 
     0.0898488456982695*lnbara*w2*z2 - 0.12863109955282465*x*z2 + 
     0.009467502496847111*u*x*z2 - 0.567682337066967*v*x*z2 - 
     0.1642900692638757*lnbara*v*x*z2 + 0.5854099721783019*w*x*z2 + 
     0.1642900692638757*lnbara*w*x*z2 + 0.014860391371313229*x2*z2 - 
     0.12863109955282465*y*z2 + 0.009179528839111438*u*y*z2 + 
     0.2937072269357538*v*y*z2 + 0.08594186774137386*lnbara*v*y*z2 - 
     0.2756916181666832*w*y*z2 - 0.08594186774137386*lnbara*w*y*z2 + 
     0.012927559000443166*x*y*z2 + 0.014860391371313229*y2*z2 - 
     0.32387026738382124*z3 + 0.019595873400524336*u*z3 - 
     0.02302537153791993*v*z3 - 0.012370155645114489*lnbara*v*z3 + 
     0.056907625247539464*w*z3 + 0.012370155645114489*lnbara*w*z3 + 
     0.019595873400524336*x*z3 + 0.019595873400524336*y*z3 + 
     0.05780009836815717*z4;
 
  Hexp[5] = 1.442468283791513 - 1.2020569031595942*u + 1.055346282161366*u2 - 
     0.6477405347676425*u3 + 0.2312003934726287*u4 - 
     0.03583388093754629*u5 - 1.2020569031595942*v + 
     0.5605449982541442*u*v - 0.2572621991056493*u2*v + 
     0.07838349360209734*u3*v - 0.01103033535263963*u4*v + 
     1.055346282161366*v2 - 0.2572621991056493*u*v2 + 
     0.059441565485252916*u2*v2 - 0.007187565837728457*u3*v2 - 
     0.6477405347676425*v3 + 0.07838349360209734*u*v3 - 
     0.007187565837728457*u2*v3 + 0.2312003934726287*v4 - 
     0.01103033535263963*u*v4 - 0.03583388093754629*v5 - 
     1.2020569031595942*w + 0.5605449982541442*u*w - 
     0.2572621991056493*u2*w + 0.07838349360209734*u3*w - 
     0.01103033535263963*u4*w + 0.5605449982541442*v*w - 
     0.1577849144326134*u*v*w + 0.037870009987388445*u2*v*w - 
     0.004645460955055343*u3*v*w - 0.2572621991056493*v2*w + 
     0.037870009987388445*u*v2*w - 0.003682012115837776*u2*v2*w + 
     0.07838349360209734*v3*w - 0.004645460955055343*u*v3*w - 
     0.01103033535263963*v4*w + 1.055346282161366*w2 - 
     0.2572621991056493*u*w2 + 0.059441565485252916*u2*w2 - 
     0.007187565837728457*u3*w2 - 0.2572621991056493*v*w2 + 
     0.037870009987388445*u*v*w2 - 0.003682012115837776*u2*v*w2 + 
     0.059441565485252916*v2*w2 - 0.003682012115837776*u*v2*w2 - 
     0.007187565837728457*v3*w2 - 0.6477405347676425*w3 + 
     0.07838349360209734*u*w3 - 0.007187565837728457*u2*w3 + 
     0.07838349360209734*v*w3 - 0.004645460955055343*u*v*w3 - 
     0.007187565837728457*v2*w3 + 0.2312003934726287*w4 - 
     0.01103033535263963*u*w4 - 0.01103033535263963*v*w4 - 
     0.03583388093754629*w5 - 1.2020569031595942*x + 
     0.5605449982541442*u*x - 0.2572621991056493*u2*x + 
     0.07838349360209734*u3*x - 0.01103033535263963*u4*x + 
     0.5605449982541442*v*x + 0.6073596595763473*u*v*x + 
     0.2604341376321621*lnbara*u*v*x + 0.3760917586139986*u2*v*x + 
     0.08940826630843093*lnbara*u2*v*x + 0.037477277457532834*u3*v*x + 
     0.01183453927120045*lnbara*u3*v*x - 1.0639274511577628*v2*x - 
     0.2604341376321621*lnbara*v2*x - 0.7026134316358068*u*v2*x - 
     0.22570026809960958*lnbara*u*v2*x - 0.1500217343901253*u2*v2*x - 
     0.03949638218639865*lnbara*u2*v2*x + 0.21773437647406627*v3*x + 
     0.049480622580457954*lnbara*v3*x + 0.2734637028141685*u*v3*x + 
     0.08124448002734691*lnbara*u*v3*x - 0.0830149629493877*v4*x - 
     0.021288058617899477*lnbara*v4*x + 0.45535505529906856*w*x - 
     0.9544130977678413*u*w*x - 0.2604341376321621*lnbara*u*w*x - 
     0.28766340725578016*u2*w*x - 0.08940826630843093*lnbara*u2*w*x - 
     0.04860539177383991*u3*w*x - 0.01183453927120045*lnbara*u3*w*x - 
     0.1477478457157279*v*w*x + 0.0294499229802987*u*v*w*x - 
     0.0031815740344914314*u2*v*w*x + 2.210207833302107*v2*w*x + 
     0.6571602770555028*lnbara*v2*w*x - 0.24603076767859458*u*v2*w*x - 
     0.07342181363031552*lnbara*u*v2*w*x - 0.379627630631495*v3*w*x - 
     0.1139718730378812*lnbara*v3*w*x + 0.6128968062935399*w2*x + 
     0.2604341376321621*lnbara*w2*x + 0.7885160927264733*u*w2*x + 
     0.22570026809960958*lnbara*u*w2*x + 0.14088351757088213*u2*w2*x + 
     0.03949638218639865*lnbara*u2*w2*x - 2.1392972928567677*v*w2*x - 
     0.6571602770555028*lnbara*v*w2*x + 0.2400067630542147*u*v*w2*x + 
     0.07342181363031552*lnbara*u*v*w2*x - 0.0034959415238012314*v2*w2*x - 
     0.08220536163558813*w3*x - 0.049480622580457954*lnbara*w3*x - 
     0.2841442232687275*u*w3*x - 0.08124448002734691*lnbara*u*w3*x + 
     0.37095314628030795*v*w3*x + 0.1139718730378812*lnbara*v*w3*x + 
     0.06407498022240934*w4*x + 0.021288058617899477*lnbara*w4*x + 
     1.055346282161366*x2 - 0.2572621991056493*u*x2 + 
     0.059441565485252916*u2*x2 - 0.007187565837728457*u3*x2 - 
     1.0639274511577628*v*x2 - 0.2604341376321621*lnbara*v*x2 - 
     0.7026134316358068*u*v*x2 - 0.22570026809960958*lnbara*u*v*x2 - 
     0.1500217343901253*u2*v*x2 - 0.03949638218639865*lnbara*u2*v*x2 + 
     1.2099171176893446*v2*x2 + 0.359395382793078*lnbara*v2*x2 + 
     0.11341087241156884*u*v2*x2 + 0.03460240492286588*lnbara*u*v2*x2 - 
     0.19765223368547283*v3*x2 - 0.05890476996783441*lnbara*v3*x2 + 
     0.6128968062935399*w*x2 + 0.2604341376321621*lnbara*w*x2 + 
     0.7885160927264733*u*w*x2 + 0.22570026809960958*lnbara*u*w*x2 + 
     0.14088351757088213*u2*w*x2 + 0.03949638218639865*lnbara*u2*w*x2 - 
     1.4187638856780294*v*w*x2 - 0.4340568960536035*lnbara*v*w*x2 + 
     0.4719083667190125*u*v*w*x2 + 0.13861380499219164*lnbara*u*v*w*x2 - 
     0.08432966934787235*v2*w*x2 - 0.029311197001213464*lnbara*v2*w*x2 + 
     0.3421460722890461*w2*x2 + 0.07466151326052549*lnbara*w2*x2 - 
     0.5971647409089695*u*w2*x2 - 0.1732162099150575*lnbara*u*w2*x2 + 
     0.23366341602051563*v*w2*x2 + 0.07821369824348423*lnbara*v*w2*x2 + 
     0.029981938909783182*w3*x2 + 0.010002268725563646*lnbara*w3*x2 - 
     0.6477405347676425*x3 + 0.07838349360209734*u*x3 - 
     0.007187565837728457*u2*x3 + 0.21773437647406627*v*x3 + 
     0.049480622580457954*lnbara*v*x3 + 0.2734637028141685*u*v*x3 + 
     0.08124448002734691*lnbara*u*v*x3 - 0.19765223368547283*v2*x3 - 
     0.05890476996783441*lnbara*v2*x3 - 0.08220536163558813*w*x3 - 
     0.049480622580457954*lnbara*w*x3 - 0.2841442232687275*u*w*x3 - 
     0.08124448002734691*lnbara*u*w*x3 + 0.15184270831316776*v*w*x3 + 
     0.04890250124227076*lnbara*v*w*x3 + 0.029981938909783182*w2*x3 + 
     0.010002268725563646*lnbara*w2*x3 + 0.2312003934726287*x4 - 
     0.01103033535263963*u*x4 - 0.0830149629493877*v*x4 - 
     0.021288058617899477*lnbara*v*x4 + 0.06407498022240934*w*x4 + 
     0.021288058617899477*lnbara*w*x4 - 0.03583388093754629*x5 - 
     1.2020569031595942*y + 0.45535505529906856*u*y - 
     0.1937684457585735*u2*y + 0.05714552123638079*u3*y - 
     0.007909647374338729*u4*y + 0.5605449982541442*v*y - 
     0.1477478457157279*u*v*y + 0.03419242508889382*u2*v*y - 
     0.004113445244719507*u3*v*y - 0.2572621991056493*v2*y + 
     0.03671811535644575*u*v2*y - 0.0034959415238012314*u2*v2*y + 
     0.07838349360209734*v3*y - 0.004561039106467548*u*v3*y - 
     0.01103033535263963*v4*y + 0.5605449982541442*w*y - 
     0.1477478457157279*u*w*y + 0.03419242508889382*u2*w*y - 
     0.004113445244719507*u3*w*y - 0.19930559247576604*v*w*y + 
     0.0294499229802987*u*v*w*y - 0.00284243058988845*u2*v*w*y + 
     0.051710236001772665*v2*w*y - 0.0031815740344914314*u*v2*w*y - 
     0.00656707520983952*v3*w*y - 0.2572621991056493*w2*y + 
     0.03671811535644575*u*w2*y - 0.0034959415238012314*u2*w2*y + 
     0.051710236001772665*v*w2*y - 0.0031815740344914314*u*v*w2*y - 
     0.005642275295441937*v2*w2*y + 0.07838349360209734*w3*y - 
     0.004561039106467548*u*w3*y - 0.00656707520983952*v*w3*y - 
     0.01103033535263963*w4*y + 0.5605449982541442*x*y - 
     0.1477478457157279*u*x*y + 0.03419242508889382*u2*x*y - 
     0.004113445244719507*u3*x*y + 2.2622108417237268*v*x*y + 
     0.7813024128964863*lnbara*v*x*y + 0.784651044201938*u*v*x*y + 
     0.2031162645172523*lnbara*u*v*x*y + 0.11792649464288701*u2*v*x*y + 
     0.03307364462356231*lnbara*u2*v*x*y - 1.0340129784736338*v2*x*y - 
     0.3437674709654954*lnbara*v2*x*y - 0.2704335518900037*u*v2*x*y - 
     0.07343720881724174*lnbara*u*v2*x*y + 0.46395116219858895*v3*x*y + 
     0.13730604359507229*lnbara*v3*x*y - 2.5677436018720683*w*x*y - 
     0.7813024128964863*lnbara*w*x*y - 0.7288112068192476*u*w*x*y - 
     0.2031162645172523*lnbara*u*w*x*y - 0.12369133963724584*u2*w*x*y - 
     0.03307364462356231*lnbara*u2*w*x*y + 0.0294499229802987*v*w*x*y - 
     0.0024426217097745406*u*v*w*x*y - 0.49099871455493777*v2*w*x*y - 
     0.13429794906684606*lnbara*v2*w*x*y + 1.106075413549916*w2*x*y + 
     0.3437674709654954*lnbara*w2*x*y + 0.2645638939826569*u*w2*x*y + 
     0.07343720881724174*lnbara*u*w2*x*y + 0.4847403793989428*v*w2*x*y + 
     0.13429794906684606*lnbara*v*w2*x*y - 0.4727100683983638*w3*x*y - 
     0.13730604359507229*lnbara*w3*x*y - 0.2572621991056493*x2*y + 
     0.03671811535644575*u*x2*y - 0.0034959415238012314*u2*x2*y - 
     1.0340129784736338*v*x2*y - 0.3437674709654954*lnbara*v*x2*y - 
     0.2704335518900037*u*v*x2*y - 0.07343720881724174*lnbara*u*v*x2*y + 
     0.031244750707668238*v2*x2*y + 0.0180532260934895*lnbara*v2*x2*y + 
     1.106075413549916*w*x2*y + 0.3437674709654954*lnbara*w*x2*y + 
     0.2645638939826569*u*w*x2*y + 0.07343720881724174*lnbara*u*w*x2*y + 
     0.35130413810496963*v*w*x2*y + 0.0998330860923288*lnbara*v*w*x2*y - 
     0.3921997968798069*w2*x2*y - 0.1178863121858183*lnbara*w2*x2*y + 
     0.07838349360209734*x3*y - 0.004561039106467548*u*x3*y + 
     0.46395116219858895*v*x3*y + 0.13730604359507229*lnbara*v*x3*y - 
     0.4727100683983638*w*x3*y - 0.13730604359507229*lnbara*w*x3*y - 
     0.01103033535263963*x4*y + 1.055346282161366*y2 - 
     0.1937684457585735*u*y2 + 0.03966531372621466*u2*y2 - 
     0.004526575380073919*u3*y2 - 0.2572621991056493*v*y2 + 
     0.03419242508889382*u*v*y2 - 0.003126465361442786*u2*v*y2 + 
     0.059441565485252916*v2*y2 - 0.0034959415238012314*u*v2*y2 - 
     0.007187565837728457*v3*y2 - 0.2572621991056493*w*y2 + 
     0.03419242508889382*u*w*y2 - 0.003126465361442786*u2*w*y2 + 
     0.051710236001772665*v*w*y2 - 0.003076761121503497*u*v*w*y2 - 
     0.005642275295441937*v2*w*y2 + 0.059441565485252916*w2*y2 - 
     0.0034959415238012314*u*w2*y2 - 0.005642275295441937*v*w2*y2 - 
     0.007187565837728457*w3*y2 - 0.2572621991056493*x*y2 + 
     0.03419242508889382*u*x*y2 - 0.003126465361442786*u2*x*y2 - 
     0.615960329857727*v*x*y2 - 0.19532560322412157*lnbara*v*x*y2 + 
     0.03305770874592587*u*v*x*y2 + 0.012242640752295034*lnbara*u*v*x*y2 - 
     0.005100940598748941*v2*x*y2 + 0.0018224798925292808*lnbara*v2*x*
      y2 + 0.6905484552015612*w*x*y2 + 0.19532560322412157*lnbara*w*x*y2 - 
     0.03858822320866964*u*w*x*y2 - 0.012242640752295034*lnbara*u*w*x*y2 - 
     0.0031815740344914314*v*w*x*y2 - 0.0020770130408900665*w2*x*y2 - 
     0.0018224798925292808*lnbara*w2*x*y2 + 0.059441565485252916*x2*y2 - 
     0.0034959415238012314*u*x2*y2 - 0.005100940598748941*v*x2*y2 + 
     0.0018224798925292808*lnbara*v*x2*y2 - 0.0020770130408900665*w*x2*
      y2 - 0.0018224798925292808*lnbara*w*x2*y2 - 
     0.007187565837728457*x3*y2 - 0.6477405347676425*y3 + 
     0.05714552123638079*u*y3 - 0.004526575380073919*u2*y3 + 
     0.07838349360209734*v*y3 - 0.004113445244719507*u*v*y3 - 
     0.007187565837728457*v2*y3 + 0.07838349360209734*w*y3 - 
     0.004113445244719507*u*w*y3 - 0.00656707520983952*v*w*y3 - 
     0.007187565837728457*w2*y3 + 0.07838349360209734*x*y3 - 
     0.004113445244719507*u*x*y3 + 0.2032231771919266*v*x*y3 + 
     0.05859768096723647*lnbara*v*x*y3 - 0.21242967725344947*w*x*y3 - 
     0.05859768096723647*lnbara*w*x*y3 - 0.007187565837728457*x2*y3 + 
     0.2312003934726287*y4 - 0.007909647374338729*u*y4 - 
     0.01103033535263963*v*y4 - 0.01103033535263963*w*y4 - 
     0.01103033535263963*x*y4 - 0.03583388093754629*y5 - 
     1.2020569031595942*z + 0.5605449982541442*u*z - 
     0.2572621991056493*u2*z + 0.07838349360209734*u3*z - 
     0.01103033535263963*u4*z + 0.45535505529906856*v*z - 
     0.9544130977678413*u*v*z - 0.2604341376321621*lnbara*u*v*z - 
     0.28766340725578016*u2*v*z - 0.08940826630843093*lnbara*u2*v*z - 
     0.04860539177383991*u3*v*z - 0.01183453927120045*lnbara*u3*v*z + 
     0.6128968062935399*v2*z + 0.2604341376321621*lnbara*v2*z + 
     0.7885160927264733*u*v2*z + 0.22570026809960958*lnbara*u*v2*z + 
     0.14088351757088213*u2*v2*z + 0.03949638218639865*lnbara*u2*v2*z - 
     0.08220536163558813*v3*z - 0.049480622580457954*lnbara*v3*z - 
     0.2841442232687275*u*v3*z - 0.08124448002734691*lnbara*u*v3*z + 
     0.06407498022240934*v4*z + 0.021288058617899477*lnbara*v4*z + 
     0.5605449982541442*w*z + 0.6073596595763473*u*w*z + 
     0.2604341376321621*lnbara*u*w*z + 0.3760917586139986*u2*w*z + 
     0.08940826630843093*lnbara*u2*w*z + 0.037477277457532834*u3*w*z + 
     0.01183453927120045*lnbara*u3*w*z - 0.1477478457157279*v*w*z + 
     0.0294499229802987*u*v*w*z - 0.0031815740344914314*u2*v*w*z - 
     2.1392972928567677*v2*w*z - 0.6571602770555028*lnbara*v2*w*z + 
     0.2400067630542147*u*v2*w*z + 0.07342181363031552*lnbara*u*v2*w*z + 
     0.37095314628030795*v3*w*z + 0.1139718730378812*lnbara*v3*w*z - 
     1.0639274511577628*w2*z - 0.2604341376321621*lnbara*w2*z - 
     0.7026134316358068*u*w2*z - 0.22570026809960958*lnbara*u*w2*z - 
     0.1500217343901253*u2*w2*z - 0.03949638218639865*lnbara*u2*w2*z + 
     2.210207833302107*v*w2*z + 0.6571602770555028*lnbara*v*w2*z - 
     0.24603076767859458*u*v*w2*z - 0.07342181363031552*lnbara*u*v*w2*z - 
     0.0034959415238012314*v2*w2*z + 0.21773437647406627*w3*z + 
     0.049480622580457954*lnbara*w3*z + 0.2734637028141685*u*w3*z + 
     0.08124448002734691*lnbara*u*w3*z - 0.379627630631495*v*w3*z - 
     0.1139718730378812*lnbara*v*w3*z - 0.0830149629493877*w4*z - 
     0.021288058617899477*lnbara*w4*z + 0.5605449982541442*x*z - 
     0.1577849144326134*u*x*z + 0.037870009987388445*u2*x*z - 
     0.004645460955055343*u3*x*z - 0.1477478457157279*v*x*z + 
     0.0294499229802987*u*v*x*z - 0.0031815740344914314*u2*v*x*z - 
     1.4187638856780294*v2*x*z - 0.4340568960536035*lnbara*v2*x*z + 
     0.4719083667190125*u*v2*x*z + 0.13861380499219164*lnbara*u*v2*x*z + 
     0.15184270831316776*v3*x*z + 0.04890250124227076*lnbara*v3*x*z - 
     0.1477478457157279*w*x*z + 0.0294499229802987*u*w*x*z - 
     0.0031815740344914314*u2*w*x*z + 2.932302535936238*v*w*x*z + 
     0.868113792107207*lnbara*v*w*x*z - 0.9524128773908065*u*v*w*x*z - 
     0.2772276099843833*lnbara*u*v*w*x*z - 0.1586442374307426*v2*w*x*z - 
     0.04890250124227076*lnbara*v2*w*x*z - 1.4187638856780294*w2*x*z - 
     0.4340568960536035*lnbara*w2*x*z + 0.4719083667190125*u*w2*x*z + 
     0.13861380499219164*lnbara*u*w2*x*z - 0.1586442374307426*v*w2*x*z - 
     0.04890250124227076*lnbara*v*w2*x*z + 0.15184270831316776*w3*x*z + 
     0.04890250124227076*lnbara*w3*x*z - 0.2572621991056493*x2*z + 
     0.037870009987388445*u*x2*z - 0.003682012115837776*u2*x2*z + 
     2.210207833302107*v*x2*z + 0.6571602770555028*lnbara*v*x2*z - 
     0.24603076767859458*u*v*x2*z - 0.07342181363031552*lnbara*u*v*x2*z - 
     0.08432966934787235*v2*x2*z - 0.029311197001213464*lnbara*v2*x2*z - 
     2.1392972928567677*w*x2*z - 0.6571602770555028*lnbara*w*x2*z + 
     0.2400067630542147*u*w*x2*z + 0.07342181363031552*lnbara*u*w*x2*z - 
     0.1586442374307426*v*w*x2*z - 0.04890250124227076*lnbara*v*w*x2*z + 
     0.23366341602051563*w2*x2*z + 0.07821369824348423*lnbara*w2*x2*z + 
     0.07838349360209734*x3*z - 0.004645460955055343*u*x3*z - 
     0.379627630631495*v*x3*z - 0.1139718730378812*lnbara*v*x3*z + 
     0.37095314628030795*w*x3*z + 0.1139718730378812*lnbara*w*x3*z - 
     0.01103033535263963*x4*z + 0.5605449982541442*y*z - 
     0.1477478457157279*u*y*z + 0.03419242508889382*u2*y*z - 
     0.004113445244719507*u3*y*z - 2.5677436018720683*v*y*z - 
     0.7813024128964863*lnbara*v*y*z - 0.7288112068192476*u*v*y*z - 
     0.2031162645172523*lnbara*u*v*y*z - 0.12369133963724584*u2*v*y*z - 
     0.03307364462356231*lnbara*u2*v*y*z + 1.106075413549916*v2*y*z + 
     0.3437674709654954*lnbara*v2*y*z + 0.2645638939826569*u*v2*y*z + 
     0.07343720881724174*lnbara*u*v2*y*z - 0.4727100683983638*v3*y*z - 
     0.13730604359507229*lnbara*v3*y*z + 2.2622108417237268*w*y*z + 
     0.7813024128964863*lnbara*w*y*z + 0.784651044201938*u*w*y*z + 
     0.2031162645172523*lnbara*u*w*y*z + 0.11792649464288701*u2*w*y*z + 
     0.03307364462356231*lnbara*u2*w*y*z + 0.0294499229802987*v*w*y*z - 
     0.0024426217097745406*u*v*w*y*z + 0.4847403793989428*v2*w*y*z + 
     0.13429794906684606*lnbara*v2*w*y*z - 1.0340129784736338*w2*y*z - 
     0.3437674709654954*lnbara*w2*y*z - 0.2704335518900037*u*w2*y*z - 
     0.07343720881724174*lnbara*u*w2*y*z - 0.49099871455493777*v*w2*y*z - 
     0.13429794906684606*lnbara*v*w2*y*z + 0.46395116219858895*w3*y*z + 
     0.13730604359507229*lnbara*w3*y*z - 0.19930559247576604*x*y*z + 
     0.0294499229802987*u*x*y*z - 0.00284243058988845*u2*x*y*z + 
     0.0294499229802987*v*x*y*z - 0.0024426217097745406*u*v*x*y*z + 
     0.35130413810496963*v2*x*y*z + 0.0998330860923288*lnbara*v2*x*y*z + 
     0.0294499229802987*w*x*y*z - 0.0024426217097745406*u*w*x*y*z - 
     0.7107357590994907*v*w*x*y*z - 0.1996661721846576*lnbara*v*w*x*y*z + 
     0.35130413810496963*w2*x*y*z + 0.0998330860923288*lnbara*w2*x*y*z + 
     0.051710236001772665*x2*y*z - 0.0031815740344914314*u*x2*y*z - 
     0.49099871455493777*v*x2*y*z - 0.13429794906684606*lnbara*v*x2*y*z + 
     0.4847403793989428*w*x2*y*z + 0.13429794906684606*lnbara*w*x2*y*z - 
     0.00656707520983952*x3*y*z - 0.2572621991056493*y2*z + 
     0.03419242508889382*u*y2*z - 0.003126465361442786*u2*y2*z + 
     0.6905484552015612*v*y2*z + 0.19532560322412157*lnbara*v*y2*z - 
     0.03858822320866964*u*v*y2*z - 0.012242640752295034*lnbara*u*v*y2*z - 
     0.0020770130408900665*v2*y2*z - 0.0018224798925292808*lnbara*v2*y2*
      z - 0.615960329857727*w*y2*z - 0.19532560322412157*lnbara*w*y2*z + 
     0.03305770874592587*u*w*y2*z + 0.012242640752295034*lnbara*u*w*y2*z - 
     0.0031815740344914314*v*w*y2*z - 0.005100940598748941*w2*y2*z + 
     0.0018224798925292808*lnbara*w2*y2*z + 0.051710236001772665*x*y2*z - 
     0.003076761121503497*u*x*y2*z - 0.0031815740344914314*v*x*y2*z - 
     0.0031815740344914314*w*x*y2*z - 0.005642275295441937*x2*y2*z + 
     0.07838349360209734*y3*z - 0.004113445244719507*u*y3*z - 
     0.21242967725344947*v*y3*z - 0.05859768096723647*lnbara*v*y3*z + 
     0.2032231771919266*w*y3*z + 0.05859768096723647*lnbara*w*y3*z - 
     0.00656707520983952*x*y3*z - 0.01103033535263963*y4*z + 
     1.055346282161366*z2 - 0.2572621991056493*u*z2 + 
     0.059441565485252916*u2*z2 - 0.007187565837728457*u3*z2 + 
     0.6128968062935399*v*z2 + 0.2604341376321621*lnbara*v*z2 + 
     0.7885160927264733*u*v*z2 + 0.22570026809960958*lnbara*u*v*z2 + 
     0.14088351757088213*u2*v*z2 + 0.03949638218639865*lnbara*u2*v*z2 + 
     0.3421460722890461*v2*z2 + 0.07466151326052549*lnbara*v2*z2 - 
     0.5971647409089695*u*v2*z2 - 0.1732162099150575*lnbara*u*v2*z2 + 
     0.029981938909783182*v3*z2 + 0.010002268725563646*lnbara*v3*z2 - 
     1.0639274511577628*w*z2 - 0.2604341376321621*lnbara*w*z2 - 
     0.7026134316358068*u*w*z2 - 0.22570026809960958*lnbara*u*w*z2 - 
     0.1500217343901253*u2*w*z2 - 0.03949638218639865*lnbara*u2*w*z2 - 
     1.4187638856780294*v*w*z2 - 0.4340568960536035*lnbara*v*w*z2 + 
     0.4719083667190125*u*v*w*z2 + 0.13861380499219164*lnbara*u*v*w*z2 + 
     0.23366341602051563*v2*w*z2 + 0.07821369824348423*lnbara*v2*w*z2 + 
     1.2099171176893446*w2*z2 + 0.359395382793078*lnbara*w2*z2 + 
     0.11341087241156884*u*w2*z2 + 0.03460240492286588*lnbara*u*w2*z2 - 
     0.08432966934787235*v*w2*z2 - 0.029311197001213464*lnbara*v*w2*z2 - 
     0.19765223368547283*w3*z2 - 0.05890476996783441*lnbara*w3*z2 - 
     0.2572621991056493*x*z2 + 0.037870009987388445*u*x*z2 - 
     0.003682012115837776*u2*x*z2 - 2.1392972928567677*v*x*z2 - 
     0.6571602770555028*lnbara*v*x*z2 + 0.2400067630542147*u*v*x*z2 + 
     0.07342181363031552*lnbara*u*v*x*z2 + 0.23366341602051563*v2*x*z2 + 
     0.07821369824348423*lnbara*v2*x*z2 + 2.210207833302107*w*x*z2 + 
     0.6571602770555028*lnbara*w*x*z2 - 0.24603076767859458*u*w*x*z2 - 
     0.07342181363031552*lnbara*u*w*x*z2 - 0.1586442374307426*v*w*x*z2 - 
     0.04890250124227076*lnbara*v*w*x*z2 - 0.08432966934787235*w2*x*z2 - 
     0.029311197001213464*lnbara*w2*x*z2 + 0.059441565485252916*x2*z2 - 
     0.003682012115837776*u*x2*z2 - 0.0034959415238012314*v*x2*z2 - 
     0.0034959415238012314*w*x2*z2 - 0.007187565837728457*x3*z2 - 
     0.2572621991056493*y*z2 + 0.03671811535644575*u*y*z2 - 
     0.0034959415238012314*u2*y*z2 + 1.106075413549916*v*y*z2 + 
     0.3437674709654954*lnbara*v*y*z2 + 0.2645638939826569*u*v*y*z2 + 
     0.07343720881724174*lnbara*u*v*y*z2 - 0.3921997968798069*v2*y*z2 - 
     0.1178863121858183*lnbara*v2*y*z2 - 1.0340129784736338*w*y*z2 - 
     0.3437674709654954*lnbara*w*y*z2 - 0.2704335518900037*u*w*y*z2 - 
     0.07343720881724174*lnbara*u*w*y*z2 + 0.35130413810496963*v*w*y*z2 + 
     0.0998330860923288*lnbara*v*w*y*z2 + 0.031244750707668238*w2*y*z2 + 
     0.0180532260934895*lnbara*w2*y*z2 + 0.051710236001772665*x*y*z2 - 
     0.0031815740344914314*u*x*y*z2 + 0.4847403793989428*v*x*y*z2 + 
     0.13429794906684606*lnbara*v*x*y*z2 - 0.49099871455493777*w*x*y*z2 - 
     0.13429794906684606*lnbara*w*x*y*z2 - 0.005642275295441937*x2*y*z2 + 
     0.059441565485252916*y2*z2 - 0.0034959415238012314*u*y2*z2 - 
     0.0020770130408900665*v*y2*z2 - 0.0018224798925292808*lnbara*v*y2*
      z2 - 0.005100940598748941*w*y2*z2 + 0.0018224798925292808*lnbara*w*
      y2*z2 - 0.005642275295441937*x*y2*z2 - 0.007187565837728457*y3*
      z2 - 0.6477405347676425*z3 + 0.07838349360209734*u*z3 - 
     0.007187565837728457*u2*z3 - 0.08220536163558813*v*z3 - 
     0.049480622580457954*lnbara*v*z3 - 0.2841442232687275*u*v*z3 - 
     0.08124448002734691*lnbara*u*v*z3 + 0.029981938909783182*v2*z3 + 
     0.010002268725563646*lnbara*v2*z3 + 0.21773437647406627*w*z3 + 
     0.049480622580457954*lnbara*w*z3 + 0.2734637028141685*u*w*z3 + 
     0.08124448002734691*lnbara*u*w*z3 + 0.15184270831316776*v*w*z3 + 
     0.04890250124227076*lnbara*v*w*z3 - 0.19765223368547283*w2*z3 - 
     0.05890476996783441*lnbara*w2*z3 + 0.07838349360209734*x*z3 - 
     0.004645460955055343*u*x*z3 + 0.37095314628030795*v*x*z3 + 
     0.1139718730378812*lnbara*v*x*z3 - 0.379627630631495*w*x*z3 - 
     0.1139718730378812*lnbara*w*x*z3 - 0.007187565837728457*x2*z3 + 
     0.07838349360209734*y*z3 - 0.004561039106467548*u*y*z3 - 
     0.4727100683983638*v*y*z3 - 0.13730604359507229*lnbara*v*y*z3 + 
     0.46395116219858895*w*y*z3 + 0.13730604359507229*lnbara*w*y*z3 - 
     0.00656707520983952*x*y*z3 - 0.007187565837728457*y2*z3 + 
     0.2312003934726287*z4 - 0.01103033535263963*u*z4 + 
     0.06407498022240934*v*z4 + 0.021288058617899477*lnbara*v*z4 - 
     0.0830149629493877*w*z4 - 0.021288058617899477*lnbara*w*z4 - 
     0.01103033535263963*x*z4 - 0.01103033535263963*y*z4 - 
     0.03583388093754629*z5;
 
  Hexp[6] = 1.2020569031595942 - 1.2020569031595942*u + 
     1.3191828527017075*u2 - 1.0795675579460708*u3 + 
     0.5780009836815717*u4 - 0.17916940468773146*u5 + 
     0.024344353217864104*u6 - 1.2020569031595942*v + 
     0.7006812478176802*u*v - 0.4287703318427488*u2*v + 
     0.19595873400524338*u3*v - 0.05515167676319815*u4*v + 
     0.007027953632525705*u5*v + 1.3191828527017075*v2 - 
     0.4287703318427488*u*v2 + 0.14860391371313228*u2*v2 - 
     0.03593782918864228*u3*v2 + 0.004169061579930593*u4*v2 - 
     1.0795675579460708*v3 + 0.19595873400524338*u*v3 - 
     0.03593782918864228*u2*v3 + 0.0035837624283894857*u3*v3 + 
     0.5780009836815717*v4 - 0.05515167676319815*u*v4 + 
     0.004169061579930593*u2*v4 - 0.17916940468773146*v5 + 
     0.007027953632525705*u*v5 + 0.024344353217864104*v6 - 
     1.2020569031595942*w + 0.7006812478176802*u*w - 
     0.4287703318427488*u2*w + 0.19595873400524338*u3*w - 
     0.05515167676319815*u4*w + 0.007027953632525705*u5*w + 
     0.7006812478176802*v*w - 0.262974857387689*u*v*w + 
     0.09467502496847112*u2*v*w - 0.023227304775276715*u3*v*w + 
     0.0027129425983649334*u4*v*w - 0.4287703318427488*v2*w + 
     0.09467502496847112*u*v2*w - 0.01841006057918888*u2*v2*w + 
     0.001880650990063473*u3*v2*w + 0.19595873400524338*v3*w - 
     0.023227304775276715*u*v3*w + 0.001880650990063473*u2*v3*w - 
     0.05515167676319815*v4*w + 0.0027129425983649334*u*v4*w + 
     0.007027953632525705*v5*w + 1.3191828527017075*w2 - 
     0.4287703318427488*u*w2 + 0.14860391371313228*u2*w2 - 
     0.03593782918864228*u3*w2 + 0.004169061579930593*u4*w2 - 
     0.4287703318427488*v*w2 + 0.09467502496847112*u*v*w2 - 
     0.01841006057918888*u2*v*w2 + 0.001880650990063473*u3*v*w2 + 
     0.14860391371313228*v2*w2 - 0.01841006057918888*u*v2*w2 + 
     0.0015174486936744638*u2*v2*w2 - 0.03593782918864228*v3*w2 + 
     0.001880650990063473*u*v3*w2 + 0.004169061579930593*v4*w2 - 
     1.0795675579460708*w3 + 0.19595873400524338*u*w3 - 
     0.03593782918864228*u2*w3 + 0.0035837624283894857*u3*w3 + 
     0.19595873400524338*v*w3 - 0.023227304775276715*u*v*w3 + 
     0.001880650990063473*u2*v*w3 - 0.03593782918864228*v2*w3 + 
     0.001880650990063473*u*v2*w3 + 0.0035837624283894857*v3*w3 + 
     0.5780009836815717*w4 - 0.05515167676319815*u*w4 + 
     0.004169061579930593*u2*w4 - 0.05515167676319815*v*w4 + 
     0.0027129425983649334*u*v*w4 + 0.004169061579930593*v2*w4 - 
     0.17916940468773146*w5 + 0.007027953632525705*u*w5 + 
     0.007027953632525705*v*w5 + 0.024344353217864104*w6 - 
     1.2020569031595942*x + 0.7006812478176802*u*x - 
     0.4287703318427488*u2*x + 0.19595873400524338*u3*x - 
     0.05515167676319815*u4*x + 0.007027953632525705*u5*x + 
     0.7006812478176802*v*x + 0.9399232832849783*u*v*x + 
     0.4340568960536035*lnbara*u*v*x + 0.9029759522398169*u2*v*x + 
     0.22352066577107735*lnbara*u2*v*x + 0.17752427122833048*u3*v*x + 
     0.05917269635600225*lnbara*u3*v*x + 0.024106166155729025*u4*v*x + 
     0.0052304795439763515*lnbara*u4*v*x - 1.700869602587337*v2*x - 
     0.4340568960536035*lnbara*v2*x - 1.6624918007146796*u*v2*x - 
     0.564250670249024*lnbara*u*v2*x - 0.7171950201286277*u2*v2*x - 
     0.19748191093199324*lnbara*u2*v2*x - 0.08627759743978855*u3*v2*x - 
     0.02432329879100571*lnbara*u3*v2*x + 0.5237190151099749*v3*x + 
     0.12370155645114488*lnbara*v3*x + 1.29961478071472*u*v3*x + 
     0.40622240013673455*lnbara*u*v3*x + 0.08694308134115901*u2*v3*x + 
     0.021986858303963977*lnbara*u2*v3*x - 0.397334765898689*v4*x - 
     0.10644029308949739*lnbara*v4*x - 0.22841481745724104*u*v4*x - 
     0.06907962057473943*lnbara*u*v4*x + 0.0487756611897797*v5*x + 
     0.013089806333713885*lnbara*v5*x + 0.5691938191238357*w*x - 
     1.5183456802708015*u*w*x - 0.4340568960536035*lnbara*u*w*x - 
     0.6819050738442709*u2*w*x - 0.22352066577107735*lnbara*u2*w*x - 
     0.23316484280986582*u3*w*x - 0.05917269635600225*lnbara*u3*w*x - 
     0.017520814568225136*u4*w*x - 0.0052304795439763515*lnbara*u4*w*x - 
     0.24624640952621318*v*w*x + 0.07362480745074675*u*v*w*x - 
     0.015907870172457157*u2*v*w*x + 0.0017051473997301786*u3*v*w*x + 
     5.2517028011488085*v2*w*x + 1.642900692638757*lnbara*v2*w*x - 
     1.1689689937010432*u*v2*w*x - 0.36710906815157757*lnbara*u*v2*w*x + 
     0.5646118353233481*u2*v2*w*x + 0.16529892872504193*lnbara*u2*v2*w*
      x - 1.803161592292574*v3*w*x - 0.5698593651894059*lnbara*v3*w*x - 
     0.025380422110629967*u*v3*w*x - 0.0022974655834007394*lnbara*u*v3*w*
      x + 0.5276000546267682*v4*w*x + 0.15634114484424524*lnbara*v4*w*x + 
     0.9491518611469659*w2*x + 0.4340568960536035*lnbara*w2*x + 
     1.8772484534413458*u*w2*x + 0.564250670249024*lnbara*u*w2*x + 
     0.6715039360324119*u2*w2*x + 0.19748191093199324*lnbara*u2*w2*x + 
     0.0911441314262025*u3*w2*x + 0.02432329879100571*lnbara*u3*w2*x - 
     5.0744264500354594*v*w2*x - 1.642900692638757*lnbara*v*w2*x + 
     1.138848970579144*u*v*w2*x + 0.36710906815157757*lnbara*u*v*w2*x - 
     0.5618212037342907*u2*v*w2*x - 0.16529892872504193*lnbara*u2*v*w2*
      x - 0.017479707619006156*v2*w2*x + 0.00130062566240163*u*v2*w2*x - 
     0.3213850461435467*v3*w2*x - 0.08664893308797812*lnbara*v3*w2*x - 
     0.1848964780137795*w3*x - 0.12370155645114488*lnbara*w3*x - 
     1.3530173829875152*u*w3*x - 0.40622240013673455*lnbara*u*w3*x - 
     0.0821346574113807*u2*w3*x - 0.021986858303963977*lnbara*u2*w3*x + 
     1.7597891705366386*v*w3*x + 0.5698593651894059*lnbara*v*w3*x + 
     0.02852820513246288*u*v*w3*x + 0.0022974655834007394*lnbara*u*v*w3*x + 
     0.3249693774147304*v2*w3*x + 0.08664893308797812*lnbara*v2*w3*x + 
     0.3026348522637972*w4*x + 0.10644029308949739*lnbara*w4*x + 
     0.23469657548791326*u*w4*x + 0.06907962057473943*lnbara*u*w4*x - 
     0.5225454785602572*v*w4*x - 0.15634114484424524*lnbara*v*w4*x - 
     0.03675623670681*w5*x - 0.013089806333713885*lnbara*w5*x + 
     1.3191828527017075*x2 - 0.4287703318427488*u*x2 + 
     0.14860391371313228*u2*x2 - 0.03593782918864228*u3*x2 + 
     0.004169061579930593*u4*x2 - 1.700869602587337*v*x2 - 
     0.4340568960536035*lnbara*v*x2 - 1.6624918007146796*u*v*x2 - 
     0.564250670249024*lnbara*u*v*x2 - 0.7171950201286277*u2*v*x2 - 
     0.19748191093199324*lnbara*u2*v*x2 - 0.08627759743978855*u3*v*x2 - 
     0.02432329879100571*lnbara*u3*v*x2 + 2.8750447180595793*v2*x2 + 
     0.898488456982695*lnbara*v2*x2 + 0.5382190246221227*u*v2*x2 + 
     0.1730120246143294*lnbara*u*v2*x2 + 0.479916000193387*u2*v2*x2 + 
     0.13727120803720705*lnbara*u2*v2*x2 - 0.9391738601208355*v3*x2 - 
     0.29452384983917207*lnbara*v3*x2 - 0.3238462135165472*u*v3*x2 - 
     0.09533400426449122*lnbara*u*v3*x2 + 0.29091212584308784*v4*x2 + 
     0.08588333011797569*lnbara*v4*x2 + 0.9491518611469659*w*x2 + 
     0.4340568960536035*lnbara*w*x2 + 1.8772484534413458*u*w*x2 + 
     0.564250670249024*lnbara*u*w*x2 + 0.6715039360324119*u2*w*x2 + 
     0.19748191093199324*lnbara*u2*w*x2 + 0.0911441314262025*u3*w*x2 + 
     0.02432329879100571*lnbara*u3*w*x2 - 3.366052674172739*v*w*x2 - 
     1.0851422401340087*lnbara*v*w*x2 + 2.244030329434903*u*v*w*x2 + 
     0.6930690249609582*lnbara*u*v*w*x2 - 0.5870531877615167*u2*v*w*x2 - 
     0.1732603915082422*lnbara*u2*v*w*x2 - 0.3972223492383506*v2*w*x2 - 
     0.1465559850060673*lnbara*v2*w*x2 - 0.20929117997162128*u*v2*w*x2 - 
     0.06205837953332923*lnbara*u*v2*w*x2 - 0.25301233477098106*v3*w*x2 - 
     0.062498180515692975*lnbara*v3*w*x2 + 0.8242562168640629*w2*x2 + 
     0.18665378315131373*lnbara*w2*x2 - 2.8414768629489666*u*w2*x2 - 
     0.8660810495752876*lnbara*u*w2*x2 + 0.11275072634761112*u2*w2*x2 + 
     0.035989183471035156*lnbara*u2*w2*x2 + 1.103138998233008*v*w2*x2 + 
     0.39106849121742115*lnbara*v*w2*x2 + 0.13104994444668858*u*v*w2*x2 + 
     0.030017221475897783*lnbara*u*v*w2*x2 + 0.4244378822310122*v2*w2*
      x2 + 0.11758155264728642*lnbara*v2*w2*x2 + 
     0.14157447061094622*w3*x2 + 0.05001134362781823*lnbara*w3*x2 + 
     0.4093655691718876*u*w3*x2 + 0.12737516232192267*lnbara*u*w3*x2 - 
     0.4837702578695747*v*w3*x2 - 0.14799273546990963*lnbara*v*w3*x2 + 
     0.03289810142239334*w4*x2 + 0.007026033220340475*lnbara*w4*x2 - 
     1.0795675579460708*x3 + 0.19595873400524338*u*x3 - 
     0.03593782918864228*u2*x3 + 0.0035837624283894857*u3*x3 + 
     0.5237190151099749*v*x3 + 0.12370155645114488*lnbara*v*x3 + 
     1.29961478071472*u*v*x3 + 0.40622240013673455*lnbara*u*v*x3 + 
     0.08694308134115901*u2*v*x3 + 0.021986858303963977*lnbara*u2*v*x3 - 
     0.9391738601208355*v2*x3 - 0.29452384983917207*lnbara*v2*x3 - 
     0.3238462135165472*u*v2*x3 - 0.09533400426449122*lnbara*u*v2*x3 + 
     0.043640908015127135*v3*x3 + 0.01752291016105757*lnbara*v3*x3 - 
     0.1848964780137795*w*x3 - 0.12370155645114488*lnbara*w*x3 - 
     1.3530173829875152*u*w*x3 - 0.40622240013673455*lnbara*u*w*x3 - 
     0.0821346574113807*u2*w*x3 - 0.021986858303963977*lnbara*u2*w*x3 + 
     0.7184614571972799*v*w*x3 + 0.24451250621135381*lnbara*v*w*x3 - 
     0.07931329913612384*u*v*w*x3 - 0.03204115805743145*lnbara*u*v*w*x3 + 
     0.5112934663833444*v2*w*x3 + 0.14720183048123822*lnbara*v2*w*x3 + 
     0.14157447061094622*w2*x3 + 0.05001134362781823*lnbara*w2*x3 + 
     0.4093655691718876*u*w2*x3 + 0.12737516232192267*lnbara*u*w2*x3 - 
     0.569080649680904*v*w2*x3 - 0.16941649028491496*lnbara*v*w2*x3 + 
     0.02313083477136829*w3*x3 + 0.004691749642619168*lnbara*w3*x3 + 
     0.5780009836815717*x4 - 0.05515167676319815*u*x4 + 
     0.004169061579930593*u2*x4 - 0.397334765898689*v*x4 - 
     0.10644029308949739*lnbara*v*x4 - 0.22841481745724104*u*v*x4 - 
     0.06907962057473943*lnbara*u*v*x4 + 0.29091212584308784*v2*x4 + 
     0.08588333011797569*lnbara*v2*x4 + 0.3026348522637972*w*x4 + 
     0.10644029308949739*lnbara*w*x4 + 0.23469657548791326*u*w*x4 + 
     0.06907962057473943*lnbara*u*w*x4 - 0.31472121563065336*v*w*x4 - 
     0.09290936333831616*lnbara*v*w*x4 + 0.03289810142239334*w2*x4 + 
     0.007026033220340475*lnbara*w2*x4 - 0.17916940468773146*x5 + 
     0.007027953632525705*u*x5 + 0.0487756611897797*v*x5 + 
     0.013089806333713885*lnbara*v*x5 - 0.03675623670681*w*x5 - 
     0.013089806333713885*lnbara*w*x5 + 0.024344353217864104*x6 - 
     1.2020569031595942*y + 0.5691938191238357*u*y - 
     0.32294740959762247*u2*y + 0.14286380309095198*u3*y - 
     0.03954823687169365*u4*y + 0.004991470850443998*u5*y + 
     0.7006812478176802*v*y - 0.24624640952621318*u*v*y + 
     0.08548106272223456*u2*v*y - 0.020567226223597537*u3*v*y + 
     0.0023754912548396108*u4*v*y - 0.4287703318427488*v2*y + 
     0.09179528839111438*u*v2*y - 0.017479707619006156*u2*v2*y + 
     0.0017631106072740343*u3*v2*y + 0.19595873400524338*v3*y - 
     0.022805195532337737*u*v3*y + 0.001821220663909673*u2*v3*y - 
     0.05515167676319815*v4*y + 0.0026790848116713074*u*v4*y + 
     0.007027953632525705*v5*y + 0.7006812478176802*w*y - 
     0.24624640952621318*u*w*y + 0.08548106272223456*u2*w*y - 
     0.020567226223597537*u3*w*y + 0.0023754912548396108*u4*w*y - 
     0.3321759874596101*v*w*y + 0.07362480745074675*u*v*w*y - 
     0.01421215294944225*u2*v*w*y + 0.001442635622102732*u3*v*w*y + 
     0.12927559000443167*v2*w*y - 0.015907870172457157*u*v2*w*y + 
     0.00130062566240163*u2*v2*w*y - 0.032835376049197604*v3*w*y + 
     0.0017051473997301786*u*v3*w*y + 0.0039062667758325825*v4*w*y - 
     0.4287703318427488*w2*y + 0.09179528839111438*u*w2*y - 
     0.017479707619006156*u2*w2*y + 0.0017631106072740343*u3*w2*y + 
     0.12927559000443167*v*w2*y - 0.015907870172457157*u*v*w2*y + 
     0.00130062566240163*u2*v*w2*y - 0.028211376477209685*v2*w2*y + 
     0.001490005926655855*u*v2*w2*y + 0.0030453133225042724*v3*w2*y + 
     0.19595873400524338*w3*y - 0.022805195532337737*u*w3*y + 
     0.001821220663909673*u2*w3*y - 0.032835376049197604*v*w3*y + 
     0.0017051473997301786*u*v*w3*y + 0.0030453133225042724*v2*w3*y - 
     0.05515167676319815*w4*y + 0.0026790848116713074*u*w4*y + 
     0.0039062667758325825*v*w4*y + 0.007027953632525705*w5*y + 
     0.7006812478176802*x*y - 0.24624640952621318*u*x*y + 
     0.08548106272223456*u2*x*y - 0.020567226223597537*u3*x*y + 
     0.0023754912548396108*u4*x*y + 3.553322954846076*v*x*y + 
     1.3021706881608104*lnbara*v*x*y + 1.8769958336226564*u*v*x*y + 
     0.5077906612931307*lnbara*u*v*x*y + 0.5620711026947999*u2*v*x*y + 
     0.16536822311781155*lnbara*u2*v*x*y + 0.06775115910844728*u3*v*x*y + 
     0.017198580632115178*lnbara*u3*v*x*y - 2.4417959999484613*v2*x*y - 
     0.8594186774137386*lnbara*v2*x*y - 1.2909700854356503*u*v2*x*y - 
     0.3671860440862087*lnbara*u*v2*x*y - 0.222192079524089*u2*v2*x*y - 
     0.06208972089809539*lnbara*u2*v2*x*y + 2.2053341079970514*v3*x*y + 
     0.6865302179753614*lnbara*v3*x*y + 0.19254447471177424*u*v3*x*y + 
     0.05112899772410982*lnbara*u*v3*x*y - 0.412831813275259*v4*x*y - 
     0.12512755974621306*lnbara*v4*x*y - 4.062544221759978*w*x*y - 
     1.3021706881608104*lnbara*w*x*y - 1.7373962401659306*u*w*x*y - 
     0.5077906612931307*lnbara*u*w*x*y - 0.5908953276665939*u2*w*x*y - 
     0.16536822311781155*lnbara*u2*w*x*y - 0.064738719421504*u3*w*x*y - 
     0.017198580632115178*lnbara*u3*w*x*y + 0.07362480745074675*v*w*x*y - 
     0.012213108548872703*u*v*w*x*y + 0.001111492150387925*u2*v*w*x*y - 
     2.343078615218984*v2*w*x*y - 0.6714897453342303*lnbara*v2*w*x*y + 
     0.4498241458617747*u*v2*w*x*y + 0.13547920846092296*lnbara*u*v2*w*x*
      y + 0.12539708324984536*v3*w*x*y + 0.034595562617565336*lnbara*v3*w*x*
      y + 2.621952087639167*w2*x*y + 0.8594186774137386*lnbara*w2*x*y + 
     1.2616217958989164*u*w2*x*y + 0.3671860440862087*lnbara*u*w2*x*y + 
     0.22484058072370788*u2*w2*x*y + 0.06208972089809539*lnbara*u2*w2*x*
      y + 2.311786939439009*v*w2*x*y + 0.6714897453342303*lnbara*v*w2*x*y - 
     0.4475358708690333*u*v*w2*x*y - 0.13547920846092296*lnbara*u*v*w2*x*
      y + 0.00146144490120512*v2*w2*x*y - 2.2491286389959257*w3*x*y - 
     0.6865302179753614*lnbara*w3*x*y - 0.18944564848149062*u*w3*x*y - 
     0.05112899772410982*lnbara*u*w3*x*y - 0.12207317499372534*v*w3*x*y - 
     0.034595562617565336*lnbara*v*w3*x*y + 0.4179202471284635*w4*x*y + 
     0.12512755974621306*lnbara*w4*x*y - 0.4287703318427488*x2*y + 
     0.09179528839111438*u*x2*y - 0.017479707619006156*u2*x2*y + 
     0.0017631106072740343*u3*x2*y - 2.4417959999484613*v*x2*y - 
     0.8594186774137386*lnbara*v*x2*y - 1.2909700854356503*u*v*x2*y - 
     0.3671860440862087*lnbara*u*v*x2*y - 0.222192079524089*u2*v*x2*y - 
     0.06208972089809539*lnbara*u2*v*x2*y + 0.1411793984604333*v2*x2*y + 
     0.09026613046744748*lnbara*v2*x2*y + 0.8636086160336289*u*v2*x2*y + 
     0.24856634396486232*lnbara*u*v2*x2*y - 0.28983228539271344*v3*x2*y - 
     0.09094784681695743*lnbara*v3*x2*y + 2.621952087639167*w*x2*y + 
     0.8594186774137386*lnbara*w*x2*y + 1.2616217958989164*u*w*x2*y + 
     0.3671860440862087*lnbara*u*w*x2*y + 0.22484058072370788*u2*w*x2*y + 
     0.06208972089809539*lnbara*u2*w*x2*y + 1.6733264521145743*v*w*x2*y + 
     0.499165430461644*lnbara*v*w*x2*y - 0.7575451375965339*u*v*w*x2*y - 
     0.21838360971474802*lnbara*u*v*w*x2*y + 0.029245523131395473*v2*w*x2*
      y + 0.008946886688429884*lnbara*v2*w*x2*y - 
     1.8627603909108525*w2*x2*y - 0.5894315609290915*lnbara*w2*x2*y - 
     0.10231789367095857*u*w2*x2*y - 0.030182734250114304*lnbara*u*w2*x2*
      y - 0.034386556943485275*v*w2*x2*y - 0.01584237786733671*lnbara*v*w2*
      x2*y + 0.3009259577639667*w3*x2*y + 0.09784333799586425*lnbara*w3*
      x2*y + 0.19595873400524338*x3*y - 0.022805195532337737*u*x3*y + 
     0.001821220663909673*u2*x3*y + 2.2053341079970514*v*x3*y + 
     0.6865302179753614*lnbara*v*x3*y + 0.19254447471177424*u*v*x3*y + 
     0.05112899772410982*lnbara*u*v*x3*y - 0.28983228539271344*v2*x3*y - 
     0.09094784681695743*lnbara*v2*x3*y - 2.2491286389959257*w*x3*y - 
     0.6865302179753614*lnbara*w*x3*y - 0.18944564848149062*u*w*x3*y - 
     0.05112899772410982*lnbara*u*w*x3*y - 0.0062284034187645365*v*w*x3*y - 
     0.006895491178906826*lnbara*v*w*x3*y + 0.3009259577639667*w2*x3*y + 
     0.09784333799586425*lnbara*w2*x3*y - 0.05515167676319815*x4*y + 
     0.0026790848116713074*u*x4*y - 0.412831813275259*v*x4*y - 
     0.12512755974621306*lnbara*v*x4*y + 0.4179202471284635*w*x4*y + 
     0.12512755974621306*lnbara*w*x4*y + 0.007027953632525705*x5*y + 
     1.3191828527017075*y2 - 0.32294740959762247*u*y2 + 
     0.09916328431553666*u2*y2 - 0.022632876900369598*u3*y2 + 
     0.0025444588000576086*u4*y2 - 0.4287703318427488*v*y2 + 
     0.08548106272223456*u*v*y2 - 0.01563232680721393*u2*v*y2 + 
     0.0015419823403225055*u3*v*y2 + 0.14860391371313228*v2*y2 - 
     0.017479707619006156*u*v2*y2 + 0.0013887934717172665*u2*v2*y2 - 
     0.03593782918864228*v3*y2 + 0.001821220663909673*u*v3*y2 + 
     0.004169061579930593*v4*y2 - 0.4287703318427488*w*y2 + 
     0.08548106272223456*u*w*y2 - 0.01563232680721393*u2*w*y2 + 
     0.0015419823403225055*u3*w*y2 + 0.12927559000443167*v*w*y2 - 
     0.015383805607517485*u*v*w*y2 + 0.001229379566375842*u2*v*w*y2 - 
     0.028211376477209685*v2*w*y2 + 0.00146144490120512*u*v2*w*y2 + 
     0.0030453133225042724*v3*w*y2 + 0.14860391371313228*w2*y2 - 
     0.017479707619006156*u*w2*y2 + 0.0013887934717172665*u2*w2*y2 - 
     0.028211376477209685*v*w2*y2 + 0.00146144490120512*u*v*w2*y2 + 
     0.002763300406558978*v2*w2*y2 - 0.03593782918864228*w3*y2 + 
     0.001821220663909673*u*w3*y2 + 0.0030453133225042724*v*w3*y2 + 
     0.004169061579930593*w4*y2 - 0.4287703318427488*x*y2 + 
     0.08548106272223456*u*x*y2 - 0.01563232680721393*u2*x*y2 + 
     0.0015419823403225055*u3*x*y2 - 1.458515156634267*v*x*y2 - 
     0.48831400806030395*lnbara*v*x*y2 + 0.15508634310271682*u*v*x*y2 + 
     0.06121320376147517*lnbara*u*v*x*y2 + 0.060610376887719246*u2*v*x*
      y2 + 0.015697459289112238*lnbara*u2*v*x*y2 - 
     0.027023436237519106*v2*x*y2 + 0.009112399462646404*lnbara*v2*x*y2 - 
     0.1308877099919966*u*v2*x*y2 - 0.03812489698291478*lnbara*u*v2*x*
      y2 - 0.14302771932672184*v3*x*y2 - 0.041476677259681335*lnbara*v3*x*
      y2 + 1.6449854699938522*w*x*y2 + 0.48831400806030395*lnbara*w*x*y2 - 
     0.18273891541643567*u*w*x*y2 - 0.06121320376147517*lnbara*u*w*x*y2 - 
     0.05823691063225072*u2*w*x*y2 - 0.015697459289112238*lnbara*u2*w*x*
      y2 - 0.015907870172457157*v*w*x*y2 + 0.001111492150387925*u*v*w*x*
      y2 + 0.7234333761756083*v2*w*x*y2 + 0.20371729223242238*lnbara*v2*w*
      x*y2 - 0.008866331960675933*w2*x*y2 - 0.009112399462646404*lnbara*
      w2*x*y2 + 0.13337539195281198*u*w2*x*y2 + 
     0.03812489698291478*lnbara*u*w2*x*y2 - 0.7204819253477474*v*w2*x*
      y2 - 0.20371729223242238*lnbara*v*w2*x*y2 + 
     0.14667148092405935*w3*x*y2 + 0.041476677259681335*lnbara*w3*x*y2 + 
     0.14860391371313228*x2*y2 - 0.017479707619006156*u*x2*y2 + 
     0.0013887934717172665*u2*x2*y2 - 0.027023436237519106*v*x2*y2 + 
     0.009112399462646404*lnbara*v*x2*y2 - 0.1308877099919966*u*v*x2*y2 - 
     0.03812489698291478*lnbara*u*v*x2*y2 + 0.3378573257997137*v2*x2*
      y2 + 0.09448041654628751*lnbara*v2*x2*y2 - 
     0.008866331960675933*w*x2*y2 - 0.009112399462646404*lnbara*w*x2*y2 + 
     0.13337539195281198*u*w*x2*y2 + 0.03812489698291478*lnbara*u*w*x2*
      y2 - 0.4477982839144112*v*w*x2*y2 - 0.12659992801563436*lnbara*v*w*
      x2*y2 + 0.1141478259424908*w2*x2*y2 + 0.03211951146934684*lnbara*
      w2*x2*y2 - 0.03593782918864228*x3*y2 + 0.001821220663909673*u*x3*
      y2 - 0.14302771932672184*v*x3*y2 - 0.041476677259681335*lnbara*v*x3*
      y2 + 0.14667148092405935*w*x3*y2 + 0.041476677259681335*lnbara*w*x3*
      y2 + 0.004169061579930593*x4*y2 - 1.0795675579460708*y3 + 
     0.14286380309095198*u*y3 - 0.022632876900369598*u2*y3 + 
     0.0020957041129497137*u3*y3 + 0.19595873400524338*v*y3 - 
     0.020567226223597537*u*v*y3 + 0.0015419823403225055*u2*v*y3 - 
     0.03593782918864228*v2*y3 + 0.0017631106072740343*u*v2*y3 + 
     0.0035837624283894857*v3*y3 + 0.19595873400524338*w*y3 - 
     0.020567226223597537*u*w*y3 + 0.0015419823403225055*u2*w*y3 - 
     0.032835376049197604*v*w*y3 + 0.0016187608563898235*u*v*w*y3 + 
     0.0030453133225042724*v2*w*y3 - 0.03593782918864228*w2*y3 + 
     0.0017631106072740343*u*w2*y3 + 0.0030453133225042724*v*w2*y3 + 
     0.0035837624283894857*w3*y3 + 0.19595873400524338*x*y3 - 
     0.020567226223597537*u*x*y3 + 0.0015419823403225055*u2*x*y3 + 
     0.9672844851536025*v*x*y3 + 0.2929884048361824*lnbara*v*x*y3 + 
     0.0816735862630184*u*v*x*y3 + 0.019963821863986493*lnbara*u*v*x*y3 - 
     0.12145181321698113*v2*x*y3 - 0.037545222670016885*lnbara*v2*x*y3 - 
     1.013316985461217*w*x*y3 - 0.2929884048361824*lnbara*w*x*y3 - 
     0.07883727181036221*u*w*x*y3 - 0.019963821863986493*lnbara*u*w*x*y3 + 
     0.0017051473997301786*v*w*x*y3 + 0.1251536848709543*w2*x*y3 + 
     0.037545222670016885*lnbara*w2*x*y3 - 0.03593782918864228*x2*y3 + 
     0.0017631106072740343*u*x2*y3 - 0.12145181321698113*v*x2*y3 - 
     0.037545222670016885*lnbara*v*x2*y3 + 0.1251536848709543*w*x2*y3 + 
     0.037545222670016885*lnbara*w*x2*y3 + 0.0035837624283894857*x3*y3 + 
     0.5780009836815717*y4 - 0.03954823687169365*u*y4 + 
     0.0025444588000576086*u2*y4 - 0.05515167676319815*v*y4 + 
     0.0023754912548396108*u*v*y4 + 0.004169061579930593*v2*y4 - 
     0.05515167676319815*w*y4 + 0.0023754912548396108*u*w*y4 + 
     0.0039062667758325825*v*w*y4 + 0.004169061579930593*w2*y4 - 
     0.05515167676319815*x*y4 + 0.0023754912548396108*u*x*y4 - 
     0.14164027833703918*v*x*y4 - 0.040692834005025325*lnbara*v*x*y4 + 
     0.14703230574707543*w*x*y4 + 0.040692834005025325*lnbara*w*x*y4 + 
     0.004169061579930593*x2*y4 - 0.17916940468773146*y5 + 
     0.004991470850443998*u*y5 + 0.007027953632525705*v*y5 + 
     0.007027953632525705*w*y5 + 0.007027953632525705*x*y5 + 
     0.024344353217864104*y6 - 1.2020569031595942*z + 
     0.7006812478176802*u*z - 0.4287703318427488*u2*z + 
     0.19595873400524338*u3*z - 0.05515167676319815*u4*z + 
     0.007027953632525705*u5*z + 0.5691938191238357*v*z - 
     1.5183456802708015*u*v*z - 0.4340568960536035*lnbara*u*v*z - 
     0.6819050738442709*u2*v*z - 0.22352066577107735*lnbara*u2*v*z - 
     0.23316484280986582*u3*v*z - 0.05917269635600225*lnbara*u3*v*z - 
     0.017520814568225136*u4*v*z - 0.0052304795439763515*lnbara*u4*v*z + 
     0.9491518611469659*v2*z + 0.4340568960536035*lnbara*v2*z + 
     1.8772484534413458*u*v2*z + 0.564250670249024*lnbara*u*v2*z + 
     0.6715039360324119*u2*v2*z + 0.19748191093199324*lnbara*u2*v2*z + 
     0.0911441314262025*u3*v2*z + 0.02432329879100571*lnbara*u3*v2*z - 
     0.1848964780137795*v3*z - 0.12370155645114488*lnbara*v3*z - 
     1.3530173829875152*u*v3*z - 0.40622240013673455*lnbara*u*v3*z - 
     0.0821346574113807*u2*v3*z - 0.021986858303963977*lnbara*u2*v3*z + 
     0.3026348522637972*v4*z + 0.10644029308949739*lnbara*v4*z + 
     0.23469657548791326*u*v4*z + 0.06907962057473943*lnbara*u*v4*z - 
     0.03675623670681*v5*z - 0.013089806333713885*lnbara*v5*z + 
     0.7006812478176802*w*z + 0.9399232832849783*u*w*z + 
     0.4340568960536035*lnbara*u*w*z + 0.9029759522398169*u2*w*z + 
     0.22352066577107735*lnbara*u2*w*z + 0.17752427122833048*u3*w*z + 
     0.05917269635600225*lnbara*u3*w*z + 0.024106166155729025*u4*w*z + 
     0.0052304795439763515*lnbara*u4*w*z - 0.24624640952621318*v*w*z + 
     0.07362480745074675*u*v*w*z - 0.015907870172457157*u2*v*w*z + 
     0.0017051473997301786*u3*v*w*z - 5.0744264500354594*v2*w*z - 
     1.642900692638757*lnbara*v2*w*z + 1.138848970579144*u*v2*w*z + 
     0.36710906815157757*lnbara*u*v2*w*z - 0.5618212037342907*u2*v2*w*z - 
     0.16529892872504193*lnbara*u2*v2*w*z + 1.7597891705366386*v3*w*z + 
     0.5698593651894059*lnbara*v3*w*z + 0.02852820513246288*u*v3*w*z + 
     0.0022974655834007394*lnbara*u*v3*w*z - 0.5225454785602572*v4*w*z - 
     0.15634114484424524*lnbara*v4*w*z - 1.700869602587337*w2*z - 
     0.4340568960536035*lnbara*w2*z - 1.6624918007146796*u*w2*z - 
     0.564250670249024*lnbara*u*w2*z - 0.7171950201286277*u2*w2*z - 
     0.19748191093199324*lnbara*u2*w2*z - 0.08627759743978855*u3*w2*z - 
     0.02432329879100571*lnbara*u3*w2*z + 5.2517028011488085*v*w2*z + 
     1.642900692638757*lnbara*v*w2*z - 1.1689689937010432*u*v*w2*z - 
     0.36710906815157757*lnbara*u*v*w2*z + 0.5646118353233481*u2*v*w2*z + 
     0.16529892872504193*lnbara*u2*v*w2*z - 0.017479707619006156*v2*w2*
      z + 0.00130062566240163*u*v2*w2*z + 0.3249693774147304*v3*w2*z + 
     0.08664893308797812*lnbara*v3*w2*z + 0.5237190151099749*w3*z + 
     0.12370155645114488*lnbara*w3*z + 1.29961478071472*u*w3*z + 
     0.40622240013673455*lnbara*u*w3*z + 0.08694308134115901*u2*w3*z + 
     0.021986858303963977*lnbara*u2*w3*z - 1.803161592292574*v*w3*z - 
     0.5698593651894059*lnbara*v*w3*z - 0.025380422110629967*u*v*w3*z - 
     0.0022974655834007394*lnbara*u*v*w3*z - 0.3213850461435467*v2*w3*z - 
     0.08664893308797812*lnbara*v2*w3*z - 0.397334765898689*w4*z - 
     0.10644029308949739*lnbara*w4*z - 0.22841481745724104*u*w4*z - 
     0.06907962057473943*lnbara*u*w4*z + 0.5276000546267682*v*w4*z + 
     0.15634114484424524*lnbara*v*w4*z + 0.0487756611897797*w5*z + 
     0.013089806333713885*lnbara*w5*z + 0.7006812478176802*x*z - 
     0.262974857387689*u*x*z + 0.09467502496847112*u2*x*z - 
     0.023227304775276715*u3*x*z + 0.0027129425983649334*u4*x*z - 
     0.24624640952621318*v*x*z + 0.07362480745074675*u*v*x*z - 
     0.015907870172457157*u2*v*x*z + 0.0017051473997301786*u3*v*x*z - 
     3.366052674172739*v2*x*z - 1.0851422401340087*lnbara*v2*x*z + 
     2.244030329434903*u*v2*x*z + 0.6930690249609582*lnbara*u*v2*x*z - 
     0.5870531877615167*u2*v2*x*z - 0.1732603915082422*lnbara*u2*v2*x*z + 
     0.7184614571972799*v3*x*z + 0.24451250621135381*lnbara*v3*x*z - 
     0.07931329913612384*u*v3*x*z - 0.03204115805743145*lnbara*u*v3*x*z - 
     0.31472121563065336*v4*x*z - 0.09290936333831616*lnbara*v4*x*z - 
     0.24624640952621318*w*x*z + 0.07362480745074675*u*w*x*z - 
     0.015907870172457157*u2*w*x*z + 0.0017051473997301786*u3*w*x*z + 
     6.969042259795925*v*w*x*z + 2.1702844802680175*lnbara*v*w*x*z - 
     4.531041378633713*u*v*w*x*z - 1.3861380499219165*lnbara*u*v*w*x*z + 
     1.1782060481677972*u2*v*w*x*z + 0.3465207830164844*lnbara*u2*v*w*x*z - 
     0.752469102785154*v2*w*x*z - 0.24451250621135381*lnbara*v2*w*x*z + 
     0.08204355214290159*u*v2*w*x*z + 0.03204115805743145*lnbara*u*v2*w*x*
      z + 0.7415394744753414*v3*w*x*z + 0.2104909159856026*lnbara*v3*w*x*
      z - 3.366052674172739*w2*x*z - 1.0851422401340087*lnbara*w2*x*z + 
     2.244030329434903*u*w2*x*z + 0.6930690249609582*lnbara*u*w2*x*z - 
     0.5870531877615167*u2*w2*x*z - 0.1732603915082422*lnbara*u2*w2*x*z - 
     0.752469102785154*v*w2*x*z - 0.24451250621135381*lnbara*v*w2*x*z + 
     0.08204355214290159*u*v*w2*x*z + 0.03204115805743145*lnbara*u*v*w2*x*
      z - 0.8449111212201761*v2*w2*x*z - 0.23516310529457285*lnbara*v2*w2*
      x*z + 0.7184614571972799*w3*x*z + 0.24451250621135381*lnbara*w3*x*z - 
     0.07931329913612384*u*w3*x*z - 0.03204115805743145*lnbara*u*w3*x*z + 
     0.7415394744753414*v*w3*x*z + 0.2104909159856026*lnbara*v*w3*x*z - 
     0.31472121563065336*w4*x*z - 0.09290936333831616*lnbara*w4*x*z - 
     0.4287703318427488*x2*z + 0.09467502496847112*u*x2*z - 
     0.01841006057918888*u2*x2*z + 0.001880650990063473*u3*x2*z + 
     5.2517028011488085*v*x2*z + 1.642900692638757*lnbara*v*x2*z - 
     1.1689689937010432*u*v*x2*z - 0.36710906815157757*lnbara*u*v*x2*z + 
     0.5646118353233481*u2*v*x2*z + 0.16529892872504193*lnbara*u2*v*x2*
      z - 0.3972223492383506*v2*x2*z - 0.1465559850060673*lnbara*v2*x2*
      z - 0.20929117997162128*u*v2*x2*z - 0.06205837953332923*lnbara*u*v2*
      x2*z + 0.5112934663833444*v3*x2*z + 0.14720183048123822*lnbara*v3*
      x2*z - 5.0744264500354594*w*x2*z - 1.642900692638757*lnbara*w*x2*z + 
     1.138848970579144*u*w*x2*z + 0.36710906815157757*lnbara*u*w*x2*z - 
     0.5618212037342907*u2*w*x2*z - 0.16529892872504193*lnbara*u2*w*x2*
      z - 0.752469102785154*v*w*x2*z - 0.24451250621135381*lnbara*v*w*x2*
      z + 0.08204355214290159*u*v*w*x2*z + 0.03204115805743145*lnbara*u*v*w*
      x2*z - 1.0464511532205378*v2*w*x2*z - 0.2896065773402419*lnbara*v2*
      w*x2*z + 1.103138998233008*w2*x2*z + 0.39106849121742115*lnbara*w2*
      x2*z + 0.13104994444668858*u*w2*x2*z + 0.030017221475897783*lnbara*u*
      w2*x2*z + 1.1098745724532006*v*w2*x2*z + 0.31182123714391863*lnbara*
      v*w2*x2*z - 0.569080649680904*w3*x2*z - 0.16941649028491496*lnbara*
      w3*x2*z + 0.19595873400524338*x3*z - 0.023227304775276715*u*x3*z + 
     0.001880650990063473*u2*x3*z - 1.803161592292574*v*x3*z - 
     0.5698593651894059*lnbara*v*x3*z - 0.025380422110629967*u*v*x3*z - 
     0.0022974655834007394*lnbara*u*v*x3*z - 0.25301233477098106*v2*x3*z - 
     0.062498180515692975*lnbara*v2*x3*z + 1.7597891705366386*w*x3*z + 
     0.5698593651894059*lnbara*w*x3*z + 0.02852820513246288*u*w*x3*z + 
     0.0022974655834007394*lnbara*u*w*x3*z + 0.7415394744753414*v*w*x3*z + 
     0.2104909159856026*lnbara*v*w*x3*z - 0.4837702578695747*w2*x3*z - 
     0.14799273546990963*lnbara*w2*x3*z - 0.05515167676319815*x4*z + 
     0.0027129425983649334*u*x4*z + 0.5276000546267682*v*x4*z + 
     0.15634114484424524*lnbara*v*x4*z - 0.5225454785602572*w*x4*z - 
     0.15634114484424524*lnbara*w*x4*z + 0.007027953632525705*x5*z + 
     0.7006812478176802*y*z - 0.24624640952621318*u*y*z + 
     0.08548106272223456*u2*y*z - 0.020567226223597537*u3*y*z + 
     0.0023754912548396108*u4*y*z - 4.062544221759978*v*y*z - 
     1.3021706881608104*lnbara*v*y*z - 1.7373962401659306*u*v*y*z - 
     0.5077906612931307*lnbara*u*v*y*z - 0.5908953276665939*u2*v*y*z - 
     0.16536822311781155*lnbara*u2*v*y*z - 0.064738719421504*u3*v*y*z - 
     0.017198580632115178*lnbara*u3*v*y*z + 2.621952087639167*v2*y*z + 
     0.8594186774137386*lnbara*v2*y*z + 1.2616217958989164*u*v2*y*z + 
     0.3671860440862087*lnbara*u*v2*y*z + 0.22484058072370788*u2*v2*y*z + 
     0.06208972089809539*lnbara*u2*v2*y*z - 2.2491286389959257*v3*y*z - 
     0.6865302179753614*lnbara*v3*y*z - 0.18944564848149062*u*v3*y*z - 
     0.05112899772410982*lnbara*u*v3*y*z + 0.4179202471284635*v4*y*z + 
     0.12512755974621306*lnbara*v4*y*z + 3.553322954846076*w*y*z + 
     1.3021706881608104*lnbara*w*y*z + 1.8769958336226564*u*w*y*z + 
     0.5077906612931307*lnbara*u*w*y*z + 0.5620711026947999*u2*w*y*z + 
     0.16536822311781155*lnbara*u2*w*y*z + 0.06775115910844728*u3*w*y*z + 
     0.017198580632115178*lnbara*u3*w*y*z + 0.07362480745074675*v*w*y*z - 
     0.012213108548872703*u*v*w*y*z + 0.001111492150387925*u2*v*w*y*z + 
     2.311786939439009*v2*w*y*z + 0.6714897453342303*lnbara*v2*w*y*z - 
     0.4475358708690333*u*v2*w*y*z - 0.13547920846092296*lnbara*u*v2*w*y*
      z - 0.12207317499372534*v3*w*y*z - 0.034595562617565336*lnbara*v3*w*y*
      z - 2.4417959999484613*w2*y*z - 0.8594186774137386*lnbara*w2*y*z - 
     1.2909700854356503*u*w2*y*z - 0.3671860440862087*lnbara*u*w2*y*z - 
     0.222192079524089*u2*w2*y*z - 0.06208972089809539*lnbara*u2*w2*y*z - 
     2.343078615218984*v*w2*y*z - 0.6714897453342303*lnbara*v*w2*y*z + 
     0.4498241458617747*u*v*w2*y*z + 0.13547920846092296*lnbara*u*v*w2*y*
      z + 0.00146144490120512*v2*w2*y*z + 2.2053341079970514*w3*y*z + 
     0.6865302179753614*lnbara*w3*y*z + 0.19254447471177424*u*w3*y*z + 
     0.05112899772410982*lnbara*u*w3*y*z + 0.12539708324984536*v*w3*y*z + 
     0.034595562617565336*lnbara*v*w3*y*z - 0.412831813275259*w4*y*z - 
     0.12512755974621306*lnbara*w4*y*z - 0.3321759874596101*x*y*z + 
     0.07362480745074675*u*x*y*z - 0.01421215294944225*u2*x*y*z + 
     0.001442635622102732*u3*x*y*z + 0.07362480745074675*v*x*y*z - 
     0.012213108548872703*u*v*x*y*z + 0.001111492150387925*u2*v*x*y*z + 
     1.6733264521145743*v2*x*y*z + 0.499165430461644*lnbara*v2*x*y*z - 
     0.7575451375965339*u*v2*x*y*z - 0.21838360971474802*lnbara*u*v2*x*y*
      z - 0.0062284034187645365*v3*x*y*z - 0.006895491178906826*lnbara*v3*x*
      y*z + 0.07362480745074675*w*x*y*z - 0.012213108548872703*u*w*x*y*z + 
     0.001111492150387925*u2*w*x*y*z - 3.3872903186769054*v*w*x*y*z - 
     0.998330860923288*lnbara*v*w*x*y*z + 1.518280865154906*u*v*w*x*y*z + 
     0.43676721942949603*lnbara*u*v*w*x*y*z + 0.008782531191255194*v2*w*x*y*
      z + 0.006895491178906826*lnbara*v2*w*x*y*z + 
     1.6733264521145743*w2*x*y*z + 0.499165430461644*lnbara*w2*x*y*z - 
     0.7575451375965339*u*w2*x*y*z - 0.21838360971474802*lnbara*u*w2*x*y*
      z + 0.008782531191255194*v*w2*x*y*z + 0.006895491178906826*lnbara*v*
      w2*x*y*z - 0.0062284034187645365*w3*x*y*z - 
     0.006895491178906826*lnbara*w3*x*y*z + 0.12927559000443167*x2*y*z - 
     0.015907870172457157*u*x2*y*z + 0.00130062566240163*u2*x2*y*z - 
     2.343078615218984*v*x2*y*z - 0.6714897453342303*lnbara*v*x2*y*z + 
     0.4498241458617747*u*v*x2*y*z + 0.13547920846092296*lnbara*u*v*x2*y*
      z + 0.029245523131395473*v2*x2*y*z + 0.008946886688429884*lnbara*v2*
      x2*y*z + 2.311786939439009*w*x2*y*z + 0.6714897453342303*lnbara*w*x2*
      y*z - 0.4475358708690333*u*w*x2*y*z - 0.13547920846092296*lnbara*u*w*
      x2*y*z + 0.008782531191255194*v*w*x2*y*z + 0.006895491178906826*
      lnbara*v*w*x2*y*z - 0.034386556943485275*w2*x2*y*z - 
     0.01584237786733671*lnbara*w2*x2*y*z - 0.032835376049197604*x3*y*z + 
     0.0017051473997301786*u*x3*y*z + 0.12539708324984536*v*x3*y*z + 
     0.034595562617565336*lnbara*v*x3*y*z - 0.12207317499372534*w*x3*y*z - 
     0.034595562617565336*lnbara*w*x3*y*z + 0.0039062667758325825*x4*y*z - 
     0.4287703318427488*y2*z + 0.08548106272223456*u*y2*z - 
     0.01563232680721393*u2*y2*z + 0.0015419823403225055*u3*y2*z + 
     1.6449854699938522*v*y2*z + 0.48831400806030395*lnbara*v*y2*z - 
     0.18273891541643567*u*v*y2*z - 0.06121320376147517*lnbara*u*v*y2*z - 
     0.05823691063225072*u2*v*y2*z - 0.015697459289112238*lnbara*u2*v*y2*
      z - 0.008866331960675933*v2*y2*z - 0.009112399462646404*lnbara*v2*
      y2*z + 0.13337539195281198*u*v2*y2*z + 0.03812489698291478*lnbara*u*
      v2*y2*z + 0.14667148092405935*v3*y2*z + 0.041476677259681335*lnbara*
      v3*y2*z - 1.458515156634267*w*y2*z - 0.48831400806030395*lnbara*w*
      y2*z + 0.15508634310271682*u*w*y2*z + 0.06121320376147517*lnbara*u*w*
      y2*z + 0.060610376887719246*u2*w*y2*z + 0.015697459289112238*lnbara*
      u2*w*y2*z - 0.015907870172457157*v*w*y2*z + 
     0.001111492150387925*u*v*w*y2*z - 0.7204819253477474*v2*w*y2*z - 
     0.20371729223242238*lnbara*v2*w*y2*z - 0.027023436237519106*w2*y2*
      z + 0.009112399462646404*lnbara*w2*y2*z - 0.1308877099919966*u*w2*
      y2*z - 0.03812489698291478*lnbara*u*w2*y2*z + 
     0.7234333761756083*v*w2*y2*z + 0.20371729223242238*lnbara*v*w2*y2*
      z - 0.14302771932672184*w3*y2*z - 0.041476677259681335*lnbara*w3*y2*
      z + 0.12927559000443167*x*y2*z - 0.015383805607517485*u*x*y2*z + 
     0.001229379566375842*u2*x*y2*z - 0.015907870172457157*v*x*y2*z + 
     0.001111492150387925*u*v*x*y2*z - 0.4477982839144112*v2*x*y2*z - 
     0.12659992801563436*lnbara*v2*x*y2*z - 0.015907870172457157*w*x*y2*
      z + 0.001111492150387925*u*w*x*y2*z + 0.8993746019959791*v*w*x*y2*z + 
     0.2531998560312687*lnbara*v*w*x*y2*z - 0.4477982839144112*w2*x*y2*z - 
     0.12659992801563436*lnbara*w2*x*y2*z - 0.028211376477209685*x2*y2*
      z + 0.00146144490120512*u*x2*y2*z + 0.7234333761756083*v*x2*y2*z + 
     0.20371729223242238*lnbara*v*x2*y2*z - 0.7204819253477474*w*x2*y2*
      z - 0.20371729223242238*lnbara*w*x2*y2*z + 0.0030453133225042724*x3*
      y2*z + 0.19595873400524338*y3*z - 0.020567226223597537*u*y3*z + 
     0.0015419823403225055*u2*y3*z - 1.013316985461217*v*y3*z - 
     0.2929884048361824*lnbara*v*y3*z - 0.07883727181036221*u*v*y3*z - 
     0.019963821863986493*lnbara*u*v*y3*z + 0.1251536848709543*v2*y3*z + 
     0.037545222670016885*lnbara*v2*y3*z + 0.9672844851536025*w*y3*z + 
     0.2929884048361824*lnbara*w*y3*z + 0.0816735862630184*u*w*y3*z + 
     0.019963821863986493*lnbara*u*w*y3*z + 0.0017051473997301786*v*w*y3*
      z - 0.12145181321698113*w2*y3*z - 0.037545222670016885*lnbara*w2*y3*
      z - 0.032835376049197604*x*y3*z + 0.0016187608563898235*u*x*y3*z + 
     0.0017051473997301786*v*x*y3*z + 0.0017051473997301786*w*x*y3*z + 
     0.0030453133225042724*x2*y3*z - 0.05515167676319815*y4*z + 
     0.0023754912548396108*u*y4*z + 0.14703230574707543*v*y4*z + 
     0.040692834005025325*lnbara*v*y4*z - 0.14164027833703918*w*y4*z - 
     0.040692834005025325*lnbara*w*y4*z + 0.0039062667758325825*x*y4*z + 
     0.007027953632525705*y5*z + 1.3191828527017075*z2 - 
     0.4287703318427488*u*z2 + 0.14860391371313228*u2*z2 - 
     0.03593782918864228*u3*z2 + 0.004169061579930593*u4*z2 + 
     0.9491518611469659*v*z2 + 0.4340568960536035*lnbara*v*z2 + 
     1.8772484534413458*u*v*z2 + 0.564250670249024*lnbara*u*v*z2 + 
     0.6715039360324119*u2*v*z2 + 0.19748191093199324*lnbara*u2*v*z2 + 
     0.0911441314262025*u3*v*z2 + 0.02432329879100571*lnbara*u3*v*z2 + 
     0.8242562168640629*v2*z2 + 0.18665378315131373*lnbara*v2*z2 - 
     2.8414768629489666*u*v2*z2 - 0.8660810495752876*lnbara*u*v2*z2 + 
     0.11275072634761112*u2*v2*z2 + 0.035989183471035156*lnbara*u2*v2*
      z2 + 0.14157447061094622*v3*z2 + 0.05001134362781823*lnbara*v3*
      z2 + 0.4093655691718876*u*v3*z2 + 0.12737516232192267*lnbara*u*v3*
      z2 + 0.03289810142239334*v4*z2 + 0.007026033220340475*lnbara*v4*
      z2 - 1.700869602587337*w*z2 - 0.4340568960536035*lnbara*w*z2 - 
     1.6624918007146796*u*w*z2 - 0.564250670249024*lnbara*u*w*z2 - 
     0.7171950201286277*u2*w*z2 - 0.19748191093199324*lnbara*u2*w*z2 - 
     0.08627759743978855*u3*w*z2 - 0.02432329879100571*lnbara*u3*w*z2 - 
     3.366052674172739*v*w*z2 - 1.0851422401340087*lnbara*v*w*z2 + 
     2.244030329434903*u*v*w*z2 + 0.6930690249609582*lnbara*u*v*w*z2 - 
     0.5870531877615167*u2*v*w*z2 - 0.1732603915082422*lnbara*u2*v*w*z2 + 
     1.103138998233008*v2*w*z2 + 0.39106849121742115*lnbara*v2*w*z2 + 
     0.13104994444668858*u*v2*w*z2 + 0.030017221475897783*lnbara*u*v2*w*
      z2 - 0.4837702578695747*v3*w*z2 - 0.14799273546990963*lnbara*v3*w*
      z2 + 2.8750447180595793*w2*z2 + 0.898488456982695*lnbara*w2*z2 + 
     0.5382190246221227*u*w2*z2 + 0.1730120246143294*lnbara*u*w2*z2 + 
     0.479916000193387*u2*w2*z2 + 0.13727120803720705*lnbara*u2*w2*z2 - 
     0.3972223492383506*v*w2*z2 - 0.1465559850060673*lnbara*v*w2*z2 - 
     0.20929117997162128*u*v*w2*z2 - 0.06205837953332923*lnbara*u*v*w2*
      z2 + 0.4244378822310122*v2*w2*z2 + 0.11758155264728642*lnbara*v2*
      w2*z2 - 0.9391738601208355*w3*z2 - 0.29452384983917207*lnbara*w3*
      z2 - 0.3238462135165472*u*w3*z2 - 0.09533400426449122*lnbara*u*w3*
      z2 - 0.25301233477098106*v*w3*z2 - 0.062498180515692975*lnbara*v*w3*
      z2 + 0.29091212584308784*w4*z2 + 0.08588333011797569*lnbara*w4*
      z2 - 0.4287703318427488*x*z2 + 0.09467502496847112*u*x*z2 - 
     0.01841006057918888*u2*x*z2 + 0.001880650990063473*u3*x*z2 - 
     5.0744264500354594*v*x*z2 - 1.642900692638757*lnbara*v*x*z2 + 
     1.138848970579144*u*v*x*z2 + 0.36710906815157757*lnbara*u*v*x*z2 - 
     0.5618212037342907*u2*v*x*z2 - 0.16529892872504193*lnbara*u2*v*x*
      z2 + 1.103138998233008*v2*x*z2 + 0.39106849121742115*lnbara*v2*x*
      z2 + 0.13104994444668858*u*v2*x*z2 + 0.030017221475897783*lnbara*u*
      v2*x*z2 - 0.569080649680904*v3*x*z2 - 0.16941649028491496*lnbara*
      v3*x*z2 + 5.2517028011488085*w*x*z2 + 1.642900692638757*lnbara*w*x*
      z2 - 1.1689689937010432*u*w*x*z2 - 0.36710906815157757*lnbara*u*w*x*
      z2 + 0.5646118353233481*u2*w*x*z2 + 0.16529892872504193*lnbara*u2*w*
      x*z2 - 0.752469102785154*v*w*x*z2 - 0.24451250621135381*lnbara*v*w*x*
      z2 + 0.08204355214290159*u*v*w*x*z2 + 0.03204115805743145*lnbara*u*v*
      w*x*z2 + 1.1098745724532006*v2*w*x*z2 + 0.31182123714391863*lnbara*
      v2*w*x*z2 - 0.3972223492383506*w2*x*z2 - 0.1465559850060673*lnbara*
      w2*x*z2 - 0.20929117997162128*u*w2*x*z2 - 
     0.06205837953332923*lnbara*u*w2*x*z2 - 1.0464511532205378*v*w2*x*
      z2 - 0.2896065773402419*lnbara*v*w2*x*z2 + 
     0.5112934663833444*w3*x*z2 + 0.14720183048123822*lnbara*w3*x*z2 + 
     0.14860391371313228*x2*z2 - 0.01841006057918888*u*x2*z2 + 
     0.0015174486936744638*u2*x2*z2 - 0.017479707619006156*v*x2*z2 + 
     0.00130062566240163*u*v*x2*z2 + 0.4244378822310122*v2*x2*z2 + 
     0.11758155264728642*lnbara*v2*x2*z2 - 0.017479707619006156*w*x2*
      z2 + 0.00130062566240163*u*w*x2*z2 - 0.8449111212201761*v*w*x2*
      z2 - 0.23516310529457285*lnbara*v*w*x2*z2 + 
     0.4244378822310122*w2*x2*z2 + 0.11758155264728642*lnbara*w2*x2*
      z2 - 0.03593782918864228*x3*z2 + 0.001880650990063473*u*x3*z2 - 
     0.3213850461435467*v*x3*z2 - 0.08664893308797812*lnbara*v*x3*z2 + 
     0.3249693774147304*w*x3*z2 + 0.08664893308797812*lnbara*w*x3*z2 + 
     0.004169061579930593*x4*z2 - 0.4287703318427488*y*z2 + 
     0.09179528839111438*u*y*z2 - 0.017479707619006156*u2*y*z2 + 
     0.0017631106072740343*u3*y*z2 + 2.621952087639167*v*y*z2 + 
     0.8594186774137386*lnbara*v*y*z2 + 1.2616217958989164*u*v*y*z2 + 
     0.3671860440862087*lnbara*u*v*y*z2 + 0.22484058072370788*u2*v*y*z2 + 
     0.06208972089809539*lnbara*u2*v*y*z2 - 1.8627603909108525*v2*y*z2 - 
     0.5894315609290915*lnbara*v2*y*z2 - 0.10231789367095857*u*v2*y*z2 - 
     0.030182734250114304*lnbara*u*v2*y*z2 + 0.3009259577639667*v3*y*z2 + 
     0.09784333799586425*lnbara*v3*y*z2 - 2.4417959999484613*w*y*z2 - 
     0.8594186774137386*lnbara*w*y*z2 - 1.2909700854356503*u*w*y*z2 - 
     0.3671860440862087*lnbara*u*w*y*z2 - 0.222192079524089*u2*w*y*z2 - 
     0.06208972089809539*lnbara*u2*w*y*z2 + 1.6733264521145743*v*w*y*z2 + 
     0.499165430461644*lnbara*v*w*y*z2 - 0.7575451375965339*u*v*w*y*z2 - 
     0.21838360971474802*lnbara*u*v*w*y*z2 - 0.034386556943485275*v2*w*y*
      z2 - 0.01584237786733671*lnbara*v2*w*y*z2 + 
     0.1411793984604333*w2*y*z2 + 0.09026613046744748*lnbara*w2*y*z2 + 
     0.8636086160336289*u*w2*y*z2 + 0.24856634396486232*lnbara*u*w2*y*
      z2 + 0.029245523131395473*v*w2*y*z2 + 0.008946886688429884*lnbara*v*
      w2*y*z2 - 0.28983228539271344*w3*y*z2 - 0.09094784681695743*lnbara*
      w3*y*z2 + 0.12927559000443167*x*y*z2 - 0.015907870172457157*u*x*y*
      z2 + 0.00130062566240163*u2*x*y*z2 + 2.311786939439009*v*x*y*z2 + 
     0.6714897453342303*lnbara*v*x*y*z2 - 0.4475358708690333*u*v*x*y*z2 - 
     0.13547920846092296*lnbara*u*v*x*y*z2 - 0.034386556943485275*v2*x*y*
      z2 - 0.01584237786733671*lnbara*v2*x*y*z2 - 
     2.343078615218984*w*x*y*z2 - 0.6714897453342303*lnbara*w*x*y*z2 + 
     0.4498241458617747*u*w*x*y*z2 + 0.13547920846092296*lnbara*u*w*x*y*
      z2 + 0.008782531191255194*v*w*x*y*z2 + 0.006895491178906826*lnbara*v*
      w*x*y*z2 + 0.029245523131395473*w2*x*y*z2 + 
     0.008946886688429884*lnbara*w2*x*y*z2 - 0.028211376477209685*x2*y*
      z2 + 0.001490005926655855*u*x2*y*z2 + 0.00146144490120512*v*x2*y*
      z2 + 0.00146144490120512*w*x2*y*z2 + 0.0030453133225042724*x3*y*
      z2 + 0.14860391371313228*y2*z2 - 0.017479707619006156*u*y2*z2 + 
     0.0013887934717172665*u2*y2*z2 - 0.008866331960675933*v*y2*z2 - 
     0.009112399462646404*lnbara*v*y2*z2 + 0.13337539195281198*u*v*y2*
      z2 + 0.03812489698291478*lnbara*u*v*y2*z2 + 
     0.1141478259424908*v2*y2*z2 + 0.03211951146934684*lnbara*v2*y2*
      z2 - 0.027023436237519106*w*y2*z2 + 0.009112399462646404*lnbara*w*
      y2*z2 - 0.1308877099919966*u*w*y2*z2 - 0.03812489698291478*lnbara*u*
      w*y2*z2 - 0.4477982839144112*v*w*y2*z2 - 0.12659992801563436*lnbara*
      v*w*y2*z2 + 0.3378573257997137*w2*y2*z2 + 
     0.09448041654628751*lnbara*w2*y2*z2 - 0.028211376477209685*x*y2*
      z2 + 0.00146144490120512*u*x*y2*z2 - 0.7204819253477474*v*x*y2*
      z2 - 0.20371729223242238*lnbara*v*x*y2*z2 + 
     0.7234333761756083*w*x*y2*z2 + 0.20371729223242238*lnbara*w*x*y2*
      z2 + 0.002763300406558978*x2*y2*z2 - 0.03593782918864228*y3*z2 + 
     0.0017631106072740343*u*y3*z2 + 0.1251536848709543*v*y3*z2 + 
     0.037545222670016885*lnbara*v*y3*z2 - 0.12145181321698113*w*y3*z2 - 
     0.037545222670016885*lnbara*w*y3*z2 + 0.0030453133225042724*x*y3*
      z2 + 0.004169061579930593*y4*z2 - 1.0795675579460708*z3 + 
     0.19595873400524338*u*z3 - 0.03593782918864228*u2*z3 + 
     0.0035837624283894857*u3*z3 - 0.1848964780137795*v*z3 - 
     0.12370155645114488*lnbara*v*z3 - 1.3530173829875152*u*v*z3 - 
     0.40622240013673455*lnbara*u*v*z3 - 0.0821346574113807*u2*v*z3 - 
     0.021986858303963977*lnbara*u2*v*z3 + 0.14157447061094622*v2*z3 + 
     0.05001134362781823*lnbara*v2*z3 + 0.4093655691718876*u*v2*z3 + 
     0.12737516232192267*lnbara*u*v2*z3 + 0.02313083477136829*v3*z3 + 
     0.004691749642619168*lnbara*v3*z3 + 0.5237190151099749*w*z3 + 
     0.12370155645114488*lnbara*w*z3 + 1.29961478071472*u*w*z3 + 
     0.40622240013673455*lnbara*u*w*z3 + 0.08694308134115901*u2*w*z3 + 
     0.021986858303963977*lnbara*u2*w*z3 + 0.7184614571972799*v*w*z3 + 
     0.24451250621135381*lnbara*v*w*z3 - 0.07931329913612384*u*v*w*z3 - 
     0.03204115805743145*lnbara*u*v*w*z3 - 0.569080649680904*v2*w*z3 - 
     0.16941649028491496*lnbara*v2*w*z3 - 0.9391738601208355*w2*z3 - 
     0.29452384983917207*lnbara*w2*z3 - 0.3238462135165472*u*w2*z3 - 
     0.09533400426449122*lnbara*u*w2*z3 + 0.5112934663833444*v*w2*z3 + 
     0.14720183048123822*lnbara*v*w2*z3 + 0.043640908015127135*w3*z3 + 
     0.01752291016105757*lnbara*w3*z3 + 0.19595873400524338*x*z3 - 
     0.023227304775276715*u*x*z3 + 0.001880650990063473*u2*x*z3 + 
     1.7597891705366386*v*x*z3 + 0.5698593651894059*lnbara*v*x*z3 + 
     0.02852820513246288*u*v*x*z3 + 0.0022974655834007394*lnbara*u*v*x*z3 - 
     0.4837702578695747*v2*x*z3 - 0.14799273546990963*lnbara*v2*x*z3 - 
     1.803161592292574*w*x*z3 - 0.5698593651894059*lnbara*w*x*z3 - 
     0.025380422110629967*u*w*x*z3 - 0.0022974655834007394*lnbara*u*w*x*
      z3 + 0.7415394744753414*v*w*x*z3 + 0.2104909159856026*lnbara*v*w*x*
      z3 - 0.25301233477098106*w2*x*z3 - 0.062498180515692975*lnbara*w2*x*
      z3 - 0.03593782918864228*x2*z3 + 0.001880650990063473*u*x2*z3 + 
     0.3249693774147304*v*x2*z3 + 0.08664893308797812*lnbara*v*x2*z3 - 
     0.3213850461435467*w*x2*z3 - 0.08664893308797812*lnbara*w*x2*z3 + 
     0.0035837624283894857*x3*z3 + 0.19595873400524338*y*z3 - 
     0.022805195532337737*u*y*z3 + 0.001821220663909673*u2*y*z3 - 
     2.2491286389959257*v*y*z3 - 0.6865302179753614*lnbara*v*y*z3 - 
     0.18944564848149062*u*v*y*z3 - 0.05112899772410982*lnbara*u*v*y*z3 + 
     0.3009259577639667*v2*y*z3 + 0.09784333799586425*lnbara*v2*y*z3 + 
     2.2053341079970514*w*y*z3 + 0.6865302179753614*lnbara*w*y*z3 + 
     0.19254447471177424*u*w*y*z3 + 0.05112899772410982*lnbara*u*w*y*z3 - 
     0.0062284034187645365*v*w*y*z3 - 0.006895491178906826*lnbara*v*w*y*
      z3 - 0.28983228539271344*w2*y*z3 - 0.09094784681695743*lnbara*w2*y*
      z3 - 0.032835376049197604*x*y*z3 + 0.0017051473997301786*u*x*y*z3 - 
     0.12207317499372534*v*x*y*z3 - 0.034595562617565336*lnbara*v*x*y*z3 + 
     0.12539708324984536*w*x*y*z3 + 0.034595562617565336*lnbara*w*x*y*z3 + 
     0.0030453133225042724*x2*y*z3 - 0.03593782918864228*y2*z3 + 
     0.001821220663909673*u*y2*z3 + 0.14667148092405935*v*y2*z3 + 
     0.041476677259681335*lnbara*v*y2*z3 - 0.14302771932672184*w*y2*z3 - 
     0.041476677259681335*lnbara*w*y2*z3 + 0.0030453133225042724*x*y2*
      z3 + 0.0035837624283894857*y3*z3 + 0.5780009836815717*z4 - 
     0.05515167676319815*u*z4 + 0.004169061579930593*u2*z4 + 
     0.3026348522637972*v*z4 + 0.10644029308949739*lnbara*v*z4 + 
     0.23469657548791326*u*v*z4 + 0.06907962057473943*lnbara*u*v*z4 + 
     0.03289810142239334*v2*z4 + 0.007026033220340475*lnbara*v2*z4 - 
     0.397334765898689*w*z4 - 0.10644029308949739*lnbara*w*z4 - 
     0.22841481745724104*u*w*z4 - 0.06907962057473943*lnbara*u*w*z4 - 
     0.31472121563065336*v*w*z4 - 0.09290936333831616*lnbara*v*w*z4 + 
     0.29091212584308784*w2*z4 + 0.08588333011797569*lnbara*w2*z4 - 
     0.05515167676319815*x*z4 + 0.0027129425983649334*u*x*z4 - 
     0.5225454785602572*v*x*z4 - 0.15634114484424524*lnbara*v*x*z4 + 
     0.5276000546267682*w*x*z4 + 0.15634114484424524*lnbara*w*x*z4 + 
     0.004169061579930593*x2*z4 - 0.05515167676319815*y*z4 + 
     0.0026790848116713074*u*y*z4 + 0.4179202471284635*v*y*z4 + 
     0.12512755974621306*lnbara*v*y*z4 - 0.412831813275259*w*y*z4 - 
     0.12512755974621306*lnbara*w*y*z4 + 0.0039062667758325825*x*y*z4 + 
     0.004169061579930593*y2*z4 - 0.17916940468773146*z5 + 
     0.007027953632525705*u*z5 - 0.03675623670681*v*z5 - 
     0.013089806333713885*lnbara*v*z5 + 0.0487756611897797*w*z5 + 
     0.013089806333713885*lnbara*w*z5 + 0.007027953632525705*x*z5 + 
     0.007027953632525705*y*z5 + 0.024344353217864104*z6;
 
  Hexp[7] = 1.0303344884225094 - 1.2020569031595942*u + 
     1.5830194232420491*u2 - 1.6193513369191062*u3 + 
     1.1560019673631434*u4 - 0.5375082140631944*u5 + 
     0.14606611930718463*u6 - 0.017602106974984624*u7 - 
     1.2020569031595942*v + 0.8408174973812163*u*v - 
     0.6431554977641232*u2*v + 0.39191746801048677*u3*v - 
     0.16545503028959443*u4*v + 0.04216772179515423*u5*v - 
     0.004855915459192057*u6*v + 1.5830194232420491*v2 - 
     0.6431554977641232*u*v2 + 0.29720782742626456*u2*v2 - 
     0.10781348756592685*u3*v2 + 0.02501436947958356*u4*v2 - 
     0.0027042708881657167*u5*v2 - 1.6193513369191062*v3 + 
     0.39191746801048677*u*v3 - 0.10781348756592685*u2*v3 + 
     0.021502574570336913*u3*v3 - 0.002116028409986158*u4*v3 + 
     1.1560019673631434*v4 - 0.16545503028959443*u*v4 + 
     0.02501436947958356*u2*v4 - 0.002116028409986158*u3*v4 - 
     0.5375082140631944*v5 + 0.04216772179515423*u*v5 - 
     0.0027042708881657167*u2*v5 + 0.14606611930718463*v6 - 
     0.004855915459192057*u*v6 - 0.017602106974984624*v7 - 
     1.2020569031595942*w + 0.8408174973812163*u*w - 
     0.6431554977641232*u2*w + 0.39191746801048677*u3*w - 
     0.16545503028959443*u4*w + 0.04216772179515423*u5*w - 
     0.004855915459192057*u6*w + 0.8408174973812163*v*w - 
     0.3944622860815335*u*v*w + 0.18935004993694224*u2*v*w - 
     0.06968191432583015*u3*v*w + 0.0162776555901896*u4*v*w - 
     0.0017664062094713614*u5*v*w - 0.6431554977641232*v2*w + 
     0.18935004993694224*u*v2*w - 0.05523018173756664*u2*v2*w + 
     0.011283905940380838*u3*v2*w - 0.0011239361348986583*u4*v2*w + 
     0.39191746801048677*v3*w - 0.06968191432583015*u*v3*w + 
     0.011283905940380838*u2*v3*w - 0.0009831878162066224*u3*v3*w - 
     0.16545503028959443*v4*w + 0.0162776555901896*u*v4*w - 
     0.0011239361348986583*u2*v4*w + 0.04216772179515423*v5*w - 
     0.0017664062094713614*u*v5*w - 0.004855915459192057*v6*w + 
     1.5830194232420491*w2 - 0.6431554977641232*u*w2 + 
     0.29720782742626456*u2*w2 - 0.10781348756592685*u3*w2 + 
     0.02501436947958356*u4*w2 - 0.0027042708881657167*u5*w2 - 
     0.6431554977641232*v*w2 + 0.18935004993694224*u*v*w2 - 
     0.05523018173756664*u2*v*w2 + 0.011283905940380838*u3*v*w2 - 
     0.0011239361348986583*u4*v*w2 + 0.29720782742626456*v2*w2 - 
     0.05523018173756664*u*v2*w2 + 0.009104692162046783*u2*v2*w2 - 
     0.0008001063695529314*u3*v2*w2 - 0.10781348756592685*v3*w2 + 
     0.011283905940380838*u*v3*w2 - 0.0008001063695529314*u2*v3*w2 + 
     0.02501436947958356*v4*w2 - 0.0011239361348986583*u*v4*w2 - 
     0.0027042708881657167*v5*w2 - 1.6193513369191062*w3 + 
     0.39191746801048677*u*w3 - 0.10781348756592685*u2*w3 + 
     0.021502574570336913*u3*w3 - 0.002116028409986158*u4*w3 + 
     0.39191746801048677*v*w3 - 0.06968191432583015*u*v*w3 + 
     0.011283905940380838*u2*v*w3 - 0.0009831878162066224*u3*v*w3 - 
     0.10781348756592685*v2*w3 + 0.011283905940380838*u*v2*w3 - 
     0.0008001063695529314*u2*v2*w3 + 0.021502574570336913*v3*w3 - 
     0.0009831878162066224*u*v3*w3 - 0.002116028409986158*v4*w3 + 
     1.1560019673631434*w4 - 0.16545503028959443*u*w4 + 
     0.02501436947958356*u2*w4 - 0.002116028409986158*u3*w4 - 
     0.16545503028959443*v*w4 + 0.0162776555901896*u*v*w4 - 
     0.0011239361348986583*u2*v*w4 + 0.02501436947958356*v2*w4 - 
     0.0011239361348986583*u*v2*w4 - 0.002116028409986158*v3*w4 - 
     0.5375082140631944*w5 + 0.04216772179515423*u*w5 - 
     0.0027042708881657167*u2*w5 + 0.04216772179515423*v*w5 - 
     0.0017664062094713614*u*v*w5 - 0.0027042708881657167*v2*w5 + 
     0.14606611930718463*w6 - 0.004855915459192057*u*w6 - 
     0.004855915459192057*v*w6 - 0.017602106974984624*w7 - 
     1.2020569031595942*x + 0.8408174973812163*u*x - 
     0.6431554977641232*u2*x + 0.39191746801048677*u3*x - 
     0.16545503028959443*u4*x + 0.04216772179515423*u5*x - 
     0.004855915459192057*u6*x + 0.8408174973812163*v*x + 
     1.316872732915981*u*v*x + 0.6510853440804052*lnbara*u*v*x + 
     1.7420888571164688*u2*v*x + 0.4470413315421547*lnbara*u2*v*x + 
     0.5072130866752762*u3*v*x + 0.17751808906800676*lnbara*u3*v*x + 
     0.14015372875382298*u4*v*x + 0.031382877263858105*lnbara*u4*v*x + 
     0.007029374644742698*u5*v*x + 0.0025756149427488186*lnbara*u5*v*x - 
     2.4582922118695194*v2*x - 0.6510853440804052*lnbara*v2*x - 
     3.1637691242153525*u*v2*x - 1.128501340498048*lnbara*u*v2*x - 
     2.066949955700743*u2*v2*x - 0.5924457327959798*lnbara*u2*v2*x - 
     0.49681704281786926*u3*v2*x - 0.14593979274603425*lnbara*u3*v2*x - 
     0.05261848185645107*u4*v2*x - 0.01365467547098468*lnbara*u4*v2*x + 
     1.0120947283767654*v3*x + 0.24740311290228975*lnbara*v3*x + 
     3.7247490277998456*u*v3*x + 1.2186672004102037*lnbara*u*v3*x + 
     0.5028126095006992*u2*v3*x + 0.13192114982378386*lnbara*u2*v3*x + 
     0.23084041467755445*u3*v3*x + 0.06654355140807486*lnbara*u3*v3*x - 
     1.1463870292291396*v4*x - 0.3193208792684922*lnbara*v4*x - 
     1.3112778013936697*u*v4*x - 0.4144777234484366*lnbara*u*v4*x - 
     0.16893584646815507*u2*v4*x - 0.0453330032683975*lnbara*u2*v4*x + 
     0.281434133138352*v5*x + 0.07853883800228331*lnbara*v5*x + 
     0.2738733074161405*u*v5*x + 0.08187135582602693*lnbara*u*v5*x - 
     0.04987109935863513*v6*x - 0.013727259996145989*lnbara*v6*x + 
     0.6830325829486028*w*x - 2.1845063283947157*u*w*x - 
     0.6510853440804052*lnbara*u*w*x - 1.2999471003253766*u2*w*x - 
     0.4470413315421547*lnbara*u2*w*x - 0.6741348014198821*u3*w*x - 
     0.17751808906800676*lnbara*u3*w*x - 0.10064161922879965*u4*w*x - 
     0.031382877263858105*lnbara*u4*w*x - 0.011350747467801053*u5*w*x - 
     0.0025756149427488186*lnbara*u5*w*x - 0.3693696142893198*v*w*x + 
     0.1472496149014935*u*v*w*x - 0.04772361051737147*u2*v*w*x + 
     0.01023088439838107*u3*v*w*x - 0.0010458019341617988*u4*v*w*x + 
     10.034005404400828*v2*w*x + 3.285801385277514*lnbara*v2*w*x - 
     3.3495745233238825*u*v2*w*x - 1.1013272044547329*lnbara*u*v2*w*x + 
     3.2459862158900528*u2*v2*w*x + 0.9917935723502516*lnbara*u2*v2*w*x - 
     0.40594313303596224*u3*v2*w*x - 0.12233847935377173*lnbara*u3*v2*w*
      x - 5.165259334653691*v3*w*x - 1.7095780955682178*lnbara*v3*w*x - 
     0.15031327644943632*u*v3*w*x - 0.013784793500404437*lnbara*u*v3*w*x - 
     0.40939098371296645*u2*v3*w*x - 0.12576336826926637*lnbara*u2*v3*w*
      x + 3.0315936321798276*v4*w*x + 0.9380468690654714*lnbara*v4*w*x - 
     0.15381418595568414*u*v4*w*x - 0.048049503963086576*lnbara*u*v4*w*x - 
     0.48018841061091755*v5*w*x - 0.14541778766038482*lnbara*v5*w*x + 
     1.3307155997089624*w2*x + 0.6510853440804052*lnbara*w2*x + 
     3.593282429668685*u*w2*x + 1.128501340498048*lnbara*u*w2*x + 
     1.9298767034120956*u2*w2*x + 0.5924457327959798*lnbara*u2*w2*x + 
     0.5260162467363529*u3*w2*x + 0.14593979274603425*lnbara*u3*w2*x + 
     0.049642889715863496*u4*w2*x + 0.01365467547098468*lnbara*u4*w2*x - 
     9.679452702174132*v*w2*x - 3.285801385277514*lnbara*v*w2*x + 
     3.2592144539581844*u*v*w2*x + 1.1013272044547329*lnbara*u*v*w2*x - 
     3.229242426355708*u2*v*w2*x - 0.9917935723502516*lnbara*u2*v*w2*x + 
     0.404386452649353*u3*v*w2*x + 0.12233847935377173*lnbara*u3*v*w2*x - 
     0.052439122857018475*v2*w2*x + 0.00780375397440978*u*v2*w2*x - 
     0.0006345782786901335*u2*v2*w2*x - 1.8540397627858702*v3*w2*x - 
     0.5198935985278687*lnbara*v3*w2*x + 0.5315078645933379*u*v3*w2*x + 
     0.1550921261368306*lnbara*u*v3*w2*x + 0.11440699548555519*v4*w2*x + 
     0.02876833807133764*lnbara*v4*w2*x - 0.3344496541843748*w3*x - 
     0.24740311290228975*lnbara*w3*x - 3.884956834618231*u*w3*x - 
     1.2186672004102037*lnbara*u*w3*x - 0.47396206592202933*u2*w3*x - 
     0.13192114982378386*lnbara*u2*w3*x - 0.23348793049311065*u3*w3*x - 
     0.06654355140807486*lnbara*u3*w3*x + 5.035142069385885*v*w3*x + 
     1.7095780955682178*lnbara*v*w3*x + 0.16919997458043376*u*v*w3*x + 
     0.013784793500404437*lnbara*u*v*w3*x + 0.4078728401620492*u2*v*w3*x + 
     0.12576336826926637*lnbara*u2*v*w3*x + 1.8755457504129727*v2*w3*x + 
     0.5198935985278687*lnbara*v2*w3*x - 0.5329098110819325*u*v2*w3*x - 
     0.1550921261368306*lnbara*u*v2*w3*x - 0.0009436436586848417*v3*w3*
      x + 0.8622872883244641*w4*x + 0.3193208792684922*lnbara*w4*x + 
     1.3489683495777027*u*w4*x + 0.4144777234484366*lnbara*u*w4*x + 
     0.1660137847279631*u2*w4*x + 0.0453330032683975*lnbara*u2*w4*x - 
     3.0012661757807617*v*w4*x - 0.9380468690654714*lnbara*v*w4*x + 
     0.15191010788660775*u*v*w4*x + 0.048049503963086576*lnbara*u*v*w4*x - 
     0.11655132895257132*v2*w4*x - 0.02876833807133764*lnbara*v2*w4*x - 
     0.2093175862405338*w5*x - 0.07853883800228331*lnbara*w5*x - 
     0.27798071288879395*u*w5*x - 0.08187135582602693*lnbara*u*w5*x + 
     0.4769026267982312*v*w5*x + 0.14541778766038482*lnbara*v*w5*x + 
     0.04158747525391902*w6*x + 0.013727259996145989*lnbara*w6*x + 
     1.5830194232420491*x2 - 0.6431554977641232*u*x2 + 
     0.29720782742626456*u2*x2 - 0.10781348756592685*u3*x2 + 
     0.02501436947958356*u4*x2 - 0.0027042708881657167*u5*x2 - 
     2.4582922118695194*v*x2 - 0.6510853440804052*lnbara*v*x2 - 
     3.1637691242153525*u*v*x2 - 1.128501340498048*lnbara*u*v*x2 - 
     2.066949955700743*u2*v*x2 - 0.5924457327959798*lnbara*u2*v*x2 - 
     0.49681704281786926*u3*v*x2 - 0.14593979274603425*lnbara*u3*v*x2 - 
     0.05261848185645107*u4*v*x2 - 0.01365467547098468*lnbara*u4*v*x2 + 
     5.493378448409817*v2*x2 + 1.79697691396539*lnbara*v2*x2 + 
     1.5405090633173697*u*v2*x2 + 0.5190360738429882*lnbara*u*v2*x2 + 
     2.7618349656998586*u2*v2*x2 + 0.8236272482232423*lnbara*u2*v2*x2 + 
     0.024812552253286976*u3*v2*x2 + 0.003229366446298125*lnbara*u3*v2*
      x2 - 2.6912970732885757*v3*x2 - 0.8835715495175162*lnbara*v3*x2 - 
     1.8613624203011476*u*v3*x2 - 0.5720040255869473*lnbara*u*v3*x2 - 
     0.48114924036453877*u2*v3*x2 - 0.13930793778849923*lnbara*u2*v3*
      x2 + 1.6718584721002623*v4*x2 + 0.5152999807078541*lnbara*v4*x2 + 
     0.17448936082610797*u*v4*x2 + 0.05426081916613278*lnbara*u*v4*x2 - 
     0.24406286627904641*v5*x2 - 0.07417184660062576*lnbara*v5*x2 + 
     1.3307155997089624*w*x2 + 0.6510853440804052*lnbara*w*x2 + 
     3.593282429668685*u*w*x2 + 1.128501340498048*lnbara*u*w*x2 + 
     1.9298767034120956*u2*w*x2 + 0.5924457327959798*lnbara*u2*w*x2 + 
     0.5260162467363529*u3*w*x2 + 0.14593979274603425*lnbara*u3*w*x2 + 
     0.049642889715863496*u4*w*x2 + 0.01365467547098468*lnbara*u4*w*x2 - 
     6.422064708307189*v*w*x2 - 2.1702844802680175*lnbara*v*w*x2 + 
     6.435061406178583*u*v*w*x2 + 2.079207074882875*lnbara*u*v*w*x2 - 
     3.373810219562036*u2*v*w*x2 - 1.0395623490494532*lnbara*u2*v*w*x2 + 
     0.5336843597652852*u3*v*w*x2 + 0.16066085741485445*lnbara*u3*v*w*
      x2 - 1.1288573398553086*v2*w*x2 - 0.43966795501820194*lnbara*v2*w*
      x2 - 1.202554183086874*u*v2*w*x2 - 0.3723502771999754*lnbara*u*v2*w*
      x2 - 0.08093775692251794*u2*v2*w*x2 - 0.025975593895628257*lnbara*
      u2*v2*w*x2 - 1.4645041396124352*v3*w*x2 - 
     0.37498908309415785*lnbara*v3*w*x2 + 1.25382676464303*u*v3*w*x2 + 
     0.3652162355419701*lnbara*u*v3*w*x2 + 0.0011325127373185734*v4*w*
      x2 - 0.009074544432852395*lnbara*v4*w*x2 + 
     1.5951827813991792*w2*x2 + 0.37330756630262746*lnbara*w2*x2 - 
     8.153252996171776*u*w2*x2 - 2.598243148725863*lnbara*u*w2*x2 + 
     0.6456564865390652*u2*w2*x2 + 0.21593510082621092*lnbara*u2*w2*
      x2 - 0.5616689792286329*u3*w2*x2 - 0.16389022386115257*lnbara*u3*
      w2*x2 + 3.1418162127487004*v*w2*x2 + 1.1732054736522635*lnbara*v*
      w2*x2 + 0.7605706197007904*u*v*w2*x2 + 0.1801033288553867*lnbara*u*
      v*w2*x2 + 0.41357773400649983*u2*v*w2*x2 + 
     0.1313398473229925*lnbara*u2*v*w2*x2 + 2.445843105402685*v2*w2*
      x2 + 0.7054893158837185*lnbara*v2*w2*x2 - 
     1.2333152992895828*u*v2*w2*x2 - 0.3521670333053898*lnbara*u*v2*w2*
      x2 - 0.061694139411974463*v3*w2*x2 - 0.01592769241839614*lnbara*v3*
      w2*x2 + 0.403289978849488*w3*x2 + 0.15003403088345468*lnbara*w3*
      x2 + 2.3470147044696774*u*w3*x2 + 0.7642509739315361*lnbara*u*w3*
      x2 + 0.14483466237687437*u2*w3*x2 + 0.03394368436113497*lnbara*u2*
      w3*x2 - 2.775770631100383*v*w3*x2 - 0.8879564128194577*lnbara*v*w3*
      x2 + 0.3619717726179578*u*v*w3*x2 + 0.10324350723878072*lnbara*u*v*
      w3*x2 - 0.04072484698113826*v2*w3*x2 - 0.013759874074562587*lnbara*
      v2*w3*x2 + 0.1913662943454968*w4*x2 + 0.04215619932204285*lnbara*
      w4*x2 - 0.5618260667674302*u*w4*x2 - 0.17055352864149378*lnbara*u*
      w4*x2 + 0.3328573832991313*v*w4*x2 + 0.11063120820095806*lnbara*v*
      w4*x2 + 0.004695610230502542*w5*x2 + 0.002302749325478825*lnbara*
      w5*x2 - 1.6193513369191062*x3 + 0.39191746801048677*u*x3 - 
     0.10781348756592685*u2*x3 + 0.021502574570336913*u3*x3 - 
     0.002116028409986158*u4*x3 + 1.0120947283767654*v*x3 + 
     0.24740311290228975*lnbara*v*x3 + 3.7247490277998456*u*v*x3 + 
     1.2186672004102037*lnbara*u*v*x3 + 0.5028126095006992*u2*v*x3 + 
     0.13192114982378386*lnbara*u2*v*x3 + 0.23084041467755445*u3*v*x3 + 
     0.06654355140807486*lnbara*u3*v*x3 - 2.6912970732885757*v2*x3 - 
     0.8835715495175162*lnbara*v2*x3 - 1.8613624203011476*u*v2*x3 - 
     0.5720040255869473*lnbara*u*v2*x3 - 0.48114924036453877*u2*v2*x3 - 
     0.13930793778849923*lnbara*u2*v2*x3 + 0.24682581080985633*v3*x3 + 
     0.10513746096634541*lnbara*v3*x3 + 0.8544575390208671*u*v3*x3 + 
     0.251383986920522*lnbara*u*v3*x3 - 0.1821625110649795*v4*x3 - 
     0.057146188727501226*lnbara*v4*x3 - 0.3344496541843748*w*x3 - 
     0.24740311290228975*lnbara*w*x3 - 3.884956834618231*u*w*x3 - 
     1.2186672004102037*lnbara*u*w*x3 - 0.47396206592202933*u2*w*x3 - 
     0.13192114982378386*lnbara*u2*w*x3 - 0.23348793049311065*u3*w*x3 - 
     0.06654355140807486*lnbara*u3*w*x3 + 2.0505932975012593*v*w*x3 + 
     0.7335375186340615*lnbara*v*w*x3 - 0.4484159450532304*u*v*w*x3 - 
     0.1922469483445887*lnbara*u*v*w*x3 + 0.33319811688498185*u2*v*w*x3 + 
     0.10536425342736426*lnbara*u2*v*w*x3 + 2.941587800744719*v2*w*x3 + 
     0.8832109828874293*lnbara*v2*w*x3 - 0.7853652276547294*u*v2*w*x3 - 
     0.21990490700155435*lnbara*u*v2*w*x3 - 0.2434872198153215*v3*w*x3 - 
     0.07483548907405872*lnbara*v3*w*x3 + 0.403289978849488*w2*x3 + 
     0.15003403088345468*lnbara*w2*x3 + 2.3470147044696774*u*w2*x3 + 
     0.7642509739315361*lnbara*u*w2*x3 + 0.14483466237687437*u2*w2*x3 + 
     0.03394368436113497*lnbara*u2*w2*x3 - 3.269269763555497*v*w2*x3 - 
     1.0164989417094896*lnbara*v*w2*x3 + 0.44168144304165097*u*v*w2*x3 + 
     0.12858123333432991*lnbara*u*v*w2*x3 + 0.05859089416673988*v2*w2*
      x3 + 0.01186851715999764*lnbara*v2*w2*x3 + 
     0.13476350893453617*w3*x3 + 0.02815049785571501*lnbara*w3*x3 - 
     0.5147171592780667*u*w3*x3 - 0.16006031325329761*lnbara*u*w3*x3 + 
     0.33499391115566624*v*w3*x3 + 0.11243984289700684*lnbara*v*w3*x3 + 
     0.026332918322492415*w4*x3 + 0.007673317744555463*lnbara*w4*x3 + 
     1.1560019673631434*x4 - 0.16545503028959443*u*x4 + 
     0.02501436947958356*u2*x4 - 0.002116028409986158*u3*x4 - 
     1.1463870292291396*v*x4 - 0.3193208792684922*lnbara*v*x4 - 
     1.3112778013936697*u*v*x4 - 0.4144777234484366*lnbara*u*v*x4 - 
     0.16893584646815507*u2*v*x4 - 0.0453330032683975*lnbara*u2*v*x4 + 
     1.6718584721002623*v2*x4 + 0.5152999807078541*lnbara*v2*x4 + 
     0.17448936082610797*u*v2*x4 + 0.05426081916613278*lnbara*u*v2*x4 - 
     0.1821625110649795*v3*x4 - 0.057146188727501226*lnbara*v3*x4 + 
     0.8622872883244641*w*x4 + 0.3193208792684922*lnbara*w*x4 + 
     1.3489683495777027*u*w*x4 + 0.4144777234484366*lnbara*u*w*x4 + 
     0.1660137847279631*u2*w*x4 + 0.0453330032683975*lnbara*u2*w*x4 - 
     1.8086906966367922*v*w*x4 - 0.5574561800298969*lnbara*v*w*x4 + 
     0.3835758469523267*u*v*w*x4 + 0.116292709475361*lnbara*u*v*w*x4 - 
     0.21163351738140398*v2*w*x4 - 0.06718897907015481*lnbara*v2*w*x4 + 
     0.1913662943454968*w2*x4 + 0.04215619932204285*lnbara*w2*x4 - 
     0.5618260667674302*u*w2*x4 - 0.17055352864149378*lnbara*u*w2*x4 + 
     0.3622126558447875*v*w2*x4 + 0.11666185005310058*lnbara*v*w2*x4 + 
     0.026332918322492415*w3*x4 + 0.007673317744555463*lnbara*w3*x4 - 
     0.5375082140631944*x5 + 0.04216772179515423*u*x5 - 
     0.0027042708881657167*u2*x5 + 0.281434133138352*v*x5 + 
     0.07853883800228331*lnbara*v*x5 + 0.2738733074161405*u*v*x5 + 
     0.08187135582602693*lnbara*u*v*x5 - 0.24406286627904641*v2*x5 - 
     0.07417184660062576*lnbara*v2*x5 - 0.2093175862405338*w*x5 - 
     0.07853883800228331*lnbara*w*x5 - 0.27798071288879395*u*w*x5 - 
     0.08187135582602693*lnbara*u*w*x5 + 0.23350760677675905*v*w*x5 + 
     0.07186909727514694*lnbara*v*w*x5 + 0.004695610230502542*w2*x5 + 
     0.002302749325478825*lnbara*w2*x5 + 0.14606611930718463*x6 - 
     0.004855915459192057*u*x6 - 0.04987109935863513*v*x6 - 
     0.013727259996145989*lnbara*v*x6 + 0.04158747525391902*w*x6 + 
     0.013727259996145989*lnbara*w*x6 - 0.017602106974984624*x7 - 
     1.2020569031595942*y + 0.6830325829486028*u*y - 
     0.4844211143964337*u2*y + 0.28572760618190396*u3*y - 
     0.11864471061508094*u4*y + 0.029948825102663987*u5*y - 
     0.0034277086455240437*u6*y + 0.8408174973812163*v*y - 
     0.3693696142893198*u*v*y + 0.17096212544446912*u2*v*y - 
     0.06170167867079261*u3*v*y + 0.014252947529037665*u4*v*y - 
     0.0015359082311407401*u5*v*y - 0.6431554977641232*v2*y + 
     0.18359057678222876*u*v2*y - 0.052439122857018475*u2*v2*y + 
     0.010578663643644204*u3*v2*y - 0.0010454015333102693*u4*v2*y + 
     0.39191746801048677*v3*y - 0.06841558659701322*u*v3*y + 
     0.010927323983458038*u2*v3*y - 0.0009436436586848417*u3*v3*y - 
     0.16545503028959443*v4*y + 0.016074508870027843*u*v4*y - 
     0.0010989319337058666*u2*v4*y + 0.04216772179515423*v5*y - 
     0.0017498755815456293*u*v5*y - 0.004855915459192057*v6*y + 
     0.8408174973812163*w*y - 0.3693696142893198*u*w*y + 
     0.17096212544446912*u2*w*y - 0.06170167867079261*u3*w*y + 
     0.014252947529037665*u4*w*y - 0.0015359082311407401*u5*w*y - 
     0.4982639811894151*v*w*y + 0.1472496149014935*u*v*w*y - 
     0.04263645884832675*u2*v*w*y + 0.008655813732616391*u3*v*w*y - 
     0.0008582761349145647*u4*v*w*y + 0.25855118000886335*v2*w*y - 
     0.04772361051737147*u*v2*w*y + 0.00780375397440978*u2*v2*w*y - 
     0.0006817048264512694*u3*v2*w*y - 0.0985061281475928*v3*w*y + 
     0.01023088439838107*u*v3*w*y - 0.0007202416621432963*u2*v3*w*y + 
     0.023437600654995495*v4*w*y - 0.0010458019341617988*u*v4*w*y - 
     0.0025714972415127255*v5*w*y - 0.6431554977641232*w2*y + 
     0.18359057678222876*u*w2*y - 0.052439122857018475*u2*w2*y + 
     0.010578663643644204*u3*w2*y - 0.0010454015333102693*u4*w2*y + 
     0.25855118000886335*v*w2*y - 0.04772361051737147*u*v*w2*y + 
     0.00780375397440978*u2*v*w2*y - 0.0006817048264512694*u3*v*w2*y - 
     0.08463412943162905*v2*w2*y + 0.00894003555993513*u*v2*w2*y - 
     0.0006345782786901335*u2*v2*w2*y + 0.018271879935025635*v3*w2*y - 
     0.0008364387244659098*u*v3*w2*y - 0.0018766602068817076*v4*w2*y + 
     0.39191746801048677*w3*y - 0.06841558659701322*u*w3*y + 
     0.010927323983458038*u2*w3*y - 0.0009436436586848417*u3*w3*y - 
     0.0985061281475928*v*w3*y + 0.01023088439838107*u*v*w3*y - 
     0.0007202416621432963*u2*v*w3*y + 0.018271879935025635*v2*w3*y - 
     0.0008364387244659098*u*v2*w3*y - 0.0017038721568713427*v3*w3*y - 
     0.16545503028959443*w4*y + 0.016074508870027843*u*w4*y - 
     0.0010989319337058666*u2*w4*y + 0.023437600654995495*v*w4*y - 
     0.0010458019341617988*u*v*w4*y - 0.0018766602068817076*v2*w4*y + 
     0.04216772179515423*w5*y - 0.0017498755815456293*u*w5*y - 
     0.0025714972415127255*v*w5*y - 0.004855915459192057*w6*y + 
     0.8408174973812163*x*y - 0.3693696142893198*u*x*y + 
     0.17096212544446912*u2*x*y - 0.06170167867079261*u3*x*y + 
     0.014252947529037665*u4*x*y - 0.0015359082311407401*u5*x*y + 
     5.050947856234655*v*x*y + 1.9532560322412158*lnbara*v*x*y + 
     3.6089086211615613*u*v*x*y + 1.0155813225862613*lnbara*u*v*x*y + 
     1.6153412124624802*u2*v*x*y + 0.49610466935343467*lnbara*u2*v*x*y + 
     0.39176531410887067*u3*v*x*y + 0.10319148379269107*lnbara*u3*v*x*y + 
     0.037264929316591394*u4*v*x*y + 0.01035774990633652*lnbara*u4*v*x*y - 
     4.63804380635014*v2*x*y - 1.7188373548274771*lnbara*v2*x*y - 
     3.7155448088414325*u*v2*x*y - 1.101558132258626*lnbara*u*v2*x*y - 
     1.279932716374738*u2*v2*x*y - 0.37253832538857234*lnbara*u2*v2*x*y - 
     0.16259660502488668*u3*v2*x*y - 0.043853270547930576*lnbara*u3*v2*x*
      y + 6.321775087715999*v3*x*y + 2.0595906539260844*lnbara*v3*x*y + 
     1.1114419930785513*u*v3*x*y + 0.30677398634465897*lnbara*u*v3*x*y + 
     0.5189008798963198*u2*v3*x*y + 0.14879519944448213*lnbara*u2*v3*x*
      y - 2.3697386855833713*v4*x*y - 0.7507653584772783*lnbara*v4*x*y - 
     0.2989574253995812*u*v4*x*y - 0.0809778929894431*lnbara*u*v4*x*y + 
     0.4787256919895696*v5*x*y + 0.1438457497350551*lnbara*v5*x*y - 
     5.814779756605509*w*x*y - 1.9532560322412158*lnbara*w*x*y - 
     3.3297094342481097*u*w*x*y - 1.0155813225862613*lnbara*u*w*x*y - 
     1.7018138873778625*u2*w*x*y - 0.49610466935343467*lnbara*u2*w*x*y - 
     0.373690675987211*u3*w*x*y - 0.10319148379269107*lnbara*u3*w*x*y - 
     0.03908459842732547*u4*w*x*y - 0.01035774990633652*lnbara*u4*w*x*y + 
     0.1472496149014935*v*w*x*y - 0.03663932564661811*u*v*w*x*y + 
     0.006668952902327549*u2*v*w*x*y - 0.0006126156464844033*u3*v*w*x*y - 
     6.7414545262279955*v2*w*x*y - 2.014469236002691*lnbara*v2*w*x*y + 
     2.582819839347*u*v2*w*x*y + 0.8128752507655377*lnbara*u*v2*w*x*y - 
     0.30225971597078544*u2*v2*w*x*y - 0.09017258826251653*lnbara*u2*v2*w*
      x*y + 0.7227291601125875*v3*w*x*y + 0.20757337570539203*lnbara*v3*w*x*
      y - 0.368445017992858*u*v3*w*x*y - 0.11464743468556707*lnbara*u*v3*w*
      x*y - 0.3710874532554622*v4*w*x*y - 0.10073935030932556*lnbara*v4*w*x*
      y + 4.998355981731551*w2*x*y + 1.7188373548274771*lnbara*w2*x*y + 
     3.6274999402312313*u*w2*x*y + 1.101558132258626*lnbara*u*w2*x*y + 
     1.2958237235724512*u2*w2*x*y + 0.37253832538857234*lnbara*u2*w2*x*
      y + 0.1611431319221526*u3*w2*x*y + 0.043853270547930576*lnbara*u3*
      w2*x*y + 6.647579498888072*v*w2*x*y + 2.014469236002691*lnbara*v*w2*
      x*y - 2.5690901893905513*u*v*w2*x*y - 0.8128752507655377*lnbara*u*v*
      w2*x*y + 0.3011577157359468*u2*v*w2*x*y + 0.09017258826251653*lnbara*
      u2*v*w2*x*y + 0.008768669407230721*v2*w2*x*y - 
     0.0005708021200927833*u*v2*w2*x*y + 0.8908183968952704*v3*w2*x*y + 
     0.2518959384495211*lnbara*v3*w2*x*y - 6.4531586807126216*w3*x*y - 
     2.0595906539260844*lnbara*w3*x*y - 1.0928490356968497*u*w3*x*y - 
     0.30677398634465897*lnbara*u*w3*x*y - 0.5203695633112709*u2*w3*x*y - 
     0.14879519944448213*lnbara*u2*w3*x*y - 0.7027857105758675*v*w3*x*y - 
     0.20757337570539203*lnbara*v*w3*x*y + 0.3671622432587578*u*v*w3*x*y + 
     0.11464743468556707*lnbara*u*v*w3*x*y - 0.8924547277374686*v2*w3*x*
      y - 0.2518959384495211*lnbara*v2*w3*x*y + 2.4002692887025985*w4*x*
      y + 0.7507653584772783*lnbara*w4*x*y + 0.29707382410626676*u*w4*x*y + 
     0.0809778929894431*lnbara*u*w4*x*y + 0.36905978156971897*v*w4*x*y + 
     0.10073935030932556*lnbara*v*w4*x*y - 0.48202800643018173*w5*x*y - 
     0.1438457497350551*lnbara*w5*x*y - 0.6431554977641232*x2*y + 
     0.18359057678222876*u*x2*y - 0.052439122857018475*u2*x2*y + 
     0.010578663643644204*u3*x2*y - 0.0010454015333102693*u4*x2*y - 
     4.63804380635014*v*x2*y - 1.7188373548274771*lnbara*v*x2*y - 
     3.7155448088414325*u*v*x2*y - 1.101558132258626*lnbara*u*v*x2*y - 
     1.279932716374738*u2*v*x2*y - 0.37253832538857234*lnbara*u2*v*x2*y - 
     0.16259660502488668*u3*v*x2*y - 0.043853270547930576*lnbara*u3*v*x2*
      y + 0.3848527108952509*v2*x2*y + 0.27079839140234246*lnbara*v2*x2*
      y + 4.968594829946177*u*v2*x2*y + 1.491398063789174*lnbara*u*v2*x2*
      y + 0.2124983335725487*u2*v2*x2*y + 0.0544834669601152*lnbara*u2*
      v2*x2*y - 1.6610384150846027*v3*x2*y - 0.5456870809017446*lnbara*
      v3*x2*y - 0.9061306963251085*u*v3*x2*y - 0.2658866648025671*lnbara*
      u*v3*x2*y + 0.1309550363078858*v4*x2*y + 0.04900052281465277*lnbara*
      v4*x2*y + 4.998355981731551*w*x2*y + 1.7188373548274771*lnbara*w*x2*
      y + 3.6274999402312313*u*w*x2*y + 1.101558132258626*lnbara*u*w*x2*y + 
     1.2958237235724512*u2*w*x2*y + 0.37253832538857234*lnbara*u2*w*x2*
      y + 0.1611431319221526*u3*w*x2*y + 0.043853270547930576*lnbara*u3*w*
      x2*y + 4.806051314717304*v*w*x2*y + 1.4974962913849321*lnbara*v*w*x2*
      y - 4.358084874395133*u*v*w*x2*y - 1.310301658288488*lnbara*u*v*w*x2*
      y + 0.44246231471333614*u2*v*w*x2*y + 0.13612276309543284*lnbara*u2*
      v*w*x2*y + 0.16780437876971865*v2*w*x2*y + 
     0.05368132013057931*lnbara*v2*w*x2*y + 0.00017982275152600617*u*v2*w*
      x2*y - 0.008148863899001088*lnbara*u*v2*w*x2*y + 
     0.8410658901759228*v3*w*x2*y + 0.24067518005948615*lnbara*v3*w*x2*
      y - 5.33566764662009*w2*x2*y - 1.7682946827872745*lnbara*w2*x2*y - 
     0.5880364469542249*u*w2*x2*y - 0.18109640550068581*lnbara*u*w2*x2*
      y - 0.6568129911274124*u2*w2*x2*y - 0.19060623005554803*lnbara*u2*
      w2*x2*y - 0.19274016063176588*v*w2*x2*y - 
     0.09505426720402026*lnbara*v*w2*x2*y + 0.5522979595539836*u*v*w2*x2*
      y + 0.17560355760261687*lnbara*u*v*w2*x2*y - 
     1.0211916139594837*v2*w2*x2*y - 0.2879966174194479*lnbara*v2*w2*x2*
      y + 1.7216900283016308*w3*x2*y + 0.5870600279751855*lnbara*w3*x2*
      y + 0.35111850002088596*u*w3*x2*y + 0.09843197109895134*lnbara*u*w3*
      x2*y + 0.4722390677423313*v*w3*x2*y + 0.13175034334097716*lnbara*v*
      w3*x2*y - 0.4270870728419753*w4*x2*y - 0.13342942879566821*lnbara*
      w4*x2*y + 0.39191746801048677*x3*y - 0.06841558659701322*u*x3*y + 
     0.010927323983458038*u2*x3*y - 0.0009436436586848417*u3*x3*y + 
     6.321775087715999*v*x3*y + 2.0595906539260844*lnbara*v*x3*y + 
     1.1114419930785513*u*v*x3*y + 0.30677398634465897*lnbara*u*v*x3*y + 
     0.5189008798963198*u2*v*x3*y + 0.14879519944448213*lnbara*u2*v*x3*
      y - 1.6610384150846027*v2*x3*y - 0.5456870809017446*lnbara*v2*x3*
      y - 0.9061306963251085*u*v2*x3*y - 0.2658866648025671*lnbara*u*v2*
      x3*y + 0.8429911783197556*v3*x3*y + 0.2503190400812598*lnbara*v3*
      x3*y - 6.4531586807126216*w*x3*y - 2.0595906539260844*lnbara*w*x3*
      y - 1.0928490356968497*u*w*x3*y - 0.30677398634465897*lnbara*u*w*x3*
      y - 0.5203695633112709*u2*w*x3*y - 0.14879519944448213*lnbara*u2*w*
      x3*y - 0.03145999950209565*v*w*x3*y - 0.041372947073440955*lnbara*v*w*
      x3*y + 0.5529561454818885*u*v*w*x3*y + 0.1674546937036158*lnbara*u*v*
      w*x3*y - 0.9153400045660104*v2*w*x3*y - 0.25553842398615445*lnbara*
      v2*w*x3*y + 1.7216900283016308*w2*x3*y + 0.5870600279751855*lnbara*
      w2*x3*y + 0.35111850002088596*u*w2*x3*y + 
     0.09843197109895134*lnbara*u*w2*x3*y + 0.569975957295371*v*w2*x3*y + 
     0.15957579625776946*lnbara*v*w2*x3*y - 0.5006975185477412*w3*x3*y - 
     0.1543564123528748*lnbara*w3*x3*y - 0.16545503028959443*x4*y + 
     0.016074508870027843*u*x4*y - 0.0010989319337058666*u2*x4*y - 
     2.3697386855833713*v*x4*y - 0.7507653584772783*lnbara*v*x4*y - 
     0.2989574253995812*u*v*x4*y - 0.0809778929894431*lnbara*u*v*x4*y + 
     0.1309550363078858*v2*x4*y + 0.04900052281465277*lnbara*v2*x4*y + 
     2.4002692887025985*w*x4*y + 0.7507653584772783*lnbara*w*x4*y + 
     0.29707382410626676*u*w*x4*y + 0.0809778929894431*lnbara*u*w*x4*y + 
     0.2932474952337439*v*w*x4*y + 0.08442890598101545*lnbara*v*w*x4*y - 
     0.4270870728419753*w2*x4*y - 0.13342942879566821*lnbara*w2*x4*y + 
     0.04216772179515423*x5*y - 0.0017498755815456293*u*x5*y + 
     0.4787256919895696*v*x5*y + 0.1438457497350551*lnbara*v*x5*y - 
     0.48202800643018173*w*x5*y - 0.1438457497350551*lnbara*w*x5*y - 
     0.004855915459192057*x6*y + 1.5830194232420491*y2 - 
     0.4844211143964337*u*y2 + 0.1983265686310733*u2*y2 - 
     0.06789863070110878*u3*y2 + 0.01526675280034565*u4*y2 - 
     0.001619470152478382*u5*y2 - 0.6431554977641232*v*y2 + 
     0.17096212544446912*u*v*y2 - 0.04689698042164179*u2*v*y2 + 
     0.009251894041935034*u3*v*y2 - 0.0009023290305323929*u4*v*y2 + 
     0.29720782742626456*v2*y2 - 0.052439122857018475*u*v2*y2 + 
     0.008332760830303598*u2*v2*y2 - 0.0007154743025816235*u3*v2*y2 - 
     0.10781348756592685*v3*y2 + 0.010927323983458038*u*v3*y2 - 
     0.000755870097742802*u2*v3*y2 + 0.02501436947958356*v4*y2 - 
     0.0010989319337058666*u*v4*y2 - 0.0027042708881657167*v5*y2 - 
     0.6431554977641232*w*y2 + 0.17096212544446912*u*w*y2 - 
     0.04689698042164179*u2*w*y2 + 0.009251894041935034*u3*w*y2 - 
     0.0009023290305323929*u4*w*y2 + 0.25855118000886335*v*w*y2 - 
     0.04615141682255245*u*v*w*y2 + 0.007376277398255051*u2*v*w*y2 - 
     0.0006353292970921247*u3*v*w*y2 - 0.08463412943162905*v2*w*y2 + 
     0.008768669407230721*u*v2*w*y2 - 0.0006133788582035897*u2*v2*w*y2 + 
     0.018271879935025635*v3*w*y2 - 0.0008257705772076173*u*v3*w*y2 - 
     0.0018766602068817076*v4*w*y2 + 0.29720782742626456*w2*y2 - 
     0.052439122857018475*u*w2*y2 + 0.008332760830303598*u2*w2*y2 - 
     0.0007154743025816235*u3*w2*y2 - 0.08463412943162905*v*w2*y2 + 
     0.008768669407230721*u*v*w2*y2 - 0.0006133788582035897*u2*v*w2*y2 + 
     0.01657980243935387*v2*w2*y2 - 0.0007553212077869194*u*v2*w2*y2 - 
     0.0015904265351103456*v3*w2*y2 - 0.10781348756592685*w3*y2 + 
     0.010927323983458038*u*w3*y2 - 0.000755870097742802*u2*w3*y2 + 
     0.018271879935025635*v*w3*y2 - 0.0008257705772076173*u*v*w3*y2 - 
     0.0015904265351103456*v2*w3*y2 + 0.02501436947958356*w4*y2 - 
     0.0010989319337058666*u*w4*y2 - 0.0018766602068817076*v*w4*y2 - 
     0.0027042708881657167*w5*y2 - 0.6431554977641232*x*y2 + 
     0.17096212544446912*u*x*y2 - 0.04689698042164179*u2*x*y2 + 
     0.009251894041935034*u3*x*y2 - 0.0009023290305323929*u4*x*y2 - 
     2.777512025251304*v*x*y2 - 0.9766280161206079*lnbara*v*x*y2 + 
     0.4390247991246611*u*v*x*y2 + 0.18363961128442552*lnbara*u*v*x*y2 + 
     0.3502072962213621*u2*v*x*y2 + 0.09418475573467341*lnbara*u2*v*x*
      y2 + 0.055393310008639646*u3*v*x*y2 + 0.015395299386842583*lnbara*
      u3*v*x*y2 - 0.08497562276797721*v2*x*y2 + 
     0.027337198387939212*lnbara*v2*x*y2 - 0.7526477768237668*u*v2*x*y2 - 
     0.22874938189748867*lnbara*u*v2*x*y2 - 0.1770896870636047*u2*v2*x*
      y2 - 0.04860228502711452*lnbara*u2*v2*x*y2 - 
     0.8226148783091756*v3*x*y2 - 0.248860063558088*lnbara*v3*x*y2 + 
     0.33716810694416005*u*v3*x*y2 + 0.09925662865561737*lnbara*u*v3*x*
      y2 + 0.04549876822052544*v4*x*y2 + 0.015411950745225283*lnbara*v4*x*
      y2 + 3.150452651970475*w*x*y2 + 0.9766280161206079*lnbara*w*x*y2 - 
     0.5219825160658177*u*w*x*y2 - 0.18363961128442552*lnbara*u*w*x*y2 - 
     0.335966498688551*u2*w*x*y2 - 0.09418475573467341*lnbara*u2*w*x*y2 - 
     0.05665911334956437*u3*w*x*y2 - 0.015395299386842583*lnbara*u3*w*x*
      y2 - 0.04772361051737147*v*w*x*y2 + 0.006668952902327549*u*v*w*x*
      y2 - 0.000520176702770554*u2*v*w*x*y2 + 4.165985435140145*v2*w*x*
      y2 + 1.2223037533945342*lnbara*v2*w*x*y2 - 
     0.15906541536540864*u*v2*w*x*y2 - 0.05241767434109274*lnbara*u*v2*w*x*
      y2 - 0.5092031405619221*v3*w*x*y2 - 0.15134262670003695*lnbara*v3*w*
      x*y2 - 0.022693681826607907*w2*x*y2 - 0.027337198387939212*lnbara*
      w2*x*y2 + 0.7675738685886592*u*w2*x*y2 + 0.22874938189748867*lnbara*
      u*w2*x*y2 + 0.17593076120022283*u2*w2*x*y2 + 
     0.04860228502711452*lnbara*u2*w2*x*y2 - 4.148276730172979*v*w2*x*
      y2 - 1.2223037533945342*lnbara*v*w2*x*y2 + 
     0.15794313861675344*u*v*w2*x*y2 + 0.05241767434109274*lnbara*u*v*w2*x*
      y2 - 0.0007553212077869194*v2*w2*x*y2 + 0.8444774478932007*w3*x*
      y2 + 0.248860063558088*lnbara*w3*x*y2 - 0.3385312614440469*u*w3*x*
      y2 - 0.09925662865561737*lnbara*u*w3*x*y2 + 
     0.5075561415724656*v*w3*x*y2 + 0.15134262670003695*lnbara*v*w3*x*
      y2 - 0.047668105888734366*w4*x*y2 - 0.015411950745225283*lnbara*w4*
      x*y2 + 0.29720782742626456*x2*y2 - 0.052439122857018475*u*x2*y2 + 
     0.008332760830303598*u2*x2*y2 - 0.0007154743025816235*u3*x2*y2 - 
     0.08497562276797721*v*x2*y2 + 0.027337198387939212*lnbara*v*x2*y2 - 
     0.7526477768237668*u*v*x2*y2 - 0.22874938189748867*lnbara*u*v*x2*
      y2 - 0.1770896870636047*u2*v*x2*y2 - 0.04860228502711452*lnbara*u2*
      v*x2*y2 + 1.9461607406157502*v2*x2*y2 + 0.566882499277725*lnbara*
      v2*x2*y2 + 0.021416517678162092*u*v2*x2*y2 + 
     0.005626359452000324*lnbara*u*v2*x2*y2 - 0.2056752271955055*v3*x2*
      y2 - 0.06026111093570525*lnbara*v3*x2*y2 - 
     0.022693681826607907*w*x2*y2 - 0.027337198387939212*lnbara*w*x2*y2 + 
     0.7675738685886592*u*w*x2*y2 + 0.22874938189748867*lnbara*u*w*x2*
      y2 + 0.17593076120022283*u2*w*x2*y2 + 0.04860228502711452*lnbara*
      u2*w*x2*y2 - 2.5782754794730662*v*w*x2*y2 - 
     0.7595995680938061*lnbara*v*w*x2*y2 + 0.6316944039353403*u*v*w*x2*
      y2 + 0.18411867864157963*lnbara*u*v*w*x2*y2 - 
     0.21678745401254226*v2*w*x2*y2 - 0.06933321435793802*lnbara*v2*w*x2*
      y2 + 0.6573559458240761*w2*x2*y2 + 0.19271706881608106*lnbara*w2*
      x2*y2 - 0.6548222450121166*u*w2*x2*y2 - 0.18974503809357995*lnbara*
      u*w2*x2*y2 + 0.37199049657356653*v*w2*x2*y2 + 
     0.11960991850943639*lnbara*v*w2*x2*y2 + 0.047708646825452936*w3*x2*
      y2 + 0.00998440678420688*lnbara*w3*x2*y2 - 
     0.10781348756592685*x3*y2 + 0.010927323983458038*u*x3*y2 - 
     0.000755870097742802*u2*x3*y2 - 0.8226148783091756*v*x3*y2 - 
     0.248860063558088*lnbara*v*x3*y2 + 0.33716810694416005*u*v*x3*y2 + 
     0.09925662865561737*lnbara*u*v*x3*y2 - 0.2056752271955055*v2*x3*
      y2 - 0.06026111093570525*lnbara*v2*x3*y2 + 
     0.8444774478932007*w*x3*y2 + 0.248860063558088*lnbara*w*x3*y2 - 
     0.3385312614440469*u*w*x3*y2 - 0.09925662865561737*lnbara*u*w*x3*
      y2 + 0.15576929487146673*v*w*x3*y2 + 0.05027670415149837*lnbara*v*w*
      x3*y2 + 0.047708646825452936*w2*x3*y2 + 0.00998440678420688*lnbara*
      w2*x3*y2 + 0.02501436947958356*x4*y2 - 0.0010989319337058666*u*x4*
      y2 + 0.04549876822052544*v*x4*y2 + 0.015411950745225283*lnbara*v*x4*
      y2 - 0.047668105888734366*w*x4*y2 - 0.015411950745225283*lnbara*w*
      x4*y2 - 0.0027042708881657167*x5*y2 - 1.6193513369191062*y3 + 
     0.28572760618190396*u*y3 - 0.06789863070110878*u2*y3 + 
     0.012574224677698281*u3*y3 - 0.001186695305274723*u4*y3 + 
     0.39191746801048677*v*y3 - 0.06170167867079261*u*v*y3 + 
     0.009251894041935034*u2*v*y3 - 0.0007701655588751243*u3*v*y3 - 
     0.10781348756592685*v2*y3 + 0.010578663643644204*u*v2*y3 - 
     0.0007154743025816235*u2*v2*y3 + 0.021502574570336913*v3*y3 - 
     0.0009436436586848417*u*v3*y3 - 0.002116028409986158*v4*y3 + 
     0.39191746801048677*w*y3 - 0.06170167867079261*u*w*y3 + 
     0.009251894041935034*u2*w*y3 - 0.0007701655588751243*u3*w*y3 - 
     0.0985061281475928*v*w*y3 + 0.009712565138338942*u*v*w*y3 - 
     0.00065880688954107*u2*v*w*y3 + 0.018271879935025635*v2*w*y3 - 
     0.0008105602649905572*u*v2*w*y3 - 0.0017038721568713427*v3*w*y3 - 
     0.10781348756592685*w2*y3 + 0.010578663643644204*u*w2*y3 - 
     0.0007154743025816235*u2*w2*y3 + 0.018271879935025635*v*w2*y3 - 
     0.0008105602649905572*u*v*w2*y3 - 0.0015904265351103456*v2*w2*y3 + 
     0.021502574570336913*w3*y3 - 0.0009436436586848417*u*w3*y3 - 
     0.0017038721568713427*v*w3*y3 - 0.002116028409986158*w4*y3 + 
     0.39191746801048677*x*y3 - 0.06170167867079261*u*x*y3 + 
     0.009251894041935034*u2*x*y3 - 0.0007701655588751243*u3*x*y3 + 
     2.7762869962453007*v*x*y3 + 0.878965214508547*lnbara*v*x*y3 + 
     0.472929670266122*u*v*x*y3 + 0.11978293118391894*lnbara*u*v*x*y3 + 
     0.06433729180422629*u2*v*x*y3 + 0.018012828737752302*lnbara*u2*v*x*
      y3 - 0.6965292598704438*v2*x*y3 - 0.2252713360201013*lnbara*v2*x*
      y3 - 0.14004639793591991*u*v2*x*y3 - 0.03825655240654065*lnbara*u*
      v2*x*y3 + 0.3560786661881267*v3*x*y3 + 0.1031812587805062*lnbara*
      v3*x*y3 - 2.914384497168144*w*x*y3 - 0.878965214508547*lnbara*w*x*
      y3 - 0.4559117835501848*u*w*x*y3 - 0.11978293118391894*lnbara*u*w*x*
      y3 - 0.06557961755270206*u2*w*x*y3 - 0.018012828737752302*lnbara*u2*
      w*x*y3 + 0.01023088439838107*v*w*x*y3 - 0.0006126156464844033*u*v*w*x*
      y3 - 0.6290677486101003*v2*w*x*y3 - 0.17563835536198544*lnbara*v2*w*
      x*y3 + 0.7187404897942826*w2*x*y3 + 0.2252713360201013*lnbara*w2*x*
      y3 + 0.1387217802717251*u*w2*x*y3 + 0.03825655240654065*lnbara*u*w2*
      x*y3 + 0.6274055393084268*v*w2*x*y3 + 0.17563835536198544*lnbara*v*
      w2*x*y3 - 0.35800549766301815*w3*x*y3 - 0.1031812587805062*lnbara*
      w3*x*y3 - 0.10781348756592685*x2*y3 + 0.010578663643644204*u*x2*
      y3 - 0.0007154743025816235*u2*x2*y3 - 0.6965292598704438*v*x2*
      y3 - 0.2252713360201013*lnbara*v*x2*y3 - 0.14004639793591991*u*v*x2*
      y3 - 0.03825655240654065*lnbara*u*v*x2*y3 - 
     0.09998366236526494*v2*x2*y3 - 0.024091921466539296*lnbara*v2*x2*
      y3 + 0.7187404897942826*w*x2*y3 + 0.2252713360201013*lnbara*w*x2*
      y3 + 0.1387217802717251*u*w*x2*y3 + 0.03825655240654065*lnbara*u*w*
      x2*y3 + 0.4419629692064293*v*w*x2*y3 + 0.124791357615411*lnbara*v*w*
      x2*y3 - 0.3442555249706034*w2*x2*y3 - 0.10069943614887171*lnbara*
      w2*x2*y3 + 0.021502574570336913*x3*y3 - 0.0009436436586848417*u*
      x3*y3 + 0.3560786661881267*v*x3*y3 + 0.1031812587805062*lnbara*v*
      x3*y3 - 0.35800549766301815*w*x3*y3 - 0.1031812587805062*lnbara*w*
      x3*y3 - 0.002116028409986158*x4*y3 + 1.1560019673631434*y4 - 
     0.11864471061508094*u*y4 + 0.01526675280034565*u2*y4 - 
     0.001186695305274723*u3*y4 - 0.16545503028959443*v*y4 + 
     0.014252947529037665*u*v*y4 - 0.0009023290305323929*u2*v*y4 + 
     0.02501436947958356*v2*y4 - 0.0010454015333102693*u*v2*y4 - 
     0.002116028409986158*v3*y4 - 0.16545503028959443*w*y4 + 
     0.014252947529037665*u*w*y4 - 0.0009023290305323929*u2*w*y4 + 
     0.023437600654995495*v*w*y4 - 0.0009818697515814416*u*v*w*y4 - 
     0.0018766602068817076*v2*w*y4 + 0.02501436947958356*w2*y4 - 
     0.0010454015333102693*u*w2*y4 - 0.0018766602068817076*v*w2*y4 - 
     0.002116028409986158*w3*y4 - 0.16545503028959443*x*y4 + 
     0.014252947529037665*u*x*y4 - 0.0009023290305323929*u2*x*y4 - 
     0.8149620980179276*v*x*y4 - 0.24415700403015198*lnbara*v*x*y4 - 
     0.010048357709481526*u*v*x*y4 - 0.0006955947583152419*lnbara*u*v*x*
      y4 + 0.03885294084404612*v2*x*y4 + 0.013253738191194095*lnbara*v2*x*
      y4 + 0.8473142624781451*w*x*y4 + 0.24415700403015198*lnbara*w*x*y4 + 
     0.008352282215414328*u*w*x*y4 + 0.0006955947583152419*lnbara*u*w*x*
      y4 - 0.0010458019341617988*v*w*x*y4 - 0.04107580891265065*w2*x*y4 - 
     0.013253738191194095*lnbara*w2*x*y4 + 0.02501436947958356*x2*y4 - 
     0.0010454015333102693*u*x2*y4 + 0.03885294084404612*v*x2*y4 + 
     0.013253738191194095*lnbara*v*x2*y4 - 0.04107580891265065*w*x2*y4 - 
     0.013253738191194095*lnbara*w*x2*y4 - 0.002116028409986158*x3*y4 - 
     0.5375082140631944*y5 + 0.029948825102663987*u*y5 - 
     0.001619470152478382*u2*y5 + 0.04216772179515423*v*y5 - 
     0.0015359082311407401*u*v*y5 - 0.0027042708881657167*v2*y5 + 
     0.04216772179515423*w*y5 - 0.0015359082311407401*u*w*y5 - 
     0.0025714972415127255*v*w*y5 - 0.0027042708881657167*w2*y5 + 
     0.04216772179515423*x*y5 - 0.0015359082311407401*u*x*y5 + 
     0.13433805924390965*v*x*y5 + 0.03836752920473817*lnbara*v*x*y5 - 
     0.13785434103492666*w*x*y5 - 0.03836752920473817*lnbara*w*x*y5 - 
     0.0027042708881657167*x2*y5 + 0.14606611930718463*y6 - 
     0.0034277086455240437*u*y6 - 0.004855915459192057*v*y6 - 
     0.004855915459192057*w*y6 - 0.004855915459192057*x*y6 - 
     0.017602106974984624*y7 - 1.2020569031595942*z + 
     0.8408174973812163*u*z - 0.6431554977641232*u2*z + 
     0.39191746801048677*u3*z - 0.16545503028959443*u4*z + 
     0.04216772179515423*u5*z - 0.004855915459192057*u6*z + 
     0.6830325829486028*v*z - 2.1845063283947157*u*v*z - 
     0.6510853440804052*lnbara*u*v*z - 1.2999471003253766*u2*v*z - 
     0.4470413315421547*lnbara*u2*v*z - 0.6741348014198821*u3*v*z - 
     0.17751808906800676*lnbara*u3*v*z - 0.10064161922879965*u4*v*z - 
     0.031382877263858105*lnbara*u4*v*z - 0.011350747467801053*u5*v*z - 
     0.0025756149427488186*lnbara*u5*v*z + 1.3307155997089624*v2*z + 
     0.6510853440804052*lnbara*v2*z + 3.593282429668685*u*v2*z + 
     1.128501340498048*lnbara*u*v2*z + 1.9298767034120956*u2*v2*z + 
     0.5924457327959798*lnbara*u2*v2*z + 0.5260162467363529*u3*v2*z + 
     0.14593979274603425*lnbara*u3*v2*z + 0.049642889715863496*u4*v2*z + 
     0.01365467547098468*lnbara*u4*v2*z - 0.3344496541843748*v3*z - 
     0.24740311290228975*lnbara*v3*z - 3.884956834618231*u*v3*z - 
     1.2186672004102037*lnbara*u*v3*z - 0.47396206592202933*u2*v3*z - 
     0.13192114982378386*lnbara*u2*v3*z - 0.23348793049311065*u3*v3*z - 
     0.06654355140807486*lnbara*u3*v3*z + 0.8622872883244641*v4*z + 
     0.3193208792684922*lnbara*v4*z + 1.3489683495777027*u*v4*z + 
     0.4144777234484366*lnbara*u*v4*z + 0.1660137847279631*u2*v4*z + 
     0.0453330032683975*lnbara*u2*v4*z - 0.2093175862405338*v5*z - 
     0.07853883800228331*lnbara*v5*z - 0.27798071288879395*u*v5*z - 
     0.08187135582602693*lnbara*u*v5*z + 0.04158747525391902*v6*z + 
     0.013727259996145989*lnbara*v6*z + 0.8408174973812163*w*z + 
     1.316872732915981*u*w*z + 0.6510853440804052*lnbara*u*w*z + 
     1.7420888571164688*u2*w*z + 0.4470413315421547*lnbara*u2*w*z + 
     0.5072130866752762*u3*w*z + 0.17751808906800676*lnbara*u3*w*z + 
     0.14015372875382298*u4*w*z + 0.031382877263858105*lnbara*u4*w*z + 
     0.007029374644742698*u5*w*z + 0.0025756149427488186*lnbara*u5*w*z - 
     0.3693696142893198*v*w*z + 0.1472496149014935*u*v*w*z - 
     0.04772361051737147*u2*v*w*z + 0.01023088439838107*u3*v*w*z - 
     0.0010458019341617988*u4*v*w*z - 9.679452702174132*v2*w*z - 
     3.285801385277514*lnbara*v2*w*z + 3.2592144539581844*u*v2*w*z + 
     1.1013272044547329*lnbara*u*v2*w*z - 3.229242426355708*u2*v2*w*z - 
     0.9917935723502516*lnbara*u2*v2*w*z + 0.404386452649353*u3*v2*w*z + 
     0.12233847935377173*lnbara*u3*v2*w*z + 5.035142069385885*v3*w*z + 
     1.7095780955682178*lnbara*v3*w*z + 0.16919997458043376*u*v3*w*z + 
     0.013784793500404437*lnbara*u*v3*w*z + 0.4078728401620492*u2*v3*w*z + 
     0.12576336826926637*lnbara*u2*v3*w*z - 3.0012661757807617*v4*w*z - 
     0.9380468690654714*lnbara*v4*w*z + 0.15191010788660775*u*v4*w*z + 
     0.048049503963086576*lnbara*u*v4*w*z + 0.4769026267982312*v5*w*z + 
     0.14541778766038482*lnbara*v5*w*z - 2.4582922118695194*w2*z - 
     0.6510853440804052*lnbara*w2*z - 3.1637691242153525*u*w2*z - 
     1.128501340498048*lnbara*u*w2*z - 2.066949955700743*u2*w2*z - 
     0.5924457327959798*lnbara*u2*w2*z - 0.49681704281786926*u3*w2*z - 
     0.14593979274603425*lnbara*u3*w2*z - 0.05261848185645107*u4*w2*z - 
     0.01365467547098468*lnbara*u4*w2*z + 10.034005404400828*v*w2*z + 
     3.285801385277514*lnbara*v*w2*z - 3.3495745233238825*u*v*w2*z - 
     1.1013272044547329*lnbara*u*v*w2*z + 3.2459862158900528*u2*v*w2*z + 
     0.9917935723502516*lnbara*u2*v*w2*z - 0.40594313303596224*u3*v*w2*
      z - 0.12233847935377173*lnbara*u3*v*w2*z - 0.052439122857018475*v2*
      w2*z + 0.00780375397440978*u*v2*w2*z - 0.0006345782786901335*u2*v2*
      w2*z + 1.8755457504129727*v3*w2*z + 0.5198935985278687*lnbara*v3*
      w2*z - 0.5329098110819325*u*v3*w2*z - 0.1550921261368306*lnbara*u*
      v3*w2*z - 0.11655132895257132*v4*w2*z - 0.02876833807133764*lnbara*
      v4*w2*z + 1.0120947283767654*w3*z + 0.24740311290228975*lnbara*w3*
      z + 3.7247490277998456*u*w3*z + 1.2186672004102037*lnbara*u*w3*z + 
     0.5028126095006992*u2*w3*z + 0.13192114982378386*lnbara*u2*w3*z + 
     0.23084041467755445*u3*w3*z + 0.06654355140807486*lnbara*u3*w3*z - 
     5.165259334653691*v*w3*z - 1.7095780955682178*lnbara*v*w3*z - 
     0.15031327644943632*u*v*w3*z - 0.013784793500404437*lnbara*u*v*w3*z - 
     0.40939098371296645*u2*v*w3*z - 0.12576336826926637*lnbara*u2*v*w3*
      z - 1.8540397627858702*v2*w3*z - 0.5198935985278687*lnbara*v2*w3*
      z + 0.5315078645933379*u*v2*w3*z + 0.1550921261368306*lnbara*u*v2*
      w3*z - 0.0009436436586848417*v3*w3*z - 1.1463870292291396*w4*z - 
     0.3193208792684922*lnbara*w4*z - 1.3112778013936697*u*w4*z - 
     0.4144777234484366*lnbara*u*w4*z - 0.16893584646815507*u2*w4*z - 
     0.0453330032683975*lnbara*u2*w4*z + 3.0315936321798276*v*w4*z + 
     0.9380468690654714*lnbara*v*w4*z - 0.15381418595568414*u*v*w4*z - 
     0.048049503963086576*lnbara*u*v*w4*z + 0.11440699548555519*v2*w4*z + 
     0.02876833807133764*lnbara*v2*w4*z + 0.281434133138352*w5*z + 
     0.07853883800228331*lnbara*w5*z + 0.2738733074161405*u*w5*z + 
     0.08187135582602693*lnbara*u*w5*z - 0.48018841061091755*v*w5*z - 
     0.14541778766038482*lnbara*v*w5*z - 0.04987109935863513*w6*z - 
     0.013727259996145989*lnbara*w6*z + 0.8408174973812163*x*z - 
     0.3944622860815335*u*x*z + 0.18935004993694224*u2*x*z - 
     0.06968191432583015*u3*x*z + 0.0162776555901896*u4*x*z - 
     0.0017664062094713614*u5*x*z - 0.3693696142893198*v*x*z + 
     0.1472496149014935*u*v*x*z - 0.04772361051737147*u2*v*x*z + 
     0.01023088439838107*u3*v*x*z - 0.0010458019341617988*u4*v*x*z - 
     6.422064708307189*v2*x*z - 2.1702844802680175*lnbara*v2*x*z + 
     6.435061406178583*u*v2*x*z + 2.079207074882875*lnbara*u*v2*x*z - 
     3.373810219562036*u2*v2*x*z - 1.0395623490494532*lnbara*u2*v2*x*z + 
     0.5336843597652852*u3*v2*x*z + 0.16066085741485445*lnbara*u3*v2*x*
      z + 2.0505932975012593*v3*x*z + 0.7335375186340615*lnbara*v3*x*z - 
     0.4484159450532304*u*v3*x*z - 0.1922469483445887*lnbara*u*v3*x*z + 
     0.33319811688498185*u2*v3*x*z + 0.10536425342736426*lnbara*u2*v3*x*
      z - 1.8086906966367922*v4*x*z - 0.5574561800298969*lnbara*v4*x*z + 
     0.3835758469523267*u*v4*x*z + 0.116292709475361*lnbara*u*v4*x*z + 
     0.23350760677675905*v5*x*z + 0.07186909727514694*lnbara*v5*x*z - 
     0.3693696142893198*w*x*z + 0.1472496149014935*u*w*x*z - 
     0.04772361051737147*u2*w*x*z + 0.01023088439838107*u3*w*x*z - 
     0.0010458019341617988*u4*w*x*z + 13.318003239515274*v*w*x*z + 
     4.340568960536035*lnbara*v*w*x*z - 12.99906497164889*u*v*w*x*z - 
     4.15841414976575*lnbara*u*v*w*x*z + 6.772218474992654*u2*v*w*x*z + 
     2.0791246980989064*lnbara*u2*v*w*x*z - 1.0696904197726014*u3*v*w*x*z - 
     0.3213217148297089*lnbara*u3*v*w*x*z - 2.152616234264882*v2*w*x*z - 
     0.7335375186340615*lnbara*v2*w*x*z + 0.46479746309389686*u*v2*w*x*z + 
     0.1922469483445887*lnbara*u*v2*w*x*z - 0.3345794792700652*u2*v2*w*x*
      z - 0.10536425342736426*lnbara*u2*v2*w*x*z + 
     4.268816061721532*v3*w*x*z + 1.2629454959136155*lnbara*v3*w*x*z - 
     1.617872252781772*u*v3*w*x*z - 0.46845974278075075*lnbara*u*v3*w*x*z - 
     0.3368289563598407*v4*w*x*z - 0.10155666376810567*lnbara*v4*w*x*z - 
     6.422064708307189*w2*x*z - 2.1702844802680175*lnbara*w2*x*z + 
     6.435061406178583*u*w2*x*z + 2.079207074882875*lnbara*u*w2*x*z - 
     3.373810219562036*u2*w2*x*z - 1.0395623490494532*lnbara*u2*w2*x*z + 
     0.5336843597652852*u3*w2*x*z + 0.16066085741485445*lnbara*u3*w2*x*
      z - 2.152616234264882*v*w2*x*z - 0.7335375186340615*lnbara*v*w2*x*z + 
     0.46479746309389686*u*v*w2*x*z + 0.1922469483445887*lnbara*u*v*w2*x*
      z - 0.3345794792700652*u2*v*w2*x*z - 0.10536425342736426*lnbara*u2*v*
      w2*x*z - 4.86789835135428*v2*w2*x*z - 1.410978631767437*lnbara*v2*
      w2*x*z + 2.4648726427480128*u*v2*w2*x*z + 0.7043340666107796*lnbara*
      u*v2*w2*x*z + 0.10030472915504478*v3*w2*x*z + 
     0.029687566492958728*lnbara*v3*w2*x*z + 2.0505932975012593*w3*x*z + 
     0.7335375186340615*lnbara*w3*x*z - 0.4484159450532304*u*w3*x*z - 
     0.1922469483445887*lnbara*u*w3*x*z + 0.33319811688498185*u2*w3*x*z + 
     0.10536425342736426*lnbara*u2*w3*x*z + 4.268816061721532*v*w3*x*z + 
     1.2629454959136155*lnbara*v*w3*x*z - 1.617872252781772*u*v*w3*x*z - 
     0.46845974278075075*lnbara*u*v*w3*x*z + 0.10030472915504478*v2*w3*x*
      z + 0.029687566492958728*lnbara*v2*w3*x*z - 
     1.8086906966367922*w4*x*z - 0.5574561800298969*lnbara*w4*x*z + 
     0.3835758469523267*u*w4*x*z + 0.116292709475361*lnbara*u*w4*x*z - 
     0.3368289563598407*v*w4*x*z - 0.10155666376810567*lnbara*v*w4*x*z + 
     0.23350760677675905*w5*x*z + 0.07186909727514694*lnbara*w5*x*z - 
     0.6431554977641232*x2*z + 0.18935004993694224*u*x2*z - 
     0.05523018173756664*u2*x2*z + 0.011283905940380838*u3*x2*z - 
     0.0011239361348986583*u4*x2*z + 10.034005404400828*v*x2*z + 
     3.285801385277514*lnbara*v*x2*z - 3.3495745233238825*u*v*x2*z - 
     1.1013272044547329*lnbara*u*v*x2*z + 3.2459862158900528*u2*v*x2*z + 
     0.9917935723502516*lnbara*u2*v*x2*z - 0.40594313303596224*u3*v*x2*
      z - 0.12233847935377173*lnbara*u3*v*x2*z - 1.1288573398553086*v2*x2*
      z - 0.43966795501820194*lnbara*v2*x2*z - 1.202554183086874*u*v2*x2*
      z - 0.3723502771999754*lnbara*u*v2*x2*z - 0.08093775692251794*u2*v2*
      x2*z - 0.025975593895628257*lnbara*u2*v2*x2*z + 
     2.941587800744719*v3*x2*z + 0.8832109828874293*lnbara*v3*x2*z - 
     0.7853652276547294*u*v3*x2*z - 0.21990490700155435*lnbara*u*v3*x2*
      z - 0.21163351738140398*v4*x2*z - 0.06718897907015481*lnbara*v4*x2*
      z - 9.679452702174132*w*x2*z - 3.285801385277514*lnbara*w*x2*z + 
     3.2592144539581844*u*w*x2*z + 1.1013272044547329*lnbara*u*w*x2*z - 
     3.229242426355708*u2*w*x2*z - 0.9917935723502516*lnbara*u2*w*x2*z + 
     0.404386452649353*u3*w*x2*z + 0.12233847935377173*lnbara*u3*w*x2*z - 
     2.152616234264882*v*w*x2*z - 0.7335375186340615*lnbara*v*w*x2*z + 
     0.46479746309389686*u*v*w*x2*z + 0.1922469483445887*lnbara*u*v*w*x2*
      z - 0.3345794792700652*u2*v*w*x2*z - 0.10536425342736426*lnbara*u2*v*
      w*x2*z - 6.030472710174449*v2*w*x2*z - 1.7376394640414514*lnbara*v2*
      w*x2*z + 2.4254958617946483*u*v2*w*x2*z + 0.7008504098797653*lnbara*
      u*v2*w*x2*z + 0.2189313721141216*v3*w*x2*z + 
     0.06175769728307968*lnbara*v3*w*x2*z + 3.1418162127487004*w2*x2*z + 
     1.1732054736522635*lnbara*w2*x2*z + 0.7605706197007904*u*w2*x2*z + 
     0.1801033288553867*lnbara*u*w2*x2*z + 0.41357773400649983*u2*w2*x2*
      z + 0.1313398473229925*lnbara*u2*w2*x2*z + 
     6.391972088595844*v*w2*x2*z + 1.8709274228635118*lnbara*v*w2*x2*z - 
     2.0843530956674345*u*v*w2*x2*z - 0.6095267362125409*lnbara*u*v*w2*x2*
      z - 0.05985191547449978*v2*w2*x2*z - 0.01186851715999764*lnbara*v2*
      w2*x2*z - 3.269269763555497*w3*x2*z - 1.0164989417094896*lnbara*w3*
      x2*z + 0.44168144304165097*u*w3*x2*z + 0.12858123333432991*lnbara*u*
      w3*x2*z - 0.3134017819611535*v*w3*x2*z - 0.0993620511060278*lnbara*
      v*w3*x2*z + 0.3622126558447875*w4*x2*z + 0.11666185005310058*lnbara*
      w4*x2*z + 0.39191746801048677*x3*z - 0.06968191432583015*u*x3*z + 
     0.011283905940380838*u2*x3*z - 0.0009831878162066224*u3*x3*z - 
     5.165259334653691*v*x3*z - 1.7095780955682178*lnbara*v*x3*z - 
     0.15031327644943632*u*v*x3*z - 0.013784793500404437*lnbara*u*v*x3*z - 
     0.40939098371296645*u2*v*x3*z - 0.12576336826926637*lnbara*u2*v*x3*
      z - 1.4645041396124352*v2*x3*z - 0.37498908309415785*lnbara*v2*x3*
      z + 1.25382676464303*u*v2*x3*z + 0.3652162355419701*lnbara*u*v2*x3*
      z - 0.2434872198153215*v3*x3*z - 0.07483548907405872*lnbara*v3*x3*
      z + 5.035142069385885*w*x3*z + 1.7095780955682178*lnbara*w*x3*z + 
     0.16919997458043376*u*w*x3*z + 0.013784793500404437*lnbara*u*w*x3*z + 
     0.4078728401620492*u2*w*x3*z + 0.12576336826926637*lnbara*u2*w*x3*
      z + 4.268816061721532*v*w*x3*z + 1.2629454959136155*lnbara*v*w*x3*z - 
     1.617872252781772*u*v*w*x3*z - 0.46845974278075075*lnbara*u*v*w*x3*z + 
     0.2189313721141216*v2*w*x3*z + 0.06175769728307968*lnbara*v2*w*x3*
      z - 2.775770631100383*w2*x3*z - 0.8879564128194577*lnbara*w2*x3*z + 
     0.3619717726179578*u*w2*x3*z + 0.10324350723878072*lnbara*u*w2*x3*
      z - 0.3134017819611535*v*w2*x3*z - 0.0993620511060278*lnbara*v*w2*
      x3*z + 0.33499391115566624*w3*x3*z + 0.11243984289700684*lnbara*w3*
      x3*z - 0.16545503028959443*x4*z + 0.0162776555901896*u*x4*z - 
     0.0011239361348986583*u2*x4*z + 3.0315936321798276*v*x4*z + 
     0.9380468690654714*lnbara*v*x4*z - 0.15381418595568414*u*v*x4*z - 
     0.048049503963086576*lnbara*u*v*x4*z + 0.0011325127373185734*v2*x4*
      z - 0.009074544432852395*lnbara*v2*x4*z - 3.0012661757807617*w*x4*
      z - 0.9380468690654714*lnbara*w*x4*z + 0.15191010788660775*u*w*x4*z + 
     0.048049503963086576*lnbara*u*w*x4*z - 0.3368289563598407*v*w*x4*z - 
     0.10155666376810567*lnbara*v*w*x4*z + 0.3328573832991313*w2*x4*z + 
     0.11063120820095806*lnbara*w2*x4*z + 0.04216772179515423*x5*z - 
     0.0017664062094713614*u*x5*z - 0.48018841061091755*v*x5*z - 
     0.14541778766038482*lnbara*v*x5*z + 0.4769026267982312*w*x5*z + 
     0.14541778766038482*lnbara*w*x5*z - 0.004855915459192057*x6*z + 
     0.8408174973812163*y*z - 0.3693696142893198*u*y*z + 
     0.17096212544446912*u2*y*z - 0.06170167867079261*u3*y*z + 
     0.014252947529037665*u4*y*z - 0.0015359082311407401*u5*y*z - 
     5.814779756605509*v*y*z - 1.9532560322412158*lnbara*v*y*z - 
     3.3297094342481097*u*v*y*z - 1.0155813225862613*lnbara*u*v*y*z - 
     1.7018138873778625*u2*v*y*z - 0.49610466935343467*lnbara*u2*v*y*z - 
     0.373690675987211*u3*v*y*z - 0.10319148379269107*lnbara*u3*v*y*z - 
     0.03908459842732547*u4*v*y*z - 0.01035774990633652*lnbara*u4*v*y*z + 
     4.998355981731551*v2*y*z + 1.7188373548274771*lnbara*v2*y*z + 
     3.6274999402312313*u*v2*y*z + 1.101558132258626*lnbara*u*v2*y*z + 
     1.2958237235724512*u2*v2*y*z + 0.37253832538857234*lnbara*u2*v2*y*
      z + 0.1611431319221526*u3*v2*y*z + 0.043853270547930576*lnbara*u3*
      v2*y*z - 6.4531586807126216*v3*y*z - 2.0595906539260844*lnbara*v3*y*
      z - 1.0928490356968497*u*v3*y*z - 0.30677398634465897*lnbara*u*v3*y*
      z - 0.5203695633112709*u2*v3*y*z - 0.14879519944448213*lnbara*u2*v3*
      y*z + 2.4002692887025985*v4*y*z + 0.7507653584772783*lnbara*v4*y*z + 
     0.29707382410626676*u*v4*y*z + 0.0809778929894431*lnbara*u*v4*y*z - 
     0.48202800643018173*v5*y*z - 0.1438457497350551*lnbara*v5*y*z + 
     5.050947856234655*w*y*z + 1.9532560322412158*lnbara*w*y*z + 
     3.6089086211615613*u*w*y*z + 1.0155813225862613*lnbara*u*w*y*z + 
     1.6153412124624802*u2*w*y*z + 0.49610466935343467*lnbara*u2*w*y*z + 
     0.39176531410887067*u3*w*y*z + 0.10319148379269107*lnbara*u3*w*y*z + 
     0.037264929316591394*u4*w*y*z + 0.01035774990633652*lnbara*u4*w*y*z + 
     0.1472496149014935*v*w*y*z - 0.03663932564661811*u*v*w*y*z + 
     0.006668952902327549*u2*v*w*y*z - 0.0006126156464844033*u3*v*w*y*z + 
     6.647579498888072*v2*w*y*z + 2.014469236002691*lnbara*v2*w*y*z - 
     2.5690901893905513*u*v2*w*y*z - 0.8128752507655377*lnbara*u*v2*w*y*z + 
     0.3011577157359468*u2*v2*w*y*z + 0.09017258826251653*lnbara*u2*v2*w*
      y*z - 0.7027857105758675*v3*w*y*z - 0.20757337570539203*lnbara*v3*w*y*
      z + 0.3671622432587578*u*v3*w*y*z + 0.11464743468556707*lnbara*u*v3*w*
      y*z + 0.36905978156971897*v4*w*y*z + 0.10073935030932556*lnbara*v4*w*
      y*z - 4.63804380635014*w2*y*z - 1.7188373548274771*lnbara*w2*y*z - 
     3.7155448088414325*u*w2*y*z - 1.101558132258626*lnbara*u*w2*y*z - 
     1.279932716374738*u2*w2*y*z - 0.37253832538857234*lnbara*u2*w2*y*z - 
     0.16259660502488668*u3*w2*y*z - 0.043853270547930576*lnbara*u3*w2*y*
      z - 6.7414545262279955*v*w2*y*z - 2.014469236002691*lnbara*v*w2*y*z + 
     2.582819839347*u*v*w2*y*z + 0.8128752507655377*lnbara*u*v*w2*y*z - 
     0.30225971597078544*u2*v*w2*y*z - 0.09017258826251653*lnbara*u2*v*w2*
      y*z + 0.008768669407230721*v2*w2*y*z - 0.0005708021200927833*u*v2*
      w2*y*z - 0.8924547277374686*v3*w2*y*z - 0.2518959384495211*lnbara*
      v3*w2*y*z + 6.321775087715999*w3*y*z + 2.0595906539260844*lnbara*w3*
      y*z + 1.1114419930785513*u*w3*y*z + 0.30677398634465897*lnbara*u*w3*y*
      z + 0.5189008798963198*u2*w3*y*z + 0.14879519944448213*lnbara*u2*w3*
      y*z + 0.7227291601125875*v*w3*y*z + 0.20757337570539203*lnbara*v*w3*y*
      z - 0.368445017992858*u*v*w3*y*z - 0.11464743468556707*lnbara*u*v*w3*
      y*z + 0.8908183968952704*v2*w3*y*z + 0.2518959384495211*lnbara*v2*
      w3*y*z - 2.3697386855833713*w4*y*z - 0.7507653584772783*lnbara*w4*y*
      z - 0.2989574253995812*u*w4*y*z - 0.0809778929894431*lnbara*u*w4*y*
      z - 0.3710874532554622*v*w4*y*z - 0.10073935030932556*lnbara*v*w4*y*
      z + 0.4787256919895696*w5*y*z + 0.1438457497350551*lnbara*w5*y*z - 
     0.4982639811894151*x*y*z + 0.1472496149014935*u*x*y*z - 
     0.04263645884832675*u2*x*y*z + 0.008655813732616391*u3*x*y*z - 
     0.0008582761349145647*u4*x*y*z + 0.1472496149014935*v*x*y*z - 
     0.03663932564661811*u*v*x*y*z + 0.006668952902327549*u2*v*x*y*z - 
     0.0006126156464844033*u3*v*x*y*z + 4.806051314717304*v2*x*y*z + 
     1.4974962913849321*lnbara*v2*x*y*z - 4.358084874395133*u*v2*x*y*z - 
     1.310301658288488*lnbara*u*v2*x*y*z + 0.44246231471333614*u2*v2*x*y*
      z + 0.13612276309543284*lnbara*u2*v2*x*y*z - 
     0.03145999950209565*v3*x*y*z - 0.041372947073440955*lnbara*v3*x*y*z + 
     0.5529561454818885*u*v3*x*y*z + 0.1674546937036158*lnbara*u*v3*x*y*z + 
     0.2932474952337439*v4*x*y*z + 0.08442890598101545*lnbara*v4*x*y*z + 
     0.1472496149014935*w*x*y*z - 0.03663932564661811*u*w*x*y*z + 
     0.006668952902327549*u2*w*x*y*z - 0.0006126156464844033*u3*w*x*y*z - 
     9.734014872777879*v*w*x*y*z - 2.9949925827698642*lnbara*v*w*x*y*z + 
     8.735313288561295*u*v*w*x*y*z + 2.620603316576976*lnbara*u*v*w*x*y*z - 
     0.8865113815143282*u2*v*w*x*y*z - 0.27224552619086567*lnbara*u2*v*w*x*
      y*z + 0.04678476613703959*v2*w*x*y*z + 0.041372947073440955*lnbara*v2*
      w*x*y*z - 0.5540525639589041*u*v2*w*x*y*z - 0.1674546937036158*lnbara*
      u*v2*w*x*y*z - 1.3152966221164228*v3*w*x*y*z - 
     0.3724255234004633*lnbara*v3*w*x*y*z + 4.806051314717304*w2*x*y*z + 
     1.4974962913849321*lnbara*w2*x*y*z - 4.358084874395133*u*w2*x*y*z - 
     1.310301658288488*lnbara*u*w2*x*y*z + 0.44246231471333614*u2*w2*x*y*
      z + 0.13612276309543284*lnbara*u2*w2*x*y*z + 
     0.04678476613703959*v*w2*x*y*z + 0.041372947073440955*lnbara*v*w2*x*y*
      z - 0.5540525639589041*u*v*w2*x*y*z - 0.1674546937036158*lnbara*u*v*
      w2*x*y*z + 2.0406049955739975*v2*w2*x*y*z + 
     0.5759932348388958*lnbara*v2*w2*x*y*z - 0.03145999950209565*w3*x*y*
      z - 0.041372947073440955*lnbara*w3*x*y*z + 0.5529561454818885*u*w3*x*
      y*z + 0.1674546937036158*lnbara*u*w3*x*y*z - 
     1.3152966221164228*v*w3*x*y*z - 0.3724255234004633*lnbara*v*w3*x*y*z + 
     0.2932474952337439*w4*x*y*z + 0.08442890598101545*lnbara*w4*x*y*z + 
     0.25855118000886335*x2*y*z - 0.04772361051737147*u*x2*y*z + 
     0.00780375397440978*u2*x2*y*z - 0.0006817048264512694*u3*x2*y*z - 
     6.7414545262279955*v*x2*y*z - 2.014469236002691*lnbara*v*x2*y*z + 
     2.582819839347*u*v*x2*y*z + 0.8128752507655377*lnbara*u*v*x2*y*z - 
     0.30225971597078544*u2*v*x2*y*z - 0.09017258826251653*lnbara*u2*v*x2*
      y*z + 0.16780437876971865*v2*x2*y*z + 0.05368132013057931*lnbara*v2*
      x2*y*z + 0.00017982275152600617*u*v2*x2*y*z - 
     0.008148863899001088*lnbara*u*v2*x2*y*z - 0.9153400045660104*v3*x2*y*
      z - 0.25553842398615445*lnbara*v3*x2*y*z + 6.647579498888072*w*x2*y*
      z + 2.014469236002691*lnbara*w*x2*y*z - 2.5690901893905513*u*w*x2*y*
      z - 0.8128752507655377*lnbara*u*w*x2*y*z + 0.3011577157359468*u2*w*
      x2*y*z + 0.09017258826251653*lnbara*u2*w*x2*y*z + 
     0.04678476613703959*v*w*x2*y*z + 0.041372947073440955*lnbara*v*w*x2*y*
      z - 0.5540525639589041*u*v*w*x2*y*z - 0.1674546937036158*lnbara*u*v*w*
      x2*y*z + 2.53079627759307*v2*w*x2*y*z + 0.7137169391031986*lnbara*
      v2*w*x2*y*z - 0.19274016063176588*w2*x2*y*z - 
     0.09505426720402026*lnbara*w2*x2*y*z + 0.5522979595539836*u*w2*x2*y*
      z + 0.17560355760261687*lnbara*u*w2*x2*y*z - 
     2.1878006392634903*v*w2*x2*y*z - 0.6177543113748135*lnbara*v*w2*x2*y*
      z + 0.569975957295371*w3*x2*y*z + 0.15957579625776946*lnbara*w3*x2*
      y*z - 0.0985061281475928*x3*y*z + 0.01023088439838107*u*x3*y*z - 
     0.0007202416621432963*u2*x3*y*z + 0.7227291601125875*v*x3*y*z + 
     0.20757337570539203*lnbara*v*x3*y*z - 0.368445017992858*u*v*x3*y*z - 
     0.11464743468556707*lnbara*u*v*x3*y*z + 0.8410658901759228*v2*x3*y*
      z + 0.24067518005948615*lnbara*v2*x3*y*z - 0.7027857105758675*w*x3*y*
      z - 0.20757337570539203*lnbara*w*x3*y*z + 0.3671622432587578*u*w*x3*y*
      z + 0.11464743468556707*lnbara*u*w*x3*y*z - 1.3152966221164228*v*w*x3*
      y*z - 0.3724255234004633*lnbara*v*w*x3*y*z + 
     0.4722390677423313*w2*x3*y*z + 0.13175034334097716*lnbara*w2*x3*y*
      z + 0.023437600654995495*x4*y*z - 0.0010458019341617988*u*x4*y*z - 
     0.3710874532554622*v*x4*y*z - 0.10073935030932556*lnbara*v*x4*y*z + 
     0.36905978156971897*w*x4*y*z + 0.10073935030932556*lnbara*w*x4*y*z - 
     0.0025714972415127255*x5*y*z - 0.6431554977641232*y2*z + 
     0.17096212544446912*u*y2*z - 0.04689698042164179*u2*y2*z + 
     0.009251894041935034*u3*y2*z - 0.0009023290305323929*u4*y2*z + 
     3.150452651970475*v*y2*z + 0.9766280161206079*lnbara*v*y2*z - 
     0.5219825160658177*u*v*y2*z - 0.18363961128442552*lnbara*u*v*y2*z - 
     0.335966498688551*u2*v*y2*z - 0.09418475573467341*lnbara*u2*v*y2*z - 
     0.05665911334956437*u3*v*y2*z - 0.015395299386842583*lnbara*u3*v*y2*
      z - 0.022693681826607907*v2*y2*z - 0.027337198387939212*lnbara*v2*
      y2*z + 0.7675738685886592*u*v2*y2*z + 0.22874938189748867*lnbara*u*
      v2*y2*z + 0.17593076120022283*u2*v2*y2*z + 
     0.04860228502711452*lnbara*u2*v2*y2*z + 0.8444774478932007*v3*y2*
      z + 0.248860063558088*lnbara*v3*y2*z - 0.3385312614440469*u*v3*y2*
      z - 0.09925662865561737*lnbara*u*v3*y2*z - 0.047668105888734366*v4*
      y2*z - 0.015411950745225283*lnbara*v4*y2*z - 
     2.777512025251304*w*y2*z - 0.9766280161206079*lnbara*w*y2*z + 
     0.4390247991246611*u*w*y2*z + 0.18363961128442552*lnbara*u*w*y2*z + 
     0.3502072962213621*u2*w*y2*z + 0.09418475573467341*lnbara*u2*w*y2*
      z + 0.055393310008639646*u3*w*y2*z + 0.015395299386842583*lnbara*u3*
      w*y2*z - 0.04772361051737147*v*w*y2*z + 0.006668952902327549*u*v*w*
      y2*z - 0.000520176702770554*u2*v*w*y2*z - 4.148276730172979*v2*w*
      y2*z - 1.2223037533945342*lnbara*v2*w*y2*z + 
     0.15794313861675344*u*v2*w*y2*z + 0.05241767434109274*lnbara*u*v2*w*
      y2*z + 0.5075561415724656*v3*w*y2*z + 0.15134262670003695*lnbara*v3*
      w*y2*z - 0.08497562276797721*w2*y2*z + 0.027337198387939212*lnbara*
      w2*y2*z - 0.7526477768237668*u*w2*y2*z - 0.22874938189748867*lnbara*
      u*w2*y2*z - 0.1770896870636047*u2*w2*y2*z - 
     0.04860228502711452*lnbara*u2*w2*y2*z + 4.165985435140145*v*w2*y2*
      z + 1.2223037533945342*lnbara*v*w2*y2*z - 0.15906541536540864*u*v*w2*
      y2*z - 0.05241767434109274*lnbara*u*v*w2*y2*z - 
     0.0007553212077869194*v2*w2*y2*z - 0.8226148783091756*w3*y2*z - 
     0.248860063558088*lnbara*w3*y2*z + 0.33716810694416005*u*w3*y2*z + 
     0.09925662865561737*lnbara*u*w3*y2*z - 0.5092031405619221*v*w3*y2*
      z - 0.15134262670003695*lnbara*v*w3*y2*z + 0.04549876822052544*w4*
      y2*z + 0.015411950745225283*lnbara*w4*y2*z + 
     0.25855118000886335*x*y2*z - 0.04615141682255245*u*x*y2*z + 
     0.007376277398255051*u2*x*y2*z - 0.0006353292970921247*u3*x*y2*z - 
     0.04772361051737147*v*x*y2*z + 0.006668952902327549*u*v*x*y2*z - 
     0.000520176702770554*u2*v*x*y2*z - 2.5782754794730662*v2*x*y2*z - 
     0.7595995680938061*lnbara*v2*x*y2*z + 0.6316944039353403*u*v2*x*y2*
      z + 0.18411867864157963*lnbara*u*v2*x*y2*z + 
     0.15576929487146673*v3*x*y2*z + 0.05027670415149837*lnbara*v3*x*y2*
      z - 0.04772361051737147*w*x*y2*z + 0.006668952902327549*u*w*x*y2*z - 
     0.000520176702770554*u2*w*x*y2*z + 5.179219163949073*v*w*x*y2*z + 
     1.5191991361876123*lnbara*v*w*x*y2*z - 1.2649350069307033*u*v*w*x*y2*
      z - 0.36823735728315926*lnbara*u*v*w*x*y2*z - 
     0.1570218018180108*v2*w*x*y2*z - 0.05027670415149837*lnbara*v2*w*x*
      y2*z - 2.5782754794730662*w2*x*y2*z - 0.7595995680938061*lnbara*w2*
      x*y2*z + 0.6316944039353403*u*w2*x*y2*z + 0.18411867864157963*lnbara*
      u*w2*x*y2*z - 0.1570218018180108*v*w2*x*y2*z - 
     0.05027670415149837*lnbara*v*w2*x*y2*z + 0.15576929487146673*w3*x*y2*
      z + 0.05027670415149837*lnbara*w3*x*y2*z - 0.08463412943162905*x2*
      y2*z + 0.008768669407230721*u*x2*y2*z - 0.0006133788582035897*u2*
      x2*y2*z + 4.165985435140145*v*x2*y2*z + 1.2223037533945342*lnbara*v*
      x2*y2*z - 0.15906541536540864*u*v*x2*y2*z - 
     0.05241767434109274*lnbara*u*v*x2*y2*z - 0.21678745401254226*v2*x2*
      y2*z - 0.06933321435793802*lnbara*v2*x2*y2*z - 
     4.148276730172979*w*x2*y2*z - 1.2223037533945342*lnbara*w*x2*y2*z + 
     0.15794313861675344*u*w*x2*y2*z + 0.05241767434109274*lnbara*u*w*x2*
      y2*z - 0.1570218018180108*v*w*x2*y2*z - 0.05027670415149837*lnbara*v*
      w*x2*y2*z + 0.37199049657356653*w2*x2*y2*z + 
     0.11960991850943639*lnbara*w2*x2*y2*z + 0.018271879935025635*x3*y2*
      z - 0.0008257705772076173*u*x3*y2*z - 0.5092031405619221*v*x3*y2*
      z - 0.15134262670003695*lnbara*v*x3*y2*z + 0.5075561415724656*w*x3*
      y2*z + 0.15134262670003695*lnbara*w*x3*y2*z - 
     0.0018766602068817076*x4*y2*z + 0.39191746801048677*y3*z - 
     0.06170167867079261*u*y3*z + 0.009251894041935034*u2*y3*z - 
     0.0007701655588751243*u3*y3*z - 2.914384497168144*v*y3*z - 
     0.878965214508547*lnbara*v*y3*z - 0.4559117835501848*u*v*y3*z - 
     0.11978293118391894*lnbara*u*v*y3*z - 0.06557961755270206*u2*v*y3*z - 
     0.018012828737752302*lnbara*u2*v*y3*z + 0.7187404897942826*v2*y3*z + 
     0.2252713360201013*lnbara*v2*y3*z + 0.1387217802717251*u*v2*y3*z + 
     0.03825655240654065*lnbara*u*v2*y3*z - 0.35800549766301815*v3*y3*z - 
     0.1031812587805062*lnbara*v3*y3*z + 2.7762869962453007*w*y3*z + 
     0.878965214508547*lnbara*w*y3*z + 0.472929670266122*u*w*y3*z + 
     0.11978293118391894*lnbara*u*w*y3*z + 0.06433729180422629*u2*w*y3*z + 
     0.018012828737752302*lnbara*u2*w*y3*z + 0.01023088439838107*v*w*y3*
      z - 0.0006126156464844033*u*v*w*y3*z + 0.6274055393084268*v2*w*y3*
      z + 0.17563835536198544*lnbara*v2*w*y3*z - 0.6965292598704438*w2*y3*
      z - 0.2252713360201013*lnbara*w2*y3*z - 0.14004639793591991*u*w2*y3*
      z - 0.03825655240654065*lnbara*u*w2*y3*z - 0.6290677486101003*v*w2*
      y3*z - 0.17563835536198544*lnbara*v*w2*y3*z + 
     0.3560786661881267*w3*y3*z + 0.1031812587805062*lnbara*w3*y3*z - 
     0.0985061281475928*x*y3*z + 0.009712565138338942*u*x*y3*z - 
     0.00065880688954107*u2*x*y3*z + 0.01023088439838107*v*x*y3*z - 
     0.0006126156464844033*u*v*x*y3*z + 0.4419629692064293*v2*x*y3*z + 
     0.124791357615411*lnbara*v2*x*y3*z + 0.01023088439838107*w*x*y3*z - 
     0.0006126156464844033*u*w*x*y3*z - 0.8860365808247611*v*w*x*y3*z - 
     0.249582715230822*lnbara*v*w*x*y3*z + 0.4419629692064293*w2*x*y3*z + 
     0.124791357615411*lnbara*w2*x*y3*z + 0.018271879935025635*x2*y3*z - 
     0.0008105602649905572*u*x2*y3*z - 0.6290677486101003*v*x2*y3*z - 
     0.17563835536198544*lnbara*v*x2*y3*z + 0.6274055393084268*w*x2*y3*
      z + 0.17563835536198544*lnbara*w*x2*y3*z - 0.0017038721568713427*x3*
      y3*z - 0.16545503028959443*y4*z + 0.014252947529037665*u*y4*z - 
     0.0009023290305323929*u2*y4*z + 0.8473142624781451*v*y4*z + 
     0.24415700403015198*lnbara*v*y4*z + 0.008352282215414328*u*v*y4*z + 
     0.0006955947583152419*lnbara*u*v*y4*z - 0.04107580891265065*v2*y4*z - 
     0.013253738191194095*lnbara*v2*y4*z - 0.8149620980179276*w*y4*z - 
     0.24415700403015198*lnbara*w*y4*z - 0.010048357709481526*u*w*y4*z - 
     0.0006955947583152419*lnbara*u*w*y4*z - 0.0010458019341617988*v*w*y4*
      z + 0.03885294084404612*w2*y4*z + 0.013253738191194095*lnbara*w2*y4*
      z + 0.023437600654995495*x*y4*z - 0.0009818697515814416*u*x*y4*z - 
     0.0010458019341617988*v*x*y4*z - 0.0010458019341617988*w*x*y4*z - 
     0.0018766602068817076*x2*y4*z + 0.04216772179515423*y5*z - 
     0.0015359082311407401*u*y5*z - 0.13785434103492666*v*y5*z - 
     0.03836752920473817*lnbara*v*y5*z + 0.13433805924390965*w*y5*z + 
     0.03836752920473817*lnbara*w*y5*z - 0.0025714972415127255*x*y5*z - 
     0.004855915459192057*y6*z + 1.5830194232420491*z2 - 
     0.6431554977641232*u*z2 + 0.29720782742626456*u2*z2 - 
     0.10781348756592685*u3*z2 + 0.02501436947958356*u4*z2 - 
     0.0027042708881657167*u5*z2 + 1.3307155997089624*v*z2 + 
     0.6510853440804052*lnbara*v*z2 + 3.593282429668685*u*v*z2 + 
     1.128501340498048*lnbara*u*v*z2 + 1.9298767034120956*u2*v*z2 + 
     0.5924457327959798*lnbara*u2*v*z2 + 0.5260162467363529*u3*v*z2 + 
     0.14593979274603425*lnbara*u3*v*z2 + 0.049642889715863496*u4*v*z2 + 
     0.01365467547098468*lnbara*u4*v*z2 + 1.5951827813991792*v2*z2 + 
     0.37330756630262746*lnbara*v2*z2 - 8.153252996171776*u*v2*z2 - 
     2.598243148725863*lnbara*u*v2*z2 + 0.6456564865390652*u2*v2*z2 + 
     0.21593510082621092*lnbara*u2*v2*z2 - 0.5616689792286329*u3*v2*
      z2 - 0.16389022386115257*lnbara*u3*v2*z2 + 
     0.403289978849488*v3*z2 + 0.15003403088345468*lnbara*v3*z2 + 
     2.3470147044696774*u*v3*z2 + 0.7642509739315361*lnbara*u*v3*z2 + 
     0.14483466237687437*u2*v3*z2 + 0.03394368436113497*lnbara*u2*v3*
      z2 + 0.1913662943454968*v4*z2 + 0.04215619932204285*lnbara*v4*z2 - 
     0.5618260667674302*u*v4*z2 - 0.17055352864149378*lnbara*u*v4*z2 + 
     0.004695610230502542*v5*z2 + 0.002302749325478825*lnbara*v5*z2 - 
     2.4582922118695194*w*z2 - 0.6510853440804052*lnbara*w*z2 - 
     3.1637691242153525*u*w*z2 - 1.128501340498048*lnbara*u*w*z2 - 
     2.066949955700743*u2*w*z2 - 0.5924457327959798*lnbara*u2*w*z2 - 
     0.49681704281786926*u3*w*z2 - 0.14593979274603425*lnbara*u3*w*z2 - 
     0.05261848185645107*u4*w*z2 - 0.01365467547098468*lnbara*u4*w*z2 - 
     6.422064708307189*v*w*z2 - 2.1702844802680175*lnbara*v*w*z2 + 
     6.435061406178583*u*v*w*z2 + 2.079207074882875*lnbara*u*v*w*z2 - 
     3.373810219562036*u2*v*w*z2 - 1.0395623490494532*lnbara*u2*v*w*z2 + 
     0.5336843597652852*u3*v*w*z2 + 0.16066085741485445*lnbara*u3*v*w*
      z2 + 3.1418162127487004*v2*w*z2 + 1.1732054736522635*lnbara*v2*w*
      z2 + 0.7605706197007904*u*v2*w*z2 + 0.1801033288553867*lnbara*u*v2*
      w*z2 + 0.41357773400649983*u2*v2*w*z2 + 0.1313398473229925*lnbara*
      u2*v2*w*z2 - 2.775770631100383*v3*w*z2 - 
     0.8879564128194577*lnbara*v3*w*z2 + 0.3619717726179578*u*v3*w*z2 + 
     0.10324350723878072*lnbara*u*v3*w*z2 + 0.3328573832991313*v4*w*z2 + 
     0.11063120820095806*lnbara*v4*w*z2 + 5.493378448409817*w2*z2 + 
     1.79697691396539*lnbara*w2*z2 + 1.5405090633173697*u*w2*z2 + 
     0.5190360738429882*lnbara*u*w2*z2 + 2.7618349656998586*u2*w2*z2 + 
     0.8236272482232423*lnbara*u2*w2*z2 + 0.024812552253286976*u3*w2*
      z2 + 0.003229366446298125*lnbara*u3*w2*z2 - 
     1.1288573398553086*v*w2*z2 - 0.43966795501820194*lnbara*v*w2*z2 - 
     1.202554183086874*u*v*w2*z2 - 0.3723502771999754*lnbara*u*v*w2*z2 - 
     0.08093775692251794*u2*v*w2*z2 - 0.025975593895628257*lnbara*u2*v*
      w2*z2 + 2.445843105402685*v2*w2*z2 + 0.7054893158837185*lnbara*v2*
      w2*z2 - 1.2333152992895828*u*v2*w2*z2 - 0.3521670333053898*lnbara*
      u*v2*w2*z2 - 0.04072484698113826*v3*w2*z2 - 
     0.013759874074562587*lnbara*v3*w2*z2 - 2.6912970732885757*w3*z2 - 
     0.8835715495175162*lnbara*w3*z2 - 1.8613624203011476*u*w3*z2 - 
     0.5720040255869473*lnbara*u*w3*z2 - 0.48114924036453877*u2*w3*z2 - 
     0.13930793778849923*lnbara*u2*w3*z2 - 1.4645041396124352*v*w3*z2 - 
     0.37498908309415785*lnbara*v*w3*z2 + 1.25382676464303*u*v*w3*z2 + 
     0.3652162355419701*lnbara*u*v*w3*z2 - 0.061694139411974463*v2*w3*
      z2 - 0.01592769241839614*lnbara*v2*w3*z2 + 
     1.6718584721002623*w4*z2 + 0.5152999807078541*lnbara*w4*z2 + 
     0.17448936082610797*u*w4*z2 + 0.05426081916613278*lnbara*u*w4*z2 + 
     0.0011325127373185734*v*w4*z2 - 0.009074544432852395*lnbara*v*w4*
      z2 - 0.24406286627904641*w5*z2 - 0.07417184660062576*lnbara*w5*
      z2 - 0.6431554977641232*x*z2 + 0.18935004993694224*u*x*z2 - 
     0.05523018173756664*u2*x*z2 + 0.011283905940380838*u3*x*z2 - 
     0.0011239361348986583*u4*x*z2 - 9.679452702174132*v*x*z2 - 
     3.285801385277514*lnbara*v*x*z2 + 3.2592144539581844*u*v*x*z2 + 
     1.1013272044547329*lnbara*u*v*x*z2 - 3.229242426355708*u2*v*x*z2 - 
     0.9917935723502516*lnbara*u2*v*x*z2 + 0.404386452649353*u3*v*x*z2 + 
     0.12233847935377173*lnbara*u3*v*x*z2 + 3.1418162127487004*v2*x*z2 + 
     1.1732054736522635*lnbara*v2*x*z2 + 0.7605706197007904*u*v2*x*z2 + 
     0.1801033288553867*lnbara*u*v2*x*z2 + 0.41357773400649983*u2*v2*x*
      z2 + 0.1313398473229925*lnbara*u2*v2*x*z2 - 
     3.269269763555497*v3*x*z2 - 1.0164989417094896*lnbara*v3*x*z2 + 
     0.44168144304165097*u*v3*x*z2 + 0.12858123333432991*lnbara*u*v3*x*
      z2 + 0.3622126558447875*v4*x*z2 + 0.11666185005310058*lnbara*v4*x*
      z2 + 10.034005404400828*w*x*z2 + 3.285801385277514*lnbara*w*x*z2 - 
     3.3495745233238825*u*w*x*z2 - 1.1013272044547329*lnbara*u*w*x*z2 + 
     3.2459862158900528*u2*w*x*z2 + 0.9917935723502516*lnbara*u2*w*x*z2 - 
     0.40594313303596224*u3*w*x*z2 - 0.12233847935377173*lnbara*u3*w*x*
      z2 - 2.152616234264882*v*w*x*z2 - 0.7335375186340615*lnbara*v*w*x*
      z2 + 0.46479746309389686*u*v*w*x*z2 + 0.1922469483445887*lnbara*u*v*w*
      x*z2 - 0.3345794792700652*u2*v*w*x*z2 - 0.10536425342736426*lnbara*
      u2*v*w*x*z2 + 6.391972088595844*v2*w*x*z2 + 
     1.8709274228635118*lnbara*v2*w*x*z2 - 2.0843530956674345*u*v2*w*x*
      z2 - 0.6095267362125409*lnbara*u*v2*w*x*z2 - 
     0.3134017819611535*v3*w*x*z2 - 0.0993620511060278*lnbara*v3*w*x*z2 - 
     1.1288573398553086*w2*x*z2 - 0.43966795501820194*lnbara*w2*x*z2 - 
     1.202554183086874*u*w2*x*z2 - 0.3723502771999754*lnbara*u*w2*x*z2 - 
     0.08093775692251794*u2*w2*x*z2 - 0.025975593895628257*lnbara*u2*w2*
      x*z2 - 6.030472710174449*v*w2*x*z2 - 1.7376394640414514*lnbara*v*w2*
      x*z2 + 2.4254958617946483*u*v*w2*x*z2 + 0.7008504098797653*lnbara*u*
      v*w2*x*z2 - 0.05985191547449978*v2*w2*x*z2 - 
     0.01186851715999764*lnbara*v2*w2*x*z2 + 2.941587800744719*w3*x*z2 + 
     0.8832109828874293*lnbara*w3*x*z2 - 0.7853652276547294*u*w3*x*z2 - 
     0.21990490700155435*lnbara*u*w3*x*z2 + 0.2189313721141216*v*w3*x*
      z2 + 0.06175769728307968*lnbara*v*w3*x*z2 - 
     0.21163351738140398*w4*x*z2 - 0.06718897907015481*lnbara*w4*x*z2 + 
     0.29720782742626456*x2*z2 - 0.05523018173756664*u*x2*z2 + 
     0.009104692162046783*u2*x2*z2 - 0.0008001063695529314*u3*x2*z2 - 
     0.052439122857018475*v*x2*z2 + 0.00780375397440978*u*v*x2*z2 - 
     0.0006345782786901335*u2*v*x2*z2 + 2.445843105402685*v2*x2*z2 + 
     0.7054893158837185*lnbara*v2*x2*z2 - 1.2333152992895828*u*v2*x2*
      z2 - 0.3521670333053898*lnbara*u*v2*x2*z2 + 
     0.05859089416673988*v3*x2*z2 + 0.01186851715999764*lnbara*v3*x2*
      z2 - 0.052439122857018475*w*x2*z2 + 0.00780375397440978*u*w*x2*
      z2 - 0.0006345782786901335*u2*w*x2*z2 - 4.86789835135428*v*w*x2*
      z2 - 1.410978631767437*lnbara*v*w*x2*z2 + 2.4648726427480128*u*v*w*
      x2*z2 + 0.7043340666107796*lnbara*u*v*w*x2*z2 - 
     0.05985191547449978*v2*w*x2*z2 - 0.01186851715999764*lnbara*v2*w*x2*
      z2 + 2.445843105402685*w2*x2*z2 + 0.7054893158837185*lnbara*w2*x2*
      z2 - 1.2333152992895828*u*w2*x2*z2 - 0.3521670333053898*lnbara*u*
      w2*x2*z2 - 0.05985191547449978*v*w2*x2*z2 - 
     0.01186851715999764*lnbara*v*w2*x2*z2 + 0.05859089416673988*w3*x2*
      z2 + 0.01186851715999764*lnbara*w3*x2*z2 - 
     0.10781348756592685*x3*z2 + 0.011283905940380838*u*x3*z2 - 
     0.0008001063695529314*u2*x3*z2 - 1.8540397627858702*v*x3*z2 - 
     0.5198935985278687*lnbara*v*x3*z2 + 0.5315078645933379*u*v*x3*z2 + 
     0.1550921261368306*lnbara*u*v*x3*z2 - 0.061694139411974463*v2*x3*
      z2 - 0.01592769241839614*lnbara*v2*x3*z2 + 
     1.8755457504129727*w*x3*z2 + 0.5198935985278687*lnbara*w*x3*z2 - 
     0.5329098110819325*u*w*x3*z2 - 0.1550921261368306*lnbara*u*w*x3*z2 + 
     0.10030472915504478*v*w*x3*z2 + 0.029687566492958728*lnbara*v*w*x3*
      z2 - 0.04072484698113826*w2*x3*z2 - 0.013759874074562587*lnbara*w2*
      x3*z2 + 0.02501436947958356*x4*z2 - 0.0011239361348986583*u*x4*
      z2 + 0.11440699548555519*v*x4*z2 + 0.02876833807133764*lnbara*v*x4*
      z2 - 0.11655132895257132*w*x4*z2 - 0.02876833807133764*lnbara*w*x4*
      z2 - 0.0027042708881657167*x5*z2 - 0.6431554977641232*y*z2 + 
     0.18359057678222876*u*y*z2 - 0.052439122857018475*u2*y*z2 + 
     0.010578663643644204*u3*y*z2 - 0.0010454015333102693*u4*y*z2 + 
     4.998355981731551*v*y*z2 + 1.7188373548274771*lnbara*v*y*z2 + 
     3.6274999402312313*u*v*y*z2 + 1.101558132258626*lnbara*u*v*y*z2 + 
     1.2958237235724512*u2*v*y*z2 + 0.37253832538857234*lnbara*u2*v*y*
      z2 + 0.1611431319221526*u3*v*y*z2 + 0.043853270547930576*lnbara*u3*
      v*y*z2 - 5.33566764662009*v2*y*z2 - 1.7682946827872745*lnbara*v2*y*
      z2 - 0.5880364469542249*u*v2*y*z2 - 0.18109640550068581*lnbara*u*v2*
      y*z2 - 0.6568129911274124*u2*v2*y*z2 - 0.19060623005554803*lnbara*
      u2*v2*y*z2 + 1.7216900283016308*v3*y*z2 + 
     0.5870600279751855*lnbara*v3*y*z2 + 0.35111850002088596*u*v3*y*z2 + 
     0.09843197109895134*lnbara*u*v3*y*z2 - 0.4270870728419753*v4*y*z2 - 
     0.13342942879566821*lnbara*v4*y*z2 - 4.63804380635014*w*y*z2 - 
     1.7188373548274771*lnbara*w*y*z2 - 3.7155448088414325*u*w*y*z2 - 
     1.101558132258626*lnbara*u*w*y*z2 - 1.279932716374738*u2*w*y*z2 - 
     0.37253832538857234*lnbara*u2*w*y*z2 - 0.16259660502488668*u3*w*y*
      z2 - 0.043853270547930576*lnbara*u3*w*y*z2 + 
     4.806051314717304*v*w*y*z2 + 1.4974962913849321*lnbara*v*w*y*z2 - 
     4.358084874395133*u*v*w*y*z2 - 1.310301658288488*lnbara*u*v*w*y*z2 + 
     0.44246231471333614*u2*v*w*y*z2 + 0.13612276309543284*lnbara*u2*v*w*y*
      z2 - 0.19274016063176588*v2*w*y*z2 - 0.09505426720402026*lnbara*v2*
      w*y*z2 + 0.5522979595539836*u*v2*w*y*z2 + 0.17560355760261687*lnbara*
      u*v2*w*y*z2 + 0.4722390677423313*v3*w*y*z2 + 
     0.13175034334097716*lnbara*v3*w*y*z2 + 0.3848527108952509*w2*y*z2 + 
     0.27079839140234246*lnbara*w2*y*z2 + 4.968594829946177*u*w2*y*z2 + 
     1.491398063789174*lnbara*u*w2*y*z2 + 0.2124983335725487*u2*w2*y*
      z2 + 0.0544834669601152*lnbara*u2*w2*y*z2 + 
     0.16780437876971865*v*w2*y*z2 + 0.05368132013057931*lnbara*v*w2*y*
      z2 + 0.00017982275152600617*u*v*w2*y*z2 - 0.008148863899001088*
      lnbara*u*v*w2*y*z2 - 1.0211916139594837*v2*w2*y*z2 - 
     0.2879966174194479*lnbara*v2*w2*y*z2 - 1.6610384150846027*w3*y*z2 - 
     0.5456870809017446*lnbara*w3*y*z2 - 0.9061306963251085*u*w3*y*z2 - 
     0.2658866648025671*lnbara*u*w3*y*z2 + 0.8410658901759228*v*w3*y*z2 + 
     0.24067518005948615*lnbara*v*w3*y*z2 + 0.1309550363078858*w4*y*z2 + 
     0.04900052281465277*lnbara*w4*y*z2 + 0.25855118000886335*x*y*z2 - 
     0.04772361051737147*u*x*y*z2 + 0.00780375397440978*u2*x*y*z2 - 
     0.0006817048264512694*u3*x*y*z2 + 6.647579498888072*v*x*y*z2 + 
     2.014469236002691*lnbara*v*x*y*z2 - 2.5690901893905513*u*v*x*y*z2 - 
     0.8128752507655377*lnbara*u*v*x*y*z2 + 0.3011577157359468*u2*v*x*y*
      z2 + 0.09017258826251653*lnbara*u2*v*x*y*z2 - 
     0.19274016063176588*v2*x*y*z2 - 0.09505426720402026*lnbara*v2*x*y*
      z2 + 0.5522979595539836*u*v2*x*y*z2 + 0.17560355760261687*lnbara*u*
      v2*x*y*z2 + 0.569975957295371*v3*x*y*z2 + 
     0.15957579625776946*lnbara*v3*x*y*z2 - 6.7414545262279955*w*x*y*z2 - 
     2.014469236002691*lnbara*w*x*y*z2 + 2.582819839347*u*w*x*y*z2 + 
     0.8128752507655377*lnbara*u*w*x*y*z2 - 0.30225971597078544*u2*w*x*y*
      z2 - 0.09017258826251653*lnbara*u2*w*x*y*z2 + 
     0.04678476613703959*v*w*x*y*z2 + 0.041372947073440955*lnbara*v*w*x*y*
      z2 - 0.5540525639589041*u*v*w*x*y*z2 - 0.1674546937036158*lnbara*u*v*
      w*x*y*z2 - 2.1878006392634903*v2*w*x*y*z2 - 
     0.6177543113748135*lnbara*v2*w*x*y*z2 + 0.16780437876971865*w2*x*y*
      z2 + 0.05368132013057931*lnbara*w2*x*y*z2 + 
     0.00017982275152600617*u*w2*x*y*z2 - 0.008148863899001088*lnbara*u*w2*
      x*y*z2 + 2.53079627759307*v*w2*x*y*z2 + 0.7137169391031986*lnbara*v*
      w2*x*y*z2 - 0.9153400045660104*w3*x*y*z2 - 
     0.25553842398615445*lnbara*w3*x*y*z2 - 0.08463412943162905*x2*y*z2 + 
     0.00894003555993513*u*x2*y*z2 - 0.0006345782786901335*u2*x2*y*z2 + 
     0.008768669407230721*v*x2*y*z2 - 0.0005708021200927833*u*v*x2*y*z2 - 
     1.0211916139594837*v2*x2*y*z2 - 0.2879966174194479*lnbara*v2*x2*y*
      z2 + 0.008768669407230721*w*x2*y*z2 - 0.0005708021200927833*u*w*x2*
      y*z2 + 2.0406049955739975*v*w*x2*y*z2 + 0.5759932348388958*lnbara*v*
      w*x2*y*z2 - 1.0211916139594837*w2*x2*y*z2 - 
     0.2879966174194479*lnbara*w2*x2*y*z2 + 0.018271879935025635*x3*y*
      z2 - 0.0008364387244659098*u*x3*y*z2 + 0.8908183968952704*v*x3*y*
      z2 + 0.2518959384495211*lnbara*v*x3*y*z2 - 
     0.8924547277374686*w*x3*y*z2 - 0.2518959384495211*lnbara*w*x3*y*z2 - 
     0.0018766602068817076*x4*y*z2 + 0.29720782742626456*y2*z2 - 
     0.052439122857018475*u*y2*z2 + 0.008332760830303598*u2*y2*z2 - 
     0.0007154743025816235*u3*y2*z2 - 0.022693681826607907*v*y2*z2 - 
     0.027337198387939212*lnbara*v*y2*z2 + 0.7675738685886592*u*v*y2*z2 + 
     0.22874938189748867*lnbara*u*v*y2*z2 + 0.17593076120022283*u2*v*y2*
      z2 + 0.04860228502711452*lnbara*u2*v*y2*z2 + 
     0.6573559458240761*v2*y2*z2 + 0.19271706881608106*lnbara*v2*y2*
      z2 - 0.6548222450121166*u*v2*y2*z2 - 0.18974503809357995*lnbara*u*
      v2*y2*z2 + 0.047708646825452936*v3*y2*z2 + 
     0.00998440678420688*lnbara*v3*y2*z2 - 0.08497562276797721*w*y2*z2 + 
     0.027337198387939212*lnbara*w*y2*z2 - 0.7526477768237668*u*w*y2*z2 - 
     0.22874938189748867*lnbara*u*w*y2*z2 - 0.1770896870636047*u2*w*y2*
      z2 - 0.04860228502711452*lnbara*u2*w*y2*z2 - 
     2.5782754794730662*v*w*y2*z2 - 0.7595995680938061*lnbara*v*w*y2*z2 + 
     0.6316944039353403*u*v*w*y2*z2 + 0.18411867864157963*lnbara*u*v*w*y2*
      z2 + 0.37199049657356653*v2*w*y2*z2 + 0.11960991850943639*lnbara*
      v2*w*y2*z2 + 1.9461607406157502*w2*y2*z2 + 
     0.566882499277725*lnbara*w2*y2*z2 + 0.021416517678162092*u*w2*y2*
      z2 + 0.005626359452000324*lnbara*u*w2*y2*z2 - 
     0.21678745401254226*v*w2*y2*z2 - 0.06933321435793802*lnbara*v*w2*y2*
      z2 - 0.2056752271955055*w3*y2*z2 - 0.06026111093570525*lnbara*w3*
      y2*z2 - 0.08463412943162905*x*y2*z2 + 0.008768669407230721*u*x*y2*
      z2 - 0.0006133788582035897*u2*x*y2*z2 - 4.148276730172979*v*x*y2*
      z2 - 1.2223037533945342*lnbara*v*x*y2*z2 + 
     0.15794313861675344*u*v*x*y2*z2 + 0.05241767434109274*lnbara*u*v*x*y2*
      z2 + 0.37199049657356653*v2*x*y2*z2 + 0.11960991850943639*lnbara*
      v2*x*y2*z2 + 4.165985435140145*w*x*y2*z2 + 
     1.2223037533945342*lnbara*w*x*y2*z2 - 0.15906541536540864*u*w*x*y2*
      z2 - 0.05241767434109274*lnbara*u*w*x*y2*z2 - 
     0.1570218018180108*v*w*x*y2*z2 - 0.05027670415149837*lnbara*v*w*x*y2*
      z2 - 0.21678745401254226*w2*x*y2*z2 - 0.06933321435793802*lnbara*
      w2*x*y2*z2 + 0.01657980243935387*x2*y2*z2 - 
     0.0007553212077869194*u*x2*y2*z2 - 0.0007553212077869194*v*x2*y2*
      z2 - 0.0007553212077869194*w*x2*y2*z2 - 0.0015904265351103456*x3*
      y2*z2 - 0.10781348756592685*y3*z2 + 0.010578663643644204*u*y3*
      z2 - 0.0007154743025816235*u2*y3*z2 + 0.7187404897942826*v*y3*
      z2 + 0.2252713360201013*lnbara*v*y3*z2 + 0.1387217802717251*u*v*y3*
      z2 + 0.03825655240654065*lnbara*u*v*y3*z2 - 
     0.3442555249706034*v2*y3*z2 - 0.10069943614887171*lnbara*v2*y3*
      z2 - 0.6965292598704438*w*y3*z2 - 0.2252713360201013*lnbara*w*y3*
      z2 - 0.14004639793591991*u*w*y3*z2 - 0.03825655240654065*lnbara*u*w*
      y3*z2 + 0.4419629692064293*v*w*y3*z2 + 0.124791357615411*lnbara*v*w*
      y3*z2 - 0.09998366236526494*w2*y3*z2 - 0.024091921466539296*lnbara*
      w2*y3*z2 + 0.018271879935025635*x*y3*z2 - 
     0.0008105602649905572*u*x*y3*z2 + 0.6274055393084268*v*x*y3*z2 + 
     0.17563835536198544*lnbara*v*x*y3*z2 - 0.6290677486101003*w*x*y3*
      z2 - 0.17563835536198544*lnbara*w*x*y3*z2 - 
     0.0015904265351103456*x2*y3*z2 + 0.02501436947958356*y4*z2 - 
     0.0010454015333102693*u*y4*z2 - 0.04107580891265065*v*y4*z2 - 
     0.013253738191194095*lnbara*v*y4*z2 + 0.03885294084404612*w*y4*z2 + 
     0.013253738191194095*lnbara*w*y4*z2 - 0.0018766602068817076*x*y4*
      z2 - 0.0027042708881657167*y5*z2 - 1.6193513369191062*z3 + 
     0.39191746801048677*u*z3 - 0.10781348756592685*u2*z3 + 
     0.021502574570336913*u3*z3 - 0.002116028409986158*u4*z3 - 
     0.3344496541843748*v*z3 - 0.24740311290228975*lnbara*v*z3 - 
     3.884956834618231*u*v*z3 - 1.2186672004102037*lnbara*u*v*z3 - 
     0.47396206592202933*u2*v*z3 - 0.13192114982378386*lnbara*u2*v*z3 - 
     0.23348793049311065*u3*v*z3 - 0.06654355140807486*lnbara*u3*v*z3 + 
     0.403289978849488*v2*z3 + 0.15003403088345468*lnbara*v2*z3 + 
     2.3470147044696774*u*v2*z3 + 0.7642509739315361*lnbara*u*v2*z3 + 
     0.14483466237687437*u2*v2*z3 + 0.03394368436113497*lnbara*u2*v2*
      z3 + 0.13476350893453617*v3*z3 + 0.02815049785571501*lnbara*v3*
      z3 - 0.5147171592780667*u*v3*z3 - 0.16006031325329761*lnbara*u*v3*
      z3 + 0.026332918322492415*v4*z3 + 0.007673317744555463*lnbara*v4*
      z3 + 1.0120947283767654*w*z3 + 0.24740311290228975*lnbara*w*z3 + 
     3.7247490277998456*u*w*z3 + 1.2186672004102037*lnbara*u*w*z3 + 
     0.5028126095006992*u2*w*z3 + 0.13192114982378386*lnbara*u2*w*z3 + 
     0.23084041467755445*u3*w*z3 + 0.06654355140807486*lnbara*u3*w*z3 + 
     2.0505932975012593*v*w*z3 + 0.7335375186340615*lnbara*v*w*z3 - 
     0.4484159450532304*u*v*w*z3 - 0.1922469483445887*lnbara*u*v*w*z3 + 
     0.33319811688498185*u2*v*w*z3 + 0.10536425342736426*lnbara*u2*v*w*
      z3 - 3.269269763555497*v2*w*z3 - 1.0164989417094896*lnbara*v2*w*
      z3 + 0.44168144304165097*u*v2*w*z3 + 0.12858123333432991*lnbara*u*
      v2*w*z3 + 0.33499391115566624*v3*w*z3 + 0.11243984289700684*lnbara*
      v3*w*z3 - 2.6912970732885757*w2*z3 - 0.8835715495175162*lnbara*w2*
      z3 - 1.8613624203011476*u*w2*z3 - 0.5720040255869473*lnbara*u*w2*
      z3 - 0.48114924036453877*u2*w2*z3 - 0.13930793778849923*lnbara*u2*
      w2*z3 + 2.941587800744719*v*w2*z3 + 0.8832109828874293*lnbara*v*w2*
      z3 - 0.7853652276547294*u*v*w2*z3 - 0.21990490700155435*lnbara*u*v*
      w2*z3 + 0.05859089416673988*v2*w2*z3 + 0.01186851715999764*lnbara*
      v2*w2*z3 + 0.24682581080985633*w3*z3 + 0.10513746096634541*lnbara*
      w3*z3 + 0.8544575390208671*u*w3*z3 + 0.251383986920522*lnbara*u*w3*
      z3 - 0.2434872198153215*v*w3*z3 - 0.07483548907405872*lnbara*v*w3*
      z3 - 0.1821625110649795*w4*z3 - 0.057146188727501226*lnbara*w4*
      z3 + 0.39191746801048677*x*z3 - 0.06968191432583015*u*x*z3 + 
     0.011283905940380838*u2*x*z3 - 0.0009831878162066224*u3*x*z3 + 
     5.035142069385885*v*x*z3 + 1.7095780955682178*lnbara*v*x*z3 + 
     0.16919997458043376*u*v*x*z3 + 0.013784793500404437*lnbara*u*v*x*z3 + 
     0.4078728401620492*u2*v*x*z3 + 0.12576336826926637*lnbara*u2*v*x*
      z3 - 2.775770631100383*v2*x*z3 - 0.8879564128194577*lnbara*v2*x*
      z3 + 0.3619717726179578*u*v2*x*z3 + 0.10324350723878072*lnbara*u*v2*
      x*z3 + 0.33499391115566624*v3*x*z3 + 0.11243984289700684*lnbara*v3*
      x*z3 - 5.165259334653691*w*x*z3 - 1.7095780955682178*lnbara*w*x*z3 - 
     0.15031327644943632*u*w*x*z3 - 0.013784793500404437*lnbara*u*w*x*z3 - 
     0.40939098371296645*u2*w*x*z3 - 0.12576336826926637*lnbara*u2*w*x*
      z3 + 4.268816061721532*v*w*x*z3 + 1.2629454959136155*lnbara*v*w*x*
      z3 - 1.617872252781772*u*v*w*x*z3 - 0.46845974278075075*lnbara*u*v*w*
      x*z3 - 0.3134017819611535*v2*w*x*z3 - 0.0993620511060278*lnbara*v2*
      w*x*z3 - 1.4645041396124352*w2*x*z3 - 0.37498908309415785*lnbara*w2*
      x*z3 + 1.25382676464303*u*w2*x*z3 + 0.3652162355419701*lnbara*u*w2*
      x*z3 + 0.2189313721141216*v*w2*x*z3 + 0.06175769728307968*lnbara*v*
      w2*x*z3 - 0.2434872198153215*w3*x*z3 - 0.07483548907405872*lnbara*
      w3*x*z3 - 0.10781348756592685*x2*z3 + 0.011283905940380838*u*x2*
      z3 - 0.0008001063695529314*u2*x2*z3 + 1.8755457504129727*v*x2*
      z3 + 0.5198935985278687*lnbara*v*x2*z3 - 0.5329098110819325*u*v*x2*
      z3 - 0.1550921261368306*lnbara*u*v*x2*z3 - 
     0.04072484698113826*v2*x2*z3 - 0.013759874074562587*lnbara*v2*x2*
      z3 - 1.8540397627858702*w*x2*z3 - 0.5198935985278687*lnbara*w*x2*
      z3 + 0.5315078645933379*u*w*x2*z3 + 0.1550921261368306*lnbara*u*w*
      x2*z3 + 0.10030472915504478*v*w*x2*z3 + 0.029687566492958728*lnbara*
      v*w*x2*z3 - 0.061694139411974463*w2*x2*z3 - 
     0.01592769241839614*lnbara*w2*x2*z3 + 0.021502574570336913*x3*z3 - 
     0.0009831878162066224*u*x3*z3 - 0.0009436436586848417*v*x3*z3 - 
     0.0009436436586848417*w*x3*z3 - 0.002116028409986158*x4*z3 + 
     0.39191746801048677*y*z3 - 0.06841558659701322*u*y*z3 + 
     0.010927323983458038*u2*y*z3 - 0.0009436436586848417*u3*y*z3 - 
     6.4531586807126216*v*y*z3 - 2.0595906539260844*lnbara*v*y*z3 - 
     1.0928490356968497*u*v*y*z3 - 0.30677398634465897*lnbara*u*v*y*z3 - 
     0.5203695633112709*u2*v*y*z3 - 0.14879519944448213*lnbara*u2*v*y*
      z3 + 1.7216900283016308*v2*y*z3 + 0.5870600279751855*lnbara*v2*y*
      z3 + 0.35111850002088596*u*v2*y*z3 + 0.09843197109895134*lnbara*u*
      v2*y*z3 - 0.5006975185477412*v3*y*z3 - 0.1543564123528748*lnbara*
      v3*y*z3 + 6.321775087715999*w*y*z3 + 2.0595906539260844*lnbara*w*y*
      z3 + 1.1114419930785513*u*w*y*z3 + 0.30677398634465897*lnbara*u*w*y*
      z3 + 0.5189008798963198*u2*w*y*z3 + 0.14879519944448213*lnbara*u2*w*
      y*z3 - 0.03145999950209565*v*w*y*z3 - 0.041372947073440955*lnbara*v*w*
      y*z3 + 0.5529561454818885*u*v*w*y*z3 + 0.1674546937036158*lnbara*u*v*
      w*y*z3 + 0.569975957295371*v2*w*y*z3 + 0.15957579625776946*lnbara*
      v2*w*y*z3 - 1.6610384150846027*w2*y*z3 - 0.5456870809017446*lnbara*
      w2*y*z3 - 0.9061306963251085*u*w2*y*z3 - 0.2658866648025671*lnbara*
      u*w2*y*z3 - 0.9153400045660104*v*w2*y*z3 - 
     0.25553842398615445*lnbara*v*w2*y*z3 + 0.8429911783197556*w3*y*z3 + 
     0.2503190400812598*lnbara*w3*y*z3 - 0.0985061281475928*x*y*z3 + 
     0.01023088439838107*u*x*y*z3 - 0.0007202416621432963*u2*x*y*z3 - 
     0.7027857105758675*v*x*y*z3 - 0.20757337570539203*lnbara*v*x*y*z3 + 
     0.3671622432587578*u*v*x*y*z3 + 0.11464743468556707*lnbara*u*v*x*y*
      z3 + 0.4722390677423313*v2*x*y*z3 + 0.13175034334097716*lnbara*v2*x*
      y*z3 + 0.7227291601125875*w*x*y*z3 + 0.20757337570539203*lnbara*w*x*y*
      z3 - 0.368445017992858*u*w*x*y*z3 - 0.11464743468556707*lnbara*u*w*x*
      y*z3 - 1.3152966221164228*v*w*x*y*z3 - 0.3724255234004633*lnbara*v*w*
      x*y*z3 + 0.8410658901759228*w2*x*y*z3 + 0.24067518005948615*lnbara*
      w2*x*y*z3 + 0.018271879935025635*x2*y*z3 - 
     0.0008364387244659098*u*x2*y*z3 - 0.8924547277374686*v*x2*y*z3 - 
     0.2518959384495211*lnbara*v*x2*y*z3 + 0.8908183968952704*w*x2*y*z3 + 
     0.2518959384495211*lnbara*w*x2*y*z3 - 0.0017038721568713427*x3*y*
      z3 - 0.10781348756592685*y2*z3 + 0.010927323983458038*u*y2*z3 - 
     0.000755870097742802*u2*y2*z3 + 0.8444774478932007*v*y2*z3 + 
     0.248860063558088*lnbara*v*y2*z3 - 0.3385312614440469*u*v*y2*z3 - 
     0.09925662865561737*lnbara*u*v*y2*z3 + 0.047708646825452936*v2*y2*
      z3 + 0.00998440678420688*lnbara*v2*y2*z3 - 
     0.8226148783091756*w*y2*z3 - 0.248860063558088*lnbara*w*y2*z3 + 
     0.33716810694416005*u*w*y2*z3 + 0.09925662865561737*lnbara*u*w*y2*
      z3 + 0.15576929487146673*v*w*y2*z3 + 0.05027670415149837*lnbara*v*w*
      y2*z3 - 0.2056752271955055*w2*y2*z3 - 0.06026111093570525*lnbara*
      w2*y2*z3 + 0.018271879935025635*x*y2*z3 - 
     0.0008257705772076173*u*x*y2*z3 + 0.5075561415724656*v*x*y2*z3 + 
     0.15134262670003695*lnbara*v*x*y2*z3 - 0.5092031405619221*w*x*y2*
      z3 - 0.15134262670003695*lnbara*w*x*y2*z3 - 
     0.0015904265351103456*x2*y2*z3 + 0.021502574570336913*y3*z3 - 
     0.0009436436586848417*u*y3*z3 - 0.35800549766301815*v*y3*z3 - 
     0.1031812587805062*lnbara*v*y3*z3 + 0.3560786661881267*w*y3*z3 + 
     0.1031812587805062*lnbara*w*y3*z3 - 0.0017038721568713427*x*y3*z3 - 
     0.002116028409986158*y4*z3 + 1.1560019673631434*z4 - 
     0.16545503028959443*u*z4 + 0.02501436947958356*u2*z4 - 
     0.002116028409986158*u3*z4 + 0.8622872883244641*v*z4 + 
     0.3193208792684922*lnbara*v*z4 + 1.3489683495777027*u*v*z4 + 
     0.4144777234484366*lnbara*u*v*z4 + 0.1660137847279631*u2*v*z4 + 
     0.0453330032683975*lnbara*u2*v*z4 + 0.1913662943454968*v2*z4 + 
     0.04215619932204285*lnbara*v2*z4 - 0.5618260667674302*u*v2*z4 - 
     0.17055352864149378*lnbara*u*v2*z4 + 0.026332918322492415*v3*z4 + 
     0.007673317744555463*lnbara*v3*z4 - 1.1463870292291396*w*z4 - 
     0.3193208792684922*lnbara*w*z4 - 1.3112778013936697*u*w*z4 - 
     0.4144777234484366*lnbara*u*w*z4 - 0.16893584646815507*u2*w*z4 - 
     0.0453330032683975*lnbara*u2*w*z4 - 1.8086906966367922*v*w*z4 - 
     0.5574561800298969*lnbara*v*w*z4 + 0.3835758469523267*u*v*w*z4 + 
     0.116292709475361*lnbara*u*v*w*z4 + 0.3622126558447875*v2*w*z4 + 
     0.11666185005310058*lnbara*v2*w*z4 + 1.6718584721002623*w2*z4 + 
     0.5152999807078541*lnbara*w2*z4 + 0.17448936082610797*u*w2*z4 + 
     0.05426081916613278*lnbara*u*w2*z4 - 0.21163351738140398*v*w2*z4 - 
     0.06718897907015481*lnbara*v*w2*z4 - 0.1821625110649795*w3*z4 - 
     0.057146188727501226*lnbara*w3*z4 - 0.16545503028959443*x*z4 + 
     0.0162776555901896*u*x*z4 - 0.0011239361348986583*u2*x*z4 - 
     3.0012661757807617*v*x*z4 - 0.9380468690654714*lnbara*v*x*z4 + 
     0.15191010788660775*u*v*x*z4 + 0.048049503963086576*lnbara*u*v*x*z4 + 
     0.3328573832991313*v2*x*z4 + 0.11063120820095806*lnbara*v2*x*z4 + 
     3.0315936321798276*w*x*z4 + 0.9380468690654714*lnbara*w*x*z4 - 
     0.15381418595568414*u*w*x*z4 - 0.048049503963086576*lnbara*u*w*x*z4 - 
     0.3368289563598407*v*w*x*z4 - 0.10155666376810567*lnbara*v*w*x*z4 + 
     0.0011325127373185734*w2*x*z4 - 0.009074544432852395*lnbara*w2*x*
      z4 + 0.02501436947958356*x2*z4 - 0.0011239361348986583*u*x2*z4 - 
     0.11655132895257132*v*x2*z4 - 0.02876833807133764*lnbara*v*x2*z4 + 
     0.11440699548555519*w*x2*z4 + 0.02876833807133764*lnbara*w*x2*z4 - 
     0.002116028409986158*x3*z4 - 0.16545503028959443*y*z4 + 
     0.016074508870027843*u*y*z4 - 0.0010989319337058666*u2*y*z4 + 
     2.4002692887025985*v*y*z4 + 0.7507653584772783*lnbara*v*y*z4 + 
     0.29707382410626676*u*v*y*z4 + 0.0809778929894431*lnbara*u*v*y*z4 - 
     0.4270870728419753*v2*y*z4 - 0.13342942879566821*lnbara*v2*y*z4 - 
     2.3697386855833713*w*y*z4 - 0.7507653584772783*lnbara*w*y*z4 - 
     0.2989574253995812*u*w*y*z4 - 0.0809778929894431*lnbara*u*w*y*z4 + 
     0.2932474952337439*v*w*y*z4 + 0.08442890598101545*lnbara*v*w*y*z4 + 
     0.1309550363078858*w2*y*z4 + 0.04900052281465277*lnbara*w2*y*z4 + 
     0.023437600654995495*x*y*z4 - 0.0010458019341617988*u*x*y*z4 + 
     0.36905978156971897*v*x*y*z4 + 0.10073935030932556*lnbara*v*x*y*z4 - 
     0.3710874532554622*w*x*y*z4 - 0.10073935030932556*lnbara*w*x*y*z4 - 
     0.0018766602068817076*x2*y*z4 + 0.02501436947958356*y2*z4 - 
     0.0010989319337058666*u*y2*z4 - 0.047668105888734366*v*y2*z4 - 
     0.015411950745225283*lnbara*v*y2*z4 + 0.04549876822052544*w*y2*z4 + 
     0.015411950745225283*lnbara*w*y2*z4 - 0.0018766602068817076*x*y2*
      z4 - 0.002116028409986158*y3*z4 - 0.5375082140631944*z5 + 
     0.04216772179515423*u*z5 - 0.0027042708881657167*u2*z5 - 
     0.2093175862405338*v*z5 - 0.07853883800228331*lnbara*v*z5 - 
     0.27798071288879395*u*v*z5 - 0.08187135582602693*lnbara*u*v*z5 + 
     0.004695610230502542*v2*z5 + 0.002302749325478825*lnbara*v2*z5 + 
     0.281434133138352*w*z5 + 0.07853883800228331*lnbara*w*z5 + 
     0.2738733074161405*u*w*z5 + 0.08187135582602693*lnbara*u*w*z5 + 
     0.23350760677675905*v*w*z5 + 0.07186909727514694*lnbara*v*w*z5 - 
     0.24406286627904641*w2*z5 - 0.07417184660062576*lnbara*w2*z5 + 
     0.04216772179515423*x*z5 - 0.0017664062094713614*u*x*z5 + 
     0.4769026267982312*v*x*z5 + 0.14541778766038482*lnbara*v*x*z5 - 
     0.48018841061091755*w*x*z5 - 0.14541778766038482*lnbara*w*x*z5 - 
     0.0027042708881657167*x2*z5 + 0.04216772179515423*y*z5 - 
     0.0017498755815456293*u*y*z5 - 0.48202800643018173*v*y*z5 - 
     0.1438457497350551*lnbara*v*y*z5 + 0.4787256919895696*w*y*z5 + 
     0.1438457497350551*lnbara*w*y*z5 - 0.0025714972415127255*x*y*z5 - 
     0.0027042708881657167*y2*z5 + 0.14606611930718463*z6 - 
     0.004855915459192057*u*z6 + 0.04158747525391902*v*z6 + 
     0.013727259996145989*lnbara*v*z6 - 0.04987109935863513*w*z6 - 
     0.013727259996145989*lnbara*w*z6 - 0.004855915459192057*x*z6 - 
     0.004855915459192057*y*z6 - 0.017602106974984624*z7;
 
  Hexp[8] = 0.9015426773696957 - 1.2020569031595942*u + 
     1.8468559937823905*u2 - 2.2670918716867487*u3 + 
     2.0230034428855013*u4 - 1.2541858328141202*u5 + 
     0.5112314175751462*u6 - 0.12321474882489236*u7 + 
     0.013314221484862214*u8 - 1.2020569031595942*v + 
     0.9809537469447522*u*v - 0.9004176968697725*u2*v + 
     0.6858555690183518*u3*v - 0.386061737342387*u4*v + 
     0.1475870262830398*u5*v - 0.0339914082143444*u6*v + 
     0.0035512018991495737*u7*v + 1.8468559937823905*v2 - 
     0.9004176968697725*u*v2 + 0.520113697995963*u2*v2 - 
     0.251564804320496*u3*v2 + 0.08755029317854246*u4*v2 - 
     0.018929896217160017*u5*v2 + 0.0018904202297008643*u6*v2 - 
     2.2670918716867487*v3 + 0.6858555690183518*u*v3 - 
     0.251564804320496*u2*v3 + 0.0752590109961792*u3*v3 - 
     0.014812198869903107*u4*v3 + 0.001388160767640295*u5*v3 + 
     2.0230034428855013*v4 - 0.386061737342387*u*v4 + 
     0.08755029317854246*u2*v4 - 0.014812198869903107*u3*v4 + 
     0.0012633787101392036*u4*v4 - 1.2541858328141202*v5 + 
     0.1475870262830398*u*v5 - 0.018929896217160017*u2*v5 + 
     0.001388160767640295*u3*v5 + 0.5112314175751462*v6 - 
     0.0339914082143444*u*v6 + 0.0018904202297008643*u2*v6 - 
     0.12321474882489236*v7 + 0.0035512018991495737*u*v7 + 
     0.013314221484862214*v8 - 1.2020569031595942*w + 
     0.9809537469447522*u*w - 0.9004176968697725*u2*w + 
     0.6858555690183518*u3*w - 0.386061737342387*u4*w + 
     0.1475870262830398*u5*w - 0.0339914082143444*u6*w + 
     0.0035512018991495737*u7*w + 0.9809537469447522*v*w - 
     0.5522472005141469*u*v*w + 0.3313625873896489*u2*v*w - 
     0.162591133426937*u3*v*w + 0.056971794565663605*u4*v*w - 
     0.012364843466299529*u5*v*w + 0.0012376963517727734*u6*v*w - 
     0.9004176968697725*v2*w + 0.3313625873896489*u*v2*w - 
     0.12887042405432217*u2*v2*w + 0.03949367079133293*u3*v2*w - 
     0.007867552944290607*u4*v2*w + 0.0007424815539584533*u5*v2*w + 
     0.6858555690183518*v3*w - 0.162591133426937*u*v3*w + 
     0.03949367079133293*u2*v3*w - 0.006882314713446356*u3*v3*w + 
     0.0005960914163116732*u4*v3*w - 0.386061737342387*v4*w + 
     0.056971794565663605*u*v4*w - 0.007867552944290607*u2*v4*w + 
     0.0005960914163116732*u3*v4*w + 0.1475870262830398*v5*w - 
     0.012364843466299529*u*v5*w + 0.0007424815539584533*u2*v5*w - 
     0.0339914082143444*v6*w + 0.0012376963517727734*u*v6*w + 
     0.0035512018991495737*v7*w + 1.8468559937823905*w2 - 
     0.9004176968697725*u*w2 + 0.520113697995963*u2*w2 - 
     0.251564804320496*u3*w2 + 0.08755029317854246*u4*w2 - 
     0.018929896217160017*u5*w2 + 0.0018904202297008643*u6*w2 - 
     0.9004176968697725*v*w2 + 0.3313625873896489*u*v*w2 - 
     0.12887042405432217*u2*v*w2 + 0.03949367079133293*u3*v*w2 - 
     0.007867552944290607*u4*v*w2 + 0.0007424815539584533*u5*v*w2 + 
     0.520113697995963*v2*w2 - 0.12887042405432217*u*v2*w2 + 
     0.03186642256716374*u2*v2*w2 - 0.00560074458687052*u3*v2*w2 + 
     0.000487363420148127*u4*v2*w2 - 0.251564804320496*v3*w2 + 
     0.03949367079133293*u*v3*w2 - 0.00560074458687052*u2*v3*w2 + 
     0.0004300153542944288*u3*v3*w2 + 0.08755029317854246*v4*w2 - 
     0.007867552944290607*u*v4*w2 + 0.000487363420148127*u2*v4*w2 - 
     0.018929896217160017*v5*w2 + 0.0007424815539584533*u*v5*w2 + 
     0.0018904202297008643*v6*w2 - 2.2670918716867487*w3 + 
     0.6858555690183518*u*w3 - 0.251564804320496*u2*w3 + 
     0.0752590109961792*u3*w3 - 0.014812198869903107*u4*w3 + 
     0.001388160767640295*u5*w3 + 0.6858555690183518*v*w3 - 
     0.162591133426937*u*v*w3 + 0.03949367079133293*u2*v*w3 - 
     0.006882314713446356*u3*v*w3 + 0.0005960914163116732*u4*v*w3 - 
     0.251564804320496*v2*w3 + 0.03949367079133293*u*v2*w3 - 
     0.00560074458687052*u2*v2*w3 + 0.0004300153542944288*u3*v2*w3 + 
     0.0752590109961792*v3*w3 - 0.006882314713446356*u*v3*w3 + 
     0.0004300153542944288*u2*v3*w3 - 0.014812198869903107*v4*w3 + 
     0.0005960914163116732*u*v4*w3 + 0.001388160767640295*v5*w3 + 
     2.0230034428855013*w4 - 0.386061737342387*u*w4 + 
     0.08755029317854246*u2*w4 - 0.014812198869903107*u3*w4 + 
     0.0012633787101392036*u4*w4 - 0.386061737342387*v*w4 + 
     0.056971794565663605*u*v*w4 - 0.007867552944290607*u2*v*w4 + 
     0.0005960914163116732*u3*v*w4 + 0.08755029317854246*v2*w4 - 
     0.007867552944290607*u*v2*w4 + 0.000487363420148127*u2*v2*w4 - 
     0.014812198869903107*v3*w4 + 0.0005960914163116732*u*v3*w4 + 
     0.0012633787101392036*v4*w4 - 1.2541858328141202*w5 + 
     0.1475870262830398*u*w5 - 0.018929896217160017*u2*w5 + 
     0.001388160767640295*u3*w5 + 0.1475870262830398*v*w5 - 
     0.012364843466299529*u*v*w5 + 0.0007424815539584533*u2*v*w5 - 
     0.018929896217160017*v2*w5 + 0.0007424815539584533*u*v2*w5 + 
     0.001388160767640295*v3*w5 + 0.5112314175751462*w6 - 
     0.0339914082143444*u*w6 + 0.0018904202297008643*u2*w6 - 
     0.0339914082143444*v*w6 + 0.0012376963517727734*u*v*w6 + 
     0.0018904202297008643*v2*w6 - 0.12321474882489236*w7 + 
     0.0035512018991495737*u*w7 + 0.0035512018991495737*v*w7 + 
     0.013314221484862214*w8 - 1.2020569031595942*x + 
     0.9809537469447522*u*x - 0.9004176968697725*u2*x + 
     0.6858555690183518*u3*x - 0.386061737342387*u4*x + 
     0.1475870262830398*u5*x - 0.0339914082143444*u6*x + 
     0.0035512018991495737*u7*x + 0.9809537469447522*v*x + 
     1.7296818908683025*u*v*x + 0.9115194817125674*lnbara*u*v*x + 
     2.950865208678974*u2*v*x + 0.7823223301987707*lnbara*u2*v*x + 
     1.131721092930809*u3*v*x + 0.41420887449201577*lnbara*u3*v*x + 
     0.47680804183544256*u4*v*x + 0.10984007042350338*lnbara*u4*v*x + 
     0.046951959438293674*u5*v*x + 0.018029304599241732*lnbara*u5*v*x + 
     0.006061422665855913*u6*v*x + 0.0010840252459144199*lnbara*u6*v*x - 
     3.3276691614032563*v2*x - 0.9115194817125674*lnbara*v2*x - 
     5.289736299142919*u*v2*x - 1.9748773458715838*lnbara*u*v2*x - 
     4.650086557902906*u2*v2*x - 1.3823733765239528*lnbara*u2*v2*x - 
     1.6750109905361525*u3*v2*x - 0.5107892746111199*lnbara*u3*v2*x - 
     0.3563815319580459*u4*v2*x - 0.09558272829689277*lnbara*u4*v2*x - 
     0.026406119658478486*u5*v2*x - 0.007442787768918609*lnbara*u5*v2*x + 
     1.7170463437119636*v3*x + 0.4329554475790071*lnbara*v3*x + 
     8.335636464746663*u*v3*x + 2.843556800957142*lnbara*u*v3*x + 
     1.7021286302045417*u2*v3*x + 0.46172402438324356*lnbara*u2*v3*x + 
     1.5576572952608156*u3*v3*x + 0.465804859856524*lnbara*u3*v3*x - 
     0.01399123854239156*u4*v3*x - 0.006838625664216695*lnbara*u4*v3*x - 
     2.581767811748015*v4*x - 0.7450820516264817*lnbara*v4*x - 
     4.408138300869153*u*v4*x - 1.450672032069528*lnbara*u*v4*x - 
     1.1428845474172378*u2*v4*x - 0.3173310228787825*lnbara*u2*v4*x - 
     0.2223252216914388*u3*v4*x - 0.06514335645328273*lnbara*u3*v4*x + 
     0.9506587243582331*v5*x + 0.2748859330079916*lnbara*v5*x + 
     1.84547571556521*u*v5*x + 0.5730994907821886*lnbara*u*v5*x + 
     0.11799273363683711*u2*v5*x + 0.03151257404486913*lnbara*u2*v5*x - 
     0.3370863430138181*v6*x - 0.09609081997302192*lnbara*v6*x - 
     0.25711383486763884*u*v6*x - 0.07776386003863338*lnbara*u*v6*x + 
     0.039205434782531995*v7*x + 0.01108690204816448*lnbara*v7*x + 
     0.7968713467733699*w*x - 2.9443689245385314*u*w*x - 
     0.9115194817125674*lnbara*u*w*x - 2.177117134294563*u2*w*x - 
     0.7823223301987707*lnbara*u2*w*x - 1.5212050940015565*u3*w*x - 
     0.41420887449201577*lnbara*u3*w*x - 0.33851565849786086*u4*w*x - 
     0.10984007042350338*lnbara*u4*w*x - 0.07720156919970216*u5*w*x - 
     0.018029304599241732*lnbara*u5*w*x - 0.003018119129298935*u6*w*x - 
     0.0010840252459144199*lnbara*u6*w*x - 0.5171174600050477*v*w*x + 
     0.2576868260776136*u*v*w*x - 0.1113550912072001*u2*v*w*x + 
     0.03580809539433375*u3*v*w*x - 0.007320613539132591*u4*v*w*x + 
     0.0007017792893363362*u5*v*w*x + 16.840740404671994*v2*w*x + 
     5.75015242423565*lnbara*v2*w*x - 7.4944534531230955*u*v2*w*x - 
     2.569763477061043*lnbara*u*v2*w*x + 10.92704206771195*u2*v2*w*x + 
     3.4712775032258807*lnbara*u2*v2*w*x - 2.7345557618171856*u3*v2*w*x - 
     0.8563693554764021*lnbara*u3*v2*w*x + 0.5508222899910821*u4*v2*w*x + 
     0.16402829156973656*lnbara*u4*v2*w*x - 11.553644836317881*v3*w*x - 
     3.9890155563258416*lnbara*v3*w*x - 0.5200656204166001*u*v3*w*x - 
     0.04824677725141553*lnbara*u*v3*w*x - 2.7556939387551567*u2*v3*w*x - 
     0.8803435778848646*lnbara*u2*v3*w*x + 0.0688751348615315*u3*v3*w*x + 
     0.029360220026995808*lnbara*u3*v3*w*x + 10.200182207413253*v4*w*x + 
     3.28316404172915*lnbara*v4*w*x - 1.034655985722088*u*v4*w*x - 
     0.336346527741606*lnbara*u*v4*w*x + 0.8981582264276017*u2*v4*w*x + 
     0.26783993094333697*lnbara*u2*v4*w*x - 3.234078310073586*v5*w*x - 
     1.0179245136226938*lnbara*v5*w*x + 0.01575761553644318*u*v5*w*x + 
     0.010275898356323341*lnbara*u*v5*w*x + 0.5245800234274332*v6*w*x + 
     0.15852336677018297*lnbara*v6*w*x + 1.7490619043784765*w2*x + 
     0.9115194817125674*lnbara*w2*x + 6.04138458368625*u*w2*x + 
     1.9748773458715838*lnbara*u*w2*x + 4.3302489692293955*u2*w2*x + 
     1.3823733765239528*lnbara*u2*w2*x + 1.7772082042508452*u3*w2*x + 
     0.5107892746111199*lnbara*u3*w2*x + 0.33555238697393286*u4*w2*x + 
     0.09558272829689277*lnbara*u4*w2*x + 0.028399509890409173*u5*w2*x + 
     0.007442787768918609*lnbara*u5*w2*x - 16.220273175775272*v*w2*x - 
     5.75015242423565*lnbara*v*w2*x + 7.283613291269799*u*v*w2*x + 
     2.569763477061043*lnbara*u*v*w2*x - 10.868438804341743*u2*v*w2*x - 
     3.4712775032258807*lnbara*u2*v*w2*x + 2.723658999110921*u3*v*w2*x + 
     0.8563693554764021*lnbara*u3*v*w2*x - 0.5498436699327358*u4*v*w2*x - 
     0.16402829156973656*lnbara*u4*v*w2*x - 0.1223579533330431*v2*w2*x + 
     0.027313138910434232*u*v2*w2*x - 0.004442047950830934*u2*v2*w2*x + 
     0.0003652020446131235*u3*v2*w2*x - 6.2616857203946035*v3*w2*x - 
     1.8196275948475404*lnbara*v3*w2*x + 3.5848494417836383*u*v3*w2*x + 
     1.0856448829578143*lnbara*u*v3*w2*x - 0.660951971257608*u2*v3*w2*
      x - 0.18951332857164516*lnbara*u2*v3*w2*x + 
     0.7756766725864659*v4*w2*x + 0.2013783664993635*lnbara*v4*w2*x - 
     0.3272951320069847*u*v4*w2*x - 0.09992430356434646*lnbara*u*v4*w2*
      x - 0.22871819549961922*v5*w2*x - 0.05887745268544395*lnbara*v5*w2*
      x - 0.5311674638752799*w3*x - 0.4329554475790071*lnbara*w3*x - 
     8.70945468065623*u*w3*x - 2.843556800957142*lnbara*u*w3*x - 
     1.6011517276791973*u2*w3*x - 0.46172402438324356*lnbara*u2*w3*x - 
     1.576189905969709*u3*w3*x - 0.465804859856524*lnbara*u3*w3*x + 
     0.015644250220599377*u4*w3*x + 0.006838625664216695*lnbara*u4*w3*x + 
     11.250037884026336*v*w3*x + 3.9890155563258416*lnbara*v*w3*x + 
     0.5861690638750913*u*v*w3*x + 0.04824677725141553*lnbara*u*v*w3*x + 
     2.7450669338987366*u2*v*w3*x + 0.8803435778848646*lnbara*u2*v*w3*x - 
     0.06800667812701873*u3*v*w3*x - 0.029360220026995808*lnbara*u3*v*w3*
      x + 6.336956677089462*v2*w3*x + 1.8196275948475404*lnbara*v2*w3*x - 
     3.5946630672038005*u*v2*w3*x - 1.0856448829578143*lnbara*u*v2*w3*x + 
     0.6616579679935063*u2*v2*w3*x + 0.18951332857164516*lnbara*u2*v2*
      w3*x - 0.006605505610793891*v3*w3*x + 0.00038509593985133696*u*v3*
      w3*x + 0.38164842058355447*v4*w3*x + 0.10823518328690307*lnbara*v4*
      w3*x + 1.9188684163037728*w4*x + 0.7450820516264817*lnbara*w4*x + 
     4.540055219513269*u*w4*x + 1.450672032069528*lnbara*u*w4*x + 
     1.122430115235894*u2*w4*x + 0.3173310228787825*lnbara*u2*w4*x + 
     0.2239680885196098*u3*w4*x + 0.06514335645328273*lnbara*u3*w4*x - 
     10.094036110016523*v*w4*x - 3.28316404172915*lnbara*v*w4*x + 
     1.0213274392385536*u*v*w4*x + 0.336346527741606*lnbara*u*v*w4*x - 
     0.8972167925745769*u2*v*w4*x - 0.26783993094333697*lnbara*u2*v*w4*
      x - 0.7906870068555788*v2*w4*x - 0.2013783664993635*lnbara*v2*w4*
      x + 0.3281593662661477*u*v2*w4*x + 0.09992430356434646*lnbara*u*v2*
      w4*x - 0.380500721540267*v3*w4*x - 0.10823518328690307*lnbara*v3*
      w4*x - 0.6982508102158693*w5*x - 0.2748859330079916*lnbara*w5*x - 
     1.8742275538737843*u*w5*x - 0.5730994907821886*lnbara*u*w5*x - 
     0.11604224021663521*u2*w5*x - 0.03151257404486913*lnbara*u2*w5*x + 
     3.2110778233847816*v*w5*x + 1.0179245136226938*lnbara*v*w5*x - 
     0.01449065568595044*u*v*w5*x - 0.010275898356323341*lnbara*u*v*w5*x + 
     0.2301351338116376*v2*w5*x + 0.05887745268544395*lnbara*v2*w5*x + 
     0.27910097428080544*w6*x + 0.09609081997302192*lnbara*w6*x + 
     0.2599997989809029*u*w6*x + 0.07776386003863338*lnbara*u*w6*x - 
     0.5222803748590084*v*w6*x - 0.15852336677018297*lnbara*v*w6*x - 
     0.033158063533986065*w7*x - 0.01108690204816448*lnbara*w7*x + 
     1.8468559937823905*x2 - 0.9004176968697725*u*x2 + 
     0.520113697995963*u2*x2 - 0.251564804320496*u3*x2 + 
     0.08755029317854246*u4*x2 - 0.018929896217160017*u5*x2 + 
     0.0018904202297008643*u6*x2 - 3.3276691614032563*v*x2 - 
     0.9115194817125674*lnbara*v*x2 - 5.289736299142919*u*v*x2 - 
     1.9748773458715838*lnbara*u*v*x2 - 4.650086557902906*u2*v*x2 - 
     1.3823733765239528*lnbara*u2*v*x2 - 1.6750109905361525*u3*v*x2 - 
     0.5107892746111199*lnbara*u3*v*x2 - 0.3563815319580459*u4*v*x2 - 
     0.09558272829689277*lnbara*u4*v*x2 - 0.026406119658478486*u5*v*x2 - 
     0.007442787768918609*lnbara*u5*v*x2 + 9.22032358478725*v2*x2 + 
     3.1447095994394325*lnbara*v2*x2 + 3.443135626202991*u*v2*x2 + 
     1.2110841723003059*lnbara*u*v2*x2 + 9.306085458851838*u2*v2*x2 + 
     2.882695368781348*lnbara*u2*v2*x2 + 0.170862170132498*u3*v2*x2 + 
     0.022605565124086877*lnbara*u3*v2*x2 + 0.363288554298664*u4*v2*
      x2 + 0.10475946602085698*lnbara*u4*v2*x2 - 
     6.021984802397401*v3*x2 - 2.0616669488742043*lnbara*v3*x2 - 
     6.264516709859727*u*v3*x2 - 2.002014089554316*lnbara*u*v3*x2 - 
     3.2461502369868342*u2*v3*x2 - 0.9751555645194946*lnbara*u2*v3*x2 - 
     0.35640766524496237*u3*v3*x2 - 0.09812666811852842*lnbara*u3*v3*
      x2 + 5.626060910791232*v4*x2 + 1.8035499324774895*lnbara*v4*x2 + 
     1.1739473090123895*u*v4*x2 + 0.3798257341629295*lnbara*u*v4*x2 + 
     0.9661938550317448*u2*v4*x2 + 0.2809590613076102*lnbara*u2*v4*x2 - 
     1.6435396981777775*v5*x2 - 0.5192029262043804*lnbara*v5*x2 - 
     0.31687310350773107*u*v5*x2 - 0.09418670348943739*lnbara*u*v5*x2 + 
     0.2767842038860737*v6*x2 + 0.08359044675304854*lnbara*v6*x2 + 
     1.7490619043784765*w*x2 + 0.9115194817125674*lnbara*w*x2 + 
     6.04138458368625*u*w*x2 + 1.9748773458715838*lnbara*u*w*x2 + 
     4.3302489692293955*u2*w*x2 + 1.3823733765239528*lnbara*u2*w*x2 + 
     1.7772082042508452*u3*w*x2 + 0.5107892746111199*lnbara*u3*w*x2 + 
     0.33555238697393286*u4*w*x2 + 0.09558272829689277*lnbara*u4*w*x2 + 
     0.028399509890409173*u5*w*x2 + 0.007442787768918609*lnbara*u5*w*x2 - 
     10.763863509478952*v*w*x2 - 3.7979978404690304*lnbara*v*w*x2 + 
     14.408707884242522*u*v*w*x2 + 4.851483174726708*lnbara*u*v*w*x2 - 
     11.35352724075799*u2*v*w*x2 - 3.638468221673086*lnbara*u2*v*w*x2 + 
     3.595212268118998*u3*v*w*x2 + 1.124626001903981*lnbara*u3*v*w*x2 - 
     0.637464961753253*u4*v*w*x2 - 0.19025408565816512*lnbara*u4*v*w*x2 - 
     2.5057639727820775*v2*w*x2 - 1.0258918950424711*lnbara*v2*w*x2 - 
     4.04603639452907*u*v2*w*x2 - 1.3032259701999138*lnbara*u*v2*w*x2 - 
     0.5438356537989508*u2*v2*w*x2 - 0.1818291572693978*lnbara*u2*v2*w*
      x2 - 0.17072856838087597*u3*v2*w*x2 - 0.04904808595029047*lnbara*
      u3*v2*w*x2 - 4.961706764789829*v3*w*x2 - 
     1.3124617908295524*lnbara*v3*w*x2 + 8.457223146401986*u*v3*w*x2 + 
     2.5565136487937905*lnbara*u*v3*w*x2 - 0.9782558475006098*u2*v3*w*
      x2 - 0.28088060701825807*lnbara*u2*v3*w*x2 + 
     0.01586781553997586*v4*w*x2 - 0.06352181102996676*lnbara*v4*w*x2 - 
     1.0816608287503775*u*v4*w*x2 - 0.32438223975151587*lnbara*u*v4*w*
      x2 - 0.14867177025989595*v5*w*x2 - 0.030113006529747105*lnbara*v5*w*
      x2 + 2.7099088373198637*w2*x2 + 0.653288241029598*lnbara*w2*x2 - 
     18.2664360726891*u*w2*x2 - 6.062567347027013*lnbara*u*w2*x2 + 
     2.165326096275261*u2*w2*x2 + 0.7557728528917382*lnbara*u2*w2*x2 - 
     3.7882789087219217*u3*w2*x2 - 1.1472315670280682*lnbara*u3*w2*x2 + 
     0.2761865963929522*u4*w2*x2 + 0.08549461963730814*lnbara*u4*w2*
      x2 + 6.988719566598391*v*w2*x2 + 2.737479438521948*lnbara*v*w2*
      x2 + 2.583201962578535*u*v*w2*x2 + 0.6303616509938534*lnbara*u*v*w2*
      x2 + 2.78012177163788*u2*v*w2*x2 + 0.9193789312609476*lnbara*u2*v*
      w2*x2 + 0.0944959815922243*u3*v*w2*x2 + 0.012699061790125279*
      lnbara*u3*v*w2*x2 + 8.25179929321027*v2*w2*x2 + 
     2.469212605593015*lnbara*v2*w2*x2 - 8.325060940884864*u*v2*w2*x2 - 
     2.4651692331377286*lnbara*u*v2*w2*x2 + 1.4020259571353741*u2*v2*w2*
      x2 + 0.40905101643700437*lnbara*u2*v2*w2*x2 - 
     0.4179222450177246*v3*w2*x2 - 0.11149384692877298*lnbara*v3*w2*
      x2 + 0.3092023748907339*u*v3*w2*x2 + 0.0785869636098416*lnbara*u*
      v3*w2*x2 + 0.581863351202302*v4*w2*x2 + 
     0.1659602153670836*lnbara*v4*w2*x2 + 0.8972500249744643*w3*x2 + 
     0.3500794053947276*lnbara*w3*x2 + 7.880191664548825*u*w3*x2 + 
     2.6748784087603763*lnbara*u*w3*x2 + 0.9841419128221275*u2*w3*x2 + 
     0.23760579052794478*lnbara*u2*w3*x2 + 0.43479804170814224*u3*w3*
      x2 + 0.1344756922786936*lnbara*u3*w3*x2 - 
     9.326716278242827*v*w3*x2 - 3.107847444868102*lnbara*v*w3*x2 + 
     2.4434643394917717*u*v*w3*x2 + 0.722704550671465*lnbara*u*v*w3*x2 - 
     1.4446667880342972*u2*v*w3*x2 - 0.43070638308913284*lnbara*u2*v*w3*
      x2 - 0.2730340390527256*v2*w3*x2 - 0.09631911852193811*lnbara*v2*
      w3*x2 + 0.5346542533590023*u*v2*w3*x2 + 0.17261962424250146*lnbara*
      u*v2*w3*x2 - 0.6281581353908926*v3*w3*x2 - 
     0.1774903697854292*lnbara*v3*w3*x2 + 0.651338693005845*w4*x2 + 
     0.14754669762714998*lnbara*w4*x2 - 3.7835481298107045*u*w4*x2 - 
     1.1938747004904564*lnbara*u*w4*x2 + 0.05725447402071905*u2*w4*x2 + 
     0.021576912362776347*lnbara*u2*w4*x2 + 2.2331993759180806*v*w4*x2 + 
     0.7744184574067065*lnbara*v*w4*x2 + 0.08267922567088919*u*v*w4*x2 + 
     0.01813757575979942*lnbara*u*v*w4*x2 + 0.3702307156554167*v2*w4*
      x2 + 0.10136840292874362*lnbara*v2*w4*x2 + 
     0.03085436595372382*w5*x2 + 0.016119245278351776*lnbara*w5*x2 + 
     0.47543762922904986*u*w5*x2 + 0.14922477962881076*lnbara*u*w5*x2 - 
     0.46033723691317213*v*w5*x2 - 0.14606708002401708*lnbara*v*w5*x2 + 
     0.013912167390291797*w6*x2 + 0.0027513912903176212*lnbara*w6*x2 - 
     2.2670918716867487*x3 + 0.6858555690183518*u*x3 - 
     0.251564804320496*u2*x3 + 0.0752590109961792*u3*x3 - 
     0.014812198869903107*u4*x3 + 0.001388160767640295*u5*x3 + 
     1.7170463437119636*v*x3 + 0.4329554475790071*lnbara*v*x3 + 
     8.335636464746663*u*v*x3 + 2.843556800957142*lnbara*u*v*x3 + 
     1.7021286302045417*u2*v*x3 + 0.46172402438324356*lnbara*u2*v*x3 + 
     1.5576572952608156*u3*v*x3 + 0.465804859856524*lnbara*u3*v*x3 - 
     0.01399123854239156*u4*v*x3 - 0.006838625664216695*lnbara*u4*v*x3 - 
     6.021984802397401*v2*x3 - 2.0616669488742043*lnbara*v2*x3 - 
     6.264516709859727*u*v2*x3 - 2.002014089554316*lnbara*u*v2*x3 - 
     3.2461502369868342*u2*v2*x3 - 0.9751555645194946*lnbara*u2*v2*x3 - 
     0.35640766524496237*u3*v2*x3 - 0.09812666811852842*lnbara*u3*v2*
      x3 + 0.817892698661721*v3*x3 + 0.36798111338220896*lnbara*v3*x3 + 
     5.761241784590613*u*v3*x3 + 1.759687908443654*lnbara*u*v3*x3 + 
     0.23971655435506595*u2*v3*x3 + 0.06962388387677895*lnbara*u2*v3*
      x3 - 1.2251346623182928*v4*x3 - 0.4000233210925086*lnbara*v4*x3 - 
     0.7794687099225679*u*v4*x3 - 0.23484948154842047*lnbara*u*v4*x3 + 
     0.0956649502705806*v5*x3 + 0.03428802753892857*lnbara*v5*x3 - 
     0.5311674638752799*w*x3 - 0.4329554475790071*lnbara*w*x3 - 
     8.70945468065623*u*w*x3 - 2.843556800957142*lnbara*u*w*x3 - 
     1.6011517276791973*u2*w*x3 - 0.46172402438324356*lnbara*u2*w*x3 - 
     1.576189905969709*u3*w*x3 - 0.465804859856524*lnbara*u3*w*x3 + 
     0.015644250220599377*u4*w*x3 + 0.006838625664216695*lnbara*u4*w*x3 + 
     4.570769251234671*v*w*x3 + 1.7115875434794767*lnbara*v*w*x3 - 
     1.485347767785549*u*v*w*x3 - 0.6728643192060604*lnbara*u*v*w*x3 + 
     2.240193096445929*u2*v*w*x3 + 0.7375497739915498*lnbara*u2*v*w*x3 - 
     0.07656531816033134*u3*v*w*x3 - 0.03634902416016519*lnbara*u3*v*w*
      x3 + 9.909152497593267*v2*w*x3 + 3.0912384401060025*lnbara*v2*w*
      x3 - 5.305139799956746*u*v2*w*x3 - 1.5393343490108804*lnbara*u*v2*w*
      x3 + 1.5675237969406215*u2*v2*w*x3 + 0.4579464223436356*lnbara*u2*
      v2*w*x3 - 1.6389294857674492*v3*w*x3 - 0.5238484235184111*lnbara*
      v3*w*x3 - 0.30529725405798797*u*v3*w*x3 - 
     0.10319808585533102*lnbara*u*v3*w*x3 + 0.8263404688000965*v4*w*x3 + 
     0.2438589036761638*lnbara*v4*w*x3 + 0.8972500249744643*w2*x3 + 
     0.3500794053947276*lnbara*w2*x3 + 7.880191664548825*u*w2*x3 + 
     2.6748784087603763*lnbara*u*w2*x3 + 0.9841419128221275*u2*w2*x3 + 
     0.23760579052794478*lnbara*u2*w2*x3 + 0.43479804170814224*u3*w2*
      x3 + 0.1344756922786936*lnbara*u3*w2*x3 - 
     10.997725885446338*v*w2*x3 - 3.557746295983214*lnbara*v*w2*x3 + 
     2.9792615221240184*u*v*w2*x3 + 0.9000686333403095*lnbara*u*v*w2*x3 - 
     1.8066447210612555*u2*v*w2*x3 - 0.5331767926648573*lnbara*u2*v*w2*
      x3 + 0.3997513066521812*v2*w2*x3 + 0.08307962011998349*lnbara*v2*
      w2*x3 + 0.5497959040120096*u*v2*w2*x3 + 0.17704276038055036*lnbara*
      u*v2*w2*x3 - 0.8243258774524206*v3*w2*x3 - 
     0.23041500895027525*lnbara*v3*w2*x3 + 0.4593564384590013*w3*x3 + 
     0.09852674249500254*lnbara*w3*x3 - 3.462967340849832*u*w3*x3 - 
     1.1204221927730833*lnbara*u*w3*x3 + 0.0015185718493600246*u2*w3*
      x3 + 0.005606486444442812*lnbara*u2*w3*x3 + 
     2.2465725155547824*v*w3*x3 + 0.7870789002790479*lnbara*v*w3*x3 + 
     0.1244773390895362*u*v*w3*x3 + 0.024916275986332885*lnbara*u*v*w3*
      x3 + 0.461488349439593*v2*w3*x3 + 0.12645487928146082*lnbara*v2*
      w3*x3 + 0.17761627523096088*w4*x3 + 0.05371322421188824*lnbara*w4*
      x3 + 0.4131831056920087*u*w4*x3 + 0.13608853103686827*lnbara*u*w4*
      x3 - 0.5665614064319183*v*w4*x3 - 0.1764702004496186*lnbara*v*w4*
      x3 + 0.01134547383834047*w5*x3 + 0.002283398903340656*lnbara*w5*
      x3 + 2.0230034428855013*x4 - 0.386061737342387*u*x4 + 
     0.08755029317854246*u2*x4 - 0.014812198869903107*u3*x4 + 
     0.0012633787101392036*u4*x4 - 2.581767811748015*v*x4 - 
     0.7450820516264817*lnbara*v*x4 - 4.408138300869153*u*v*x4 - 
     1.450672032069528*lnbara*u*v*x4 - 1.1428845474172378*u2*v*x4 - 
     0.3173310228787825*lnbara*u2*v*x4 - 0.2223252216914388*u3*v*x4 - 
     0.06514335645328273*lnbara*u3*v*x4 + 5.626060910791232*v2*x4 + 
     1.8035499324774895*lnbara*v2*x4 + 1.1739473090123895*u*v2*x4 + 
     0.3798257341629295*lnbara*u*v2*x4 + 0.9661938550317448*u2*v2*x4 + 
     0.2809590613076102*lnbara*u2*v2*x4 - 1.2251346623182928*v3*x4 - 
     0.4000233210925086*lnbara*v3*x4 - 0.7794687099225679*u*v3*x4 - 
     0.23484948154842047*lnbara*u*v3*x4 + 0.43991353613455403*v4*x4 + 
     0.1319480025020833*lnbara*v4*x4 + 1.9188684163037728*w*x4 + 
     0.7450820516264817*lnbara*w*x4 + 4.540055219513269*u*w*x4 + 
     1.450672032069528*lnbara*u*w*x4 + 1.122430115235894*u2*w*x4 + 
     0.3173310228787825*lnbara*u2*w*x4 + 0.2239680885196098*u3*w*x4 + 
     0.06514335645328273*lnbara*u3*w*x4 - 6.0865303594656925*v*w*x4 - 
     1.9510966301046395*lnbara*v*w*x4 + 2.583274807875346*u*v*w*x4 + 
     0.814048966327527*lnbara*u*v*w*x4 - 1.0214921534358323*u2*v*w*x4 - 
     0.3025359736703866*lnbara*u2*v*w*x4 - 1.4226442649834423*v2*w*x4 - 
     0.4703228534910837*lnbara*v2*w*x4 + 0.2656466906715195*u*v2*w*x4 + 
     0.08058867723285898*lnbara*u*v2*w*x4 - 0.31318215309396386*v3*w*x4 - 
     0.07872508228066824*lnbara*v3*w*x4 + 0.651338693005845*w2*x4 + 
     0.14754669762714998*lnbara*w2*x4 - 3.7835481298107045*u*w2*x4 - 
     1.1938747004904564*lnbara*u*w2*x4 + 0.05725447402071905*u2*w2*x4 + 
     0.021576912362776347*lnbara*u2*w2*x4 + 2.4334094721170496*v*w2*x4 + 
     0.816632950371704*lnbara*v*w2*x4 + 0.10307841593891565*u*v*w2*x4 + 
     0.018172273278693238*lnbara*u*v*w2*x4 + 0.39505594064956606*v2*w2*
      x4 + 0.11018942250071202*lnbara*v2*w2*x4 + 
     0.17761627523096088*w3*x4 + 0.05371322421188824*lnbara*w3*x4 + 
     0.4131831056920087*u*w3*x4 + 0.13608853103686827*lnbara*u*w3*x4 - 
     0.5397225167390112*v*w3*x4 - 0.16840858737356737*lnbara*v*w3*x4 + 
     0.021325233798974344*w4*x4 + 0.004996244651440274*lnbara*w4*x4 - 
     1.2541858328141202*x5 + 0.1475870262830398*u*x5 - 
     0.018929896217160017*u2*x5 + 0.001388160767640295*u3*x5 + 
     0.9506587243582331*v*x5 + 0.2748859330079916*lnbara*v*x5 + 
     1.84547571556521*u*v*x5 + 0.5730994907821886*lnbara*u*v*x5 + 
     0.11799273363683711*u2*v*x5 + 0.03151257404486913*lnbara*u2*v*x5 - 
     1.6435396981777775*v2*x5 - 0.5192029262043804*lnbara*v2*x5 - 
     0.31687310350773107*u*v2*x5 - 0.09418670348943739*lnbara*u*v2*x5 + 
     0.0956649502705806*v3*x5 + 0.03428802753892857*lnbara*v3*x5 - 
     0.6982508102158693*w*x5 - 0.2748859330079916*lnbara*w*x5 - 
     1.8742275538737843*u*w*x5 - 0.5730994907821886*lnbara*u*w*x5 - 
     0.11604224021663521*u2*w*x5 - 0.03151257404486913*lnbara*u2*w*x5 + 
     1.5716677873215597*v*w*x5 + 0.5030836809260285*lnbara*v*w*x5 - 
     0.15605883003169002*u*v*w*x5 - 0.05503807613937338*lnbara*u*v*w*x5 + 
     0.4211542429001266*v2*w*x5 + 0.12424611305742275*lnbara*v2*w*x5 + 
     0.03085436595372382*w2*x5 + 0.016119245278351776*lnbara*w2*x5 + 
     0.47543762922904986*u*w2*x5 + 0.14922477962881076*lnbara*u*w2*x5 - 
     0.5247428948621341*v*w2*x5 - 0.16081753949969196*lnbara*v*w2*x5 + 
     0.01134547383834047*w3*x5 + 0.002283398903340656*lnbara*w3*x5 + 
     0.5112314175751462*x6 - 0.0339914082143444*u*x6 + 
     0.0018904202297008643*u2*x6 - 0.3370863430138181*v*x6 - 
     0.09609081997302192*lnbara*v*x6 - 0.25711383486763884*u*v*x6 - 
     0.07776386003863338*lnbara*u*v*x6 + 0.2767842038860737*v2*x6 + 
     0.08359044675304854*lnbara*v2*x6 + 0.27910097428080544*w*x6 + 
     0.09609081997302192*lnbara*w*x6 + 0.2599997989809029*u*w*x6 + 
     0.07776386003863338*lnbara*u*w*x6 - 0.2866167180827836*v*w*x6 - 
     0.08634183804336616*lnbara*v*w*x6 + 0.013912167390291797*w2*x6 + 
     0.0027513912903176212*lnbara*w2*x6 - 0.12321474882489236*x7 + 
     0.0035512018991495737*u*x7 + 0.039205434782531995*v*x7 + 
     0.01108690204816448*lnbara*v*x7 - 0.033158063533986065*w*x7 - 
     0.01108690204816448*lnbara*w*x7 + 0.013314221484862214*x8 - 
     1.2020569031595942*y + 0.7968713467733699*u*y - 
     0.6781895601550072*u2*y + 0.5000233108183318*u3*y - 
     0.27683765810185557*u4*y + 0.10482088785932395*u5*y - 
     0.023993960518668304*u6*y + 0.002496169349396353*u7*y + 
     0.9809537469447522*v*y - 0.5171174600050477*u*v*y + 
     0.29918371952782097*u2*v*y - 0.14397058356518277*u3*v*y + 
     0.04988531635163183*u4*v*y - 0.01075135761798518*u5*v*y + 
     0.0010711545725659018*u6*v*y - 0.9004176968697725*v2*y + 
     0.3212835093689003*u*v2*y - 0.1223579533330431*u2*v2*y + 
     0.03702532275275472*u3*v2*y - 0.007317810733171885*u4*v2*y + 
     0.0006870207501448052*u5*v2*y + 0.6858555690183518*v3*y - 
     0.15963636872636416*u*v3*y + 0.03824563394210313*u2*v3*y - 
     0.006605505610793891*u3*v3*y + 0.0005687770966253271*u4*v3*y - 
     0.386061737342387*v4*y + 0.056260781045097456*u*v4*y - 
     0.007692523535941066*u2*v4*y + 0.0005789219466621661*u3*v4*y + 
     0.1475870262830398*v5*y - 0.012249129070819405*u*v5*y + 
     0.0007299175618735945*u2*v5*y - 0.0339914082143444*v6*y + 
     0.001228493995858818*u*v6*y + 0.0035512018991495737*v7*y + 
     0.9809537469447522*w*y - 0.5171174600050477*u*w*y + 
     0.29918371952782097*u2*w*y - 0.14397058356518277*u3*w*y + 
     0.04988531635163183*u4*w*y - 0.01075135761798518*u5*w*y + 
     0.0010711545725659018*u6*w*y - 0.6975695736651812*v*w*y + 
     0.2576868260776136*u*v*w*y - 0.09948507064609574*u2*v*w*y + 
     0.030295348064157374*u3*v*w*y - 0.006007932944401952*u4*v*w*y + 
     0.000565180561156404*u5*v*w*y + 0.45246456501551086*v2*w*y - 
     0.1113550912072001*u*v2*w*y + 0.027313138910434232*u2*v2*w*y - 
     0.004771933785158886*u3*v2*w*y + 0.0004135240269207944*u4*v2*w*y - 
     0.2298476323443832*v3*w*y + 0.03580809539433375*u*v3*w*y - 
     0.005041691635003074*u2*v3*w*y + 0.00038509593985133696*u3*v3*w*y + 
     0.08203160229248424*v4*w*y - 0.007320613539132591*u*v4*w*y + 
     0.0004507102322422891*u2*v4*w*y - 0.018000480690589078*v5*w*y + 
     0.0007017792893363362*u*v5*w*y + 0.0018148095406981609*v6*w*y - 
     0.9004176968697725*w2*y + 0.3212835093689003*u*w2*y - 
     0.1223579533330431*u2*w2*y + 0.03702532275275472*u3*w2*y - 
     0.007317810733171885*u4*w2*y + 0.0006870207501448052*u5*w2*y + 
     0.45246456501551086*v*w2*y - 0.1113550912072001*u*v*w2*y + 
     0.027313138910434232*u2*v*w2*y - 0.004771933785158886*u3*v*w2*y + 
     0.0004135240269207944*u4*v*w2*y - 0.1974796353404678*v2*w2*y + 
     0.03129012445977296*u*v2*w2*y - 0.004442047950830934*u2*v2*w2*y + 
     0.0003407946912851918*u3*v2*w2*y + 0.06395157977258972*v3*w2*y - 
     0.005855071071261368*u*v3*w2*y + 0.0003652020446131235*u2*v3*w2*y - 
     0.013136621448171954*v4*w2*y + 0.000527909826103984*u*v4*w2*y + 
     0.001263472670057093*v5*w2*y + 0.6858555690183518*w3*y - 
     0.15963636872636416*u*w3*y + 0.03824563394210313*u2*w3*y - 
     0.006605505610793891*u3*w3*y + 0.0005687770966253271*u4*w3*y - 
     0.2298476323443832*v*w3*y + 0.03580809539433375*u*v*w3*y - 
     0.005041691635003074*u2*v*w3*y + 0.00038509593985133696*u3*v*w3*y + 
     0.06395157977258972*v2*w3*y - 0.005855071071261368*u*v2*w3*y + 
     0.0003652020446131235*u2*v2*w3*y - 0.011927105098099399*v3*w3*y + 
     0.00048336079466143985*u*v3*w3*y + 0.0010740897315456516*v4*w3*y - 
     0.386061737342387*w4*y + 0.056260781045097456*u*w4*y - 
     0.007692523535941066*u2*w4*y + 0.0005789219466621661*u3*w4*y + 
     0.08203160229248424*v*w4*y - 0.007320613539132591*u*v*w4*y + 
     0.0004507102322422891*u2*v*w4*y - 0.013136621448171954*v2*w4*y + 
     0.000527909826103984*u*v2*w4*y + 0.0010740897315456516*v3*w4*y + 
     0.1475870262830398*w5*y - 0.012249129070819405*u*w5*y + 
     0.0007299175618735945*u2*w5*y - 0.018000480690589078*v*w5*y + 
     0.0007017792893363362*u*v*w5*y + 0.001263472670057093*v2*w5*y - 
     0.0339914082143444*w6*y + 0.001228493995858818*u*w6*y + 
     0.0018148095406981609*v*w6*y + 0.0035512018991495737*w7*y + 
     0.9809537469447522*x*y - 0.5171174600050477*u*x*y + 
     0.29918371952782097*u2*x*y - 0.14397058356518277*u3*x*y + 
     0.04988531635163183*u4*x*y - 0.01075135761798518*u5*x*y + 
     0.0010711545725659018*u6*x*y + 6.729507193086304*v*x*y + 
     2.734558445137702*lnbara*v*x*y + 6.093431672716988*u*v*x*y + 
     1.7772673145259574*lnbara*u*v*x*y + 3.6244323005177024*u2*v*x*y + 
     1.1575775618246809*lnbara*u2*v*x*y + 1.326032325221745*u3*v*x*y + 
     0.36117019327441874*lnbara*u3*v*x*y + 0.2517914740480953*u4*v*x*y + 
     0.07250424934435563*lnbara*u4*v*x*y + 0.020900424621812116*u5*v*x*y + 
     0.0052376950009461855*lnbara*u5*v*x*y - 7.740580989744235*v2*x*y - 
     3.007965370948085*lnbara*v2*x*y - 8.34831676538791*u*v2*x*y - 
     2.570302308603461*lnbara*u*v2*x*y - 4.3167789899540825*u2*v2*x*y - 
     1.303884138860003*lnbara*u2*v2*x*y - 1.0998046234447676*u3*v2*x*y - 
     0.306972893835514*lnbara*u3*v2*x*y - 0.10954859095754421*u4*v2*x*y - 
     0.029980593819149505*lnbara*u4*v2*x*y + 14.150094597275556*v3*x*y + 
     4.8057115258275305*lnbara*v3*x*y + 3.7558333567491418*u*v3*x*y + 
     1.0737089522063064*lnbara*u*v3*x*y + 3.5021103597603167*u2*v3*x*y + 
     1.041566396111375*lnbara*u2*v3*x*y + 0.1081140594632303*u3*v3*x*y + 
     0.025365208983699924*lnbara*u3*v3*x*y - 7.96562555520799*v4*x*y - 
     2.6276787546704745*lnbara*v4*x*y - 2.0218463214313056*u*v4*x*y - 
     0.5668452509261017*lnbara*u*v4*x*y - 0.5891572387478903*u2*v4*x*y - 
     0.17202385787388583*lnbara*u2*v4*x*y + 3.2252148129088143*v5*x*y + 
     1.0069202481453856*lnbara*v5*x*y + 0.23526924787765785*u*v5*x*y + 
     0.06359410612179822*lnbara*u*v5*x*y - 0.4717695787130639*v6*x*y - 
     0.14311605655840995*lnbara*v6*x*y - 7.798871853605498*w*x*y - 
     2.734558445137702*lnbara*w*x*y - 5.604833095618447*u*w*x*y - 
     1.7772673145259574*lnbara*u*w*x*y - 3.826201875320261*u2*w*x*y - 
     1.1575775618246809*lnbara*u2*w*x*y - 1.262771091795936*u3*w*x*y - 
     0.36117019327441874*lnbara*u3*w*x*y - 0.26452915782323383*u4*w*x*y - 
     0.07250424934435563*lnbara*u4*w*x*y - 0.0196913141648136*u5*w*x*y - 
     0.0052376950009461855*lnbara*u5*w*x*y + 0.2576868260776136*v*w*x*y - 
     0.08549175984210892*u*v*w*x*y + 0.023341335158146424*u2*v*w*x*y - 
     0.004288309525390823*u3*v*w*x*y + 0.0003820180212080083*u4*v*w*x*y - 
     15.142507034031205*v2*w*x*y - 4.700428217339612*lnbara*v2*w*x*y + 
     8.684236515504576*u*v2*w*x*y + 2.845063377679382*lnbara*u*v2*w*x*y - 
     2.0369169970657963*u2*v2*w*x*y - 0.6312081178376158*lnbara*u2*v2*w*x*
      y + 0.5879915453536685*u3*v2*w*x*y + 0.17392267685688417*lnbara*u3*
      v2*w*x*y + 2.4387387085229473*v3*w*x*y + 0.726506814968872*lnbara*v3*
      w*x*y - 2.478798620600135*u*v3*w*x*y - 0.8025320427989695*lnbara*u*v3*
      w*x*y - 0.1891735970499443*u2*v3*w*x*y - 0.049895924897055216*lnbara*
      u2*v3*w*x*y - 2.5094652412675753*v4*w*x*y - 
     0.7051754521652789*lnbara*v4*w*x*y + 0.977194806205722*u*v4*w*x*y + 
     0.2916555967133236*lnbara*u*v4*w*x*y + 0.2147385756550609*v5*w*x*y + 
     0.0583092465218906*lnbara*v5*w*x*y + 8.371127296661705*w2*x*y + 
     3.007965370948085*lnbara*w2*x*y + 8.142878738630774*u*w2*x*y + 
     2.570302308603461*lnbara*u*w2*x*y + 4.372397515146079*u2*w2*x*y + 
     1.303884138860003*lnbara*u2*w2*x*y + 1.089630311725629*u3*w2*x*y + 
     0.306972893835514*lnbara*u3*w2*x*y + 0.11045275181230746*u4*w2*x*y + 
     0.029980593819149505*lnbara*u4*w2*x*y + 14.923465303571382*v*w2*x*y + 
     4.700428217339612*lnbara*v*w2*x*y - 8.636182740657008*u*v*w2*x*y - 
     2.845063377679382*lnbara*u*v*w2*x*y + 2.0292029954219255*u2*v*w2*x*
      y + 0.6312081178376158*lnbara*u2*v*w2*x*y - 
     0.5873630368597165*u3*v*w2*x*y - 0.17392267685688417*lnbara*u3*v*w2*
      x*y + 0.03069034292530752*v2*w2*x*y - 0.003995614840649483*u*v2*w2*
      x*y + 0.0002863764888716214*u2*v2*w2*x*y + 
     6.015319832123562*v3*w2*x*y + 1.7632715691466476*lnbara*v3*w2*x*y - 
     0.7304529490550218*u*v3*w2*x*y - 0.21288835092447814*lnbara*u*v3*w2*
      x*y - 0.6626623249676024*v4*w2*x*y - 0.19636416180684646*lnbara*v4*
      w2*x*y - 14.456656314267676*w3*x*y - 4.8057115258275305*lnbara*w3*x*
      y - 3.6907580059131853*u*w3*x*y - 1.0737089522063064*lnbara*u*w3*x*
      y - 3.5123911436649746*u2*w3*x*y - 1.041566396111375*lnbara*u2*w3*x*
      y - 0.10728353612006211*u3*w3*x*y - 0.025365208983699924*lnbara*u3*
      w3*x*y - 2.3689366351444274*v*w3*x*y - 0.726506814968872*lnbara*v*w3*
      x*y + 2.469819197461433*u*v*w3*x*y + 0.8025320427989695*lnbara*u*v*w3*
      x*y + 0.18981304783567338*u2*v*w3*x*y + 0.049895924897055216*lnbara*
      u2*v*w3*x*y - 6.026774148018949*v2*w3*x*y - 
     1.7632715691466476*lnbara*v2*w3*x*y + 0.731117931401122*u*v2*w3*x*
      y + 0.21288835092447814*lnbara*u*v2*w3*x*y + 
     0.00047310410784989136*v3*w3*x*y + 8.072482666125286*w4*x*y + 
     2.6276787546704745*lnbara*w4*x*y + 2.0086611123781046*u*w4*x*y + 
     0.5668452509261017*lnbara*u*w4*x*y + 0.590076818257424*u2*w4*x*y + 
     0.17202385787388583*lnbara*u2*w4*x*y + 2.495271539467373*v*w4*x*y + 
     0.7051754521652789*lnbara*v*w4*x*y - 0.9763863300273927*u*v*w4*x*y - 
     0.2916555967133236*lnbara*u*v*w4*x*y + 0.663692907110261*v2*w4*x*y + 
     0.19636416180684646*lnbara*v2*w4*x*y - 3.248331013993099*w5*x*y - 
     1.0069202481453856*lnbara*w5*x*y - 0.23401253231882208*u*w5*x*y - 
     0.06359410612179822*lnbara*u*w5*x*y - 0.2133826221782255*v*w5*x*y - 
     0.0583092465218906*lnbara*v*w5*x*y + 0.47407842963740254*w6*x*y + 
     0.14311605655840995*lnbara*w6*x*y - 0.9004176968697725*x2*y + 
     0.3212835093689003*u*x2*y - 0.1223579533330431*u2*x2*y + 
     0.03702532275275472*u3*x2*y - 0.007317810733171885*u4*x2*y + 
     0.0006870207501448052*u5*x2*y - 7.740580989744235*v*x2*y - 
     3.007965370948085*lnbara*v*x2*y - 8.34831676538791*u*v*x2*y - 
     2.570302308603461*lnbara*u*v*x2*y - 4.3167789899540825*u2*v*x2*y - 
     1.303884138860003*lnbara*u2*v*x2*y - 1.0998046234447676*u3*v*x2*y - 
     0.306972893835514*lnbara*u3*v*x2*y - 0.10954859095754421*u4*v*x2*y - 
     0.029980593819149505*lnbara*u4*v*x2*y + 0.8190067945965689*v2*x2*y + 
     0.6318629132721324*lnbara*v2*x2*y + 16.737595251903855*u*v2*x2*y + 
     5.219893223262108*lnbara*u*v2*x2*y + 1.43981530141774*u2*v2*x2*y + 
     0.3813842687208064*lnbara*u2*v2*x2*y + 0.8558867409043719*u3*v2*x2*
      y + 0.24521275140423673*lnbara*u3*v2*x2*y - 
     5.574896354901597*v3*x2*y - 1.909904783156106*lnbara*v3*x2*y - 
     6.110264042573513*u*v3*x2*y - 1.8612066536179699*lnbara*u*v3*x2*y - 
     0.8578197999104673*u2*v3*x2*y - 0.24023814331107549*lnbara*u2*v3*
      x2*y + 0.8738097966923795*v4*x2*y + 0.34300365970256935*lnbara*v4*
      x2*y + 1.632450766309338*u*v4*x2*y + 0.47753166783295525*lnbara*u*
      v4*x2*y - 0.2597697450829062*v5*x2*y - 0.08515023996893491*lnbara*
      v5*x2*y + 8.371127296661705*w*x2*y + 3.007965370948085*lnbara*w*x2*
      y + 8.142878738630774*u*w*x2*y + 2.570302308603461*lnbara*u*w*x2*y + 
     4.372397515146079*u2*w*x2*y + 1.303884138860003*lnbara*u2*w*x2*y + 
     1.089630311725629*u3*w*x2*y + 0.306972893835514*lnbara*u3*w*x2*y + 
     0.11045275181230746*u4*w*x2*y + 0.029980593819149505*lnbara*u4*w*x2*
      y + 10.777349982686436*v*w*x2*y + 3.494158013231508*lnbara*v*w*x2*y - 
     14.680040084881753*u*v*w*x2*y - 4.586055804009709*lnbara*u*v*w*x2*y + 
     2.978128785284849*u2*v*w*x2*y + 0.9528593416680298*lnbara*u2*v*w*x2*
      y - 0.7405676776348092*u3*v*w*x2*y - 0.21821356263065994*lnbara*u3*v*
      w*x2*y + 0.5638297481368869*v2*w*x2*y + 0.18788462045702756*lnbara*
      v2*w*x2*y + 0.008389015172307996*u*v2*w*x2*y - 
     0.057042047293007624*lnbara*u*v2*w*x2*y - 0.2581203920240232*u2*v2*w*
      x2*y - 0.07638602271158683*lnbara*u2*v2*w*x2*y + 
     5.676870448679409*v3*w*x2*y + 1.684726260416403*lnbara*v3*w*x2*y - 
     1.182692665484111*u*v3*w*x2*y - 0.32670867639523105*lnbara*u*v3*w*x2*
      y - 0.566190372464217*v4*w*x2*y - 0.1702185650157698*lnbara*v4*w*x2*
      y - 11.93413855963392*w2*x2*y - 4.1260209265036405*lnbara*w2*x2*y - 
     1.9788978869332372*u*w2*x2*y - 0.6338374192524003*lnbara*u*w2*x2*y - 
     4.430910486593282*u2*w2*x2*y - 1.3342436103888362*lnbara*u2*w2*x2*
      y - 0.1142476902831892*u3*w2*x2*y - 0.0269991887735768*lnbara*u3*
      w2*x2*y - 0.6330043203094217*v*w2*x2*y - 0.3326899352140709*lnbara*
      v*w2*x2*y + 3.7124326039755955*u*v*w2*x2*y + 
     1.229224903218318*lnbara*u*v*w2*x2*y + 0.3880630632568426*u2*v*w2*
      x2*y + 0.0969009664171316*lnbara*u2*v*w2*x2*y - 
     6.896344257474369*v2*w2*x2*y - 2.015976321936135*lnbara*v2*w2*x2*
      y + 1.8892318082935522*u*v2*w2*x2*y + 0.5393619159455898*lnbara*u*
      v2*w2*x2*y + 0.030715686889802068*v3*w2*x2*y - 
     0.0026175022546954534*lnbara*v3*w2*x2*y + 5.769076336816564*w3*x2*
      y + 2.0547100979131496*lnbara*w3*x2*y + 2.371701525434619*u*w3*x2*
      y + 0.6890237976926594*lnbara*u*w3*x2*y + 0.729201859518042*u2*w3*
      x2*y + 0.21972319960553072*lnbara*u2*w3*x2*y + 
     3.1903919237729643*v*w3*x2*y + 0.9222524033868401*lnbara*v*w3*x2*y - 
     2.203724440304821*u*v*w3*x2*y - 0.6502064428550666*lnbara*u*v*w3*x2*
      y + 0.501248699467762*v2*w3*x2*y + 0.1627718847199586*lnbara*v2*w3*
      x2*y - 2.872858759697617*w4*x2*y - 0.9340060015696775*lnbara*w4*x2*
      y - 0.13345592306181395*u*w4*x2*y - 0.03997846452824747*lnbara*u*w4*
      x2*y - 0.06439562415041984*v*w4*x2*y - 0.022967627645177263*lnbara*v*
      w4*x2*y + 0.3612787107357604*w5*x2*y + 0.11818205016461883*lnbara*
      w5*x2*y + 0.6858555690183518*x3*y - 0.15963636872636416*u*x3*y + 
     0.03824563394210313*u2*x3*y - 0.006605505610793891*u3*x3*y + 
     0.0005687770966253271*u4*x3*y + 14.150094597275556*v*x3*y + 
     4.8057115258275305*lnbara*v*x3*y + 3.7558333567491418*u*v*x3*y + 
     1.0737089522063064*lnbara*u*v*x3*y + 3.5021103597603167*u2*v*x3*y + 
     1.041566396111375*lnbara*u2*v*x3*y + 0.1081140594632303*u3*v*x3*y + 
     0.025365208983699924*lnbara*u3*v*x3*y - 5.574896354901597*v2*x3*y - 
     1.909904783156106*lnbara*v2*x3*y - 6.110264042573513*u*v2*x3*y - 
     1.8612066536179699*lnbara*u*v2*x3*y - 0.8578197999104673*u2*v2*x3*
      y - 0.24023814331107549*lnbara*u2*v2*x3*y + 
     5.6819090881671865*v3*x3*y + 1.7522332805688188*lnbara*v3*x3*y + 
     0.6056692941957744*u*v3*x3*y + 0.1862563479008138*lnbara*u*v3*x3*y - 
     0.7465338393563911*v4*x3*y - 0.22949316645754256*lnbara*v4*x3*y - 
     14.456656314267676*w*x3*y - 4.8057115258275305*lnbara*w*x3*y - 
     3.6907580059131853*u*w*x3*y - 1.0737089522063064*lnbara*u*w*x3*y - 
     3.5123911436649746*u2*w*x3*y - 1.041566396111375*lnbara*u2*w*x3*y - 
     0.10728353612006211*u3*w*x3*y - 0.025365208983699924*lnbara*u3*w*x3*
      y - 0.09200933391270437*v*w*x3*y - 0.14480531475704334*lnbara*v*w*x3*
      y + 3.724170161382556*u*v*w*x3*y + 1.1721828559253105*lnbara*u*v*w*x3*
      y + 0.1296884569931269*u2*v*w*x3*y + 0.020514943705544776*lnbara*u2*
      v*w*x3*y - 6.183783910974188*v2*w*x3*y - 1.7887689679030814*lnbara*
      v2*w*x3*y + 1.617693856062018*u*v2*w*x3*y + 
     0.4746900012067356*lnbara*u*v2*w*x3*y + 0.032891805121808636*v3*w*x3*
      y - 0.002730352711332793*lnbara*v3*w*x3*y + 
     5.769076336816564*w2*x3*y + 2.0547100979131496*lnbara*w2*x3*y + 
     2.371701525434619*u*w2*x3*y + 0.6890237976926594*lnbara*u*w2*x3*y + 
     0.729201859518042*u2*w2*x3*y + 0.21972319960553072*lnbara*u2*w2*x3*
      y + 3.8502028793420484*v*w2*x3*y + 1.1170305738043862*lnbara*v*w2*
      x3*y - 2.127445819261271*u*v*w2*x3*y - 0.6316147687241351*lnbara*u*v*
      w2*x3*y + 0.2951846833599234*v2*w2*x3*y + 
     0.10231487175528582*lnbara*v2*w2*x3*y - 3.369820769025423*w3*x3*y - 
     1.0804948864701236*lnbara*w3*x3*y - 0.09448932656403045*u*w3*x3*y - 
     0.029331580383414237*lnbara*u*w3*x3*y + 0.04710638371571019*v*w3*x3*
      y + 0.006233587961624442*lnbara*v*w3*x3*y + 
     0.3734562378603602*w4*x3*y + 0.1236750594519651*lnbara*w4*x3*y - 
     0.386061737342387*x4*y + 0.056260781045097456*u*x4*y - 
     0.007692523535941066*u2*x4*y + 0.0005789219466621661*u3*x4*y - 
     7.96562555520799*v*x4*y - 2.6276787546704745*lnbara*v*x4*y - 
     2.0218463214313056*u*v*x4*y - 0.5668452509261017*lnbara*u*v*x4*y - 
     0.5891572387478903*u2*v*x4*y - 0.17202385787388583*lnbara*u2*v*x4*
      y + 0.8738097966923795*v2*x4*y + 0.34300365970256935*lnbara*v2*x4*
      y + 1.632450766309338*u*v2*x4*y + 0.47753166783295525*lnbara*u*v2*
      x4*y - 0.7465338393563911*v3*x4*y - 0.22949316645754256*lnbara*v3*
      x4*y + 8.072482666125286*w*x4*y + 2.6276787546704745*lnbara*w*x4*y + 
     2.0086611123781046*u*w*x4*y + 0.5668452509261017*lnbara*u*w*x4*y + 
     0.590076818257424*u2*w*x4*y + 0.17202385787388583*lnbara*u2*w*x4*y + 
     1.9788571739028187*v*w*x4*y + 0.5910023418671081*lnbara*v*w*x4*y - 
     1.497715134559773*u*v*w*x4*y - 0.43755320330470776*lnbara*u*v*w*x4*y + 
     0.4680668096248443*v2*w*x4*y + 0.13910950612836231*lnbara*v2*w*x4*
      y - 2.872858759697617*w2*x4*y - 0.9340060015696775*lnbara*w2*x4*y - 
     0.13345592306181395*u*w2*x4*y - 0.03997846452824747*lnbara*u*w2*x4*
      y - 0.09314373993962562*v*w2*x4*y - 0.033291399122784844*lnbara*v*w2*
      x4*y + 0.3734562378603602*w3*x4*y + 0.1236750594519651*lnbara*w3*
      x4*y + 0.1475870262830398*x5*y - 0.012249129070819405*u*x5*y + 
     0.0007299175618735945*u2*x5*y + 3.2252148129088143*v*x5*y + 
     1.0069202481453856*lnbara*v*x5*y + 0.23526924787765785*u*v*x5*y + 
     0.06359410612179822*lnbara*u*v*x5*y - 0.2597697450829062*v2*x5*y - 
     0.08515023996893491*lnbara*v2*x5*y - 3.248331013993099*w*x5*y - 
     1.0069202481453856*lnbara*w*x5*y - 0.23401253231882208*u*w*x5*y - 
     0.06359410612179822*lnbara*u*w*x5*y - 0.09961325470566673*v*w*x5*y - 
     0.03303181019568392*lnbara*v*w*x5*y + 0.3612787107357604*w2*x5*y + 
     0.11818205016461883*lnbara*w2*x5*y - 0.0339914082143444*x6*y + 
     0.001228493995858818*u*x6*y - 0.4717695787130639*v*x6*y - 
     0.14311605655840995*lnbara*v*x6*y + 0.47407842963740254*w*x6*y + 
     0.14311605655840995*lnbara*w*x6*y + 0.0035512018991495737*x7*y + 
     1.8468559937823905*y2 - 0.6781895601550072*u*y2 + 
     0.3470714951043783*u2*y2 - 0.15843013830258718*u3*y2 + 
     0.05343363480120978*u4*y2 - 0.011336291067348674*u5*y2 + 
     0.0011180783913151146*u6*y2 - 0.9004176968697725*v*y2 + 
     0.29918371952782097*u*v*y2 - 0.10942628765049751*u2*v*y2 + 
     0.03238162914677262*u3*v*y2 - 0.00631630321372675*u4*v*y2 + 
     0.0005880488320726199*u5*v*y2 + 0.520113697995963*v2*y2 - 
     0.1223579533330431*u*v2*y2 + 0.029164662906062597*u2*v2*y2 - 
     0.0050083201180713645*u3*v2*y2 + 0.00042934540544391214*u4*v2*y2 - 
     0.251564804320496*v3*y2 + 0.03824563394210313*u*v3*y2 - 
     0.005291090684199614*u2*v3*y2 + 0.00039954507731219854*u3*v3*y2 + 
     0.08755029317854246*v4*y2 - 0.007692523535941066*u*v4*y2 + 
     0.0004679400724052549*u2*v4*y2 - 0.018929896217160017*v5*y2 + 
     0.0007299175618735945*u*v5*y2 + 0.0018904202297008643*v6*y2 - 
     0.9004176968697725*w*y2 + 0.29918371952782097*u*w*y2 - 
     0.10942628765049751*u2*w*y2 + 0.03238162914677262*u3*w*y2 - 
     0.00631630321372675*u4*w*y2 + 0.0005880488320726199*u5*w*y2 + 
     0.45246456501551086*v*w*y2 - 0.1076866392526224*u*v*w*y2 + 
     0.02581697089389268*u2*v*w*y2 - 0.004447305079644873*u3*v*w*y2 + 
     0.00038194899426036925*u4*v*w*y2 - 0.1974796353404678*v2*w*y2 + 
     0.03069034292530752*u*v2*w*y2 - 0.0042936520074251285*u2*v2*w*y2 + 
     0.00032622599069967407*u3*v2*w*y2 + 0.06395157977258972*v3*w*y2 - 
     0.005780394040453321*u*v3*w*y2 + 0.0003568540057622896*u2*v3*w*y2 - 
     0.013136621448171954*v4*w*y2 + 0.0005230003987145713*u*v4*w*y2 + 
     0.001263472670057093*v5*w*y2 + 0.520113697995963*w2*y2 - 
     0.1223579533330431*u*w2*y2 + 0.029164662906062597*u2*w2*y2 - 
     0.0050083201180713645*u3*w2*y2 + 0.00042934540544391214*u4*w2*y2 - 
     0.1974796353404678*v*w2*y2 + 0.03069034292530752*u*v*w2*y2 - 
     0.0042936520074251285*u2*v*w2*y2 + 0.00032622599069967407*u3*v*w2*
      y2 + 0.05802930853773854*v2*w2*y2 - 0.005287248454508436*u*v2*w2*
      y2 + 0.00032784931577668774*u2*v2*w2*y2 - 
     0.01113298574577242*v3*w2*y2 + 0.0004489814737672761*u*v3*w2*y2 + 
     0.001019248467243472*v4*w2*y2 - 0.251564804320496*w3*y2 + 
     0.03824563394210313*u*w3*y2 - 0.005291090684199614*u2*w3*y2 + 
     0.00039954507731219854*u3*w3*y2 + 0.06395157977258972*v*w3*y2 - 
     0.005780394040453321*u*v*w3*y2 + 0.0003568540057622896*u2*v*w3*y2 - 
     0.01113298574577242*v2*w3*y2 + 0.0004489814737672761*u*v2*w3*y2 + 
     0.0009524091176864817*v3*w3*y2 + 0.08755029317854246*w4*y2 - 
     0.007692523535941066*u*w4*y2 + 0.0004679400724052549*u2*w4*y2 - 
     0.013136621448171954*v*w4*y2 + 0.0005230003987145713*u*v*w4*y2 + 
     0.001019248467243472*v2*w4*y2 - 0.018929896217160017*w5*y2 + 
     0.0007299175618735945*u*w5*y2 + 0.001263472670057093*v*w5*y2 + 
     0.0018904202297008643*w6*y2 - 0.9004176968697725*x*y2 + 
     0.29918371952782097*u*x*y2 - 0.10942628765049751*u2*x*y2 + 
     0.03238162914677262*u3*x*y2 - 0.00631630321372675*u4*x*y2 + 
     0.0005880488320726199*u5*x*y2 - 4.647008665663399*v*x*y2 - 
     1.7090990282110639*lnbara*v*x*y2 + 0.9708296446662518*u*v*x*y2 + 
     0.4284924263303262*lnbara*u*v*x*y2 + 1.184519706140848*u2*v*x*y2 + 
     0.329646645071357*lnbara*u2*v*x*y2 + 0.3742822830969903*u3*v*x*y2 + 
     0.10776709570789808*lnbara*u3*v*x*y2 + 0.04056836531530246*u4*v*x*
      y2 + 0.010614825463755726*lnbara*u4*v*x*y2 - 
     0.20624980265509574*v2*x*y2 + 0.06378679623852483*lnbara*v2*x*y2 - 
     2.5341893643030327*u*v2*x*y2 - 0.8006228366412104*lnbara*u*v2*x*y2 - 
     1.1971008100465077*u2*v2*x*y2 - 0.3402159951898016*lnbara*u2*v2*x*
      y2 - 0.17277027032117476*u3*v2*x*y2 - 0.04766748743684271*lnbara*
      u3*v2*x*y2 - 2.770275796275451*v3*x*y2 - 
     0.8710102224533081*lnbara*v3*x*y2 + 2.2733271985354553*u*v3*x*y2 + 
     0.6947964005893216*lnbara*u*v3*x*y2 + 0.15265057835965268*u2*v3*x*
      y2 + 0.04014533812336501*lnbara*u2*v3*x*y2 + 
     0.305005920641606*v4*x*y2 + 0.10788365521657699*lnbara*v4*x*y2 - 
     0.3602833223629141*u*v4*x*y2 - 0.10812164650482199*lnbara*u*v4*x*
      y2 - 0.11272620964583531*v5*x*y2 - 0.03339748565253056*lnbara*v5*x*
      y2 + 5.299654762421948*w*x*y2 + 1.7090990282110639*lnbara*w*x*y2 - 
     1.1643976508622838*u*w*x*y2 - 0.4284924263303262*lnbara*u*w*x*y2 - 
     1.1346769147760087*u2*w*x*y2 - 0.329646645071357*lnbara*u2*w*x*y2 - 
     0.3831429064834633*u3*w*x*y2 - 0.10776709570789808*lnbara*u3*w*x*
      y2 - 0.03979465732217202*u4*w*x*y2 - 0.010614825463755726*lnbara*u4*
      w*x*y2 - 0.1113550912072001*v*w*x*y2 + 0.023341335158146424*u*v*w*x*
      y2 - 0.0036412369193938783*u2*v*w*x*y2 + 0.0002913721656218036*u3*v*
      w*x*y2 + 14.046191130880398*v2*w*x*y2 + 4.27806313688087*lnbara*v2*
      w*x*y2 - 1.0675924425094043*u*v2*w*x*y2 - 0.36692372038764914*lnbara*
      u*v2*w*x*y2 + 0.7426221290311992*u2*v2*w*x*y2 + 
     0.21620742554342368*lnbara*u2*v2*w*x*y2 - 3.431997185570922*v3*w*x*
      y2 - 1.0593983869002586*lnbara*v3*w*x*y2 - 
     0.3063351018757037*u*v3*w*x*y2 - 0.0810097794355468*lnbara*u*v3*w*x*
      y2 + 1.1422301498147045*v4*w*x*y2 + 0.33097644191339387*lnbara*v4*w*
      x*y2 - 0.044978574732269516*w2*x*y2 - 0.06378679623852483*lnbara*w2*
      x*y2 + 2.5864306854801558*u*w2*x*y2 + 0.8006228366412104*lnbara*u*
      w2*x*y2 + 1.1889883290028347*u2*w2*x*y2 + 
     0.3402159951898016*lnbara*u2*w2*x*y2 + 0.17341769222311584*u3*w2*x*
      y2 + 0.04766748743684271*lnbara*u3*w2*x*y2 - 
     13.984210663495318*v*w2*x*y2 - 4.27806313688087*lnbara*v*w2*x*y2 + 
     1.059736505268818*u*v*w2*x*y2 + 0.36692372038764914*lnbara*u*v*w2*x*
      y2 - 0.7420694165018765*u2*v*w2*x*y2 - 0.21620742554342368*lnbara*
      u2*v*w2*x*y2 - 0.005287248454508436*v2*w2*x*y2 + 
     0.0003034691554580678*u*v2*w2*x*y2 - 1.0830627587062924*v3*w2*x*
      y2 - 0.2997685920100648*lnbara*v3*w2*x*y2 + 
     2.8467947898195387*w3*x*y2 + 0.8710102224533081*lnbara*w3*x*y2 - 
     2.2828692800346633*u*w3*x*y2 - 0.6947964005893216*lnbara*u*w3*x*y2 - 
     0.15198537243470955*u2*w3*x*y2 - 0.04014533812336501*lnbara*u2*w3*x*
      y2 + 3.420468192644727*v*w3*x*y2 + 1.0593983869002586*lnbara*v*w3*x*
      y2 + 0.3069925110386899*u*v*w3*x*y2 + 0.0810097794355468*lnbara*u*v*
      w3*x*y2 + 1.0839553144032048*v2*w3*x*y2 + 
     0.2997685920100648*lnbara*v2*w3*x*y2 - 0.32019128431906846*w4*x*
      y2 - 0.10788365521657699*lnbara*w4*x*y2 + 0.3611306117059756*u*w4*x*
      y2 + 0.10812164650482199*lnbara*u*w4*x*y2 - 
     1.1411946582446566*v*w4*x*y2 - 0.33097644191339387*lnbara*v*w4*x*
      y2 + 0.11415571194993857*w5*x*y2 + 0.03339748565253056*lnbara*w5*x*
      y2 + 0.520113697995963*x2*y2 - 0.1223579533330431*u*x2*y2 + 
     0.029164662906062597*u2*x2*y2 - 0.0050083201180713645*u3*x2*y2 + 
     0.00042934540544391214*u4*x2*y2 - 0.20624980265509574*v*x2*y2 + 
     0.06378679623852483*lnbara*v*x2*y2 - 2.5341893643030327*u*v*x2*y2 - 
     0.8006228366412104*lnbara*u*v*x2*y2 - 1.1971008100465077*u2*v*x2*
      y2 - 0.3402159951898016*lnbara*u2*v*x2*y2 - 
     0.17277027032117476*u3*v*x2*y2 - 0.04766748743684271*lnbara*u3*v*x2*
      y2 + 6.563551498721121*v2*x2*y2 + 1.9840887474720377*lnbara*v2*x2*
      y2 + 0.14499255922663437*u*v2*x2*y2 + 0.03938451616400227*lnbara*u*
      v2*x2*y2 + 0.8278420296899973*u2*v2*x2*y2 + 
     0.238589191178055*lnbara*u2*v2*x2*y2 - 1.3869981182997964*v3*x2*
      y2 - 0.4218277765499367*lnbara*v3*x2*y2 - 
     0.4779542480040674*u*v3*x2*y2 - 0.13759704980282306*lnbara*u*v3*x2*
      y2 + 0.5651566216606796*v4*x2*y2 + 0.16240221528459253*lnbara*v4*
      x2*y2 - 0.044978574732269516*w*x2*y2 - 0.06378679623852483*lnbara*w*
      x2*y2 + 2.5864306854801558*u*w*x2*y2 + 0.8006228366412104*lnbara*u*
      w*x2*y2 + 1.1889883290028347*u2*w*x2*y2 + 
     0.3402159951898016*lnbara*u2*w*x2*y2 + 0.17341769222311584*u3*w*x2*
      y2 + 0.04766748743684271*lnbara*u3*w*x2*y2 - 
     8.691639367114691*v*w*x2*y2 - 2.6585984883283214*lnbara*v*w*x2*y2 + 
     4.260756983735999*u*v*w*x2*y2 + 1.2888307504910574*lnbara*u*v*w*x2*
      y2 - 0.6381316954939846*u2*v*w*x2*y2 - 0.19013921429911454*lnbara*
      u2*v*w*x2*y2 - 1.4568456155246001*v2*w*x2*y2 - 
     0.48533250050556614*lnbara*v2*w*x2*y2 - 0.5429343918861453*u*v2*w*
      x2*y2 - 0.1557593910339726*lnbara*u*v2*w*x2*y2 - 
     0.7046868197766329*v3*w*x2*y2 - 0.18518041263910326*lnbara*v3*w*x2*
      y2 + 2.216432092777231*w2*x2*y2 + 0.6745097408562837*lnbara*w2*x2*
      y2 - 4.417728806752933*u*w2*x2*y2 - 1.3282152666550597*lnbara*u*w2*
      x2*y2 - 0.18884186308954337*u2*w2*x2*y2 - 
     0.048449976878940454*lnbara*u2*w2*x2*y2 + 2.499274797319209*v*w2*
      x2*y2 + 0.8372694295660547*lnbara*v*w2*x2*y2 + 
     0.36293415511749816*u*v*w2*x2*y2 + 0.09272663386743893*lnbara*u*v*w2*
      x2*y2 + 1.4101267527346482*v2*w2*x2*y2 + 
     0.39576075754209206*lnbara*v2*w2*x2*y2 + 0.32522417184198954*w3*x2*
      y2 + 0.06989084748944815*lnbara*w3*x2*y2 + 
     0.6591783435173679*u*w3*x2*y2 + 0.20062980696935676*lnbara*u*w3*x2*
      y2 - 1.3871873506084558*v*w3*x2*y2 - 0.4071365920310302*lnbara*v*
      w3*x2*y2 + 0.11854588979479715*w4*x2*y2 + 
     0.03415403184344883*lnbara*w4*x2*y2 - 0.251564804320496*x3*y2 + 
     0.03824563394210313*u*x3*y2 - 0.005291090684199614*u2*x3*y2 + 
     0.00039954507731219854*u3*x3*y2 - 2.770275796275451*v*x3*y2 - 
     0.8710102224533081*lnbara*v*x3*y2 + 2.2733271985354553*u*v*x3*y2 + 
     0.6947964005893216*lnbara*u*v*x3*y2 + 0.15265057835965268*u2*v*x3*
      y2 + 0.04014533812336501*lnbara*u2*v*x3*y2 - 
     1.3869981182997964*v2*x3*y2 - 0.4218277765499367*lnbara*v2*x3*y2 - 
     0.4779542480040674*u*v2*x3*y2 - 0.13759704980282306*lnbara*u*v2*x3*
      y2 - 0.19054563887772494*v3*x3*y2 - 0.04854179569331324*lnbara*v3*
      x3*y2 + 2.8467947898195387*w*x3*y2 + 0.8710102224533081*lnbara*w*
      x3*y2 - 2.2828692800346633*u*w*x3*y2 - 0.6947964005893216*lnbara*u*
      w*x3*y2 - 0.15198537243470955*u2*w*x3*y2 - 
     0.04014533812336501*lnbara*u2*w*x3*y2 + 1.046392947967706*v*w*x3*
      y2 + 0.35193692906048857*lnbara*v*w*x3*y2 - 
     0.18025531671901043*u*v*w*x3*y2 - 0.06303275716653367*lnbara*u*v*w*x3*
      y2 + 1.492463738628857*v2*w*x3*y2 + 0.4272793464214101*lnbara*v2*w*
      x3*y2 + 0.32522417184198954*w2*x3*y2 + 0.06989084748944815*lnbara*
      w2*x3*y2 + 0.6591783435173679*u*w2*x3*y2 + 
     0.20062980696935676*lnbara*u*w2*x3*y2 - 1.4916718912285227*v*w2*x3*
      y2 - 0.432716958385381*lnbara*v*w2*x3*y2 + 
     0.19122371263757587*w3*x3*y2 + 0.05397940765728412*lnbara*w3*x3*
      y2 + 0.08755029317854246*x4*y2 - 0.007692523535941066*u*x4*y2 + 
     0.0004679400724052549*u2*x4*y2 + 0.305005920641606*v*x4*y2 + 
     0.10788365521657699*lnbara*v*x4*y2 - 0.3602833223629141*u*v*x4*y2 - 
     0.10812164650482199*lnbara*u*v*x4*y2 + 0.5651566216606796*v2*x4*
      y2 + 0.16240221528459253*lnbara*v2*x4*y2 - 
     0.32019128431906846*w*x4*y2 - 0.10788365521657699*lnbara*w*x4*y2 + 
     0.3611306117059756*u*w*x4*y2 + 0.10812164650482199*lnbara*u*w*x4*
      y2 - 0.6823722786029639*v*w*x4*y2 - 0.19655624712804137*lnbara*v*w*
      x4*y2 + 0.11854588979479715*w2*x4*y2 + 0.03415403184344883*lnbara*
      w2*x4*y2 - 0.018929896217160017*x5*y2 + 0.0007299175618735945*u*
      x5*y2 - 0.11272620964583531*v*x5*y2 - 0.03339748565253056*lnbara*v*
      x5*y2 + 0.11415571194993857*w*x5*y2 + 0.03339748565253056*lnbara*w*
      x5*y2 + 0.0018904202297008643*x6*y2 - 2.2670918716867487*y3 + 
     0.5000233108183318*u*y3 - 0.15843013830258718*u2*y3 + 
     0.044009786371943985*u3*y3 - 0.00830686713692306*u4*y3 + 
     0.0007585417970558358*u5*y3 + 0.6858555690183518*v*y3 - 
     0.14397058356518277*u*v*y3 + 0.03238162914677262*u2*v*y3 - 
     0.0053911589121258706*u3*v*y3 + 0.0004539037516950064*u4*v*y3 - 
     0.251564804320496*v2*y3 + 0.03702532275275472*u*v2*y3 - 
     0.0050083201180713645*u2*v2*y3 + 0.0003728851239061164*u3*v2*y3 + 
     0.0752590109961792*v3*y3 - 0.006605505610793891*u*v3*y3 + 
     0.00039954507731219854*u2*v3*y3 - 0.014812198869903107*v4*y3 + 
     0.0005789219466621661*u*v4*y3 + 0.001388160767640295*v5*y3 + 
     0.6858555690183518*w*y3 - 0.14397058356518277*u*w*y3 + 
     0.03238162914677262*u2*w*y3 - 0.0053911589121258706*u3*w*y3 + 
     0.0004539037516950064*u4*w*y3 - 0.2298476323443832*v*w*y3 + 
     0.0339939779841863*u*v*w*y3 - 0.004611648226787489*u2*v*w*y3 + 
     0.0003439536028536974*u3*v*w*y3 + 0.06395157977258972*v2*w*y3 - 
     0.005673921854933901*u*v2*w*y3 + 0.00034533361905439837*u2*v2*w*
      y3 - 0.011927105098099399*v3*w*y3 + 0.00047310410784989136*u*v3*w*
      y3 + 0.0010740897315456516*v4*w*y3 - 0.251564804320496*w2*y3 + 
     0.03702532275275472*u*w2*y3 - 0.0050083201180713645*u2*w2*y3 + 
     0.0003728851239061164*u3*w2*y3 + 0.06395157977258972*v*w2*y3 - 
     0.005673921854933901*u*v*w2*y3 + 0.00034533361905439837*u2*v*w2*
      y3 - 0.01113298574577242*v2*w2*y3 + 0.00044357422314508923*u*v2*
      w2*y3 + 0.0009524091176864817*v3*w2*y3 + 
     0.0752590109961792*w3*y3 - 0.006605505610793891*u*w3*y3 + 
     0.00039954507731219854*u2*w3*y3 - 0.011927105098099399*v*w3*y3 + 
     0.00047310410784989136*u*v*w3*y3 + 0.0009524091176864817*v2*w3*y3 - 
     0.014812198869903107*w4*y3 + 0.0005789219466621661*u*w4*y3 + 
     0.0010740897315456516*v*w4*y3 + 0.001388160767640295*w5*y3 + 
     0.6858555690183518*x*y3 - 0.14397058356518277*u*x*y3 + 
     0.03238162914677262*u2*x*y3 - 0.0053911589121258706*u3*x*y3 + 
     0.0004539037516950064*u4*x*y3 + 6.221638137007376*v*x*y3 + 
     2.0509188338532764*lnbara*v*x*y3 + 1.6028488135384624*u*v*x*y3 + 
     0.4192402591437163*lnbara*u*v*x*y3 + 0.43459981748405074*u2*v*x*y3 + 
     0.12608980116426613*lnbara*u2*v*x*y3 + 0.05131419753175176*u3*v*x*
      y3 + 0.013481184061483282*lnbara*u3*v*x*y3 - 
     2.339296200037759*v2*x*y3 - 0.7884496760703545*lnbara*v2*x*y3 - 
     0.9468503021957163*u*v2*x*y3 - 0.26779586684578455*lnbara*u*v2*x*
      y3 - 0.16757153566204508*u2*v2*x*y3 - 0.046821383490061465*lnbara*
      u2*v2*x*y3 + 2.402267061883944*v3*x*y3 + 
     0.7222688114635434*lnbara*v3*x*y3 + 0.10166794959991486*u*v3*x*y3 + 
     0.025545381796987655*lnbara*u*v3*x*y3 - 0.3237542215275384*v4*x*y3 - 
     0.09733306860121325*lnbara*v4*x*y3 - 6.543865639160677*w*x*y3 - 
     2.0509188338532764*lnbara*w*x*y3 - 1.5432862100326823*u*w*x*y3 - 
     0.4192402591437163*lnbara*u*w*x*y3 - 0.44329609772338113*u2*w*x*y3 - 
     0.12608980116426613*lnbara*u2*w*x*y3 - 0.050643992483851874*u3*w*x*
      y3 - 0.013481184061483282*lnbara*u3*w*x*y3 + 
     0.03580809539433375*v*w*x*y3 - 0.004288309525390823*u*v*w*x*y3 + 
     0.0002913721656218036*u2*v*w*x*y3 - 4.2497906793289655*v2*w*x*y3 - 
     1.229468487533898*lnbara*v2*w*x*y3 + 0.4686008650126944*u*v2*w*x*
      y3 + 0.14200807750295125*lnbara*u*v2*w*x*y3 + 
     0.21733303969258025*v3*w*x*y3 + 0.06469446777014416*lnbara*v3*w*x*
      y3 + 2.417035504771195*w2*x*y3 + 0.7884496760703545*lnbara*w2*x*
      y3 + 0.9375779785463527*u*w2*x*y3 + 0.26779586684578455*lnbara*u*w2*
      x*y3 + 0.16819984993563145*u2*w2*x*y3 + 0.046821383490061465*lnbara*
      u2*w2*x*y3 + 4.23815521421725*v*w2*x*y3 + 
     1.229468487533898*lnbara*v*w2*x*y3 - 0.46795439814148526*u*v*w2*x*
      y3 - 0.14200807750295125*lnbara*u*v*w2*x*y3 + 
     0.0004489814737672761*v2*w2*x*y3 - 2.4157548822081845*w3*x*y3 - 
     0.7222688114635434*lnbara*w3*x*y3 - 0.10092543442474523*u*w3*x*y3 - 
     0.025545381796987655*lnbara*u*w3*x*y3 - 0.21637657479006892*v*w3*x*
      y3 - 0.06469446777014416*lnbara*v*w3*x*y3 + 
     0.3249190900404754*w4*x*y3 + 0.09733306860121325*lnbara*w4*x*y3 - 
     0.251564804320496*x2*y3 + 0.03702532275275472*u*x2*y3 - 
     0.0050083201180713645*u2*x2*y3 + 0.0003728851239061164*u3*x2*y3 - 
     2.339296200037759*v*x2*y3 - 0.7884496760703545*lnbara*v*x2*y3 - 
     0.9468503021957163*u*v*x2*y3 - 0.26779586684578455*lnbara*u*v*x2*
      y3 - 0.16757153566204508*u2*v*x2*y3 - 0.046821383490061465*lnbara*
      u2*v*x2*y3 - 0.6788052052736328*v2*x2*y3 - 
     0.16864345026577507*lnbara*v2*x2*y3 + 0.717639217730022*u*v2*x2*
      y3 + 0.20684606078825712*lnbara*u*v2*x2*y3 - 
     0.20393804085263473*v3*x2*y3 - 0.06221919382932583*lnbara*v3*x2*
      y3 + 2.417035504771195*w*x2*y3 + 0.7884496760703545*lnbara*w*x2*
      y3 + 0.9375779785463527*u*w*x2*y3 + 0.26779586684578455*lnbara*u*w*
      x2*y3 + 0.16819984993563145*u2*w*x2*y3 + 
     0.046821383490061465*lnbara*u2*w*x2*y3 + 2.9845483465315206*v*w*x2*
      y3 + 0.873539503307877*lnbara*v*w*x2*y3 - 0.7548466365170377*u*v*w*
      x2*y3 - 0.21880709774223123*lnbara*u*v*w*x2*y3 + 
     0.029077792864669175*v2*w*x2*y3 + 0.011972114859594282*lnbara*v2*w*
      x2*y3 - 2.321676668163961*w2*x2*y3 - 0.704896053042102*lnbara*w2*
      x2*y3 + 0.038158053864272005*u*w2*x2*y3 + 
     0.011961036953974104*lnbara*u*w2*x2*y3 - 0.10578881199195397*v*w2*
      x2*y3 - 0.04001243623425464*lnbara*v*w2*x2*y3 + 
     0.28220067646190156*w3*x2*y3 + 0.09025951520398619*lnbara*w3*x2*
      y3 + 0.0752590109961792*x3*y3 - 0.006605505610793891*u*x3*y3 + 
     0.00039954507731219854*u2*x3*y3 + 2.402267061883944*v*x3*y3 + 
     0.7222688114635434*lnbara*v*x3*y3 + 0.10166794959991486*u*v*x3*y3 + 
     0.025545381796987655*lnbara*u*v*x3*y3 - 0.20393804085263473*v2*x3*
      y3 - 0.06221919382932583*lnbara*v2*x3*y3 - 
     2.4157548822081845*w*x3*y3 - 0.7222688114635434*lnbara*w*x3*y3 - 
     0.10092543442474523*u*w*x3*y3 - 0.025545381796987655*lnbara*u*w*x3*
      y3 - 0.07704797923780887*v*w*x3*y3 - 0.028040321374660362*lnbara*v*w*
      x3*y3 + 0.28220067646190156*w2*x3*y3 + 0.09025951520398619*lnbara*
      w2*x3*y3 - 0.014812198869903107*x4*y3 + 0.0005789219466621661*u*
      x4*y3 - 0.3237542215275384*v*x4*y3 - 0.09733306860121325*lnbara*v*
      x4*y3 + 0.3249190900404754*w*x4*y3 + 0.09733306860121325*lnbara*w*
      x4*y3 + 0.001388160767640295*x5*y3 + 2.0230034428855013*y4 - 
     0.27683765810185557*u*y4 + 0.05343363480120978*u2*y4 - 
     0.00830686713692306*u3*y4 + 0.0006746357862159641*u4*y4 - 
     0.386061737342387*v*y4 + 0.04988531635163183*u*v*y4 - 
     0.00631630321372675*u2*v*y4 + 0.0004539037516950064*u3*v*y4 + 
     0.08755029317854246*v2*y4 - 0.007317810733171885*u*v2*y4 + 
     0.00042934540544391214*u2*v2*y4 - 0.014812198869903107*v3*y4 + 
     0.0005687770966253271*u*v3*y4 + 0.0012633787101392036*v4*y4 - 
     0.386061737342387*w*y4 + 0.04988531635163183*u*w*y4 - 
     0.00631630321372675*u2*w*y4 + 0.0004539037516950064*u3*w*y4 + 
     0.08203160229248424*v*w*y4 - 0.006873088261070091*u*v*w*y4 + 
     0.00040392715269129916*u2*v*w*y4 - 0.013136621448171954*v2*w*y4 + 
     0.000507581743944034*u*v2*w*y4 + 0.0010740897315456516*v3*w*y4 + 
     0.08755029317854246*w2*y4 - 0.007317810733171885*u*w2*y4 + 
     0.00042934540544391214*u2*w2*y4 - 0.013136621448171954*v*w2*y4 + 
     0.000507581743944034*u*v*w2*y4 + 0.001019248467243472*v2*w2*y4 - 
     0.014812198869903107*w3*y4 + 0.0005687770966253271*u*w3*y4 + 
     0.0010740897315456516*v*w3*y4 + 0.0012633787101392036*w4*y4 - 
     0.386061737342387*x*y4 + 0.04988531635163183*u*x*y4 - 
     0.00631630321372675*u2*x*y4 + 0.0004539037516950064*u3*x*y4 - 
     2.7455486537995553*v*x*y4 - 0.8545495141055319*lnbara*v*x*y4 - 
     0.06972985855284485*u*v*x*y4 - 0.004869163308206694*lnbara*u*v*x*y4 + 
     0.02029024705940767*u2*v*x*y4 + 0.005188045152882472*lnbara*u2*v*x*
      y4 + 0.260373564991028*v2*x*y4 + 0.09277616733835867*lnbara*v2*x*
      y4 - 0.04301078480890973*u*v2*x*y4 - 0.013022653225572441*lnbara*u*
      v2*x*y4 - 0.2353317760308018*v3*x*y4 - 0.06767441939812754*lnbara*
      v3*x*y4 + 2.8587812294103165*w*x*y4 + 0.8545495141055319*lnbara*w*x*
      y4 + 0.05785733009437446*u*w*x*y4 + 0.004869163308206694*lnbara*u*w*x*
      y4 - 0.01953851722470816*u2*w*x*y4 - 0.005188045152882472*lnbara*u2*
      w*x*y4 - 0.007320613539132591*v*w*x*y4 + 0.0003820180212080083*u*v*w*
      x*y4 + 0.7480915272662646*v2*w*x*y4 + 0.21034811390666633*lnbara*v2*
      w*x*y4 - 0.2759336414712597*w2*x*y4 - 0.09277616733835867*lnbara*w2*
      x*y4 + 0.04382088794664975*u*w2*x*y4 + 0.013022653225572441*lnbara*u*
      w2*x*y4 - 0.747040617041446*v*w2*x*y4 - 0.21034811390666633*lnbara*
      v*w2*x*y4 + 0.23650678939377565*w3*x*y4 + 
     0.06767441939812754*lnbara*w3*x*y4 + 0.08755029317854246*x2*y4 - 
     0.007317810733171885*u*x2*y4 + 0.00042934540544391214*u2*x2*y4 + 
     0.260373564991028*v*x2*y4 + 0.09277616733835867*lnbara*v*x2*y4 - 
     0.04301078480890973*u*v*x2*y4 - 0.013022653225572441*lnbara*u*v*x2*
      y4 + 0.28165966173252016*v2*x2*y4 + 0.07779208068357109*lnbara*v2*
      x2*y4 - 0.2759336414712597*w*x2*y4 - 0.09277616733835867*lnbara*w*
      x2*y4 + 0.04382088794664975*u*w*x2*y4 + 0.013022653225572441*lnbara*
      u*w*x2*y4 - 0.4807266928857714*v*w*x2*y4 - 
     0.1356427800167511*lnbara*v*w*x2*y4 + 0.20047304487804693*w2*x2*
      y4 + 0.05785069933318*lnbara*w2*x2*y4 - 0.014812198869903107*x3*
      y4 + 0.0005687770966253271*u*x3*y4 - 0.2353317760308018*v*x3*y4 - 
     0.06767441939812754*lnbara*v*x3*y4 + 0.23650678939377565*w*x3*y4 + 
     0.06767441939812754*lnbara*w*x3*y4 + 0.0012633787101392036*x4*y4 - 
     1.2541858328141202*y5 + 0.10482088785932395*u*y5 - 
     0.011336291067348674*u2*y5 + 0.0007585417970558358*u3*y5 + 
     0.1475870262830398*v*y5 - 0.01075135761798518*u*v*y5 + 
     0.0005880488320726199*u2*v*y5 - 0.018929896217160017*v2*y5 + 
     0.0006870207501448052*u*v2*y5 + 0.001388160767640295*v3*y5 + 
     0.1475870262830398*w*y5 - 0.01075135761798518*u*w*y5 + 
     0.0005880488320726199*u2*w*y5 - 0.018000480690589078*v*w*y5 + 
     0.0006541741874990681*u*v*w*y5 + 0.001263472670057093*v2*w*y5 - 
     0.018929896217160017*w2*y5 + 0.0006870207501448052*u*w2*y5 + 
     0.001263472670057093*v*w2*y5 + 0.001388160767640295*w3*y5 + 
     0.1475870262830398*x*y5 - 0.01075135761798518*u*x*y5 + 
     0.0005880488320726199*u2*x*y5 + 0.9067948266532218*v*x*y5 + 
     0.26857270443316716*lnbara*v*x*y5 + 0.04008569352545048*u*v*x*y5 + 
     0.009316737180946303*lnbara*u*v*x*y5 - 0.06841032198704927*v2*x*y5 - 
     0.021403950235092197*lnbara*v2*x*y5 - 0.9314087991903407*w*x*y5 - 
     0.26857270443316716*lnbara*w*x*y5 - 0.038965576694794626*u*w*x*y5 - 
     0.009316737180946303*lnbara*u*w*x*y5 + 0.0007017792893363362*v*w*x*
      y5 + 0.06988272110288132*w2*x*y5 + 0.021403950235092197*lnbara*w2*x*
      y5 - 0.018929896217160017*x2*y5 + 0.0006870207501448052*u*x2*y5 - 
     0.06841032198704927*v*x2*y5 - 0.021403950235092197*lnbara*v*x2*y5 + 
     0.06988272110288132*w*x2*y5 + 0.021403950235092197*lnbara*w*x2*y5 + 
     0.001388160767640295*x3*y5 + 0.5112314175751462*y6 - 
     0.023993960518668304*u*y6 + 0.0011180783913151146*u2*y6 - 
     0.0339914082143444*v*y6 + 0.0010711545725659018*u*v*y6 + 
     0.0018904202297008643*v2*y6 - 0.0339914082143444*w*y6 + 
     0.0010711545725659018*u*w*y6 + 0.0018148095406981609*v*w*y6 + 
     0.0018904202297008643*w2*y6 - 0.0339914082143444*x*y6 + 
     0.0010711545725659018*u*x*y6 - 0.11244046513485796*v*x*y6 - 
     0.032045606778957444*lnbara*v*x*y6 + 0.11490665548248956*w*x*y6 + 
     0.032045606778957444*lnbara*w*x*y6 + 0.0018904202297008643*x2*y6 - 
     0.12321474882489236*y7 + 0.002496169349396353*u*y7 + 
     0.0035512018991495737*v*y7 + 0.0035512018991495737*w*y7 + 
     0.0035512018991495737*x*y7 + 0.013314221484862214*y8 - 
     1.2020569031595942*z + 0.9809537469447522*u*z - 
     0.9004176968697725*u2*z + 0.6858555690183518*u3*z - 
     0.386061737342387*u4*z + 0.1475870262830398*u5*z - 
     0.0339914082143444*u6*z + 0.0035512018991495737*u7*z + 
     0.7968713467733699*v*z - 2.9443689245385314*u*v*z - 
     0.9115194817125674*lnbara*u*v*z - 2.177117134294563*u2*v*z - 
     0.7823223301987707*lnbara*u2*v*z - 1.5212050940015565*u3*v*z - 
     0.41420887449201577*lnbara*u3*v*z - 0.33851565849786086*u4*v*z - 
     0.10984007042350338*lnbara*u4*v*z - 0.07720156919970216*u5*v*z - 
     0.018029304599241732*lnbara*u5*v*z - 0.003018119129298935*u6*v*z - 
     0.0010840252459144199*lnbara*u6*v*z + 1.7490619043784765*v2*z + 
     0.9115194817125674*lnbara*v2*z + 6.04138458368625*u*v2*z + 
     1.9748773458715838*lnbara*u*v2*z + 4.3302489692293955*u2*v2*z + 
     1.3823733765239528*lnbara*u2*v2*z + 1.7772082042508452*u3*v2*z + 
     0.5107892746111199*lnbara*u3*v2*z + 0.33555238697393286*u4*v2*z + 
     0.09558272829689277*lnbara*u4*v2*z + 0.028399509890409173*u5*v2*z + 
     0.007442787768918609*lnbara*u5*v2*z - 0.5311674638752799*v3*z - 
     0.4329554475790071*lnbara*v3*z - 8.70945468065623*u*v3*z - 
     2.843556800957142*lnbara*u*v3*z - 1.6011517276791973*u2*v3*z - 
     0.46172402438324356*lnbara*u2*v3*z - 1.576189905969709*u3*v3*z - 
     0.465804859856524*lnbara*u3*v3*z + 0.015644250220599377*u4*v3*z + 
     0.006838625664216695*lnbara*u4*v3*z + 1.9188684163037728*v4*z + 
     0.7450820516264817*lnbara*v4*z + 4.540055219513269*u*v4*z + 
     1.450672032069528*lnbara*u*v4*z + 1.122430115235894*u2*v4*z + 
     0.3173310228787825*lnbara*u2*v4*z + 0.2239680885196098*u3*v4*z + 
     0.06514335645328273*lnbara*u3*v4*z - 0.6982508102158693*v5*z - 
     0.2748859330079916*lnbara*v5*z - 1.8742275538737843*u*v5*z - 
     0.5730994907821886*lnbara*u*v5*z - 0.11604224021663521*u2*v5*z - 
     0.03151257404486913*lnbara*u2*v5*z + 0.27910097428080544*v6*z + 
     0.09609081997302192*lnbara*v6*z + 0.2599997989809029*u*v6*z + 
     0.07776386003863338*lnbara*u*v6*z - 0.033158063533986065*v7*z - 
     0.01108690204816448*lnbara*v7*z + 0.9809537469447522*w*z + 
     1.7296818908683025*u*w*z + 0.9115194817125674*lnbara*u*w*z + 
     2.950865208678974*u2*w*z + 0.7823223301987707*lnbara*u2*w*z + 
     1.131721092930809*u3*w*z + 0.41420887449201577*lnbara*u3*w*z + 
     0.47680804183544256*u4*w*z + 0.10984007042350338*lnbara*u4*w*z + 
     0.046951959438293674*u5*w*z + 0.018029304599241732*lnbara*u5*w*z + 
     0.006061422665855913*u6*w*z + 0.0010840252459144199*lnbara*u6*w*z - 
     0.5171174600050477*v*w*z + 0.2576868260776136*u*v*w*z - 
     0.1113550912072001*u2*v*w*z + 0.03580809539433375*u3*v*w*z - 
     0.007320613539132591*u4*v*w*z + 0.0007017792893363362*u5*v*w*z - 
     16.220273175775272*v2*w*z - 5.75015242423565*lnbara*v2*w*z + 
     7.283613291269799*u*v2*w*z + 2.569763477061043*lnbara*u*v2*w*z - 
     10.868438804341743*u2*v2*w*z - 3.4712775032258807*lnbara*u2*v2*w*z + 
     2.723658999110921*u3*v2*w*z + 0.8563693554764021*lnbara*u3*v2*w*z - 
     0.5498436699327358*u4*v2*w*z - 0.16402829156973656*lnbara*u4*v2*w*
      z + 11.250037884026336*v3*w*z + 3.9890155563258416*lnbara*v3*w*z + 
     0.5861690638750913*u*v3*w*z + 0.04824677725141553*lnbara*u*v3*w*z + 
     2.7450669338987366*u2*v3*w*z + 0.8803435778848646*lnbara*u2*v3*w*z - 
     0.06800667812701873*u3*v3*w*z - 0.029360220026995808*lnbara*u3*v3*w*
      z - 10.094036110016523*v4*w*z - 3.28316404172915*lnbara*v4*w*z + 
     1.0213274392385536*u*v4*w*z + 0.336346527741606*lnbara*u*v4*w*z - 
     0.8972167925745769*u2*v4*w*z - 0.26783993094333697*lnbara*u2*v4*w*
      z + 3.2110778233847816*v5*w*z + 1.0179245136226938*lnbara*v5*w*z - 
     0.01449065568595044*u*v5*w*z - 0.010275898356323341*lnbara*u*v5*w*z - 
     0.5222803748590084*v6*w*z - 0.15852336677018297*lnbara*v6*w*z - 
     3.3276691614032563*w2*z - 0.9115194817125674*lnbara*w2*z - 
     5.289736299142919*u*w2*z - 1.9748773458715838*lnbara*u*w2*z - 
     4.650086557902906*u2*w2*z - 1.3823733765239528*lnbara*u2*w2*z - 
     1.6750109905361525*u3*w2*z - 0.5107892746111199*lnbara*u3*w2*z - 
     0.3563815319580459*u4*w2*z - 0.09558272829689277*lnbara*u4*w2*z - 
     0.026406119658478486*u5*w2*z - 0.007442787768918609*lnbara*u5*w2*z + 
     16.840740404671994*v*w2*z + 5.75015242423565*lnbara*v*w2*z - 
     7.4944534531230955*u*v*w2*z - 2.569763477061043*lnbara*u*v*w2*z + 
     10.92704206771195*u2*v*w2*z + 3.4712775032258807*lnbara*u2*v*w2*z - 
     2.7345557618171856*u3*v*w2*z - 0.8563693554764021*lnbara*u3*v*w2*z + 
     0.5508222899910821*u4*v*w2*z + 0.16402829156973656*lnbara*u4*v*w2*
      z - 0.1223579533330431*v2*w2*z + 0.027313138910434232*u*v2*w2*z - 
     0.004442047950830934*u2*v2*w2*z + 0.0003652020446131235*u3*v2*w2*
      z + 6.336956677089462*v3*w2*z + 1.8196275948475404*lnbara*v3*w2*z - 
     3.5946630672038005*u*v3*w2*z - 1.0856448829578143*lnbara*u*v3*w2*z + 
     0.6616579679935063*u2*v3*w2*z + 0.18951332857164516*lnbara*u2*v3*
      w2*z - 0.7906870068555788*v4*w2*z - 0.2013783664993635*lnbara*v4*
      w2*z + 0.3281593662661477*u*v4*w2*z + 0.09992430356434646*lnbara*u*
      v4*w2*z + 0.2301351338116376*v5*w2*z + 0.05887745268544395*lnbara*
      v5*w2*z + 1.7170463437119636*w3*z + 0.4329554475790071*lnbara*w3*
      z + 8.335636464746663*u*w3*z + 2.843556800957142*lnbara*u*w3*z + 
     1.7021286302045417*u2*w3*z + 0.46172402438324356*lnbara*u2*w3*z + 
     1.5576572952608156*u3*w3*z + 0.465804859856524*lnbara*u3*w3*z - 
     0.01399123854239156*u4*w3*z - 0.006838625664216695*lnbara*u4*w3*z - 
     11.553644836317881*v*w3*z - 3.9890155563258416*lnbara*v*w3*z - 
     0.5200656204166001*u*v*w3*z - 0.04824677725141553*lnbara*u*v*w3*z - 
     2.7556939387551567*u2*v*w3*z - 0.8803435778848646*lnbara*u2*v*w3*z + 
     0.0688751348615315*u3*v*w3*z + 0.029360220026995808*lnbara*u3*v*w3*
      z - 6.2616857203946035*v2*w3*z - 1.8196275948475404*lnbara*v2*w3*
      z + 3.5848494417836383*u*v2*w3*z + 1.0856448829578143*lnbara*u*v2*
      w3*z - 0.660951971257608*u2*v2*w3*z - 0.18951332857164516*lnbara*
      u2*v2*w3*z - 0.006605505610793891*v3*w3*z + 
     0.00038509593985133696*u*v3*w3*z - 0.380500721540267*v4*w3*z - 
     0.10823518328690307*lnbara*v4*w3*z - 2.581767811748015*w4*z - 
     0.7450820516264817*lnbara*w4*z - 4.408138300869153*u*w4*z - 
     1.450672032069528*lnbara*u*w4*z - 1.1428845474172378*u2*w4*z - 
     0.3173310228787825*lnbara*u2*w4*z - 0.2223252216914388*u3*w4*z - 
     0.06514335645328273*lnbara*u3*w4*z + 10.200182207413253*v*w4*z + 
     3.28316404172915*lnbara*v*w4*z - 1.034655985722088*u*v*w4*z - 
     0.336346527741606*lnbara*u*v*w4*z + 0.8981582264276017*u2*v*w4*z + 
     0.26783993094333697*lnbara*u2*v*w4*z + 0.7756766725864659*v2*w4*z + 
     0.2013783664993635*lnbara*v2*w4*z - 0.3272951320069847*u*v2*w4*z - 
     0.09992430356434646*lnbara*u*v2*w4*z + 0.38164842058355447*v3*w4*z + 
     0.10823518328690307*lnbara*v3*w4*z + 0.9506587243582331*w5*z + 
     0.2748859330079916*lnbara*w5*z + 1.84547571556521*u*w5*z + 
     0.5730994907821886*lnbara*u*w5*z + 0.11799273363683711*u2*w5*z + 
     0.03151257404486913*lnbara*u2*w5*z - 3.234078310073586*v*w5*z - 
     1.0179245136226938*lnbara*v*w5*z + 0.01575761553644318*u*v*w5*z + 
     0.010275898356323341*lnbara*u*v*w5*z - 0.22871819549961922*v2*w5*z - 
     0.05887745268544395*lnbara*v2*w5*z - 0.3370863430138181*w6*z - 
     0.09609081997302192*lnbara*w6*z - 0.25711383486763884*u*w6*z - 
     0.07776386003863338*lnbara*u*w6*z + 0.5245800234274332*v*w6*z + 
     0.15852336677018297*lnbara*v*w6*z + 0.039205434782531995*w7*z + 
     0.01108690204816448*lnbara*w7*z + 0.9809537469447522*x*z - 
     0.5522472005141469*u*x*z + 0.3313625873896489*u2*x*z - 
     0.162591133426937*u3*x*z + 0.056971794565663605*u4*x*z - 
     0.012364843466299529*u5*x*z + 0.0012376963517727734*u6*x*z - 
     0.5171174600050477*v*x*z + 0.2576868260776136*u*v*x*z - 
     0.1113550912072001*u2*v*x*z + 0.03580809539433375*u3*v*x*z - 
     0.007320613539132591*u4*v*x*z + 0.0007017792893363362*u5*v*x*z - 
     10.763863509478952*v2*x*z - 3.7979978404690304*lnbara*v2*x*z + 
     14.408707884242522*u*v2*x*z + 4.851483174726708*lnbara*u*v2*x*z - 
     11.35352724075799*u2*v2*x*z - 3.638468221673086*lnbara*u2*v2*x*z + 
     3.595212268118998*u3*v2*x*z + 1.124626001903981*lnbara*u3*v2*x*z - 
     0.637464961753253*u4*v2*x*z - 0.19025408565816512*lnbara*u4*v2*x*z + 
     4.570769251234671*v3*x*z + 1.7115875434794767*lnbara*v3*x*z - 
     1.485347767785549*u*v3*x*z - 0.6728643192060604*lnbara*u*v3*x*z + 
     2.240193096445929*u2*v3*x*z + 0.7375497739915498*lnbara*u2*v3*x*z - 
     0.07656531816033134*u3*v3*x*z - 0.03634902416016519*lnbara*u3*v3*x*
      z - 6.0865303594656925*v4*x*z - 1.9510966301046395*lnbara*v4*x*z + 
     2.583274807875346*u*v4*x*z + 0.814048966327527*lnbara*u*v4*x*z - 
     1.0214921534358323*u2*v4*x*z - 0.3025359736703866*lnbara*u2*v4*x*z + 
     1.5716677873215597*v5*x*z + 0.5030836809260285*lnbara*v5*x*z - 
     0.15605883003169002*u*v5*x*z - 0.05503807613937338*lnbara*u*v5*x*z - 
     0.2866167180827836*v6*x*z - 0.08634183804336616*lnbara*v6*x*z - 
     0.5171174600050477*w*x*z + 0.2576868260776136*u*w*x*z - 
     0.1113550912072001*u2*w*x*z + 0.03580809539433375*u3*w*x*z - 
     0.007320613539132591*u4*w*x*z + 0.0007017792893363362*u5*w*x*z + 
     22.357006209034473*v*w*x*z + 7.595995680938061*lnbara*v*w*x*z - 
     29.1182808068324*u*v*w*x*z - 9.702966349453416*lnbara*u*v*w*x*z + 
     22.793147607056017*u2*v*w*x*z + 7.276936443346172*lnbara*u2*v*w*x*z - 
     7.206676437932214*u3*v*w*x*z - 2.249252003807962*lnbara*u3*v*w*x*z + 
     1.2764023824610562*u4*v*w*x*z + 0.38050817131633025*lnbara*u4*v*w*x*
      z - 4.808822770349789*v2*w*x*z - 1.7115875434794767*lnbara*v2*w*x*z + 
     1.5426830809278815*u*v2*w*x*z + 0.6728643192060604*lnbara*u*v2*w*x*z - 
     2.2498626331415124*u2*v2*w*x*z - 0.7375497739915498*lnbara*u2*v2*w*x*
      z + 0.07737638458711994*u3*v2*w*x*z + 0.03634902416016519*lnbara*u3*
      v2*w*x*z + 14.388317561563154*v3*w*x*z + 4.420309235697655*lnbara*v3*
      w*x*z - 10.915203494539247*u*v3*w*x*z - 3.2792181994652556*lnbara*u*
      v3*w*x*z + 2.4240248630265353*u2*v3*w*x*z + 
     0.7115869901073909*lnbara*u2*v3*w*x*z - 2.2689406137217927*v4*w*x*z - 
     0.7108966463767397*lnbara*v4*w*x*z + 1.0002685704936713*u*v4*w*x*z + 
     0.30624466399171646*lnbara*u*v4*w*x*z + 0.6108819098365138*v5*w*x*z + 
     0.1761800865537642*lnbara*v5*w*x*z - 10.763863509478952*w2*x*z - 
     3.7979978404690304*lnbara*w2*x*z + 14.408707884242522*u*w2*x*z + 
     4.851483174726708*lnbara*u*w2*x*z - 11.35352724075799*u2*w2*x*z - 
     3.638468221673086*lnbara*u2*w2*x*z + 3.595212268118998*u3*w2*x*z + 
     1.124626001903981*lnbara*u3*w2*x*z - 0.637464961753253*u4*w2*x*z - 
     0.19025408565816512*lnbara*u4*w2*x*z - 4.808822770349789*v*w2*x*z - 
     1.7115875434794767*lnbara*v*w2*x*z + 1.5426830809278815*u*v*w2*x*z + 
     0.6728643192060604*lnbara*u*v*w2*x*z - 2.2498626331415124*u2*v*w2*x*
      z - 0.7375497739915498*lnbara*u2*v*w2*x*z + 
     0.07737638458711994*u3*v*w2*x*z + 0.03634902416016519*lnbara*u3*v*w2*
      x*z - 16.42034107834173*v2*w2*x*z - 4.93842521118603*lnbara*v2*w2*x*
      z + 16.637816190951657*u*v2*w2*x*z + 4.930338466275457*lnbara*u*v2*
      w2*x*z - 2.803109839150323*u2*v2*w2*x*z - 
     0.8181020328740087*lnbara*u2*v2*w2*x*z + 0.6761564834039746*v3*w2*x*
      z + 0.2078129654507111*lnbara*v3*w2*x*z - 0.8428720597864838*u*v3*
      w2*x*z - 0.25120658785234307*lnbara*u*v3*w2*x*z - 
     0.9508002022690504*v4*w2*x*z - 0.2673286182958272*lnbara*v4*w2*x*z + 
     4.570769251234671*w3*x*z + 1.7115875434794767*lnbara*w3*x*z - 
     1.485347767785549*u*w3*x*z - 0.6728643192060604*lnbara*u*w3*x*z + 
     2.240193096445929*u2*w3*x*z + 0.7375497739915498*lnbara*u2*w3*x*z - 
     0.07656531816033134*u3*w3*x*z - 0.03634902416016519*lnbara*u3*w3*x*
      z + 14.388317561563154*v*w3*x*z + 4.420309235697655*lnbara*v*w3*x*z - 
     10.915203494539247*u*v*w3*x*z - 3.2792181994652556*lnbara*u*v*w3*x*z + 
     2.4240248630265353*u2*v*w3*x*z + 0.7115869901073909*lnbara*u2*v*w3*x*
      z + 0.6761564834039746*v2*w3*x*z + 0.2078129654507111*lnbara*v2*w3*
      x*z - 0.8428720597864838*u*v2*w3*x*z - 0.25120658785234307*lnbara*u*
      v2*w3*x*z + 1.2574727801717278*v3*w3*x*z + 
     0.3549807395708584*lnbara*v3*w3*x*z - 6.0865303594656925*w4*x*z - 
     1.9510966301046395*lnbara*w4*x*z + 2.583274807875346*u*w4*x*z + 
     0.814048966327527*lnbara*u*w4*x*z - 1.0214921534358323*u2*w4*x*z - 
     0.3025359736703866*lnbara*u2*w4*x*z - 2.2689406137217927*v*w4*x*z - 
     0.7108966463767397*lnbara*v*w4*x*z + 1.0002685704936713*u*v*w4*x*z + 
     0.30624466399171646*lnbara*u*v*w4*x*z - 0.9508002022690504*v2*w4*x*
      z - 0.2673286182958272*lnbara*v2*w4*x*z + 1.5716677873215597*w5*x*
      z + 0.5030836809260285*lnbara*w5*x*z - 0.15605883003169002*u*w5*x*z - 
     0.05503807613937338*lnbara*u*w5*x*z + 0.6108819098365138*v*w5*x*z + 
     0.1761800865537642*lnbara*v*w5*x*z - 0.2866167180827836*w6*x*z - 
     0.08634183804336616*lnbara*w6*x*z - 0.9004176968697725*x2*z + 
     0.3313625873896489*u*x2*z - 0.12887042405432217*u2*x2*z + 
     0.03949367079133293*u3*x2*z - 0.007867552944290607*u4*x2*z + 
     0.0007424815539584533*u5*x2*z + 16.840740404671994*v*x2*z + 
     5.75015242423565*lnbara*v*x2*z - 7.4944534531230955*u*v*x2*z - 
     2.569763477061043*lnbara*u*v*x2*z + 10.92704206771195*u2*v*x2*z + 
     3.4712775032258807*lnbara*u2*v*x2*z - 2.7345557618171856*u3*v*x2*z - 
     0.8563693554764021*lnbara*u3*v*x2*z + 0.5508222899910821*u4*v*x2*z + 
     0.16402829156973656*lnbara*u4*v*x2*z - 2.5057639727820775*v2*x2*z - 
     1.0258918950424711*lnbara*v2*x2*z - 4.04603639452907*u*v2*x2*z - 
     1.3032259701999138*lnbara*u*v2*x2*z - 0.5438356537989508*u2*v2*x2*
      z - 0.1818291572693978*lnbara*u2*v2*x2*z - 
     0.17072856838087597*u3*v2*x2*z - 0.04904808595029047*lnbara*u3*v2*
      x2*z + 9.909152497593267*v3*x2*z + 3.0912384401060025*lnbara*v3*x2*
      z - 5.305139799956746*u*v3*x2*z - 1.5393343490108804*lnbara*u*v3*x2*
      z + 1.5675237969406215*u2*v3*x2*z + 0.4579464223436356*lnbara*u2*
      v3*x2*z - 1.4226442649834423*v4*x2*z - 0.4703228534910837*lnbara*
      v4*x2*z + 0.2656466906715195*u*v4*x2*z + 0.08058867723285898*lnbara*
      u*v4*x2*z + 0.4211542429001266*v5*x2*z + 0.12424611305742275*lnbara*
      v5*x2*z - 16.220273175775272*w*x2*z - 5.75015242423565*lnbara*w*x2*
      z + 7.283613291269799*u*w*x2*z + 2.569763477061043*lnbara*u*w*x2*z - 
     10.868438804341743*u2*w*x2*z - 3.4712775032258807*lnbara*u2*w*x2*z + 
     2.723658999110921*u3*w*x2*z + 0.8563693554764021*lnbara*u3*w*x2*z - 
     0.5498436699327358*u4*w*x2*z - 0.16402829156973656*lnbara*u4*w*x2*
      z - 4.808822770349789*v*w*x2*z - 1.7115875434794767*lnbara*v*w*x2*z + 
     1.5426830809278815*u*v*w*x2*z + 0.6728643192060604*lnbara*u*v*w*x2*z - 
     2.2498626331415124*u2*v*w*x2*z - 0.7375497739915498*lnbara*u2*v*w*x2*
      z + 0.07737638458711994*u3*v*w*x2*z + 0.03634902416016519*lnbara*u3*
      v*w*x2*z - 20.346437220092437*v2*w*x2*z - 6.08173812414508*lnbara*
      v2*w*x2*z + 16.365226923917742*u*v2*w*x2*z + 
     4.905952869158358*lnbara*u*v2*w*x2*z - 3.2577140116251813*u2*v2*w*
      x2*z - 0.9439400410182851*lnbara*u2*v2*w*x2*z + 
     1.4784816196761565*v3*w*x2*z + 0.43230388098155775*lnbara*v3*w*x2*
      z - 0.6938884585226485*u*v3*w*x2*z - 0.21809446900966986*lnbara*u*v3*
      w*x2*z - 1.267405871830795*v4*w*x2*z - 0.35480949331667067*lnbara*
      v4*w*x2*z + 6.988719566598391*w2*x2*z + 2.737479438521948*lnbara*
      w2*x2*z + 2.583201962578535*u*w2*x2*z + 0.6303616509938534*lnbara*u*
      w2*x2*z + 2.78012177163788*u2*w2*x2*z + 0.9193789312609476*lnbara*
      u2*w2*x2*z + 0.0944959815922243*u3*w2*x2*z + 
     0.012699061790125279*lnbara*u3*w2*x2*z + 21.55337156258267*v*w2*x2*
      z + 6.548245980022291*lnbara*v*w2*x2*z - 14.057135775486067*u*v*w2*
      x2*z - 4.266687153487787*lnbara*u*v*w2*x2*z + 
     3.4982098333445135*u2*v*w2*x2*z + 1.0191704113395068*lnbara*u2*v*w2*
      x2*z - 0.40857845580650054*v2*w2*x2*z - 0.08307962011998349*lnbara*
      v2*w2*x2*z - 0.5491803784646733*u*v2*w2*x2*z - 
     0.17704276038055036*lnbara*u*v2*w2*x2*z + 2.1080028169579204*v3*w2*
      x2*z + 0.5953996603124057*lnbara*v3*w2*x2*z - 
     10.997725885446338*w3*x2*z - 3.557746295983214*lnbara*w3*x2*z + 
     2.9792615221240184*u*w3*x2*z + 0.9000686333403095*lnbara*u*w3*x2*z - 
     1.8066447210612555*u2*w3*x2*z - 0.5331767926648573*lnbara*u2*w3*x2*
      z - 2.1068706790103*v*w3*x2*z - 0.6955343577421946*lnbara*v*w3*x2*
      z + 0.8761362502114731*u*v*w3*x2*z + 0.296376278878668*lnbara*u*v*w3*
      x2*z - 1.7437824185418067*v2*w3*x2*z - 0.4914395306435913*lnbara*
      v2*w3*x2*z + 2.4334094721170496*w4*x2*z + 
     0.816632950371704*lnbara*w4*x2*z + 0.10307841593891565*u*w4*x2*z + 
     0.018172273278693238*lnbara*u*w4*x2*z + 1.0094259951122324*v*w4*x2*
      z + 0.2874207900901255*lnbara*v*w4*x2*z - 0.5247428948621341*w5*x2*
      z - 0.16081753949969196*lnbara*w5*x2*z + 0.6858555690183518*x3*z - 
     0.162591133426937*u*x3*z + 0.03949367079133293*u2*x3*z - 
     0.006882314713446356*u3*x3*z + 0.0005960914163116732*u4*x3*z - 
     11.553644836317881*v*x3*z - 3.9890155563258416*lnbara*v*x3*z - 
     0.5200656204166001*u*v*x3*z - 0.04824677725141553*lnbara*u*v*x3*z - 
     2.7556939387551567*u2*v*x3*z - 0.8803435778848646*lnbara*u2*v*x3*z + 
     0.0688751348615315*u3*v*x3*z + 0.029360220026995808*lnbara*u3*v*x3*
      z - 4.961706764789829*v2*x3*z - 1.3124617908295524*lnbara*v2*x3*z + 
     8.457223146401986*u*v2*x3*z + 2.5565136487937905*lnbara*u*v2*x3*z - 
     0.9782558475006098*u2*v2*x3*z - 0.28088060701825807*lnbara*u2*v2*
      x3*z - 1.6389294857674492*v3*x3*z - 0.5238484235184111*lnbara*v3*
      x3*z - 0.30529725405798797*u*v3*x3*z - 0.10319808585533102*lnbara*u*
      v3*x3*z - 0.31318215309396386*v4*x3*z - 0.07872508228066824*lnbara*
      v4*x3*z + 11.250037884026336*w*x3*z + 3.9890155563258416*lnbara*w*
      x3*z + 0.5861690638750913*u*w*x3*z + 0.04824677725141553*lnbara*u*w*
      x3*z + 2.7450669338987366*u2*w*x3*z + 0.8803435778848646*lnbara*u2*
      w*x3*z - 0.06800667812701873*u3*w*x3*z - 0.029360220026995808*lnbara*
      u3*w*x3*z + 14.388317561563154*v*w*x3*z + 4.420309235697655*lnbara*v*
      w*x3*z - 10.915203494539247*u*v*w*x3*z - 3.2792181994652556*lnbara*u*
      v*w*x3*z + 2.4240248630265353*u2*v*w*x3*z + 
     0.7115869901073909*lnbara*u2*v*w*x3*z + 1.4784816196761565*v2*w*x3*
      z + 0.43230388098155775*lnbara*v2*w*x3*z - 0.6938884585226485*u*v2*w*
      x3*z - 0.21809446900966986*lnbara*u*v2*w*x3*z + 
     1.5073534267839392*v3*w*x3*z + 0.4300715944036431*lnbara*v3*w*x3*z - 
     9.326716278242827*w2*x3*z - 3.107847444868102*lnbara*w2*x3*z + 
     2.4434643394917717*u*w2*x3*z + 0.722704550671465*lnbara*u*w2*x3*z - 
     1.4446667880342972*u2*w2*x3*z - 0.43070638308913284*lnbara*u2*w2*
      x3*z - 2.1068706790103*v*w2*x3*z - 0.6955343577421946*lnbara*v*w2*
      x3*z + 0.8761362502114731*u*v*w2*x3*z + 0.296376278878668*lnbara*u*v*
      w2*x3*z - 1.7364430725838955*v2*w2*x3*z - 
     0.4913145532522245*lnbara*v2*w2*x3*z + 2.2465725155547824*w3*x3*z + 
     0.7870789002790479*lnbara*w3*x3*z + 0.1244773390895362*u*w3*x3*z + 
     0.024916275986332885*lnbara*u*w3*x3*z + 1.0840027554445029*v*w3*x3*
      z + 0.30837662850281694*lnbara*v*w3*x3*z - 0.5397225167390112*w4*x3*
      z - 0.16840858737356737*lnbara*w4*x3*z - 0.386061737342387*x4*z + 
     0.056971794565663605*u*x4*z - 0.007867552944290607*u2*x4*z + 
     0.0005960914163116732*u3*x4*z + 10.200182207413253*v*x4*z + 
     3.28316404172915*lnbara*v*x4*z - 1.034655985722088*u*v*x4*z - 
     0.336346527741606*lnbara*u*v*x4*z + 0.8981582264276017*u2*v*x4*z + 
     0.26783993094333697*lnbara*u2*v*x4*z + 0.01586781553997586*v2*x4*z - 
     0.06352181102996676*lnbara*v2*x4*z - 1.0816608287503775*u*v2*x4*z - 
     0.32438223975151587*lnbara*u*v2*x4*z + 0.8263404688000965*v3*x4*z + 
     0.2438589036761638*lnbara*v3*x4*z - 10.094036110016523*w*x4*z - 
     3.28316404172915*lnbara*w*x4*z + 1.0213274392385536*u*w*x4*z + 
     0.336346527741606*lnbara*u*w*x4*z - 0.8972167925745769*u2*w*x4*z - 
     0.26783993094333697*lnbara*u2*w*x4*z - 2.2689406137217927*v*w*x4*z - 
     0.7108966463767397*lnbara*v*w*x4*z + 1.0002685704936713*u*v*w*x4*z + 
     0.30624466399171646*lnbara*u*v*w*x4*z - 1.267405871830795*v2*w*x4*z - 
     0.35480949331667067*lnbara*v2*w*x4*z + 2.2331993759180806*w2*x4*z + 
     0.7744184574067065*lnbara*w2*x4*z + 0.08267922567088919*u*w2*x4*z + 
     0.01813757575979942*lnbara*u*w2*x4*z + 1.0094259951122324*v*w2*x4*
      z + 0.2874207900901255*lnbara*v*w2*x4*z - 0.5665614064319183*w3*x4*
      z - 0.1764702004496186*lnbara*w3*x4*z + 0.1475870262830398*x5*z - 
     0.012364843466299529*u*x5*z + 0.0007424815539584533*u2*x5*z - 
     3.234078310073586*v*x5*z - 1.0179245136226938*lnbara*v*x5*z + 
     0.01575761553644318*u*v*x5*z + 0.010275898356323341*lnbara*u*v*x5*z - 
     0.14867177025989595*v2*x5*z - 0.030113006529747105*lnbara*v2*x5*z + 
     3.2110778233847816*w*x5*z + 1.0179245136226938*lnbara*w*x5*z - 
     0.01449065568595044*u*w*x5*z - 0.010275898356323341*lnbara*u*w*x5*z + 
     0.6108819098365138*v*w*x5*z + 0.1761800865537642*lnbara*v*w*x5*z - 
     0.46033723691317213*w2*x5*z - 0.14606708002401708*lnbara*w2*x5*z - 
     0.0339914082143444*x6*z + 0.0012376963517727734*u*x6*z + 
     0.5245800234274332*v*x6*z + 0.15852336677018297*lnbara*v*x6*z - 
     0.5222803748590084*w*x6*z - 0.15852336677018297*lnbara*w*x6*z + 
     0.0035512018991495737*x7*z + 0.9809537469447522*y*z - 
     0.5171174600050477*u*y*z + 0.29918371952782097*u2*y*z - 
     0.14397058356518277*u3*y*z + 0.04988531635163183*u4*y*z - 
     0.01075135761798518*u5*y*z + 0.0010711545725659018*u6*y*z - 
     7.798871853605498*v*y*z - 2.734558445137702*lnbara*v*y*z - 
     5.604833095618447*u*v*y*z - 1.7772673145259574*lnbara*u*v*y*z - 
     3.826201875320261*u2*v*y*z - 1.1575775618246809*lnbara*u2*v*y*z - 
     1.262771091795936*u3*v*y*z - 0.36117019327441874*lnbara*u3*v*y*z - 
     0.26452915782323383*u4*v*y*z - 0.07250424934435563*lnbara*u4*v*y*z - 
     0.0196913141648136*u5*v*y*z - 0.0052376950009461855*lnbara*u5*v*y*z + 
     8.371127296661705*v2*y*z + 3.007965370948085*lnbara*v2*y*z + 
     8.142878738630774*u*v2*y*z + 2.570302308603461*lnbara*u*v2*y*z + 
     4.372397515146079*u2*v2*y*z + 1.303884138860003*lnbara*u2*v2*y*z + 
     1.089630311725629*u3*v2*y*z + 0.306972893835514*lnbara*u3*v2*y*z + 
     0.11045275181230746*u4*v2*y*z + 0.029980593819149505*lnbara*u4*v2*y*
      z - 14.456656314267676*v3*y*z - 4.8057115258275305*lnbara*v3*y*z - 
     3.6907580059131853*u*v3*y*z - 1.0737089522063064*lnbara*u*v3*y*z - 
     3.5123911436649746*u2*v3*y*z - 1.041566396111375*lnbara*u2*v3*y*z - 
     0.10728353612006211*u3*v3*y*z - 0.025365208983699924*lnbara*u3*v3*y*
      z + 8.072482666125286*v4*y*z + 2.6276787546704745*lnbara*v4*y*z + 
     2.0086611123781046*u*v4*y*z + 0.5668452509261017*lnbara*u*v4*y*z + 
     0.590076818257424*u2*v4*y*z + 0.17202385787388583*lnbara*u2*v4*y*z - 
     3.248331013993099*v5*y*z - 1.0069202481453856*lnbara*v5*y*z - 
     0.23401253231882208*u*v5*y*z - 0.06359410612179822*lnbara*u*v5*y*z + 
     0.47407842963740254*v6*y*z + 0.14311605655840995*lnbara*v6*y*z + 
     6.729507193086304*w*y*z + 2.734558445137702*lnbara*w*y*z + 
     6.093431672716988*u*w*y*z + 1.7772673145259574*lnbara*u*w*y*z + 
     3.6244323005177024*u2*w*y*z + 1.1575775618246809*lnbara*u2*w*y*z + 
     1.326032325221745*u3*w*y*z + 0.36117019327441874*lnbara*u3*w*y*z + 
     0.2517914740480953*u4*w*y*z + 0.07250424934435563*lnbara*u4*w*y*z + 
     0.020900424621812116*u5*w*y*z + 0.0052376950009461855*lnbara*u5*w*y*
      z + 0.2576868260776136*v*w*y*z - 0.08549175984210892*u*v*w*y*z + 
     0.023341335158146424*u2*v*w*y*z - 0.004288309525390823*u3*v*w*y*z + 
     0.0003820180212080083*u4*v*w*y*z + 14.923465303571382*v2*w*y*z + 
     4.700428217339612*lnbara*v2*w*y*z - 8.636182740657008*u*v2*w*y*z - 
     2.845063377679382*lnbara*u*v2*w*y*z + 2.0292029954219255*u2*v2*w*y*
      z + 0.6312081178376158*lnbara*u2*v2*w*y*z - 
     0.5873630368597165*u3*v2*w*y*z - 0.17392267685688417*lnbara*u3*v2*w*
      y*z - 2.3689366351444274*v3*w*y*z - 0.726506814968872*lnbara*v3*w*y*
      z + 2.469819197461433*u*v3*w*y*z + 0.8025320427989695*lnbara*u*v3*w*y*
      z + 0.18981304783567338*u2*v3*w*y*z + 0.049895924897055216*lnbara*u2*
      v3*w*y*z + 2.495271539467373*v4*w*y*z + 0.7051754521652789*lnbara*v4*
      w*y*z - 0.9763863300273927*u*v4*w*y*z - 0.2916555967133236*lnbara*u*
      v4*w*y*z - 0.2133826221782255*v5*w*y*z - 0.0583092465218906*lnbara*
      v5*w*y*z - 7.740580989744235*w2*y*z - 3.007965370948085*lnbara*w2*y*
      z - 8.34831676538791*u*w2*y*z - 2.570302308603461*lnbara*u*w2*y*z - 
     4.3167789899540825*u2*w2*y*z - 1.303884138860003*lnbara*u2*w2*y*z - 
     1.0998046234447676*u3*w2*y*z - 0.306972893835514*lnbara*u3*w2*y*z - 
     0.10954859095754421*u4*w2*y*z - 0.029980593819149505*lnbara*u4*w2*y*
      z - 15.142507034031205*v*w2*y*z - 4.700428217339612*lnbara*v*w2*y*z + 
     8.684236515504576*u*v*w2*y*z + 2.845063377679382*lnbara*u*v*w2*y*z - 
     2.0369169970657963*u2*v*w2*y*z - 0.6312081178376158*lnbara*u2*v*w2*y*
      z + 0.5879915453536685*u3*v*w2*y*z + 0.17392267685688417*lnbara*u3*v*
      w2*y*z + 0.03069034292530752*v2*w2*y*z - 0.003995614840649483*u*v2*
      w2*y*z + 0.0002863764888716214*u2*v2*w2*y*z - 
     6.026774148018949*v3*w2*y*z - 1.7632715691466476*lnbara*v3*w2*y*z + 
     0.731117931401122*u*v3*w2*y*z + 0.21288835092447814*lnbara*u*v3*w2*y*
      z + 0.663692907110261*v4*w2*y*z + 0.19636416180684646*lnbara*v4*w2*
      y*z + 14.150094597275556*w3*y*z + 4.8057115258275305*lnbara*w3*y*z + 
     3.7558333567491418*u*w3*y*z + 1.0737089522063064*lnbara*u*w3*y*z + 
     3.5021103597603167*u2*w3*y*z + 1.041566396111375*lnbara*u2*w3*y*z + 
     0.1081140594632303*u3*w3*y*z + 0.025365208983699924*lnbara*u3*w3*y*
      z + 2.4387387085229473*v*w3*y*z + 0.726506814968872*lnbara*v*w3*y*z - 
     2.478798620600135*u*v*w3*y*z - 0.8025320427989695*lnbara*u*v*w3*y*z - 
     0.1891735970499443*u2*v*w3*y*z - 0.049895924897055216*lnbara*u2*v*w3*
      y*z + 6.015319832123562*v2*w3*y*z + 1.7632715691466476*lnbara*v2*w3*
      y*z - 0.7304529490550218*u*v2*w3*y*z - 0.21288835092447814*lnbara*u*
      v2*w3*y*z + 0.00047310410784989136*v3*w3*y*z - 
     7.96562555520799*w4*y*z - 2.6276787546704745*lnbara*w4*y*z - 
     2.0218463214313056*u*w4*y*z - 0.5668452509261017*lnbara*u*w4*y*z - 
     0.5891572387478903*u2*w4*y*z - 0.17202385787388583*lnbara*u2*w4*y*
      z - 2.5094652412675753*v*w4*y*z - 0.7051754521652789*lnbara*v*w4*y*
      z + 0.977194806205722*u*v*w4*y*z + 0.2916555967133236*lnbara*u*v*w4*y*
      z - 0.6626623249676024*v2*w4*y*z - 0.19636416180684646*lnbara*v2*w4*
      y*z + 3.2252148129088143*w5*y*z + 1.0069202481453856*lnbara*w5*y*z + 
     0.23526924787765785*u*w5*y*z + 0.06359410612179822*lnbara*u*w5*y*z + 
     0.2147385756550609*v*w5*y*z + 0.0583092465218906*lnbara*v*w5*y*z - 
     0.4717695787130639*w6*y*z - 0.14311605655840995*lnbara*w6*y*z - 
     0.6975695736651812*x*y*z + 0.2576868260776136*u*x*y*z - 
     0.09948507064609574*u2*x*y*z + 0.030295348064157374*u3*x*y*z - 
     0.006007932944401952*u4*x*y*z + 0.000565180561156404*u5*x*y*z + 
     0.2576868260776136*v*x*y*z - 0.08549175984210892*u*v*x*y*z + 
     0.023341335158146424*u2*v*x*y*z - 0.004288309525390823*u3*v*x*y*z + 
     0.0003820180212080083*u4*v*x*y*z + 10.777349982686436*v2*x*y*z + 
     3.494158013231508*lnbara*v2*x*y*z - 14.680040084881753*u*v2*x*y*z - 
     4.586055804009709*lnbara*u*v2*x*y*z + 2.978128785284849*u2*v2*x*y*z + 
     0.9528593416680298*lnbara*u2*v2*x*y*z - 0.7405676776348092*u3*v2*x*y*
      z - 0.21821356263065994*lnbara*u3*v2*x*y*z - 
     0.09200933391270437*v3*x*y*z - 0.14480531475704334*lnbara*v3*x*y*z + 
     3.724170161382556*u*v3*x*y*z + 1.1721828559253105*lnbara*u*v3*x*y*z + 
     0.1296884569931269*u2*v3*x*y*z + 0.020514943705544776*lnbara*u2*v3*x*
      y*z + 1.9788571739028187*v4*x*y*z + 0.5910023418671081*lnbara*v4*x*y*
      z - 1.497715134559773*u*v4*x*y*z - 0.43755320330470776*lnbara*u*v4*x*
      y*z - 0.09961325470566673*v5*x*y*z - 0.03303181019568392*lnbara*v5*x*
      y*z + 0.2576868260776136*w*x*y*z - 0.08549175984210892*u*w*x*y*z + 
     0.023341335158146424*u2*w*x*y*z - 0.004288309525390823*u3*w*x*y*z + 
     0.0003820180212080083*u4*w*x*y*z - 21.839161866507176*v*w*x*y*z - 
     6.988316026463016*lnbara*v*w*x*y*z + 29.42708255896211*u*v*w*x*y*z + 
     9.172111608019417*lnbara*u*v*w*x*y*z - 5.96736483518329*u2*v*w*x*y*z - 
     1.9057186833360595*lnbara*u2*v*w*x*y*z + 1.4820558313910297*u3*v*w*x*y*
      z + 0.4364271252613199*lnbara*u3*v*w*x*y*z + 
     0.14564601713500816*v2*w*x*y*z + 0.14480531475704334*lnbara*v2*w*x*y*
      z - 3.731845090721665*u*v2*w*x*y*z - 1.1721828559253105*lnbara*u*v2*w*
      x*y*z - 0.12911307483747164*u2*v2*w*x*y*z - 
     0.020514943705544776*lnbara*u2*v2*w*x*y*z - 8.881204021839554*v3*w*x*
      y*z - 2.6069786638032433*lnbara*v3*w*x*y*z + 
     3.3873280227068086*u*v3*w*x*y*z + 0.9769151192502976*lnbara*u*v3*w*x*y*
      z + 0.6318226520207785*v4*w*x*y*z + 0.19318619266094705*lnbara*v4*w*x*
      y*z + 10.777349982686436*w2*x*y*z + 3.494158013231508*lnbara*w2*x*y*
      z - 14.680040084881753*u*w2*x*y*z - 4.586055804009709*lnbara*u*w2*x*y*
      z + 2.978128785284849*u2*w2*x*y*z + 0.9528593416680298*lnbara*u2*w2*
      x*y*z - 0.7405676776348092*u3*w2*x*y*z - 0.21821356263065994*lnbara*
      u3*w2*x*y*z + 0.14564601713500816*v*w2*x*y*z + 
     0.14480531475704334*lnbara*v*w2*x*y*z - 3.731845090721665*u*v*w2*x*y*
      z - 1.1721828559253105*lnbara*u*v*w2*x*y*z - 
     0.12911307483747164*u2*v*w2*x*y*z - 0.020514943705544776*lnbara*u2*v*
      w2*x*y*z + 13.78024088853395*v2*w2*x*y*z + 
     4.03195264387227*lnbara*v2*w2*x*y*z - 3.777642640073415*u*v2*w2*x*y*
      z - 1.0787238318911796*lnbara*u*v2*w2*x*y*z - 
     0.5309427518886998*v3*w2*x*y*z - 0.16015438246526315*lnbara*v3*w2*x*
      y*z - 0.09200933391270437*w3*x*y*z - 0.14480531475704334*lnbara*w3*x*
      y*z + 3.724170161382556*u*w3*x*y*z + 1.1721828559253105*lnbara*u*w3*x*
      y*z + 0.1296884569931269*u2*w3*x*y*z + 0.020514943705544776*lnbara*
      u2*w3*x*y*z - 8.881204021839554*v*w3*x*y*z - 
     2.6069786638032433*lnbara*v*w3*x*y*z + 3.3873280227068086*u*v*w3*x*y*
      z + 0.9769151192502976*lnbara*u*v*w3*x*y*z - 
     0.5309427518886998*v2*w3*x*y*z - 0.16015438246526315*lnbara*v2*w3*x*
      y*z + 1.9788571739028187*w4*x*y*z + 0.5910023418671081*lnbara*w4*x*y*
      z - 1.497715134559773*u*w4*x*y*z - 0.43755320330470776*lnbara*u*w4*x*
      y*z + 0.6318226520207785*v*w4*x*y*z + 0.19318619266094705*lnbara*v*w4*
      x*y*z - 0.09961325470566673*w5*x*y*z - 0.03303181019568392*lnbara*w5*
      x*y*z + 0.45246456501551086*x2*y*z - 0.1113550912072001*u*x2*y*z + 
     0.027313138910434232*u2*x2*y*z - 0.004771933785158886*u3*x2*y*z + 
     0.0004135240269207944*u4*x2*y*z - 15.142507034031205*v*x2*y*z - 
     4.700428217339612*lnbara*v*x2*y*z + 8.684236515504576*u*v*x2*y*z + 
     2.845063377679382*lnbara*u*v*x2*y*z - 2.0369169970657963*u2*v*x2*y*
      z - 0.6312081178376158*lnbara*u2*v*x2*y*z + 
     0.5879915453536685*u3*v*x2*y*z + 0.17392267685688417*lnbara*u3*v*x2*
      y*z + 0.5638297481368869*v2*x2*y*z + 0.18788462045702756*lnbara*v2*
      x2*y*z + 0.008389015172307996*u*v2*x2*y*z - 
     0.057042047293007624*lnbara*u*v2*x2*y*z - 0.2581203920240232*u2*v2*
      x2*y*z - 0.07638602271158683*lnbara*u2*v2*x2*y*z - 
     6.183783910974188*v3*x2*y*z - 1.7887689679030814*lnbara*v3*x2*y*z + 
     1.617693856062018*u*v3*x2*y*z + 0.4746900012067356*lnbara*u*v3*x2*y*
      z + 0.4680668096248443*v4*x2*y*z + 0.13910950612836231*lnbara*v4*x2*
      y*z + 14.923465303571382*w*x2*y*z + 4.700428217339612*lnbara*w*x2*y*
      z - 8.636182740657008*u*w*x2*y*z - 2.845063377679382*lnbara*u*w*x2*y*
      z + 2.0292029954219255*u2*w*x2*y*z + 0.6312081178376158*lnbara*u2*w*
      x2*y*z - 0.5873630368597165*u3*w*x2*y*z - 0.17392267685688417*lnbara*
      u3*w*x2*y*z + 0.14564601713500816*v*w*x2*y*z + 
     0.14480531475704334*lnbara*v*w*x2*y*z - 3.731845090721665*u*v*w*x2*y*
      z - 1.1721828559253105*lnbara*u*v*w*x2*y*z - 
     0.12911307483747164*u2*v*w*x2*y*z - 0.020514943705544776*lnbara*u2*v*
      w*x2*y*z + 17.091071621436193*v2*w*x2*y*z + 
     4.99601857372239*lnbara*v2*w*x2*y*z - 3.489746635672796*u*v2*w*x2*y*
      z - 1.0013349315415667*lnbara*u*v2*w*x2*y*z - 
     0.9801494429301945*v3*w*x2*y*z - 0.293356436036497*lnbara*v3*w*x2*y*
      z - 0.6330043203094217*w2*x2*y*z - 0.3326899352140709*lnbara*w2*x2*
      y*z + 3.7124326039755955*u*w2*x2*y*z + 1.229224903218318*lnbara*u*w2*
      x2*y*z + 0.3880630632568426*u2*w2*x2*y*z + 
     0.0969009664171316*lnbara*u2*w2*x2*y*z - 14.77406945239147*v*w2*x2*
      y*z - 4.3242801796236945*lnbara*v*w2*x2*y*z + 
     4.000620121386894*u*v*w2*x2*y*z + 1.1582596990589664*lnbara*u*v*w2*
      x2*y*z - 0.2945882654409418*v2*w2*x2*y*z - 
     0.10231487175528582*lnbara*v2*w2*x2*y*z + 3.8502028793420484*w3*x2*
      y*z + 1.1170305738043862*lnbara*w3*x2*y*z - 
     2.127445819261271*u*w3*x2*y*z - 0.6316147687241351*lnbara*u*w3*x2*y*
      z + 0.9014731642677928*v*w3*x2*y*z + 0.28985320078620536*lnbara*v*w3*
      x2*y*z - 0.09314373993962562*w4*x2*y*z - 0.033291399122784844*lnbara*
      w4*x2*y*z - 0.2298476323443832*x3*y*z + 0.03580809539433375*u*x3*y*
      z - 0.005041691635003074*u2*x3*y*z + 0.00038509593985133696*u3*x3*y*
      z + 2.4387387085229473*v*x3*y*z + 0.726506814968872*lnbara*v*x3*y*z - 
     2.478798620600135*u*v*x3*y*z - 0.8025320427989695*lnbara*u*v*x3*y*z - 
     0.1891735970499443*u2*v*x3*y*z - 0.049895924897055216*lnbara*u2*v*x3*
      y*z + 5.676870448679409*v2*x3*y*z + 1.684726260416403*lnbara*v2*x3*
      y*z - 1.182692665484111*u*v2*x3*y*z - 0.32670867639523105*lnbara*u*
      v2*x3*y*z + 0.032891805121808636*v3*x3*y*z - 
     0.002730352711332793*lnbara*v3*x3*y*z - 2.3689366351444274*w*x3*y*z - 
     0.726506814968872*lnbara*w*x3*y*z + 2.469819197461433*u*w*x3*y*z + 
     0.8025320427989695*lnbara*u*w*x3*y*z + 0.18981304783567338*u2*w*x3*y*
      z + 0.049895924897055216*lnbara*u2*w*x3*y*z - 
     8.881204021839554*v*w*x3*y*z - 2.6069786638032433*lnbara*v*w*x3*y*z + 
     3.3873280227068086*u*v*w*x3*y*z + 0.9769151192502976*lnbara*u*v*w*x3*y*
      z - 0.9801494429301945*v2*w*x3*y*z - 0.293356436036497*lnbara*v2*w*
      x3*y*z + 3.1903919237729643*w2*x3*y*z + 0.9222524033868401*lnbara*
      w2*x3*y*z - 2.203724440304821*u*w2*x3*y*z - 
     0.6502064428550666*lnbara*u*w2*x3*y*z + 0.9014731642677928*v*w2*x3*y*
      z + 0.28985320078620536*lnbara*v*w2*x3*y*z + 
     0.04710638371571019*w3*x3*y*z + 0.006233587961624442*lnbara*w3*x3*y*
      z + 0.08203160229248424*x4*y*z - 0.007320613539132591*u*x4*y*z + 
     0.0004507102322422891*u2*x4*y*z - 2.5094652412675753*v*x4*y*z - 
     0.7051754521652789*lnbara*v*x4*y*z + 0.977194806205722*u*v*x4*y*z + 
     0.2916555967133236*lnbara*u*v*x4*y*z - 0.566190372464217*v2*x4*y*z - 
     0.1702185650157698*lnbara*v2*x4*y*z + 2.495271539467373*w*x4*y*z + 
     0.7051754521652789*lnbara*w*x4*y*z - 0.9763863300273927*u*w*x4*y*z - 
     0.2916555967133236*lnbara*u*w*x4*y*z + 0.6318226520207785*v*w*x4*y*z + 
     0.19318619266094705*lnbara*v*w*x4*y*z - 0.06439562415041984*w2*x4*y*
      z - 0.022967627645177263*lnbara*w2*x4*y*z - 
     0.018000480690589078*x5*y*z + 0.0007017792893363362*u*x5*y*z + 
     0.2147385756550609*v*x5*y*z + 0.0583092465218906*lnbara*v*x5*y*z - 
     0.2133826221782255*w*x5*y*z - 0.0583092465218906*lnbara*w*x5*y*z + 
     0.0018148095406981609*x6*y*z - 0.9004176968697725*y2*z + 
     0.29918371952782097*u*y2*z - 0.10942628765049751*u2*y2*z + 
     0.03238162914677262*u3*y2*z - 0.00631630321372675*u4*y2*z + 
     0.0005880488320726199*u5*y2*z + 5.299654762421948*v*y2*z + 
     1.7090990282110639*lnbara*v*y2*z - 1.1643976508622838*u*v*y2*z - 
     0.4284924263303262*lnbara*u*v*y2*z - 1.1346769147760087*u2*v*y2*z - 
     0.329646645071357*lnbara*u2*v*y2*z - 0.3831429064834633*u3*v*y2*z - 
     0.10776709570789808*lnbara*u3*v*y2*z - 0.03979465732217202*u4*v*y2*
      z - 0.010614825463755726*lnbara*u4*v*y2*z - 
     0.044978574732269516*v2*y2*z - 0.06378679623852483*lnbara*v2*y2*z + 
     2.5864306854801558*u*v2*y2*z + 0.8006228366412104*lnbara*u*v2*y2*z + 
     1.1889883290028347*u2*v2*y2*z + 0.3402159951898016*lnbara*u2*v2*y2*
      z + 0.17341769222311584*u3*v2*y2*z + 0.04766748743684271*lnbara*u3*
      v2*y2*z + 2.8467947898195387*v3*y2*z + 0.8710102224533081*lnbara*
      v3*y2*z - 2.2828692800346633*u*v3*y2*z - 0.6947964005893216*lnbara*
      u*v3*y2*z - 0.15198537243470955*u2*v3*y2*z - 
     0.04014533812336501*lnbara*u2*v3*y2*z - 0.32019128431906846*v4*y2*
      z - 0.10788365521657699*lnbara*v4*y2*z + 0.3611306117059756*u*v4*y2*
      z + 0.10812164650482199*lnbara*u*v4*y2*z + 0.11415571194993857*v5*
      y2*z + 0.03339748565253056*lnbara*v5*y2*z - 
     4.647008665663399*w*y2*z - 1.7090990282110639*lnbara*w*y2*z + 
     0.9708296446662518*u*w*y2*z + 0.4284924263303262*lnbara*u*w*y2*z + 
     1.184519706140848*u2*w*y2*z + 0.329646645071357*lnbara*u2*w*y2*z + 
     0.3742822830969903*u3*w*y2*z + 0.10776709570789808*lnbara*u3*w*y2*
      z + 0.04056836531530246*u4*w*y2*z + 0.010614825463755726*lnbara*u4*w*
      y2*z - 0.1113550912072001*v*w*y2*z + 0.023341335158146424*u*v*w*y2*
      z - 0.0036412369193938783*u2*v*w*y2*z + 0.0002913721656218036*u3*v*w*
      y2*z - 13.984210663495318*v2*w*y2*z - 4.27806313688087*lnbara*v2*w*
      y2*z + 1.059736505268818*u*v2*w*y2*z + 0.36692372038764914*lnbara*u*
      v2*w*y2*z - 0.7420694165018765*u2*v2*w*y2*z - 
     0.21620742554342368*lnbara*u2*v2*w*y2*z + 3.420468192644727*v3*w*y2*
      z + 1.0593983869002586*lnbara*v3*w*y2*z + 0.3069925110386899*u*v3*w*
      y2*z + 0.0810097794355468*lnbara*u*v3*w*y2*z - 
     1.1411946582446566*v4*w*y2*z - 0.33097644191339387*lnbara*v4*w*y2*
      z - 0.20624980265509574*w2*y2*z + 0.06378679623852483*lnbara*w2*y2*
      z - 2.5341893643030327*u*w2*y2*z - 0.8006228366412104*lnbara*u*w2*
      y2*z - 1.1971008100465077*u2*w2*y2*z - 0.3402159951898016*lnbara*
      u2*w2*y2*z - 0.17277027032117476*u3*w2*y2*z - 
     0.04766748743684271*lnbara*u3*w2*y2*z + 14.046191130880398*v*w2*y2*
      z + 4.27806313688087*lnbara*v*w2*y2*z - 1.0675924425094043*u*v*w2*
      y2*z - 0.36692372038764914*lnbara*u*v*w2*y2*z + 
     0.7426221290311992*u2*v*w2*y2*z + 0.21620742554342368*lnbara*u2*v*
      w2*y2*z - 0.005287248454508436*v2*w2*y2*z + 
     0.0003034691554580678*u*v2*w2*y2*z + 1.0839553144032048*v3*w2*y2*
      z + 0.2997685920100648*lnbara*v3*w2*y2*z - 
     2.770275796275451*w3*y2*z - 0.8710102224533081*lnbara*w3*y2*z + 
     2.2733271985354553*u*w3*y2*z + 0.6947964005893216*lnbara*u*w3*y2*z + 
     0.15265057835965268*u2*w3*y2*z + 0.04014533812336501*lnbara*u2*w3*
      y2*z - 3.431997185570922*v*w3*y2*z - 1.0593983869002586*lnbara*v*w3*
      y2*z - 0.3063351018757037*u*v*w3*y2*z - 0.0810097794355468*lnbara*u*
      v*w3*y2*z - 1.0830627587062924*v2*w3*y2*z - 
     0.2997685920100648*lnbara*v2*w3*y2*z + 0.305005920641606*w4*y2*z + 
     0.10788365521657699*lnbara*w4*y2*z - 0.3602833223629141*u*w4*y2*z - 
     0.10812164650482199*lnbara*u*w4*y2*z + 1.1422301498147045*v*w4*y2*
      z + 0.33097644191339387*lnbara*v*w4*y2*z - 0.11272620964583531*w5*
      y2*z - 0.03339748565253056*lnbara*w5*y2*z + 
     0.45246456501551086*x*y2*z - 0.1076866392526224*u*x*y2*z + 
     0.02581697089389268*u2*x*y2*z - 0.004447305079644873*u3*x*y2*z + 
     0.00038194899426036925*u4*x*y2*z - 0.1113550912072001*v*x*y2*z + 
     0.023341335158146424*u*v*x*y2*z - 0.0036412369193938783*u2*v*x*y2*z + 
     0.0002913721656218036*u3*v*x*y2*z - 8.691639367114691*v2*x*y2*z - 
     2.6585984883283214*lnbara*v2*x*y2*z + 4.260756983735999*u*v2*x*y2*
      z + 1.2888307504910574*lnbara*u*v2*x*y2*z - 
     0.6381316954939846*u2*v2*x*y2*z - 0.19013921429911454*lnbara*u2*v2*
      x*y2*z + 1.046392947967706*v3*x*y2*z + 0.35193692906048857*lnbara*
      v3*x*y2*z - 0.18025531671901043*u*v3*x*y2*z - 
     0.06303275716653367*lnbara*u*v3*x*y2*z - 0.6823722786029639*v4*x*y2*
      z - 0.19655624712804137*lnbara*v4*x*y2*z - 0.1113550912072001*w*x*y2*
      z + 0.023341335158146424*u*w*x*y2*z - 0.0036412369193938783*u2*w*x*
      y2*z + 0.0002913721656218036*u3*w*x*y2*z + 
     17.462617451739675*v*w*x*y2*z + 5.317196976656643*lnbara*v*w*x*y2*z - 
     8.532337360892159*u*v*w*x*y2*z - 2.577661500982115*lnbara*u*v*w*x*y2*
      z + 1.277050969854987*u2*v*w*x*y2*z + 0.3802784285982291*lnbara*u2*v*
      w*x*y2*z - 1.0551604965935144*v2*w*x*y2*z - 
     0.35193692906048857*lnbara*v2*w*x*y2*z + 0.18081274049740853*u*v2*w*x*
      y2*z + 0.06303275716653367*lnbara*u*v2*w*x*y2*z + 
     2.0929117260759176*v3*w*x*y2*z + 0.5923170046701335*lnbara*v3*w*x*y2*
      z - 8.691639367114691*w2*x*y2*z - 2.6585984883283214*lnbara*w2*x*y2*
      z + 4.260756983735999*u*w2*x*y2*z + 1.2888307504910574*lnbara*u*w2*x*
      y2*z - 0.6381316954939846*u2*w2*x*y2*z - 0.19013921429911454*lnbara*
      u2*w2*x*y2*z - 1.0551604965935144*v*w2*x*y2*z - 
     0.35193692906048857*lnbara*v*w2*x*y2*z + 0.18081274049740853*u*v*w2*x*
      y2*z + 0.06303275716653367*lnbara*u*v*w2*x*y2*z - 
     2.8192943376822845*v2*w2*x*y2*z - 0.7915215150841841*lnbara*v2*w2*x*
      y2*z + 1.046392947967706*w3*x*y2*z + 0.35193692906048857*lnbara*w3*
      x*y2*z - 0.18025531671901043*u*w3*x*y2*z - 
     0.06303275716653367*lnbara*u*w3*x*y2*z + 2.0929117260759176*v*w3*x*
      y2*z + 0.5923170046701335*lnbara*v*w3*x*y2*z - 
     0.6823722786029639*w4*x*y2*z - 0.19655624712804137*lnbara*w4*x*y2*
      z - 0.1974796353404678*x2*y2*z + 0.03069034292530752*u*x2*y2*z - 
     0.0042936520074251285*u2*x2*y2*z + 0.00032622599069967407*u3*x2*y2*
      z + 14.046191130880398*v*x2*y2*z + 4.27806313688087*lnbara*v*x2*y2*
      z - 1.0675924425094043*u*v*x2*y2*z - 0.36692372038764914*lnbara*u*v*
      x2*y2*z + 0.7426221290311992*u2*v*x2*y2*z + 
     0.21620742554342368*lnbara*u2*v*x2*y2*z - 1.4568456155246001*v2*x2*
      y2*z - 0.48533250050556614*lnbara*v2*x2*y2*z - 
     0.5429343918861453*u*v2*x2*y2*z - 0.1557593910339726*lnbara*u*v2*x2*
      y2*z + 1.492463738628857*v3*x2*y2*z + 0.4272793464214101*lnbara*v3*
      x2*y2*z - 13.984210663495318*w*x2*y2*z - 4.27806313688087*lnbara*w*
      x2*y2*z + 1.059736505268818*u*w*x2*y2*z + 
     0.36692372038764914*lnbara*u*w*x2*y2*z - 0.7420694165018765*u2*w*x2*
      y2*z - 0.21620742554342368*lnbara*u2*w*x2*y2*z - 
     1.0551604965935144*v*w*x2*y2*z - 0.35193692906048857*lnbara*v*w*x2*
      y2*z + 0.18081274049740853*u*v*w*x2*y2*z + 
     0.06303275716653367*lnbara*u*v*w*x2*y2*z - 3.271037633209231*v2*w*x2*
      y2*z - 0.9126427694703488*lnbara*v2*w*x2*y2*z + 
     2.499274797319209*w2*x2*y2*z + 0.8372694295660547*lnbara*w2*x2*y2*
      z + 0.36293415511749816*u*w2*x2*y2*z + 0.09272663386743893*lnbara*u*
      w2*x2*y2*z + 3.2714817035807937*v*w2*x2*y2*z + 
     0.9180803814343197*lnbara*v*w2*x2*y2*z - 1.4916718912285227*w3*x2*
      y2*z - 0.432716958385381*lnbara*w3*x2*y2*z + 
     0.06395157977258972*x3*y2*z - 0.005780394040453321*u*x3*y2*z + 
     0.0003568540057622896*u2*x3*y2*z - 3.431997185570922*v*x3*y2*z - 
     1.0593983869002586*lnbara*v*x3*y2*z - 0.3063351018757037*u*v*x3*y2*
      z - 0.0810097794355468*lnbara*u*v*x3*y2*z - 
     0.7046868197766329*v2*x3*y2*z - 0.18518041263910326*lnbara*v2*x3*
      y2*z + 3.420468192644727*w*x3*y2*z + 1.0593983869002586*lnbara*w*x3*
      y2*z + 0.3069925110386899*u*w*x3*y2*z + 0.0810097794355468*lnbara*u*
      w*x3*y2*z + 2.0929117260759176*v*w*x3*y2*z + 
     0.5923170046701335*lnbara*v*w*x3*y2*z - 1.3871873506084558*w2*x3*y2*
      z - 0.4071365920310302*lnbara*w2*x3*y2*z - 
     0.013136621448171954*x4*y2*z + 0.0005230003987145713*u*x4*y2*z + 
     1.1422301498147045*v*x4*y2*z + 0.33097644191339387*lnbara*v*x4*y2*
      z - 1.1411946582446566*w*x4*y2*z - 0.33097644191339387*lnbara*w*x4*
      y2*z + 0.001263472670057093*x5*y2*z + 0.6858555690183518*y3*z - 
     0.14397058356518277*u*y3*z + 0.03238162914677262*u2*y3*z - 
     0.0053911589121258706*u3*y3*z + 0.0004539037516950064*u4*y3*z - 
     6.543865639160677*v*y3*z - 2.0509188338532764*lnbara*v*y3*z - 
     1.5432862100326823*u*v*y3*z - 0.4192402591437163*lnbara*u*v*y3*z - 
     0.44329609772338113*u2*v*y3*z - 0.12608980116426613*lnbara*u2*v*y3*
      z - 0.050643992483851874*u3*v*y3*z - 0.013481184061483282*lnbara*u3*
      v*y3*z + 2.417035504771195*v2*y3*z + 0.7884496760703545*lnbara*v2*
      y3*z + 0.9375779785463527*u*v2*y3*z + 0.26779586684578455*lnbara*u*
      v2*y3*z + 0.16819984993563145*u2*v2*y3*z + 
     0.046821383490061465*lnbara*u2*v2*y3*z - 2.4157548822081845*v3*y3*
      z - 0.7222688114635434*lnbara*v3*y3*z - 0.10092543442474523*u*v3*y3*
      z - 0.025545381796987655*lnbara*u*v3*y3*z + 
     0.3249190900404754*v4*y3*z + 0.09733306860121325*lnbara*v4*y3*z + 
     6.221638137007376*w*y3*z + 2.0509188338532764*lnbara*w*y3*z + 
     1.6028488135384624*u*w*y3*z + 0.4192402591437163*lnbara*u*w*y3*z + 
     0.43459981748405074*u2*w*y3*z + 0.12608980116426613*lnbara*u2*w*y3*
      z + 0.05131419753175176*u3*w*y3*z + 0.013481184061483282*lnbara*u3*w*
      y3*z + 0.03580809539433375*v*w*y3*z - 0.004288309525390823*u*v*w*y3*
      z + 0.0002913721656218036*u2*v*w*y3*z + 4.23815521421725*v2*w*y3*
      z + 1.229468487533898*lnbara*v2*w*y3*z - 0.46795439814148526*u*v2*w*
      y3*z - 0.14200807750295125*lnbara*u*v2*w*y3*z - 
     0.21637657479006892*v3*w*y3*z - 0.06469446777014416*lnbara*v3*w*y3*
      z - 2.339296200037759*w2*y3*z - 0.7884496760703545*lnbara*w2*y3*z - 
     0.9468503021957163*u*w2*y3*z - 0.26779586684578455*lnbara*u*w2*y3*
      z - 0.16757153566204508*u2*w2*y3*z - 0.046821383490061465*lnbara*u2*
      w2*y3*z - 4.2497906793289655*v*w2*y3*z - 1.229468487533898*lnbara*v*
      w2*y3*z + 0.4686008650126944*u*v*w2*y3*z + 
     0.14200807750295125*lnbara*u*v*w2*y3*z + 0.0004489814737672761*v2*w2*
      y3*z + 2.402267061883944*w3*y3*z + 0.7222688114635434*lnbara*w3*y3*
      z + 0.10166794959991486*u*w3*y3*z + 0.025545381796987655*lnbara*u*w3*
      y3*z + 0.21733303969258025*v*w3*y3*z + 0.06469446777014416*lnbara*v*
      w3*y3*z - 0.3237542215275384*w4*y3*z - 0.09733306860121325*lnbara*
      w4*y3*z - 0.2298476323443832*x*y3*z + 0.0339939779841863*u*x*y3*z - 
     0.004611648226787489*u2*x*y3*z + 0.0003439536028536974*u3*x*y3*z + 
     0.03580809539433375*v*x*y3*z - 0.004288309525390823*u*v*x*y3*z + 
     0.0002913721656218036*u2*v*x*y3*z + 2.9845483465315206*v2*x*y3*z + 
     0.873539503307877*lnbara*v2*x*y3*z - 0.7548466365170377*u*v2*x*y3*
      z - 0.21880709774223123*lnbara*u*v2*x*y3*z - 
     0.07704797923780887*v3*x*y3*z - 0.028040321374660362*lnbara*v3*x*y3*
      z + 0.03580809539433375*w*x*y3*z - 0.004288309525390823*u*w*x*y3*z + 
     0.0002913721656218036*u2*w*x*y3*z - 5.983871189946358*v*w*x*y3*z - 
     1.747079006615754*lnbara*v*w*x*y3*z + 1.5105778324009722*u*v*w*x*y3*
      z + 0.43761419548446245*lnbara*u*v*w*x*y3*z + 
     0.07777103749659893*v2*w*x*y3*z + 0.028040321374660362*lnbara*v2*w*x*
      y3*z + 2.9845483465315206*w2*x*y3*z + 0.873539503307877*lnbara*w2*x*
      y3*z - 0.7548466365170377*u*w2*x*y3*z - 0.21880709774223123*lnbara*u*
      w2*x*y3*z + 0.07777103749659893*v*w2*x*y3*z + 
     0.028040321374660362*lnbara*v*w2*x*y3*z - 0.07704797923780887*w3*x*
      y3*z - 0.028040321374660362*lnbara*w3*x*y3*z + 
     0.06395157977258972*x2*y3*z - 0.005673921854933901*u*x2*y3*z + 
     0.00034533361905439837*u2*x2*y3*z - 4.2497906793289655*v*x2*y3*z - 
     1.229468487533898*lnbara*v*x2*y3*z + 0.4686008650126944*u*v*x2*y3*
      z + 0.14200807750295125*lnbara*u*v*x2*y3*z + 
     0.029077792864669175*v2*x2*y3*z + 0.011972114859594282*lnbara*v2*x2*
      y3*z + 4.23815521421725*w*x2*y3*z + 1.229468487533898*lnbara*w*x2*
      y3*z - 0.46795439814148526*u*w*x2*y3*z - 0.14200807750295125*lnbara*
      u*w*x2*y3*z + 0.07777103749659893*v*w*x2*y3*z + 
     0.028040321374660362*lnbara*v*w*x2*y3*z - 0.10578881199195397*w2*x2*
      y3*z - 0.04001243623425464*lnbara*w2*x2*y3*z - 
     0.011927105098099399*x3*y3*z + 0.00047310410784989136*u*x3*y3*z + 
     0.21733303969258025*v*x3*y3*z + 0.06469446777014416*lnbara*v*x3*y3*
      z - 0.21637657479006892*w*x3*y3*z - 0.06469446777014416*lnbara*w*x3*
      y3*z + 0.0010740897315456516*x4*y3*z - 0.386061737342387*y4*z + 
     0.04988531635163183*u*y4*z - 0.00631630321372675*u2*y4*z + 
     0.0004539037516950064*u3*y4*z + 2.8587812294103165*v*y4*z + 
     0.8545495141055319*lnbara*v*y4*z + 0.05785733009437446*u*v*y4*z + 
     0.004869163308206694*lnbara*u*v*y4*z - 0.01953851722470816*u2*v*y4*
      z - 0.005188045152882472*lnbara*u2*v*y4*z - 
     0.2759336414712597*v2*y4*z - 0.09277616733835867*lnbara*v2*y4*z + 
     0.04382088794664975*u*v2*y4*z + 0.013022653225572441*lnbara*u*v2*y4*
      z + 0.23650678939377565*v3*y4*z + 0.06767441939812754*lnbara*v3*y4*
      z - 2.7455486537995553*w*y4*z - 0.8545495141055319*lnbara*w*y4*z - 
     0.06972985855284485*u*w*y4*z - 0.004869163308206694*lnbara*u*w*y4*z + 
     0.02029024705940767*u2*w*y4*z + 0.005188045152882472*lnbara*u2*w*y4*
      z - 0.007320613539132591*v*w*y4*z + 0.0003820180212080083*u*v*w*y4*
      z - 0.747040617041446*v2*w*y4*z - 0.21034811390666633*lnbara*v2*w*
      y4*z + 0.260373564991028*w2*y4*z + 0.09277616733835867*lnbara*w2*
      y4*z - 0.04301078480890973*u*w2*y4*z - 0.013022653225572441*lnbara*u*
      w2*y4*z + 0.7480915272662646*v*w2*y4*z + 0.21034811390666633*lnbara*
      v*w2*y4*z - 0.2353317760308018*w3*y4*z - 0.06767441939812754*lnbara*
      w3*y4*z + 0.08203160229248424*x*y4*z - 0.006873088261070091*u*x*y4*
      z + 0.00040392715269129916*u2*x*y4*z - 0.007320613539132591*v*x*y4*
      z + 0.0003820180212080083*u*v*x*y4*z - 0.4807266928857714*v2*x*y4*
      z - 0.1356427800167511*lnbara*v2*x*y4*z - 0.007320613539132591*w*x*
      y4*z + 0.0003820180212080083*u*w*x*y4*z + 0.9627812643931486*v*w*x*
      y4*z + 0.2712855600335022*lnbara*v*w*x*y4*z - 
     0.4807266928857714*w2*x*y4*z - 0.1356427800167511*lnbara*w2*x*y4*z - 
     0.013136621448171954*x2*y4*z + 0.000507581743944034*u*x2*y4*z + 
     0.7480915272662646*v*x2*y4*z + 0.21034811390666633*lnbara*v*x2*y4*
      z - 0.747040617041446*w*x2*y4*z - 0.21034811390666633*lnbara*w*x2*
      y4*z + 0.0010740897315456516*x3*y4*z + 0.1475870262830398*y5*z - 
     0.01075135761798518*u*y5*z + 0.0005880488320726199*u2*y5*z - 
     0.9314087991903407*v*y5*z - 0.26857270443316716*lnbara*v*y5*z - 
     0.038965576694794626*u*v*y5*z - 0.009316737180946303*lnbara*u*v*y5*z + 
     0.06988272110288132*v2*y5*z + 0.021403950235092197*lnbara*v2*y5*z + 
     0.9067948266532218*w*y5*z + 0.26857270443316716*lnbara*w*y5*z + 
     0.04008569352545048*u*w*y5*z + 0.009316737180946303*lnbara*u*w*y5*z + 
     0.0007017792893363362*v*w*y5*z - 0.06841032198704927*w2*y5*z - 
     0.021403950235092197*lnbara*w2*y5*z - 0.018000480690589078*x*y5*z + 
     0.0006541741874990681*u*x*y5*z + 0.0007017792893363362*v*x*y5*z + 
     0.0007017792893363362*w*x*y5*z + 0.001263472670057093*x2*y5*z - 
     0.0339914082143444*y6*z + 0.0010711545725659018*u*y6*z + 
     0.11490665548248956*v*y6*z + 0.032045606778957444*lnbara*v*y6*z - 
     0.11244046513485796*w*y6*z - 0.032045606778957444*lnbara*w*y6*z + 
     0.0018148095406981609*x*y6*z + 0.0035512018991495737*y7*z + 
     1.8468559937823905*z2 - 0.9004176968697725*u*z2 + 
     0.520113697995963*u2*z2 - 0.251564804320496*u3*z2 + 
     0.08755029317854246*u4*z2 - 0.018929896217160017*u5*z2 + 
     0.0018904202297008643*u6*z2 + 1.7490619043784765*v*z2 + 
     0.9115194817125674*lnbara*v*z2 + 6.04138458368625*u*v*z2 + 
     1.9748773458715838*lnbara*u*v*z2 + 4.3302489692293955*u2*v*z2 + 
     1.3823733765239528*lnbara*u2*v*z2 + 1.7772082042508452*u3*v*z2 + 
     0.5107892746111199*lnbara*u3*v*z2 + 0.33555238697393286*u4*v*z2 + 
     0.09558272829689277*lnbara*u4*v*z2 + 0.028399509890409173*u5*v*z2 + 
     0.007442787768918609*lnbara*u5*v*z2 + 2.7099088373198637*v2*z2 + 
     0.653288241029598*lnbara*v2*z2 - 18.2664360726891*u*v2*z2 - 
     6.062567347027013*lnbara*u*v2*z2 + 2.165326096275261*u2*v2*z2 + 
     0.7557728528917382*lnbara*u2*v2*z2 - 3.7882789087219217*u3*v2*z2 - 
     1.1472315670280682*lnbara*u3*v2*z2 + 0.2761865963929522*u4*v2*z2 + 
     0.08549461963730814*lnbara*u4*v2*z2 + 0.8972500249744643*v3*z2 + 
     0.3500794053947276*lnbara*v3*z2 + 7.880191664548825*u*v3*z2 + 
     2.6748784087603763*lnbara*u*v3*z2 + 0.9841419128221275*u2*v3*z2 + 
     0.23760579052794478*lnbara*u2*v3*z2 + 0.43479804170814224*u3*v3*
      z2 + 0.1344756922786936*lnbara*u3*v3*z2 + 
     0.651338693005845*v4*z2 + 0.14754669762714998*lnbara*v4*z2 - 
     3.7835481298107045*u*v4*z2 - 1.1938747004904564*lnbara*u*v4*z2 + 
     0.05725447402071905*u2*v4*z2 + 0.021576912362776347*lnbara*u2*v4*
      z2 + 0.03085436595372382*v5*z2 + 0.016119245278351776*lnbara*v5*
      z2 + 0.47543762922904986*u*v5*z2 + 0.14922477962881076*lnbara*u*v5*
      z2 + 0.013912167390291797*v6*z2 + 0.0027513912903176212*lnbara*v6*
      z2 - 3.3276691614032563*w*z2 - 0.9115194817125674*lnbara*w*z2 - 
     5.289736299142919*u*w*z2 - 1.9748773458715838*lnbara*u*w*z2 - 
     4.650086557902906*u2*w*z2 - 1.3823733765239528*lnbara*u2*w*z2 - 
     1.6750109905361525*u3*w*z2 - 0.5107892746111199*lnbara*u3*w*z2 - 
     0.3563815319580459*u4*w*z2 - 0.09558272829689277*lnbara*u4*w*z2 - 
     0.026406119658478486*u5*w*z2 - 0.007442787768918609*lnbara*u5*w*z2 - 
     10.763863509478952*v*w*z2 - 3.7979978404690304*lnbara*v*w*z2 + 
     14.408707884242522*u*v*w*z2 + 4.851483174726708*lnbara*u*v*w*z2 - 
     11.35352724075799*u2*v*w*z2 - 3.638468221673086*lnbara*u2*v*w*z2 + 
     3.595212268118998*u3*v*w*z2 + 1.124626001903981*lnbara*u3*v*w*z2 - 
     0.637464961753253*u4*v*w*z2 - 0.19025408565816512*lnbara*u4*v*w*z2 + 
     6.988719566598391*v2*w*z2 + 2.737479438521948*lnbara*v2*w*z2 + 
     2.583201962578535*u*v2*w*z2 + 0.6303616509938534*lnbara*u*v2*w*z2 + 
     2.78012177163788*u2*v2*w*z2 + 0.9193789312609476*lnbara*u2*v2*w*
      z2 + 0.0944959815922243*u3*v2*w*z2 + 0.012699061790125279*lnbara*
      u3*v2*w*z2 - 9.326716278242827*v3*w*z2 - 
     3.107847444868102*lnbara*v3*w*z2 + 2.4434643394917717*u*v3*w*z2 + 
     0.722704550671465*lnbara*u*v3*w*z2 - 1.4446667880342972*u2*v3*w*
      z2 - 0.43070638308913284*lnbara*u2*v3*w*z2 + 
     2.2331993759180806*v4*w*z2 + 0.7744184574067065*lnbara*v4*w*z2 + 
     0.08267922567088919*u*v4*w*z2 + 0.01813757575979942*lnbara*u*v4*w*
      z2 - 0.46033723691317213*v5*w*z2 - 0.14606708002401708*lnbara*v5*w*
      z2 + 9.22032358478725*w2*z2 + 3.1447095994394325*lnbara*w2*z2 + 
     3.443135626202991*u*w2*z2 + 1.2110841723003059*lnbara*u*w2*z2 + 
     9.306085458851838*u2*w2*z2 + 2.882695368781348*lnbara*u2*w2*z2 + 
     0.170862170132498*u3*w2*z2 + 0.022605565124086877*lnbara*u3*w2*
      z2 + 0.363288554298664*u4*w2*z2 + 0.10475946602085698*lnbara*u4*
      w2*z2 - 2.5057639727820775*v*w2*z2 - 1.0258918950424711*lnbara*v*
      w2*z2 - 4.04603639452907*u*v*w2*z2 - 1.3032259701999138*lnbara*u*v*
      w2*z2 - 0.5438356537989508*u2*v*w2*z2 - 0.1818291572693978*lnbara*
      u2*v*w2*z2 - 0.17072856838087597*u3*v*w2*z2 - 
     0.04904808595029047*lnbara*u3*v*w2*z2 + 8.25179929321027*v2*w2*
      z2 + 2.469212605593015*lnbara*v2*w2*z2 - 8.325060940884864*u*v2*
      w2*z2 - 2.4651692331377286*lnbara*u*v2*w2*z2 + 
     1.4020259571353741*u2*v2*w2*z2 + 0.40905101643700437*lnbara*u2*v2*
      w2*z2 - 0.2730340390527256*v3*w2*z2 - 0.09631911852193811*lnbara*
      v3*w2*z2 + 0.5346542533590023*u*v3*w2*z2 + 
     0.17261962424250146*lnbara*u*v3*w2*z2 + 0.3702307156554167*v4*w2*
      z2 + 0.10136840292874362*lnbara*v4*w2*z2 - 
     6.021984802397401*w3*z2 - 2.0616669488742043*lnbara*w3*z2 - 
     6.264516709859727*u*w3*z2 - 2.002014089554316*lnbara*u*w3*z2 - 
     3.2461502369868342*u2*w3*z2 - 0.9751555645194946*lnbara*u2*w3*z2 - 
     0.35640766524496237*u3*w3*z2 - 0.09812666811852842*lnbara*u3*w3*
      z2 - 4.961706764789829*v*w3*z2 - 1.3124617908295524*lnbara*v*w3*
      z2 + 8.457223146401986*u*v*w3*z2 + 2.5565136487937905*lnbara*u*v*w3*
      z2 - 0.9782558475006098*u2*v*w3*z2 - 0.28088060701825807*lnbara*u2*
      v*w3*z2 - 0.4179222450177246*v2*w3*z2 - 0.11149384692877298*lnbara*
      v2*w3*z2 + 0.3092023748907339*u*v2*w3*z2 + 
     0.0785869636098416*lnbara*u*v2*w3*z2 - 0.6281581353908926*v3*w3*
      z2 - 0.1774903697854292*lnbara*v3*w3*z2 + 
     5.626060910791232*w4*z2 + 1.8035499324774895*lnbara*w4*z2 + 
     1.1739473090123895*u*w4*z2 + 0.3798257341629295*lnbara*u*w4*z2 + 
     0.9661938550317448*u2*w4*z2 + 0.2809590613076102*lnbara*u2*w4*z2 + 
     0.01586781553997586*v*w4*z2 - 0.06352181102996676*lnbara*v*w4*z2 - 
     1.0816608287503775*u*v*w4*z2 - 0.32438223975151587*lnbara*u*v*w4*
      z2 + 0.581863351202302*v2*w4*z2 + 0.1659602153670836*lnbara*v2*w4*
      z2 - 1.6435396981777775*w5*z2 - 0.5192029262043804*lnbara*w5*z2 - 
     0.31687310350773107*u*w5*z2 - 0.09418670348943739*lnbara*u*w5*z2 - 
     0.14867177025989595*v*w5*z2 - 0.030113006529747105*lnbara*v*w5*z2 + 
     0.2767842038860737*w6*z2 + 0.08359044675304854*lnbara*w6*z2 - 
     0.9004176968697725*x*z2 + 0.3313625873896489*u*x*z2 - 
     0.12887042405432217*u2*x*z2 + 0.03949367079133293*u3*x*z2 - 
     0.007867552944290607*u4*x*z2 + 0.0007424815539584533*u5*x*z2 - 
     16.220273175775272*v*x*z2 - 5.75015242423565*lnbara*v*x*z2 + 
     7.283613291269799*u*v*x*z2 + 2.569763477061043*lnbara*u*v*x*z2 - 
     10.868438804341743*u2*v*x*z2 - 3.4712775032258807*lnbara*u2*v*x*z2 + 
     2.723658999110921*u3*v*x*z2 + 0.8563693554764021*lnbara*u3*v*x*z2 - 
     0.5498436699327358*u4*v*x*z2 - 0.16402829156973656*lnbara*u4*v*x*
      z2 + 6.988719566598391*v2*x*z2 + 2.737479438521948*lnbara*v2*x*
      z2 + 2.583201962578535*u*v2*x*z2 + 0.6303616509938534*lnbara*u*v2*x*
      z2 + 2.78012177163788*u2*v2*x*z2 + 0.9193789312609476*lnbara*u2*
      v2*x*z2 + 0.0944959815922243*u3*v2*x*z2 + 
     0.012699061790125279*lnbara*u3*v2*x*z2 - 10.997725885446338*v3*x*
      z2 - 3.557746295983214*lnbara*v3*x*z2 + 2.9792615221240184*u*v3*x*
      z2 + 0.9000686333403095*lnbara*u*v3*x*z2 - 
     1.8066447210612555*u2*v3*x*z2 - 0.5331767926648573*lnbara*u2*v3*x*
      z2 + 2.4334094721170496*v4*x*z2 + 0.816632950371704*lnbara*v4*x*
      z2 + 0.10307841593891565*u*v4*x*z2 + 0.018172273278693238*lnbara*u*
      v4*x*z2 - 0.5247428948621341*v5*x*z2 - 0.16081753949969196*lnbara*
      v5*x*z2 + 16.840740404671994*w*x*z2 + 5.75015242423565*lnbara*w*x*
      z2 - 7.4944534531230955*u*w*x*z2 - 2.569763477061043*lnbara*u*w*x*
      z2 + 10.92704206771195*u2*w*x*z2 + 3.4712775032258807*lnbara*u2*w*x*
      z2 - 2.7345557618171856*u3*w*x*z2 - 0.8563693554764021*lnbara*u3*w*
      x*z2 + 0.5508222899910821*u4*w*x*z2 + 0.16402829156973656*lnbara*u4*
      w*x*z2 - 4.808822770349789*v*w*x*z2 - 1.7115875434794767*lnbara*v*w*x*
      z2 + 1.5426830809278815*u*v*w*x*z2 + 0.6728643192060604*lnbara*u*v*w*
      x*z2 - 2.2498626331415124*u2*v*w*x*z2 - 0.7375497739915498*lnbara*
      u2*v*w*x*z2 + 0.07737638458711994*u3*v*w*x*z2 + 
     0.03634902416016519*lnbara*u3*v*w*x*z2 + 21.55337156258267*v2*w*x*
      z2 + 6.548245980022291*lnbara*v2*w*x*z2 - 14.057135775486067*u*v2*w*
      x*z2 - 4.266687153487787*lnbara*u*v2*w*x*z2 + 
     3.4982098333445135*u2*v2*w*x*z2 + 1.0191704113395068*lnbara*u2*v2*w*
      x*z2 - 2.1068706790103*v3*w*x*z2 - 0.6955343577421946*lnbara*v3*w*x*
      z2 + 0.8761362502114731*u*v3*w*x*z2 + 0.296376278878668*lnbara*u*v3*
      w*x*z2 + 1.0094259951122324*v4*w*x*z2 + 0.2874207900901255*lnbara*
      v4*w*x*z2 - 2.5057639727820775*w2*x*z2 - 1.0258918950424711*lnbara*
      w2*x*z2 - 4.04603639452907*u*w2*x*z2 - 1.3032259701999138*lnbara*u*
      w2*x*z2 - 0.5438356537989508*u2*w2*x*z2 - 
     0.1818291572693978*lnbara*u2*w2*x*z2 - 0.17072856838087597*u3*w2*x*
      z2 - 0.04904808595029047*lnbara*u3*w2*x*z2 - 
     20.346437220092437*v*w2*x*z2 - 6.08173812414508*lnbara*v*w2*x*z2 + 
     16.365226923917742*u*v*w2*x*z2 + 4.905952869158358*lnbara*u*v*w2*x*
      z2 - 3.2577140116251813*u2*v*w2*x*z2 - 0.9439400410182851*lnbara*
      u2*v*w2*x*z2 - 0.40857845580650054*v2*w2*x*z2 - 
     0.08307962011998349*lnbara*v2*w2*x*z2 - 0.5491803784646733*u*v2*w2*
      x*z2 - 0.17704276038055036*lnbara*u*v2*w2*x*z2 - 
     1.7437824185418067*v3*w2*x*z2 - 0.4914395306435913*lnbara*v3*w2*x*
      z2 + 9.909152497593267*w3*x*z2 + 3.0912384401060025*lnbara*w3*x*
      z2 - 5.305139799956746*u*w3*x*z2 - 1.5393343490108804*lnbara*u*w3*x*
      z2 + 1.5675237969406215*u2*w3*x*z2 + 0.4579464223436356*lnbara*u2*
      w3*x*z2 + 1.4784816196761565*v*w3*x*z2 + 0.43230388098155775*lnbara*
      v*w3*x*z2 - 0.6938884585226485*u*v*w3*x*z2 - 
     0.21809446900966986*lnbara*u*v*w3*x*z2 + 2.1080028169579204*v2*w3*x*
      z2 + 0.5953996603124057*lnbara*v2*w3*x*z2 - 
     1.4226442649834423*w4*x*z2 - 0.4703228534910837*lnbara*w4*x*z2 + 
     0.2656466906715195*u*w4*x*z2 + 0.08058867723285898*lnbara*u*w4*x*
      z2 - 1.267405871830795*v*w4*x*z2 - 0.35480949331667067*lnbara*v*w4*
      x*z2 + 0.4211542429001266*w5*x*z2 + 0.12424611305742275*lnbara*w5*x*
      z2 + 0.520113697995963*x2*z2 - 0.12887042405432217*u*x2*z2 + 
     0.03186642256716374*u2*x2*z2 - 0.00560074458687052*u3*x2*z2 + 
     0.000487363420148127*u4*x2*z2 - 0.1223579533330431*v*x2*z2 + 
     0.027313138910434232*u*v*x2*z2 - 0.004442047950830934*u2*v*x2*z2 + 
     0.0003652020446131235*u3*v*x2*z2 + 8.25179929321027*v2*x2*z2 + 
     2.469212605593015*lnbara*v2*x2*z2 - 8.325060940884864*u*v2*x2*z2 - 
     2.4651692331377286*lnbara*u*v2*x2*z2 + 1.4020259571353741*u2*v2*x2*
      z2 + 0.40905101643700437*lnbara*u2*v2*x2*z2 + 
     0.3997513066521812*v3*x2*z2 + 0.08307962011998349*lnbara*v3*x2*
      z2 + 0.5497959040120096*u*v3*x2*z2 + 0.17704276038055036*lnbara*u*
      v3*x2*z2 + 0.39505594064956606*v4*x2*z2 + 
     0.11018942250071202*lnbara*v4*x2*z2 - 0.1223579533330431*w*x2*z2 + 
     0.027313138910434232*u*w*x2*z2 - 0.004442047950830934*u2*w*x2*z2 + 
     0.0003652020446131235*u3*w*x2*z2 - 16.42034107834173*v*w*x2*z2 - 
     4.93842521118603*lnbara*v*w*x2*z2 + 16.637816190951657*u*v*w*x2*z2 + 
     4.930338466275457*lnbara*u*v*w*x2*z2 - 2.803109839150323*u2*v*w*x2*
      z2 - 0.8181020328740087*lnbara*u2*v*w*x2*z2 - 
     0.40857845580650054*v2*w*x2*z2 - 0.08307962011998349*lnbara*v2*w*x2*
      z2 - 0.5491803784646733*u*v2*w*x2*z2 - 0.17704276038055036*lnbara*u*
      v2*w*x2*z2 - 1.7364430725838955*v3*w*x2*z2 - 
     0.4913145532522245*lnbara*v3*w*x2*z2 + 8.25179929321027*w2*x2*z2 + 
     2.469212605593015*lnbara*w2*x2*z2 - 8.325060940884864*u*w2*x2*z2 - 
     2.4651692331377286*lnbara*u*w2*x2*z2 + 1.4020259571353741*u2*w2*x2*
      z2 + 0.40905101643700437*lnbara*u2*w2*x2*z2 - 
     0.40857845580650054*v*w2*x2*z2 - 0.08307962011998349*lnbara*v*w2*x2*
      z2 - 0.5491803784646733*u*v*w2*x2*z2 - 0.17704276038055036*lnbara*u*
      v*w2*x2*z2 + 2.684507561107731*v2*w2*x2*z2 + 
     0.7622502615030249*lnbara*v2*w2*x2*z2 + 0.3997513066521812*w3*x2*
      z2 + 0.08307962011998349*lnbara*w3*x2*z2 + 
     0.5497959040120096*u*w3*x2*z2 + 0.17704276038055036*lnbara*u*w3*x2*
      z2 - 1.7364430725838955*v*w3*x2*z2 - 0.4913145532522245*lnbara*v*
      w3*x2*z2 + 0.39505594064956606*w4*x2*z2 + 
     0.11018942250071202*lnbara*w4*x2*z2 - 0.251564804320496*x3*z2 + 
     0.03949367079133293*u*x3*z2 - 0.00560074458687052*u2*x3*z2 + 
     0.0004300153542944288*u3*x3*z2 - 6.2616857203946035*v*x3*z2 - 
     1.8196275948475404*lnbara*v*x3*z2 + 3.5848494417836383*u*v*x3*z2 + 
     1.0856448829578143*lnbara*u*v*x3*z2 - 0.660951971257608*u2*v*x3*
      z2 - 0.18951332857164516*lnbara*u2*v*x3*z2 - 
     0.4179222450177246*v2*x3*z2 - 0.11149384692877298*lnbara*v2*x3*
      z2 + 0.3092023748907339*u*v2*x3*z2 + 0.0785869636098416*lnbara*u*
      v2*x3*z2 - 0.8243258774524206*v3*x3*z2 - 
     0.23041500895027525*lnbara*v3*x3*z2 + 6.336956677089462*w*x3*z2 + 
     1.8196275948475404*lnbara*w*x3*z2 - 3.5946630672038005*u*w*x3*z2 - 
     1.0856448829578143*lnbara*u*w*x3*z2 + 0.6616579679935063*u2*w*x3*
      z2 + 0.18951332857164516*lnbara*u2*w*x3*z2 + 
     0.6761564834039746*v*w*x3*z2 + 0.2078129654507111*lnbara*v*w*x3*z2 - 
     0.8428720597864838*u*v*w*x3*z2 - 0.25120658785234307*lnbara*u*v*w*x3*
      z2 + 2.1080028169579204*v2*w*x3*z2 + 0.5953996603124057*lnbara*v2*
      w*x3*z2 - 0.2730340390527256*w2*x3*z2 - 0.09631911852193811*lnbara*
      w2*x3*z2 + 0.5346542533590023*u*w2*x3*z2 + 
     0.17261962424250146*lnbara*u*w2*x3*z2 - 1.7437824185418067*v*w2*x3*
      z2 - 0.4914395306435913*lnbara*v*w2*x3*z2 + 
     0.461488349439593*w3*x3*z2 + 0.12645487928146082*lnbara*w3*x3*z2 + 
     0.08755029317854246*x4*z2 - 0.007867552944290607*u*x4*z2 + 
     0.000487363420148127*u2*x4*z2 + 0.7756766725864659*v*x4*z2 + 
     0.2013783664993635*lnbara*v*x4*z2 - 0.3272951320069847*u*v*x4*z2 - 
     0.09992430356434646*lnbara*u*v*x4*z2 + 0.581863351202302*v2*x4*z2 + 
     0.1659602153670836*lnbara*v2*x4*z2 - 0.7906870068555788*w*x4*z2 - 
     0.2013783664993635*lnbara*w*x4*z2 + 0.3281593662661477*u*w*x4*z2 + 
     0.09992430356434646*lnbara*u*w*x4*z2 - 0.9508002022690504*v*w*x4*
      z2 - 0.2673286182958272*lnbara*v*w*x4*z2 + 
     0.3702307156554167*w2*x4*z2 + 0.10136840292874362*lnbara*w2*x4*
      z2 - 0.018929896217160017*x5*z2 + 0.0007424815539584533*u*x5*z2 - 
     0.22871819549961922*v*x5*z2 - 0.05887745268544395*lnbara*v*x5*z2 + 
     0.2301351338116376*w*x5*z2 + 0.05887745268544395*lnbara*w*x5*z2 + 
     0.0018904202297008643*x6*z2 - 0.9004176968697725*y*z2 + 
     0.3212835093689003*u*y*z2 - 0.1223579533330431*u2*y*z2 + 
     0.03702532275275472*u3*y*z2 - 0.007317810733171885*u4*y*z2 + 
     0.0006870207501448052*u5*y*z2 + 8.371127296661705*v*y*z2 + 
     3.007965370948085*lnbara*v*y*z2 + 8.142878738630774*u*v*y*z2 + 
     2.570302308603461*lnbara*u*v*y*z2 + 4.372397515146079*u2*v*y*z2 + 
     1.303884138860003*lnbara*u2*v*y*z2 + 1.089630311725629*u3*v*y*z2 + 
     0.306972893835514*lnbara*u3*v*y*z2 + 0.11045275181230746*u4*v*y*z2 + 
     0.029980593819149505*lnbara*u4*v*y*z2 - 11.93413855963392*v2*y*z2 - 
     4.1260209265036405*lnbara*v2*y*z2 - 1.9788978869332372*u*v2*y*z2 - 
     0.6338374192524003*lnbara*u*v2*y*z2 - 4.430910486593282*u2*v2*y*
      z2 - 1.3342436103888362*lnbara*u2*v2*y*z2 - 
     0.1142476902831892*u3*v2*y*z2 - 0.0269991887735768*lnbara*u3*v2*y*
      z2 + 5.769076336816564*v3*y*z2 + 2.0547100979131496*lnbara*v3*y*
      z2 + 2.371701525434619*u*v3*y*z2 + 0.6890237976926594*lnbara*u*v3*y*
      z2 + 0.729201859518042*u2*v3*y*z2 + 0.21972319960553072*lnbara*u2*
      v3*y*z2 - 2.872858759697617*v4*y*z2 - 0.9340060015696775*lnbara*v4*
      y*z2 - 0.13345592306181395*u*v4*y*z2 - 0.03997846452824747*lnbara*u*
      v4*y*z2 + 0.3612787107357604*v5*y*z2 + 0.11818205016461883*lnbara*
      v5*y*z2 - 7.740580989744235*w*y*z2 - 3.007965370948085*lnbara*w*y*
      z2 - 8.34831676538791*u*w*y*z2 - 2.570302308603461*lnbara*u*w*y*z2 - 
     4.3167789899540825*u2*w*y*z2 - 1.303884138860003*lnbara*u2*w*y*z2 - 
     1.0998046234447676*u3*w*y*z2 - 0.306972893835514*lnbara*u3*w*y*z2 - 
     0.10954859095754421*u4*w*y*z2 - 0.029980593819149505*lnbara*u4*w*y*
      z2 + 10.777349982686436*v*w*y*z2 + 3.494158013231508*lnbara*v*w*y*
      z2 - 14.680040084881753*u*v*w*y*z2 - 4.586055804009709*lnbara*u*v*w*y*
      z2 + 2.978128785284849*u2*v*w*y*z2 + 0.9528593416680298*lnbara*u2*v*
      w*y*z2 - 0.7405676776348092*u3*v*w*y*z2 - 0.21821356263065994*lnbara*
      u3*v*w*y*z2 - 0.6330043203094217*v2*w*y*z2 - 
     0.3326899352140709*lnbara*v2*w*y*z2 + 3.7124326039755955*u*v2*w*y*
      z2 + 1.229224903218318*lnbara*u*v2*w*y*z2 + 
     0.3880630632568426*u2*v2*w*y*z2 + 0.0969009664171316*lnbara*u2*v2*w*
      y*z2 + 3.1903919237729643*v3*w*y*z2 + 0.9222524033868401*lnbara*v3*
      w*y*z2 - 2.203724440304821*u*v3*w*y*z2 - 0.6502064428550666*lnbara*u*
      v3*w*y*z2 - 0.06439562415041984*v4*w*y*z2 - 
     0.022967627645177263*lnbara*v4*w*y*z2 + 0.8190067945965689*w2*y*z2 + 
     0.6318629132721324*lnbara*w2*y*z2 + 16.737595251903855*u*w2*y*z2 + 
     5.219893223262108*lnbara*u*w2*y*z2 + 1.43981530141774*u2*w2*y*z2 + 
     0.3813842687208064*lnbara*u2*w2*y*z2 + 0.8558867409043719*u3*w2*y*
      z2 + 0.24521275140423673*lnbara*u3*w2*y*z2 + 
     0.5638297481368869*v*w2*y*z2 + 0.18788462045702756*lnbara*v*w2*y*
      z2 + 0.008389015172307996*u*v*w2*y*z2 - 0.057042047293007624*lnbara*
      u*v*w2*y*z2 - 0.2581203920240232*u2*v*w2*y*z2 - 
     0.07638602271158683*lnbara*u2*v*w2*y*z2 - 6.896344257474369*v2*w2*y*
      z2 - 2.015976321936135*lnbara*v2*w2*y*z2 + 
     1.8892318082935522*u*v2*w2*y*z2 + 0.5393619159455898*lnbara*u*v2*w2*
      y*z2 + 0.501248699467762*v3*w2*y*z2 + 0.1627718847199586*lnbara*v3*
      w2*y*z2 - 5.574896354901597*w3*y*z2 - 1.909904783156106*lnbara*w3*
      y*z2 - 6.110264042573513*u*w3*y*z2 - 1.8612066536179699*lnbara*u*w3*
      y*z2 - 0.8578197999104673*u2*w3*y*z2 - 0.24023814331107549*lnbara*
      u2*w3*y*z2 + 5.676870448679409*v*w3*y*z2 + 
     1.684726260416403*lnbara*v*w3*y*z2 - 1.182692665484111*u*v*w3*y*z2 - 
     0.32670867639523105*lnbara*u*v*w3*y*z2 + 0.030715686889802068*v2*w3*
      y*z2 - 0.0026175022546954534*lnbara*v2*w3*y*z2 + 
     0.8738097966923795*w4*y*z2 + 0.34300365970256935*lnbara*w4*y*z2 + 
     1.632450766309338*u*w4*y*z2 + 0.47753166783295525*lnbara*u*w4*y*z2 - 
     0.566190372464217*v*w4*y*z2 - 0.1702185650157698*lnbara*v*w4*y*z2 - 
     0.2597697450829062*w5*y*z2 - 0.08515023996893491*lnbara*w5*y*z2 + 
     0.45246456501551086*x*y*z2 - 0.1113550912072001*u*x*y*z2 + 
     0.027313138910434232*u2*x*y*z2 - 0.004771933785158886*u3*x*y*z2 + 
     0.0004135240269207944*u4*x*y*z2 + 14.923465303571382*v*x*y*z2 + 
     4.700428217339612*lnbara*v*x*y*z2 - 8.636182740657008*u*v*x*y*z2 - 
     2.845063377679382*lnbara*u*v*x*y*z2 + 2.0292029954219255*u2*v*x*y*
      z2 + 0.6312081178376158*lnbara*u2*v*x*y*z2 - 
     0.5873630368597165*u3*v*x*y*z2 - 0.17392267685688417*lnbara*u3*v*x*y*
      z2 - 0.6330043203094217*v2*x*y*z2 - 0.3326899352140709*lnbara*v2*x*
      y*z2 + 3.7124326039755955*u*v2*x*y*z2 + 1.229224903218318*lnbara*u*
      v2*x*y*z2 + 0.3880630632568426*u2*v2*x*y*z2 + 
     0.0969009664171316*lnbara*u2*v2*x*y*z2 + 3.8502028793420484*v3*x*y*
      z2 + 1.1170305738043862*lnbara*v3*x*y*z2 - 
     2.127445819261271*u*v3*x*y*z2 - 0.6316147687241351*lnbara*u*v3*x*y*
      z2 - 0.09314373993962562*v4*x*y*z2 - 0.033291399122784844*lnbara*v4*
      x*y*z2 - 15.142507034031205*w*x*y*z2 - 4.700428217339612*lnbara*w*x*y*
      z2 + 8.684236515504576*u*w*x*y*z2 + 2.845063377679382*lnbara*u*w*x*y*
      z2 - 2.0369169970657963*u2*w*x*y*z2 - 0.6312081178376158*lnbara*u2*
      w*x*y*z2 + 0.5879915453536685*u3*w*x*y*z2 + 
     0.17392267685688417*lnbara*u3*w*x*y*z2 + 0.14564601713500816*v*w*x*y*
      z2 + 0.14480531475704334*lnbara*v*w*x*y*z2 - 
     3.731845090721665*u*v*w*x*y*z2 - 1.1721828559253105*lnbara*u*v*w*x*y*
      z2 - 0.12911307483747164*u2*v*w*x*y*z2 - 0.020514943705544776*lnbara*
      u2*v*w*x*y*z2 - 14.77406945239147*v2*w*x*y*z2 - 
     4.3242801796236945*lnbara*v2*w*x*y*z2 + 4.000620121386894*u*v2*w*x*y*
      z2 + 1.1582596990589664*lnbara*u*v2*w*x*y*z2 + 
     0.9014731642677928*v3*w*x*y*z2 + 0.28985320078620536*lnbara*v3*w*x*y*
      z2 + 0.5638297481368869*w2*x*y*z2 + 0.18788462045702756*lnbara*w2*x*
      y*z2 + 0.008389015172307996*u*w2*x*y*z2 - 0.057042047293007624*
      lnbara*u*w2*x*y*z2 - 0.2581203920240232*u2*w2*x*y*z2 - 
     0.07638602271158683*lnbara*u2*w2*x*y*z2 + 17.091071621436193*v*w2*x*
      y*z2 + 4.99601857372239*lnbara*v*w2*x*y*z2 - 
     3.489746635672796*u*v*w2*x*y*z2 - 1.0013349315415667*lnbara*u*v*w2*x*
      y*z2 - 0.2945882654409418*v2*w2*x*y*z2 - 0.10231487175528582*lnbara*
      v2*w2*x*y*z2 - 6.183783910974188*w3*x*y*z2 - 
     1.7887689679030814*lnbara*w3*x*y*z2 + 1.617693856062018*u*w3*x*y*
      z2 + 0.4746900012067356*lnbara*u*w3*x*y*z2 - 
     0.9801494429301945*v*w3*x*y*z2 - 0.293356436036497*lnbara*v*w3*x*y*
      z2 + 0.4680668096248443*w4*x*y*z2 + 0.13910950612836231*lnbara*w4*x*
      y*z2 - 0.1974796353404678*x2*y*z2 + 0.03129012445977296*u*x2*y*
      z2 - 0.004442047950830934*u2*x2*y*z2 + 0.0003407946912851918*u3*
      x2*y*z2 + 0.03069034292530752*v*x2*y*z2 - 
     0.003995614840649483*u*v*x2*y*z2 + 0.0002863764888716214*u2*v*x2*y*
      z2 - 6.896344257474369*v2*x2*y*z2 - 2.015976321936135*lnbara*v2*
      x2*y*z2 + 1.8892318082935522*u*v2*x2*y*z2 + 
     0.5393619159455898*lnbara*u*v2*x2*y*z2 + 0.2951846833599234*v3*x2*y*
      z2 + 0.10231487175528582*lnbara*v3*x2*y*z2 + 
     0.03069034292530752*w*x2*y*z2 - 0.003995614840649483*u*w*x2*y*z2 + 
     0.0002863764888716214*u2*w*x2*y*z2 + 13.78024088853395*v*w*x2*y*
      z2 + 4.03195264387227*lnbara*v*w*x2*y*z2 - 
     3.777642640073415*u*v*w*x2*y*z2 - 1.0787238318911796*lnbara*u*v*w*x2*
      y*z2 - 0.2945882654409418*v2*w*x2*y*z2 - 0.10231487175528582*lnbara*
      v2*w*x2*y*z2 - 6.896344257474369*w2*x2*y*z2 - 
     2.015976321936135*lnbara*w2*x2*y*z2 + 1.8892318082935522*u*w2*x2*y*
      z2 + 0.5393619159455898*lnbara*u*w2*x2*y*z2 - 
     0.2945882654409418*v*w2*x2*y*z2 - 0.10231487175528582*lnbara*v*w2*
      x2*y*z2 + 0.2951846833599234*w3*x2*y*z2 + 
     0.10231487175528582*lnbara*w3*x2*y*z2 + 0.06395157977258972*x3*y*
      z2 - 0.005855071071261368*u*x3*y*z2 + 0.0003652020446131235*u2*x3*
      y*z2 + 6.015319832123562*v*x3*y*z2 + 1.7632715691466476*lnbara*v*x3*
      y*z2 - 0.7304529490550218*u*v*x3*y*z2 - 0.21288835092447814*lnbara*u*
      v*x3*y*z2 + 0.030715686889802068*v2*x3*y*z2 - 
     0.0026175022546954534*lnbara*v2*x3*y*z2 - 6.026774148018949*w*x3*y*
      z2 - 1.7632715691466476*lnbara*w*x3*y*z2 + 
     0.731117931401122*u*w*x3*y*z2 + 0.21288835092447814*lnbara*u*w*x3*y*
      z2 - 0.5309427518886998*v*w*x3*y*z2 - 0.16015438246526315*lnbara*v*w*
      x3*y*z2 + 0.501248699467762*w2*x3*y*z2 + 
     0.1627718847199586*lnbara*w2*x3*y*z2 - 0.013136621448171954*x4*y*
      z2 + 0.000527909826103984*u*x4*y*z2 - 0.6626623249676024*v*x4*y*
      z2 - 0.19636416180684646*lnbara*v*x4*y*z2 + 
     0.663692907110261*w*x4*y*z2 + 0.19636416180684646*lnbara*w*x4*y*z2 + 
     0.001263472670057093*x5*y*z2 + 0.520113697995963*y2*z2 - 
     0.1223579533330431*u*y2*z2 + 0.029164662906062597*u2*y2*z2 - 
     0.0050083201180713645*u3*y2*z2 + 0.00042934540544391214*u4*y2*z2 - 
     0.044978574732269516*v*y2*z2 - 0.06378679623852483*lnbara*v*y2*z2 + 
     2.5864306854801558*u*v*y2*z2 + 0.8006228366412104*lnbara*u*v*y2*z2 + 
     1.1889883290028347*u2*v*y2*z2 + 0.3402159951898016*lnbara*u2*v*y2*
      z2 + 0.17341769222311584*u3*v*y2*z2 + 0.04766748743684271*lnbara*
      u3*v*y2*z2 + 2.216432092777231*v2*y2*z2 + 
     0.6745097408562837*lnbara*v2*y2*z2 - 4.417728806752933*u*v2*y2*
      z2 - 1.3282152666550597*lnbara*u*v2*y2*z2 - 
     0.18884186308954337*u2*v2*y2*z2 - 0.048449976878940454*lnbara*u2*
      v2*y2*z2 + 0.32522417184198954*v3*y2*z2 + 
     0.06989084748944815*lnbara*v3*y2*z2 + 0.6591783435173679*u*v3*y2*
      z2 + 0.20062980696935676*lnbara*u*v3*y2*z2 + 
     0.11854588979479715*v4*y2*z2 + 0.03415403184344883*lnbara*v4*y2*
      z2 - 0.20624980265509574*w*y2*z2 + 0.06378679623852483*lnbara*w*y2*
      z2 - 2.5341893643030327*u*w*y2*z2 - 0.8006228366412104*lnbara*u*w*
      y2*z2 - 1.1971008100465077*u2*w*y2*z2 - 0.3402159951898016*lnbara*
      u2*w*y2*z2 - 0.17277027032117476*u3*w*y2*z2 - 
     0.04766748743684271*lnbara*u3*w*y2*z2 - 8.691639367114691*v*w*y2*
      z2 - 2.6585984883283214*lnbara*v*w*y2*z2 + 
     4.260756983735999*u*v*w*y2*z2 + 1.2888307504910574*lnbara*u*v*w*y2*
      z2 - 0.6381316954939846*u2*v*w*y2*z2 - 0.19013921429911454*lnbara*
      u2*v*w*y2*z2 + 2.499274797319209*v2*w*y2*z2 + 
     0.8372694295660547*lnbara*v2*w*y2*z2 + 0.36293415511749816*u*v2*w*
      y2*z2 + 0.09272663386743893*lnbara*u*v2*w*y2*z2 - 
     1.3871873506084558*v3*w*y2*z2 - 0.4071365920310302*lnbara*v3*w*y2*
      z2 + 6.563551498721121*w2*y2*z2 + 1.9840887474720377*lnbara*w2*y2*
      z2 + 0.14499255922663437*u*w2*y2*z2 + 0.03938451616400227*lnbara*u*
      w2*y2*z2 + 0.8278420296899973*u2*w2*y2*z2 + 
     0.238589191178055*lnbara*u2*w2*y2*z2 - 1.4568456155246001*v*w2*y2*
      z2 - 0.48533250050556614*lnbara*v*w2*y2*z2 - 
     0.5429343918861453*u*v*w2*y2*z2 - 0.1557593910339726*lnbara*u*v*w2*
      y2*z2 + 1.4101267527346482*v2*w2*y2*z2 + 
     0.39576075754209206*lnbara*v2*w2*y2*z2 - 1.3869981182997964*w3*y2*
      z2 - 0.4218277765499367*lnbara*w3*y2*z2 - 
     0.4779542480040674*u*w3*y2*z2 - 0.13759704980282306*lnbara*u*w3*y2*
      z2 - 0.7046868197766329*v*w3*y2*z2 - 0.18518041263910326*lnbara*v*
      w3*y2*z2 + 0.5651566216606796*w4*y2*z2 + 
     0.16240221528459253*lnbara*w4*y2*z2 - 0.1974796353404678*x*y2*z2 + 
     0.03069034292530752*u*x*y2*z2 - 0.0042936520074251285*u2*x*y2*z2 + 
     0.00032622599069967407*u3*x*y2*z2 - 13.984210663495318*v*x*y2*z2 - 
     4.27806313688087*lnbara*v*x*y2*z2 + 1.059736505268818*u*v*x*y2*z2 + 
     0.36692372038764914*lnbara*u*v*x*y2*z2 - 0.7420694165018765*u2*v*x*
      y2*z2 - 0.21620742554342368*lnbara*u2*v*x*y2*z2 + 
     2.499274797319209*v2*x*y2*z2 + 0.8372694295660547*lnbara*v2*x*y2*
      z2 + 0.36293415511749816*u*v2*x*y2*z2 + 0.09272663386743893*lnbara*
      u*v2*x*y2*z2 - 1.4916718912285227*v3*x*y2*z2 - 
     0.432716958385381*lnbara*v3*x*y2*z2 + 14.046191130880398*w*x*y2*
      z2 + 4.27806313688087*lnbara*w*x*y2*z2 - 1.0675924425094043*u*w*x*
      y2*z2 - 0.36692372038764914*lnbara*u*w*x*y2*z2 + 
     0.7426221290311992*u2*w*x*y2*z2 + 0.21620742554342368*lnbara*u2*w*x*
      y2*z2 - 1.0551604965935144*v*w*x*y2*z2 - 0.35193692906048857*lnbara*
      v*w*x*y2*z2 + 0.18081274049740853*u*v*w*x*y2*z2 + 
     0.06303275716653367*lnbara*u*v*w*x*y2*z2 + 3.2714817035807937*v2*w*x*
      y2*z2 + 0.9180803814343197*lnbara*v2*w*x*y2*z2 - 
     1.4568456155246001*w2*x*y2*z2 - 0.48533250050556614*lnbara*w2*x*y2*
      z2 - 0.5429343918861453*u*w2*x*y2*z2 - 0.1557593910339726*lnbara*u*
      w2*x*y2*z2 - 3.271037633209231*v*w2*x*y2*z2 - 
     0.9126427694703488*lnbara*v*w2*x*y2*z2 + 1.492463738628857*w3*x*y2*
      z2 + 0.4272793464214101*lnbara*w3*x*y2*z2 + 
     0.05802930853773854*x2*y2*z2 - 0.005287248454508436*u*x2*y2*z2 + 
     0.00032784931577668774*u2*x2*y2*z2 - 0.005287248454508436*v*x2*y2*
      z2 + 0.0003034691554580678*u*v*x2*y2*z2 + 
     1.4101267527346482*v2*x2*y2*z2 + 0.39576075754209206*lnbara*v2*x2*
      y2*z2 - 0.005287248454508436*w*x2*y2*z2 + 
     0.0003034691554580678*u*w*x2*y2*z2 - 2.8192943376822845*v*w*x2*y2*
      z2 - 0.7915215150841841*lnbara*v*w*x2*y2*z2 + 
     1.4101267527346482*w2*x2*y2*z2 + 0.39576075754209206*lnbara*w2*x2*
      y2*z2 - 0.01113298574577242*x3*y2*z2 + 0.0004489814737672761*u*x3*
      y2*z2 - 1.0830627587062924*v*x3*y2*z2 - 0.2997685920100648*lnbara*
      v*x3*y2*z2 + 1.0839553144032048*w*x3*y2*z2 + 
     0.2997685920100648*lnbara*w*x3*y2*z2 + 0.001019248467243472*x4*y2*
      z2 - 0.251564804320496*y3*z2 + 0.03702532275275472*u*y3*z2 - 
     0.0050083201180713645*u2*y3*z2 + 0.0003728851239061164*u3*y3*z2 + 
     2.417035504771195*v*y3*z2 + 0.7884496760703545*lnbara*v*y3*z2 + 
     0.9375779785463527*u*v*y3*z2 + 0.26779586684578455*lnbara*u*v*y3*
      z2 + 0.16819984993563145*u2*v*y3*z2 + 0.046821383490061465*lnbara*
      u2*v*y3*z2 - 2.321676668163961*v2*y3*z2 - 
     0.704896053042102*lnbara*v2*y3*z2 + 0.038158053864272005*u*v2*y3*
      z2 + 0.011961036953974104*lnbara*u*v2*y3*z2 + 
     0.28220067646190156*v3*y3*z2 + 0.09025951520398619*lnbara*v3*y3*
      z2 - 2.339296200037759*w*y3*z2 - 0.7884496760703545*lnbara*w*y3*
      z2 - 0.9468503021957163*u*w*y3*z2 - 0.26779586684578455*lnbara*u*w*
      y3*z2 - 0.16757153566204508*u2*w*y3*z2 - 
     0.046821383490061465*lnbara*u2*w*y3*z2 + 2.9845483465315206*v*w*y3*
      z2 + 0.873539503307877*lnbara*v*w*y3*z2 - 0.7548466365170377*u*v*w*
      y3*z2 - 0.21880709774223123*lnbara*u*v*w*y3*z2 - 
     0.10578881199195397*v2*w*y3*z2 - 0.04001243623425464*lnbara*v2*w*y3*
      z2 - 0.6788052052736328*w2*y3*z2 - 0.16864345026577507*lnbara*w2*
      y3*z2 + 0.717639217730022*u*w2*y3*z2 + 0.20684606078825712*lnbara*
      u*w2*y3*z2 + 0.029077792864669175*v*w2*y3*z2 + 
     0.011972114859594282*lnbara*v*w2*y3*z2 - 0.20393804085263473*w3*y3*
      z2 - 0.06221919382932583*lnbara*w3*y3*z2 + 
     0.06395157977258972*x*y3*z2 - 0.005673921854933901*u*x*y3*z2 + 
     0.00034533361905439837*u2*x*y3*z2 + 4.23815521421725*v*x*y3*z2 + 
     1.229468487533898*lnbara*v*x*y3*z2 - 0.46795439814148526*u*v*x*y3*
      z2 - 0.14200807750295125*lnbara*u*v*x*y3*z2 - 
     0.10578881199195397*v2*x*y3*z2 - 0.04001243623425464*lnbara*v2*x*y3*
      z2 - 4.2497906793289655*w*x*y3*z2 - 1.229468487533898*lnbara*w*x*y3*
      z2 + 0.4686008650126944*u*w*x*y3*z2 + 0.14200807750295125*lnbara*u*w*
      x*y3*z2 + 0.07777103749659893*v*w*x*y3*z2 + 
     0.028040321374660362*lnbara*v*w*x*y3*z2 + 0.029077792864669175*w2*x*
      y3*z2 + 0.011972114859594282*lnbara*w2*x*y3*z2 - 
     0.01113298574577242*x2*y3*z2 + 0.00044357422314508923*u*x2*y3*z2 + 
     0.0004489814737672761*v*x2*y3*z2 + 0.0004489814737672761*w*x2*y3*
      z2 + 0.0009524091176864817*x3*y3*z2 + 0.08755029317854246*y4*z2 - 
     0.007317810733171885*u*y4*z2 + 0.00042934540544391214*u2*y4*z2 - 
     0.2759336414712597*v*y4*z2 - 0.09277616733835867*lnbara*v*y4*z2 + 
     0.04382088794664975*u*v*y4*z2 + 0.013022653225572441*lnbara*u*v*y4*
      z2 + 0.20047304487804693*v2*y4*z2 + 0.05785069933318*lnbara*v2*y4*
      z2 + 0.260373564991028*w*y4*z2 + 0.09277616733835867*lnbara*w*y4*
      z2 - 0.04301078480890973*u*w*y4*z2 - 0.013022653225572441*lnbara*u*w*
      y4*z2 - 0.4807266928857714*v*w*y4*z2 - 0.1356427800167511*lnbara*v*
      w*y4*z2 + 0.28165966173252016*w2*y4*z2 + 
     0.07779208068357109*lnbara*w2*y4*z2 - 0.013136621448171954*x*y4*
      z2 + 0.000507581743944034*u*x*y4*z2 - 0.747040617041446*v*x*y4*
      z2 - 0.21034811390666633*lnbara*v*x*y4*z2 + 
     0.7480915272662646*w*x*y4*z2 + 0.21034811390666633*lnbara*w*x*y4*
      z2 + 0.001019248467243472*x2*y4*z2 - 0.018929896217160017*y5*z2 + 
     0.0006870207501448052*u*y5*z2 + 0.06988272110288132*v*y5*z2 + 
     0.021403950235092197*lnbara*v*y5*z2 - 0.06841032198704927*w*y5*z2 - 
     0.021403950235092197*lnbara*w*y5*z2 + 0.001263472670057093*x*y5*z2 + 
     0.0018904202297008643*y6*z2 - 2.2670918716867487*z3 + 
     0.6858555690183518*u*z3 - 0.251564804320496*u2*z3 + 
     0.0752590109961792*u3*z3 - 0.014812198869903107*u4*z3 + 
     0.001388160767640295*u5*z3 - 0.5311674638752799*v*z3 - 
     0.4329554475790071*lnbara*v*z3 - 8.70945468065623*u*v*z3 - 
     2.843556800957142*lnbara*u*v*z3 - 1.6011517276791973*u2*v*z3 - 
     0.46172402438324356*lnbara*u2*v*z3 - 1.576189905969709*u3*v*z3 - 
     0.465804859856524*lnbara*u3*v*z3 + 0.015644250220599377*u4*v*z3 + 
     0.006838625664216695*lnbara*u4*v*z3 + 0.8972500249744643*v2*z3 + 
     0.3500794053947276*lnbara*v2*z3 + 7.880191664548825*u*v2*z3 + 
     2.6748784087603763*lnbara*u*v2*z3 + 0.9841419128221275*u2*v2*z3 + 
     0.23760579052794478*lnbara*u2*v2*z3 + 0.43479804170814224*u3*v2*
      z3 + 0.1344756922786936*lnbara*u3*v2*z3 + 
     0.4593564384590013*v3*z3 + 0.09852674249500254*lnbara*v3*z3 - 
     3.462967340849832*u*v3*z3 - 1.1204221927730833*lnbara*u*v3*z3 + 
     0.0015185718493600246*u2*v3*z3 + 0.005606486444442812*lnbara*u2*v3*
      z3 + 0.17761627523096088*v4*z3 + 0.05371322421188824*lnbara*v4*
      z3 + 0.4131831056920087*u*v4*z3 + 0.13608853103686827*lnbara*u*v4*
      z3 + 0.01134547383834047*v5*z3 + 0.002283398903340656*lnbara*v5*
      z3 + 1.7170463437119636*w*z3 + 0.4329554475790071*lnbara*w*z3 + 
     8.335636464746663*u*w*z3 + 2.843556800957142*lnbara*u*w*z3 + 
     1.7021286302045417*u2*w*z3 + 0.46172402438324356*lnbara*u2*w*z3 + 
     1.5576572952608156*u3*w*z3 + 0.465804859856524*lnbara*u3*w*z3 - 
     0.01399123854239156*u4*w*z3 - 0.006838625664216695*lnbara*u4*w*z3 + 
     4.570769251234671*v*w*z3 + 1.7115875434794767*lnbara*v*w*z3 - 
     1.485347767785549*u*v*w*z3 - 0.6728643192060604*lnbara*u*v*w*z3 + 
     2.240193096445929*u2*v*w*z3 + 0.7375497739915498*lnbara*u2*v*w*z3 - 
     0.07656531816033134*u3*v*w*z3 - 0.03634902416016519*lnbara*u3*v*w*
      z3 - 10.997725885446338*v2*w*z3 - 3.557746295983214*lnbara*v2*w*
      z3 + 2.9792615221240184*u*v2*w*z3 + 0.9000686333403095*lnbara*u*v2*
      w*z3 - 1.8066447210612555*u2*v2*w*z3 - 0.5331767926648573*lnbara*
      u2*v2*w*z3 + 2.2465725155547824*v3*w*z3 + 
     0.7870789002790479*lnbara*v3*w*z3 + 0.1244773390895362*u*v3*w*z3 + 
     0.024916275986332885*lnbara*u*v3*w*z3 - 0.5665614064319183*v4*w*z3 - 
     0.1764702004496186*lnbara*v4*w*z3 - 6.021984802397401*w2*z3 - 
     2.0616669488742043*lnbara*w2*z3 - 6.264516709859727*u*w2*z3 - 
     2.002014089554316*lnbara*u*w2*z3 - 3.2461502369868342*u2*w2*z3 - 
     0.9751555645194946*lnbara*u2*w2*z3 - 0.35640766524496237*u3*w2*
      z3 - 0.09812666811852842*lnbara*u3*w2*z3 + 
     9.909152497593267*v*w2*z3 + 3.0912384401060025*lnbara*v*w2*z3 - 
     5.305139799956746*u*v*w2*z3 - 1.5393343490108804*lnbara*u*v*w2*z3 + 
     1.5675237969406215*u2*v*w2*z3 + 0.4579464223436356*lnbara*u2*v*w2*
      z3 + 0.3997513066521812*v2*w2*z3 + 0.08307962011998349*lnbara*v2*
      w2*z3 + 0.5497959040120096*u*v2*w2*z3 + 0.17704276038055036*lnbara*
      u*v2*w2*z3 + 0.461488349439593*v3*w2*z3 + 
     0.12645487928146082*lnbara*v3*w2*z3 + 0.817892698661721*w3*z3 + 
     0.36798111338220896*lnbara*w3*z3 + 5.761241784590613*u*w3*z3 + 
     1.759687908443654*lnbara*u*w3*z3 + 0.23971655435506595*u2*w3*z3 + 
     0.06962388387677895*lnbara*u2*w3*z3 - 1.6389294857674492*v*w3*z3 - 
     0.5238484235184111*lnbara*v*w3*z3 - 0.30529725405798797*u*v*w3*z3 - 
     0.10319808585533102*lnbara*u*v*w3*z3 - 0.8243258774524206*v2*w3*
      z3 - 0.23041500895027525*lnbara*v2*w3*z3 - 
     1.2251346623182928*w4*z3 - 0.4000233210925086*lnbara*w4*z3 - 
     0.7794687099225679*u*w4*z3 - 0.23484948154842047*lnbara*u*w4*z3 + 
     0.8263404688000965*v*w4*z3 + 0.2438589036761638*lnbara*v*w4*z3 + 
     0.0956649502705806*w5*z3 + 0.03428802753892857*lnbara*w5*z3 + 
     0.6858555690183518*x*z3 - 0.162591133426937*u*x*z3 + 
     0.03949367079133293*u2*x*z3 - 0.006882314713446356*u3*x*z3 + 
     0.0005960914163116732*u4*x*z3 + 11.250037884026336*v*x*z3 + 
     3.9890155563258416*lnbara*v*x*z3 + 0.5861690638750913*u*v*x*z3 + 
     0.04824677725141553*lnbara*u*v*x*z3 + 2.7450669338987366*u2*v*x*z3 + 
     0.8803435778848646*lnbara*u2*v*x*z3 - 0.06800667812701873*u3*v*x*
      z3 - 0.029360220026995808*lnbara*u3*v*x*z3 - 
     9.326716278242827*v2*x*z3 - 3.107847444868102*lnbara*v2*x*z3 + 
     2.4434643394917717*u*v2*x*z3 + 0.722704550671465*lnbara*u*v2*x*z3 - 
     1.4446667880342972*u2*v2*x*z3 - 0.43070638308913284*lnbara*u2*v2*x*
      z3 + 2.2465725155547824*v3*x*z3 + 0.7870789002790479*lnbara*v3*x*
      z3 + 0.1244773390895362*u*v3*x*z3 + 0.024916275986332885*lnbara*u*
      v3*x*z3 - 0.5397225167390112*v4*x*z3 - 0.16840858737356737*lnbara*
      v4*x*z3 - 11.553644836317881*w*x*z3 - 3.9890155563258416*lnbara*w*x*
      z3 - 0.5200656204166001*u*w*x*z3 - 0.04824677725141553*lnbara*u*w*x*
      z3 - 2.7556939387551567*u2*w*x*z3 - 0.8803435778848646*lnbara*u2*w*
      x*z3 + 0.0688751348615315*u3*w*x*z3 + 0.029360220026995808*lnbara*
      u3*w*x*z3 + 14.388317561563154*v*w*x*z3 + 4.420309235697655*lnbara*v*
      w*x*z3 - 10.915203494539247*u*v*w*x*z3 - 3.2792181994652556*lnbara*u*
      v*w*x*z3 + 2.4240248630265353*u2*v*w*x*z3 + 
     0.7115869901073909*lnbara*u2*v*w*x*z3 - 2.1068706790103*v2*w*x*z3 - 
     0.6955343577421946*lnbara*v2*w*x*z3 + 0.8761362502114731*u*v2*w*x*
      z3 + 0.296376278878668*lnbara*u*v2*w*x*z3 + 
     1.0840027554445029*v3*w*x*z3 + 0.30837662850281694*lnbara*v3*w*x*
      z3 - 4.961706764789829*w2*x*z3 - 1.3124617908295524*lnbara*w2*x*
      z3 + 8.457223146401986*u*w2*x*z3 + 2.5565136487937905*lnbara*u*w2*x*
      z3 - 0.9782558475006098*u2*w2*x*z3 - 0.28088060701825807*lnbara*u2*
      w2*x*z3 + 1.4784816196761565*v*w2*x*z3 + 0.43230388098155775*lnbara*
      v*w2*x*z3 - 0.6938884585226485*u*v*w2*x*z3 - 
     0.21809446900966986*lnbara*u*v*w2*x*z3 - 1.7364430725838955*v2*w2*x*
      z3 - 0.4913145532522245*lnbara*v2*w2*x*z3 - 
     1.6389294857674492*w3*x*z3 - 0.5238484235184111*lnbara*w3*x*z3 - 
     0.30529725405798797*u*w3*x*z3 - 0.10319808585533102*lnbara*u*w3*x*
      z3 + 1.5073534267839392*v*w3*x*z3 + 0.4300715944036431*lnbara*v*w3*
      x*z3 - 0.31318215309396386*w4*x*z3 - 0.07872508228066824*lnbara*w4*
      x*z3 - 0.251564804320496*x2*z3 + 0.03949367079133293*u*x2*z3 - 
     0.00560074458687052*u2*x2*z3 + 0.0004300153542944288*u3*x2*z3 + 
     6.336956677089462*v*x2*z3 + 1.8196275948475404*lnbara*v*x2*z3 - 
     3.5946630672038005*u*v*x2*z3 - 1.0856448829578143*lnbara*u*v*x2*z3 + 
     0.6616579679935063*u2*v*x2*z3 + 0.18951332857164516*lnbara*u2*v*x2*
      z3 - 0.2730340390527256*v2*x2*z3 - 0.09631911852193811*lnbara*v2*
      x2*z3 + 0.5346542533590023*u*v2*x2*z3 + 0.17261962424250146*lnbara*
      u*v2*x2*z3 + 0.461488349439593*v3*x2*z3 + 
     0.12645487928146082*lnbara*v3*x2*z3 - 6.2616857203946035*w*x2*z3 - 
     1.8196275948475404*lnbara*w*x2*z3 + 3.5848494417836383*u*w*x2*z3 + 
     1.0856448829578143*lnbara*u*w*x2*z3 - 0.660951971257608*u2*w*x2*
      z3 - 0.18951332857164516*lnbara*u2*w*x2*z3 + 
     0.6761564834039746*v*w*x2*z3 + 0.2078129654507111*lnbara*v*w*x2*z3 - 
     0.8428720597864838*u*v*w*x2*z3 - 0.25120658785234307*lnbara*u*v*w*x2*
      z3 - 1.7437824185418067*v2*w*x2*z3 - 0.4914395306435913*lnbara*v2*
      w*x2*z3 - 0.4179222450177246*w2*x2*z3 - 0.11149384692877298*lnbara*
      w2*x2*z3 + 0.3092023748907339*u*w2*x2*z3 + 
     0.0785869636098416*lnbara*u*w2*x2*z3 + 2.1080028169579204*v*w2*x2*
      z3 + 0.5953996603124057*lnbara*v*w2*x2*z3 - 
     0.8243258774524206*w3*x2*z3 - 0.23041500895027525*lnbara*w3*x2*
      z3 + 0.0752590109961792*x3*z3 - 0.006882314713446356*u*x3*z3 + 
     0.0004300153542944288*u2*x3*z3 - 0.006605505610793891*v*x3*z3 + 
     0.00038509593985133696*u*v*x3*z3 - 0.6281581353908926*v2*x3*z3 - 
     0.1774903697854292*lnbara*v2*x3*z3 - 0.006605505610793891*w*x3*z3 + 
     0.00038509593985133696*u*w*x3*z3 + 1.2574727801717278*v*w*x3*z3 + 
     0.3549807395708584*lnbara*v*w*x3*z3 - 0.6281581353908926*w2*x3*z3 - 
     0.1774903697854292*lnbara*w2*x3*z3 - 0.014812198869903107*x4*z3 + 
     0.0005960914163116732*u*x4*z3 + 0.38164842058355447*v*x4*z3 + 
     0.10823518328690307*lnbara*v*x4*z3 - 0.380500721540267*w*x4*z3 - 
     0.10823518328690307*lnbara*w*x4*z3 + 0.001388160767640295*x5*z3 + 
     0.6858555690183518*y*z3 - 0.15963636872636416*u*y*z3 + 
     0.03824563394210313*u2*y*z3 - 0.006605505610793891*u3*y*z3 + 
     0.0005687770966253271*u4*y*z3 - 14.456656314267676*v*y*z3 - 
     4.8057115258275305*lnbara*v*y*z3 - 3.6907580059131853*u*v*y*z3 - 
     1.0737089522063064*lnbara*u*v*y*z3 - 3.5123911436649746*u2*v*y*z3 - 
     1.041566396111375*lnbara*u2*v*y*z3 - 0.10728353612006211*u3*v*y*z3 - 
     0.025365208983699924*lnbara*u3*v*y*z3 + 5.769076336816564*v2*y*z3 + 
     2.0547100979131496*lnbara*v2*y*z3 + 2.371701525434619*u*v2*y*z3 + 
     0.6890237976926594*lnbara*u*v2*y*z3 + 0.729201859518042*u2*v2*y*
      z3 + 0.21972319960553072*lnbara*u2*v2*y*z3 - 
     3.369820769025423*v3*y*z3 - 1.0804948864701236*lnbara*v3*y*z3 - 
     0.09448932656403045*u*v3*y*z3 - 0.029331580383414237*lnbara*u*v3*y*
      z3 + 0.3734562378603602*v4*y*z3 + 0.1236750594519651*lnbara*v4*y*
      z3 + 14.150094597275556*w*y*z3 + 4.8057115258275305*lnbara*w*y*z3 + 
     3.7558333567491418*u*w*y*z3 + 1.0737089522063064*lnbara*u*w*y*z3 + 
     3.5021103597603167*u2*w*y*z3 + 1.041566396111375*lnbara*u2*w*y*z3 + 
     0.1081140594632303*u3*w*y*z3 + 0.025365208983699924*lnbara*u3*w*y*
      z3 - 0.09200933391270437*v*w*y*z3 - 0.14480531475704334*lnbara*v*w*y*
      z3 + 3.724170161382556*u*v*w*y*z3 + 1.1721828559253105*lnbara*u*v*w*y*
      z3 + 0.1296884569931269*u2*v*w*y*z3 + 0.020514943705544776*lnbara*
      u2*v*w*y*z3 + 3.8502028793420484*v2*w*y*z3 + 
     1.1170305738043862*lnbara*v2*w*y*z3 - 2.127445819261271*u*v2*w*y*
      z3 - 0.6316147687241351*lnbara*u*v2*w*y*z3 + 
     0.04710638371571019*v3*w*y*z3 + 0.006233587961624442*lnbara*v3*w*y*
      z3 - 5.574896354901597*w2*y*z3 - 1.909904783156106*lnbara*w2*y*
      z3 - 6.110264042573513*u*w2*y*z3 - 1.8612066536179699*lnbara*u*w2*y*
      z3 - 0.8578197999104673*u2*w2*y*z3 - 0.24023814331107549*lnbara*u2*
      w2*y*z3 - 6.183783910974188*v*w2*y*z3 - 1.7887689679030814*lnbara*v*
      w2*y*z3 + 1.617693856062018*u*v*w2*y*z3 + 
     0.4746900012067356*lnbara*u*v*w2*y*z3 + 0.2951846833599234*v2*w2*y*
      z3 + 0.10231487175528582*lnbara*v2*w2*y*z3 + 
     5.6819090881671865*w3*y*z3 + 1.7522332805688188*lnbara*w3*y*z3 + 
     0.6056692941957744*u*w3*y*z3 + 0.1862563479008138*lnbara*u*w3*y*z3 + 
     0.032891805121808636*v*w3*y*z3 - 0.002730352711332793*lnbara*v*w3*y*
      z3 - 0.7465338393563911*w4*y*z3 - 0.22949316645754256*lnbara*w4*y*
      z3 - 0.2298476323443832*x*y*z3 + 0.03580809539433375*u*x*y*z3 - 
     0.005041691635003074*u2*x*y*z3 + 0.00038509593985133696*u3*x*y*z3 - 
     2.3689366351444274*v*x*y*z3 - 0.726506814968872*lnbara*v*x*y*z3 + 
     2.469819197461433*u*v*x*y*z3 + 0.8025320427989695*lnbara*u*v*x*y*z3 + 
     0.18981304783567338*u2*v*x*y*z3 + 0.049895924897055216*lnbara*u2*v*x*
      y*z3 + 3.1903919237729643*v2*x*y*z3 + 0.9222524033868401*lnbara*v2*
      x*y*z3 - 2.203724440304821*u*v2*x*y*z3 - 0.6502064428550666*lnbara*u*
      v2*x*y*z3 + 0.04710638371571019*v3*x*y*z3 + 
     0.006233587961624442*lnbara*v3*x*y*z3 + 2.4387387085229473*w*x*y*z3 + 
     0.726506814968872*lnbara*w*x*y*z3 - 2.478798620600135*u*w*x*y*z3 - 
     0.8025320427989695*lnbara*u*w*x*y*z3 - 0.1891735970499443*u2*w*x*y*
      z3 - 0.049895924897055216*lnbara*u2*w*x*y*z3 - 
     8.881204021839554*v*w*x*y*z3 - 2.6069786638032433*lnbara*v*w*x*y*z3 + 
     3.3873280227068086*u*v*w*x*y*z3 + 0.9769151192502976*lnbara*u*v*w*x*y*
      z3 + 0.9014731642677928*v2*w*x*y*z3 + 0.28985320078620536*lnbara*v2*
      w*x*y*z3 + 5.676870448679409*w2*x*y*z3 + 1.684726260416403*lnbara*
      w2*x*y*z3 - 1.182692665484111*u*w2*x*y*z3 - 
     0.32670867639523105*lnbara*u*w2*x*y*z3 - 0.9801494429301945*v*w2*x*y*
      z3 - 0.293356436036497*lnbara*v*w2*x*y*z3 + 
     0.032891805121808636*w3*x*y*z3 - 0.002730352711332793*lnbara*w3*x*y*
      z3 + 0.06395157977258972*x2*y*z3 - 0.005855071071261368*u*x2*y*
      z3 + 0.0003652020446131235*u2*x2*y*z3 - 6.026774148018949*v*x2*y*
      z3 - 1.7632715691466476*lnbara*v*x2*y*z3 + 
     0.731117931401122*u*v*x2*y*z3 + 0.21288835092447814*lnbara*u*v*x2*y*
      z3 + 0.501248699467762*v2*x2*y*z3 + 0.1627718847199586*lnbara*v2*
      x2*y*z3 + 6.015319832123562*w*x2*y*z3 + 1.7632715691466476*lnbara*w*
      x2*y*z3 - 0.7304529490550218*u*w*x2*y*z3 - 
     0.21288835092447814*lnbara*u*w*x2*y*z3 - 0.5309427518886998*v*w*x2*y*
      z3 - 0.16015438246526315*lnbara*v*w*x2*y*z3 + 
     0.030715686889802068*w2*x2*y*z3 - 0.0026175022546954534*lnbara*w2*
      x2*y*z3 - 0.011927105098099399*x3*y*z3 + 0.00048336079466143985*u*
      x3*y*z3 + 0.00047310410784989136*v*x3*y*z3 + 
     0.00047310410784989136*w*x3*y*z3 + 0.0010740897315456516*x4*y*z3 - 
     0.251564804320496*y2*z3 + 0.03824563394210313*u*y2*z3 - 
     0.005291090684199614*u2*y2*z3 + 0.00039954507731219854*u3*y2*z3 + 
     2.8467947898195387*v*y2*z3 + 0.8710102224533081*lnbara*v*y2*z3 - 
     2.2828692800346633*u*v*y2*z3 - 0.6947964005893216*lnbara*u*v*y2*z3 - 
     0.15198537243470955*u2*v*y2*z3 - 0.04014533812336501*lnbara*u2*v*y2*
      z3 + 0.32522417184198954*v2*y2*z3 + 0.06989084748944815*lnbara*v2*
      y2*z3 + 0.6591783435173679*u*v2*y2*z3 + 0.20062980696935676*lnbara*
      u*v2*y2*z3 + 0.19122371263757587*v3*y2*z3 + 
     0.05397940765728412*lnbara*v3*y2*z3 - 2.770275796275451*w*y2*z3 - 
     0.8710102224533081*lnbara*w*y2*z3 + 2.2733271985354553*u*w*y2*z3 + 
     0.6947964005893216*lnbara*u*w*y2*z3 + 0.15265057835965268*u2*w*y2*
      z3 + 0.04014533812336501*lnbara*u2*w*y2*z3 + 
     1.046392947967706*v*w*y2*z3 + 0.35193692906048857*lnbara*v*w*y2*z3 - 
     0.18025531671901043*u*v*w*y2*z3 - 0.06303275716653367*lnbara*u*v*w*y2*
      z3 - 1.4916718912285227*v2*w*y2*z3 - 0.432716958385381*lnbara*v2*w*
      y2*z3 - 1.3869981182997964*w2*y2*z3 - 0.4218277765499367*lnbara*
      w2*y2*z3 - 0.4779542480040674*u*w2*y2*z3 - 
     0.13759704980282306*lnbara*u*w2*y2*z3 + 1.492463738628857*v*w2*y2*
      z3 + 0.4272793464214101*lnbara*v*w2*y2*z3 - 
     0.19054563887772494*w3*y2*z3 - 0.04854179569331324*lnbara*w3*y2*
      z3 + 0.06395157977258972*x*y2*z3 - 0.005780394040453321*u*x*y2*
      z3 + 0.0003568540057622896*u2*x*y2*z3 + 3.420468192644727*v*x*y2*
      z3 + 1.0593983869002586*lnbara*v*x*y2*z3 + 
     0.3069925110386899*u*v*x*y2*z3 + 0.0810097794355468*lnbara*u*v*x*y2*
      z3 - 1.3871873506084558*v2*x*y2*z3 - 0.4071365920310302*lnbara*v2*
      x*y2*z3 - 3.431997185570922*w*x*y2*z3 - 1.0593983869002586*lnbara*w*
      x*y2*z3 - 0.3063351018757037*u*w*x*y2*z3 - 
     0.0810097794355468*lnbara*u*w*x*y2*z3 + 2.0929117260759176*v*w*x*y2*
      z3 + 0.5923170046701335*lnbara*v*w*x*y2*z3 - 
     0.7046868197766329*w2*x*y2*z3 - 0.18518041263910326*lnbara*w2*x*y2*
      z3 - 0.01113298574577242*x2*y2*z3 + 0.0004489814737672761*u*x2*y2*
      z3 + 1.0839553144032048*v*x2*y2*z3 + 0.2997685920100648*lnbara*v*
      x2*y2*z3 - 1.0830627587062924*w*x2*y2*z3 - 
     0.2997685920100648*lnbara*w*x2*y2*z3 + 0.0009524091176864817*x3*y2*
      z3 + 0.0752590109961792*y3*z3 - 0.006605505610793891*u*y3*z3 + 
     0.00039954507731219854*u2*y3*z3 - 2.4157548822081845*v*y3*z3 - 
     0.7222688114635434*lnbara*v*y3*z3 - 0.10092543442474523*u*v*y3*z3 - 
     0.025545381796987655*lnbara*u*v*y3*z3 + 0.28220067646190156*v2*y3*
      z3 + 0.09025951520398619*lnbara*v2*y3*z3 + 
     2.402267061883944*w*y3*z3 + 0.7222688114635434*lnbara*w*y3*z3 + 
     0.10166794959991486*u*w*y3*z3 + 0.025545381796987655*lnbara*u*w*y3*
      z3 - 0.07704797923780887*v*w*y3*z3 - 0.028040321374660362*lnbara*v*w*
      y3*z3 - 0.20393804085263473*w2*y3*z3 - 0.06221919382932583*lnbara*
      w2*y3*z3 - 0.011927105098099399*x*y3*z3 + 
     0.00047310410784989136*u*x*y3*z3 - 0.21637657479006892*v*x*y3*z3 - 
     0.06469446777014416*lnbara*v*x*y3*z3 + 0.21733303969258025*w*x*y3*
      z3 + 0.06469446777014416*lnbara*w*x*y3*z3 + 
     0.0009524091176864817*x2*y3*z3 - 0.014812198869903107*y4*z3 + 
     0.0005687770966253271*u*y4*z3 + 0.23650678939377565*v*y4*z3 + 
     0.06767441939812754*lnbara*v*y4*z3 - 0.2353317760308018*w*y4*z3 - 
     0.06767441939812754*lnbara*w*y4*z3 + 0.0010740897315456516*x*y4*z3 + 
     0.001388160767640295*y5*z3 + 2.0230034428855013*z4 - 
     0.386061737342387*u*z4 + 0.08755029317854246*u2*z4 - 
     0.014812198869903107*u3*z4 + 0.0012633787101392036*u4*z4 + 
     1.9188684163037728*v*z4 + 0.7450820516264817*lnbara*v*z4 + 
     4.540055219513269*u*v*z4 + 1.450672032069528*lnbara*u*v*z4 + 
     1.122430115235894*u2*v*z4 + 0.3173310228787825*lnbara*u2*v*z4 + 
     0.2239680885196098*u3*v*z4 + 0.06514335645328273*lnbara*u3*v*z4 + 
     0.651338693005845*v2*z4 + 0.14754669762714998*lnbara*v2*z4 - 
     3.7835481298107045*u*v2*z4 - 1.1938747004904564*lnbara*u*v2*z4 + 
     0.05725447402071905*u2*v2*z4 + 0.021576912362776347*lnbara*u2*v2*
      z4 + 0.17761627523096088*v3*z4 + 0.05371322421188824*lnbara*v3*
      z4 + 0.4131831056920087*u*v3*z4 + 0.13608853103686827*lnbara*u*v3*
      z4 + 0.021325233798974344*v4*z4 + 0.004996244651440274*lnbara*v4*
      z4 - 2.581767811748015*w*z4 - 0.7450820516264817*lnbara*w*z4 - 
     4.408138300869153*u*w*z4 - 1.450672032069528*lnbara*u*w*z4 - 
     1.1428845474172378*u2*w*z4 - 0.3173310228787825*lnbara*u2*w*z4 - 
     0.2223252216914388*u3*w*z4 - 0.06514335645328273*lnbara*u3*w*z4 - 
     6.0865303594656925*v*w*z4 - 1.9510966301046395*lnbara*v*w*z4 + 
     2.583274807875346*u*v*w*z4 + 0.814048966327527*lnbara*u*v*w*z4 - 
     1.0214921534358323*u2*v*w*z4 - 0.3025359736703866*lnbara*u2*v*w*z4 + 
     2.4334094721170496*v2*w*z4 + 0.816632950371704*lnbara*v2*w*z4 + 
     0.10307841593891565*u*v2*w*z4 + 0.018172273278693238*lnbara*u*v2*w*
      z4 - 0.5397225167390112*v3*w*z4 - 0.16840858737356737*lnbara*v3*w*
      z4 + 5.626060910791232*w2*z4 + 1.8035499324774895*lnbara*w2*z4 + 
     1.1739473090123895*u*w2*z4 + 0.3798257341629295*lnbara*u*w2*z4 + 
     0.9661938550317448*u2*w2*z4 + 0.2809590613076102*lnbara*u2*w2*z4 - 
     1.4226442649834423*v*w2*z4 - 0.4703228534910837*lnbara*v*w2*z4 + 
     0.2656466906715195*u*v*w2*z4 + 0.08058867723285898*lnbara*u*v*w2*
      z4 + 0.39505594064956606*v2*w2*z4 + 0.11018942250071202*lnbara*v2*
      w2*z4 - 1.2251346623182928*w3*z4 - 0.4000233210925086*lnbara*w3*
      z4 - 0.7794687099225679*u*w3*z4 - 0.23484948154842047*lnbara*u*w3*
      z4 - 0.31318215309396386*v*w3*z4 - 0.07872508228066824*lnbara*v*w3*
      z4 + 0.43991353613455403*w4*z4 + 0.1319480025020833*lnbara*w4*z4 - 
     0.386061737342387*x*z4 + 0.056971794565663605*u*x*z4 - 
     0.007867552944290607*u2*x*z4 + 0.0005960914163116732*u3*x*z4 - 
     10.094036110016523*v*x*z4 - 3.28316404172915*lnbara*v*x*z4 + 
     1.0213274392385536*u*v*x*z4 + 0.336346527741606*lnbara*u*v*x*z4 - 
     0.8972167925745769*u2*v*x*z4 - 0.26783993094333697*lnbara*u2*v*x*
      z4 + 2.2331993759180806*v2*x*z4 + 0.7744184574067065*lnbara*v2*x*
      z4 + 0.08267922567088919*u*v2*x*z4 + 0.01813757575979942*lnbara*u*
      v2*x*z4 - 0.5665614064319183*v3*x*z4 - 0.1764702004496186*lnbara*
      v3*x*z4 + 10.200182207413253*w*x*z4 + 3.28316404172915*lnbara*w*x*
      z4 - 1.034655985722088*u*w*x*z4 - 0.336346527741606*lnbara*u*w*x*
      z4 + 0.8981582264276017*u2*w*x*z4 + 0.26783993094333697*lnbara*u2*w*
      x*z4 - 2.2689406137217927*v*w*x*z4 - 0.7108966463767397*lnbara*v*w*x*
      z4 + 1.0002685704936713*u*v*w*x*z4 + 0.30624466399171646*lnbara*u*v*w*
      x*z4 + 1.0094259951122324*v2*w*x*z4 + 0.2874207900901255*lnbara*v2*
      w*x*z4 + 0.01586781553997586*w2*x*z4 - 0.06352181102996676*lnbara*
      w2*x*z4 - 1.0816608287503775*u*w2*x*z4 - 0.32438223975151587*lnbara*
      u*w2*x*z4 - 1.267405871830795*v*w2*x*z4 - 
     0.35480949331667067*lnbara*v*w2*x*z4 + 0.8263404688000965*w3*x*z4 + 
     0.2438589036761638*lnbara*w3*x*z4 + 0.08755029317854246*x2*z4 - 
     0.007867552944290607*u*x2*z4 + 0.000487363420148127*u2*x2*z4 - 
     0.7906870068555788*v*x2*z4 - 0.2013783664993635*lnbara*v*x2*z4 + 
     0.3281593662661477*u*v*x2*z4 + 0.09992430356434646*lnbara*u*v*x2*
      z4 + 0.3702307156554167*v2*x2*z4 + 0.10136840292874362*lnbara*v2*
      x2*z4 + 0.7756766725864659*w*x2*z4 + 0.2013783664993635*lnbara*w*
      x2*z4 - 0.3272951320069847*u*w*x2*z4 - 0.09992430356434646*lnbara*u*
      w*x2*z4 - 0.9508002022690504*v*w*x2*z4 - 0.2673286182958272*lnbara*
      v*w*x2*z4 + 0.581863351202302*w2*x2*z4 + 
     0.1659602153670836*lnbara*w2*x2*z4 - 0.014812198869903107*x3*z4 + 
     0.0005960914163116732*u*x3*z4 - 0.380500721540267*v*x3*z4 - 
     0.10823518328690307*lnbara*v*x3*z4 + 0.38164842058355447*w*x3*z4 + 
     0.10823518328690307*lnbara*w*x3*z4 + 0.0012633787101392036*x4*z4 - 
     0.386061737342387*y*z4 + 0.056260781045097456*u*y*z4 - 
     0.007692523535941066*u2*y*z4 + 0.0005789219466621661*u3*y*z4 + 
     8.072482666125286*v*y*z4 + 2.6276787546704745*lnbara*v*y*z4 + 
     2.0086611123781046*u*v*y*z4 + 0.5668452509261017*lnbara*u*v*y*z4 + 
     0.590076818257424*u2*v*y*z4 + 0.17202385787388583*lnbara*u2*v*y*z4 - 
     2.872858759697617*v2*y*z4 - 0.9340060015696775*lnbara*v2*y*z4 - 
     0.13345592306181395*u*v2*y*z4 - 0.03997846452824747*lnbara*u*v2*y*
      z4 + 0.3734562378603602*v3*y*z4 + 0.1236750594519651*lnbara*v3*y*
      z4 - 7.96562555520799*w*y*z4 - 2.6276787546704745*lnbara*w*y*z4 - 
     2.0218463214313056*u*w*y*z4 - 0.5668452509261017*lnbara*u*w*y*z4 - 
     0.5891572387478903*u2*w*y*z4 - 0.17202385787388583*lnbara*u2*w*y*
      z4 + 1.9788571739028187*v*w*y*z4 + 0.5910023418671081*lnbara*v*w*y*
      z4 - 1.497715134559773*u*v*w*y*z4 - 0.43755320330470776*lnbara*u*v*w*
      y*z4 - 0.09314373993962562*v2*w*y*z4 - 0.033291399122784844*lnbara*
      v2*w*y*z4 + 0.8738097966923795*w2*y*z4 + 0.34300365970256935*lnbara*
      w2*y*z4 + 1.632450766309338*u*w2*y*z4 + 0.47753166783295525*lnbara*
      u*w2*y*z4 + 0.4680668096248443*v*w2*y*z4 + 
     0.13910950612836231*lnbara*v*w2*y*z4 - 0.7465338393563911*w3*y*z4 - 
     0.22949316645754256*lnbara*w3*y*z4 + 0.08203160229248424*x*y*z4 - 
     0.007320613539132591*u*x*y*z4 + 0.0004507102322422891*u2*x*y*z4 + 
     2.495271539467373*v*x*y*z4 + 0.7051754521652789*lnbara*v*x*y*z4 - 
     0.9763863300273927*u*v*x*y*z4 - 0.2916555967133236*lnbara*u*v*x*y*z4 - 
     0.06439562415041984*v2*x*y*z4 - 0.022967627645177263*lnbara*v2*x*y*
      z4 - 2.5094652412675753*w*x*y*z4 - 0.7051754521652789*lnbara*w*x*y*
      z4 + 0.977194806205722*u*w*x*y*z4 + 0.2916555967133236*lnbara*u*w*x*y*
      z4 + 0.6318226520207785*v*w*x*y*z4 + 0.19318619266094705*lnbara*v*w*x*
      y*z4 - 0.566190372464217*w2*x*y*z4 - 0.1702185650157698*lnbara*w2*x*
      y*z4 - 0.013136621448171954*x2*y*z4 + 0.000527909826103984*u*x2*y*
      z4 + 0.663692907110261*v*x2*y*z4 + 0.19636416180684646*lnbara*v*x2*
      y*z4 - 0.6626623249676024*w*x2*y*z4 - 0.19636416180684646*lnbara*w*
      x2*y*z4 + 0.0010740897315456516*x3*y*z4 + 
     0.08755029317854246*y2*z4 - 0.007692523535941066*u*y2*z4 + 
     0.0004679400724052549*u2*y2*z4 - 0.32019128431906846*v*y2*z4 - 
     0.10788365521657699*lnbara*v*y2*z4 + 0.3611306117059756*u*v*y2*z4 + 
     0.10812164650482199*lnbara*u*v*y2*z4 + 0.11854588979479715*v2*y2*
      z4 + 0.03415403184344883*lnbara*v2*y2*z4 + 
     0.305005920641606*w*y2*z4 + 0.10788365521657699*lnbara*w*y2*z4 - 
     0.3602833223629141*u*w*y2*z4 - 0.10812164650482199*lnbara*u*w*y2*
      z4 - 0.6823722786029639*v*w*y2*z4 - 0.19655624712804137*lnbara*v*w*
      y2*z4 + 0.5651566216606796*w2*y2*z4 + 0.16240221528459253*lnbara*
      w2*y2*z4 - 0.013136621448171954*x*y2*z4 + 
     0.0005230003987145713*u*x*y2*z4 - 1.1411946582446566*v*x*y2*z4 - 
     0.33097644191339387*lnbara*v*x*y2*z4 + 1.1422301498147045*w*x*y2*
      z4 + 0.33097644191339387*lnbara*w*x*y2*z4 + 
     0.001019248467243472*x2*y2*z4 - 0.014812198869903107*y3*z4 + 
     0.0005789219466621661*u*y3*z4 + 0.3249190900404754*v*y3*z4 + 
     0.09733306860121325*lnbara*v*y3*z4 - 0.3237542215275384*w*y3*z4 - 
     0.09733306860121325*lnbara*w*y3*z4 + 0.0010740897315456516*x*y3*z4 + 
     0.0012633787101392036*y4*z4 - 1.2541858328141202*z5 + 
     0.1475870262830398*u*z5 - 0.018929896217160017*u2*z5 + 
     0.001388160767640295*u3*z5 - 0.6982508102158693*v*z5 - 
     0.2748859330079916*lnbara*v*z5 - 1.8742275538737843*u*v*z5 - 
     0.5730994907821886*lnbara*u*v*z5 - 0.11604224021663521*u2*v*z5 - 
     0.03151257404486913*lnbara*u2*v*z5 + 0.03085436595372382*v2*z5 + 
     0.016119245278351776*lnbara*v2*z5 + 0.47543762922904986*u*v2*z5 + 
     0.14922477962881076*lnbara*u*v2*z5 + 0.01134547383834047*v3*z5 + 
     0.002283398903340656*lnbara*v3*z5 + 0.9506587243582331*w*z5 + 
     0.2748859330079916*lnbara*w*z5 + 1.84547571556521*u*w*z5 + 
     0.5730994907821886*lnbara*u*w*z5 + 0.11799273363683711*u2*w*z5 + 
     0.03151257404486913*lnbara*u2*w*z5 + 1.5716677873215597*v*w*z5 + 
     0.5030836809260285*lnbara*v*w*z5 - 0.15605883003169002*u*v*w*z5 - 
     0.05503807613937338*lnbara*u*v*w*z5 - 0.5247428948621341*v2*w*z5 - 
     0.16081753949969196*lnbara*v2*w*z5 - 1.6435396981777775*w2*z5 - 
     0.5192029262043804*lnbara*w2*z5 - 0.31687310350773107*u*w2*z5 - 
     0.09418670348943739*lnbara*u*w2*z5 + 0.4211542429001266*v*w2*z5 + 
     0.12424611305742275*lnbara*v*w2*z5 + 0.0956649502705806*w3*z5 + 
     0.03428802753892857*lnbara*w3*z5 + 0.1475870262830398*x*z5 - 
     0.012364843466299529*u*x*z5 + 0.0007424815539584533*u2*x*z5 + 
     3.2110778233847816*v*x*z5 + 1.0179245136226938*lnbara*v*x*z5 - 
     0.01449065568595044*u*v*x*z5 - 0.010275898356323341*lnbara*u*v*x*z5 - 
     0.46033723691317213*v2*x*z5 - 0.14606708002401708*lnbara*v2*x*z5 - 
     3.234078310073586*w*x*z5 - 1.0179245136226938*lnbara*w*x*z5 + 
     0.01575761553644318*u*w*x*z5 + 0.010275898356323341*lnbara*u*w*x*z5 + 
     0.6108819098365138*v*w*x*z5 + 0.1761800865537642*lnbara*v*w*x*z5 - 
     0.14867177025989595*w2*x*z5 - 0.030113006529747105*lnbara*w2*x*z5 - 
     0.018929896217160017*x2*z5 + 0.0007424815539584533*u*x2*z5 + 
     0.2301351338116376*v*x2*z5 + 0.05887745268544395*lnbara*v*x2*z5 - 
     0.22871819549961922*w*x2*z5 - 0.05887745268544395*lnbara*w*x2*z5 + 
     0.001388160767640295*x3*z5 + 0.1475870262830398*y*z5 - 
     0.012249129070819405*u*y*z5 + 0.0007299175618735945*u2*y*z5 - 
     3.248331013993099*v*y*z5 - 1.0069202481453856*lnbara*v*y*z5 - 
     0.23401253231882208*u*v*y*z5 - 0.06359410612179822*lnbara*u*v*y*z5 + 
     0.3612787107357604*v2*y*z5 + 0.11818205016461883*lnbara*v2*y*z5 + 
     3.2252148129088143*w*y*z5 + 1.0069202481453856*lnbara*w*y*z5 + 
     0.23526924787765785*u*w*y*z5 + 0.06359410612179822*lnbara*u*w*y*z5 - 
     0.09961325470566673*v*w*y*z5 - 0.03303181019568392*lnbara*v*w*y*z5 - 
     0.2597697450829062*w2*y*z5 - 0.08515023996893491*lnbara*w2*y*z5 - 
     0.018000480690589078*x*y*z5 + 0.0007017792893363362*u*x*y*z5 - 
     0.2133826221782255*v*x*y*z5 - 0.0583092465218906*lnbara*v*x*y*z5 + 
     0.2147385756550609*w*x*y*z5 + 0.0583092465218906*lnbara*w*x*y*z5 + 
     0.001263472670057093*x2*y*z5 - 0.018929896217160017*y2*z5 + 
     0.0007299175618735945*u*y2*z5 + 0.11415571194993857*v*y2*z5 + 
     0.03339748565253056*lnbara*v*y2*z5 - 0.11272620964583531*w*y2*z5 - 
     0.03339748565253056*lnbara*w*y2*z5 + 0.001263472670057093*x*y2*z5 + 
     0.001388160767640295*y3*z5 + 0.5112314175751462*z6 - 
     0.0339914082143444*u*z6 + 0.0018904202297008643*u2*z6 + 
     0.27910097428080544*v*z6 + 0.09609081997302192*lnbara*v*z6 + 
     0.2599997989809029*u*v*z6 + 0.07776386003863338*lnbara*u*v*z6 + 
     0.013912167390291797*v2*z6 + 0.0027513912903176212*lnbara*v2*z6 - 
     0.3370863430138181*w*z6 - 0.09609081997302192*lnbara*w*z6 - 
     0.25711383486763884*u*w*z6 - 0.07776386003863338*lnbara*u*w*z6 - 
     0.2866167180827836*v*w*z6 - 0.08634183804336616*lnbara*v*w*z6 + 
     0.2767842038860737*w2*z6 + 0.08359044675304854*lnbara*w2*z6 - 
     0.0339914082143444*x*z6 + 0.0012376963517727734*u*x*z6 - 
     0.5222803748590084*v*x*z6 - 0.15852336677018297*lnbara*v*x*z6 + 
     0.5245800234274332*w*x*z6 + 0.15852336677018297*lnbara*w*x*z6 + 
     0.0018904202297008643*x2*z6 - 0.0339914082143444*y*z6 + 
     0.001228493995858818*u*y*z6 + 0.47407842963740254*v*y*z6 + 
     0.14311605655840995*lnbara*v*y*z6 - 0.4717695787130639*w*y*z6 - 
     0.14311605655840995*lnbara*w*y*z6 + 0.0018148095406981609*x*y*z6 + 
     0.0018904202297008643*y2*z6 - 0.12321474882489236*z7 + 
     0.0035512018991495737*u*z7 - 0.033158063533986065*v*z7 - 
     0.01108690204816448*lnbara*v*z7 + 0.039205434782531995*w*z7 + 
     0.01108690204816448*lnbara*w*z7 + 0.0035512018991495737*x*z7 + 
     0.0035512018991495737*y*z7 + 0.013314221484862214*z8;
 
  ttothei = 1.0L;
  result = Hexp[0];

  for (i=1; i<nTerms; i++) {
    ttothei *= tinit;
    result += Hexp[i]*ttothei;
  }
  foo->value = result;

  ttothei = 1.0L;
  result = Hexp[1];

  for (i=2; i<nTerms; i++) {
    ttothei *= tinit;
    result += i*Hexp[i]*ttothei;
  }
  foo->deriv = result;

  return 0;
}
