/*
#define VERBOSE 1
#define VERBOSE (TVIL_CABS(t-1) < 0.01L)
*/
#define VERBOSE 0

/* Evaluation for the generic case. */

#include "internal.h"
#include "3vil_params.h"

/* ******************************************************************* */
     
int TVIL_MaxSteps (TVIL_DATA *foo, TVIL_COMPLEX z)
{
  return (foo->nStepsMaxCon 
          + floor((double) (TVIL_CABS(z) * foo->nStepsMaxVar)));
}

/* **************************************************************** */

void TVIL_InitialValue (TVIL_DATA *foo, TVIL_COMPLEX tinit)
{
  /* For convenience: */
  TVIL_REAL qq = foo->qq;
  int i;

  /* Set initial values for integration... */
  for (i=0; i<NUM_I_FUNCS; i++)
    TVIL_InitialI (&(foo->II[i]), qq, tinit);

  for (i=0; i<NUM_F_FUNCS; i++)
    TVIL_InitialFBAR (&(foo->FBAR[i]), qq, tinit);

  for (i=0; i<NUM_G_FUNCS; i++)
    TVIL_InitialG (&(foo->G[i]), qq, tinit);

  TVIL_InitialH (&(foo->H), qq, tinit);

  return;
}

/* ************************************************************************ */

int TVIL_Integrate (TVIL_DATA    *foo,
		    TVIL_COMPLEX  t0,
		    TVIL_COMPLEX  t1,
		    TVIL_REAL    *dt_start) 
{
  TVIL_COMPLEX t, dt;
  TVIL_REAL pre_error = foo->precisionGoal;

  TVIL_RESULT save_point[65];
  TVIL_COMPLEX t_save[65];
  TVIL_COMPLEX H_save[65];
  TVIL_COMPLEX bestH;
  int saved[65];
  int num_saved = 0;
  int err_bin;
  TVIL_REAL Hscore, best_Hscore;

  int i;
  int force_step;
  int rk6status = 1; /* 1 for success or forced; 
                        0 for need retry, 
		       -1 when the H error is big and step was forced. 
		       -2 when the non-H error is big and step was forced. */
  /* int status_H = 1;  */     /* Set to 0 when H error reported by rk6 gets big. */
  int status_others = 1; /* Set to 0 when other function errors get big.   */

  for (i=0; i<65; i++) saved[i] = 0;

  t  = t0;
  dt = TVIL_CABS(*dt_start) * (t1 - t0)/TVIL_CABS(t1 - t0);

  /* Continue taking steps until we are less than 2 steps away from 
     the endpoint. However, if the error reported by rk6 gets bigger than
     MAX_ALLOWED_ERR defined in rk6.c, and we're near the endpoint, then 
     things probably aren't going to get any better. So, also in that case 
     we will take only one more step. */
  while ( (TVIL_CABS(dt) < 0.5*TVIL_CABS(t1 - t)) && (1 == status_others) )
  {
    force_step = 0;

    for (;;) {

      if (TVIL_CABS(dt) < TVIL_MIN_STEP_SIZE) {
        force_step = 1;
        dt = TVIL_MIN_STEP_SIZE * dt/TVIL_CABS(dt);
      }

      if (VERBOSE) {
        printf("force = %d: t = %Le + %Le I,  deltat = %Le + %Le I\n",
        force_step, TVIL_CREAL(t), TVIL_CIMAG(t), TVIL_CREAL(dt), TVIL_CIMAG(dt));
      }

      rk6status = foo->RKstepper6 (foo, &t, &dt, pre_error, force_step);
      if (0 != rk6status) break;
    }

    /* If H error reported by rk6 got big, save a bunch of points so that
       for each we can do just one more step from here to find H.  */

    /* If H is NaN, and we aren't close to t=1, something is very wrong. */
    if ((1 == TVIL_IsInfinite(foo->H.value)) && (TVIL_CABS(t-1) > 0.01L)) 
      TVIL_Error ("TVIL_Integrate", "H is non-numeric; evaluation terminated. This should never happen. Please contact the authors with details of this case.", -666);

    err_bin = floor((double) (3.0L * TVIL_CLOG(err_H))) - 12;

    if ((0 < err_bin) && (err_bin < 65) && (saved[err_bin] == 0) && (TVIL_CABS(t-1) < 0.01L)) {
      num_saved++;
      TVIL_Backup (foo, &(save_point[num_saved]) );
      t_save[num_saved] = t;
      for (i=1; i <= err_bin; i++) saved[i] = 1;
      /* printf("SAVE t = 1 + %Le I, err_H = %Le, err_noH = %Le\n", */
        /* TVIL_CIMAG(t), err_H, maxerr_noH); */
    }

    /* If the errors in non-H functions got too big, just do one more step. 
       Set status_others to 0 so that the while loop knows to stop. */
    if ((maxerr_noH > 1e9) && (TVIL_CABS(t-1) < 0.01L)) {
      status_others = 0;
    }
  }

  t_save[0] = t;
  TVIL_Backup (foo, &(save_point[0]));

  /* Save the step size magnitude, to be used at start of the next leg,
     if there is one. */
  *dt_start = TVIL_CABS(dt);

  /* The remaining distance is less than twice the step size. So, take 
     one more step arranged to land exactly on t1, and force it. 
     Do it for each saved point, running backwards through the list of saved points,
     so that the last point run is the one that should have the best non-H functions. */

  for (i=num_saved; i >= 0; i--) {
    TVIL_Restore (foo, &(save_point[i]));
    t = t_save[i];
    dt = t1 - t;

    if ((foo->poleAt1) && (TVIL_CABS(t-1) < 0.01L))
      foo->RKstepper5 (foo, &t, dt); 
    else 
      foo->RKstepper6 (foo, &t, &dt, pre_error, 1); 

    H_save[i] = foo->H.value;
  }

  /* Now we need to choose the best H. If there were no saved points with 
     large err_H, then our course is clear... */
  if (0 == num_saved) 
    bestH = H_save[0];
  else 
    /* printf("Number of points saved for retry = %d\n",num_saved); */

  /* ... and if we have 1 to 3 saved points with large err_H, which
     should almost never happen, the best we can do is just take the 
     one with the smallest Im[H] ... */
  if ((1 <= num_saved) && (num_saved <= 3) && (TVIL_CABS(t-1) < 0.01L)){
    bestH = H_save[1];
    /* printf("Point 1: H = %Le + I %Le\n", TVIL_CREAL(H_save[1]), TVIL_CIMAG(H_save[1])); */
    for (i=2; i <= num_saved; i++) {
      if (TVIL_FABS(TVIL_CIMAG(H_save[i])) < TVIL_FABS(TVIL_CIMAG(bestH)))
       bestH = H_save[i];
    /* printf("Point %d: H = %Le + I %Le\n", i, TVIL_CREAL(H_save[i]), TVIL_CIMAG(H_save[i])); */
    }
  }

  /* ... and if more than 3 saved points, we take into account both Im[H]
     and closeness to its neighbors... */
  if ((4 <= num_saved) && (num_saved <= 65) && (TVIL_CABS(t-1) < 0.01L)){
    bestH = H_save[2];
    best_Hscore = 666;
    /* printf("Point 1: H = %Le + I %Le\n", TVIL_CREAL(H_save[1]), TVIL_CIMAG(H_save[1])); */
    for (i=2; i < num_saved; i++) {
      /* The following metric is up for debate. */
      Hscore = 2.L*TVIL_FABS(TVIL_CIMAG(H_save[i])) +
               TVIL_FABS(TVIL_CREAL(H_save[i]) - TVIL_CREAL(H_save[i-1]))+
               TVIL_FABS(TVIL_CREAL(H_save[i]) - TVIL_CREAL(H_save[i+1]));

      if (Hscore < best_Hscore) {
        best_Hscore = Hscore;
        bestH = H_save[i];
      }
      /* printf("Point %d: H = %Le + I %Le\n", i, TVIL_CREAL(H_save[i]), TVIL_CIMAG(H_save[i])); */
    }
    /* printf("Point %d: H = %Le + I %Le\n", num_saved, TVIL_CREAL(H_save[num_saved]), TVIL_CIMAG(H_save[num_saved])); */
  }

  foo->H.value = bestH;

  return 0;
}

/* **************************************************************** */
/* Handling of all generic (i.e. non-analytic) cases.               */

void TVIL_CaseGeneric (TVIL_DATA *foo)
{
  TVIL_COMPLEX tInit, tFinal;
  TVIL_REAL deltat_start;

  /* Set up derivative coefficients, etc., in all sub-objects, along
     with global a parameter and powers: */
  TVIL_SetupFunctions (foo);

  deltat_start = 0.0001L;

  if (foo->reAxisOK && foo->forceContour == NO) {

    tInit = foo->tInit;
    TVIL_InitialValue (foo, tInit);
    tFinal = 1.0L;
    TVIL_Integrate (foo, tInit, tFinal, &deltat_start);
    foo->status = REAXIS;

  } else {

    tInit = (foo->tInit)*I;
    TVIL_InitialValue (foo, tInit);

    /* First leg: */
    tFinal = (foo->imDispl)*I;
    TVIL_Integrate (foo, tInit, tFinal, &deltat_start);

    /* Second leg: */
    tInit = tFinal;
    tFinal = 1.0L + (foo->imDispl)*I;
    TVIL_Integrate (foo, tInit, tFinal, &deltat_start);

    /* Last leg: */
    tInit = tFinal;
    tFinal = 1.0L + 0.0L*I;
    TVIL_Integrate (foo, tInit, tFinal, &deltat_start);

    foo->status = CONTOUR;
  }

  return;
}
