/* Miscellaneous and general functions */

#include "internal.h"
#include <string.h>

/* ******************************************************************* */
/* This was Delta in TSIL... */

TVIL_REAL TVIL_Lambda (TVIL_REAL x, TVIL_REAL y, TVIL_REAL z)
{
  return x*x + y*y + z*z - 2.0L*x*y - 2.0L*x*z - 2.0L*y*z;
}

/* ******************************************************************* */

TVIL_REAL TVIL_Kappa (TVIL_REAL x, TVIL_REAL y, TVIL_REAL z)
{
  return x*x + y*y + z*z - x*y - x*z - y*z;
}

/* ******************************************************************* */

TVIL_REAL TVIL_Delta (TVIL_REAL w, 
		      TVIL_REAL x,
		      TVIL_REAL y,
		      TVIL_REAL z)
{
  return TVIL_Lambda(x,y,z) + 2.L*w*(x + y + z) - 3*w*w;
}

/* ******************************************************************* */

TVIL_REAL TVIL_Phi (TVIL_REAL w, 
		    TVIL_REAL x,
		    TVIL_REAL y,
		    TVIL_REAL z)
{
  TVIL_REAL w3, w2, x3, x2, y3, y2, z3, z2;

  w2 = w*w;
  w3 = w2*w;
  x2 = x*x;
  x3 = x2*x;
  y2 = y*y;
  y3 = y2*y;
  z2 = z*z;
  z3 = z2*z;
    
  return 
    w2*w2 - 4*w3*x + 6*w2*x2 - 4*w*x3 + 
    x2*x2 - 4*w3*y + 4*w2*x*y + 4*w*x2*y - 4*x3*y + 6*w2*y2 + 
    4*w*x*y2 + 6*x2*y2 - 4*w*y3 - 4*x*y3 + y2*y2 - 4*w3*z + 
    4*w2*x*z + 4*w*x2*z - 4*x3*z + 4*w2*y*z - 40*w*x*y*z + 
    4*x2*y*z + 4*w*y2*z + 4*x*y2*z - 4*y3*z + 6*w2*z2 + 
    4*w*x*z2 + 6*x2*z2 + 4*w*y*z2 + 4*x*y*z2 + 
    6*y2*z2 - 4*w*z3 - 4*x*z3 - 4*y*z3 + z2*z2 +
    8*a*(w+x-y-z)*(w-x+y-z)*(w-x-y+z);
}

/* ******************************************************************* */

TVIL_REAL TVIL_Rplus (TVIL_REAL x, TVIL_REAL y, TVIL_REAL z)
{
  return (-3*a*a + a*(x + y + z) +
	  2*a*TVIL_SQRT(TVIL_FABS(TVIL_Kappa(x,y,z))))/TVIL_Lambda(a-x,a-y,a-z);
}

/* ******************************************************************* */

TVIL_REAL TVIL_Rminus (TVIL_REAL x, TVIL_REAL y, TVIL_REAL z)
{
  return (-3*a*a + a*(x + y + z) -
	  2*a*TVIL_SQRT(TVIL_FABS(TVIL_Kappa(x,y,z))))/TVIL_Lambda(a-x,a-y,a-z);
}

/* ******************************************************************* */

TVIL_REAL TVIL_R4 (TVIL_REAL w, 
		   TVIL_REAL x,
		   TVIL_REAL y,
		   TVIL_REAL z)
{
  if ( (TVIL_FABS(TVIL_Delta(w,x,y,z)/(w*w + x*x + y*y + z*z + 1e-99)) < 
        TVIL_TOL) ||
       ((TVIL_FABS((w-x)/(w+x + 1e-99)) < TVIL_TOL) && 
        (TVIL_FABS((y-z)/(y+z + 1e-99)) < TVIL_TOL)) ||        
       ((TVIL_FABS((w-y)/(w+y + 1e-99)) < TVIL_TOL) && 
        (TVIL_FABS((x-z)/(x+z + 1e-99)) < TVIL_TOL)) ||        
       ((TVIL_FABS((w-z)/(w+z + 1e-99)) < TVIL_TOL) && 
        (TVIL_FABS((x-y)/(x+y + 1e-99)) < TVIL_TOL)) )
    return a/(a - w);
  else
    return 8*a*(w + x - y - z)*(w - x + y - z)*(w - x - y + z)/TVIL_Phi(w,x,y,z);
}

/* ******************************************************************* */

int TVIL_Compare (const void * a, const void * b)
{
    TVIL_REAL f = *((TVIL_REAL *)a);
    TVIL_REAL s = *((TVIL_REAL *)b);

    /* printf("In Compare...\n"); */
    /* printf("1st arg = %Le\n", f); */
    /* printf("2nd arg = %Le\n", s); */

    if (f > s) return  1;
    if (f < s) return -1;
    return 0;
}

/* ******************************************************************* */
/* Returns true/false indicating if z is TVIL_Infinity               */

int TVIL_IsInfinite (TVIL_COMPLEX z)
{
  /* DGR - modified to work with gcc4 */
#if __GNUC__==4
  if (isnan(TVIL_CREAL(z)) || isinf(TVIL_CREAL(z)) ||
      isnan(TVIL_CIMAG(z)) || isinf(TVIL_CIMAG(z)))
    return TRUE;
  else
    return FALSE;
#else
  if (isnan(TVIL_CREAL(z)) || isnan(TVIL_CIMAG(z)))
    return TRUE;
  else
    return FALSE;
#endif
}

/* ******************************************************************* */

int TVIL_NonZero (TVIL_REAL x)
{
  if (TVIL_FABS (x) > TVIL_TOL)
    return TRUE;
  else
    return FALSE;
}

/* ******************************************************************* */

int TVIL_Sign (TVIL_REAL x)
{
  if (x > TVIL_TOL)
    return +1;
  else if (x < TVIL_TOL)
    return -1;
  else
    return 0;
}

/* ******************************************************************* */
/* Generic printing of complexes                                       */

void TVIL_cprintf (TVIL_COMPLEX z)
{
  TVIL_cfprintf (stdout, (double complex) z);
  return;
}

/* ******************************************************************* */
/* Mathematica-compatible printing of complexes                        */

void TVIL_cprintfM (TVIL_COMPLEX z)
{
  TVIL_cfprintfM (stdout, (double complex) z);
  return;
}

/* ******************************************************************* */
/* Generic printing of complexes                                       */

void TVIL_cfprintf (FILE *fp, double complex z)
{
  TVIL_REAL log10z, temp;
  int rightdigits; /* Number of digits to right of decimal. */
  char zformat[20]; /* Format for z */

  if (TVIL_IsInfinite (z))
    fprintf(fp, " ComplexInfinity");
  else {
    /*
    fprintf(fp, "% 1.15le, % 1.15le", creal(z), cimag(z));
    fprintf(fp, "% 18.16lf, % 18.16lf", creal(z), cimag(z)); 
    */
    log10z = TVIL_LOG(TVIL_FABS(creal(z)));
    temp = TVIL_LOG(TVIL_FABS(cimag(z)));
    if (temp > log10z) log10z = temp;
    log10z *= 0.43429448190325176L;
    rightdigits = 17 - ceil (log10z);
    if (rightdigits > 16) rightdigits = 16;
    if (rightdigits < 1) rightdigits = 1;
    sprintf(zformat, "%% .%dlf, %% .%dlf",rightdigits,rightdigits);

    fprintf(fp, zformat, creal(z), cimag(z));
  }

  return;
}


/* ******************************************************************* */
/* Mathematica-compatible printing of complexes                        */

void TVIL_cfprintfM (FILE *fp, double complex z)
{
  TVIL_REAL log10z, temp;
  int rightdigits; /* Number of digits to right of decimal. */
  char zformat[20]; /* Format for z */

  if (TVIL_IsInfinite (z))
    fprintf(fp, " ComplexInfinity");
  else {
    /*
    fprintf(fp, "% 1.15le + % 1.15le I", creal(z), cimag(z));
    fprintf(fp, "% 18.16lf + % 18.16lf I", creal(z), cimag(z));
    */
    log10z = TVIL_LOG(TVIL_FABS(creal(z)));
    temp = TVIL_LOG(TVIL_FABS(cimag(z)));
    if (temp > log10z) log10z = temp;
    log10z *= 0.43429448190325176L;
    rightdigits = 17 - ceil (log10z);
    if (rightdigits > 16) rightdigits = 16;
    if (rightdigits < 1) rightdigits = 1;

    sprintf(zformat, "%% .%dlf + %% .%dlf I",rightdigits,rightdigits);

    /* Only for making TestData files with all positive arguments: */
    /* sprintf(zformat, "%% .%dlf + 0.0 I",rightdigits); */

    fprintf(fp, zformat, creal(z), cimag(z));
  }

  return;
}

/* ******************************************************************* */

void TVIL_PrintData (TVIL_DATA *foo)
{
  TVIL_WriteData (stdout, foo);
  return;
}

/* ******************************************************************* */

void TVIL_PrintDataM (TVIL_DATA *foo)
{
  TVIL_WriteDataM (stdout, foo);
  return;
}

/* ******************************************************************* */

void TVIL_WriteData (FILE *fp, TVIL_DATA *foo)
{
#include "3vil_names.h"

  int i, j;

  if (foo->status == UNEVALUATED) {
    TVIL_Warn ("Write/PrintDataM", "This case has not yet been evaluated!");
    return;
  }

  fprintf(fp, "u  = %.12Lf\n", (long double) (foo->u));
  fprintf(fp, "v  = %.12Lf\n", (long double) (foo->v));
  fprintf(fp, "w  = %.12Lf\n", (long double) (foo->w));
  fprintf(fp, "x  = %.12Lf\n", (long double) (foo->x));
  fprintf(fp, "y  = %.12Lf\n", (long double) (foo->y));
  fprintf(fp, "z  = %.12Lf\n", (long double) (foo->z));
  fprintf(fp, "qq = %.12Lf\n", (long double) (foo->qq));

  if (printBold) {
  fprintf(fp, "\n");
  fprintf(fp, "(* Bold Functions: *)\n");
  for (j=0; j<NUM_E_FUNCS; j++) {
    for (i=0; i<4; i++) {
      fprintf(fp, "%s[%d]  = ", ename[j][0], i);
      TVIL_cprintf (foo->E[j].bold[i]); fprintf(fp, "\n");
    }
    fprintf(fp, "\n");
  }

  for (j=0; j<NUM_F_FUNCS; j++) {
    for (i=0; i<4; i++) {
      fprintf(fp, "%s[%d] = ", fname[j][0], i);
      TVIL_cprintf (foo->FBAR[j].boldF[i]); fprintf(fp, "\n");
    }
    fprintf(fp, "\n");
  }

  for (j=0; j<NUM_G_FUNCS; j++) {
    for (i=0; i<4; i++) {
      fprintf(fp, "%s[%d]   = ", gname[j][0], i);
      if (i == 0 && foo->G[j].hasNegativeArg)
	printf("Not yet implemented for negative mass^2 arguments!\n");
      else
	{TVIL_cprintf (foo->G[j].bold[i]); fprintf(fp, "\n");}
    }
    fprintf(fp, "\n");
  }

  for (i=0; i<2; i++) {
    fprintf(fp, "Huvwxyz[%d]  = ", i);
    TVIL_cprintf (foo->H.bold[i]); fprintf(fp, "\n");
  }
  }

  fprintf(fp, "\n");
  fprintf(fp, "(* Basis Functions: *)\n");
  for (j=0; j<NUM_I_FUNCS; j++) {
    fprintf(fp, "%s     = ", iname[j][0]);
    TVIL_cprintf (foo->II[j].value); fprintf(fp, "\n");
  }
  fprintf(fp, "\n");
  for (j=0; j<NUM_E_FUNCS; j++) {
    fprintf(fp, "%s    = ", ename[j][0]);
    TVIL_cprintf (foo->E[j].value); fprintf(fp, "\n");
  }
  fprintf(fp, "\n");
  for (j=0; j<NUM_F_FUNCS; j++) {
    fprintf(fp, "%s    = ", fname[j][0]);
    TVIL_cprintf (foo->FBAR[j].fValue); fprintf(fp, "\n");
  }
  fprintf(fp, "\n");
  for (j=0; j<NUM_F_FUNCS; j++) {
    fprintf(fp, "%s = ", fbarname[j][0]);
    TVIL_cprintf (foo->FBAR[j].value); fprintf(fp, "\n");
  }
  fprintf(fp, "\n");
  for (j=0; j<NUM_G_FUNCS; j++) {
    fprintf(fp, "%s   = ", gname[j][0]);
    TVIL_cprintf (foo->G[j].value); fprintf(fp, "\n");
  }
  fprintf(fp, "\n");
  fprintf(fp, "Huvwxyz  = ");
  TVIL_cprintf (foo->H.value); fprintf(fp, "\n");

  return;
}

/* ******************************************************************* */
/* Print output in Mathematica-compatible format                       */

void TVIL_WriteDataM (FILE *fp, TVIL_DATA *foo)
{
#include "3vil_names.h"

  int i, j;

  if (foo->status == UNEVALUATED) {
    TVIL_Warn ("Write/PrintDataM", "This case has not yet been evaluated!");
    return;
  }

  fprintf(fp, "u  = %.12Lf;\n", (long double) (foo->u));
  fprintf(fp, "v  = %.12Lf;\n", (long double) (foo->v));
  fprintf(fp, "w  = %.12Lf;\n", (long double) (foo->w));
  fprintf(fp, "x  = %.12Lf;\n", (long double) (foo->x));
  fprintf(fp, "y  = %.12Lf;\n", (long double) (foo->y));
  fprintf(fp, "z  = %.12Lf;\n", (long double) (foo->z));
  fprintf(fp, "qq = %.12Lf;\n", (long double) (foo->qq));

  if (printBold) {
  fprintf(fp, "\n");
  fprintf(fp, "(* Bold Functions: *)\n");
  for (j=0; j<NUM_E_FUNCS; j++) {
    for (i=0; i<4; i++) {
      fprintf(fp, "%s[%d]  = ", ename[j][0], i);
      TVIL_cprintfM (foo->E[j].bold[i]); fprintf(fp, ";\n");
    }
    fprintf(fp, "\n");
  }

  for (j=0; j<NUM_F_FUNCS; j++) {
    for (i=0; i<4; i++) {
      fprintf(fp, "%s[%d] = ", fname[j][0], i);
      TVIL_cprintfM (foo->FBAR[j].boldF[i]); fprintf(fp, ";\n");
    }
    fprintf(fp, "\n");
  }

  for (j=0; j<NUM_G_FUNCS; j++) {
    for (i=0; i<4; i++) {
      fprintf(fp, "%s[%d]   = ", gname[j][0], i);
      if (i == 0 && foo->G[j].hasNegativeArg)
	printf("Not yet implemented for negative mass^2 arguments!\n");
      else
	{TVIL_cprintfM (foo->G[j].bold[i]); fprintf(fp, ";\n");}
    }
    fprintf(fp, "\n");
  }

  for (i=0; i<2; i++) {
    fprintf(fp, "Huvwxyz[%d]  = ", i);
    TVIL_cprintfM (foo->H.bold[i]); fprintf(fp, ";\n");
  }
  }

  fprintf(fp, "\n");
  fprintf(fp, "(* Basis Functions: *)\n");
  for (j=0; j<NUM_I_FUNCS; j++) {
    fprintf(fp, "%s     = ", iname[j][0]);
    TVIL_cprintfM (foo->II[j].value); fprintf(fp, ";\n");
  }
  fprintf(fp, "\n");
  for (j=0; j<NUM_E_FUNCS; j++) {
    fprintf(fp, "%s    = ", ename[j][0]);
    TVIL_cprintfM (foo->E[j].value); fprintf(fp, ";\n");
  }
  fprintf(fp, "\n");
  for (j=0; j<NUM_F_FUNCS; j++) {
    fprintf(fp, "%s    = ", fname[j][0]);
    TVIL_cprintfM (foo->FBAR[j].fValue); fprintf(fp, ";\n");
  }
  fprintf(fp, "\n");
  for (j=0; j<NUM_F_FUNCS; j++) {
    fprintf(fp, "%s = ", fbarname[j][0]);
    TVIL_cprintfM (foo->FBAR[j].value); fprintf(fp, ";\n");
  }
  fprintf(fp, "\n");
  for (j=0; j<NUM_G_FUNCS; j++) {
    fprintf(fp, "%s   = ", gname[j][0]);
    TVIL_cprintfM (foo->G[j].value); fprintf(fp, ";\n");
  }
  fprintf(fp, "\n");
  fprintf(fp, "Huvwxyz  = ");
  TVIL_cprintfM (foo->H.value); fprintf(fp, ";\n");

  return;
}

/* ******************************************************************* */

int TVIL_GetStatus (TVIL_DATA *foo)
{
  return foo->status;
}

/* ******************************************************************* */
/* Prints the evaluation status of the specified data object: whether
   unevaluated, or evaluaed analytically, numerically by integration
   along the real s axis, or numerically by integration along the
   displaced contour. */

void TVIL_PrintStatus (TVIL_DATA *foo)
{
  if (foo->status == UNEVALUATED)
    TVIL_Warn("TVIL_PrintStatus", "Functions not yet evaluated!");
  else {
    printf("(* Evaluation method: ");
    if (foo->status == ANALYTIC)
      printf("Analytic");
    else if (foo->status == REAXIS)
      printf("Integration along real t-axis");
    else if (foo->status == CONTOUR)
      printf("Integration along displaced contour");
    printf(" *)\n");

    if (foo->poleAt1 && foo->status != ANALYTIC)
      printf("(* Pole at or near t = 1 *)\n");

    if (foo->status != ANALYTIC)
      printf("(* a = %Lf *)\n", a);
  }
  return;
}

/* ******************************************************************* */

void TVIL_PrintVersion (void)
{
  printf("(* 3VIL Version: %s *)\n", TVIL_VERSION);
  return;
}

/* ******************************************************************* */

void TVIL_Warn (const char *func, const char *msg)
{
  if (printWarns == YES) {
    fprintf (stderr, "WARNING (%s): %s\n", func, msg);
    fflush (stderr);
  }
  return;
}

/* ******************************************************************* */
/* Prints error message, stack trace, and exits. */

#include <execinfo.h>

void TVIL_Error (const char *func, const char *msg, int val)
{
  void *callstack[128];
  int i, frames = backtrace(callstack, 128);
  char **strs = backtrace_symbols(callstack, frames);

  fprintf (stderr, "ERROR (%s): %s\n", func, msg);

  for (i=0; i<frames; i++)
    fprintf(stderr, "%s\n", strs[i]);

  fflush (stderr);
  exit (val);
}

/* ******************************************************************* */

void TVIL_Set_aParameter (TVIL_DATA *foo, TVIL_REAL newval)
{
  foo->aParameter = newval;
  return;
}

/* ******************************************************************* */

void TVIL_SetContourDisplacement (TVIL_DATA *foo, TVIL_REAL d)
{
  foo->imDispl = d;
  return;
}

/* ******************************************************************* */

void TVIL_Set_tInitial (TVIL_DATA *foo, TVIL_REAL newval)
{
  foo->tInit = newval;
  return;
}

/* ******************************************************************* */

void TVIL_SetDoAnalytic (TVIL_DATA *foo)
{
  foo->doAnalytic = YES;
  return;
}

/* ******************************************************************* */

void TVIL_UnsetDoAnalytic (TVIL_DATA *foo)
{
  foo->doAnalytic = NO;
  return;
}

/* ******************************************************************* */

void TVIL_SetForceContour (TVIL_DATA *foo)
{
  foo->forceContour = YES;
  return;
}

/* ******************************************************************* */

void TVIL_UnsetForceContour (TVIL_DATA *foo)
{
  foo->forceContour = NO;
  return;
}

/* ******************************************************************* */

void TVIL_WarnsOn (void)
{
  printWarns = YES;
  return;
}

/* ******************************************************************* */

void TVIL_WarnsOff (void)
{
  printWarns = NO;
  return;
}

/* ******************************************************************* */

void TVIL_SetPrintBold (void)
{
  printBold = YES;
  return;
}

/* ******************************************************************* */

void TVIL_UnsetPrintBold (void)
{
  printBold = NO;
  return;
}

/* **************************************************************** */

void TVIL_ResetStepSizeParams (TVIL_DATA *foo,
                               TVIL_REAL precisionGoal,
                               int nstepsStart,
                               int nstepsMaxCon,
                               int nstepsMaxVar,
                               int nstepsMin)
{
  foo->precisionGoal = precisionGoal;
  foo->nStepsStart   = nstepsStart;
  foo->nStepsMaxCon  = nstepsMaxCon;
  foo->nStepsMaxVar  = nstepsMaxVar;
  foo->nStepsMin     = nstepsMin;

  return;
}

/* ******************************************************************* */
/*
  Returns 1 (TRUE) if there is a pole within minDistance of 1.0, and
  sets *npole to the pole value nearest to 1.0.  Returns 0 (FALSE) if
  we are further away than minDistance from all poles.
*/

int TVIL_NearPole (TVIL_DATA *foo,
		   TVIL_REAL *nearestPole,
		   TVIL_REAL minDistance)
{
  TVIL_REAL distance;
  int i;
  int areWeCloseToAPole = NO;

  *nearestPole = 0.0L; /* This should be ignored if it survives. */

  for (i=0; i<(foo->nPoles); i++) {
    distance = TVIL_FABS(1.0L - (foo->tPole[i]));
    if (distance < minDistance) {
      minDistance = distance;
      *nearestPole = foo->tPole[i];
      areWeCloseToAPole = YES;
    }
  }
  return areWeCloseToAPole;
}

/* **************************************************************** */

void TVIL_Backup (TVIL_DATA *d, TVIL_RESULT *r)
{
  int i;

  r->u = d->u;
  r->v = d->v;
  r->w = d->w;
  r->x = d->x;
  r->y = d->y;
  r->z = d->z;

   for (i=0; i<NUM_I_FUNCS; i++)
    r->II[i] = d->II[i].value;
  for (i=0; i<NUM_F_FUNCS; i++)
    r->FBAR[i] = d->FBAR[i].value;
  for (i=0; i<NUM_G_FUNCS; i++)
    r->G[i] = d->G[i].value;
  r->H = d->H.value;

   for (i=0; i<NUM_I_FUNCS; i++)
    r->IIderiv[i] = d->II[i].deriv;
  for (i=0; i<NUM_F_FUNCS; i++)
    r->FBARderiv[i] = d->FBAR[i].deriv;
  for (i=0; i<NUM_G_FUNCS; i++)
    r->Gderiv[i] = d->G[i].deriv;
  r->Hderiv = d->H.deriv;

  return;
}

/* **************************************************************** */

void TVIL_Restore (TVIL_DATA *d, TVIL_RESULT *r)
{
  int i;

  d->u = r->u;
  d->v = r->v;
  d->w = r->w;
  d->x = r->x;
  d->y = r->y;
  d->z = r->z;

  for (i=0; i<NUM_I_FUNCS; i++)
    d->II[i].value = r->II[i];
  for (i=0; i<NUM_F_FUNCS; i++)
    d->FBAR[i].value = r->FBAR[i];
  for (i=0; i<NUM_G_FUNCS; i++)
    d->G[i].value = r->G[i];
  d->H.value = r->H;

  for (i=0; i<NUM_I_FUNCS; i++)
    d->II[i].deriv = r->IIderiv[i];
  for (i=0; i<NUM_F_FUNCS; i++)
    d->FBAR[i].deriv = r->FBARderiv[i];
  for (i=0; i<NUM_G_FUNCS; i++)
    d->G[i].deriv = r->Gderiv[i];
  d->H.deriv = r->Hderiv;

  return;
}

/* ******************************************************************* */
/* Check to see if a string represents a floating-point value. */

int TVIL_IsNumeric (const char * s)
{
  if (s == NULL || *s == '\0' || isspace(*s))
    return 0;
  char *p;
  strtod (s, &p);
  return *p == '\0';
}

/* ******************************************************************* */
/* Check input function identifier for sanity */

int TVIL_ValidIdentifier (const char* which)
{ 
#include "3vil_names.h"

  int i,j;

  if (!strcmp(which, "H")) return YES;

  for (i=0; i<NUM_G_FUNCS; i++)
    for (j=0; j<NUM_G_PERMS; j++)
      if (!strcmp(which, gname[i][j])) return YES;

  for (i=0; i<NUM_F_FUNCS; i++)
    for (j=0; j<NUM_F_PERMS; j++)
      if (!strcmp(which, fname[i][j])) return YES;

  for (i=0; i<NUM_F_FUNCS; i++)
    for (j=0; j<NUM_F_PERMS; j++)
      if (!strcmp(which, fbarname[i][j])) return YES;

  for (i=0; i<NUM_E_FUNCS; i++)
    for (j=0; j<NUM_E_PERMS; j++)
      if (!strcmp(which, ename[i][j])) return YES;

  for (i=0; i<NUM_I_FUNCS; i++)
    for (j=0; j<NUM_I_PERMS; j++)
      if (!strcmp(which, iname[i][j])) return YES;

  /* If we get here the identifier is invalid: */
  return NO;
}

/* **************************************************************** */
/* Extract a single function value from the data object             */

TVIL_COMPLEX TVIL_GetFunction (TVIL_DATA *foo, const char *which)
{
#include "3vil_names.h"

  int i,j;

  TVIL_COMPLEX result = (TVIL_COMPLEX) 0.0;
  char funcname[] = "TVIL_GetFunction";
  char errmsg[55] = "Function not defined for these parameters: ";

  if (!TVIL_ValidIdentifier (which))
    TVIL_Error (funcname, "Invalid function identifier specified.", 23);

  if (foo->status == UNEVALUATED)
    TVIL_Error (funcname, "This case has not yet been evaluated!", 22);

  if (!strncmp(which, "H", 1)) {
    result = foo->H.value;
  }
  else if (!strncmp(which, "G", 1)) {
    for (i=0; i<NUM_G_FUNCS; i++)
      for (j=0; j<NUM_G_PERMS; j++)
	if (!strcmp(which, gname[i][j]))
	  result = foo->G[i].value;
  }
  else if (!strncmp(which, "FBAR", 4)) {
    for (i=0; i<NUM_F_FUNCS; i++)
      for (j=0; j<NUM_F_PERMS; j++)
	if (!strcmp(which, fbarname[i][j]))
	  result = foo->FBAR[i].value;
  }
  else if (!strncmp(which, "F", 1)) {
    for (i=0; i<NUM_F_FUNCS; i++)
      for (j=0; j<NUM_F_PERMS; j++)
	if (!strcmp(which, fname[i][j]))
	  result = foo->FBAR[i].fValue;
  }
  else if (!strncmp(which, "E", 1)) {
    for (i=0; i<NUM_E_FUNCS; i++)
      for (j=0; j<NUM_E_PERMS; j++)
	if (!strcmp(which, ename[i][j]))
	  result = foo->E[i].value;
  }
  else if (!strncmp(which, "I", 1)) {
    for (i=0; i<NUM_I_FUNCS; i++)
      for (j=0; j<NUM_I_PERMS; j++)
	if (!strcmp(which, iname[i][j]))
	  result = foo->II[i].value;
  }
  else {
    /* Should never get here, but... */
    TVIL_Error (funcname, "Invalid identifier specified.", 1);
  }

  if (TVIL_IsInfinite (result))
    TVIL_Warn (funcname, strncat (errmsg, which, 7));

  return result;
}

/* **************************************************************** */
/* Extract a single function value from the data object             */

TVIL_COMPLEX TVIL_GetBoldFunction (TVIL_DATA *foo,
				   const char *which,
				   int n)
{
#include "3vil_names.h"

  int i,j;

  TVIL_COMPLEX result = (TVIL_COMPLEX) 0.0;
  char funcname[] = "TVIL_GetBoldFunction";
  char errmsg[55] = "Function not defined for these parameters: ";

  if (!TVIL_ValidIdentifier (which))
    TVIL_Error (funcname, "Invalid function identifier specified.", 23);

  if (foo->status == UNEVALUATED)
    TVIL_Error (funcname, "This case has not yet been evaluated!", 22);

  if (!strncmp(which, "H", 1)) {
    if (n < 0 || n > 1)
      TVIL_Error (funcname, "Invalid pole specification for bold H.", 24);

    result = foo->H.bold[n];
  }
  else if (!strncmp(which, "G", 1)) {
    if (n < 0 || n > 3)
      TVIL_Error (funcname, "Invalid pole specification for bold G.", 24);

    for (i=0; i<NUM_G_FUNCS; i++)
      for (j=0; j<NUM_G_PERMS; j++)
	if (!strcmp(which, gname[i][j])) {
	  if (foo->G[i].hasNegativeArg && n == 0)
	    TVIL_Warn (funcname,
		       "eps^0 term in bold G not yet implemented for negative argument(s)! Do not use this value.");
	  result = foo->G[i].bold[n];
	}
  }
  else if (!strncmp(which, "F", 1) && strncmp(which, "FBAR", 4)) {
    if (n < 0 || n > 3)
      TVIL_Error (funcname, "Invalid pole specification for bold F.", 24);

    for (i=0; i<NUM_F_FUNCS; i++)
      for (j=0; j<NUM_F_PERMS; j++)
	if (!strcmp(which, fname[i][j]))
	  result = foo->FBAR[i].boldF[n];
  }
  else if (!strncmp(which, "E", 1)) {
    if (n < 0 || n > 3)
      TVIL_Error (funcname, "Invalid pole specification for bold E.", 24);

    for (i=0; i<NUM_E_FUNCS; i++)
      for (j=0; j<NUM_E_PERMS; j++)
	if (!strcmp(which, ename[i][j]))
	  result = foo->E[i].bold[n];
  }
  else if (!strncmp(which, "I", 1)) {
    if (n < 0 || n > 2)
      TVIL_Error (funcname, "Invalid pole specification for bold I.", 24);

    for (i=0; i<NUM_I_FUNCS; i++)
      for (j=0; j<NUM_I_PERMS; j++)
	if (!strcmp(which, iname[i][j])) {
	  if (n == 0)
	    result = TVIL_II0 (foo->II[i].arg[0], foo->II[i].arg[1], foo->II[i].arg[2], foo->qq);
	  else if (n == 1)
	    result = TVIL_II1 (foo->II[i].arg[0], foo->II[i].arg[1], foo->II[i].arg[2], foo->qq);
	  else
	    result = TVIL_II2 (foo->II[i].arg[0], foo->II[i].arg[1], foo->II[i].arg[2], foo->qq);
	}
  }
  else {
    /* Should never get here, but... */
    TVIL_Error (funcname, "Invalid identifier specified.", 1);
  }

  if (TVIL_IsInfinite (result))
    TVIL_Warn (funcname, strncat (errmsg, which, 7));

  return result;
}

/* **************************************************************** */
