/* 3VIL v1.0 */

/* General header file for the user API and all 3VIL types.  This file
   must be included in all applications of 3VIL.  */

#ifndef TVIL_H
#define TVIL_H

#define TVIL_VERSION "1.0"

#ifdef __cplusplus
extern "C" {
#endif

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <float.h>
#include <complex.h>
#include "3vil_global.h"

typedef long double          TVIL_REAL;
typedef long double _Complex TVIL_COMPLEX;
#define TVIL_EXP             expl
#define TVIL_CEXP            cexpl
#define TVIL_LOG             logl
#define TVIL_CLOG            clogl
#define TVIL_FABS            fabsl
#define TVIL_CABS            cabsl
#define TVIL_SQRT            sqrtl
#define TVIL_CSQRT           csqrtl
#define TVIL_POW             powl
#define TVIL_CPOW            cpowl
#define TVIL_ATAN            atanl
#define TVIL_ATAN2           atan2l
#define TVIL_CREAL           creall
#define TVIL_CIMAG           cimagl
#define TVIL_CONJ            conjl
#define TVIL_EPSILON         LDBL_EPSILON
#define TVIL_TOL             1000.0L*LDBL_EPSILON


/* Global parameters */
TVIL_REAL a, a2, a3, a4, a5, a6, a7, a8, a9;
TVIL_REAL maxerr, err_H, maxerr_noH; /* From the rk6 stepper. */

/* ========= Datatype Definitions ========= */

/* ======================================== */
/*            I-type function               */
/* ======================================== */
struct TVIL_Itype {

  int          which;
  TVIL_REAL    arg[3];
  TVIL_COMPLEX value;
  TVIL_COMPLEX deriv;
  TVIL_COMPLEX bold[4];

  /* Evolution coefficients: */
  TVIL_REAL p[2];
  TVIL_REAL cII[2];
  TVIL_REAL cILL_012[2], cILL_021[2], cILL_120[2];
  TVIL_REAL cIL_012[3], cIL_102[3], cIL_201[3];
  TVIL_REAL cI[3];

  TVIL_REAL sumargs;
};

typedef struct TVIL_Itype TVIL_ITYPE;

/* ======================================== */
/*          Fbar-type function              */
/* ======================================== */
struct TVIL_FBARtype {

  int          which;
  TVIL_REAL    arg[4];
  TVIL_COMPLEX value;
  TVIL_COMPLEX deriv;
  TVIL_COMPLEX fValue;
  TVIL_COMPLEX boldF[4];

  /* Pole locations (they are the same for each term!) */
  TVIL_REAL p2, p3, p4, p5;

  /* Evolution coefficients: */
  TVIL_REAL cFF11, cFF13;
  TVIL_REAL cFF21, cFF22[3], cFF23[3];
  TVIL_REAL cFLLL11, cFLLL12, cFLLL13;
  TVIL_REAL cFLLL21, cFLLL23[3], cFLLL24[3], cFLLL25[3];
  TVIL_REAL cFLL11, cFLL12[3], cFLL13[3];
  TVIL_REAL cFLL21, cFLL23[3], cFLL24[3], cFLL25[3];
  TVIL_REAL cFIL4, cFIL5;
  TVIL_REAL cFI22;
  TVIL_REAL cFL10, cFL11, cFL13, cFL14, cFL15;
  TVIL_REAL cFL20[3], cFL21, cFL22[3], cFL23[3];
  TVIL_REAL cF0, cF2, cF3;

  /* Pointers to needed data: */
  TVIL_COMPLEX *fval[3];
  TVIL_COMPLEX *ival[4];

  TVIL_REAL sumargs;
};

typedef struct TVIL_FBARtype TVIL_FBARTYPE;

/* ======================================== */
/*            E-type function               */
/* ======================================== */
struct TVIL_Etype {

  int          which;
  TVIL_REAL    arg[4];
  TVIL_COMPLEX value;
  TVIL_COMPLEX bold[4];

  TVIL_COMPLEX *fval[4];
};

typedef struct TVIL_Etype TVIL_ETYPE;

/* ======================================== */
/*             G-type function              */
/* ======================================== */
struct TVIL_Gtype {

  int          which;
  TVIL_REAL    arg[5];
  TVIL_COMPLEX value;
  TVIL_COMPLEX deriv;
  TVIL_COMPLEX bold[4];
  int          hasNegativeArg;

  /* Pole locations (default arg ordering): */
  TVIL_REAL p1, p2, p3, p4, p5;

  /* These correspond to the four F terms, in order */
  TVIL_REAL cGF1[4], cGF4[4], cGF5[4];
  TVIL_REAL cGF_p4[4], cGF_p5[4];

  TVIL_REAL cGI_p2[2], cGI_p3[2];

  TVIL_REAL cGL_p2[4], cGL_p3[4];

  TVIL_REAL cG0, cG1, cG2, cG3, cG4, cG5;

  /* Pointers to needed data: */
  TVIL_COMPLEX *fval[4];
  TVIL_COMPLEX *ival[10];

  TVIL_REAL sumargs;
};

typedef struct TVIL_Gtype TVIL_GTYPE;

/* ======================================== */
/*             H-type function              */
/* ======================================== */
struct TVIL_Htype {

  TVIL_REAL    arg[6];
  TVIL_COMPLEX value;
  TVIL_COMPLEX deriv;
  TVIL_COMPLEX bold[2];

  TVIL_REAL cHGnum[6][5];
  TVIL_REAL cHGden[6][8];
  TVIL_REAL cHnum[3];
  TVIL_REAL cHden[4];

  TVIL_REAL cHd1roots[3];

  /* Pointers to needed data: */
  TVIL_COMPLEX *gval[6];
  TVIL_COMPLEX *fval[6][2];
  TVIL_COMPLEX *ival[6][4];

  TVIL_REAL sumargs;
};

typedef struct TVIL_Htype TVIL_HTYPE;


/* ======================================== */
/*          General Data Structure          */
/* ======================================== */
struct TVIL_Data
{
  int isAligned;     /* Indicates whether the data sub-objects have
                        been constructed */

  int isInitialized; /* Indicates whether initial values have been
			set. */

  int reAxisOK;      /* YES/NO indicatinging whether there is a pole
			on the real-t axis between t=0 and t=1. */

  int poleAt1;       /* YES/NO indicating whether there is a pole near
			t=1. */

  int status;        /* Current evaluation status: whether evaluated
                        and, if so, how (see tsil_global.h for value
                        definitions). */

  int doAnalytic;    /* If YES (NO), put in (don't put in) all
			analytically known functions after generic
			evaulation. */

  int forceContour;  /* If YES, force evaulation by integration along
			contour in complex plane, even if integration
			along real t-axis is possible. */

  /* Data */
  TVIL_REAL u, v, w, x, y, z, qq;

  TVIL_ITYPE      II[NUM_I_FUNCS];
  TVIL_FBARTYPE FBAR[NUM_F_FUNCS];
  TVIL_ETYPE       E[NUM_E_FUNCS];
  TVIL_GTYPE       G[NUM_G_FUNCS];
  TVIL_HTYPE       H;

  TVIL_REAL *tPole;
  int        nPoles;

  /* Parameters for numerical integration; defaults are defined in
     3vil_params.h: */
  TVIL_REAL imDispl;
  TVIL_REAL tInit;
  TVIL_REAL aParameter;

  /* Parameters for automatic step-size control in Runge-Kutta: */
  TVIL_REAL precisionGoal;
  int       nStepsStart, nStepsMaxCon, nStepsMaxVar, nStepsMin;

  /* Pointers to RK stepper functions: */
  int  (*RKstepper6) ();
  void (*RKstepper5) ();
};

typedef struct TVIL_Data TVIL_DATA;

/* ======================================== */
/*          Results Data Structure          */
/* ======================================== */
struct TVIL_Result
{
  TVIL_REAL u, v, w, x, y, z, qq;

  TVIL_COMPLEX    II[NUM_I_FUNCS];
  TVIL_COMPLEX  FBAR[NUM_F_FUNCS];
  TVIL_COMPLEX     G[NUM_G_FUNCS];
  TVIL_COMPLEX     H;

  TVIL_COMPLEX    IIderiv[NUM_I_FUNCS];
  TVIL_COMPLEX  FBARderiv[NUM_F_FUNCS];
  TVIL_COMPLEX     Gderiv[NUM_G_FUNCS];
  TVIL_COMPLEX     Hderiv;
};

typedef struct TVIL_Result TVIL_RESULT;


/* Toggle to control printing of warning messages */
extern int printWarns;

/* Toggle to control printing of bold functions */
extern int printBold;

/* === Prototypes for functions in the user API: === */

/* Basic evaluation functions: */
int          TVIL_SetParameters (TVIL_DATA *, TVIL_REAL, TVIL_REAL, TVIL_REAL,
				 TVIL_REAL, TVIL_REAL, TVIL_REAL, TVIL_REAL);
int          TVIL_Evaluate (TVIL_DATA *);
TVIL_COMPLEX TVIL_GetFunction (TVIL_DATA *, const char *);
TVIL_COMPLEX TVIL_GetBoldFunction (TVIL_DATA *, const char *, int);

/* I/O and related functions: */
void TVIL_PrintData (TVIL_DATA *);
void TVIL_PrintDataM (TVIL_DATA *);
void TVIL_WriteData (FILE *, TVIL_DATA *);
void TVIL_WriteDataM (FILE *, TVIL_DATA *);
void TVIL_Error (const char *, const char *, int);
void TVIL_Warn (const char *, const char *);
void TVIL_WarnsOff (void);
void TVIL_WarnsOn (void);
void TVIL_SetPrintBold (void);
void TVIL_UnsetPrintBold (void);
void TVIL_cfprintf (FILE *, double complex);
void TVIL_cfprintfM (FILE *, double complex);
void TVIL_cprintf (TVIL_COMPLEX);
void TVIL_cprintfM (TVIL_COMPLEX);

/* Utilities: */
int  TVIL_IsNumeric (const char *);
int  TVIL_GetStatus (TVIL_DATA *);
int  TVIL_IsInfinite (TVIL_COMPLEX);
void TVIL_PrintStatus (TVIL_DATA *);
void TVIL_PrintVersion (void);
void TVIL_ResetStepSizeParams (TVIL_DATA *, TVIL_REAL, int, int, int, int);
void TVIL_Set_aParameter (TVIL_DATA *, TVIL_REAL);
void TVIL_SetContourDisplacement (TVIL_DATA *, TVIL_REAL);
void TVIL_Set_tInitial (TVIL_DATA *, TVIL_REAL);
void TVIL_SetDoAnalytic (TVIL_DATA *);
void TVIL_UnsetDoAnalytic (TVIL_DATA *);
void TVIL_SetForceContour (TVIL_DATA *);
void TVIL_UnsetForceContour (TVIL_DATA *);

/* Analytic cases: */
TVIL_COMPLEX TVIL_Dilog (TVIL_COMPLEX);
TVIL_COMPLEX TVIL_Trilog (TVIL_COMPLEX);
TVIL_COMPLEX TVIL_A (TVIL_REAL, TVIL_REAL);
TVIL_COMPLEX TVIL_Aeps (TVIL_REAL, TVIL_REAL);
TVIL_COMPLEX TVIL_Aeps2 (TVIL_REAL, TVIL_REAL);
TVIL_COMPLEX TVIL_I2  (TVIL_REAL, TVIL_REAL, TVIL_REAL, TVIL_REAL);
TVIL_COMPLEX TVIL_II2 (TVIL_REAL, TVIL_REAL, TVIL_REAL, TVIL_REAL);
TVIL_COMPLEX TVIL_II1 (TVIL_REAL, TVIL_REAL, TVIL_REAL, TVIL_REAL);
TVIL_COMPLEX TVIL_II0 (TVIL_REAL, TVIL_REAL, TVIL_REAL, TVIL_REAL);
TVIL_COMPLEX TVIL_Ieps (TVIL_REAL, TVIL_REAL, TVIL_REAL, TVIL_REAL);
int TVIL_FBARanalytic (TVIL_REAL, TVIL_REAL, TVIL_REAL, TVIL_REAL, TVIL_REAL, TVIL_COMPLEX *);
int TVIL_Ganalytic (TVIL_REAL, TVIL_REAL, TVIL_REAL, TVIL_REAL, TVIL_REAL, TVIL_REAL, TVIL_COMPLEX *);
int TVIL_Hanalytic (TVIL_REAL, TVIL_REAL, TVIL_REAL, TVIL_REAL, TVIL_REAL, TVIL_REAL, TVIL_REAL, TVIL_COMPLEX *);

#ifdef __cplusplus
}
#endif

#endif /* 3vil.h */
