/* 
  Implements a general 5-stage, 4th-order Runge-Kutta step.  The
  purpose of this (as opposed to classical 4-substep 4th order
  Runge-Kutta) is to never have to evaluate anything at the endpoint.
  Butcher coefficients a[i,j], b[i], c[i] have to be selected
  according to theory; for our purposes it is important that c[5] < 1.

  This code can be made more efficient by removing terms in the code
  involving coefficient a[i,j] that are 0. However, efficiency is not
  a big concern, because the function TVIL_rk5 should only be called
  once in each run. So all of the 0 coefficients are left in as a nod
  to generality.

  If RKmode = 0, then RKindvar = s is incremented by an amount
  RKdelta.  (Here tthresh is ignored.)

  If RKmode = 1, then RKindvar = ln(1-s/tthresh) is incremented by
  RKdelta.  (Here tthresh is the nearest threshold.)

  The component members of *foo and *RKindvar are updated at the end.
  The derivatives are NOT updated, because they might diverge causing
  a major speed hit, and because they are not needed (since this is
  ONLY to be used for the last step!)

  NOTE: do NOT call rk4 AFTER calling TVIL_rk5, because rk4 expects
  the derivatives to be up-to-date in the struct, and TVIL_rk5 doesn't
  update them!  It is permitted to call TVIL_rk5 repeatedly as a test,
  because TVIL_rk5 doesn't assume that the derivatives in the struct
  are updated. But that would be needlessly inefficient.  

*/

#include "internal.h"

/* Here is a nice set of Butcher coefficients. Others are possible.*/
#define Butcherc2 0.25L
#define Butcherc3 0.375L
#define Butcherc4 0.5L
#define Butcherc5 0.625L

#define Butchera21 0.25L
#define Butchera31 0.0L
#define Butchera32 0.375L
#define Butchera41 0.0L
#define Butchera42 0.5L
#define Butchera43 0.0L
#define Butchera51 0.0L
#define Butchera52 35.0L/72.0L
#define Butchera53 0.0L
#define Butchera54 (5.0L/36.0L)

#define Butcherb1 (-1.0L/15.0L)
#define Butcherb2 (2.0L/3.0L)
#define Butcherb3 (4.0L/3.0L)
#define Butcherb4 (-10.0L/3.0L)
#define Butcherb5 (12.0L/5.0L)

/* **************************************************************** */

void TVIL_rk5 (TVIL_DATA    *foo, 
	       TVIL_COMPLEX *RKindvar,
	       TVIL_COMPLEX  RKdelta)
{
  /* The following two used to be arguments, maybe restore someday. */
  int RKmode = 0; 
  TVIL_REAL tthresh = 0;

  static TVIL_COMPLEX k1I[NUM_I_FUNCS], k2I[NUM_I_FUNCS], k3I[NUM_I_FUNCS];
  static TVIL_COMPLEX k4I[NUM_I_FUNCS], k5I[NUM_I_FUNCS];
  static TVIL_COMPLEX startingI[NUM_I_FUNCS];

  static TVIL_COMPLEX k1FBAR[NUM_F_FUNCS], k2FBAR[NUM_F_FUNCS], k3FBAR[NUM_F_FUNCS];
  static TVIL_COMPLEX k4FBAR[NUM_F_FUNCS], k5FBAR[NUM_F_FUNCS];
  static TVIL_COMPLEX startingFBAR[NUM_F_FUNCS];

  static TVIL_COMPLEX k1G[NUM_G_FUNCS], k2G[NUM_G_FUNCS], k3G[NUM_G_FUNCS];
  static TVIL_COMPLEX k4G[NUM_G_FUNCS], k5G[NUM_G_FUNCS];
  static TVIL_COMPLEX startingG[NUM_G_FUNCS];

  static TVIL_COMPLEX k1H, k2H, k3H;
  static TVIL_COMPLEX k4H, k5H;
  static TVIL_COMPLEX startingH;

  TVIL_COMPLEX t, dt;
  int i;

  /* For convenience: */
  TVIL_REAL qq = foo->qq;

  /* Relate t, dt to the independent variable. */
  if (0 == RKmode) {
    t = *RKindvar;
    dt = RKdelta;
  }
  else {
    t = tthresh*(1.0L - TVIL_CEXP(*RKindvar));
    dt = (t - tthresh)*RKdelta;
  }

  /* Set the starting values */
  for (i=0; i<NUM_I_FUNCS; i++)
    startingI[i] = foo->II[i].value;

  for (i=0; i<NUM_F_FUNCS; i++)
    startingFBAR[i] = foo->FBAR[i].value;

  for (i=0; i<NUM_G_FUNCS; i++)
    startingG[i] = foo->G[i].value;

  startingH = foo->H.value;

  /* Fill up k1 arrays: */
  for (i=0; i<NUM_I_FUNCS; i++)
    k1I[i] = dt * TVIL_dIdt (foo->II[i], t, qq);

  for (i=0; i<NUM_F_FUNCS; i++)
    k1FBAR[i] = dt * TVIL_dFBARdt (foo->FBAR[i], t, qq);

  for (i=0; i<NUM_G_FUNCS; i++)
    k1G[i] = dt * TVIL_dGdt (foo->G[i], t, qq);

  k1H = dt * TVIL_dHdt (foo->H, t, qq);

  /* Update independent variable. */
  if (0 == RKmode) {
    t = *RKindvar + Butcherc2 * RKdelta;
  }
  else {
    t = tthresh*(1.0L - TVIL_CEXP(*RKindvar + Butcherc2 * RKdelta));
    dt = (t - tthresh) * RKdelta;
  }

  /* Adjust data values */
  for (i=0; i<NUM_I_FUNCS; i++)
    foo->II[i].value = startingI[i] + Butchera21 * k1I[i];

  for (i=0; i<NUM_F_FUNCS; i++)
    foo->FBAR[i].value = startingFBAR[i] + Butchera21 * k1FBAR[i];

  for (i=0; i<NUM_G_FUNCS; i++)
    foo->G[i].value = startingG[i] + Butchera21 * k1G[i];

  foo->H.value = startingH + Butchera21 * k1H;

  /* Fill up k2 arrays: */
  for (i=0; i<NUM_I_FUNCS; i++)
    k2I[i] = dt * TVIL_dIdt (foo->II[i], t, qq);

  for (i=0; i<NUM_F_FUNCS; i++)
    k2FBAR[i] = dt * TVIL_dFBARdt (foo->FBAR[i], t, qq);

  for (i=0; i<NUM_G_FUNCS; i++)
    k2G[i] = dt * TVIL_dGdt (foo->G[i], t, qq);

  k2H = dt * TVIL_dHdt (foo->H, t, qq);

  /* Update independent variabl. */
  if (0 == RKmode) {
    t = *RKindvar + Butcherc3 * RKdelta;
  }
  else {
    t = tthresh*(1.0L - TVIL_CEXP(*RKindvar + Butcherc3 * RKdelta));
    dt = (t - tthresh) * RKdelta;
  }

  /* Adjust data values */
  for (i=0; i<NUM_I_FUNCS; i++)
    foo->II[i].value = startingI[i] + Butchera31 * k1I[i]
                                    + Butchera32 * k2I[i];

  for (i=0; i<NUM_F_FUNCS; i++)
    foo->FBAR[i].value = startingFBAR[i] + Butchera31 * k1FBAR[i]
                                         + Butchera32 * k2FBAR[i];

  for (i=0; i<NUM_G_FUNCS; i++)
    foo->G[i].value = startingG[i] + Butchera31 * k1G[i]
                                   + Butchera32 * k2G[i];

  foo->H.value = startingH + Butchera31 * k1H
                           + Butchera32 * k2H;

  /* Fill up k3 arrays: */
  for (i=0; i<NUM_I_FUNCS; i++)
    k3I[i] = dt * TVIL_dIdt (foo->II[i], t, qq);

  for (i=0; i<NUM_F_FUNCS; i++)
    k3FBAR[i] = dt * TVIL_dFBARdt (foo->FBAR[i], t, qq);

  for (i=0; i<NUM_G_FUNCS; i++)
    k3G[i] = dt * TVIL_dGdt (foo->G[i], t, qq);

  k3H = dt * TVIL_dHdt (foo->H, t, qq);

  /* Update independent variable. */
  if (0 == RKmode) {
    t = *RKindvar + Butcherc4 * RKdelta;
  }
  else {
    t = tthresh*(1.0L - TVIL_CEXP(*RKindvar + Butcherc4 * RKdelta));
    dt = (t - tthresh) * RKdelta;
  }

  /* Adjust data values */
  for (i=0; i<NUM_I_FUNCS; i++)
    foo->II[i].value = startingI[i] + Butchera41 * k1I[i]
                                    + Butchera42 * k2I[i]
                                    + Butchera43 * k3I[i];

  for (i=0; i<NUM_F_FUNCS; i++)
    foo->FBAR[i].value = startingFBAR[i] + Butchera41 * k1FBAR[i]
                                         + Butchera42 * k2FBAR[i]
                                         + Butchera43 * k3FBAR[i];

  for (i=0; i<NUM_G_FUNCS; i++)
    foo->G[i].value = startingG[i] + Butchera41 * k1G[i]
                                   + Butchera42 * k2G[i]
                                   + Butchera43 * k3G[i];

  foo->H.value = startingH + Butchera41 * k1H
                           + Butchera42 * k2H
                           + Butchera43 * k3H;

  /* Fill up k4 arrays: */
  for (i=0; i<NUM_I_FUNCS; i++)
    k4I[i] = dt * TVIL_dIdt (foo->II[i], t, qq);

  for (i=0; i<NUM_F_FUNCS; i++)
    k4FBAR[i] = dt * TVIL_dFBARdt (foo->FBAR[i], t, qq);

  for (i=0; i<NUM_G_FUNCS; i++)
    k4G[i] = dt * TVIL_dGdt (foo->G[i], t, qq);

  k4H = dt * TVIL_dHdt (foo->H, t, qq);

  /* Update independent variable. */
  if (0 == RKmode) {
    t = *RKindvar + Butcherc5 * RKdelta;
  }
  else {
    t = tthresh*(1.0L - TVIL_CEXP(*RKindvar + Butcherc5 * RKdelta));
    dt = (t - tthresh) * RKdelta;
  }

  /* Adjust data values */
  for (i=0; i<NUM_I_FUNCS; i++)
    foo->II[i].value = startingI[i] + Butchera51 * k1I[i]
                                    + Butchera52 * k2I[i]
                                    + Butchera53 * k3I[i]
                                    + Butchera54 * k4I[i];

  for (i=0; i<NUM_F_FUNCS; i++)
    foo->FBAR[i].value = startingFBAR[i] + Butchera51 * k1FBAR[i]
                                         + Butchera52 * k2FBAR[i]
                                         + Butchera53 * k3FBAR[i]
                                         + Butchera54 * k4FBAR[i];

  for (i=0; i<NUM_G_FUNCS; i++)
    foo->G[i].value = startingG[i] + Butchera51 * k1G[i]
                                   + Butchera52 * k2G[i]
                                   + Butchera53 * k3G[i]
                                   + Butchera54 * k4G[i];

  foo->H.value = startingH + Butchera51 * k1H
                           + Butchera52 * k2H
                           + Butchera53 * k3H
                           + Butchera54 * k4H;

  /* Fill up k5 arrays: */
  for (i=0; i<NUM_I_FUNCS; i++)
    k5I[i] = dt * TVIL_dIdt (foo->II[i], t, qq);

  for (i=0; i<NUM_F_FUNCS; i++)
    k5FBAR[i] = dt * TVIL_dFBARdt (foo->FBAR[i], t, qq);

  for (i=0; i<NUM_G_FUNCS; i++)
    k5G[i] = dt * TVIL_dGdt (foo->G[i], t, qq);

  k5H = dt * TVIL_dHdt (foo->H, t, qq);

  /* printf("At the end of rk5...\n"); */
  /* printf("Last t  = "); TVIL_cprintf(t); printf("\n"); */
  /* printf("Last dt = "); TVIL_cprintf(dt); printf("\n"); */

  /* Increment data values */
  for (i=0; i<NUM_I_FUNCS; i++)
    foo->II[i].value =   startingI[i]
                       + Butcherb1 * k1I[i] + Butcherb2 * k2I[i] 
                       + Butcherb3 * k3I[i] + Butcherb4 * k4I[i] 
                       + Butcherb5 * k5I[i];

  for (i=0; i<NUM_F_FUNCS; i++)
    foo->FBAR[i].value =   startingFBAR[i]
                         + Butcherb1 * k1FBAR[i] + Butcherb2 * k2FBAR[i]
                         + Butcherb3 * k3FBAR[i] + Butcherb4 * k4FBAR[i]
                         + Butcherb5 * k5FBAR[i];

  for (i=0; i<NUM_G_FUNCS; i++)
    foo->G[i].value =   startingG[i]
                         + Butcherb1 * k1G[i] + Butcherb2 * k2G[i]
                         + Butcherb3 * k3G[i] + Butcherb4 * k4G[i]
                         + Butcherb5 * k5G[i];

  foo->H.value = startingH
                  + Butcherb1 * k1H + Butcherb2 * k2H
                  + Butcherb3 * k3H + Butcherb4 * k4H
                  + Butcherb5 * k5H;

  /* Update independent variable for next step, if there is one. */
  *RKindvar += RKdelta;

  return;
}
