/*
  Initialization (pointer alignment) in general data struct.

  This is everything that is independent of the values of the
  arguments; everything that *does* depend on u,v,w,x,y,z,qq is set in
  TVIL_SetParameters below.  This routine only ever needs to be called
  once on any TVIL_DATA object.
*/

#include "internal.h"

/* Default values: */
int printWarns = YES;
int printBold = NO;

/* **************************************************************** */

void TVIL_Construct (TVIL_DATA *foo)
{
  int i, j;

  /* These define which functions go where in the evolution
     equations: */
  static int FrefF[NUM_F_FUNCS][3] = {
    {uwxy, xuwy, yuwx}, 
    {vwxz, xvwz, zvwx},
    {uwxy, wuxy, yuwx},
    {vwxz, wvxz, zvwx},
    {vuyz, yuvz, zuvy},
    {wuxy, xuwy, yuwx}, 
    {uvyz, vuyz, zuvy},
    {uwxy, wuxy, xuwy},
    {uvyz, yuvz, zuvy},
    {wvxz, xvwz, zvwx},
    {uvyz, vuyz, yuvz},
    {vwxz, wvxz, xvwz}
  };

  /* In the following the 4 elements correspond (in order) to the four
     F terms in dFbar/dt (eq. 6.20 of current preprint): */
  static int FrefI[NUM_F_FUNCS][4] = {
    {uxy, wxy, uwy, uwx}, 
    {vxz, wxz, vwz, vwx},
    {uwy, wxy, uxy, uwx},
    {vwz, wxz, vxz, vwx},
    {vyz, uyz, uvz, uvy},
    {wxy, uxy, uwy, uwx},
    {uvz, vyz, uyz, uvy},
    {uwx, wxy, uxy, uwy},
    {uyz, vyz, uvz, uvy},
    {wxz, vxz, vwz, vwx},
    {uvy, vyz, uyz, uvz},
    {vwx, wxz, vxz, vwz}
  };

  static int ErefF[NUM_E_FUNCS][4] = {
    {wuxy, uwxy, xuwy, yuwx}, 
    {vwxz, wvxz, xvwz, zvwx},
    {uvyz, vuyz, yuvz, zuvy}
  };

  static int GrefF[NUM_G_FUNCS][4] = {
    {uvyz, zuvy, vuyz, yuvz},
    {uvyz, vuyz, yuvz, zuvy},
    {vwxz, xvwz, wvxz, zvwx},
    {vwxz, wvxz, xvwz, zvwx},
    {uwxy, xuwy, wuxy, yuwx},
    {uwxy, wuxy, xuwy, yuwx}
  };

  static int GrefI[NUM_G_FUNCS][10] = {
    {uwz, vwy, vyz, uvy, uyz, uvz, vwy, vwy, uwz, uwz},
    {uvx, xyz, vyz, uyz, uvz, uvy, xyz, xyz, uvx, uvx},
    {uvx, uwz, wxz, vwz, vxz, vwx, uwz, uwz, uvx, uvx},
    {vwy, xyz, wxz, vxz, vwz, vwx, xyz, xyz, vwy, vwy},
    {uvx, vwy, wxy, uwy, uxy, uwx, vwy, vwy, uvx, uvx},
    {uwz, xyz, wxy, uxy, uwy, uwx, xyz, xyz, uwz, uwz}
  };

  static int HrefG[6] = {
    xuvyz, wuzvy, zuwxy, vuxwy, yvwxz, uvxwz
  };

  static int HrefF[6][2] = {
    {xvwz, xuwy},
    {wvxz, wuxy},
    {zvwx, zuvy},
    {vwxz, vuyz},
    {yuvz, yuwx},
    {uvyz, uwxy}
  };

  /* First 2 (of 4) are for the Fs, last 2 are the explicit Is */
  static int HrefI[6][4] = {
    {vwz, uwy, uwz, vwy},
    {vxz, uxy, uvx, xyz},
    {vwx, uvy, uvx, vwy},
    {wxz, uyz, uwz, xyz},
    {uvz, uwx, uwz, uvx},
    {vyz, wxy, xyz, vwy}
  };

  /* Align pointers: */

  /* Fbar-type objects */
  for (i=0; i<NUM_F_FUNCS; i++)
    for (j=0; j<3; j++)
      foo->FBAR[i].fval[j] = &(foo->FBAR[FrefF[i][j]].value);

  for (i=0; i<NUM_F_FUNCS; i++)
    for (j=0; j<4; j++)
      foo->FBAR[i].ival[j] = &(foo->II[FrefI[i][j]].value);

  /* E-type objects (these point to values of F, not FBAR!) */
  for (i=0; i<NUM_E_FUNCS; i++)
    for (j=0; j<4; j++)
      foo->E[i].fval[j] = &(foo->FBAR[ErefF[i][j]].fValue);

  /* G-type objects */
  for (i=0; i<NUM_G_FUNCS; i++)
    for (j=0; j<4; j++)
      foo->G[i].fval[j] = &(foo->FBAR[GrefF[i][j]].value);

  for (i=0; i<NUM_G_FUNCS; i++)
    for (j=0; j<10; j++)
      foo->G[i].ival[j] = &(foo->II[GrefI[i][j]].value);

  for (i=0; i<6; i++) {
    foo->H.gval[i] = &(foo->G[HrefG[i]].value);
    for (j=0; j<2; j++) {
      foo->H.fval[i][j] = &(foo->FBAR[HrefF[i][j]].value);
      foo->H.ival[i][j] = &(foo->II[HrefI[i][j]].value);
      foo->H.ival[i][j+2] = &(foo->II[HrefI[i][j+2]].value);
    }
  }

  foo->tPole = calloc (sizeof(TVIL_REAL), NUM_POLES);

  /* Later a is set to 2.0*MAX(u,v,w,x,y,z) */
  a = a2 = a3 = a4 = a5 = a6 = a7 = a8 = a9 = 1.0L;

  /* Set default Runge-Kutta parameters.  */
  /* Evaluation routines: */
  foo->RKstepper6 = &TVIL_rk6;
  foo->RKstepper5 = &TVIL_rk5;

  /* These can be reset by calling TVIL_ResetStepSizeParams */
  foo->precisionGoal = TVIL_PRECISION_GOAL;
  foo->nStepsStart   = TVIL_NSTEPS_START;
  foo->nStepsMin     = TVIL_NSTEPS_MIN;
  foo->nStepsMaxCon  = TVIL_NSTEPS_MAX_CON;
  foo->nStepsMaxVar  = TVIL_NSTEPS_MAX_VAR;
  /* This can be reset by calling TVIL_SetContourDisplacement: */
  foo->imDispl       = IM_DISPL;
  /* This can be reset by calling TVIL_Set_tInitial: */
  foo->tInit         = T_INIT;
  /* This can be toggled by TVIL_(Set/Unset)DoAnalytic: */
  foo->doAnalytic    = DOANALYTIC;
  /* This can be toggled with TVIL_(Set/Unset)ForceContour */
  foo->forceContour  = NO;

  foo->isAligned     = YES;
  foo->status        = UNEVALUATED;

  return;
}

/* **************************************************************** */
/* Sets parameters (u,v,w,x,y,z,qq) in a data struct and sub-objects,
   and determines a default value for a. */

int TVIL_SetParameters (TVIL_DATA *foo,
			TVIL_REAL u,
			TVIL_REAL v,
			TVIL_REAL w,
			TVIL_REAL x,
			TVIL_REAL y,
			TVIL_REAL z,
			TVIL_REAL qq)
{
  TVIL_REAL tmp[7], largestArg, tmpabs;
  int i;
  TVIL_REAL aMult[] = {2.0, 1.9, 1.95, 2.05};
  char funcname[] = "TVIL_SetParameters";

  /* Basic sanity checks on arguments: */
  tmp[0] = u;
  tmp[1] = v;
  tmp[2] = w;
  tmp[3] = x;
  tmp[4] = y;
  tmp[5] = z;
  tmp[6] = qq;

  if (tmp[6] < TVIL_TOL*TVIL_TOL)
    TVIL_Error("TVIL_SetParameters",
	       "Renormalization scale squared must be positive", 2);

  /* Set up data object if necessary */
  if (foo->isAligned != YES)
    TVIL_Construct (foo);

  /* Set values in the data object */
  foo->u  = u;
  foo->v  = v;
  foo->w  = w;
  foo->x  = x;
  foo->y  = y;
  foo->z  = z;
  foo->qq = qq;

  TVIL_SetParamsI (&(foo->II[uvx]), uvx, u, v, x);
  TVIL_SetParamsI (&(foo->II[xyz]), xyz, x, y, z);
  TVIL_SetParamsI (&(foo->II[uxy]), uxy, u, x, y);
  TVIL_SetParamsI (&(foo->II[vxz]), vxz, v, x, z);
  TVIL_SetParamsI (&(foo->II[uwz]), uwz, u, w, z);
  TVIL_SetParamsI (&(foo->II[vwy]), vwy, v, w, y);
  TVIL_SetParamsI (&(foo->II[uwy]), uwy, u, w, y);
  TVIL_SetParamsI (&(foo->II[vwz]), vwz, v, w, z);
  TVIL_SetParamsI (&(foo->II[vyz]), vyz, v, y, z);
  TVIL_SetParamsI (&(foo->II[wxy]), wxy, w, x, y);
  TVIL_SetParamsI (&(foo->II[uvz]), uvz, u, v, z);
  TVIL_SetParamsI (&(foo->II[uwx]), uwx, u, w, x);
  TVIL_SetParamsI (&(foo->II[uyz]), uyz, u, y, z);
  TVIL_SetParamsI (&(foo->II[wxz]), wxz, w, x, z);
  TVIL_SetParamsI (&(foo->II[uvy]), uvy, u, v, y);
  TVIL_SetParamsI (&(foo->II[vwx]), vwx, v, w, x);

  TVIL_SetParamsFBAR (&(foo->FBAR[wuxy]), wuxy, w, u, x, y);
  TVIL_SetParamsFBAR (&(foo->FBAR[wvxz]), wvxz, w, v, x, z);
  TVIL_SetParamsFBAR (&(foo->FBAR[xuwy]), xuwy, x, u, w, y);
  TVIL_SetParamsFBAR (&(foo->FBAR[xvwz]), xvwz, x, v, w, z);
  TVIL_SetParamsFBAR (&(foo->FBAR[uvyz]), uvyz, u, v, y, z);
  TVIL_SetParamsFBAR (&(foo->FBAR[uwxy]), uwxy, u, w, x, y);
  TVIL_SetParamsFBAR (&(foo->FBAR[yuvz]), yuvz, y, u, v, z);
  TVIL_SetParamsFBAR (&(foo->FBAR[yuwx]), yuwx, y, u, w, x);
  TVIL_SetParamsFBAR (&(foo->FBAR[vuyz]), vuyz, v, u, y, z);
  TVIL_SetParamsFBAR (&(foo->FBAR[vwxz]), vwxz, v, w, x, z);
  TVIL_SetParamsFBAR (&(foo->FBAR[zuvy]), zuvy, z, u, v, y);
  TVIL_SetParamsFBAR (&(foo->FBAR[zvwx]), zvwx, z, v, w, x);

  TVIL_SetParamsE (&(foo->E[0]), wuxy, w, u, x, y);
  TVIL_SetParamsE (&(foo->E[1]), vwxz, v, w, x, z);
  TVIL_SetParamsE (&(foo->E[2]), uvyz, u, v, y, z);

  TVIL_SetParamsG (&(foo->G[wuzvy]), wuzvy, w, u, z, v, y);
  TVIL_SetParamsG (&(foo->G[xuvyz]), xuvyz, x, u, v, y, z);
  TVIL_SetParamsG (&(foo->G[uvxwz]), uvxwz, u, v, x, w, z);
  TVIL_SetParamsG (&(foo->G[yvwxz]), yvwxz, y, v, w, x, z);
  TVIL_SetParamsG (&(foo->G[vuxwy]), vuxwy, v, u, x, w, y);
  TVIL_SetParamsG (&(foo->G[zuwxy]), zuwxy, z, u, w, x, y);

  TVIL_SetParamsH (&(foo->H), u, v, w, x, y, z);

  /* Find a good a value... */
  largestArg = TVIL_FABS(tmp[0]);
  for (i=1; i<6; i++)
    if ((tmpabs = TVIL_FABS(tmp[i])) > largestArg) largestArg = tmpabs;

  for (i=0; i<4; i++) {

    a = aMult[i]*largestArg;

    /* printf("Testing a multiplier %Lf...", aMult[i]); */

    if ( (TVIL_FABS(TVIL_Phi (u,w,x,y)) < 
            (u*u*u*u + w*w*w*w + x*x*x*x + y*y*y*y) * TVIL_TOL && 
	  TVIL_FABS(TVIL_Delta (u,w,x,y)) > (u*u + w*w + x*x + y*y) * TVIL_TOL)
	 ||
	 (TVIL_FABS(TVIL_Phi (w,v,x,z)) < 
            (w*w*w*w + v*v*v*v + x*x*x*x + z*z*z*z) * TVIL_TOL &&
	  TVIL_FABS(TVIL_Delta (w,v,x,z)) > (w*w + v*v + x*x + z*z) * TVIL_TOL)
	 ||
	 (TVIL_FABS(TVIL_Phi (u,v,y,z)) < 
            (u*u*u*u + v*v*v*v + y*y*y*y + z*z*z*z) * TVIL_TOL &&
	  TVIL_FABS(TVIL_Delta (u,v,y,z)) > (u*u + v*v + y*y + z*z) * TVIL_TOL) ) {
      /* printf ("problematic!\n"); */
      ;
    }
    else {
      /* printf ("good value.\n"); */
      break;
    }
  }
  if (i==4)
    TVIL_Error (funcname, "This should never happen! No valid a found.", 45);

  /* User can change this later if desired: */
  foo->aParameter = a;

  /* Turn on warning messages by default: */
  printWarns         = YES;
  foo->isInitialized = TRUE;
  foo->status        = UNEVALUATED;

  return 0;
}

/* **************************************************************** */
/* This should be called immediately prior to generic evaluation. */

int TVIL_SetupFunctions (TVIL_DATA *foo)
{
  int i, j;

  /* Not all of these need to be set for what happens in this routine,
     but they are needed for the generic evaulation which must
     inevitably follow. */
  a = foo->aParameter;
  a2 = a*a;
  a3 = a2*a;
  a4 = a2*a2;
  a5 = a2*a3;
  a6 = a3*a3;
  a7 = a4*a3;
  a8 = a4*a4;
  a9 = a4*a5;

  /* Set values in all sub-objects: */
  for (i=0; i<NUM_I_FUNCS; i++)
    TVIL_ConstructI (&(foo->II[i]));

  for (i=0; i<NUM_F_FUNCS; i++)
    TVIL_ConstructFBAR (&(foo->FBAR[i]));

  for (i=0; i<NUM_G_FUNCS; i++)
    TVIL_ConstructG (&(foo->G[i]));

  TVIL_ConstructH (&(foo->H));

  /* Calculate pole locations... */
  j = 0;

  /* Poles associated with I functions: */
  for (i=0; i<NUM_I_FUNCS; i++) {
    if ((foo->II[i]).p[0] > 0 && (foo->II[i]).p[0] < 1.0L + THRESH_CUTOFF) {
      foo->tPole[j++] = (foo->II[i]).p[0];
      /* printf("Pole added, j=%d now\n", j); */
    }
    if ((foo->II[i]).p[1] > 0 && (foo->II[i]).p[1] < 1.0L + THRESH_CUTOFF) {
      foo->tPole[j++] = (foo->II[i]).p[1];
      /* printf("Pole added, j=%d now\n", j); */
    }
  }

  /* Poles associated with FBAR functions: */
  for (i=0; i<NUM_F_FUNCS; i++) {
    if ((i % 2 == 0) && (foo->FBAR[i]).p2 > 0 && (foo->FBAR[i]).p2 < 1.0L + THRESH_CUTOFF) {
      foo->tPole[j++] = (foo->FBAR[i]).p2;
      /* printf("Pole added, j=%d now\n", j); */
    }
    if ((foo->FBAR[i]).p3 > 0 && (foo->FBAR[i]).p3 < 1.0L + THRESH_CUTOFF) {
      foo->tPole[j++] = (foo->FBAR[i]).p3;
      /* printf("Pole added, j=%d now\n", j); */
    }
  }

  /* Poles associated with H function: */
  for (i=0; i<3; i++) {
    if ((foo->H).cHd1roots[i] > 0 && (foo->H).cHd1roots[i] < 1.0L + THRESH_CUTOFF) {
      foo->tPole[j++] = (foo->H).cHd1roots[i];
	/* printf("Pole added, j=%d now\n", j); */
    }
  }

  foo->nPoles = j;

  qsort (foo->tPole, foo->nPoles, sizeof(TVIL_REAL), TVIL_Compare);

  if (foo->nPoles > 0 && foo->tPole[0] < 1.0L)
    foo->reAxisOK = NO;
  else
    foo->reAxisOK = YES;

  foo->poleAt1 = NO;
  for (i=0; i<foo->nPoles; i++)
    if (TVIL_FABS (foo->tPole[i] - 1.0L) < THRESH_CUTOFF) {
      foo->poleAt1 = YES;
      break;
    }

  return 0;
}
