/* Routines for setup and initialization of H-type functions */

#include "internal.h"

/* ****************************************************************** */

TVIL_COMPLEX TVIL_H00000x (TVIL_REAL x, TVIL_REAL qq)
{
  TVIL_COMPLEX lnbarx = TVIL_LOG (TVIL_FABS(x)/qq);
  if (x<0) lnbarx = lnbarx - I * PI;

  return PI2*PI2/30.L + 6.L*Zeta3 - 6.L*lnbarx*Zeta3;
}

/* ****************************************************************** */

TVIL_COMPLEX TVIL_H0000xx (TVIL_REAL x, TVIL_REAL qq)
{
  TVIL_COMPLEX lnbarx = TVIL_LOG (TVIL_FABS(x)/qq);
  if (x<0) lnbarx = lnbarx - I * PI;

  return -PI2*PI2/18.L + 6.L*Zeta3 - 6.L*lnbarx*Zeta3;
}

/* ****************************************************************** */

TVIL_COMPLEX TVIL_H00xx00 (TVIL_REAL x, TVIL_REAL qq)
{
  TVIL_COMPLEX lnbarx = TVIL_LOG (TVIL_FABS(x)/qq);
  if (x<0) lnbarx = lnbarx - I * PI;

  return 16.L*A4TRANS - (7.L*PI2*PI2)/60.L - (2.L*PI2*LN2*LN2)/3.L +
      (2.L*TVIL_POW(LN2,4))/3.L + 6.L*Zeta3 - 6.L*lnbarx*Zeta3;
}

/* ****************************************************************** */

TVIL_COMPLEX TVIL_H000xxx (TVIL_REAL x, TVIL_REAL qq)
{
  TVIL_COMPLEX lnbarx = TVIL_LOG (TVIL_FABS(x)/qq);
  if (x<0) lnbarx = lnbarx - I * PI;

  return (-11.L*PI2*PI2)/180.L - 9.L*LS2*LS2 + 6.L*Zeta3 - 6.L*lnbarx*Zeta3;
}

/* ****************************************************************** */

TVIL_COMPLEX TVIL_H00x0xx (TVIL_REAL x, TVIL_REAL qq)
{
  TVIL_COMPLEX lnbarx = TVIL_LOG (TVIL_FABS(x)/qq);
  if (x<0) lnbarx = lnbarx - I * PI;

  return -PI2*PI2/10.L + 6.L*Zeta3 - 6.L*lnbarx*Zeta3;
}

/* ****************************************************************** */

TVIL_COMPLEX TVIL_H00xx0x (TVIL_REAL x, TVIL_REAL qq)
{
  TVIL_COMPLEX lnbarx = TVIL_LOG (TVIL_FABS(x)/qq);
  if (x<0) lnbarx = lnbarx - I * PI;

  return -PI2*PI2/24.L - (27.L*LS2*LS2)/2.L + 6.L*Zeta3 - 6.L*lnbarx*Zeta3;
}

/* ****************************************************************** */

TVIL_COMPLEX TVIL_H00xxxx (TVIL_REAL x, TVIL_REAL qq)
{
  TVIL_COMPLEX lnbarx = TVIL_LOG (TVIL_FABS(x)/qq);
  if (x<0) lnbarx = lnbarx - I * PI;

  return (-77.L*PI2*PI2)/1080.L - (27.L*LS2*LS2)/2.L + 6.L*Zeta3 - 6.L*lnbarx*Zeta3;
}

/* ****************************************************************** */

TVIL_COMPLEX TVIL_H0xxx0x (TVIL_REAL x, TVIL_REAL qq)
{
  TVIL_COMPLEX lnbarx = TVIL_LOG (TVIL_FABS(x)/qq);
  if (x<0) lnbarx = lnbarx - I * PI;

  return 32.L*A4TRANS - (11.L*PI2*PI2)/45.L - (4.L*PI2*LN2*LN2)/3.L +
      (4.L*TVIL_POW(LN2,4))/3.L + 6.L*Zeta3 - 6.L*lnbarx*Zeta3;
}

/* ****************************************************************** */

TVIL_COMPLEX TVIL_H0xxxxx (TVIL_REAL x, TVIL_REAL qq)
{
  TVIL_COMPLEX lnbarx = TVIL_LOG (TVIL_FABS(x)/qq);
  if (x<0) lnbarx = lnbarx - I * PI;

  return (7.L*PI2*PI2)/30.L - 6.L*LS2*LS2 + 4.L*PI*LS3 - 
    6.L*LSP4 + 6.L*Zeta3 - (26.L*TVIL_LOG(3.L)*Zeta3)/3.L - 6.L*lnbarx*Zeta3;
}

/* ****************************************************************** */

TVIL_COMPLEX TVIL_Hxxxxxx (TVIL_REAL x, TVIL_REAL qq)
{
  TVIL_COMPLEX lnbarx = TVIL_LOG (TVIL_FABS(x)/qq);
  if (x<0) lnbarx = lnbarx - I * PI;

  return 16.L*A4TRANS - (17.L*PI2*PI2)/90.L - (2.L*PI2*LN2*LN2)/3.L +
      (2.L*TVIL_POW(LN2,4))/3.L - 9.L*LS2*LS2 + 6.L*Zeta3 - 6.L*lnbarx*Zeta3;
}

/* ****************************************************************** */

TVIL_COMPLEX TVIL_H0000xy (TVIL_REAL X, TVIL_REAL Y, TVIL_REAL qq)
{
  TVIL_REAL temp, x, y, r, rn;
  TVIL_COMPLEX lnyox, lnyox2, lnyox3, result;
  int n;

  TVIL_REAL H0000xytableA[] = 
{0, -5.771508654754528, -0.9908202605290377, -0.4554997637691356, 
-0.2722553172561659, -0.18340689113100384, -0.13264301025945624, 
-0.10065188364596245, -0.07910335295050075, -0.06386299666644402, 
-0.05267152266389563, -0.044203457720624506, -0.037637189824228756, 
-0.03244045032314036, -0.02825566197451801, -0.024835167451514817, 
-0.022002920684058513, -0.019630936576686797, -0.017624326231044263, 
-0.015911510773175264, -0.014437658599678937, -0.013160185208758335, 
-0.012045605508904489, -0.011067292553681857, -0.010203855801378753, 
-0.00943795038884998, -0.008755391144546015, -0.008144485256976947, 
-0.007595523964870454, -0.007100391345518764, -0.006652260325854115, 
-0.006245354357771404, -0.00587475901885709, -0.00553627192336581, 
-0.005226282284630873, -0.004941673612817055, -0.004679744600887353, 
-0.004438144411544276, -0.0042148194431325114, -0.004007969303397238, 
-0.003816010213595261, -0.0036375444426091225, -0.0034713346609491907, 
-0.0033162823294021204, -0.0031714094124283415, -0.0030358428439824087, 
-0.0029088012819856734, -0.00278958377382435, -0.0026775600239582753, 
-0.0025721620098190234, -0.0024728767365571197, -0.00237923995711857, 
-0.0022908307133314987, -0.0022072665775220157, -0.002128199493719271};

  TVIL_REAL H0000xytableAL[] = 
{0, 6.289868133696452, 2.519934066848226, 1.707733822343262, 
1.320730922313002, 1.0847514045170683, 0.9232187630234829, 
0.8048386287528784, 0.7140153690363196, 0.6419756690081806, 
0.5833609945235882, 0.5346961183563276, 0.4936210803474843, 
0.4584735278777824, 0.42804693073942135, 0.40144343343261524, 
0.37798055165205013, 0.35712994121711616, 0.33847602069749777, 
0.32168729427003384, 0.30649603279040655, 0.2926835938334291, 
0.28006963046066846, 0.2685040345737776, 0.2578608371971804, 
0.24803353146632695, 0.23893144485609086, 0.2304768953898546, 
0.2226029406763021, 0.21525158018351165, 0.20837230756127628, 
0.20192093586638576, 0.19585863741054743, 0.19015115377287606, 
0.18476814175476985, 0.1796826287106614, 0.17487055646735347, 
0.17031039744545307, 0.16598282997502245, 0.1618704624114089, 
0.15795759769408763, 0.15423003158950496, 0.15067487912096633, 
0.14728042469138464, 0.14403599220610044, 0.14093183214699528, 
0.13795902306937347, 0.1351093854154818, 0.13237540588311347, 
0.12975017087012625, 0.1272273077481267, 0.1248009329106759, 
0.12246560570078144, 0.12021628745522983, 0.11804830501433407};

  TVIL_REAL H0000xytableAL2[] = 
{0, -1., -0.25, -0.1111111111111111, -0.0625, -0.04, -0.027777777777777776, 
-0.02040816326530612, -0.015625, -0.012345679012345678, -0.01, 
-0.008264462809917356, -0.006944444444444444, -0.005917159763313609, 
-0.00510204081632653, -0.0044444444444444444, -0.00390625, 
-0.0034602076124567475, -0.0030864197530864196, -0.002770083102493075, 
-0.0025, -0.0022675736961451248, -0.002066115702479339, 
-0.001890359168241966, -0.001736111111111111, -0.0016, 
-0.0014792899408284023, -0.0013717421124828531, -0.0012755102040816326, 
-0.0011890606420927466, -0.0011111111111111111, -0.001040582726326743, 
-0.0009765625, -0.0009182736455463728, -0.0008650519031141869, 
-0.0008163265306122449, -0.0007716049382716049, -0.0007304601899196494, 
-0.0006925207756232687, -0.0006574621959237344, -0.000625, 
-0.000594883997620464, -0.0005668934240362812, -0.0005408328826392645, 
-0.0005165289256198347, -0.0004938271604938272, -0.0004725897920604915, 
-0.0004526935264825713, -0.00043402777777777775, 
-0.00041649312786339027, -0.0004, -0.00038446751249519417, 
-0.0003698224852071006, -0.000355998576005696, -0.0003429355281207133};

  TVIL_REAL H0000xytableAL3[] = 
{0, 0.16666666666666666, 0.08333333333333333, 0.05555555555555555, 
0.041666666666666664, 0.03333333333333333, 0.027777777777777776, 
0.023809523809523808, 0.020833333333333332, 0.018518518518518517, 
0.016666666666666666, 0.015151515151515152, 0.013888888888888888, 
0.01282051282051282, 0.011904761904761904, 0.011111111111111112, 
0.010416666666666666, 0.00980392156862745, 0.009259259259259259, 
0.008771929824561403, 0.008333333333333333, 0.007936507936507936, 
0.007575757575757576, 0.007246376811594203, 0.006944444444444444, 
0.006666666666666667, 0.00641025641025641, 0.006172839506172839, 
0.005952380952380952, 0.005747126436781609, 0.005555555555555556, 
0.005376344086021506, 0.005208333333333333, 0.005050505050505051, 
0.004901960784313725, 0.004761904761904762, 0.004629629629629629, 
0.0045045045045045045, 0.0043859649122807015, 0.004273504273504274, 
0.004166666666666667, 0.0040650406504065045, 0.003968253968253968, 
0.003875968992248062, 0.003787878787878788, 0.003703703703703704, 
0.0036231884057971015, 0.0035460992907801418, 0.003472222222222222, 
0.003401360544217687, 0.0033333333333333335, 0.0032679738562091504, 
0.003205128205128205, 0.0031446540880503146, 0.0030864197530864196};

  TVIL_REAL H0000xytableB[] = 
{0, -3.606170709478783, 1.3030853547393915, -0.7020569031595942, 
0.44899638107339945, -0.3161415493031639, 0.23677227874029094, 
-0.18513020717421758, 0.14942386507238703, -0.12358586339347805, 
0.10421485228998595, -0.08927459372384941, 0.07748095467614455, 
-0.06798911436855568, 0.06022342172231385, -0.053779853118521376, 
0.04836757525410332, -0.04377260022553214, 0.03983443999339681, 
-0.03643068369021216, 0.033466554347372, -0.030867680989995816, 
0.02857499751036313, -0.026541078850438687, 0.024727467512195206, 
-0.023102694466010182, 0.021640794766869628, -0.020320180772936194, 
0.019122777326742807, -0.018033351205936965, 0.01703898628391136, 
-0.01612866913134285, 0.015292959146436497, -0.01452372397143166, 
0.013813925762716603, -0.01315744738809714, 0.012548950206661622, 
-0.011983757006052656, 0.011457755111565354, -0.010967315770349928, 
0.010509226744095143, -0.010080635681143882, 0.009679002332162678, 
-0.009302058057522974, 0.008947771375507549, -0.008614318537730273, 
0.00830005830630419, -0.008003510257300686, 0.007723336055267219, 
-0.007458323240407891, 0.007207371148400615, -0.0069694786465391845, 
0.006743733421913201, -0.006529302599998385, 0.0063254245071484305};

  x = X; y = Y;
  if (x < y) {temp = x; x = y; y = temp;}
  lnyox = TVIL_CLOG (y/x);
  lnyox2 = lnyox * lnyox;
  lnyox3 = lnyox2 * lnyox;

  if (y < 0.5 * x) {
    result = TVIL_H00000x (x, qq);
    rn = r = y/x;
    for (n = 1; n < 55; n++) {
      result += rn * (H0000xytableA[n] + 
                      H0000xytableAL[n] * lnyox +
                      H0000xytableAL2[n] * lnyox2 +
                      H0000xytableAL3[n] * lnyox3);
      rn *= r;
    }
  }
  else {
    result = TVIL_H0000xx (x, qq);
    rn = r = (y/x - 1.L);
    for (n = 1; n < 55; n++) {
      result += rn * H0000xytableB[n]; 
      rn *= r;
    }
  }

  return result;
}

/* ****************************************************************** */

TVIL_COMPLEX TVIL_H00xy00 (TVIL_REAL X, TVIL_REAL Y, TVIL_REAL qq)
{
  TVIL_REAL temp, x, y, r, rn;
  TVIL_COMPLEX lnyox, lnyox2, lnyox3, result;
  int n;

  TVIL_REAL H00xy00tableA[] = 
     {0, -9.79207768635047, 0.7511047763270091, -0.4191462310912395, 
     0.20621701959959626, -0.13722069745019216, 0.09193846614273049, 
     -0.06843562038543838, 0.05165372870784368, -0.04105357365173534, 
     0.033019836413298975, -0.02737467270849847, 0.022910932916912707, 
     -0.019558060340949424, 0.016822339978673106, -0.014671608279901882, 
     0.012873979604384522, -0.011413178354922782, 0.010168751558653266, 
     -0.009131772457131593, 0.008234679617107499, -0.007472254436954373, 
     0.006804238346668323, -0.00622741874813848, 0.005716602689091299, 
     -0.005269726260277052, 0.0048703735969110595, -0.004517170409153615, 
     0.004199050275951688, -0.003915069785280917, 0.003657549501413143, 
     -0.003425824577737352, 0.003214431127257694, -0.0030228897954743816, 
     0.0028472274732536385, -0.002687091549342435, 0.0025395370938717243, 
     -0.002404300086827793, 0.0022791603410184103, -0.002163917021350853, 
     0.002056870781620146, -0.0019578666706048797, 0.0018655861862000023, 
     -0.001779908241255207, 0.0016997971848202176, -0.0016251572911653855, 
     0.001555165964983285, -0.0014897467282653787, 0.0014282395391937985, 
     -0.0013705827057139408, 0.0013162416688140832, -0.0012651662216932036, 
     0.0012169197168695745, -0.0011714609604231707, 0.001128430461449892};
 
  TVIL_REAL H00xy00tableAL[] = 
     {0, 6.579736267392905, -1.7898681336964524, 1.1932454224643019, 
     -0.846322955737115, 0.6770583645896922, -0.5560671556765954, 
     0.47662899057993885, -0.41465878512479337, 0.3685855867775942, 
     -0.3307887564948964, 0.30071705135899685, -0.2752172909572561, 
     0.2540467301143903, -0.2356676439785089, 0.219956467713275, 
     -0.20607463987008415, 0.19395260223066746, -0.1830943936935481, 
     0.17345784665704553, -0.16473093770369465, 0.15688660733685203, 
     -0.14971876918632868, 0.1432092574825752, -0.13721649707794614, 
     0.13172783719482833, -0.12664281113977002, 0.12195233665311186, 
     -0.11758314864287198, 0.11352855731035913, -0.10973387879588299, 
     0.10619407625408032, -0.10286750884284943, 0.09975031160518737, 
     -0.09681021752945451, 0.09404421131432726, -0.09142690304533586, 
     0.08895590566573218, -0.08661096715723433, 0.08439017312756168, 
     -0.08227717257978025, 0.08027041227295637, -0.07835654621183032, 
     0.07653430095109008, -0.07479267556974536, 0.07313061611263992, 
     -0.07153897338250136, 0.0700168675658524, -0.06855662734583841, 
     0.06715751250204577, -0.06581304280177581, 0.06452259098213316, 
     -0.0632806453842238, 0.06208667094301203, -0.060935950144224024};
 
  TVIL_REAL H00xy00tableAL2[] = 
     {0, -1., 0.25, -0.1111111111111111, 0.0625, -0.04, 
     0.027777777777777776, -0.02040816326530612, 0.015625, 
     -0.012345679012345678, 0.01, -0.008264462809917356, 
     0.006944444444444444, -0.005917159763313609, 0.00510204081632653, 
     -0.0044444444444444444, 0.00390625, -0.0034602076124567475, 
     0.0030864197530864196, -0.002770083102493075, 0.0025, 
     -0.0022675736961451248, 0.002066115702479339, -0.001890359168241966, 
     0.001736111111111111, -0.0016, 0.0014792899408284023, 
     -0.0013717421124828531, 0.0012755102040816326, -0.0011890606420927466, 
     0.0011111111111111111, -0.001040582726326743, 0.0009765625, 
     -0.0009182736455463728, 0.0008650519031141869, -0.0008163265306122449, 
     0.0007716049382716049, -0.0007304601899196494, 0.0006925207756232687, 
     -0.0006574621959237344, 0.000625, -0.000594883997620464, 
     0.0005668934240362812, -0.0005408328826392645, 0.0005165289256198347, 
     -0.0004938271604938272, 0.0004725897920604915, -0.0004526935264825713, 
     0.00043402777777777775, -0.00041649312786339027, 0.0004, 
     -0.00038446751249519417, 0.0003698224852071006, -0.000355998576005696, 
     0.0003429355281207133};
 
  TVIL_REAL H00xy00tableAL3[] = 
     {0, 0.3333333333333333, -0.16666666666666666, 
     0.1111111111111111, -0.08333333333333333, 0.06666666666666667, 
     -0.05555555555555555, 0.047619047619047616, -0.041666666666666664, 
     0.037037037037037035, -0.03333333333333333, 0.030303030303030304, 
     -0.027777777777777776, 0.02564102564102564, -0.023809523809523808, 
     0.022222222222222223, -0.020833333333333332, 0.0196078431372549, 
     -0.018518518518518517, 0.017543859649122806, -0.016666666666666666, 
     0.015873015873015872, -0.015151515151515152, 0.014492753623188406, 
     -0.013888888888888888, 0.013333333333333334, -0.01282051282051282, 
     0.012345679012345678, -0.011904761904761904, 0.011494252873563218, 
     -0.011111111111111112, 0.010752688172043012, -0.010416666666666666, 
     0.010101010101010102, -0.00980392156862745, 0.009523809523809525, 
     -0.009259259259259259, 0.009009009009009009, -0.008771929824561403, 
     0.008547008547008548, -0.008333333333333333, 0.008130081300813009, 
     -0.007936507936507936, 0.007751937984496124, -0.007575757575757576, 
     0.007407407407407408, -0.007246376811594203, 0.0070921985815602835, 
     -0.006944444444444444, 0.006802721088435374, -0.006666666666666667, 
     0.006535947712418301, -0.00641025641025641, 0.006289308176100629, 
     -0.006172839506172839};
 
  TVIL_REAL H00xy00tableB[] = 
     {0, -3.606170709478783, 1.3444198410583277, -0.7433913894785305, 
     0.4854159390174413, -0.34764617887231086, 0.26410724373726213, 
     -0.20904077140173205, 0.17052736121150414, -0.14237169300359953, 
     0.12106815577282903, -0.10449992028742372, 0.09132132775302415, 
     -0.08064049248844635, 0.07184548325983225, -0.06450403333709659, 
     0.05830309176370446, -0.053011010668709224, 0.04845312414495573, 
     -0.04449553120072153, 0.04103406033139695, -0.03798659580825092, 
     0.035287637580184106, -0.03288437720791906, 0.03073382269569433, 
     -0.02880066171427778, 0.027055652884628245, -0.025474400177053547, 
     0.024036408956822714, -0.02272435161000705, 0.021523490880894364, 
     -0.020421223127560756, 0.01940671364231006, -0.01847060329077066, 
     0.017604770863683256, -0.016802139293195906, 0.016056516660319542, 
     -0.015362464988553981, 0.014715191374112147, -0.014110457182627087, 
     0.013544501943618573, -0.013013979268006029, 0.012515902652041028, 
     -0.012047599451010346, 0.01160667163591271, -0.011190962206949526, 
     0.010798526344768966, -0.01042760654587789, 0.010076611121519571, 
     -0.009744095546565093, 0.00942874623192927, -0.009129366364862737, 
     0.008844863519416657, -0.008574238786983476, 0.008316577216074247};

  x = X; y = Y;
  if (x < y) {temp = x; x = y; y = temp;}
  lnyox = TVIL_CLOG (y/x);
  lnyox2 = lnyox * lnyox;
  lnyox3 = lnyox2 * lnyox;

  if (y < 0.5 * x) {
    result = TVIL_H00000x (x, qq);
    rn = r = y/x;
    for (n = 1; n < 55; n++) {
      result += rn * (H00xy00tableA[n] + 
                      H00xy00tableAL[n] * lnyox +
                      H00xy00tableAL2[n] * lnyox2 +
                      H00xy00tableAL3[n] * lnyox3);
      rn *= r;
    }
  }
  else {
    result = TVIL_H00xx00 (x, qq);
    rn = r = (y/x - 1.L);
    for (n = 1; n < 55; n++) {
      result += rn * H00xy00tableB[n]; 
      rn *= r;
    }
  }

  return result;
}

/* ****************************************************************** */

TVIL_COMPLEX TVIL_Hxx00yy (TVIL_REAL X, TVIL_REAL Y, TVIL_REAL qq)
{
  TVIL_REAL temp, x, y, r, rn;
  TVIL_COMPLEX lnyox, lnyox2, result;
  int n;

  TVIL_REAL Hxx00yytableA[] = 
      {0, -5.803808507330341, -1.1070235056985824, 
     -0.39659361124680936, -0.18764389686941385, -0.10403747946959108, 
     -0.06392637368694487, -0.042216980337300924, -0.029409935226498707, 
     -0.021349313511786594, -0.016013382287494937, -0.012335206198216042, 
     -0.009714170980258285, -0.0077939341522410555, -0.006353655064130437, 
     -0.005251338280434115, -0.004392763214592366, -0.003713682498096005, 
     -0.003169230399269085, -0.002727410486290549, -0.002364974043723156, 
     -0.002064743885343223, -0.0018138364827109954, -0.001602456357289339, 
     -0.0014230632207882232, -0.0012697868405164292, -0.001138009579365884, 
     -0.0010240643396874916, -0.0009250131615211044, -0.0008384829898046564, 
     -0.0007625424951059889, -0.0006956087331940197, -0.000636375736878303, 
     -0.0005837593978922917, -0.0005368545667244762, -0.0004949014002961771, 
     -0.0004572587696264341, -0.00042338310080741737, -0.000392811429245965, 
     -0.0003651477445470175, -0.00034005192288846797, 
     -0.00031723070704706147, -0.0002964303167277138, 
     -0.00027743036440311397, -0.00026003882231806497, 
     -0.0002440878402823199, -0.00022943025550087315, 
     -0.00021593666798258582, -0.00020349298027200202, 
     -0.00019199832002797512, -0.00018136327957818388, 
     -0.00017150841895413604, -0.0001623629887732292, 
     -0.00015386383723026142, -0.0001459544718103445};
 
  TVIL_REAL Hxx00yytableAL[] = 
     {0, 4.710131866303548, 1.6328437109295515, 0.8396735850641455, 
     0.5160080232652293, 0.35128966376209514, 0.2556253487463144, 
     0.19493365057579692, 0.15390900397981067, 0.12482139132031672, 
     0.10341336426723124, 0.0871781336316711, 0.07455918422684954, 
     0.06454727506201285, 0.05646402596920924, 0.04983936503964295, 
     0.04433896518515787, 0.03971961617652872, 0.03580086331623464, 
     0.03244646126274395, 0.029551937447689886, 0.027036064359842765, 
     0.024834894479562414, 0.02289751239201923, 0.021182960405754042, 
     0.019657980545688153, 0.018295333751607606, 0.017072533268224235, 
     0.015970879315570147, 0.014974715666670801, 0.014070851569770601, 
     0.01324810819596009, 0.012496959807985639, 0.011809247650515442, 
     0.011177950157322133, 0.010596997126397376, 0.010061118483949666, 
     0.009565720454195376, 0.00910678359030076, 0.00868077835475943, 
     0.008284594872754611, 0.007915484196918454, 0.007571008972245399, 
     0.007249001816492226, 0.006947530064186927, 0.0066648657835836345, 
     0.0063994601821420055, 0.00614992167984893, 0.005914997060393681, 
     0.005693555215049564, 0.005484573078630137, 0.005287123425333204, 
     0.0051003642479615086, 0.004923529489493095, 0.0047559209332825245};

  TVIL_REAL Hxx00yytableAL2[] = 
     {0, -1., -0.5833333333333334, -0.4111111111111111, 
     -0.31726190476190474, -0.25825396825396824, -0.21773689273689273, 
     -0.18820148105862392, -0.1657179625929626, -0.1480310720506799, 
     -0.1337542806350856, -0.12198834642544183, -0.11212458325713812, 
     -0.10373630170481388, -0.0965155303434345, -0.09023441835885304, 
     -0.08472077527594087, -0.0798420550358441, -0.07549456873314729, 
     -0.07159603748964931, -0.06808033817926941, -0.06489372398910566, 
     -0.061992058464711355, -0.05933875955645778, -0.05690324977588434, 
     -0.054659772846073464, -0.05258647957556952, -0.05066471411053239, 
     -0.048878451129775895, -0.04721384801073383, -0.04565888546875645, 
     -0.04420307692878923, -0.04283723176682806, -0.0415532611213545, 
     -0.04034401760492567, -0.03920316220692631, -0.038125053154143065, 
     -0.0371046526161107, -0.03613744799971259, -0.03521938523902766, 
     -0.03434681200045703, -0.03351642912537886, -0.03272524894936139, 
     -0.031970559388009204, -0.031249892879642034, -0.030560999435430815, 
     -0.029901823176904922, -0.029270481845468755, -0.028665248853805114, 
     -0.02808453751874225, -0.027526887172407814, -0.02699095089570893, 
     -0.02647548465727964, -0.025979337673544503, -0.025501443832673828};
 
  TVIL_REAL Hxx00yytableB[] = 
     {0, -3.606170709478783, 1.2512855644747463, -0.6502571128949493, 
     0.40349958040532413, -0.2769477382316584, 0.2029143496567781, 
     -0.1556410524701204, 0.12350357552552654, -0.10060172816807414, 
     0.08367027685191816, -0.07077801775591966, 0.06072083684993612, 
     -0.052714983972890705, 0.046231770792222265, -0.04090364270863467, 
     0.0364683840117164, -0.032734697929903384, 0.029560269323487598, 
     -0.026837380162998276, 0.024483238744118313, -0.022433332269491985, 
     0.02063676627266295, -0.019052938404268707, 0.017649126081082545, 
     -0.016398711194005544, 0.015279856125493874, -0.014274504227028348, 
     0.013367616731442974, -0.012546584110916748, 0.01180076763039252, 
     -0.011121139110225886, 0.010499995506321483, -0.009930731015061187, 
     0.009407653789464038, -0.008925837531778946, 0.008481000558839866, 
     -0.00806940666249338, 0.007687783376888871, -0.007333254235995304, 
     0.007003282342622416, -0.006695623134924875, 0.006408284671608699, 
     -0.00613949409480033, 0.00588766919331174, -0.005651394196297593, 
     0.005429399091112341, -0.005220541889360816, 0.005023793369147683, 
     -0.0048382239050493575, 0.004662992064719984, -0.0044973347056663365, 
     0.004340558350199891, -0.00419203165293526, 0.004051178805057246};

  x = X; y = Y;
  if (x < y) {temp = x; x = y; y = temp;}
  lnyox = TVIL_CLOG (y/x);
  lnyox2 = lnyox * lnyox;

  if (y < 0.5 * x) {
    result = TVIL_H0000xx (x, qq);
    rn = r = y/x;
    for (n = 1; n < 55; n++) {
      result += rn * (Hxx00yytableA[n] + 
                      Hxx00yytableAL[n] * lnyox +
                      Hxx00yytableAL2[n] * lnyox2);
      rn *= r;
    }
  }
  else {
    result = TVIL_H0xxx0x (x, qq);
    rn = r = (y/x - 1.L);
    for (n = 1; n < 55; n++) {
      result += rn * Hxx00yytableB[n]; 
      rn *= r;
    }
  }

  return result;
}

/* ****************************************************************** */

TVIL_COMPLEX TVIL_Hxx0yxx (TVIL_REAL x, TVIL_REAL y, TVIL_REAL qq)
{
  TVIL_REAL r, rn;
  TVIL_COMPLEX lnyox, lnyox2, lnyox3, result;
  int n;
  
  lnyox = TVIL_CLOG(y/x);
  lnyox2 = lnyox  * lnyox;
  lnyox3 = lnyox2  * lnyox;

  TVIL_REAL Hxx0yxxtableA[] = 
      {0,-7.17120335576568, -4.770786863068025, -3.643141859329301, 
     -2.976352507227782, -2.5310036189189566, -2.2102281744852936, 
     -1.9669857405038749, -1.7755223624616479, -1.6204849056779387, 
     -1.4921183676049312, -1.38390980430293, -1.2913349040190347, 
     -1.211147018675306, -1.1409523441336409, -1.0789448913293271, 
     -1.0237350221853105, -0.9742350165134863, -0.9295806197089809, 
     -0.8890759795708817, -0.8521541906742954, -0.8183484975970985, 
     -0.7872709290008437, -0.7585962087489505, -0.7320494774569928, 
     -0.7073968073103726, -0.6844377928392268, -0.6629997040534177, 
     -0.6429328290467945, -0.6241067318527449, -0.606407221502185, 
     -0.5897338787820147, -0.5739980240414656, -0.5591210365572382, 
     -0.5450329561999209, -0.53167131335898, -0.5189801446294756, 
     -0.5069091605998957, -0.4954130388970907, -0.4844508209424492, 
     -0.47398539502065434, -0.46398305153042935, -0.4544130988783056, 
     -0.44524753054401983, -0.4364607355050638, -0.42802924554613536, 
     -0.41993151406437873, -0.4121477218655252, -0.40465960616992336, 
     -0.39745030964276085, -0.39050424675438716, -0.38380698518435025, 
     -0.37734514032215666, -0.3711062812013789, -0.36507884644160693, 
     -0.35925206897289885, -0.3536159084864234, -0.34816099069815465, 
     -0.3428785526341339, -0.3377603932494973, -0.3327988287820708, 
     -0.3279866523172766, -0.32331709710635176, -0.31878380323609823, 
     -0.3143807872969408, -0.31010241473810474, -0.30594337463520366, 
     -0.30189865662725535, -0.2979635298077941, -0.29413352337890697, 
     -0.2904044088981544, -0.2867721839668873, -0.2832330572247561, 
     -0.27978343452955245, -0.27641990621417567, -0.27313923532369117, 
     -0.26993834674533324, -0.2668143171530848, -0.2637643656962408, 
     -0.26078584536829363, -0.257876234998645, -0.2550331318151497, 
     -0.2522542445304148, -0.2495373869091722, -0.24688047177798583, 
     -0.24428150544208632, -0.24173858247730343, -0.23924988086791957, 
     -0.23681365746384253, -0.2344282437328125, -0.2320920417854528, 
     -0.22980352065286586, -0.22756121279818867, -0.22536371084507228, 
     -0.22320966450745705, -0.22109777770629258, -0.21902680586001128, 
     -0.2169955533366217, -0.21500287105624588, -0.21304765423380767, 
     -0.21112884025237058};
 
  TVIL_REAL Hxx0yxxtableAL[] = 
     {0, 4., 2.888888888888889, 2.299259259259259, 
     1.9261678004535148, 1.6659027462836986, 1.4726476196028866, 
     1.3227621890501868, 1.202711730370738, 1.1041442648931712, 
     1.0216051547131357, 0.9513698057702162, 0.8908019330821674, 
     0.8379794676182001, 0.7914657923327295, 0.7501641594940478, 
     0.7132218629696401, 0.6799653071622836, 0.649854890756228, 
     0.6224529599568244, 0.5974005974171258, 0.5744005165582854, 
     0.5532042577651595, 0.5336024692596654, 0.5154174351319091, 
     0.49849726408944117, 0.4827113217132884, 0.4679466050623757, 
     0.4541048393252593, 0.4411001333789486, 0.4288570720688297, 
     0.4173091527321683, 0.40639749528743474, 0.3960697713800168, 
     0.38627931018658795, 0.3769843476381838, 0.36814739280702297, 
     0.3597346905736244, 0.351715763853482, 0.3440630219119889, 
     0.3367514238502791, 0.3297581883650753, 0.3230625424937986, 
     0.31664550334371305, 0.3104896878403842, 0.3045791463695734, 
     0.29889921686896503, 0.29343639648371395, 0.28817822835757445, 
     0.28311320150883246, 0.2782306620528089, 0.2735207342925209, 
     0.26897425041590967, 0.2645826877196444, 0.26033811243215016, 
     0.256233129337238, 0.25226083650863257, 0.24841478455814034, 
     0.24468893987890186, 0.24107765143236579, 0.23757562068515362, 
     0.23417787435137488, 0.23087973963846603, 0.22767682173130985, 
     0.22456498328112687, 0.22154032569314583, 0.21859917203096998, 
     0.21573805137638433, 0.21295368450152533, 0.2102429707262352, 
     0.20760297584735501, 0.20503092103894857, 0.2025241726332127, 
     0.20008023270131786, 0.1976967303618006, 0.19537141375154088, 
     0.19310214260092046, 0.190886881360588, 0.18872369283243387, 
     0.18661073226198893, 0.1845462418535739, 0.18252854567319587, 
     0.18055604490747068, 0.1786272134497923, 0.17674059378760218, 
     0.17489479316698273, 0.17308848001292304, 0.17132038058552407, 
     0.1695892758541342, 0.16789399857296625, 0.16623343054315362, 
     0.16460650004747687, 0.16301217944514554, 0.16144948291506417, 
     0.15991746433696222, 0.15841521530062647, 0.15694186323426207, 
     0.1554965696437177, 0.15407852845496448, 0.15268696445280858, 
     0.15132113180936044};
 
  TVIL_REAL Hxx0yxxtableB[] = 
     {0, -2.6759565668419647, 0.8076906476165622, 
     -0.3845247324846479, 0.22409036295871276, -0.1465401051537547, 
     0.10324708396793292, -0.07664686926049558, 0.059143131402314086, 
     -0.047015939227991854, 0.03826930297945497, -0.03175429240711414, 
     0.02677161705321085, -0.02287587903255217, 0.019772484679844826, 
     -0.017260278758420955, 0.015198112410752196, -0.013484577691433075, 
     0.012045299517347507, -0.010824727687532454, 0.009780695839648579, 
     -0.008880730515275603, 0.0080994953619839, -0.007416988342716535, 
     0.00681724867829514, -0.006287415224468018, 0.005817031216611146, 
     -0.005397524376068885, 0.005021813592000823, -0.004684008146971785, 
     0.0043791754125867, -0.004103159764159814, 0.0038524402033980887, 
     -0.0036240175139667374, 0.0034153241508877996, -0.0032241517762809924, 
     0.003048592599838279, -0.0028869915982926986, 0.0027379073676870036, 
     -0.002600079870831884, 0.002472403726131972, -0.002353905975820275, 
     0.0022437274952280817, -0.002141107377199497, 0.002045369759696103, 
     -0.001955912669294818, 0.0018721985355548143, -0.0017937460962647866, 
     0.0017201234652828301, -0.0016509421759837162, 0.0015858520464931691, 
     -0.0015245367396447274, 0.0014667099122778193, -0.0014121118661420684, 
     0.0013605066271007993, -0.0013116793911590874, 0.00126543428559589, 
     -0.0012215924015395174, 0.001179990061016739, -0.001140477287074504, 
     0.001102916450226643, -0.0010671810683768879, 0.0010331547406474684, 
     -0.0010007301983055161, 0.0009698084583168176, -0.0009402980670369931, 
     0.000912114423235244, -0.0008851791710801082, 0.0008594196549452371, 
     -0.0008347684289431029, 0.000811162814996505, -0.0007885445040354405, 
     0.0007668591955751897, -0.0007460562715128627, 0.0007260885004773399, 
     -0.0007069117695072125, 0.0006884848402061161, -0.0006707691268571792, 
     0.0006537284942650656, -0.0006373290733462517, 0.0006215390927088033, 
     -0.0006063287246567406, 0.0005916699442242018, -0.0005775363999942773, 
     0.000563903295589932, -0.0005507472808414793, 0.0005380463517377643, 
     -0.0005257797583600315, 0.0005139279200789548, -0.0005024723473673331, 
     0.0004913955696449618, -0.00048068106862964033, 0.00047031321671955895, 
     -0.00046027721997694837, 0.0004505590653255259, -0.00044114547160849227, 
     0.0004320238441883617, -0.0004231822327989271, 0.00041460929238491817, 
     -0.0004062942466910441, 0.000398226854380776};
 
  TVIL_REAL Hxx0yxxtableC[] = 
     {0, 2.3265870915803015, 2.2558960783562805, 1.8212156257749559, 
     1.4923901848114964, 1.261341516685507, 1.0932804754914427, 
     0.9660552164362657, 0.8664292044707946, 0.7862452865629423, 
     0.720256565158887, 0.6649488091183019, 0.6178833600881172, 
     0.5773155309218692, 0.5419639774761948, 0.510866119368717, 
     0.4832844919071698, 0.45864430363278175, 0.43649071791800276, 
     0.4164589538441205, 0.398252929292225, 0.3816297251355881, 
     0.36638809704213715, 0.35235985348179233, 0.3394032971914295, 
     0.32739817470854116, 0.3162417433354313, 0.30584567659332845, 
     0.2961336062005154, 0.28703915245820155, 0.2785043331266954, 
     0.2704782683220972, 0.2629161189232382, 0.25577821065515044, 
     0.24902930691985142, 0.24263800162612392, 0.2365762094635951, 
     0.23081873579569953, 0.22534291198637302, 0.2201282847986069, 
     0.215156350708345, 0.21041032771152968, 0.20587495857475713, 
     0.20153634057299555, 0.19738177763315976, 0.19339965150720118, 
     0.18957930916892737, 0.1859109640928988, 0.1823856094530979, 
     0.17899494159052698, 0.17573129235571527, 0.17258756914476525, 
     0.16955720162434343, 0.16663409428852918, 0.16381258411396976, 
     0.1610874026836145, 0.15845364223685102, 0.1559067251779298, 
     0.15344237663741928, 0.15105659973493096, 0.14874565323702515, 
     0.14650603134329787, 0.14433444536720727, 0.14222780710707197, 
     0.1401832137275852, 0.13819793399373337, 0.13626939571768307, 
     0.1343951742954255, 0.13257298222409677, 0.13080065950322012, 
     0.12907616483389733, 0.1273975675394199, 0.12576304013906475, 
     0.12417085151412971, 0.12261936061169187, 0.12110701063724282, 
     0.11963232369237232, 0.11819389581811329, 0.11679039240850378, 
     0.11542054396242382, 0.11408314214488219, 0.11277703613170784, 
     0.1115011292140799, 0.11025437564154855, 0.1090357776841836, 
     0.1078443828962676, 0.10667928156554687, 0.10553960433348998, 
     0.1044245199732951, 0.10333323331355168, 0.10226498329651232, 
     0.10121904116087871, 0.10019470873986433, 0.09919131686607201, 
     0.09820822387542871, 0.09724481420305836, 0.09630049706455328, 
     0.09537470521663205, 0.09446689379165156, 0.09357653920087722, 
     0.09269269444023234};
 
  TVIL_REAL Hxx0yxxtableCL[] = 
     {0, 4.644934066848226, 2.220567308492198, 1.4533560751286048, 
     1.1152464830063975, 0.9235331306124643, 0.7975701115890522, 
     0.7069497674710449, 0.6377663787362929, 0.5827302065227348, 
     0.5376185538375832, 0.49979465337677537, 0.46751397909386133, 
     0.4395685227095246, 0.4150906129948624, 0.39343810399358137, 
     0.3741237978586313, 0.3567702345957645, 0.34107970188381587, 
     0.3268137371005917, 0.31377875027611113, 0.3018157103943566, 
     0.29079259871212504, 0.28059878917568026, 0.27114079804794167, 
     0.2623390238861646, 0.2541252154226907, 0.24644048225767892, 
     0.23923371569358856, 0.23246032320231713, 0.22608120536871337, 
     0.22006192219509474, 0.21437200866540937, 0.20898440897412893, 
     0.2038750058502728, 0.19902222665485095, 0.19440671188954478, 
     0.19001103476986836, 0.18581946283253264, 0.18181775434083297, 
     0.17799298365204208, 0.1743333908113904, 0.17082825150821043, 
     0.16746776422352078, 0.16424295195416033, 0.16114557634649837, 
     0.15816806243565526, 0.15530343248171632, 0.15254524763628882, 
     0.14988755637161455, 0.14732484876867386, 0.14485201589691826, 
     0.14246431363167733, 0.14015733035008965, 0.13792695802594587, 
     0.135769366310805, 0.1336809792453257, 0.13165845429271456, 
     0.12969866342697278, 0.12779867604340056, 0.12595574348856703, 
     0.12416728503246689, 0.12243087512753105, 0.12074423181807775, 
     0.11910520618015033, 0.11751177268585987, 0.11596202039866269, 
     0.11445414491672078, 0.11298644099084339, 0.11155729575168283, 
     0.11016518248802121, 0.10880865492427155, 0.10748634195084748, 
     0.10619694276592895, 0.10493922239145392, 0.10371200752997067, 
     0.10251418273235577, 0.10134468684939459, 0.10020250974287873, 
     0.0990866892342421, 0.09799630827086701, 0.09693049229207584, 
     0.0958884067785089, 0.09486925497009731, 0.09387227573919274, 
     0.09289674160663138, 0.09194195688960055, 0.09100725597116059, 
     0.09009200168216082, 0.08919558378708804, 0.08831741756610922, 
     0.08745694248622356, 0.08661362095503151, 0.08578693715116516, 
     0.08497639592591148, 0.08418152177100355, 0.083401857847956, 
     0.08263696507468979, 0.0818864212655232, 0.0811498203209132, 
     0.08041142994939152};
 
  TVIL_REAL Hxx0yxxtableCL2[] = 
     {0, -1., -0.4375, -0.19444444444444445, -0.0849609375, 
     -0.028802083333333332, 0.002658420138888889, 0.021332908163265305, 
     0.03283125559488932, 0.04005289767364556, 0.04460699535551525, 
     0.04743788097531217, 0.04912504459184314, 0.050037248249814995, 
     0.0504162860338134, 0.05042469376194829, 0.05017402344228727, 
     0.04974217756410621, 0.049184352129238114, 0.04854012348577984, 
     0.04783814180305653, 0.04709930157015638, 0.04633892124472436, 
     0.04556826535397117, 0.04479562239985248, 0.04402707784047107, 
     0.04326707468961388, 0.042518824226114436, 0.04178460963864882, 
     0.04106601235290838, 0.04036408196103562, 0.039679464634130156, 
     0.03901250071476344, 0.038363299253967593, 0.037731795179499004, 
     0.037117793295303826, 0.03652100223804877, 0.03594106073392141, 
     0.03537755792392641, 0.034830049100270384, 0.03429806787909263, 
     0.033781135596623914, 0.033278768535970275, 0.03279048345505984, 
     0.03231580178189861, 0.031854252763125444, 0.03140537579001272, 
     0.030968722078128962, 0.030543855839575012, 0.030130355057557286, 
     0.029727811950203388, 0.02933583319253828, 0.028954039951342436, 
     0.028582067776374794, 0.028219566382524337, 0.027866199350359264, 
     0.02752164376688673, 0.027185589823820088, 0.02685774038703802, 
     0.026537810548027792, 0.02622552716578656, 0.02592062840579549, 
     0.02562286328119039, 0.025331991200056792, 0.02504778152181892, 
     0.02477001312492507, 0.024498473987418994, 0.02423296078149824, 
     0.023973278482772524, 0.02371923999462835, 0.023470665787864633, 
     0.023227383555576086, 0.022989227883115457, 0.022756039932854643, 
     0.022527667143381818, 0.02230396294271056, 0.022084786475034264, 
     0.02187000234053035, 0.0216594803477018, 0.02145309527773511, 
     0.021250726660352767, 0.021052258560642724, 0.020857579376355934, 
     0.020666581645174806, 0.020479161861469683, 0.020295220302076266, 
     0.020114660860644035, 0.019937390890123325, 0.019763321052977085, 
     0.019592365178721403, 0.019424440128417236, 0.019259465665753515, 
     0.019097364334379526, 0.01893806134116122, 0.018781484445052964, 
     0.018627563851291794, 0.01847623211063677, 0.01832742402339047, 
     0.018181076547953777, 0.018037128713678383, 0.01788988668989317};
 
  TVIL_REAL Hxx0yxxtableCL3[] = 
     {0, 0.16666666666666666, 0.14583333333333334, 
     0.11805555555555555, 0.09830729166666667, 0.08411458333333334, 
     0.0735134548611111, 0.06531343005952381, 0.05878575642903646, 
     0.05346623173466435, 0.049046961466471355, 0.04531623377944484, 
     0.042123867405785456, 0.039360434581071906, 0.03694433470567068, 
     0.03481348123815325, 0.03291979578595298, 0.03122546512396166, 
     0.029700337060839282, 0.028320070843747408, 0.02706479895235437, 
     0.02591814263899915, 0.024866476724879027, 0.023898372990828833, 
     0.02300417349804916, 0.022175659760297985, 0.021405793536277067, 
     0.020688511769311708, 0.020018562910774588, 0.01939137519159507, 
     0.018802949788603238, 0.018249773558694376, 0.017728747278709368, 
     0.017237126265556544, 0.016772470951475858, 0.0163326055178894, 
     0.0159155830936041, 0.015519656331869336, 0.015143252419530497, 
     0.014784951757466436, 0.014443469697318986, 0.014117640834616354, 
     0.013806405449791984, 0.013508797761607369, 0.01322393571612599, 
     0.012951012081732357, 0.012689286659111688, 0.01243807944643424, 
     0.012196764625650076, 0.011964765256912415, 0.011741548585589758, 
     0.011526621880794875, 0.011319528736405723, 0.011119845775621558, 
     0.010927179708541927, 0.01074116469936272, 0.010561460005783813, 
     0.010387747858305114, 0.010219731551405573, 0.010057133722278781, 
     0.0098996947959425, 0.009747171578232895, 0.009599335980508335, 
     0.009455973861880356, 0.00931688397651018, 0.009181877014998192, 
     0.009050774730185522, 0.008923409138809889, 0.008799621791436214, 
     0.008679263103936824, 0.00856219174454334, 0.008448274071147534, 
     0.00833738361410374, 0.008229400600291772, 0.008124211514645515, 
     0.0080217086957465, 0.007921789962430382, 0.007824358268663109, 
     0.0077293213842176915, 0.007636591598926175, 0.00754608544849831, 
     0.007457723460091823, 0.007371429915991894, 0.007287132633911815, 
     0.007204762762565075, 0.007124254591283078, 0.007045545372563967, 
     0.0069685751565380745, 0.006893286636425545, 0.006819625004142846, 
     0.006747537815288046, 0.006676974862800926, 0.006607888058653738, 
     0.006540231322982617, 0.006473960480118682, 0.006409033161022448, 
     0.006345408711665525, 0.006283048106940458, 0.0062219138697129014, 
     0.00616196999466095, 0.006103181876574102};

  if (y < 2. * x) {     
    result = TVIL_H0xxx0x (x,qq);     
    rn = r = 0.25L * y/x;
    for (n=1; n < 100; n++) {
      result += rn * (Hxx0yxxtableA[n] + Hxx0yxxtableAL[n] * lnyox);
      rn *= r;
    }
  }
  else if (y > 6.47 * x) {
    result = TVIL_H00000x (y, qq);
    rn = r = 4.L * x/y;
    for (n=1; n < 100; n++) {
      result += rn * (Hxx0yxxtableC[n] +
                     -Hxx0yxxtableCL[n] * lnyox +
                     +Hxx0yxxtableCL2[n] * lnyox2 +
                     -Hxx0yxxtableCL3[n] * lnyox3);
      rn *= r;
    }
  }
  else {
    result = -10.9900479678501708324319020119L - 6.L * Zeta3 * TVIL_Ap(x,qq); 
    rn = r = 0.25L * y/x - 1.L;
    for (n=1; n < 100; n++) {
      result += rn * Hxx0yxxtableB[n];
      rn *= r;
    }
  }

  return result;
}

/* ****************************************************************** */

TVIL_COMPLEX TVIL_H00xyxx (TVIL_REAL x, TVIL_REAL y, TVIL_REAL qq)
{
  TVIL_REAL r, rn;
  TVIL_COMPLEX lnyox, lnyox2, lnyox3, result;
  int n;
  
  TVIL_REAL H00xyxxtableA[] = 
     {0, -13.289868133696451, -0.9669930816025213, 
     -0.12106297685467804, 0.02269210315254011, 0.05305705124267418, 
     0.056962172362317404, 0.053817860206711726, 0.04898548386116558, 
     0.04409630208175254, 0.03963634922671204, 0.035711810187721085, 
     0.03230179376483015, 0.02934741462276283, 0.026783839197765377, 
     0.024551306237142002, 0.022598195814536655, 0.02088113647382457, 
     0.019364107976896472, 0.018017307514450152, 0.016816067264322616, 
     0.015739915948772273, 0.014771799271321738, 0.013897445421188783, 
     0.013104853496650144, 0.012383882466561, 0.011725920932249775, 
     0.011123621285347031, 0.01057068502372949, 0.010061688703854748, 
     0.009591942228289858, 0.009157372938046443, 0.008754430373090143, 
     0.008380007654238283, 0.00803137628966647, 0.007706131872125611, 
     0.0074021486506852825, 0.007117541366178745, 0.006850633057925321, 
     0.006599927800311574, 0.006364087526459925, 0.006141912254062708, 
     0.005932323154408588, 0.0057343480065412036, 0.005547108659678668, 
     0.00536981019260855, 0.00520173151196892, 0.005042217174641751, 
     0.004890670254892031, 0.0047465461059366994, 0.004609346889550437, 
     0.004478616767085412, 0.00435393766167708, 0.004234925515049275, 
     0.004121226973719345, 0.004012516448939682, 0.003908493502721968, 
     0.0038088805190383294, 0.003713420624995103, 0.003621875831605604, 
     0.003534025367892218, 0.0034496641855440617, 0.003368601614341886, 
     0.003290660151117803, 0.0032156743672108173, 0.0031434899212660133, 
     0.0030739626658519053, 0.0030069578377760475, 0.002942349323196156, 
     0.002880018989680172, 0.00281985607828696, 0.0027617566495394775, 
     0.0027056230778606067, 0.0026513635896526257, 0.002598891840736391, 
     0.002548126529336018, 0.00249899104120778, 0.0024514131238756003, 
     0.0024053245872563828, 0.0023606610282419054, 0.002317361577054807, 
     0.0022753686634186185, 0.002234627800779078, 0.0021950873869894553, 
     0.002156698520028768, 0.0021194148274609717, 0.0020831923084674935, 
     0.002047989187396551, 0.00201376577787209, 0.0019804843565942643, 
     0.0019481090460433201, 0.0019166057053704982, 0.0018859418288241937, 
     0.0018560864511177047, 0.0018270100591973489, 0.0017986845099169854, 
     0.0017710829531677398, 0.001744179760050358, 0.0017179504557125838, 
     0.0016923716565056934, 0.0016674210111430328};
 
  TVIL_REAL H00xyxxtableAL[] = 
     {0, 6., 1., 0.35555555555555557, 0.17142857142857143, 
     0.09752380952380953, 0.06156806156806157, 0.0417541642031438, 
     0.029836829836829837, 0.022188027416785588, 0.017026391565091257, 
     0.01340133141683688, 0.01077123920036829, 0.008810746195614866, 
     0.007315649807923605, 0.0061529940453463215, 0.00523345637677516, 
     0.004495383730580068, 0.003895206396532606, 0.0034014889128663856, 
     0.0029911298017116076, 0.0026468711258993264, 0.0023556288196188413, 
     0.0021073501929259756, 0.0018942176268713314, 0.0017100842105384022, 
     0.001550067629055722, 0.0014102538317303814, 0.0012874780227249117, 
     0.001179160880107527, 0.001083184734649434, 0.0009977990083940814, 
     0.000921547317059402, 0.0008532107787074236, 0.0007917635633530104, 
     0.0007363377721153913, 0.0006861954872945445, 0.0006407063781225025, 
     0.0005993296430478138, 0.0005615993608668193, 0.0005271125393553616, 
     0.0004955193119724905, 0.00046651485534595554, 0.00043983269306615544, 
     0.0004152391223474259, 0.0003925285548402621, 0.00037151960530774306, 
     0.0003520517949759243, 0.00033398276233356533, 0.0003171858946400654, 
     0.00030154830964451766, 0.00028696912996573824, 0.0002733580029520103, 
     0.000260633827181856, 0.00024872365350918027, 0.00023756173402786907, 
     0.00022708869678988967, 0.0002172508277588653, 0.0002079994444768006, 
     0.00019929034839068997, 0.0001910833448278782, 0.00018334182130379817, 
     0.00017603237625676764, 0.00016912449148308558, 0.00016259024253303052, 
     0.00015640404215798816, 0.00015054241259800468, 0.0001449837830897447, 
     0.00013970830947526876, 0.00013469771321711444, 0.00012993513748712685, 
     0.00012540501830545417, 0.00012109296897048317, 0.00011698567624719628, 
     0.0001130708069762813, 0.00010933692393415447, 0.00010577340991891872, 
     0.00010237039916256808, 0.00009911871527832853, 0.0000960098150463013, 
     0.00009303573742258913, 0.00009018905722856146, 0.00008746284303931697, 
     0.00008485061884497837, 0.00008234632910626645, 0.00007994430686775546, 
     0.00007763924462908719, 0.0000754261677068827, 0.00007330040984870924, 
     0.00007125759088573047, 0.00006929359623301264, 0.0000674045580662466, 
     0.00006558683802119102, 0.00006383701127772334, 0.0000621518519042368, 
     0.0000605283193504564, 0.00005896354598773992, 0.000057454825605744105, 
     0.00005599960278310456, 0.000054595463057624134, 0.00005324012382849211};
 
  TVIL_REAL H00xyxxtableAL2[] = 
     {0, -1., -0.3333333333333333, -0.17777777777777778, 
     -0.11428571428571428, -0.08126984126984127, -0.06156806156806157, 
     -0.04871319157033443, -0.03978243978243978, -0.03328204112517838, 
     -0.028377319275152094, -0.02456910759753428, -0.02154247840073658, 
     -0.019089950090498877, -0.017069849551821746, -0.015382485113365802, 
     -0.013955883671400425, -0.01273692056997686, -0.011685619189597818, 
     -0.01077138155741022, -0.009970432672372026, -0.009264048940647642, 
     -0.008637305671935751, -0.008078175739549573, -0.007576870507485326, 
     -0.007125350877243342, -0.006716959725908129, -0.006346142242786716, 
     -0.006008230772716254, -0.00569927758718638, -0.00541592367324717, 
     -0.005155294876702753, -0.004914919024316812, -0.00469265928289083, 
     -0.004486660192333725, -0.0042953036706731154, -0.004117172923767266, 
     -0.0039510226650887655, -0.0037957544059694878, -0.0036503958456343254, 
     -0.003514083595702411, -0.003386048631812019, -0.003265603987421689, 
     -0.003152134300307447, -0.003045086897214456, -0.002943964161301966, 
     -0.0028483169740260304, -0.0027577390606447405, -0.0026718620986685227, 
     -0.0025903514728938676, -0.00251290258037098, -0.002439237604708775, 
     -0.002369102692250756, -0.0023022654734397283, -0.0022385128815826227, 
     -0.0021776492285887996, -0.0021194945033723035, -0.0020638828637092204, 
     -0.0020106612966090728, -0.001959688425841785, -0.0019108334482787822, 
     -0.0018639751832552813, -0.0018190012213199324, -0.0017758071605723986, 
     -0.0017342959203523252, -0.0016943771233782052, -0.0016559665385780515, 
     -0.0016189855778354825, -0.0015833608407197128, -0.0015490237019968162, 
     -0.0015159099373498132, -0.0014839593832812075, -0.001453115627645798, 
     -0.0014233257276742213, -0.001394539952707469, -0.001366711549176931, 
     -0.001339796525639637, -0.0013137534559196236, -0.001288543298618271, 
     -0.0012641292314429671, -0.0012404764989678552, -0.0012175522725855798, 
     -0.0011953255215373319, -0.0011737668940222006, -0.0011528486074877304, 
     -0.001132544347293202, -0.0011128291730169164, -0.001093679431749799, 
     -0.001075072677781069, -0.0010569875981383352, -0.0010394039434951897, 
     -0.0010223024640047401, -0.0010056648496582622, -0.0009894736748047117, 
     -0.0009737123464997101, -0.0009583650563822264, -0.0009434167358038387, 
     -0.0009288530139595296, -0.000914660178790708, -0.0009008251404507983, 
     -0.000887335397141535};
 
  TVIL_REAL H00xyxxtableC[] = 
     {0, -2.2653441971127455, 0.3031253907398125, 0.2594319089125304, 
     0.1954223295332606, 0.15149392686662788, 0.12137308667201503, 
     0.09992003367833141, 0.08406826016981268, 0.07198488138710939, 
     0.06253258882759302, 0.05497701253003841, 0.0488265326643646, 
     0.04374148696116405, 0.03948068574885861, 0.035868734447709216, 
     0.03277539956280122, 0.03010219054008656, 0.027773402597548038, 
     0.025729996223518285, 0.02392532670501184, 0.02232210801044078, 
     0.020890217347775332, 0.01960508303889128, 0.01844648404888097, 
     0.017397644550895437, 0.016444542959218055, 0.015575378905917153, 
     0.014780157937692277, 0.014050364932089823, 0.013378705068325997, 
     0.012758896731596425, 0.012185504699704947, 0.01165380483632161, 
     0.011159673620146013, 0.010699497395567525, 0.010270097391864965, 
     0.009868667432393612, 0.009492721918967523, 0.00914005218444652, 
     0.008808689697894896, 0.008496874910404425, 0.008203030766951544, 
     0.007925740096191498, 0.007663726237608897, 0.007415836382769102, 
     0.007181027201232715, 0.006958352397103688, 0.006746951903091604, 
     0.0065460424683999666, 0.006354909437049024, 0.0061728995462337006, 
     0.0059994146014404545, 0.005833905907434118, 0.005675869352774105, 
     0.005524841060943822, 0.005380393534048518, 0.005242132225814785, 
     0.005109692489678793, 0.004982736855380443, 0.004860952593929939, 
     0.00474404953628081, 0.004631758115691577, 0.0045238276077207195, 
     0.004420024545186425, 0.0043201312883248645, 0.004223944732873823, 
     0.004131275140955373, 0.004041945081484222, 0.003955788468431157, 
     0.0038726496866606512, 0.003792382796268878, 0.00371485080739931, 
     0.0036399250184298766, 0.0035674844112267897, 0.0034974150978618053, 
     0.0034296098138050833, 0.003363967453146711, 0.0033003926418760775, 
     0.0032387953456681764, 0.0031790905089967717, 0.00312119772272246, 
     0.003065040917594397, 0.0030105480813624316, 0.0029576509974256903, 
     0.0029062850031477366, 0.0028563887661502895, 0.002807904077059863, 
     0.002760775657326733, 0.002714950980865516, 0.002670380108382929, 
     0.0026270155333627415, 0.002584812038771613, 0.0025437265636338996, 
     0.0025037180786993867, 0.0024647474704963604, 0.0024267774331241942, 
     0.002389772367195435, 0.0023536982853878045, 0.002318522724112259, 
     0.002294658322427868};
 
  TVIL_REAL H00xyxxtableCL[] = 
     {0, 3.144934066848226, 0.5699752750680849, 0.28366681948226935, 
     0.18546461851440593, 0.1356028847053723, 0.10539841670202049, 
     0.08521684817788971, 0.07085570305816472, 0.06017178767415825, 
     0.051953555993096756, 0.04546414520256157, 0.040230069189885566, 
     0.03593368626992976, 0.03235434804170108, 0.029334291545215926, 
     0.02675793151952626, 0.02453879385812589, 0.022610997270675923, 
     0.020923542244883145, 0.01943638769904172, 0.018117697386749492, 
     0.016941870224394567, 0.015888107235179097, 0.014939352857354803, 
     0.014081501923717378, 0.013302798117995781, 0.012593372392857806, 
     0.011944885021668895, 0.0113502452996259, 0.010803390064048018, 
     0.010299107222650199, 0.009832894046253626, 0.00940084254932681, 
     0.008999546149657044, 0.00862602317206406, 0.008277653781167786, 
     0.00795212769279736, 0.007647400591659313, 0.007361657623451097, 
     0.007093282667984005, 0.0068408323616579066, 0.006603014041538995, 
     0.0063786669431569396, 0.0061667461102455174, 0.005966308574712265, 
     0.005776501444959889, 0.005596551604720992, 0.005425756776194239, 
     0.005263477743091294, 0.005109131563238527, 0.004962185628197663, 
     0.0048221524502061455, 0.0046885850755598564, 0.004561073039133981, 
     0.004439238787671262, 0.004322734510246542, 0.0042112393233321105, 
     0.004104456765453622, 0.004002112562795005, 0.003903952632488476, 
     0.003809741294880403, 0.003719259669932254, 0.003632304236210753, 
     0.003548685533735206, 0.0034682269943592725, 0.00339076388543248, 
     0.0033161423542662494, 0.0032442185624639514, 0.0031748579005012106, 
     0.0031079342740920747, 0.0030433294548745777, 0.002980932488817444, 
     0.0029206391565065547, 0.0028623514801309306, 0.0028059772725665686, 
     0.002751429724463754, 0.0026986270256890623, 0.0026474920178653147, 
     0.0025979518750983135, 0.0025499378102842724, 0.0025033848046616416, 
     0.00245823135850995, 0.0024144192611102555, 0.002371893378270076, 
     0.0023306014558831995, 0.0022904939381439967, 0.0022515237991690452, 
     0.0022136463868978142, 0.002176819278250568, 0.002141002144616999, 
     0.0021061566268345557, 0.002072246218892224, 0.002039236159664528, 
     0.0020070933320426444, 0.0019757861688855074, 0.0019452845652642994, 
     0.00191555979651932, 0.0018865844416894938, 0.0018583323119120978, 
     0.0018461198976401447};
 
  TVIL_REAL H00xyxxtableCL2[] = 
     {0, -0.75, -0.078125, -0.005208333333333333, 
     0.009195963541666666, 0.01208984375, 0.012026638454861112, 
     0.011161627737032312, 0.01013500349862235, 0.009148183953825129, 
     0.008259613930232942, 0.0074785252904760605, 0.006797519925768528, 
     0.00620448389778889, 0.005686970059064114, 0.005233704296158512, 
     0.00483497459793171, 0.0044825972355188115, 0.004169740119403678, 
     0.0038907156705757806, 0.003640785967356354, 0.0034159939272717025, 
     0.003213022476427887, 0.0030290792183634532, 0.0028618028032834832, 
     0.002709187164393746, 0.0025695202204833833, 0.0024413341955944173, 
     0.0023233652370024433, 0.0022145204728455654, 0.002113851030921681, 
     0.002020529846389728, 0.0019338333293778282, 0.00185312615538956, 
     0.0017778485922894942, 0.0017075058962295756, 0.0016416594021635984, 
     0.0015799190081383693, 0.0015219368106894873, 0.0014674016947836242, 
     0.0014160347184560282, 0.0013675851616206966, 0.0013218271320547658, 
     0.001278556640500603, 0.0012375890721409242, 0.0011987569941275802, 
     0.001161908248966095, 0.0011269042918337407, 0.001093618736700303, 
     0.0010619360817139968, 0.0010317505889376824, 0.0010029652973541018, 
     0.000975491151248516, 0.000949246228739647, 0.0009241550574592431, 
     0.0009001480062530808, 0.0008771607433535518, 0.0008551337528065131, 
     0.0008340119020639474, 0.0008137440546128824, 0.0007942827223277077, 
     0.0007755837529303653, 0.0007576060485397872, 0.0007403113118041256, 
     0.0007236638165498064, 0.0007076302002611443, 0.0006921792760322862, 
     0.0006772818619172585, 0.0006629106258502837, 0.0006490399445227385, 
     0.0006356457747897206, 0.0006227055363420233, 0.000610198004521725, 
     0.0005981032122843189, 0.0005864023604197804, 0.0005750777352411842, 
     0.0005641126330342194, 0.0005534912906356821, 0.0005431988215750459, 
     0.0005332211572716338, 0.000523544992831674, 0.0005141577370354743, 
     0.0005050474661457814, 0.0004962028812047315, 0.00048761326851920353, 
     0.00047926846306328556, 0.0004711588145524129, 0.00046327515596685296, 
     0.0004556087743229288, 0.000448151383508959, 0.00044089509901958237, 
     0.00043383241443714746, 0.00042695617952236104, 0.00042025957978856875, 
     0.00041373611744503297, 0.00040737959360449984, 0.0004011840916593257, 
     0.00039514396173855954, 0.00038925380616573963, 0.00038350846584384566, 
     0.0003835378554008345};
 
  TVIL_REAL H00xyxxtableCL3[] = 
     {0, 0.16666666666666666, 0.0625, 0.034722222222222224, 
     0.022786458333333332, 0.01640625, 0.012532552083333334, 
     0.009974888392857142, 0.008182525634765625, 0.00686928077980324, 
     0.005873235066731771, 0.0050966089422052555, 0.0044772293832567, 
     0.003973872233659794, 0.0035582376377923147, 0.003210321068763733, 
     0.002915623626904562, 0.0026634070501827143, 0.0024455666587325847, 
     0.0022558828181660686, 0.002089511460326321, 0.0019426297023441986, 
     0.001812184526050425, 0.001695711418515993, 0.0015912014265762053, 
     0.001497002302122894, 0.0014117440349161907, 0.0013342820851265506, 
     0.0012636536329164079, 0.001199043518510456, 0.0011397574778729946, 
     0.0010852009284638814, 0.0010348620182079886, 0.0009882979788212196, 
     0.0009451240619263478, 0.0009050045099098826, 0.0008676451416381397, 
     0.0008327872287162275, 0.0008002024098641037, 0.000769688445516886, 
     0.0007410656564492269, 0.0007141739223365542, 0.0006888701410746298, 
     0.0006650260691282932, 0.0006425264784630539, 0.0006212675776941528, 
     0.0006011556546922391, 0.0005821059055621093, 0.0005640414210796046, 
     0.0005468923066527778, 0.000530594915914525, 0.0005150911813472418, 
     0.0005003280280356992, 0.0004862568588598998, 0.000472833101267917, 
     0.0004600158072831338, 0.0004477672996593641, 0.000436052858147808, 
     0.0004248404407192154, 0.00041410043532326857, 0.0004038054383895373, 
     0.00039393005680032797, 0.0003844507305115584, 0.00037534557337674194, 
     0.00036659423005216214, 0.0003581777471373551, 0.0003500784569415265, 
     0.0003422798724696725, 0.0003347665923970791, 0.00032752421495181567, 
     0.0003205392597553943, 0.00031379909678493767, 0.0003072918817185062, 
     0.00030100649701080923, 0.0002949324981211645, 0.00028906006438079643, 
     0.0002833799540436738, 0.0002778834631151829, 0.00027256238759695167, 
     0.00026740898882488944, 0.00026241596161167473, 0.0002575764049350991, 
     0.0002528837949403713, 0.0002483319600481365, 0.00024391505798095374, 
     0.00023962755453962764, 0.0002354642039773856, 0.00023142003083469038, 
     0.00022749031311067671, 0.0002236705666590014, 0.0002199565307064538, 
     0.00021634415440213335, 0.00021282958431348813, 0.00020940915279313028, 
     0.0002060793671472006, 0.00020283689954222523, 0.00019967857759297292, 
     0.00019660137557883954, 0.00019360240624082412, 0.00019067891311526123, 
     0.00018782826336418806};
 
  TVIL_REAL H00xyxxtableB[] = 
     {0, -3.2027181327479024, 1.0836780701489368, 
     -0.5586508236979613, 0.345704442565157, -0.23712107307382535, 
     0.17378044034801188, -0.13338962730616433, 0.10594518330463032, 
     -0.08638813320563787, 0.07192549347710508, -0.060907975150488625, 
     0.05230828719048042, -0.04545819225801295, 0.03990707934314597, 
     -0.03534171964407373, 0.03153865207400919, -0.028334833241990487, 
     0.025608942182765104, -0.023269126570914456, 0.02124476658862029, 
     -0.019480814495540844, 0.017933826282116316, -0.01656912946201866, 
     0.01535876885765941, -0.014279994689045394, 0.01331413483913646, 
     -0.012445743325006975, 0.011661950056237547, -0.0109519591229299, 
     0.01030665795254565, -0.0097183101112574, 0.009180311838921151, 
     -0.008686997596723645, 0.008233483633042299, -0.007815541278211935, 
     0.007429493662858111, -0.007072131023536793, 0.006740640857096691, 
     -0.006432550012346022, 0.006145676435929143, -0.005878088770232909, 
     0.005628072371859694, -0.0053941006069346265, 0.005174810504262467, 
     -0.00496898202398785, 0.004775520339042938, -0.004593440637659568, 
     0.004421855043916928, -0.004259961324530659, 0.004107033107577346, 
     -0.003962411385456999, 0.003825497112353804, -0.0036957447374920682, 
     0.0035726565409773814, -0.0034557776600199763, 0.003344691710724629, 
     -0.0032390169250685052, 0.003138402734717355, -0.003042526743391949, 
     0.002951092037931785, -0.0028638247953020335, 0.0027804721487847406, 
     -0.0027008002816604204, 0.002624592720995189, -0.0025516488078090177, 
     0.0024817823230267877, -0.002414820251287156, 0.0023506016669782304, 
     -0.0022889767288382795, 0.0022298057711591934, -0.0021729584810964847, 
     0.002118313152857612, -0.00206575601064192, 0.0020151805931611783, 
     -0.0019664871934027273, 0.001919582348022239, -0.0018743783713905254, 
     0.0018307929298716358, -0.0017887486523996687, 0.0017481727738485886, 
     -0.0017089968080675477, 0.0016711562477856323, -0.001634590288884713, 
     0.0015992415767975, -0.0015650559730196884, 0.0015319823399267407, 
     -0.0014999723422693678, 0.0014689802638817306, -0.0014389628382804054, 
     0.0014098790919603588, -0.0013816901993086023, 0.00135435934815821, 
     -0.0013278516150980885, 0.001302133849734835, -0.001277174567177547, 
     0.0012529438480829297, -0.0012294132456565632, 0.0012065556990621396, 
     -0.0011843454527359493, 0.001162757981151022};

  lnyox = TVIL_CLOG(y/x);
  lnyox2 = lnyox  * lnyox;
  lnyox3 = lnyox2  * lnyox;

  if (y < 2. * x) {     
    result = TVIL_H00x0xx (x,qq);     
    rn = r = 0.25L * y/x;
    for (n=1; n < 100; n++) {
      result += rn * (H00xyxxtableA[n] + 
                      H00xyxxtableAL[n] * lnyox +
                      H00xyxxtableAL2[n] * lnyox2);
      rn *= r;
    }
  }
  else if (y > 6.47 * x) {
    result = TVIL_H00000x (y, qq);
    rn = r = 4.L * x/y;
    for (n=1; n < 100; n++) {
      result += rn * (H00xyxxtableC[n] +
                     -H00xyxxtableCL[n] * lnyox +
                     +H00xyxxtableCL2[n] * lnyox2 +
                     -H00xyxxtableCL3[n] * lnyox3);
      rn *= r;
    }
  }
  else {
    result = -9.403507956757080718L - 6.L * Zeta3 * TVIL_Ap(x,qq); 
    rn = r = 0.25L * y/x - 1.L;
    for (n=1; n < 100; n++) {
      result += rn * H00xyxxtableB[n];
      rn *= r;
    }
  }

  return result;
}

/* ****************************************************************** */

TVIL_COMPLEX TVIL_H00xxxy (TVIL_REAL x, TVIL_REAL y, TVIL_REAL qq)
{
  TVIL_REAL r, rn;
  TVIL_COMPLEX lnyox, lnyox2, lnyox3, result;
  int n;
  
  TVIL_REAL H00xxxytableA[] = 
      {0, -2.0018374258594345, -0.5237717540537673, 
     -0.17977238330662387, -0.07532513896071008, -0.03698279344431988, 
     -0.020537181387706258, -0.012517297358194722, -0.008182086212874022, 
     -0.005641000802617092, -0.0040538915808804345, -0.0030114338806556153, 
     -0.0022983379215537098, -0.0017939851302012047, -0.0014271725329481444, 
     -0.0011539939564214213, -0.0009463589133344908, -0.000785719826366258, 
     -0.0006594912764238116, -0.0005589272023354966, -0.0004778203183260859, 
     -0.00041168190788527185, -0.0003572111461558853, -0.0003119438105629195, 
     -0.00027401491572767305, -0.00024199532297983082, 
     -0.0002147773587920781, -0.00019149350092271928, 
     -0.00017145775193257406, -0.00015412281851842323, 
     -0.00013904845885774595, -0.00012587782453086488, 
     -0.00011431959489204814, -0.00010413435574443687, 
     -0.00009512412067096044, -0.00008712420218687674, 
     -0.00007999685606429336, -0.00007362627524114822, 
     -0.00006791461925357034, -0.00006277884428970507, 
     -0.00005814815671434321, -0.00005396195542147482, -0.000050168159922686, 
     -0.00004672184468368209, -0.00004358411801364419, 
     -0.00004072119731956327, -0.00003810364286181029, 
     -0.000035705720089247126, -0.00003350486677942201, 
     -0.00003148124599539414, -0.000029617369617840967, 
     -0.000027897780160569198, -0.000026308780911088536, 
     -0.000024838206293217418, -0.00002347522583073245, 
     -0.00002221017628030099, -0.00002103441746034766, 
     -0.000019940208078124577, -0.000018920598487465914, 
     -0.000017969337823749914, -0.00001708079338344079, 
     -0.000016249880461371667, -0.000015472001144029248, 
     -0.00001474299079294827, -0.000014059071148057986, 
     -0.000013416809143772704, -0.000012813080666684826, 
     -0.000012245038597668416, -0.00001171008457689898, 
     -0.000011205844010884181, -0.000010730143908646078, 
     -0.00001028099319179823, -9.856565172144843e-6, -9.4551819320144e-6, 
     -9.075300377999613e-6, -8.715499769080541e-6, -8.374470546064993e-6, 
     -8.05100431156166e-6, -7.743984828867321e-6, -7.452379924669301e-6, 
     -7.175234194731711e-6, -6.911662424079725e-6, -6.660843643899264e-6, 
     -6.4220157566655e-6, -6.194470669101962e-6, -5.977549879621921e-6, 
     -5.770640473058848e-6, -5.5731714808756425e-6, -5.384610569757469e-6, 
     -5.204461025629754e-6, -5.032259003777795e-6, -4.867571018943172e-6, 
     -4.709991652091184e-6, -4.559141453031559e-6, -4.4146650202736035e-6, 
     -4.276229241443063e-6, -4.143521679312646e-6, -4.016249090028573e-6, 
     -3.894136061475339e-6, -3.776923760930652e-6, -3.6643687822400234e-6};
 
  TVIL_REAL H00xxxytableAL[] = 
     {0, -1.6449340668482262, -0.5724670334241131, 
     -0.2566446889494088, -0.1383168500453899, -0.08503840067123256, 
     -0.057326974104334044, -0.04123697347471053, -0.031091672826478456, 
     -0.02428517447305728, -0.01949604432690355, -0.015997840783815498, 
     -0.01336437827741245, -0.011332105477754165, -0.009730851742749125, 
     -0.008446718429967762, -0.0074011186889088865, -0.006538399712167894, 
     -0.005818250483743376, -0.0052108899355867155, -0.004693929076570069, 
     -0.004250272777776204, -0.0038666851943456454, -0.0035327892399835414, 
     -0.003240356205963635, -0.0029827931703419186, -0.002754767656171317, 
     -0.0025519290804592726, -0.0023706994780381407, -0.002208114483812626, 
     -0.002061701235477137, -0.0019293837135462982, -0.0018094086908285248, 
     -0.0017002873175084102, -0.00160074867898799, -0.0015097026015761447, 
     -0.001426209659552695, -0.0013494568329290857, -0.0012787376310495688, 
     -0.0012134357695534892, -0.0011530116927521358, -0.0010969913883026317, 
     -0.001044957059156533, -0.0009965393084890237, -0.0009514105634925371, 
     -0.00090927951855483, -0.0008698864211390545, -0.000832999057404811, 
     -0.0007984093213278323, -0.0007659302723587169, -0.0007353936037006042, 
     -0.0007066474569941902, -0.000679554530278615, -0.0006539904350921846, 
     -0.0006298422659117842, -0.0006070073511343956, -0.000585392159739722, 
     -0.0005649113418449286, -0.0005454868847343912, -0.0005270473687490305, 
     -0.0005095273097561609, -0.0004928665768747335, -0.00047700987577062315, 
     -0.00046190628921681867, -0.0004475088677783068, 
     -0.00043377426446765746, -0.0004206624080545056, 
     -0.00040813621042464984, -0.000396161303992313, -0.0003847058056893615, 
     -0.00037374010450101636, -0.00036323666990098827, 
     -0.00035316987886915147, -0.00034351585945962756, 
     -0.00033425234913416965, -0.0003253585662890933, 
     -0.00031681509359005045, -0.00030860377189027496, 
     -0.00030070760364930144, -0.0002931106648917589, 
     -0.00028579802485424743, -0.0002787556725622961, -0.0002719704496627201, 
     -0.0002654299889096323, -0.0002591226577666472, -0.00025303750664476543, 
     -0.0002471642213454049, -0.0002414930793227417, -0.00023601490941871708, 
     -0.0002307210547591275, -0.0002256033385304064, -0.00022065403238415507, 
     -0.0002158658272415001, -0.0002112318062910401, -0.00020674541999413446, 
     -0.00020240046292873543, -0.00019819105231878506, 
     -0.00019411160811047892, -0.00019015683446923018, 
     -0.0001863217025828076, -0.0001826014346662519};

  TVIL_REAL H00xxxytableB[] = 
     {0, -3.1367204059992617, 1.02405686873614, -0.5111637812120541, 
     0.3075805973977935, -0.2058676274757182, 0.14764174262949725, 
     -0.11115024596756605, 0.08674814846215805, -0.06961418050897154, 
     0.05711622772615627, -0.04771657458910572, 0.04046719489009554, 
     -0.034757424578047806, 0.03017931562756071, -0.026451825996391745, 
     0.02337617155577874, -0.020808503661621524, 0.01864262747624206, 
     -0.016798751751043776, 0.015215977090940374, -0.013847166406816285, 
     0.012655371080989252, -0.011611295698685334, 0.010691469973331352, 
     -0.009876910933648566, 0.00915213057259579, -0.008504390579798005, 
     0.007923136227167631, -0.007399561798916166, 0.006926273738759701, 
     -0.0064970271725368475, 0.0061065180828012095, -0.005750218088812486, 
     0.0054242421295496775, -0.005125241765171428, 0.004850318578515205, 
     -0.004596953460932227, 0.004362948536394483, -0.004146379205745648, 
     0.003945554343880191, -0.003758983102766872, 0.0035853470959516386, 
     -0.0034234769897689473, 0.003272332720786766, -0.003130986711196491, 
     0.0029986095737634865, -0.0028744578929544404, 0.002757863744529584, 
     -0.002648225676471537, 0.002545000922863391, -0.0024476986617249343, 
     0.002355874159801613, -0.0022691236733813494, 0.0021870799955673575, 
     -0.0021094085579867108, 0.0020358040093961054, -0.001965987205638705, 
     0.001899702555371882, -0.001836715674295628, 0.0017768113075631651, 
     -0.0017197914858891356, 0.001665473885781362, -0.0016136903684677662, 
     0.0015642856755997405};

  TVIL_REAL H00xxxytableC[] = 
    {0, 0.585870317864344, 1.0749440871750642, 0.7945196888382672, 
     0.5895917448978106, 0.4545582718893676, 0.36275303657612046, 
     0.29765529667343843, 0.24972739347131054, 0.21331138625131096, 
     0.1849083442465826, 0.16226500145691228, 0.14387679028992437, 
     0.12870682605896683, 0.11602064385723486, 0.10528540022463197, 
     0.09610634001964423, 0.08818555963660289, 0.0812945329914469, 
     0.07525537461357738, 0.06992779118333772, 0.06519982184902946, 
     0.060981154473962684, 0.057198226177995797, 0.05379058097785075, 
     0.05070812693471472, 0.04790904614809814, 0.04535818480595726, 
     0.04302580051470128, 0.04088657851551592, 0.03891885236746102, 
     0.03710398161514831, 0.035425851071982116, 0.033870465112308616, 
     0.032425616772186465, 0.031080616189315652, 0.029826066439047106, 
     0.02865367747524076, 0.027556110895544186, 0.026526849787374185, 
     0.025560089093999896, 0.024650642857412478, 0.023793865410588803, 
     0.0229855841540511, 0.02222204199591147, 0.021499847887642205, 
     0.02081593416989338, 0.020167519669225566, 0.019552077669473557, 
     0.01896730802973494, 0.01841111284176682, 0.017881575118397965, 
     0.017376940085753394, 0.01689559871905688, 0.01643607321722688, 
     0.015997004157563333, 0.015577139110255201, 0.015175322524600337, 
     0.0147904867258322, 0.014421643884191105, 0.014067878837091624, 
     0.013728342661517989, 0.013402246907613043, 0.013088858416213359, 
     0.012787494653153207, 0.012497519501785221, 0.012218339462571623, 
     0.011949400214974447, 0.011690183502372685, 0.011440204305489612, 
     0.011199008273934477, 0.010966169389041384, 0.010741287834302499, 
     0.010523988052408256, 0.010313916970279662, 0.01011074237555459, 
     0.009914151429811035, 0.009723849305410181, 0.009539557934250181, 
     0.009361014857962643, 0.009187972170179859, 0.009020195542470036, 
     0.008857463326396446, 0.008699565724918073, 0.008546304027026178, 
     0.008397489900113308, 0.008252944735107915, 0.008112499039886454, 
     0.007975991876902632, 0.007843270341356208, 0.007714189076566506, 
     0.007588609823523402, 0.007466401001864631, 0.007347437319776609, 
     0.007231599410539449, 0.007118773493638315, 0.007008851058545002, 
     0.006901728569437787, 0.006797307189276052, 0.006695492521780521, 
     0.006611859862366555};
 
  TVIL_REAL H00xxxytableCL[] = 
      {0, -3.894934066848226, -1.4733962917801415, 
     -0.804765568127423, -0.5256161807781027, -0.3791501568585194, 
     -0.2908160363011469, -0.23250205316651362, -0.1915268235960676, 
     -0.16139293732498686, -0.13844758503845897, -0.12048881936677944, 
     -0.10611549485993722, -0.09439667297840391, -0.0846915674493769, 
     -0.07654589365154811, -0.06962941177263608, -0.06369682031828511, 
     -0.05856245566358826, -0.05408345410681581, -0.050148265495223034, 
     -0.04666864483296625, -0.04357395944447823, -0.04080707127702656, 
     -0.0383213115387549, -0.03607822615007485, -0.03404587377863209, 
     -0.032197525754650756, -0.03051066214090776, -0.028966188705421368, 
     -0.027547820517899387, -0.0262415925348443, -0.025035467900106485, 
     -0.02391902211084343, -0.022883186578186235, -0.021920039052144534, 
     -0.02102263129543962, -0.020184846568046973, -0.019401281124499896, 
     -0.01866714517210502, -0.017978179692290683, -0.01733058626319474, 
     -0.01672096759316628, -0.016146276921738936, -0.01560377479622663, 
     -0.015090992010366285, -0.01460569771292576, -0.014145871871448031, 
     -0.013709681418881759, -0.013295459526101151, -0.012901687536919414, 
     -0.012526979178559354, -0.012170066723106159, -0.011829788826935672, 
     -0.01150507981762116, -0.011194960233065421, -0.010898528446931541, 
     -0.010614953238933982, -0.010343467189069158, -0.010083360792110987, 
     -0.009833977203239233, -0.009594707537967609, -0.00936498665997112, 
     -0.00914428939928617, -0.008932127150925822, -0.008728044810425384, 
     -0.008531618008382945, -0.008342450610828937, -0.00816017245636795, 
     -0.00798443730458409, -0.007814920973272011, -0.007651319644719056, 
     -0.007493348323578776, -0.007340739430892208, -0.007193241520572635, 
     -0.007050618106207846, -0.006912646587381483, -0.00677911726589759, 
     -0.006649832443332239, -0.0065246055922515995, -0.006403260594243657, 
     -0.00628563103862433, -0.006171559576310465, -0.0060608973239119615, 
     -0.005953503313592334, -0.005849243984688793, -0.005747992713476381, 
     -0.005649629377811419, -0.005554039953702582, -0.0054611161411379765, 
     -0.005370755016747068, -0.005282858711101009, -0.005197334108656421, 
     -0.0051140925685288175, -0.005033049664444813, -0.004954124942368973, 
     -0.00487724169443347, -0.0048023267479182065, -0.004729310268136915, 
     -0.0046581255741824645, -0.004611721237897026};
 
  TVIL_REAL H00xxxytableCL2[] = 
     {0, -0.75, -0.234375, -0.08333333333333333, -0.032958984375, 
     -0.0137890625, -0.005594889322916667, -0.0017523940728635203, 
     0.00017075538635253907, 0.001168960490554729, 0.0016894442694527763, 
     0.0019501629822463418, 0.0020644755235740115, 0.0020944497269474766, 
     0.0020754579919745747, 0.0020282866014746576, 0.0019654306225108443, 
     0.0018945178921172827, 0.001820243263293511, 0.0017454965785438726, 
     0.0016720383510939873, 0.001600913806469657, 0.0015327115033696027, 
     0.001467727481187457, 0.0014060708201278156, 0.0013477322303520093, 
     0.001292628956354538, 0.0012406343101957077, 0.0011915971190656492, 
     0.0011453544951925063, 0.001101740153088354, 0.0010605897427283005, 
     0.0010217441772264095, 0.0009850516123224874, 0.000950368522143078, 
     0.0009175601733289971, 0.0008865007035928135, 0.0008570729455246915, 
     0.0008291680918595764, 0.0008026852677678019, 0.0007775310545861888, 
     0.000753618994780464, 0.000730869097799995, 0.000709207359472893, 
     0.0006885653027460001, 0.0006688795442454127, 0.0006500913888640359, 
     0.0006321464530556135, 0.0006149943165070315, 0.0005985882012158699, 
     0.0005828846766077036, 0.0005678433891095446, 0.0005534268144969155, 
     0.0005396000313134176, 0.0005263305136957979, 0.0005135879420051747, 
     0.0005013440297527001, 0.000489572365406228, 0.00047824826776709085, 
     0.0004673486537085726, 0.0004568519171673062, 0.0004467378183738378, 
     0.00043698738239797945, 0.00042758280616779666, 0.0004185073731979724, 
     0.0004097453753339396, 0.0004012820408827782, 0.0003931034685607557, 
     0.0003851965667409129, 0.0003775489975326414, 0.00037014912526916945, 
     0.0003629859690186454, 0.0003560491587704594, 0.0003493288949809166, 
     0.0003428159111917038, 0.00033650143946106314, 0.00033037717837149533, 
     0.0003244352633993941, 0.00031866823945150564, 0.0003130690353907162, 
     0.00030763094038958216, 0.00030234758196440284, 0.00029721290555564706, 
     0.000292221155532323, 0.0002873668575085405, 0.0002826448018701781, 
     0.0002780500284183236, 0.0002735778120441017, 0.0002692236493567174, 
     0.000264983246193089, 0.00026085250594340653, 0.0002568275186323608, 
     0.00025290455070072333, 0.0002490800354364453, 0.00024535056400853766, 
     0.00024171287706072683, 0.00023816385682529054, 0.00023470051972058738, 
     0.00023132000939864143, 0.000228019590211745, 0.00023324891292081119};
 
  TVIL_REAL H00xxxytableCL3[] = 
     {0, -0.125, -0.078125, -0.049479166666666664, -0.03369140625, 
     -0.02451171875, -0.018778483072916668, -0.014957972935267858, 
     -0.012272834777832031, -0.010303709242078993, -0.008809804916381836, 
     -0.007644902576099743, -0.006715841591358185, -0.005960807777368105, 
     -0.00533735632364239, -0.004815481572101514, -0.0043734354330808856, 
     -0.003995110573562081, -0.0036683499876946574, -0.0033838242271533665, 
     -0.003134267190466744, -0.002913944553510884, -0.0027182767890743457, 
     -0.0025435671277736806, -0.002386802139864234, -0.0022455034531843232, 
     -0.0021176160523742817, -0.002001423127689825, -0.0018954804493746117, 
     -0.0017985652777656842, -0.001709636216809492, -0.0016278013926958222, 
     -0.0015522930273119827, -0.0014824469682318295, -0.0014176860928895218, 
     -0.001357506764864824, -0.0013014677124572095, -0.0012491808430743414, 
     -0.0012003036147961554, -0.0011545326682753292, -0.0011115984846738404, 
     -0.0010712608835048312, -0.0010333052116119445, -0.0009975391036924398, 
     -0.0009637897176945809, -0.0009319013665412294, -0.0009017334820383588, 
     -0.0008731588583431641, -0.000846062131619407, -0.0008203384599791668, 
     -0.0007958923738717877, -0.0007726367720208627, -0.0007504920420535488, 
     -0.0007293852882898498, -0.0007092496519018755, -0.0006900237109247007, 
     -0.0006716509494890463, -0.0006540792872217119, -0.0006372606610788231, 
     -0.0006211506529849029, -0.000605708157584306, -0.000590895085200492, 
     -0.0005766760957673376, -0.0005630183600651129, -0.0005498913450782433, 
     -0.0005372666207060327, -0.0005251176854122897, -0.0005134198087045087, 
     -0.0005021498885956187, -0.0004912863224277235, -0.0004808088896330914, 
     -0.00047069864517740653, -0.0004609378225777593, 
     -0.00045150974551621387, -0.00044239874718174675, 
     -0.00043359009657119465, -0.00042506993106551075, 
     -0.0004168251946727744, -0.00040884358139542756, -0.0004011134832373342, 
     -0.00039362394241751204, -0.0003863646074026487, -0.000379325692410557, 
     -0.0003724979400722047, -0.00036587258697143066, 
     -0.00035944133180944147, -0.0003531963059660784, -0.0003471300462520356, 
     -0.00034123546966601507, -0.0003355058499885021, 
     -0.00032993479605968073, -0.00032451623160320006, 
     -0.00031924437647023215, -0.0003141137291896954, -0.0003091190507208009, 
     -0.0003042553493133379, -0.0002995178663894594, -0.0002949020633682593, 
     -0.0002904036093612362, -0.0002860183696728918, -0.0002817423950462821};

  TVIL_REAL H00xxxytableD[] = 
     {0, -1.7565021624839214, 0.4486609263461504, 
     -0.19666799319246364, 0.10924868529394295, -0.06926246094895291, 
     0.04775912516360917, -0.03489772034546111, 0.026604194270673367, 
     -0.02094810232142802, 0.016919899144109213, -0.01395031293885434, 
     0.011698655113410304, -0.009950963585654834, 0.008567396603633088, 
     -0.007453437993533912, 0.006543351739846642, -0.005790274969628928, 
     0.005160071840011441, -0.004627401297154714, 0.0041731322929471105, 
     -0.003782603579979171, 0.003444426862955732, -0.003149647751071713, 
     0.0028911473055271072, -0.002663208458957223, 0.0024611973740282208, 
     -0.0022813261970076513, 0.0021204742855948948, -0.0019760520032654737, 
     0.0018458958800162465, -0.0017281871486281553, 0.0016213878847469697, 
     -0.0015241905340876447, 0.001435477713169881, -0.0013542899615538882, 
     0.0012797996976804695, -0.001211290051059335, 0.001148137554654336, 
     -0.0010897979134598058, 0.0010357942399446586, -0.000985707279527106, 
     0.0009391672504795593, -0.0008958470005708232, 0.0008554562431110046, 
     -0.0008177366821258894, 0.0007824578733044684, -0.0007494136964902495, 
     0.000718419338594653, -0.000689308704239683, 0.0006619321862085234, 
     -0.0006361547396795855, 0.0006118542138454012, -0.00058891990234051, 
     0.0005672512802868224, -0.0005467569009965211, 0.0005273534296762701, 
     -0.0005089647950302033, 0.0004915214426041574, -0.0004749596761626571, 
     0.00045922107543360224, -0.00044425198026612305, 0.00043000303268337975, 
     -0.0004164287695217614, 0.0004034872593697211, -0.00039113977838497045, 
     0.0003793505203038601, -0.00036808633658276394, 0.0003573165031457331, 
     -0.00034701251067009026, 0.00033714787573404826, 
     -0.00032769797048786884, 0.0003186398688008873, -0.0003099522070879044, 
     0.0003016150582358489, -0.0002936098172401759, 0.0002859190973243344, 
     -0.00027852663545832975, 0.0002714172063168899, -0.00026457654382654487, 
     0.00025799126954619304, -0.000251648827209281, 0.00024553742282913937, 
     -0.00023964596983362124, 0.00023396403875213803, 
     -0.00022848181102846937, 0.00022319003657717603, 
     -0.00021807999474081694, 0.0002131434583400828, -0.0002083726605399665, 
     0.00020376026428266592, -0.00019929933406247257, 0.00019498330983979783, 
     -0.000190805982911043, 0.0001867614735684998, -0.00018284421040011713, 
     0.0001790489110929939, -0.00017537056461704224, 0.0001718044146765702, 
     -0.00016834594432769983, 0.00016499086166869156};

  lnyox = TVIL_CLOG(y/x);
  lnyox2 = lnyox  * lnyox;
  lnyox3 = lnyox2  * lnyox;

  if (y < 0.5 * x) {
    result = TVIL_H00xx0x (x,qq);
    rn = r = y/x;
    for (n=1; n < 100; n++) {
      result += rn * (H00xxxytableA[n] - H00xxxytableAL[n] * lnyox); 
      rn *= r;
    }
  } 
  else if (y < 1.6 * x) {
    result = TVIL_H00xxxx (x,qq);
    rn = r = y/x - 1.L;
    for (n=1; n < 100; n++) {
      result += rn * H00xxxytableD[n]; 
      rn *= r;
    }
  } 
  else if (y < 6. * x) {
    result = -9.267575567256300L - 6.L * Zeta3 * TVIL_Ap(x,qq);
    rn = r = y/(4.L*x) - 1.L;
    for (n=1; n < 64; n++) {
      result += rn * H00xxxytableB[n]; 
      rn *= r;
    }
  }
  else {
    result = TVIL_H00000x (y, qq);
    rn = r = 4.L * x/y;
    for (n=1; n < 100; n++) {
      result += rn * (H00xxxytableC[n] +
                     +H00xxxytableCL[n] * lnyox +
                     +H00xxxytableCL2[n] * lnyox2 +
                     +H00xxxytableCL3[n] * lnyox3);
      rn *= r;
    }
  }

  return result;
}

/* ****************************************************************** */

TVIL_COMPLEX TVIL_H00y0xx (TVIL_REAL x, TVIL_REAL y, TVIL_REAL qq)
{
  TVIL_REAL r, rn;
  TVIL_COMPLEX lnyox, lnyox2, lnyox3, result;
  int n;
  
  TVIL_REAL H00y0xxtableA[] = 
      {0, -1.7715086547545278, -0.7408202605290377, 
     -0.40611704771975266, -0.2566303172561659, -0.17700689113100387, 
     -0.1295565905063698, -0.09898591113450883, -0.07812679045050075, 
     -0.06325333350534057, -0.052271522663895675, -0.04393025233847847, 
     -0.03744428858966081, -0.03230039920448213, -0.028151538692552214, 
     -0.024756155105835842, -0.021941885527808458, -0.019583044429801577, 
     -0.017586222283475283, -0.01588081733159641, -0.014412658599678996, 
     -0.013139617646888563, -0.012028530172520396, -0.011052998722542062, 
     -0.01019179947421825, -0.00942771038884993, -0.008746637949629849, 
     -0.008136958551284323, -0.0075890162597475885, -0.007094735884676462, 
     -0.006647322054249172, -0.006241023108130073, -0.005870944321591465, 
     -0.005532899017413384, -0.005223289025450578, -0.004939008056798727, 
     -0.004677363104164278, -0.004436010123188033, -0.004212901103033823, 
     -0.0040062402772409605, -0.0038144477135952787, -0.0036361288947266007, 
     -0.0034700491883323387, -0.0033151123285743295, -0.0031703422039043444, 
     -0.003034867382924633, -0.0029079079175394207, -0.002788764048108669, 
     -0.002676806503510737, -0.0025714681437167958, -0.0024722367365571096, 
     -0.0023786486960459124, -0.002290283638649235, -0.002206759637577549, 
     -0.002127729074613491};
 
  TVIL_REAL H00y0xxtableAL[] = 
     {0, 3.2898681336964524, 2.144934066848226, 1.596622711232151, 
     1.273855922313002, 1.0607514045170683, 0.909329874134594, 
     0.7960922730677472, 0.7081559940363196, 0.637860442670732, 
     0.5803609945235884, 0.5324421739536229, 0.49188496923637315, 
     0.4571080294708639, 0.42695363627878, 0.4005545445437263, 
     0.37724812977705013, 0.35651931634432965, 0.3379616174053167, 
     0.32124991272753495, 0.30612103279040653, 0.29235965473397985, 
     0.2797878874103304, 0.2682574659866156, 0.2576438233082915, 
     0.24784153146632698, 0.23876075755522602, 0.23032447959957875, 
     0.22246627886872192, 0.21512857391019172, 0.20826119645016516, 
     0.2018202343122251, 0.19576708467617246, 0.19006767435055363, 
     0.18469181364567153, 0.17961265786518035, 0.17480625605583083, 
     0.17025117094356768, 0.1659281572822101, 0.16181988839633785, 
     0.15791072269408762, 0.1541865034921181, 0.15063438673353519, 
     0.14724269216468885, 0.14400077432480818, 0.1408989103362957, 
     0.13792820199597822, 0.13508049008400413, 0.13234827914700237, 
     0.1297246712908693, 0.12720330774812671, 0.12477831717464677, 
     0.12244426978817335, 0.12019613659243704, 0.11802925304054959};
 
  TVIL_REAL H00y0xxtableB[] = 
     {0, -2.4041138063191885, 0.7020569031595942, 
     -0.32914904655084054, 0.19014882195016747, -0.12364683533791176, 
     0.08677668376924741, -0.06423451201385233, 0.04945610402928957, 
     -0.039246715766299956, 0.031900338120411015, -0.026438707826544672, 
     0.02226826980820068, -0.019011956510902345, 0.01642091895935105, 
     -0.014325558749997136, 0.012607065347161384, -0.011180202275942103, 
     0.009982534269668844, -0.008967481477756264, 0.008099722339680843, 
     -0.007352079497221442, 0.006703366159663909, -0.006136868725103578, 
     0.005639259567403007, -0.005199806064228096, 0.00480978708500146, 
     -0.004462057005823708, 0.0041507161154377, -0.003870858744220093, 
     0.003618378854547591, -0.0033898185850389922, 0.0031822492353346665, 
     -0.0029931769867646285, 0.0028204676533238743, -0.002662286196320071, 
     0.00251704778280204, -0.0023833779368807873, 0.0022600799032780824, 
     -0.002146107768977938, 0.002040544210554558, -0.0019425819792737165, 
     0.0018515084232989532, -0.0017666924907055945, 0.0016875737690661974, 
     -0.001613653204899805, 0.0015444852150559019, -0.0014796709564563887, 
     0.0014188525638071797, -0.001361708199383832, 0.0013079477866839787, 
     -0.001257309322068162, 0.0012095556766017965, -0.0011644718150307931, 
     0.0011218623708515799, -0.001081549526303828, 0.0010433711542404758, 
     -0.0010071791855473106, 0.0009728381713567241, -0.0009402240139391241, 
     0.0009092228440294864, -0.0008797300255928495, 0.000851649271760041, 
     -0.0008248918579652417, 0.0007993759202603035, -0.0007750258284293168, 
     0.0007517716249272957, -0.000729548521861044, 0.0007082964492497942, 
     -0.0006879596486771902, 0.0006684863071958896, -0.0006498282269914515, 
     0.0006319405268685463, -0.0006147813721038703, 0.00059831172962619, 
     -0.0005824951458458169, 0.0005672975447700186, -0.0005526870443148708, 
     0.0005386337889628119, -0.0005251097971247162, 0.0005120888217479344, 
     -0.000499546222872755, 0.00048745885098107, -0.00047580494010507, 
     0.0004645640097739358, -0.00045371677497306473, 0.00044324506337613256, 
     -0.000433131739186441, 0.0004233606329910218, -0.0004139164770911932, 
     0.0004047848458261852, -0.00039595210045411874, 0.0003874053381969639, 
     -0.00037913234509363847, 0.0003711215523395754, -0.0003633619958211473, 
     0.00035584327858046924, -0.00034855553597066755, 0.0003414894032833342, 
     -0.0003346359856497595, 0.00032798683003531365};

  TVIL_REAL H00y0xxtableC[] = 
     {0, -9.771508654754527, -1.2408202605290377, 
     -0.5048824798185181, -0.2878803172561659, -0.1898068911310038, 
     -0.13572943001254256, -0.10231785615741606, -0.08007991545050075, 
     -0.06447265982754746, -0.0530715226638957, -0.04447666310277054, 
     -0.037830091058796644, -0.03258050144179869, -0.028359785256483916, 
     -0.024914179797193903, -0.022063955840308513, -0.019678828723572017, 
     -0.017662430178613242, -0.01594220421475412, -0.014462658599678935, 
     -0.013180752770628162, -0.01206268084528861, -0.011081586384821707, 
     -0.010215912128539256, -0.009448190388849947, -0.008764144339462127, 
     -0.00815201196266957, -0.00760203166999332, -0.007106046806361066, 
     -0.0066571985974590575, -0.006249685607412708, -0.005878573716122715, 
     -0.005539644829318235, -0.005229275543811196, -0.004944339168835382, 
     -0.004682126097610428, -0.004440278699900491, -0.0042167377832312, 
     -0.004009698329553488, -0.0038175727135952714, -0.0036389599904916164, 
     -0.003472620133566043, -0.0033174523302298836, -0.0031724766209523664, 
     -0.0030368183050401704, -0.0029096946464319123, -0.0027904034995400173, 
     -0.0026783135444058, -0.002572855875921251, -0.002473516736557116, 
     -0.002379831218191214, -0.002291377788013735, -0.0022077735174664964, 
     -0.0021286699128250652, -0.002053749262200602, -0.0019827214270896343, 
     -0.0019153210176665186, -0.001851304900217357, -0.0017904499927947626, 
     -0.0017325513116035612, -0.0017505745439953813, -0.0016944131909985144, 
     -0.0016409074191613138, -0.0015898924680629312, -0.0015412161579330536, 
     -0.0014947377543294699, -0.0014503269505709987, -0.001407862954171446, 
     -0.0013672336653019757, -0.001328334936837386, -0.0012910699068568732, 
     -0.001255348395603345, -0.0012210863598844518, -0.0011882053987462143, 
     -0.001156632304985819, -0.0011262986577091072, -0.0010971404516955663, 
     -0.0010690977598189835, -0.0010421144251971867, -0.0010161377801158472, 
     -0.0009911183890980516, -0.0009670098137776403, -0.000943768397487374, 
     -0.0009213530676948425, -0.0008997251546162197, -0.0008788482245111959, 
     -0.0008586879263168326, -0.0008392118504141205, -0.0008203893984429569, 
     -0.0008021916631882897, -0.0007845913176573183, -0.0007675625125521807, 
     -0.000751080781420034, -0.0007351229528304165, -0.0007196670689911255, 
     -0.0007046923102686647, -0.0006901789251288828, -0.0006761081650574485, 
     -0.0006624622240595998, -0.0006492241823748482};
 
  TVIL_REAL H00y0xxtableCL[] = 
     {0, 9.289868133696451, 2.894934066848226, 1.8188449334543733, 
     1.367605922313002, 1.1087514045170683, 0.9371076519123718, 
     0.8135849844380096, 0.7198747440363196, 0.6460908953456292, 
     0.5863609945235884, 0.5369500627590323, 0.49535719145859536, 
     0.4598390262847009, 0.4291402252000628, 0.40233232232150407, 
     0.37871297352705013, 0.3577405660899026, 0.3389904239896788, 
     0.3221246758125328, 0.30687103279040656, 0.29300753293287846, 
     0.2803513735110066, 0.26875060316093957, 0.2580778510860693, 
     0.24822553146632698, 0.23910213215695564, 0.2306293111801305, 
     0.22273960248388225, 0.2153745864568316, 0.2084834186723874, 
     0.2020216374205464, 0.19595019014492246, 0.19023463319519845, 
     0.18484446986386815, 0.17975259955614245, 0.1749348568788761, 
     0.17036962394733846, 0.16603750266783485, 0.16192103642647998, 
     0.1580044726940876, 0.15427355968689183, 0.1507153715083975, 
     0.1473181572180804, 0.1440712100873927, 0.14096475395769487, 
     0.13798984414276871, 0.13513828074695938, 0.13240253261922458, 
     0.12977567044938318, 0.1272513077481267, 0.12482354864670503, 
     0.12248694161338955, 0.12023643831802261, 0.11806735698811854, 
     0.11597534969404188, 0.11395637307994151, 0.11200666212395151, 
     0.11012270656777007, 0.10830122970381631, 0.10653916924913555, 
     0.10733142063336483, 0.1055999912915541, 0.10392354085680952, 
     0.10229949140567485, 0.10072542371529623, 0.09919906523539339, 
     0.0977182791379043, 0.09628105433330511, 0.0948854963554834, 
     0.09352981902826424, 0.09221233683648503, 0.09093145793308785, 
     0.08968567772121164, 0.08847357295686689, 0.08729379632358328, 
     0.08614507143553948, 0.08502618823020572, 0.0839359987155276, 
     0.08287341304022404, 0.08183739585891488, 0.08082696296658941, 
     0.07984117817941463, 0.07887915044109986, 0.07794003113601439, 
     0.07702301159202574, 0.07612732075761058, 0.07525222303921186, 
     0.07439701628609091, 0.07356102991107011, 0.07274362313659374, 
     0.07194418335646399, 0.07116212460444804, 0.07039688612171012, 
     0.06964793101570671, 0.0689147450038041, 0.06819683523543832, 
     0.06749372918714854, 0.06680497362527675, 0.0661301336315486, 
     0.0654687916871319};
 
  TVIL_REAL H00y0xxtableCL2[] = 
     {0, -2., -0.5, -0.2222222222222222, -0.125, -0.08, 
     -0.05555555555555555, -0.04081632653061224, -0.03125, 
     -0.024691358024691357, -0.02, -0.01652892561983471, 
     -0.013888888888888888, -0.011834319526627219, -0.01020408163265306, 
     -0.008888888888888889, -0.0078125, -0.006920415224913495, 
     -0.006172839506172839, -0.00554016620498615, -0.005, 
     -0.0045351473922902496, -0.004132231404958678, -0.003780718336483932, 
     -0.003472222222222222, -0.0032, -0.0029585798816568047, 
     -0.0027434842249657062, -0.002551020408163265, -0.0023781212841854932, 
     -0.0022222222222222222, -0.002081165452653486, -0.001953125, 
     -0.0018365472910927456, -0.0017301038062283738, -0.0016326530612244899, 
     -0.0015432098765432098, -0.0014609203798392988, -0.0013850415512465374, 
     -0.0013149243918474688, -0.00125, -0.001189767995240928, 
     -0.0011337868480725624, -0.001081665765278529, -0.0010330578512396695, 
     -0.0009876543209876543, -0.000945179584120983, -0.0009053870529651426, 
     -0.0008680555555555555, -0.0008329862557267805, -0.0008, 
     -0.0007689350249903883, -0.0007396449704142012, -0.000711997152011392, 
     -0.0006858710562414266, -0.0006611570247933885, -0.0006377551020408163, 
     -0.0006155740227762388, -0.0005945303210463733, -0.0005745475438092502, 
     -0.0005555555555555556, -0.0002687449610319806, -0.00026014568158168577, 
     -0.0002519526329050139, -0.000244140625, -0.00023668639053254438, 
     -0.0002295684113865932, -0.00022276676319893073, 
     -0.00021626297577854672, -0.00021003990758244065, 
     -0.00020408163265306123, -0.00019837333862328903, 
     -0.00019290123456790122, -0.00018765246762994934, 
     -0.00018261504747991235, -0.00017777777777777779, 
     -0.00017313019390581717, -0.00016866250632484398, 
     -0.0001643655489809336, -0.0001602307322544464, -0.00015625, 
     -0.00015241579027587258, -0.000148720999405116, -0.00014515894904920887, 
     -0.0001417233560090703, -0.0001384083044982699, -0.00013520822065981613, 
     -0.0001321178491214163, -0.00012913223140495868, 
     -0.00012624668602449185, -0.0001234567901234568, 
     -0.00012075836251660427, -0.00011814744801512288, 
     -0.00011562030292519366, -0.00011317338162064282, 
     -0.00011080332409972299, -0.00010850694444444444, 
     -0.00010628122010840685, -0.00010412328196584757, 
     -0.00010203040506070809, -0.0001};
 
  TVIL_REAL H00y0xxtableCL3[] = 
     {0, 0.3333333333333333, 0.16666666666666666, 
     0.1111111111111111, 0.08333333333333333, 0.06666666666666667, 
     0.05555555555555555, 0.047619047619047616, 0.041666666666666664, 
     0.037037037037037035, 0.03333333333333333, 0.030303030303030304, 
     0.027777777777777776, 0.02564102564102564, 0.023809523809523808, 
     0.022222222222222223, 0.020833333333333332, 0.0196078431372549, 
     0.018518518518518517, 0.017543859649122806, 0.016666666666666666, 
     0.015873015873015872, 0.015151515151515152, 0.014492753623188406, 
     0.013888888888888888, 0.013333333333333334, 0.01282051282051282, 
     0.012345679012345678, 0.011904761904761904, 0.011494252873563218, 
     0.011111111111111112, 0.010752688172043012, 0.010416666666666666, 
     0.010101010101010102, 0.00980392156862745, 0.009523809523809525, 
     0.009259259259259259, 0.009009009009009009, 0.008771929824561403, 
     0.008547008547008548, 0.008333333333333333, 0.008130081300813009, 
     0.007936507936507936, 0.007751937984496124, 0.007575757575757576, 
     0.007407407407407408, 0.007246376811594203, 0.0070921985815602835, 
     0.006944444444444444, 0.006802721088435374, 0.006666666666666667, 
     0.006535947712418301, 0.00641025641025641, 0.006289308176100629, 
     0.006172839506172839, 0.006060606060606061, 0.005952380952380952, 
     0.005847953216374269, 0.005747126436781609, 0.005649717514124294, 
     0.005555555555555556, 0.00546448087431694, 0.005376344086021506, 
     0.005291005291005291, 0.005208333333333333, 0.005128205128205128, 
     0.005050505050505051, 0.004975124378109453, 0.004901960784313725, 
     0.004830917874396135, 0.004761904761904762, 0.004694835680751174, 
     0.004629629629629629, 0.0045662100456621, 0.0045045045045045045, 
     0.0044444444444444444, 0.0043859649122807015, 0.004329004329004329, 
     0.004273504273504274, 0.004219409282700422, 0.004166666666666667, 
     0.00411522633744856, 0.0040650406504065045, 0.004016064257028112, 
     0.003968253968253968, 0.00392156862745098, 0.003875968992248062, 
     0.0038314176245210726, 0.003787878787878788, 0.003745318352059925, 
     0.003703703703703704, 0.003663003663003663, 0.0036231884057971015, 
     0.0035842293906810036, 0.0035460992907801418, 0.0035087719298245615, 
     0.003472222222222222, 0.003436426116838488, 0.003401360544217687, 
     0.003367003367003367, 0.0033333333333333335};

  lnyox = TVIL_CLOG(y/x);
  lnyox2 = lnyox  * lnyox;
  lnyox3 = lnyox2  * lnyox;

  if (y < 0.5 * x) {     
    result = TVIL_H0000xx (x,qq);     
    rn = r = y/x;
    for (n=1; n < 55; n++) {
      result += rn * (H00y0xxtableA[n] + H00y0xxtableAL[n] * lnyox); 
      rn *= r;
    }
  }
  else if (x < 0.67 * y) {
    result = TVIL_H00000x (y, qq);
    rn = r = x/y;
    for (n=1; n < 100; n++) {
      result += rn * (H00y0xxtableC[n] +
                     -H00y0xxtableCL[n] * lnyox +
                     +H00y0xxtableCL2[n] * lnyox2 +
                     -H00y0xxtableCL3[n] * lnyox3);
      rn *= r;
    }
  }
  else {
    result = TVIL_H00x0xx (x,qq); 
    rn = r = y/x - 1.L;
    for (n=1; n < 100; n++) {
      result += rn * H00y0xxtableB[n];
      rn *= r;
    }
  }

  return result;
}

/* ****************************************************************** */

TVIL_COMPLEX TVIL_H0xyxxy (TVIL_REAL x, TVIL_REAL y, TVIL_REAL qq)
{
  TVIL_REAL r, rn;
  TVIL_COMPLEX lnyox, lnyox2, lnyox3, result;
  int n;

  TVIL_REAL H0xyxxytableA[] = 
     {0, -5., -0.5578703703703703, -0.09913580246913581, 
     -0.022110689315408702, -0.005895757597571657, -0.0018702317387822198, 
     -0.0007042900382162473, -0.0003097663721546232, -0.000154640983596654, 
     -0.00008502588227017905, -0.000050254114096208755, 
     -0.00003138452816573706, -0.00002046893559725157, 
     -0.000013829165881557104, -9.622650109638615e-6, 
     -6.8659629150144794e-6, -5.006656173975466e-6, 
     -3.7209996255646362e-6, -2.812405962491567e-6, -2.157776521066482e-6, 
     -1.677933689697068e-6, -1.3207265838036512e-6, 
     -1.0510694877493304e-6, -8.449028750850438e-7, -6.854406632537833e-7, 
     -5.607856518275605e-7, -4.6238342897279424e-7, -3.840013021044869e-7, 
     -3.2104250060974094e-7, -2.700783643948213e-7, 
     -2.2852462784409347e-7, -1.944144237065131e-7, 
     -1.6623713473355173e-7, -1.4282267012976278e-7, 
     -1.232574658352298e-7, -1.06822896276704e-7, -9.294969027163774e-8, 
     -8.118389176376313e-8, -7.116122822750638e-8, -6.258765724291877e-8, 
     -5.522449157606733e-8, -4.887694459095076e-8, -4.33852502642664e-8, 
     -3.861773520567292e-8, -3.446538080926251e-8, -3.08375303802943e-8, 
     -2.765848150315061e-8, -2.4864766897187687e-8, 
     -2.2402973770339716e-8, -2.0227986636158453e-8, 
     -1.830156485803571e-8, -1.6591186090882456e-8, -1.506910194780858e-8, 
     -1.3711563825649354e-8, -1.2498185758590793e-8, 
     -1.1411418084085544e-8, -1.0436111083339332e-8, -9.55915196155443e-9, 
     -8.769161832823747e-9, -8.056241976710532e-9, -7.411760694397611e-9, 
     -6.828173731200117e-9, -6.298872540799211e-9, -5.8180557153865544e-9, 
     -5.3806197496947645e-9, -4.98206598815944e-9, -4.618421156430481e-9, 
     -4.286169327171724e-9, -3.98219353606324e-9, -3.703725563352776e-9, 
     -3.448302642042155e-9, -3.2137300560685934e-9, 
     -2.9980487588180047e-9, -2.799507280533015e-9, 
     -2.6165373079142367e-9, -2.4477324147029622e-9, 
     -2.2918295017092746e-9, -2.147692571396526e-9, 
     -2.0142985180171567e-9, -1.890724661264768e-9, 
     -1.7761377909759638e-9, -1.6697845238236598e-9, 
     -1.5709828012120412e-9, -1.4791143815536753e-9, 
     -1.3936182004771612e-9, -1.3139844898554839e-9, 
     -1.2397495613396567e-9, -1.1704911727277594e-9, 
     -1.105824406328517e-9, -1.0453979977690702e-9, 
     -9.888910616814057e-10, -9.360101675765507e-10, 
     -8.864867251445692e-10, -8.400746433402494e-10, 
     -7.965482320462488e-10, -7.557003189464212e-10, 
     -7.173405575760918e-10, -6.81293905414484e-10, 
     -6.473992534080244e-10, -6.155081905135674e-10};
 
  TVIL_REAL H0xyxxytableAL[] = 
     {0, 3., 0.4861111111111111, 0.11, 0.03072987528344671, 
     0.010370622323003275, 0.004184146314377195, 0.0019735454549450076, 
     0.0010539306324510703, 0.0006177285713118969, 0.0003879177322815684, 
     0.0002566532913514061, 0.00017687755488052944, 0.00012597861479456812, 
     0.00009220683648964006, 0.00006906169092669979, 0.000052759657530502066, 
     0.000041004639502432355, 0.00003235335719037635, 
     0.000025870850909841243, 0.000020935371064498435, 
     0.00001712380429765053, 0.000014142248661849423, 
     0.000011782770572046298, 9.895801889896504e-6, 8.372129312630723e-6, 
     7.130907379785522e-6, 6.111535581124009e-6, 5.268062727203476e-6, 
     4.5652737429049065e-6, 3.975914796126812e-6, 3.478700235396734e-6, 
     3.0568639511594176e-6, 2.6970947488465306e-6, 2.3887458384266142e-6, 
     2.1232421851831665e-6, 1.8936321742283025e-6, 1.6942455650488228e-6, 
     1.5204304516826837e-6, 1.3683494567443115e-6, 1.234820698262328e-6, 
     1.1171928597405451e-6, 1.0132464259726425e-6, 9.2111513332285e-7, 
     8.392231391815126e-7, 7.662344911177193e-7, 7.01012277147418e-7, 
     6.425854390432468e-7, 5.901216839440126e-7, 5.429052739754287e-7, 
     5.003187369446169e-7, 4.6182774370977286e-7, 4.269685544798783e-7, 
     3.95337558121203e-7, 3.6658252377564447e-7, 3.403952589078247e-7, 
     3.1650542695229955e-7, 2.9467532456292314e-7, 2.7469545576808934e-7, 
     2.5638077017382086e-7, 2.3956745632281135e-7, 2.2411020064330328e-7, 
     2.0987983806505043e-7, 1.9676133308884265e-7, 1.846520404582163e-7, 
     1.7346020305949688e-7, 1.6310365163503562e-7, 1.5350867662484614e-7, 
     1.44609047185342e-7, 1.3634515635569537e-7, 1.2866327460114794e-7, 
     1.2151489667814626e-7, 1.148561690352201e-7, 1.0864738686453261e-7, 
     1.0285255151581073e-7, 9.743898032897191e-8, 9.237696207677455e-8, 
     8.763945216918299e-8, 8.320180258559341e-8, 7.904152219333376e-8, 
     7.51380637005394e-8, 7.147263399488595e-8, 6.802802505024948e-8, 
     6.478846295243082e-8, 6.173947291203579e-8, 5.886775840532525e-8, 
     5.6161092818948146e-8, 5.360822217750751e-8, 5.119877770856538e-8, 
     4.8923197151925716e-8, 4.677265385218974e-8, 4.473899278849349e-8, 
     4.281467279542371e-8, 4.099271431641551e-8, 3.926665210721296e-8, 
     3.76304923737164e-8, 3.607867388702416e-8, 3.4606032669798975e-8, 
     3.320776989319017e-8, 3.187942266323077e-8, 3.061683741060088e-8};
 
  TVIL_REAL H0xyxxytableAL2[] = 
     {0, -0.5, -0.125, -0.03888888888888889, -0.014583333333333334, 
     -0.006507936507936508, -0.003373015873015873, -0.001965098393669822, 
     -0.0012482309357309358, -0.0008447253545292761, -0.0005994169229463347, 
     -0.0004412285653490269, -0.0003344261374447133, -0.0002596209981508759, 
     -0.00020562644360790623, -0.00016565875809551185, 
     -0.00013543245400771069, -0.00011214589769858343, 
     -0.00009391327366203405, -0.00007943277158715695, 
     -0.0000677856611192188, -0.00005831078320426595, 
     -0.00005052398936614752, -0.00004406517310776336, 
     -0.00003866268757366699, -0.00003410897699046893, 
     -0.000030243594001871133, -0.000026941174889970454, 
     -0.000024102800840456915, -0.000021649708574869066, 
     -0.00001951865489533295, -0.000017658461240620136, 
     -0.000016027410629843768, -0.000014591267449433058, 
     -0.000013321757248188267, -0.000012195389684574449, 
     -0.000011192539856528917, -0.000010296725895482213, 
     -9.494036871434965e-6, -8.772676710086096e-6, -8.122598305759567e-6, 
     -7.535208244168854e-6, -7.003127164113113e-6, -6.519994233405224e-6, 
     -6.080306807419199e-6, -5.6792883039682125e-6, -5.312778828030176e-6, 
     -4.977144231895299e-6, -4.669200186774124e-6, -4.386148534271297e-6, 
     -4.12552372752809e-6, -3.885147597482366e-6, -3.6630910160739466e-6, 
     -3.4576412953790097e-6, -3.267274374845677e-6, 
     -3.0906310197037756e-6, -2.9264963912240766e-6, 
     -2.773782460761312e-6, -2.631512829838938e-6, -2.498809592147531e-6, 
     -2.374881933549511e-6, -2.2590162156258552e-6, -2.150567329036213e-6, 
     -2.0489511366389495e-6, -1.9536378542465587e-6, 
     -1.8641462401279175e-6, -1.7800384837593947e-6, 
     -1.700915700555825e-6, -1.6264139529340455e-6, 
     -1.5562007295263819e-6, -1.489971824036601e-6, -1.427448563416806e-6, 
     -1.3683753419872312e-6, -1.3125174240247563e-6, 
     -1.2596589823778036e-6, -1.2096013449637103e-6, 
     -1.162161424684811e-6, -1.117170311457072e-6, -1.074472007760107e-6, 
     -1.0339222914566158e-6, -9.953876916487387e-7, -9.587445650856359e-7, 
     -9.238782621504256e-7, -8.906823727689567e-7, -8.590580437260953e-7, 
     -8.289133598712746e-7, -8.001627825644343e-7, -7.727266394735279e-7, 
     -7.465306605003313e-7, -7.215055551950275e-7, -6.975866275328055e-7, 
     -6.747134243768213e-7, -6.528294143493228e-7, -6.318816941834177e-7, 
     -6.118207199378154e-7, -5.926000607310363e-7, -5.741761728944654e-7, 
     -5.565081926591294e-7, -5.395577456824833e-7, -5.232887718917511e-7, 
     -5.076673642719785e-7};
 
  TVIL_REAL H0xyxxytableC[] = 
     {0, -1.893829290521217, 0.1733144102723232, 0.14314759539822597, 
     0.10693668774301324, 0.0812332913152306, 0.06277377862958178, 
     0.04915106770575772, 0.038849242865190536, 0.030894429083372643, 
     0.024641473156461646, 0.019650536265279517, 0.015613973918725765, 
     0.012311659789973728, 0.00958287805435415, 0.00730816016794944, 
     0.005397259375023944, 0.0037810117521402953, 0.00240571660112171, 
     0.001229185265249344, 0.0002179166812376343, -0.0006549523902822374, 
     -0.0014111556771840839, -0.002068421491650041, -0.0026413203584760847, 
     -0.0031419111284282653, -0.003580238445972689, -0.0039647193489054655, 
     -0.004302446320150671, -0.004599426772116333, -0.004860773729838147, 
     -0.005090858733121428, -0.005293435258081236, -0.005471738964086842, 
     -0.005628569595963556, -0.005766358269096106, -0.005887223035237948, 
     -0.005993014997146509, -0.0060853567588044916, -0.006165674627429021, 
     -0.006235225696295463, -0.0062951207134444576, -0.006346343465624418, 
     -0.006389767268167701, -0.006426169041487767, -0.006456241367148119, 
     -0.006480602846128919, -0.0064998070252829665, -0.0065143501121536496, 
     -0.006524677661097822, -0.006531190383279234, -0.006534249208211371, 
     -0.006534179704062643, -0.006531275947043391, -0.006525803916199821, 
     -0.006518004478307887, -0.006508096017859677, -0.006496276759017983, 
     -0.006482726819605683, -0.0064676100314623175, -0.0064510755566636185, 
     -0.006433259325002841, -0.006414285314658555, -0.006394266695015152, 
     -0.006373306848081267, -0.006351500282793313, -0.006328933454643762, 
     -0.006305685501485858, -0.006281828904999523, -0.006257430086123936, 
     -0.00623254994174352, -0.006207244329029818, -0.006181564503076886, 
     -0.006155557512799989, -0.006129266559487895, -0.0061027313218916315, 
     -0.00607598825129016, -0.006049070839585238, -0.006022009863137445, 
     -0.0059948336047566594, -0.005967568055996887, -0.005940237101673997, 
     -0.005912862688319628, -0.005885464978104403, -0.005858062489602953, 
     -0.005830672226631847, -0.0058033097962650465, -0.005775989517020359, 
     -0.00574872451810985, -0.005721526830559348, -0.0056944074709225564, 
     -0.005667376518244609, -0.0056404431848668946, -0.005613615881608497, 
     -0.0055869022778083435, -0.005560309356667556, -0.005533843466289899, 
     -0.005507510366781765, -0.005481315273740461, -0.005455262898429021, 
     -0.005418913823326536};
 
  TVIL_REAL H0xyxxytableCL[] = 
     {0, -2., -0.3248775943990546, -0.1780488906953509, 
     -0.10794843429489148, -0.06466857334904913, -0.03624111656783163, 
     -0.016878259768595416, -0.0033209484243580667, 0.006379022395829492, 
     0.013436408754181395, 0.018635825278401047, 0.022499739929363688, 
     0.025385413915747584, 0.02754286982436241, 0.029150673880992323, 
     0.030338660015575517, 0.031202733227660207, 0.03181473598847406, 
     0.03222916077986704, 0.03248780245587715, 0.032623037350458, 
     0.03266017001201346, 0.032619136182270206, 0.032515754413808357, 
     0.03236265672884696, 0.0321699880684205, 0.031945937180456824, 
     0.03169714325325892, 0.03142901001102131, 0.031145950232723478, 
     0.030851577492567582, 0.030548857532098878, 0.03024022851663284, 
     0.029927697133841635, 0.02961291580895685, 0.029297245065263787, 
     0.028981804128989153, 0.028667512178637465, 0.028355122109261402, 
     0.028045248278176615, 0.027738389388424656, 0.027434947426599125, 
     0.027135243385345426, 0.02683953035521964, 0.026548004456152935, 
     0.026260813988377825, 0.02597806711092998, 0.025699838298633447, 
     0.0254261737826607, 0.025157096142909577, 0.02489260819068131, 
     0.02463269625602152, 0.02437733297445785, 0.024126479651841826, 
     0.02388008827287192, 0.02363810320808242, 0.023400462665183683, 
     0.023167099923280924, 0.02293794438239504, 0.022712922455634093, 
     0.022491958327131297, 0.022274974595328557, 0.022061892818219197, 
     0.021852633974674615, 0.021647118853882697, 0.021445268383158377, 
     0.02124700390289134, 0.021052247396130236, 0.020860921679228043, 
     0.020672950559059784, 0.020488258961545814, 0.020306773035550212, 
     0.020128420235656655, 0.01995312938683887, 0.019780830733626877, 
     0.01961145597601325, 0.019444938294036825, 0.019281212362717733, 
     0.019120214358789595, 0.018961881960479522, 0.01880615434141646, 
     0.018652972159602817, 0.018502277542257355, 0.018354014067227937, 
     0.018208126741577943, 0.018064561977867596, 0.017923267568580425, 
     0.01778419265908271, 0.01764728771945031, 0.01751250451545025, 
     0.017379796078923967, 0.017249116677783526, 0.01712042178580181, 
     0.016993668052350433, 0.016868813272216357, 0.016745816355607665, 
     0.01662463729844156, 0.01650523715299227, 0.016387577998963447, 
     0.0162562814008219};
 
  TVIL_REAL H0xyxxytableCL2[] = 
     {0, -0.25, 0.026041666666666668, -0.019444444444444445, 
     -0.04700753348214286, -0.06036427331349206, -0.06637948143075097, 
     -0.0686103126009655, -0.06883963681654073, -0.06799407655959407, 
     -0.06657504725072716, -0.06486307011799483, -0.0630189105926035, 
     -0.06113599546287946, -0.05926873640074436, -0.05744839967638495, 
     -0.05569227745021364, -0.05400912804943074, -0.05240247376080019, 
     -0.050872635358436745, -0.04941800455671499, -0.04803584762274223, 
     -0.0467228157002505, -0.04547526909304419, -0.04428948220449073, 
     -0.04316177126133637, -0.042088571793058596, -0.04106648333411533, 
     -0.04009229277050895, -0.039162983856229884, -0.03827573788620914, 
     -0.0374279288414178, -0.036617115212710355, -0.03584102996879567, 
     -0.03509756963562481, -0.03438478311847195, -0.0337008606708424, 
     -0.0330441232607957, -0.032413012481739986, -0.03180608108534567, 
     -0.0312219841681655, -0.03065947101347804, -0.030117377570750267, 
     -0.02959461954346898, -0.029090186049520423, -0.02860313381514466, 
     -0.028132581862604616, -0.02767770665229556, -0.027237737641528585, 
     -0.026811953224276684, -0.02639967701851549, -0.026000274470251807, 
     -0.025613149745796232, -0.025237742886229517, -0.02487352720028958, 
     -0.024520006874043238, -0.02417671477769015, -0.02384321045167416, 
     -0.023519078255950728, -0.023203925667785067, -0.022897381714842444, 
     -0.022599095531588946, -0.02230873502815826, -0.0220259856618674, 
     -0.021750549302491602, -0.02148214318324507, -0.02122049893016824, 
     -0.02096536166330211, -0.020716489163642934, -0.020473651100422905, 
     -0.020236628313760816, -0.020005212148175956, -0.019779203832864137, 
     -0.019558413905000974, -0.019342661672668397, -0.0191317747142995, 
     -0.0189255884118074, -0.018723945514808715, -0.01852669573357398, 
     -0.018333695358538786, -0.018144806904391478, -0.01795989877691908, 
     -0.017778844960943643, -0.017601524727817876, -0.01742782236107356, 
     -0.017257626898929575, -0.01709083189246962, -0.016927335178394114, 
     -0.016767038665336582, -0.016609848132813586, -0.01645567304194891, 
     -0.016304426357178582, -0.016156024378203397, -0.016010386581510626, 
     -0.015867435470837393, -0.015727096435994316, -0.015589297619510716, 
     -0.01545396979060177, -0.015321046225994122, -0.015190462597179266, 
     -0.015056522015793905};
 
  TVIL_REAL H0xyxxytableCL3[] = 
     {0, 0., 0.0625, 0.0625, 0.056640625, 0.05078125, 
     0.045735677083333336, 0.04150390625, 0.037952423095703125, 
     0.034947713216145836, 0.032380294799804685, 0.030164718627929688, 
     0.028234978516896565, 0.026539921760559082, 0.02503957280090877, 
     0.023702370127042134, 0.022503129119286314, 0.02142154355533421, 
     0.020441077801580023, 0.019548141019186005, 0.01873146561902104, 
     0.017981634702491214, 0.01729071914912145, 0.01665199617923463, 
     0.016059729053604716, 0.015508993093631318, 0.014995537126020659, 
     0.014515672263138867, 0.014066181958393634, 0.01364424875481346, 
     0.013247394233047683, 0.012873429472672872, 0.012520413945376034, 
     0.012186621215051494, 0.011870510167162132, 0.011570700755984637, 
     0.01128595346397447, 0.01101515182736483, 0.010757287507249795, 
     0.010511447483962162, 0.010276803030652319, 0.01005260018420985, 
     0.009838151481538016, 0.009632828769359307, 0.009436056928247201, 
     0.009247308378028653, 0.009066098253314588, 0.008891980155654098, 
     0.008724542403427854, 0.008563404712694728, 0.008408215252256424, 
     0.008258648024585726, 0.008114400531277518, 0.007975191687571244, 
     0.007840759955455509, 0.00771086166905969, 0.007585269529593336, 
     0.007463771250117979, 0.007346168333014768, 0.007232274965216635, 
     0.007121917018164722, 0.007014931141074426, 0.006911163937497583, 
     0.006810471216377711, 0.006712717309843514, 0.006617774450895628, 
     0.0065255222049329965, 0.006435846949755162, 0.006348641399279351, 
     0.006263804166738756, 0.00618123936359096, 0.0061008562307719475, 
     0.006022568799288926, 0.005946295577460723, 0.005871959262393264, 
     0.0057994864735242775, 0.005728807506290031, 0.005659856104160944, 
     0.005592569247465555, 0.005526886957575965, 0.005462752115164976, 
     0.005400110291367543, 0.005338909590788642, 0.005279100505397759, 
     0.005220635778438091, 0.005163470277557587, 0.005107560876439936, 
     0.005052866344277538, 0.004999347242486151, 0.004946965828112883, 
     0.004895685963436194, 0.004845473031299095, 0.004796293855755188, 
     0.004748116627642114, 0.004700910834728611, 0.004654647196110166, 
     0.004609297600554414, 0.004564835048521214, 0.004521233597604059, 
     0.004478468311159267, 0.004436515209907435};
 
  TVIL_REAL H0xyxxytableB[] = 
     {0, -4.46174402057842, 1.6878607023212888, -0.9209228826750575, 
     0.5900508061969214, -0.41441277337672533, 0.3089926768655442, 
     -0.2403153849559233, 0.19286542590750178, -0.15859526282017034, 
     0.13297044119648305, -0.11326784738468865, 0.09776713805548942, 
     -0.08533562811320572, 0.07520147410429556, -0.06682321386828394, 
     0.05981142497972208, -0.05387992167663397, 0.048814374574228214, 
     -0.0444515680967183, 0.04066535208137678, -0.0373569196999512, 
     0.034447948437782705, -0.03187567639913741, 0.029589312059499995, 
     -0.027547378802056877, 0.025715725163576952, -0.024066016038960747, 
     0.022574575985693734, -0.021221493447636436, 0.01998992051530074, 
     -0.018865520758518523, 0.0178360302799363, -0.016890906125774457, 
     0.016021042669228047, -0.015218541302766647, 0.014476522249988882, 
     -0.01378896988906661, 0.013150604914649533, -0.012556778127493257, 
     0.012003381755200493, -0.011486775062431953, 0.011003721669632371, 
     -0.010551336513795473, 0.010127040786961694, -0.009728523505581815, 
     0.00935370861486079, -0.009000726732681263, 0.0086678907970541, 
     -0.008353675011578882, 0.008056696585638956, -0.007775699852434204, 
     0.007509542416146764, -0.007257183036237624, 0.0070176710038425405, 
     -0.006790136802234031, 0.0065737838784031, -0.006367881372709383, 
     0.006171757689493448, -0.005984794783221947, 0.005806423090822493, 
     -0.005636117009214181, 0.005473390863326114, -0.005317795298578273, 
     0.005168914053284429, -0.005026361058371592, 0.004889777831890124, 
     -0.0047588311306051845, 0.0046332108448814135, -0.004512628065414373, 
     0.004396813402577721, -0.004285515344946866, 0.004178498917400603, 
     -0.004075544345548743, 0.003976445886843649, -0.0038810108045110314, 
     0.0037890582751878704, -0.0037004187026731805, 0.0036149326337039656, 
     -0.0035324502922396694, 0.003452830854337119, -0.0033759414483697514, 
     0.0033016569021807746, -0.003229859406645455, 0.003160437995612775, 
     -0.0030932874828539236, 0.0030283082015369472, -0.002965405681930794, 
     0.002904493624959495, -0.0028454850208619042, 0.0027883022477690755, 
     -0.002732869872929214, 0.002679115056783324, -0.002626970193210315, 
     0.0025763696262876834, -0.0025272539380618853, 0.0024795664316384775, 
     -0.002433247855058024, 0.002388246122303622, -0.002344509359620675, 
     0.0023020097683009166};
 
  TVIL_REAL H0xyxxytableD[] = 
     {0, -2.9076415309340558, 0.9075528692303481, -0.444529021879714, 
     0.26572809604126735, -0.1777554768694839, 0.12780651820329558, 
     -0.09662408670107187, 0.07579727698279626, -0.06116592307686569, 
     0.05047555006610807, -0.042415741881200564, 0.03618133519416877, 
     -0.031254783711776785, 0.027290741079000147, -0.02405138780562699, 
     0.02136848178463538, -0.019120199303275548, 0.017216516624285083, 
     -0.01558970724338038, 0.014188002964374155, -0.012971267577422466, 
     0.011907983253944196, -0.010973112526879616, 0.0101465561586494, 
     -0.009412023993733865, 0.008756196807222824, -0.008168096313968753, 
     0.007638606158674219, -0.007160103817632393, 0.006726174940648658, 
     -0.006331389641343069, 0.005971125810778327, -0.005641428462986112, 
     0.005338896933981568, -0.005060593790053414, 0.004803970787526465, 
     -0.004566808322994893, 0.004347165629798769, -0.0041433395900270995, 
     0.003953830495936596, -0.0037773134492230104, 0.0036126143591247925, 
     -0.003458689711285018, 0.003314609443645304, -0.0031795423944797965, 
     0.0030527438892752334, -0.0029335451137307705, 0.0028213439843883734, 
     -0.002715597279881146, 0.002615813837243075, -0.002521548651263951, 
     0.0024323977421330384, -0.0023479936788640687, 0.0022680016642300307, 
     -0.0021921161019405402, 0.0021200575791884557, -0.002051570207965897, 
     0.0019864192770975356, -0.0019243891740729074, 0.0018652815417350421, 
     -0.0018089136399023903, 0.0017551168862296948, -0.0017037355541899418, 
     0.0016546256090878703, -0.0016076536655911175, 0.001562696052458859, 
     -0.0015196379720223072, 0.0014783727435769556, -0.001438801121224173, 
     0.0014008306778859544, -0.0013643752482395896, 0.001329354424202362, 
     -0.0012956930973631944, 0.001263321043421476, -0.0012321725442727555, 
     0.0012021860438837205, -0.0011733038345390773, 0.001145471770427726, 
     -0.0011186390058724595, 0.0010927577558038257, -0.0010677830763382967, 
     0.001043672663550733, -0.0010203866687329437, 0.0009978875286088392, 
     -0.0009761398091349846, 0.0009551100616550473, -0.0009347666903015402, 
     0.0009150798296483415, -0.0008960212317160792, 0.0008775641615202016, 
     -0.0008596833004296296, 0.0008423546566741469, -0.0008255554824008108, 
     0.0008092641967362201, -0.0007934603143615092, 0.0007781243791520671, 
     -0.0007632379024747016, 0.0007487833057715284, -0.0007347438670927479, 
     0.0007211036712702042};

  lnyox = TVIL_CLOG(y/x);
  lnyox2 = lnyox  * lnyox;
  lnyox3 = lnyox2  * lnyox;

  if (y < 0.5 * x) {
    result = TVIL_H00x0xx (x,qq);
    rn = r = y/x;
    for (n=1; n < 100; n++) {
      result += rn * (H0xyxxytableA[n] + 
                      H0xyxxytableAL[n] * lnyox + 
                      H0xyxxytableAL2[n] * lnyox2);
      rn *= r;
    }
  }
  else if (y < 1.6 * x) {
    result = TVIL_H0xxxxx (x,qq);
    rn = r = y/x - 1.L;
    for (n=1; n < 100; n++) {
      result += rn * H0xyxxytableD[n];
      rn *= r;
    }
  }
  else if (y < 6. * x) {
    result = -13.325948659996127462L - 6.L * Zeta3 * TVIL_Ap(x,qq);
    rn = r = y/(4.L*x) - 1.L;
    for (n=1; n < 64; n++) {
      result += rn * H0xyxxytableB[n];
      rn *= r;
    }
  }
  else {
    result = TVIL_H0000xx (y, qq);
    rn = r = 4.L * x/y;
    for (n=1; n < 100; n++) {
      result += rn * (H0xyxxytableC[n] 
                     +H0xyxxytableCL[n] * lnyox 
                     +H0xyxxytableCL2[n] * lnyox2
                     +H0xyxxytableCL3[n] * lnyox3);
      rn *= r;
    }
  }

  return result;
}

/* ****************************************************************** */

TVIL_COMPLEX TVIL_H00yxyx (TVIL_REAL x, TVIL_REAL y, TVIL_REAL qq)
{
  TVIL_REAL r, rn;
  TVIL_COMPLEX lnyox, lnyox2, lnyox3, result;
  int n;

  TVIL_REAL H00yxyxtableA[] = 
     {0, -1.5479430968404058, -0.06628527877928236, 
     -0.006013133100024864, -0.0012565424995303748, -0.0005429208076882179, 
     -0.00031842964941132757, -0.0002074264683930396, 
     -0.00014219693920383686, -0.0001009927672878716, 
     -0.00007380953218811489, -0.00005526909950618017, 
     -0.00004226384992596835, -0.00003291614993198787, 
     -0.000026051715650225568, -0.000020914194910195504, 
     -0.000017003563198997238, -0.000013981443347461545, 
     -0.000011613976721705294, -9.736431331362152e-6, 
     -8.230744739693364e-6, -7.010959380318245e-6, -6.013585436323936e-6, 
     -5.191105846785796e-6, -4.507524291977443e-6, -3.935265571618351e-6, 
     -3.4529862216682757e-6, -3.044007276621232e-6, 
     -2.6951783875554098e-6, -2.396045018151705e-6, -2.138231244982706e-6, 
     -1.914977721463788e-6, -1.7207925252279861e-6, 
     -1.5511849685016866e-6, -1.4024609665763318e-6, 
     -1.271564494251785e-6, -1.155953841333315e-6, -1.0535043542751414e-6, 
     -9.624314898002755e-7, -8.812295574387769e-7, -8.086226626313544e-7, 
     -7.435251989626099e-7, -6.850098601921659e-7, -6.322816086142482e-7, 
     -5.846563876035117e-7, -5.415436329409432e-7, -5.024318413271902e-7, 
     -4.6687661118043236e-7, -4.3449069198371377e-7, 
     -4.049356726710922e-7, -3.7791501320786035e-7, -3.531681814096223e-7, 
     -3.304657027615226e-7, -3.0960496727338945e-7, -2.904066663202017e-7, 
     -2.7271175556476427e-7, -2.5637885866882797e-7, 
     -2.412820415205713e-7, -2.2730889887947758e-7, 
     -2.1435890524156034e-7, -2.0234198981170795e-7, 
     -1.9117730209261916e-7, -1.8079214004391953e-7, 
     -1.7112101725500268e-7, -1.6210484928991086e-7, 
     -1.5369024244544462e-7, -1.4582887072974615e-7, 
     -1.3847692901065148e-7, -1.3159465207618452e-7, 
     -1.2514589085460473e-7, -1.1909773830799366e-7, 
     -1.1342019858198318e-7, -1.0808589389805641e-7, 
     -1.0306980444113894e-7, -9.834903714639362e-8, -9.390261984377932e-8, 
     -8.971131769238711e-8, -8.57574692415865e-8, -8.202483980320719e-8, 
     -7.849849011722189e-8, -7.516465855007946e-8, -7.201065528619457e-8, 
     -6.902476716434412e-8, -6.619617197625336e-8, -6.351486118819093e-8, 
     -6.097157017106918e-8, -5.855771513303032e-8, -5.6265336043045125e-8, 
     -5.408704491659095e-8, -5.2015978906639477e-8, 
     -5.0045757706383837e-8, -4.81704448255585e-8, -4.638451235088823e-8, 
     -4.468280884402222e-8, -4.3060530068028836e-8, -4.15131922667952e-8, 
     -4.003660775106336e-8, -3.8626862570823624e-8, -3.728029607680258e-8, 
     -3.599348219419139e-8, -2.6456786080566956e-6};
 
  TVIL_REAL H00yxyxtableAL[] = 
     {0, 1.0887664832879436, 0.06665136596654848, 
     0.003524648957387709, -0.0011551691899625933, -0.0010375836099557022, 
     -0.0006754636432639293, -0.0004333983799966759, -0.0002867223298754863, 
     -0.00019669025767321614, -0.00013951244223977483, 
     -0.00010187560601047128, -0.00007627552343351132, 
     -0.000058351699759020975, -0.00004548097430499353, 
     -0.00003603185243482929, -0.000028958269436163335, 
     -0.000023570945324908037, -0.000019404458681708046, 
     -0.000016137588414549017, -0.00001354423196453057, 
     -0.000011462389107299852, -9.774106606182764e-6, 
     -8.392236470960101e-6, -7.251524773771063e-6, -6.30250931594681e-6, 
     -5.507273403733252e-6, -4.836447350762219e-6, -4.267062096940133e-6, 
     -3.7809933355511067e-6, -3.3638204404508167e-6, 
     -3.003980462037761e-6, -2.6921344995786835e-6, -2.420688617036703e-6, 
     -2.183428378181675e-6, -1.9752377215987406e-6, -1.791881010173032e-6, 
     -1.6298328057488942e-6, -1.4861439883947052e-6, 
     -1.3583357641968449e-6, -1.2443152269941285e-6, 
     -1.1423076918983264e-6, -1.0508021638788313e-6, 
     -9.685071564654698e-7, -8.943147137556771e-7, -8.272709703805006e-7, 
     -7.665519498046856e-7, -7.114435809253089e-7, -6.613251279900092e-7, 
     -6.156553952462022e-7, -5.739611971926719e-7, -5.358276865727287e-7, 
     -5.008902118710688e-7, -4.688274389900825e-7, -4.39355521727846e-7, 
     -4.1222314551257775e-7, -3.872073007540416e-7, -3.641096678363139e-7, 
     -3.427535165032966e-7, -3.2298103919345577e-7, 
     -3.0465105155689046e-7, -2.876370045588277e-7, 
     -2.7182526173004746e-7, -2.571136026552947e-7, 
     -2.4340992000441755e-7, -2.3063108255428727e-7, -2.18701940919936e-7, 
     -2.0755445626946508e-7, -1.971269352671039e-7, 
     -1.8736335697590179e-7, -1.782127795400016e-7, 
     -1.6962881622475692e-7, -1.6156917187706902e-7, 
     -1.539952321240879e-7, -1.4687169869351864e-7, 
     -1.4016626514425897e-7, -1.3384932806762512e-7, 
     -1.2789372947826523e-7, -1.222745266776769e-7, 
     -1.1696878635674869e-7, -1.119554001192174e-7, 
     -1.0721491896564987e-7, -1.0272940458614331e-7, 
     -9.848229557663834e-8, -9.445828692466114e-8, -9.06432213106117e-8, 
     -8.702399094475773e-8, -8.358844881157562e-8, -8.032532832514711e-8, 
     -7.72241705146436e-8, -7.427525795979586e-8, -7.146955478459982e-8, 
     -6.879865209502116e-8, -6.625471831456187e-8, -6.383045393146997e-8, 
     -6.151905022416628e-8, -5.93141515780426e-8, -5.720982104793783e-8, 
     -5.520050885700588e-8, -5.328102355493788e-8, -3.886825059595448e-6};
 
  TVIL_REAL H00yxyxtableAL2[] = 
     {0, -0.25, -0.036458333333333336, -0.00798611111111111, 
     -0.0024483816964285714, -0.0009768725198412698, -0.00047134171890031266, 
     -0.00025940013994561983, -0.00015649799237254987, 
     -0.00010091993737563782, -0.00006844520579171265, 
     -0.00004829679217216003, -0.000035190304357214175, 
     -0.000026330671140142776, -0.00002014760314455338, 
     -0.00001571444432354641, -0.000012461483493098061, 
     -0.000010026059691991769, -8.170288070682427e-6, 
     -6.733996470556322e-6, -5.606826041035188e-6, -4.711168536303925e-6, 
     -3.991439725433649e-6, -3.407168576038599e-6, -2.928445748155955e-6, 
     -2.5328657945517278e-6, -2.203435488025686e-6, -1.927119301437979e-6, 
     -1.6938126070610009e-6, -1.495606712499644e-6, -1.326256023295854e-6, 
     -1.180787148515242e-6, -1.0552089709901362e-6, -9.462953944500218e-7, 
     -8.514209889752808e-7, -7.684355399411732e-7, -6.955674870864235e-7, 
     -6.313490139052001e-7, -5.745575013214496e-7, -5.241694503563882e-7, 
     -4.793239782858142e-7, -4.392937182211531e-7, -4.034614830611753e-7, 
     -3.7130144670001085e-7, -3.4236388695189644e-7, 
     -3.162627531909837e-7, -2.9266548666781525e-7, -2.71284646830833e-7, 
     -2.5187099287129456e-7, -2.34207743505751e-7, -2.1810579513656038e-7, 
     -2.0339972300180948e-7, -1.899444247312922e-7, 
     -1.7761229310598309e-7, -1.6629082646487042e-7, 
     -1.558806023971127e-7, -1.462935540771716e-7, -1.3745149959580005e-7, 
     -1.2928488348861816e-7, -1.2173169681431257e-7, 
     -1.1473654793482443e-7, -1.0824986087280442e-7, 
     -1.0222718198119188e-7, -9.662857882486735e-8, -9.1418117778677e-8, 
     -8.656340899607392e-8, -8.203520918305839e-8, -7.780707409103632e-8, 
     -7.385505387434488e-8, -7.015742548764587e-8, -6.66944571608855e-8, 
     -6.344820071408775e-8, -6.0402308084533e-8, -5.75418689542012e-8, 
     -5.4853266801577097e-8, -5.232405107204684e-8, -4.994282347590575e-8, 
     -4.769913669132104e-8, -4.558340397881412e-8, -4.358681841006078e-8, 
     -4.1701280582149274e-8, -3.99193338331441e-8, -3.8234106099439277e-8, 
     -3.663925766293935e-8, -3.512893412909419e-8, -3.369772405735375e-8, 
     -3.234062073549239e-8, -3.1052987649993674e-8, 
     -2.9830527257571426e-8, -2.8669252709023574e-8, 
     -2.7565462216901613e-8, -2.651571579372435e-8, 
     -2.5516814118347675e-8, -2.4565779315203507e-8, 
     -2.365983745493836e-8, -2.2796402605943872e-8, 
     -2.1973062284745697e-8, -2.1187564169521625e-8, 
     -2.0437803955428343e-8, -1.972181424316656e-8, -1.4277497295949218e-6};
 
  TVIL_REAL H00yxyxtableC[] = 
     {0, -1.5795867438169813, -0.018418253829740383, 
     0.009758330737013488, 0.006650200100248638, 0.0038658156894913476, 
     0.002180382876794451, 0.001171479357063826, 0.0005538130225843374, 
     0.00016722332207315393, -0.00007861199362786675, 
     -0.00023620162809684375, -0.00033707796159149206, 
     -0.00040074984809852776, -0.0004396098926429809, 
     -0.00046171353339917454, -0.00047240236370479916, 
     -0.00047527961637904015, -0.00047281159331771794, 
     -0.00046670715967191134, -0.0004581622476124911, 
     -0.00044802037647224263, -0.0004368798374117591, -0.0004251663663479219, 
     -0.00041318310124429866, -0.00040114535349422553, 
     -0.00038920508188943284, -0.0003774682927160239, -0.0003660075223262463, 
     -0.00035487086384511715, -0.00034408854094841734, -0.000333677724694368, 
     -0.00032364608144470203, -0.0003139943974128957, 
     -0.00030471852667960167, -0.0002958108404716511, 
     -0.00028726130675093067, -0.00027905829443855903, 
     -0.0002711891716698639, -0.00026364074944045315, -0.000256399608863183, 
     -0.0002494523406193355, -0.0002427857180763663, -0.00023638682026741985, 
     -0.00023024311699084907, -0.00022434252533628553, 
     -0.00021867344472063614, -0.0002132247758362734, -0.0002079859276374, 
     -0.00020294681551826128, -0.00019809785309412783, 
     -0.00019342993942687847, -0.00018893444309991785, 
     -0.00018460318421103378, -0.00018042841509277076, 
     -0.0001764028003701356, -0.00017251939681128467, 
     -0.00016877163330799964, -0.0001651532912310726, 
     -0.00016165848533516037, -0.0001582816453335155, -0.0001550174982215078, 
     -0.00015186105139637727, -0.00014880757659667398, 
     -0.00014585259466705879, -0.00014299186114081174, 
     -0.00014022135262278714, -0.0001375372539487059, 
     -0.00013493594609188114, -0.00013241399478557646, 
     -0.0001299681398272091, -0.0001275952850299314, -0.00012529248878692667, 
     -0.00012305695521416804, -0.000120886025838228, -0.0001187771717967882, 
     -0.00011672798652072585, -0.00011473617886810158, 
     -0.00011279956668178246, -0.00011091607074391243, 
     -0.00010908370910196752, -0.00010730059174255367, 
     -0.0001055649155905173, -0.00010387495981239222, 
     -0.00010222908140442022, -0.00010062571104670678, 
     -0.00009906334920627754, -0.00009754056247286268, 
     -0.00009605598011237435, -0.000094608290824005, -0.00009319623968782453, 
     -0.00009181862529062963, -0.0000904742970186485, 
     -0.00008916215250640708, -0.00008788113523190482, 
     -0.00008663023224874354, -0.00008540847204667771, 
     -0.00008421492253245103, -0.00008304868912346539, 
     -0.00008190891294724417, -0.00008601659993178737};
 
  TVIL_REAL H00yxyxtableCL[] = 
     {0, -1.4112335167120564, -0.09193738538829793, 
     -0.014849784929117482, 0.0004300047643115146, 0.005388765603540138, 
     0.007101268070572209, 0.00752378697477582, 0.007400526992062867, 
     0.007051437008622784, 0.006621219527949362, 0.00617641292364822, 
     0.005747122199022479, 0.00534600718336509, 0.004977277392315901, 
     0.004641083397975716, 0.0043357061516381655, 0.004058659894496157, 
     0.003807245748767625, 0.0035788226049503474, 0.003370931114042559, 
     0.0031813414133182397, 0.003008061880454308, 0.0028493287846982837, 
     0.0027035874300549677, 0.0025694703840264815, 0.0024457756657599704, 
     0.002331446285496858, 0.002225551724640581, 0.0021272715189835393, 
     0.002035880889315952, 0.001950738260546825, 0.0018712744702783961, 
     0.0017969834610299835, 0.0017274142604457353, 0.0016621640716132247, 
     0.0016008723160165954, 0.001543215491981424, 0.0014889027304221718, 
     0.0014376719467156778, 0.00138928650244677, 0.0013435323036705927, 
     0.0013002152733796474, 0.001259159145259908, 0.0012202035337844502, 
     0.0011832022424269977, 0.0011480217774652913, 0.0011145400396456026, 
     0.001082645170034622, 0.0010522345298112842, 0.0010232137966498076, 
     0.0009954961628003836, 0.0009690016220567422, 0.0009436563345695398, 
     0.0009193920599708211, 0.0008961456505592342, 0.0008738585973930135, 
     0.0008524766230768952, 0.0008319493158345632, 0.0008122298001501768, 
     0.0007932744398583104, 0.0007750425700753451, 0.0007574962548093714, 
     0.0007406000674700193, 0.0007243208918329977, 0.0007086277413038133, 
     0.0006934915945773028, 0.0006788852460094914, 0.000664783169210385, 
     0.0006511613925343912, 0.0006379973852923672, 0.0006252699536386571, 
     0.0006129591452002045, 0.0006010461616150001, 0.0005895132782355176, 
     0.0005783437703308098, 0.0005675218451900623, 0.0005570325795915688, 
     0.0005468618621554163, 0.0005369963401464178, 0.0005274233703367816, 
     0.0005181309735762334, 0.0005091077927514823, 0.0005003430538473776, 
     0.0004918265298493887, 0.00048354850725146597, 0.00047549975495523905, 
     0.00046767149536616173, 0.0004600553775098914, 0.00045264345200807504, 
     0.000445428147767055, 0.0004384022502458972, 0.0004315588811818242, 
     0.00042489147966165083, 0.00041839378443737766, 0.0004120598173926915, 
     0.0004058838680749803, 0.00039986047921454883, 0.0003939844331591715, 
     0.0003882507391579776, 0.0003903253785420243};
 
  TVIL_REAL H00yxyxtableCL2[] = 
     {0, -0.125, 0.0234375, -0.008680555555555556, 
     -0.018513997395833332, -0.019306640625, -0.017647298177083333, 
     -0.015583915450946003, -0.013676367487226214, -0.012039869634772944, 
     -0.010664665396251376, -0.009511881348937573, -0.008541330480486685, 
     -0.007718572087732053, -0.007015761619988097, -0.006410815335304361, 
     -0.005886286353644617, -0.005428352284129359, -0.005025998041558661, 
     -0.004670382479609359, -0.004354354812155635, -0.0040720867245976145, 
     -0.0038187919388523966, -0.0035905114035325154, -0.003383947712155125, 
     -0.0031963365722494363, -0.003025346308444636, -0.002868998711781585, 
     -0.0027256062544083123, -0.0025937219394674012, -0.002472098974964268, 
     -0.0023596581388208105, -0.002255461205984571, -0.0021586891847225724, 
     -0.0020686243921376004, -0.0019846356130654, -0.001906165749633353, 
     -0.0018327214938346568, -0.001763864651977448, -0.001699204824788186, 
     -0.0016383932054568559, -0.0015811173038658508, -0.0015270964415384659, 
     -0.0014760778906605994, -0.0014278335535282847, -0.0013821570972197975, 
     -0.0013388614731560178, -0.0012977767632466838, -0.00125874830410524, 
     -0.0012216350488052235, -0.0011863081322024416, -0.0011526496112398256, 
     -0.0011205513561076065, -0.0010899140718265271, -0.0010606464328969066, 
     -0.001032664316224185, -0.0010058901196828643, -0.0009802521554887725, 
     -0.0009556841090737805, -0.0009321245554458097, -0.0009095165261097215, 
     -0.0008878071205538189, -0.0008669471570988466, -0.0008468908585834803, 
     -0.0008275955689405372, -0.0008090214972165607, -0.0007911314860166122, 
     -0.0007738908017264716, -0.0007572669441848179, -0.0007412294737556408, 
     -0.000725749853992344, -0.0007108013082949492, -0.0006963586891449007, 
     -0.0006823983586619689, -0.000668898079367805, -0.0006558369141635426, 
     -0.000643195134636767, -0.0006309541369081476, -0.0006190963643117832, 
     -0.0006076052362772336, -0.000596465082846623, -0.0005856610843181224, 
     -0.0005751792155585142, -0.0005650061945732083, -0.0005551294349627002, 
     -0.000545537001930667, -0.0005362175715411978, -0.0005271603929515074, 
     -0.0005183552533723077, -0.0005097924455311276, -0.0005014627374346227, 
     -0.0004933573442445345, -0.0004854679020987112, -0.00047778644372366866, 
     -0.0004703053756987606, -0.00046301745724427524, -0.0004559157804168495, 
     -0.0004489937516055932, -0.00044224507423137654, -0.0004356637325599396, 
     -0.0004320614004973855};
 
  TVIL_REAL H00yxyxtableCL3[] = 
     {0, 0., 0.03125, 0.0234375, 0.0166015625, 0.01220703125, 
     0.009379069010416666, 0.007476806640625, 0.0061359405517578125, 
     0.0051517486572265625, 0.004404878616333008, 0.0038224458694458008, 
     0.0033579195539156594, 0.0029804036021232605, 0.002668678095298154, 
     0.002407740770528714, 0.002186717712902464, 0.0019975552859250456, 
     0.0018341749936452187, 0.001691912113528815, 0.0015671335952220033, 
     0.0014569722767527353, 0.001359138394536527, 0.001271783563886686, 
     0.00119340106993208, 0.0011227517265921527, 0.0010588080261871387, 
     0.001000711563844912, 0.0009477402246873057, 0.000899282638882842, 
     0.0008548181084047459, 0.0008139006963479111, 0.0007761465136559913, 
     0.0007412234841159147, 0.0007088430464447609, 0.0006787533824324118, 
     0.0006507338562286047, 0.0006245904215371707, 0.0006001518073980777, 
     0.0005772663341376646, 0.0005557992423369202, 0.0005356304417524156, 
     0.0005166526058059723, 0.0004987695518462199, 0.00048189485884729045, 
     0.0004659506832706147, 0.0004508667410191794, 0.000436579429171582, 
     0.0004230310658097035, 0.0004101692299895834, 0.00039794618693589384, 
     0.0003863183860104313, 0.0003752460210267744, 0.00036469264414492486, 
     0.00035462482595093774, 0.00034501185546235037, 0.00033582547474452314, 
     0.00032703964361085596, 0.0003186303305394116, 0.00031057532649245146, 
     0.000302854078792153, 0.000295447542600246, 0.0002883380478836688, 
     0.00028150918003255645, 0.00027494567253912163, 0.00026863331035301635, 
     0.00026255884270614486, 0.00025670990435225436, 0.00025107494429780933, 
     0.00024564316121386174, 0.0002404044448165457, 0.00023534932258870327, 
     0.00023046891128887965, 0.00022575487275810693, 0.00022119937359087337, 
     0.00021679504828559732, 0.00021253496553275537, 0.0002084125973363872, 
     0.00020442179069771375, 0.0002005567416186671, 0.00019681197120875602, 
     0.00019318230370132434, 0.0001896628462052785, 0.00018624897003610233, 
     0.00018293629348571533, 0.00017972066590472074, 0.0001765981529830392, 
     0.0001735650231260178, 0.00017061773483300754, 0.00016775292499425108, 
     0.00016496739802984036, 0.00016225811580160003, 0.0001596221882351161, 
     0.0001570568645948477, 0.00015455952536040044, 0.00015212767465666894, 
     0.0001497589331947297, 0.00014745103168412965, 0.0001452018046806181, 
     0.0001430091848364459, 0.00014087119752314106};
 
  TVIL_REAL H00yxyxtableB[] = 
     {0, -5.0628893982817065, 2.047382391789873, -1.1672770690327166, 
     0.772255800105581, -0.5560106642317105, 0.422943722005418, 
     -0.3344389283970494, 0.27220196060490776, -0.2265614535962275, 
     0.19197570946928966, -0.16506657253788587, 0.14367099553650503, 
     -0.12634738119636682, 0.11210243516911511, -0.10023223091851192, 
     0.09022574480502424, -0.08170416516629384, 0.07438150688205568, 
     -0.06803835230488098, 0.06250392269519765, -0.05764357655678174, 
     0.05334992643815072, -0.04953641910391005, 0.04613262441662885, 
     -0.04308072973108008, 0.04033289801304472, -0.03784925358608762, 
     0.03559632988468433, -0.03354586137338588, 0.031673834686541384, 
     -0.02995973701153302, 0.028385955987024904, -0.02693729702286894, 
     0.025600592374667525, -0.024364382473258565, 0.02321865456803054, 
     -0.022154627143935946, 0.021164571131532365, -0.02024166087140699, 
     0.01937984927931747, -0.01857376280219966, 0.017818612642342, 
     -0.01711011941962562, 0.0164444489858171, -0.01581815753485574, 
     0.015228144494758036, -0.014671611959751725, 0.014146029640477948, 
     -0.013649104487013425, 0.013178754282899536, -0.012733084625178342, 
     0.012310368800975639, -0.011909030149633903, 0.01152762656408979, 
     -0.011164836838731098, 0.01081944861544039, -0.010490347716601843, 
     0.0101765086848512, -0.009876986375362617, 0.009590908468362032, 
     -0.009317468788042203, 0.009055921329706233, -0.008805574910256138, 
     0.00856578836845793, -0.008335966251073511, 0.008115554929215956, 
     -0.007904039096376617, 0.0077009386056712615, -0.007505805609109486, 
     0.007318221966233182, -0.007137796893401787, 0.006964164828413531, 
     -0.0067969834881177445, 0.006635932099256966, -0.006480709785032618, 
     0.006331034091859974, -0.006186639642505641, 0.0060472769033169185, 
     -0.005912711054585209, 0.005782720954259468, -0.005657098186260945, 
     0.005535646185565204, -0.005418179433026761, 0.0053045227136389655, 
     -0.005194510432558343, 0.005087985983788426, -0.004984801166921723, 
     0.0048848156477873175, -0.004787896459252228, 0.004693917538782634, 
     -0.004602759299691443, 0.004514308233285714, -0.004428456539384857, 
     0.004345101782911832, -0.0042641465744674405, 0.004185498272984997, 
     -0.004109068708731341, 0.004034773925072337, -0.003962533937558449, 
     0.003892272509010289};
 
  TVIL_REAL H00yxyxtableD[] = 
     {0, -3.5763898990065126, 1.2274618429079514, 
     -0.6340363571501366, 0.3921105816691218, -0.2686028418892603, 
     0.1965681572973041, -0.1506651404515779, 0.11950340571515362, 
     -0.09731903267077452, 0.0809296391355988, -0.06845643375456997, 
     0.05872964975859569, -0.050988753484022216, 0.044721181103483644, 
     -0.0395708473007439, 0.03528386813419134, -0.03167511369705323, 
     0.028606910226645223, -0.025975085708470087, 0.023699594477764326, 
     -0.021718076488503886, 0.019981343838727014, -0.018450160884602357, 
     0.017092909842187854, -0.015883873393459307, 0.01480195422462823, 
     -0.013829708581359575, 0.012952608580543446, -0.012158473258279574, 
     0.011437025523104386, -0.010779544063061183, 0.010178587577407859, 
     -0.009627774607790565, 0.009121606481659317, -0.008655323956217471, 
     0.008224790406107402, -0.007826396067246858, 0.007456979096075285, 
     -0.007113760142763737, 0.0067942878502373115, -0.006496393236645102, 
     0.006218151339527252, -0.0059578488262846795, 0.005713956530412623, 
     -0.005485106073199955, 0.0052700698888442, -0.0050677440966888235, 
     0.00487713376475549, -0.004697340189411833, 0.004527549881099592, 
     -0.004367024998802188, 0.004215095018880194, -0.00407114945901863, 
     0.003934631506857297, -0.003805032426633137, 0.0036818866368170107, 
     -0.0035647673680440244, 0.0034532828242303856, -0.003347072781132945, 
     0.003245805566136853, -0.003149175371072986, 0.003056899856630486, 
     -0.0029687180126531697, 0.002884388243465315, -0.002803686651504797, 
     0.0027264054960669864, -0.0026523518069780346, 0.002581346135601335, 
     -0.002513221427802692, 0.002447822005413555, -0.0023850026443839175, 
     0.00232462773924606, -0.0022665705447497773, 0.002210712486606601, 
     -0.0021569425342180043, 0.002105156629080245, -0.002055257163273058, 
     0.0020071525030649754, -0.001960756553216672, 0.0019159883580457666, 
     -0.0018727717357406703, 0.0018310349427849801, -0.0017907103656840573, 
     0.0017517342374773215, -0.0017140463767783408, 0.0016775899473140856, 
     -0.0016423112361383895, 0.001608159448875818, -0.0015750865205135132, 
     0.001543046940402494, -0.00151199759025844, 0.0014818975940669194, 
     -0.0014527081789009364, 0.0014243925457509416, -0.0013969157495502514, 
     0.0013702445876532339, -0.0013443474960905726, 0.0013191944529861999, 
     -0.0012947568885748732, 0.0012710076013084212};
 
  TVIL_REAL H00yxyxtableE[] = 
     {0, 2.115819489896223, -1.5261975767837797, 1.211112021756433, 
     -1.0109381608448256, 0.8709855746324195, -0.7669526366543788, 
     0.6862411600816268, -0.6216106387528415, 0.5685791112946103, 
     -0.5242119289812345, 0.4865010344690192, -0.45402270325719357, 
     0.42573750451792086, -0.4008676899069004, 0.3788190047198143, 
     -0.3591291273973895, 0.3414326767348997, -0.32543686305383734, 
     0.3109041710931282, -0.2976398035479569, 0.2854824184432958, 
     -0.2742971900632005, 0.26397053774846296, -0.2544060708689525, 
     0.2455214333480462, -0.23724582226102467, 0.22951801760188661, 
     -0.22228480395283515, 0.21549969566984917, -0.20912189934085199, 
     0.20311546334847613, -0.19744857617409292, 0.19209298384089216, 
     -0.18702350346083407, 0.18221761481853038, -0.17765511571654502, 
     0.1733178297236498, -0.16918935722914114, 0.1652548624724195, 
     -0.16150089060563927, 0.1579152099460984, -0.15448667544994207, 
     0.15120511013938798, -0.14806120177985435, 0.1450464125600454, 
     -0.14215289989954977, 0.13937344681216424, -0.13670140050248106, 
     0.13413061807888454, -0.13165541843636103, 0.12927053950407266, 
     -0.12697110017077093, 0.12475256630006519, -0.12261072033072223, 
     0.12054163402730676, -0.11854164400581119, 0.11660732970928565, 
     -0.11473549355135162, 0.11292314298208292, -0.11116747426207368, 
     0.10946585775741124, -0.10781582459142199, 0.1062150545090334, 
     -0.10466136482687145, 0.1031527003571897, -0.10168712420674156, 
     0.1002628093630393, -0.09887803099033442, 0.09753115936630244, 
     -0.09622065339799428, 0.09494505466226942, -0.0937029819217792, 
     0.0924931260727256, -0.09131424548517401, 0.09016516170072492, 
     -0.08904475545591516, 0.08795196300288516, -0.08688577270165938, 
     0.08584522186089084, -0.08482939380615202, 0.08383741515684526, 
     -0.08286845329458771, 0.0819217140075187, -0.08099643929640762, 
     0.08009190532972232, -0.07920742053597168, 0.07834232382267359, 
     -0.0774959829122341, 0.07666779278586805, -0.0758571742274524, 
     0.07506357245989317, -0.07428645586721043, 0.07352531479611095, 
     -0.07277966043133127, 0.07204902373949955, -0.07133295447668797, 
     0.07063102025521337, -0.06994280566559412, 0.06926791144989229, 
     -0.06860595372296145};

  lnyox = TVIL_CLOG(y/x);
  lnyox2 = lnyox  * lnyox;
  lnyox3 = lnyox2  * lnyox;

  if (y < 0.1545 * x) {
    result = TVIL_H0000xx (x,qq);
    rn = r = 4.L * y/x;
    for (n=1; n < 100; n++) {
      result += rn * (H00yxyxtableA[n] + 
                      H00yxyxtableAL[n] * lnyox + 
                      H00yxyxtableAL2[n] * lnyox2);
      rn *= r;
    }
  }
  else if (y < 0.5 * x) {
    result = -1.9978533782973847021L - 6.L * Zeta3 * TVIL_Ap(x,qq);
    rn = r = 0.25L * x/y - 1.L;
    for (n=1; n < 100; n++) {
      result += rn * H00yxyxtableE[n];
      rn *= r;
    }
  }
  else if (y < 1.6 * x) {
    result = TVIL_H00xxxx (x,qq);
    rn = r = y/x - 1.L;
    for (n=1; n < 100; n++) {
      result += rn * H00yxyxtableD[n];
      rn *= r;
    }
  }
  else if (y < 6.47 * x) {
    result = -11.926489899965666927L - 6.L * Zeta3 * TVIL_Ap(x,qq);
    rn = r = y/(4.L*x) - 1.L;
    for (n=1; n < 64; n++) {
      result += rn * H00yxyxtableB[n];
      rn *= r;
    }
  }
  else {
    result = TVIL_H0000xx (y, qq);
    rn = r = 4.L * x/y;
    for (n=1; n < 100; n++) {
      result += rn * (H00yxyxtableC[n] 
                     +H00yxyxtableCL[n] * lnyox 
                     +H00yxyxtableCL2[n] * lnyox2
                     +H00yxyxtableCL3[n] * lnyox3);
      rn *= r;
    }
  }

  return result;
}

/* ****************************************************************** */

int TVIL_AnalyticHTYPE (TVIL_HTYPE *foo, TVIL_REAL qq)
{
  int retval;

  retval = TVIL_Hanalytic (foo->arg[0],
			   foo->arg[1],
			   foo->arg[2],
			   foo->arg[3],
			   foo->arg[4],
			   foo->arg[5],
			   qq,
			   &(foo->value));

  return retval;
}

/* ****************************************************************** */

int TVIL_Hanalytic (TVIL_REAL a1,
		    TVIL_REAL a2,
		    TVIL_REAL a3,
		    TVIL_REAL a4,
		    TVIL_REAL a5,
		    TVIL_REAL a6,
		    TVIL_REAL QQ,
		    TVIL_COMPLEX *result)
{
  int success = 1;

  if (SAME7(0,a1,a2,a3,a4,a5,a6))
    *result = TVIL_ComplexInfinity;

  else if (SAME6(0,a2,a3,a4,a5,a6))
    *result = TVIL_H00000x (a1, QQ);
  else if (SAME6(0,a1,a3,a4,a5,a6))
    *result = TVIL_H00000x (a2, QQ);
  else if (SAME6(0,a1,a2,a4,a5,a6))
    *result = TVIL_H00000x (a3, QQ);
  else if (SAME6(0,a1,a2,a3,a5,a6))
    *result = TVIL_H00000x (a4, QQ);
  else if (SAME6(0,a1,a2,a3,a4,a6))
    *result = TVIL_H00000x (a5, QQ);
  else if (SAME6(0,a1,a2,a3,a4,a5))
    *result = TVIL_H00000x (a6, QQ);
	
  else if (SAME5(0,a1,a2,a3,a4) && SAME2(a5,a6))
    *result = TVIL_H0000xx (a6, QQ);
  else if (SAME5(0,a1,a2,a3,a5) && SAME2(a4,a6))
    *result = TVIL_H0000xx (a6, QQ);
  else if (SAME5(0,a1,a2,a4,a5) && SAME2(a3,a6))
    *result = TVIL_H0000xx (a6, QQ);
  else if (SAME5(0,a2,a3,a4,a5) && SAME2(a1,a6))
    *result = TVIL_H0000xx (a6, QQ);
  else if (SAME5(0,a1,a2,a3,a6) && SAME2(a4,a5))
    *result = TVIL_H0000xx (a5, QQ);
  else if (SAME5(0,a1,a2,a4,a6) && SAME2(a3,a5))
    *result = TVIL_H0000xx (a5, QQ);
  else if (SAME5(0,a1,a3,a4,a6) && SAME2(a2,a5))
    *result = TVIL_H0000xx (a5, QQ);
  else if (SAME5(0,a1,a3,a5,a6) && SAME2(a2,a4))
    *result = TVIL_H0000xx (a4, QQ);
  else if (SAME5(0,a2,a3,a5,a6) && SAME2(a1,a4))
    *result = TVIL_H0000xx (a4, QQ);
  else if (SAME5(0,a1,a4,a5,a6) && SAME2(a2,a3)) 
    *result = TVIL_H0000xx (a3, QQ);
  else if (SAME5(0,a2,a4,a5,a6) && SAME2(a1,a3))
    *result = TVIL_H0000xx (a3, QQ);
  else if (SAME5(0,a3,a4,a5,a6) && SAME2(a1,a2))
    *result = TVIL_H0000xx (a2, QQ);

  else if (SAME5(0,a1,a3,a4,a5) && SAME2(a2,a6))
    *result = TVIL_H00xx00 (a6, QQ);
  else if (SAME5(0,a2,a3,a4,a6) && SAME2(a1,a5))
    *result = TVIL_H00xx00 (a5, QQ);
  else if (SAME5(0,a1,a2,a5,a6) && SAME2(a3,a4))
    *result = TVIL_H00xx00 (a4, QQ);

  else if (SAME4(0,a1,a2,a3) && SAME3(a4,a5,a6))
    *result = TVIL_H000xxx (a6, QQ);
  else if (SAME4(0,a2,a4,a5) && SAME3(a1,a3,a6))
    *result = TVIL_H000xxx (a6, QQ);
  else if (SAME4(0,a1,a4,a6) && SAME3(a2,a3,a5))
    *result = TVIL_H000xxx (a5, QQ);
  else if (SAME4(0,a3,a5,a6) && SAME3(a1,a2,a4))
    *result = TVIL_H000xxx (a4, QQ);

  else if (SAME4(0,a1,a2,a4) && SAME3(a3,a5,a6))
    *result = TVIL_H00x0xx (a6, QQ);
  else if (SAME4(0,a2,a3,a5) && SAME3(a1,a4,a6))
    *result = TVIL_H00x0xx (a6, QQ);
  else if (SAME4(0,a1,a3,a6) && SAME3(a2,a4,a5))
    *result = TVIL_H00x0xx (a5, QQ);
  else if (SAME4(0,a4,a5,a6) && SAME3(a1,a2,a3))
    *result = TVIL_H00x0xx (a3, QQ);

  else if (SAME4(0,a1,a2,a5) && SAME3(a3,a4,a6))
    *result = TVIL_H00xx0x (a6, QQ);
  else if (SAME4(0,a1,a3,a5) && SAME3(a2,a4,a6)) 
    *result = TVIL_H00xx0x (a6, QQ);
  else if (SAME4(0,a1,a3,a4) && SAME3(a2,a5,a6))
    *result = TVIL_H00xx0x (a6, QQ);
  else if (SAME4(0,a2,a3,a4) && SAME3(a1,a5,a6))
    *result = TVIL_H00xx0x (a6, QQ);
  else if (SAME4(0,a3,a4,a5) && SAME3(a1,a2,a6))
    *result = TVIL_H00xx0x (a6, QQ);
  else if (SAME4(0,a1,a4,a5) && SAME3(a2,a3,a6))
    *result = TVIL_H00xx0x (a6, QQ);
  else if (SAME4(0,a2,a3,a6) && SAME3(a1,a4,a5))
    *result = TVIL_H00xx0x (a5, QQ);
  else if (SAME4(0,a1,a2,a6) && SAME3(a3,a4,a5))
    *result = TVIL_H00xx0x (a5, QQ);
  else if (SAME4(0,a2,a4,a6) && SAME3(a1,a3,a5))
    *result = TVIL_H00xx0x (a5, QQ);
  else if (SAME4(0,a3,a4,a6) && SAME3(a1,a2,a5))
    *result = TVIL_H00xx0x (a5, QQ);
  else if (SAME4(0,a1,a5,a6) && SAME3(a2,a3,a4))
    *result = TVIL_H00xx0x (a4, QQ);
  else if (SAME4(0,a2,a5,a6) && SAME3(a1,a3,a4))
    *result = TVIL_H00xx0x (a4, QQ);

  else if (SAME3(0,a1,a2) && SAME4(a3,a4,a5,a6))
    *result = TVIL_H00xxxx (a6, QQ);
  else if (SAME3(0,a1,a3) && SAME4(a2,a4,a5,a6))
    *result = TVIL_H00xxxx (a6, QQ);
  else if (SAME3(0,a1,a4) && SAME4(a2,a3,a5,a6))
    *result = TVIL_H00xxxx (a6, QQ);
  else if (SAME3(0,a2,a3) && SAME4(a1,a4,a5,a6))
    *result = TVIL_H00xxxx (a6, QQ);
  else if (SAME3(0,a2,a4) && SAME4(a1,a3,a5,a6))
    *result = TVIL_H00xxxx (a6, QQ);
  else if (SAME3(0,a2,a5) && SAME4(a1,a3,a4,a6))
    *result = TVIL_H00xxxx (a6, QQ);
  else if (SAME3(0,a3,a5) && SAME4(a1,a2,a4,a6))
    *result = TVIL_H00xxxx (a6, QQ);
  else if (SAME3(0,a4,a5) && SAME4(a1,a2,a3,a6))
    *result = TVIL_H00xxxx (a6, QQ);
  else if (SAME3(0,a1,a6) && SAME4(a2,a3,a4,a5))
    *result = TVIL_H00xxxx (a5, QQ);
  else if (SAME3(0,a3,a6) && SAME4(a1,a2,a4,a5))
    *result = TVIL_H00xxxx (a5, QQ);
  else if (SAME3(0,a4,a6) && SAME4(a1,a2,a3,a5))
    *result = TVIL_H00xxxx (a5, QQ);
  else if (SAME3(0,a5,a6) && SAME4(a1,a2,a3,a4))
    *result = TVIL_H00xxxx (a4, QQ);

  else if (SAME3(0,a1,a5) && SAME4(a2,a3,a4,a6))
    *result = TVIL_H0xxx0x (a6, QQ);
  else if (SAME3(0,a3,a4) && SAME4(a1,a2,a5,a6))
    *result = TVIL_H0xxx0x (a6, QQ);
  else if (SAME3(0,a2,a6) && SAME4(a1,a3,a4,a5))
    *result = TVIL_H0xxx0x (a5, QQ);

  else if (SAME2(0,a1) && SAME5(a2,a3,a4,a5,a6))
    *result = TVIL_H0xxxxx (a6, QQ);
  else if (SAME2(0,a2) && SAME5(a1,a3,a4,a5,a6))
    *result = TVIL_H0xxxxx (a6, QQ);
  else if (SAME2(0,a3) && SAME5(a1,a2,a4,a5,a6))
    *result = TVIL_H0xxxxx (a6, QQ);
  else if (SAME2(0,a4) && SAME5(a1,a2,a3,a5,a6))
    *result = TVIL_H0xxxxx (a6, QQ);
  else if (SAME2(0,a5) && SAME5(a1,a2,a3,a4,a6))
    *result = TVIL_H0xxxxx (a6, QQ);
  else if (SAME2(0,a6) && SAME5(a1,a2,a3,a4,a5))
    *result = TVIL_H0xxxxx (a5, QQ);

  else if (SAME6(a1,a2,a3,a4,a5,a6))
    *result = TVIL_Hxxxxxx (a6, QQ);

  else if (SAME5(0,a1,a2,a3,a4) && (a5>0) && (a6>0))
    *result = TVIL_H0000xy (a5, a6, QQ);

  else if (SAME5(0,a1,a2,a3,a5) && (a4>0) && (a6>0))
    *result = TVIL_H0000xy (a4, a6, QQ);
  else if (SAME5(0,a1,a2,a4,a5) && (a3>0) && (a6>0))
    *result = TVIL_H0000xy (a3, a6, QQ);
  else if (SAME5(0,a2,a3,a4,a5) && (a1>0) && (a6>0))
    *result = TVIL_H0000xy (a1, a6, QQ);
  else if (SAME5(0,a1,a2,a3,a6) && (a4>0) && (a5>0))
    *result = TVIL_H0000xy (a4, a5, QQ);
  else if (SAME5(0,a1,a2,a4,a6) && (a3>0) && (a5>0))
    *result = TVIL_H0000xy (a3, a5, QQ);
  else if (SAME5(0,a1,a3,a4,a6) && (a2>0) && (a5>0))
    *result = TVIL_H0000xy (a2, a5, QQ);
  else if (SAME5(0,a1,a3,a5,a6) && (a2>0) && (a4>0))
    *result = TVIL_H0000xy (a2, a4, QQ);
  else if (SAME5(0,a2,a3,a5,a6) && (a1>0) && (a4>0))
    *result = TVIL_H0000xy (a1, a4, QQ);
  else if (SAME5(0,a1,a4,a5,a6) && (a2>0) && (a3>0)) 
    *result = TVIL_H0000xy (a2, a3, QQ);
  else if (SAME5(0,a2,a4,a5,a6) && (a1>0) && (a3>0))
    *result = TVIL_H0000xy (a1, a3, QQ);
  else if (SAME5(0,a3,a4,a5,a6) && (a1>0) && (a2>0))
    *result = TVIL_H0000xy (a1, a2, QQ);

  else if (SAME5(0,a1,a3,a4,a5) && (a2>0) && (a6>0))
    *result = TVIL_H00xy00 (a2, a6, QQ);
  else if (SAME5(0,a2,a3,a4,a6) && (a1>0) && (a5>0))
    *result = TVIL_H00xy00 (a1, a5, QQ);
  else if (SAME5(0,a1,a2,a5,a6) && (a3>0) && (a4>0))
    *result = TVIL_H00xy00 (a3, a4, QQ);

  else if (SAME3(0,a1,a5) && SAME2(a2,a3) && SAME2(a4,a6) && (a2>0) && (a4>0))
    *result = TVIL_Hxx00yy (a2, a4, QQ);
  else if (SAME3(0,a1,a5) && SAME2(a2,a4) && SAME2(a3,a6) && (a2>0) && (a3>0))
    *result = TVIL_Hxx00yy (a2, a3, QQ);
  else if (SAME3(0,a2,a6) && SAME2(a1,a4) && SAME2(a3,a5) && (a1>0) && (a3>0))
    *result = TVIL_Hxx00yy (a1, a3, QQ);
  else if (SAME3(0,a2,a6) && SAME2(a1,a3) && SAME2(a4,a5) && (a1>0) && (a4>0))
    *result = TVIL_Hxx00yy (a1, a4, QQ);
  else if (SAME3(0,a3,a4) && SAME2(a1,a2) && SAME2(a5,a6) && (a1>0) && (a5>0))
    *result = TVIL_Hxx00yy (a1, a5, QQ);
  else if (SAME3(0,a3,a4) && SAME2(a1,a6) && SAME2(a2,a5) && (a1>0) && (a2>0))
    *result = TVIL_Hxx00yy (a1, a2, QQ);

  else if (SAME4(a1,a2,a5,a6) && SAME2(0,a3) && (a1>0) && (a4>0))
    *result = TVIL_Hxx0yxx (a1, a4, QQ);
  else if (SAME4(a1,a2,a5,a6) && SAME2(0,a4) && (a1>0) && (a3>0))
    *result = TVIL_Hxx0yxx (a1, a3, QQ);
  else if (SAME4(a1,a3,a4,a5) && SAME2(0,a2) && (a1>0) && (a6>0))
    *result = TVIL_Hxx0yxx (a1, a6, QQ);
  else if (SAME4(a1,a3,a4,a5) && SAME2(0,a6) && (a1>0) && (a2>0))
    *result = TVIL_Hxx0yxx (a1, a2, QQ);
  else if (SAME4(a2,a3,a4,a6) && SAME2(0,a1) && (a2>0) && (a5>0))
    *result = TVIL_Hxx0yxx (a2, a5, QQ);
  else if (SAME4(a2,a3,a4,a6) && SAME2(0,a5) && (a2>0) && (a1>0))
    *result = TVIL_Hxx0yxx (a2, a1, QQ);

  else if (SAME3(0,a1,a2) && SAME3(a3,a5,a6) && (a3>0) && (a4>0))
    *result = TVIL_H00xyxx (a3, a4, QQ);
  else if (SAME3(0,a1,a3) && SAME3(a2,a4,a5) && (a2>0) && (a6>0))
    *result = TVIL_H00xyxx (a2, a6, QQ);
  else if (SAME3(0,a1,a4) && SAME3(a3,a5,a6) && (a3>0) && (a2>0))
    *result = TVIL_H00xyxx (a3, a2, QQ);
  else if (SAME3(0,a1,a6) && SAME3(a2,a4,a5) && (a2>0) && (a3>0))
    *result = TVIL_H00xyxx (a2, a3, QQ);
  else if (SAME3(0,a2,a3) && SAME3(a1,a4,a6) && (a1>0) && (a5>0))
    *result = TVIL_H00xyxx (a1, a5, QQ);
  else if (SAME3(0,a2,a4) && SAME3(a3,a5,a6) && (a3>0) && (a1>0))
    *result = TVIL_H00xyxx (a3, a1, QQ);
  else if (SAME3(0,a2,a5) && SAME3(a1,a4,a6) && (a1>0) && (a3>0))
    *result = TVIL_H00xyxx (a1, a3, QQ);
  else if (SAME3(0,a3,a5) && SAME3(a1,a4,a6) && (a1>0) && (a2>0))
    *result = TVIL_H00xyxx (a1, a2, QQ);
  else if (SAME3(0,a3,a6) && SAME3(a2,a4,a5) && (a2>0) && (a1>0))
    *result = TVIL_H00xyxx (a2, a1, QQ);
  else if (SAME3(0,a4,a5) && SAME3(a1,a2,a3) && (a1>0) && (a6>0))
    *result = TVIL_H00xyxx (a1, a6, QQ);
  else if (SAME3(0,a4,a6) && SAME3(a1,a2,a3) && (a1>0) && (a5>0))
    *result = TVIL_H00xyxx (a1, a5, QQ);
  else if (SAME3(0,a5,a6) && SAME3(a1,a2,a3) && (a1>0) && (a4>0))
    *result = TVIL_H00xyxx (a1, a4, QQ);

  else if (SAME3(0,a1,a2) && SAME3(a3,a4,a5) && (a3>0) && (a6>0))
    *result = TVIL_H00xxxy (a3, a6, QQ);
  else if (SAME3(0,a1,a2) && SAME3(a3,a4,a6) && (a3>0) && (a5>0))
    *result = TVIL_H00xxxy (a3, a5, QQ);
  else if (SAME3(0,a1,a3) && SAME3(a2,a4,a6) && (a2>0) && (a5>0))
    *result = TVIL_H00xxxy (a2, a5, QQ);
  else if (SAME3(0,a1,a3) && SAME3(a2,a5,a6) && (a2>0) && (a4>0))
    *result = TVIL_H00xxxy (a2, a4, QQ);
  else if (SAME3(0,a1,a4) && SAME3(a2,a3,a6) && (a2>0) && (a5>0))
    *result = TVIL_H00xxxy (a2, a5, QQ);
  else if (SAME3(0,a1,a4) && SAME3(a2,a5,a6) && (a2>0) && (a3>0))
    *result = TVIL_H00xxxy (a2, a3, QQ);
  else if (SAME3(0,a1,a6) && SAME3(a2,a3,a4) && (a2>0) && (a5>0))
    *result = TVIL_H00xxxy (a2, a5, QQ);
  else if (SAME3(0,a1,a6) && SAME3(a3,a4,a5) && (a3>0) && (a2>0))
    *result = TVIL_H00xxxy (a3, a2, QQ);
  else if (SAME3(0,a2,a3) && SAME3(a1,a4,a5) && (a1>0) && (a6>0))
    *result = TVIL_H00xxxy (a1, a6, QQ);
  else if (SAME3(0,a2,a3) && SAME3(a1,a5,a6) && (a1>0) && (a4>0))
    *result = TVIL_H00xxxy (a1, a4, QQ);
  else if (SAME3(0,a2,a4) && SAME3(a1,a3,a5) && (a1>0) && (a6>0))
    *result = TVIL_H00xxxy (a1, a6, QQ);
  else if (SAME3(0,a2,a4) && SAME3(a1,a5,a6) && (a1>0) && (a3>0))
    *result = TVIL_H00xxxy (a1, a3, QQ);
  else if (SAME3(0,a2,a5) && SAME3(a1,a3,a4) && (a1>0) && (a6>0))
    *result = TVIL_H00xxxy (a1, a6, QQ);
  else if (SAME3(0,a2,a5) && SAME3(a3,a4,a6) && (a3>0) && (a1>0))
    *result = TVIL_H00xxxy (a3, a1, QQ);
  else if (SAME3(0,a3,a5) && SAME3(a1,a2,a6) && (a1>0) && (a4>0))
    *result = TVIL_H00xxxy (a1, a4, QQ);
  else if (SAME3(0,a3,a5) && SAME3(a2,a4,a6) && (a2>0) && (a1>0))
    *result = TVIL_H00xxxy (a2, a1, QQ);
  else if (SAME3(0,a3,a6) && SAME3(a1,a2,a5) && (a1>0) && (a4>0))
    *result = TVIL_H00xxxy (a1, a4, QQ);
  else if (SAME3(0,a3,a6) && SAME3(a1,a4,a5) && (a1>0) && (a2>0))
    *result = TVIL_H00xxxy (a1, a2, QQ);
  else if (SAME3(0,a4,a5) && SAME3(a1,a2,a6) && (a1>0) && (a3>0))
    *result = TVIL_H00xxxy (a1, a3, QQ);
  else if (SAME3(0,a4,a5) && SAME3(a2,a3,a6) && (a2>0) && (a1>0))
    *result = TVIL_H00xxxy (a2, a1, QQ);
  else if (SAME3(0,a4,a6) && SAME3(a1,a2,a5) && (a1>0) && (a3>0))
    *result = TVIL_H00xxxy (a1, a3, QQ);
  else if (SAME3(0,a4,a6) && SAME3(a1,a3,a5) && (a1>0) && (a2>0))
    *result = TVIL_H00xxxy (a1, a2, QQ);
  else if (SAME3(0,a5,a6) && SAME3(a1,a3,a4) && (a1>0) && (a2>0))
    *result = TVIL_H00xxxy (a1, a2, QQ);
  else if (SAME3(0,a5,a6) && SAME3(a2,a3,a4) && (a2>0) && (a1>0))
    *result = TVIL_H00xxxy (a2, a1, QQ);

  else if (SAME4(0,a1,a2,a4) && SAME2(a5,a6) && (a5>0) && (a3>0))
    *result = TVIL_H00y0xx (a5, a3, QQ);
  else if (SAME4(0,a1,a2,a4) && SAME2(a3,a6) && (a3>0) && (a5>0))
    *result = TVIL_H00y0xx (a3, a5, QQ);
  else if (SAME4(0,a1,a2,a4) && SAME2(a3,a5) && (a3>0) && (a6>0))
    *result = TVIL_H00y0xx (a3, a6, QQ);
  else if (SAME4(0,a4,a5,a6) && SAME2(a1,a2) && (a1>0) && (a3>0))
    *result = TVIL_H00y0xx (a1, a3, QQ);
  else if (SAME4(0,a4,a5,a6) && SAME2(a2,a3) && (a2>0) && (a1>0))
    *result = TVIL_H00y0xx (a2, a1, QQ);
  else if (SAME4(0,a4,a5,a6) && SAME2(a1,a3) && (a1>0) && (a2>0))
    *result = TVIL_H00y0xx (a1, a2, QQ);
  else if (SAME4(0,a1,a3,a6) && SAME2(a2,a5) && (a2>0) && (a4>0))
    *result = TVIL_H00y0xx (a2, a4, QQ);
  else if (SAME4(0,a1,a3,a6) && SAME2(a2,a4) && (a2>0) && (a5>0))
    *result = TVIL_H00y0xx (a2, a5, QQ);
  else if (SAME4(0,a1,a3,a6) && SAME2(a4,a5) && (a4>0) && (a2>0))
    *result = TVIL_H00y0xx (a4, a2, QQ);
  else if (SAME4(0,a2,a3,a5) && SAME2(a1,a6) && (a1>0) && (a4>0))
    *result = TVIL_H00y0xx (a1, a4, QQ);
  else if (SAME4(0,a2,a3,a5) && SAME2(a4,a6) && (a4>0) && (a1>0))
    *result = TVIL_H00y0xx (a4, a1, QQ);
  else if (SAME4(0,a2,a3,a5) && SAME2(a1,a4) && (a1>0) && (a6>0))
    *result = TVIL_H00y0xx (a1, a6, QQ);

  else if (SAME2(0,a1) && SAME3(a2,a4,a5) && SAME2(a3,a6) && (a2>0) && (a6>0))
    *result = TVIL_H0xyxxy (a2, a3, QQ);
  else if (SAME2(0,a1) && SAME3(a3,a5,a6) && SAME2(a2,a4) && (a3>0) && (a2>0))
    *result = TVIL_H0xyxxy (a3, a2, QQ);
  else if (SAME2(0,a2) && SAME3(a1,a4,a6) && SAME2(a3,a5) && (a1>0) && (a3>0))
    *result = TVIL_H0xyxxy (a1, a3, QQ);
  else if (SAME2(0,a2) && SAME3(a3,a5,a6) && SAME2(a1,a4) && (a3>0) && (a1>0))
    *result = TVIL_H0xyxxy (a3, a1, QQ);
  else if (SAME2(0,a3) && SAME3(a2,a4,a5) && SAME2(a1,a6) && (a2>0) && (a1>0))
    *result = TVIL_H0xyxxy (a2, a1, QQ);
  else if (SAME2(0,a3) && SAME3(a1,a4,a6) && SAME2(a2,a5) && (a1>0) && (a2>0))
    *result = TVIL_H0xyxxy (a1, a2, QQ);
  else if (SAME2(0,a4) && SAME3(a1,a2,a3) && SAME2(a5,a6) && (a1>0) && (a5>0))
    *result = TVIL_H0xyxxy (a1, a5, QQ);
  else if (SAME2(0,a4) && SAME3(a3,a5,a6) && SAME2(a1,a2) && (a3>0) && (a1>0))
    *result = TVIL_H0xyxxy (a3, a1, QQ);
  else if (SAME2(0,a5) && SAME3(a1,a2,a3) && SAME2(a4,a6) && (a1>0) && (a4>0))
    *result = TVIL_H0xyxxy (a1, a4, QQ);
  else if (SAME2(0,a5) && SAME3(a1,a4,a6) && SAME2(a2,a3) && (a1>0) && (a2>0))
    *result = TVIL_H0xyxxy (a1, a2, QQ);
  else if (SAME2(0,a6) && SAME3(a1,a2,a3) && SAME2(a4,a5) && (a1>0) && (a4>0))
    *result = TVIL_H0xyxxy (a1, a4, QQ);
  else if (SAME2(0,a6) && SAME3(a2,a4,a5) && SAME2(a1,a3) && (a2>0) && (a1>0))
    *result = TVIL_H0xyxxy (a2, a1, QQ);

  else if (SAME3(0,a1,a2) && SAME2(a3,a5) && SAME2(a4,a6) && (a3>0) && (a4>0))
    *result = TVIL_H00yxyx (a4, a3, QQ);
  else if (SAME3(0,a1,a2) && SAME2(a3,a6) && SAME2(a4,a5) && (a3>0) && (a4>0))
    *result = TVIL_H00yxyx (a4, a3, QQ);
  else if (SAME3(0,a1,a3) && SAME2(a2,a5) && SAME2(a4,a6) && (a2>0) && (a4>0))
    *result = TVIL_H00yxyx (a4, a2, QQ);
  else if (SAME3(0,a1,a3) && SAME2(a2,a4) && SAME2(a5,a6) && (a2>0) && (a5>0))
    *result = TVIL_H00yxyx (a5, a2, QQ);
  else if (SAME3(0,a1,a4) && SAME2(a3,a6) && SAME2(a2,a5) && (a3>0) && (a2>0))
    *result = TVIL_H00yxyx (a2, a3, QQ);
  else if (SAME3(0,a1,a4) && SAME2(a5,a6) && SAME2(a2,a3) && (a5>0) && (a2>0))
    *result = TVIL_H00yxyx (a2, a5, QQ);
  else if (SAME3(0,a1,a6) && SAME2(a4,a5) && SAME2(a2,a3) && (a4>0) && (a2>0))
    *result = TVIL_H00yxyx (a2, a4, QQ);
  else if (SAME3(0,a1,a6) && SAME2(a4,a2) && SAME2(a5,a3) && (a4>0) && (a5>0))
    *result = TVIL_H00yxyx (a5, a4, QQ);
  else if (SAME3(0,a2,a3) && SAME2(a1,a6) && SAME2(a4,a5) && (a1>0) && (a4>0))
    *result = TVIL_H00yxyx (a4, a1, QQ);
  else if (SAME3(0,a2,a3) && SAME2(a1,a4) && SAME2(a6,a5) && (a1>0) && (a6>0))
    *result = TVIL_H00yxyx (a6, a1, QQ);
  else if (SAME3(0,a2,a4) && SAME2(a5,a6) && SAME2(a1,a3) && (a5>0) && (a1>0))
    *result = TVIL_H00yxyx (a1, a5, QQ);
  else if (SAME3(0,a2,a4) && SAME2(a5,a3) && SAME2(a1,a6) && (a5>0) && (a1>0))
    *result = TVIL_H00yxyx (a1, a5, QQ);
  else if (SAME3(0,a2,a5) && SAME2(a4,a6) && SAME2(a1,a3) && (a4>0) && (a1>0))
    *result = TVIL_H00yxyx (a1, a4, QQ);
  else if (SAME3(0,a2,a5) && SAME2(a4,a1) && SAME2(a6,a3) && (a4>0) && (a6>0))
    *result = TVIL_H00yxyx (a6, a4, QQ);
  else if (SAME3(0,a3,a5) && SAME2(a4,a6) && SAME2(a1,a2) && (a4>0) && (a1>0))
    *result = TVIL_H00yxyx (a1, a4, QQ);
  else if (SAME3(0,a3,a5) && SAME2(a1,a6) && SAME2(a4,a2) && (a1>0) && (a4>0))
    *result = TVIL_H00yxyx (a4, a1, QQ);
  else if (SAME3(0,a3,a6) && SAME2(a4,a5) && SAME2(a1,a2) && (a4>0) && (a1>0))
    *result = TVIL_H00yxyx (a1, a4, QQ);
  else if (SAME3(0,a3,a6) && SAME2(a2,a5) && SAME2(a1,a4) && (a2>0) && (a1>0))
    *result = TVIL_H00yxyx (a1, a2, QQ);
  else if (SAME3(0,a4,a5) && SAME2(a2,a3) && SAME2(a1,a6) && (a2>0) && (a1>0))
    *result = TVIL_H00yxyx (a1, a2, QQ);
  else if (SAME3(0,a4,a5) && SAME2(a2,a1) && SAME2(a3,a6) && (a2>0) && (a3>0))
    *result = TVIL_H00yxyx (a3, a2, QQ);
  else if (SAME3(0,a4,a6) && SAME2(a1,a3) && SAME2(a2,a5) && (a1>0) && (a2>0))
    *result = TVIL_H00yxyx (a2, a1, QQ);
  else if (SAME3(0,a4,a6) && SAME2(a1,a2) && SAME2(a3,a5) && (a1>0) && (a3>0))
    *result = TVIL_H00yxyx (a3, a1, QQ);
  else if (SAME3(0,a5,a6) && SAME2(a2,a3) && SAME2(a1,a4) && (a2>0) && (a1>0))
    *result = TVIL_H00yxyx (a1, a2, QQ);
  else if (SAME3(0,a5,a6) && SAME2(a1,a3) && SAME2(a2,a4) && (a1>0) && (a2>0))
    *result = TVIL_H00yxyx (a2, a1, QQ);

  else
    success = 0;

  return success;
}

/* ****************************************************************** */
/* Sets initial value for the H function at a point near t = 0. */

int TVIL_InitialH (TVIL_HTYPE  *foo, 
		   TVIL_REAL    qq,
		   TVIL_COMPLEX tinit)
{
  TVIL_REAL u = (foo->arg[0])/a;
  TVIL_REAL v = (foo->arg[1])/a;
  TVIL_REAL w = (foo->arg[2])/a;
  TVIL_REAL x = (foo->arg[3])/a;
  TVIL_REAL y = (foo->arg[4])/a;
  TVIL_REAL z = (foo->arg[5])/a;

  TVIL_REAL Hexp[9];
  TVIL_REAL u2, u3, u4, u5, u6, u7, u8;
  TVIL_REAL v2, v3, v4, v5, v6, v7, v8;
  TVIL_REAL w2, w3, w4, w5, w6, w7, w8;
  TVIL_REAL x2, x3, x4, x5, x6, x7, x8;
  TVIL_REAL y2, y3, y4, y5, y6, y7, y8;
  TVIL_REAL z2, z3, z4, z5, z6, z7, z8;

  TVIL_REAL lnbara;
  TVIL_COMPLEX ttothei, result;
  int i, nTerms = 9;

  lnbara = TVIL_LOG (a/qq);

  u2 = u*u;
  u3 = u2*u;
  u4 = u2*u2;
  u5 = u2*u3;
  u6 = u3*u3;
  u7 = u4*u3;
  u8 = u4*u4;

  v2 = v*v;
  v3 = v2*v;
  v4 = v2*v2;
  v5 = v2*v3;
  v6 = v3*v3;
  v7 = v4*v3;
  v8 = v4*v4;

  w2 = w*w;
  w3 = w2*w;
  w4 = w2*w2;
  w5 = w2*w3;
  w6 = w3*w3;
  w7 = w4*w3;
  w8 = w4*w4;

  x2 = x*x;
  x3 = x2*x;
  x4 = x2*x2;
  x5 = x2*x3;
  x6 = x3*x3;
  x7 = x4*x3;
  x8 = x4*x4;

  y2 = y*y;
  y3 = y2*y;
  y4 = y2*y2;
  y5 = y2*y3;
  y6 = y3*y3;
  y7 = y4*y3;
  y8 = y4*y4;

  z2 = z*z;
  z3 = z2*z;
  z4 = z2*z2;
  z5 = z2*z3;
  z6 = z3*z3;
  z7 = z4*z3;
  z8 = z4*z4;

  Hexp[0] = -10.03527847976879 - 7.212341418957566*lnbara;
 
  Hexp[1] = 7.212341418957566 - 1.2020569031595942*u - 1.2020569031595942*v - 
     1.2020569031595942*w - 1.2020569031595942*x - 1.2020569031595942*y - 
     1.2020569031595942*z;
 
  Hexp[2] = 3.606170709478783 - 1.2020569031595942*u + 0.2638365705403415*u2 - 
     1.2020569031595942*v + 0.14013624956353604*u*v + 
     0.2638365705403415*v2 - 1.2020569031595942*w + 
     0.14013624956353604*u*w + 0.14013624956353604*v*w + 
     0.2638365705403415*w2 - 1.2020569031595942*x + 
     0.14013624956353604*u*x + 0.14013624956353604*v*x + 
     0.11383876382476714*w*x + 0.2638365705403415*x2 - 
     1.2020569031595942*y + 0.11383876382476714*u*y + 
     0.14013624956353604*v*y + 0.14013624956353604*w*y + 
     0.14013624956353604*x*y + 0.2638365705403415*y2 - 
     1.2020569031595942*z + 0.14013624956353604*u*z + 
     0.11383876382476714*v*z + 0.14013624956353604*w*z + 
     0.14013624956353604*x*z + 0.14013624956353604*y*z + 
     0.2638365705403415*z2;
 
  Hexp[3] = 2.4041138063191885 - 1.2020569031595942*u + 0.527673141080683*u2 - 
     0.10795675579460708*u3 - 1.2020569031595942*v + 
     0.2802724991270721*u*v - 0.04287703318427488*u2*v + 
     0.527673141080683*v2 - 0.04287703318427488*u*v2 - 
     0.10795675579460708*v3 - 1.2020569031595942*w + 
     0.2802724991270721*u*w - 0.04287703318427488*u2*w + 
     0.2802724991270721*v*w - 0.0262974857387689*u*v*w - 
     0.04287703318427488*v2*w + 0.527673141080683*w2 - 
     0.04287703318427488*u*w2 - 0.04287703318427488*v*w2 - 
     0.10795675579460708*w3 - 1.2020569031595942*x + 
     0.2802724991270721*u*x - 0.04287703318427488*u2*x + 
     0.2802724991270721*v*x + 0.12075917025180338*u*v*x + 
     0.04340568960536035*lnbara*u*v*x - 0.19685380218203927*v2*x - 
     0.04340568960536035*lnbara*v2*x + 0.22767752764953428*w*x - 
     0.1786014099503857*u*w*x - 0.04340568960536035*lnbara*u*w*x - 
     0.02462464095262132*v*w*x + 0.12168202803800214*w2*x + 
     0.04340568960536035*lnbara*w2*x + 0.527673141080683*x2 - 
     0.04287703318427488*u*x2 - 0.19685380218203927*v*x2 - 
     0.04340568960536035*lnbara*v*x2 + 0.12168202803800214*w*x2 + 
     0.04340568960536035*lnbara*w*x2 - 0.10795675579460708*x3 - 
     1.2020569031595942*y + 0.22767752764953428*u*y - 
     0.03229474095976225*u2*y + 0.2802724991270721*v*y - 
     0.02462464095262132*u*v*y - 0.04287703318427488*v2*y + 
     0.2802724991270721*w*y - 0.02462464095262132*u*w*y - 
     0.03321759874596101*v*w*y - 0.04287703318427488*w2*y + 
     0.2802724991270721*x*y - 0.02462464095262132*u*x*y + 
     0.4356328212545243*v*x*y + 0.13021706881608106*lnbara*v*x*y - 
     0.4865549479459145*w*x*y - 0.13021706881608106*lnbara*w*x*y - 
     0.04287703318427488*x2*y + 0.527673141080683*y2 - 
     0.03229474095976225*u*y2 - 0.04287703318427488*v*y2 - 
     0.04287703318427488*w*y2 - 0.04287703318427488*x*y2 - 
     0.10795675579460708*y3 - 1.2020569031595942*z + 
     0.2802724991270721*u*z - 0.04287703318427488*u2*z + 
     0.22767752764953428*v*z - 0.1786014099503857*u*v*z - 
     0.04340568960536035*lnbara*u*v*z + 0.12168202803800214*v2*z + 
     0.04340568960536035*lnbara*v2*z + 0.2802724991270721*w*z + 
     0.12075917025180338*u*w*z + 0.04340568960536035*lnbara*u*w*z - 
     0.02462464095262132*v*w*z - 0.19685380218203927*w2*z - 
     0.04340568960536035*lnbara*w2*z + 0.2802724991270721*x*z - 
     0.0262974857387689*u*x*z - 0.02462464095262132*v*x*z - 
     0.02462464095262132*w*x*z - 0.04287703318427488*x2*z + 
     0.2802724991270721*y*z - 0.02462464095262132*u*y*z - 
     0.4865549479459145*v*y*z - 0.13021706881608106*lnbara*v*y*z + 
     0.4356328212545243*w*y*z + 0.13021706881608106*lnbara*w*y*z - 
     0.03321759874596101*x*y*z - 0.04287703318427488*y2*z + 
     0.527673141080683*z2 - 0.04287703318427488*u*z2 + 
     0.12168202803800214*v*z2 + 0.04340568960536035*lnbara*v*z2 - 
     0.19685380218203927*w*z2 - 0.04340568960536035*lnbara*w*z2 - 
     0.04287703318427488*x*z2 - 0.04287703318427488*y*z2 - 
     0.10795675579460708*z3;
 
  Hexp[4] = 1.8030853547393915 - 1.2020569031595942*u + 
     0.7915097116210246*u2 - 0.32387026738382124*u3 + 
     0.05780009836815717*u4 - 1.2020569031595942*v + 
     0.4204087486906081*u*v - 0.12863109955282465*u2*v + 
     0.019595873400524336*u3*v + 0.7915097116210246*v2 - 
     0.12863109955282465*u*v2 + 0.014860391371313229*u2*v2 - 
     0.32387026738382124*v3 + 0.019595873400524336*u*v3 + 
     0.05780009836815717*v4 - 1.2020569031595942*w + 
     0.4204087486906081*u*w - 0.12863109955282465*u2*w + 
     0.019595873400524336*u3*w + 0.4204087486906081*v*w - 
     0.0788924572163067*u*v*w + 0.009467502496847111*u2*v*w - 
     0.12863109955282465*v2*w + 0.009467502496847111*u*v2*w + 
     0.019595873400524336*v3*w + 0.7915097116210246*w2 - 
     0.12863109955282465*u*w2 + 0.014860391371313229*u2*w2 - 
     0.12863109955282465*v*w2 + 0.009467502496847111*u*v*w2 + 
     0.014860391371313229*v2*w2 - 0.32387026738382124*w3 + 
     0.019595873400524336*u*w3 + 0.019595873400524336*v*w3 + 
     0.05780009836815717*w4 - 1.2020569031595942*x + 
     0.4204087486906081*u*x - 0.12863109955282465*u2*x + 
     0.019595873400524336*u3*x + 0.4204087486906081*v*x + 
     0.3297232435513899*u*v*x + 0.13021706881608106*lnbara*u*v*x + 
     0.09849335296892119*u2*v*x + 0.022352066577107733*lnbara*u2*v*x - 
     0.5580071393420976*v2*x - 0.13021706881608106*lnbara*v2*x - 
     0.18693837131393218*u*v2*x - 0.056425067024902395*lnbara*u*v2*x + 
     0.056907625247539464*v3*x + 0.012370155645114489*lnbara*v3*x + 
     0.3415162914743014*w*x - 0.5032499626471368*u*w*x - 
     0.13021706881608106*lnbara*u*w*x - 0.07638626512936658*u2*w*x - 
     0.022352066577107733*lnbara*u2*w*x - 0.07387392285786396*v*w*x + 
     0.007362480745074675*u*v*w*x + 0.5854099721783019*v2*w*x + 
     0.1642900692638757*lnbara*v2*w*x + 0.33249181690998614*w2*x + 
     0.13021706881608106*lnbara*w2*x + 0.2084140365865988*u*w2*x + 
     0.056425067024902395*lnbara*u*w2*x - 0.567682337066967*v*w2*x - 
     0.1642900692638757*lnbara*v*w2*x - 0.02302537153791993*w3*x - 
     0.012370155645114489*lnbara*w3*x + 0.7915097116210246*x2 - 
     0.12863109955282465*u*x2 + 0.014860391371313229*u2*x2 - 
     0.5580071393420976*v*x2 - 0.13021706881608106*lnbara*v*x2 - 
     0.18693837131393218*u*v*x2 - 0.056425067024902395*lnbara*u*v*x2 + 
     0.3204490485619901*v2*x2 + 0.0898488456982695*lnbara*v2*x2 + 
     0.33249181690998614*w*x2 + 0.13021706881608106*lnbara*w*x2 + 
     0.2084140365865988*u*w*x2 + 0.056425067024902395*lnbara*u*w*x2 - 
     0.37639381622218754*v*w*x2 - 0.10851422401340087*lnbara*v*w*x2 + 
     0.0892695937352878*w2*x2 + 0.018665378315131372*lnbara*w2*x2 - 
     0.32387026738382124*x3 + 0.019595873400524336*u*x3 + 
     0.056907625247539464*v*x3 + 0.012370155645114489*lnbara*v*x3 - 
     0.02302537153791993*w*x3 - 0.012370155645114489*lnbara*w*x3 + 
     0.05780009836815717*x4 - 1.2020569031595942*y + 
     0.3415162914743014*u*y - 0.09688422287928675*u2*y + 
     0.014286380309095197*u3*y + 0.4204087486906081*v*y - 
     0.07387392285786396*u*v*y + 0.008548106272223455*u2*v*y - 
     0.12863109955282465*v2*y + 0.009179528839111438*u*v2*y + 
     0.019595873400524336*v3*y + 0.4204087486906081*w*y - 
     0.07387392285786396*u*w*y + 0.008548106272223455*u2*w*y - 
     0.09965279623788302*v*w*y + 0.007362480745074675*u*v*w*y + 
     0.012927559000443166*v2*w*y - 0.12863109955282465*w2*y + 
     0.009179528839111438*u*w2*y + 0.012927559000443166*v*w2*y + 
     0.019595873400524336*w3*y + 0.4204087486906081*x*y - 
     0.07387392285786396*u*x*y + 0.008548106272223455*u2*x*y + 
     1.209235662151512*v*x*y + 0.39065120644824314*lnbara*v*x*y + 
     0.20631857427634712*u*v*x*y + 0.05077906612931307*lnbara*u*v*x*y - 
     0.2756916181666832*v2*x*y - 0.08594186774137386*lnbara*v2*x*y - 
     1.3620020422256827*w*x*y - 0.39065120644824314*lnbara*w*x*y - 
     0.19235861493067452*u*w*x*y - 0.05077906612931307*lnbara*u*w*x*y + 
     0.007362480745074675*v*w*x*y + 0.2937072269357538*w2*x*y + 
     0.08594186774137386*lnbara*w2*x*y - 0.12863109955282465*x2*y + 
     0.009179528839111438*u*x2*y - 0.2756916181666832*v*x2*y - 
     0.08594186774137386*lnbara*v*x2*y + 0.2937072269357538*w*x2*y + 
     0.08594186774137386*lnbara*w*x2*y + 0.019595873400524336*x3*y + 
     0.7915097116210246*y2 - 0.09688422287928675*u*y2 + 
     0.009916328431553665*u2*y2 - 0.12863109955282465*v*y2 + 
     0.008548106272223455*u*v*y2 + 0.014860391371313229*v2*y2 - 
     0.12863109955282465*w*y2 + 0.008548106272223455*u*w*y2 + 
     0.012927559000443166*v*w*y2 + 0.014860391371313229*w2*y2 - 
     0.12863109955282465*x*y2 + 0.008548106272223455*u*x*y2 - 
     0.16375636262563784*v*x*y2 - 0.04883140080603039*lnbara*v*x*y2 + 
     0.18240339396159638*w*x*y2 + 0.04883140080603039*lnbara*w*x*y2 + 
     0.014860391371313229*x2*y2 - 0.32387026738382124*y3 + 
     0.014286380309095197*u*y3 + 0.019595873400524336*v*y3 + 
     0.019595873400524336*w*y3 + 0.019595873400524336*x*y3 + 
     0.05780009836815717*y4 - 1.2020569031595942*z + 
     0.4204087486906081*u*z - 0.12863109955282465*u2*z + 
     0.019595873400524336*u3*z + 0.3415162914743014*v*z - 
     0.5032499626471368*u*v*z - 0.13021706881608106*lnbara*u*v*z - 
     0.07638626512936658*u2*v*z - 0.022352066577107733*lnbara*u2*v*z + 
     0.33249181690998614*v2*z + 0.13021706881608106*lnbara*v2*z + 
     0.2084140365865988*u*v2*z + 0.056425067024902395*lnbara*u*v2*z - 
     0.02302537153791993*v3*z - 0.012370155645114489*lnbara*v3*z + 
     0.4204087486906081*w*z + 0.3297232435513899*u*w*z + 
     0.13021706881608106*lnbara*u*w*z + 0.09849335296892119*u2*w*z + 
     0.022352066577107733*lnbara*u2*w*z - 0.07387392285786396*v*w*z + 
     0.007362480745074675*u*v*w*z - 0.567682337066967*v2*w*z - 
     0.1642900692638757*lnbara*v2*w*z - 0.5580071393420976*w2*z - 
     0.13021706881608106*lnbara*w2*z - 0.18693837131393218*u*w2*z - 
     0.056425067024902395*lnbara*u*w2*z + 0.5854099721783019*v*w2*z + 
     0.1642900692638757*lnbara*v*w2*z + 0.056907625247539464*w3*z + 
     0.012370155645114489*lnbara*w3*z + 0.4204087486906081*x*z - 
     0.0788924572163067*u*x*z + 0.009467502496847111*u2*x*z - 
     0.07387392285786396*v*x*z + 0.007362480745074675*u*v*x*z - 
     0.37639381622218754*v2*x*z - 0.10851422401340087*lnbara*v2*x*z - 
     0.07387392285786396*w*x*z + 0.007362480745074675*u*w*x*z + 
     0.7764813235894199*v*w*x*z + 0.21702844802680174*lnbara*v*w*x*z - 
     0.37639381622218754*w2*x*z - 0.10851422401340087*lnbara*w2*x*z - 
     0.12863109955282465*x2*z + 0.009467502496847111*u*x2*z + 
     0.5854099721783019*v*x2*z + 0.1642900692638757*lnbara*v*x2*z - 
     0.567682337066967*w*x2*z - 0.1642900692638757*lnbara*w*x2*z + 
     0.019595873400524336*x3*z + 0.4204087486906081*y*z - 
     0.07387392285786396*u*y*z + 0.008548106272223455*u2*y*z - 
     1.3620020422256827*v*y*z - 0.39065120644824314*lnbara*v*y*z - 
     0.19235861493067452*u*v*y*z - 0.05077906612931307*lnbara*u*v*y*z + 
     0.2937072269357538*v2*y*z + 0.08594186774137386*lnbara*v2*y*z + 
     1.209235662151512*w*y*z + 0.39065120644824314*lnbara*w*y*z + 
     0.20631857427634712*u*w*y*z + 0.05077906612931307*lnbara*u*w*y*z + 
     0.007362480745074675*v*w*y*z - 0.2756916181666832*w2*y*z - 
     0.08594186774137386*lnbara*w2*y*z - 0.09965279623788302*x*y*z + 
     0.007362480745074675*u*x*y*z + 0.007362480745074675*v*x*y*z + 
     0.007362480745074675*w*x*y*z + 0.012927559000443166*x2*y*z - 
     0.12863109955282465*y2*z + 0.008548106272223455*u*y2*z + 
     0.18240339396159638*v*y2*z + 0.04883140080603039*lnbara*v*y2*z - 
     0.16375636262563784*w*y2*z - 0.04883140080603039*lnbara*w*y2*z + 
     0.012927559000443166*x*y2*z + 0.019595873400524336*y3*z + 
     0.7915097116210246*z2 - 0.12863109955282465*u*z2 + 
     0.014860391371313229*u2*z2 + 0.33249181690998614*v*z2 + 
     0.13021706881608106*lnbara*v*z2 + 0.2084140365865988*u*v*z2 + 
     0.056425067024902395*lnbara*u*v*z2 + 0.0892695937352878*v2*z2 + 
     0.018665378315131372*lnbara*v2*z2 - 0.5580071393420976*w*z2 - 
     0.13021706881608106*lnbara*w*z2 - 0.18693837131393218*u*w*z2 - 
     0.056425067024902395*lnbara*u*w*z2 - 0.37639381622218754*v*w*z2 - 
     0.10851422401340087*lnbara*v*w*z2 + 0.3204490485619901*w2*z2 + 
     0.0898488456982695*lnbara*w2*z2 - 0.12863109955282465*x*z2 + 
     0.009467502496847111*u*x*z2 - 0.567682337066967*v*x*z2 - 
     0.1642900692638757*lnbara*v*x*z2 + 0.5854099721783019*w*x*z2 + 
     0.1642900692638757*lnbara*w*x*z2 + 0.014860391371313229*x2*z2 - 
     0.12863109955282465*y*z2 + 0.009179528839111438*u*y*z2 + 
     0.2937072269357538*v*y*z2 + 0.08594186774137386*lnbara*v*y*z2 - 
     0.2756916181666832*w*y*z2 - 0.08594186774137386*lnbara*w*y*z2 + 
     0.012927559000443166*x*y*z2 + 0.014860391371313229*y2*z2 - 
     0.32387026738382124*z3 + 0.019595873400524336*u*z3 - 
     0.02302537153791993*v*z3 - 0.012370155645114489*lnbara*v*z3 + 
     0.056907625247539464*w*z3 + 0.012370155645114489*lnbara*w*z3 + 
     0.019595873400524336*x*z3 + 0.019595873400524336*y*z3 + 
     0.05780009836815717*z4;
 
  Hexp[5] = 1.442468283791513 - 1.2020569031595942*u + 1.055346282161366*u2 - 
     0.6477405347676425*u3 + 0.2312003934726287*u4 - 
     0.03583388093754629*u5 - 1.2020569031595942*v + 
     0.5605449982541442*u*v - 0.2572621991056493*u2*v + 
     0.07838349360209734*u3*v - 0.01103033535263963*u4*v + 
     1.055346282161366*v2 - 0.2572621991056493*u*v2 + 
     0.059441565485252916*u2*v2 - 0.007187565837728457*u3*v2 - 
     0.6477405347676425*v3 + 0.07838349360209734*u*v3 - 
     0.007187565837728457*u2*v3 + 0.2312003934726287*v4 - 
     0.01103033535263963*u*v4 - 0.03583388093754629*v5 - 
     1.2020569031595942*w + 0.5605449982541442*u*w - 
     0.2572621991056493*u2*w + 0.07838349360209734*u3*w - 
     0.01103033535263963*u4*w + 0.5605449982541442*v*w - 
     0.1577849144326134*u*v*w + 0.037870009987388445*u2*v*w - 
     0.004645460955055343*u3*v*w - 0.2572621991056493*v2*w + 
     0.037870009987388445*u*v2*w - 0.003682012115837776*u2*v2*w + 
     0.07838349360209734*v3*w - 0.004645460955055343*u*v3*w - 
     0.01103033535263963*v4*w + 1.055346282161366*w2 - 
     0.2572621991056493*u*w2 + 0.059441565485252916*u2*w2 - 
     0.007187565837728457*u3*w2 - 0.2572621991056493*v*w2 + 
     0.037870009987388445*u*v*w2 - 0.003682012115837776*u2*v*w2 + 
     0.059441565485252916*v2*w2 - 0.003682012115837776*u*v2*w2 - 
     0.007187565837728457*v3*w2 - 0.6477405347676425*w3 + 
     0.07838349360209734*u*w3 - 0.007187565837728457*u2*w3 + 
     0.07838349360209734*v*w3 - 0.004645460955055343*u*v*w3 - 
     0.007187565837728457*v2*w3 + 0.2312003934726287*w4 - 
     0.01103033535263963*u*w4 - 0.01103033535263963*v*w4 - 
     0.03583388093754629*w5 - 1.2020569031595942*x + 
     0.5605449982541442*u*x - 0.2572621991056493*u2*x + 
     0.07838349360209734*u3*x - 0.01103033535263963*u4*x + 
     0.5605449982541442*v*x + 0.6073596595763473*u*v*x + 
     0.2604341376321621*lnbara*u*v*x + 0.3760917586139986*u2*v*x + 
     0.08940826630843093*lnbara*u2*v*x + 0.037477277457532834*u3*v*x + 
     0.01183453927120045*lnbara*u3*v*x - 1.0639274511577628*v2*x - 
     0.2604341376321621*lnbara*v2*x - 0.7026134316358068*u*v2*x - 
     0.22570026809960958*lnbara*u*v2*x - 0.1500217343901253*u2*v2*x - 
     0.03949638218639865*lnbara*u2*v2*x + 0.21773437647406627*v3*x + 
     0.049480622580457954*lnbara*v3*x + 0.2734637028141685*u*v3*x + 
     0.08124448002734691*lnbara*u*v3*x - 0.0830149629493877*v4*x - 
     0.021288058617899477*lnbara*v4*x + 0.45535505529906856*w*x - 
     0.9544130977678413*u*w*x - 0.2604341376321621*lnbara*u*w*x - 
     0.28766340725578016*u2*w*x - 0.08940826630843093*lnbara*u2*w*x - 
     0.04860539177383991*u3*w*x - 0.01183453927120045*lnbara*u3*w*x - 
     0.1477478457157279*v*w*x + 0.0294499229802987*u*v*w*x - 
     0.0031815740344914314*u2*v*w*x + 2.210207833302107*v2*w*x + 
     0.6571602770555028*lnbara*v2*w*x - 0.24603076767859458*u*v2*w*x - 
     0.07342181363031552*lnbara*u*v2*w*x - 0.379627630631495*v3*w*x - 
     0.1139718730378812*lnbara*v3*w*x + 0.6128968062935399*w2*x + 
     0.2604341376321621*lnbara*w2*x + 0.7885160927264733*u*w2*x + 
     0.22570026809960958*lnbara*u*w2*x + 0.14088351757088213*u2*w2*x + 
     0.03949638218639865*lnbara*u2*w2*x - 2.1392972928567677*v*w2*x - 
     0.6571602770555028*lnbara*v*w2*x + 0.2400067630542147*u*v*w2*x + 
     0.07342181363031552*lnbara*u*v*w2*x - 0.0034959415238012314*v2*w2*x - 
     0.08220536163558813*w3*x - 0.049480622580457954*lnbara*w3*x - 
     0.2841442232687275*u*w3*x - 0.08124448002734691*lnbara*u*w3*x + 
     0.37095314628030795*v*w3*x + 0.1139718730378812*lnbara*v*w3*x + 
     0.06407498022240934*w4*x + 0.021288058617899477*lnbara*w4*x + 
     1.055346282161366*x2 - 0.2572621991056493*u*x2 + 
     0.059441565485252916*u2*x2 - 0.007187565837728457*u3*x2 - 
     1.0639274511577628*v*x2 - 0.2604341376321621*lnbara*v*x2 - 
     0.7026134316358068*u*v*x2 - 0.22570026809960958*lnbara*u*v*x2 - 
     0.1500217343901253*u2*v*x2 - 0.03949638218639865*lnbara*u2*v*x2 + 
     1.2099171176893446*v2*x2 + 0.359395382793078*lnbara*v2*x2 + 
     0.11341087241156884*u*v2*x2 + 0.03460240492286588*lnbara*u*v2*x2 - 
     0.19765223368547283*v3*x2 - 0.05890476996783441*lnbara*v3*x2 + 
     0.6128968062935399*w*x2 + 0.2604341376321621*lnbara*w*x2 + 
     0.7885160927264733*u*w*x2 + 0.22570026809960958*lnbara*u*w*x2 + 
     0.14088351757088213*u2*w*x2 + 0.03949638218639865*lnbara*u2*w*x2 - 
     1.4187638856780294*v*w*x2 - 0.4340568960536035*lnbara*v*w*x2 + 
     0.4719083667190125*u*v*w*x2 + 0.13861380499219164*lnbara*u*v*w*x2 - 
     0.08432966934787235*v2*w*x2 - 0.029311197001213464*lnbara*v2*w*x2 + 
     0.3421460722890461*w2*x2 + 0.07466151326052549*lnbara*w2*x2 - 
     0.5971647409089695*u*w2*x2 - 0.1732162099150575*lnbara*u*w2*x2 + 
     0.23366341602051563*v*w2*x2 + 0.07821369824348423*lnbara*v*w2*x2 + 
     0.029981938909783182*w3*x2 + 0.010002268725563646*lnbara*w3*x2 - 
     0.6477405347676425*x3 + 0.07838349360209734*u*x3 - 
     0.007187565837728457*u2*x3 + 0.21773437647406627*v*x3 + 
     0.049480622580457954*lnbara*v*x3 + 0.2734637028141685*u*v*x3 + 
     0.08124448002734691*lnbara*u*v*x3 - 0.19765223368547283*v2*x3 - 
     0.05890476996783441*lnbara*v2*x3 - 0.08220536163558813*w*x3 - 
     0.049480622580457954*lnbara*w*x3 - 0.2841442232687275*u*w*x3 - 
     0.08124448002734691*lnbara*u*w*x3 + 0.15184270831316776*v*w*x3 + 
     0.04890250124227076*lnbara*v*w*x3 + 0.029981938909783182*w2*x3 + 
     0.010002268725563646*lnbara*w2*x3 + 0.2312003934726287*x4 - 
     0.01103033535263963*u*x4 - 0.0830149629493877*v*x4 - 
     0.021288058617899477*lnbara*v*x4 + 0.06407498022240934*w*x4 + 
     0.021288058617899477*lnbara*w*x4 - 0.03583388093754629*x5 - 
     1.2020569031595942*y + 0.45535505529906856*u*y - 
     0.1937684457585735*u2*y + 0.05714552123638079*u3*y - 
     0.007909647374338729*u4*y + 0.5605449982541442*v*y - 
     0.1477478457157279*u*v*y + 0.03419242508889382*u2*v*y - 
     0.004113445244719507*u3*v*y - 0.2572621991056493*v2*y + 
     0.03671811535644575*u*v2*y - 0.0034959415238012314*u2*v2*y + 
     0.07838349360209734*v3*y - 0.004561039106467548*u*v3*y - 
     0.01103033535263963*v4*y + 0.5605449982541442*w*y - 
     0.1477478457157279*u*w*y + 0.03419242508889382*u2*w*y - 
     0.004113445244719507*u3*w*y - 0.19930559247576604*v*w*y + 
     0.0294499229802987*u*v*w*y - 0.00284243058988845*u2*v*w*y + 
     0.051710236001772665*v2*w*y - 0.0031815740344914314*u*v2*w*y - 
     0.00656707520983952*v3*w*y - 0.2572621991056493*w2*y + 
     0.03671811535644575*u*w2*y - 0.0034959415238012314*u2*w2*y + 
     0.051710236001772665*v*w2*y - 0.0031815740344914314*u*v*w2*y - 
     0.005642275295441937*v2*w2*y + 0.07838349360209734*w3*y - 
     0.004561039106467548*u*w3*y - 0.00656707520983952*v*w3*y - 
     0.01103033535263963*w4*y + 0.5605449982541442*x*y - 
     0.1477478457157279*u*x*y + 0.03419242508889382*u2*x*y - 
     0.004113445244719507*u3*x*y + 2.2622108417237268*v*x*y + 
     0.7813024128964863*lnbara*v*x*y + 0.784651044201938*u*v*x*y + 
     0.2031162645172523*lnbara*u*v*x*y + 0.11792649464288701*u2*v*x*y + 
     0.03307364462356231*lnbara*u2*v*x*y - 1.0340129784736338*v2*x*y - 
     0.3437674709654954*lnbara*v2*x*y - 0.2704335518900037*u*v2*x*y - 
     0.07343720881724174*lnbara*u*v2*x*y + 0.46395116219858895*v3*x*y + 
     0.13730604359507229*lnbara*v3*x*y - 2.5677436018720683*w*x*y - 
     0.7813024128964863*lnbara*w*x*y - 0.7288112068192476*u*w*x*y - 
     0.2031162645172523*lnbara*u*w*x*y - 0.12369133963724584*u2*w*x*y - 
     0.03307364462356231*lnbara*u2*w*x*y + 0.0294499229802987*v*w*x*y - 
     0.0024426217097745406*u*v*w*x*y - 0.49099871455493777*v2*w*x*y - 
     0.13429794906684606*lnbara*v2*w*x*y + 1.106075413549916*w2*x*y + 
     0.3437674709654954*lnbara*w2*x*y + 0.2645638939826569*u*w2*x*y + 
     0.07343720881724174*lnbara*u*w2*x*y + 0.4847403793989428*v*w2*x*y + 
     0.13429794906684606*lnbara*v*w2*x*y - 0.4727100683983638*w3*x*y - 
     0.13730604359507229*lnbara*w3*x*y - 0.2572621991056493*x2*y + 
     0.03671811535644575*u*x2*y - 0.0034959415238012314*u2*x2*y - 
     1.0340129784736338*v*x2*y - 0.3437674709654954*lnbara*v*x2*y - 
     0.2704335518900037*u*v*x2*y - 0.07343720881724174*lnbara*u*v*x2*y + 
     0.031244750707668238*v2*x2*y + 0.0180532260934895*lnbara*v2*x2*y + 
     1.106075413549916*w*x2*y + 0.3437674709654954*lnbara*w*x2*y + 
     0.2645638939826569*u*w*x2*y + 0.07343720881724174*lnbara*u*w*x2*y + 
     0.35130413810496963*v*w*x2*y + 0.0998330860923288*lnbara*v*w*x2*y - 
     0.3921997968798069*w2*x2*y - 0.1178863121858183*lnbara*w2*x2*y + 
     0.07838349360209734*x3*y - 0.004561039106467548*u*x3*y + 
     0.46395116219858895*v*x3*y + 0.13730604359507229*lnbara*v*x3*y - 
     0.4727100683983638*w*x3*y - 0.13730604359507229*lnbara*w*x3*y - 
     0.01103033535263963*x4*y + 1.055346282161366*y2 - 
     0.1937684457585735*u*y2 + 0.03966531372621466*u2*y2 - 
     0.004526575380073919*u3*y2 - 0.2572621991056493*v*y2 + 
     0.03419242508889382*u*v*y2 - 0.003126465361442786*u2*v*y2 + 
     0.059441565485252916*v2*y2 - 0.0034959415238012314*u*v2*y2 - 
     0.007187565837728457*v3*y2 - 0.2572621991056493*w*y2 + 
     0.03419242508889382*u*w*y2 - 0.003126465361442786*u2*w*y2 + 
     0.051710236001772665*v*w*y2 - 0.003076761121503497*u*v*w*y2 - 
     0.005642275295441937*v2*w*y2 + 0.059441565485252916*w2*y2 - 
     0.0034959415238012314*u*w2*y2 - 0.005642275295441937*v*w2*y2 - 
     0.007187565837728457*w3*y2 - 0.2572621991056493*x*y2 + 
     0.03419242508889382*u*x*y2 - 0.003126465361442786*u2*x*y2 - 
     0.615960329857727*v*x*y2 - 0.19532560322412157*lnbara*v*x*y2 + 
     0.03305770874592587*u*v*x*y2 + 0.012242640752295034*lnbara*u*v*x*y2 - 
     0.005100940598748941*v2*x*y2 + 0.0018224798925292808*lnbara*v2*x*
      y2 + 0.6905484552015612*w*x*y2 + 0.19532560322412157*lnbara*w*x*y2 - 
     0.03858822320866964*u*w*x*y2 - 0.012242640752295034*lnbara*u*w*x*y2 - 
     0.0031815740344914314*v*w*x*y2 - 0.0020770130408900665*w2*x*y2 - 
     0.0018224798925292808*lnbara*w2*x*y2 + 0.059441565485252916*x2*y2 - 
     0.0034959415238012314*u*x2*y2 - 0.005100940598748941*v*x2*y2 + 
     0.0018224798925292808*lnbara*v*x2*y2 - 0.0020770130408900665*w*x2*
      y2 - 0.0018224798925292808*lnbara*w*x2*y2 - 
     0.007187565837728457*x3*y2 - 0.6477405347676425*y3 + 
     0.05714552123638079*u*y3 - 0.004526575380073919*u2*y3 + 
     0.07838349360209734*v*y3 - 0.004113445244719507*u*v*y3 - 
     0.007187565837728457*v2*y3 + 0.07838349360209734*w*y3 - 
     0.004113445244719507*u*w*y3 - 0.00656707520983952*v*w*y3 - 
     0.007187565837728457*w2*y3 + 0.07838349360209734*x*y3 - 
     0.004113445244719507*u*x*y3 + 0.2032231771919266*v*x*y3 + 
     0.05859768096723647*lnbara*v*x*y3 - 0.21242967725344947*w*x*y3 - 
     0.05859768096723647*lnbara*w*x*y3 - 0.007187565837728457*x2*y3 + 
     0.2312003934726287*y4 - 0.007909647374338729*u*y4 - 
     0.01103033535263963*v*y4 - 0.01103033535263963*w*y4 - 
     0.01103033535263963*x*y4 - 0.03583388093754629*y5 - 
     1.2020569031595942*z + 0.5605449982541442*u*z - 
     0.2572621991056493*u2*z + 0.07838349360209734*u3*z - 
     0.01103033535263963*u4*z + 0.45535505529906856*v*z - 
     0.9544130977678413*u*v*z - 0.2604341376321621*lnbara*u*v*z - 
     0.28766340725578016*u2*v*z - 0.08940826630843093*lnbara*u2*v*z - 
     0.04860539177383991*u3*v*z - 0.01183453927120045*lnbara*u3*v*z + 
     0.6128968062935399*v2*z + 0.2604341376321621*lnbara*v2*z + 
     0.7885160927264733*u*v2*z + 0.22570026809960958*lnbara*u*v2*z + 
     0.14088351757088213*u2*v2*z + 0.03949638218639865*lnbara*u2*v2*z - 
     0.08220536163558813*v3*z - 0.049480622580457954*lnbara*v3*z - 
     0.2841442232687275*u*v3*z - 0.08124448002734691*lnbara*u*v3*z + 
     0.06407498022240934*v4*z + 0.021288058617899477*lnbara*v4*z + 
     0.5605449982541442*w*z + 0.6073596595763473*u*w*z + 
     0.2604341376321621*lnbara*u*w*z + 0.3760917586139986*u2*w*z + 
     0.08940826630843093*lnbara*u2*w*z + 0.037477277457532834*u3*w*z + 
     0.01183453927120045*lnbara*u3*w*z - 0.1477478457157279*v*w*z + 
     0.0294499229802987*u*v*w*z - 0.0031815740344914314*u2*v*w*z - 
     2.1392972928567677*v2*w*z - 0.6571602770555028*lnbara*v2*w*z + 
     0.2400067630542147*u*v2*w*z + 0.07342181363031552*lnbara*u*v2*w*z + 
     0.37095314628030795*v3*w*z + 0.1139718730378812*lnbara*v3*w*z - 
     1.0639274511577628*w2*z - 0.2604341376321621*lnbara*w2*z - 
     0.7026134316358068*u*w2*z - 0.22570026809960958*lnbara*u*w2*z - 
     0.1500217343901253*u2*w2*z - 0.03949638218639865*lnbara*u2*w2*z + 
     2.210207833302107*v*w2*z + 0.6571602770555028*lnbara*v*w2*z - 
     0.24603076767859458*u*v*w2*z - 0.07342181363031552*lnbara*u*v*w2*z - 
     0.0034959415238012314*v2*w2*z + 0.21773437647406627*w3*z + 
     0.049480622580457954*lnbara*w3*z + 0.2734637028141685*u*w3*z + 
     0.08124448002734691*lnbara*u*w3*z - 0.379627630631495*v*w3*z - 
     0.1139718730378812*lnbara*v*w3*z - 0.0830149629493877*w4*z - 
     0.021288058617899477*lnbara*w4*z + 0.5605449982541442*x*z - 
     0.1577849144326134*u*x*z + 0.037870009987388445*u2*x*z - 
     0.004645460955055343*u3*x*z - 0.1477478457157279*v*x*z + 
     0.0294499229802987*u*v*x*z - 0.0031815740344914314*u2*v*x*z - 
     1.4187638856780294*v2*x*z - 0.4340568960536035*lnbara*v2*x*z + 
     0.4719083667190125*u*v2*x*z + 0.13861380499219164*lnbara*u*v2*x*z + 
     0.15184270831316776*v3*x*z + 0.04890250124227076*lnbara*v3*x*z - 
     0.1477478457157279*w*x*z + 0.0294499229802987*u*w*x*z - 
     0.0031815740344914314*u2*w*x*z + 2.932302535936238*v*w*x*z + 
     0.868113792107207*lnbara*v*w*x*z - 0.9524128773908065*u*v*w*x*z - 
     0.2772276099843833*lnbara*u*v*w*x*z - 0.1586442374307426*v2*w*x*z - 
     0.04890250124227076*lnbara*v2*w*x*z - 1.4187638856780294*w2*x*z - 
     0.4340568960536035*lnbara*w2*x*z + 0.4719083667190125*u*w2*x*z + 
     0.13861380499219164*lnbara*u*w2*x*z - 0.1586442374307426*v*w2*x*z - 
     0.04890250124227076*lnbara*v*w2*x*z + 0.15184270831316776*w3*x*z + 
     0.04890250124227076*lnbara*w3*x*z - 0.2572621991056493*x2*z + 
     0.037870009987388445*u*x2*z - 0.003682012115837776*u2*x2*z + 
     2.210207833302107*v*x2*z + 0.6571602770555028*lnbara*v*x2*z - 
     0.24603076767859458*u*v*x2*z - 0.07342181363031552*lnbara*u*v*x2*z - 
     0.08432966934787235*v2*x2*z - 0.029311197001213464*lnbara*v2*x2*z - 
     2.1392972928567677*w*x2*z - 0.6571602770555028*lnbara*w*x2*z + 
     0.2400067630542147*u*w*x2*z + 0.07342181363031552*lnbara*u*w*x2*z - 
     0.1586442374307426*v*w*x2*z - 0.04890250124227076*lnbara*v*w*x2*z + 
     0.23366341602051563*w2*x2*z + 0.07821369824348423*lnbara*w2*x2*z + 
     0.07838349360209734*x3*z - 0.004645460955055343*u*x3*z - 
     0.379627630631495*v*x3*z - 0.1139718730378812*lnbara*v*x3*z + 
     0.37095314628030795*w*x3*z + 0.1139718730378812*lnbara*w*x3*z - 
     0.01103033535263963*x4*z + 0.5605449982541442*y*z - 
     0.1477478457157279*u*y*z + 0.03419242508889382*u2*y*z - 
     0.004113445244719507*u3*y*z - 2.5677436018720683*v*y*z - 
     0.7813024128964863*lnbara*v*y*z - 0.7288112068192476*u*v*y*z - 
     0.2031162645172523*lnbara*u*v*y*z - 0.12369133963724584*u2*v*y*z - 
     0.03307364462356231*lnbara*u2*v*y*z + 1.106075413549916*v2*y*z + 
     0.3437674709654954*lnbara*v2*y*z + 0.2645638939826569*u*v2*y*z + 
     0.07343720881724174*lnbara*u*v2*y*z - 0.4727100683983638*v3*y*z - 
     0.13730604359507229*lnbara*v3*y*z + 2.2622108417237268*w*y*z + 
     0.7813024128964863*lnbara*w*y*z + 0.784651044201938*u*w*y*z + 
     0.2031162645172523*lnbara*u*w*y*z + 0.11792649464288701*u2*w*y*z + 
     0.03307364462356231*lnbara*u2*w*y*z + 0.0294499229802987*v*w*y*z - 
     0.0024426217097745406*u*v*w*y*z + 0.4847403793989428*v2*w*y*z + 
     0.13429794906684606*lnbara*v2*w*y*z - 1.0340129784736338*w2*y*z - 
     0.3437674709654954*lnbara*w2*y*z - 0.2704335518900037*u*w2*y*z - 
     0.07343720881724174*lnbara*u*w2*y*z - 0.49099871455493777*v*w2*y*z - 
     0.13429794906684606*lnbara*v*w2*y*z + 0.46395116219858895*w3*y*z + 
     0.13730604359507229*lnbara*w3*y*z - 0.19930559247576604*x*y*z + 
     0.0294499229802987*u*x*y*z - 0.00284243058988845*u2*x*y*z + 
     0.0294499229802987*v*x*y*z - 0.0024426217097745406*u*v*x*y*z + 
     0.35130413810496963*v2*x*y*z + 0.0998330860923288*lnbara*v2*x*y*z + 
     0.0294499229802987*w*x*y*z - 0.0024426217097745406*u*w*x*y*z - 
     0.7107357590994907*v*w*x*y*z - 0.1996661721846576*lnbara*v*w*x*y*z + 
     0.35130413810496963*w2*x*y*z + 0.0998330860923288*lnbara*w2*x*y*z + 
     0.051710236001772665*x2*y*z - 0.0031815740344914314*u*x2*y*z - 
     0.49099871455493777*v*x2*y*z - 0.13429794906684606*lnbara*v*x2*y*z + 
     0.4847403793989428*w*x2*y*z + 0.13429794906684606*lnbara*w*x2*y*z - 
     0.00656707520983952*x3*y*z - 0.2572621991056493*y2*z + 
     0.03419242508889382*u*y2*z - 0.003126465361442786*u2*y2*z + 
     0.6905484552015612*v*y2*z + 0.19532560322412157*lnbara*v*y2*z - 
     0.03858822320866964*u*v*y2*z - 0.012242640752295034*lnbara*u*v*y2*z - 
     0.0020770130408900665*v2*y2*z - 0.0018224798925292808*lnbara*v2*y2*
      z - 0.615960329857727*w*y2*z - 0.19532560322412157*lnbara*w*y2*z + 
     0.03305770874592587*u*w*y2*z + 0.012242640752295034*lnbara*u*w*y2*z - 
     0.0031815740344914314*v*w*y2*z - 0.005100940598748941*w2*y2*z + 
     0.0018224798925292808*lnbara*w2*y2*z + 0.051710236001772665*x*y2*z - 
     0.003076761121503497*u*x*y2*z - 0.0031815740344914314*v*x*y2*z - 
     0.0031815740344914314*w*x*y2*z - 0.005642275295441937*x2*y2*z + 
     0.07838349360209734*y3*z - 0.004113445244719507*u*y3*z - 
     0.21242967725344947*v*y3*z - 0.05859768096723647*lnbara*v*y3*z + 
     0.2032231771919266*w*y3*z + 0.05859768096723647*lnbara*w*y3*z - 
     0.00656707520983952*x*y3*z - 0.01103033535263963*y4*z + 
     1.055346282161366*z2 - 0.2572621991056493*u*z2 + 
     0.059441565485252916*u2*z2 - 0.007187565837728457*u3*z2 + 
     0.6128968062935399*v*z2 + 0.2604341376321621*lnbara*v*z2 + 
     0.7885160927264733*u*v*z2 + 0.22570026809960958*lnbara*u*v*z2 + 
     0.14088351757088213*u2*v*z2 + 0.03949638218639865*lnbara*u2*v*z2 + 
     0.3421460722890461*v2*z2 + 0.07466151326052549*lnbara*v2*z2 - 
     0.5971647409089695*u*v2*z2 - 0.1732162099150575*lnbara*u*v2*z2 + 
     0.029981938909783182*v3*z2 + 0.010002268725563646*lnbara*v3*z2 - 
     1.0639274511577628*w*z2 - 0.2604341376321621*lnbara*w*z2 - 
     0.7026134316358068*u*w*z2 - 0.22570026809960958*lnbara*u*w*z2 - 
     0.1500217343901253*u2*w*z2 - 0.03949638218639865*lnbara*u2*w*z2 - 
     1.4187638856780294*v*w*z2 - 0.4340568960536035*lnbara*v*w*z2 + 
     0.4719083667190125*u*v*w*z2 + 0.13861380499219164*lnbara*u*v*w*z2 + 
     0.23366341602051563*v2*w*z2 + 0.07821369824348423*lnbara*v2*w*z2 + 
     1.2099171176893446*w2*z2 + 0.359395382793078*lnbara*w2*z2 + 
     0.11341087241156884*u*w2*z2 + 0.03460240492286588*lnbara*u*w2*z2 - 
     0.08432966934787235*v*w2*z2 - 0.029311197001213464*lnbara*v*w2*z2 - 
     0.19765223368547283*w3*z2 - 0.05890476996783441*lnbara*w3*z2 - 
     0.2572621991056493*x*z2 + 0.037870009987388445*u*x*z2 - 
     0.003682012115837776*u2*x*z2 - 2.1392972928567677*v*x*z2 - 
     0.6571602770555028*lnbara*v*x*z2 + 0.2400067630542147*u*v*x*z2 + 
     0.07342181363031552*lnbara*u*v*x*z2 + 0.23366341602051563*v2*x*z2 + 
     0.07821369824348423*lnbara*v2*x*z2 + 2.210207833302107*w*x*z2 + 
     0.6571602770555028*lnbara*w*x*z2 - 0.24603076767859458*u*w*x*z2 - 
     0.07342181363031552*lnbara*u*w*x*z2 - 0.1586442374307426*v*w*x*z2 - 
     0.04890250124227076*lnbara*v*w*x*z2 - 0.08432966934787235*w2*x*z2 - 
     0.029311197001213464*lnbara*w2*x*z2 + 0.059441565485252916*x2*z2 - 
     0.003682012115837776*u*x2*z2 - 0.0034959415238012314*v*x2*z2 - 
     0.0034959415238012314*w*x2*z2 - 0.007187565837728457*x3*z2 - 
     0.2572621991056493*y*z2 + 0.03671811535644575*u*y*z2 - 
     0.0034959415238012314*u2*y*z2 + 1.106075413549916*v*y*z2 + 
     0.3437674709654954*lnbara*v*y*z2 + 0.2645638939826569*u*v*y*z2 + 
     0.07343720881724174*lnbara*u*v*y*z2 - 0.3921997968798069*v2*y*z2 - 
     0.1178863121858183*lnbara*v2*y*z2 - 1.0340129784736338*w*y*z2 - 
     0.3437674709654954*lnbara*w*y*z2 - 0.2704335518900037*u*w*y*z2 - 
     0.07343720881724174*lnbara*u*w*y*z2 + 0.35130413810496963*v*w*y*z2 + 
     0.0998330860923288*lnbara*v*w*y*z2 + 0.031244750707668238*w2*y*z2 + 
     0.0180532260934895*lnbara*w2*y*z2 + 0.051710236001772665*x*y*z2 - 
     0.0031815740344914314*u*x*y*z2 + 0.4847403793989428*v*x*y*z2 + 
     0.13429794906684606*lnbara*v*x*y*z2 - 0.49099871455493777*w*x*y*z2 - 
     0.13429794906684606*lnbara*w*x*y*z2 - 0.005642275295441937*x2*y*z2 + 
     0.059441565485252916*y2*z2 - 0.0034959415238012314*u*y2*z2 - 
     0.0020770130408900665*v*y2*z2 - 0.0018224798925292808*lnbara*v*y2*
      z2 - 0.005100940598748941*w*y2*z2 + 0.0018224798925292808*lnbara*w*
      y2*z2 - 0.005642275295441937*x*y2*z2 - 0.007187565837728457*y3*
      z2 - 0.6477405347676425*z3 + 0.07838349360209734*u*z3 - 
     0.007187565837728457*u2*z3 - 0.08220536163558813*v*z3 - 
     0.049480622580457954*lnbara*v*z3 - 0.2841442232687275*u*v*z3 - 
     0.08124448002734691*lnbara*u*v*z3 + 0.029981938909783182*v2*z3 + 
     0.010002268725563646*lnbara*v2*z3 + 0.21773437647406627*w*z3 + 
     0.049480622580457954*lnbara*w*z3 + 0.2734637028141685*u*w*z3 + 
     0.08124448002734691*lnbara*u*w*z3 + 0.15184270831316776*v*w*z3 + 
     0.04890250124227076*lnbara*v*w*z3 - 0.19765223368547283*w2*z3 - 
     0.05890476996783441*lnbara*w2*z3 + 0.07838349360209734*x*z3 - 
     0.004645460955055343*u*x*z3 + 0.37095314628030795*v*x*z3 + 
     0.1139718730378812*lnbara*v*x*z3 - 0.379627630631495*w*x*z3 - 
     0.1139718730378812*lnbara*w*x*z3 - 0.007187565837728457*x2*z3 + 
     0.07838349360209734*y*z3 - 0.004561039106467548*u*y*z3 - 
     0.4727100683983638*v*y*z3 - 0.13730604359507229*lnbara*v*y*z3 + 
     0.46395116219858895*w*y*z3 + 0.13730604359507229*lnbara*w*y*z3 - 
     0.00656707520983952*x*y*z3 - 0.007187565837728457*y2*z3 + 
     0.2312003934726287*z4 - 0.01103033535263963*u*z4 + 
     0.06407498022240934*v*z4 + 0.021288058617899477*lnbara*v*z4 - 
     0.0830149629493877*w*z4 - 0.021288058617899477*lnbara*w*z4 - 
     0.01103033535263963*x*z4 - 0.01103033535263963*y*z4 - 
     0.03583388093754629*z5;
 
  Hexp[6] = 1.2020569031595942 - 1.2020569031595942*u + 
     1.3191828527017075*u2 - 1.0795675579460708*u3 + 
     0.5780009836815717*u4 - 0.17916940468773146*u5 + 
     0.024344353217864104*u6 - 1.2020569031595942*v + 
     0.7006812478176802*u*v - 0.4287703318427488*u2*v + 
     0.19595873400524338*u3*v - 0.05515167676319815*u4*v + 
     0.007027953632525705*u5*v + 1.3191828527017075*v2 - 
     0.4287703318427488*u*v2 + 0.14860391371313228*u2*v2 - 
     0.03593782918864228*u3*v2 + 0.004169061579930593*u4*v2 - 
     1.0795675579460708*v3 + 0.19595873400524338*u*v3 - 
     0.03593782918864228*u2*v3 + 0.0035837624283894857*u3*v3 + 
     0.5780009836815717*v4 - 0.05515167676319815*u*v4 + 
     0.004169061579930593*u2*v4 - 0.17916940468773146*v5 + 
     0.007027953632525705*u*v5 + 0.024344353217864104*v6 - 
     1.2020569031595942*w + 0.7006812478176802*u*w - 
     0.4287703318427488*u2*w + 0.19595873400524338*u3*w - 
     0.05515167676319815*u4*w + 0.007027953632525705*u5*w + 
     0.7006812478176802*v*w - 0.262974857387689*u*v*w + 
     0.09467502496847112*u2*v*w - 0.023227304775276715*u3*v*w + 
     0.0027129425983649334*u4*v*w - 0.4287703318427488*v2*w + 
     0.09467502496847112*u*v2*w - 0.01841006057918888*u2*v2*w + 
     0.001880650990063473*u3*v2*w + 0.19595873400524338*v3*w - 
     0.023227304775276715*u*v3*w + 0.001880650990063473*u2*v3*w - 
     0.05515167676319815*v4*w + 0.0027129425983649334*u*v4*w + 
     0.007027953632525705*v5*w + 1.3191828527017075*w2 - 
     0.4287703318427488*u*w2 + 0.14860391371313228*u2*w2 - 
     0.03593782918864228*u3*w2 + 0.004169061579930593*u4*w2 - 
     0.4287703318427488*v*w2 + 0.09467502496847112*u*v*w2 - 
     0.01841006057918888*u2*v*w2 + 0.001880650990063473*u3*v*w2 + 
     0.14860391371313228*v2*w2 - 0.01841006057918888*u*v2*w2 + 
     0.0015174486936744638*u2*v2*w2 - 0.03593782918864228*v3*w2 + 
     0.001880650990063473*u*v3*w2 + 0.004169061579930593*v4*w2 - 
     1.0795675579460708*w3 + 0.19595873400524338*u*w3 - 
     0.03593782918864228*u2*w3 + 0.0035837624283894857*u3*w3 + 
     0.19595873400524338*v*w3 - 0.023227304775276715*u*v*w3 + 
     0.001880650990063473*u2*v*w3 - 0.03593782918864228*v2*w3 + 
     0.001880650990063473*u*v2*w3 + 0.0035837624283894857*v3*w3 + 
     0.5780009836815717*w4 - 0.05515167676319815*u*w4 + 
     0.004169061579930593*u2*w4 - 0.05515167676319815*v*w4 + 
     0.0027129425983649334*u*v*w4 + 0.004169061579930593*v2*w4 - 
     0.17916940468773146*w5 + 0.007027953632525705*u*w5 + 
     0.007027953632525705*v*w5 + 0.024344353217864104*w6 - 
     1.2020569031595942*x + 0.7006812478176802*u*x - 
     0.4287703318427488*u2*x + 0.19595873400524338*u3*x - 
     0.05515167676319815*u4*x + 0.007027953632525705*u5*x + 
     0.7006812478176802*v*x + 0.9399232832849783*u*v*x + 
     0.4340568960536035*lnbara*u*v*x + 0.9029759522398169*u2*v*x + 
     0.22352066577107735*lnbara*u2*v*x + 0.17752427122833048*u3*v*x + 
     0.05917269635600225*lnbara*u3*v*x + 0.024106166155729025*u4*v*x + 
     0.0052304795439763515*lnbara*u4*v*x - 1.700869602587337*v2*x - 
     0.4340568960536035*lnbara*v2*x - 1.6624918007146796*u*v2*x - 
     0.564250670249024*lnbara*u*v2*x - 0.7171950201286277*u2*v2*x - 
     0.19748191093199324*lnbara*u2*v2*x - 0.08627759743978855*u3*v2*x - 
     0.02432329879100571*lnbara*u3*v2*x + 0.5237190151099749*v3*x + 
     0.12370155645114488*lnbara*v3*x + 1.29961478071472*u*v3*x + 
     0.40622240013673455*lnbara*u*v3*x + 0.08694308134115901*u2*v3*x + 
     0.021986858303963977*lnbara*u2*v3*x - 0.397334765898689*v4*x - 
     0.10644029308949739*lnbara*v4*x - 0.22841481745724104*u*v4*x - 
     0.06907962057473943*lnbara*u*v4*x + 0.0487756611897797*v5*x + 
     0.013089806333713885*lnbara*v5*x + 0.5691938191238357*w*x - 
     1.5183456802708015*u*w*x - 0.4340568960536035*lnbara*u*w*x - 
     0.6819050738442709*u2*w*x - 0.22352066577107735*lnbara*u2*w*x - 
     0.23316484280986582*u3*w*x - 0.05917269635600225*lnbara*u3*w*x - 
     0.017520814568225136*u4*w*x - 0.0052304795439763515*lnbara*u4*w*x - 
     0.24624640952621318*v*w*x + 0.07362480745074675*u*v*w*x - 
     0.015907870172457157*u2*v*w*x + 0.0017051473997301786*u3*v*w*x + 
     5.2517028011488085*v2*w*x + 1.642900692638757*lnbara*v2*w*x - 
     1.1689689937010432*u*v2*w*x - 0.36710906815157757*lnbara*u*v2*w*x + 
     0.5646118353233481*u2*v2*w*x + 0.16529892872504193*lnbara*u2*v2*w*
      x - 1.803161592292574*v3*w*x - 0.5698593651894059*lnbara*v3*w*x - 
     0.025380422110629967*u*v3*w*x - 0.0022974655834007394*lnbara*u*v3*w*
      x + 0.5276000546267682*v4*w*x + 0.15634114484424524*lnbara*v4*w*x + 
     0.9491518611469659*w2*x + 0.4340568960536035*lnbara*w2*x + 
     1.8772484534413458*u*w2*x + 0.564250670249024*lnbara*u*w2*x + 
     0.6715039360324119*u2*w2*x + 0.19748191093199324*lnbara*u2*w2*x + 
     0.0911441314262025*u3*w2*x + 0.02432329879100571*lnbara*u3*w2*x - 
     5.0744264500354594*v*w2*x - 1.642900692638757*lnbara*v*w2*x + 
     1.138848970579144*u*v*w2*x + 0.36710906815157757*lnbara*u*v*w2*x - 
     0.5618212037342907*u2*v*w2*x - 0.16529892872504193*lnbara*u2*v*w2*
      x - 0.017479707619006156*v2*w2*x + 0.00130062566240163*u*v2*w2*x - 
     0.3213850461435467*v3*w2*x - 0.08664893308797812*lnbara*v3*w2*x - 
     0.1848964780137795*w3*x - 0.12370155645114488*lnbara*w3*x - 
     1.3530173829875152*u*w3*x - 0.40622240013673455*lnbara*u*w3*x - 
     0.0821346574113807*u2*w3*x - 0.021986858303963977*lnbara*u2*w3*x + 
     1.7597891705366386*v*w3*x + 0.5698593651894059*lnbara*v*w3*x + 
     0.02852820513246288*u*v*w3*x + 0.0022974655834007394*lnbara*u*v*w3*x + 
     0.3249693774147304*v2*w3*x + 0.08664893308797812*lnbara*v2*w3*x + 
     0.3026348522637972*w4*x + 0.10644029308949739*lnbara*w4*x + 
     0.23469657548791326*u*w4*x + 0.06907962057473943*lnbara*u*w4*x - 
     0.5225454785602572*v*w4*x - 0.15634114484424524*lnbara*v*w4*x - 
     0.03675623670681*w5*x - 0.013089806333713885*lnbara*w5*x + 
     1.3191828527017075*x2 - 0.4287703318427488*u*x2 + 
     0.14860391371313228*u2*x2 - 0.03593782918864228*u3*x2 + 
     0.004169061579930593*u4*x2 - 1.700869602587337*v*x2 - 
     0.4340568960536035*lnbara*v*x2 - 1.6624918007146796*u*v*x2 - 
     0.564250670249024*lnbara*u*v*x2 - 0.7171950201286277*u2*v*x2 - 
     0.19748191093199324*lnbara*u2*v*x2 - 0.08627759743978855*u3*v*x2 - 
     0.02432329879100571*lnbara*u3*v*x2 + 2.8750447180595793*v2*x2 + 
     0.898488456982695*lnbara*v2*x2 + 0.5382190246221227*u*v2*x2 + 
     0.1730120246143294*lnbara*u*v2*x2 + 0.479916000193387*u2*v2*x2 + 
     0.13727120803720705*lnbara*u2*v2*x2 - 0.9391738601208355*v3*x2 - 
     0.29452384983917207*lnbara*v3*x2 - 0.3238462135165472*u*v3*x2 - 
     0.09533400426449122*lnbara*u*v3*x2 + 0.29091212584308784*v4*x2 + 
     0.08588333011797569*lnbara*v4*x2 + 0.9491518611469659*w*x2 + 
     0.4340568960536035*lnbara*w*x2 + 1.8772484534413458*u*w*x2 + 
     0.564250670249024*lnbara*u*w*x2 + 0.6715039360324119*u2*w*x2 + 
     0.19748191093199324*lnbara*u2*w*x2 + 0.0911441314262025*u3*w*x2 + 
     0.02432329879100571*lnbara*u3*w*x2 - 3.366052674172739*v*w*x2 - 
     1.0851422401340087*lnbara*v*w*x2 + 2.244030329434903*u*v*w*x2 + 
     0.6930690249609582*lnbara*u*v*w*x2 - 0.5870531877615167*u2*v*w*x2 - 
     0.1732603915082422*lnbara*u2*v*w*x2 - 0.3972223492383506*v2*w*x2 - 
     0.1465559850060673*lnbara*v2*w*x2 - 0.20929117997162128*u*v2*w*x2 - 
     0.06205837953332923*lnbara*u*v2*w*x2 - 0.25301233477098106*v3*w*x2 - 
     0.062498180515692975*lnbara*v3*w*x2 + 0.8242562168640629*w2*x2 + 
     0.18665378315131373*lnbara*w2*x2 - 2.8414768629489666*u*w2*x2 - 
     0.8660810495752876*lnbara*u*w2*x2 + 0.11275072634761112*u2*w2*x2 + 
     0.035989183471035156*lnbara*u2*w2*x2 + 1.103138998233008*v*w2*x2 + 
     0.39106849121742115*lnbara*v*w2*x2 + 0.13104994444668858*u*v*w2*x2 + 
     0.030017221475897783*lnbara*u*v*w2*x2 + 0.4244378822310122*v2*w2*
      x2 + 0.11758155264728642*lnbara*v2*w2*x2 + 
     0.14157447061094622*w3*x2 + 0.05001134362781823*lnbara*w3*x2 + 
     0.4093655691718876*u*w3*x2 + 0.12737516232192267*lnbara*u*w3*x2 - 
     0.4837702578695747*v*w3*x2 - 0.14799273546990963*lnbara*v*w3*x2 + 
     0.03289810142239334*w4*x2 + 0.007026033220340475*lnbara*w4*x2 - 
     1.0795675579460708*x3 + 0.19595873400524338*u*x3 - 
     0.03593782918864228*u2*x3 + 0.0035837624283894857*u3*x3 + 
     0.5237190151099749*v*x3 + 0.12370155645114488*lnbara*v*x3 + 
     1.29961478071472*u*v*x3 + 0.40622240013673455*lnbara*u*v*x3 + 
     0.08694308134115901*u2*v*x3 + 0.021986858303963977*lnbara*u2*v*x3 - 
     0.9391738601208355*v2*x3 - 0.29452384983917207*lnbara*v2*x3 - 
     0.3238462135165472*u*v2*x3 - 0.09533400426449122*lnbara*u*v2*x3 + 
     0.043640908015127135*v3*x3 + 0.01752291016105757*lnbara*v3*x3 - 
     0.1848964780137795*w*x3 - 0.12370155645114488*lnbara*w*x3 - 
     1.3530173829875152*u*w*x3 - 0.40622240013673455*lnbara*u*w*x3 - 
     0.0821346574113807*u2*w*x3 - 0.021986858303963977*lnbara*u2*w*x3 + 
     0.7184614571972799*v*w*x3 + 0.24451250621135381*lnbara*v*w*x3 - 
     0.07931329913612384*u*v*w*x3 - 0.03204115805743145*lnbara*u*v*w*x3 + 
     0.5112934663833444*v2*w*x3 + 0.14720183048123822*lnbara*v2*w*x3 + 
     0.14157447061094622*w2*x3 + 0.05001134362781823*lnbara*w2*x3 + 
     0.4093655691718876*u*w2*x3 + 0.12737516232192267*lnbara*u*w2*x3 - 
     0.569080649680904*v*w2*x3 - 0.16941649028491496*lnbara*v*w2*x3 + 
     0.02313083477136829*w3*x3 + 0.004691749642619168*lnbara*w3*x3 + 
     0.5780009836815717*x4 - 0.05515167676319815*u*x4 + 
     0.004169061579930593*u2*x4 - 0.397334765898689*v*x4 - 
     0.10644029308949739*lnbara*v*x4 - 0.22841481745724104*u*v*x4 - 
     0.06907962057473943*lnbara*u*v*x4 + 0.29091212584308784*v2*x4 + 
     0.08588333011797569*lnbara*v2*x4 + 0.3026348522637972*w*x4 + 
     0.10644029308949739*lnbara*w*x4 + 0.23469657548791326*u*w*x4 + 
     0.06907962057473943*lnbara*u*w*x4 - 0.31472121563065336*v*w*x4 - 
     0.09290936333831616*lnbara*v*w*x4 + 0.03289810142239334*w2*x4 + 
     0.007026033220340475*lnbara*w2*x4 - 0.17916940468773146*x5 + 
     0.007027953632525705*u*x5 + 0.0487756611897797*v*x5 + 
     0.013089806333713885*lnbara*v*x5 - 0.03675623670681*w*x5 - 
     0.013089806333713885*lnbara*w*x5 + 0.024344353217864104*x6 - 
     1.2020569031595942*y + 0.5691938191238357*u*y - 
     0.32294740959762247*u2*y + 0.14286380309095198*u3*y - 
     0.03954823687169365*u4*y + 0.004991470850443998*u5*y + 
     0.7006812478176802*v*y - 0.24624640952621318*u*v*y + 
     0.08548106272223456*u2*v*y - 0.020567226223597537*u3*v*y + 
     0.0023754912548396108*u4*v*y - 0.4287703318427488*v2*y + 
     0.09179528839111438*u*v2*y - 0.017479707619006156*u2*v2*y + 
     0.0017631106072740343*u3*v2*y + 0.19595873400524338*v3*y - 
     0.022805195532337737*u*v3*y + 0.001821220663909673*u2*v3*y - 
     0.05515167676319815*v4*y + 0.0026790848116713074*u*v4*y + 
     0.007027953632525705*v5*y + 0.7006812478176802*w*y - 
     0.24624640952621318*u*w*y + 0.08548106272223456*u2*w*y - 
     0.020567226223597537*u3*w*y + 0.0023754912548396108*u4*w*y - 
     0.3321759874596101*v*w*y + 0.07362480745074675*u*v*w*y - 
     0.01421215294944225*u2*v*w*y + 0.001442635622102732*u3*v*w*y + 
     0.12927559000443167*v2*w*y - 0.015907870172457157*u*v2*w*y + 
     0.00130062566240163*u2*v2*w*y - 0.032835376049197604*v3*w*y + 
     0.0017051473997301786*u*v3*w*y + 0.0039062667758325825*v4*w*y - 
     0.4287703318427488*w2*y + 0.09179528839111438*u*w2*y - 
     0.017479707619006156*u2*w2*y + 0.0017631106072740343*u3*w2*y + 
     0.12927559000443167*v*w2*y - 0.015907870172457157*u*v*w2*y + 
     0.00130062566240163*u2*v*w2*y - 0.028211376477209685*v2*w2*y + 
     0.001490005926655855*u*v2*w2*y + 0.0030453133225042724*v3*w2*y + 
     0.19595873400524338*w3*y - 0.022805195532337737*u*w3*y + 
     0.001821220663909673*u2*w3*y - 0.032835376049197604*v*w3*y + 
     0.0017051473997301786*u*v*w3*y + 0.0030453133225042724*v2*w3*y - 
     0.05515167676319815*w4*y + 0.0026790848116713074*u*w4*y + 
     0.0039062667758325825*v*w4*y + 0.007027953632525705*w5*y + 
     0.7006812478176802*x*y - 0.24624640952621318*u*x*y + 
     0.08548106272223456*u2*x*y - 0.020567226223597537*u3*x*y + 
     0.0023754912548396108*u4*x*y + 3.553322954846076*v*x*y + 
     1.3021706881608104*lnbara*v*x*y + 1.8769958336226564*u*v*x*y + 
     0.5077906612931307*lnbara*u*v*x*y + 0.5620711026947999*u2*v*x*y + 
     0.16536822311781155*lnbara*u2*v*x*y + 0.06775115910844728*u3*v*x*y + 
     0.017198580632115178*lnbara*u3*v*x*y - 2.4417959999484613*v2*x*y - 
     0.8594186774137386*lnbara*v2*x*y - 1.2909700854356503*u*v2*x*y - 
     0.3671860440862087*lnbara*u*v2*x*y - 0.222192079524089*u2*v2*x*y - 
     0.06208972089809539*lnbara*u2*v2*x*y + 2.2053341079970514*v3*x*y + 
     0.6865302179753614*lnbara*v3*x*y + 0.19254447471177424*u*v3*x*y + 
     0.05112899772410982*lnbara*u*v3*x*y - 0.412831813275259*v4*x*y - 
     0.12512755974621306*lnbara*v4*x*y - 4.062544221759978*w*x*y - 
     1.3021706881608104*lnbara*w*x*y - 1.7373962401659306*u*w*x*y - 
     0.5077906612931307*lnbara*u*w*x*y - 0.5908953276665939*u2*w*x*y - 
     0.16536822311781155*lnbara*u2*w*x*y - 0.064738719421504*u3*w*x*y - 
     0.017198580632115178*lnbara*u3*w*x*y + 0.07362480745074675*v*w*x*y - 
     0.012213108548872703*u*v*w*x*y + 0.001111492150387925*u2*v*w*x*y - 
     2.343078615218984*v2*w*x*y - 0.6714897453342303*lnbara*v2*w*x*y + 
     0.4498241458617747*u*v2*w*x*y + 0.13547920846092296*lnbara*u*v2*w*x*
      y + 0.12539708324984536*v3*w*x*y + 0.034595562617565336*lnbara*v3*w*x*
      y + 2.621952087639167*w2*x*y + 0.8594186774137386*lnbara*w2*x*y + 
     1.2616217958989164*u*w2*x*y + 0.3671860440862087*lnbara*u*w2*x*y + 
     0.22484058072370788*u2*w2*x*y + 0.06208972089809539*lnbara*u2*w2*x*
      y + 2.311786939439009*v*w2*x*y + 0.6714897453342303*lnbara*v*w2*x*y - 
     0.4475358708690333*u*v*w2*x*y - 0.13547920846092296*lnbara*u*v*w2*x*
      y + 0.00146144490120512*v2*w2*x*y - 2.2491286389959257*w3*x*y - 
     0.6865302179753614*lnbara*w3*x*y - 0.18944564848149062*u*w3*x*y - 
     0.05112899772410982*lnbara*u*w3*x*y - 0.12207317499372534*v*w3*x*y - 
     0.034595562617565336*lnbara*v*w3*x*y + 0.4179202471284635*w4*x*y + 
     0.12512755974621306*lnbara*w4*x*y - 0.4287703318427488*x2*y + 
     0.09179528839111438*u*x2*y - 0.017479707619006156*u2*x2*y + 
     0.0017631106072740343*u3*x2*y - 2.4417959999484613*v*x2*y - 
     0.8594186774137386*lnbara*v*x2*y - 1.2909700854356503*u*v*x2*y - 
     0.3671860440862087*lnbara*u*v*x2*y - 0.222192079524089*u2*v*x2*y - 
     0.06208972089809539*lnbara*u2*v*x2*y + 0.1411793984604333*v2*x2*y + 
     0.09026613046744748*lnbara*v2*x2*y + 0.8636086160336289*u*v2*x2*y + 
     0.24856634396486232*lnbara*u*v2*x2*y - 0.28983228539271344*v3*x2*y - 
     0.09094784681695743*lnbara*v3*x2*y + 2.621952087639167*w*x2*y + 
     0.8594186774137386*lnbara*w*x2*y + 1.2616217958989164*u*w*x2*y + 
     0.3671860440862087*lnbara*u*w*x2*y + 0.22484058072370788*u2*w*x2*y + 
     0.06208972089809539*lnbara*u2*w*x2*y + 1.6733264521145743*v*w*x2*y + 
     0.499165430461644*lnbara*v*w*x2*y - 0.7575451375965339*u*v*w*x2*y - 
     0.21838360971474802*lnbara*u*v*w*x2*y + 0.029245523131395473*v2*w*x2*
      y + 0.008946886688429884*lnbara*v2*w*x2*y - 
     1.8627603909108525*w2*x2*y - 0.5894315609290915*lnbara*w2*x2*y - 
     0.10231789367095857*u*w2*x2*y - 0.030182734250114304*lnbara*u*w2*x2*
      y - 0.034386556943485275*v*w2*x2*y - 0.01584237786733671*lnbara*v*w2*
      x2*y + 0.3009259577639667*w3*x2*y + 0.09784333799586425*lnbara*w3*
      x2*y + 0.19595873400524338*x3*y - 0.022805195532337737*u*x3*y + 
     0.001821220663909673*u2*x3*y + 2.2053341079970514*v*x3*y + 
     0.6865302179753614*lnbara*v*x3*y + 0.19254447471177424*u*v*x3*y + 
     0.05112899772410982*lnbara*u*v*x3*y - 0.28983228539271344*v2*x3*y - 
     0.09094784681695743*lnbara*v2*x3*y - 2.2491286389959257*w*x3*y - 
     0.6865302179753614*lnbara*w*x3*y - 0.18944564848149062*u*w*x3*y - 
     0.05112899772410982*lnbara*u*w*x3*y - 0.0062284034187645365*v*w*x3*y - 
     0.006895491178906826*lnbara*v*w*x3*y + 0.3009259577639667*w2*x3*y + 
     0.09784333799586425*lnbara*w2*x3*y - 0.05515167676319815*x4*y + 
     0.0026790848116713074*u*x4*y - 0.412831813275259*v*x4*y - 
     0.12512755974621306*lnbara*v*x4*y + 0.4179202471284635*w*x4*y + 
     0.12512755974621306*lnbara*w*x4*y + 0.007027953632525705*x5*y + 
     1.3191828527017075*y2 - 0.32294740959762247*u*y2 + 
     0.09916328431553666*u2*y2 - 0.022632876900369598*u3*y2 + 
     0.0025444588000576086*u4*y2 - 0.4287703318427488*v*y2 + 
     0.08548106272223456*u*v*y2 - 0.01563232680721393*u2*v*y2 + 
     0.0015419823403225055*u3*v*y2 + 0.14860391371313228*v2*y2 - 
     0.017479707619006156*u*v2*y2 + 0.0013887934717172665*u2*v2*y2 - 
     0.03593782918864228*v3*y2 + 0.001821220663909673*u*v3*y2 + 
     0.004169061579930593*v4*y2 - 0.4287703318427488*w*y2 + 
     0.08548106272223456*u*w*y2 - 0.01563232680721393*u2*w*y2 + 
     0.0015419823403225055*u3*w*y2 + 0.12927559000443167*v*w*y2 - 
     0.015383805607517485*u*v*w*y2 + 0.001229379566375842*u2*v*w*y2 - 
     0.028211376477209685*v2*w*y2 + 0.00146144490120512*u*v2*w*y2 + 
     0.0030453133225042724*v3*w*y2 + 0.14860391371313228*w2*y2 - 
     0.017479707619006156*u*w2*y2 + 0.0013887934717172665*u2*w2*y2 - 
     0.028211376477209685*v*w2*y2 + 0.00146144490120512*u*v*w2*y2 + 
     0.002763300406558978*v2*w2*y2 - 0.03593782918864228*w3*y2 + 
     0.001821220663909673*u*w3*y2 + 0.0030453133225042724*v*w3*y2 + 
     0.004169061579930593*w4*y2 - 0.4287703318427488*x*y2 + 
     0.08548106272223456*u*x*y2 - 0.01563232680721393*u2*x*y2 + 
     0.0015419823403225055*u3*x*y2 - 1.458515156634267*v*x*y2 - 
     0.48831400806030395*lnbara*v*x*y2 + 0.15508634310271682*u*v*x*y2 + 
     0.06121320376147517*lnbara*u*v*x*y2 + 0.060610376887719246*u2*v*x*
      y2 + 0.015697459289112238*lnbara*u2*v*x*y2 - 
     0.027023436237519106*v2*x*y2 + 0.009112399462646404*lnbara*v2*x*y2 - 
     0.1308877099919966*u*v2*x*y2 - 0.03812489698291478*lnbara*u*v2*x*
      y2 - 0.14302771932672184*v3*x*y2 - 0.041476677259681335*lnbara*v3*x*
      y2 + 1.6449854699938522*w*x*y2 + 0.48831400806030395*lnbara*w*x*y2 - 
     0.18273891541643567*u*w*x*y2 - 0.06121320376147517*lnbara*u*w*x*y2 - 
     0.05823691063225072*u2*w*x*y2 - 0.015697459289112238*lnbara*u2*w*x*
      y2 - 0.015907870172457157*v*w*x*y2 + 0.001111492150387925*u*v*w*x*
      y2 + 0.7234333761756083*v2*w*x*y2 + 0.20371729223242238*lnbara*v2*w*
      x*y2 - 0.008866331960675933*w2*x*y2 - 0.009112399462646404*lnbara*
      w2*x*y2 + 0.13337539195281198*u*w2*x*y2 + 
     0.03812489698291478*lnbara*u*w2*x*y2 - 0.7204819253477474*v*w2*x*
      y2 - 0.20371729223242238*lnbara*v*w2*x*y2 + 
     0.14667148092405935*w3*x*y2 + 0.041476677259681335*lnbara*w3*x*y2 + 
     0.14860391371313228*x2*y2 - 0.017479707619006156*u*x2*y2 + 
     0.0013887934717172665*u2*x2*y2 - 0.027023436237519106*v*x2*y2 + 
     0.009112399462646404*lnbara*v*x2*y2 - 0.1308877099919966*u*v*x2*y2 - 
     0.03812489698291478*lnbara*u*v*x2*y2 + 0.3378573257997137*v2*x2*
      y2 + 0.09448041654628751*lnbara*v2*x2*y2 - 
     0.008866331960675933*w*x2*y2 - 0.009112399462646404*lnbara*w*x2*y2 + 
     0.13337539195281198*u*w*x2*y2 + 0.03812489698291478*lnbara*u*w*x2*
      y2 - 0.4477982839144112*v*w*x2*y2 - 0.12659992801563436*lnbara*v*w*
      x2*y2 + 0.1141478259424908*w2*x2*y2 + 0.03211951146934684*lnbara*
      w2*x2*y2 - 0.03593782918864228*x3*y2 + 0.001821220663909673*u*x3*
      y2 - 0.14302771932672184*v*x3*y2 - 0.041476677259681335*lnbara*v*x3*
      y2 + 0.14667148092405935*w*x3*y2 + 0.041476677259681335*lnbara*w*x3*
      y2 + 0.004169061579930593*x4*y2 - 1.0795675579460708*y3 + 
     0.14286380309095198*u*y3 - 0.022632876900369598*u2*y3 + 
     0.0020957041129497137*u3*y3 + 0.19595873400524338*v*y3 - 
     0.020567226223597537*u*v*y3 + 0.0015419823403225055*u2*v*y3 - 
     0.03593782918864228*v2*y3 + 0.0017631106072740343*u*v2*y3 + 
     0.0035837624283894857*v3*y3 + 0.19595873400524338*w*y3 - 
     0.020567226223597537*u*w*y3 + 0.0015419823403225055*u2*w*y3 - 
     0.032835376049197604*v*w*y3 + 0.0016187608563898235*u*v*w*y3 + 
     0.0030453133225042724*v2*w*y3 - 0.03593782918864228*w2*y3 + 
     0.0017631106072740343*u*w2*y3 + 0.0030453133225042724*v*w2*y3 + 
     0.0035837624283894857*w3*y3 + 0.19595873400524338*x*y3 - 
     0.020567226223597537*u*x*y3 + 0.0015419823403225055*u2*x*y3 + 
     0.9672844851536025*v*x*y3 + 0.2929884048361824*lnbara*v*x*y3 + 
     0.0816735862630184*u*v*x*y3 + 0.019963821863986493*lnbara*u*v*x*y3 - 
     0.12145181321698113*v2*x*y3 - 0.037545222670016885*lnbara*v2*x*y3 - 
     1.013316985461217*w*x*y3 - 0.2929884048361824*lnbara*w*x*y3 - 
     0.07883727181036221*u*w*x*y3 - 0.019963821863986493*lnbara*u*w*x*y3 + 
     0.0017051473997301786*v*w*x*y3 + 0.1251536848709543*w2*x*y3 + 
     0.037545222670016885*lnbara*w2*x*y3 - 0.03593782918864228*x2*y3 + 
     0.0017631106072740343*u*x2*y3 - 0.12145181321698113*v*x2*y3 - 
     0.037545222670016885*lnbara*v*x2*y3 + 0.1251536848709543*w*x2*y3 + 
     0.037545222670016885*lnbara*w*x2*y3 + 0.0035837624283894857*x3*y3 + 
     0.5780009836815717*y4 - 0.03954823687169365*u*y4 + 
     0.0025444588000576086*u2*y4 - 0.05515167676319815*v*y4 + 
     0.0023754912548396108*u*v*y4 + 0.004169061579930593*v2*y4 - 
     0.05515167676319815*w*y4 + 0.0023754912548396108*u*w*y4 + 
     0.0039062667758325825*v*w*y4 + 0.004169061579930593*w2*y4 - 
     0.05515167676319815*x*y4 + 0.0023754912548396108*u*x*y4 - 
     0.14164027833703918*v*x*y4 - 0.040692834005025325*lnbara*v*x*y4 + 
     0.14703230574707543*w*x*y4 + 0.040692834005025325*lnbara*w*x*y4 + 
     0.004169061579930593*x2*y4 - 0.17916940468773146*y5 + 
     0.004991470850443998*u*y5 + 0.007027953632525705*v*y5 + 
     0.007027953632525705*w*y5 + 0.007027953632525705*x*y5 + 
     0.024344353217864104*y6 - 1.2020569031595942*z + 
     0.7006812478176802*u*z - 0.4287703318427488*u2*z + 
     0.19595873400524338*u3*z - 0.05515167676319815*u4*z + 
     0.007027953632525705*u5*z + 0.5691938191238357*v*z - 
     1.5183456802708015*u*v*z - 0.4340568960536035*lnbara*u*v*z - 
     0.6819050738442709*u2*v*z - 0.22352066577107735*lnbara*u2*v*z - 
     0.23316484280986582*u3*v*z - 0.05917269635600225*lnbara*u3*v*z - 
     0.017520814568225136*u4*v*z - 0.0052304795439763515*lnbara*u4*v*z + 
     0.9491518611469659*v2*z + 0.4340568960536035*lnbara*v2*z + 
     1.8772484534413458*u*v2*z + 0.564250670249024*lnbara*u*v2*z + 
     0.6715039360324119*u2*v2*z + 0.19748191093199324*lnbara*u2*v2*z + 
     0.0911441314262025*u3*v2*z + 0.02432329879100571*lnbara*u3*v2*z - 
     0.1848964780137795*v3*z - 0.12370155645114488*lnbara*v3*z - 
     1.3530173829875152*u*v3*z - 0.40622240013673455*lnbara*u*v3*z - 
     0.0821346574113807*u2*v3*z - 0.021986858303963977*lnbara*u2*v3*z + 
     0.3026348522637972*v4*z + 0.10644029308949739*lnbara*v4*z + 
     0.23469657548791326*u*v4*z + 0.06907962057473943*lnbara*u*v4*z - 
     0.03675623670681*v5*z - 0.013089806333713885*lnbara*v5*z + 
     0.7006812478176802*w*z + 0.9399232832849783*u*w*z + 
     0.4340568960536035*lnbara*u*w*z + 0.9029759522398169*u2*w*z + 
     0.22352066577107735*lnbara*u2*w*z + 0.17752427122833048*u3*w*z + 
     0.05917269635600225*lnbara*u3*w*z + 0.024106166155729025*u4*w*z + 
     0.0052304795439763515*lnbara*u4*w*z - 0.24624640952621318*v*w*z + 
     0.07362480745074675*u*v*w*z - 0.015907870172457157*u2*v*w*z + 
     0.0017051473997301786*u3*v*w*z - 5.0744264500354594*v2*w*z - 
     1.642900692638757*lnbara*v2*w*z + 1.138848970579144*u*v2*w*z + 
     0.36710906815157757*lnbara*u*v2*w*z - 0.5618212037342907*u2*v2*w*z - 
     0.16529892872504193*lnbara*u2*v2*w*z + 1.7597891705366386*v3*w*z + 
     0.5698593651894059*lnbara*v3*w*z + 0.02852820513246288*u*v3*w*z + 
     0.0022974655834007394*lnbara*u*v3*w*z - 0.5225454785602572*v4*w*z - 
     0.15634114484424524*lnbara*v4*w*z - 1.700869602587337*w2*z - 
     0.4340568960536035*lnbara*w2*z - 1.6624918007146796*u*w2*z - 
     0.564250670249024*lnbara*u*w2*z - 0.7171950201286277*u2*w2*z - 
     0.19748191093199324*lnbara*u2*w2*z - 0.08627759743978855*u3*w2*z - 
     0.02432329879100571*lnbara*u3*w2*z + 5.2517028011488085*v*w2*z + 
     1.642900692638757*lnbara*v*w2*z - 1.1689689937010432*u*v*w2*z - 
     0.36710906815157757*lnbara*u*v*w2*z + 0.5646118353233481*u2*v*w2*z + 
     0.16529892872504193*lnbara*u2*v*w2*z - 0.017479707619006156*v2*w2*
      z + 0.00130062566240163*u*v2*w2*z + 0.3249693774147304*v3*w2*z + 
     0.08664893308797812*lnbara*v3*w2*z + 0.5237190151099749*w3*z + 
     0.12370155645114488*lnbara*w3*z + 1.29961478071472*u*w3*z + 
     0.40622240013673455*lnbara*u*w3*z + 0.08694308134115901*u2*w3*z + 
     0.021986858303963977*lnbara*u2*w3*z - 1.803161592292574*v*w3*z - 
     0.5698593651894059*lnbara*v*w3*z - 0.025380422110629967*u*v*w3*z - 
     0.0022974655834007394*lnbara*u*v*w3*z - 0.3213850461435467*v2*w3*z - 
     0.08664893308797812*lnbara*v2*w3*z - 0.397334765898689*w4*z - 
     0.10644029308949739*lnbara*w4*z - 0.22841481745724104*u*w4*z - 
     0.06907962057473943*lnbara*u*w4*z + 0.5276000546267682*v*w4*z + 
     0.15634114484424524*lnbara*v*w4*z + 0.0487756611897797*w5*z + 
     0.013089806333713885*lnbara*w5*z + 0.7006812478176802*x*z - 
     0.262974857387689*u*x*z + 0.09467502496847112*u2*x*z - 
     0.023227304775276715*u3*x*z + 0.0027129425983649334*u4*x*z - 
     0.24624640952621318*v*x*z + 0.07362480745074675*u*v*x*z - 
     0.015907870172457157*u2*v*x*z + 0.0017051473997301786*u3*v*x*z - 
     3.366052674172739*v2*x*z - 1.0851422401340087*lnbara*v2*x*z + 
     2.244030329434903*u*v2*x*z + 0.6930690249609582*lnbara*u*v2*x*z - 
     0.5870531877615167*u2*v2*x*z - 0.1732603915082422*lnbara*u2*v2*x*z + 
     0.7184614571972799*v3*x*z + 0.24451250621135381*lnbara*v3*x*z - 
     0.07931329913612384*u*v3*x*z - 0.03204115805743145*lnbara*u*v3*x*z - 
     0.31472121563065336*v4*x*z - 0.09290936333831616*lnbara*v4*x*z - 
     0.24624640952621318*w*x*z + 0.07362480745074675*u*w*x*z - 
     0.015907870172457157*u2*w*x*z + 0.0017051473997301786*u3*w*x*z + 
     6.969042259795925*v*w*x*z + 2.1702844802680175*lnbara*v*w*x*z - 
     4.531041378633713*u*v*w*x*z - 1.3861380499219165*lnbara*u*v*w*x*z + 
     1.1782060481677972*u2*v*w*x*z + 0.3465207830164844*lnbara*u2*v*w*x*z - 
     0.752469102785154*v2*w*x*z - 0.24451250621135381*lnbara*v2*w*x*z + 
     0.08204355214290159*u*v2*w*x*z + 0.03204115805743145*lnbara*u*v2*w*x*
      z + 0.7415394744753414*v3*w*x*z + 0.2104909159856026*lnbara*v3*w*x*
      z - 3.366052674172739*w2*x*z - 1.0851422401340087*lnbara*w2*x*z + 
     2.244030329434903*u*w2*x*z + 0.6930690249609582*lnbara*u*w2*x*z - 
     0.5870531877615167*u2*w2*x*z - 0.1732603915082422*lnbara*u2*w2*x*z - 
     0.752469102785154*v*w2*x*z - 0.24451250621135381*lnbara*v*w2*x*z + 
     0.08204355214290159*u*v*w2*x*z + 0.03204115805743145*lnbara*u*v*w2*x*
      z - 0.8449111212201761*v2*w2*x*z - 0.23516310529457285*lnbara*v2*w2*
      x*z + 0.7184614571972799*w3*x*z + 0.24451250621135381*lnbara*w3*x*z - 
     0.07931329913612384*u*w3*x*z - 0.03204115805743145*lnbara*u*w3*x*z + 
     0.7415394744753414*v*w3*x*z + 0.2104909159856026*lnbara*v*w3*x*z - 
     0.31472121563065336*w4*x*z - 0.09290936333831616*lnbara*w4*x*z - 
     0.4287703318427488*x2*z + 0.09467502496847112*u*x2*z - 
     0.01841006057918888*u2*x2*z + 0.001880650990063473*u3*x2*z + 
     5.2517028011488085*v*x2*z + 1.642900692638757*lnbara*v*x2*z - 
     1.1689689937010432*u*v*x2*z - 0.36710906815157757*lnbara*u*v*x2*z + 
     0.5646118353233481*u2*v*x2*z + 0.16529892872504193*lnbara*u2*v*x2*
      z - 0.3972223492383506*v2*x2*z - 0.1465559850060673*lnbara*v2*x2*
      z - 0.20929117997162128*u*v2*x2*z - 0.06205837953332923*lnbara*u*v2*
      x2*z + 0.5112934663833444*v3*x2*z + 0.14720183048123822*lnbara*v3*
      x2*z - 5.0744264500354594*w*x2*z - 1.642900692638757*lnbara*w*x2*z + 
     1.138848970579144*u*w*x2*z + 0.36710906815157757*lnbara*u*w*x2*z - 
     0.5618212037342907*u2*w*x2*z - 0.16529892872504193*lnbara*u2*w*x2*
      z - 0.752469102785154*v*w*x2*z - 0.24451250621135381*lnbara*v*w*x2*
      z + 0.08204355214290159*u*v*w*x2*z + 0.03204115805743145*lnbara*u*v*w*
      x2*z - 1.0464511532205378*v2*w*x2*z - 0.2896065773402419*lnbara*v2*
      w*x2*z + 1.103138998233008*w2*x2*z + 0.39106849121742115*lnbara*w2*
      x2*z + 0.13104994444668858*u*w2*x2*z + 0.030017221475897783*lnbara*u*
      w2*x2*z + 1.1098745724532006*v*w2*x2*z + 0.31182123714391863*lnbara*
      v*w2*x2*z - 0.569080649680904*w3*x2*z - 0.16941649028491496*lnbara*
      w3*x2*z + 0.19595873400524338*x3*z - 0.023227304775276715*u*x3*z + 
     0.001880650990063473*u2*x3*z - 1.803161592292574*v*x3*z - 
     0.5698593651894059*lnbara*v*x3*z - 0.025380422110629967*u*v*x3*z - 
     0.0022974655834007394*lnbara*u*v*x3*z - 0.25301233477098106*v2*x3*z - 
     0.062498180515692975*lnbara*v2*x3*z + 1.7597891705366386*w*x3*z + 
     0.5698593651894059*lnbara*w*x3*z + 0.02852820513246288*u*w*x3*z + 
     0.0022974655834007394*lnbara*u*w*x3*z + 0.7415394744753414*v*w*x3*z + 
     0.2104909159856026*lnbara*v*w*x3*z - 0.4837702578695747*w2*x3*z - 
     0.14799273546990963*lnbara*w2*x3*z - 0.05515167676319815*x4*z + 
     0.0027129425983649334*u*x4*z + 0.5276000546267682*v*x4*z + 
     0.15634114484424524*lnbara*v*x4*z - 0.5225454785602572*w*x4*z - 
     0.15634114484424524*lnbara*w*x4*z + 0.007027953632525705*x5*z + 
     0.7006812478176802*y*z - 0.24624640952621318*u*y*z + 
     0.08548106272223456*u2*y*z - 0.020567226223597537*u3*y*z + 
     0.0023754912548396108*u4*y*z - 4.062544221759978*v*y*z - 
     1.3021706881608104*lnbara*v*y*z - 1.7373962401659306*u*v*y*z - 
     0.5077906612931307*lnbara*u*v*y*z - 0.5908953276665939*u2*v*y*z - 
     0.16536822311781155*lnbara*u2*v*y*z - 0.064738719421504*u3*v*y*z - 
     0.017198580632115178*lnbara*u3*v*y*z + 2.621952087639167*v2*y*z + 
     0.8594186774137386*lnbara*v2*y*z + 1.2616217958989164*u*v2*y*z + 
     0.3671860440862087*lnbara*u*v2*y*z + 0.22484058072370788*u2*v2*y*z + 
     0.06208972089809539*lnbara*u2*v2*y*z - 2.2491286389959257*v3*y*z - 
     0.6865302179753614*lnbara*v3*y*z - 0.18944564848149062*u*v3*y*z - 
     0.05112899772410982*lnbara*u*v3*y*z + 0.4179202471284635*v4*y*z + 
     0.12512755974621306*lnbara*v4*y*z + 3.553322954846076*w*y*z + 
     1.3021706881608104*lnbara*w*y*z + 1.8769958336226564*u*w*y*z + 
     0.5077906612931307*lnbara*u*w*y*z + 0.5620711026947999*u2*w*y*z + 
     0.16536822311781155*lnbara*u2*w*y*z + 0.06775115910844728*u3*w*y*z + 
     0.017198580632115178*lnbara*u3*w*y*z + 0.07362480745074675*v*w*y*z - 
     0.012213108548872703*u*v*w*y*z + 0.001111492150387925*u2*v*w*y*z + 
     2.311786939439009*v2*w*y*z + 0.6714897453342303*lnbara*v2*w*y*z - 
     0.4475358708690333*u*v2*w*y*z - 0.13547920846092296*lnbara*u*v2*w*y*
      z - 0.12207317499372534*v3*w*y*z - 0.034595562617565336*lnbara*v3*w*y*
      z - 2.4417959999484613*w2*y*z - 0.8594186774137386*lnbara*w2*y*z - 
     1.2909700854356503*u*w2*y*z - 0.3671860440862087*lnbara*u*w2*y*z - 
     0.222192079524089*u2*w2*y*z - 0.06208972089809539*lnbara*u2*w2*y*z - 
     2.343078615218984*v*w2*y*z - 0.6714897453342303*lnbara*v*w2*y*z + 
     0.4498241458617747*u*v*w2*y*z + 0.13547920846092296*lnbara*u*v*w2*y*
      z + 0.00146144490120512*v2*w2*y*z + 2.2053341079970514*w3*y*z + 
     0.6865302179753614*lnbara*w3*y*z + 0.19254447471177424*u*w3*y*z + 
     0.05112899772410982*lnbara*u*w3*y*z + 0.12539708324984536*v*w3*y*z + 
     0.034595562617565336*lnbara*v*w3*y*z - 0.412831813275259*w4*y*z - 
     0.12512755974621306*lnbara*w4*y*z - 0.3321759874596101*x*y*z + 
     0.07362480745074675*u*x*y*z - 0.01421215294944225*u2*x*y*z + 
     0.001442635622102732*u3*x*y*z + 0.07362480745074675*v*x*y*z - 
     0.012213108548872703*u*v*x*y*z + 0.001111492150387925*u2*v*x*y*z + 
     1.6733264521145743*v2*x*y*z + 0.499165430461644*lnbara*v2*x*y*z - 
     0.7575451375965339*u*v2*x*y*z - 0.21838360971474802*lnbara*u*v2*x*y*
      z - 0.0062284034187645365*v3*x*y*z - 0.006895491178906826*lnbara*v3*x*
      y*z + 0.07362480745074675*w*x*y*z - 0.012213108548872703*u*w*x*y*z + 
     0.001111492150387925*u2*w*x*y*z - 3.3872903186769054*v*w*x*y*z - 
     0.998330860923288*lnbara*v*w*x*y*z + 1.518280865154906*u*v*w*x*y*z + 
     0.43676721942949603*lnbara*u*v*w*x*y*z + 0.008782531191255194*v2*w*x*y*
      z + 0.006895491178906826*lnbara*v2*w*x*y*z + 
     1.6733264521145743*w2*x*y*z + 0.499165430461644*lnbara*w2*x*y*z - 
     0.7575451375965339*u*w2*x*y*z - 0.21838360971474802*lnbara*u*w2*x*y*
      z + 0.008782531191255194*v*w2*x*y*z + 0.006895491178906826*lnbara*v*
      w2*x*y*z - 0.0062284034187645365*w3*x*y*z - 
     0.006895491178906826*lnbara*w3*x*y*z + 0.12927559000443167*x2*y*z - 
     0.015907870172457157*u*x2*y*z + 0.00130062566240163*u2*x2*y*z - 
     2.343078615218984*v*x2*y*z - 0.6714897453342303*lnbara*v*x2*y*z + 
     0.4498241458617747*u*v*x2*y*z + 0.13547920846092296*lnbara*u*v*x2*y*
      z + 0.029245523131395473*v2*x2*y*z + 0.008946886688429884*lnbara*v2*
      x2*y*z + 2.311786939439009*w*x2*y*z + 0.6714897453342303*lnbara*w*x2*
      y*z - 0.4475358708690333*u*w*x2*y*z - 0.13547920846092296*lnbara*u*w*
      x2*y*z + 0.008782531191255194*v*w*x2*y*z + 0.006895491178906826*
      lnbara*v*w*x2*y*z - 0.034386556943485275*w2*x2*y*z - 
     0.01584237786733671*lnbara*w2*x2*y*z - 0.032835376049197604*x3*y*z + 
     0.0017051473997301786*u*x3*y*z + 0.12539708324984536*v*x3*y*z + 
     0.034595562617565336*lnbara*v*x3*y*z - 0.12207317499372534*w*x3*y*z - 
     0.034595562617565336*lnbara*w*x3*y*z + 0.0039062667758325825*x4*y*z - 
     0.4287703318427488*y2*z + 0.08548106272223456*u*y2*z - 
     0.01563232680721393*u2*y2*z + 0.0015419823403225055*u3*y2*z + 
     1.6449854699938522*v*y2*z + 0.48831400806030395*lnbara*v*y2*z - 
     0.18273891541643567*u*v*y2*z - 0.06121320376147517*lnbara*u*v*y2*z - 
     0.05823691063225072*u2*v*y2*z - 0.015697459289112238*lnbara*u2*v*y2*
      z - 0.008866331960675933*v2*y2*z - 0.009112399462646404*lnbara*v2*
      y2*z + 0.13337539195281198*u*v2*y2*z + 0.03812489698291478*lnbara*u*
      v2*y2*z + 0.14667148092405935*v3*y2*z + 0.041476677259681335*lnbara*
      v3*y2*z - 1.458515156634267*w*y2*z - 0.48831400806030395*lnbara*w*
      y2*z + 0.15508634310271682*u*w*y2*z + 0.06121320376147517*lnbara*u*w*
      y2*z + 0.060610376887719246*u2*w*y2*z + 0.015697459289112238*lnbara*
      u2*w*y2*z - 0.015907870172457157*v*w*y2*z + 
     0.001111492150387925*u*v*w*y2*z - 0.7204819253477474*v2*w*y2*z - 
     0.20371729223242238*lnbara*v2*w*y2*z - 0.027023436237519106*w2*y2*
      z + 0.009112399462646404*lnbara*w2*y2*z - 0.1308877099919966*u*w2*
      y2*z - 0.03812489698291478*lnbara*u*w2*y2*z + 
     0.7234333761756083*v*w2*y2*z + 0.20371729223242238*lnbara*v*w2*y2*
      z - 0.14302771932672184*w3*y2*z - 0.041476677259681335*lnbara*w3*y2*
      z + 0.12927559000443167*x*y2*z - 0.015383805607517485*u*x*y2*z + 
     0.001229379566375842*u2*x*y2*z - 0.015907870172457157*v*x*y2*z + 
     0.001111492150387925*u*v*x*y2*z - 0.4477982839144112*v2*x*y2*z - 
     0.12659992801563436*lnbara*v2*x*y2*z - 0.015907870172457157*w*x*y2*
      z + 0.001111492150387925*u*w*x*y2*z + 0.8993746019959791*v*w*x*y2*z + 
     0.2531998560312687*lnbara*v*w*x*y2*z - 0.4477982839144112*w2*x*y2*z - 
     0.12659992801563436*lnbara*w2*x*y2*z - 0.028211376477209685*x2*y2*
      z + 0.00146144490120512*u*x2*y2*z + 0.7234333761756083*v*x2*y2*z + 
     0.20371729223242238*lnbara*v*x2*y2*z - 0.7204819253477474*w*x2*y2*
      z - 0.20371729223242238*lnbara*w*x2*y2*z + 0.0030453133225042724*x3*
      y2*z + 0.19595873400524338*y3*z - 0.020567226223597537*u*y3*z + 
     0.0015419823403225055*u2*y3*z - 1.013316985461217*v*y3*z - 
     0.2929884048361824*lnbara*v*y3*z - 0.07883727181036221*u*v*y3*z - 
     0.019963821863986493*lnbara*u*v*y3*z + 0.1251536848709543*v2*y3*z + 
     0.037545222670016885*lnbara*v2*y3*z + 0.9672844851536025*w*y3*z + 
     0.2929884048361824*lnbara*w*y3*z + 0.0816735862630184*u*w*y3*z + 
     0.019963821863986493*lnbara*u*w*y3*z + 0.0017051473997301786*v*w*y3*
      z - 0.12145181321698113*w2*y3*z - 0.037545222670016885*lnbara*w2*y3*
      z - 0.032835376049197604*x*y3*z + 0.0016187608563898235*u*x*y3*z + 
     0.0017051473997301786*v*x*y3*z + 0.0017051473997301786*w*x*y3*z + 
     0.0030453133225042724*x2*y3*z - 0.05515167676319815*y4*z + 
     0.0023754912548396108*u*y4*z + 0.14703230574707543*v*y4*z + 
     0.040692834005025325*lnbara*v*y4*z - 0.14164027833703918*w*y4*z - 
     0.040692834005025325*lnbara*w*y4*z + 0.0039062667758325825*x*y4*z + 
     0.007027953632525705*y5*z + 1.3191828527017075*z2 - 
     0.4287703318427488*u*z2 + 0.14860391371313228*u2*z2 - 
     0.03593782918864228*u3*z2 + 0.004169061579930593*u4*z2 + 
     0.9491518611469659*v*z2 + 0.4340568960536035*lnbara*v*z2 + 
     1.8772484534413458*u*v*z2 + 0.564250670249024*lnbara*u*v*z2 + 
     0.6715039360324119*u2*v*z2 + 0.19748191093199324*lnbara*u2*v*z2 + 
     0.0911441314262025*u3*v*z2 + 0.02432329879100571*lnbara*u3*v*z2 + 
     0.8242562168640629*v2*z2 + 0.18665378315131373*lnbara*v2*z2 - 
     2.8414768629489666*u*v2*z2 - 0.8660810495752876*lnbara*u*v2*z2 + 
     0.11275072634761112*u2*v2*z2 + 0.035989183471035156*lnbara*u2*v2*
      z2 + 0.14157447061094622*v3*z2 + 0.05001134362781823*lnbara*v3*
      z2 + 0.4093655691718876*u*v3*z2 + 0.12737516232192267*lnbara*u*v3*
      z2 + 0.03289810142239334*v4*z2 + 0.007026033220340475*lnbara*v4*
      z2 - 1.700869602587337*w*z2 - 0.4340568960536035*lnbara*w*z2 - 
     1.6624918007146796*u*w*z2 - 0.564250670249024*lnbara*u*w*z2 - 
     0.7171950201286277*u2*w*z2 - 0.19748191093199324*lnbara*u2*w*z2 - 
     0.08627759743978855*u3*w*z2 - 0.02432329879100571*lnbara*u3*w*z2 - 
     3.366052674172739*v*w*z2 - 1.0851422401340087*lnbara*v*w*z2 + 
     2.244030329434903*u*v*w*z2 + 0.6930690249609582*lnbara*u*v*w*z2 - 
     0.5870531877615167*u2*v*w*z2 - 0.1732603915082422*lnbara*u2*v*w*z2 + 
     1.103138998233008*v2*w*z2 + 0.39106849121742115*lnbara*v2*w*z2 + 
     0.13104994444668858*u*v2*w*z2 + 0.030017221475897783*lnbara*u*v2*w*
      z2 - 0.4837702578695747*v3*w*z2 - 0.14799273546990963*lnbara*v3*w*
      z2 + 2.8750447180595793*w2*z2 + 0.898488456982695*lnbara*w2*z2 + 
     0.5382190246221227*u*w2*z2 + 0.1730120246143294*lnbara*u*w2*z2 + 
     0.479916000193387*u2*w2*z2 + 0.13727120803720705*lnbara*u2*w2*z2 - 
     0.3972223492383506*v*w2*z2 - 0.1465559850060673*lnbara*v*w2*z2 - 
     0.20929117997162128*u*v*w2*z2 - 0.06205837953332923*lnbara*u*v*w2*
      z2 + 0.4244378822310122*v2*w2*z2 + 0.11758155264728642*lnbara*v2*
      w2*z2 - 0.9391738601208355*w3*z2 - 0.29452384983917207*lnbara*w3*
      z2 - 0.3238462135165472*u*w3*z2 - 0.09533400426449122*lnbara*u*w3*
      z2 - 0.25301233477098106*v*w3*z2 - 0.062498180515692975*lnbara*v*w3*
      z2 + 0.29091212584308784*w4*z2 + 0.08588333011797569*lnbara*w4*
      z2 - 0.4287703318427488*x*z2 + 0.09467502496847112*u*x*z2 - 
     0.01841006057918888*u2*x*z2 + 0.001880650990063473*u3*x*z2 - 
     5.0744264500354594*v*x*z2 - 1.642900692638757*lnbara*v*x*z2 + 
     1.138848970579144*u*v*x*z2 + 0.36710906815157757*lnbara*u*v*x*z2 - 
     0.5618212037342907*u2*v*x*z2 - 0.16529892872504193*lnbara*u2*v*x*
      z2 + 1.103138998233008*v2*x*z2 + 0.39106849121742115*lnbara*v2*x*
      z2 + 0.13104994444668858*u*v2*x*z2 + 0.030017221475897783*lnbara*u*
      v2*x*z2 - 0.569080649680904*v3*x*z2 - 0.16941649028491496*lnbara*
      v3*x*z2 + 5.2517028011488085*w*x*z2 + 1.642900692638757*lnbara*w*x*
      z2 - 1.1689689937010432*u*w*x*z2 - 0.36710906815157757*lnbara*u*w*x*
      z2 + 0.5646118353233481*u2*w*x*z2 + 0.16529892872504193*lnbara*u2*w*
      x*z2 - 0.752469102785154*v*w*x*z2 - 0.24451250621135381*lnbara*v*w*x*
      z2 + 0.08204355214290159*u*v*w*x*z2 + 0.03204115805743145*lnbara*u*v*
      w*x*z2 + 1.1098745724532006*v2*w*x*z2 + 0.31182123714391863*lnbara*
      v2*w*x*z2 - 0.3972223492383506*w2*x*z2 - 0.1465559850060673*lnbara*
      w2*x*z2 - 0.20929117997162128*u*w2*x*z2 - 
     0.06205837953332923*lnbara*u*w2*x*z2 - 1.0464511532205378*v*w2*x*
      z2 - 0.2896065773402419*lnbara*v*w2*x*z2 + 
     0.5112934663833444*w3*x*z2 + 0.14720183048123822*lnbara*w3*x*z2 + 
     0.14860391371313228*x2*z2 - 0.01841006057918888*u*x2*z2 + 
     0.0015174486936744638*u2*x2*z2 - 0.017479707619006156*v*x2*z2 + 
     0.00130062566240163*u*v*x2*z2 + 0.4244378822310122*v2*x2*z2 + 
     0.11758155264728642*lnbara*v2*x2*z2 - 0.017479707619006156*w*x2*
      z2 + 0.00130062566240163*u*w*x2*z2 - 0.8449111212201761*v*w*x2*
      z2 - 0.23516310529457285*lnbara*v*w*x2*z2 + 
     0.4244378822310122*w2*x2*z2 + 0.11758155264728642*lnbara*w2*x2*
      z2 - 0.03593782918864228*x3*z2 + 0.001880650990063473*u*x3*z2 - 
     0.3213850461435467*v*x3*z2 - 0.08664893308797812*lnbara*v*x3*z2 + 
     0.3249693774147304*w*x3*z2 + 0.08664893308797812*lnbara*w*x3*z2 + 
     0.004169061579930593*x4*z2 - 0.4287703318427488*y*z2 + 
     0.09179528839111438*u*y*z2 - 0.017479707619006156*u2*y*z2 + 
     0.0017631106072740343*u3*y*z2 + 2.621952087639167*v*y*z2 + 
     0.8594186774137386*lnbara*v*y*z2 + 1.2616217958989164*u*v*y*z2 + 
     0.3671860440862087*lnbara*u*v*y*z2 + 0.22484058072370788*u2*v*y*z2 + 
     0.06208972089809539*lnbara*u2*v*y*z2 - 1.8627603909108525*v2*y*z2 - 
     0.5894315609290915*lnbara*v2*y*z2 - 0.10231789367095857*u*v2*y*z2 - 
     0.030182734250114304*lnbara*u*v2*y*z2 + 0.3009259577639667*v3*y*z2 + 
     0.09784333799586425*lnbara*v3*y*z2 - 2.4417959999484613*w*y*z2 - 
     0.8594186774137386*lnbara*w*y*z2 - 1.2909700854356503*u*w*y*z2 - 
     0.3671860440862087*lnbara*u*w*y*z2 - 0.222192079524089*u2*w*y*z2 - 
     0.06208972089809539*lnbara*u2*w*y*z2 + 1.6733264521145743*v*w*y*z2 + 
     0.499165430461644*lnbara*v*w*y*z2 - 0.7575451375965339*u*v*w*y*z2 - 
     0.21838360971474802*lnbara*u*v*w*y*z2 - 0.034386556943485275*v2*w*y*
      z2 - 0.01584237786733671*lnbara*v2*w*y*z2 + 
     0.1411793984604333*w2*y*z2 + 0.09026613046744748*lnbara*w2*y*z2 + 
     0.8636086160336289*u*w2*y*z2 + 0.24856634396486232*lnbara*u*w2*y*
      z2 + 0.029245523131395473*v*w2*y*z2 + 0.008946886688429884*lnbara*v*
      w2*y*z2 - 0.28983228539271344*w3*y*z2 - 0.09094784681695743*lnbara*
      w3*y*z2 + 0.12927559000443167*x*y*z2 - 0.015907870172457157*u*x*y*
      z2 + 0.00130062566240163*u2*x*y*z2 + 2.311786939439009*v*x*y*z2 + 
     0.6714897453342303*lnbara*v*x*y*z2 - 0.4475358708690333*u*v*x*y*z2 - 
     0.13547920846092296*lnbara*u*v*x*y*z2 - 0.034386556943485275*v2*x*y*
      z2 - 0.01584237786733671*lnbara*v2*x*y*z2 - 
     2.343078615218984*w*x*y*z2 - 0.6714897453342303*lnbara*w*x*y*z2 + 
     0.4498241458617747*u*w*x*y*z2 + 0.13547920846092296*lnbara*u*w*x*y*
      z2 + 0.008782531191255194*v*w*x*y*z2 + 0.006895491178906826*lnbara*v*
      w*x*y*z2 + 0.029245523131395473*w2*x*y*z2 + 
     0.008946886688429884*lnbara*w2*x*y*z2 - 0.028211376477209685*x2*y*
      z2 + 0.001490005926655855*u*x2*y*z2 + 0.00146144490120512*v*x2*y*
      z2 + 0.00146144490120512*w*x2*y*z2 + 0.0030453133225042724*x3*y*
      z2 + 0.14860391371313228*y2*z2 - 0.017479707619006156*u*y2*z2 + 
     0.0013887934717172665*u2*y2*z2 - 0.008866331960675933*v*y2*z2 - 
     0.009112399462646404*lnbara*v*y2*z2 + 0.13337539195281198*u*v*y2*
      z2 + 0.03812489698291478*lnbara*u*v*y2*z2 + 
     0.1141478259424908*v2*y2*z2 + 0.03211951146934684*lnbara*v2*y2*
      z2 - 0.027023436237519106*w*y2*z2 + 0.009112399462646404*lnbara*w*
      y2*z2 - 0.1308877099919966*u*w*y2*z2 - 0.03812489698291478*lnbara*u*
      w*y2*z2 - 0.4477982839144112*v*w*y2*z2 - 0.12659992801563436*lnbara*
      v*w*y2*z2 + 0.3378573257997137*w2*y2*z2 + 
     0.09448041654628751*lnbara*w2*y2*z2 - 0.028211376477209685*x*y2*
      z2 + 0.00146144490120512*u*x*y2*z2 - 0.7204819253477474*v*x*y2*
      z2 - 0.20371729223242238*lnbara*v*x*y2*z2 + 
     0.7234333761756083*w*x*y2*z2 + 0.20371729223242238*lnbara*w*x*y2*
      z2 + 0.002763300406558978*x2*y2*z2 - 0.03593782918864228*y3*z2 + 
     0.0017631106072740343*u*y3*z2 + 0.1251536848709543*v*y3*z2 + 
     0.037545222670016885*lnbara*v*y3*z2 - 0.12145181321698113*w*y3*z2 - 
     0.037545222670016885*lnbara*w*y3*z2 + 0.0030453133225042724*x*y3*
      z2 + 0.004169061579930593*y4*z2 - 1.0795675579460708*z3 + 
     0.19595873400524338*u*z3 - 0.03593782918864228*u2*z3 + 
     0.0035837624283894857*u3*z3 - 0.1848964780137795*v*z3 - 
     0.12370155645114488*lnbara*v*z3 - 1.3530173829875152*u*v*z3 - 
     0.40622240013673455*lnbara*u*v*z3 - 0.0821346574113807*u2*v*z3 - 
     0.021986858303963977*lnbara*u2*v*z3 + 0.14157447061094622*v2*z3 + 
     0.05001134362781823*lnbara*v2*z3 + 0.4093655691718876*u*v2*z3 + 
     0.12737516232192267*lnbara*u*v2*z3 + 0.02313083477136829*v3*z3 + 
     0.004691749642619168*lnbara*v3*z3 + 0.5237190151099749*w*z3 + 
     0.12370155645114488*lnbara*w*z3 + 1.29961478071472*u*w*z3 + 
     0.40622240013673455*lnbara*u*w*z3 + 0.08694308134115901*u2*w*z3 + 
     0.021986858303963977*lnbara*u2*w*z3 + 0.7184614571972799*v*w*z3 + 
     0.24451250621135381*lnbara*v*w*z3 - 0.07931329913612384*u*v*w*z3 - 
     0.03204115805743145*lnbara*u*v*w*z3 - 0.569080649680904*v2*w*z3 - 
     0.16941649028491496*lnbara*v2*w*z3 - 0.9391738601208355*w2*z3 - 
     0.29452384983917207*lnbara*w2*z3 - 0.3238462135165472*u*w2*z3 - 
     0.09533400426449122*lnbara*u*w2*z3 + 0.5112934663833444*v*w2*z3 + 
     0.14720183048123822*lnbara*v*w2*z3 + 0.043640908015127135*w3*z3 + 
     0.01752291016105757*lnbara*w3*z3 + 0.19595873400524338*x*z3 - 
     0.023227304775276715*u*x*z3 + 0.001880650990063473*u2*x*z3 + 
     1.7597891705366386*v*x*z3 + 0.5698593651894059*lnbara*v*x*z3 + 
     0.02852820513246288*u*v*x*z3 + 0.0022974655834007394*lnbara*u*v*x*z3 - 
     0.4837702578695747*v2*x*z3 - 0.14799273546990963*lnbara*v2*x*z3 - 
     1.803161592292574*w*x*z3 - 0.5698593651894059*lnbara*w*x*z3 - 
     0.025380422110629967*u*w*x*z3 - 0.0022974655834007394*lnbara*u*w*x*
      z3 + 0.7415394744753414*v*w*x*z3 + 0.2104909159856026*lnbara*v*w*x*
      z3 - 0.25301233477098106*w2*x*z3 - 0.062498180515692975*lnbara*w2*x*
      z3 - 0.03593782918864228*x2*z3 + 0.001880650990063473*u*x2*z3 + 
     0.3249693774147304*v*x2*z3 + 0.08664893308797812*lnbara*v*x2*z3 - 
     0.3213850461435467*w*x2*z3 - 0.08664893308797812*lnbara*w*x2*z3 + 
     0.0035837624283894857*x3*z3 + 0.19595873400524338*y*z3 - 
     0.022805195532337737*u*y*z3 + 0.001821220663909673*u2*y*z3 - 
     2.2491286389959257*v*y*z3 - 0.6865302179753614*lnbara*v*y*z3 - 
     0.18944564848149062*u*v*y*z3 - 0.05112899772410982*lnbara*u*v*y*z3 + 
     0.3009259577639667*v2*y*z3 + 0.09784333799586425*lnbara*v2*y*z3 + 
     2.2053341079970514*w*y*z3 + 0.6865302179753614*lnbara*w*y*z3 + 
     0.19254447471177424*u*w*y*z3 + 0.05112899772410982*lnbara*u*w*y*z3 - 
     0.0062284034187645365*v*w*y*z3 - 0.006895491178906826*lnbara*v*w*y*
      z3 - 0.28983228539271344*w2*y*z3 - 0.09094784681695743*lnbara*w2*y*
      z3 - 0.032835376049197604*x*y*z3 + 0.0017051473997301786*u*x*y*z3 - 
     0.12207317499372534*v*x*y*z3 - 0.034595562617565336*lnbara*v*x*y*z3 + 
     0.12539708324984536*w*x*y*z3 + 0.034595562617565336*lnbara*w*x*y*z3 + 
     0.0030453133225042724*x2*y*z3 - 0.03593782918864228*y2*z3 + 
     0.001821220663909673*u*y2*z3 + 0.14667148092405935*v*y2*z3 + 
     0.041476677259681335*lnbara*v*y2*z3 - 0.14302771932672184*w*y2*z3 - 
     0.041476677259681335*lnbara*w*y2*z3 + 0.0030453133225042724*x*y2*
      z3 + 0.0035837624283894857*y3*z3 + 0.5780009836815717*z4 - 
     0.05515167676319815*u*z4 + 0.004169061579930593*u2*z4 + 
     0.3026348522637972*v*z4 + 0.10644029308949739*lnbara*v*z4 + 
     0.23469657548791326*u*v*z4 + 0.06907962057473943*lnbara*u*v*z4 + 
     0.03289810142239334*v2*z4 + 0.007026033220340475*lnbara*v2*z4 - 
     0.397334765898689*w*z4 - 0.10644029308949739*lnbara*w*z4 - 
     0.22841481745724104*u*w*z4 - 0.06907962057473943*lnbara*u*w*z4 - 
     0.31472121563065336*v*w*z4 - 0.09290936333831616*lnbara*v*w*z4 + 
     0.29091212584308784*w2*z4 + 0.08588333011797569*lnbara*w2*z4 - 
     0.05515167676319815*x*z4 + 0.0027129425983649334*u*x*z4 - 
     0.5225454785602572*v*x*z4 - 0.15634114484424524*lnbara*v*x*z4 + 
     0.5276000546267682*w*x*z4 + 0.15634114484424524*lnbara*w*x*z4 + 
     0.004169061579930593*x2*z4 - 0.05515167676319815*y*z4 + 
     0.0026790848116713074*u*y*z4 + 0.4179202471284635*v*y*z4 + 
     0.12512755974621306*lnbara*v*y*z4 - 0.412831813275259*w*y*z4 - 
     0.12512755974621306*lnbara*w*y*z4 + 0.0039062667758325825*x*y*z4 + 
     0.004169061579930593*y2*z4 - 0.17916940468773146*z5 + 
     0.007027953632525705*u*z5 - 0.03675623670681*v*z5 - 
     0.013089806333713885*lnbara*v*z5 + 0.0487756611897797*w*z5 + 
     0.013089806333713885*lnbara*w*z5 + 0.007027953632525705*x*z5 + 
     0.007027953632525705*y*z5 + 0.024344353217864104*z6;
 
  Hexp[7] = 1.0303344884225094 - 1.2020569031595942*u + 
     1.5830194232420491*u2 - 1.6193513369191062*u3 + 
     1.1560019673631434*u4 - 0.5375082140631944*u5 + 
     0.14606611930718463*u6 - 0.017602106974984624*u7 - 
     1.2020569031595942*v + 0.8408174973812163*u*v - 
     0.6431554977641232*u2*v + 0.39191746801048677*u3*v - 
     0.16545503028959443*u4*v + 0.04216772179515423*u5*v - 
     0.004855915459192057*u6*v + 1.5830194232420491*v2 - 
     0.6431554977641232*u*v2 + 0.29720782742626456*u2*v2 - 
     0.10781348756592685*u3*v2 + 0.02501436947958356*u4*v2 - 
     0.0027042708881657167*u5*v2 - 1.6193513369191062*v3 + 
     0.39191746801048677*u*v3 - 0.10781348756592685*u2*v3 + 
     0.021502574570336913*u3*v3 - 0.002116028409986158*u4*v3 + 
     1.1560019673631434*v4 - 0.16545503028959443*u*v4 + 
     0.02501436947958356*u2*v4 - 0.002116028409986158*u3*v4 - 
     0.5375082140631944*v5 + 0.04216772179515423*u*v5 - 
     0.0027042708881657167*u2*v5 + 0.14606611930718463*v6 - 
     0.004855915459192057*u*v6 - 0.017602106974984624*v7 - 
     1.2020569031595942*w + 0.8408174973812163*u*w - 
     0.6431554977641232*u2*w + 0.39191746801048677*u3*w - 
     0.16545503028959443*u4*w + 0.04216772179515423*u5*w - 
     0.004855915459192057*u6*w + 0.8408174973812163*v*w - 
     0.3944622860815335*u*v*w + 0.18935004993694224*u2*v*w - 
     0.06968191432583015*u3*v*w + 0.0162776555901896*u4*v*w - 
     0.0017664062094713614*u5*v*w - 0.6431554977641232*v2*w + 
     0.18935004993694224*u*v2*w - 0.05523018173756664*u2*v2*w + 
     0.011283905940380838*u3*v2*w - 0.0011239361348986583*u4*v2*w + 
     0.39191746801048677*v3*w - 0.06968191432583015*u*v3*w + 
     0.011283905940380838*u2*v3*w - 0.0009831878162066224*u3*v3*w - 
     0.16545503028959443*v4*w + 0.0162776555901896*u*v4*w - 
     0.0011239361348986583*u2*v4*w + 0.04216772179515423*v5*w - 
     0.0017664062094713614*u*v5*w - 0.004855915459192057*v6*w + 
     1.5830194232420491*w2 - 0.6431554977641232*u*w2 + 
     0.29720782742626456*u2*w2 - 0.10781348756592685*u3*w2 + 
     0.02501436947958356*u4*w2 - 0.0027042708881657167*u5*w2 - 
     0.6431554977641232*v*w2 + 0.18935004993694224*u*v*w2 - 
     0.05523018173756664*u2*v*w2 + 0.011283905940380838*u3*v*w2 - 
     0.0011239361348986583*u4*v*w2 + 0.29720782742626456*v2*w2 - 
     0.05523018173756664*u*v2*w2 + 0.009104692162046783*u2*v2*w2 - 
     0.0008001063695529314*u3*v2*w2 - 0.10781348756592685*v3*w2 + 
     0.011283905940380838*u*v3*w2 - 0.0008001063695529314*u2*v3*w2 + 
     0.02501436947958356*v4*w2 - 0.0011239361348986583*u*v4*w2 - 
     0.0027042708881657167*v5*w2 - 1.6193513369191062*w3 + 
     0.39191746801048677*u*w3 - 0.10781348756592685*u2*w3 + 
     0.021502574570336913*u3*w3 - 0.002116028409986158*u4*w3 + 
     0.39191746801048677*v*w3 - 0.06968191432583015*u*v*w3 + 
     0.011283905940380838*u2*v*w3 - 0.0009831878162066224*u3*v*w3 - 
     0.10781348756592685*v2*w3 + 0.011283905940380838*u*v2*w3 - 
     0.0008001063695529314*u2*v2*w3 + 0.021502574570336913*v3*w3 - 
     0.0009831878162066224*u*v3*w3 - 0.002116028409986158*v4*w3 + 
     1.1560019673631434*w4 - 0.16545503028959443*u*w4 + 
     0.02501436947958356*u2*w4 - 0.002116028409986158*u3*w4 - 
     0.16545503028959443*v*w4 + 0.0162776555901896*u*v*w4 - 
     0.0011239361348986583*u2*v*w4 + 0.02501436947958356*v2*w4 - 
     0.0011239361348986583*u*v2*w4 - 0.002116028409986158*v3*w4 - 
     0.5375082140631944*w5 + 0.04216772179515423*u*w5 - 
     0.0027042708881657167*u2*w5 + 0.04216772179515423*v*w5 - 
     0.0017664062094713614*u*v*w5 - 0.0027042708881657167*v2*w5 + 
     0.14606611930718463*w6 - 0.004855915459192057*u*w6 - 
     0.004855915459192057*v*w6 - 0.017602106974984624*w7 - 
     1.2020569031595942*x + 0.8408174973812163*u*x - 
     0.6431554977641232*u2*x + 0.39191746801048677*u3*x - 
     0.16545503028959443*u4*x + 0.04216772179515423*u5*x - 
     0.004855915459192057*u6*x + 0.8408174973812163*v*x + 
     1.316872732915981*u*v*x + 0.6510853440804052*lnbara*u*v*x + 
     1.7420888571164688*u2*v*x + 0.4470413315421547*lnbara*u2*v*x + 
     0.5072130866752762*u3*v*x + 0.17751808906800676*lnbara*u3*v*x + 
     0.14015372875382298*u4*v*x + 0.031382877263858105*lnbara*u4*v*x + 
     0.007029374644742698*u5*v*x + 0.0025756149427488186*lnbara*u5*v*x - 
     2.4582922118695194*v2*x - 0.6510853440804052*lnbara*v2*x - 
     3.1637691242153525*u*v2*x - 1.128501340498048*lnbara*u*v2*x - 
     2.066949955700743*u2*v2*x - 0.5924457327959798*lnbara*u2*v2*x - 
     0.49681704281786926*u3*v2*x - 0.14593979274603425*lnbara*u3*v2*x - 
     0.05261848185645107*u4*v2*x - 0.01365467547098468*lnbara*u4*v2*x + 
     1.0120947283767654*v3*x + 0.24740311290228975*lnbara*v3*x + 
     3.7247490277998456*u*v3*x + 1.2186672004102037*lnbara*u*v3*x + 
     0.5028126095006992*u2*v3*x + 0.13192114982378386*lnbara*u2*v3*x + 
     0.23084041467755445*u3*v3*x + 0.06654355140807486*lnbara*u3*v3*x - 
     1.1463870292291396*v4*x - 0.3193208792684922*lnbara*v4*x - 
     1.3112778013936697*u*v4*x - 0.4144777234484366*lnbara*u*v4*x - 
     0.16893584646815507*u2*v4*x - 0.0453330032683975*lnbara*u2*v4*x + 
     0.281434133138352*v5*x + 0.07853883800228331*lnbara*v5*x + 
     0.2738733074161405*u*v5*x + 0.08187135582602693*lnbara*u*v5*x - 
     0.04987109935863513*v6*x - 0.013727259996145989*lnbara*v6*x + 
     0.6830325829486028*w*x - 2.1845063283947157*u*w*x - 
     0.6510853440804052*lnbara*u*w*x - 1.2999471003253766*u2*w*x - 
     0.4470413315421547*lnbara*u2*w*x - 0.6741348014198821*u3*w*x - 
     0.17751808906800676*lnbara*u3*w*x - 0.10064161922879965*u4*w*x - 
     0.031382877263858105*lnbara*u4*w*x - 0.011350747467801053*u5*w*x - 
     0.0025756149427488186*lnbara*u5*w*x - 0.3693696142893198*v*w*x + 
     0.1472496149014935*u*v*w*x - 0.04772361051737147*u2*v*w*x + 
     0.01023088439838107*u3*v*w*x - 0.0010458019341617988*u4*v*w*x + 
     10.034005404400828*v2*w*x + 3.285801385277514*lnbara*v2*w*x - 
     3.3495745233238825*u*v2*w*x - 1.1013272044547329*lnbara*u*v2*w*x + 
     3.2459862158900528*u2*v2*w*x + 0.9917935723502516*lnbara*u2*v2*w*x - 
     0.40594313303596224*u3*v2*w*x - 0.12233847935377173*lnbara*u3*v2*w*
      x - 5.165259334653691*v3*w*x - 1.7095780955682178*lnbara*v3*w*x - 
     0.15031327644943632*u*v3*w*x - 0.013784793500404437*lnbara*u*v3*w*x - 
     0.40939098371296645*u2*v3*w*x - 0.12576336826926637*lnbara*u2*v3*w*
      x + 3.0315936321798276*v4*w*x + 0.9380468690654714*lnbara*v4*w*x - 
     0.15381418595568414*u*v4*w*x - 0.048049503963086576*lnbara*u*v4*w*x - 
     0.48018841061091755*v5*w*x - 0.14541778766038482*lnbara*v5*w*x + 
     1.3307155997089624*w2*x + 0.6510853440804052*lnbara*w2*x + 
     3.593282429668685*u*w2*x + 1.128501340498048*lnbara*u*w2*x + 
     1.9298767034120956*u2*w2*x + 0.5924457327959798*lnbara*u2*w2*x + 
     0.5260162467363529*u3*w2*x + 0.14593979274603425*lnbara*u3*w2*x + 
     0.049642889715863496*u4*w2*x + 0.01365467547098468*lnbara*u4*w2*x - 
     9.679452702174132*v*w2*x - 3.285801385277514*lnbara*v*w2*x + 
     3.2592144539581844*u*v*w2*x + 1.1013272044547329*lnbara*u*v*w2*x - 
     3.229242426355708*u2*v*w2*x - 0.9917935723502516*lnbara*u2*v*w2*x + 
     0.404386452649353*u3*v*w2*x + 0.12233847935377173*lnbara*u3*v*w2*x - 
     0.052439122857018475*v2*w2*x + 0.00780375397440978*u*v2*w2*x - 
     0.0006345782786901335*u2*v2*w2*x - 1.8540397627858702*v3*w2*x - 
     0.5198935985278687*lnbara*v3*w2*x + 0.5315078645933379*u*v3*w2*x + 
     0.1550921261368306*lnbara*u*v3*w2*x + 0.11440699548555519*v4*w2*x + 
     0.02876833807133764*lnbara*v4*w2*x - 0.3344496541843748*w3*x - 
     0.24740311290228975*lnbara*w3*x - 3.884956834618231*u*w3*x - 
     1.2186672004102037*lnbara*u*w3*x - 0.47396206592202933*u2*w3*x - 
     0.13192114982378386*lnbara*u2*w3*x - 0.23348793049311065*u3*w3*x - 
     0.06654355140807486*lnbara*u3*w3*x + 5.035142069385885*v*w3*x + 
     1.7095780955682178*lnbara*v*w3*x + 0.16919997458043376*u*v*w3*x + 
     0.013784793500404437*lnbara*u*v*w3*x + 0.4078728401620492*u2*v*w3*x + 
     0.12576336826926637*lnbara*u2*v*w3*x + 1.8755457504129727*v2*w3*x + 
     0.5198935985278687*lnbara*v2*w3*x - 0.5329098110819325*u*v2*w3*x - 
     0.1550921261368306*lnbara*u*v2*w3*x - 0.0009436436586848417*v3*w3*
      x + 0.8622872883244641*w4*x + 0.3193208792684922*lnbara*w4*x + 
     1.3489683495777027*u*w4*x + 0.4144777234484366*lnbara*u*w4*x + 
     0.1660137847279631*u2*w4*x + 0.0453330032683975*lnbara*u2*w4*x - 
     3.0012661757807617*v*w4*x - 0.9380468690654714*lnbara*v*w4*x + 
     0.15191010788660775*u*v*w4*x + 0.048049503963086576*lnbara*u*v*w4*x - 
     0.11655132895257132*v2*w4*x - 0.02876833807133764*lnbara*v2*w4*x - 
     0.2093175862405338*w5*x - 0.07853883800228331*lnbara*w5*x - 
     0.27798071288879395*u*w5*x - 0.08187135582602693*lnbara*u*w5*x + 
     0.4769026267982312*v*w5*x + 0.14541778766038482*lnbara*v*w5*x + 
     0.04158747525391902*w6*x + 0.013727259996145989*lnbara*w6*x + 
     1.5830194232420491*x2 - 0.6431554977641232*u*x2 + 
     0.29720782742626456*u2*x2 - 0.10781348756592685*u3*x2 + 
     0.02501436947958356*u4*x2 - 0.0027042708881657167*u5*x2 - 
     2.4582922118695194*v*x2 - 0.6510853440804052*lnbara*v*x2 - 
     3.1637691242153525*u*v*x2 - 1.128501340498048*lnbara*u*v*x2 - 
     2.066949955700743*u2*v*x2 - 0.5924457327959798*lnbara*u2*v*x2 - 
     0.49681704281786926*u3*v*x2 - 0.14593979274603425*lnbara*u3*v*x2 - 
     0.05261848185645107*u4*v*x2 - 0.01365467547098468*lnbara*u4*v*x2 + 
     5.493378448409817*v2*x2 + 1.79697691396539*lnbara*v2*x2 + 
     1.5405090633173697*u*v2*x2 + 0.5190360738429882*lnbara*u*v2*x2 + 
     2.7618349656998586*u2*v2*x2 + 0.8236272482232423*lnbara*u2*v2*x2 + 
     0.024812552253286976*u3*v2*x2 + 0.003229366446298125*lnbara*u3*v2*
      x2 - 2.6912970732885757*v3*x2 - 0.8835715495175162*lnbara*v3*x2 - 
     1.8613624203011476*u*v3*x2 - 0.5720040255869473*lnbara*u*v3*x2 - 
     0.48114924036453877*u2*v3*x2 - 0.13930793778849923*lnbara*u2*v3*
      x2 + 1.6718584721002623*v4*x2 + 0.5152999807078541*lnbara*v4*x2 + 
     0.17448936082610797*u*v4*x2 + 0.05426081916613278*lnbara*u*v4*x2 - 
     0.24406286627904641*v5*x2 - 0.07417184660062576*lnbara*v5*x2 + 
     1.3307155997089624*w*x2 + 0.6510853440804052*lnbara*w*x2 + 
     3.593282429668685*u*w*x2 + 1.128501340498048*lnbara*u*w*x2 + 
     1.9298767034120956*u2*w*x2 + 0.5924457327959798*lnbara*u2*w*x2 + 
     0.5260162467363529*u3*w*x2 + 0.14593979274603425*lnbara*u3*w*x2 + 
     0.049642889715863496*u4*w*x2 + 0.01365467547098468*lnbara*u4*w*x2 - 
     6.422064708307189*v*w*x2 - 2.1702844802680175*lnbara*v*w*x2 + 
     6.435061406178583*u*v*w*x2 + 2.079207074882875*lnbara*u*v*w*x2 - 
     3.373810219562036*u2*v*w*x2 - 1.0395623490494532*lnbara*u2*v*w*x2 + 
     0.5336843597652852*u3*v*w*x2 + 0.16066085741485445*lnbara*u3*v*w*
      x2 - 1.1288573398553086*v2*w*x2 - 0.43966795501820194*lnbara*v2*w*
      x2 - 1.202554183086874*u*v2*w*x2 - 0.3723502771999754*lnbara*u*v2*w*
      x2 - 0.08093775692251794*u2*v2*w*x2 - 0.025975593895628257*lnbara*
      u2*v2*w*x2 - 1.4645041396124352*v3*w*x2 - 
     0.37498908309415785*lnbara*v3*w*x2 + 1.25382676464303*u*v3*w*x2 + 
     0.3652162355419701*lnbara*u*v3*w*x2 + 0.0011325127373185734*v4*w*
      x2 - 0.009074544432852395*lnbara*v4*w*x2 + 
     1.5951827813991792*w2*x2 + 0.37330756630262746*lnbara*w2*x2 - 
     8.153252996171776*u*w2*x2 - 2.598243148725863*lnbara*u*w2*x2 + 
     0.6456564865390652*u2*w2*x2 + 0.21593510082621092*lnbara*u2*w2*
      x2 - 0.5616689792286329*u3*w2*x2 - 0.16389022386115257*lnbara*u3*
      w2*x2 + 3.1418162127487004*v*w2*x2 + 1.1732054736522635*lnbara*v*
      w2*x2 + 0.7605706197007904*u*v*w2*x2 + 0.1801033288553867*lnbara*u*
      v*w2*x2 + 0.41357773400649983*u2*v*w2*x2 + 
     0.1313398473229925*lnbara*u2*v*w2*x2 + 2.445843105402685*v2*w2*
      x2 + 0.7054893158837185*lnbara*v2*w2*x2 - 
     1.2333152992895828*u*v2*w2*x2 - 0.3521670333053898*lnbara*u*v2*w2*
      x2 - 0.061694139411974463*v3*w2*x2 - 0.01592769241839614*lnbara*v3*
      w2*x2 + 0.403289978849488*w3*x2 + 0.15003403088345468*lnbara*w3*
      x2 + 2.3470147044696774*u*w3*x2 + 0.7642509739315361*lnbara*u*w3*
      x2 + 0.14483466237687437*u2*w3*x2 + 0.03394368436113497*lnbara*u2*
      w3*x2 - 2.775770631100383*v*w3*x2 - 0.8879564128194577*lnbara*v*w3*
      x2 + 0.3619717726179578*u*v*w3*x2 + 0.10324350723878072*lnbara*u*v*
      w3*x2 - 0.04072484698113826*v2*w3*x2 - 0.013759874074562587*lnbara*
      v2*w3*x2 + 0.1913662943454968*w4*x2 + 0.04215619932204285*lnbara*
      w4*x2 - 0.5618260667674302*u*w4*x2 - 0.17055352864149378*lnbara*u*
      w4*x2 + 0.3328573832991313*v*w4*x2 + 0.11063120820095806*lnbara*v*
      w4*x2 + 0.004695610230502542*w5*x2 + 0.002302749325478825*lnbara*
      w5*x2 - 1.6193513369191062*x3 + 0.39191746801048677*u*x3 - 
     0.10781348756592685*u2*x3 + 0.021502574570336913*u3*x3 - 
     0.002116028409986158*u4*x3 + 1.0120947283767654*v*x3 + 
     0.24740311290228975*lnbara*v*x3 + 3.7247490277998456*u*v*x3 + 
     1.2186672004102037*lnbara*u*v*x3 + 0.5028126095006992*u2*v*x3 + 
     0.13192114982378386*lnbara*u2*v*x3 + 0.23084041467755445*u3*v*x3 + 
     0.06654355140807486*lnbara*u3*v*x3 - 2.6912970732885757*v2*x3 - 
     0.8835715495175162*lnbara*v2*x3 - 1.8613624203011476*u*v2*x3 - 
     0.5720040255869473*lnbara*u*v2*x3 - 0.48114924036453877*u2*v2*x3 - 
     0.13930793778849923*lnbara*u2*v2*x3 + 0.24682581080985633*v3*x3 + 
     0.10513746096634541*lnbara*v3*x3 + 0.8544575390208671*u*v3*x3 + 
     0.251383986920522*lnbara*u*v3*x3 - 0.1821625110649795*v4*x3 - 
     0.057146188727501226*lnbara*v4*x3 - 0.3344496541843748*w*x3 - 
     0.24740311290228975*lnbara*w*x3 - 3.884956834618231*u*w*x3 - 
     1.2186672004102037*lnbara*u*w*x3 - 0.47396206592202933*u2*w*x3 - 
     0.13192114982378386*lnbara*u2*w*x3 - 0.23348793049311065*u3*w*x3 - 
     0.06654355140807486*lnbara*u3*w*x3 + 2.0505932975012593*v*w*x3 + 
     0.7335375186340615*lnbara*v*w*x3 - 0.4484159450532304*u*v*w*x3 - 
     0.1922469483445887*lnbara*u*v*w*x3 + 0.33319811688498185*u2*v*w*x3 + 
     0.10536425342736426*lnbara*u2*v*w*x3 + 2.941587800744719*v2*w*x3 + 
     0.8832109828874293*lnbara*v2*w*x3 - 0.7853652276547294*u*v2*w*x3 - 
     0.21990490700155435*lnbara*u*v2*w*x3 - 0.2434872198153215*v3*w*x3 - 
     0.07483548907405872*lnbara*v3*w*x3 + 0.403289978849488*w2*x3 + 
     0.15003403088345468*lnbara*w2*x3 + 2.3470147044696774*u*w2*x3 + 
     0.7642509739315361*lnbara*u*w2*x3 + 0.14483466237687437*u2*w2*x3 + 
     0.03394368436113497*lnbara*u2*w2*x3 - 3.269269763555497*v*w2*x3 - 
     1.0164989417094896*lnbara*v*w2*x3 + 0.44168144304165097*u*v*w2*x3 + 
     0.12858123333432991*lnbara*u*v*w2*x3 + 0.05859089416673988*v2*w2*
      x3 + 0.01186851715999764*lnbara*v2*w2*x3 + 
     0.13476350893453617*w3*x3 + 0.02815049785571501*lnbara*w3*x3 - 
     0.5147171592780667*u*w3*x3 - 0.16006031325329761*lnbara*u*w3*x3 + 
     0.33499391115566624*v*w3*x3 + 0.11243984289700684*lnbara*v*w3*x3 + 
     0.026332918322492415*w4*x3 + 0.007673317744555463*lnbara*w4*x3 + 
     1.1560019673631434*x4 - 0.16545503028959443*u*x4 + 
     0.02501436947958356*u2*x4 - 0.002116028409986158*u3*x4 - 
     1.1463870292291396*v*x4 - 0.3193208792684922*lnbara*v*x4 - 
     1.3112778013936697*u*v*x4 - 0.4144777234484366*lnbara*u*v*x4 - 
     0.16893584646815507*u2*v*x4 - 0.0453330032683975*lnbara*u2*v*x4 + 
     1.6718584721002623*v2*x4 + 0.5152999807078541*lnbara*v2*x4 + 
     0.17448936082610797*u*v2*x4 + 0.05426081916613278*lnbara*u*v2*x4 - 
     0.1821625110649795*v3*x4 - 0.057146188727501226*lnbara*v3*x4 + 
     0.8622872883244641*w*x4 + 0.3193208792684922*lnbara*w*x4 + 
     1.3489683495777027*u*w*x4 + 0.4144777234484366*lnbara*u*w*x4 + 
     0.1660137847279631*u2*w*x4 + 0.0453330032683975*lnbara*u2*w*x4 - 
     1.8086906966367922*v*w*x4 - 0.5574561800298969*lnbara*v*w*x4 + 
     0.3835758469523267*u*v*w*x4 + 0.116292709475361*lnbara*u*v*w*x4 - 
     0.21163351738140398*v2*w*x4 - 0.06718897907015481*lnbara*v2*w*x4 + 
     0.1913662943454968*w2*x4 + 0.04215619932204285*lnbara*w2*x4 - 
     0.5618260667674302*u*w2*x4 - 0.17055352864149378*lnbara*u*w2*x4 + 
     0.3622126558447875*v*w2*x4 + 0.11666185005310058*lnbara*v*w2*x4 + 
     0.026332918322492415*w3*x4 + 0.007673317744555463*lnbara*w3*x4 - 
     0.5375082140631944*x5 + 0.04216772179515423*u*x5 - 
     0.0027042708881657167*u2*x5 + 0.281434133138352*v*x5 + 
     0.07853883800228331*lnbara*v*x5 + 0.2738733074161405*u*v*x5 + 
     0.08187135582602693*lnbara*u*v*x5 - 0.24406286627904641*v2*x5 - 
     0.07417184660062576*lnbara*v2*x5 - 0.2093175862405338*w*x5 - 
     0.07853883800228331*lnbara*w*x5 - 0.27798071288879395*u*w*x5 - 
     0.08187135582602693*lnbara*u*w*x5 + 0.23350760677675905*v*w*x5 + 
     0.07186909727514694*lnbara*v*w*x5 + 0.004695610230502542*w2*x5 + 
     0.002302749325478825*lnbara*w2*x5 + 0.14606611930718463*x6 - 
     0.004855915459192057*u*x6 - 0.04987109935863513*v*x6 - 
     0.013727259996145989*lnbara*v*x6 + 0.04158747525391902*w*x6 + 
     0.013727259996145989*lnbara*w*x6 - 0.017602106974984624*x7 - 
     1.2020569031595942*y + 0.6830325829486028*u*y - 
     0.4844211143964337*u2*y + 0.28572760618190396*u3*y - 
     0.11864471061508094*u4*y + 0.029948825102663987*u5*y - 
     0.0034277086455240437*u6*y + 0.8408174973812163*v*y - 
     0.3693696142893198*u*v*y + 0.17096212544446912*u2*v*y - 
     0.06170167867079261*u3*v*y + 0.014252947529037665*u4*v*y - 
     0.0015359082311407401*u5*v*y - 0.6431554977641232*v2*y + 
     0.18359057678222876*u*v2*y - 0.052439122857018475*u2*v2*y + 
     0.010578663643644204*u3*v2*y - 0.0010454015333102693*u4*v2*y + 
     0.39191746801048677*v3*y - 0.06841558659701322*u*v3*y + 
     0.010927323983458038*u2*v3*y - 0.0009436436586848417*u3*v3*y - 
     0.16545503028959443*v4*y + 0.016074508870027843*u*v4*y - 
     0.0010989319337058666*u2*v4*y + 0.04216772179515423*v5*y - 
     0.0017498755815456293*u*v5*y - 0.004855915459192057*v6*y + 
     0.8408174973812163*w*y - 0.3693696142893198*u*w*y + 
     0.17096212544446912*u2*w*y - 0.06170167867079261*u3*w*y + 
     0.014252947529037665*u4*w*y - 0.0015359082311407401*u5*w*y - 
     0.4982639811894151*v*w*y + 0.1472496149014935*u*v*w*y - 
     0.04263645884832675*u2*v*w*y + 0.008655813732616391*u3*v*w*y - 
     0.0008582761349145647*u4*v*w*y + 0.25855118000886335*v2*w*y - 
     0.04772361051737147*u*v2*w*y + 0.00780375397440978*u2*v2*w*y - 
     0.0006817048264512694*u3*v2*w*y - 0.0985061281475928*v3*w*y + 
     0.01023088439838107*u*v3*w*y - 0.0007202416621432963*u2*v3*w*y + 
     0.023437600654995495*v4*w*y - 0.0010458019341617988*u*v4*w*y - 
     0.0025714972415127255*v5*w*y - 0.6431554977641232*w2*y + 
     0.18359057678222876*u*w2*y - 0.052439122857018475*u2*w2*y + 
     0.010578663643644204*u3*w2*y - 0.0010454015333102693*u4*w2*y + 
     0.25855118000886335*v*w2*y - 0.04772361051737147*u*v*w2*y + 
     0.00780375397440978*u2*v*w2*y - 0.0006817048264512694*u3*v*w2*y - 
     0.08463412943162905*v2*w2*y + 0.00894003555993513*u*v2*w2*y - 
     0.0006345782786901335*u2*v2*w2*y + 0.018271879935025635*v3*w2*y - 
     0.0008364387244659098*u*v3*w2*y - 0.0018766602068817076*v4*w2*y + 
     0.39191746801048677*w3*y - 0.06841558659701322*u*w3*y + 
     0.010927323983458038*u2*w3*y - 0.0009436436586848417*u3*w3*y - 
     0.0985061281475928*v*w3*y + 0.01023088439838107*u*v*w3*y - 
     0.0007202416621432963*u2*v*w3*y + 0.018271879935025635*v2*w3*y - 
     0.0008364387244659098*u*v2*w3*y - 0.0017038721568713427*v3*w3*y - 
     0.16545503028959443*w4*y + 0.016074508870027843*u*w4*y - 
     0.0010989319337058666*u2*w4*y + 0.023437600654995495*v*w4*y - 
     0.0010458019341617988*u*v*w4*y - 0.0018766602068817076*v2*w4*y + 
     0.04216772179515423*w5*y - 0.0017498755815456293*u*w5*y - 
     0.0025714972415127255*v*w5*y - 0.004855915459192057*w6*y + 
     0.8408174973812163*x*y - 0.3693696142893198*u*x*y + 
     0.17096212544446912*u2*x*y - 0.06170167867079261*u3*x*y + 
     0.014252947529037665*u4*x*y - 0.0015359082311407401*u5*x*y + 
     5.050947856234655*v*x*y + 1.9532560322412158*lnbara*v*x*y + 
     3.6089086211615613*u*v*x*y + 1.0155813225862613*lnbara*u*v*x*y + 
     1.6153412124624802*u2*v*x*y + 0.49610466935343467*lnbara*u2*v*x*y + 
     0.39176531410887067*u3*v*x*y + 0.10319148379269107*lnbara*u3*v*x*y + 
     0.037264929316591394*u4*v*x*y + 0.01035774990633652*lnbara*u4*v*x*y - 
     4.63804380635014*v2*x*y - 1.7188373548274771*lnbara*v2*x*y - 
     3.7155448088414325*u*v2*x*y - 1.101558132258626*lnbara*u*v2*x*y - 
     1.279932716374738*u2*v2*x*y - 0.37253832538857234*lnbara*u2*v2*x*y - 
     0.16259660502488668*u3*v2*x*y - 0.043853270547930576*lnbara*u3*v2*x*
      y + 6.321775087715999*v3*x*y + 2.0595906539260844*lnbara*v3*x*y + 
     1.1114419930785513*u*v3*x*y + 0.30677398634465897*lnbara*u*v3*x*y + 
     0.5189008798963198*u2*v3*x*y + 0.14879519944448213*lnbara*u2*v3*x*
      y - 2.3697386855833713*v4*x*y - 0.7507653584772783*lnbara*v4*x*y - 
     0.2989574253995812*u*v4*x*y - 0.0809778929894431*lnbara*u*v4*x*y + 
     0.4787256919895696*v5*x*y + 0.1438457497350551*lnbara*v5*x*y - 
     5.814779756605509*w*x*y - 1.9532560322412158*lnbara*w*x*y - 
     3.3297094342481097*u*w*x*y - 1.0155813225862613*lnbara*u*w*x*y - 
     1.7018138873778625*u2*w*x*y - 0.49610466935343467*lnbara*u2*w*x*y - 
     0.373690675987211*u3*w*x*y - 0.10319148379269107*lnbara*u3*w*x*y - 
     0.03908459842732547*u4*w*x*y - 0.01035774990633652*lnbara*u4*w*x*y + 
     0.1472496149014935*v*w*x*y - 0.03663932564661811*u*v*w*x*y + 
     0.006668952902327549*u2*v*w*x*y - 0.0006126156464844033*u3*v*w*x*y - 
     6.7414545262279955*v2*w*x*y - 2.014469236002691*lnbara*v2*w*x*y + 
     2.582819839347*u*v2*w*x*y + 0.8128752507655377*lnbara*u*v2*w*x*y - 
     0.30225971597078544*u2*v2*w*x*y - 0.09017258826251653*lnbara*u2*v2*w*
      x*y + 0.7227291601125875*v3*w*x*y + 0.20757337570539203*lnbara*v3*w*x*
      y - 0.368445017992858*u*v3*w*x*y - 0.11464743468556707*lnbara*u*v3*w*
      x*y - 0.3710874532554622*v4*w*x*y - 0.10073935030932556*lnbara*v4*w*x*
      y + 4.998355981731551*w2*x*y + 1.7188373548274771*lnbara*w2*x*y + 
     3.6274999402312313*u*w2*x*y + 1.101558132258626*lnbara*u*w2*x*y + 
     1.2958237235724512*u2*w2*x*y + 0.37253832538857234*lnbara*u2*w2*x*
      y + 0.1611431319221526*u3*w2*x*y + 0.043853270547930576*lnbara*u3*
      w2*x*y + 6.647579498888072*v*w2*x*y + 2.014469236002691*lnbara*v*w2*
      x*y - 2.5690901893905513*u*v*w2*x*y - 0.8128752507655377*lnbara*u*v*
      w2*x*y + 0.3011577157359468*u2*v*w2*x*y + 0.09017258826251653*lnbara*
      u2*v*w2*x*y + 0.008768669407230721*v2*w2*x*y - 
     0.0005708021200927833*u*v2*w2*x*y + 0.8908183968952704*v3*w2*x*y + 
     0.2518959384495211*lnbara*v3*w2*x*y - 6.4531586807126216*w3*x*y - 
     2.0595906539260844*lnbara*w3*x*y - 1.0928490356968497*u*w3*x*y - 
     0.30677398634465897*lnbara*u*w3*x*y - 0.5203695633112709*u2*w3*x*y - 
     0.14879519944448213*lnbara*u2*w3*x*y - 0.7027857105758675*v*w3*x*y - 
     0.20757337570539203*lnbara*v*w3*x*y + 0.3671622432587578*u*v*w3*x*y + 
     0.11464743468556707*lnbara*u*v*w3*x*y - 0.8924547277374686*v2*w3*x*
      y - 0.2518959384495211*lnbara*v2*w3*x*y + 2.4002692887025985*w4*x*
      y + 0.7507653584772783*lnbara*w4*x*y + 0.29707382410626676*u*w4*x*y + 
     0.0809778929894431*lnbara*u*w4*x*y + 0.36905978156971897*v*w4*x*y + 
     0.10073935030932556*lnbara*v*w4*x*y - 0.48202800643018173*w5*x*y - 
     0.1438457497350551*lnbara*w5*x*y - 0.6431554977641232*x2*y + 
     0.18359057678222876*u*x2*y - 0.052439122857018475*u2*x2*y + 
     0.010578663643644204*u3*x2*y - 0.0010454015333102693*u4*x2*y - 
     4.63804380635014*v*x2*y - 1.7188373548274771*lnbara*v*x2*y - 
     3.7155448088414325*u*v*x2*y - 1.101558132258626*lnbara*u*v*x2*y - 
     1.279932716374738*u2*v*x2*y - 0.37253832538857234*lnbara*u2*v*x2*y - 
     0.16259660502488668*u3*v*x2*y - 0.043853270547930576*lnbara*u3*v*x2*
      y + 0.3848527108952509*v2*x2*y + 0.27079839140234246*lnbara*v2*x2*
      y + 4.968594829946177*u*v2*x2*y + 1.491398063789174*lnbara*u*v2*x2*
      y + 0.2124983335725487*u2*v2*x2*y + 0.0544834669601152*lnbara*u2*
      v2*x2*y - 1.6610384150846027*v3*x2*y - 0.5456870809017446*lnbara*
      v3*x2*y - 0.9061306963251085*u*v3*x2*y - 0.2658866648025671*lnbara*
      u*v3*x2*y + 0.1309550363078858*v4*x2*y + 0.04900052281465277*lnbara*
      v4*x2*y + 4.998355981731551*w*x2*y + 1.7188373548274771*lnbara*w*x2*
      y + 3.6274999402312313*u*w*x2*y + 1.101558132258626*lnbara*u*w*x2*y + 
     1.2958237235724512*u2*w*x2*y + 0.37253832538857234*lnbara*u2*w*x2*
      y + 0.1611431319221526*u3*w*x2*y + 0.043853270547930576*lnbara*u3*w*
      x2*y + 4.806051314717304*v*w*x2*y + 1.4974962913849321*lnbara*v*w*x2*
      y - 4.358084874395133*u*v*w*x2*y - 1.310301658288488*lnbara*u*v*w*x2*
      y + 0.44246231471333614*u2*v*w*x2*y + 0.13612276309543284*lnbara*u2*
      v*w*x2*y + 0.16780437876971865*v2*w*x2*y + 
     0.05368132013057931*lnbara*v2*w*x2*y + 0.00017982275152600617*u*v2*w*
      x2*y - 0.008148863899001088*lnbara*u*v2*w*x2*y + 
     0.8410658901759228*v3*w*x2*y + 0.24067518005948615*lnbara*v3*w*x2*
      y - 5.33566764662009*w2*x2*y - 1.7682946827872745*lnbara*w2*x2*y - 
     0.5880364469542249*u*w2*x2*y - 0.18109640550068581*lnbara*u*w2*x2*
      y - 0.6568129911274124*u2*w2*x2*y - 0.19060623005554803*lnbara*u2*
      w2*x2*y - 0.19274016063176588*v*w2*x2*y - 
     0.09505426720402026*lnbara*v*w2*x2*y + 0.5522979595539836*u*v*w2*x2*
      y + 0.17560355760261687*lnbara*u*v*w2*x2*y - 
     1.0211916139594837*v2*w2*x2*y - 0.2879966174194479*lnbara*v2*w2*x2*
      y + 1.7216900283016308*w3*x2*y + 0.5870600279751855*lnbara*w3*x2*
      y + 0.35111850002088596*u*w3*x2*y + 0.09843197109895134*lnbara*u*w3*
      x2*y + 0.4722390677423313*v*w3*x2*y + 0.13175034334097716*lnbara*v*
      w3*x2*y - 0.4270870728419753*w4*x2*y - 0.13342942879566821*lnbara*
      w4*x2*y + 0.39191746801048677*x3*y - 0.06841558659701322*u*x3*y + 
     0.010927323983458038*u2*x3*y - 0.0009436436586848417*u3*x3*y + 
     6.321775087715999*v*x3*y + 2.0595906539260844*lnbara*v*x3*y + 
     1.1114419930785513*u*v*x3*y + 0.30677398634465897*lnbara*u*v*x3*y + 
     0.5189008798963198*u2*v*x3*y + 0.14879519944448213*lnbara*u2*v*x3*
      y - 1.6610384150846027*v2*x3*y - 0.5456870809017446*lnbara*v2*x3*
      y - 0.9061306963251085*u*v2*x3*y - 0.2658866648025671*lnbara*u*v2*
      x3*y + 0.8429911783197556*v3*x3*y + 0.2503190400812598*lnbara*v3*
      x3*y - 6.4531586807126216*w*x3*y - 2.0595906539260844*lnbara*w*x3*
      y - 1.0928490356968497*u*w*x3*y - 0.30677398634465897*lnbara*u*w*x3*
      y - 0.5203695633112709*u2*w*x3*y - 0.14879519944448213*lnbara*u2*w*
      x3*y - 0.03145999950209565*v*w*x3*y - 0.041372947073440955*lnbara*v*w*
      x3*y + 0.5529561454818885*u*v*w*x3*y + 0.1674546937036158*lnbara*u*v*
      w*x3*y - 0.9153400045660104*v2*w*x3*y - 0.25553842398615445*lnbara*
      v2*w*x3*y + 1.7216900283016308*w2*x3*y + 0.5870600279751855*lnbara*
      w2*x3*y + 0.35111850002088596*u*w2*x3*y + 
     0.09843197109895134*lnbara*u*w2*x3*y + 0.569975957295371*v*w2*x3*y + 
     0.15957579625776946*lnbara*v*w2*x3*y - 0.5006975185477412*w3*x3*y - 
     0.1543564123528748*lnbara*w3*x3*y - 0.16545503028959443*x4*y + 
     0.016074508870027843*u*x4*y - 0.0010989319337058666*u2*x4*y - 
     2.3697386855833713*v*x4*y - 0.7507653584772783*lnbara*v*x4*y - 
     0.2989574253995812*u*v*x4*y - 0.0809778929894431*lnbara*u*v*x4*y + 
     0.1309550363078858*v2*x4*y + 0.04900052281465277*lnbara*v2*x4*y + 
     2.4002692887025985*w*x4*y + 0.7507653584772783*lnbara*w*x4*y + 
     0.29707382410626676*u*w*x4*y + 0.0809778929894431*lnbara*u*w*x4*y + 
     0.2932474952337439*v*w*x4*y + 0.08442890598101545*lnbara*v*w*x4*y - 
     0.4270870728419753*w2*x4*y - 0.13342942879566821*lnbara*w2*x4*y + 
     0.04216772179515423*x5*y - 0.0017498755815456293*u*x5*y + 
     0.4787256919895696*v*x5*y + 0.1438457497350551*lnbara*v*x5*y - 
     0.48202800643018173*w*x5*y - 0.1438457497350551*lnbara*w*x5*y - 
     0.004855915459192057*x6*y + 1.5830194232420491*y2 - 
     0.4844211143964337*u*y2 + 0.1983265686310733*u2*y2 - 
     0.06789863070110878*u3*y2 + 0.01526675280034565*u4*y2 - 
     0.001619470152478382*u5*y2 - 0.6431554977641232*v*y2 + 
     0.17096212544446912*u*v*y2 - 0.04689698042164179*u2*v*y2 + 
     0.009251894041935034*u3*v*y2 - 0.0009023290305323929*u4*v*y2 + 
     0.29720782742626456*v2*y2 - 0.052439122857018475*u*v2*y2 + 
     0.008332760830303598*u2*v2*y2 - 0.0007154743025816235*u3*v2*y2 - 
     0.10781348756592685*v3*y2 + 0.010927323983458038*u*v3*y2 - 
     0.000755870097742802*u2*v3*y2 + 0.02501436947958356*v4*y2 - 
     0.0010989319337058666*u*v4*y2 - 0.0027042708881657167*v5*y2 - 
     0.6431554977641232*w*y2 + 0.17096212544446912*u*w*y2 - 
     0.04689698042164179*u2*w*y2 + 0.009251894041935034*u3*w*y2 - 
     0.0009023290305323929*u4*w*y2 + 0.25855118000886335*v*w*y2 - 
     0.04615141682255245*u*v*w*y2 + 0.007376277398255051*u2*v*w*y2 - 
     0.0006353292970921247*u3*v*w*y2 - 0.08463412943162905*v2*w*y2 + 
     0.008768669407230721*u*v2*w*y2 - 0.0006133788582035897*u2*v2*w*y2 + 
     0.018271879935025635*v3*w*y2 - 0.0008257705772076173*u*v3*w*y2 - 
     0.0018766602068817076*v4*w*y2 + 0.29720782742626456*w2*y2 - 
     0.052439122857018475*u*w2*y2 + 0.008332760830303598*u2*w2*y2 - 
     0.0007154743025816235*u3*w2*y2 - 0.08463412943162905*v*w2*y2 + 
     0.008768669407230721*u*v*w2*y2 - 0.0006133788582035897*u2*v*w2*y2 + 
     0.01657980243935387*v2*w2*y2 - 0.0007553212077869194*u*v2*w2*y2 - 
     0.0015904265351103456*v3*w2*y2 - 0.10781348756592685*w3*y2 + 
     0.010927323983458038*u*w3*y2 - 0.000755870097742802*u2*w3*y2 + 
     0.018271879935025635*v*w3*y2 - 0.0008257705772076173*u*v*w3*y2 - 
     0.0015904265351103456*v2*w3*y2 + 0.02501436947958356*w4*y2 - 
     0.0010989319337058666*u*w4*y2 - 0.0018766602068817076*v*w4*y2 - 
     0.0027042708881657167*w5*y2 - 0.6431554977641232*x*y2 + 
     0.17096212544446912*u*x*y2 - 0.04689698042164179*u2*x*y2 + 
     0.009251894041935034*u3*x*y2 - 0.0009023290305323929*u4*x*y2 - 
     2.777512025251304*v*x*y2 - 0.9766280161206079*lnbara*v*x*y2 + 
     0.4390247991246611*u*v*x*y2 + 0.18363961128442552*lnbara*u*v*x*y2 + 
     0.3502072962213621*u2*v*x*y2 + 0.09418475573467341*lnbara*u2*v*x*
      y2 + 0.055393310008639646*u3*v*x*y2 + 0.015395299386842583*lnbara*
      u3*v*x*y2 - 0.08497562276797721*v2*x*y2 + 
     0.027337198387939212*lnbara*v2*x*y2 - 0.7526477768237668*u*v2*x*y2 - 
     0.22874938189748867*lnbara*u*v2*x*y2 - 0.1770896870636047*u2*v2*x*
      y2 - 0.04860228502711452*lnbara*u2*v2*x*y2 - 
     0.8226148783091756*v3*x*y2 - 0.248860063558088*lnbara*v3*x*y2 + 
     0.33716810694416005*u*v3*x*y2 + 0.09925662865561737*lnbara*u*v3*x*
      y2 + 0.04549876822052544*v4*x*y2 + 0.015411950745225283*lnbara*v4*x*
      y2 + 3.150452651970475*w*x*y2 + 0.9766280161206079*lnbara*w*x*y2 - 
     0.5219825160658177*u*w*x*y2 - 0.18363961128442552*lnbara*u*w*x*y2 - 
     0.335966498688551*u2*w*x*y2 - 0.09418475573467341*lnbara*u2*w*x*y2 - 
     0.05665911334956437*u3*w*x*y2 - 0.015395299386842583*lnbara*u3*w*x*
      y2 - 0.04772361051737147*v*w*x*y2 + 0.006668952902327549*u*v*w*x*
      y2 - 0.000520176702770554*u2*v*w*x*y2 + 4.165985435140145*v2*w*x*
      y2 + 1.2223037533945342*lnbara*v2*w*x*y2 - 
     0.15906541536540864*u*v2*w*x*y2 - 0.05241767434109274*lnbara*u*v2*w*x*
      y2 - 0.5092031405619221*v3*w*x*y2 - 0.15134262670003695*lnbara*v3*w*
      x*y2 - 0.022693681826607907*w2*x*y2 - 0.027337198387939212*lnbara*
      w2*x*y2 + 0.7675738685886592*u*w2*x*y2 + 0.22874938189748867*lnbara*
      u*w2*x*y2 + 0.17593076120022283*u2*w2*x*y2 + 
     0.04860228502711452*lnbara*u2*w2*x*y2 - 4.148276730172979*v*w2*x*
      y2 - 1.2223037533945342*lnbara*v*w2*x*y2 + 
     0.15794313861675344*u*v*w2*x*y2 + 0.05241767434109274*lnbara*u*v*w2*x*
      y2 - 0.0007553212077869194*v2*w2*x*y2 + 0.8444774478932007*w3*x*
      y2 + 0.248860063558088*lnbara*w3*x*y2 - 0.3385312614440469*u*w3*x*
      y2 - 0.09925662865561737*lnbara*u*w3*x*y2 + 
     0.5075561415724656*v*w3*x*y2 + 0.15134262670003695*lnbara*v*w3*x*
      y2 - 0.047668105888734366*w4*x*y2 - 0.015411950745225283*lnbara*w4*
      x*y2 + 0.29720782742626456*x2*y2 - 0.052439122857018475*u*x2*y2 + 
     0.008332760830303598*u2*x2*y2 - 0.0007154743025816235*u3*x2*y2 - 
     0.08497562276797721*v*x2*y2 + 0.027337198387939212*lnbara*v*x2*y2 - 
     0.7526477768237668*u*v*x2*y2 - 0.22874938189748867*lnbara*u*v*x2*
      y2 - 0.1770896870636047*u2*v*x2*y2 - 0.04860228502711452*lnbara*u2*
      v*x2*y2 + 1.9461607406157502*v2*x2*y2 + 0.566882499277725*lnbara*
      v2*x2*y2 + 0.021416517678162092*u*v2*x2*y2 + 
     0.005626359452000324*lnbara*u*v2*x2*y2 - 0.2056752271955055*v3*x2*
      y2 - 0.06026111093570525*lnbara*v3*x2*y2 - 
     0.022693681826607907*w*x2*y2 - 0.027337198387939212*lnbara*w*x2*y2 + 
     0.7675738685886592*u*w*x2*y2 + 0.22874938189748867*lnbara*u*w*x2*
      y2 + 0.17593076120022283*u2*w*x2*y2 + 0.04860228502711452*lnbara*
      u2*w*x2*y2 - 2.5782754794730662*v*w*x2*y2 - 
     0.7595995680938061*lnbara*v*w*x2*y2 + 0.6316944039353403*u*v*w*x2*
      y2 + 0.18411867864157963*lnbara*u*v*w*x2*y2 - 
     0.21678745401254226*v2*w*x2*y2 - 0.06933321435793802*lnbara*v2*w*x2*
      y2 + 0.6573559458240761*w2*x2*y2 + 0.19271706881608106*lnbara*w2*
      x2*y2 - 0.6548222450121166*u*w2*x2*y2 - 0.18974503809357995*lnbara*
      u*w2*x2*y2 + 0.37199049657356653*v*w2*x2*y2 + 
     0.11960991850943639*lnbara*v*w2*x2*y2 + 0.047708646825452936*w3*x2*
      y2 + 0.00998440678420688*lnbara*w3*x2*y2 - 
     0.10781348756592685*x3*y2 + 0.010927323983458038*u*x3*y2 - 
     0.000755870097742802*u2*x3*y2 - 0.8226148783091756*v*x3*y2 - 
     0.248860063558088*lnbara*v*x3*y2 + 0.33716810694416005*u*v*x3*y2 + 
     0.09925662865561737*lnbara*u*v*x3*y2 - 0.2056752271955055*v2*x3*
      y2 - 0.06026111093570525*lnbara*v2*x3*y2 + 
     0.8444774478932007*w*x3*y2 + 0.248860063558088*lnbara*w*x3*y2 - 
     0.3385312614440469*u*w*x3*y2 - 0.09925662865561737*lnbara*u*w*x3*
      y2 + 0.15576929487146673*v*w*x3*y2 + 0.05027670415149837*lnbara*v*w*
      x3*y2 + 0.047708646825452936*w2*x3*y2 + 0.00998440678420688*lnbara*
      w2*x3*y2 + 0.02501436947958356*x4*y2 - 0.0010989319337058666*u*x4*
      y2 + 0.04549876822052544*v*x4*y2 + 0.015411950745225283*lnbara*v*x4*
      y2 - 0.047668105888734366*w*x4*y2 - 0.015411950745225283*lnbara*w*
      x4*y2 - 0.0027042708881657167*x5*y2 - 1.6193513369191062*y3 + 
     0.28572760618190396*u*y3 - 0.06789863070110878*u2*y3 + 
     0.012574224677698281*u3*y3 - 0.001186695305274723*u4*y3 + 
     0.39191746801048677*v*y3 - 0.06170167867079261*u*v*y3 + 
     0.009251894041935034*u2*v*y3 - 0.0007701655588751243*u3*v*y3 - 
     0.10781348756592685*v2*y3 + 0.010578663643644204*u*v2*y3 - 
     0.0007154743025816235*u2*v2*y3 + 0.021502574570336913*v3*y3 - 
     0.0009436436586848417*u*v3*y3 - 0.002116028409986158*v4*y3 + 
     0.39191746801048677*w*y3 - 0.06170167867079261*u*w*y3 + 
     0.009251894041935034*u2*w*y3 - 0.0007701655588751243*u3*w*y3 - 
     0.0985061281475928*v*w*y3 + 0.009712565138338942*u*v*w*y3 - 
     0.00065880688954107*u2*v*w*y3 + 0.018271879935025635*v2*w*y3 - 
     0.0008105602649905572*u*v2*w*y3 - 0.0017038721568713427*v3*w*y3 - 
     0.10781348756592685*w2*y3 + 0.010578663643644204*u*w2*y3 - 
     0.0007154743025816235*u2*w2*y3 + 0.018271879935025635*v*w2*y3 - 
     0.0008105602649905572*u*v*w2*y3 - 0.0015904265351103456*v2*w2*y3 + 
     0.021502574570336913*w3*y3 - 0.0009436436586848417*u*w3*y3 - 
     0.0017038721568713427*v*w3*y3 - 0.002116028409986158*w4*y3 + 
     0.39191746801048677*x*y3 - 0.06170167867079261*u*x*y3 + 
     0.009251894041935034*u2*x*y3 - 0.0007701655588751243*u3*x*y3 + 
     2.7762869962453007*v*x*y3 + 0.878965214508547*lnbara*v*x*y3 + 
     0.472929670266122*u*v*x*y3 + 0.11978293118391894*lnbara*u*v*x*y3 + 
     0.06433729180422629*u2*v*x*y3 + 0.018012828737752302*lnbara*u2*v*x*
      y3 - 0.6965292598704438*v2*x*y3 - 0.2252713360201013*lnbara*v2*x*
      y3 - 0.14004639793591991*u*v2*x*y3 - 0.03825655240654065*lnbara*u*
      v2*x*y3 + 0.3560786661881267*v3*x*y3 + 0.1031812587805062*lnbara*
      v3*x*y3 - 2.914384497168144*w*x*y3 - 0.878965214508547*lnbara*w*x*
      y3 - 0.4559117835501848*u*w*x*y3 - 0.11978293118391894*lnbara*u*w*x*
      y3 - 0.06557961755270206*u2*w*x*y3 - 0.018012828737752302*lnbara*u2*
      w*x*y3 + 0.01023088439838107*v*w*x*y3 - 0.0006126156464844033*u*v*w*x*
      y3 - 0.6290677486101003*v2*w*x*y3 - 0.17563835536198544*lnbara*v2*w*
      x*y3 + 0.7187404897942826*w2*x*y3 + 0.2252713360201013*lnbara*w2*x*
      y3 + 0.1387217802717251*u*w2*x*y3 + 0.03825655240654065*lnbara*u*w2*
      x*y3 + 0.6274055393084268*v*w2*x*y3 + 0.17563835536198544*lnbara*v*
      w2*x*y3 - 0.35800549766301815*w3*x*y3 - 0.1031812587805062*lnbara*
      w3*x*y3 - 0.10781348756592685*x2*y3 + 0.010578663643644204*u*x2*
      y3 - 0.0007154743025816235*u2*x2*y3 - 0.6965292598704438*v*x2*
      y3 - 0.2252713360201013*lnbara*v*x2*y3 - 0.14004639793591991*u*v*x2*
      y3 - 0.03825655240654065*lnbara*u*v*x2*y3 - 
     0.09998366236526494*v2*x2*y3 - 0.024091921466539296*lnbara*v2*x2*
      y3 + 0.7187404897942826*w*x2*y3 + 0.2252713360201013*lnbara*w*x2*
      y3 + 0.1387217802717251*u*w*x2*y3 + 0.03825655240654065*lnbara*u*w*
      x2*y3 + 0.4419629692064293*v*w*x2*y3 + 0.124791357615411*lnbara*v*w*
      x2*y3 - 0.3442555249706034*w2*x2*y3 - 0.10069943614887171*lnbara*
      w2*x2*y3 + 0.021502574570336913*x3*y3 - 0.0009436436586848417*u*
      x3*y3 + 0.3560786661881267*v*x3*y3 + 0.1031812587805062*lnbara*v*
      x3*y3 - 0.35800549766301815*w*x3*y3 - 0.1031812587805062*lnbara*w*
      x3*y3 - 0.002116028409986158*x4*y3 + 1.1560019673631434*y4 - 
     0.11864471061508094*u*y4 + 0.01526675280034565*u2*y4 - 
     0.001186695305274723*u3*y4 - 0.16545503028959443*v*y4 + 
     0.014252947529037665*u*v*y4 - 0.0009023290305323929*u2*v*y4 + 
     0.02501436947958356*v2*y4 - 0.0010454015333102693*u*v2*y4 - 
     0.002116028409986158*v3*y4 - 0.16545503028959443*w*y4 + 
     0.014252947529037665*u*w*y4 - 0.0009023290305323929*u2*w*y4 + 
     0.023437600654995495*v*w*y4 - 0.0009818697515814416*u*v*w*y4 - 
     0.0018766602068817076*v2*w*y4 + 0.02501436947958356*w2*y4 - 
     0.0010454015333102693*u*w2*y4 - 0.0018766602068817076*v*w2*y4 - 
     0.002116028409986158*w3*y4 - 0.16545503028959443*x*y4 + 
     0.014252947529037665*u*x*y4 - 0.0009023290305323929*u2*x*y4 - 
     0.8149620980179276*v*x*y4 - 0.24415700403015198*lnbara*v*x*y4 - 
     0.010048357709481526*u*v*x*y4 - 0.0006955947583152419*lnbara*u*v*x*
      y4 + 0.03885294084404612*v2*x*y4 + 0.013253738191194095*lnbara*v2*x*
      y4 + 0.8473142624781451*w*x*y4 + 0.24415700403015198*lnbara*w*x*y4 + 
     0.008352282215414328*u*w*x*y4 + 0.0006955947583152419*lnbara*u*w*x*
      y4 - 0.0010458019341617988*v*w*x*y4 - 0.04107580891265065*w2*x*y4 - 
     0.013253738191194095*lnbara*w2*x*y4 + 0.02501436947958356*x2*y4 - 
     0.0010454015333102693*u*x2*y4 + 0.03885294084404612*v*x2*y4 + 
     0.013253738191194095*lnbara*v*x2*y4 - 0.04107580891265065*w*x2*y4 - 
     0.013253738191194095*lnbara*w*x2*y4 - 0.002116028409986158*x3*y4 - 
     0.5375082140631944*y5 + 0.029948825102663987*u*y5 - 
     0.001619470152478382*u2*y5 + 0.04216772179515423*v*y5 - 
     0.0015359082311407401*u*v*y5 - 0.0027042708881657167*v2*y5 + 
     0.04216772179515423*w*y5 - 0.0015359082311407401*u*w*y5 - 
     0.0025714972415127255*v*w*y5 - 0.0027042708881657167*w2*y5 + 
     0.04216772179515423*x*y5 - 0.0015359082311407401*u*x*y5 + 
     0.13433805924390965*v*x*y5 + 0.03836752920473817*lnbara*v*x*y5 - 
     0.13785434103492666*w*x*y5 - 0.03836752920473817*lnbara*w*x*y5 - 
     0.0027042708881657167*x2*y5 + 0.14606611930718463*y6 - 
     0.0034277086455240437*u*y6 - 0.004855915459192057*v*y6 - 
     0.004855915459192057*w*y6 - 0.004855915459192057*x*y6 - 
     0.017602106974984624*y7 - 1.2020569031595942*z + 
     0.8408174973812163*u*z - 0.6431554977641232*u2*z + 
     0.39191746801048677*u3*z - 0.16545503028959443*u4*z + 
     0.04216772179515423*u5*z - 0.004855915459192057*u6*z + 
     0.6830325829486028*v*z - 2.1845063283947157*u*v*z - 
     0.6510853440804052*lnbara*u*v*z - 1.2999471003253766*u2*v*z - 
     0.4470413315421547*lnbara*u2*v*z - 0.6741348014198821*u3*v*z - 
     0.17751808906800676*lnbara*u3*v*z - 0.10064161922879965*u4*v*z - 
     0.031382877263858105*lnbara*u4*v*z - 0.011350747467801053*u5*v*z - 
     0.0025756149427488186*lnbara*u5*v*z + 1.3307155997089624*v2*z + 
     0.6510853440804052*lnbara*v2*z + 3.593282429668685*u*v2*z + 
     1.128501340498048*lnbara*u*v2*z + 1.9298767034120956*u2*v2*z + 
     0.5924457327959798*lnbara*u2*v2*z + 0.5260162467363529*u3*v2*z + 
     0.14593979274603425*lnbara*u3*v2*z + 0.049642889715863496*u4*v2*z + 
     0.01365467547098468*lnbara*u4*v2*z - 0.3344496541843748*v3*z - 
     0.24740311290228975*lnbara*v3*z - 3.884956834618231*u*v3*z - 
     1.2186672004102037*lnbara*u*v3*z - 0.47396206592202933*u2*v3*z - 
     0.13192114982378386*lnbara*u2*v3*z - 0.23348793049311065*u3*v3*z - 
     0.06654355140807486*lnbara*u3*v3*z + 0.8622872883244641*v4*z + 
     0.3193208792684922*lnbara*v4*z + 1.3489683495777027*u*v4*z + 
     0.4144777234484366*lnbara*u*v4*z + 0.1660137847279631*u2*v4*z + 
     0.0453330032683975*lnbara*u2*v4*z - 0.2093175862405338*v5*z - 
     0.07853883800228331*lnbara*v5*z - 0.27798071288879395*u*v5*z - 
     0.08187135582602693*lnbara*u*v5*z + 0.04158747525391902*v6*z + 
     0.013727259996145989*lnbara*v6*z + 0.8408174973812163*w*z + 
     1.316872732915981*u*w*z + 0.6510853440804052*lnbara*u*w*z + 
     1.7420888571164688*u2*w*z + 0.4470413315421547*lnbara*u2*w*z + 
     0.5072130866752762*u3*w*z + 0.17751808906800676*lnbara*u3*w*z + 
     0.14015372875382298*u4*w*z + 0.031382877263858105*lnbara*u4*w*z + 
     0.007029374644742698*u5*w*z + 0.0025756149427488186*lnbara*u5*w*z - 
     0.3693696142893198*v*w*z + 0.1472496149014935*u*v*w*z - 
     0.04772361051737147*u2*v*w*z + 0.01023088439838107*u3*v*w*z - 
     0.0010458019341617988*u4*v*w*z - 9.679452702174132*v2*w*z - 
     3.285801385277514*lnbara*v2*w*z + 3.2592144539581844*u*v2*w*z + 
     1.1013272044547329*lnbara*u*v2*w*z - 3.229242426355708*u2*v2*w*z - 
     0.9917935723502516*lnbara*u2*v2*w*z + 0.404386452649353*u3*v2*w*z + 
     0.12233847935377173*lnbara*u3*v2*w*z + 5.035142069385885*v3*w*z + 
     1.7095780955682178*lnbara*v3*w*z + 0.16919997458043376*u*v3*w*z + 
     0.013784793500404437*lnbara*u*v3*w*z + 0.4078728401620492*u2*v3*w*z + 
     0.12576336826926637*lnbara*u2*v3*w*z - 3.0012661757807617*v4*w*z - 
     0.9380468690654714*lnbara*v4*w*z + 0.15191010788660775*u*v4*w*z + 
     0.048049503963086576*lnbara*u*v4*w*z + 0.4769026267982312*v5*w*z + 
     0.14541778766038482*lnbara*v5*w*z - 2.4582922118695194*w2*z - 
     0.6510853440804052*lnbara*w2*z - 3.1637691242153525*u*w2*z - 
     1.128501340498048*lnbara*u*w2*z - 2.066949955700743*u2*w2*z - 
     0.5924457327959798*lnbara*u2*w2*z - 0.49681704281786926*u3*w2*z - 
     0.14593979274603425*lnbara*u3*w2*z - 0.05261848185645107*u4*w2*z - 
     0.01365467547098468*lnbara*u4*w2*z + 10.034005404400828*v*w2*z + 
     3.285801385277514*lnbara*v*w2*z - 3.3495745233238825*u*v*w2*z - 
     1.1013272044547329*lnbara*u*v*w2*z + 3.2459862158900528*u2*v*w2*z + 
     0.9917935723502516*lnbara*u2*v*w2*z - 0.40594313303596224*u3*v*w2*
      z - 0.12233847935377173*lnbara*u3*v*w2*z - 0.052439122857018475*v2*
      w2*z + 0.00780375397440978*u*v2*w2*z - 0.0006345782786901335*u2*v2*
      w2*z + 1.8755457504129727*v3*w2*z + 0.5198935985278687*lnbara*v3*
      w2*z - 0.5329098110819325*u*v3*w2*z - 0.1550921261368306*lnbara*u*
      v3*w2*z - 0.11655132895257132*v4*w2*z - 0.02876833807133764*lnbara*
      v4*w2*z + 1.0120947283767654*w3*z + 0.24740311290228975*lnbara*w3*
      z + 3.7247490277998456*u*w3*z + 1.2186672004102037*lnbara*u*w3*z + 
     0.5028126095006992*u2*w3*z + 0.13192114982378386*lnbara*u2*w3*z + 
     0.23084041467755445*u3*w3*z + 0.06654355140807486*lnbara*u3*w3*z - 
     5.165259334653691*v*w3*z - 1.7095780955682178*lnbara*v*w3*z - 
     0.15031327644943632*u*v*w3*z - 0.013784793500404437*lnbara*u*v*w3*z - 
     0.40939098371296645*u2*v*w3*z - 0.12576336826926637*lnbara*u2*v*w3*
      z - 1.8540397627858702*v2*w3*z - 0.5198935985278687*lnbara*v2*w3*
      z + 0.5315078645933379*u*v2*w3*z + 0.1550921261368306*lnbara*u*v2*
      w3*z - 0.0009436436586848417*v3*w3*z - 1.1463870292291396*w4*z - 
     0.3193208792684922*lnbara*w4*z - 1.3112778013936697*u*w4*z - 
     0.4144777234484366*lnbara*u*w4*z - 0.16893584646815507*u2*w4*z - 
     0.0453330032683975*lnbara*u2*w4*z + 3.0315936321798276*v*w4*z + 
     0.9380468690654714*lnbara*v*w4*z - 0.15381418595568414*u*v*w4*z - 
     0.048049503963086576*lnbara*u*v*w4*z + 0.11440699548555519*v2*w4*z + 
     0.02876833807133764*lnbara*v2*w4*z + 0.281434133138352*w5*z + 
     0.07853883800228331*lnbara*w5*z + 0.2738733074161405*u*w5*z + 
     0.08187135582602693*lnbara*u*w5*z - 0.48018841061091755*v*w5*z - 
     0.14541778766038482*lnbara*v*w5*z - 0.04987109935863513*w6*z - 
     0.013727259996145989*lnbara*w6*z + 0.8408174973812163*x*z - 
     0.3944622860815335*u*x*z + 0.18935004993694224*u2*x*z - 
     0.06968191432583015*u3*x*z + 0.0162776555901896*u4*x*z - 
     0.0017664062094713614*u5*x*z - 0.3693696142893198*v*x*z + 
     0.1472496149014935*u*v*x*z - 0.04772361051737147*u2*v*x*z + 
     0.01023088439838107*u3*v*x*z - 0.0010458019341617988*u4*v*x*z - 
     6.422064708307189*v2*x*z - 2.1702844802680175*lnbara*v2*x*z + 
     6.435061406178583*u*v2*x*z + 2.079207074882875*lnbara*u*v2*x*z - 
     3.373810219562036*u2*v2*x*z - 1.0395623490494532*lnbara*u2*v2*x*z + 
     0.5336843597652852*u3*v2*x*z + 0.16066085741485445*lnbara*u3*v2*x*
      z + 2.0505932975012593*v3*x*z + 0.7335375186340615*lnbara*v3*x*z - 
     0.4484159450532304*u*v3*x*z - 0.1922469483445887*lnbara*u*v3*x*z + 
     0.33319811688498185*u2*v3*x*z + 0.10536425342736426*lnbara*u2*v3*x*
      z - 1.8086906966367922*v4*x*z - 0.5574561800298969*lnbara*v4*x*z + 
     0.3835758469523267*u*v4*x*z + 0.116292709475361*lnbara*u*v4*x*z + 
     0.23350760677675905*v5*x*z + 0.07186909727514694*lnbara*v5*x*z - 
     0.3693696142893198*w*x*z + 0.1472496149014935*u*w*x*z - 
     0.04772361051737147*u2*w*x*z + 0.01023088439838107*u3*w*x*z - 
     0.0010458019341617988*u4*w*x*z + 13.318003239515274*v*w*x*z + 
     4.340568960536035*lnbara*v*w*x*z - 12.99906497164889*u*v*w*x*z - 
     4.15841414976575*lnbara*u*v*w*x*z + 6.772218474992654*u2*v*w*x*z + 
     2.0791246980989064*lnbara*u2*v*w*x*z - 1.0696904197726014*u3*v*w*x*z - 
     0.3213217148297089*lnbara*u3*v*w*x*z - 2.152616234264882*v2*w*x*z - 
     0.7335375186340615*lnbara*v2*w*x*z + 0.46479746309389686*u*v2*w*x*z + 
     0.1922469483445887*lnbara*u*v2*w*x*z - 0.3345794792700652*u2*v2*w*x*
      z - 0.10536425342736426*lnbara*u2*v2*w*x*z + 
     4.268816061721532*v3*w*x*z + 1.2629454959136155*lnbara*v3*w*x*z - 
     1.617872252781772*u*v3*w*x*z - 0.46845974278075075*lnbara*u*v3*w*x*z - 
     0.3368289563598407*v4*w*x*z - 0.10155666376810567*lnbara*v4*w*x*z - 
     6.422064708307189*w2*x*z - 2.1702844802680175*lnbara*w2*x*z + 
     6.435061406178583*u*w2*x*z + 2.079207074882875*lnbara*u*w2*x*z - 
     3.373810219562036*u2*w2*x*z - 1.0395623490494532*lnbara*u2*w2*x*z + 
     0.5336843597652852*u3*w2*x*z + 0.16066085741485445*lnbara*u3*w2*x*
      z - 2.152616234264882*v*w2*x*z - 0.7335375186340615*lnbara*v*w2*x*z + 
     0.46479746309389686*u*v*w2*x*z + 0.1922469483445887*lnbara*u*v*w2*x*
      z - 0.3345794792700652*u2*v*w2*x*z - 0.10536425342736426*lnbara*u2*v*
      w2*x*z - 4.86789835135428*v2*w2*x*z - 1.410978631767437*lnbara*v2*
      w2*x*z + 2.4648726427480128*u*v2*w2*x*z + 0.7043340666107796*lnbara*
      u*v2*w2*x*z + 0.10030472915504478*v3*w2*x*z + 
     0.029687566492958728*lnbara*v3*w2*x*z + 2.0505932975012593*w3*x*z + 
     0.7335375186340615*lnbara*w3*x*z - 0.4484159450532304*u*w3*x*z - 
     0.1922469483445887*lnbara*u*w3*x*z + 0.33319811688498185*u2*w3*x*z + 
     0.10536425342736426*lnbara*u2*w3*x*z + 4.268816061721532*v*w3*x*z + 
     1.2629454959136155*lnbara*v*w3*x*z - 1.617872252781772*u*v*w3*x*z - 
     0.46845974278075075*lnbara*u*v*w3*x*z + 0.10030472915504478*v2*w3*x*
      z + 0.029687566492958728*lnbara*v2*w3*x*z - 
     1.8086906966367922*w4*x*z - 0.5574561800298969*lnbara*w4*x*z + 
     0.3835758469523267*u*w4*x*z + 0.116292709475361*lnbara*u*w4*x*z - 
     0.3368289563598407*v*w4*x*z - 0.10155666376810567*lnbara*v*w4*x*z + 
     0.23350760677675905*w5*x*z + 0.07186909727514694*lnbara*w5*x*z - 
     0.6431554977641232*x2*z + 0.18935004993694224*u*x2*z - 
     0.05523018173756664*u2*x2*z + 0.011283905940380838*u3*x2*z - 
     0.0011239361348986583*u4*x2*z + 10.034005404400828*v*x2*z + 
     3.285801385277514*lnbara*v*x2*z - 3.3495745233238825*u*v*x2*z - 
     1.1013272044547329*lnbara*u*v*x2*z + 3.2459862158900528*u2*v*x2*z + 
     0.9917935723502516*lnbara*u2*v*x2*z - 0.40594313303596224*u3*v*x2*
      z - 0.12233847935377173*lnbara*u3*v*x2*z - 1.1288573398553086*v2*x2*
      z - 0.43966795501820194*lnbara*v2*x2*z - 1.202554183086874*u*v2*x2*
      z - 0.3723502771999754*lnbara*u*v2*x2*z - 0.08093775692251794*u2*v2*
      x2*z - 0.025975593895628257*lnbara*u2*v2*x2*z + 
     2.941587800744719*v3*x2*z + 0.8832109828874293*lnbara*v3*x2*z - 
     0.7853652276547294*u*v3*x2*z - 0.21990490700155435*lnbara*u*v3*x2*
      z - 0.21163351738140398*v4*x2*z - 0.06718897907015481*lnbara*v4*x2*
      z - 9.679452702174132*w*x2*z - 3.285801385277514*lnbara*w*x2*z + 
     3.2592144539581844*u*w*x2*z + 1.1013272044547329*lnbara*u*w*x2*z - 
     3.229242426355708*u2*w*x2*z - 0.9917935723502516*lnbara*u2*w*x2*z + 
     0.404386452649353*u3*w*x2*z + 0.12233847935377173*lnbara*u3*w*x2*z - 
     2.152616234264882*v*w*x2*z - 0.7335375186340615*lnbara*v*w*x2*z + 
     0.46479746309389686*u*v*w*x2*z + 0.1922469483445887*lnbara*u*v*w*x2*
      z - 0.3345794792700652*u2*v*w*x2*z - 0.10536425342736426*lnbara*u2*v*
      w*x2*z - 6.030472710174449*v2*w*x2*z - 1.7376394640414514*lnbara*v2*
      w*x2*z + 2.4254958617946483*u*v2*w*x2*z + 0.7008504098797653*lnbara*
      u*v2*w*x2*z + 0.2189313721141216*v3*w*x2*z + 
     0.06175769728307968*lnbara*v3*w*x2*z + 3.1418162127487004*w2*x2*z + 
     1.1732054736522635*lnbara*w2*x2*z + 0.7605706197007904*u*w2*x2*z + 
     0.1801033288553867*lnbara*u*w2*x2*z + 0.41357773400649983*u2*w2*x2*
      z + 0.1313398473229925*lnbara*u2*w2*x2*z + 
     6.391972088595844*v*w2*x2*z + 1.8709274228635118*lnbara*v*w2*x2*z - 
     2.0843530956674345*u*v*w2*x2*z - 0.6095267362125409*lnbara*u*v*w2*x2*
      z - 0.05985191547449978*v2*w2*x2*z - 0.01186851715999764*lnbara*v2*
      w2*x2*z - 3.269269763555497*w3*x2*z - 1.0164989417094896*lnbara*w3*
      x2*z + 0.44168144304165097*u*w3*x2*z + 0.12858123333432991*lnbara*u*
      w3*x2*z - 0.3134017819611535*v*w3*x2*z - 0.0993620511060278*lnbara*
      v*w3*x2*z + 0.3622126558447875*w4*x2*z + 0.11666185005310058*lnbara*
      w4*x2*z + 0.39191746801048677*x3*z - 0.06968191432583015*u*x3*z + 
     0.011283905940380838*u2*x3*z - 0.0009831878162066224*u3*x3*z - 
     5.165259334653691*v*x3*z - 1.7095780955682178*lnbara*v*x3*z - 
     0.15031327644943632*u*v*x3*z - 0.013784793500404437*lnbara*u*v*x3*z - 
     0.40939098371296645*u2*v*x3*z - 0.12576336826926637*lnbara*u2*v*x3*
      z - 1.4645041396124352*v2*x3*z - 0.37498908309415785*lnbara*v2*x3*
      z + 1.25382676464303*u*v2*x3*z + 0.3652162355419701*lnbara*u*v2*x3*
      z - 0.2434872198153215*v3*x3*z - 0.07483548907405872*lnbara*v3*x3*
      z + 5.035142069385885*w*x3*z + 1.7095780955682178*lnbara*w*x3*z + 
     0.16919997458043376*u*w*x3*z + 0.013784793500404437*lnbara*u*w*x3*z + 
     0.4078728401620492*u2*w*x3*z + 0.12576336826926637*lnbara*u2*w*x3*
      z + 4.268816061721532*v*w*x3*z + 1.2629454959136155*lnbara*v*w*x3*z - 
     1.617872252781772*u*v*w*x3*z - 0.46845974278075075*lnbara*u*v*w*x3*z + 
     0.2189313721141216*v2*w*x3*z + 0.06175769728307968*lnbara*v2*w*x3*
      z - 2.775770631100383*w2*x3*z - 0.8879564128194577*lnbara*w2*x3*z + 
     0.3619717726179578*u*w2*x3*z + 0.10324350723878072*lnbara*u*w2*x3*
      z - 0.3134017819611535*v*w2*x3*z - 0.0993620511060278*lnbara*v*w2*
      x3*z + 0.33499391115566624*w3*x3*z + 0.11243984289700684*lnbara*w3*
      x3*z - 0.16545503028959443*x4*z + 0.0162776555901896*u*x4*z - 
     0.0011239361348986583*u2*x4*z + 3.0315936321798276*v*x4*z + 
     0.9380468690654714*lnbara*v*x4*z - 0.15381418595568414*u*v*x4*z - 
     0.048049503963086576*lnbara*u*v*x4*z + 0.0011325127373185734*v2*x4*
      z - 0.009074544432852395*lnbara*v2*x4*z - 3.0012661757807617*w*x4*
      z - 0.9380468690654714*lnbara*w*x4*z + 0.15191010788660775*u*w*x4*z + 
     0.048049503963086576*lnbara*u*w*x4*z - 0.3368289563598407*v*w*x4*z - 
     0.10155666376810567*lnbara*v*w*x4*z + 0.3328573832991313*w2*x4*z + 
     0.11063120820095806*lnbara*w2*x4*z + 0.04216772179515423*x5*z - 
     0.0017664062094713614*u*x5*z - 0.48018841061091755*v*x5*z - 
     0.14541778766038482*lnbara*v*x5*z + 0.4769026267982312*w*x5*z + 
     0.14541778766038482*lnbara*w*x5*z - 0.004855915459192057*x6*z + 
     0.8408174973812163*y*z - 0.3693696142893198*u*y*z + 
     0.17096212544446912*u2*y*z - 0.06170167867079261*u3*y*z + 
     0.014252947529037665*u4*y*z - 0.0015359082311407401*u5*y*z - 
     5.814779756605509*v*y*z - 1.9532560322412158*lnbara*v*y*z - 
     3.3297094342481097*u*v*y*z - 1.0155813225862613*lnbara*u*v*y*z - 
     1.7018138873778625*u2*v*y*z - 0.49610466935343467*lnbara*u2*v*y*z - 
     0.373690675987211*u3*v*y*z - 0.10319148379269107*lnbara*u3*v*y*z - 
     0.03908459842732547*u4*v*y*z - 0.01035774990633652*lnbara*u4*v*y*z + 
     4.998355981731551*v2*y*z + 1.7188373548274771*lnbara*v2*y*z + 
     3.6274999402312313*u*v2*y*z + 1.101558132258626*lnbara*u*v2*y*z + 
     1.2958237235724512*u2*v2*y*z + 0.37253832538857234*lnbara*u2*v2*y*
      z + 0.1611431319221526*u3*v2*y*z + 0.043853270547930576*lnbara*u3*
      v2*y*z - 6.4531586807126216*v3*y*z - 2.0595906539260844*lnbara*v3*y*
      z - 1.0928490356968497*u*v3*y*z - 0.30677398634465897*lnbara*u*v3*y*
      z - 0.5203695633112709*u2*v3*y*z - 0.14879519944448213*lnbara*u2*v3*
      y*z + 2.4002692887025985*v4*y*z + 0.7507653584772783*lnbara*v4*y*z + 
     0.29707382410626676*u*v4*y*z + 0.0809778929894431*lnbara*u*v4*y*z - 
     0.48202800643018173*v5*y*z - 0.1438457497350551*lnbara*v5*y*z + 
     5.050947856234655*w*y*z + 1.9532560322412158*lnbara*w*y*z + 
     3.6089086211615613*u*w*y*z + 1.0155813225862613*lnbara*u*w*y*z + 
     1.6153412124624802*u2*w*y*z + 0.49610466935343467*lnbara*u2*w*y*z + 
     0.39176531410887067*u3*w*y*z + 0.10319148379269107*lnbara*u3*w*y*z + 
     0.037264929316591394*u4*w*y*z + 0.01035774990633652*lnbara*u4*w*y*z + 
     0.1472496149014935*v*w*y*z - 0.03663932564661811*u*v*w*y*z + 
     0.006668952902327549*u2*v*w*y*z - 0.0006126156464844033*u3*v*w*y*z + 
     6.647579498888072*v2*w*y*z + 2.014469236002691*lnbara*v2*w*y*z - 
     2.5690901893905513*u*v2*w*y*z - 0.8128752507655377*lnbara*u*v2*w*y*z + 
     0.3011577157359468*u2*v2*w*y*z + 0.09017258826251653*lnbara*u2*v2*w*
      y*z - 0.7027857105758675*v3*w*y*z - 0.20757337570539203*lnbara*v3*w*y*
      z + 0.3671622432587578*u*v3*w*y*z + 0.11464743468556707*lnbara*u*v3*w*
      y*z + 0.36905978156971897*v4*w*y*z + 0.10073935030932556*lnbara*v4*w*
      y*z - 4.63804380635014*w2*y*z - 1.7188373548274771*lnbara*w2*y*z - 
     3.7155448088414325*u*w2*y*z - 1.101558132258626*lnbara*u*w2*y*z - 
     1.279932716374738*u2*w2*y*z - 0.37253832538857234*lnbara*u2*w2*y*z - 
     0.16259660502488668*u3*w2*y*z - 0.043853270547930576*lnbara*u3*w2*y*
      z - 6.7414545262279955*v*w2*y*z - 2.014469236002691*lnbara*v*w2*y*z + 
     2.582819839347*u*v*w2*y*z + 0.8128752507655377*lnbara*u*v*w2*y*z - 
     0.30225971597078544*u2*v*w2*y*z - 0.09017258826251653*lnbara*u2*v*w2*
      y*z + 0.008768669407230721*v2*w2*y*z - 0.0005708021200927833*u*v2*
      w2*y*z - 0.8924547277374686*v3*w2*y*z - 0.2518959384495211*lnbara*
      v3*w2*y*z + 6.321775087715999*w3*y*z + 2.0595906539260844*lnbara*w3*
      y*z + 1.1114419930785513*u*w3*y*z + 0.30677398634465897*lnbara*u*w3*y*
      z + 0.5189008798963198*u2*w3*y*z + 0.14879519944448213*lnbara*u2*w3*
      y*z + 0.7227291601125875*v*w3*y*z + 0.20757337570539203*lnbara*v*w3*y*
      z - 0.368445017992858*u*v*w3*y*z - 0.11464743468556707*lnbara*u*v*w3*
      y*z + 0.8908183968952704*v2*w3*y*z + 0.2518959384495211*lnbara*v2*
      w3*y*z - 2.3697386855833713*w4*y*z - 0.7507653584772783*lnbara*w4*y*
      z - 0.2989574253995812*u*w4*y*z - 0.0809778929894431*lnbara*u*w4*y*
      z - 0.3710874532554622*v*w4*y*z - 0.10073935030932556*lnbara*v*w4*y*
      z + 0.4787256919895696*w5*y*z + 0.1438457497350551*lnbara*w5*y*z - 
     0.4982639811894151*x*y*z + 0.1472496149014935*u*x*y*z - 
     0.04263645884832675*u2*x*y*z + 0.008655813732616391*u3*x*y*z - 
     0.0008582761349145647*u4*x*y*z + 0.1472496149014935*v*x*y*z - 
     0.03663932564661811*u*v*x*y*z + 0.006668952902327549*u2*v*x*y*z - 
     0.0006126156464844033*u3*v*x*y*z + 4.806051314717304*v2*x*y*z + 
     1.4974962913849321*lnbara*v2*x*y*z - 4.358084874395133*u*v2*x*y*z - 
     1.310301658288488*lnbara*u*v2*x*y*z + 0.44246231471333614*u2*v2*x*y*
      z + 0.13612276309543284*lnbara*u2*v2*x*y*z - 
     0.03145999950209565*v3*x*y*z - 0.041372947073440955*lnbara*v3*x*y*z + 
     0.5529561454818885*u*v3*x*y*z + 0.1674546937036158*lnbara*u*v3*x*y*z + 
     0.2932474952337439*v4*x*y*z + 0.08442890598101545*lnbara*v4*x*y*z + 
     0.1472496149014935*w*x*y*z - 0.03663932564661811*u*w*x*y*z + 
     0.006668952902327549*u2*w*x*y*z - 0.0006126156464844033*u3*w*x*y*z - 
     9.734014872777879*v*w*x*y*z - 2.9949925827698642*lnbara*v*w*x*y*z + 
     8.735313288561295*u*v*w*x*y*z + 2.620603316576976*lnbara*u*v*w*x*y*z - 
     0.8865113815143282*u2*v*w*x*y*z - 0.27224552619086567*lnbara*u2*v*w*x*
      y*z + 0.04678476613703959*v2*w*x*y*z + 0.041372947073440955*lnbara*v2*
      w*x*y*z - 0.5540525639589041*u*v2*w*x*y*z - 0.1674546937036158*lnbara*
      u*v2*w*x*y*z - 1.3152966221164228*v3*w*x*y*z - 
     0.3724255234004633*lnbara*v3*w*x*y*z + 4.806051314717304*w2*x*y*z + 
     1.4974962913849321*lnbara*w2*x*y*z - 4.358084874395133*u*w2*x*y*z - 
     1.310301658288488*lnbara*u*w2*x*y*z + 0.44246231471333614*u2*w2*x*y*
      z + 0.13612276309543284*lnbara*u2*w2*x*y*z + 
     0.04678476613703959*v*w2*x*y*z + 0.041372947073440955*lnbara*v*w2*x*y*
      z - 0.5540525639589041*u*v*w2*x*y*z - 0.1674546937036158*lnbara*u*v*
      w2*x*y*z + 2.0406049955739975*v2*w2*x*y*z + 
     0.5759932348388958*lnbara*v2*w2*x*y*z - 0.03145999950209565*w3*x*y*
      z - 0.041372947073440955*lnbara*w3*x*y*z + 0.5529561454818885*u*w3*x*
      y*z + 0.1674546937036158*lnbara*u*w3*x*y*z - 
     1.3152966221164228*v*w3*x*y*z - 0.3724255234004633*lnbara*v*w3*x*y*z + 
     0.2932474952337439*w4*x*y*z + 0.08442890598101545*lnbara*w4*x*y*z + 
     0.25855118000886335*x2*y*z - 0.04772361051737147*u*x2*y*z + 
     0.00780375397440978*u2*x2*y*z - 0.0006817048264512694*u3*x2*y*z - 
     6.7414545262279955*v*x2*y*z - 2.014469236002691*lnbara*v*x2*y*z + 
     2.582819839347*u*v*x2*y*z + 0.8128752507655377*lnbara*u*v*x2*y*z - 
     0.30225971597078544*u2*v*x2*y*z - 0.09017258826251653*lnbara*u2*v*x2*
      y*z + 0.16780437876971865*v2*x2*y*z + 0.05368132013057931*lnbara*v2*
      x2*y*z + 0.00017982275152600617*u*v2*x2*y*z - 
     0.008148863899001088*lnbara*u*v2*x2*y*z - 0.9153400045660104*v3*x2*y*
      z - 0.25553842398615445*lnbara*v3*x2*y*z + 6.647579498888072*w*x2*y*
      z + 2.014469236002691*lnbara*w*x2*y*z - 2.5690901893905513*u*w*x2*y*
      z - 0.8128752507655377*lnbara*u*w*x2*y*z + 0.3011577157359468*u2*w*
      x2*y*z + 0.09017258826251653*lnbara*u2*w*x2*y*z + 
     0.04678476613703959*v*w*x2*y*z + 0.041372947073440955*lnbara*v*w*x2*y*
      z - 0.5540525639589041*u*v*w*x2*y*z - 0.1674546937036158*lnbara*u*v*w*
      x2*y*z + 2.53079627759307*v2*w*x2*y*z + 0.7137169391031986*lnbara*
      v2*w*x2*y*z - 0.19274016063176588*w2*x2*y*z - 
     0.09505426720402026*lnbara*w2*x2*y*z + 0.5522979595539836*u*w2*x2*y*
      z + 0.17560355760261687*lnbara*u*w2*x2*y*z - 
     2.1878006392634903*v*w2*x2*y*z - 0.6177543113748135*lnbara*v*w2*x2*y*
      z + 0.569975957295371*w3*x2*y*z + 0.15957579625776946*lnbara*w3*x2*
      y*z - 0.0985061281475928*x3*y*z + 0.01023088439838107*u*x3*y*z - 
     0.0007202416621432963*u2*x3*y*z + 0.7227291601125875*v*x3*y*z + 
     0.20757337570539203*lnbara*v*x3*y*z - 0.368445017992858*u*v*x3*y*z - 
     0.11464743468556707*lnbara*u*v*x3*y*z + 0.8410658901759228*v2*x3*y*
      z + 0.24067518005948615*lnbara*v2*x3*y*z - 0.7027857105758675*w*x3*y*
      z - 0.20757337570539203*lnbara*w*x3*y*z + 0.3671622432587578*u*w*x3*y*
      z + 0.11464743468556707*lnbara*u*w*x3*y*z - 1.3152966221164228*v*w*x3*
      y*z - 0.3724255234004633*lnbara*v*w*x3*y*z + 
     0.4722390677423313*w2*x3*y*z + 0.13175034334097716*lnbara*w2*x3*y*
      z + 0.023437600654995495*x4*y*z - 0.0010458019341617988*u*x4*y*z - 
     0.3710874532554622*v*x4*y*z - 0.10073935030932556*lnbara*v*x4*y*z + 
     0.36905978156971897*w*x4*y*z + 0.10073935030932556*lnbara*w*x4*y*z - 
     0.0025714972415127255*x5*y*z - 0.6431554977641232*y2*z + 
     0.17096212544446912*u*y2*z - 0.04689698042164179*u2*y2*z + 
     0.009251894041935034*u3*y2*z - 0.0009023290305323929*u4*y2*z + 
     3.150452651970475*v*y2*z + 0.9766280161206079*lnbara*v*y2*z - 
     0.5219825160658177*u*v*y2*z - 0.18363961128442552*lnbara*u*v*y2*z - 
     0.335966498688551*u2*v*y2*z - 0.09418475573467341*lnbara*u2*v*y2*z - 
     0.05665911334956437*u3*v*y2*z - 0.015395299386842583*lnbara*u3*v*y2*
      z - 0.022693681826607907*v2*y2*z - 0.027337198387939212*lnbara*v2*
      y2*z + 0.7675738685886592*u*v2*y2*z + 0.22874938189748867*lnbara*u*
      v2*y2*z + 0.17593076120022283*u2*v2*y2*z + 
     0.04860228502711452*lnbara*u2*v2*y2*z + 0.8444774478932007*v3*y2*
      z + 0.248860063558088*lnbara*v3*y2*z - 0.3385312614440469*u*v3*y2*
      z - 0.09925662865561737*lnbara*u*v3*y2*z - 0.047668105888734366*v4*
      y2*z - 0.015411950745225283*lnbara*v4*y2*z - 
     2.777512025251304*w*y2*z - 0.9766280161206079*lnbara*w*y2*z + 
     0.4390247991246611*u*w*y2*z + 0.18363961128442552*lnbara*u*w*y2*z + 
     0.3502072962213621*u2*w*y2*z + 0.09418475573467341*lnbara*u2*w*y2*
      z + 0.055393310008639646*u3*w*y2*z + 0.015395299386842583*lnbara*u3*
      w*y2*z - 0.04772361051737147*v*w*y2*z + 0.006668952902327549*u*v*w*
      y2*z - 0.000520176702770554*u2*v*w*y2*z - 4.148276730172979*v2*w*
      y2*z - 1.2223037533945342*lnbara*v2*w*y2*z + 
     0.15794313861675344*u*v2*w*y2*z + 0.05241767434109274*lnbara*u*v2*w*
      y2*z + 0.5075561415724656*v3*w*y2*z + 0.15134262670003695*lnbara*v3*
      w*y2*z - 0.08497562276797721*w2*y2*z + 0.027337198387939212*lnbara*
      w2*y2*z - 0.7526477768237668*u*w2*y2*z - 0.22874938189748867*lnbara*
      u*w2*y2*z - 0.1770896870636047*u2*w2*y2*z - 
     0.04860228502711452*lnbara*u2*w2*y2*z + 4.165985435140145*v*w2*y2*
      z + 1.2223037533945342*lnbara*v*w2*y2*z - 0.15906541536540864*u*v*w2*
      y2*z - 0.05241767434109274*lnbara*u*v*w2*y2*z - 
     0.0007553212077869194*v2*w2*y2*z - 0.8226148783091756*w3*y2*z - 
     0.248860063558088*lnbara*w3*y2*z + 0.33716810694416005*u*w3*y2*z + 
     0.09925662865561737*lnbara*u*w3*y2*z - 0.5092031405619221*v*w3*y2*
      z - 0.15134262670003695*lnbara*v*w3*y2*z + 0.04549876822052544*w4*
      y2*z + 0.015411950745225283*lnbara*w4*y2*z + 
     0.25855118000886335*x*y2*z - 0.04615141682255245*u*x*y2*z + 
     0.007376277398255051*u2*x*y2*z - 0.0006353292970921247*u3*x*y2*z - 
     0.04772361051737147*v*x*y2*z + 0.006668952902327549*u*v*x*y2*z - 
     0.000520176702770554*u2*v*x*y2*z - 2.5782754794730662*v2*x*y2*z - 
     0.7595995680938061*lnbara*v2*x*y2*z + 0.6316944039353403*u*v2*x*y2*
      z + 0.18411867864157963*lnbara*u*v2*x*y2*z + 
     0.15576929487146673*v3*x*y2*z + 0.05027670415149837*lnbara*v3*x*y2*
      z - 0.04772361051737147*w*x*y2*z + 0.006668952902327549*u*w*x*y2*z - 
     0.000520176702770554*u2*w*x*y2*z + 5.179219163949073*v*w*x*y2*z + 
     1.5191991361876123*lnbara*v*w*x*y2*z - 1.2649350069307033*u*v*w*x*y2*
      z - 0.36823735728315926*lnbara*u*v*w*x*y2*z - 
     0.1570218018180108*v2*w*x*y2*z - 0.05027670415149837*lnbara*v2*w*x*
      y2*z - 2.5782754794730662*w2*x*y2*z - 0.7595995680938061*lnbara*w2*
      x*y2*z + 0.6316944039353403*u*w2*x*y2*z + 0.18411867864157963*lnbara*
      u*w2*x*y2*z - 0.1570218018180108*v*w2*x*y2*z - 
     0.05027670415149837*lnbara*v*w2*x*y2*z + 0.15576929487146673*w3*x*y2*
      z + 0.05027670415149837*lnbara*w3*x*y2*z - 0.08463412943162905*x2*
      y2*z + 0.008768669407230721*u*x2*y2*z - 0.0006133788582035897*u2*
      x2*y2*z + 4.165985435140145*v*x2*y2*z + 1.2223037533945342*lnbara*v*
      x2*y2*z - 0.15906541536540864*u*v*x2*y2*z - 
     0.05241767434109274*lnbara*u*v*x2*y2*z - 0.21678745401254226*v2*x2*
      y2*z - 0.06933321435793802*lnbara*v2*x2*y2*z - 
     4.148276730172979*w*x2*y2*z - 1.2223037533945342*lnbara*w*x2*y2*z + 
     0.15794313861675344*u*w*x2*y2*z + 0.05241767434109274*lnbara*u*w*x2*
      y2*z - 0.1570218018180108*v*w*x2*y2*z - 0.05027670415149837*lnbara*v*
      w*x2*y2*z + 0.37199049657356653*w2*x2*y2*z + 
     0.11960991850943639*lnbara*w2*x2*y2*z + 0.018271879935025635*x3*y2*
      z - 0.0008257705772076173*u*x3*y2*z - 0.5092031405619221*v*x3*y2*
      z - 0.15134262670003695*lnbara*v*x3*y2*z + 0.5075561415724656*w*x3*
      y2*z + 0.15134262670003695*lnbara*w*x3*y2*z - 
     0.0018766602068817076*x4*y2*z + 0.39191746801048677*y3*z - 
     0.06170167867079261*u*y3*z + 0.009251894041935034*u2*y3*z - 
     0.0007701655588751243*u3*y3*z - 2.914384497168144*v*y3*z - 
     0.878965214508547*lnbara*v*y3*z - 0.4559117835501848*u*v*y3*z - 
     0.11978293118391894*lnbara*u*v*y3*z - 0.06557961755270206*u2*v*y3*z - 
     0.018012828737752302*lnbara*u2*v*y3*z + 0.7187404897942826*v2*y3*z + 
     0.2252713360201013*lnbara*v2*y3*z + 0.1387217802717251*u*v2*y3*z + 
     0.03825655240654065*lnbara*u*v2*y3*z - 0.35800549766301815*v3*y3*z - 
     0.1031812587805062*lnbara*v3*y3*z + 2.7762869962453007*w*y3*z + 
     0.878965214508547*lnbara*w*y3*z + 0.472929670266122*u*w*y3*z + 
     0.11978293118391894*lnbara*u*w*y3*z + 0.06433729180422629*u2*w*y3*z + 
     0.018012828737752302*lnbara*u2*w*y3*z + 0.01023088439838107*v*w*y3*
      z - 0.0006126156464844033*u*v*w*y3*z + 0.6274055393084268*v2*w*y3*
      z + 0.17563835536198544*lnbara*v2*w*y3*z - 0.6965292598704438*w2*y3*
      z - 0.2252713360201013*lnbara*w2*y3*z - 0.14004639793591991*u*w2*y3*
      z - 0.03825655240654065*lnbara*u*w2*y3*z - 0.6290677486101003*v*w2*
      y3*z - 0.17563835536198544*lnbara*v*w2*y3*z + 
     0.3560786661881267*w3*y3*z + 0.1031812587805062*lnbara*w3*y3*z - 
     0.0985061281475928*x*y3*z + 0.009712565138338942*u*x*y3*z - 
     0.00065880688954107*u2*x*y3*z + 0.01023088439838107*v*x*y3*z - 
     0.0006126156464844033*u*v*x*y3*z + 0.4419629692064293*v2*x*y3*z + 
     0.124791357615411*lnbara*v2*x*y3*z + 0.01023088439838107*w*x*y3*z - 
     0.0006126156464844033*u*w*x*y3*z - 0.8860365808247611*v*w*x*y3*z - 
     0.249582715230822*lnbara*v*w*x*y3*z + 0.4419629692064293*w2*x*y3*z + 
     0.124791357615411*lnbara*w2*x*y3*z + 0.018271879935025635*x2*y3*z - 
     0.0008105602649905572*u*x2*y3*z - 0.6290677486101003*v*x2*y3*z - 
     0.17563835536198544*lnbara*v*x2*y3*z + 0.6274055393084268*w*x2*y3*
      z + 0.17563835536198544*lnbara*w*x2*y3*z - 0.0017038721568713427*x3*
      y3*z - 0.16545503028959443*y4*z + 0.014252947529037665*u*y4*z - 
     0.0009023290305323929*u2*y4*z + 0.8473142624781451*v*y4*z + 
     0.24415700403015198*lnbara*v*y4*z + 0.008352282215414328*u*v*y4*z + 
     0.0006955947583152419*lnbara*u*v*y4*z - 0.04107580891265065*v2*y4*z - 
     0.013253738191194095*lnbara*v2*y4*z - 0.8149620980179276*w*y4*z - 
     0.24415700403015198*lnbara*w*y4*z - 0.010048357709481526*u*w*y4*z - 
     0.0006955947583152419*lnbara*u*w*y4*z - 0.0010458019341617988*v*w*y4*
      z + 0.03885294084404612*w2*y4*z + 0.013253738191194095*lnbara*w2*y4*
      z + 0.023437600654995495*x*y4*z - 0.0009818697515814416*u*x*y4*z - 
     0.0010458019341617988*v*x*y4*z - 0.0010458019341617988*w*x*y4*z - 
     0.0018766602068817076*x2*y4*z + 0.04216772179515423*y5*z - 
     0.0015359082311407401*u*y5*z - 0.13785434103492666*v*y5*z - 
     0.03836752920473817*lnbara*v*y5*z + 0.13433805924390965*w*y5*z + 
     0.03836752920473817*lnbara*w*y5*z - 0.0025714972415127255*x*y5*z - 
     0.004855915459192057*y6*z + 1.5830194232420491*z2 - 
     0.6431554977641232*u*z2 + 0.29720782742626456*u2*z2 - 
     0.10781348756592685*u3*z2 + 0.02501436947958356*u4*z2 - 
     0.0027042708881657167*u5*z2 + 1.3307155997089624*v*z2 + 
     0.6510853440804052*lnbara*v*z2 + 3.593282429668685*u*v*z2 + 
     1.128501340498048*lnbara*u*v*z2 + 1.9298767034120956*u2*v*z2 + 
     0.5924457327959798*lnbara*u2*v*z2 + 0.5260162467363529*u3*v*z2 + 
     0.14593979274603425*lnbara*u3*v*z2 + 0.049642889715863496*u4*v*z2 + 
     0.01365467547098468*lnbara*u4*v*z2 + 1.5951827813991792*v2*z2 + 
     0.37330756630262746*lnbara*v2*z2 - 8.153252996171776*u*v2*z2 - 
     2.598243148725863*lnbara*u*v2*z2 + 0.6456564865390652*u2*v2*z2 + 
     0.21593510082621092*lnbara*u2*v2*z2 - 0.5616689792286329*u3*v2*
      z2 - 0.16389022386115257*lnbara*u3*v2*z2 + 
     0.403289978849488*v3*z2 + 0.15003403088345468*lnbara*v3*z2 + 
     2.3470147044696774*u*v3*z2 + 0.7642509739315361*lnbara*u*v3*z2 + 
     0.14483466237687437*u2*v3*z2 + 0.03394368436113497*lnbara*u2*v3*
      z2 + 0.1913662943454968*v4*z2 + 0.04215619932204285*lnbara*v4*z2 - 
     0.5618260667674302*u*v4*z2 - 0.17055352864149378*lnbara*u*v4*z2 + 
     0.004695610230502542*v5*z2 + 0.002302749325478825*lnbara*v5*z2 - 
     2.4582922118695194*w*z2 - 0.6510853440804052*lnbara*w*z2 - 
     3.1637691242153525*u*w*z2 - 1.128501340498048*lnbara*u*w*z2 - 
     2.066949955700743*u2*w*z2 - 0.5924457327959798*lnbara*u2*w*z2 - 
     0.49681704281786926*u3*w*z2 - 0.14593979274603425*lnbara*u3*w*z2 - 
     0.05261848185645107*u4*w*z2 - 0.01365467547098468*lnbara*u4*w*z2 - 
     6.422064708307189*v*w*z2 - 2.1702844802680175*lnbara*v*w*z2 + 
     6.435061406178583*u*v*w*z2 + 2.079207074882875*lnbara*u*v*w*z2 - 
     3.373810219562036*u2*v*w*z2 - 1.0395623490494532*lnbara*u2*v*w*z2 + 
     0.5336843597652852*u3*v*w*z2 + 0.16066085741485445*lnbara*u3*v*w*
      z2 + 3.1418162127487004*v2*w*z2 + 1.1732054736522635*lnbara*v2*w*
      z2 + 0.7605706197007904*u*v2*w*z2 + 0.1801033288553867*lnbara*u*v2*
      w*z2 + 0.41357773400649983*u2*v2*w*z2 + 0.1313398473229925*lnbara*
      u2*v2*w*z2 - 2.775770631100383*v3*w*z2 - 
     0.8879564128194577*lnbara*v3*w*z2 + 0.3619717726179578*u*v3*w*z2 + 
     0.10324350723878072*lnbara*u*v3*w*z2 + 0.3328573832991313*v4*w*z2 + 
     0.11063120820095806*lnbara*v4*w*z2 + 5.493378448409817*w2*z2 + 
     1.79697691396539*lnbara*w2*z2 + 1.5405090633173697*u*w2*z2 + 
     0.5190360738429882*lnbara*u*w2*z2 + 2.7618349656998586*u2*w2*z2 + 
     0.8236272482232423*lnbara*u2*w2*z2 + 0.024812552253286976*u3*w2*
      z2 + 0.003229366446298125*lnbara*u3*w2*z2 - 
     1.1288573398553086*v*w2*z2 - 0.43966795501820194*lnbara*v*w2*z2 - 
     1.202554183086874*u*v*w2*z2 - 0.3723502771999754*lnbara*u*v*w2*z2 - 
     0.08093775692251794*u2*v*w2*z2 - 0.025975593895628257*lnbara*u2*v*
      w2*z2 + 2.445843105402685*v2*w2*z2 + 0.7054893158837185*lnbara*v2*
      w2*z2 - 1.2333152992895828*u*v2*w2*z2 - 0.3521670333053898*lnbara*
      u*v2*w2*z2 - 0.04072484698113826*v3*w2*z2 - 
     0.013759874074562587*lnbara*v3*w2*z2 - 2.6912970732885757*w3*z2 - 
     0.8835715495175162*lnbara*w3*z2 - 1.8613624203011476*u*w3*z2 - 
     0.5720040255869473*lnbara*u*w3*z2 - 0.48114924036453877*u2*w3*z2 - 
     0.13930793778849923*lnbara*u2*w3*z2 - 1.4645041396124352*v*w3*z2 - 
     0.37498908309415785*lnbara*v*w3*z2 + 1.25382676464303*u*v*w3*z2 + 
     0.3652162355419701*lnbara*u*v*w3*z2 - 0.061694139411974463*v2*w3*
      z2 - 0.01592769241839614*lnbara*v2*w3*z2 + 
     1.6718584721002623*w4*z2 + 0.5152999807078541*lnbara*w4*z2 + 
     0.17448936082610797*u*w4*z2 + 0.05426081916613278*lnbara*u*w4*z2 + 
     0.0011325127373185734*v*w4*z2 - 0.009074544432852395*lnbara*v*w4*
      z2 - 0.24406286627904641*w5*z2 - 0.07417184660062576*lnbara*w5*
      z2 - 0.6431554977641232*x*z2 + 0.18935004993694224*u*x*z2 - 
     0.05523018173756664*u2*x*z2 + 0.011283905940380838*u3*x*z2 - 
     0.0011239361348986583*u4*x*z2 - 9.679452702174132*v*x*z2 - 
     3.285801385277514*lnbara*v*x*z2 + 3.2592144539581844*u*v*x*z2 + 
     1.1013272044547329*lnbara*u*v*x*z2 - 3.229242426355708*u2*v*x*z2 - 
     0.9917935723502516*lnbara*u2*v*x*z2 + 0.404386452649353*u3*v*x*z2 + 
     0.12233847935377173*lnbara*u3*v*x*z2 + 3.1418162127487004*v2*x*z2 + 
     1.1732054736522635*lnbara*v2*x*z2 + 0.7605706197007904*u*v2*x*z2 + 
     0.1801033288553867*lnbara*u*v2*x*z2 + 0.41357773400649983*u2*v2*x*
      z2 + 0.1313398473229925*lnbara*u2*v2*x*z2 - 
     3.269269763555497*v3*x*z2 - 1.0164989417094896*lnbara*v3*x*z2 + 
     0.44168144304165097*u*v3*x*z2 + 0.12858123333432991*lnbara*u*v3*x*
      z2 + 0.3622126558447875*v4*x*z2 + 0.11666185005310058*lnbara*v4*x*
      z2 + 10.034005404400828*w*x*z2 + 3.285801385277514*lnbara*w*x*z2 - 
     3.3495745233238825*u*w*x*z2 - 1.1013272044547329*lnbara*u*w*x*z2 + 
     3.2459862158900528*u2*w*x*z2 + 0.9917935723502516*lnbara*u2*w*x*z2 - 
     0.40594313303596224*u3*w*x*z2 - 0.12233847935377173*lnbara*u3*w*x*
      z2 - 2.152616234264882*v*w*x*z2 - 0.7335375186340615*lnbara*v*w*x*
      z2 + 0.46479746309389686*u*v*w*x*z2 + 0.1922469483445887*lnbara*u*v*w*
      x*z2 - 0.3345794792700652*u2*v*w*x*z2 - 0.10536425342736426*lnbara*
      u2*v*w*x*z2 + 6.391972088595844*v2*w*x*z2 + 
     1.8709274228635118*lnbara*v2*w*x*z2 - 2.0843530956674345*u*v2*w*x*
      z2 - 0.6095267362125409*lnbara*u*v2*w*x*z2 - 
     0.3134017819611535*v3*w*x*z2 - 0.0993620511060278*lnbara*v3*w*x*z2 - 
     1.1288573398553086*w2*x*z2 - 0.43966795501820194*lnbara*w2*x*z2 - 
     1.202554183086874*u*w2*x*z2 - 0.3723502771999754*lnbara*u*w2*x*z2 - 
     0.08093775692251794*u2*w2*x*z2 - 0.025975593895628257*lnbara*u2*w2*
      x*z2 - 6.030472710174449*v*w2*x*z2 - 1.7376394640414514*lnbara*v*w2*
      x*z2 + 2.4254958617946483*u*v*w2*x*z2 + 0.7008504098797653*lnbara*u*
      v*w2*x*z2 - 0.05985191547449978*v2*w2*x*z2 - 
     0.01186851715999764*lnbara*v2*w2*x*z2 + 2.941587800744719*w3*x*z2 + 
     0.8832109828874293*lnbara*w3*x*z2 - 0.7853652276547294*u*w3*x*z2 - 
     0.21990490700155435*lnbara*u*w3*x*z2 + 0.2189313721141216*v*w3*x*
      z2 + 0.06175769728307968*lnbara*v*w3*x*z2 - 
     0.21163351738140398*w4*x*z2 - 0.06718897907015481*lnbara*w4*x*z2 + 
     0.29720782742626456*x2*z2 - 0.05523018173756664*u*x2*z2 + 
     0.009104692162046783*u2*x2*z2 - 0.0008001063695529314*u3*x2*z2 - 
     0.052439122857018475*v*x2*z2 + 0.00780375397440978*u*v*x2*z2 - 
     0.0006345782786901335*u2*v*x2*z2 + 2.445843105402685*v2*x2*z2 + 
     0.7054893158837185*lnbara*v2*x2*z2 - 1.2333152992895828*u*v2*x2*
      z2 - 0.3521670333053898*lnbara*u*v2*x2*z2 + 
     0.05859089416673988*v3*x2*z2 + 0.01186851715999764*lnbara*v3*x2*
      z2 - 0.052439122857018475*w*x2*z2 + 0.00780375397440978*u*w*x2*
      z2 - 0.0006345782786901335*u2*w*x2*z2 - 4.86789835135428*v*w*x2*
      z2 - 1.410978631767437*lnbara*v*w*x2*z2 + 2.4648726427480128*u*v*w*
      x2*z2 + 0.7043340666107796*lnbara*u*v*w*x2*z2 - 
     0.05985191547449978*v2*w*x2*z2 - 0.01186851715999764*lnbara*v2*w*x2*
      z2 + 2.445843105402685*w2*x2*z2 + 0.7054893158837185*lnbara*w2*x2*
      z2 - 1.2333152992895828*u*w2*x2*z2 - 0.3521670333053898*lnbara*u*
      w2*x2*z2 - 0.05985191547449978*v*w2*x2*z2 - 
     0.01186851715999764*lnbara*v*w2*x2*z2 + 0.05859089416673988*w3*x2*
      z2 + 0.01186851715999764*lnbara*w3*x2*z2 - 
     0.10781348756592685*x3*z2 + 0.011283905940380838*u*x3*z2 - 
     0.0008001063695529314*u2*x3*z2 - 1.8540397627858702*v*x3*z2 - 
     0.5198935985278687*lnbara*v*x3*z2 + 0.5315078645933379*u*v*x3*z2 + 
     0.1550921261368306*lnbara*u*v*x3*z2 - 0.061694139411974463*v2*x3*
      z2 - 0.01592769241839614*lnbara*v2*x3*z2 + 
     1.8755457504129727*w*x3*z2 + 0.5198935985278687*lnbara*w*x3*z2 - 
     0.5329098110819325*u*w*x3*z2 - 0.1550921261368306*lnbara*u*w*x3*z2 + 
     0.10030472915504478*v*w*x3*z2 + 0.029687566492958728*lnbara*v*w*x3*
      z2 - 0.04072484698113826*w2*x3*z2 - 0.013759874074562587*lnbara*w2*
      x3*z2 + 0.02501436947958356*x4*z2 - 0.0011239361348986583*u*x4*
      z2 + 0.11440699548555519*v*x4*z2 + 0.02876833807133764*lnbara*v*x4*
      z2 - 0.11655132895257132*w*x4*z2 - 0.02876833807133764*lnbara*w*x4*
      z2 - 0.0027042708881657167*x5*z2 - 0.6431554977641232*y*z2 + 
     0.18359057678222876*u*y*z2 - 0.052439122857018475*u2*y*z2 + 
     0.010578663643644204*u3*y*z2 - 0.0010454015333102693*u4*y*z2 + 
     4.998355981731551*v*y*z2 + 1.7188373548274771*lnbara*v*y*z2 + 
     3.6274999402312313*u*v*y*z2 + 1.101558132258626*lnbara*u*v*y*z2 + 
     1.2958237235724512*u2*v*y*z2 + 0.37253832538857234*lnbara*u2*v*y*
      z2 + 0.1611431319221526*u3*v*y*z2 + 0.043853270547930576*lnbara*u3*
      v*y*z2 - 5.33566764662009*v2*y*z2 - 1.7682946827872745*lnbara*v2*y*
      z2 - 0.5880364469542249*u*v2*y*z2 - 0.18109640550068581*lnbara*u*v2*
      y*z2 - 0.6568129911274124*u2*v2*y*z2 - 0.19060623005554803*lnbara*
      u2*v2*y*z2 + 1.7216900283016308*v3*y*z2 + 
     0.5870600279751855*lnbara*v3*y*z2 + 0.35111850002088596*u*v3*y*z2 + 
     0.09843197109895134*lnbara*u*v3*y*z2 - 0.4270870728419753*v4*y*z2 - 
     0.13342942879566821*lnbara*v4*y*z2 - 4.63804380635014*w*y*z2 - 
     1.7188373548274771*lnbara*w*y*z2 - 3.7155448088414325*u*w*y*z2 - 
     1.101558132258626*lnbara*u*w*y*z2 - 1.279932716374738*u2*w*y*z2 - 
     0.37253832538857234*lnbara*u2*w*y*z2 - 0.16259660502488668*u3*w*y*
      z2 - 0.043853270547930576*lnbara*u3*w*y*z2 + 
     4.806051314717304*v*w*y*z2 + 1.4974962913849321*lnbara*v*w*y*z2 - 
     4.358084874395133*u*v*w*y*z2 - 1.310301658288488*lnbara*u*v*w*y*z2 + 
     0.44246231471333614*u2*v*w*y*z2 + 0.13612276309543284*lnbara*u2*v*w*y*
      z2 - 0.19274016063176588*v2*w*y*z2 - 0.09505426720402026*lnbara*v2*
      w*y*z2 + 0.5522979595539836*u*v2*w*y*z2 + 0.17560355760261687*lnbara*
      u*v2*w*y*z2 + 0.4722390677423313*v3*w*y*z2 + 
     0.13175034334097716*lnbara*v3*w*y*z2 + 0.3848527108952509*w2*y*z2 + 
     0.27079839140234246*lnbara*w2*y*z2 + 4.968594829946177*u*w2*y*z2 + 
     1.491398063789174*lnbara*u*w2*y*z2 + 0.2124983335725487*u2*w2*y*
      z2 + 0.0544834669601152*lnbara*u2*w2*y*z2 + 
     0.16780437876971865*v*w2*y*z2 + 0.05368132013057931*lnbara*v*w2*y*
      z2 + 0.00017982275152600617*u*v*w2*y*z2 - 0.008148863899001088*
      lnbara*u*v*w2*y*z2 - 1.0211916139594837*v2*w2*y*z2 - 
     0.2879966174194479*lnbara*v2*w2*y*z2 - 1.6610384150846027*w3*y*z2 - 
     0.5456870809017446*lnbara*w3*y*z2 - 0.9061306963251085*u*w3*y*z2 - 
     0.2658866648025671*lnbara*u*w3*y*z2 + 0.8410658901759228*v*w3*y*z2 + 
     0.24067518005948615*lnbara*v*w3*y*z2 + 0.1309550363078858*w4*y*z2 + 
     0.04900052281465277*lnbara*w4*y*z2 + 0.25855118000886335*x*y*z2 - 
     0.04772361051737147*u*x*y*z2 + 0.00780375397440978*u2*x*y*z2 - 
     0.0006817048264512694*u3*x*y*z2 + 6.647579498888072*v*x*y*z2 + 
     2.014469236002691*lnbara*v*x*y*z2 - 2.5690901893905513*u*v*x*y*z2 - 
     0.8128752507655377*lnbara*u*v*x*y*z2 + 0.3011577157359468*u2*v*x*y*
      z2 + 0.09017258826251653*lnbara*u2*v*x*y*z2 - 
     0.19274016063176588*v2*x*y*z2 - 0.09505426720402026*lnbara*v2*x*y*
      z2 + 0.5522979595539836*u*v2*x*y*z2 + 0.17560355760261687*lnbara*u*
      v2*x*y*z2 + 0.569975957295371*v3*x*y*z2 + 
     0.15957579625776946*lnbara*v3*x*y*z2 - 6.7414545262279955*w*x*y*z2 - 
     2.014469236002691*lnbara*w*x*y*z2 + 2.582819839347*u*w*x*y*z2 + 
     0.8128752507655377*lnbara*u*w*x*y*z2 - 0.30225971597078544*u2*w*x*y*
      z2 - 0.09017258826251653*lnbara*u2*w*x*y*z2 + 
     0.04678476613703959*v*w*x*y*z2 + 0.041372947073440955*lnbara*v*w*x*y*
      z2 - 0.5540525639589041*u*v*w*x*y*z2 - 0.1674546937036158*lnbara*u*v*
      w*x*y*z2 - 2.1878006392634903*v2*w*x*y*z2 - 
     0.6177543113748135*lnbara*v2*w*x*y*z2 + 0.16780437876971865*w2*x*y*
      z2 + 0.05368132013057931*lnbara*w2*x*y*z2 + 
     0.00017982275152600617*u*w2*x*y*z2 - 0.008148863899001088*lnbara*u*w2*
      x*y*z2 + 2.53079627759307*v*w2*x*y*z2 + 0.7137169391031986*lnbara*v*
      w2*x*y*z2 - 0.9153400045660104*w3*x*y*z2 - 
     0.25553842398615445*lnbara*w3*x*y*z2 - 0.08463412943162905*x2*y*z2 + 
     0.00894003555993513*u*x2*y*z2 - 0.0006345782786901335*u2*x2*y*z2 + 
     0.008768669407230721*v*x2*y*z2 - 0.0005708021200927833*u*v*x2*y*z2 - 
     1.0211916139594837*v2*x2*y*z2 - 0.2879966174194479*lnbara*v2*x2*y*
      z2 + 0.008768669407230721*w*x2*y*z2 - 0.0005708021200927833*u*w*x2*
      y*z2 + 2.0406049955739975*v*w*x2*y*z2 + 0.5759932348388958*lnbara*v*
      w*x2*y*z2 - 1.0211916139594837*w2*x2*y*z2 - 
     0.2879966174194479*lnbara*w2*x2*y*z2 + 0.018271879935025635*x3*y*
      z2 - 0.0008364387244659098*u*x3*y*z2 + 0.8908183968952704*v*x3*y*
      z2 + 0.2518959384495211*lnbara*v*x3*y*z2 - 
     0.8924547277374686*w*x3*y*z2 - 0.2518959384495211*lnbara*w*x3*y*z2 - 
     0.0018766602068817076*x4*y*z2 + 0.29720782742626456*y2*z2 - 
     0.052439122857018475*u*y2*z2 + 0.008332760830303598*u2*y2*z2 - 
     0.0007154743025816235*u3*y2*z2 - 0.022693681826607907*v*y2*z2 - 
     0.027337198387939212*lnbara*v*y2*z2 + 0.7675738685886592*u*v*y2*z2 + 
     0.22874938189748867*lnbara*u*v*y2*z2 + 0.17593076120022283*u2*v*y2*
      z2 + 0.04860228502711452*lnbara*u2*v*y2*z2 + 
     0.6573559458240761*v2*y2*z2 + 0.19271706881608106*lnbara*v2*y2*
      z2 - 0.6548222450121166*u*v2*y2*z2 - 0.18974503809357995*lnbara*u*
      v2*y2*z2 + 0.047708646825452936*v3*y2*z2 + 
     0.00998440678420688*lnbara*v3*y2*z2 - 0.08497562276797721*w*y2*z2 + 
     0.027337198387939212*lnbara*w*y2*z2 - 0.7526477768237668*u*w*y2*z2 - 
     0.22874938189748867*lnbara*u*w*y2*z2 - 0.1770896870636047*u2*w*y2*
      z2 - 0.04860228502711452*lnbara*u2*w*y2*z2 - 
     2.5782754794730662*v*w*y2*z2 - 0.7595995680938061*lnbara*v*w*y2*z2 + 
     0.6316944039353403*u*v*w*y2*z2 + 0.18411867864157963*lnbara*u*v*w*y2*
      z2 + 0.37199049657356653*v2*w*y2*z2 + 0.11960991850943639*lnbara*
      v2*w*y2*z2 + 1.9461607406157502*w2*y2*z2 + 
     0.566882499277725*lnbara*w2*y2*z2 + 0.021416517678162092*u*w2*y2*
      z2 + 0.005626359452000324*lnbara*u*w2*y2*z2 - 
     0.21678745401254226*v*w2*y2*z2 - 0.06933321435793802*lnbara*v*w2*y2*
      z2 - 0.2056752271955055*w3*y2*z2 - 0.06026111093570525*lnbara*w3*
      y2*z2 - 0.08463412943162905*x*y2*z2 + 0.008768669407230721*u*x*y2*
      z2 - 0.0006133788582035897*u2*x*y2*z2 - 4.148276730172979*v*x*y2*
      z2 - 1.2223037533945342*lnbara*v*x*y2*z2 + 
     0.15794313861675344*u*v*x*y2*z2 + 0.05241767434109274*lnbara*u*v*x*y2*
      z2 + 0.37199049657356653*v2*x*y2*z2 + 0.11960991850943639*lnbara*
      v2*x*y2*z2 + 4.165985435140145*w*x*y2*z2 + 
     1.2223037533945342*lnbara*w*x*y2*z2 - 0.15906541536540864*u*w*x*y2*
      z2 - 0.05241767434109274*lnbara*u*w*x*y2*z2 - 
     0.1570218018180108*v*w*x*y2*z2 - 0.05027670415149837*lnbara*v*w*x*y2*
      z2 - 0.21678745401254226*w2*x*y2*z2 - 0.06933321435793802*lnbara*
      w2*x*y2*z2 + 0.01657980243935387*x2*y2*z2 - 
     0.0007553212077869194*u*x2*y2*z2 - 0.0007553212077869194*v*x2*y2*
      z2 - 0.0007553212077869194*w*x2*y2*z2 - 0.0015904265351103456*x3*
      y2*z2 - 0.10781348756592685*y3*z2 + 0.010578663643644204*u*y3*
      z2 - 0.0007154743025816235*u2*y3*z2 + 0.7187404897942826*v*y3*
      z2 + 0.2252713360201013*lnbara*v*y3*z2 + 0.1387217802717251*u*v*y3*
      z2 + 0.03825655240654065*lnbara*u*v*y3*z2 - 
     0.3442555249706034*v2*y3*z2 - 0.10069943614887171*lnbara*v2*y3*
      z2 - 0.6965292598704438*w*y3*z2 - 0.2252713360201013*lnbara*w*y3*
      z2 - 0.14004639793591991*u*w*y3*z2 - 0.03825655240654065*lnbara*u*w*
      y3*z2 + 0.4419629692064293*v*w*y3*z2 + 0.124791357615411*lnbara*v*w*
      y3*z2 - 0.09998366236526494*w2*y3*z2 - 0.024091921466539296*lnbara*
      w2*y3*z2 + 0.018271879935025635*x*y3*z2 - 
     0.0008105602649905572*u*x*y3*z2 + 0.6274055393084268*v*x*y3*z2 + 
     0.17563835536198544*lnbara*v*x*y3*z2 - 0.6290677486101003*w*x*y3*
      z2 - 0.17563835536198544*lnbara*w*x*y3*z2 - 
     0.0015904265351103456*x2*y3*z2 + 0.02501436947958356*y4*z2 - 
     0.0010454015333102693*u*y4*z2 - 0.04107580891265065*v*y4*z2 - 
     0.013253738191194095*lnbara*v*y4*z2 + 0.03885294084404612*w*y4*z2 + 
     0.013253738191194095*lnbara*w*y4*z2 - 0.0018766602068817076*x*y4*
      z2 - 0.0027042708881657167*y5*z2 - 1.6193513369191062*z3 + 
     0.39191746801048677*u*z3 - 0.10781348756592685*u2*z3 + 
     0.021502574570336913*u3*z3 - 0.002116028409986158*u4*z3 - 
     0.3344496541843748*v*z3 - 0.24740311290228975*lnbara*v*z3 - 
     3.884956834618231*u*v*z3 - 1.2186672004102037*lnbara*u*v*z3 - 
     0.47396206592202933*u2*v*z3 - 0.13192114982378386*lnbara*u2*v*z3 - 
     0.23348793049311065*u3*v*z3 - 0.06654355140807486*lnbara*u3*v*z3 + 
     0.403289978849488*v2*z3 + 0.15003403088345468*lnbara*v2*z3 + 
     2.3470147044696774*u*v2*z3 + 0.7642509739315361*lnbara*u*v2*z3 + 
     0.14483466237687437*u2*v2*z3 + 0.03394368436113497*lnbara*u2*v2*
      z3 + 0.13476350893453617*v3*z3 + 0.02815049785571501*lnbara*v3*
      z3 - 0.5147171592780667*u*v3*z3 - 0.16006031325329761*lnbara*u*v3*
      z3 + 0.026332918322492415*v4*z3 + 0.007673317744555463*lnbara*v4*
      z3 + 1.0120947283767654*w*z3 + 0.24740311290228975*lnbara*w*z3 + 
     3.7247490277998456*u*w*z3 + 1.2186672004102037*lnbara*u*w*z3 + 
     0.5028126095006992*u2*w*z3 + 0.13192114982378386*lnbara*u2*w*z3 + 
     0.23084041467755445*u3*w*z3 + 0.06654355140807486*lnbara*u3*w*z3 + 
     2.0505932975012593*v*w*z3 + 0.7335375186340615*lnbara*v*w*z3 - 
     0.4484159450532304*u*v*w*z3 - 0.1922469483445887*lnbara*u*v*w*z3 + 
     0.33319811688498185*u2*v*w*z3 + 0.10536425342736426*lnbara*u2*v*w*
      z3 - 3.269269763555497*v2*w*z3 - 1.0164989417094896*lnbara*v2*w*
      z3 + 0.44168144304165097*u*v2*w*z3 + 0.12858123333432991*lnbara*u*
      v2*w*z3 + 0.33499391115566624*v3*w*z3 + 0.11243984289700684*lnbara*
      v3*w*z3 - 2.6912970732885757*w2*z3 - 0.8835715495175162*lnbara*w2*
      z3 - 1.8613624203011476*u*w2*z3 - 0.5720040255869473*lnbara*u*w2*
      z3 - 0.48114924036453877*u2*w2*z3 - 0.13930793778849923*lnbara*u2*
      w2*z3 + 2.941587800744719*v*w2*z3 + 0.8832109828874293*lnbara*v*w2*
      z3 - 0.7853652276547294*u*v*w2*z3 - 0.21990490700155435*lnbara*u*v*
      w2*z3 + 0.05859089416673988*v2*w2*z3 + 0.01186851715999764*lnbara*
      v2*w2*z3 + 0.24682581080985633*w3*z3 + 0.10513746096634541*lnbara*
      w3*z3 + 0.8544575390208671*u*w3*z3 + 0.251383986920522*lnbara*u*w3*
      z3 - 0.2434872198153215*v*w3*z3 - 0.07483548907405872*lnbara*v*w3*
      z3 - 0.1821625110649795*w4*z3 - 0.057146188727501226*lnbara*w4*
      z3 + 0.39191746801048677*x*z3 - 0.06968191432583015*u*x*z3 + 
     0.011283905940380838*u2*x*z3 - 0.0009831878162066224*u3*x*z3 + 
     5.035142069385885*v*x*z3 + 1.7095780955682178*lnbara*v*x*z3 + 
     0.16919997458043376*u*v*x*z3 + 0.013784793500404437*lnbara*u*v*x*z3 + 
     0.4078728401620492*u2*v*x*z3 + 0.12576336826926637*lnbara*u2*v*x*
      z3 - 2.775770631100383*v2*x*z3 - 0.8879564128194577*lnbara*v2*x*
      z3 + 0.3619717726179578*u*v2*x*z3 + 0.10324350723878072*lnbara*u*v2*
      x*z3 + 0.33499391115566624*v3*x*z3 + 0.11243984289700684*lnbara*v3*
      x*z3 - 5.165259334653691*w*x*z3 - 1.7095780955682178*lnbara*w*x*z3 - 
     0.15031327644943632*u*w*x*z3 - 0.013784793500404437*lnbara*u*w*x*z3 - 
     0.40939098371296645*u2*w*x*z3 - 0.12576336826926637*lnbara*u2*w*x*
      z3 + 4.268816061721532*v*w*x*z3 + 1.2629454959136155*lnbara*v*w*x*
      z3 - 1.617872252781772*u*v*w*x*z3 - 0.46845974278075075*lnbara*u*v*w*
      x*z3 - 0.3134017819611535*v2*w*x*z3 - 0.0993620511060278*lnbara*v2*
      w*x*z3 - 1.4645041396124352*w2*x*z3 - 0.37498908309415785*lnbara*w2*
      x*z3 + 1.25382676464303*u*w2*x*z3 + 0.3652162355419701*lnbara*u*w2*
      x*z3 + 0.2189313721141216*v*w2*x*z3 + 0.06175769728307968*lnbara*v*
      w2*x*z3 - 0.2434872198153215*w3*x*z3 - 0.07483548907405872*lnbara*
      w3*x*z3 - 0.10781348756592685*x2*z3 + 0.011283905940380838*u*x2*
      z3 - 0.0008001063695529314*u2*x2*z3 + 1.8755457504129727*v*x2*
      z3 + 0.5198935985278687*lnbara*v*x2*z3 - 0.5329098110819325*u*v*x2*
      z3 - 0.1550921261368306*lnbara*u*v*x2*z3 - 
     0.04072484698113826*v2*x2*z3 - 0.013759874074562587*lnbara*v2*x2*
      z3 - 1.8540397627858702*w*x2*z3 - 0.5198935985278687*lnbara*w*x2*
      z3 + 0.5315078645933379*u*w*x2*z3 + 0.1550921261368306*lnbara*u*w*
      x2*z3 + 0.10030472915504478*v*w*x2*z3 + 0.029687566492958728*lnbara*
      v*w*x2*z3 - 0.061694139411974463*w2*x2*z3 - 
     0.01592769241839614*lnbara*w2*x2*z3 + 0.021502574570336913*x3*z3 - 
     0.0009831878162066224*u*x3*z3 - 0.0009436436586848417*v*x3*z3 - 
     0.0009436436586848417*w*x3*z3 - 0.002116028409986158*x4*z3 + 
     0.39191746801048677*y*z3 - 0.06841558659701322*u*y*z3 + 
     0.010927323983458038*u2*y*z3 - 0.0009436436586848417*u3*y*z3 - 
     6.4531586807126216*v*y*z3 - 2.0595906539260844*lnbara*v*y*z3 - 
     1.0928490356968497*u*v*y*z3 - 0.30677398634465897*lnbara*u*v*y*z3 - 
     0.5203695633112709*u2*v*y*z3 - 0.14879519944448213*lnbara*u2*v*y*
      z3 + 1.7216900283016308*v2*y*z3 + 0.5870600279751855*lnbara*v2*y*
      z3 + 0.35111850002088596*u*v2*y*z3 + 0.09843197109895134*lnbara*u*
      v2*y*z3 - 0.5006975185477412*v3*y*z3 - 0.1543564123528748*lnbara*
      v3*y*z3 + 6.321775087715999*w*y*z3 + 2.0595906539260844*lnbara*w*y*
      z3 + 1.1114419930785513*u*w*y*z3 + 0.30677398634465897*lnbara*u*w*y*
      z3 + 0.5189008798963198*u2*w*y*z3 + 0.14879519944448213*lnbara*u2*w*
      y*z3 - 0.03145999950209565*v*w*y*z3 - 0.041372947073440955*lnbara*v*w*
      y*z3 + 0.5529561454818885*u*v*w*y*z3 + 0.1674546937036158*lnbara*u*v*
      w*y*z3 + 0.569975957295371*v2*w*y*z3 + 0.15957579625776946*lnbara*
      v2*w*y*z3 - 1.6610384150846027*w2*y*z3 - 0.5456870809017446*lnbara*
      w2*y*z3 - 0.9061306963251085*u*w2*y*z3 - 0.2658866648025671*lnbara*
      u*w2*y*z3 - 0.9153400045660104*v*w2*y*z3 - 
     0.25553842398615445*lnbara*v*w2*y*z3 + 0.8429911783197556*w3*y*z3 + 
     0.2503190400812598*lnbara*w3*y*z3 - 0.0985061281475928*x*y*z3 + 
     0.01023088439838107*u*x*y*z3 - 0.0007202416621432963*u2*x*y*z3 - 
     0.7027857105758675*v*x*y*z3 - 0.20757337570539203*lnbara*v*x*y*z3 + 
     0.3671622432587578*u*v*x*y*z3 + 0.11464743468556707*lnbara*u*v*x*y*
      z3 + 0.4722390677423313*v2*x*y*z3 + 0.13175034334097716*lnbara*v2*x*
      y*z3 + 0.7227291601125875*w*x*y*z3 + 0.20757337570539203*lnbara*w*x*y*
      z3 - 0.368445017992858*u*w*x*y*z3 - 0.11464743468556707*lnbara*u*w*x*
      y*z3 - 1.3152966221164228*v*w*x*y*z3 - 0.3724255234004633*lnbara*v*w*
      x*y*z3 + 0.8410658901759228*w2*x*y*z3 + 0.24067518005948615*lnbara*
      w2*x*y*z3 + 0.018271879935025635*x2*y*z3 - 
     0.0008364387244659098*u*x2*y*z3 - 0.8924547277374686*v*x2*y*z3 - 
     0.2518959384495211*lnbara*v*x2*y*z3 + 0.8908183968952704*w*x2*y*z3 + 
     0.2518959384495211*lnbara*w*x2*y*z3 - 0.0017038721568713427*x3*y*
      z3 - 0.10781348756592685*y2*z3 + 0.010927323983458038*u*y2*z3 - 
     0.000755870097742802*u2*y2*z3 + 0.8444774478932007*v*y2*z3 + 
     0.248860063558088*lnbara*v*y2*z3 - 0.3385312614440469*u*v*y2*z3 - 
     0.09925662865561737*lnbara*u*v*y2*z3 + 0.047708646825452936*v2*y2*
      z3 + 0.00998440678420688*lnbara*v2*y2*z3 - 
     0.8226148783091756*w*y2*z3 - 0.248860063558088*lnbara*w*y2*z3 + 
     0.33716810694416005*u*w*y2*z3 + 0.09925662865561737*lnbara*u*w*y2*
      z3 + 0.15576929487146673*v*w*y2*z3 + 0.05027670415149837*lnbara*v*w*
      y2*z3 - 0.2056752271955055*w2*y2*z3 - 0.06026111093570525*lnbara*
      w2*y2*z3 + 0.018271879935025635*x*y2*z3 - 
     0.0008257705772076173*u*x*y2*z3 + 0.5075561415724656*v*x*y2*z3 + 
     0.15134262670003695*lnbara*v*x*y2*z3 - 0.5092031405619221*w*x*y2*
      z3 - 0.15134262670003695*lnbara*w*x*y2*z3 - 
     0.0015904265351103456*x2*y2*z3 + 0.021502574570336913*y3*z3 - 
     0.0009436436586848417*u*y3*z3 - 0.35800549766301815*v*y3*z3 - 
     0.1031812587805062*lnbara*v*y3*z3 + 0.3560786661881267*w*y3*z3 + 
     0.1031812587805062*lnbara*w*y3*z3 - 0.0017038721568713427*x*y3*z3 - 
     0.002116028409986158*y4*z3 + 1.1560019673631434*z4 - 
     0.16545503028959443*u*z4 + 0.02501436947958356*u2*z4 - 
     0.002116028409986158*u3*z4 + 0.8622872883244641*v*z4 + 
     0.3193208792684922*lnbara*v*z4 + 1.3489683495777027*u*v*z4 + 
     0.4144777234484366*lnbara*u*v*z4 + 0.1660137847279631*u2*v*z4 + 
     0.0453330032683975*lnbara*u2*v*z4 + 0.1913662943454968*v2*z4 + 
     0.04215619932204285*lnbara*v2*z4 - 0.5618260667674302*u*v2*z4 - 
     0.17055352864149378*lnbara*u*v2*z4 + 0.026332918322492415*v3*z4 + 
     0.007673317744555463*lnbara*v3*z4 - 1.1463870292291396*w*z4 - 
     0.3193208792684922*lnbara*w*z4 - 1.3112778013936697*u*w*z4 - 
     0.4144777234484366*lnbara*u*w*z4 - 0.16893584646815507*u2*w*z4 - 
     0.0453330032683975*lnbara*u2*w*z4 - 1.8086906966367922*v*w*z4 - 
     0.5574561800298969*lnbara*v*w*z4 + 0.3835758469523267*u*v*w*z4 + 
     0.116292709475361*lnbara*u*v*w*z4 + 0.3622126558447875*v2*w*z4 + 
     0.11666185005310058*lnbara*v2*w*z4 + 1.6718584721002623*w2*z4 + 
     0.5152999807078541*lnbara*w2*z4 + 0.17448936082610797*u*w2*z4 + 
     0.05426081916613278*lnbara*u*w2*z4 - 0.21163351738140398*v*w2*z4 - 
     0.06718897907015481*lnbara*v*w2*z4 - 0.1821625110649795*w3*z4 - 
     0.057146188727501226*lnbara*w3*z4 - 0.16545503028959443*x*z4 + 
     0.0162776555901896*u*x*z4 - 0.0011239361348986583*u2*x*z4 - 
     3.0012661757807617*v*x*z4 - 0.9380468690654714*lnbara*v*x*z4 + 
     0.15191010788660775*u*v*x*z4 + 0.048049503963086576*lnbara*u*v*x*z4 + 
     0.3328573832991313*v2*x*z4 + 0.11063120820095806*lnbara*v2*x*z4 + 
     3.0315936321798276*w*x*z4 + 0.9380468690654714*lnbara*w*x*z4 - 
     0.15381418595568414*u*w*x*z4 - 0.048049503963086576*lnbara*u*w*x*z4 - 
     0.3368289563598407*v*w*x*z4 - 0.10155666376810567*lnbara*v*w*x*z4 + 
     0.0011325127373185734*w2*x*z4 - 0.009074544432852395*lnbara*w2*x*
      z4 + 0.02501436947958356*x2*z4 - 0.0011239361348986583*u*x2*z4 - 
     0.11655132895257132*v*x2*z4 - 0.02876833807133764*lnbara*v*x2*z4 + 
     0.11440699548555519*w*x2*z4 + 0.02876833807133764*lnbara*w*x2*z4 - 
     0.002116028409986158*x3*z4 - 0.16545503028959443*y*z4 + 
     0.016074508870027843*u*y*z4 - 0.0010989319337058666*u2*y*z4 + 
     2.4002692887025985*v*y*z4 + 0.7507653584772783*lnbara*v*y*z4 + 
     0.29707382410626676*u*v*y*z4 + 0.0809778929894431*lnbara*u*v*y*z4 - 
     0.4270870728419753*v2*y*z4 - 0.13342942879566821*lnbara*v2*y*z4 - 
     2.3697386855833713*w*y*z4 - 0.7507653584772783*lnbara*w*y*z4 - 
     0.2989574253995812*u*w*y*z4 - 0.0809778929894431*lnbara*u*w*y*z4 + 
     0.2932474952337439*v*w*y*z4 + 0.08442890598101545*lnbara*v*w*y*z4 + 
     0.1309550363078858*w2*y*z4 + 0.04900052281465277*lnbara*w2*y*z4 + 
     0.023437600654995495*x*y*z4 - 0.0010458019341617988*u*x*y*z4 + 
     0.36905978156971897*v*x*y*z4 + 0.10073935030932556*lnbara*v*x*y*z4 - 
     0.3710874532554622*w*x*y*z4 - 0.10073935030932556*lnbara*w*x*y*z4 - 
     0.0018766602068817076*x2*y*z4 + 0.02501436947958356*y2*z4 - 
     0.0010989319337058666*u*y2*z4 - 0.047668105888734366*v*y2*z4 - 
     0.015411950745225283*lnbara*v*y2*z4 + 0.04549876822052544*w*y2*z4 + 
     0.015411950745225283*lnbara*w*y2*z4 - 0.0018766602068817076*x*y2*
      z4 - 0.002116028409986158*y3*z4 - 0.5375082140631944*z5 + 
     0.04216772179515423*u*z5 - 0.0027042708881657167*u2*z5 - 
     0.2093175862405338*v*z5 - 0.07853883800228331*lnbara*v*z5 - 
     0.27798071288879395*u*v*z5 - 0.08187135582602693*lnbara*u*v*z5 + 
     0.004695610230502542*v2*z5 + 0.002302749325478825*lnbara*v2*z5 + 
     0.281434133138352*w*z5 + 0.07853883800228331*lnbara*w*z5 + 
     0.2738733074161405*u*w*z5 + 0.08187135582602693*lnbara*u*w*z5 + 
     0.23350760677675905*v*w*z5 + 0.07186909727514694*lnbara*v*w*z5 - 
     0.24406286627904641*w2*z5 - 0.07417184660062576*lnbara*w2*z5 + 
     0.04216772179515423*x*z5 - 0.0017664062094713614*u*x*z5 + 
     0.4769026267982312*v*x*z5 + 0.14541778766038482*lnbara*v*x*z5 - 
     0.48018841061091755*w*x*z5 - 0.14541778766038482*lnbara*w*x*z5 - 
     0.0027042708881657167*x2*z5 + 0.04216772179515423*y*z5 - 
     0.0017498755815456293*u*y*z5 - 0.48202800643018173*v*y*z5 - 
     0.1438457497350551*lnbara*v*y*z5 + 0.4787256919895696*w*y*z5 + 
     0.1438457497350551*lnbara*w*y*z5 - 0.0025714972415127255*x*y*z5 - 
     0.0027042708881657167*y2*z5 + 0.14606611930718463*z6 - 
     0.004855915459192057*u*z6 + 0.04158747525391902*v*z6 + 
     0.013727259996145989*lnbara*v*z6 - 0.04987109935863513*w*z6 - 
     0.013727259996145989*lnbara*w*z6 - 0.004855915459192057*x*z6 - 
     0.004855915459192057*y*z6 - 0.017602106974984624*z7;
 
  Hexp[8] = 0.9015426773696957 - 1.2020569031595942*u + 
     1.8468559937823905*u2 - 2.2670918716867487*u3 + 
     2.0230034428855013*u4 - 1.2541858328141202*u5 + 
     0.5112314175751462*u6 - 0.12321474882489236*u7 + 
     0.013314221484862214*u8 - 1.2020569031595942*v + 
     0.9809537469447522*u*v - 0.9004176968697725*u2*v + 
     0.6858555690183518*u3*v - 0.386061737342387*u4*v + 
     0.1475870262830398*u5*v - 0.0339914082143444*u6*v + 
     0.0035512018991495737*u7*v + 1.8468559937823905*v2 - 
     0.9004176968697725*u*v2 + 0.520113697995963*u2*v2 - 
     0.251564804320496*u3*v2 + 0.08755029317854246*u4*v2 - 
     0.018929896217160017*u5*v2 + 0.0018904202297008643*u6*v2 - 
     2.2670918716867487*v3 + 0.6858555690183518*u*v3 - 
     0.251564804320496*u2*v3 + 0.0752590109961792*u3*v3 - 
     0.014812198869903107*u4*v3 + 0.001388160767640295*u5*v3 + 
     2.0230034428855013*v4 - 0.386061737342387*u*v4 + 
     0.08755029317854246*u2*v4 - 0.014812198869903107*u3*v4 + 
     0.0012633787101392036*u4*v4 - 1.2541858328141202*v5 + 
     0.1475870262830398*u*v5 - 0.018929896217160017*u2*v5 + 
     0.001388160767640295*u3*v5 + 0.5112314175751462*v6 - 
     0.0339914082143444*u*v6 + 0.0018904202297008643*u2*v6 - 
     0.12321474882489236*v7 + 0.0035512018991495737*u*v7 + 
     0.013314221484862214*v8 - 1.2020569031595942*w + 
     0.9809537469447522*u*w - 0.9004176968697725*u2*w + 
     0.6858555690183518*u3*w - 0.386061737342387*u4*w + 
     0.1475870262830398*u5*w - 0.0339914082143444*u6*w + 
     0.0035512018991495737*u7*w + 0.9809537469447522*v*w - 
     0.5522472005141469*u*v*w + 0.3313625873896489*u2*v*w - 
     0.162591133426937*u3*v*w + 0.056971794565663605*u4*v*w - 
     0.012364843466299529*u5*v*w + 0.0012376963517727734*u6*v*w - 
     0.9004176968697725*v2*w + 0.3313625873896489*u*v2*w - 
     0.12887042405432217*u2*v2*w + 0.03949367079133293*u3*v2*w - 
     0.007867552944290607*u4*v2*w + 0.0007424815539584533*u5*v2*w + 
     0.6858555690183518*v3*w - 0.162591133426937*u*v3*w + 
     0.03949367079133293*u2*v3*w - 0.006882314713446356*u3*v3*w + 
     0.0005960914163116732*u4*v3*w - 0.386061737342387*v4*w + 
     0.056971794565663605*u*v4*w - 0.007867552944290607*u2*v4*w + 
     0.0005960914163116732*u3*v4*w + 0.1475870262830398*v5*w - 
     0.012364843466299529*u*v5*w + 0.0007424815539584533*u2*v5*w - 
     0.0339914082143444*v6*w + 0.0012376963517727734*u*v6*w + 
     0.0035512018991495737*v7*w + 1.8468559937823905*w2 - 
     0.9004176968697725*u*w2 + 0.520113697995963*u2*w2 - 
     0.251564804320496*u3*w2 + 0.08755029317854246*u4*w2 - 
     0.018929896217160017*u5*w2 + 0.0018904202297008643*u6*w2 - 
     0.9004176968697725*v*w2 + 0.3313625873896489*u*v*w2 - 
     0.12887042405432217*u2*v*w2 + 0.03949367079133293*u3*v*w2 - 
     0.007867552944290607*u4*v*w2 + 0.0007424815539584533*u5*v*w2 + 
     0.520113697995963*v2*w2 - 0.12887042405432217*u*v2*w2 + 
     0.03186642256716374*u2*v2*w2 - 0.00560074458687052*u3*v2*w2 + 
     0.000487363420148127*u4*v2*w2 - 0.251564804320496*v3*w2 + 
     0.03949367079133293*u*v3*w2 - 0.00560074458687052*u2*v3*w2 + 
     0.0004300153542944288*u3*v3*w2 + 0.08755029317854246*v4*w2 - 
     0.007867552944290607*u*v4*w2 + 0.000487363420148127*u2*v4*w2 - 
     0.018929896217160017*v5*w2 + 0.0007424815539584533*u*v5*w2 + 
     0.0018904202297008643*v6*w2 - 2.2670918716867487*w3 + 
     0.6858555690183518*u*w3 - 0.251564804320496*u2*w3 + 
     0.0752590109961792*u3*w3 - 0.014812198869903107*u4*w3 + 
     0.001388160767640295*u5*w3 + 0.6858555690183518*v*w3 - 
     0.162591133426937*u*v*w3 + 0.03949367079133293*u2*v*w3 - 
     0.006882314713446356*u3*v*w3 + 0.0005960914163116732*u4*v*w3 - 
     0.251564804320496*v2*w3 + 0.03949367079133293*u*v2*w3 - 
     0.00560074458687052*u2*v2*w3 + 0.0004300153542944288*u3*v2*w3 + 
     0.0752590109961792*v3*w3 - 0.006882314713446356*u*v3*w3 + 
     0.0004300153542944288*u2*v3*w3 - 0.014812198869903107*v4*w3 + 
     0.0005960914163116732*u*v4*w3 + 0.001388160767640295*v5*w3 + 
     2.0230034428855013*w4 - 0.386061737342387*u*w4 + 
     0.08755029317854246*u2*w4 - 0.014812198869903107*u3*w4 + 
     0.0012633787101392036*u4*w4 - 0.386061737342387*v*w4 + 
     0.056971794565663605*u*v*w4 - 0.007867552944290607*u2*v*w4 + 
     0.0005960914163116732*u3*v*w4 + 0.08755029317854246*v2*w4 - 
     0.007867552944290607*u*v2*w4 + 0.000487363420148127*u2*v2*w4 - 
     0.014812198869903107*v3*w4 + 0.0005960914163116732*u*v3*w4 + 
     0.0012633787101392036*v4*w4 - 1.2541858328141202*w5 + 
     0.1475870262830398*u*w5 - 0.018929896217160017*u2*w5 + 
     0.001388160767640295*u3*w5 + 0.1475870262830398*v*w5 - 
     0.012364843466299529*u*v*w5 + 0.0007424815539584533*u2*v*w5 - 
     0.018929896217160017*v2*w5 + 0.0007424815539584533*u*v2*w5 + 
     0.001388160767640295*v3*w5 + 0.5112314175751462*w6 - 
     0.0339914082143444*u*w6 + 0.0018904202297008643*u2*w6 - 
     0.0339914082143444*v*w6 + 0.0012376963517727734*u*v*w6 + 
     0.0018904202297008643*v2*w6 - 0.12321474882489236*w7 + 
     0.0035512018991495737*u*w7 + 0.0035512018991495737*v*w7 + 
     0.013314221484862214*w8 - 1.2020569031595942*x + 
     0.9809537469447522*u*x - 0.9004176968697725*u2*x + 
     0.6858555690183518*u3*x - 0.386061737342387*u4*x + 
     0.1475870262830398*u5*x - 0.0339914082143444*u6*x + 
     0.0035512018991495737*u7*x + 0.9809537469447522*v*x + 
     1.7296818908683025*u*v*x + 0.9115194817125674*lnbara*u*v*x + 
     2.950865208678974*u2*v*x + 0.7823223301987707*lnbara*u2*v*x + 
     1.131721092930809*u3*v*x + 0.41420887449201577*lnbara*u3*v*x + 
     0.47680804183544256*u4*v*x + 0.10984007042350338*lnbara*u4*v*x + 
     0.046951959438293674*u5*v*x + 0.018029304599241732*lnbara*u5*v*x + 
     0.006061422665855913*u6*v*x + 0.0010840252459144199*lnbara*u6*v*x - 
     3.3276691614032563*v2*x - 0.9115194817125674*lnbara*v2*x - 
     5.289736299142919*u*v2*x - 1.9748773458715838*lnbara*u*v2*x - 
     4.650086557902906*u2*v2*x - 1.3823733765239528*lnbara*u2*v2*x - 
     1.6750109905361525*u3*v2*x - 0.5107892746111199*lnbara*u3*v2*x - 
     0.3563815319580459*u4*v2*x - 0.09558272829689277*lnbara*u4*v2*x - 
     0.026406119658478486*u5*v2*x - 0.007442787768918609*lnbara*u5*v2*x + 
     1.7170463437119636*v3*x + 0.4329554475790071*lnbara*v3*x + 
     8.335636464746663*u*v3*x + 2.843556800957142*lnbara*u*v3*x + 
     1.7021286302045417*u2*v3*x + 0.46172402438324356*lnbara*u2*v3*x + 
     1.5576572952608156*u3*v3*x + 0.465804859856524*lnbara*u3*v3*x - 
     0.01399123854239156*u4*v3*x - 0.006838625664216695*lnbara*u4*v3*x - 
     2.581767811748015*v4*x - 0.7450820516264817*lnbara*v4*x - 
     4.408138300869153*u*v4*x - 1.450672032069528*lnbara*u*v4*x - 
     1.1428845474172378*u2*v4*x - 0.3173310228787825*lnbara*u2*v4*x - 
     0.2223252216914388*u3*v4*x - 0.06514335645328273*lnbara*u3*v4*x + 
     0.9506587243582331*v5*x + 0.2748859330079916*lnbara*v5*x + 
     1.84547571556521*u*v5*x + 0.5730994907821886*lnbara*u*v5*x + 
     0.11799273363683711*u2*v5*x + 0.03151257404486913*lnbara*u2*v5*x - 
     0.3370863430138181*v6*x - 0.09609081997302192*lnbara*v6*x - 
     0.25711383486763884*u*v6*x - 0.07776386003863338*lnbara*u*v6*x + 
     0.039205434782531995*v7*x + 0.01108690204816448*lnbara*v7*x + 
     0.7968713467733699*w*x - 2.9443689245385314*u*w*x - 
     0.9115194817125674*lnbara*u*w*x - 2.177117134294563*u2*w*x - 
     0.7823223301987707*lnbara*u2*w*x - 1.5212050940015565*u3*w*x - 
     0.41420887449201577*lnbara*u3*w*x - 0.33851565849786086*u4*w*x - 
     0.10984007042350338*lnbara*u4*w*x - 0.07720156919970216*u5*w*x - 
     0.018029304599241732*lnbara*u5*w*x - 0.003018119129298935*u6*w*x - 
     0.0010840252459144199*lnbara*u6*w*x - 0.5171174600050477*v*w*x + 
     0.2576868260776136*u*v*w*x - 0.1113550912072001*u2*v*w*x + 
     0.03580809539433375*u3*v*w*x - 0.007320613539132591*u4*v*w*x + 
     0.0007017792893363362*u5*v*w*x + 16.840740404671994*v2*w*x + 
     5.75015242423565*lnbara*v2*w*x - 7.4944534531230955*u*v2*w*x - 
     2.569763477061043*lnbara*u*v2*w*x + 10.92704206771195*u2*v2*w*x + 
     3.4712775032258807*lnbara*u2*v2*w*x - 2.7345557618171856*u3*v2*w*x - 
     0.8563693554764021*lnbara*u3*v2*w*x + 0.5508222899910821*u4*v2*w*x + 
     0.16402829156973656*lnbara*u4*v2*w*x - 11.553644836317881*v3*w*x - 
     3.9890155563258416*lnbara*v3*w*x - 0.5200656204166001*u*v3*w*x - 
     0.04824677725141553*lnbara*u*v3*w*x - 2.7556939387551567*u2*v3*w*x - 
     0.8803435778848646*lnbara*u2*v3*w*x + 0.0688751348615315*u3*v3*w*x + 
     0.029360220026995808*lnbara*u3*v3*w*x + 10.200182207413253*v4*w*x + 
     3.28316404172915*lnbara*v4*w*x - 1.034655985722088*u*v4*w*x - 
     0.336346527741606*lnbara*u*v4*w*x + 0.8981582264276017*u2*v4*w*x + 
     0.26783993094333697*lnbara*u2*v4*w*x - 3.234078310073586*v5*w*x - 
     1.0179245136226938*lnbara*v5*w*x + 0.01575761553644318*u*v5*w*x + 
     0.010275898356323341*lnbara*u*v5*w*x + 0.5245800234274332*v6*w*x + 
     0.15852336677018297*lnbara*v6*w*x + 1.7490619043784765*w2*x + 
     0.9115194817125674*lnbara*w2*x + 6.04138458368625*u*w2*x + 
     1.9748773458715838*lnbara*u*w2*x + 4.3302489692293955*u2*w2*x + 
     1.3823733765239528*lnbara*u2*w2*x + 1.7772082042508452*u3*w2*x + 
     0.5107892746111199*lnbara*u3*w2*x + 0.33555238697393286*u4*w2*x + 
     0.09558272829689277*lnbara*u4*w2*x + 0.028399509890409173*u5*w2*x + 
     0.007442787768918609*lnbara*u5*w2*x - 16.220273175775272*v*w2*x - 
     5.75015242423565*lnbara*v*w2*x + 7.283613291269799*u*v*w2*x + 
     2.569763477061043*lnbara*u*v*w2*x - 10.868438804341743*u2*v*w2*x - 
     3.4712775032258807*lnbara*u2*v*w2*x + 2.723658999110921*u3*v*w2*x + 
     0.8563693554764021*lnbara*u3*v*w2*x - 0.5498436699327358*u4*v*w2*x - 
     0.16402829156973656*lnbara*u4*v*w2*x - 0.1223579533330431*v2*w2*x + 
     0.027313138910434232*u*v2*w2*x - 0.004442047950830934*u2*v2*w2*x + 
     0.0003652020446131235*u3*v2*w2*x - 6.2616857203946035*v3*w2*x - 
     1.8196275948475404*lnbara*v3*w2*x + 3.5848494417836383*u*v3*w2*x + 
     1.0856448829578143*lnbara*u*v3*w2*x - 0.660951971257608*u2*v3*w2*
      x - 0.18951332857164516*lnbara*u2*v3*w2*x + 
     0.7756766725864659*v4*w2*x + 0.2013783664993635*lnbara*v4*w2*x - 
     0.3272951320069847*u*v4*w2*x - 0.09992430356434646*lnbara*u*v4*w2*
      x - 0.22871819549961922*v5*w2*x - 0.05887745268544395*lnbara*v5*w2*
      x - 0.5311674638752799*w3*x - 0.4329554475790071*lnbara*w3*x - 
     8.70945468065623*u*w3*x - 2.843556800957142*lnbara*u*w3*x - 
     1.6011517276791973*u2*w3*x - 0.46172402438324356*lnbara*u2*w3*x - 
     1.576189905969709*u3*w3*x - 0.465804859856524*lnbara*u3*w3*x + 
     0.015644250220599377*u4*w3*x + 0.006838625664216695*lnbara*u4*w3*x + 
     11.250037884026336*v*w3*x + 3.9890155563258416*lnbara*v*w3*x + 
     0.5861690638750913*u*v*w3*x + 0.04824677725141553*lnbara*u*v*w3*x + 
     2.7450669338987366*u2*v*w3*x + 0.8803435778848646*lnbara*u2*v*w3*x - 
     0.06800667812701873*u3*v*w3*x - 0.029360220026995808*lnbara*u3*v*w3*
      x + 6.336956677089462*v2*w3*x + 1.8196275948475404*lnbara*v2*w3*x - 
     3.5946630672038005*u*v2*w3*x - 1.0856448829578143*lnbara*u*v2*w3*x + 
     0.6616579679935063*u2*v2*w3*x + 0.18951332857164516*lnbara*u2*v2*
      w3*x - 0.006605505610793891*v3*w3*x + 0.00038509593985133696*u*v3*
      w3*x + 0.38164842058355447*v4*w3*x + 0.10823518328690307*lnbara*v4*
      w3*x + 1.9188684163037728*w4*x + 0.7450820516264817*lnbara*w4*x + 
     4.540055219513269*u*w4*x + 1.450672032069528*lnbara*u*w4*x + 
     1.122430115235894*u2*w4*x + 0.3173310228787825*lnbara*u2*w4*x + 
     0.2239680885196098*u3*w4*x + 0.06514335645328273*lnbara*u3*w4*x - 
     10.094036110016523*v*w4*x - 3.28316404172915*lnbara*v*w4*x + 
     1.0213274392385536*u*v*w4*x + 0.336346527741606*lnbara*u*v*w4*x - 
     0.8972167925745769*u2*v*w4*x - 0.26783993094333697*lnbara*u2*v*w4*
      x - 0.7906870068555788*v2*w4*x - 0.2013783664993635*lnbara*v2*w4*
      x + 0.3281593662661477*u*v2*w4*x + 0.09992430356434646*lnbara*u*v2*
      w4*x - 0.380500721540267*v3*w4*x - 0.10823518328690307*lnbara*v3*
      w4*x - 0.6982508102158693*w5*x - 0.2748859330079916*lnbara*w5*x - 
     1.8742275538737843*u*w5*x - 0.5730994907821886*lnbara*u*w5*x - 
     0.11604224021663521*u2*w5*x - 0.03151257404486913*lnbara*u2*w5*x + 
     3.2110778233847816*v*w5*x + 1.0179245136226938*lnbara*v*w5*x - 
     0.01449065568595044*u*v*w5*x - 0.010275898356323341*lnbara*u*v*w5*x + 
     0.2301351338116376*v2*w5*x + 0.05887745268544395*lnbara*v2*w5*x + 
     0.27910097428080544*w6*x + 0.09609081997302192*lnbara*w6*x + 
     0.2599997989809029*u*w6*x + 0.07776386003863338*lnbara*u*w6*x - 
     0.5222803748590084*v*w6*x - 0.15852336677018297*lnbara*v*w6*x - 
     0.033158063533986065*w7*x - 0.01108690204816448*lnbara*w7*x + 
     1.8468559937823905*x2 - 0.9004176968697725*u*x2 + 
     0.520113697995963*u2*x2 - 0.251564804320496*u3*x2 + 
     0.08755029317854246*u4*x2 - 0.018929896217160017*u5*x2 + 
     0.0018904202297008643*u6*x2 - 3.3276691614032563*v*x2 - 
     0.9115194817125674*lnbara*v*x2 - 5.289736299142919*u*v*x2 - 
     1.9748773458715838*lnbara*u*v*x2 - 4.650086557902906*u2*v*x2 - 
     1.3823733765239528*lnbara*u2*v*x2 - 1.6750109905361525*u3*v*x2 - 
     0.5107892746111199*lnbara*u3*v*x2 - 0.3563815319580459*u4*v*x2 - 
     0.09558272829689277*lnbara*u4*v*x2 - 0.026406119658478486*u5*v*x2 - 
     0.007442787768918609*lnbara*u5*v*x2 + 9.22032358478725*v2*x2 + 
     3.1447095994394325*lnbara*v2*x2 + 3.443135626202991*u*v2*x2 + 
     1.2110841723003059*lnbara*u*v2*x2 + 9.306085458851838*u2*v2*x2 + 
     2.882695368781348*lnbara*u2*v2*x2 + 0.170862170132498*u3*v2*x2 + 
     0.022605565124086877*lnbara*u3*v2*x2 + 0.363288554298664*u4*v2*
      x2 + 0.10475946602085698*lnbara*u4*v2*x2 - 
     6.021984802397401*v3*x2 - 2.0616669488742043*lnbara*v3*x2 - 
     6.264516709859727*u*v3*x2 - 2.002014089554316*lnbara*u*v3*x2 - 
     3.2461502369868342*u2*v3*x2 - 0.9751555645194946*lnbara*u2*v3*x2 - 
     0.35640766524496237*u3*v3*x2 - 0.09812666811852842*lnbara*u3*v3*
      x2 + 5.626060910791232*v4*x2 + 1.8035499324774895*lnbara*v4*x2 + 
     1.1739473090123895*u*v4*x2 + 0.3798257341629295*lnbara*u*v4*x2 + 
     0.9661938550317448*u2*v4*x2 + 0.2809590613076102*lnbara*u2*v4*x2 - 
     1.6435396981777775*v5*x2 - 0.5192029262043804*lnbara*v5*x2 - 
     0.31687310350773107*u*v5*x2 - 0.09418670348943739*lnbara*u*v5*x2 + 
     0.2767842038860737*v6*x2 + 0.08359044675304854*lnbara*v6*x2 + 
     1.7490619043784765*w*x2 + 0.9115194817125674*lnbara*w*x2 + 
     6.04138458368625*u*w*x2 + 1.9748773458715838*lnbara*u*w*x2 + 
     4.3302489692293955*u2*w*x2 + 1.3823733765239528*lnbara*u2*w*x2 + 
     1.7772082042508452*u3*w*x2 + 0.5107892746111199*lnbara*u3*w*x2 + 
     0.33555238697393286*u4*w*x2 + 0.09558272829689277*lnbara*u4*w*x2 + 
     0.028399509890409173*u5*w*x2 + 0.007442787768918609*lnbara*u5*w*x2 - 
     10.763863509478952*v*w*x2 - 3.7979978404690304*lnbara*v*w*x2 + 
     14.408707884242522*u*v*w*x2 + 4.851483174726708*lnbara*u*v*w*x2 - 
     11.35352724075799*u2*v*w*x2 - 3.638468221673086*lnbara*u2*v*w*x2 + 
     3.595212268118998*u3*v*w*x2 + 1.124626001903981*lnbara*u3*v*w*x2 - 
     0.637464961753253*u4*v*w*x2 - 0.19025408565816512*lnbara*u4*v*w*x2 - 
     2.5057639727820775*v2*w*x2 - 1.0258918950424711*lnbara*v2*w*x2 - 
     4.04603639452907*u*v2*w*x2 - 1.3032259701999138*lnbara*u*v2*w*x2 - 
     0.5438356537989508*u2*v2*w*x2 - 0.1818291572693978*lnbara*u2*v2*w*
      x2 - 0.17072856838087597*u3*v2*w*x2 - 0.04904808595029047*lnbara*
      u3*v2*w*x2 - 4.961706764789829*v3*w*x2 - 
     1.3124617908295524*lnbara*v3*w*x2 + 8.457223146401986*u*v3*w*x2 + 
     2.5565136487937905*lnbara*u*v3*w*x2 - 0.9782558475006098*u2*v3*w*
      x2 - 0.28088060701825807*lnbara*u2*v3*w*x2 + 
     0.01586781553997586*v4*w*x2 - 0.06352181102996676*lnbara*v4*w*x2 - 
     1.0816608287503775*u*v4*w*x2 - 0.32438223975151587*lnbara*u*v4*w*
      x2 - 0.14867177025989595*v5*w*x2 - 0.030113006529747105*lnbara*v5*w*
      x2 + 2.7099088373198637*w2*x2 + 0.653288241029598*lnbara*w2*x2 - 
     18.2664360726891*u*w2*x2 - 6.062567347027013*lnbara*u*w2*x2 + 
     2.165326096275261*u2*w2*x2 + 0.7557728528917382*lnbara*u2*w2*x2 - 
     3.7882789087219217*u3*w2*x2 - 1.1472315670280682*lnbara*u3*w2*x2 + 
     0.2761865963929522*u4*w2*x2 + 0.08549461963730814*lnbara*u4*w2*
      x2 + 6.988719566598391*v*w2*x2 + 2.737479438521948*lnbara*v*w2*
      x2 + 2.583201962578535*u*v*w2*x2 + 0.6303616509938534*lnbara*u*v*w2*
      x2 + 2.78012177163788*u2*v*w2*x2 + 0.9193789312609476*lnbara*u2*v*
      w2*x2 + 0.0944959815922243*u3*v*w2*x2 + 0.012699061790125279*
      lnbara*u3*v*w2*x2 + 8.25179929321027*v2*w2*x2 + 
     2.469212605593015*lnbara*v2*w2*x2 - 8.325060940884864*u*v2*w2*x2 - 
     2.4651692331377286*lnbara*u*v2*w2*x2 + 1.4020259571353741*u2*v2*w2*
      x2 + 0.40905101643700437*lnbara*u2*v2*w2*x2 - 
     0.4179222450177246*v3*w2*x2 - 0.11149384692877298*lnbara*v3*w2*
      x2 + 0.3092023748907339*u*v3*w2*x2 + 0.0785869636098416*lnbara*u*
      v3*w2*x2 + 0.581863351202302*v4*w2*x2 + 
     0.1659602153670836*lnbara*v4*w2*x2 + 0.8972500249744643*w3*x2 + 
     0.3500794053947276*lnbara*w3*x2 + 7.880191664548825*u*w3*x2 + 
     2.6748784087603763*lnbara*u*w3*x2 + 0.9841419128221275*u2*w3*x2 + 
     0.23760579052794478*lnbara*u2*w3*x2 + 0.43479804170814224*u3*w3*
      x2 + 0.1344756922786936*lnbara*u3*w3*x2 - 
     9.326716278242827*v*w3*x2 - 3.107847444868102*lnbara*v*w3*x2 + 
     2.4434643394917717*u*v*w3*x2 + 0.722704550671465*lnbara*u*v*w3*x2 - 
     1.4446667880342972*u2*v*w3*x2 - 0.43070638308913284*lnbara*u2*v*w3*
      x2 - 0.2730340390527256*v2*w3*x2 - 0.09631911852193811*lnbara*v2*
      w3*x2 + 0.5346542533590023*u*v2*w3*x2 + 0.17261962424250146*lnbara*
      u*v2*w3*x2 - 0.6281581353908926*v3*w3*x2 - 
     0.1774903697854292*lnbara*v3*w3*x2 + 0.651338693005845*w4*x2 + 
     0.14754669762714998*lnbara*w4*x2 - 3.7835481298107045*u*w4*x2 - 
     1.1938747004904564*lnbara*u*w4*x2 + 0.05725447402071905*u2*w4*x2 + 
     0.021576912362776347*lnbara*u2*w4*x2 + 2.2331993759180806*v*w4*x2 + 
     0.7744184574067065*lnbara*v*w4*x2 + 0.08267922567088919*u*v*w4*x2 + 
     0.01813757575979942*lnbara*u*v*w4*x2 + 0.3702307156554167*v2*w4*
      x2 + 0.10136840292874362*lnbara*v2*w4*x2 + 
     0.03085436595372382*w5*x2 + 0.016119245278351776*lnbara*w5*x2 + 
     0.47543762922904986*u*w5*x2 + 0.14922477962881076*lnbara*u*w5*x2 - 
     0.46033723691317213*v*w5*x2 - 0.14606708002401708*lnbara*v*w5*x2 + 
     0.013912167390291797*w6*x2 + 0.0027513912903176212*lnbara*w6*x2 - 
     2.2670918716867487*x3 + 0.6858555690183518*u*x3 - 
     0.251564804320496*u2*x3 + 0.0752590109961792*u3*x3 - 
     0.014812198869903107*u4*x3 + 0.001388160767640295*u5*x3 + 
     1.7170463437119636*v*x3 + 0.4329554475790071*lnbara*v*x3 + 
     8.335636464746663*u*v*x3 + 2.843556800957142*lnbara*u*v*x3 + 
     1.7021286302045417*u2*v*x3 + 0.46172402438324356*lnbara*u2*v*x3 + 
     1.5576572952608156*u3*v*x3 + 0.465804859856524*lnbara*u3*v*x3 - 
     0.01399123854239156*u4*v*x3 - 0.006838625664216695*lnbara*u4*v*x3 - 
     6.021984802397401*v2*x3 - 2.0616669488742043*lnbara*v2*x3 - 
     6.264516709859727*u*v2*x3 - 2.002014089554316*lnbara*u*v2*x3 - 
     3.2461502369868342*u2*v2*x3 - 0.9751555645194946*lnbara*u2*v2*x3 - 
     0.35640766524496237*u3*v2*x3 - 0.09812666811852842*lnbara*u3*v2*
      x3 + 0.817892698661721*v3*x3 + 0.36798111338220896*lnbara*v3*x3 + 
     5.761241784590613*u*v3*x3 + 1.759687908443654*lnbara*u*v3*x3 + 
     0.23971655435506595*u2*v3*x3 + 0.06962388387677895*lnbara*u2*v3*
      x3 - 1.2251346623182928*v4*x3 - 0.4000233210925086*lnbara*v4*x3 - 
     0.7794687099225679*u*v4*x3 - 0.23484948154842047*lnbara*u*v4*x3 + 
     0.0956649502705806*v5*x3 + 0.03428802753892857*lnbara*v5*x3 - 
     0.5311674638752799*w*x3 - 0.4329554475790071*lnbara*w*x3 - 
     8.70945468065623*u*w*x3 - 2.843556800957142*lnbara*u*w*x3 - 
     1.6011517276791973*u2*w*x3 - 0.46172402438324356*lnbara*u2*w*x3 - 
     1.576189905969709*u3*w*x3 - 0.465804859856524*lnbara*u3*w*x3 + 
     0.015644250220599377*u4*w*x3 + 0.006838625664216695*lnbara*u4*w*x3 + 
     4.570769251234671*v*w*x3 + 1.7115875434794767*lnbara*v*w*x3 - 
     1.485347767785549*u*v*w*x3 - 0.6728643192060604*lnbara*u*v*w*x3 + 
     2.240193096445929*u2*v*w*x3 + 0.7375497739915498*lnbara*u2*v*w*x3 - 
     0.07656531816033134*u3*v*w*x3 - 0.03634902416016519*lnbara*u3*v*w*
      x3 + 9.909152497593267*v2*w*x3 + 3.0912384401060025*lnbara*v2*w*
      x3 - 5.305139799956746*u*v2*w*x3 - 1.5393343490108804*lnbara*u*v2*w*
      x3 + 1.5675237969406215*u2*v2*w*x3 + 0.4579464223436356*lnbara*u2*
      v2*w*x3 - 1.6389294857674492*v3*w*x3 - 0.5238484235184111*lnbara*
      v3*w*x3 - 0.30529725405798797*u*v3*w*x3 - 
     0.10319808585533102*lnbara*u*v3*w*x3 + 0.8263404688000965*v4*w*x3 + 
     0.2438589036761638*lnbara*v4*w*x3 + 0.8972500249744643*w2*x3 + 
     0.3500794053947276*lnbara*w2*x3 + 7.880191664548825*u*w2*x3 + 
     2.6748784087603763*lnbara*u*w2*x3 + 0.9841419128221275*u2*w2*x3 + 
     0.23760579052794478*lnbara*u2*w2*x3 + 0.43479804170814224*u3*w2*
      x3 + 0.1344756922786936*lnbara*u3*w2*x3 - 
     10.997725885446338*v*w2*x3 - 3.557746295983214*lnbara*v*w2*x3 + 
     2.9792615221240184*u*v*w2*x3 + 0.9000686333403095*lnbara*u*v*w2*x3 - 
     1.8066447210612555*u2*v*w2*x3 - 0.5331767926648573*lnbara*u2*v*w2*
      x3 + 0.3997513066521812*v2*w2*x3 + 0.08307962011998349*lnbara*v2*
      w2*x3 + 0.5497959040120096*u*v2*w2*x3 + 0.17704276038055036*lnbara*
      u*v2*w2*x3 - 0.8243258774524206*v3*w2*x3 - 
     0.23041500895027525*lnbara*v3*w2*x3 + 0.4593564384590013*w3*x3 + 
     0.09852674249500254*lnbara*w3*x3 - 3.462967340849832*u*w3*x3 - 
     1.1204221927730833*lnbara*u*w3*x3 + 0.0015185718493600246*u2*w3*
      x3 + 0.005606486444442812*lnbara*u2*w3*x3 + 
     2.2465725155547824*v*w3*x3 + 0.7870789002790479*lnbara*v*w3*x3 + 
     0.1244773390895362*u*v*w3*x3 + 0.024916275986332885*lnbara*u*v*w3*
      x3 + 0.461488349439593*v2*w3*x3 + 0.12645487928146082*lnbara*v2*
      w3*x3 + 0.17761627523096088*w4*x3 + 0.05371322421188824*lnbara*w4*
      x3 + 0.4131831056920087*u*w4*x3 + 0.13608853103686827*lnbara*u*w4*
      x3 - 0.5665614064319183*v*w4*x3 - 0.1764702004496186*lnbara*v*w4*
      x3 + 0.01134547383834047*w5*x3 + 0.002283398903340656*lnbara*w5*
      x3 + 2.0230034428855013*x4 - 0.386061737342387*u*x4 + 
     0.08755029317854246*u2*x4 - 0.014812198869903107*u3*x4 + 
     0.0012633787101392036*u4*x4 - 2.581767811748015*v*x4 - 
     0.7450820516264817*lnbara*v*x4 - 4.408138300869153*u*v*x4 - 
     1.450672032069528*lnbara*u*v*x4 - 1.1428845474172378*u2*v*x4 - 
     0.3173310228787825*lnbara*u2*v*x4 - 0.2223252216914388*u3*v*x4 - 
     0.06514335645328273*lnbara*u3*v*x4 + 5.626060910791232*v2*x4 + 
     1.8035499324774895*lnbara*v2*x4 + 1.1739473090123895*u*v2*x4 + 
     0.3798257341629295*lnbara*u*v2*x4 + 0.9661938550317448*u2*v2*x4 + 
     0.2809590613076102*lnbara*u2*v2*x4 - 1.2251346623182928*v3*x4 - 
     0.4000233210925086*lnbara*v3*x4 - 0.7794687099225679*u*v3*x4 - 
     0.23484948154842047*lnbara*u*v3*x4 + 0.43991353613455403*v4*x4 + 
     0.1319480025020833*lnbara*v4*x4 + 1.9188684163037728*w*x4 + 
     0.7450820516264817*lnbara*w*x4 + 4.540055219513269*u*w*x4 + 
     1.450672032069528*lnbara*u*w*x4 + 1.122430115235894*u2*w*x4 + 
     0.3173310228787825*lnbara*u2*w*x4 + 0.2239680885196098*u3*w*x4 + 
     0.06514335645328273*lnbara*u3*w*x4 - 6.0865303594656925*v*w*x4 - 
     1.9510966301046395*lnbara*v*w*x4 + 2.583274807875346*u*v*w*x4 + 
     0.814048966327527*lnbara*u*v*w*x4 - 1.0214921534358323*u2*v*w*x4 - 
     0.3025359736703866*lnbara*u2*v*w*x4 - 1.4226442649834423*v2*w*x4 - 
     0.4703228534910837*lnbara*v2*w*x4 + 0.2656466906715195*u*v2*w*x4 + 
     0.08058867723285898*lnbara*u*v2*w*x4 - 0.31318215309396386*v3*w*x4 - 
     0.07872508228066824*lnbara*v3*w*x4 + 0.651338693005845*w2*x4 + 
     0.14754669762714998*lnbara*w2*x4 - 3.7835481298107045*u*w2*x4 - 
     1.1938747004904564*lnbara*u*w2*x4 + 0.05725447402071905*u2*w2*x4 + 
     0.021576912362776347*lnbara*u2*w2*x4 + 2.4334094721170496*v*w2*x4 + 
     0.816632950371704*lnbara*v*w2*x4 + 0.10307841593891565*u*v*w2*x4 + 
     0.018172273278693238*lnbara*u*v*w2*x4 + 0.39505594064956606*v2*w2*
      x4 + 0.11018942250071202*lnbara*v2*w2*x4 + 
     0.17761627523096088*w3*x4 + 0.05371322421188824*lnbara*w3*x4 + 
     0.4131831056920087*u*w3*x4 + 0.13608853103686827*lnbara*u*w3*x4 - 
     0.5397225167390112*v*w3*x4 - 0.16840858737356737*lnbara*v*w3*x4 + 
     0.021325233798974344*w4*x4 + 0.004996244651440274*lnbara*w4*x4 - 
     1.2541858328141202*x5 + 0.1475870262830398*u*x5 - 
     0.018929896217160017*u2*x5 + 0.001388160767640295*u3*x5 + 
     0.9506587243582331*v*x5 + 0.2748859330079916*lnbara*v*x5 + 
     1.84547571556521*u*v*x5 + 0.5730994907821886*lnbara*u*v*x5 + 
     0.11799273363683711*u2*v*x5 + 0.03151257404486913*lnbara*u2*v*x5 - 
     1.6435396981777775*v2*x5 - 0.5192029262043804*lnbara*v2*x5 - 
     0.31687310350773107*u*v2*x5 - 0.09418670348943739*lnbara*u*v2*x5 + 
     0.0956649502705806*v3*x5 + 0.03428802753892857*lnbara*v3*x5 - 
     0.6982508102158693*w*x5 - 0.2748859330079916*lnbara*w*x5 - 
     1.8742275538737843*u*w*x5 - 0.5730994907821886*lnbara*u*w*x5 - 
     0.11604224021663521*u2*w*x5 - 0.03151257404486913*lnbara*u2*w*x5 + 
     1.5716677873215597*v*w*x5 + 0.5030836809260285*lnbara*v*w*x5 - 
     0.15605883003169002*u*v*w*x5 - 0.05503807613937338*lnbara*u*v*w*x5 + 
     0.4211542429001266*v2*w*x5 + 0.12424611305742275*lnbara*v2*w*x5 + 
     0.03085436595372382*w2*x5 + 0.016119245278351776*lnbara*w2*x5 + 
     0.47543762922904986*u*w2*x5 + 0.14922477962881076*lnbara*u*w2*x5 - 
     0.5247428948621341*v*w2*x5 - 0.16081753949969196*lnbara*v*w2*x5 + 
     0.01134547383834047*w3*x5 + 0.002283398903340656*lnbara*w3*x5 + 
     0.5112314175751462*x6 - 0.0339914082143444*u*x6 + 
     0.0018904202297008643*u2*x6 - 0.3370863430138181*v*x6 - 
     0.09609081997302192*lnbara*v*x6 - 0.25711383486763884*u*v*x6 - 
     0.07776386003863338*lnbara*u*v*x6 + 0.2767842038860737*v2*x6 + 
     0.08359044675304854*lnbara*v2*x6 + 0.27910097428080544*w*x6 + 
     0.09609081997302192*lnbara*w*x6 + 0.2599997989809029*u*w*x6 + 
     0.07776386003863338*lnbara*u*w*x6 - 0.2866167180827836*v*w*x6 - 
     0.08634183804336616*lnbara*v*w*x6 + 0.013912167390291797*w2*x6 + 
     0.0027513912903176212*lnbara*w2*x6 - 0.12321474882489236*x7 + 
     0.0035512018991495737*u*x7 + 0.039205434782531995*v*x7 + 
     0.01108690204816448*lnbara*v*x7 - 0.033158063533986065*w*x7 - 
     0.01108690204816448*lnbara*w*x7 + 0.013314221484862214*x8 - 
     1.2020569031595942*y + 0.7968713467733699*u*y - 
     0.6781895601550072*u2*y + 0.5000233108183318*u3*y - 
     0.27683765810185557*u4*y + 0.10482088785932395*u5*y - 
     0.023993960518668304*u6*y + 0.002496169349396353*u7*y + 
     0.9809537469447522*v*y - 0.5171174600050477*u*v*y + 
     0.29918371952782097*u2*v*y - 0.14397058356518277*u3*v*y + 
     0.04988531635163183*u4*v*y - 0.01075135761798518*u5*v*y + 
     0.0010711545725659018*u6*v*y - 0.9004176968697725*v2*y + 
     0.3212835093689003*u*v2*y - 0.1223579533330431*u2*v2*y + 
     0.03702532275275472*u3*v2*y - 0.007317810733171885*u4*v2*y + 
     0.0006870207501448052*u5*v2*y + 0.6858555690183518*v3*y - 
     0.15963636872636416*u*v3*y + 0.03824563394210313*u2*v3*y - 
     0.006605505610793891*u3*v3*y + 0.0005687770966253271*u4*v3*y - 
     0.386061737342387*v4*y + 0.056260781045097456*u*v4*y - 
     0.007692523535941066*u2*v4*y + 0.0005789219466621661*u3*v4*y + 
     0.1475870262830398*v5*y - 0.012249129070819405*u*v5*y + 
     0.0007299175618735945*u2*v5*y - 0.0339914082143444*v6*y + 
     0.001228493995858818*u*v6*y + 0.0035512018991495737*v7*y + 
     0.9809537469447522*w*y - 0.5171174600050477*u*w*y + 
     0.29918371952782097*u2*w*y - 0.14397058356518277*u3*w*y + 
     0.04988531635163183*u4*w*y - 0.01075135761798518*u5*w*y + 
     0.0010711545725659018*u6*w*y - 0.6975695736651812*v*w*y + 
     0.2576868260776136*u*v*w*y - 0.09948507064609574*u2*v*w*y + 
     0.030295348064157374*u3*v*w*y - 0.006007932944401952*u4*v*w*y + 
     0.000565180561156404*u5*v*w*y + 0.45246456501551086*v2*w*y - 
     0.1113550912072001*u*v2*w*y + 0.027313138910434232*u2*v2*w*y - 
     0.004771933785158886*u3*v2*w*y + 0.0004135240269207944*u4*v2*w*y - 
     0.2298476323443832*v3*w*y + 0.03580809539433375*u*v3*w*y - 
     0.005041691635003074*u2*v3*w*y + 0.00038509593985133696*u3*v3*w*y + 
     0.08203160229248424*v4*w*y - 0.007320613539132591*u*v4*w*y + 
     0.0004507102322422891*u2*v4*w*y - 0.018000480690589078*v5*w*y + 
     0.0007017792893363362*u*v5*w*y + 0.0018148095406981609*v6*w*y - 
     0.9004176968697725*w2*y + 0.3212835093689003*u*w2*y - 
     0.1223579533330431*u2*w2*y + 0.03702532275275472*u3*w2*y - 
     0.007317810733171885*u4*w2*y + 0.0006870207501448052*u5*w2*y + 
     0.45246456501551086*v*w2*y - 0.1113550912072001*u*v*w2*y + 
     0.027313138910434232*u2*v*w2*y - 0.004771933785158886*u3*v*w2*y + 
     0.0004135240269207944*u4*v*w2*y - 0.1974796353404678*v2*w2*y + 
     0.03129012445977296*u*v2*w2*y - 0.004442047950830934*u2*v2*w2*y + 
     0.0003407946912851918*u3*v2*w2*y + 0.06395157977258972*v3*w2*y - 
     0.005855071071261368*u*v3*w2*y + 0.0003652020446131235*u2*v3*w2*y - 
     0.013136621448171954*v4*w2*y + 0.000527909826103984*u*v4*w2*y + 
     0.001263472670057093*v5*w2*y + 0.6858555690183518*w3*y - 
     0.15963636872636416*u*w3*y + 0.03824563394210313*u2*w3*y - 
     0.006605505610793891*u3*w3*y + 0.0005687770966253271*u4*w3*y - 
     0.2298476323443832*v*w3*y + 0.03580809539433375*u*v*w3*y - 
     0.005041691635003074*u2*v*w3*y + 0.00038509593985133696*u3*v*w3*y + 
     0.06395157977258972*v2*w3*y - 0.005855071071261368*u*v2*w3*y + 
     0.0003652020446131235*u2*v2*w3*y - 0.011927105098099399*v3*w3*y + 
     0.00048336079466143985*u*v3*w3*y + 0.0010740897315456516*v4*w3*y - 
     0.386061737342387*w4*y + 0.056260781045097456*u*w4*y - 
     0.007692523535941066*u2*w4*y + 0.0005789219466621661*u3*w4*y + 
     0.08203160229248424*v*w4*y - 0.007320613539132591*u*v*w4*y + 
     0.0004507102322422891*u2*v*w4*y - 0.013136621448171954*v2*w4*y + 
     0.000527909826103984*u*v2*w4*y + 0.0010740897315456516*v3*w4*y + 
     0.1475870262830398*w5*y - 0.012249129070819405*u*w5*y + 
     0.0007299175618735945*u2*w5*y - 0.018000480690589078*v*w5*y + 
     0.0007017792893363362*u*v*w5*y + 0.001263472670057093*v2*w5*y - 
     0.0339914082143444*w6*y + 0.001228493995858818*u*w6*y + 
     0.0018148095406981609*v*w6*y + 0.0035512018991495737*w7*y + 
     0.9809537469447522*x*y - 0.5171174600050477*u*x*y + 
     0.29918371952782097*u2*x*y - 0.14397058356518277*u3*x*y + 
     0.04988531635163183*u4*x*y - 0.01075135761798518*u5*x*y + 
     0.0010711545725659018*u6*x*y + 6.729507193086304*v*x*y + 
     2.734558445137702*lnbara*v*x*y + 6.093431672716988*u*v*x*y + 
     1.7772673145259574*lnbara*u*v*x*y + 3.6244323005177024*u2*v*x*y + 
     1.1575775618246809*lnbara*u2*v*x*y + 1.326032325221745*u3*v*x*y + 
     0.36117019327441874*lnbara*u3*v*x*y + 0.2517914740480953*u4*v*x*y + 
     0.07250424934435563*lnbara*u4*v*x*y + 0.020900424621812116*u5*v*x*y + 
     0.0052376950009461855*lnbara*u5*v*x*y - 7.740580989744235*v2*x*y - 
     3.007965370948085*lnbara*v2*x*y - 8.34831676538791*u*v2*x*y - 
     2.570302308603461*lnbara*u*v2*x*y - 4.3167789899540825*u2*v2*x*y - 
     1.303884138860003*lnbara*u2*v2*x*y - 1.0998046234447676*u3*v2*x*y - 
     0.306972893835514*lnbara*u3*v2*x*y - 0.10954859095754421*u4*v2*x*y - 
     0.029980593819149505*lnbara*u4*v2*x*y + 14.150094597275556*v3*x*y + 
     4.8057115258275305*lnbara*v3*x*y + 3.7558333567491418*u*v3*x*y + 
     1.0737089522063064*lnbara*u*v3*x*y + 3.5021103597603167*u2*v3*x*y + 
     1.041566396111375*lnbara*u2*v3*x*y + 0.1081140594632303*u3*v3*x*y + 
     0.025365208983699924*lnbara*u3*v3*x*y - 7.96562555520799*v4*x*y - 
     2.6276787546704745*lnbara*v4*x*y - 2.0218463214313056*u*v4*x*y - 
     0.5668452509261017*lnbara*u*v4*x*y - 0.5891572387478903*u2*v4*x*y - 
     0.17202385787388583*lnbara*u2*v4*x*y + 3.2252148129088143*v5*x*y + 
     1.0069202481453856*lnbara*v5*x*y + 0.23526924787765785*u*v5*x*y + 
     0.06359410612179822*lnbara*u*v5*x*y - 0.4717695787130639*v6*x*y - 
     0.14311605655840995*lnbara*v6*x*y - 7.798871853605498*w*x*y - 
     2.734558445137702*lnbara*w*x*y - 5.604833095618447*u*w*x*y - 
     1.7772673145259574*lnbara*u*w*x*y - 3.826201875320261*u2*w*x*y - 
     1.1575775618246809*lnbara*u2*w*x*y - 1.262771091795936*u3*w*x*y - 
     0.36117019327441874*lnbara*u3*w*x*y - 0.26452915782323383*u4*w*x*y - 
     0.07250424934435563*lnbara*u4*w*x*y - 0.0196913141648136*u5*w*x*y - 
     0.0052376950009461855*lnbara*u5*w*x*y + 0.2576868260776136*v*w*x*y - 
     0.08549175984210892*u*v*w*x*y + 0.023341335158146424*u2*v*w*x*y - 
     0.004288309525390823*u3*v*w*x*y + 0.0003820180212080083*u4*v*w*x*y - 
     15.142507034031205*v2*w*x*y - 4.700428217339612*lnbara*v2*w*x*y + 
     8.684236515504576*u*v2*w*x*y + 2.845063377679382*lnbara*u*v2*w*x*y - 
     2.0369169970657963*u2*v2*w*x*y - 0.6312081178376158*lnbara*u2*v2*w*x*
      y + 0.5879915453536685*u3*v2*w*x*y + 0.17392267685688417*lnbara*u3*
      v2*w*x*y + 2.4387387085229473*v3*w*x*y + 0.726506814968872*lnbara*v3*
      w*x*y - 2.478798620600135*u*v3*w*x*y - 0.8025320427989695*lnbara*u*v3*
      w*x*y - 0.1891735970499443*u2*v3*w*x*y - 0.049895924897055216*lnbara*
      u2*v3*w*x*y - 2.5094652412675753*v4*w*x*y - 
     0.7051754521652789*lnbara*v4*w*x*y + 0.977194806205722*u*v4*w*x*y + 
     0.2916555967133236*lnbara*u*v4*w*x*y + 0.2147385756550609*v5*w*x*y + 
     0.0583092465218906*lnbara*v5*w*x*y + 8.371127296661705*w2*x*y + 
     3.007965370948085*lnbara*w2*x*y + 8.142878738630774*u*w2*x*y + 
     2.570302308603461*lnbara*u*w2*x*y + 4.372397515146079*u2*w2*x*y + 
     1.303884138860003*lnbara*u2*w2*x*y + 1.089630311725629*u3*w2*x*y + 
     0.306972893835514*lnbara*u3*w2*x*y + 0.11045275181230746*u4*w2*x*y + 
     0.029980593819149505*lnbara*u4*w2*x*y + 14.923465303571382*v*w2*x*y + 
     4.700428217339612*lnbara*v*w2*x*y - 8.636182740657008*u*v*w2*x*y - 
     2.845063377679382*lnbara*u*v*w2*x*y + 2.0292029954219255*u2*v*w2*x*
      y + 0.6312081178376158*lnbara*u2*v*w2*x*y - 
     0.5873630368597165*u3*v*w2*x*y - 0.17392267685688417*lnbara*u3*v*w2*
      x*y + 0.03069034292530752*v2*w2*x*y - 0.003995614840649483*u*v2*w2*
      x*y + 0.0002863764888716214*u2*v2*w2*x*y + 
     6.015319832123562*v3*w2*x*y + 1.7632715691466476*lnbara*v3*w2*x*y - 
     0.7304529490550218*u*v3*w2*x*y - 0.21288835092447814*lnbara*u*v3*w2*
      x*y - 0.6626623249676024*v4*w2*x*y - 0.19636416180684646*lnbara*v4*
      w2*x*y - 14.456656314267676*w3*x*y - 4.8057115258275305*lnbara*w3*x*
      y - 3.6907580059131853*u*w3*x*y - 1.0737089522063064*lnbara*u*w3*x*
      y - 3.5123911436649746*u2*w3*x*y - 1.041566396111375*lnbara*u2*w3*x*
      y - 0.10728353612006211*u3*w3*x*y - 0.025365208983699924*lnbara*u3*
      w3*x*y - 2.3689366351444274*v*w3*x*y - 0.726506814968872*lnbara*v*w3*
      x*y + 2.469819197461433*u*v*w3*x*y + 0.8025320427989695*lnbara*u*v*w3*
      x*y + 0.18981304783567338*u2*v*w3*x*y + 0.049895924897055216*lnbara*
      u2*v*w3*x*y - 6.026774148018949*v2*w3*x*y - 
     1.7632715691466476*lnbara*v2*w3*x*y + 0.731117931401122*u*v2*w3*x*
      y + 0.21288835092447814*lnbara*u*v2*w3*x*y + 
     0.00047310410784989136*v3*w3*x*y + 8.072482666125286*w4*x*y + 
     2.6276787546704745*lnbara*w4*x*y + 2.0086611123781046*u*w4*x*y + 
     0.5668452509261017*lnbara*u*w4*x*y + 0.590076818257424*u2*w4*x*y + 
     0.17202385787388583*lnbara*u2*w4*x*y + 2.495271539467373*v*w4*x*y + 
     0.7051754521652789*lnbara*v*w4*x*y - 0.9763863300273927*u*v*w4*x*y - 
     0.2916555967133236*lnbara*u*v*w4*x*y + 0.663692907110261*v2*w4*x*y + 
     0.19636416180684646*lnbara*v2*w4*x*y - 3.248331013993099*w5*x*y - 
     1.0069202481453856*lnbara*w5*x*y - 0.23401253231882208*u*w5*x*y - 
     0.06359410612179822*lnbara*u*w5*x*y - 0.2133826221782255*v*w5*x*y - 
     0.0583092465218906*lnbara*v*w5*x*y + 0.47407842963740254*w6*x*y + 
     0.14311605655840995*lnbara*w6*x*y - 0.9004176968697725*x2*y + 
     0.3212835093689003*u*x2*y - 0.1223579533330431*u2*x2*y + 
     0.03702532275275472*u3*x2*y - 0.007317810733171885*u4*x2*y + 
     0.0006870207501448052*u5*x2*y - 7.740580989744235*v*x2*y - 
     3.007965370948085*lnbara*v*x2*y - 8.34831676538791*u*v*x2*y - 
     2.570302308603461*lnbara*u*v*x2*y - 4.3167789899540825*u2*v*x2*y - 
     1.303884138860003*lnbara*u2*v*x2*y - 1.0998046234447676*u3*v*x2*y - 
     0.306972893835514*lnbara*u3*v*x2*y - 0.10954859095754421*u4*v*x2*y - 
     0.029980593819149505*lnbara*u4*v*x2*y + 0.8190067945965689*v2*x2*y + 
     0.6318629132721324*lnbara*v2*x2*y + 16.737595251903855*u*v2*x2*y + 
     5.219893223262108*lnbara*u*v2*x2*y + 1.43981530141774*u2*v2*x2*y + 
     0.3813842687208064*lnbara*u2*v2*x2*y + 0.8558867409043719*u3*v2*x2*
      y + 0.24521275140423673*lnbara*u3*v2*x2*y - 
     5.574896354901597*v3*x2*y - 1.909904783156106*lnbara*v3*x2*y - 
     6.110264042573513*u*v3*x2*y - 1.8612066536179699*lnbara*u*v3*x2*y - 
     0.8578197999104673*u2*v3*x2*y - 0.24023814331107549*lnbara*u2*v3*
      x2*y + 0.8738097966923795*v4*x2*y + 0.34300365970256935*lnbara*v4*
      x2*y + 1.632450766309338*u*v4*x2*y + 0.47753166783295525*lnbara*u*
      v4*x2*y - 0.2597697450829062*v5*x2*y - 0.08515023996893491*lnbara*
      v5*x2*y + 8.371127296661705*w*x2*y + 3.007965370948085*lnbara*w*x2*
      y + 8.142878738630774*u*w*x2*y + 2.570302308603461*lnbara*u*w*x2*y + 
     4.372397515146079*u2*w*x2*y + 1.303884138860003*lnbara*u2*w*x2*y + 
     1.089630311725629*u3*w*x2*y + 0.306972893835514*lnbara*u3*w*x2*y + 
     0.11045275181230746*u4*w*x2*y + 0.029980593819149505*lnbara*u4*w*x2*
      y + 10.777349982686436*v*w*x2*y + 3.494158013231508*lnbara*v*w*x2*y - 
     14.680040084881753*u*v*w*x2*y - 4.586055804009709*lnbara*u*v*w*x2*y + 
     2.978128785284849*u2*v*w*x2*y + 0.9528593416680298*lnbara*u2*v*w*x2*
      y - 0.7405676776348092*u3*v*w*x2*y - 0.21821356263065994*lnbara*u3*v*
      w*x2*y + 0.5638297481368869*v2*w*x2*y + 0.18788462045702756*lnbara*
      v2*w*x2*y + 0.008389015172307996*u*v2*w*x2*y - 
     0.057042047293007624*lnbara*u*v2*w*x2*y - 0.2581203920240232*u2*v2*w*
      x2*y - 0.07638602271158683*lnbara*u2*v2*w*x2*y + 
     5.676870448679409*v3*w*x2*y + 1.684726260416403*lnbara*v3*w*x2*y - 
     1.182692665484111*u*v3*w*x2*y - 0.32670867639523105*lnbara*u*v3*w*x2*
      y - 0.566190372464217*v4*w*x2*y - 0.1702185650157698*lnbara*v4*w*x2*
      y - 11.93413855963392*w2*x2*y - 4.1260209265036405*lnbara*w2*x2*y - 
     1.9788978869332372*u*w2*x2*y - 0.6338374192524003*lnbara*u*w2*x2*y - 
     4.430910486593282*u2*w2*x2*y - 1.3342436103888362*lnbara*u2*w2*x2*
      y - 0.1142476902831892*u3*w2*x2*y - 0.0269991887735768*lnbara*u3*
      w2*x2*y - 0.6330043203094217*v*w2*x2*y - 0.3326899352140709*lnbara*
      v*w2*x2*y + 3.7124326039755955*u*v*w2*x2*y + 
     1.229224903218318*lnbara*u*v*w2*x2*y + 0.3880630632568426*u2*v*w2*
      x2*y + 0.0969009664171316*lnbara*u2*v*w2*x2*y - 
     6.896344257474369*v2*w2*x2*y - 2.015976321936135*lnbara*v2*w2*x2*
      y + 1.8892318082935522*u*v2*w2*x2*y + 0.5393619159455898*lnbara*u*
      v2*w2*x2*y + 0.030715686889802068*v3*w2*x2*y - 
     0.0026175022546954534*lnbara*v3*w2*x2*y + 5.769076336816564*w3*x2*
      y + 2.0547100979131496*lnbara*w3*x2*y + 2.371701525434619*u*w3*x2*
      y + 0.6890237976926594*lnbara*u*w3*x2*y + 0.729201859518042*u2*w3*
      x2*y + 0.21972319960553072*lnbara*u2*w3*x2*y + 
     3.1903919237729643*v*w3*x2*y + 0.9222524033868401*lnbara*v*w3*x2*y - 
     2.203724440304821*u*v*w3*x2*y - 0.6502064428550666*lnbara*u*v*w3*x2*
      y + 0.501248699467762*v2*w3*x2*y + 0.1627718847199586*lnbara*v2*w3*
      x2*y - 2.872858759697617*w4*x2*y - 0.9340060015696775*lnbara*w4*x2*
      y - 0.13345592306181395*u*w4*x2*y - 0.03997846452824747*lnbara*u*w4*
      x2*y - 0.06439562415041984*v*w4*x2*y - 0.022967627645177263*lnbara*v*
      w4*x2*y + 0.3612787107357604*w5*x2*y + 0.11818205016461883*lnbara*
      w5*x2*y + 0.6858555690183518*x3*y - 0.15963636872636416*u*x3*y + 
     0.03824563394210313*u2*x3*y - 0.006605505610793891*u3*x3*y + 
     0.0005687770966253271*u4*x3*y + 14.150094597275556*v*x3*y + 
     4.8057115258275305*lnbara*v*x3*y + 3.7558333567491418*u*v*x3*y + 
     1.0737089522063064*lnbara*u*v*x3*y + 3.5021103597603167*u2*v*x3*y + 
     1.041566396111375*lnbara*u2*v*x3*y + 0.1081140594632303*u3*v*x3*y + 
     0.025365208983699924*lnbara*u3*v*x3*y - 5.574896354901597*v2*x3*y - 
     1.909904783156106*lnbara*v2*x3*y - 6.110264042573513*u*v2*x3*y - 
     1.8612066536179699*lnbara*u*v2*x3*y - 0.8578197999104673*u2*v2*x3*
      y - 0.24023814331107549*lnbara*u2*v2*x3*y + 
     5.6819090881671865*v3*x3*y + 1.7522332805688188*lnbara*v3*x3*y + 
     0.6056692941957744*u*v3*x3*y + 0.1862563479008138*lnbara*u*v3*x3*y - 
     0.7465338393563911*v4*x3*y - 0.22949316645754256*lnbara*v4*x3*y - 
     14.456656314267676*w*x3*y - 4.8057115258275305*lnbara*w*x3*y - 
     3.6907580059131853*u*w*x3*y - 1.0737089522063064*lnbara*u*w*x3*y - 
     3.5123911436649746*u2*w*x3*y - 1.041566396111375*lnbara*u2*w*x3*y - 
     0.10728353612006211*u3*w*x3*y - 0.025365208983699924*lnbara*u3*w*x3*
      y - 0.09200933391270437*v*w*x3*y - 0.14480531475704334*lnbara*v*w*x3*
      y + 3.724170161382556*u*v*w*x3*y + 1.1721828559253105*lnbara*u*v*w*x3*
      y + 0.1296884569931269*u2*v*w*x3*y + 0.020514943705544776*lnbara*u2*
      v*w*x3*y - 6.183783910974188*v2*w*x3*y - 1.7887689679030814*lnbara*
      v2*w*x3*y + 1.617693856062018*u*v2*w*x3*y + 
     0.4746900012067356*lnbara*u*v2*w*x3*y + 0.032891805121808636*v3*w*x3*
      y - 0.002730352711332793*lnbara*v3*w*x3*y + 
     5.769076336816564*w2*x3*y + 2.0547100979131496*lnbara*w2*x3*y + 
     2.371701525434619*u*w2*x3*y + 0.6890237976926594*lnbara*u*w2*x3*y + 
     0.729201859518042*u2*w2*x3*y + 0.21972319960553072*lnbara*u2*w2*x3*
      y + 3.8502028793420484*v*w2*x3*y + 1.1170305738043862*lnbara*v*w2*
      x3*y - 2.127445819261271*u*v*w2*x3*y - 0.6316147687241351*lnbara*u*v*
      w2*x3*y + 0.2951846833599234*v2*w2*x3*y + 
     0.10231487175528582*lnbara*v2*w2*x3*y - 3.369820769025423*w3*x3*y - 
     1.0804948864701236*lnbara*w3*x3*y - 0.09448932656403045*u*w3*x3*y - 
     0.029331580383414237*lnbara*u*w3*x3*y + 0.04710638371571019*v*w3*x3*
      y + 0.006233587961624442*lnbara*v*w3*x3*y + 
     0.3734562378603602*w4*x3*y + 0.1236750594519651*lnbara*w4*x3*y - 
     0.386061737342387*x4*y + 0.056260781045097456*u*x4*y - 
     0.007692523535941066*u2*x4*y + 0.0005789219466621661*u3*x4*y - 
     7.96562555520799*v*x4*y - 2.6276787546704745*lnbara*v*x4*y - 
     2.0218463214313056*u*v*x4*y - 0.5668452509261017*lnbara*u*v*x4*y - 
     0.5891572387478903*u2*v*x4*y - 0.17202385787388583*lnbara*u2*v*x4*
      y + 0.8738097966923795*v2*x4*y + 0.34300365970256935*lnbara*v2*x4*
      y + 1.632450766309338*u*v2*x4*y + 0.47753166783295525*lnbara*u*v2*
      x4*y - 0.7465338393563911*v3*x4*y - 0.22949316645754256*lnbara*v3*
      x4*y + 8.072482666125286*w*x4*y + 2.6276787546704745*lnbara*w*x4*y + 
     2.0086611123781046*u*w*x4*y + 0.5668452509261017*lnbara*u*w*x4*y + 
     0.590076818257424*u2*w*x4*y + 0.17202385787388583*lnbara*u2*w*x4*y + 
     1.9788571739028187*v*w*x4*y + 0.5910023418671081*lnbara*v*w*x4*y - 
     1.497715134559773*u*v*w*x4*y - 0.43755320330470776*lnbara*u*v*w*x4*y + 
     0.4680668096248443*v2*w*x4*y + 0.13910950612836231*lnbara*v2*w*x4*
      y - 2.872858759697617*w2*x4*y - 0.9340060015696775*lnbara*w2*x4*y - 
     0.13345592306181395*u*w2*x4*y - 0.03997846452824747*lnbara*u*w2*x4*
      y - 0.09314373993962562*v*w2*x4*y - 0.033291399122784844*lnbara*v*w2*
      x4*y + 0.3734562378603602*w3*x4*y + 0.1236750594519651*lnbara*w3*
      x4*y + 0.1475870262830398*x5*y - 0.012249129070819405*u*x5*y + 
     0.0007299175618735945*u2*x5*y + 3.2252148129088143*v*x5*y + 
     1.0069202481453856*lnbara*v*x5*y + 0.23526924787765785*u*v*x5*y + 
     0.06359410612179822*lnbara*u*v*x5*y - 0.2597697450829062*v2*x5*y - 
     0.08515023996893491*lnbara*v2*x5*y - 3.248331013993099*w*x5*y - 
     1.0069202481453856*lnbara*w*x5*y - 0.23401253231882208*u*w*x5*y - 
     0.06359410612179822*lnbara*u*w*x5*y - 0.09961325470566673*v*w*x5*y - 
     0.03303181019568392*lnbara*v*w*x5*y + 0.3612787107357604*w2*x5*y + 
     0.11818205016461883*lnbara*w2*x5*y - 0.0339914082143444*x6*y + 
     0.001228493995858818*u*x6*y - 0.4717695787130639*v*x6*y - 
     0.14311605655840995*lnbara*v*x6*y + 0.47407842963740254*w*x6*y + 
     0.14311605655840995*lnbara*w*x6*y + 0.0035512018991495737*x7*y + 
     1.8468559937823905*y2 - 0.6781895601550072*u*y2 + 
     0.3470714951043783*u2*y2 - 0.15843013830258718*u3*y2 + 
     0.05343363480120978*u4*y2 - 0.011336291067348674*u5*y2 + 
     0.0011180783913151146*u6*y2 - 0.9004176968697725*v*y2 + 
     0.29918371952782097*u*v*y2 - 0.10942628765049751*u2*v*y2 + 
     0.03238162914677262*u3*v*y2 - 0.00631630321372675*u4*v*y2 + 
     0.0005880488320726199*u5*v*y2 + 0.520113697995963*v2*y2 - 
     0.1223579533330431*u*v2*y2 + 0.029164662906062597*u2*v2*y2 - 
     0.0050083201180713645*u3*v2*y2 + 0.00042934540544391214*u4*v2*y2 - 
     0.251564804320496*v3*y2 + 0.03824563394210313*u*v3*y2 - 
     0.005291090684199614*u2*v3*y2 + 0.00039954507731219854*u3*v3*y2 + 
     0.08755029317854246*v4*y2 - 0.007692523535941066*u*v4*y2 + 
     0.0004679400724052549*u2*v4*y2 - 0.018929896217160017*v5*y2 + 
     0.0007299175618735945*u*v5*y2 + 0.0018904202297008643*v6*y2 - 
     0.9004176968697725*w*y2 + 0.29918371952782097*u*w*y2 - 
     0.10942628765049751*u2*w*y2 + 0.03238162914677262*u3*w*y2 - 
     0.00631630321372675*u4*w*y2 + 0.0005880488320726199*u5*w*y2 + 
     0.45246456501551086*v*w*y2 - 0.1076866392526224*u*v*w*y2 + 
     0.02581697089389268*u2*v*w*y2 - 0.004447305079644873*u3*v*w*y2 + 
     0.00038194899426036925*u4*v*w*y2 - 0.1974796353404678*v2*w*y2 + 
     0.03069034292530752*u*v2*w*y2 - 0.0042936520074251285*u2*v2*w*y2 + 
     0.00032622599069967407*u3*v2*w*y2 + 0.06395157977258972*v3*w*y2 - 
     0.005780394040453321*u*v3*w*y2 + 0.0003568540057622896*u2*v3*w*y2 - 
     0.013136621448171954*v4*w*y2 + 0.0005230003987145713*u*v4*w*y2 + 
     0.001263472670057093*v5*w*y2 + 0.520113697995963*w2*y2 - 
     0.1223579533330431*u*w2*y2 + 0.029164662906062597*u2*w2*y2 - 
     0.0050083201180713645*u3*w2*y2 + 0.00042934540544391214*u4*w2*y2 - 
     0.1974796353404678*v*w2*y2 + 0.03069034292530752*u*v*w2*y2 - 
     0.0042936520074251285*u2*v*w2*y2 + 0.00032622599069967407*u3*v*w2*
      y2 + 0.05802930853773854*v2*w2*y2 - 0.005287248454508436*u*v2*w2*
      y2 + 0.00032784931577668774*u2*v2*w2*y2 - 
     0.01113298574577242*v3*w2*y2 + 0.0004489814737672761*u*v3*w2*y2 + 
     0.001019248467243472*v4*w2*y2 - 0.251564804320496*w3*y2 + 
     0.03824563394210313*u*w3*y2 - 0.005291090684199614*u2*w3*y2 + 
     0.00039954507731219854*u3*w3*y2 + 0.06395157977258972*v*w3*y2 - 
     0.005780394040453321*u*v*w3*y2 + 0.0003568540057622896*u2*v*w3*y2 - 
     0.01113298574577242*v2*w3*y2 + 0.0004489814737672761*u*v2*w3*y2 + 
     0.0009524091176864817*v3*w3*y2 + 0.08755029317854246*w4*y2 - 
     0.007692523535941066*u*w4*y2 + 0.0004679400724052549*u2*w4*y2 - 
     0.013136621448171954*v*w4*y2 + 0.0005230003987145713*u*v*w4*y2 + 
     0.001019248467243472*v2*w4*y2 - 0.018929896217160017*w5*y2 + 
     0.0007299175618735945*u*w5*y2 + 0.001263472670057093*v*w5*y2 + 
     0.0018904202297008643*w6*y2 - 0.9004176968697725*x*y2 + 
     0.29918371952782097*u*x*y2 - 0.10942628765049751*u2*x*y2 + 
     0.03238162914677262*u3*x*y2 - 0.00631630321372675*u4*x*y2 + 
     0.0005880488320726199*u5*x*y2 - 4.647008665663399*v*x*y2 - 
     1.7090990282110639*lnbara*v*x*y2 + 0.9708296446662518*u*v*x*y2 + 
     0.4284924263303262*lnbara*u*v*x*y2 + 1.184519706140848*u2*v*x*y2 + 
     0.329646645071357*lnbara*u2*v*x*y2 + 0.3742822830969903*u3*v*x*y2 + 
     0.10776709570789808*lnbara*u3*v*x*y2 + 0.04056836531530246*u4*v*x*
      y2 + 0.010614825463755726*lnbara*u4*v*x*y2 - 
     0.20624980265509574*v2*x*y2 + 0.06378679623852483*lnbara*v2*x*y2 - 
     2.5341893643030327*u*v2*x*y2 - 0.8006228366412104*lnbara*u*v2*x*y2 - 
     1.1971008100465077*u2*v2*x*y2 - 0.3402159951898016*lnbara*u2*v2*x*
      y2 - 0.17277027032117476*u3*v2*x*y2 - 0.04766748743684271*lnbara*
      u3*v2*x*y2 - 2.770275796275451*v3*x*y2 - 
     0.8710102224533081*lnbara*v3*x*y2 + 2.2733271985354553*u*v3*x*y2 + 
     0.6947964005893216*lnbara*u*v3*x*y2 + 0.15265057835965268*u2*v3*x*
      y2 + 0.04014533812336501*lnbara*u2*v3*x*y2 + 
     0.305005920641606*v4*x*y2 + 0.10788365521657699*lnbara*v4*x*y2 - 
     0.3602833223629141*u*v4*x*y2 - 0.10812164650482199*lnbara*u*v4*x*
      y2 - 0.11272620964583531*v5*x*y2 - 0.03339748565253056*lnbara*v5*x*
      y2 + 5.299654762421948*w*x*y2 + 1.7090990282110639*lnbara*w*x*y2 - 
     1.1643976508622838*u*w*x*y2 - 0.4284924263303262*lnbara*u*w*x*y2 - 
     1.1346769147760087*u2*w*x*y2 - 0.329646645071357*lnbara*u2*w*x*y2 - 
     0.3831429064834633*u3*w*x*y2 - 0.10776709570789808*lnbara*u3*w*x*
      y2 - 0.03979465732217202*u4*w*x*y2 - 0.010614825463755726*lnbara*u4*
      w*x*y2 - 0.1113550912072001*v*w*x*y2 + 0.023341335158146424*u*v*w*x*
      y2 - 0.0036412369193938783*u2*v*w*x*y2 + 0.0002913721656218036*u3*v*
      w*x*y2 + 14.046191130880398*v2*w*x*y2 + 4.27806313688087*lnbara*v2*
      w*x*y2 - 1.0675924425094043*u*v2*w*x*y2 - 0.36692372038764914*lnbara*
      u*v2*w*x*y2 + 0.7426221290311992*u2*v2*w*x*y2 + 
     0.21620742554342368*lnbara*u2*v2*w*x*y2 - 3.431997185570922*v3*w*x*
      y2 - 1.0593983869002586*lnbara*v3*w*x*y2 - 
     0.3063351018757037*u*v3*w*x*y2 - 0.0810097794355468*lnbara*u*v3*w*x*
      y2 + 1.1422301498147045*v4*w*x*y2 + 0.33097644191339387*lnbara*v4*w*
      x*y2 - 0.044978574732269516*w2*x*y2 - 0.06378679623852483*lnbara*w2*
      x*y2 + 2.5864306854801558*u*w2*x*y2 + 0.8006228366412104*lnbara*u*
      w2*x*y2 + 1.1889883290028347*u2*w2*x*y2 + 
     0.3402159951898016*lnbara*u2*w2*x*y2 + 0.17341769222311584*u3*w2*x*
      y2 + 0.04766748743684271*lnbara*u3*w2*x*y2 - 
     13.984210663495318*v*w2*x*y2 - 4.27806313688087*lnbara*v*w2*x*y2 + 
     1.059736505268818*u*v*w2*x*y2 + 0.36692372038764914*lnbara*u*v*w2*x*
      y2 - 0.7420694165018765*u2*v*w2*x*y2 - 0.21620742554342368*lnbara*
      u2*v*w2*x*y2 - 0.005287248454508436*v2*w2*x*y2 + 
     0.0003034691554580678*u*v2*w2*x*y2 - 1.0830627587062924*v3*w2*x*
      y2 - 0.2997685920100648*lnbara*v3*w2*x*y2 + 
     2.8467947898195387*w3*x*y2 + 0.8710102224533081*lnbara*w3*x*y2 - 
     2.2828692800346633*u*w3*x*y2 - 0.6947964005893216*lnbara*u*w3*x*y2 - 
     0.15198537243470955*u2*w3*x*y2 - 0.04014533812336501*lnbara*u2*w3*x*
      y2 + 3.420468192644727*v*w3*x*y2 + 1.0593983869002586*lnbara*v*w3*x*
      y2 + 0.3069925110386899*u*v*w3*x*y2 + 0.0810097794355468*lnbara*u*v*
      w3*x*y2 + 1.0839553144032048*v2*w3*x*y2 + 
     0.2997685920100648*lnbara*v2*w3*x*y2 - 0.32019128431906846*w4*x*
      y2 - 0.10788365521657699*lnbara*w4*x*y2 + 0.3611306117059756*u*w4*x*
      y2 + 0.10812164650482199*lnbara*u*w4*x*y2 - 
     1.1411946582446566*v*w4*x*y2 - 0.33097644191339387*lnbara*v*w4*x*
      y2 + 0.11415571194993857*w5*x*y2 + 0.03339748565253056*lnbara*w5*x*
      y2 + 0.520113697995963*x2*y2 - 0.1223579533330431*u*x2*y2 + 
     0.029164662906062597*u2*x2*y2 - 0.0050083201180713645*u3*x2*y2 + 
     0.00042934540544391214*u4*x2*y2 - 0.20624980265509574*v*x2*y2 + 
     0.06378679623852483*lnbara*v*x2*y2 - 2.5341893643030327*u*v*x2*y2 - 
     0.8006228366412104*lnbara*u*v*x2*y2 - 1.1971008100465077*u2*v*x2*
      y2 - 0.3402159951898016*lnbara*u2*v*x2*y2 - 
     0.17277027032117476*u3*v*x2*y2 - 0.04766748743684271*lnbara*u3*v*x2*
      y2 + 6.563551498721121*v2*x2*y2 + 1.9840887474720377*lnbara*v2*x2*
      y2 + 0.14499255922663437*u*v2*x2*y2 + 0.03938451616400227*lnbara*u*
      v2*x2*y2 + 0.8278420296899973*u2*v2*x2*y2 + 
     0.238589191178055*lnbara*u2*v2*x2*y2 - 1.3869981182997964*v3*x2*
      y2 - 0.4218277765499367*lnbara*v3*x2*y2 - 
     0.4779542480040674*u*v3*x2*y2 - 0.13759704980282306*lnbara*u*v3*x2*
      y2 + 0.5651566216606796*v4*x2*y2 + 0.16240221528459253*lnbara*v4*
      x2*y2 - 0.044978574732269516*w*x2*y2 - 0.06378679623852483*lnbara*w*
      x2*y2 + 2.5864306854801558*u*w*x2*y2 + 0.8006228366412104*lnbara*u*
      w*x2*y2 + 1.1889883290028347*u2*w*x2*y2 + 
     0.3402159951898016*lnbara*u2*w*x2*y2 + 0.17341769222311584*u3*w*x2*
      y2 + 0.04766748743684271*lnbara*u3*w*x2*y2 - 
     8.691639367114691*v*w*x2*y2 - 2.6585984883283214*lnbara*v*w*x2*y2 + 
     4.260756983735999*u*v*w*x2*y2 + 1.2888307504910574*lnbara*u*v*w*x2*
      y2 - 0.6381316954939846*u2*v*w*x2*y2 - 0.19013921429911454*lnbara*
      u2*v*w*x2*y2 - 1.4568456155246001*v2*w*x2*y2 - 
     0.48533250050556614*lnbara*v2*w*x2*y2 - 0.5429343918861453*u*v2*w*
      x2*y2 - 0.1557593910339726*lnbara*u*v2*w*x2*y2 - 
     0.7046868197766329*v3*w*x2*y2 - 0.18518041263910326*lnbara*v3*w*x2*
      y2 + 2.216432092777231*w2*x2*y2 + 0.6745097408562837*lnbara*w2*x2*
      y2 - 4.417728806752933*u*w2*x2*y2 - 1.3282152666550597*lnbara*u*w2*
      x2*y2 - 0.18884186308954337*u2*w2*x2*y2 - 
     0.048449976878940454*lnbara*u2*w2*x2*y2 + 2.499274797319209*v*w2*
      x2*y2 + 0.8372694295660547*lnbara*v*w2*x2*y2 + 
     0.36293415511749816*u*v*w2*x2*y2 + 0.09272663386743893*lnbara*u*v*w2*
      x2*y2 + 1.4101267527346482*v2*w2*x2*y2 + 
     0.39576075754209206*lnbara*v2*w2*x2*y2 + 0.32522417184198954*w3*x2*
      y2 + 0.06989084748944815*lnbara*w3*x2*y2 + 
     0.6591783435173679*u*w3*x2*y2 + 0.20062980696935676*lnbara*u*w3*x2*
      y2 - 1.3871873506084558*v*w3*x2*y2 - 0.4071365920310302*lnbara*v*
      w3*x2*y2 + 0.11854588979479715*w4*x2*y2 + 
     0.03415403184344883*lnbara*w4*x2*y2 - 0.251564804320496*x3*y2 + 
     0.03824563394210313*u*x3*y2 - 0.005291090684199614*u2*x3*y2 + 
     0.00039954507731219854*u3*x3*y2 - 2.770275796275451*v*x3*y2 - 
     0.8710102224533081*lnbara*v*x3*y2 + 2.2733271985354553*u*v*x3*y2 + 
     0.6947964005893216*lnbara*u*v*x3*y2 + 0.15265057835965268*u2*v*x3*
      y2 + 0.04014533812336501*lnbara*u2*v*x3*y2 - 
     1.3869981182997964*v2*x3*y2 - 0.4218277765499367*lnbara*v2*x3*y2 - 
     0.4779542480040674*u*v2*x3*y2 - 0.13759704980282306*lnbara*u*v2*x3*
      y2 - 0.19054563887772494*v3*x3*y2 - 0.04854179569331324*lnbara*v3*
      x3*y2 + 2.8467947898195387*w*x3*y2 + 0.8710102224533081*lnbara*w*
      x3*y2 - 2.2828692800346633*u*w*x3*y2 - 0.6947964005893216*lnbara*u*
      w*x3*y2 - 0.15198537243470955*u2*w*x3*y2 - 
     0.04014533812336501*lnbara*u2*w*x3*y2 + 1.046392947967706*v*w*x3*
      y2 + 0.35193692906048857*lnbara*v*w*x3*y2 - 
     0.18025531671901043*u*v*w*x3*y2 - 0.06303275716653367*lnbara*u*v*w*x3*
      y2 + 1.492463738628857*v2*w*x3*y2 + 0.4272793464214101*lnbara*v2*w*
      x3*y2 + 0.32522417184198954*w2*x3*y2 + 0.06989084748944815*lnbara*
      w2*x3*y2 + 0.6591783435173679*u*w2*x3*y2 + 
     0.20062980696935676*lnbara*u*w2*x3*y2 - 1.4916718912285227*v*w2*x3*
      y2 - 0.432716958385381*lnbara*v*w2*x3*y2 + 
     0.19122371263757587*w3*x3*y2 + 0.05397940765728412*lnbara*w3*x3*
      y2 + 0.08755029317854246*x4*y2 - 0.007692523535941066*u*x4*y2 + 
     0.0004679400724052549*u2*x4*y2 + 0.305005920641606*v*x4*y2 + 
     0.10788365521657699*lnbara*v*x4*y2 - 0.3602833223629141*u*v*x4*y2 - 
     0.10812164650482199*lnbara*u*v*x4*y2 + 0.5651566216606796*v2*x4*
      y2 + 0.16240221528459253*lnbara*v2*x4*y2 - 
     0.32019128431906846*w*x4*y2 - 0.10788365521657699*lnbara*w*x4*y2 + 
     0.3611306117059756*u*w*x4*y2 + 0.10812164650482199*lnbara*u*w*x4*
      y2 - 0.6823722786029639*v*w*x4*y2 - 0.19655624712804137*lnbara*v*w*
      x4*y2 + 0.11854588979479715*w2*x4*y2 + 0.03415403184344883*lnbara*
      w2*x4*y2 - 0.018929896217160017*x5*y2 + 0.0007299175618735945*u*
      x5*y2 - 0.11272620964583531*v*x5*y2 - 0.03339748565253056*lnbara*v*
      x5*y2 + 0.11415571194993857*w*x5*y2 + 0.03339748565253056*lnbara*w*
      x5*y2 + 0.0018904202297008643*x6*y2 - 2.2670918716867487*y3 + 
     0.5000233108183318*u*y3 - 0.15843013830258718*u2*y3 + 
     0.044009786371943985*u3*y3 - 0.00830686713692306*u4*y3 + 
     0.0007585417970558358*u5*y3 + 0.6858555690183518*v*y3 - 
     0.14397058356518277*u*v*y3 + 0.03238162914677262*u2*v*y3 - 
     0.0053911589121258706*u3*v*y3 + 0.0004539037516950064*u4*v*y3 - 
     0.251564804320496*v2*y3 + 0.03702532275275472*u*v2*y3 - 
     0.0050083201180713645*u2*v2*y3 + 0.0003728851239061164*u3*v2*y3 + 
     0.0752590109961792*v3*y3 - 0.006605505610793891*u*v3*y3 + 
     0.00039954507731219854*u2*v3*y3 - 0.014812198869903107*v4*y3 + 
     0.0005789219466621661*u*v4*y3 + 0.001388160767640295*v5*y3 + 
     0.6858555690183518*w*y3 - 0.14397058356518277*u*w*y3 + 
     0.03238162914677262*u2*w*y3 - 0.0053911589121258706*u3*w*y3 + 
     0.0004539037516950064*u4*w*y3 - 0.2298476323443832*v*w*y3 + 
     0.0339939779841863*u*v*w*y3 - 0.004611648226787489*u2*v*w*y3 + 
     0.0003439536028536974*u3*v*w*y3 + 0.06395157977258972*v2*w*y3 - 
     0.005673921854933901*u*v2*w*y3 + 0.00034533361905439837*u2*v2*w*
      y3 - 0.011927105098099399*v3*w*y3 + 0.00047310410784989136*u*v3*w*
      y3 + 0.0010740897315456516*v4*w*y3 - 0.251564804320496*w2*y3 + 
     0.03702532275275472*u*w2*y3 - 0.0050083201180713645*u2*w2*y3 + 
     0.0003728851239061164*u3*w2*y3 + 0.06395157977258972*v*w2*y3 - 
     0.005673921854933901*u*v*w2*y3 + 0.00034533361905439837*u2*v*w2*
      y3 - 0.01113298574577242*v2*w2*y3 + 0.00044357422314508923*u*v2*
      w2*y3 + 0.0009524091176864817*v3*w2*y3 + 
     0.0752590109961792*w3*y3 - 0.006605505610793891*u*w3*y3 + 
     0.00039954507731219854*u2*w3*y3 - 0.011927105098099399*v*w3*y3 + 
     0.00047310410784989136*u*v*w3*y3 + 0.0009524091176864817*v2*w3*y3 - 
     0.014812198869903107*w4*y3 + 0.0005789219466621661*u*w4*y3 + 
     0.0010740897315456516*v*w4*y3 + 0.001388160767640295*w5*y3 + 
     0.6858555690183518*x*y3 - 0.14397058356518277*u*x*y3 + 
     0.03238162914677262*u2*x*y3 - 0.0053911589121258706*u3*x*y3 + 
     0.0004539037516950064*u4*x*y3 + 6.221638137007376*v*x*y3 + 
     2.0509188338532764*lnbara*v*x*y3 + 1.6028488135384624*u*v*x*y3 + 
     0.4192402591437163*lnbara*u*v*x*y3 + 0.43459981748405074*u2*v*x*y3 + 
     0.12608980116426613*lnbara*u2*v*x*y3 + 0.05131419753175176*u3*v*x*
      y3 + 0.013481184061483282*lnbara*u3*v*x*y3 - 
     2.339296200037759*v2*x*y3 - 0.7884496760703545*lnbara*v2*x*y3 - 
     0.9468503021957163*u*v2*x*y3 - 0.26779586684578455*lnbara*u*v2*x*
      y3 - 0.16757153566204508*u2*v2*x*y3 - 0.046821383490061465*lnbara*
      u2*v2*x*y3 + 2.402267061883944*v3*x*y3 + 
     0.7222688114635434*lnbara*v3*x*y3 + 0.10166794959991486*u*v3*x*y3 + 
     0.025545381796987655*lnbara*u*v3*x*y3 - 0.3237542215275384*v4*x*y3 - 
     0.09733306860121325*lnbara*v4*x*y3 - 6.543865639160677*w*x*y3 - 
     2.0509188338532764*lnbara*w*x*y3 - 1.5432862100326823*u*w*x*y3 - 
     0.4192402591437163*lnbara*u*w*x*y3 - 0.44329609772338113*u2*w*x*y3 - 
     0.12608980116426613*lnbara*u2*w*x*y3 - 0.050643992483851874*u3*w*x*
      y3 - 0.013481184061483282*lnbara*u3*w*x*y3 + 
     0.03580809539433375*v*w*x*y3 - 0.004288309525390823*u*v*w*x*y3 + 
     0.0002913721656218036*u2*v*w*x*y3 - 4.2497906793289655*v2*w*x*y3 - 
     1.229468487533898*lnbara*v2*w*x*y3 + 0.4686008650126944*u*v2*w*x*
      y3 + 0.14200807750295125*lnbara*u*v2*w*x*y3 + 
     0.21733303969258025*v3*w*x*y3 + 0.06469446777014416*lnbara*v3*w*x*
      y3 + 2.417035504771195*w2*x*y3 + 0.7884496760703545*lnbara*w2*x*
      y3 + 0.9375779785463527*u*w2*x*y3 + 0.26779586684578455*lnbara*u*w2*
      x*y3 + 0.16819984993563145*u2*w2*x*y3 + 0.046821383490061465*lnbara*
      u2*w2*x*y3 + 4.23815521421725*v*w2*x*y3 + 
     1.229468487533898*lnbara*v*w2*x*y3 - 0.46795439814148526*u*v*w2*x*
      y3 - 0.14200807750295125*lnbara*u*v*w2*x*y3 + 
     0.0004489814737672761*v2*w2*x*y3 - 2.4157548822081845*w3*x*y3 - 
     0.7222688114635434*lnbara*w3*x*y3 - 0.10092543442474523*u*w3*x*y3 - 
     0.025545381796987655*lnbara*u*w3*x*y3 - 0.21637657479006892*v*w3*x*
      y3 - 0.06469446777014416*lnbara*v*w3*x*y3 + 
     0.3249190900404754*w4*x*y3 + 0.09733306860121325*lnbara*w4*x*y3 - 
     0.251564804320496*x2*y3 + 0.03702532275275472*u*x2*y3 - 
     0.0050083201180713645*u2*x2*y3 + 0.0003728851239061164*u3*x2*y3 - 
     2.339296200037759*v*x2*y3 - 0.7884496760703545*lnbara*v*x2*y3 - 
     0.9468503021957163*u*v*x2*y3 - 0.26779586684578455*lnbara*u*v*x2*
      y3 - 0.16757153566204508*u2*v*x2*y3 - 0.046821383490061465*lnbara*
      u2*v*x2*y3 - 0.6788052052736328*v2*x2*y3 - 
     0.16864345026577507*lnbara*v2*x2*y3 + 0.717639217730022*u*v2*x2*
      y3 + 0.20684606078825712*lnbara*u*v2*x2*y3 - 
     0.20393804085263473*v3*x2*y3 - 0.06221919382932583*lnbara*v3*x2*
      y3 + 2.417035504771195*w*x2*y3 + 0.7884496760703545*lnbara*w*x2*
      y3 + 0.9375779785463527*u*w*x2*y3 + 0.26779586684578455*lnbara*u*w*
      x2*y3 + 0.16819984993563145*u2*w*x2*y3 + 
     0.046821383490061465*lnbara*u2*w*x2*y3 + 2.9845483465315206*v*w*x2*
      y3 + 0.873539503307877*lnbara*v*w*x2*y3 - 0.7548466365170377*u*v*w*
      x2*y3 - 0.21880709774223123*lnbara*u*v*w*x2*y3 + 
     0.029077792864669175*v2*w*x2*y3 + 0.011972114859594282*lnbara*v2*w*
      x2*y3 - 2.321676668163961*w2*x2*y3 - 0.704896053042102*lnbara*w2*
      x2*y3 + 0.038158053864272005*u*w2*x2*y3 + 
     0.011961036953974104*lnbara*u*w2*x2*y3 - 0.10578881199195397*v*w2*
      x2*y3 - 0.04001243623425464*lnbara*v*w2*x2*y3 + 
     0.28220067646190156*w3*x2*y3 + 0.09025951520398619*lnbara*w3*x2*
      y3 + 0.0752590109961792*x3*y3 - 0.006605505610793891*u*x3*y3 + 
     0.00039954507731219854*u2*x3*y3 + 2.402267061883944*v*x3*y3 + 
     0.7222688114635434*lnbara*v*x3*y3 + 0.10166794959991486*u*v*x3*y3 + 
     0.025545381796987655*lnbara*u*v*x3*y3 - 0.20393804085263473*v2*x3*
      y3 - 0.06221919382932583*lnbara*v2*x3*y3 - 
     2.4157548822081845*w*x3*y3 - 0.7222688114635434*lnbara*w*x3*y3 - 
     0.10092543442474523*u*w*x3*y3 - 0.025545381796987655*lnbara*u*w*x3*
      y3 - 0.07704797923780887*v*w*x3*y3 - 0.028040321374660362*lnbara*v*w*
      x3*y3 + 0.28220067646190156*w2*x3*y3 + 0.09025951520398619*lnbara*
      w2*x3*y3 - 0.014812198869903107*x4*y3 + 0.0005789219466621661*u*
      x4*y3 - 0.3237542215275384*v*x4*y3 - 0.09733306860121325*lnbara*v*
      x4*y3 + 0.3249190900404754*w*x4*y3 + 0.09733306860121325*lnbara*w*
      x4*y3 + 0.001388160767640295*x5*y3 + 2.0230034428855013*y4 - 
     0.27683765810185557*u*y4 + 0.05343363480120978*u2*y4 - 
     0.00830686713692306*u3*y4 + 0.0006746357862159641*u4*y4 - 
     0.386061737342387*v*y4 + 0.04988531635163183*u*v*y4 - 
     0.00631630321372675*u2*v*y4 + 0.0004539037516950064*u3*v*y4 + 
     0.08755029317854246*v2*y4 - 0.007317810733171885*u*v2*y4 + 
     0.00042934540544391214*u2*v2*y4 - 0.014812198869903107*v3*y4 + 
     0.0005687770966253271*u*v3*y4 + 0.0012633787101392036*v4*y4 - 
     0.386061737342387*w*y4 + 0.04988531635163183*u*w*y4 - 
     0.00631630321372675*u2*w*y4 + 0.0004539037516950064*u3*w*y4 + 
     0.08203160229248424*v*w*y4 - 0.006873088261070091*u*v*w*y4 + 
     0.00040392715269129916*u2*v*w*y4 - 0.013136621448171954*v2*w*y4 + 
     0.000507581743944034*u*v2*w*y4 + 0.0010740897315456516*v3*w*y4 + 
     0.08755029317854246*w2*y4 - 0.007317810733171885*u*w2*y4 + 
     0.00042934540544391214*u2*w2*y4 - 0.013136621448171954*v*w2*y4 + 
     0.000507581743944034*u*v*w2*y4 + 0.001019248467243472*v2*w2*y4 - 
     0.014812198869903107*w3*y4 + 0.0005687770966253271*u*w3*y4 + 
     0.0010740897315456516*v*w3*y4 + 0.0012633787101392036*w4*y4 - 
     0.386061737342387*x*y4 + 0.04988531635163183*u*x*y4 - 
     0.00631630321372675*u2*x*y4 + 0.0004539037516950064*u3*x*y4 - 
     2.7455486537995553*v*x*y4 - 0.8545495141055319*lnbara*v*x*y4 - 
     0.06972985855284485*u*v*x*y4 - 0.004869163308206694*lnbara*u*v*x*y4 + 
     0.02029024705940767*u2*v*x*y4 + 0.005188045152882472*lnbara*u2*v*x*
      y4 + 0.260373564991028*v2*x*y4 + 0.09277616733835867*lnbara*v2*x*
      y4 - 0.04301078480890973*u*v2*x*y4 - 0.013022653225572441*lnbara*u*
      v2*x*y4 - 0.2353317760308018*v3*x*y4 - 0.06767441939812754*lnbara*
      v3*x*y4 + 2.8587812294103165*w*x*y4 + 0.8545495141055319*lnbara*w*x*
      y4 + 0.05785733009437446*u*w*x*y4 + 0.004869163308206694*lnbara*u*w*x*
      y4 - 0.01953851722470816*u2*w*x*y4 - 0.005188045152882472*lnbara*u2*
      w*x*y4 - 0.007320613539132591*v*w*x*y4 + 0.0003820180212080083*u*v*w*
      x*y4 + 0.7480915272662646*v2*w*x*y4 + 0.21034811390666633*lnbara*v2*
      w*x*y4 - 0.2759336414712597*w2*x*y4 - 0.09277616733835867*lnbara*w2*
      x*y4 + 0.04382088794664975*u*w2*x*y4 + 0.013022653225572441*lnbara*u*
      w2*x*y4 - 0.747040617041446*v*w2*x*y4 - 0.21034811390666633*lnbara*
      v*w2*x*y4 + 0.23650678939377565*w3*x*y4 + 
     0.06767441939812754*lnbara*w3*x*y4 + 0.08755029317854246*x2*y4 - 
     0.007317810733171885*u*x2*y4 + 0.00042934540544391214*u2*x2*y4 + 
     0.260373564991028*v*x2*y4 + 0.09277616733835867*lnbara*v*x2*y4 - 
     0.04301078480890973*u*v*x2*y4 - 0.013022653225572441*lnbara*u*v*x2*
      y4 + 0.28165966173252016*v2*x2*y4 + 0.07779208068357109*lnbara*v2*
      x2*y4 - 0.2759336414712597*w*x2*y4 - 0.09277616733835867*lnbara*w*
      x2*y4 + 0.04382088794664975*u*w*x2*y4 + 0.013022653225572441*lnbara*
      u*w*x2*y4 - 0.4807266928857714*v*w*x2*y4 - 
     0.1356427800167511*lnbara*v*w*x2*y4 + 0.20047304487804693*w2*x2*
      y4 + 0.05785069933318*lnbara*w2*x2*y4 - 0.014812198869903107*x3*
      y4 + 0.0005687770966253271*u*x3*y4 - 0.2353317760308018*v*x3*y4 - 
     0.06767441939812754*lnbara*v*x3*y4 + 0.23650678939377565*w*x3*y4 + 
     0.06767441939812754*lnbara*w*x3*y4 + 0.0012633787101392036*x4*y4 - 
     1.2541858328141202*y5 + 0.10482088785932395*u*y5 - 
     0.011336291067348674*u2*y5 + 0.0007585417970558358*u3*y5 + 
     0.1475870262830398*v*y5 - 0.01075135761798518*u*v*y5 + 
     0.0005880488320726199*u2*v*y5 - 0.018929896217160017*v2*y5 + 
     0.0006870207501448052*u*v2*y5 + 0.001388160767640295*v3*y5 + 
     0.1475870262830398*w*y5 - 0.01075135761798518*u*w*y5 + 
     0.0005880488320726199*u2*w*y5 - 0.018000480690589078*v*w*y5 + 
     0.0006541741874990681*u*v*w*y5 + 0.001263472670057093*v2*w*y5 - 
     0.018929896217160017*w2*y5 + 0.0006870207501448052*u*w2*y5 + 
     0.001263472670057093*v*w2*y5 + 0.001388160767640295*w3*y5 + 
     0.1475870262830398*x*y5 - 0.01075135761798518*u*x*y5 + 
     0.0005880488320726199*u2*x*y5 + 0.9067948266532218*v*x*y5 + 
     0.26857270443316716*lnbara*v*x*y5 + 0.04008569352545048*u*v*x*y5 + 
     0.009316737180946303*lnbara*u*v*x*y5 - 0.06841032198704927*v2*x*y5 - 
     0.021403950235092197*lnbara*v2*x*y5 - 0.9314087991903407*w*x*y5 - 
     0.26857270443316716*lnbara*w*x*y5 - 0.038965576694794626*u*w*x*y5 - 
     0.009316737180946303*lnbara*u*w*x*y5 + 0.0007017792893363362*v*w*x*
      y5 + 0.06988272110288132*w2*x*y5 + 0.021403950235092197*lnbara*w2*x*
      y5 - 0.018929896217160017*x2*y5 + 0.0006870207501448052*u*x2*y5 - 
     0.06841032198704927*v*x2*y5 - 0.021403950235092197*lnbara*v*x2*y5 + 
     0.06988272110288132*w*x2*y5 + 0.021403950235092197*lnbara*w*x2*y5 + 
     0.001388160767640295*x3*y5 + 0.5112314175751462*y6 - 
     0.023993960518668304*u*y6 + 0.0011180783913151146*u2*y6 - 
     0.0339914082143444*v*y6 + 0.0010711545725659018*u*v*y6 + 
     0.0018904202297008643*v2*y6 - 0.0339914082143444*w*y6 + 
     0.0010711545725659018*u*w*y6 + 0.0018148095406981609*v*w*y6 + 
     0.0018904202297008643*w2*y6 - 0.0339914082143444*x*y6 + 
     0.0010711545725659018*u*x*y6 - 0.11244046513485796*v*x*y6 - 
     0.032045606778957444*lnbara*v*x*y6 + 0.11490665548248956*w*x*y6 + 
     0.032045606778957444*lnbara*w*x*y6 + 0.0018904202297008643*x2*y6 - 
     0.12321474882489236*y7 + 0.002496169349396353*u*y7 + 
     0.0035512018991495737*v*y7 + 0.0035512018991495737*w*y7 + 
     0.0035512018991495737*x*y7 + 0.013314221484862214*y8 - 
     1.2020569031595942*z + 0.9809537469447522*u*z - 
     0.9004176968697725*u2*z + 0.6858555690183518*u3*z - 
     0.386061737342387*u4*z + 0.1475870262830398*u5*z - 
     0.0339914082143444*u6*z + 0.0035512018991495737*u7*z + 
     0.7968713467733699*v*z - 2.9443689245385314*u*v*z - 
     0.9115194817125674*lnbara*u*v*z - 2.177117134294563*u2*v*z - 
     0.7823223301987707*lnbara*u2*v*z - 1.5212050940015565*u3*v*z - 
     0.41420887449201577*lnbara*u3*v*z - 0.33851565849786086*u4*v*z - 
     0.10984007042350338*lnbara*u4*v*z - 0.07720156919970216*u5*v*z - 
     0.018029304599241732*lnbara*u5*v*z - 0.003018119129298935*u6*v*z - 
     0.0010840252459144199*lnbara*u6*v*z + 1.7490619043784765*v2*z + 
     0.9115194817125674*lnbara*v2*z + 6.04138458368625*u*v2*z + 
     1.9748773458715838*lnbara*u*v2*z + 4.3302489692293955*u2*v2*z + 
     1.3823733765239528*lnbara*u2*v2*z + 1.7772082042508452*u3*v2*z + 
     0.5107892746111199*lnbara*u3*v2*z + 0.33555238697393286*u4*v2*z + 
     0.09558272829689277*lnbara*u4*v2*z + 0.028399509890409173*u5*v2*z + 
     0.007442787768918609*lnbara*u5*v2*z - 0.5311674638752799*v3*z - 
     0.4329554475790071*lnbara*v3*z - 8.70945468065623*u*v3*z - 
     2.843556800957142*lnbara*u*v3*z - 1.6011517276791973*u2*v3*z - 
     0.46172402438324356*lnbara*u2*v3*z - 1.576189905969709*u3*v3*z - 
     0.465804859856524*lnbara*u3*v3*z + 0.015644250220599377*u4*v3*z + 
     0.006838625664216695*lnbara*u4*v3*z + 1.9188684163037728*v4*z + 
     0.7450820516264817*lnbara*v4*z + 4.540055219513269*u*v4*z + 
     1.450672032069528*lnbara*u*v4*z + 1.122430115235894*u2*v4*z + 
     0.3173310228787825*lnbara*u2*v4*z + 0.2239680885196098*u3*v4*z + 
     0.06514335645328273*lnbara*u3*v4*z - 0.6982508102158693*v5*z - 
     0.2748859330079916*lnbara*v5*z - 1.8742275538737843*u*v5*z - 
     0.5730994907821886*lnbara*u*v5*z - 0.11604224021663521*u2*v5*z - 
     0.03151257404486913*lnbara*u2*v5*z + 0.27910097428080544*v6*z + 
     0.09609081997302192*lnbara*v6*z + 0.2599997989809029*u*v6*z + 
     0.07776386003863338*lnbara*u*v6*z - 0.033158063533986065*v7*z - 
     0.01108690204816448*lnbara*v7*z + 0.9809537469447522*w*z + 
     1.7296818908683025*u*w*z + 0.9115194817125674*lnbara*u*w*z + 
     2.950865208678974*u2*w*z + 0.7823223301987707*lnbara*u2*w*z + 
     1.131721092930809*u3*w*z + 0.41420887449201577*lnbara*u3*w*z + 
     0.47680804183544256*u4*w*z + 0.10984007042350338*lnbara*u4*w*z + 
     0.046951959438293674*u5*w*z + 0.018029304599241732*lnbara*u5*w*z + 
     0.006061422665855913*u6*w*z + 0.0010840252459144199*lnbara*u6*w*z - 
     0.5171174600050477*v*w*z + 0.2576868260776136*u*v*w*z - 
     0.1113550912072001*u2*v*w*z + 0.03580809539433375*u3*v*w*z - 
     0.007320613539132591*u4*v*w*z + 0.0007017792893363362*u5*v*w*z - 
     16.220273175775272*v2*w*z - 5.75015242423565*lnbara*v2*w*z + 
     7.283613291269799*u*v2*w*z + 2.569763477061043*lnbara*u*v2*w*z - 
     10.868438804341743*u2*v2*w*z - 3.4712775032258807*lnbara*u2*v2*w*z + 
     2.723658999110921*u3*v2*w*z + 0.8563693554764021*lnbara*u3*v2*w*z - 
     0.5498436699327358*u4*v2*w*z - 0.16402829156973656*lnbara*u4*v2*w*
      z + 11.250037884026336*v3*w*z + 3.9890155563258416*lnbara*v3*w*z + 
     0.5861690638750913*u*v3*w*z + 0.04824677725141553*lnbara*u*v3*w*z + 
     2.7450669338987366*u2*v3*w*z + 0.8803435778848646*lnbara*u2*v3*w*z - 
     0.06800667812701873*u3*v3*w*z - 0.029360220026995808*lnbara*u3*v3*w*
      z - 10.094036110016523*v4*w*z - 3.28316404172915*lnbara*v4*w*z + 
     1.0213274392385536*u*v4*w*z + 0.336346527741606*lnbara*u*v4*w*z - 
     0.8972167925745769*u2*v4*w*z - 0.26783993094333697*lnbara*u2*v4*w*
      z + 3.2110778233847816*v5*w*z + 1.0179245136226938*lnbara*v5*w*z - 
     0.01449065568595044*u*v5*w*z - 0.010275898356323341*lnbara*u*v5*w*z - 
     0.5222803748590084*v6*w*z - 0.15852336677018297*lnbara*v6*w*z - 
     3.3276691614032563*w2*z - 0.9115194817125674*lnbara*w2*z - 
     5.289736299142919*u*w2*z - 1.9748773458715838*lnbara*u*w2*z - 
     4.650086557902906*u2*w2*z - 1.3823733765239528*lnbara*u2*w2*z - 
     1.6750109905361525*u3*w2*z - 0.5107892746111199*lnbara*u3*w2*z - 
     0.3563815319580459*u4*w2*z - 0.09558272829689277*lnbara*u4*w2*z - 
     0.026406119658478486*u5*w2*z - 0.007442787768918609*lnbara*u5*w2*z + 
     16.840740404671994*v*w2*z + 5.75015242423565*lnbara*v*w2*z - 
     7.4944534531230955*u*v*w2*z - 2.569763477061043*lnbara*u*v*w2*z + 
     10.92704206771195*u2*v*w2*z + 3.4712775032258807*lnbara*u2*v*w2*z - 
     2.7345557618171856*u3*v*w2*z - 0.8563693554764021*lnbara*u3*v*w2*z + 
     0.5508222899910821*u4*v*w2*z + 0.16402829156973656*lnbara*u4*v*w2*
      z - 0.1223579533330431*v2*w2*z + 0.027313138910434232*u*v2*w2*z - 
     0.004442047950830934*u2*v2*w2*z + 0.0003652020446131235*u3*v2*w2*
      z + 6.336956677089462*v3*w2*z + 1.8196275948475404*lnbara*v3*w2*z - 
     3.5946630672038005*u*v3*w2*z - 1.0856448829578143*lnbara*u*v3*w2*z + 
     0.6616579679935063*u2*v3*w2*z + 0.18951332857164516*lnbara*u2*v3*
      w2*z - 0.7906870068555788*v4*w2*z - 0.2013783664993635*lnbara*v4*
      w2*z + 0.3281593662661477*u*v4*w2*z + 0.09992430356434646*lnbara*u*
      v4*w2*z + 0.2301351338116376*v5*w2*z + 0.05887745268544395*lnbara*
      v5*w2*z + 1.7170463437119636*w3*z + 0.4329554475790071*lnbara*w3*
      z + 8.335636464746663*u*w3*z + 2.843556800957142*lnbara*u*w3*z + 
     1.7021286302045417*u2*w3*z + 0.46172402438324356*lnbara*u2*w3*z + 
     1.5576572952608156*u3*w3*z + 0.465804859856524*lnbara*u3*w3*z - 
     0.01399123854239156*u4*w3*z - 0.006838625664216695*lnbara*u4*w3*z - 
     11.553644836317881*v*w3*z - 3.9890155563258416*lnbara*v*w3*z - 
     0.5200656204166001*u*v*w3*z - 0.04824677725141553*lnbara*u*v*w3*z - 
     2.7556939387551567*u2*v*w3*z - 0.8803435778848646*lnbara*u2*v*w3*z + 
     0.0688751348615315*u3*v*w3*z + 0.029360220026995808*lnbara*u3*v*w3*
      z - 6.2616857203946035*v2*w3*z - 1.8196275948475404*lnbara*v2*w3*
      z + 3.5848494417836383*u*v2*w3*z + 1.0856448829578143*lnbara*u*v2*
      w3*z - 0.660951971257608*u2*v2*w3*z - 0.18951332857164516*lnbara*
      u2*v2*w3*z - 0.006605505610793891*v3*w3*z + 
     0.00038509593985133696*u*v3*w3*z - 0.380500721540267*v4*w3*z - 
     0.10823518328690307*lnbara*v4*w3*z - 2.581767811748015*w4*z - 
     0.7450820516264817*lnbara*w4*z - 4.408138300869153*u*w4*z - 
     1.450672032069528*lnbara*u*w4*z - 1.1428845474172378*u2*w4*z - 
     0.3173310228787825*lnbara*u2*w4*z - 0.2223252216914388*u3*w4*z - 
     0.06514335645328273*lnbara*u3*w4*z + 10.200182207413253*v*w4*z + 
     3.28316404172915*lnbara*v*w4*z - 1.034655985722088*u*v*w4*z - 
     0.336346527741606*lnbara*u*v*w4*z + 0.8981582264276017*u2*v*w4*z + 
     0.26783993094333697*lnbara*u2*v*w4*z + 0.7756766725864659*v2*w4*z + 
     0.2013783664993635*lnbara*v2*w4*z - 0.3272951320069847*u*v2*w4*z - 
     0.09992430356434646*lnbara*u*v2*w4*z + 0.38164842058355447*v3*w4*z + 
     0.10823518328690307*lnbara*v3*w4*z + 0.9506587243582331*w5*z + 
     0.2748859330079916*lnbara*w5*z + 1.84547571556521*u*w5*z + 
     0.5730994907821886*lnbara*u*w5*z + 0.11799273363683711*u2*w5*z + 
     0.03151257404486913*lnbara*u2*w5*z - 3.234078310073586*v*w5*z - 
     1.0179245136226938*lnbara*v*w5*z + 0.01575761553644318*u*v*w5*z + 
     0.010275898356323341*lnbara*u*v*w5*z - 0.22871819549961922*v2*w5*z - 
     0.05887745268544395*lnbara*v2*w5*z - 0.3370863430138181*w6*z - 
     0.09609081997302192*lnbara*w6*z - 0.25711383486763884*u*w6*z - 
     0.07776386003863338*lnbara*u*w6*z + 0.5245800234274332*v*w6*z + 
     0.15852336677018297*lnbara*v*w6*z + 0.039205434782531995*w7*z + 
     0.01108690204816448*lnbara*w7*z + 0.9809537469447522*x*z - 
     0.5522472005141469*u*x*z + 0.3313625873896489*u2*x*z - 
     0.162591133426937*u3*x*z + 0.056971794565663605*u4*x*z - 
     0.012364843466299529*u5*x*z + 0.0012376963517727734*u6*x*z - 
     0.5171174600050477*v*x*z + 0.2576868260776136*u*v*x*z - 
     0.1113550912072001*u2*v*x*z + 0.03580809539433375*u3*v*x*z - 
     0.007320613539132591*u4*v*x*z + 0.0007017792893363362*u5*v*x*z - 
     10.763863509478952*v2*x*z - 3.7979978404690304*lnbara*v2*x*z + 
     14.408707884242522*u*v2*x*z + 4.851483174726708*lnbara*u*v2*x*z - 
     11.35352724075799*u2*v2*x*z - 3.638468221673086*lnbara*u2*v2*x*z + 
     3.595212268118998*u3*v2*x*z + 1.124626001903981*lnbara*u3*v2*x*z - 
     0.637464961753253*u4*v2*x*z - 0.19025408565816512*lnbara*u4*v2*x*z + 
     4.570769251234671*v3*x*z + 1.7115875434794767*lnbara*v3*x*z - 
     1.485347767785549*u*v3*x*z - 0.6728643192060604*lnbara*u*v3*x*z + 
     2.240193096445929*u2*v3*x*z + 0.7375497739915498*lnbara*u2*v3*x*z - 
     0.07656531816033134*u3*v3*x*z - 0.03634902416016519*lnbara*u3*v3*x*
      z - 6.0865303594656925*v4*x*z - 1.9510966301046395*lnbara*v4*x*z + 
     2.583274807875346*u*v4*x*z + 0.814048966327527*lnbara*u*v4*x*z - 
     1.0214921534358323*u2*v4*x*z - 0.3025359736703866*lnbara*u2*v4*x*z + 
     1.5716677873215597*v5*x*z + 0.5030836809260285*lnbara*v5*x*z - 
     0.15605883003169002*u*v5*x*z - 0.05503807613937338*lnbara*u*v5*x*z - 
     0.2866167180827836*v6*x*z - 0.08634183804336616*lnbara*v6*x*z - 
     0.5171174600050477*w*x*z + 0.2576868260776136*u*w*x*z - 
     0.1113550912072001*u2*w*x*z + 0.03580809539433375*u3*w*x*z - 
     0.007320613539132591*u4*w*x*z + 0.0007017792893363362*u5*w*x*z + 
     22.357006209034473*v*w*x*z + 7.595995680938061*lnbara*v*w*x*z - 
     29.1182808068324*u*v*w*x*z - 9.702966349453416*lnbara*u*v*w*x*z + 
     22.793147607056017*u2*v*w*x*z + 7.276936443346172*lnbara*u2*v*w*x*z - 
     7.206676437932214*u3*v*w*x*z - 2.249252003807962*lnbara*u3*v*w*x*z + 
     1.2764023824610562*u4*v*w*x*z + 0.38050817131633025*lnbara*u4*v*w*x*
      z - 4.808822770349789*v2*w*x*z - 1.7115875434794767*lnbara*v2*w*x*z + 
     1.5426830809278815*u*v2*w*x*z + 0.6728643192060604*lnbara*u*v2*w*x*z - 
     2.2498626331415124*u2*v2*w*x*z - 0.7375497739915498*lnbara*u2*v2*w*x*
      z + 0.07737638458711994*u3*v2*w*x*z + 0.03634902416016519*lnbara*u3*
      v2*w*x*z + 14.388317561563154*v3*w*x*z + 4.420309235697655*lnbara*v3*
      w*x*z - 10.915203494539247*u*v3*w*x*z - 3.2792181994652556*lnbara*u*
      v3*w*x*z + 2.4240248630265353*u2*v3*w*x*z + 
     0.7115869901073909*lnbara*u2*v3*w*x*z - 2.2689406137217927*v4*w*x*z - 
     0.7108966463767397*lnbara*v4*w*x*z + 1.0002685704936713*u*v4*w*x*z + 
     0.30624466399171646*lnbara*u*v4*w*x*z + 0.6108819098365138*v5*w*x*z + 
     0.1761800865537642*lnbara*v5*w*x*z - 10.763863509478952*w2*x*z - 
     3.7979978404690304*lnbara*w2*x*z + 14.408707884242522*u*w2*x*z + 
     4.851483174726708*lnbara*u*w2*x*z - 11.35352724075799*u2*w2*x*z - 
     3.638468221673086*lnbara*u2*w2*x*z + 3.595212268118998*u3*w2*x*z + 
     1.124626001903981*lnbara*u3*w2*x*z - 0.637464961753253*u4*w2*x*z - 
     0.19025408565816512*lnbara*u4*w2*x*z - 4.808822770349789*v*w2*x*z - 
     1.7115875434794767*lnbara*v*w2*x*z + 1.5426830809278815*u*v*w2*x*z + 
     0.6728643192060604*lnbara*u*v*w2*x*z - 2.2498626331415124*u2*v*w2*x*
      z - 0.7375497739915498*lnbara*u2*v*w2*x*z + 
     0.07737638458711994*u3*v*w2*x*z + 0.03634902416016519*lnbara*u3*v*w2*
      x*z - 16.42034107834173*v2*w2*x*z - 4.93842521118603*lnbara*v2*w2*x*
      z + 16.637816190951657*u*v2*w2*x*z + 4.930338466275457*lnbara*u*v2*
      w2*x*z - 2.803109839150323*u2*v2*w2*x*z - 
     0.8181020328740087*lnbara*u2*v2*w2*x*z + 0.6761564834039746*v3*w2*x*
      z + 0.2078129654507111*lnbara*v3*w2*x*z - 0.8428720597864838*u*v3*
      w2*x*z - 0.25120658785234307*lnbara*u*v3*w2*x*z - 
     0.9508002022690504*v4*w2*x*z - 0.2673286182958272*lnbara*v4*w2*x*z + 
     4.570769251234671*w3*x*z + 1.7115875434794767*lnbara*w3*x*z - 
     1.485347767785549*u*w3*x*z - 0.6728643192060604*lnbara*u*w3*x*z + 
     2.240193096445929*u2*w3*x*z + 0.7375497739915498*lnbara*u2*w3*x*z - 
     0.07656531816033134*u3*w3*x*z - 0.03634902416016519*lnbara*u3*w3*x*
      z + 14.388317561563154*v*w3*x*z + 4.420309235697655*lnbara*v*w3*x*z - 
     10.915203494539247*u*v*w3*x*z - 3.2792181994652556*lnbara*u*v*w3*x*z + 
     2.4240248630265353*u2*v*w3*x*z + 0.7115869901073909*lnbara*u2*v*w3*x*
      z + 0.6761564834039746*v2*w3*x*z + 0.2078129654507111*lnbara*v2*w3*
      x*z - 0.8428720597864838*u*v2*w3*x*z - 0.25120658785234307*lnbara*u*
      v2*w3*x*z + 1.2574727801717278*v3*w3*x*z + 
     0.3549807395708584*lnbara*v3*w3*x*z - 6.0865303594656925*w4*x*z - 
     1.9510966301046395*lnbara*w4*x*z + 2.583274807875346*u*w4*x*z + 
     0.814048966327527*lnbara*u*w4*x*z - 1.0214921534358323*u2*w4*x*z - 
     0.3025359736703866*lnbara*u2*w4*x*z - 2.2689406137217927*v*w4*x*z - 
     0.7108966463767397*lnbara*v*w4*x*z + 1.0002685704936713*u*v*w4*x*z + 
     0.30624466399171646*lnbara*u*v*w4*x*z - 0.9508002022690504*v2*w4*x*
      z - 0.2673286182958272*lnbara*v2*w4*x*z + 1.5716677873215597*w5*x*
      z + 0.5030836809260285*lnbara*w5*x*z - 0.15605883003169002*u*w5*x*z - 
     0.05503807613937338*lnbara*u*w5*x*z + 0.6108819098365138*v*w5*x*z + 
     0.1761800865537642*lnbara*v*w5*x*z - 0.2866167180827836*w6*x*z - 
     0.08634183804336616*lnbara*w6*x*z - 0.9004176968697725*x2*z + 
     0.3313625873896489*u*x2*z - 0.12887042405432217*u2*x2*z + 
     0.03949367079133293*u3*x2*z - 0.007867552944290607*u4*x2*z + 
     0.0007424815539584533*u5*x2*z + 16.840740404671994*v*x2*z + 
     5.75015242423565*lnbara*v*x2*z - 7.4944534531230955*u*v*x2*z - 
     2.569763477061043*lnbara*u*v*x2*z + 10.92704206771195*u2*v*x2*z + 
     3.4712775032258807*lnbara*u2*v*x2*z - 2.7345557618171856*u3*v*x2*z - 
     0.8563693554764021*lnbara*u3*v*x2*z + 0.5508222899910821*u4*v*x2*z + 
     0.16402829156973656*lnbara*u4*v*x2*z - 2.5057639727820775*v2*x2*z - 
     1.0258918950424711*lnbara*v2*x2*z - 4.04603639452907*u*v2*x2*z - 
     1.3032259701999138*lnbara*u*v2*x2*z - 0.5438356537989508*u2*v2*x2*
      z - 0.1818291572693978*lnbara*u2*v2*x2*z - 
     0.17072856838087597*u3*v2*x2*z - 0.04904808595029047*lnbara*u3*v2*
      x2*z + 9.909152497593267*v3*x2*z + 3.0912384401060025*lnbara*v3*x2*
      z - 5.305139799956746*u*v3*x2*z - 1.5393343490108804*lnbara*u*v3*x2*
      z + 1.5675237969406215*u2*v3*x2*z + 0.4579464223436356*lnbara*u2*
      v3*x2*z - 1.4226442649834423*v4*x2*z - 0.4703228534910837*lnbara*
      v4*x2*z + 0.2656466906715195*u*v4*x2*z + 0.08058867723285898*lnbara*
      u*v4*x2*z + 0.4211542429001266*v5*x2*z + 0.12424611305742275*lnbara*
      v5*x2*z - 16.220273175775272*w*x2*z - 5.75015242423565*lnbara*w*x2*
      z + 7.283613291269799*u*w*x2*z + 2.569763477061043*lnbara*u*w*x2*z - 
     10.868438804341743*u2*w*x2*z - 3.4712775032258807*lnbara*u2*w*x2*z + 
     2.723658999110921*u3*w*x2*z + 0.8563693554764021*lnbara*u3*w*x2*z - 
     0.5498436699327358*u4*w*x2*z - 0.16402829156973656*lnbara*u4*w*x2*
      z - 4.808822770349789*v*w*x2*z - 1.7115875434794767*lnbara*v*w*x2*z + 
     1.5426830809278815*u*v*w*x2*z + 0.6728643192060604*lnbara*u*v*w*x2*z - 
     2.2498626331415124*u2*v*w*x2*z - 0.7375497739915498*lnbara*u2*v*w*x2*
      z + 0.07737638458711994*u3*v*w*x2*z + 0.03634902416016519*lnbara*u3*
      v*w*x2*z - 20.346437220092437*v2*w*x2*z - 6.08173812414508*lnbara*
      v2*w*x2*z + 16.365226923917742*u*v2*w*x2*z + 
     4.905952869158358*lnbara*u*v2*w*x2*z - 3.2577140116251813*u2*v2*w*
      x2*z - 0.9439400410182851*lnbara*u2*v2*w*x2*z + 
     1.4784816196761565*v3*w*x2*z + 0.43230388098155775*lnbara*v3*w*x2*
      z - 0.6938884585226485*u*v3*w*x2*z - 0.21809446900966986*lnbara*u*v3*
      w*x2*z - 1.267405871830795*v4*w*x2*z - 0.35480949331667067*lnbara*
      v4*w*x2*z + 6.988719566598391*w2*x2*z + 2.737479438521948*lnbara*
      w2*x2*z + 2.583201962578535*u*w2*x2*z + 0.6303616509938534*lnbara*u*
      w2*x2*z + 2.78012177163788*u2*w2*x2*z + 0.9193789312609476*lnbara*
      u2*w2*x2*z + 0.0944959815922243*u3*w2*x2*z + 
     0.012699061790125279*lnbara*u3*w2*x2*z + 21.55337156258267*v*w2*x2*
      z + 6.548245980022291*lnbara*v*w2*x2*z - 14.057135775486067*u*v*w2*
      x2*z - 4.266687153487787*lnbara*u*v*w2*x2*z + 
     3.4982098333445135*u2*v*w2*x2*z + 1.0191704113395068*lnbara*u2*v*w2*
      x2*z - 0.40857845580650054*v2*w2*x2*z - 0.08307962011998349*lnbara*
      v2*w2*x2*z - 0.5491803784646733*u*v2*w2*x2*z - 
     0.17704276038055036*lnbara*u*v2*w2*x2*z + 2.1080028169579204*v3*w2*
      x2*z + 0.5953996603124057*lnbara*v3*w2*x2*z - 
     10.997725885446338*w3*x2*z - 3.557746295983214*lnbara*w3*x2*z + 
     2.9792615221240184*u*w3*x2*z + 0.9000686333403095*lnbara*u*w3*x2*z - 
     1.8066447210612555*u2*w3*x2*z - 0.5331767926648573*lnbara*u2*w3*x2*
      z - 2.1068706790103*v*w3*x2*z - 0.6955343577421946*lnbara*v*w3*x2*
      z + 0.8761362502114731*u*v*w3*x2*z + 0.296376278878668*lnbara*u*v*w3*
      x2*z - 1.7437824185418067*v2*w3*x2*z - 0.4914395306435913*lnbara*
      v2*w3*x2*z + 2.4334094721170496*w4*x2*z + 
     0.816632950371704*lnbara*w4*x2*z + 0.10307841593891565*u*w4*x2*z + 
     0.018172273278693238*lnbara*u*w4*x2*z + 1.0094259951122324*v*w4*x2*
      z + 0.2874207900901255*lnbara*v*w4*x2*z - 0.5247428948621341*w5*x2*
      z - 0.16081753949969196*lnbara*w5*x2*z + 0.6858555690183518*x3*z - 
     0.162591133426937*u*x3*z + 0.03949367079133293*u2*x3*z - 
     0.006882314713446356*u3*x3*z + 0.0005960914163116732*u4*x3*z - 
     11.553644836317881*v*x3*z - 3.9890155563258416*lnbara*v*x3*z - 
     0.5200656204166001*u*v*x3*z - 0.04824677725141553*lnbara*u*v*x3*z - 
     2.7556939387551567*u2*v*x3*z - 0.8803435778848646*lnbara*u2*v*x3*z + 
     0.0688751348615315*u3*v*x3*z + 0.029360220026995808*lnbara*u3*v*x3*
      z - 4.961706764789829*v2*x3*z - 1.3124617908295524*lnbara*v2*x3*z + 
     8.457223146401986*u*v2*x3*z + 2.5565136487937905*lnbara*u*v2*x3*z - 
     0.9782558475006098*u2*v2*x3*z - 0.28088060701825807*lnbara*u2*v2*
      x3*z - 1.6389294857674492*v3*x3*z - 0.5238484235184111*lnbara*v3*
      x3*z - 0.30529725405798797*u*v3*x3*z - 0.10319808585533102*lnbara*u*
      v3*x3*z - 0.31318215309396386*v4*x3*z - 0.07872508228066824*lnbara*
      v4*x3*z + 11.250037884026336*w*x3*z + 3.9890155563258416*lnbara*w*
      x3*z + 0.5861690638750913*u*w*x3*z + 0.04824677725141553*lnbara*u*w*
      x3*z + 2.7450669338987366*u2*w*x3*z + 0.8803435778848646*lnbara*u2*
      w*x3*z - 0.06800667812701873*u3*w*x3*z - 0.029360220026995808*lnbara*
      u3*w*x3*z + 14.388317561563154*v*w*x3*z + 4.420309235697655*lnbara*v*
      w*x3*z - 10.915203494539247*u*v*w*x3*z - 3.2792181994652556*lnbara*u*
      v*w*x3*z + 2.4240248630265353*u2*v*w*x3*z + 
     0.7115869901073909*lnbara*u2*v*w*x3*z + 1.4784816196761565*v2*w*x3*
      z + 0.43230388098155775*lnbara*v2*w*x3*z - 0.6938884585226485*u*v2*w*
      x3*z - 0.21809446900966986*lnbara*u*v2*w*x3*z + 
     1.5073534267839392*v3*w*x3*z + 0.4300715944036431*lnbara*v3*w*x3*z - 
     9.326716278242827*w2*x3*z - 3.107847444868102*lnbara*w2*x3*z + 
     2.4434643394917717*u*w2*x3*z + 0.722704550671465*lnbara*u*w2*x3*z - 
     1.4446667880342972*u2*w2*x3*z - 0.43070638308913284*lnbara*u2*w2*
      x3*z - 2.1068706790103*v*w2*x3*z - 0.6955343577421946*lnbara*v*w2*
      x3*z + 0.8761362502114731*u*v*w2*x3*z + 0.296376278878668*lnbara*u*v*
      w2*x3*z - 1.7364430725838955*v2*w2*x3*z - 
     0.4913145532522245*lnbara*v2*w2*x3*z + 2.2465725155547824*w3*x3*z + 
     0.7870789002790479*lnbara*w3*x3*z + 0.1244773390895362*u*w3*x3*z + 
     0.024916275986332885*lnbara*u*w3*x3*z + 1.0840027554445029*v*w3*x3*
      z + 0.30837662850281694*lnbara*v*w3*x3*z - 0.5397225167390112*w4*x3*
      z - 0.16840858737356737*lnbara*w4*x3*z - 0.386061737342387*x4*z + 
     0.056971794565663605*u*x4*z - 0.007867552944290607*u2*x4*z + 
     0.0005960914163116732*u3*x4*z + 10.200182207413253*v*x4*z + 
     3.28316404172915*lnbara*v*x4*z - 1.034655985722088*u*v*x4*z - 
     0.336346527741606*lnbara*u*v*x4*z + 0.8981582264276017*u2*v*x4*z + 
     0.26783993094333697*lnbara*u2*v*x4*z + 0.01586781553997586*v2*x4*z - 
     0.06352181102996676*lnbara*v2*x4*z - 1.0816608287503775*u*v2*x4*z - 
     0.32438223975151587*lnbara*u*v2*x4*z + 0.8263404688000965*v3*x4*z + 
     0.2438589036761638*lnbara*v3*x4*z - 10.094036110016523*w*x4*z - 
     3.28316404172915*lnbara*w*x4*z + 1.0213274392385536*u*w*x4*z + 
     0.336346527741606*lnbara*u*w*x4*z - 0.8972167925745769*u2*w*x4*z - 
     0.26783993094333697*lnbara*u2*w*x4*z - 2.2689406137217927*v*w*x4*z - 
     0.7108966463767397*lnbara*v*w*x4*z + 1.0002685704936713*u*v*w*x4*z + 
     0.30624466399171646*lnbara*u*v*w*x4*z - 1.267405871830795*v2*w*x4*z - 
     0.35480949331667067*lnbara*v2*w*x4*z + 2.2331993759180806*w2*x4*z + 
     0.7744184574067065*lnbara*w2*x4*z + 0.08267922567088919*u*w2*x4*z + 
     0.01813757575979942*lnbara*u*w2*x4*z + 1.0094259951122324*v*w2*x4*
      z + 0.2874207900901255*lnbara*v*w2*x4*z - 0.5665614064319183*w3*x4*
      z - 0.1764702004496186*lnbara*w3*x4*z + 0.1475870262830398*x5*z - 
     0.012364843466299529*u*x5*z + 0.0007424815539584533*u2*x5*z - 
     3.234078310073586*v*x5*z - 1.0179245136226938*lnbara*v*x5*z + 
     0.01575761553644318*u*v*x5*z + 0.010275898356323341*lnbara*u*v*x5*z - 
     0.14867177025989595*v2*x5*z - 0.030113006529747105*lnbara*v2*x5*z + 
     3.2110778233847816*w*x5*z + 1.0179245136226938*lnbara*w*x5*z - 
     0.01449065568595044*u*w*x5*z - 0.010275898356323341*lnbara*u*w*x5*z + 
     0.6108819098365138*v*w*x5*z + 0.1761800865537642*lnbara*v*w*x5*z - 
     0.46033723691317213*w2*x5*z - 0.14606708002401708*lnbara*w2*x5*z - 
     0.0339914082143444*x6*z + 0.0012376963517727734*u*x6*z + 
     0.5245800234274332*v*x6*z + 0.15852336677018297*lnbara*v*x6*z - 
     0.5222803748590084*w*x6*z - 0.15852336677018297*lnbara*w*x6*z + 
     0.0035512018991495737*x7*z + 0.9809537469447522*y*z - 
     0.5171174600050477*u*y*z + 0.29918371952782097*u2*y*z - 
     0.14397058356518277*u3*y*z + 0.04988531635163183*u4*y*z - 
     0.01075135761798518*u5*y*z + 0.0010711545725659018*u6*y*z - 
     7.798871853605498*v*y*z - 2.734558445137702*lnbara*v*y*z - 
     5.604833095618447*u*v*y*z - 1.7772673145259574*lnbara*u*v*y*z - 
     3.826201875320261*u2*v*y*z - 1.1575775618246809*lnbara*u2*v*y*z - 
     1.262771091795936*u3*v*y*z - 0.36117019327441874*lnbara*u3*v*y*z - 
     0.26452915782323383*u4*v*y*z - 0.07250424934435563*lnbara*u4*v*y*z - 
     0.0196913141648136*u5*v*y*z - 0.0052376950009461855*lnbara*u5*v*y*z + 
     8.371127296661705*v2*y*z + 3.007965370948085*lnbara*v2*y*z + 
     8.142878738630774*u*v2*y*z + 2.570302308603461*lnbara*u*v2*y*z + 
     4.372397515146079*u2*v2*y*z + 1.303884138860003*lnbara*u2*v2*y*z + 
     1.089630311725629*u3*v2*y*z + 0.306972893835514*lnbara*u3*v2*y*z + 
     0.11045275181230746*u4*v2*y*z + 0.029980593819149505*lnbara*u4*v2*y*
      z - 14.456656314267676*v3*y*z - 4.8057115258275305*lnbara*v3*y*z - 
     3.6907580059131853*u*v3*y*z - 1.0737089522063064*lnbara*u*v3*y*z - 
     3.5123911436649746*u2*v3*y*z - 1.041566396111375*lnbara*u2*v3*y*z - 
     0.10728353612006211*u3*v3*y*z - 0.025365208983699924*lnbara*u3*v3*y*
      z + 8.072482666125286*v4*y*z + 2.6276787546704745*lnbara*v4*y*z + 
     2.0086611123781046*u*v4*y*z + 0.5668452509261017*lnbara*u*v4*y*z + 
     0.590076818257424*u2*v4*y*z + 0.17202385787388583*lnbara*u2*v4*y*z - 
     3.248331013993099*v5*y*z - 1.0069202481453856*lnbara*v5*y*z - 
     0.23401253231882208*u*v5*y*z - 0.06359410612179822*lnbara*u*v5*y*z + 
     0.47407842963740254*v6*y*z + 0.14311605655840995*lnbara*v6*y*z + 
     6.729507193086304*w*y*z + 2.734558445137702*lnbara*w*y*z + 
     6.093431672716988*u*w*y*z + 1.7772673145259574*lnbara*u*w*y*z + 
     3.6244323005177024*u2*w*y*z + 1.1575775618246809*lnbara*u2*w*y*z + 
     1.326032325221745*u3*w*y*z + 0.36117019327441874*lnbara*u3*w*y*z + 
     0.2517914740480953*u4*w*y*z + 0.07250424934435563*lnbara*u4*w*y*z + 
     0.020900424621812116*u5*w*y*z + 0.0052376950009461855*lnbara*u5*w*y*
      z + 0.2576868260776136*v*w*y*z - 0.08549175984210892*u*v*w*y*z + 
     0.023341335158146424*u2*v*w*y*z - 0.004288309525390823*u3*v*w*y*z + 
     0.0003820180212080083*u4*v*w*y*z + 14.923465303571382*v2*w*y*z + 
     4.700428217339612*lnbara*v2*w*y*z - 8.636182740657008*u*v2*w*y*z - 
     2.845063377679382*lnbara*u*v2*w*y*z + 2.0292029954219255*u2*v2*w*y*
      z + 0.6312081178376158*lnbara*u2*v2*w*y*z - 
     0.5873630368597165*u3*v2*w*y*z - 0.17392267685688417*lnbara*u3*v2*w*
      y*z - 2.3689366351444274*v3*w*y*z - 0.726506814968872*lnbara*v3*w*y*
      z + 2.469819197461433*u*v3*w*y*z + 0.8025320427989695*lnbara*u*v3*w*y*
      z + 0.18981304783567338*u2*v3*w*y*z + 0.049895924897055216*lnbara*u2*
      v3*w*y*z + 2.495271539467373*v4*w*y*z + 0.7051754521652789*lnbara*v4*
      w*y*z - 0.9763863300273927*u*v4*w*y*z - 0.2916555967133236*lnbara*u*
      v4*w*y*z - 0.2133826221782255*v5*w*y*z - 0.0583092465218906*lnbara*
      v5*w*y*z - 7.740580989744235*w2*y*z - 3.007965370948085*lnbara*w2*y*
      z - 8.34831676538791*u*w2*y*z - 2.570302308603461*lnbara*u*w2*y*z - 
     4.3167789899540825*u2*w2*y*z - 1.303884138860003*lnbara*u2*w2*y*z - 
     1.0998046234447676*u3*w2*y*z - 0.306972893835514*lnbara*u3*w2*y*z - 
     0.10954859095754421*u4*w2*y*z - 0.029980593819149505*lnbara*u4*w2*y*
      z - 15.142507034031205*v*w2*y*z - 4.700428217339612*lnbara*v*w2*y*z + 
     8.684236515504576*u*v*w2*y*z + 2.845063377679382*lnbara*u*v*w2*y*z - 
     2.0369169970657963*u2*v*w2*y*z - 0.6312081178376158*lnbara*u2*v*w2*y*
      z + 0.5879915453536685*u3*v*w2*y*z + 0.17392267685688417*lnbara*u3*v*
      w2*y*z + 0.03069034292530752*v2*w2*y*z - 0.003995614840649483*u*v2*
      w2*y*z + 0.0002863764888716214*u2*v2*w2*y*z - 
     6.026774148018949*v3*w2*y*z - 1.7632715691466476*lnbara*v3*w2*y*z + 
     0.731117931401122*u*v3*w2*y*z + 0.21288835092447814*lnbara*u*v3*w2*y*
      z + 0.663692907110261*v4*w2*y*z + 0.19636416180684646*lnbara*v4*w2*
      y*z + 14.150094597275556*w3*y*z + 4.8057115258275305*lnbara*w3*y*z + 
     3.7558333567491418*u*w3*y*z + 1.0737089522063064*lnbara*u*w3*y*z + 
     3.5021103597603167*u2*w3*y*z + 1.041566396111375*lnbara*u2*w3*y*z + 
     0.1081140594632303*u3*w3*y*z + 0.025365208983699924*lnbara*u3*w3*y*
      z + 2.4387387085229473*v*w3*y*z + 0.726506814968872*lnbara*v*w3*y*z - 
     2.478798620600135*u*v*w3*y*z - 0.8025320427989695*lnbara*u*v*w3*y*z - 
     0.1891735970499443*u2*v*w3*y*z - 0.049895924897055216*lnbara*u2*v*w3*
      y*z + 6.015319832123562*v2*w3*y*z + 1.7632715691466476*lnbara*v2*w3*
      y*z - 0.7304529490550218*u*v2*w3*y*z - 0.21288835092447814*lnbara*u*
      v2*w3*y*z + 0.00047310410784989136*v3*w3*y*z - 
     7.96562555520799*w4*y*z - 2.6276787546704745*lnbara*w4*y*z - 
     2.0218463214313056*u*w4*y*z - 0.5668452509261017*lnbara*u*w4*y*z - 
     0.5891572387478903*u2*w4*y*z - 0.17202385787388583*lnbara*u2*w4*y*
      z - 2.5094652412675753*v*w4*y*z - 0.7051754521652789*lnbara*v*w4*y*
      z + 0.977194806205722*u*v*w4*y*z + 0.2916555967133236*lnbara*u*v*w4*y*
      z - 0.6626623249676024*v2*w4*y*z - 0.19636416180684646*lnbara*v2*w4*
      y*z + 3.2252148129088143*w5*y*z + 1.0069202481453856*lnbara*w5*y*z + 
     0.23526924787765785*u*w5*y*z + 0.06359410612179822*lnbara*u*w5*y*z + 
     0.2147385756550609*v*w5*y*z + 0.0583092465218906*lnbara*v*w5*y*z - 
     0.4717695787130639*w6*y*z - 0.14311605655840995*lnbara*w6*y*z - 
     0.6975695736651812*x*y*z + 0.2576868260776136*u*x*y*z - 
     0.09948507064609574*u2*x*y*z + 0.030295348064157374*u3*x*y*z - 
     0.006007932944401952*u4*x*y*z + 0.000565180561156404*u5*x*y*z + 
     0.2576868260776136*v*x*y*z - 0.08549175984210892*u*v*x*y*z + 
     0.023341335158146424*u2*v*x*y*z - 0.004288309525390823*u3*v*x*y*z + 
     0.0003820180212080083*u4*v*x*y*z + 10.777349982686436*v2*x*y*z + 
     3.494158013231508*lnbara*v2*x*y*z - 14.680040084881753*u*v2*x*y*z - 
     4.586055804009709*lnbara*u*v2*x*y*z + 2.978128785284849*u2*v2*x*y*z + 
     0.9528593416680298*lnbara*u2*v2*x*y*z - 0.7405676776348092*u3*v2*x*y*
      z - 0.21821356263065994*lnbara*u3*v2*x*y*z - 
     0.09200933391270437*v3*x*y*z - 0.14480531475704334*lnbara*v3*x*y*z + 
     3.724170161382556*u*v3*x*y*z + 1.1721828559253105*lnbara*u*v3*x*y*z + 
     0.1296884569931269*u2*v3*x*y*z + 0.020514943705544776*lnbara*u2*v3*x*
      y*z + 1.9788571739028187*v4*x*y*z + 0.5910023418671081*lnbara*v4*x*y*
      z - 1.497715134559773*u*v4*x*y*z - 0.43755320330470776*lnbara*u*v4*x*
      y*z - 0.09961325470566673*v5*x*y*z - 0.03303181019568392*lnbara*v5*x*
      y*z + 0.2576868260776136*w*x*y*z - 0.08549175984210892*u*w*x*y*z + 
     0.023341335158146424*u2*w*x*y*z - 0.004288309525390823*u3*w*x*y*z + 
     0.0003820180212080083*u4*w*x*y*z - 21.839161866507176*v*w*x*y*z - 
     6.988316026463016*lnbara*v*w*x*y*z + 29.42708255896211*u*v*w*x*y*z + 
     9.172111608019417*lnbara*u*v*w*x*y*z - 5.96736483518329*u2*v*w*x*y*z - 
     1.9057186833360595*lnbara*u2*v*w*x*y*z + 1.4820558313910297*u3*v*w*x*y*
      z + 0.4364271252613199*lnbara*u3*v*w*x*y*z + 
     0.14564601713500816*v2*w*x*y*z + 0.14480531475704334*lnbara*v2*w*x*y*
      z - 3.731845090721665*u*v2*w*x*y*z - 1.1721828559253105*lnbara*u*v2*w*
      x*y*z - 0.12911307483747164*u2*v2*w*x*y*z - 
     0.020514943705544776*lnbara*u2*v2*w*x*y*z - 8.881204021839554*v3*w*x*
      y*z - 2.6069786638032433*lnbara*v3*w*x*y*z + 
     3.3873280227068086*u*v3*w*x*y*z + 0.9769151192502976*lnbara*u*v3*w*x*y*
      z + 0.6318226520207785*v4*w*x*y*z + 0.19318619266094705*lnbara*v4*w*x*
      y*z + 10.777349982686436*w2*x*y*z + 3.494158013231508*lnbara*w2*x*y*
      z - 14.680040084881753*u*w2*x*y*z - 4.586055804009709*lnbara*u*w2*x*y*
      z + 2.978128785284849*u2*w2*x*y*z + 0.9528593416680298*lnbara*u2*w2*
      x*y*z - 0.7405676776348092*u3*w2*x*y*z - 0.21821356263065994*lnbara*
      u3*w2*x*y*z + 0.14564601713500816*v*w2*x*y*z + 
     0.14480531475704334*lnbara*v*w2*x*y*z - 3.731845090721665*u*v*w2*x*y*
      z - 1.1721828559253105*lnbara*u*v*w2*x*y*z - 
     0.12911307483747164*u2*v*w2*x*y*z - 0.020514943705544776*lnbara*u2*v*
      w2*x*y*z + 13.78024088853395*v2*w2*x*y*z + 
     4.03195264387227*lnbara*v2*w2*x*y*z - 3.777642640073415*u*v2*w2*x*y*
      z - 1.0787238318911796*lnbara*u*v2*w2*x*y*z - 
     0.5309427518886998*v3*w2*x*y*z - 0.16015438246526315*lnbara*v3*w2*x*
      y*z - 0.09200933391270437*w3*x*y*z - 0.14480531475704334*lnbara*w3*x*
      y*z + 3.724170161382556*u*w3*x*y*z + 1.1721828559253105*lnbara*u*w3*x*
      y*z + 0.1296884569931269*u2*w3*x*y*z + 0.020514943705544776*lnbara*
      u2*w3*x*y*z - 8.881204021839554*v*w3*x*y*z - 
     2.6069786638032433*lnbara*v*w3*x*y*z + 3.3873280227068086*u*v*w3*x*y*
      z + 0.9769151192502976*lnbara*u*v*w3*x*y*z - 
     0.5309427518886998*v2*w3*x*y*z - 0.16015438246526315*lnbara*v2*w3*x*
      y*z + 1.9788571739028187*w4*x*y*z + 0.5910023418671081*lnbara*w4*x*y*
      z - 1.497715134559773*u*w4*x*y*z - 0.43755320330470776*lnbara*u*w4*x*
      y*z + 0.6318226520207785*v*w4*x*y*z + 0.19318619266094705*lnbara*v*w4*
      x*y*z - 0.09961325470566673*w5*x*y*z - 0.03303181019568392*lnbara*w5*
      x*y*z + 0.45246456501551086*x2*y*z - 0.1113550912072001*u*x2*y*z + 
     0.027313138910434232*u2*x2*y*z - 0.004771933785158886*u3*x2*y*z + 
     0.0004135240269207944*u4*x2*y*z - 15.142507034031205*v*x2*y*z - 
     4.700428217339612*lnbara*v*x2*y*z + 8.684236515504576*u*v*x2*y*z + 
     2.845063377679382*lnbara*u*v*x2*y*z - 2.0369169970657963*u2*v*x2*y*
      z - 0.6312081178376158*lnbara*u2*v*x2*y*z + 
     0.5879915453536685*u3*v*x2*y*z + 0.17392267685688417*lnbara*u3*v*x2*
      y*z + 0.5638297481368869*v2*x2*y*z + 0.18788462045702756*lnbara*v2*
      x2*y*z + 0.008389015172307996*u*v2*x2*y*z - 
     0.057042047293007624*lnbara*u*v2*x2*y*z - 0.2581203920240232*u2*v2*
      x2*y*z - 0.07638602271158683*lnbara*u2*v2*x2*y*z - 
     6.183783910974188*v3*x2*y*z - 1.7887689679030814*lnbara*v3*x2*y*z + 
     1.617693856062018*u*v3*x2*y*z + 0.4746900012067356*lnbara*u*v3*x2*y*
      z + 0.4680668096248443*v4*x2*y*z + 0.13910950612836231*lnbara*v4*x2*
      y*z + 14.923465303571382*w*x2*y*z + 4.700428217339612*lnbara*w*x2*y*
      z - 8.636182740657008*u*w*x2*y*z - 2.845063377679382*lnbara*u*w*x2*y*
      z + 2.0292029954219255*u2*w*x2*y*z + 0.6312081178376158*lnbara*u2*w*
      x2*y*z - 0.5873630368597165*u3*w*x2*y*z - 0.17392267685688417*lnbara*
      u3*w*x2*y*z + 0.14564601713500816*v*w*x2*y*z + 
     0.14480531475704334*lnbara*v*w*x2*y*z - 3.731845090721665*u*v*w*x2*y*
      z - 1.1721828559253105*lnbara*u*v*w*x2*y*z - 
     0.12911307483747164*u2*v*w*x2*y*z - 0.020514943705544776*lnbara*u2*v*
      w*x2*y*z + 17.091071621436193*v2*w*x2*y*z + 
     4.99601857372239*lnbara*v2*w*x2*y*z - 3.489746635672796*u*v2*w*x2*y*
      z - 1.0013349315415667*lnbara*u*v2*w*x2*y*z - 
     0.9801494429301945*v3*w*x2*y*z - 0.293356436036497*lnbara*v3*w*x2*y*
      z - 0.6330043203094217*w2*x2*y*z - 0.3326899352140709*lnbara*w2*x2*
      y*z + 3.7124326039755955*u*w2*x2*y*z + 1.229224903218318*lnbara*u*w2*
      x2*y*z + 0.3880630632568426*u2*w2*x2*y*z + 
     0.0969009664171316*lnbara*u2*w2*x2*y*z - 14.77406945239147*v*w2*x2*
      y*z - 4.3242801796236945*lnbara*v*w2*x2*y*z + 
     4.000620121386894*u*v*w2*x2*y*z + 1.1582596990589664*lnbara*u*v*w2*
      x2*y*z - 0.2945882654409418*v2*w2*x2*y*z - 
     0.10231487175528582*lnbara*v2*w2*x2*y*z + 3.8502028793420484*w3*x2*
      y*z + 1.1170305738043862*lnbara*w3*x2*y*z - 
     2.127445819261271*u*w3*x2*y*z - 0.6316147687241351*lnbara*u*w3*x2*y*
      z + 0.9014731642677928*v*w3*x2*y*z + 0.28985320078620536*lnbara*v*w3*
      x2*y*z - 0.09314373993962562*w4*x2*y*z - 0.033291399122784844*lnbara*
      w4*x2*y*z - 0.2298476323443832*x3*y*z + 0.03580809539433375*u*x3*y*
      z - 0.005041691635003074*u2*x3*y*z + 0.00038509593985133696*u3*x3*y*
      z + 2.4387387085229473*v*x3*y*z + 0.726506814968872*lnbara*v*x3*y*z - 
     2.478798620600135*u*v*x3*y*z - 0.8025320427989695*lnbara*u*v*x3*y*z - 
     0.1891735970499443*u2*v*x3*y*z - 0.049895924897055216*lnbara*u2*v*x3*
      y*z + 5.676870448679409*v2*x3*y*z + 1.684726260416403*lnbara*v2*x3*
      y*z - 1.182692665484111*u*v2*x3*y*z - 0.32670867639523105*lnbara*u*
      v2*x3*y*z + 0.032891805121808636*v3*x3*y*z - 
     0.002730352711332793*lnbara*v3*x3*y*z - 2.3689366351444274*w*x3*y*z - 
     0.726506814968872*lnbara*w*x3*y*z + 2.469819197461433*u*w*x3*y*z + 
     0.8025320427989695*lnbara*u*w*x3*y*z + 0.18981304783567338*u2*w*x3*y*
      z + 0.049895924897055216*lnbara*u2*w*x3*y*z - 
     8.881204021839554*v*w*x3*y*z - 2.6069786638032433*lnbara*v*w*x3*y*z + 
     3.3873280227068086*u*v*w*x3*y*z + 0.9769151192502976*lnbara*u*v*w*x3*y*
      z - 0.9801494429301945*v2*w*x3*y*z - 0.293356436036497*lnbara*v2*w*
      x3*y*z + 3.1903919237729643*w2*x3*y*z + 0.9222524033868401*lnbara*
      w2*x3*y*z - 2.203724440304821*u*w2*x3*y*z - 
     0.6502064428550666*lnbara*u*w2*x3*y*z + 0.9014731642677928*v*w2*x3*y*
      z + 0.28985320078620536*lnbara*v*w2*x3*y*z + 
     0.04710638371571019*w3*x3*y*z + 0.006233587961624442*lnbara*w3*x3*y*
      z + 0.08203160229248424*x4*y*z - 0.007320613539132591*u*x4*y*z + 
     0.0004507102322422891*u2*x4*y*z - 2.5094652412675753*v*x4*y*z - 
     0.7051754521652789*lnbara*v*x4*y*z + 0.977194806205722*u*v*x4*y*z + 
     0.2916555967133236*lnbara*u*v*x4*y*z - 0.566190372464217*v2*x4*y*z - 
     0.1702185650157698*lnbara*v2*x4*y*z + 2.495271539467373*w*x4*y*z + 
     0.7051754521652789*lnbara*w*x4*y*z - 0.9763863300273927*u*w*x4*y*z - 
     0.2916555967133236*lnbara*u*w*x4*y*z + 0.6318226520207785*v*w*x4*y*z + 
     0.19318619266094705*lnbara*v*w*x4*y*z - 0.06439562415041984*w2*x4*y*
      z - 0.022967627645177263*lnbara*w2*x4*y*z - 
     0.018000480690589078*x5*y*z + 0.0007017792893363362*u*x5*y*z + 
     0.2147385756550609*v*x5*y*z + 0.0583092465218906*lnbara*v*x5*y*z - 
     0.2133826221782255*w*x5*y*z - 0.0583092465218906*lnbara*w*x5*y*z + 
     0.0018148095406981609*x6*y*z - 0.9004176968697725*y2*z + 
     0.29918371952782097*u*y2*z - 0.10942628765049751*u2*y2*z + 
     0.03238162914677262*u3*y2*z - 0.00631630321372675*u4*y2*z + 
     0.0005880488320726199*u5*y2*z + 5.299654762421948*v*y2*z + 
     1.7090990282110639*lnbara*v*y2*z - 1.1643976508622838*u*v*y2*z - 
     0.4284924263303262*lnbara*u*v*y2*z - 1.1346769147760087*u2*v*y2*z - 
     0.329646645071357*lnbara*u2*v*y2*z - 0.3831429064834633*u3*v*y2*z - 
     0.10776709570789808*lnbara*u3*v*y2*z - 0.03979465732217202*u4*v*y2*
      z - 0.010614825463755726*lnbara*u4*v*y2*z - 
     0.044978574732269516*v2*y2*z - 0.06378679623852483*lnbara*v2*y2*z + 
     2.5864306854801558*u*v2*y2*z + 0.8006228366412104*lnbara*u*v2*y2*z + 
     1.1889883290028347*u2*v2*y2*z + 0.3402159951898016*lnbara*u2*v2*y2*
      z + 0.17341769222311584*u3*v2*y2*z + 0.04766748743684271*lnbara*u3*
      v2*y2*z + 2.8467947898195387*v3*y2*z + 0.8710102224533081*lnbara*
      v3*y2*z - 2.2828692800346633*u*v3*y2*z - 0.6947964005893216*lnbara*
      u*v3*y2*z - 0.15198537243470955*u2*v3*y2*z - 
     0.04014533812336501*lnbara*u2*v3*y2*z - 0.32019128431906846*v4*y2*
      z - 0.10788365521657699*lnbara*v4*y2*z + 0.3611306117059756*u*v4*y2*
      z + 0.10812164650482199*lnbara*u*v4*y2*z + 0.11415571194993857*v5*
      y2*z + 0.03339748565253056*lnbara*v5*y2*z - 
     4.647008665663399*w*y2*z - 1.7090990282110639*lnbara*w*y2*z + 
     0.9708296446662518*u*w*y2*z + 0.4284924263303262*lnbara*u*w*y2*z + 
     1.184519706140848*u2*w*y2*z + 0.329646645071357*lnbara*u2*w*y2*z + 
     0.3742822830969903*u3*w*y2*z + 0.10776709570789808*lnbara*u3*w*y2*
      z + 0.04056836531530246*u4*w*y2*z + 0.010614825463755726*lnbara*u4*w*
      y2*z - 0.1113550912072001*v*w*y2*z + 0.023341335158146424*u*v*w*y2*
      z - 0.0036412369193938783*u2*v*w*y2*z + 0.0002913721656218036*u3*v*w*
      y2*z - 13.984210663495318*v2*w*y2*z - 4.27806313688087*lnbara*v2*w*
      y2*z + 1.059736505268818*u*v2*w*y2*z + 0.36692372038764914*lnbara*u*
      v2*w*y2*z - 0.7420694165018765*u2*v2*w*y2*z - 
     0.21620742554342368*lnbara*u2*v2*w*y2*z + 3.420468192644727*v3*w*y2*
      z + 1.0593983869002586*lnbara*v3*w*y2*z + 0.3069925110386899*u*v3*w*
      y2*z + 0.0810097794355468*lnbara*u*v3*w*y2*z - 
     1.1411946582446566*v4*w*y2*z - 0.33097644191339387*lnbara*v4*w*y2*
      z - 0.20624980265509574*w2*y2*z + 0.06378679623852483*lnbara*w2*y2*
      z - 2.5341893643030327*u*w2*y2*z - 0.8006228366412104*lnbara*u*w2*
      y2*z - 1.1971008100465077*u2*w2*y2*z - 0.3402159951898016*lnbara*
      u2*w2*y2*z - 0.17277027032117476*u3*w2*y2*z - 
     0.04766748743684271*lnbara*u3*w2*y2*z + 14.046191130880398*v*w2*y2*
      z + 4.27806313688087*lnbara*v*w2*y2*z - 1.0675924425094043*u*v*w2*
      y2*z - 0.36692372038764914*lnbara*u*v*w2*y2*z + 
     0.7426221290311992*u2*v*w2*y2*z + 0.21620742554342368*lnbara*u2*v*
      w2*y2*z - 0.005287248454508436*v2*w2*y2*z + 
     0.0003034691554580678*u*v2*w2*y2*z + 1.0839553144032048*v3*w2*y2*
      z + 0.2997685920100648*lnbara*v3*w2*y2*z - 
     2.770275796275451*w3*y2*z - 0.8710102224533081*lnbara*w3*y2*z + 
     2.2733271985354553*u*w3*y2*z + 0.6947964005893216*lnbara*u*w3*y2*z + 
     0.15265057835965268*u2*w3*y2*z + 0.04014533812336501*lnbara*u2*w3*
      y2*z - 3.431997185570922*v*w3*y2*z - 1.0593983869002586*lnbara*v*w3*
      y2*z - 0.3063351018757037*u*v*w3*y2*z - 0.0810097794355468*lnbara*u*
      v*w3*y2*z - 1.0830627587062924*v2*w3*y2*z - 
     0.2997685920100648*lnbara*v2*w3*y2*z + 0.305005920641606*w4*y2*z + 
     0.10788365521657699*lnbara*w4*y2*z - 0.3602833223629141*u*w4*y2*z - 
     0.10812164650482199*lnbara*u*w4*y2*z + 1.1422301498147045*v*w4*y2*
      z + 0.33097644191339387*lnbara*v*w4*y2*z - 0.11272620964583531*w5*
      y2*z - 0.03339748565253056*lnbara*w5*y2*z + 
     0.45246456501551086*x*y2*z - 0.1076866392526224*u*x*y2*z + 
     0.02581697089389268*u2*x*y2*z - 0.004447305079644873*u3*x*y2*z + 
     0.00038194899426036925*u4*x*y2*z - 0.1113550912072001*v*x*y2*z + 
     0.023341335158146424*u*v*x*y2*z - 0.0036412369193938783*u2*v*x*y2*z + 
     0.0002913721656218036*u3*v*x*y2*z - 8.691639367114691*v2*x*y2*z - 
     2.6585984883283214*lnbara*v2*x*y2*z + 4.260756983735999*u*v2*x*y2*
      z + 1.2888307504910574*lnbara*u*v2*x*y2*z - 
     0.6381316954939846*u2*v2*x*y2*z - 0.19013921429911454*lnbara*u2*v2*
      x*y2*z + 1.046392947967706*v3*x*y2*z + 0.35193692906048857*lnbara*
      v3*x*y2*z - 0.18025531671901043*u*v3*x*y2*z - 
     0.06303275716653367*lnbara*u*v3*x*y2*z - 0.6823722786029639*v4*x*y2*
      z - 0.19655624712804137*lnbara*v4*x*y2*z - 0.1113550912072001*w*x*y2*
      z + 0.023341335158146424*u*w*x*y2*z - 0.0036412369193938783*u2*w*x*
      y2*z + 0.0002913721656218036*u3*w*x*y2*z + 
     17.462617451739675*v*w*x*y2*z + 5.317196976656643*lnbara*v*w*x*y2*z - 
     8.532337360892159*u*v*w*x*y2*z - 2.577661500982115*lnbara*u*v*w*x*y2*
      z + 1.277050969854987*u2*v*w*x*y2*z + 0.3802784285982291*lnbara*u2*v*
      w*x*y2*z - 1.0551604965935144*v2*w*x*y2*z - 
     0.35193692906048857*lnbara*v2*w*x*y2*z + 0.18081274049740853*u*v2*w*x*
      y2*z + 0.06303275716653367*lnbara*u*v2*w*x*y2*z + 
     2.0929117260759176*v3*w*x*y2*z + 0.5923170046701335*lnbara*v3*w*x*y2*
      z - 8.691639367114691*w2*x*y2*z - 2.6585984883283214*lnbara*w2*x*y2*
      z + 4.260756983735999*u*w2*x*y2*z + 1.2888307504910574*lnbara*u*w2*x*
      y2*z - 0.6381316954939846*u2*w2*x*y2*z - 0.19013921429911454*lnbara*
      u2*w2*x*y2*z - 1.0551604965935144*v*w2*x*y2*z - 
     0.35193692906048857*lnbara*v*w2*x*y2*z + 0.18081274049740853*u*v*w2*x*
      y2*z + 0.06303275716653367*lnbara*u*v*w2*x*y2*z - 
     2.8192943376822845*v2*w2*x*y2*z - 0.7915215150841841*lnbara*v2*w2*x*
      y2*z + 1.046392947967706*w3*x*y2*z + 0.35193692906048857*lnbara*w3*
      x*y2*z - 0.18025531671901043*u*w3*x*y2*z - 
     0.06303275716653367*lnbara*u*w3*x*y2*z + 2.0929117260759176*v*w3*x*
      y2*z + 0.5923170046701335*lnbara*v*w3*x*y2*z - 
     0.6823722786029639*w4*x*y2*z - 0.19655624712804137*lnbara*w4*x*y2*
      z - 0.1974796353404678*x2*y2*z + 0.03069034292530752*u*x2*y2*z - 
     0.0042936520074251285*u2*x2*y2*z + 0.00032622599069967407*u3*x2*y2*
      z + 14.046191130880398*v*x2*y2*z + 4.27806313688087*lnbara*v*x2*y2*
      z - 1.0675924425094043*u*v*x2*y2*z - 0.36692372038764914*lnbara*u*v*
      x2*y2*z + 0.7426221290311992*u2*v*x2*y2*z + 
     0.21620742554342368*lnbara*u2*v*x2*y2*z - 1.4568456155246001*v2*x2*
      y2*z - 0.48533250050556614*lnbara*v2*x2*y2*z - 
     0.5429343918861453*u*v2*x2*y2*z - 0.1557593910339726*lnbara*u*v2*x2*
      y2*z + 1.492463738628857*v3*x2*y2*z + 0.4272793464214101*lnbara*v3*
      x2*y2*z - 13.984210663495318*w*x2*y2*z - 4.27806313688087*lnbara*w*
      x2*y2*z + 1.059736505268818*u*w*x2*y2*z + 
     0.36692372038764914*lnbara*u*w*x2*y2*z - 0.7420694165018765*u2*w*x2*
      y2*z - 0.21620742554342368*lnbara*u2*w*x2*y2*z - 
     1.0551604965935144*v*w*x2*y2*z - 0.35193692906048857*lnbara*v*w*x2*
      y2*z + 0.18081274049740853*u*v*w*x2*y2*z + 
     0.06303275716653367*lnbara*u*v*w*x2*y2*z - 3.271037633209231*v2*w*x2*
      y2*z - 0.9126427694703488*lnbara*v2*w*x2*y2*z + 
     2.499274797319209*w2*x2*y2*z + 0.8372694295660547*lnbara*w2*x2*y2*
      z + 0.36293415511749816*u*w2*x2*y2*z + 0.09272663386743893*lnbara*u*
      w2*x2*y2*z + 3.2714817035807937*v*w2*x2*y2*z + 
     0.9180803814343197*lnbara*v*w2*x2*y2*z - 1.4916718912285227*w3*x2*
      y2*z - 0.432716958385381*lnbara*w3*x2*y2*z + 
     0.06395157977258972*x3*y2*z - 0.005780394040453321*u*x3*y2*z + 
     0.0003568540057622896*u2*x3*y2*z - 3.431997185570922*v*x3*y2*z - 
     1.0593983869002586*lnbara*v*x3*y2*z - 0.3063351018757037*u*v*x3*y2*
      z - 0.0810097794355468*lnbara*u*v*x3*y2*z - 
     0.7046868197766329*v2*x3*y2*z - 0.18518041263910326*lnbara*v2*x3*
      y2*z + 3.420468192644727*w*x3*y2*z + 1.0593983869002586*lnbara*w*x3*
      y2*z + 0.3069925110386899*u*w*x3*y2*z + 0.0810097794355468*lnbara*u*
      w*x3*y2*z + 2.0929117260759176*v*w*x3*y2*z + 
     0.5923170046701335*lnbara*v*w*x3*y2*z - 1.3871873506084558*w2*x3*y2*
      z - 0.4071365920310302*lnbara*w2*x3*y2*z - 
     0.013136621448171954*x4*y2*z + 0.0005230003987145713*u*x4*y2*z + 
     1.1422301498147045*v*x4*y2*z + 0.33097644191339387*lnbara*v*x4*y2*
      z - 1.1411946582446566*w*x4*y2*z - 0.33097644191339387*lnbara*w*x4*
      y2*z + 0.001263472670057093*x5*y2*z + 0.6858555690183518*y3*z - 
     0.14397058356518277*u*y3*z + 0.03238162914677262*u2*y3*z - 
     0.0053911589121258706*u3*y3*z + 0.0004539037516950064*u4*y3*z - 
     6.543865639160677*v*y3*z - 2.0509188338532764*lnbara*v*y3*z - 
     1.5432862100326823*u*v*y3*z - 0.4192402591437163*lnbara*u*v*y3*z - 
     0.44329609772338113*u2*v*y3*z - 0.12608980116426613*lnbara*u2*v*y3*
      z - 0.050643992483851874*u3*v*y3*z - 0.013481184061483282*lnbara*u3*
      v*y3*z + 2.417035504771195*v2*y3*z + 0.7884496760703545*lnbara*v2*
      y3*z + 0.9375779785463527*u*v2*y3*z + 0.26779586684578455*lnbara*u*
      v2*y3*z + 0.16819984993563145*u2*v2*y3*z + 
     0.046821383490061465*lnbara*u2*v2*y3*z - 2.4157548822081845*v3*y3*
      z - 0.7222688114635434*lnbara*v3*y3*z - 0.10092543442474523*u*v3*y3*
      z - 0.025545381796987655*lnbara*u*v3*y3*z + 
     0.3249190900404754*v4*y3*z + 0.09733306860121325*lnbara*v4*y3*z + 
     6.221638137007376*w*y3*z + 2.0509188338532764*lnbara*w*y3*z + 
     1.6028488135384624*u*w*y3*z + 0.4192402591437163*lnbara*u*w*y3*z + 
     0.43459981748405074*u2*w*y3*z + 0.12608980116426613*lnbara*u2*w*y3*
      z + 0.05131419753175176*u3*w*y3*z + 0.013481184061483282*lnbara*u3*w*
      y3*z + 0.03580809539433375*v*w*y3*z - 0.004288309525390823*u*v*w*y3*
      z + 0.0002913721656218036*u2*v*w*y3*z + 4.23815521421725*v2*w*y3*
      z + 1.229468487533898*lnbara*v2*w*y3*z - 0.46795439814148526*u*v2*w*
      y3*z - 0.14200807750295125*lnbara*u*v2*w*y3*z - 
     0.21637657479006892*v3*w*y3*z - 0.06469446777014416*lnbara*v3*w*y3*
      z - 2.339296200037759*w2*y3*z - 0.7884496760703545*lnbara*w2*y3*z - 
     0.9468503021957163*u*w2*y3*z - 0.26779586684578455*lnbara*u*w2*y3*
      z - 0.16757153566204508*u2*w2*y3*z - 0.046821383490061465*lnbara*u2*
      w2*y3*z - 4.2497906793289655*v*w2*y3*z - 1.229468487533898*lnbara*v*
      w2*y3*z + 0.4686008650126944*u*v*w2*y3*z + 
     0.14200807750295125*lnbara*u*v*w2*y3*z + 0.0004489814737672761*v2*w2*
      y3*z + 2.402267061883944*w3*y3*z + 0.7222688114635434*lnbara*w3*y3*
      z + 0.10166794959991486*u*w3*y3*z + 0.025545381796987655*lnbara*u*w3*
      y3*z + 0.21733303969258025*v*w3*y3*z + 0.06469446777014416*lnbara*v*
      w3*y3*z - 0.3237542215275384*w4*y3*z - 0.09733306860121325*lnbara*
      w4*y3*z - 0.2298476323443832*x*y3*z + 0.0339939779841863*u*x*y3*z - 
     0.004611648226787489*u2*x*y3*z + 0.0003439536028536974*u3*x*y3*z + 
     0.03580809539433375*v*x*y3*z - 0.004288309525390823*u*v*x*y3*z + 
     0.0002913721656218036*u2*v*x*y3*z + 2.9845483465315206*v2*x*y3*z + 
     0.873539503307877*lnbara*v2*x*y3*z - 0.7548466365170377*u*v2*x*y3*
      z - 0.21880709774223123*lnbara*u*v2*x*y3*z - 
     0.07704797923780887*v3*x*y3*z - 0.028040321374660362*lnbara*v3*x*y3*
      z + 0.03580809539433375*w*x*y3*z - 0.004288309525390823*u*w*x*y3*z + 
     0.0002913721656218036*u2*w*x*y3*z - 5.983871189946358*v*w*x*y3*z - 
     1.747079006615754*lnbara*v*w*x*y3*z + 1.5105778324009722*u*v*w*x*y3*
      z + 0.43761419548446245*lnbara*u*v*w*x*y3*z + 
     0.07777103749659893*v2*w*x*y3*z + 0.028040321374660362*lnbara*v2*w*x*
      y3*z + 2.9845483465315206*w2*x*y3*z + 0.873539503307877*lnbara*w2*x*
      y3*z - 0.7548466365170377*u*w2*x*y3*z - 0.21880709774223123*lnbara*u*
      w2*x*y3*z + 0.07777103749659893*v*w2*x*y3*z + 
     0.028040321374660362*lnbara*v*w2*x*y3*z - 0.07704797923780887*w3*x*
      y3*z - 0.028040321374660362*lnbara*w3*x*y3*z + 
     0.06395157977258972*x2*y3*z - 0.005673921854933901*u*x2*y3*z + 
     0.00034533361905439837*u2*x2*y3*z - 4.2497906793289655*v*x2*y3*z - 
     1.229468487533898*lnbara*v*x2*y3*z + 0.4686008650126944*u*v*x2*y3*
      z + 0.14200807750295125*lnbara*u*v*x2*y3*z + 
     0.029077792864669175*v2*x2*y3*z + 0.011972114859594282*lnbara*v2*x2*
      y3*z + 4.23815521421725*w*x2*y3*z + 1.229468487533898*lnbara*w*x2*
      y3*z - 0.46795439814148526*u*w*x2*y3*z - 0.14200807750295125*lnbara*
      u*w*x2*y3*z + 0.07777103749659893*v*w*x2*y3*z + 
     0.028040321374660362*lnbara*v*w*x2*y3*z - 0.10578881199195397*w2*x2*
      y3*z - 0.04001243623425464*lnbara*w2*x2*y3*z - 
     0.011927105098099399*x3*y3*z + 0.00047310410784989136*u*x3*y3*z + 
     0.21733303969258025*v*x3*y3*z + 0.06469446777014416*lnbara*v*x3*y3*
      z - 0.21637657479006892*w*x3*y3*z - 0.06469446777014416*lnbara*w*x3*
      y3*z + 0.0010740897315456516*x4*y3*z - 0.386061737342387*y4*z + 
     0.04988531635163183*u*y4*z - 0.00631630321372675*u2*y4*z + 
     0.0004539037516950064*u3*y4*z + 2.8587812294103165*v*y4*z + 
     0.8545495141055319*lnbara*v*y4*z + 0.05785733009437446*u*v*y4*z + 
     0.004869163308206694*lnbara*u*v*y4*z - 0.01953851722470816*u2*v*y4*
      z - 0.005188045152882472*lnbara*u2*v*y4*z - 
     0.2759336414712597*v2*y4*z - 0.09277616733835867*lnbara*v2*y4*z + 
     0.04382088794664975*u*v2*y4*z + 0.013022653225572441*lnbara*u*v2*y4*
      z + 0.23650678939377565*v3*y4*z + 0.06767441939812754*lnbara*v3*y4*
      z - 2.7455486537995553*w*y4*z - 0.8545495141055319*lnbara*w*y4*z - 
     0.06972985855284485*u*w*y4*z - 0.004869163308206694*lnbara*u*w*y4*z + 
     0.02029024705940767*u2*w*y4*z + 0.005188045152882472*lnbara*u2*w*y4*
      z - 0.007320613539132591*v*w*y4*z + 0.0003820180212080083*u*v*w*y4*
      z - 0.747040617041446*v2*w*y4*z - 0.21034811390666633*lnbara*v2*w*
      y4*z + 0.260373564991028*w2*y4*z + 0.09277616733835867*lnbara*w2*
      y4*z - 0.04301078480890973*u*w2*y4*z - 0.013022653225572441*lnbara*u*
      w2*y4*z + 0.7480915272662646*v*w2*y4*z + 0.21034811390666633*lnbara*
      v*w2*y4*z - 0.2353317760308018*w3*y4*z - 0.06767441939812754*lnbara*
      w3*y4*z + 0.08203160229248424*x*y4*z - 0.006873088261070091*u*x*y4*
      z + 0.00040392715269129916*u2*x*y4*z - 0.007320613539132591*v*x*y4*
      z + 0.0003820180212080083*u*v*x*y4*z - 0.4807266928857714*v2*x*y4*
      z - 0.1356427800167511*lnbara*v2*x*y4*z - 0.007320613539132591*w*x*
      y4*z + 0.0003820180212080083*u*w*x*y4*z + 0.9627812643931486*v*w*x*
      y4*z + 0.2712855600335022*lnbara*v*w*x*y4*z - 
     0.4807266928857714*w2*x*y4*z - 0.1356427800167511*lnbara*w2*x*y4*z - 
     0.013136621448171954*x2*y4*z + 0.000507581743944034*u*x2*y4*z + 
     0.7480915272662646*v*x2*y4*z + 0.21034811390666633*lnbara*v*x2*y4*
      z - 0.747040617041446*w*x2*y4*z - 0.21034811390666633*lnbara*w*x2*
      y4*z + 0.0010740897315456516*x3*y4*z + 0.1475870262830398*y5*z - 
     0.01075135761798518*u*y5*z + 0.0005880488320726199*u2*y5*z - 
     0.9314087991903407*v*y5*z - 0.26857270443316716*lnbara*v*y5*z - 
     0.038965576694794626*u*v*y5*z - 0.009316737180946303*lnbara*u*v*y5*z + 
     0.06988272110288132*v2*y5*z + 0.021403950235092197*lnbara*v2*y5*z + 
     0.9067948266532218*w*y5*z + 0.26857270443316716*lnbara*w*y5*z + 
     0.04008569352545048*u*w*y5*z + 0.009316737180946303*lnbara*u*w*y5*z + 
     0.0007017792893363362*v*w*y5*z - 0.06841032198704927*w2*y5*z - 
     0.021403950235092197*lnbara*w2*y5*z - 0.018000480690589078*x*y5*z + 
     0.0006541741874990681*u*x*y5*z + 0.0007017792893363362*v*x*y5*z + 
     0.0007017792893363362*w*x*y5*z + 0.001263472670057093*x2*y5*z - 
     0.0339914082143444*y6*z + 0.0010711545725659018*u*y6*z + 
     0.11490665548248956*v*y6*z + 0.032045606778957444*lnbara*v*y6*z - 
     0.11244046513485796*w*y6*z - 0.032045606778957444*lnbara*w*y6*z + 
     0.0018148095406981609*x*y6*z + 0.0035512018991495737*y7*z + 
     1.8468559937823905*z2 - 0.9004176968697725*u*z2 + 
     0.520113697995963*u2*z2 - 0.251564804320496*u3*z2 + 
     0.08755029317854246*u4*z2 - 0.018929896217160017*u5*z2 + 
     0.0018904202297008643*u6*z2 + 1.7490619043784765*v*z2 + 
     0.9115194817125674*lnbara*v*z2 + 6.04138458368625*u*v*z2 + 
     1.9748773458715838*lnbara*u*v*z2 + 4.3302489692293955*u2*v*z2 + 
     1.3823733765239528*lnbara*u2*v*z2 + 1.7772082042508452*u3*v*z2 + 
     0.5107892746111199*lnbara*u3*v*z2 + 0.33555238697393286*u4*v*z2 + 
     0.09558272829689277*lnbara*u4*v*z2 + 0.028399509890409173*u5*v*z2 + 
     0.007442787768918609*lnbara*u5*v*z2 + 2.7099088373198637*v2*z2 + 
     0.653288241029598*lnbara*v2*z2 - 18.2664360726891*u*v2*z2 - 
     6.062567347027013*lnbara*u*v2*z2 + 2.165326096275261*u2*v2*z2 + 
     0.7557728528917382*lnbara*u2*v2*z2 - 3.7882789087219217*u3*v2*z2 - 
     1.1472315670280682*lnbara*u3*v2*z2 + 0.2761865963929522*u4*v2*z2 + 
     0.08549461963730814*lnbara*u4*v2*z2 + 0.8972500249744643*v3*z2 + 
     0.3500794053947276*lnbara*v3*z2 + 7.880191664548825*u*v3*z2 + 
     2.6748784087603763*lnbara*u*v3*z2 + 0.9841419128221275*u2*v3*z2 + 
     0.23760579052794478*lnbara*u2*v3*z2 + 0.43479804170814224*u3*v3*
      z2 + 0.1344756922786936*lnbara*u3*v3*z2 + 
     0.651338693005845*v4*z2 + 0.14754669762714998*lnbara*v4*z2 - 
     3.7835481298107045*u*v4*z2 - 1.1938747004904564*lnbara*u*v4*z2 + 
     0.05725447402071905*u2*v4*z2 + 0.021576912362776347*lnbara*u2*v4*
      z2 + 0.03085436595372382*v5*z2 + 0.016119245278351776*lnbara*v5*
      z2 + 0.47543762922904986*u*v5*z2 + 0.14922477962881076*lnbara*u*v5*
      z2 + 0.013912167390291797*v6*z2 + 0.0027513912903176212*lnbara*v6*
      z2 - 3.3276691614032563*w*z2 - 0.9115194817125674*lnbara*w*z2 - 
     5.289736299142919*u*w*z2 - 1.9748773458715838*lnbara*u*w*z2 - 
     4.650086557902906*u2*w*z2 - 1.3823733765239528*lnbara*u2*w*z2 - 
     1.6750109905361525*u3*w*z2 - 0.5107892746111199*lnbara*u3*w*z2 - 
     0.3563815319580459*u4*w*z2 - 0.09558272829689277*lnbara*u4*w*z2 - 
     0.026406119658478486*u5*w*z2 - 0.007442787768918609*lnbara*u5*w*z2 - 
     10.763863509478952*v*w*z2 - 3.7979978404690304*lnbara*v*w*z2 + 
     14.408707884242522*u*v*w*z2 + 4.851483174726708*lnbara*u*v*w*z2 - 
     11.35352724075799*u2*v*w*z2 - 3.638468221673086*lnbara*u2*v*w*z2 + 
     3.595212268118998*u3*v*w*z2 + 1.124626001903981*lnbara*u3*v*w*z2 - 
     0.637464961753253*u4*v*w*z2 - 0.19025408565816512*lnbara*u4*v*w*z2 + 
     6.988719566598391*v2*w*z2 + 2.737479438521948*lnbara*v2*w*z2 + 
     2.583201962578535*u*v2*w*z2 + 0.6303616509938534*lnbara*u*v2*w*z2 + 
     2.78012177163788*u2*v2*w*z2 + 0.9193789312609476*lnbara*u2*v2*w*
      z2 + 0.0944959815922243*u3*v2*w*z2 + 0.012699061790125279*lnbara*
      u3*v2*w*z2 - 9.326716278242827*v3*w*z2 - 
     3.107847444868102*lnbara*v3*w*z2 + 2.4434643394917717*u*v3*w*z2 + 
     0.722704550671465*lnbara*u*v3*w*z2 - 1.4446667880342972*u2*v3*w*
      z2 - 0.43070638308913284*lnbara*u2*v3*w*z2 + 
     2.2331993759180806*v4*w*z2 + 0.7744184574067065*lnbara*v4*w*z2 + 
     0.08267922567088919*u*v4*w*z2 + 0.01813757575979942*lnbara*u*v4*w*
      z2 - 0.46033723691317213*v5*w*z2 - 0.14606708002401708*lnbara*v5*w*
      z2 + 9.22032358478725*w2*z2 + 3.1447095994394325*lnbara*w2*z2 + 
     3.443135626202991*u*w2*z2 + 1.2110841723003059*lnbara*u*w2*z2 + 
     9.306085458851838*u2*w2*z2 + 2.882695368781348*lnbara*u2*w2*z2 + 
     0.170862170132498*u3*w2*z2 + 0.022605565124086877*lnbara*u3*w2*
      z2 + 0.363288554298664*u4*w2*z2 + 0.10475946602085698*lnbara*u4*
      w2*z2 - 2.5057639727820775*v*w2*z2 - 1.0258918950424711*lnbara*v*
      w2*z2 - 4.04603639452907*u*v*w2*z2 - 1.3032259701999138*lnbara*u*v*
      w2*z2 - 0.5438356537989508*u2*v*w2*z2 - 0.1818291572693978*lnbara*
      u2*v*w2*z2 - 0.17072856838087597*u3*v*w2*z2 - 
     0.04904808595029047*lnbara*u3*v*w2*z2 + 8.25179929321027*v2*w2*
      z2 + 2.469212605593015*lnbara*v2*w2*z2 - 8.325060940884864*u*v2*
      w2*z2 - 2.4651692331377286*lnbara*u*v2*w2*z2 + 
     1.4020259571353741*u2*v2*w2*z2 + 0.40905101643700437*lnbara*u2*v2*
      w2*z2 - 0.2730340390527256*v3*w2*z2 - 0.09631911852193811*lnbara*
      v3*w2*z2 + 0.5346542533590023*u*v3*w2*z2 + 
     0.17261962424250146*lnbara*u*v3*w2*z2 + 0.3702307156554167*v4*w2*
      z2 + 0.10136840292874362*lnbara*v4*w2*z2 - 
     6.021984802397401*w3*z2 - 2.0616669488742043*lnbara*w3*z2 - 
     6.264516709859727*u*w3*z2 - 2.002014089554316*lnbara*u*w3*z2 - 
     3.2461502369868342*u2*w3*z2 - 0.9751555645194946*lnbara*u2*w3*z2 - 
     0.35640766524496237*u3*w3*z2 - 0.09812666811852842*lnbara*u3*w3*
      z2 - 4.961706764789829*v*w3*z2 - 1.3124617908295524*lnbara*v*w3*
      z2 + 8.457223146401986*u*v*w3*z2 + 2.5565136487937905*lnbara*u*v*w3*
      z2 - 0.9782558475006098*u2*v*w3*z2 - 0.28088060701825807*lnbara*u2*
      v*w3*z2 - 0.4179222450177246*v2*w3*z2 - 0.11149384692877298*lnbara*
      v2*w3*z2 + 0.3092023748907339*u*v2*w3*z2 + 
     0.0785869636098416*lnbara*u*v2*w3*z2 - 0.6281581353908926*v3*w3*
      z2 - 0.1774903697854292*lnbara*v3*w3*z2 + 
     5.626060910791232*w4*z2 + 1.8035499324774895*lnbara*w4*z2 + 
     1.1739473090123895*u*w4*z2 + 0.3798257341629295*lnbara*u*w4*z2 + 
     0.9661938550317448*u2*w4*z2 + 0.2809590613076102*lnbara*u2*w4*z2 + 
     0.01586781553997586*v*w4*z2 - 0.06352181102996676*lnbara*v*w4*z2 - 
     1.0816608287503775*u*v*w4*z2 - 0.32438223975151587*lnbara*u*v*w4*
      z2 + 0.581863351202302*v2*w4*z2 + 0.1659602153670836*lnbara*v2*w4*
      z2 - 1.6435396981777775*w5*z2 - 0.5192029262043804*lnbara*w5*z2 - 
     0.31687310350773107*u*w5*z2 - 0.09418670348943739*lnbara*u*w5*z2 - 
     0.14867177025989595*v*w5*z2 - 0.030113006529747105*lnbara*v*w5*z2 + 
     0.2767842038860737*w6*z2 + 0.08359044675304854*lnbara*w6*z2 - 
     0.9004176968697725*x*z2 + 0.3313625873896489*u*x*z2 - 
     0.12887042405432217*u2*x*z2 + 0.03949367079133293*u3*x*z2 - 
     0.007867552944290607*u4*x*z2 + 0.0007424815539584533*u5*x*z2 - 
     16.220273175775272*v*x*z2 - 5.75015242423565*lnbara*v*x*z2 + 
     7.283613291269799*u*v*x*z2 + 2.569763477061043*lnbara*u*v*x*z2 - 
     10.868438804341743*u2*v*x*z2 - 3.4712775032258807*lnbara*u2*v*x*z2 + 
     2.723658999110921*u3*v*x*z2 + 0.8563693554764021*lnbara*u3*v*x*z2 - 
     0.5498436699327358*u4*v*x*z2 - 0.16402829156973656*lnbara*u4*v*x*
      z2 + 6.988719566598391*v2*x*z2 + 2.737479438521948*lnbara*v2*x*
      z2 + 2.583201962578535*u*v2*x*z2 + 0.6303616509938534*lnbara*u*v2*x*
      z2 + 2.78012177163788*u2*v2*x*z2 + 0.9193789312609476*lnbara*u2*
      v2*x*z2 + 0.0944959815922243*u3*v2*x*z2 + 
     0.012699061790125279*lnbara*u3*v2*x*z2 - 10.997725885446338*v3*x*
      z2 - 3.557746295983214*lnbara*v3*x*z2 + 2.9792615221240184*u*v3*x*
      z2 + 0.9000686333403095*lnbara*u*v3*x*z2 - 
     1.8066447210612555*u2*v3*x*z2 - 0.5331767926648573*lnbara*u2*v3*x*
      z2 + 2.4334094721170496*v4*x*z2 + 0.816632950371704*lnbara*v4*x*
      z2 + 0.10307841593891565*u*v4*x*z2 + 0.018172273278693238*lnbara*u*
      v4*x*z2 - 0.5247428948621341*v5*x*z2 - 0.16081753949969196*lnbara*
      v5*x*z2 + 16.840740404671994*w*x*z2 + 5.75015242423565*lnbara*w*x*
      z2 - 7.4944534531230955*u*w*x*z2 - 2.569763477061043*lnbara*u*w*x*
      z2 + 10.92704206771195*u2*w*x*z2 + 3.4712775032258807*lnbara*u2*w*x*
      z2 - 2.7345557618171856*u3*w*x*z2 - 0.8563693554764021*lnbara*u3*w*
      x*z2 + 0.5508222899910821*u4*w*x*z2 + 0.16402829156973656*lnbara*u4*
      w*x*z2 - 4.808822770349789*v*w*x*z2 - 1.7115875434794767*lnbara*v*w*x*
      z2 + 1.5426830809278815*u*v*w*x*z2 + 0.6728643192060604*lnbara*u*v*w*
      x*z2 - 2.2498626331415124*u2*v*w*x*z2 - 0.7375497739915498*lnbara*
      u2*v*w*x*z2 + 0.07737638458711994*u3*v*w*x*z2 + 
     0.03634902416016519*lnbara*u3*v*w*x*z2 + 21.55337156258267*v2*w*x*
      z2 + 6.548245980022291*lnbara*v2*w*x*z2 - 14.057135775486067*u*v2*w*
      x*z2 - 4.266687153487787*lnbara*u*v2*w*x*z2 + 
     3.4982098333445135*u2*v2*w*x*z2 + 1.0191704113395068*lnbara*u2*v2*w*
      x*z2 - 2.1068706790103*v3*w*x*z2 - 0.6955343577421946*lnbara*v3*w*x*
      z2 + 0.8761362502114731*u*v3*w*x*z2 + 0.296376278878668*lnbara*u*v3*
      w*x*z2 + 1.0094259951122324*v4*w*x*z2 + 0.2874207900901255*lnbara*
      v4*w*x*z2 - 2.5057639727820775*w2*x*z2 - 1.0258918950424711*lnbara*
      w2*x*z2 - 4.04603639452907*u*w2*x*z2 - 1.3032259701999138*lnbara*u*
      w2*x*z2 - 0.5438356537989508*u2*w2*x*z2 - 
     0.1818291572693978*lnbara*u2*w2*x*z2 - 0.17072856838087597*u3*w2*x*
      z2 - 0.04904808595029047*lnbara*u3*w2*x*z2 - 
     20.346437220092437*v*w2*x*z2 - 6.08173812414508*lnbara*v*w2*x*z2 + 
     16.365226923917742*u*v*w2*x*z2 + 4.905952869158358*lnbara*u*v*w2*x*
      z2 - 3.2577140116251813*u2*v*w2*x*z2 - 0.9439400410182851*lnbara*
      u2*v*w2*x*z2 - 0.40857845580650054*v2*w2*x*z2 - 
     0.08307962011998349*lnbara*v2*w2*x*z2 - 0.5491803784646733*u*v2*w2*
      x*z2 - 0.17704276038055036*lnbara*u*v2*w2*x*z2 - 
     1.7437824185418067*v3*w2*x*z2 - 0.4914395306435913*lnbara*v3*w2*x*
      z2 + 9.909152497593267*w3*x*z2 + 3.0912384401060025*lnbara*w3*x*
      z2 - 5.305139799956746*u*w3*x*z2 - 1.5393343490108804*lnbara*u*w3*x*
      z2 + 1.5675237969406215*u2*w3*x*z2 + 0.4579464223436356*lnbara*u2*
      w3*x*z2 + 1.4784816196761565*v*w3*x*z2 + 0.43230388098155775*lnbara*
      v*w3*x*z2 - 0.6938884585226485*u*v*w3*x*z2 - 
     0.21809446900966986*lnbara*u*v*w3*x*z2 + 2.1080028169579204*v2*w3*x*
      z2 + 0.5953996603124057*lnbara*v2*w3*x*z2 - 
     1.4226442649834423*w4*x*z2 - 0.4703228534910837*lnbara*w4*x*z2 + 
     0.2656466906715195*u*w4*x*z2 + 0.08058867723285898*lnbara*u*w4*x*
      z2 - 1.267405871830795*v*w4*x*z2 - 0.35480949331667067*lnbara*v*w4*
      x*z2 + 0.4211542429001266*w5*x*z2 + 0.12424611305742275*lnbara*w5*x*
      z2 + 0.520113697995963*x2*z2 - 0.12887042405432217*u*x2*z2 + 
     0.03186642256716374*u2*x2*z2 - 0.00560074458687052*u3*x2*z2 + 
     0.000487363420148127*u4*x2*z2 - 0.1223579533330431*v*x2*z2 + 
     0.027313138910434232*u*v*x2*z2 - 0.004442047950830934*u2*v*x2*z2 + 
     0.0003652020446131235*u3*v*x2*z2 + 8.25179929321027*v2*x2*z2 + 
     2.469212605593015*lnbara*v2*x2*z2 - 8.325060940884864*u*v2*x2*z2 - 
     2.4651692331377286*lnbara*u*v2*x2*z2 + 1.4020259571353741*u2*v2*x2*
      z2 + 0.40905101643700437*lnbara*u2*v2*x2*z2 + 
     0.3997513066521812*v3*x2*z2 + 0.08307962011998349*lnbara*v3*x2*
      z2 + 0.5497959040120096*u*v3*x2*z2 + 0.17704276038055036*lnbara*u*
      v3*x2*z2 + 0.39505594064956606*v4*x2*z2 + 
     0.11018942250071202*lnbara*v4*x2*z2 - 0.1223579533330431*w*x2*z2 + 
     0.027313138910434232*u*w*x2*z2 - 0.004442047950830934*u2*w*x2*z2 + 
     0.0003652020446131235*u3*w*x2*z2 - 16.42034107834173*v*w*x2*z2 - 
     4.93842521118603*lnbara*v*w*x2*z2 + 16.637816190951657*u*v*w*x2*z2 + 
     4.930338466275457*lnbara*u*v*w*x2*z2 - 2.803109839150323*u2*v*w*x2*
      z2 - 0.8181020328740087*lnbara*u2*v*w*x2*z2 - 
     0.40857845580650054*v2*w*x2*z2 - 0.08307962011998349*lnbara*v2*w*x2*
      z2 - 0.5491803784646733*u*v2*w*x2*z2 - 0.17704276038055036*lnbara*u*
      v2*w*x2*z2 - 1.7364430725838955*v3*w*x2*z2 - 
     0.4913145532522245*lnbara*v3*w*x2*z2 + 8.25179929321027*w2*x2*z2 + 
     2.469212605593015*lnbara*w2*x2*z2 - 8.325060940884864*u*w2*x2*z2 - 
     2.4651692331377286*lnbara*u*w2*x2*z2 + 1.4020259571353741*u2*w2*x2*
      z2 + 0.40905101643700437*lnbara*u2*w2*x2*z2 - 
     0.40857845580650054*v*w2*x2*z2 - 0.08307962011998349*lnbara*v*w2*x2*
      z2 - 0.5491803784646733*u*v*w2*x2*z2 - 0.17704276038055036*lnbara*u*
      v*w2*x2*z2 + 2.684507561107731*v2*w2*x2*z2 + 
     0.7622502615030249*lnbara*v2*w2*x2*z2 + 0.3997513066521812*w3*x2*
      z2 + 0.08307962011998349*lnbara*w3*x2*z2 + 
     0.5497959040120096*u*w3*x2*z2 + 0.17704276038055036*lnbara*u*w3*x2*
      z2 - 1.7364430725838955*v*w3*x2*z2 - 0.4913145532522245*lnbara*v*
      w3*x2*z2 + 0.39505594064956606*w4*x2*z2 + 
     0.11018942250071202*lnbara*w4*x2*z2 - 0.251564804320496*x3*z2 + 
     0.03949367079133293*u*x3*z2 - 0.00560074458687052*u2*x3*z2 + 
     0.0004300153542944288*u3*x3*z2 - 6.2616857203946035*v*x3*z2 - 
     1.8196275948475404*lnbara*v*x3*z2 + 3.5848494417836383*u*v*x3*z2 + 
     1.0856448829578143*lnbara*u*v*x3*z2 - 0.660951971257608*u2*v*x3*
      z2 - 0.18951332857164516*lnbara*u2*v*x3*z2 - 
     0.4179222450177246*v2*x3*z2 - 0.11149384692877298*lnbara*v2*x3*
      z2 + 0.3092023748907339*u*v2*x3*z2 + 0.0785869636098416*lnbara*u*
      v2*x3*z2 - 0.8243258774524206*v3*x3*z2 - 
     0.23041500895027525*lnbara*v3*x3*z2 + 6.336956677089462*w*x3*z2 + 
     1.8196275948475404*lnbara*w*x3*z2 - 3.5946630672038005*u*w*x3*z2 - 
     1.0856448829578143*lnbara*u*w*x3*z2 + 0.6616579679935063*u2*w*x3*
      z2 + 0.18951332857164516*lnbara*u2*w*x3*z2 + 
     0.6761564834039746*v*w*x3*z2 + 0.2078129654507111*lnbara*v*w*x3*z2 - 
     0.8428720597864838*u*v*w*x3*z2 - 0.25120658785234307*lnbara*u*v*w*x3*
      z2 + 2.1080028169579204*v2*w*x3*z2 + 0.5953996603124057*lnbara*v2*
      w*x3*z2 - 0.2730340390527256*w2*x3*z2 - 0.09631911852193811*lnbara*
      w2*x3*z2 + 0.5346542533590023*u*w2*x3*z2 + 
     0.17261962424250146*lnbara*u*w2*x3*z2 - 1.7437824185418067*v*w2*x3*
      z2 - 0.4914395306435913*lnbara*v*w2*x3*z2 + 
     0.461488349439593*w3*x3*z2 + 0.12645487928146082*lnbara*w3*x3*z2 + 
     0.08755029317854246*x4*z2 - 0.007867552944290607*u*x4*z2 + 
     0.000487363420148127*u2*x4*z2 + 0.7756766725864659*v*x4*z2 + 
     0.2013783664993635*lnbara*v*x4*z2 - 0.3272951320069847*u*v*x4*z2 - 
     0.09992430356434646*lnbara*u*v*x4*z2 + 0.581863351202302*v2*x4*z2 + 
     0.1659602153670836*lnbara*v2*x4*z2 - 0.7906870068555788*w*x4*z2 - 
     0.2013783664993635*lnbara*w*x4*z2 + 0.3281593662661477*u*w*x4*z2 + 
     0.09992430356434646*lnbara*u*w*x4*z2 - 0.9508002022690504*v*w*x4*
      z2 - 0.2673286182958272*lnbara*v*w*x4*z2 + 
     0.3702307156554167*w2*x4*z2 + 0.10136840292874362*lnbara*w2*x4*
      z2 - 0.018929896217160017*x5*z2 + 0.0007424815539584533*u*x5*z2 - 
     0.22871819549961922*v*x5*z2 - 0.05887745268544395*lnbara*v*x5*z2 + 
     0.2301351338116376*w*x5*z2 + 0.05887745268544395*lnbara*w*x5*z2 + 
     0.0018904202297008643*x6*z2 - 0.9004176968697725*y*z2 + 
     0.3212835093689003*u*y*z2 - 0.1223579533330431*u2*y*z2 + 
     0.03702532275275472*u3*y*z2 - 0.007317810733171885*u4*y*z2 + 
     0.0006870207501448052*u5*y*z2 + 8.371127296661705*v*y*z2 + 
     3.007965370948085*lnbara*v*y*z2 + 8.142878738630774*u*v*y*z2 + 
     2.570302308603461*lnbara*u*v*y*z2 + 4.372397515146079*u2*v*y*z2 + 
     1.303884138860003*lnbara*u2*v*y*z2 + 1.089630311725629*u3*v*y*z2 + 
     0.306972893835514*lnbara*u3*v*y*z2 + 0.11045275181230746*u4*v*y*z2 + 
     0.029980593819149505*lnbara*u4*v*y*z2 - 11.93413855963392*v2*y*z2 - 
     4.1260209265036405*lnbara*v2*y*z2 - 1.9788978869332372*u*v2*y*z2 - 
     0.6338374192524003*lnbara*u*v2*y*z2 - 4.430910486593282*u2*v2*y*
      z2 - 1.3342436103888362*lnbara*u2*v2*y*z2 - 
     0.1142476902831892*u3*v2*y*z2 - 0.0269991887735768*lnbara*u3*v2*y*
      z2 + 5.769076336816564*v3*y*z2 + 2.0547100979131496*lnbara*v3*y*
      z2 + 2.371701525434619*u*v3*y*z2 + 0.6890237976926594*lnbara*u*v3*y*
      z2 + 0.729201859518042*u2*v3*y*z2 + 0.21972319960553072*lnbara*u2*
      v3*y*z2 - 2.872858759697617*v4*y*z2 - 0.9340060015696775*lnbara*v4*
      y*z2 - 0.13345592306181395*u*v4*y*z2 - 0.03997846452824747*lnbara*u*
      v4*y*z2 + 0.3612787107357604*v5*y*z2 + 0.11818205016461883*lnbara*
      v5*y*z2 - 7.740580989744235*w*y*z2 - 3.007965370948085*lnbara*w*y*
      z2 - 8.34831676538791*u*w*y*z2 - 2.570302308603461*lnbara*u*w*y*z2 - 
     4.3167789899540825*u2*w*y*z2 - 1.303884138860003*lnbara*u2*w*y*z2 - 
     1.0998046234447676*u3*w*y*z2 - 0.306972893835514*lnbara*u3*w*y*z2 - 
     0.10954859095754421*u4*w*y*z2 - 0.029980593819149505*lnbara*u4*w*y*
      z2 + 10.777349982686436*v*w*y*z2 + 3.494158013231508*lnbara*v*w*y*
      z2 - 14.680040084881753*u*v*w*y*z2 - 4.586055804009709*lnbara*u*v*w*y*
      z2 + 2.978128785284849*u2*v*w*y*z2 + 0.9528593416680298*lnbara*u2*v*
      w*y*z2 - 0.7405676776348092*u3*v*w*y*z2 - 0.21821356263065994*lnbara*
      u3*v*w*y*z2 - 0.6330043203094217*v2*w*y*z2 - 
     0.3326899352140709*lnbara*v2*w*y*z2 + 3.7124326039755955*u*v2*w*y*
      z2 + 1.229224903218318*lnbara*u*v2*w*y*z2 + 
     0.3880630632568426*u2*v2*w*y*z2 + 0.0969009664171316*lnbara*u2*v2*w*
      y*z2 + 3.1903919237729643*v3*w*y*z2 + 0.9222524033868401*lnbara*v3*
      w*y*z2 - 2.203724440304821*u*v3*w*y*z2 - 0.6502064428550666*lnbara*u*
      v3*w*y*z2 - 0.06439562415041984*v4*w*y*z2 - 
     0.022967627645177263*lnbara*v4*w*y*z2 + 0.8190067945965689*w2*y*z2 + 
     0.6318629132721324*lnbara*w2*y*z2 + 16.737595251903855*u*w2*y*z2 + 
     5.219893223262108*lnbara*u*w2*y*z2 + 1.43981530141774*u2*w2*y*z2 + 
     0.3813842687208064*lnbara*u2*w2*y*z2 + 0.8558867409043719*u3*w2*y*
      z2 + 0.24521275140423673*lnbara*u3*w2*y*z2 + 
     0.5638297481368869*v*w2*y*z2 + 0.18788462045702756*lnbara*v*w2*y*
      z2 + 0.008389015172307996*u*v*w2*y*z2 - 0.057042047293007624*lnbara*
      u*v*w2*y*z2 - 0.2581203920240232*u2*v*w2*y*z2 - 
     0.07638602271158683*lnbara*u2*v*w2*y*z2 - 6.896344257474369*v2*w2*y*
      z2 - 2.015976321936135*lnbara*v2*w2*y*z2 + 
     1.8892318082935522*u*v2*w2*y*z2 + 0.5393619159455898*lnbara*u*v2*w2*
      y*z2 + 0.501248699467762*v3*w2*y*z2 + 0.1627718847199586*lnbara*v3*
      w2*y*z2 - 5.574896354901597*w3*y*z2 - 1.909904783156106*lnbara*w3*
      y*z2 - 6.110264042573513*u*w3*y*z2 - 1.8612066536179699*lnbara*u*w3*
      y*z2 - 0.8578197999104673*u2*w3*y*z2 - 0.24023814331107549*lnbara*
      u2*w3*y*z2 + 5.676870448679409*v*w3*y*z2 + 
     1.684726260416403*lnbara*v*w3*y*z2 - 1.182692665484111*u*v*w3*y*z2 - 
     0.32670867639523105*lnbara*u*v*w3*y*z2 + 0.030715686889802068*v2*w3*
      y*z2 - 0.0026175022546954534*lnbara*v2*w3*y*z2 + 
     0.8738097966923795*w4*y*z2 + 0.34300365970256935*lnbara*w4*y*z2 + 
     1.632450766309338*u*w4*y*z2 + 0.47753166783295525*lnbara*u*w4*y*z2 - 
     0.566190372464217*v*w4*y*z2 - 0.1702185650157698*lnbara*v*w4*y*z2 - 
     0.2597697450829062*w5*y*z2 - 0.08515023996893491*lnbara*w5*y*z2 + 
     0.45246456501551086*x*y*z2 - 0.1113550912072001*u*x*y*z2 + 
     0.027313138910434232*u2*x*y*z2 - 0.004771933785158886*u3*x*y*z2 + 
     0.0004135240269207944*u4*x*y*z2 + 14.923465303571382*v*x*y*z2 + 
     4.700428217339612*lnbara*v*x*y*z2 - 8.636182740657008*u*v*x*y*z2 - 
     2.845063377679382*lnbara*u*v*x*y*z2 + 2.0292029954219255*u2*v*x*y*
      z2 + 0.6312081178376158*lnbara*u2*v*x*y*z2 - 
     0.5873630368597165*u3*v*x*y*z2 - 0.17392267685688417*lnbara*u3*v*x*y*
      z2 - 0.6330043203094217*v2*x*y*z2 - 0.3326899352140709*lnbara*v2*x*
      y*z2 + 3.7124326039755955*u*v2*x*y*z2 + 1.229224903218318*lnbara*u*
      v2*x*y*z2 + 0.3880630632568426*u2*v2*x*y*z2 + 
     0.0969009664171316*lnbara*u2*v2*x*y*z2 + 3.8502028793420484*v3*x*y*
      z2 + 1.1170305738043862*lnbara*v3*x*y*z2 - 
     2.127445819261271*u*v3*x*y*z2 - 0.6316147687241351*lnbara*u*v3*x*y*
      z2 - 0.09314373993962562*v4*x*y*z2 - 0.033291399122784844*lnbara*v4*
      x*y*z2 - 15.142507034031205*w*x*y*z2 - 4.700428217339612*lnbara*w*x*y*
      z2 + 8.684236515504576*u*w*x*y*z2 + 2.845063377679382*lnbara*u*w*x*y*
      z2 - 2.0369169970657963*u2*w*x*y*z2 - 0.6312081178376158*lnbara*u2*
      w*x*y*z2 + 0.5879915453536685*u3*w*x*y*z2 + 
     0.17392267685688417*lnbara*u3*w*x*y*z2 + 0.14564601713500816*v*w*x*y*
      z2 + 0.14480531475704334*lnbara*v*w*x*y*z2 - 
     3.731845090721665*u*v*w*x*y*z2 - 1.1721828559253105*lnbara*u*v*w*x*y*
      z2 - 0.12911307483747164*u2*v*w*x*y*z2 - 0.020514943705544776*lnbara*
      u2*v*w*x*y*z2 - 14.77406945239147*v2*w*x*y*z2 - 
     4.3242801796236945*lnbara*v2*w*x*y*z2 + 4.000620121386894*u*v2*w*x*y*
      z2 + 1.1582596990589664*lnbara*u*v2*w*x*y*z2 + 
     0.9014731642677928*v3*w*x*y*z2 + 0.28985320078620536*lnbara*v3*w*x*y*
      z2 + 0.5638297481368869*w2*x*y*z2 + 0.18788462045702756*lnbara*w2*x*
      y*z2 + 0.008389015172307996*u*w2*x*y*z2 - 0.057042047293007624*
      lnbara*u*w2*x*y*z2 - 0.2581203920240232*u2*w2*x*y*z2 - 
     0.07638602271158683*lnbara*u2*w2*x*y*z2 + 17.091071621436193*v*w2*x*
      y*z2 + 4.99601857372239*lnbara*v*w2*x*y*z2 - 
     3.489746635672796*u*v*w2*x*y*z2 - 1.0013349315415667*lnbara*u*v*w2*x*
      y*z2 - 0.2945882654409418*v2*w2*x*y*z2 - 0.10231487175528582*lnbara*
      v2*w2*x*y*z2 - 6.183783910974188*w3*x*y*z2 - 
     1.7887689679030814*lnbara*w3*x*y*z2 + 1.617693856062018*u*w3*x*y*
      z2 + 0.4746900012067356*lnbara*u*w3*x*y*z2 - 
     0.9801494429301945*v*w3*x*y*z2 - 0.293356436036497*lnbara*v*w3*x*y*
      z2 + 0.4680668096248443*w4*x*y*z2 + 0.13910950612836231*lnbara*w4*x*
      y*z2 - 0.1974796353404678*x2*y*z2 + 0.03129012445977296*u*x2*y*
      z2 - 0.004442047950830934*u2*x2*y*z2 + 0.0003407946912851918*u3*
      x2*y*z2 + 0.03069034292530752*v*x2*y*z2 - 
     0.003995614840649483*u*v*x2*y*z2 + 0.0002863764888716214*u2*v*x2*y*
      z2 - 6.896344257474369*v2*x2*y*z2 - 2.015976321936135*lnbara*v2*
      x2*y*z2 + 1.8892318082935522*u*v2*x2*y*z2 + 
     0.5393619159455898*lnbara*u*v2*x2*y*z2 + 0.2951846833599234*v3*x2*y*
      z2 + 0.10231487175528582*lnbara*v3*x2*y*z2 + 
     0.03069034292530752*w*x2*y*z2 - 0.003995614840649483*u*w*x2*y*z2 + 
     0.0002863764888716214*u2*w*x2*y*z2 + 13.78024088853395*v*w*x2*y*
      z2 + 4.03195264387227*lnbara*v*w*x2*y*z2 - 
     3.777642640073415*u*v*w*x2*y*z2 - 1.0787238318911796*lnbara*u*v*w*x2*
      y*z2 - 0.2945882654409418*v2*w*x2*y*z2 - 0.10231487175528582*lnbara*
      v2*w*x2*y*z2 - 6.896344257474369*w2*x2*y*z2 - 
     2.015976321936135*lnbara*w2*x2*y*z2 + 1.8892318082935522*u*w2*x2*y*
      z2 + 0.5393619159455898*lnbara*u*w2*x2*y*z2 - 
     0.2945882654409418*v*w2*x2*y*z2 - 0.10231487175528582*lnbara*v*w2*
      x2*y*z2 + 0.2951846833599234*w3*x2*y*z2 + 
     0.10231487175528582*lnbara*w3*x2*y*z2 + 0.06395157977258972*x3*y*
      z2 - 0.005855071071261368*u*x3*y*z2 + 0.0003652020446131235*u2*x3*
      y*z2 + 6.015319832123562*v*x3*y*z2 + 1.7632715691466476*lnbara*v*x3*
      y*z2 - 0.7304529490550218*u*v*x3*y*z2 - 0.21288835092447814*lnbara*u*
      v*x3*y*z2 + 0.030715686889802068*v2*x3*y*z2 - 
     0.0026175022546954534*lnbara*v2*x3*y*z2 - 6.026774148018949*w*x3*y*
      z2 - 1.7632715691466476*lnbara*w*x3*y*z2 + 
     0.731117931401122*u*w*x3*y*z2 + 0.21288835092447814*lnbara*u*w*x3*y*
      z2 - 0.5309427518886998*v*w*x3*y*z2 - 0.16015438246526315*lnbara*v*w*
      x3*y*z2 + 0.501248699467762*w2*x3*y*z2 + 
     0.1627718847199586*lnbara*w2*x3*y*z2 - 0.013136621448171954*x4*y*
      z2 + 0.000527909826103984*u*x4*y*z2 - 0.6626623249676024*v*x4*y*
      z2 - 0.19636416180684646*lnbara*v*x4*y*z2 + 
     0.663692907110261*w*x4*y*z2 + 0.19636416180684646*lnbara*w*x4*y*z2 + 
     0.001263472670057093*x5*y*z2 + 0.520113697995963*y2*z2 - 
     0.1223579533330431*u*y2*z2 + 0.029164662906062597*u2*y2*z2 - 
     0.0050083201180713645*u3*y2*z2 + 0.00042934540544391214*u4*y2*z2 - 
     0.044978574732269516*v*y2*z2 - 0.06378679623852483*lnbara*v*y2*z2 + 
     2.5864306854801558*u*v*y2*z2 + 0.8006228366412104*lnbara*u*v*y2*z2 + 
     1.1889883290028347*u2*v*y2*z2 + 0.3402159951898016*lnbara*u2*v*y2*
      z2 + 0.17341769222311584*u3*v*y2*z2 + 0.04766748743684271*lnbara*
      u3*v*y2*z2 + 2.216432092777231*v2*y2*z2 + 
     0.6745097408562837*lnbara*v2*y2*z2 - 4.417728806752933*u*v2*y2*
      z2 - 1.3282152666550597*lnbara*u*v2*y2*z2 - 
     0.18884186308954337*u2*v2*y2*z2 - 0.048449976878940454*lnbara*u2*
      v2*y2*z2 + 0.32522417184198954*v3*y2*z2 + 
     0.06989084748944815*lnbara*v3*y2*z2 + 0.6591783435173679*u*v3*y2*
      z2 + 0.20062980696935676*lnbara*u*v3*y2*z2 + 
     0.11854588979479715*v4*y2*z2 + 0.03415403184344883*lnbara*v4*y2*
      z2 - 0.20624980265509574*w*y2*z2 + 0.06378679623852483*lnbara*w*y2*
      z2 - 2.5341893643030327*u*w*y2*z2 - 0.8006228366412104*lnbara*u*w*
      y2*z2 - 1.1971008100465077*u2*w*y2*z2 - 0.3402159951898016*lnbara*
      u2*w*y2*z2 - 0.17277027032117476*u3*w*y2*z2 - 
     0.04766748743684271*lnbara*u3*w*y2*z2 - 8.691639367114691*v*w*y2*
      z2 - 2.6585984883283214*lnbara*v*w*y2*z2 + 
     4.260756983735999*u*v*w*y2*z2 + 1.2888307504910574*lnbara*u*v*w*y2*
      z2 - 0.6381316954939846*u2*v*w*y2*z2 - 0.19013921429911454*lnbara*
      u2*v*w*y2*z2 + 2.499274797319209*v2*w*y2*z2 + 
     0.8372694295660547*lnbara*v2*w*y2*z2 + 0.36293415511749816*u*v2*w*
      y2*z2 + 0.09272663386743893*lnbara*u*v2*w*y2*z2 - 
     1.3871873506084558*v3*w*y2*z2 - 0.4071365920310302*lnbara*v3*w*y2*
      z2 + 6.563551498721121*w2*y2*z2 + 1.9840887474720377*lnbara*w2*y2*
      z2 + 0.14499255922663437*u*w2*y2*z2 + 0.03938451616400227*lnbara*u*
      w2*y2*z2 + 0.8278420296899973*u2*w2*y2*z2 + 
     0.238589191178055*lnbara*u2*w2*y2*z2 - 1.4568456155246001*v*w2*y2*
      z2 - 0.48533250050556614*lnbara*v*w2*y2*z2 - 
     0.5429343918861453*u*v*w2*y2*z2 - 0.1557593910339726*lnbara*u*v*w2*
      y2*z2 + 1.4101267527346482*v2*w2*y2*z2 + 
     0.39576075754209206*lnbara*v2*w2*y2*z2 - 1.3869981182997964*w3*y2*
      z2 - 0.4218277765499367*lnbara*w3*y2*z2 - 
     0.4779542480040674*u*w3*y2*z2 - 0.13759704980282306*lnbara*u*w3*y2*
      z2 - 0.7046868197766329*v*w3*y2*z2 - 0.18518041263910326*lnbara*v*
      w3*y2*z2 + 0.5651566216606796*w4*y2*z2 + 
     0.16240221528459253*lnbara*w4*y2*z2 - 0.1974796353404678*x*y2*z2 + 
     0.03069034292530752*u*x*y2*z2 - 0.0042936520074251285*u2*x*y2*z2 + 
     0.00032622599069967407*u3*x*y2*z2 - 13.984210663495318*v*x*y2*z2 - 
     4.27806313688087*lnbara*v*x*y2*z2 + 1.059736505268818*u*v*x*y2*z2 + 
     0.36692372038764914*lnbara*u*v*x*y2*z2 - 0.7420694165018765*u2*v*x*
      y2*z2 - 0.21620742554342368*lnbara*u2*v*x*y2*z2 + 
     2.499274797319209*v2*x*y2*z2 + 0.8372694295660547*lnbara*v2*x*y2*
      z2 + 0.36293415511749816*u*v2*x*y2*z2 + 0.09272663386743893*lnbara*
      u*v2*x*y2*z2 - 1.4916718912285227*v3*x*y2*z2 - 
     0.432716958385381*lnbara*v3*x*y2*z2 + 14.046191130880398*w*x*y2*
      z2 + 4.27806313688087*lnbara*w*x*y2*z2 - 1.0675924425094043*u*w*x*
      y2*z2 - 0.36692372038764914*lnbara*u*w*x*y2*z2 + 
     0.7426221290311992*u2*w*x*y2*z2 + 0.21620742554342368*lnbara*u2*w*x*
      y2*z2 - 1.0551604965935144*v*w*x*y2*z2 - 0.35193692906048857*lnbara*
      v*w*x*y2*z2 + 0.18081274049740853*u*v*w*x*y2*z2 + 
     0.06303275716653367*lnbara*u*v*w*x*y2*z2 + 3.2714817035807937*v2*w*x*
      y2*z2 + 0.9180803814343197*lnbara*v2*w*x*y2*z2 - 
     1.4568456155246001*w2*x*y2*z2 - 0.48533250050556614*lnbara*w2*x*y2*
      z2 - 0.5429343918861453*u*w2*x*y2*z2 - 0.1557593910339726*lnbara*u*
      w2*x*y2*z2 - 3.271037633209231*v*w2*x*y2*z2 - 
     0.9126427694703488*lnbara*v*w2*x*y2*z2 + 1.492463738628857*w3*x*y2*
      z2 + 0.4272793464214101*lnbara*w3*x*y2*z2 + 
     0.05802930853773854*x2*y2*z2 - 0.005287248454508436*u*x2*y2*z2 + 
     0.00032784931577668774*u2*x2*y2*z2 - 0.005287248454508436*v*x2*y2*
      z2 + 0.0003034691554580678*u*v*x2*y2*z2 + 
     1.4101267527346482*v2*x2*y2*z2 + 0.39576075754209206*lnbara*v2*x2*
      y2*z2 - 0.005287248454508436*w*x2*y2*z2 + 
     0.0003034691554580678*u*w*x2*y2*z2 - 2.8192943376822845*v*w*x2*y2*
      z2 - 0.7915215150841841*lnbara*v*w*x2*y2*z2 + 
     1.4101267527346482*w2*x2*y2*z2 + 0.39576075754209206*lnbara*w2*x2*
      y2*z2 - 0.01113298574577242*x3*y2*z2 + 0.0004489814737672761*u*x3*
      y2*z2 - 1.0830627587062924*v*x3*y2*z2 - 0.2997685920100648*lnbara*
      v*x3*y2*z2 + 1.0839553144032048*w*x3*y2*z2 + 
     0.2997685920100648*lnbara*w*x3*y2*z2 + 0.001019248467243472*x4*y2*
      z2 - 0.251564804320496*y3*z2 + 0.03702532275275472*u*y3*z2 - 
     0.0050083201180713645*u2*y3*z2 + 0.0003728851239061164*u3*y3*z2 + 
     2.417035504771195*v*y3*z2 + 0.7884496760703545*lnbara*v*y3*z2 + 
     0.9375779785463527*u*v*y3*z2 + 0.26779586684578455*lnbara*u*v*y3*
      z2 + 0.16819984993563145*u2*v*y3*z2 + 0.046821383490061465*lnbara*
      u2*v*y3*z2 - 2.321676668163961*v2*y3*z2 - 
     0.704896053042102*lnbara*v2*y3*z2 + 0.038158053864272005*u*v2*y3*
      z2 + 0.011961036953974104*lnbara*u*v2*y3*z2 + 
     0.28220067646190156*v3*y3*z2 + 0.09025951520398619*lnbara*v3*y3*
      z2 - 2.339296200037759*w*y3*z2 - 0.7884496760703545*lnbara*w*y3*
      z2 - 0.9468503021957163*u*w*y3*z2 - 0.26779586684578455*lnbara*u*w*
      y3*z2 - 0.16757153566204508*u2*w*y3*z2 - 
     0.046821383490061465*lnbara*u2*w*y3*z2 + 2.9845483465315206*v*w*y3*
      z2 + 0.873539503307877*lnbara*v*w*y3*z2 - 0.7548466365170377*u*v*w*
      y3*z2 - 0.21880709774223123*lnbara*u*v*w*y3*z2 - 
     0.10578881199195397*v2*w*y3*z2 - 0.04001243623425464*lnbara*v2*w*y3*
      z2 - 0.6788052052736328*w2*y3*z2 - 0.16864345026577507*lnbara*w2*
      y3*z2 + 0.717639217730022*u*w2*y3*z2 + 0.20684606078825712*lnbara*
      u*w2*y3*z2 + 0.029077792864669175*v*w2*y3*z2 + 
     0.011972114859594282*lnbara*v*w2*y3*z2 - 0.20393804085263473*w3*y3*
      z2 - 0.06221919382932583*lnbara*w3*y3*z2 + 
     0.06395157977258972*x*y3*z2 - 0.005673921854933901*u*x*y3*z2 + 
     0.00034533361905439837*u2*x*y3*z2 + 4.23815521421725*v*x*y3*z2 + 
     1.229468487533898*lnbara*v*x*y3*z2 - 0.46795439814148526*u*v*x*y3*
      z2 - 0.14200807750295125*lnbara*u*v*x*y3*z2 - 
     0.10578881199195397*v2*x*y3*z2 - 0.04001243623425464*lnbara*v2*x*y3*
      z2 - 4.2497906793289655*w*x*y3*z2 - 1.229468487533898*lnbara*w*x*y3*
      z2 + 0.4686008650126944*u*w*x*y3*z2 + 0.14200807750295125*lnbara*u*w*
      x*y3*z2 + 0.07777103749659893*v*w*x*y3*z2 + 
     0.028040321374660362*lnbara*v*w*x*y3*z2 + 0.029077792864669175*w2*x*
      y3*z2 + 0.011972114859594282*lnbara*w2*x*y3*z2 - 
     0.01113298574577242*x2*y3*z2 + 0.00044357422314508923*u*x2*y3*z2 + 
     0.0004489814737672761*v*x2*y3*z2 + 0.0004489814737672761*w*x2*y3*
      z2 + 0.0009524091176864817*x3*y3*z2 + 0.08755029317854246*y4*z2 - 
     0.007317810733171885*u*y4*z2 + 0.00042934540544391214*u2*y4*z2 - 
     0.2759336414712597*v*y4*z2 - 0.09277616733835867*lnbara*v*y4*z2 + 
     0.04382088794664975*u*v*y4*z2 + 0.013022653225572441*lnbara*u*v*y4*
      z2 + 0.20047304487804693*v2*y4*z2 + 0.05785069933318*lnbara*v2*y4*
      z2 + 0.260373564991028*w*y4*z2 + 0.09277616733835867*lnbara*w*y4*
      z2 - 0.04301078480890973*u*w*y4*z2 - 0.013022653225572441*lnbara*u*w*
      y4*z2 - 0.4807266928857714*v*w*y4*z2 - 0.1356427800167511*lnbara*v*
      w*y4*z2 + 0.28165966173252016*w2*y4*z2 + 
     0.07779208068357109*lnbara*w2*y4*z2 - 0.013136621448171954*x*y4*
      z2 + 0.000507581743944034*u*x*y4*z2 - 0.747040617041446*v*x*y4*
      z2 - 0.21034811390666633*lnbara*v*x*y4*z2 + 
     0.7480915272662646*w*x*y4*z2 + 0.21034811390666633*lnbara*w*x*y4*
      z2 + 0.001019248467243472*x2*y4*z2 - 0.018929896217160017*y5*z2 + 
     0.0006870207501448052*u*y5*z2 + 0.06988272110288132*v*y5*z2 + 
     0.021403950235092197*lnbara*v*y5*z2 - 0.06841032198704927*w*y5*z2 - 
     0.021403950235092197*lnbara*w*y5*z2 + 0.001263472670057093*x*y5*z2 + 
     0.0018904202297008643*y6*z2 - 2.2670918716867487*z3 + 
     0.6858555690183518*u*z3 - 0.251564804320496*u2*z3 + 
     0.0752590109961792*u3*z3 - 0.014812198869903107*u4*z3 + 
     0.001388160767640295*u5*z3 - 0.5311674638752799*v*z3 - 
     0.4329554475790071*lnbara*v*z3 - 8.70945468065623*u*v*z3 - 
     2.843556800957142*lnbara*u*v*z3 - 1.6011517276791973*u2*v*z3 - 
     0.46172402438324356*lnbara*u2*v*z3 - 1.576189905969709*u3*v*z3 - 
     0.465804859856524*lnbara*u3*v*z3 + 0.015644250220599377*u4*v*z3 + 
     0.006838625664216695*lnbara*u4*v*z3 + 0.8972500249744643*v2*z3 + 
     0.3500794053947276*lnbara*v2*z3 + 7.880191664548825*u*v2*z3 + 
     2.6748784087603763*lnbara*u*v2*z3 + 0.9841419128221275*u2*v2*z3 + 
     0.23760579052794478*lnbara*u2*v2*z3 + 0.43479804170814224*u3*v2*
      z3 + 0.1344756922786936*lnbara*u3*v2*z3 + 
     0.4593564384590013*v3*z3 + 0.09852674249500254*lnbara*v3*z3 - 
     3.462967340849832*u*v3*z3 - 1.1204221927730833*lnbara*u*v3*z3 + 
     0.0015185718493600246*u2*v3*z3 + 0.005606486444442812*lnbara*u2*v3*
      z3 + 0.17761627523096088*v4*z3 + 0.05371322421188824*lnbara*v4*
      z3 + 0.4131831056920087*u*v4*z3 + 0.13608853103686827*lnbara*u*v4*
      z3 + 0.01134547383834047*v5*z3 + 0.002283398903340656*lnbara*v5*
      z3 + 1.7170463437119636*w*z3 + 0.4329554475790071*lnbara*w*z3 + 
     8.335636464746663*u*w*z3 + 2.843556800957142*lnbara*u*w*z3 + 
     1.7021286302045417*u2*w*z3 + 0.46172402438324356*lnbara*u2*w*z3 + 
     1.5576572952608156*u3*w*z3 + 0.465804859856524*lnbara*u3*w*z3 - 
     0.01399123854239156*u4*w*z3 - 0.006838625664216695*lnbara*u4*w*z3 + 
     4.570769251234671*v*w*z3 + 1.7115875434794767*lnbara*v*w*z3 - 
     1.485347767785549*u*v*w*z3 - 0.6728643192060604*lnbara*u*v*w*z3 + 
     2.240193096445929*u2*v*w*z3 + 0.7375497739915498*lnbara*u2*v*w*z3 - 
     0.07656531816033134*u3*v*w*z3 - 0.03634902416016519*lnbara*u3*v*w*
      z3 - 10.997725885446338*v2*w*z3 - 3.557746295983214*lnbara*v2*w*
      z3 + 2.9792615221240184*u*v2*w*z3 + 0.9000686333403095*lnbara*u*v2*
      w*z3 - 1.8066447210612555*u2*v2*w*z3 - 0.5331767926648573*lnbara*
      u2*v2*w*z3 + 2.2465725155547824*v3*w*z3 + 
     0.7870789002790479*lnbara*v3*w*z3 + 0.1244773390895362*u*v3*w*z3 + 
     0.024916275986332885*lnbara*u*v3*w*z3 - 0.5665614064319183*v4*w*z3 - 
     0.1764702004496186*lnbara*v4*w*z3 - 6.021984802397401*w2*z3 - 
     2.0616669488742043*lnbara*w2*z3 - 6.264516709859727*u*w2*z3 - 
     2.002014089554316*lnbara*u*w2*z3 - 3.2461502369868342*u2*w2*z3 - 
     0.9751555645194946*lnbara*u2*w2*z3 - 0.35640766524496237*u3*w2*
      z3 - 0.09812666811852842*lnbara*u3*w2*z3 + 
     9.909152497593267*v*w2*z3 + 3.0912384401060025*lnbara*v*w2*z3 - 
     5.305139799956746*u*v*w2*z3 - 1.5393343490108804*lnbara*u*v*w2*z3 + 
     1.5675237969406215*u2*v*w2*z3 + 0.4579464223436356*lnbara*u2*v*w2*
      z3 + 0.3997513066521812*v2*w2*z3 + 0.08307962011998349*lnbara*v2*
      w2*z3 + 0.5497959040120096*u*v2*w2*z3 + 0.17704276038055036*lnbara*
      u*v2*w2*z3 + 0.461488349439593*v3*w2*z3 + 
     0.12645487928146082*lnbara*v3*w2*z3 + 0.817892698661721*w3*z3 + 
     0.36798111338220896*lnbara*w3*z3 + 5.761241784590613*u*w3*z3 + 
     1.759687908443654*lnbara*u*w3*z3 + 0.23971655435506595*u2*w3*z3 + 
     0.06962388387677895*lnbara*u2*w3*z3 - 1.6389294857674492*v*w3*z3 - 
     0.5238484235184111*lnbara*v*w3*z3 - 0.30529725405798797*u*v*w3*z3 - 
     0.10319808585533102*lnbara*u*v*w3*z3 - 0.8243258774524206*v2*w3*
      z3 - 0.23041500895027525*lnbara*v2*w3*z3 - 
     1.2251346623182928*w4*z3 - 0.4000233210925086*lnbara*w4*z3 - 
     0.7794687099225679*u*w4*z3 - 0.23484948154842047*lnbara*u*w4*z3 + 
     0.8263404688000965*v*w4*z3 + 0.2438589036761638*lnbara*v*w4*z3 + 
     0.0956649502705806*w5*z3 + 0.03428802753892857*lnbara*w5*z3 + 
     0.6858555690183518*x*z3 - 0.162591133426937*u*x*z3 + 
     0.03949367079133293*u2*x*z3 - 0.006882314713446356*u3*x*z3 + 
     0.0005960914163116732*u4*x*z3 + 11.250037884026336*v*x*z3 + 
     3.9890155563258416*lnbara*v*x*z3 + 0.5861690638750913*u*v*x*z3 + 
     0.04824677725141553*lnbara*u*v*x*z3 + 2.7450669338987366*u2*v*x*z3 + 
     0.8803435778848646*lnbara*u2*v*x*z3 - 0.06800667812701873*u3*v*x*
      z3 - 0.029360220026995808*lnbara*u3*v*x*z3 - 
     9.326716278242827*v2*x*z3 - 3.107847444868102*lnbara*v2*x*z3 + 
     2.4434643394917717*u*v2*x*z3 + 0.722704550671465*lnbara*u*v2*x*z3 - 
     1.4446667880342972*u2*v2*x*z3 - 0.43070638308913284*lnbara*u2*v2*x*
      z3 + 2.2465725155547824*v3*x*z3 + 0.7870789002790479*lnbara*v3*x*
      z3 + 0.1244773390895362*u*v3*x*z3 + 0.024916275986332885*lnbara*u*
      v3*x*z3 - 0.5397225167390112*v4*x*z3 - 0.16840858737356737*lnbara*
      v4*x*z3 - 11.553644836317881*w*x*z3 - 3.9890155563258416*lnbara*w*x*
      z3 - 0.5200656204166001*u*w*x*z3 - 0.04824677725141553*lnbara*u*w*x*
      z3 - 2.7556939387551567*u2*w*x*z3 - 0.8803435778848646*lnbara*u2*w*
      x*z3 + 0.0688751348615315*u3*w*x*z3 + 0.029360220026995808*lnbara*
      u3*w*x*z3 + 14.388317561563154*v*w*x*z3 + 4.420309235697655*lnbara*v*
      w*x*z3 - 10.915203494539247*u*v*w*x*z3 - 3.2792181994652556*lnbara*u*
      v*w*x*z3 + 2.4240248630265353*u2*v*w*x*z3 + 
     0.7115869901073909*lnbara*u2*v*w*x*z3 - 2.1068706790103*v2*w*x*z3 - 
     0.6955343577421946*lnbara*v2*w*x*z3 + 0.8761362502114731*u*v2*w*x*
      z3 + 0.296376278878668*lnbara*u*v2*w*x*z3 + 
     1.0840027554445029*v3*w*x*z3 + 0.30837662850281694*lnbara*v3*w*x*
      z3 - 4.961706764789829*w2*x*z3 - 1.3124617908295524*lnbara*w2*x*
      z3 + 8.457223146401986*u*w2*x*z3 + 2.5565136487937905*lnbara*u*w2*x*
      z3 - 0.9782558475006098*u2*w2*x*z3 - 0.28088060701825807*lnbara*u2*
      w2*x*z3 + 1.4784816196761565*v*w2*x*z3 + 0.43230388098155775*lnbara*
      v*w2*x*z3 - 0.6938884585226485*u*v*w2*x*z3 - 
     0.21809446900966986*lnbara*u*v*w2*x*z3 - 1.7364430725838955*v2*w2*x*
      z3 - 0.4913145532522245*lnbara*v2*w2*x*z3 - 
     1.6389294857674492*w3*x*z3 - 0.5238484235184111*lnbara*w3*x*z3 - 
     0.30529725405798797*u*w3*x*z3 - 0.10319808585533102*lnbara*u*w3*x*
      z3 + 1.5073534267839392*v*w3*x*z3 + 0.4300715944036431*lnbara*v*w3*
      x*z3 - 0.31318215309396386*w4*x*z3 - 0.07872508228066824*lnbara*w4*
      x*z3 - 0.251564804320496*x2*z3 + 0.03949367079133293*u*x2*z3 - 
     0.00560074458687052*u2*x2*z3 + 0.0004300153542944288*u3*x2*z3 + 
     6.336956677089462*v*x2*z3 + 1.8196275948475404*lnbara*v*x2*z3 - 
     3.5946630672038005*u*v*x2*z3 - 1.0856448829578143*lnbara*u*v*x2*z3 + 
     0.6616579679935063*u2*v*x2*z3 + 0.18951332857164516*lnbara*u2*v*x2*
      z3 - 0.2730340390527256*v2*x2*z3 - 0.09631911852193811*lnbara*v2*
      x2*z3 + 0.5346542533590023*u*v2*x2*z3 + 0.17261962424250146*lnbara*
      u*v2*x2*z3 + 0.461488349439593*v3*x2*z3 + 
     0.12645487928146082*lnbara*v3*x2*z3 - 6.2616857203946035*w*x2*z3 - 
     1.8196275948475404*lnbara*w*x2*z3 + 3.5848494417836383*u*w*x2*z3 + 
     1.0856448829578143*lnbara*u*w*x2*z3 - 0.660951971257608*u2*w*x2*
      z3 - 0.18951332857164516*lnbara*u2*w*x2*z3 + 
     0.6761564834039746*v*w*x2*z3 + 0.2078129654507111*lnbara*v*w*x2*z3 - 
     0.8428720597864838*u*v*w*x2*z3 - 0.25120658785234307*lnbara*u*v*w*x2*
      z3 - 1.7437824185418067*v2*w*x2*z3 - 0.4914395306435913*lnbara*v2*
      w*x2*z3 - 0.4179222450177246*w2*x2*z3 - 0.11149384692877298*lnbara*
      w2*x2*z3 + 0.3092023748907339*u*w2*x2*z3 + 
     0.0785869636098416*lnbara*u*w2*x2*z3 + 2.1080028169579204*v*w2*x2*
      z3 + 0.5953996603124057*lnbara*v*w2*x2*z3 - 
     0.8243258774524206*w3*x2*z3 - 0.23041500895027525*lnbara*w3*x2*
      z3 + 0.0752590109961792*x3*z3 - 0.006882314713446356*u*x3*z3 + 
     0.0004300153542944288*u2*x3*z3 - 0.006605505610793891*v*x3*z3 + 
     0.00038509593985133696*u*v*x3*z3 - 0.6281581353908926*v2*x3*z3 - 
     0.1774903697854292*lnbara*v2*x3*z3 - 0.006605505610793891*w*x3*z3 + 
     0.00038509593985133696*u*w*x3*z3 + 1.2574727801717278*v*w*x3*z3 + 
     0.3549807395708584*lnbara*v*w*x3*z3 - 0.6281581353908926*w2*x3*z3 - 
     0.1774903697854292*lnbara*w2*x3*z3 - 0.014812198869903107*x4*z3 + 
     0.0005960914163116732*u*x4*z3 + 0.38164842058355447*v*x4*z3 + 
     0.10823518328690307*lnbara*v*x4*z3 - 0.380500721540267*w*x4*z3 - 
     0.10823518328690307*lnbara*w*x4*z3 + 0.001388160767640295*x5*z3 + 
     0.6858555690183518*y*z3 - 0.15963636872636416*u*y*z3 + 
     0.03824563394210313*u2*y*z3 - 0.006605505610793891*u3*y*z3 + 
     0.0005687770966253271*u4*y*z3 - 14.456656314267676*v*y*z3 - 
     4.8057115258275305*lnbara*v*y*z3 - 3.6907580059131853*u*v*y*z3 - 
     1.0737089522063064*lnbara*u*v*y*z3 - 3.5123911436649746*u2*v*y*z3 - 
     1.041566396111375*lnbara*u2*v*y*z3 - 0.10728353612006211*u3*v*y*z3 - 
     0.025365208983699924*lnbara*u3*v*y*z3 + 5.769076336816564*v2*y*z3 + 
     2.0547100979131496*lnbara*v2*y*z3 + 2.371701525434619*u*v2*y*z3 + 
     0.6890237976926594*lnbara*u*v2*y*z3 + 0.729201859518042*u2*v2*y*
      z3 + 0.21972319960553072*lnbara*u2*v2*y*z3 - 
     3.369820769025423*v3*y*z3 - 1.0804948864701236*lnbara*v3*y*z3 - 
     0.09448932656403045*u*v3*y*z3 - 0.029331580383414237*lnbara*u*v3*y*
      z3 + 0.3734562378603602*v4*y*z3 + 0.1236750594519651*lnbara*v4*y*
      z3 + 14.150094597275556*w*y*z3 + 4.8057115258275305*lnbara*w*y*z3 + 
     3.7558333567491418*u*w*y*z3 + 1.0737089522063064*lnbara*u*w*y*z3 + 
     3.5021103597603167*u2*w*y*z3 + 1.041566396111375*lnbara*u2*w*y*z3 + 
     0.1081140594632303*u3*w*y*z3 + 0.025365208983699924*lnbara*u3*w*y*
      z3 - 0.09200933391270437*v*w*y*z3 - 0.14480531475704334*lnbara*v*w*y*
      z3 + 3.724170161382556*u*v*w*y*z3 + 1.1721828559253105*lnbara*u*v*w*y*
      z3 + 0.1296884569931269*u2*v*w*y*z3 + 0.020514943705544776*lnbara*
      u2*v*w*y*z3 + 3.8502028793420484*v2*w*y*z3 + 
     1.1170305738043862*lnbara*v2*w*y*z3 - 2.127445819261271*u*v2*w*y*
      z3 - 0.6316147687241351*lnbara*u*v2*w*y*z3 + 
     0.04710638371571019*v3*w*y*z3 + 0.006233587961624442*lnbara*v3*w*y*
      z3 - 5.574896354901597*w2*y*z3 - 1.909904783156106*lnbara*w2*y*
      z3 - 6.110264042573513*u*w2*y*z3 - 1.8612066536179699*lnbara*u*w2*y*
      z3 - 0.8578197999104673*u2*w2*y*z3 - 0.24023814331107549*lnbara*u2*
      w2*y*z3 - 6.183783910974188*v*w2*y*z3 - 1.7887689679030814*lnbara*v*
      w2*y*z3 + 1.617693856062018*u*v*w2*y*z3 + 
     0.4746900012067356*lnbara*u*v*w2*y*z3 + 0.2951846833599234*v2*w2*y*
      z3 + 0.10231487175528582*lnbara*v2*w2*y*z3 + 
     5.6819090881671865*w3*y*z3 + 1.7522332805688188*lnbara*w3*y*z3 + 
     0.6056692941957744*u*w3*y*z3 + 0.1862563479008138*lnbara*u*w3*y*z3 + 
     0.032891805121808636*v*w3*y*z3 - 0.002730352711332793*lnbara*v*w3*y*
      z3 - 0.7465338393563911*w4*y*z3 - 0.22949316645754256*lnbara*w4*y*
      z3 - 0.2298476323443832*x*y*z3 + 0.03580809539433375*u*x*y*z3 - 
     0.005041691635003074*u2*x*y*z3 + 0.00038509593985133696*u3*x*y*z3 - 
     2.3689366351444274*v*x*y*z3 - 0.726506814968872*lnbara*v*x*y*z3 + 
     2.469819197461433*u*v*x*y*z3 + 0.8025320427989695*lnbara*u*v*x*y*z3 + 
     0.18981304783567338*u2*v*x*y*z3 + 0.049895924897055216*lnbara*u2*v*x*
      y*z3 + 3.1903919237729643*v2*x*y*z3 + 0.9222524033868401*lnbara*v2*
      x*y*z3 - 2.203724440304821*u*v2*x*y*z3 - 0.6502064428550666*lnbara*u*
      v2*x*y*z3 + 0.04710638371571019*v3*x*y*z3 + 
     0.006233587961624442*lnbara*v3*x*y*z3 + 2.4387387085229473*w*x*y*z3 + 
     0.726506814968872*lnbara*w*x*y*z3 - 2.478798620600135*u*w*x*y*z3 - 
     0.8025320427989695*lnbara*u*w*x*y*z3 - 0.1891735970499443*u2*w*x*y*
      z3 - 0.049895924897055216*lnbara*u2*w*x*y*z3 - 
     8.881204021839554*v*w*x*y*z3 - 2.6069786638032433*lnbara*v*w*x*y*z3 + 
     3.3873280227068086*u*v*w*x*y*z3 + 0.9769151192502976*lnbara*u*v*w*x*y*
      z3 + 0.9014731642677928*v2*w*x*y*z3 + 0.28985320078620536*lnbara*v2*
      w*x*y*z3 + 5.676870448679409*w2*x*y*z3 + 1.684726260416403*lnbara*
      w2*x*y*z3 - 1.182692665484111*u*w2*x*y*z3 - 
     0.32670867639523105*lnbara*u*w2*x*y*z3 - 0.9801494429301945*v*w2*x*y*
      z3 - 0.293356436036497*lnbara*v*w2*x*y*z3 + 
     0.032891805121808636*w3*x*y*z3 - 0.002730352711332793*lnbara*w3*x*y*
      z3 + 0.06395157977258972*x2*y*z3 - 0.005855071071261368*u*x2*y*
      z3 + 0.0003652020446131235*u2*x2*y*z3 - 6.026774148018949*v*x2*y*
      z3 - 1.7632715691466476*lnbara*v*x2*y*z3 + 
     0.731117931401122*u*v*x2*y*z3 + 0.21288835092447814*lnbara*u*v*x2*y*
      z3 + 0.501248699467762*v2*x2*y*z3 + 0.1627718847199586*lnbara*v2*
      x2*y*z3 + 6.015319832123562*w*x2*y*z3 + 1.7632715691466476*lnbara*w*
      x2*y*z3 - 0.7304529490550218*u*w*x2*y*z3 - 
     0.21288835092447814*lnbara*u*w*x2*y*z3 - 0.5309427518886998*v*w*x2*y*
      z3 - 0.16015438246526315*lnbara*v*w*x2*y*z3 + 
     0.030715686889802068*w2*x2*y*z3 - 0.0026175022546954534*lnbara*w2*
      x2*y*z3 - 0.011927105098099399*x3*y*z3 + 0.00048336079466143985*u*
      x3*y*z3 + 0.00047310410784989136*v*x3*y*z3 + 
     0.00047310410784989136*w*x3*y*z3 + 0.0010740897315456516*x4*y*z3 - 
     0.251564804320496*y2*z3 + 0.03824563394210313*u*y2*z3 - 
     0.005291090684199614*u2*y2*z3 + 0.00039954507731219854*u3*y2*z3 + 
     2.8467947898195387*v*y2*z3 + 0.8710102224533081*lnbara*v*y2*z3 - 
     2.2828692800346633*u*v*y2*z3 - 0.6947964005893216*lnbara*u*v*y2*z3 - 
     0.15198537243470955*u2*v*y2*z3 - 0.04014533812336501*lnbara*u2*v*y2*
      z3 + 0.32522417184198954*v2*y2*z3 + 0.06989084748944815*lnbara*v2*
      y2*z3 + 0.6591783435173679*u*v2*y2*z3 + 0.20062980696935676*lnbara*
      u*v2*y2*z3 + 0.19122371263757587*v3*y2*z3 + 
     0.05397940765728412*lnbara*v3*y2*z3 - 2.770275796275451*w*y2*z3 - 
     0.8710102224533081*lnbara*w*y2*z3 + 2.2733271985354553*u*w*y2*z3 + 
     0.6947964005893216*lnbara*u*w*y2*z3 + 0.15265057835965268*u2*w*y2*
      z3 + 0.04014533812336501*lnbara*u2*w*y2*z3 + 
     1.046392947967706*v*w*y2*z3 + 0.35193692906048857*lnbara*v*w*y2*z3 - 
     0.18025531671901043*u*v*w*y2*z3 - 0.06303275716653367*lnbara*u*v*w*y2*
      z3 - 1.4916718912285227*v2*w*y2*z3 - 0.432716958385381*lnbara*v2*w*
      y2*z3 - 1.3869981182997964*w2*y2*z3 - 0.4218277765499367*lnbara*
      w2*y2*z3 - 0.4779542480040674*u*w2*y2*z3 - 
     0.13759704980282306*lnbara*u*w2*y2*z3 + 1.492463738628857*v*w2*y2*
      z3 + 0.4272793464214101*lnbara*v*w2*y2*z3 - 
     0.19054563887772494*w3*y2*z3 - 0.04854179569331324*lnbara*w3*y2*
      z3 + 0.06395157977258972*x*y2*z3 - 0.005780394040453321*u*x*y2*
      z3 + 0.0003568540057622896*u2*x*y2*z3 + 3.420468192644727*v*x*y2*
      z3 + 1.0593983869002586*lnbara*v*x*y2*z3 + 
     0.3069925110386899*u*v*x*y2*z3 + 0.0810097794355468*lnbara*u*v*x*y2*
      z3 - 1.3871873506084558*v2*x*y2*z3 - 0.4071365920310302*lnbara*v2*
      x*y2*z3 - 3.431997185570922*w*x*y2*z3 - 1.0593983869002586*lnbara*w*
      x*y2*z3 - 0.3063351018757037*u*w*x*y2*z3 - 
     0.0810097794355468*lnbara*u*w*x*y2*z3 + 2.0929117260759176*v*w*x*y2*
      z3 + 0.5923170046701335*lnbara*v*w*x*y2*z3 - 
     0.7046868197766329*w2*x*y2*z3 - 0.18518041263910326*lnbara*w2*x*y2*
      z3 - 0.01113298574577242*x2*y2*z3 + 0.0004489814737672761*u*x2*y2*
      z3 + 1.0839553144032048*v*x2*y2*z3 + 0.2997685920100648*lnbara*v*
      x2*y2*z3 - 1.0830627587062924*w*x2*y2*z3 - 
     0.2997685920100648*lnbara*w*x2*y2*z3 + 0.0009524091176864817*x3*y2*
      z3 + 0.0752590109961792*y3*z3 - 0.006605505610793891*u*y3*z3 + 
     0.00039954507731219854*u2*y3*z3 - 2.4157548822081845*v*y3*z3 - 
     0.7222688114635434*lnbara*v*y3*z3 - 0.10092543442474523*u*v*y3*z3 - 
     0.025545381796987655*lnbara*u*v*y3*z3 + 0.28220067646190156*v2*y3*
      z3 + 0.09025951520398619*lnbara*v2*y3*z3 + 
     2.402267061883944*w*y3*z3 + 0.7222688114635434*lnbara*w*y3*z3 + 
     0.10166794959991486*u*w*y3*z3 + 0.025545381796987655*lnbara*u*w*y3*
      z3 - 0.07704797923780887*v*w*y3*z3 - 0.028040321374660362*lnbara*v*w*
      y3*z3 - 0.20393804085263473*w2*y3*z3 - 0.06221919382932583*lnbara*
      w2*y3*z3 - 0.011927105098099399*x*y3*z3 + 
     0.00047310410784989136*u*x*y3*z3 - 0.21637657479006892*v*x*y3*z3 - 
     0.06469446777014416*lnbara*v*x*y3*z3 + 0.21733303969258025*w*x*y3*
      z3 + 0.06469446777014416*lnbara*w*x*y3*z3 + 
     0.0009524091176864817*x2*y3*z3 - 0.014812198869903107*y4*z3 + 
     0.0005687770966253271*u*y4*z3 + 0.23650678939377565*v*y4*z3 + 
     0.06767441939812754*lnbara*v*y4*z3 - 0.2353317760308018*w*y4*z3 - 
     0.06767441939812754*lnbara*w*y4*z3 + 0.0010740897315456516*x*y4*z3 + 
     0.001388160767640295*y5*z3 + 2.0230034428855013*z4 - 
     0.386061737342387*u*z4 + 0.08755029317854246*u2*z4 - 
     0.014812198869903107*u3*z4 + 0.0012633787101392036*u4*z4 + 
     1.9188684163037728*v*z4 + 0.7450820516264817*lnbara*v*z4 + 
     4.540055219513269*u*v*z4 + 1.450672032069528*lnbara*u*v*z4 + 
     1.122430115235894*u2*v*z4 + 0.3173310228787825*lnbara*u2*v*z4 + 
     0.2239680885196098*u3*v*z4 + 0.06514335645328273*lnbara*u3*v*z4 + 
     0.651338693005845*v2*z4 + 0.14754669762714998*lnbara*v2*z4 - 
     3.7835481298107045*u*v2*z4 - 1.1938747004904564*lnbara*u*v2*z4 + 
     0.05725447402071905*u2*v2*z4 + 0.021576912362776347*lnbara*u2*v2*
      z4 + 0.17761627523096088*v3*z4 + 0.05371322421188824*lnbara*v3*
      z4 + 0.4131831056920087*u*v3*z4 + 0.13608853103686827*lnbara*u*v3*
      z4 + 0.021325233798974344*v4*z4 + 0.004996244651440274*lnbara*v4*
      z4 - 2.581767811748015*w*z4 - 0.7450820516264817*lnbara*w*z4 - 
     4.408138300869153*u*w*z4 - 1.450672032069528*lnbara*u*w*z4 - 
     1.1428845474172378*u2*w*z4 - 0.3173310228787825*lnbara*u2*w*z4 - 
     0.2223252216914388*u3*w*z4 - 0.06514335645328273*lnbara*u3*w*z4 - 
     6.0865303594656925*v*w*z4 - 1.9510966301046395*lnbara*v*w*z4 + 
     2.583274807875346*u*v*w*z4 + 0.814048966327527*lnbara*u*v*w*z4 - 
     1.0214921534358323*u2*v*w*z4 - 0.3025359736703866*lnbara*u2*v*w*z4 + 
     2.4334094721170496*v2*w*z4 + 0.816632950371704*lnbara*v2*w*z4 + 
     0.10307841593891565*u*v2*w*z4 + 0.018172273278693238*lnbara*u*v2*w*
      z4 - 0.5397225167390112*v3*w*z4 - 0.16840858737356737*lnbara*v3*w*
      z4 + 5.626060910791232*w2*z4 + 1.8035499324774895*lnbara*w2*z4 + 
     1.1739473090123895*u*w2*z4 + 0.3798257341629295*lnbara*u*w2*z4 + 
     0.9661938550317448*u2*w2*z4 + 0.2809590613076102*lnbara*u2*w2*z4 - 
     1.4226442649834423*v*w2*z4 - 0.4703228534910837*lnbara*v*w2*z4 + 
     0.2656466906715195*u*v*w2*z4 + 0.08058867723285898*lnbara*u*v*w2*
      z4 + 0.39505594064956606*v2*w2*z4 + 0.11018942250071202*lnbara*v2*
      w2*z4 - 1.2251346623182928*w3*z4 - 0.4000233210925086*lnbara*w3*
      z4 - 0.7794687099225679*u*w3*z4 - 0.23484948154842047*lnbara*u*w3*
      z4 - 0.31318215309396386*v*w3*z4 - 0.07872508228066824*lnbara*v*w3*
      z4 + 0.43991353613455403*w4*z4 + 0.1319480025020833*lnbara*w4*z4 - 
     0.386061737342387*x*z4 + 0.056971794565663605*u*x*z4 - 
     0.007867552944290607*u2*x*z4 + 0.0005960914163116732*u3*x*z4 - 
     10.094036110016523*v*x*z4 - 3.28316404172915*lnbara*v*x*z4 + 
     1.0213274392385536*u*v*x*z4 + 0.336346527741606*lnbara*u*v*x*z4 - 
     0.8972167925745769*u2*v*x*z4 - 0.26783993094333697*lnbara*u2*v*x*
      z4 + 2.2331993759180806*v2*x*z4 + 0.7744184574067065*lnbara*v2*x*
      z4 + 0.08267922567088919*u*v2*x*z4 + 0.01813757575979942*lnbara*u*
      v2*x*z4 - 0.5665614064319183*v3*x*z4 - 0.1764702004496186*lnbara*
      v3*x*z4 + 10.200182207413253*w*x*z4 + 3.28316404172915*lnbara*w*x*
      z4 - 1.034655985722088*u*w*x*z4 - 0.336346527741606*lnbara*u*w*x*
      z4 + 0.8981582264276017*u2*w*x*z4 + 0.26783993094333697*lnbara*u2*w*
      x*z4 - 2.2689406137217927*v*w*x*z4 - 0.7108966463767397*lnbara*v*w*x*
      z4 + 1.0002685704936713*u*v*w*x*z4 + 0.30624466399171646*lnbara*u*v*w*
      x*z4 + 1.0094259951122324*v2*w*x*z4 + 0.2874207900901255*lnbara*v2*
      w*x*z4 + 0.01586781553997586*w2*x*z4 - 0.06352181102996676*lnbara*
      w2*x*z4 - 1.0816608287503775*u*w2*x*z4 - 0.32438223975151587*lnbara*
      u*w2*x*z4 - 1.267405871830795*v*w2*x*z4 - 
     0.35480949331667067*lnbara*v*w2*x*z4 + 0.8263404688000965*w3*x*z4 + 
     0.2438589036761638*lnbara*w3*x*z4 + 0.08755029317854246*x2*z4 - 
     0.007867552944290607*u*x2*z4 + 0.000487363420148127*u2*x2*z4 - 
     0.7906870068555788*v*x2*z4 - 0.2013783664993635*lnbara*v*x2*z4 + 
     0.3281593662661477*u*v*x2*z4 + 0.09992430356434646*lnbara*u*v*x2*
      z4 + 0.3702307156554167*v2*x2*z4 + 0.10136840292874362*lnbara*v2*
      x2*z4 + 0.7756766725864659*w*x2*z4 + 0.2013783664993635*lnbara*w*
      x2*z4 - 0.3272951320069847*u*w*x2*z4 - 0.09992430356434646*lnbara*u*
      w*x2*z4 - 0.9508002022690504*v*w*x2*z4 - 0.2673286182958272*lnbara*
      v*w*x2*z4 + 0.581863351202302*w2*x2*z4 + 
     0.1659602153670836*lnbara*w2*x2*z4 - 0.014812198869903107*x3*z4 + 
     0.0005960914163116732*u*x3*z4 - 0.380500721540267*v*x3*z4 - 
     0.10823518328690307*lnbara*v*x3*z4 + 0.38164842058355447*w*x3*z4 + 
     0.10823518328690307*lnbara*w*x3*z4 + 0.0012633787101392036*x4*z4 - 
     0.386061737342387*y*z4 + 0.056260781045097456*u*y*z4 - 
     0.007692523535941066*u2*y*z4 + 0.0005789219466621661*u3*y*z4 + 
     8.072482666125286*v*y*z4 + 2.6276787546704745*lnbara*v*y*z4 + 
     2.0086611123781046*u*v*y*z4 + 0.5668452509261017*lnbara*u*v*y*z4 + 
     0.590076818257424*u2*v*y*z4 + 0.17202385787388583*lnbara*u2*v*y*z4 - 
     2.872858759697617*v2*y*z4 - 0.9340060015696775*lnbara*v2*y*z4 - 
     0.13345592306181395*u*v2*y*z4 - 0.03997846452824747*lnbara*u*v2*y*
      z4 + 0.3734562378603602*v3*y*z4 + 0.1236750594519651*lnbara*v3*y*
      z4 - 7.96562555520799*w*y*z4 - 2.6276787546704745*lnbara*w*y*z4 - 
     2.0218463214313056*u*w*y*z4 - 0.5668452509261017*lnbara*u*w*y*z4 - 
     0.5891572387478903*u2*w*y*z4 - 0.17202385787388583*lnbara*u2*w*y*
      z4 + 1.9788571739028187*v*w*y*z4 + 0.5910023418671081*lnbara*v*w*y*
      z4 - 1.497715134559773*u*v*w*y*z4 - 0.43755320330470776*lnbara*u*v*w*
      y*z4 - 0.09314373993962562*v2*w*y*z4 - 0.033291399122784844*lnbara*
      v2*w*y*z4 + 0.8738097966923795*w2*y*z4 + 0.34300365970256935*lnbara*
      w2*y*z4 + 1.632450766309338*u*w2*y*z4 + 0.47753166783295525*lnbara*
      u*w2*y*z4 + 0.4680668096248443*v*w2*y*z4 + 
     0.13910950612836231*lnbara*v*w2*y*z4 - 0.7465338393563911*w3*y*z4 - 
     0.22949316645754256*lnbara*w3*y*z4 + 0.08203160229248424*x*y*z4 - 
     0.007320613539132591*u*x*y*z4 + 0.0004507102322422891*u2*x*y*z4 + 
     2.495271539467373*v*x*y*z4 + 0.7051754521652789*lnbara*v*x*y*z4 - 
     0.9763863300273927*u*v*x*y*z4 - 0.2916555967133236*lnbara*u*v*x*y*z4 - 
     0.06439562415041984*v2*x*y*z4 - 0.022967627645177263*lnbara*v2*x*y*
      z4 - 2.5094652412675753*w*x*y*z4 - 0.7051754521652789*lnbara*w*x*y*
      z4 + 0.977194806205722*u*w*x*y*z4 + 0.2916555967133236*lnbara*u*w*x*y*
      z4 + 0.6318226520207785*v*w*x*y*z4 + 0.19318619266094705*lnbara*v*w*x*
      y*z4 - 0.566190372464217*w2*x*y*z4 - 0.1702185650157698*lnbara*w2*x*
      y*z4 - 0.013136621448171954*x2*y*z4 + 0.000527909826103984*u*x2*y*
      z4 + 0.663692907110261*v*x2*y*z4 + 0.19636416180684646*lnbara*v*x2*
      y*z4 - 0.6626623249676024*w*x2*y*z4 - 0.19636416180684646*lnbara*w*
      x2*y*z4 + 0.0010740897315456516*x3*y*z4 + 
     0.08755029317854246*y2*z4 - 0.007692523535941066*u*y2*z4 + 
     0.0004679400724052549*u2*y2*z4 - 0.32019128431906846*v*y2*z4 - 
     0.10788365521657699*lnbara*v*y2*z4 + 0.3611306117059756*u*v*y2*z4 + 
     0.10812164650482199*lnbara*u*v*y2*z4 + 0.11854588979479715*v2*y2*
      z4 + 0.03415403184344883*lnbara*v2*y2*z4 + 
     0.305005920641606*w*y2*z4 + 0.10788365521657699*lnbara*w*y2*z4 - 
     0.3602833223629141*u*w*y2*z4 - 0.10812164650482199*lnbara*u*w*y2*
      z4 - 0.6823722786029639*v*w*y2*z4 - 0.19655624712804137*lnbara*v*w*
      y2*z4 + 0.5651566216606796*w2*y2*z4 + 0.16240221528459253*lnbara*
      w2*y2*z4 - 0.013136621448171954*x*y2*z4 + 
     0.0005230003987145713*u*x*y2*z4 - 1.1411946582446566*v*x*y2*z4 - 
     0.33097644191339387*lnbara*v*x*y2*z4 + 1.1422301498147045*w*x*y2*
      z4 + 0.33097644191339387*lnbara*w*x*y2*z4 + 
     0.001019248467243472*x2*y2*z4 - 0.014812198869903107*y3*z4 + 
     0.0005789219466621661*u*y3*z4 + 0.3249190900404754*v*y3*z4 + 
     0.09733306860121325*lnbara*v*y3*z4 - 0.3237542215275384*w*y3*z4 - 
     0.09733306860121325*lnbara*w*y3*z4 + 0.0010740897315456516*x*y3*z4 + 
     0.0012633787101392036*y4*z4 - 1.2541858328141202*z5 + 
     0.1475870262830398*u*z5 - 0.018929896217160017*u2*z5 + 
     0.001388160767640295*u3*z5 - 0.6982508102158693*v*z5 - 
     0.2748859330079916*lnbara*v*z5 - 1.8742275538737843*u*v*z5 - 
     0.5730994907821886*lnbara*u*v*z5 - 0.11604224021663521*u2*v*z5 - 
     0.03151257404486913*lnbara*u2*v*z5 + 0.03085436595372382*v2*z5 + 
     0.016119245278351776*lnbara*v2*z5 + 0.47543762922904986*u*v2*z5 + 
     0.14922477962881076*lnbara*u*v2*z5 + 0.01134547383834047*v3*z5 + 
     0.002283398903340656*lnbara*v3*z5 + 0.9506587243582331*w*z5 + 
     0.2748859330079916*lnbara*w*z5 + 1.84547571556521*u*w*z5 + 
     0.5730994907821886*lnbara*u*w*z5 + 0.11799273363683711*u2*w*z5 + 
     0.03151257404486913*lnbara*u2*w*z5 + 1.5716677873215597*v*w*z5 + 
     0.5030836809260285*lnbara*v*w*z5 - 0.15605883003169002*u*v*w*z5 - 
     0.05503807613937338*lnbara*u*v*w*z5 - 0.5247428948621341*v2*w*z5 - 
     0.16081753949969196*lnbara*v2*w*z5 - 1.6435396981777775*w2*z5 - 
     0.5192029262043804*lnbara*w2*z5 - 0.31687310350773107*u*w2*z5 - 
     0.09418670348943739*lnbara*u*w2*z5 + 0.4211542429001266*v*w2*z5 + 
     0.12424611305742275*lnbara*v*w2*z5 + 0.0956649502705806*w3*z5 + 
     0.03428802753892857*lnbara*w3*z5 + 0.1475870262830398*x*z5 - 
     0.012364843466299529*u*x*z5 + 0.0007424815539584533*u2*x*z5 + 
     3.2110778233847816*v*x*z5 + 1.0179245136226938*lnbara*v*x*z5 - 
     0.01449065568595044*u*v*x*z5 - 0.010275898356323341*lnbara*u*v*x*z5 - 
     0.46033723691317213*v2*x*z5 - 0.14606708002401708*lnbara*v2*x*z5 - 
     3.234078310073586*w*x*z5 - 1.0179245136226938*lnbara*w*x*z5 + 
     0.01575761553644318*u*w*x*z5 + 0.010275898356323341*lnbara*u*w*x*z5 + 
     0.6108819098365138*v*w*x*z5 + 0.1761800865537642*lnbara*v*w*x*z5 - 
     0.14867177025989595*w2*x*z5 - 0.030113006529747105*lnbara*w2*x*z5 - 
     0.018929896217160017*x2*z5 + 0.0007424815539584533*u*x2*z5 + 
     0.2301351338116376*v*x2*z5 + 0.05887745268544395*lnbara*v*x2*z5 - 
     0.22871819549961922*w*x2*z5 - 0.05887745268544395*lnbara*w*x2*z5 + 
     0.001388160767640295*x3*z5 + 0.1475870262830398*y*z5 - 
     0.012249129070819405*u*y*z5 + 0.0007299175618735945*u2*y*z5 - 
     3.248331013993099*v*y*z5 - 1.0069202481453856*lnbara*v*y*z5 - 
     0.23401253231882208*u*v*y*z5 - 0.06359410612179822*lnbara*u*v*y*z5 + 
     0.3612787107357604*v2*y*z5 + 0.11818205016461883*lnbara*v2*y*z5 + 
     3.2252148129088143*w*y*z5 + 1.0069202481453856*lnbara*w*y*z5 + 
     0.23526924787765785*u*w*y*z5 + 0.06359410612179822*lnbara*u*w*y*z5 - 
     0.09961325470566673*v*w*y*z5 - 0.03303181019568392*lnbara*v*w*y*z5 - 
     0.2597697450829062*w2*y*z5 - 0.08515023996893491*lnbara*w2*y*z5 - 
     0.018000480690589078*x*y*z5 + 0.0007017792893363362*u*x*y*z5 - 
     0.2133826221782255*v*x*y*z5 - 0.0583092465218906*lnbara*v*x*y*z5 + 
     0.2147385756550609*w*x*y*z5 + 0.0583092465218906*lnbara*w*x*y*z5 + 
     0.001263472670057093*x2*y*z5 - 0.018929896217160017*y2*z5 + 
     0.0007299175618735945*u*y2*z5 + 0.11415571194993857*v*y2*z5 + 
     0.03339748565253056*lnbara*v*y2*z5 - 0.11272620964583531*w*y2*z5 - 
     0.03339748565253056*lnbara*w*y2*z5 + 0.001263472670057093*x*y2*z5 + 
     0.001388160767640295*y3*z5 + 0.5112314175751462*z6 - 
     0.0339914082143444*u*z6 + 0.0018904202297008643*u2*z6 + 
     0.27910097428080544*v*z6 + 0.09609081997302192*lnbara*v*z6 + 
     0.2599997989809029*u*v*z6 + 0.07776386003863338*lnbara*u*v*z6 + 
     0.013912167390291797*v2*z6 + 0.0027513912903176212*lnbara*v2*z6 - 
     0.3370863430138181*w*z6 - 0.09609081997302192*lnbara*w*z6 - 
     0.25711383486763884*u*w*z6 - 0.07776386003863338*lnbara*u*w*z6 - 
     0.2866167180827836*v*w*z6 - 0.08634183804336616*lnbara*v*w*z6 + 
     0.2767842038860737*w2*z6 + 0.08359044675304854*lnbara*w2*z6 - 
     0.0339914082143444*x*z6 + 0.0012376963517727734*u*x*z6 - 
     0.5222803748590084*v*x*z6 - 0.15852336677018297*lnbara*v*x*z6 + 
     0.5245800234274332*w*x*z6 + 0.15852336677018297*lnbara*w*x*z6 + 
     0.0018904202297008643*x2*z6 - 0.0339914082143444*y*z6 + 
     0.001228493995858818*u*y*z6 + 0.47407842963740254*v*y*z6 + 
     0.14311605655840995*lnbara*v*y*z6 - 0.4717695787130639*w*y*z6 - 
     0.14311605655840995*lnbara*w*y*z6 + 0.0018148095406981609*x*y*z6 + 
     0.0018904202297008643*y2*z6 - 0.12321474882489236*z7 + 
     0.0035512018991495737*u*z7 - 0.033158063533986065*v*z7 - 
     0.01108690204816448*lnbara*v*z7 + 0.039205434782531995*w*z7 + 
     0.01108690204816448*lnbara*w*z7 + 0.0035512018991495737*x*z7 + 
     0.0035512018991495737*y*z7 + 0.013314221484862214*z8;
 
  ttothei = 1.0L;
  result = Hexp[0];

  for (i=1; i<nTerms; i++) {
    ttothei *= tinit;
    result += Hexp[i]*ttothei;
  }
  foo->value = result;

  ttothei = 1.0L;
  result = Hexp[1];

  for (i=2; i<nTerms; i++) {
    ttothei *= tinit;
    result += i*Hexp[i]*ttothei;
  }
  foo->deriv = result;

  return 0;
}
